
#include <detached.h>
#include <string.h>
#include <localworker.h>



bool LocalWorker::serverLevelRestartPhp()
{
    return (getConfig().isPhpHandler() && m_pDetached
            && m_pDetached->pid_info.last_modify < HttpGlobals::s_tmRestartPhp);
}


void LocalWorker::checkAndStopWorker()
{
    int s = 0;
    if (getState() == ST_GOOD && m_pRestartMarker)
    {
        if (m_pRestartMarker->checkRestart(DateTime::s_curTime)
            || serverLevelRestartPhp())
        {
            LS_INFO("[%s] detect restart request, stopping ...", getName());
            s = 1;
        }
    }
    if (!s)
    {
        if (m_forceStop)
        {
            m_forceStop = 0;
            LS_INFO("[%s] force stop requested, stopping ...", getName());
        }
        else
            return;
    }
    if (getConfig().isDetached())
    {
        if (m_pDetached && m_pDetached->pid_info.pid > 0
            && DateTime::s_curTime - m_pDetached->last_stop_time > 60)
        {
            m_pDetached->last_stop_time = DateTime::s_curTime;
            killOldDetachedInstance(&m_pDetached->pid_info);
        }
    }
    else
    {
        if (getCurrentProcGroup())
            gracefulStopCurrentProcGroup();
        else
            stop();
    }
}



static int lockFile(int fd, short lockType)
{
    int ret;
    struct flock lock;
    lock.l_type = lockType;
    lock.l_start = 0;
    lock.l_whence = SEEK_SET;
    lock.l_len = 0;
    while (1)
    {
        ret = fcntl(fd, F_SETLK, &lock);
        if ((ret == -1) && (errno == EINTR))
            continue;
        if (ret == 0)
            return ret;
        int err = errno;
        lock.l_pid = 0;
        if (fcntl(fd, F_GETLK, &lock) == 0 && lock.l_pid > 0)
            ret = lock.l_pid;
        errno = err;
        return ret;
    }
}


int LocalWorker::openLockPidFile(const char *pSocketPath)
{
    char bufPidFile[4096];
    snprintf(bufPidFile, sizeof(bufPidFile), "%s.pid",
             pSocketPath);
    int fd = nio_open(bufPidFile, O_WRONLY | O_CREAT | O_NOFOLLOW | O_TRUNC, 0644);
    if (fd == -1)
    {
        int err = errno;
        LS_NOTICE("[%s]: Failed to open pid file [%s]: %s",
                 getName(), bufPidFile, strerror(errno));
        errno = err;
        return -1;
    }
    int ret;
    if ((ret = lockFile(fd, F_WRLCK)) != 0)
    {
        int err = errno;
        LS_NOTICE("[%s]: Failed to lock pid file [%s]: %s, locked by PID: %d",
                  getName(), bufPidFile, strerror(errno), ret);
        close(fd);
        errno = err;
        return -1;
    }
    LS_INFO("[%s]: locked pid file [%s].", getName(), bufPidFile);
    fcntl(fd, F_SETFD, FD_CLOEXEC);
    return fd;
}


static int unlockClosePidFile(int fd)
{
    lockFile(fd, F_UNLCK);
    close(fd);
    return 0;
}


int LocalWorker::processPid(int pid,  ExtConnProcInfo *pProcInfo)
{
    if (pid > 0)
    {
        const char *pLogId = getName();
        if (getConfig().getRunOnStartUp() != EXTAPP_RUNONSTART_DAEMON
            && !getConfig().isDetached())
        {
            //if ( getConfig().getStartByServer() == EXTAPP_AUTOSTART_CGID )
            //    addPid( pid );
            //else
            PidRegistry::add(pid, this, pProcInfo);
        }
        if (pProcInfo)
        {
            pProcInfo->setPid(pid);
            if (pProcInfo->getExtConn())
            {
                if (pProcInfo->getExtConn()->getReq())
                    pLogId = pProcInfo->getExtConn()->getReq()->getLogId();
            }
        }
        LS_INFO("[%s] add child process pid: %d, procinfo: %p",
                pLogId, pid, pProcInfo);
        if (getConfig().isDetached())
        {
            m_pDetached->pid_info.pid = pid;
            saveDetachedPid(m_pDetached->fd_pid_file,
                           &m_pDetached->pid_info,
                           pProcInfo->getAddrListen().getUnix());
        }

    }
    else
    {
        pProcInfo->setPid(-1);
        if (pid <= -600)
            pid = -SC_500;
        if (pid < -100)
            pid = -HttpStatusCode::codeToIndex(-pid);
        LS_ERROR("[%s]: Failed to start one instance. pid: %d %s",
                 getName(), pid, (pid == -SC_508) ? "Resource limit reached!" : "");
        if (pid == -SC_508)
            m_tmResourceLimited = DateTime::s_curTime;
        if (getConfig().isDetached())
            setState(ST_NOTSTARTED);
    }
    if (getConfig().isDetached())
    {
        close(pProcInfo->getListenfd());
        pProcInfo->setListenfd(-1);
        if (m_pDetached->fd_pid_file != -1)
        {
            unlockClosePidFile(m_pDetached->fd_pid_file);
            LS_INFO("[%s]: unlocked pid file [%s.pid].",
                    getName(), getConfig().getServerAddr().getUnix());
            m_pDetached->fd_pid_file = -1;
        }
    }
    return pid;
}


void LocalWorker::onSpawnDone(ExtConnProcInfo* pProcInfo, int pid)
{
    pProcInfo->onSpawnDone(pid);
    if (getConfig().isDetached())
    {
        delete pProcInfo;
    }
    if (getConfig().getSelfManaged())
    {
        resumeWaitingConns();
        processPending();
    }
}


int LocalWorker::startOneWorker(ExtConnProcInfo *pProcInfo)
{
    int pid;
    int fd;
    if (m_tmResourceLimited == DateTime::s_curTime)
        return -SC_508;
    fd = testFd(pProcInfo);
    LocalWorkerConfig &config = getConfig();
    if (pProcInfo)
        if (pProcInfo->getPid() != -1)
        {
            pProcInfo->closeListenSock(0);
            fd = -1;
            moveToStopList(pProcInfo->getPid());
        }
    if (fd == -1)
    {
        //config.altServerAddr();
        fd = ExtWorker::startServerSock(&config, config.getBackLog());
        pProcInfo->getAddrListen() = config.getServerAddr();
        pProcInfo->setListenfd(fd);
    }
    pid = workerExec(config, fd, pProcInfo);
    if (pid != 0)
        return processPid(pid, pProcInfo);
    else
        pProcInfo->setPid(0);
    return 0;
}


bool LocalWorker::loadDetachedPid(int fd, DetachedPidInfo_t *detached_pid)
{
    return pread(fd, detached_pid, sizeof(DetachedPidInfo_t), 0)
            == sizeof(DetachedPidInfo_t);
}


void LocalWorker::saveDetachedPid(int fd, DetachedPidInfo_t *detached_pid,
                                 const char *path)
{
    struct stat st;
    nio_stat(path, &st);
    m_pDetached->pid_info.inode = st.st_ino;
    m_pDetached->pid_info.last_modify = st.st_mtime;

    pwrite(fd, detached_pid, sizeof(DetachedPidInfo_t), 0);
}


void LocalWorker::removeOldDetachedSocket()
{
    LS_INFO("[%s] remove unix socket for detached process: %s",
            getName(), getConfig().getServerAddr().getUnix());
    unlink( getConfig().getServerAddr().getUnix() );
}


void LocalWorker::killOldDetachedInstance(DetachedPidInfo_t *detached_pid)
{
    LS_INFO("[%s] kill current detached process: %d",
            getName(), m_pDetached->pid_info.pid);
    removeOldDetachedSocket();
    int pid = detached_pid->pid;
    if (pid <= 1)
        return;
    PidRegistry::addMarkToStop(pid, KILL_TYPE_TERM,
                                m_pDetached->pid_info.last_modify);
}


int LocalWorker::stopDetachedWorker()
{
    if (!getConfig().isDetached())
        return -1;
    if (!m_pDetached)
        return 0;
    if (isDetachedAlreadyRunning())
        killOldDetachedInstance(&m_pDetached->pid_info);
    else
        removeOldDetachedSocket();
    setState(ST_NOTSTARTED);
    return 0;
}


int LocalWorker::isDetachedAlreadyRunning()
{
    char bufPidFile[4096];
    int ret;
    if (m_pDetached)
    {
        if (m_pDetached->last_check_time == DateTime::s_curTime
            && m_pDetached->pid_info.pid > 0)
        {
            LS_DBG_L("[%s] is running as pid: %d.",
                     getName(), m_pDetached->pid_info.pid);
            return 1;
        }
        else if (m_pDetached->fd_pid_file != -1) //starting
        {
            LS_DBG_L("[%s] is starting up by current process.", getName());
            return 2;
        }
    }

    const GSockAddr &service_addr = getConfig().getServerAddr();
    if (service_addr.family() != PF_UNIX)
        return -1;

    struct stat st;
    if (nio_stat(service_addr.getUnix(), &st) == -1)
        return -1;

    snprintf(bufPidFile, sizeof(bufPidFile), "%s.pid",
             service_addr.getUnix());
    int fd = open(bufPidFile, O_RDONLY );
    if (fd == -1)
        return -1;

    if (!m_pDetached)
        m_pDetached = new DetachedProcess_t();
    ret = loadDetachedPid(fd, &m_pDetached->pid_info);
    close(fd);
    if (ret != 1)
    {
        m_pDetached->pid_info.pid = -1;
        return -1;
    }
    if (m_pDetached->pid_info.pid > 0
        && m_pDetached->pid_info.inode == st.st_ino
        && m_pDetached->pid_info.last_modify == st.st_mtime)
    {
        ret = kill(m_pDetached->pid_info.pid, 0);
        if (ret == 0 || errno != ESRCH)
            return 1;
    }
    m_pDetached->pid_info.pid = -1;
    return 0;
}


// return 0    start in progress
// return > 0  started/already running
// return < 0  something wrong, cannot start
//             can be `-HTTP_Status_Code`

int LocalWorker::startDetachedWorker(int force)
{
    int pid;
    int fd;
    int ret;
    int tries = 0;
    if (m_tmResourceLimited == DateTime::s_curTime)
        return -SC_508;

TRY_AGAIN:
    ret = isDetachedAlreadyRunning();
    if (ret == 1)
    {
        if (!force && m_pRestartMarker)
        {
            if (m_pRestartMarker->getLastReset()
                > m_pDetached->pid_info.last_modify)
                force = 1;
        }
        if (force || serverLevelRestartPhp())
        {
            killOldDetachedInstance(&m_pDetached->pid_info);
        }
        else
            return 1;
    }
    else if (ret == 2)
        return 0;

    LocalWorkerConfig &config = getConfig();
    const GSockAddr &service_addr = config.getServerAddr();
    int fd_pid_file = openLockPidFile(service_addr.getUnix());
    if (fd_pid_file == -1)
    {
        if (errno == EAGAIN)
        {
            if (++tries < 5)
            {
                if (tries < 4)
                {
                    LS_NOTICE("[%s]: Could be detached process being started, wait and retry: %d",
                        getName(), tries);
                    usleep(10000);
                }
                else
                {
                    char bufPidFile[4096];
                    snprintf(bufPidFile, sizeof(bufPidFile), "%s.pid",
                             service_addr.getUnix());

                    LS_NOTICE("[%s]: Could be dead lock, remove pid file [%s] and retry",
                              getName(), bufPidFile);
                    unlink(bufPidFile);
                }
                goto TRY_AGAIN;
            }
        }
        LS_ERROR("[%s]: Failed to lock pid file for [%s]: %s",
                 getName(), service_addr.getUnix(), strerror(errno));
        return -1;
    }

    removeOldDetachedSocket();
    ret = CoreSocket::listen(service_addr, config.getBackLog(), &fd, 1, -1, -1);
    if (fd == -1)
    {
        LS_ERROR("[%s]: Failed to listen socket [%s]: %s",
                 getName(), service_addr.getUnix(), strerror(errno));
        unlockClosePidFile(fd_pid_file);
        return -1;
    }
    fcntl(fd, F_SETFD, FD_CLOEXEC);

    if (!m_pDetached)
        m_pDetached = new DetachedProcess();
    m_pDetached->fd_pid_file = fd_pid_file;
    m_pDetached->last_check_time = DateTime::s_curTime;
    m_pDetached->pid_info.pid = -1;
    saveDetachedPid(fd_pid_file, &m_pDetached->pid_info,
                   service_addr.getUnix());

    ExtConnProcInfo *pProcInfo = getCurrentProcGroup();
    if (!pProcInfo)
    {
        pProcInfo = new ExtConnProcInfo();
    }
    pProcInfo->getAddrListen() = config.getServerAddr();
    pProcInfo->setListenfd(fd);
    pProcInfo->setPid(0);
    pid = workerExec(config, fd, pProcInfo);
    if (pid != 0)
    {
        ret = processPid(pid, pProcInfo);
        delete pProcInfo;
        return ret;
    }
    else
    {
        LS_DBG_L("[%s] async exec in background.", getName());
        m_pDetached->pid_info.pid = 0;
    }
    return 0;
}


static void sendKillCmdToWatchdog(pid_t pid, int kill_type, long lastmod)
{
    if (HttpGlobals::s_fdCmd == -1)
        return;
    char buf[256];
    int len = snprintf(buf, 256, "extappkill:%d:%d:%ld", pid, kill_type, lastmod);
    LS_NOTICE("sendKillCmdToWatchdog: '%.*s'.", len, buf);
    write(HttpGlobals::s_fdCmd, buf, len);
}


int PidRegistry::markToStop(pid_t pid, int kill_type)
{
    if (kill_type == KILL_TYPE_NEVER)
    {
        if (s_pSimpleList)
            return s_pSimpleList->markToStop(pid, kill_type);
    }
    else
    {
        sendKillCmdToWatchdog(pid, kill_type, 0);
    }
    return 1;
}


void PidRegistry::addMarkToStop(pid_t pid, int kill_type, long lastmod)
{
    if (kill_type == KILL_TYPE_NEVER)
    {
        if (s_pSimpleList)
            s_pSimpleList->add(pid, kill_type, NULL, NULL);
    }
    else
    {
        sendKillCmdToWatchdog(pid, kill_type, lastmod);
    }
}



int LshttpdMain::startCmdChannel()
{
    int fds[2];
    if (socketpair(AF_UNIX, SOCK_DGRAM, 0, fds) == -1)
    {
        m_fdCmd = -1;
    }
    else
    {
        fcntl(fds[0], F_SETFD, FD_CLOEXEC);
        fcntl(fds[1], F_SETFD, FD_CLOEXEC);
        fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
        m_fdCmd = fds[0];
        HttpGlobals::s_fdCmd = fds[1];
    }
    return 0;
}


enum
{
    FIX_CACHE_CREATE_ROOT,
    FIX_CACHE_UPDATE_PERM,
    FIX_CACHE_REMOVE_OLD,
    EXTAPP_KILL,
};


static int parseChildCmd(const char *pAction)
{
    if (strcasecmp(pAction, "createcacheroot") == 0)
        return FIX_CACHE_CREATE_ROOT;
    else if (strcasecmp(pAction, "fixcacheperm") == 0)
        return FIX_CACHE_UPDATE_PERM;
    else if (strcasecmp(pAction, "removecacheroot") == 0)
        return FIX_CACHE_REMOVE_OLD;
    else if (strcasecmp(pAction, "extappkill") == 0)
        return EXTAPP_KILL;
    else
        return -1;
}


#if defined(__FreeBSD__ ) || defined(__NetBSD__) || defined(__OpenBSD__) \
    || defined(macintosh) || defined(__APPLE__) || defined(__APPLE_CC__)
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/user.h>
static long getProcessStartTime(int pid)
{
    int             mib[4];
    size_t          len;
    struct kinfo_proc   kp;

    len = 4;
    mib[0] = CTL_KERN;
    mib[1] = KERN_PROC;
    mib[2] = KERN_PROC_PID;
    mib[3] = pid;
    len = sizeof(kp);
    if (sysctl(mib, 4, &kp, &len, NULL, 0) != 0) {
        return -1;
    }

    return kp.ki_start.tv_sec;
}
#endif


#if defined(linux) || defined(__linux) || defined(__linux__) || defined(__gnu_linux__)
static long getProcessStartTime(int pid)
{
    char proc_pid_path[80];
    struct stat st;
    snprintf(proc_pid_path, sizeof(proc_pid_path), "/proc/%d/exe", pid);
    int ret = lstat(proc_pid_path, &st);
    if (ret == -1)
    {
        return -1;
    }
    return st.st_mtime;
}
#endif


static int killExtApp(char *pCmd, int cmdLen)
{
    char *colon = (char *)memchr(pCmd, ':', cmdLen);
    if (!colon)
        return -1;
    *colon++ = 0;
    int pid = atoi(pCmd);
    int kill_type = atoi(colon);

    colon = (char *)memchr(colon, ':', pCmd + cmdLen - colon);
    long timestamp = 0;
    if (colon)
    {
        ++colon;
        timestamp = atol(colon);
    }

    long start_time = getProcessStartTime(pid);
    if (start_time == -1)
    {
        LS_INFO("Failed to get process [%d] start time, not running, skip killing.", pid);
        return -1;
    }

    if (timestamp && start_time - timestamp > 60)
    {
        LS_INFO("process [%d] start time does match, skip killing.", pid);
        return -1;
    }

    static int s_sigKillType[3] = { SIGUSR1, SIGTERM, SIGKILL };
    if (pid > 1 && kill_type >= KILL_TYPE_USR1 && kill_type <= KILL_TYPE_KILL)
    {
        if (::kill(pid, s_sigKillType[ kill_type + 4 ]) == 0)
        {
            LS_INFO("[CLEANUP] Send signal: %d to process: %d",
                    s_sigKillType[ kill_type + 4 ], pid);
        }
    }
    return 0;
}


int LshttpdMain::processChildCmd()
{
    char achBuf[4096];
    int n = read(m_fdCmd, &achBuf, 4095);
    if (n > 0)
    {
        achBuf[n] = 0;
        LS_NOTICE("[%d] Cmd from child: [%.*s]", m_pid, n, achBuf);

        char *colon = (char *)memchr(achBuf, ':', n);
        if (!colon)
            return 0;
        *colon++ = 0;
        int action = parseChildCmd(achBuf);
        if (action == EXTAPP_KILL)
        {
            return killExtApp(colon, &achBuf[n] - colon);
        }
        else if (action >= 0)
        {
            return fixCacheDir(action, colon, &achBuf[n] - colon);
        }
    }
    return 0;
}





