/*
MIT License

Copyright (c) 2018 LiteSpeed Technologies Inc

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#include <assert.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <sys/queue.h>

#include <lsr/xxhash.h>

#include "lshpack.h"

#define HPACK_STATIC_TABLE_SIZE   61
#define INITIAL_DYNAMIC_TABLE_SIZE  4096

/* RFC 7541, Section 4.1:
 *
 * " The size of the dynamic table is the sum of the size of its entries.
 * "
 * " The size of an entry is the sum of its name's length in octets (as
 * " defined in Section 5.2), its value's length in octets, and 32.
 */
#define DYNAMIC_ENTRY_OVERHEAD 32

#define NAME_VAL(a, b) sizeof(a) - 1, sizeof(b) - 1, (a), (b)

static const struct
{
    lshpack_strlen_t  name_len;
    lshpack_strlen_t  val_len;
    const char       *name;
    const char       *val;
}
static_table[HPACK_STATIC_TABLE_SIZE] =
{
    { NAME_VAL(":authority",                    "") },
    { NAME_VAL(":method",                       "GET") },
    { NAME_VAL(":method",                       "POST") },
    { NAME_VAL(":path",                         "/") },
    { NAME_VAL(":path",                         "/index.html") },
    { NAME_VAL(":scheme",                       "http") },
    { NAME_VAL(":scheme",                       "https") },
    { NAME_VAL(":status",                       "200") },
    { NAME_VAL(":status",                       "204") },
    { NAME_VAL(":status",                       "206") },
    { NAME_VAL(":status",                       "304") },
    { NAME_VAL(":status",                       "400") },
    { NAME_VAL(":status",                       "404") },
    { NAME_VAL(":status",                       "500") },
    { NAME_VAL("accept-charset",                "") },
    { NAME_VAL("accept-encoding",               "gzip, deflate") },
    { NAME_VAL("accept-language",               "") },
    { NAME_VAL("accept-ranges",                 "") },
    { NAME_VAL("accept",                        "") },
    { NAME_VAL("access-control-allow-origin",   "") },
    { NAME_VAL("age",                           "") },
    { NAME_VAL("allow",                         "") },
    { NAME_VAL("authorization",                 "") },
    { NAME_VAL("cache-control",                 "") },
    { NAME_VAL("content-disposition",           "") },
    { NAME_VAL("content-encoding",              "") },
    { NAME_VAL("content-language",              "") },
    { NAME_VAL("content-length",                "") },
    { NAME_VAL("content-location",              "") },
    { NAME_VAL("content-range",                 "") },
    { NAME_VAL("content-type",                  "") },
    { NAME_VAL("cookie",                        "") },
    { NAME_VAL("date",                          "") },
    { NAME_VAL("etag",                          "") },
    { NAME_VAL("expect",                        "") },
    { NAME_VAL("expires",                       "") },
    { NAME_VAL("from",                          "") },
    { NAME_VAL("host",                          "") },
    { NAME_VAL("if-match",                      "") },
    { NAME_VAL("if-modified-since",             "") },
    { NAME_VAL("if-none-match",                 "") },
    { NAME_VAL("if-range",                      "") },
    { NAME_VAL("if-unmodified-since",           "") },
    { NAME_VAL("last-modified",                 "") },
    { NAME_VAL("link",                          "") },
    { NAME_VAL("location",                      "") },
    { NAME_VAL("max-forwards",                  "") },
    { NAME_VAL("proxy-authenticate",            "") },
    { NAME_VAL("proxy-authorization",           "") },
    { NAME_VAL("range",                         "") },
    { NAME_VAL("referer",                       "") },
    { NAME_VAL("refresh",                       "") },
    { NAME_VAL("retry-after",                   "") },
    { NAME_VAL("server",                        "") },
    { NAME_VAL("set-cookie",                    "") },
    { NAME_VAL("strict-transport-security",     "") },
    { NAME_VAL("transfer-encoding",             "") },
    { NAME_VAL("user-agent",                    "") },
    { NAME_VAL("vary",                          "") },
    { NAME_VAL("via",                           "") },
    { NAME_VAL("www-authenticate",              "") }
};


struct encode_el
{
    uint32_t code;
    int      bits;
};

static const struct encode_el encode_table[257] =
{
    {     0x1ff8,    13},    //        (  0)
    {   0x7fffd8,    23},    //        (  1)
    {  0xfffffe2,    28},    //        (  2)
    {  0xfffffe3,    28},    //        (  3)
    {  0xfffffe4,    28},    //        (  4)
    {  0xfffffe5,    28},    //        (  5)
    {  0xfffffe6,    28},    //        (  6)
    {  0xfffffe7,    28},    //        (  7)
    {  0xfffffe8,    28},    //        (  8)
    {   0xffffea,    24},    //        (  9)
    { 0x3ffffffc,    30},    //        ( 10)
    {  0xfffffe9,    28},    //        ( 11)
    {  0xfffffea,    28},    //        ( 12)
    { 0x3ffffffd,    30},    //        ( 13)
    {  0xfffffeb,    28},    //        ( 14)
    {  0xfffffec,    28},    //        ( 15)
    {  0xfffffed,    28},    //        ( 16)
    {  0xfffffee,    28},    //        ( 17)
    {  0xfffffef,    28},    //        ( 18)
    {  0xffffff0,    28},    //        ( 19)
    {  0xffffff1,    28},    //        ( 20)
    {  0xffffff2,    28},    //        ( 21)
    { 0x3ffffffe,    30},    //        ( 22)
    {  0xffffff3,    28},    //        ( 23)
    {  0xffffff4,    28},    //        ( 24)
    {  0xffffff5,    28},    //        ( 25)
    {  0xffffff6,    28},    //        ( 26)
    {  0xffffff7,    28},    //        ( 27)
    {  0xffffff8,    28},    //        ( 28)
    {  0xffffff9,    28},    //        ( 29)
    {  0xffffffa,    28},    //        ( 30)
    {  0xffffffb,    28},    //        ( 31)
    {       0x14,     6},    //    ' ' ( 32)
    {      0x3f8,    10},    //    '!' ( 33)
    {      0x3f9,    10},    //    '"' ( 34)
    {      0xffa,    12},    //    '#' ( 35)
    {     0x1ff9,    13},    //    '$' ( 36)
    {       0x15,     6},    //    '%' ( 37)
    {       0xf8,     8},    //    '&' ( 38)
    {      0x7fa,    11},    //    ''' ( 39)
    {      0x3fa,    10},    //    '(' ( 40)
    {      0x3fb,    10},    //    ')' ( 41)
    {       0xf9,     8},    //    '*' ( 42)
    {      0x7fb,    11},    //    '+' ( 43)
    {       0xfa,     8},    //    ',' ( 44)
    {       0x16,     6},    //    '-' ( 45)
    {       0x17,     6},    //    '.' ( 46)
    {       0x18,     6},    //    '/' ( 47)
    {        0x0,     5},    //    '0' ( 48)
    {        0x1,     5},    //    '1' ( 49)
    {        0x2,     5},    //    '2' ( 50)
    {       0x19,     6},    //    '3' ( 51)
    {       0x1a,     6},    //    '4' ( 52)
    {       0x1b,     6},    //    '5' ( 53)
    {       0x1c,     6},    //    '6' ( 54)
    {       0x1d,     6},    //    '7' ( 55)
    {       0x1e,     6},    //    '8' ( 56)
    {       0x1f,     6},    //    '9' ( 57)
    {       0x5c,     7},    //    ':' ( 58)
    {       0xfb,     8},    //    ';' ( 59)
    {     0x7ffc,    15},    //    '<' ( 60)
    {       0x20,     6},    //    '=' ( 61)
    {      0xffb,    12},    //    '>' ( 62)
    {      0x3fc,    10},    //    '?' ( 63)
    {     0x1ffa,    13},    //    '@' ( 64)
    {       0x21,     6},    //    'A' ( 65)
    {       0x5d,     7},    //    'B' ( 66)
    {       0x5e,     7},    //    'C' ( 67)
    {       0x5f,     7},    //    'D' ( 68)
    {       0x60,     7},    //    'E' ( 69)
    {       0x61,     7},    //    'F' ( 70)
    {       0x62,     7},    //    'G' ( 71)
    {       0x63,     7},    //    'H' ( 72)
    {       0x64,     7},    //    'I' ( 73)
    {       0x65,     7},    //    'J' ( 74)
    {       0x66,     7},    //    'K' ( 75)
    {       0x67,     7},    //    'L' ( 76)
    {       0x68,     7},    //    'M' ( 77)
    {       0x69,     7},    //    'N' ( 78)
    {       0x6a,     7},    //    'O' ( 79)
    {       0x6b,     7},    //    'P' ( 80)
    {       0x6c,     7},    //    'Q' ( 81)
    {       0x6d,     7},    //    'R' ( 82)
    {       0x6e,     7},    //    'S' ( 83)
    {       0x6f,     7},    //    'T' ( 84)
    {       0x70,     7},    //    'U' ( 85)
    {       0x71,     7},    //    'V' ( 86)
    {       0x72,     7},    //    'W' ( 87)
    {       0xfc,     8},    //    'X' ( 88)
    {       0x73,     7},    //    'Y' ( 89)
    {       0xfd,     8},    //    'Z' ( 90)
    {     0x1ffb,    13},    //    '[' ( 91)
    {    0x7fff0,    19},    //    '\' ( 92)
    {     0x1ffc,    13},    //    ']' ( 93)
    {     0x3ffc,    14},    //    '^' ( 94)
    {       0x22,     6},    //    '_' ( 95)
    {     0x7ffd,    15},    //    '`' ( 96)
    {        0x3,     5},    //    'a' ( 97)
    {       0x23,     6},    //    'b' ( 98)
    {        0x4,     5},    //    'c' ( 99)
    {       0x24,     6},    //    'd' (100)
    {        0x5,     5},    //    'e' (101)
    {       0x25,     6},    //    'f' (102)
    {       0x26,     6},    //    'g' (103)
    {       0x27,     6},    //    'h' (104)
    {        0x6,     5},    //    'i' (105)
    {       0x74,     7},    //    'j' (106)
    {       0x75,     7},    //    'k' (107)
    {       0x28,     6},    //    'l' (108)
    {       0x29,     6},    //    'm' (109)
    {       0x2a,     6},    //    'n' (110)
    {        0x7,     5},    //    'o' (111)
    {       0x2b,     6},    //    'p' (112)
    {       0x76,     7},    //    'q' (113)
    {       0x2c,     6},    //    'r' (114)
    {        0x8,     5},    //    's' (115)
    {        0x9,     5},    //    't' (116)
    {       0x2d,     6},    //    'u' (117)
    {       0x77,     7},    //    'v' (118)
    {       0x78,     7},    //    'w' (119)
    {       0x79,     7},    //    'x' (120)
    {       0x7a,     7},    //    'y' (121)
    {       0x7b,     7},    //    'z' (122)
    {     0x7ffe,    15},    //    '{' (123)
    {      0x7fc,    11},    //    '|' (124)
    {     0x3ffd,    14},    //    '}' (125)
    {     0x1ffd,    13},    //    '~' (126)
    {  0xffffffc,    28},    //        (127)
    {    0xfffe6,    20},    //        (128)
    {   0x3fffd2,    22},    //        (129)
    {    0xfffe7,    20},    //        (130)
    {    0xfffe8,    20},    //        (131)
    {   0x3fffd3,    22},    //        (132)
    {   0x3fffd4,    22},    //        (133)
    {   0x3fffd5,    22},    //        (134)
    {   0x7fffd9,    23},    //        (135)
    {   0x3fffd6,    22},    //        (136)
    {   0x7fffda,    23},    //        (137)
    {   0x7fffdb,    23},    //        (138)
    {   0x7fffdc,    23},    //        (139)
    {   0x7fffdd,    23},    //        (140)
    {   0x7fffde,    23},    //        (141)
    {   0xffffeb,    24},    //        (142)
    {   0x7fffdf,    23},    //        (143)
    {   0xffffec,    24},    //        (144)
    {   0xffffed,    24},    //        (145)
    {   0x3fffd7,    22},    //        (146)
    {   0x7fffe0,    23},    //        (147)
    {   0xffffee,    24},    //        (148)
    {   0x7fffe1,    23},    //        (149)
    {   0x7fffe2,    23},    //        (150)
    {   0x7fffe3,    23},    //        (151)
    {   0x7fffe4,    23},    //        (152)
    {   0x1fffdc,    21},    //        (153)
    {   0x3fffd8,    22},    //        (154)
    {   0x7fffe5,    23},    //        (155)
    {   0x3fffd9,    22},    //        (156)
    {   0x7fffe6,    23},    //        (157)
    {   0x7fffe7,    23},    //        (158)
    {   0xffffef,    24},    //        (159)
    {   0x3fffda,    22},    //        (160)
    {   0x1fffdd,    21},    //        (161)
    {    0xfffe9,    20},    //        (162)
    {   0x3fffdb,    22},    //        (163)
    {   0x3fffdc,    22},    //        (164)
    {   0x7fffe8,    23},    //        (165)
    {   0x7fffe9,    23},    //        (166)
    {   0x1fffde,    21},    //        (167)
    {   0x7fffea,    23},    //        (168)
    {   0x3fffdd,    22},    //        (169)
    {   0x3fffde,    22},    //        (170)
    {   0xfffff0,    24},    //        (171)
    {   0x1fffdf,    21},    //        (172)
    {   0x3fffdf,    22},    //        (173)
    {   0x7fffeb,    23},    //        (174)
    {   0x7fffec,    23},    //        (175)
    {   0x1fffe0,    21},    //        (176)
    {   0x1fffe1,    21},    //        (177)
    {   0x3fffe0,    22},    //        (178)
    {   0x1fffe2,    21},    //        (179)
    {   0x7fffed,    23},    //        (180)
    {   0x3fffe1,    22},    //        (181)
    {   0x7fffee,    23},    //        (182)
    {   0x7fffef,    23},    //        (183)
    {    0xfffea,    20},    //        (184)
    {   0x3fffe2,    22},    //        (185)
    {   0x3fffe3,    22},    //        (186)
    {   0x3fffe4,    22},    //        (187)
    {   0x7ffff0,    23},    //        (188)
    {   0x3fffe5,    22},    //        (189)
    {   0x3fffe6,    22},    //        (190)
    {   0x7ffff1,    23},    //        (191)
    {  0x3ffffe0,    26},    //        (192)
    {  0x3ffffe1,    26},    //        (193)
    {    0xfffeb,    20},    //        (194)
    {    0x7fff1,    19},    //        (195)
    {   0x3fffe7,    22},    //        (196)
    {   0x7ffff2,    23},    //        (197)
    {   0x3fffe8,    22},    //        (198)
    {  0x1ffffec,    25},    //        (199)
    {  0x3ffffe2,    26},    //        (200)
    {  0x3ffffe3,    26},    //        (201)
    {  0x3ffffe4,    26},    //        (202)
    {  0x7ffffde,    27},    //        (203)
    {  0x7ffffdf,    27},    //        (204)
    {  0x3ffffe5,    26},    //        (205)
    {   0xfffff1,    24},    //        (206)
    {  0x1ffffed,    25},    //        (207)
    {    0x7fff2,    19},    //        (208)
    {   0x1fffe3,    21},    //        (209)
    {  0x3ffffe6,    26},    //        (210)
    {  0x7ffffe0,    27},    //        (211)
    {  0x7ffffe1,    27},    //        (212)
    {  0x3ffffe7,    26},    //        (213)
    {  0x7ffffe2,    27},    //        (214)
    {   0xfffff2,    24},    //        (215)
    {   0x1fffe4,    21},    //        (216)
    {   0x1fffe5,    21},    //        (217)
    {  0x3ffffe8,    26},    //        (218)
    {  0x3ffffe9,    26},    //        (219)
    {  0xffffffd,    28},    //        (220)
    {  0x7ffffe3,    27},    //        (221)
    {  0x7ffffe4,    27},    //        (222)
    {  0x7ffffe5,    27},    //        (223)
    {    0xfffec,    20},    //        (224)
    {   0xfffff3,    24},    //        (225)
    {    0xfffed,    20},    //        (226)
    {   0x1fffe6,    21},    //        (227)
    {   0x3fffe9,    22},    //        (228)
    {   0x1fffe7,    21},    //        (229)
    {   0x1fffe8,    21},    //        (230)
    {   0x7ffff3,    23},    //        (231)
    {   0x3fffea,    22},    //        (232)
    {   0x3fffeb,    22},    //        (233)
    {  0x1ffffee,    25},    //        (234)
    {  0x1ffffef,    25},    //        (235)
    {   0xfffff4,    24},    //        (236)
    {   0xfffff5,    24},    //        (237)
    {  0x3ffffea,    26},    //        (238)
    {   0x7ffff4,    23},    //        (239)
    {  0x3ffffeb,    26},    //        (240)
    {  0x7ffffe6,    27},    //        (241)
    {  0x3ffffec,    26},    //        (242)
    {  0x3ffffed,    26},    //        (243)
    {  0x7ffffe7,    27},    //        (244)
    {  0x7ffffe8,    27},    //        (245)
    {  0x7ffffe9,    27},    //        (246)
    {  0x7ffffea,    27},    //        (247)
    {  0x7ffffeb,    27},    //        (248)
    {  0xffffffe,    28},    //        (249)
    {  0x7ffffec,    27},    //        (250)
    {  0x7ffffed,    27},    //        (251)
    {  0x7ffffee,    27},    //        (252)
    {  0x7ffffef,    27},    //        (253)
    {  0x7fffff0,    27},    //        (254)
    {  0x3ffffee,    26},    //        (255)
    { 0x3fffffff,    30}    //    EOS (256)
};


struct decode_el
{
    uint8_t state;
    uint8_t flags;
    uint8_t sym;
};

static const struct decode_el decode_tables[256][16] =
{
    /* 0 */
    {
        { 4, 0x00, 0 },
        { 5, 0x00, 0 },
        { 7, 0x00, 0 },
        { 8, 0x00, 0 },
        { 11, 0x00, 0 },
        { 12, 0x00, 0 },
        { 16, 0x00, 0 },
        { 19, 0x00, 0 },
        { 25, 0x00, 0 },
        { 28, 0x00, 0 },
        { 32, 0x00, 0 },
        { 35, 0x00, 0 },
        { 42, 0x00, 0 },
        { 49, 0x00, 0 },
        { 57, 0x00, 0 },
        { 64, 0x01, 0 },
    },
    /* 1 */
    {
        { 0, 0x03, 48 },
        { 0, 0x03, 49 },
        { 0, 0x03, 50 },
        { 0, 0x03, 97 },
        { 0, 0x03, 99 },
        { 0, 0x03, 101 },
        { 0, 0x03, 105 },
        { 0, 0x03, 111 },
        { 0, 0x03, 115 },
        { 0, 0x03, 116 },
        { 13, 0x00, 0 },
        { 14, 0x00, 0 },
        { 17, 0x00, 0 },
        { 18, 0x00, 0 },
        { 20, 0x00, 0 },
        { 21, 0x00, 0 },
    },
    /* 2 */
    {
        { 1, 0x02, 48 },
        { 22, 0x03, 48 },
        { 1, 0x02, 49 },
        { 22, 0x03, 49 },
        { 1, 0x02, 50 },
        { 22, 0x03, 50 },
        { 1, 0x02, 97 },
        { 22, 0x03, 97 },
        { 1, 0x02, 99 },
        { 22, 0x03, 99 },
        { 1, 0x02, 101 },
        { 22, 0x03, 101 },
        { 1, 0x02, 105 },
        { 22, 0x03, 105 },
        { 1, 0x02, 111 },
        { 22, 0x03, 111 },
    },
    /* 3 */
    {
        { 2, 0x02, 48 },
        { 9, 0x02, 48 },
        { 23, 0x02, 48 },
        { 40, 0x03, 48 },
        { 2, 0x02, 49 },
        { 9, 0x02, 49 },
        { 23, 0x02, 49 },
        { 40, 0x03, 49 },
        { 2, 0x02, 50 },
        { 9, 0x02, 50 },
        { 23, 0x02, 50 },
        { 40, 0x03, 50 },
        { 2, 0x02, 97 },
        { 9, 0x02, 97 },
        { 23, 0x02, 97 },
        { 40, 0x03, 97 },
    },
    /* 4 */
    {
        { 3, 0x02, 48 },
        { 6, 0x02, 48 },
        { 10, 0x02, 48 },
        { 15, 0x02, 48 },
        { 24, 0x02, 48 },
        { 31, 0x02, 48 },
        { 41, 0x02, 48 },
        { 56, 0x03, 48 },
        { 3, 0x02, 49 },
        { 6, 0x02, 49 },
        { 10, 0x02, 49 },
        { 15, 0x02, 49 },
        { 24, 0x02, 49 },
        { 31, 0x02, 49 },
        { 41, 0x02, 49 },
        { 56, 0x03, 49 },
    },
    /* 5 */
    {
        { 3, 0x02, 50 },
        { 6, 0x02, 50 },
        { 10, 0x02, 50 },
        { 15, 0x02, 50 },
        { 24, 0x02, 50 },
        { 31, 0x02, 50 },
        { 41, 0x02, 50 },
        { 56, 0x03, 50 },
        { 3, 0x02, 97 },
        { 6, 0x02, 97 },
        { 10, 0x02, 97 },
        { 15, 0x02, 97 },
        { 24, 0x02, 97 },
        { 31, 0x02, 97 },
        { 41, 0x02, 97 },
        { 56, 0x03, 97 },
    },
    /* 6 */
    {
        { 2, 0x02, 99 },
        { 9, 0x02, 99 },
        { 23, 0x02, 99 },
        { 40, 0x03, 99 },
        { 2, 0x02, 101 },
        { 9, 0x02, 101 },
        { 23, 0x02, 101 },
        { 40, 0x03, 101 },
        { 2, 0x02, 105 },
        { 9, 0x02, 105 },
        { 23, 0x02, 105 },
        { 40, 0x03, 105 },
        { 2, 0x02, 111 },
        { 9, 0x02, 111 },
        { 23, 0x02, 111 },
        { 40, 0x03, 111 },
    },
    /* 7 */
    {
        { 3, 0x02, 99 },
        { 6, 0x02, 99 },
        { 10, 0x02, 99 },
        { 15, 0x02, 99 },
        { 24, 0x02, 99 },
        { 31, 0x02, 99 },
        { 41, 0x02, 99 },
        { 56, 0x03, 99 },
        { 3, 0x02, 101 },
        { 6, 0x02, 101 },
        { 10, 0x02, 101 },
        { 15, 0x02, 101 },
        { 24, 0x02, 101 },
        { 31, 0x02, 101 },
        { 41, 0x02, 101 },
        { 56, 0x03, 101 },
    },
    /* 8 */
    {
        { 3, 0x02, 105 },
        { 6, 0x02, 105 },
        { 10, 0x02, 105 },
        { 15, 0x02, 105 },
        { 24, 0x02, 105 },
        { 31, 0x02, 105 },
        { 41, 0x02, 105 },
        { 56, 0x03, 105 },
        { 3, 0x02, 111 },
        { 6, 0x02, 111 },
        { 10, 0x02, 111 },
        { 15, 0x02, 111 },
        { 24, 0x02, 111 },
        { 31, 0x02, 111 },
        { 41, 0x02, 111 },
        { 56, 0x03, 111 },
    },
    /* 9 */
    {
        { 1, 0x02, 115 },
        { 22, 0x03, 115 },
        { 1, 0x02, 116 },
        { 22, 0x03, 116 },
        { 0, 0x03, 32 },
        { 0, 0x03, 37 },
        { 0, 0x03, 45 },
        { 0, 0x03, 46 },
        { 0, 0x03, 47 },
        { 0, 0x03, 51 },
        { 0, 0x03, 52 },
        { 0, 0x03, 53 },
        { 0, 0x03, 54 },
        { 0, 0x03, 55 },
        { 0, 0x03, 56 },
        { 0, 0x03, 57 },
    },
    /* 10 */
    {
        { 2, 0x02, 115 },
        { 9, 0x02, 115 },
        { 23, 0x02, 115 },
        { 40, 0x03, 115 },
        { 2, 0x02, 116 },
        { 9, 0x02, 116 },
        { 23, 0x02, 116 },
        { 40, 0x03, 116 },
        { 1, 0x02, 32 },
        { 22, 0x03, 32 },
        { 1, 0x02, 37 },
        { 22, 0x03, 37 },
        { 1, 0x02, 45 },
        { 22, 0x03, 45 },
        { 1, 0x02, 46 },
        { 22, 0x03, 46 },
    },
    /* 11 */
    {
        { 3, 0x02, 115 },
        { 6, 0x02, 115 },
        { 10, 0x02, 115 },
        { 15, 0x02, 115 },
        { 24, 0x02, 115 },
        { 31, 0x02, 115 },
        { 41, 0x02, 115 },
        { 56, 0x03, 115 },
        { 3, 0x02, 116 },
        { 6, 0x02, 116 },
        { 10, 0x02, 116 },
        { 15, 0x02, 116 },
        { 24, 0x02, 116 },
        { 31, 0x02, 116 },
        { 41, 0x02, 116 },
        { 56, 0x03, 116 },
    },
    /* 12 */
    {
        { 2, 0x02, 32 },
        { 9, 0x02, 32 },
        { 23, 0x02, 32 },
        { 40, 0x03, 32 },
        { 2, 0x02, 37 },
        { 9, 0x02, 37 },
        { 23, 0x02, 37 },
        { 40, 0x03, 37 },
        { 2, 0x02, 45 },
        { 9, 0x02, 45 },
        { 23, 0x02, 45 },
        { 40, 0x03, 45 },
        { 2, 0x02, 46 },
        { 9, 0x02, 46 },
        { 23, 0x02, 46 },
        { 40, 0x03, 46 },
    },
    /* 13 */
    {
        { 3, 0x02, 32 },
        { 6, 0x02, 32 },
        { 10, 0x02, 32 },
        { 15, 0x02, 32 },
        { 24, 0x02, 32 },
        { 31, 0x02, 32 },
        { 41, 0x02, 32 },
        { 56, 0x03, 32 },
        { 3, 0x02, 37 },
        { 6, 0x02, 37 },
        { 10, 0x02, 37 },
        { 15, 0x02, 37 },
        { 24, 0x02, 37 },
        { 31, 0x02, 37 },
        { 41, 0x02, 37 },
        { 56, 0x03, 37 },
    },
    /* 14 */
    {
        { 3, 0x02, 45 },
        { 6, 0x02, 45 },
        { 10, 0x02, 45 },
        { 15, 0x02, 45 },
        { 24, 0x02, 45 },
        { 31, 0x02, 45 },
        { 41, 0x02, 45 },
        { 56, 0x03, 45 },
        { 3, 0x02, 46 },
        { 6, 0x02, 46 },
        { 10, 0x02, 46 },
        { 15, 0x02, 46 },
        { 24, 0x02, 46 },
        { 31, 0x02, 46 },
        { 41, 0x02, 46 },
        { 56, 0x03, 46 },
    },
    /* 15 */
    {
        { 1, 0x02, 47 },
        { 22, 0x03, 47 },
        { 1, 0x02, 51 },
        { 22, 0x03, 51 },
        { 1, 0x02, 52 },
        { 22, 0x03, 52 },
        { 1, 0x02, 53 },
        { 22, 0x03, 53 },
        { 1, 0x02, 54 },
        { 22, 0x03, 54 },
        { 1, 0x02, 55 },
        { 22, 0x03, 55 },
        { 1, 0x02, 56 },
        { 22, 0x03, 56 },
        { 1, 0x02, 57 },
        { 22, 0x03, 57 },
    },
    /* 16 */
    {
        { 2, 0x02, 47 },
        { 9, 0x02, 47 },
        { 23, 0x02, 47 },
        { 40, 0x03, 47 },
        { 2, 0x02, 51 },
        { 9, 0x02, 51 },
        { 23, 0x02, 51 },
        { 40, 0x03, 51 },
        { 2, 0x02, 52 },
        { 9, 0x02, 52 },
        { 23, 0x02, 52 },
        { 40, 0x03, 52 },
        { 2, 0x02, 53 },
        { 9, 0x02, 53 },
        { 23, 0x02, 53 },
        { 40, 0x03, 53 },
    },
    /* 17 */
    {
        { 3, 0x02, 47 },
        { 6, 0x02, 47 },
        { 10, 0x02, 47 },
        { 15, 0x02, 47 },
        { 24, 0x02, 47 },
        { 31, 0x02, 47 },
        { 41, 0x02, 47 },
        { 56, 0x03, 47 },
        { 3, 0x02, 51 },
        { 6, 0x02, 51 },
        { 10, 0x02, 51 },
        { 15, 0x02, 51 },
        { 24, 0x02, 51 },
        { 31, 0x02, 51 },
        { 41, 0x02, 51 },
        { 56, 0x03, 51 },
    },
    /* 18 */
    {
        { 3, 0x02, 52 },
        { 6, 0x02, 52 },
        { 10, 0x02, 52 },
        { 15, 0x02, 52 },
        { 24, 0x02, 52 },
        { 31, 0x02, 52 },
        { 41, 0x02, 52 },
        { 56, 0x03, 52 },
        { 3, 0x02, 53 },
        { 6, 0x02, 53 },
        { 10, 0x02, 53 },
        { 15, 0x02, 53 },
        { 24, 0x02, 53 },
        { 31, 0x02, 53 },
        { 41, 0x02, 53 },
        { 56, 0x03, 53 },
    },
    /* 19 */
    {
        { 2, 0x02, 54 },
        { 9, 0x02, 54 },
        { 23, 0x02, 54 },
        { 40, 0x03, 54 },
        { 2, 0x02, 55 },
        { 9, 0x02, 55 },
        { 23, 0x02, 55 },
        { 40, 0x03, 55 },
        { 2, 0x02, 56 },
        { 9, 0x02, 56 },
        { 23, 0x02, 56 },
        { 40, 0x03, 56 },
        { 2, 0x02, 57 },
        { 9, 0x02, 57 },
        { 23, 0x02, 57 },
        { 40, 0x03, 57 },
    },
    /* 20 */
    {
        { 3, 0x02, 54 },
        { 6, 0x02, 54 },
        { 10, 0x02, 54 },
        { 15, 0x02, 54 },
        { 24, 0x02, 54 },
        { 31, 0x02, 54 },
        { 41, 0x02, 54 },
        { 56, 0x03, 54 },
        { 3, 0x02, 55 },
        { 6, 0x02, 55 },
        { 10, 0x02, 55 },
        { 15, 0x02, 55 },
        { 24, 0x02, 55 },
        { 31, 0x02, 55 },
        { 41, 0x02, 55 },
        { 56, 0x03, 55 },
    },
    /* 21 */
    {
        { 3, 0x02, 56 },
        { 6, 0x02, 56 },
        { 10, 0x02, 56 },
        { 15, 0x02, 56 },
        { 24, 0x02, 56 },
        { 31, 0x02, 56 },
        { 41, 0x02, 56 },
        { 56, 0x03, 56 },
        { 3, 0x02, 57 },
        { 6, 0x02, 57 },
        { 10, 0x02, 57 },
        { 15, 0x02, 57 },
        { 24, 0x02, 57 },
        { 31, 0x02, 57 },
        { 41, 0x02, 57 },
        { 56, 0x03, 57 },
    },
    /* 22 */
    {
        { 26, 0x00, 0 },
        { 27, 0x00, 0 },
        { 29, 0x00, 0 },
        { 30, 0x00, 0 },
        { 33, 0x00, 0 },
        { 34, 0x00, 0 },
        { 36, 0x00, 0 },
        { 37, 0x00, 0 },
        { 43, 0x00, 0 },
        { 46, 0x00, 0 },
        { 50, 0x00, 0 },
        { 53, 0x00, 0 },
        { 58, 0x00, 0 },
        { 61, 0x00, 0 },
        { 65, 0x00, 0 },
        { 68, 0x01, 0 },
    },
    /* 23 */
    {
        { 0, 0x03, 61 },
        { 0, 0x03, 65 },
        { 0, 0x03, 95 },
        { 0, 0x03, 98 },
        { 0, 0x03, 100 },
        { 0, 0x03, 102 },
        { 0, 0x03, 103 },
        { 0, 0x03, 104 },
        { 0, 0x03, 108 },
        { 0, 0x03, 109 },
        { 0, 0x03, 110 },
        { 0, 0x03, 112 },
        { 0, 0x03, 114 },
        { 0, 0x03, 117 },
        { 38, 0x00, 0 },
        { 39, 0x00, 0 },
    },
    /* 24 */
    {
        { 1, 0x02, 61 },
        { 22, 0x03, 61 },
        { 1, 0x02, 65 },
        { 22, 0x03, 65 },
        { 1, 0x02, 95 },
        { 22, 0x03, 95 },
        { 1, 0x02, 98 },
        { 22, 0x03, 98 },
        { 1, 0x02, 100 },
        { 22, 0x03, 100 },
        { 1, 0x02, 102 },
        { 22, 0x03, 102 },
        { 1, 0x02, 103 },
        { 22, 0x03, 103 },
        { 1, 0x02, 104 },
        { 22, 0x03, 104 },
    },
    /* 25 */
    {
        { 2, 0x02, 61 },
        { 9, 0x02, 61 },
        { 23, 0x02, 61 },
        { 40, 0x03, 61 },
        { 2, 0x02, 65 },
        { 9, 0x02, 65 },
        { 23, 0x02, 65 },
        { 40, 0x03, 65 },
        { 2, 0x02, 95 },
        { 9, 0x02, 95 },
        { 23, 0x02, 95 },
        { 40, 0x03, 95 },
        { 2, 0x02, 98 },
        { 9, 0x02, 98 },
        { 23, 0x02, 98 },
        { 40, 0x03, 98 },
    },
    /* 26 */
    {
        { 3, 0x02, 61 },
        { 6, 0x02, 61 },
        { 10, 0x02, 61 },
        { 15, 0x02, 61 },
        { 24, 0x02, 61 },
        { 31, 0x02, 61 },
        { 41, 0x02, 61 },
        { 56, 0x03, 61 },
        { 3, 0x02, 65 },
        { 6, 0x02, 65 },
        { 10, 0x02, 65 },
        { 15, 0x02, 65 },
        { 24, 0x02, 65 },
        { 31, 0x02, 65 },
        { 41, 0x02, 65 },
        { 56, 0x03, 65 },
    },
    /* 27 */
    {
        { 3, 0x02, 95 },
        { 6, 0x02, 95 },
        { 10, 0x02, 95 },
        { 15, 0x02, 95 },
        { 24, 0x02, 95 },
        { 31, 0x02, 95 },
        { 41, 0x02, 95 },
        { 56, 0x03, 95 },
        { 3, 0x02, 98 },
        { 6, 0x02, 98 },
        { 10, 0x02, 98 },
        { 15, 0x02, 98 },
        { 24, 0x02, 98 },
        { 31, 0x02, 98 },
        { 41, 0x02, 98 },
        { 56, 0x03, 98 },
    },
    /* 28 */
    {
        { 2, 0x02, 100 },
        { 9, 0x02, 100 },
        { 23, 0x02, 100 },
        { 40, 0x03, 100 },
        { 2, 0x02, 102 },
        { 9, 0x02, 102 },
        { 23, 0x02, 102 },
        { 40, 0x03, 102 },
        { 2, 0x02, 103 },
        { 9, 0x02, 103 },
        { 23, 0x02, 103 },
        { 40, 0x03, 103 },
        { 2, 0x02, 104 },
        { 9, 0x02, 104 },
        { 23, 0x02, 104 },
        { 40, 0x03, 104 },
    },
    /* 29 */
    {
        { 3, 0x02, 100 },
        { 6, 0x02, 100 },
        { 10, 0x02, 100 },
        { 15, 0x02, 100 },
        { 24, 0x02, 100 },
        { 31, 0x02, 100 },
        { 41, 0x02, 100 },
        { 56, 0x03, 100 },
        { 3, 0x02, 102 },
        { 6, 0x02, 102 },
        { 10, 0x02, 102 },
        { 15, 0x02, 102 },
        { 24, 0x02, 102 },
        { 31, 0x02, 102 },
        { 41, 0x02, 102 },
        { 56, 0x03, 102 },
    },
    /* 30 */
    {
        { 3, 0x02, 103 },
        { 6, 0x02, 103 },
        { 10, 0x02, 103 },
        { 15, 0x02, 103 },
        { 24, 0x02, 103 },
        { 31, 0x02, 103 },
        { 41, 0x02, 103 },
        { 56, 0x03, 103 },
        { 3, 0x02, 104 },
        { 6, 0x02, 104 },
        { 10, 0x02, 104 },
        { 15, 0x02, 104 },
        { 24, 0x02, 104 },
        { 31, 0x02, 104 },
        { 41, 0x02, 104 },
        { 56, 0x03, 104 },
    },
    /* 31 */
    {
        { 1, 0x02, 108 },
        { 22, 0x03, 108 },
        { 1, 0x02, 109 },
        { 22, 0x03, 109 },
        { 1, 0x02, 110 },
        { 22, 0x03, 110 },
        { 1, 0x02, 112 },
        { 22, 0x03, 112 },
        { 1, 0x02, 114 },
        { 22, 0x03, 114 },
        { 1, 0x02, 117 },
        { 22, 0x03, 117 },
        { 0, 0x03, 58 },
        { 0, 0x03, 66 },
        { 0, 0x03, 67 },
        { 0, 0x03, 68 },
    },
    /* 32 */
    {
        { 2, 0x02, 108 },
        { 9, 0x02, 108 },
        { 23, 0x02, 108 },
        { 40, 0x03, 108 },
        { 2, 0x02, 109 },
        { 9, 0x02, 109 },
        { 23, 0x02, 109 },
        { 40, 0x03, 109 },
        { 2, 0x02, 110 },
        { 9, 0x02, 110 },
        { 23, 0x02, 110 },
        { 40, 0x03, 110 },
        { 2, 0x02, 112 },
        { 9, 0x02, 112 },
        { 23, 0x02, 112 },
        { 40, 0x03, 112 },
    },
    /* 33 */
    {
        { 3, 0x02, 108 },
        { 6, 0x02, 108 },
        { 10, 0x02, 108 },
        { 15, 0x02, 108 },
        { 24, 0x02, 108 },
        { 31, 0x02, 108 },
        { 41, 0x02, 108 },
        { 56, 0x03, 108 },
        { 3, 0x02, 109 },
        { 6, 0x02, 109 },
        { 10, 0x02, 109 },
        { 15, 0x02, 109 },
        { 24, 0x02, 109 },
        { 31, 0x02, 109 },
        { 41, 0x02, 109 },
        { 56, 0x03, 109 },
    },
    /* 34 */
    {
        { 3, 0x02, 110 },
        { 6, 0x02, 110 },
        { 10, 0x02, 110 },
        { 15, 0x02, 110 },
        { 24, 0x02, 110 },
        { 31, 0x02, 110 },
        { 41, 0x02, 110 },
        { 56, 0x03, 110 },
        { 3, 0x02, 112 },
        { 6, 0x02, 112 },
        { 10, 0x02, 112 },
        { 15, 0x02, 112 },
        { 24, 0x02, 112 },
        { 31, 0x02, 112 },
        { 41, 0x02, 112 },
        { 56, 0x03, 112 },
    },
    /* 35 */
    {
        { 2, 0x02, 114 },
        { 9, 0x02, 114 },
        { 23, 0x02, 114 },
        { 40, 0x03, 114 },
        { 2, 0x02, 117 },
        { 9, 0x02, 117 },
        { 23, 0x02, 117 },
        { 40, 0x03, 117 },
        { 1, 0x02, 58 },
        { 22, 0x03, 58 },
        { 1, 0x02, 66 },
        { 22, 0x03, 66 },
        { 1, 0x02, 67 },
        { 22, 0x03, 67 },
        { 1, 0x02, 68 },
        { 22, 0x03, 68 },
    },
    /* 36 */
    {
        { 3, 0x02, 114 },
        { 6, 0x02, 114 },
        { 10, 0x02, 114 },
        { 15, 0x02, 114 },
        { 24, 0x02, 114 },
        { 31, 0x02, 114 },
        { 41, 0x02, 114 },
        { 56, 0x03, 114 },
        { 3, 0x02, 117 },
        { 6, 0x02, 117 },
        { 10, 0x02, 117 },
        { 15, 0x02, 117 },
        { 24, 0x02, 117 },
        { 31, 0x02, 117 },
        { 41, 0x02, 117 },
        { 56, 0x03, 117 },
    },
    /* 37 */
    {
        { 2, 0x02, 58 },
        { 9, 0x02, 58 },
        { 23, 0x02, 58 },
        { 40, 0x03, 58 },
        { 2, 0x02, 66 },
        { 9, 0x02, 66 },
        { 23, 0x02, 66 },
        { 40, 0x03, 66 },
        { 2, 0x02, 67 },
        { 9, 0x02, 67 },
        { 23, 0x02, 67 },
        { 40, 0x03, 67 },
        { 2, 0x02, 68 },
        { 9, 0x02, 68 },
        { 23, 0x02, 68 },
        { 40, 0x03, 68 },
    },
    /* 38 */
    {
        { 3, 0x02, 58 },
        { 6, 0x02, 58 },
        { 10, 0x02, 58 },
        { 15, 0x02, 58 },
        { 24, 0x02, 58 },
        { 31, 0x02, 58 },
        { 41, 0x02, 58 },
        { 56, 0x03, 58 },
        { 3, 0x02, 66 },
        { 6, 0x02, 66 },
        { 10, 0x02, 66 },
        { 15, 0x02, 66 },
        { 24, 0x02, 66 },
        { 31, 0x02, 66 },
        { 41, 0x02, 66 },
        { 56, 0x03, 66 },
    },
    /* 39 */
    {
        { 3, 0x02, 67 },
        { 6, 0x02, 67 },
        { 10, 0x02, 67 },
        { 15, 0x02, 67 },
        { 24, 0x02, 67 },
        { 31, 0x02, 67 },
        { 41, 0x02, 67 },
        { 56, 0x03, 67 },
        { 3, 0x02, 68 },
        { 6, 0x02, 68 },
        { 10, 0x02, 68 },
        { 15, 0x02, 68 },
        { 24, 0x02, 68 },
        { 31, 0x02, 68 },
        { 41, 0x02, 68 },
        { 56, 0x03, 68 },
    },
    /* 40 */
    {
        { 44, 0x00, 0 },
        { 45, 0x00, 0 },
        { 47, 0x00, 0 },
        { 48, 0x00, 0 },
        { 51, 0x00, 0 },
        { 52, 0x00, 0 },
        { 54, 0x00, 0 },
        { 55, 0x00, 0 },
        { 59, 0x00, 0 },
        { 60, 0x00, 0 },
        { 62, 0x00, 0 },
        { 63, 0x00, 0 },
        { 66, 0x00, 0 },
        { 67, 0x00, 0 },
        { 69, 0x00, 0 },
        { 72, 0x01, 0 },
    },
    /* 41 */
    {
        { 0, 0x03, 69 },
        { 0, 0x03, 70 },
        { 0, 0x03, 71 },
        { 0, 0x03, 72 },
        { 0, 0x03, 73 },
        { 0, 0x03, 74 },
        { 0, 0x03, 75 },
        { 0, 0x03, 76 },
        { 0, 0x03, 77 },
        { 0, 0x03, 78 },
        { 0, 0x03, 79 },
        { 0, 0x03, 80 },
        { 0, 0x03, 81 },
        { 0, 0x03, 82 },
        { 0, 0x03, 83 },
        { 0, 0x03, 84 },
    },
    /* 42 */
    {
        { 1, 0x02, 69 },
        { 22, 0x03, 69 },
        { 1, 0x02, 70 },
        { 22, 0x03, 70 },
        { 1, 0x02, 71 },
        { 22, 0x03, 71 },
        { 1, 0x02, 72 },
        { 22, 0x03, 72 },
        { 1, 0x02, 73 },
        { 22, 0x03, 73 },
        { 1, 0x02, 74 },
        { 22, 0x03, 74 },
        { 1, 0x02, 75 },
        { 22, 0x03, 75 },
        { 1, 0x02, 76 },
        { 22, 0x03, 76 },
    },
    /* 43 */
    {
        { 2, 0x02, 69 },
        { 9, 0x02, 69 },
        { 23, 0x02, 69 },
        { 40, 0x03, 69 },
        { 2, 0x02, 70 },
        { 9, 0x02, 70 },
        { 23, 0x02, 70 },
        { 40, 0x03, 70 },
        { 2, 0x02, 71 },
        { 9, 0x02, 71 },
        { 23, 0x02, 71 },
        { 40, 0x03, 71 },
        { 2, 0x02, 72 },
        { 9, 0x02, 72 },
        { 23, 0x02, 72 },
        { 40, 0x03, 72 },
    },
    /* 44 */
    {
        { 3, 0x02, 69 },
        { 6, 0x02, 69 },
        { 10, 0x02, 69 },
        { 15, 0x02, 69 },
        { 24, 0x02, 69 },
        { 31, 0x02, 69 },
        { 41, 0x02, 69 },
        { 56, 0x03, 69 },
        { 3, 0x02, 70 },
        { 6, 0x02, 70 },
        { 10, 0x02, 70 },
        { 15, 0x02, 70 },
        { 24, 0x02, 70 },
        { 31, 0x02, 70 },
        { 41, 0x02, 70 },
        { 56, 0x03, 70 },
    },
    /* 45 */
    {
        { 3, 0x02, 71 },
        { 6, 0x02, 71 },
        { 10, 0x02, 71 },
        { 15, 0x02, 71 },
        { 24, 0x02, 71 },
        { 31, 0x02, 71 },
        { 41, 0x02, 71 },
        { 56, 0x03, 71 },
        { 3, 0x02, 72 },
        { 6, 0x02, 72 },
        { 10, 0x02, 72 },
        { 15, 0x02, 72 },
        { 24, 0x02, 72 },
        { 31, 0x02, 72 },
        { 41, 0x02, 72 },
        { 56, 0x03, 72 },
    },
    /* 46 */
    {
        { 2, 0x02, 73 },
        { 9, 0x02, 73 },
        { 23, 0x02, 73 },
        { 40, 0x03, 73 },
        { 2, 0x02, 74 },
        { 9, 0x02, 74 },
        { 23, 0x02, 74 },
        { 40, 0x03, 74 },
        { 2, 0x02, 75 },
        { 9, 0x02, 75 },
        { 23, 0x02, 75 },
        { 40, 0x03, 75 },
        { 2, 0x02, 76 },
        { 9, 0x02, 76 },
        { 23, 0x02, 76 },
        { 40, 0x03, 76 },
    },
    /* 47 */
    {
        { 3, 0x02, 73 },
        { 6, 0x02, 73 },
        { 10, 0x02, 73 },
        { 15, 0x02, 73 },
        { 24, 0x02, 73 },
        { 31, 0x02, 73 },
        { 41, 0x02, 73 },
        { 56, 0x03, 73 },
        { 3, 0x02, 74 },
        { 6, 0x02, 74 },
        { 10, 0x02, 74 },
        { 15, 0x02, 74 },
        { 24, 0x02, 74 },
        { 31, 0x02, 74 },
        { 41, 0x02, 74 },
        { 56, 0x03, 74 },
    },
    /* 48 */
    {
        { 3, 0x02, 75 },
        { 6, 0x02, 75 },
        { 10, 0x02, 75 },
        { 15, 0x02, 75 },
        { 24, 0x02, 75 },
        { 31, 0x02, 75 },
        { 41, 0x02, 75 },
        { 56, 0x03, 75 },
        { 3, 0x02, 76 },
        { 6, 0x02, 76 },
        { 10, 0x02, 76 },
        { 15, 0x02, 76 },
        { 24, 0x02, 76 },
        { 31, 0x02, 76 },
        { 41, 0x02, 76 },
        { 56, 0x03, 76 },
    },
    /* 49 */
    {
        { 1, 0x02, 77 },
        { 22, 0x03, 77 },
        { 1, 0x02, 78 },
        { 22, 0x03, 78 },
        { 1, 0x02, 79 },
        { 22, 0x03, 79 },
        { 1, 0x02, 80 },
        { 22, 0x03, 80 },
        { 1, 0x02, 81 },
        { 22, 0x03, 81 },
        { 1, 0x02, 82 },
        { 22, 0x03, 82 },
        { 1, 0x02, 83 },
        { 22, 0x03, 83 },
        { 1, 0x02, 84 },
        { 22, 0x03, 84 },
    },
    /* 50 */
    {
        { 2, 0x02, 77 },
        { 9, 0x02, 77 },
        { 23, 0x02, 77 },
        { 40, 0x03, 77 },
        { 2, 0x02, 78 },
        { 9, 0x02, 78 },
        { 23, 0x02, 78 },
        { 40, 0x03, 78 },
        { 2, 0x02, 79 },
        { 9, 0x02, 79 },
        { 23, 0x02, 79 },
        { 40, 0x03, 79 },
        { 2, 0x02, 80 },
        { 9, 0x02, 80 },
        { 23, 0x02, 80 },
        { 40, 0x03, 80 },
    },
    /* 51 */
    {
        { 3, 0x02, 77 },
        { 6, 0x02, 77 },
        { 10, 0x02, 77 },
        { 15, 0x02, 77 },
        { 24, 0x02, 77 },
        { 31, 0x02, 77 },
        { 41, 0x02, 77 },
        { 56, 0x03, 77 },
        { 3, 0x02, 78 },
        { 6, 0x02, 78 },
        { 10, 0x02, 78 },
        { 15, 0x02, 78 },
        { 24, 0x02, 78 },
        { 31, 0x02, 78 },
        { 41, 0x02, 78 },
        { 56, 0x03, 78 },
    },
    /* 52 */
    {
        { 3, 0x02, 79 },
        { 6, 0x02, 79 },
        { 10, 0x02, 79 },
        { 15, 0x02, 79 },
        { 24, 0x02, 79 },
        { 31, 0x02, 79 },
        { 41, 0x02, 79 },
        { 56, 0x03, 79 },
        { 3, 0x02, 80 },
        { 6, 0x02, 80 },
        { 10, 0x02, 80 },
        { 15, 0x02, 80 },
        { 24, 0x02, 80 },
        { 31, 0x02, 80 },
        { 41, 0x02, 80 },
        { 56, 0x03, 80 },
    },
    /* 53 */
    {
        { 2, 0x02, 81 },
        { 9, 0x02, 81 },
        { 23, 0x02, 81 },
        { 40, 0x03, 81 },
        { 2, 0x02, 82 },
        { 9, 0x02, 82 },
        { 23, 0x02, 82 },
        { 40, 0x03, 82 },
        { 2, 0x02, 83 },
        { 9, 0x02, 83 },
        { 23, 0x02, 83 },
        { 40, 0x03, 83 },
        { 2, 0x02, 84 },
        { 9, 0x02, 84 },
        { 23, 0x02, 84 },
        { 40, 0x03, 84 },
    },
    /* 54 */
    {
        { 3, 0x02, 81 },
        { 6, 0x02, 81 },
        { 10, 0x02, 81 },
        { 15, 0x02, 81 },
        { 24, 0x02, 81 },
        { 31, 0x02, 81 },
        { 41, 0x02, 81 },
        { 56, 0x03, 81 },
        { 3, 0x02, 82 },
        { 6, 0x02, 82 },
        { 10, 0x02, 82 },
        { 15, 0x02, 82 },
        { 24, 0x02, 82 },
        { 31, 0x02, 82 },
        { 41, 0x02, 82 },
        { 56, 0x03, 82 },
    },
    /* 55 */
    {
        { 3, 0x02, 83 },
        { 6, 0x02, 83 },
        { 10, 0x02, 83 },
        { 15, 0x02, 83 },
        { 24, 0x02, 83 },
        { 31, 0x02, 83 },
        { 41, 0x02, 83 },
        { 56, 0x03, 83 },
        { 3, 0x02, 84 },
        { 6, 0x02, 84 },
        { 10, 0x02, 84 },
        { 15, 0x02, 84 },
        { 24, 0x02, 84 },
        { 31, 0x02, 84 },
        { 41, 0x02, 84 },
        { 56, 0x03, 84 },
    },
    /* 56 */
    {
        { 0, 0x03, 85 },
        { 0, 0x03, 86 },
        { 0, 0x03, 87 },
        { 0, 0x03, 89 },
        { 0, 0x03, 106 },
        { 0, 0x03, 107 },
        { 0, 0x03, 113 },
        { 0, 0x03, 118 },
        { 0, 0x03, 119 },
        { 0, 0x03, 120 },
        { 0, 0x03, 121 },
        { 0, 0x03, 122 },
        { 70, 0x00, 0 },
        { 71, 0x00, 0 },
        { 73, 0x00, 0 },
        { 74, 0x01, 0 },
    },
    /* 57 */
    {
        { 1, 0x02, 85 },
        { 22, 0x03, 85 },
        { 1, 0x02, 86 },
        { 22, 0x03, 86 },
        { 1, 0x02, 87 },
        { 22, 0x03, 87 },
        { 1, 0x02, 89 },
        { 22, 0x03, 89 },
        { 1, 0x02, 106 },
        { 22, 0x03, 106 },
        { 1, 0x02, 107 },
        { 22, 0x03, 107 },
        { 1, 0x02, 113 },
        { 22, 0x03, 113 },
        { 1, 0x02, 118 },
        { 22, 0x03, 118 },
    },
    /* 58 */
    {
        { 2, 0x02, 85 },
        { 9, 0x02, 85 },
        { 23, 0x02, 85 },
        { 40, 0x03, 85 },
        { 2, 0x02, 86 },
        { 9, 0x02, 86 },
        { 23, 0x02, 86 },
        { 40, 0x03, 86 },
        { 2, 0x02, 87 },
        { 9, 0x02, 87 },
        { 23, 0x02, 87 },
        { 40, 0x03, 87 },
        { 2, 0x02, 89 },
        { 9, 0x02, 89 },
        { 23, 0x02, 89 },
        { 40, 0x03, 89 },
    },
    /* 59 */
    {
        { 3, 0x02, 85 },
        { 6, 0x02, 85 },
        { 10, 0x02, 85 },
        { 15, 0x02, 85 },
        { 24, 0x02, 85 },
        { 31, 0x02, 85 },
        { 41, 0x02, 85 },
        { 56, 0x03, 85 },
        { 3, 0x02, 86 },
        { 6, 0x02, 86 },
        { 10, 0x02, 86 },
        { 15, 0x02, 86 },
        { 24, 0x02, 86 },
        { 31, 0x02, 86 },
        { 41, 0x02, 86 },
        { 56, 0x03, 86 },
    },
    /* 60 */
    {
        { 3, 0x02, 87 },
        { 6, 0x02, 87 },
        { 10, 0x02, 87 },
        { 15, 0x02, 87 },
        { 24, 0x02, 87 },
        { 31, 0x02, 87 },
        { 41, 0x02, 87 },
        { 56, 0x03, 87 },
        { 3, 0x02, 89 },
        { 6, 0x02, 89 },
        { 10, 0x02, 89 },
        { 15, 0x02, 89 },
        { 24, 0x02, 89 },
        { 31, 0x02, 89 },
        { 41, 0x02, 89 },
        { 56, 0x03, 89 },
    },
    /* 61 */
    {
        { 2, 0x02, 106 },
        { 9, 0x02, 106 },
        { 23, 0x02, 106 },
        { 40, 0x03, 106 },
        { 2, 0x02, 107 },
        { 9, 0x02, 107 },
        { 23, 0x02, 107 },
        { 40, 0x03, 107 },
        { 2, 0x02, 113 },
        { 9, 0x02, 113 },
        { 23, 0x02, 113 },
        { 40, 0x03, 113 },
        { 2, 0x02, 118 },
        { 9, 0x02, 118 },
        { 23, 0x02, 118 },
        { 40, 0x03, 118 },
    },
    /* 62 */
    {
        { 3, 0x02, 106 },
        { 6, 0x02, 106 },
        { 10, 0x02, 106 },
        { 15, 0x02, 106 },
        { 24, 0x02, 106 },
        { 31, 0x02, 106 },
        { 41, 0x02, 106 },
        { 56, 0x03, 106 },
        { 3, 0x02, 107 },
        { 6, 0x02, 107 },
        { 10, 0x02, 107 },
        { 15, 0x02, 107 },
        { 24, 0x02, 107 },
        { 31, 0x02, 107 },
        { 41, 0x02, 107 },
        { 56, 0x03, 107 },
    },
    /* 63 */
    {
        { 3, 0x02, 113 },
        { 6, 0x02, 113 },
        { 10, 0x02, 113 },
        { 15, 0x02, 113 },
        { 24, 0x02, 113 },
        { 31, 0x02, 113 },
        { 41, 0x02, 113 },
        { 56, 0x03, 113 },
        { 3, 0x02, 118 },
        { 6, 0x02, 118 },
        { 10, 0x02, 118 },
        { 15, 0x02, 118 },
        { 24, 0x02, 118 },
        { 31, 0x02, 118 },
        { 41, 0x02, 118 },
        { 56, 0x03, 118 },
    },
    /* 64 */
    {
        { 1, 0x02, 119 },
        { 22, 0x03, 119 },
        { 1, 0x02, 120 },
        { 22, 0x03, 120 },
        { 1, 0x02, 121 },
        { 22, 0x03, 121 },
        { 1, 0x02, 122 },
        { 22, 0x03, 122 },
        { 0, 0x03, 38 },
        { 0, 0x03, 42 },
        { 0, 0x03, 44 },
        { 0, 0x03, 59 },
        { 0, 0x03, 88 },
        { 0, 0x03, 90 },
        { 75, 0x00, 0 },
        { 78, 0x00, 0 },
    },
    /* 65 */
    {
        { 2, 0x02, 119 },
        { 9, 0x02, 119 },
        { 23, 0x02, 119 },
        { 40, 0x03, 119 },
        { 2, 0x02, 120 },
        { 9, 0x02, 120 },
        { 23, 0x02, 120 },
        { 40, 0x03, 120 },
        { 2, 0x02, 121 },
        { 9, 0x02, 121 },
        { 23, 0x02, 121 },
        { 40, 0x03, 121 },
        { 2, 0x02, 122 },
        { 9, 0x02, 122 },
        { 23, 0x02, 122 },
        { 40, 0x03, 122 },
    },
    /* 66 */
    {
        { 3, 0x02, 119 },
        { 6, 0x02, 119 },
        { 10, 0x02, 119 },
        { 15, 0x02, 119 },
        { 24, 0x02, 119 },
        { 31, 0x02, 119 },
        { 41, 0x02, 119 },
        { 56, 0x03, 119 },
        { 3, 0x02, 120 },
        { 6, 0x02, 120 },
        { 10, 0x02, 120 },
        { 15, 0x02, 120 },
        { 24, 0x02, 120 },
        { 31, 0x02, 120 },
        { 41, 0x02, 120 },
        { 56, 0x03, 120 },
    },
    /* 67 */
    {
        { 3, 0x02, 121 },
        { 6, 0x02, 121 },
        { 10, 0x02, 121 },
        { 15, 0x02, 121 },
        { 24, 0x02, 121 },
        { 31, 0x02, 121 },
        { 41, 0x02, 121 },
        { 56, 0x03, 121 },
        { 3, 0x02, 122 },
        { 6, 0x02, 122 },
        { 10, 0x02, 122 },
        { 15, 0x02, 122 },
        { 24, 0x02, 122 },
        { 31, 0x02, 122 },
        { 41, 0x02, 122 },
        { 56, 0x03, 122 },
    },
    /* 68 */
    {
        { 1, 0x02, 38 },
        { 22, 0x03, 38 },
        { 1, 0x02, 42 },
        { 22, 0x03, 42 },
        { 1, 0x02, 44 },
        { 22, 0x03, 44 },
        { 1, 0x02, 59 },
        { 22, 0x03, 59 },
        { 1, 0x02, 88 },
        { 22, 0x03, 88 },
        { 1, 0x02, 90 },
        { 22, 0x03, 90 },
        { 76, 0x00, 0 },
        { 77, 0x00, 0 },
        { 79, 0x00, 0 },
        { 81, 0x00, 0 },
    },
    /* 69 */
    {
        { 2, 0x02, 38 },
        { 9, 0x02, 38 },
        { 23, 0x02, 38 },
        { 40, 0x03, 38 },
        { 2, 0x02, 42 },
        { 9, 0x02, 42 },
        { 23, 0x02, 42 },
        { 40, 0x03, 42 },
        { 2, 0x02, 44 },
        { 9, 0x02, 44 },
        { 23, 0x02, 44 },
        { 40, 0x03, 44 },
        { 2, 0x02, 59 },
        { 9, 0x02, 59 },
        { 23, 0x02, 59 },
        { 40, 0x03, 59 },
    },
    /* 70 */
    {
        { 3, 0x02, 38 },
        { 6, 0x02, 38 },
        { 10, 0x02, 38 },
        { 15, 0x02, 38 },
        { 24, 0x02, 38 },
        { 31, 0x02, 38 },
        { 41, 0x02, 38 },
        { 56, 0x03, 38 },
        { 3, 0x02, 42 },
        { 6, 0x02, 42 },
        { 10, 0x02, 42 },
        { 15, 0x02, 42 },
        { 24, 0x02, 42 },
        { 31, 0x02, 42 },
        { 41, 0x02, 42 },
        { 56, 0x03, 42 },
    },
    /* 71 */
    {
        { 3, 0x02, 44 },
        { 6, 0x02, 44 },
        { 10, 0x02, 44 },
        { 15, 0x02, 44 },
        { 24, 0x02, 44 },
        { 31, 0x02, 44 },
        { 41, 0x02, 44 },
        { 56, 0x03, 44 },
        { 3, 0x02, 59 },
        { 6, 0x02, 59 },
        { 10, 0x02, 59 },
        { 15, 0x02, 59 },
        { 24, 0x02, 59 },
        { 31, 0x02, 59 },
        { 41, 0x02, 59 },
        { 56, 0x03, 59 },
    },
    /* 72 */
    {
        { 2, 0x02, 88 },
        { 9, 0x02, 88 },
        { 23, 0x02, 88 },
        { 40, 0x03, 88 },
        { 2, 0x02, 90 },
        { 9, 0x02, 90 },
        { 23, 0x02, 90 },
        { 40, 0x03, 90 },
        { 0, 0x03, 33 },
        { 0, 0x03, 34 },
        { 0, 0x03, 40 },
        { 0, 0x03, 41 },
        { 0, 0x03, 63 },
        { 80, 0x00, 0 },
        { 82, 0x00, 0 },
        { 84, 0x00, 0 },
    },
    /* 73 */
    {
        { 3, 0x02, 88 },
        { 6, 0x02, 88 },
        { 10, 0x02, 88 },
        { 15, 0x02, 88 },
        { 24, 0x02, 88 },
        { 31, 0x02, 88 },
        { 41, 0x02, 88 },
        { 56, 0x03, 88 },
        { 3, 0x02, 90 },
        { 6, 0x02, 90 },
        { 10, 0x02, 90 },
        { 15, 0x02, 90 },
        { 24, 0x02, 90 },
        { 31, 0x02, 90 },
        { 41, 0x02, 90 },
        { 56, 0x03, 90 },
    },
    /* 74 */
    {
        { 1, 0x02, 33 },
        { 22, 0x03, 33 },
        { 1, 0x02, 34 },
        { 22, 0x03, 34 },
        { 1, 0x02, 40 },
        { 22, 0x03, 40 },
        { 1, 0x02, 41 },
        { 22, 0x03, 41 },
        { 1, 0x02, 63 },
        { 22, 0x03, 63 },
        { 0, 0x03, 39 },
        { 0, 0x03, 43 },
        { 0, 0x03, 124 },
        { 83, 0x00, 0 },
        { 85, 0x00, 0 },
        { 88, 0x00, 0 },
    },
    /* 75 */
    {
        { 2, 0x02, 33 },
        { 9, 0x02, 33 },
        { 23, 0x02, 33 },
        { 40, 0x03, 33 },
        { 2, 0x02, 34 },
        { 9, 0x02, 34 },
        { 23, 0x02, 34 },
        { 40, 0x03, 34 },
        { 2, 0x02, 40 },
        { 9, 0x02, 40 },
        { 23, 0x02, 40 },
        { 40, 0x03, 40 },
        { 2, 0x02, 41 },
        { 9, 0x02, 41 },
        { 23, 0x02, 41 },
        { 40, 0x03, 41 },
    },
    /* 76 */
    {
        { 3, 0x02, 33 },
        { 6, 0x02, 33 },
        { 10, 0x02, 33 },
        { 15, 0x02, 33 },
        { 24, 0x02, 33 },
        { 31, 0x02, 33 },
        { 41, 0x02, 33 },
        { 56, 0x03, 33 },
        { 3, 0x02, 34 },
        { 6, 0x02, 34 },
        { 10, 0x02, 34 },
        { 15, 0x02, 34 },
        { 24, 0x02, 34 },
        { 31, 0x02, 34 },
        { 41, 0x02, 34 },
        { 56, 0x03, 34 },
    },
    /* 77 */
    {
        { 3, 0x02, 40 },
        { 6, 0x02, 40 },
        { 10, 0x02, 40 },
        { 15, 0x02, 40 },
        { 24, 0x02, 40 },
        { 31, 0x02, 40 },
        { 41, 0x02, 40 },
        { 56, 0x03, 40 },
        { 3, 0x02, 41 },
        { 6, 0x02, 41 },
        { 10, 0x02, 41 },
        { 15, 0x02, 41 },
        { 24, 0x02, 41 },
        { 31, 0x02, 41 },
        { 41, 0x02, 41 },
        { 56, 0x03, 41 },
    },
    /* 78 */
    {
        { 2, 0x02, 63 },
        { 9, 0x02, 63 },
        { 23, 0x02, 63 },
        { 40, 0x03, 63 },
        { 1, 0x02, 39 },
        { 22, 0x03, 39 },
        { 1, 0x02, 43 },
        { 22, 0x03, 43 },
        { 1, 0x02, 124 },
        { 22, 0x03, 124 },
        { 0, 0x03, 35 },
        { 0, 0x03, 62 },
        { 86, 0x00, 0 },
        { 87, 0x00, 0 },
        { 89, 0x00, 0 },
        { 90, 0x00, 0 },
    },
    /* 79 */
    {
        { 3, 0x02, 63 },
        { 6, 0x02, 63 },
        { 10, 0x02, 63 },
        { 15, 0x02, 63 },
        { 24, 0x02, 63 },
        { 31, 0x02, 63 },
        { 41, 0x02, 63 },
        { 56, 0x03, 63 },
        { 2, 0x02, 39 },
        { 9, 0x02, 39 },
        { 23, 0x02, 39 },
        { 40, 0x03, 39 },
        { 2, 0x02, 43 },
        { 9, 0x02, 43 },
        { 23, 0x02, 43 },
        { 40, 0x03, 43 },
    },
    /* 80 */
    {
        { 3, 0x02, 39 },
        { 6, 0x02, 39 },
        { 10, 0x02, 39 },
        { 15, 0x02, 39 },
        { 24, 0x02, 39 },
        { 31, 0x02, 39 },
        { 41, 0x02, 39 },
        { 56, 0x03, 39 },
        { 3, 0x02, 43 },
        { 6, 0x02, 43 },
        { 10, 0x02, 43 },
        { 15, 0x02, 43 },
        { 24, 0x02, 43 },
        { 31, 0x02, 43 },
        { 41, 0x02, 43 },
        { 56, 0x03, 43 },
    },
    /* 81 */
    {
        { 2, 0x02, 124 },
        { 9, 0x02, 124 },
        { 23, 0x02, 124 },
        { 40, 0x03, 124 },
        { 1, 0x02, 35 },
        { 22, 0x03, 35 },
        { 1, 0x02, 62 },
        { 22, 0x03, 62 },
        { 0, 0x03, 0 },
        { 0, 0x03, 36 },
        { 0, 0x03, 64 },
        { 0, 0x03, 91 },
        { 0, 0x03, 93 },
        { 0, 0x03, 126 },
        { 91, 0x00, 0 },
        { 92, 0x00, 0 },
    },
    /* 82 */
    {
        { 3, 0x02, 124 },
        { 6, 0x02, 124 },
        { 10, 0x02, 124 },
        { 15, 0x02, 124 },
        { 24, 0x02, 124 },
        { 31, 0x02, 124 },
        { 41, 0x02, 124 },
        { 56, 0x03, 124 },
        { 2, 0x02, 35 },
        { 9, 0x02, 35 },
        { 23, 0x02, 35 },
        { 40, 0x03, 35 },
        { 2, 0x02, 62 },
        { 9, 0x02, 62 },
        { 23, 0x02, 62 },
        { 40, 0x03, 62 },
    },
    /* 83 */
    {
        { 3, 0x02, 35 },
        { 6, 0x02, 35 },
        { 10, 0x02, 35 },
        { 15, 0x02, 35 },
        { 24, 0x02, 35 },
        { 31, 0x02, 35 },
        { 41, 0x02, 35 },
        { 56, 0x03, 35 },
        { 3, 0x02, 62 },
        { 6, 0x02, 62 },
        { 10, 0x02, 62 },
        { 15, 0x02, 62 },
        { 24, 0x02, 62 },
        { 31, 0x02, 62 },
        { 41, 0x02, 62 },
        { 56, 0x03, 62 },
    },
    /* 84 */
    {
        { 1, 0x02, 0 },
        { 22, 0x03, 0 },
        { 1, 0x02, 36 },
        { 22, 0x03, 36 },
        { 1, 0x02, 64 },
        { 22, 0x03, 64 },
        { 1, 0x02, 91 },
        { 22, 0x03, 91 },
        { 1, 0x02, 93 },
        { 22, 0x03, 93 },
        { 1, 0x02, 126 },
        { 22, 0x03, 126 },
        { 0, 0x03, 94 },
        { 0, 0x03, 125 },
        { 93, 0x00, 0 },
        { 94, 0x00, 0 },
    },
    /* 85 */
    {
        { 2, 0x02, 0 },
        { 9, 0x02, 0 },
        { 23, 0x02, 0 },
        { 40, 0x03, 0 },
        { 2, 0x02, 36 },
        { 9, 0x02, 36 },
        { 23, 0x02, 36 },
        { 40, 0x03, 36 },
        { 2, 0x02, 64 },
        { 9, 0x02, 64 },
        { 23, 0x02, 64 },
        { 40, 0x03, 64 },
        { 2, 0x02, 91 },
        { 9, 0x02, 91 },
        { 23, 0x02, 91 },
        { 40, 0x03, 91 },
    },
    /* 86 */
    {
        { 3, 0x02, 0 },
        { 6, 0x02, 0 },
        { 10, 0x02, 0 },
        { 15, 0x02, 0 },
        { 24, 0x02, 0 },
        { 31, 0x02, 0 },
        { 41, 0x02, 0 },
        { 56, 0x03, 0 },
        { 3, 0x02, 36 },
        { 6, 0x02, 36 },
        { 10, 0x02, 36 },
        { 15, 0x02, 36 },
        { 24, 0x02, 36 },
        { 31, 0x02, 36 },
        { 41, 0x02, 36 },
        { 56, 0x03, 36 },
    },
    /* 87 */
    {
        { 3, 0x02, 64 },
        { 6, 0x02, 64 },
        { 10, 0x02, 64 },
        { 15, 0x02, 64 },
        { 24, 0x02, 64 },
        { 31, 0x02, 64 },
        { 41, 0x02, 64 },
        { 56, 0x03, 64 },
        { 3, 0x02, 91 },
        { 6, 0x02, 91 },
        { 10, 0x02, 91 },
        { 15, 0x02, 91 },
        { 24, 0x02, 91 },
        { 31, 0x02, 91 },
        { 41, 0x02, 91 },
        { 56, 0x03, 91 },
    },
    /* 88 */
    {
        { 2, 0x02, 93 },
        { 9, 0x02, 93 },
        { 23, 0x02, 93 },
        { 40, 0x03, 93 },
        { 2, 0x02, 126 },
        { 9, 0x02, 126 },
        { 23, 0x02, 126 },
        { 40, 0x03, 126 },
        { 1, 0x02, 94 },
        { 22, 0x03, 94 },
        { 1, 0x02, 125 },
        { 22, 0x03, 125 },
        { 0, 0x03, 60 },
        { 0, 0x03, 96 },
        { 0, 0x03, 123 },
        { 95, 0x00, 0 },
    },
    /* 89 */
    {
        { 3, 0x02, 93 },
        { 6, 0x02, 93 },
        { 10, 0x02, 93 },
        { 15, 0x02, 93 },
        { 24, 0x02, 93 },
        { 31, 0x02, 93 },
        { 41, 0x02, 93 },
        { 56, 0x03, 93 },
        { 3, 0x02, 126 },
        { 6, 0x02, 126 },
        { 10, 0x02, 126 },
        { 15, 0x02, 126 },
        { 24, 0x02, 126 },
        { 31, 0x02, 126 },
        { 41, 0x02, 126 },
        { 56, 0x03, 126 },
    },
    /* 90 */
    {
        { 2, 0x02, 94 },
        { 9, 0x02, 94 },
        { 23, 0x02, 94 },
        { 40, 0x03, 94 },
        { 2, 0x02, 125 },
        { 9, 0x02, 125 },
        { 23, 0x02, 125 },
        { 40, 0x03, 125 },
        { 1, 0x02, 60 },
        { 22, 0x03, 60 },
        { 1, 0x02, 96 },
        { 22, 0x03, 96 },
        { 1, 0x02, 123 },
        { 22, 0x03, 123 },
        { 96, 0x00, 0 },
        { 110, 0x00, 0 },
    },
    /* 91 */
    {
        { 3, 0x02, 94 },
        { 6, 0x02, 94 },
        { 10, 0x02, 94 },
        { 15, 0x02, 94 },
        { 24, 0x02, 94 },
        { 31, 0x02, 94 },
        { 41, 0x02, 94 },
        { 56, 0x03, 94 },
        { 3, 0x02, 125 },
        { 6, 0x02, 125 },
        { 10, 0x02, 125 },
        { 15, 0x02, 125 },
        { 24, 0x02, 125 },
        { 31, 0x02, 125 },
        { 41, 0x02, 125 },
        { 56, 0x03, 125 },
    },
    /* 92 */
    {
        { 2, 0x02, 60 },
        { 9, 0x02, 60 },
        { 23, 0x02, 60 },
        { 40, 0x03, 60 },
        { 2, 0x02, 96 },
        { 9, 0x02, 96 },
        { 23, 0x02, 96 },
        { 40, 0x03, 96 },
        { 2, 0x02, 123 },
        { 9, 0x02, 123 },
        { 23, 0x02, 123 },
        { 40, 0x03, 123 },
        { 97, 0x00, 0 },
        { 101, 0x00, 0 },
        { 111, 0x00, 0 },
        { 133, 0x00, 0 },
    },
    /* 93 */
    {
        { 3, 0x02, 60 },
        { 6, 0x02, 60 },
        { 10, 0x02, 60 },
        { 15, 0x02, 60 },
        { 24, 0x02, 60 },
        { 31, 0x02, 60 },
        { 41, 0x02, 60 },
        { 56, 0x03, 60 },
        { 3, 0x02, 96 },
        { 6, 0x02, 96 },
        { 10, 0x02, 96 },
        { 15, 0x02, 96 },
        { 24, 0x02, 96 },
        { 31, 0x02, 96 },
        { 41, 0x02, 96 },
        { 56, 0x03, 96 },
    },
    /* 94 */
    {
        { 3, 0x02, 123 },
        { 6, 0x02, 123 },
        { 10, 0x02, 123 },
        { 15, 0x02, 123 },
        { 24, 0x02, 123 },
        { 31, 0x02, 123 },
        { 41, 0x02, 123 },
        { 56, 0x03, 123 },
        { 98, 0x00, 0 },
        { 99, 0x00, 0 },
        { 102, 0x00, 0 },
        { 105, 0x00, 0 },
        { 112, 0x00, 0 },
        { 119, 0x00, 0 },
        { 134, 0x00, 0 },
        { 153, 0x00, 0 },
    },
    /* 95 */
    {
        { 0, 0x03, 92 },
        { 0, 0x03, 195 },
        { 0, 0x03, 208 },
        { 100, 0x00, 0 },
        { 103, 0x00, 0 },
        { 104, 0x00, 0 },
        { 106, 0x00, 0 },
        { 107, 0x00, 0 },
        { 113, 0x00, 0 },
        { 116, 0x00, 0 },
        { 120, 0x00, 0 },
        { 126, 0x00, 0 },
        { 135, 0x00, 0 },
        { 142, 0x00, 0 },
        { 154, 0x00, 0 },
        { 169, 0x00, 0 },
    },
    /* 96 */
    {
        { 1, 0x02, 92 },
        { 22, 0x03, 92 },
        { 1, 0x02, 195 },
        { 22, 0x03, 195 },
        { 1, 0x02, 208 },
        { 22, 0x03, 208 },
        { 0, 0x03, 128 },
        { 0, 0x03, 130 },
        { 0, 0x03, 131 },
        { 0, 0x03, 162 },
        { 0, 0x03, 184 },
        { 0, 0x03, 194 },
        { 0, 0x03, 224 },
        { 0, 0x03, 226 },
        { 108, 0x00, 0 },
        { 109, 0x00, 0 },
    },
    /* 97 */
    {
        { 2, 0x02, 92 },
        { 9, 0x02, 92 },
        { 23, 0x02, 92 },
        { 40, 0x03, 92 },
        { 2, 0x02, 195 },
        { 9, 0x02, 195 },
        { 23, 0x02, 195 },
        { 40, 0x03, 195 },
        { 2, 0x02, 208 },
        { 9, 0x02, 208 },
        { 23, 0x02, 208 },
        { 40, 0x03, 208 },
        { 1, 0x02, 128 },
        { 22, 0x03, 128 },
        { 1, 0x02, 130 },
        { 22, 0x03, 130 },
    },
    /* 98 */
    {
        { 3, 0x02, 92 },
        { 6, 0x02, 92 },
        { 10, 0x02, 92 },
        { 15, 0x02, 92 },
        { 24, 0x02, 92 },
        { 31, 0x02, 92 },
        { 41, 0x02, 92 },
        { 56, 0x03, 92 },
        { 3, 0x02, 195 },
        { 6, 0x02, 195 },
        { 10, 0x02, 195 },
        { 15, 0x02, 195 },
        { 24, 0x02, 195 },
        { 31, 0x02, 195 },
        { 41, 0x02, 195 },
        { 56, 0x03, 195 },
    },
    /* 99 */
    {
        { 3, 0x02, 208 },
        { 6, 0x02, 208 },
        { 10, 0x02, 208 },
        { 15, 0x02, 208 },
        { 24, 0x02, 208 },
        { 31, 0x02, 208 },
        { 41, 0x02, 208 },
        { 56, 0x03, 208 },
        { 2, 0x02, 128 },
        { 9, 0x02, 128 },
        { 23, 0x02, 128 },
        { 40, 0x03, 128 },
        { 2, 0x02, 130 },
        { 9, 0x02, 130 },
        { 23, 0x02, 130 },
        { 40, 0x03, 130 },
    },
    /* 100 */
    {
        { 3, 0x02, 128 },
        { 6, 0x02, 128 },
        { 10, 0x02, 128 },
        { 15, 0x02, 128 },
        { 24, 0x02, 128 },
        { 31, 0x02, 128 },
        { 41, 0x02, 128 },
        { 56, 0x03, 128 },
        { 3, 0x02, 130 },
        { 6, 0x02, 130 },
        { 10, 0x02, 130 },
        { 15, 0x02, 130 },
        { 24, 0x02, 130 },
        { 31, 0x02, 130 },
        { 41, 0x02, 130 },
        { 56, 0x03, 130 },
    },
    /* 101 */
    {
        { 1, 0x02, 131 },
        { 22, 0x03, 131 },
        { 1, 0x02, 162 },
        { 22, 0x03, 162 },
        { 1, 0x02, 184 },
        { 22, 0x03, 184 },
        { 1, 0x02, 194 },
        { 22, 0x03, 194 },
        { 1, 0x02, 224 },
        { 22, 0x03, 224 },
        { 1, 0x02, 226 },
        { 22, 0x03, 226 },
        { 0, 0x03, 153 },
        { 0, 0x03, 161 },
        { 0, 0x03, 167 },
        { 0, 0x03, 172 },
    },
    /* 102 */
    {
        { 2, 0x02, 131 },
        { 9, 0x02, 131 },
        { 23, 0x02, 131 },
        { 40, 0x03, 131 },
        { 2, 0x02, 162 },
        { 9, 0x02, 162 },
        { 23, 0x02, 162 },
        { 40, 0x03, 162 },
        { 2, 0x02, 184 },
        { 9, 0x02, 184 },
        { 23, 0x02, 184 },
        { 40, 0x03, 184 },
        { 2, 0x02, 194 },
        { 9, 0x02, 194 },
        { 23, 0x02, 194 },
        { 40, 0x03, 194 },
    },
    /* 103 */
    {
        { 3, 0x02, 131 },
        { 6, 0x02, 131 },
        { 10, 0x02, 131 },
        { 15, 0x02, 131 },
        { 24, 0x02, 131 },
        { 31, 0x02, 131 },
        { 41, 0x02, 131 },
        { 56, 0x03, 131 },
        { 3, 0x02, 162 },
        { 6, 0x02, 162 },
        { 10, 0x02, 162 },
        { 15, 0x02, 162 },
        { 24, 0x02, 162 },
        { 31, 0x02, 162 },
        { 41, 0x02, 162 },
        { 56, 0x03, 162 },
    },
    /* 104 */
    {
        { 3, 0x02, 184 },
        { 6, 0x02, 184 },
        { 10, 0x02, 184 },
        { 15, 0x02, 184 },
        { 24, 0x02, 184 },
        { 31, 0x02, 184 },
        { 41, 0x02, 184 },
        { 56, 0x03, 184 },
        { 3, 0x02, 194 },
        { 6, 0x02, 194 },
        { 10, 0x02, 194 },
        { 15, 0x02, 194 },
        { 24, 0x02, 194 },
        { 31, 0x02, 194 },
        { 41, 0x02, 194 },
        { 56, 0x03, 194 },
    },
    /* 105 */
    {
        { 2, 0x02, 224 },
        { 9, 0x02, 224 },
        { 23, 0x02, 224 },
        { 40, 0x03, 224 },
        { 2, 0x02, 226 },
        { 9, 0x02, 226 },
        { 23, 0x02, 226 },
        { 40, 0x03, 226 },
        { 1, 0x02, 153 },
        { 22, 0x03, 153 },
        { 1, 0x02, 161 },
        { 22, 0x03, 161 },
        { 1, 0x02, 167 },
        { 22, 0x03, 167 },
        { 1, 0x02, 172 },
        { 22, 0x03, 172 },
    },
    /* 106 */
    {
        { 3, 0x02, 224 },
        { 6, 0x02, 224 },
        { 10, 0x02, 224 },
        { 15, 0x02, 224 },
        { 24, 0x02, 224 },
        { 31, 0x02, 224 },
        { 41, 0x02, 224 },
        { 56, 0x03, 224 },
        { 3, 0x02, 226 },
        { 6, 0x02, 226 },
        { 10, 0x02, 226 },
        { 15, 0x02, 226 },
        { 24, 0x02, 226 },
        { 31, 0x02, 226 },
        { 41, 0x02, 226 },
        { 56, 0x03, 226 },
    },
    /* 107 */
    {
        { 2, 0x02, 153 },
        { 9, 0x02, 153 },
        { 23, 0x02, 153 },
        { 40, 0x03, 153 },
        { 2, 0x02, 161 },
        { 9, 0x02, 161 },
        { 23, 0x02, 161 },
        { 40, 0x03, 161 },
        { 2, 0x02, 167 },
        { 9, 0x02, 167 },
        { 23, 0x02, 167 },
        { 40, 0x03, 167 },
        { 2, 0x02, 172 },
        { 9, 0x02, 172 },
        { 23, 0x02, 172 },
        { 40, 0x03, 172 },
    },
    /* 108 */
    {
        { 3, 0x02, 153 },
        { 6, 0x02, 153 },
        { 10, 0x02, 153 },
        { 15, 0x02, 153 },
        { 24, 0x02, 153 },
        { 31, 0x02, 153 },
        { 41, 0x02, 153 },
        { 56, 0x03, 153 },
        { 3, 0x02, 161 },
        { 6, 0x02, 161 },
        { 10, 0x02, 161 },
        { 15, 0x02, 161 },
        { 24, 0x02, 161 },
        { 31, 0x02, 161 },
        { 41, 0x02, 161 },
        { 56, 0x03, 161 },
    },
    /* 109 */
    {
        { 3, 0x02, 167 },
        { 6, 0x02, 167 },
        { 10, 0x02, 167 },
        { 15, 0x02, 167 },
        { 24, 0x02, 167 },
        { 31, 0x02, 167 },
        { 41, 0x02, 167 },
        { 56, 0x03, 167 },
        { 3, 0x02, 172 },
        { 6, 0x02, 172 },
        { 10, 0x02, 172 },
        { 15, 0x02, 172 },
        { 24, 0x02, 172 },
        { 31, 0x02, 172 },
        { 41, 0x02, 172 },
        { 56, 0x03, 172 },
    },
    /* 110 */
    {
        { 114, 0x00, 0 },
        { 115, 0x00, 0 },
        { 117, 0x00, 0 },
        { 118, 0x00, 0 },
        { 121, 0x00, 0 },
        { 123, 0x00, 0 },
        { 127, 0x00, 0 },
        { 130, 0x00, 0 },
        { 136, 0x00, 0 },
        { 139, 0x00, 0 },
        { 143, 0x00, 0 },
        { 146, 0x00, 0 },
        { 155, 0x00, 0 },
        { 162, 0x00, 0 },
        { 170, 0x00, 0 },
        { 180, 0x00, 0 },
    },
    /* 111 */
    {
        { 0, 0x03, 176 },
        { 0, 0x03, 177 },
        { 0, 0x03, 179 },
        { 0, 0x03, 209 },
        { 0, 0x03, 216 },
        { 0, 0x03, 217 },
        { 0, 0x03, 227 },
        { 0, 0x03, 229 },
        { 0, 0x03, 230 },
        { 122, 0x00, 0 },
        { 124, 0x00, 0 },
        { 125, 0x00, 0 },
        { 128, 0x00, 0 },
        { 129, 0x00, 0 },
        { 131, 0x00, 0 },
        { 132, 0x00, 0 },
    },
    /* 112 */
    {
        { 1, 0x02, 176 },
        { 22, 0x03, 176 },
        { 1, 0x02, 177 },
        { 22, 0x03, 177 },
        { 1, 0x02, 179 },
        { 22, 0x03, 179 },
        { 1, 0x02, 209 },
        { 22, 0x03, 209 },
        { 1, 0x02, 216 },
        { 22, 0x03, 216 },
        { 1, 0x02, 217 },
        { 22, 0x03, 217 },
        { 1, 0x02, 227 },
        { 22, 0x03, 227 },
        { 1, 0x02, 229 },
        { 22, 0x03, 229 },
    },
    /* 113 */
    {
        { 2, 0x02, 176 },
        { 9, 0x02, 176 },
        { 23, 0x02, 176 },
        { 40, 0x03, 176 },
        { 2, 0x02, 177 },
        { 9, 0x02, 177 },
        { 23, 0x02, 177 },
        { 40, 0x03, 177 },
        { 2, 0x02, 179 },
        { 9, 0x02, 179 },
        { 23, 0x02, 179 },
        { 40, 0x03, 179 },
        { 2, 0x02, 209 },
        { 9, 0x02, 209 },
        { 23, 0x02, 209 },
        { 40, 0x03, 209 },
    },
    /* 114 */
    {
        { 3, 0x02, 176 },
        { 6, 0x02, 176 },
        { 10, 0x02, 176 },
        { 15, 0x02, 176 },
        { 24, 0x02, 176 },
        { 31, 0x02, 176 },
        { 41, 0x02, 176 },
        { 56, 0x03, 176 },
        { 3, 0x02, 177 },
        { 6, 0x02, 177 },
        { 10, 0x02, 177 },
        { 15, 0x02, 177 },
        { 24, 0x02, 177 },
        { 31, 0x02, 177 },
        { 41, 0x02, 177 },
        { 56, 0x03, 177 },
    },
    /* 115 */
    {
        { 3, 0x02, 179 },
        { 6, 0x02, 179 },
        { 10, 0x02, 179 },
        { 15, 0x02, 179 },
        { 24, 0x02, 179 },
        { 31, 0x02, 179 },
        { 41, 0x02, 179 },
        { 56, 0x03, 179 },
        { 3, 0x02, 209 },
        { 6, 0x02, 209 },
        { 10, 0x02, 209 },
        { 15, 0x02, 209 },
        { 24, 0x02, 209 },
        { 31, 0x02, 209 },
        { 41, 0x02, 209 },
        { 56, 0x03, 209 },
    },
    /* 116 */
    {
        { 2, 0x02, 216 },
        { 9, 0x02, 216 },
        { 23, 0x02, 216 },
        { 40, 0x03, 216 },
        { 2, 0x02, 217 },
        { 9, 0x02, 217 },
        { 23, 0x02, 217 },
        { 40, 0x03, 217 },
        { 2, 0x02, 227 },
        { 9, 0x02, 227 },
        { 23, 0x02, 227 },
        { 40, 0x03, 227 },
        { 2, 0x02, 229 },
        { 9, 0x02, 229 },
        { 23, 0x02, 229 },
        { 40, 0x03, 229 },
    },
    /* 117 */
    {
        { 3, 0x02, 216 },
        { 6, 0x02, 216 },
        { 10, 0x02, 216 },
        { 15, 0x02, 216 },
        { 24, 0x02, 216 },
        { 31, 0x02, 216 },
        { 41, 0x02, 216 },
        { 56, 0x03, 216 },
        { 3, 0x02, 217 },
        { 6, 0x02, 217 },
        { 10, 0x02, 217 },
        { 15, 0x02, 217 },
        { 24, 0x02, 217 },
        { 31, 0x02, 217 },
        { 41, 0x02, 217 },
        { 56, 0x03, 217 },
    },
    /* 118 */
    {
        { 3, 0x02, 227 },
        { 6, 0x02, 227 },
        { 10, 0x02, 227 },
        { 15, 0x02, 227 },
        { 24, 0x02, 227 },
        { 31, 0x02, 227 },
        { 41, 0x02, 227 },
        { 56, 0x03, 227 },
        { 3, 0x02, 229 },
        { 6, 0x02, 229 },
        { 10, 0x02, 229 },
        { 15, 0x02, 229 },
        { 24, 0x02, 229 },
        { 31, 0x02, 229 },
        { 41, 0x02, 229 },
        { 56, 0x03, 229 },
    },
    /* 119 */
    {
        { 1, 0x02, 230 },
        { 22, 0x03, 230 },
        { 0, 0x03, 129 },
        { 0, 0x03, 132 },
        { 0, 0x03, 133 },
        { 0, 0x03, 134 },
        { 0, 0x03, 136 },
        { 0, 0x03, 146 },
        { 0, 0x03, 154 },
        { 0, 0x03, 156 },
        { 0, 0x03, 160 },
        { 0, 0x03, 163 },
        { 0, 0x03, 164 },
        { 0, 0x03, 169 },
        { 0, 0x03, 170 },
        { 0, 0x03, 173 },
    },
    /* 120 */
    {
        { 2, 0x02, 230 },
        { 9, 0x02, 230 },
        { 23, 0x02, 230 },
        { 40, 0x03, 230 },
        { 1, 0x02, 129 },
        { 22, 0x03, 129 },
        { 1, 0x02, 132 },
        { 22, 0x03, 132 },
        { 1, 0x02, 133 },
        { 22, 0x03, 133 },
        { 1, 0x02, 134 },
        { 22, 0x03, 134 },
        { 1, 0x02, 136 },
        { 22, 0x03, 136 },
        { 1, 0x02, 146 },
        { 22, 0x03, 146 },
    },
    /* 121 */
    {
        { 3, 0x02, 230 },
        { 6, 0x02, 230 },
        { 10, 0x02, 230 },
        { 15, 0x02, 230 },
        { 24, 0x02, 230 },
        { 31, 0x02, 230 },
        { 41, 0x02, 230 },
        { 56, 0x03, 230 },
        { 2, 0x02, 129 },
        { 9, 0x02, 129 },
        { 23, 0x02, 129 },
        { 40, 0x03, 129 },
        { 2, 0x02, 132 },
        { 9, 0x02, 132 },
        { 23, 0x02, 132 },
        { 40, 0x03, 132 },
    },
    /* 122 */
    {
        { 3, 0x02, 129 },
        { 6, 0x02, 129 },
        { 10, 0x02, 129 },
        { 15, 0x02, 129 },
        { 24, 0x02, 129 },
        { 31, 0x02, 129 },
        { 41, 0x02, 129 },
        { 56, 0x03, 129 },
        { 3, 0x02, 132 },
        { 6, 0x02, 132 },
        { 10, 0x02, 132 },
        { 15, 0x02, 132 },
        { 24, 0x02, 132 },
        { 31, 0x02, 132 },
        { 41, 0x02, 132 },
        { 56, 0x03, 132 },
    },
    /* 123 */
    {
        { 2, 0x02, 133 },
        { 9, 0x02, 133 },
        { 23, 0x02, 133 },
        { 40, 0x03, 133 },
        { 2, 0x02, 134 },
        { 9, 0x02, 134 },
        { 23, 0x02, 134 },
        { 40, 0x03, 134 },
        { 2, 0x02, 136 },
        { 9, 0x02, 136 },
        { 23, 0x02, 136 },
        { 40, 0x03, 136 },
        { 2, 0x02, 146 },
        { 9, 0x02, 146 },
        { 23, 0x02, 146 },
        { 40, 0x03, 146 },
    },
    /* 124 */
    {
        { 3, 0x02, 133 },
        { 6, 0x02, 133 },
        { 10, 0x02, 133 },
        { 15, 0x02, 133 },
        { 24, 0x02, 133 },
        { 31, 0x02, 133 },
        { 41, 0x02, 133 },
        { 56, 0x03, 133 },
        { 3, 0x02, 134 },
        { 6, 0x02, 134 },
        { 10, 0x02, 134 },
        { 15, 0x02, 134 },
        { 24, 0x02, 134 },
        { 31, 0x02, 134 },
        { 41, 0x02, 134 },
        { 56, 0x03, 134 },
    },
    /* 125 */
    {
        { 3, 0x02, 136 },
        { 6, 0x02, 136 },
        { 10, 0x02, 136 },
        { 15, 0x02, 136 },
        { 24, 0x02, 136 },
        { 31, 0x02, 136 },
        { 41, 0x02, 136 },
        { 56, 0x03, 136 },
        { 3, 0x02, 146 },
        { 6, 0x02, 146 },
        { 10, 0x02, 146 },
        { 15, 0x02, 146 },
        { 24, 0x02, 146 },
        { 31, 0x02, 146 },
        { 41, 0x02, 146 },
        { 56, 0x03, 146 },
    },
    /* 126 */
    {
        { 1, 0x02, 154 },
        { 22, 0x03, 154 },
        { 1, 0x02, 156 },
        { 22, 0x03, 156 },
        { 1, 0x02, 160 },
        { 22, 0x03, 160 },
        { 1, 0x02, 163 },
        { 22, 0x03, 163 },
        { 1, 0x02, 164 },
        { 22, 0x03, 164 },
        { 1, 0x02, 169 },
        { 22, 0x03, 169 },
        { 1, 0x02, 170 },
        { 22, 0x03, 170 },
        { 1, 0x02, 173 },
        { 22, 0x03, 173 },
    },
    /* 127 */
    {
        { 2, 0x02, 154 },
        { 9, 0x02, 154 },
        { 23, 0x02, 154 },
        { 40, 0x03, 154 },
        { 2, 0x02, 156 },
        { 9, 0x02, 156 },
        { 23, 0x02, 156 },
        { 40, 0x03, 156 },
        { 2, 0x02, 160 },
        { 9, 0x02, 160 },
        { 23, 0x02, 160 },
        { 40, 0x03, 160 },
        { 2, 0x02, 163 },
        { 9, 0x02, 163 },
        { 23, 0x02, 163 },
        { 40, 0x03, 163 },
    },
    /* 128 */
    {
        { 3, 0x02, 154 },
        { 6, 0x02, 154 },
        { 10, 0x02, 154 },
        { 15, 0x02, 154 },
        { 24, 0x02, 154 },
        { 31, 0x02, 154 },
        { 41, 0x02, 154 },
        { 56, 0x03, 154 },
        { 3, 0x02, 156 },
        { 6, 0x02, 156 },
        { 10, 0x02, 156 },
        { 15, 0x02, 156 },
        { 24, 0x02, 156 },
        { 31, 0x02, 156 },
        { 41, 0x02, 156 },
        { 56, 0x03, 156 },
    },
    /* 129 */
    {
        { 3, 0x02, 160 },
        { 6, 0x02, 160 },
        { 10, 0x02, 160 },
        { 15, 0x02, 160 },
        { 24, 0x02, 160 },
        { 31, 0x02, 160 },
        { 41, 0x02, 160 },
        { 56, 0x03, 160 },
        { 3, 0x02, 163 },
        { 6, 0x02, 163 },
        { 10, 0x02, 163 },
        { 15, 0x02, 163 },
        { 24, 0x02, 163 },
        { 31, 0x02, 163 },
        { 41, 0x02, 163 },
        { 56, 0x03, 163 },
    },
    /* 130 */
    {
        { 2, 0x02, 164 },
        { 9, 0x02, 164 },
        { 23, 0x02, 164 },
        { 40, 0x03, 164 },
        { 2, 0x02, 169 },
        { 9, 0x02, 169 },
        { 23, 0x02, 169 },
        { 40, 0x03, 169 },
        { 2, 0x02, 170 },
        { 9, 0x02, 170 },
        { 23, 0x02, 170 },
        { 40, 0x03, 170 },
        { 2, 0x02, 173 },
        { 9, 0x02, 173 },
        { 23, 0x02, 173 },
        { 40, 0x03, 173 },
    },
    /* 131 */
    {
        { 3, 0x02, 164 },
        { 6, 0x02, 164 },
        { 10, 0x02, 164 },
        { 15, 0x02, 164 },
        { 24, 0x02, 164 },
        { 31, 0x02, 164 },
        { 41, 0x02, 164 },
        { 56, 0x03, 164 },
        { 3, 0x02, 169 },
        { 6, 0x02, 169 },
        { 10, 0x02, 169 },
        { 15, 0x02, 169 },
        { 24, 0x02, 169 },
        { 31, 0x02, 169 },
        { 41, 0x02, 169 },
        { 56, 0x03, 169 },
    },
    /* 132 */
    {
        { 3, 0x02, 170 },
        { 6, 0x02, 170 },
        { 10, 0x02, 170 },
        { 15, 0x02, 170 },
        { 24, 0x02, 170 },
        { 31, 0x02, 170 },
        { 41, 0x02, 170 },
        { 56, 0x03, 170 },
        { 3, 0x02, 173 },
        { 6, 0x02, 173 },
        { 10, 0x02, 173 },
        { 15, 0x02, 173 },
        { 24, 0x02, 173 },
        { 31, 0x02, 173 },
        { 41, 0x02, 173 },
        { 56, 0x03, 173 },
    },
    /* 133 */
    {
        { 137, 0x00, 0 },
        { 138, 0x00, 0 },
        { 140, 0x00, 0 },
        { 141, 0x00, 0 },
        { 144, 0x00, 0 },
        { 145, 0x00, 0 },
        { 147, 0x00, 0 },
        { 150, 0x00, 0 },
        { 156, 0x00, 0 },
        { 159, 0x00, 0 },
        { 163, 0x00, 0 },
        { 166, 0x00, 0 },
        { 171, 0x00, 0 },
        { 174, 0x00, 0 },
        { 181, 0x00, 0 },
        { 190, 0x00, 0 },
    },
    /* 134 */
    {
        { 0, 0x03, 178 },
        { 0, 0x03, 181 },
        { 0, 0x03, 185 },
        { 0, 0x03, 186 },
        { 0, 0x03, 187 },
        { 0, 0x03, 189 },
        { 0, 0x03, 190 },
        { 0, 0x03, 196 },
        { 0, 0x03, 198 },
        { 0, 0x03, 228 },
        { 0, 0x03, 232 },
        { 0, 0x03, 233 },
        { 148, 0x00, 0 },
        { 149, 0x00, 0 },
        { 151, 0x00, 0 },
        { 152, 0x00, 0 },
    },
    /* 135 */
    {
        { 1, 0x02, 178 },
        { 22, 0x03, 178 },
        { 1, 0x02, 181 },
        { 22, 0x03, 181 },
        { 1, 0x02, 185 },
        { 22, 0x03, 185 },
        { 1, 0x02, 186 },
        { 22, 0x03, 186 },
        { 1, 0x02, 187 },
        { 22, 0x03, 187 },
        { 1, 0x02, 189 },
        { 22, 0x03, 189 },
        { 1, 0x02, 190 },
        { 22, 0x03, 190 },
        { 1, 0x02, 196 },
        { 22, 0x03, 196 },
    },
    /* 136 */
    {
        { 2, 0x02, 178 },
        { 9, 0x02, 178 },
        { 23, 0x02, 178 },
        { 40, 0x03, 178 },
        { 2, 0x02, 181 },
        { 9, 0x02, 181 },
        { 23, 0x02, 181 },
        { 40, 0x03, 181 },
        { 2, 0x02, 185 },
        { 9, 0x02, 185 },
        { 23, 0x02, 185 },
        { 40, 0x03, 185 },
        { 2, 0x02, 186 },
        { 9, 0x02, 186 },
        { 23, 0x02, 186 },
        { 40, 0x03, 186 },
    },
    /* 137 */
    {
        { 3, 0x02, 178 },
        { 6, 0x02, 178 },
        { 10, 0x02, 178 },
        { 15, 0x02, 178 },
        { 24, 0x02, 178 },
        { 31, 0x02, 178 },
        { 41, 0x02, 178 },
        { 56, 0x03, 178 },
        { 3, 0x02, 181 },
        { 6, 0x02, 181 },
        { 10, 0x02, 181 },
        { 15, 0x02, 181 },
        { 24, 0x02, 181 },
        { 31, 0x02, 181 },
        { 41, 0x02, 181 },
        { 56, 0x03, 181 },
    },
    /* 138 */
    {
        { 3, 0x02, 185 },
        { 6, 0x02, 185 },
        { 10, 0x02, 185 },
        { 15, 0x02, 185 },
        { 24, 0x02, 185 },
        { 31, 0x02, 185 },
        { 41, 0x02, 185 },
        { 56, 0x03, 185 },
        { 3, 0x02, 186 },
        { 6, 0x02, 186 },
        { 10, 0x02, 186 },
        { 15, 0x02, 186 },
        { 24, 0x02, 186 },
        { 31, 0x02, 186 },
        { 41, 0x02, 186 },
        { 56, 0x03, 186 },
    },
    /* 139 */
    {
        { 2, 0x02, 187 },
        { 9, 0x02, 187 },
        { 23, 0x02, 187 },
        { 40, 0x03, 187 },
        { 2, 0x02, 189 },
        { 9, 0x02, 189 },
        { 23, 0x02, 189 },
        { 40, 0x03, 189 },
        { 2, 0x02, 190 },
        { 9, 0x02, 190 },
        { 23, 0x02, 190 },
        { 40, 0x03, 190 },
        { 2, 0x02, 196 },
        { 9, 0x02, 196 },
        { 23, 0x02, 196 },
        { 40, 0x03, 196 },
    },
    /* 140 */
    {
        { 3, 0x02, 187 },
        { 6, 0x02, 187 },
        { 10, 0x02, 187 },
        { 15, 0x02, 187 },
        { 24, 0x02, 187 },
        { 31, 0x02, 187 },
        { 41, 0x02, 187 },
        { 56, 0x03, 187 },
        { 3, 0x02, 189 },
        { 6, 0x02, 189 },
        { 10, 0x02, 189 },
        { 15, 0x02, 189 },
        { 24, 0x02, 189 },
        { 31, 0x02, 189 },
        { 41, 0x02, 189 },
        { 56, 0x03, 189 },
    },
    /* 141 */
    {
        { 3, 0x02, 190 },
        { 6, 0x02, 190 },
        { 10, 0x02, 190 },
        { 15, 0x02, 190 },
        { 24, 0x02, 190 },
        { 31, 0x02, 190 },
        { 41, 0x02, 190 },
        { 56, 0x03, 190 },
        { 3, 0x02, 196 },
        { 6, 0x02, 196 },
        { 10, 0x02, 196 },
        { 15, 0x02, 196 },
        { 24, 0x02, 196 },
        { 31, 0x02, 196 },
        { 41, 0x02, 196 },
        { 56, 0x03, 196 },
    },
    /* 142 */
    {
        { 1, 0x02, 198 },
        { 22, 0x03, 198 },
        { 1, 0x02, 228 },
        { 22, 0x03, 228 },
        { 1, 0x02, 232 },
        { 22, 0x03, 232 },
        { 1, 0x02, 233 },
        { 22, 0x03, 233 },
        { 0, 0x03, 1 },
        { 0, 0x03, 135 },
        { 0, 0x03, 137 },
        { 0, 0x03, 138 },
        { 0, 0x03, 139 },
        { 0, 0x03, 140 },
        { 0, 0x03, 141 },
        { 0, 0x03, 143 },
    },
    /* 143 */
    {
        { 2, 0x02, 198 },
        { 9, 0x02, 198 },
        { 23, 0x02, 198 },
        { 40, 0x03, 198 },
        { 2, 0x02, 228 },
        { 9, 0x02, 228 },
        { 23, 0x02, 228 },
        { 40, 0x03, 228 },
        { 2, 0x02, 232 },
        { 9, 0x02, 232 },
        { 23, 0x02, 232 },
        { 40, 0x03, 232 },
        { 2, 0x02, 233 },
        { 9, 0x02, 233 },
        { 23, 0x02, 233 },
        { 40, 0x03, 233 },
    },
    /* 144 */
    {
        { 3, 0x02, 198 },
        { 6, 0x02, 198 },
        { 10, 0x02, 198 },
        { 15, 0x02, 198 },
        { 24, 0x02, 198 },
        { 31, 0x02, 198 },
        { 41, 0x02, 198 },
        { 56, 0x03, 198 },
        { 3, 0x02, 228 },
        { 6, 0x02, 228 },
        { 10, 0x02, 228 },
        { 15, 0x02, 228 },
        { 24, 0x02, 228 },
        { 31, 0x02, 228 },
        { 41, 0x02, 228 },
        { 56, 0x03, 228 },
    },
    /* 145 */
    {
        { 3, 0x02, 232 },
        { 6, 0x02, 232 },
        { 10, 0x02, 232 },
        { 15, 0x02, 232 },
        { 24, 0x02, 232 },
        { 31, 0x02, 232 },
        { 41, 0x02, 232 },
        { 56, 0x03, 232 },
        { 3, 0x02, 233 },
        { 6, 0x02, 233 },
        { 10, 0x02, 233 },
        { 15, 0x02, 233 },
        { 24, 0x02, 233 },
        { 31, 0x02, 233 },
        { 41, 0x02, 233 },
        { 56, 0x03, 233 },
    },
    /* 146 */
    {
        { 1, 0x02, 1 },
        { 22, 0x03, 1 },
        { 1, 0x02, 135 },
        { 22, 0x03, 135 },
        { 1, 0x02, 137 },
        { 22, 0x03, 137 },
        { 1, 0x02, 138 },
        { 22, 0x03, 138 },
        { 1, 0x02, 139 },
        { 22, 0x03, 139 },
        { 1, 0x02, 140 },
        { 22, 0x03, 140 },
        { 1, 0x02, 141 },
        { 22, 0x03, 141 },
        { 1, 0x02, 143 },
        { 22, 0x03, 143 },
    },
    /* 147 */
    {
        { 2, 0x02, 1 },
        { 9, 0x02, 1 },
        { 23, 0x02, 1 },
        { 40, 0x03, 1 },
        { 2, 0x02, 135 },
        { 9, 0x02, 135 },
        { 23, 0x02, 135 },
        { 40, 0x03, 135 },
        { 2, 0x02, 137 },
        { 9, 0x02, 137 },
        { 23, 0x02, 137 },
        { 40, 0x03, 137 },
        { 2, 0x02, 138 },
        { 9, 0x02, 138 },
        { 23, 0x02, 138 },
        { 40, 0x03, 138 },
    },
    /* 148 */
    {
        { 3, 0x02, 1 },
        { 6, 0x02, 1 },
        { 10, 0x02, 1 },
        { 15, 0x02, 1 },
        { 24, 0x02, 1 },
        { 31, 0x02, 1 },
        { 41, 0x02, 1 },
        { 56, 0x03, 1 },
        { 3, 0x02, 135 },
        { 6, 0x02, 135 },
        { 10, 0x02, 135 },
        { 15, 0x02, 135 },
        { 24, 0x02, 135 },
        { 31, 0x02, 135 },
        { 41, 0x02, 135 },
        { 56, 0x03, 135 },
    },
    /* 149 */
    {
        { 3, 0x02, 137 },
        { 6, 0x02, 137 },
        { 10, 0x02, 137 },
        { 15, 0x02, 137 },
        { 24, 0x02, 137 },
        { 31, 0x02, 137 },
        { 41, 0x02, 137 },
        { 56, 0x03, 137 },
        { 3, 0x02, 138 },
        { 6, 0x02, 138 },
        { 10, 0x02, 138 },
        { 15, 0x02, 138 },
        { 24, 0x02, 138 },
        { 31, 0x02, 138 },
        { 41, 0x02, 138 },
        { 56, 0x03, 138 },
    },
    /* 150 */
    {
        { 2, 0x02, 139 },
        { 9, 0x02, 139 },
        { 23, 0x02, 139 },
        { 40, 0x03, 139 },
        { 2, 0x02, 140 },
        { 9, 0x02, 140 },
        { 23, 0x02, 140 },
        { 40, 0x03, 140 },
        { 2, 0x02, 141 },
        { 9, 0x02, 141 },
        { 23, 0x02, 141 },
        { 40, 0x03, 141 },
        { 2, 0x02, 143 },
        { 9, 0x02, 143 },
        { 23, 0x02, 143 },
        { 40, 0x03, 143 },
    },
    /* 151 */
    {
        { 3, 0x02, 139 },
        { 6, 0x02, 139 },
        { 10, 0x02, 139 },
        { 15, 0x02, 139 },
        { 24, 0x02, 139 },
        { 31, 0x02, 139 },
        { 41, 0x02, 139 },
        { 56, 0x03, 139 },
        { 3, 0x02, 140 },
        { 6, 0x02, 140 },
        { 10, 0x02, 140 },
        { 15, 0x02, 140 },
        { 24, 0x02, 140 },
        { 31, 0x02, 140 },
        { 41, 0x02, 140 },
        { 56, 0x03, 140 },
    },
    /* 152 */
    {
        { 3, 0x02, 141 },
        { 6, 0x02, 141 },
        { 10, 0x02, 141 },
        { 15, 0x02, 141 },
        { 24, 0x02, 141 },
        { 31, 0x02, 141 },
        { 41, 0x02, 141 },
        { 56, 0x03, 141 },
        { 3, 0x02, 143 },
        { 6, 0x02, 143 },
        { 10, 0x02, 143 },
        { 15, 0x02, 143 },
        { 24, 0x02, 143 },
        { 31, 0x02, 143 },
        { 41, 0x02, 143 },
        { 56, 0x03, 143 },
    },
    /* 153 */
    {
        { 157, 0x00, 0 },
        { 158, 0x00, 0 },
        { 160, 0x00, 0 },
        { 161, 0x00, 0 },
        { 164, 0x00, 0 },
        { 165, 0x00, 0 },
        { 167, 0x00, 0 },
        { 168, 0x00, 0 },
        { 172, 0x00, 0 },
        { 173, 0x00, 0 },
        { 175, 0x00, 0 },
        { 177, 0x00, 0 },
        { 182, 0x00, 0 },
        { 185, 0x00, 0 },
        { 191, 0x00, 0 },
        { 207, 0x00, 0 },
    },
    /* 154 */
    {
        { 0, 0x03, 147 },
        { 0, 0x03, 149 },
        { 0, 0x03, 150 },
        { 0, 0x03, 151 },
        { 0, 0x03, 152 },
        { 0, 0x03, 155 },
        { 0, 0x03, 157 },
        { 0, 0x03, 158 },
        { 0, 0x03, 165 },
        { 0, 0x03, 166 },
        { 0, 0x03, 168 },
        { 0, 0x03, 174 },
        { 0, 0x03, 175 },
        { 0, 0x03, 180 },
        { 0, 0x03, 182 },
        { 0, 0x03, 183 },
    },
    /* 155 */
    {
        { 1, 0x02, 147 },
        { 22, 0x03, 147 },
        { 1, 0x02, 149 },
        { 22, 0x03, 149 },
        { 1, 0x02, 150 },
        { 22, 0x03, 150 },
        { 1, 0x02, 151 },
        { 22, 0x03, 151 },
        { 1, 0x02, 152 },
        { 22, 0x03, 152 },
        { 1, 0x02, 155 },
        { 22, 0x03, 155 },
        { 1, 0x02, 157 },
        { 22, 0x03, 157 },
        { 1, 0x02, 158 },
        { 22, 0x03, 158 },
    },
    /* 156 */
    {
        { 2, 0x02, 147 },
        { 9, 0x02, 147 },
        { 23, 0x02, 147 },
        { 40, 0x03, 147 },
        { 2, 0x02, 149 },
        { 9, 0x02, 149 },
        { 23, 0x02, 149 },
        { 40, 0x03, 149 },
        { 2, 0x02, 150 },
        { 9, 0x02, 150 },
        { 23, 0x02, 150 },
        { 40, 0x03, 150 },
        { 2, 0x02, 151 },
        { 9, 0x02, 151 },
        { 23, 0x02, 151 },
        { 40, 0x03, 151 },
    },
    /* 157 */
    {
        { 3, 0x02, 147 },
        { 6, 0x02, 147 },
        { 10, 0x02, 147 },
        { 15, 0x02, 147 },
        { 24, 0x02, 147 },
        { 31, 0x02, 147 },
        { 41, 0x02, 147 },
        { 56, 0x03, 147 },
        { 3, 0x02, 149 },
        { 6, 0x02, 149 },
        { 10, 0x02, 149 },
        { 15, 0x02, 149 },
        { 24, 0x02, 149 },
        { 31, 0x02, 149 },
        { 41, 0x02, 149 },
        { 56, 0x03, 149 },
    },
    /* 158 */
    {
        { 3, 0x02, 150 },
        { 6, 0x02, 150 },
        { 10, 0x02, 150 },
        { 15, 0x02, 150 },
        { 24, 0x02, 150 },
        { 31, 0x02, 150 },
        { 41, 0x02, 150 },
        { 56, 0x03, 150 },
        { 3, 0x02, 151 },
        { 6, 0x02, 151 },
        { 10, 0x02, 151 },
        { 15, 0x02, 151 },
        { 24, 0x02, 151 },
        { 31, 0x02, 151 },
        { 41, 0x02, 151 },
        { 56, 0x03, 151 },
    },
    /* 159 */
    {
        { 2, 0x02, 152 },
        { 9, 0x02, 152 },
        { 23, 0x02, 152 },
        { 40, 0x03, 152 },
        { 2, 0x02, 155 },
        { 9, 0x02, 155 },
        { 23, 0x02, 155 },
        { 40, 0x03, 155 },
        { 2, 0x02, 157 },
        { 9, 0x02, 157 },
        { 23, 0x02, 157 },
        { 40, 0x03, 157 },
        { 2, 0x02, 158 },
        { 9, 0x02, 158 },
        { 23, 0x02, 158 },
        { 40, 0x03, 158 },
    },
    /* 160 */
    {
        { 3, 0x02, 152 },
        { 6, 0x02, 152 },
        { 10, 0x02, 152 },
        { 15, 0x02, 152 },
        { 24, 0x02, 152 },
        { 31, 0x02, 152 },
        { 41, 0x02, 152 },
        { 56, 0x03, 152 },
        { 3, 0x02, 155 },
        { 6, 0x02, 155 },
        { 10, 0x02, 155 },
        { 15, 0x02, 155 },
        { 24, 0x02, 155 },
        { 31, 0x02, 155 },
        { 41, 0x02, 155 },
        { 56, 0x03, 155 },
    },
    /* 161 */
    {
        { 3, 0x02, 157 },
        { 6, 0x02, 157 },
        { 10, 0x02, 157 },
        { 15, 0x02, 157 },
        { 24, 0x02, 157 },
        { 31, 0x02, 157 },
        { 41, 0x02, 157 },
        { 56, 0x03, 157 },
        { 3, 0x02, 158 },
        { 6, 0x02, 158 },
        { 10, 0x02, 158 },
        { 15, 0x02, 158 },
        { 24, 0x02, 158 },
        { 31, 0x02, 158 },
        { 41, 0x02, 158 },
        { 56, 0x03, 158 },
    },
    /* 162 */
    {
        { 1, 0x02, 165 },
        { 22, 0x03, 165 },
        { 1, 0x02, 166 },
        { 22, 0x03, 166 },
        { 1, 0x02, 168 },
        { 22, 0x03, 168 },
        { 1, 0x02, 174 },
        { 22, 0x03, 174 },
        { 1, 0x02, 175 },
        { 22, 0x03, 175 },
        { 1, 0x02, 180 },
        { 22, 0x03, 180 },
        { 1, 0x02, 182 },
        { 22, 0x03, 182 },
        { 1, 0x02, 183 },
        { 22, 0x03, 183 },
    },
    /* 163 */
    {
        { 2, 0x02, 165 },
        { 9, 0x02, 165 },
        { 23, 0x02, 165 },
        { 40, 0x03, 165 },
        { 2, 0x02, 166 },
        { 9, 0x02, 166 },
        { 23, 0x02, 166 },
        { 40, 0x03, 166 },
        { 2, 0x02, 168 },
        { 9, 0x02, 168 },
        { 23, 0x02, 168 },
        { 40, 0x03, 168 },
        { 2, 0x02, 174 },
        { 9, 0x02, 174 },
        { 23, 0x02, 174 },
        { 40, 0x03, 174 },
    },
    /* 164 */
    {
        { 3, 0x02, 165 },
        { 6, 0x02, 165 },
        { 10, 0x02, 165 },
        { 15, 0x02, 165 },
        { 24, 0x02, 165 },
        { 31, 0x02, 165 },
        { 41, 0x02, 165 },
        { 56, 0x03, 165 },
        { 3, 0x02, 166 },
        { 6, 0x02, 166 },
        { 10, 0x02, 166 },
        { 15, 0x02, 166 },
        { 24, 0x02, 166 },
        { 31, 0x02, 166 },
        { 41, 0x02, 166 },
        { 56, 0x03, 166 },
    },
    /* 165 */
    {
        { 3, 0x02, 168 },
        { 6, 0x02, 168 },
        { 10, 0x02, 168 },
        { 15, 0x02, 168 },
        { 24, 0x02, 168 },
        { 31, 0x02, 168 },
        { 41, 0x02, 168 },
        { 56, 0x03, 168 },
        { 3, 0x02, 174 },
        { 6, 0x02, 174 },
        { 10, 0x02, 174 },
        { 15, 0x02, 174 },
        { 24, 0x02, 174 },
        { 31, 0x02, 174 },
        { 41, 0x02, 174 },
        { 56, 0x03, 174 },
    },
    /* 166 */
    {
        { 2, 0x02, 175 },
        { 9, 0x02, 175 },
        { 23, 0x02, 175 },
        { 40, 0x03, 175 },
        { 2, 0x02, 180 },
        { 9, 0x02, 180 },
        { 23, 0x02, 180 },
        { 40, 0x03, 180 },
        { 2, 0x02, 182 },
        { 9, 0x02, 182 },
        { 23, 0x02, 182 },
        { 40, 0x03, 182 },
        { 2, 0x02, 183 },
        { 9, 0x02, 183 },
        { 23, 0x02, 183 },
        { 40, 0x03, 183 },
    },
    /* 167 */
    {
        { 3, 0x02, 175 },
        { 6, 0x02, 175 },
        { 10, 0x02, 175 },
        { 15, 0x02, 175 },
        { 24, 0x02, 175 },
        { 31, 0x02, 175 },
        { 41, 0x02, 175 },
        { 56, 0x03, 175 },
        { 3, 0x02, 180 },
        { 6, 0x02, 180 },
        { 10, 0x02, 180 },
        { 15, 0x02, 180 },
        { 24, 0x02, 180 },
        { 31, 0x02, 180 },
        { 41, 0x02, 180 },
        { 56, 0x03, 180 },
    },
    /* 168 */
    {
        { 3, 0x02, 182 },
        { 6, 0x02, 182 },
        { 10, 0x02, 182 },
        { 15, 0x02, 182 },
        { 24, 0x02, 182 },
        { 31, 0x02, 182 },
        { 41, 0x02, 182 },
        { 56, 0x03, 182 },
        { 3, 0x02, 183 },
        { 6, 0x02, 183 },
        { 10, 0x02, 183 },
        { 15, 0x02, 183 },
        { 24, 0x02, 183 },
        { 31, 0x02, 183 },
        { 41, 0x02, 183 },
        { 56, 0x03, 183 },
    },
    /* 169 */
    {
        { 0, 0x03, 188 },
        { 0, 0x03, 191 },
        { 0, 0x03, 197 },
        { 0, 0x03, 231 },
        { 0, 0x03, 239 },
        { 176, 0x00, 0 },
        { 178, 0x00, 0 },
        { 179, 0x00, 0 },
        { 183, 0x00, 0 },
        { 184, 0x00, 0 },
        { 186, 0x00, 0 },
        { 187, 0x00, 0 },
        { 192, 0x00, 0 },
        { 199, 0x00, 0 },
        { 208, 0x00, 0 },
        { 223, 0x00, 0 },
    },
    /* 170 */
    {
        { 1, 0x02, 188 },
        { 22, 0x03, 188 },
        { 1, 0x02, 191 },
        { 22, 0x03, 191 },
        { 1, 0x02, 197 },
        { 22, 0x03, 197 },
        { 1, 0x02, 231 },
        { 22, 0x03, 231 },
        { 1, 0x02, 239 },
        { 22, 0x03, 239 },
        { 0, 0x03, 9 },
        { 0, 0x03, 142 },
        { 0, 0x03, 144 },
        { 0, 0x03, 145 },
        { 0, 0x03, 148 },
        { 0, 0x03, 159 },
    },
    /* 171 */
    {
        { 2, 0x02, 188 },
        { 9, 0x02, 188 },
        { 23, 0x02, 188 },
        { 40, 0x03, 188 },
        { 2, 0x02, 191 },
        { 9, 0x02, 191 },
        { 23, 0x02, 191 },
        { 40, 0x03, 191 },
        { 2, 0x02, 197 },
        { 9, 0x02, 197 },
        { 23, 0x02, 197 },
        { 40, 0x03, 197 },
        { 2, 0x02, 231 },
        { 9, 0x02, 231 },
        { 23, 0x02, 231 },
        { 40, 0x03, 231 },
    },
    /* 172 */
    {
        { 3, 0x02, 188 },
        { 6, 0x02, 188 },
        { 10, 0x02, 188 },
        { 15, 0x02, 188 },
        { 24, 0x02, 188 },
        { 31, 0x02, 188 },
        { 41, 0x02, 188 },
        { 56, 0x03, 188 },
        { 3, 0x02, 191 },
        { 6, 0x02, 191 },
        { 10, 0x02, 191 },
        { 15, 0x02, 191 },
        { 24, 0x02, 191 },
        { 31, 0x02, 191 },
        { 41, 0x02, 191 },
        { 56, 0x03, 191 },
    },
    /* 173 */
    {
        { 3, 0x02, 197 },
        { 6, 0x02, 197 },
        { 10, 0x02, 197 },
        { 15, 0x02, 197 },
        { 24, 0x02, 197 },
        { 31, 0x02, 197 },
        { 41, 0x02, 197 },
        { 56, 0x03, 197 },
        { 3, 0x02, 231 },
        { 6, 0x02, 231 },
        { 10, 0x02, 231 },
        { 15, 0x02, 231 },
        { 24, 0x02, 231 },
        { 31, 0x02, 231 },
        { 41, 0x02, 231 },
        { 56, 0x03, 231 },
    },
    /* 174 */
    {
        { 2, 0x02, 239 },
        { 9, 0x02, 239 },
        { 23, 0x02, 239 },
        { 40, 0x03, 239 },
        { 1, 0x02, 9 },
        { 22, 0x03, 9 },
        { 1, 0x02, 142 },
        { 22, 0x03, 142 },
        { 1, 0x02, 144 },
        { 22, 0x03, 144 },
        { 1, 0x02, 145 },
        { 22, 0x03, 145 },
        { 1, 0x02, 148 },
        { 22, 0x03, 148 },
        { 1, 0x02, 159 },
        { 22, 0x03, 159 },
    },
    /* 175 */
    {
        { 3, 0x02, 239 },
        { 6, 0x02, 239 },
        { 10, 0x02, 239 },
        { 15, 0x02, 239 },
        { 24, 0x02, 239 },
        { 31, 0x02, 239 },
        { 41, 0x02, 239 },
        { 56, 0x03, 239 },
        { 2, 0x02, 9 },
        { 9, 0x02, 9 },
        { 23, 0x02, 9 },
        { 40, 0x03, 9 },
        { 2, 0x02, 142 },
        { 9, 0x02, 142 },
        { 23, 0x02, 142 },
        { 40, 0x03, 142 },
    },
    /* 176 */
    {
        { 3, 0x02, 9 },
        { 6, 0x02, 9 },
        { 10, 0x02, 9 },
        { 15, 0x02, 9 },
        { 24, 0x02, 9 },
        { 31, 0x02, 9 },
        { 41, 0x02, 9 },
        { 56, 0x03, 9 },
        { 3, 0x02, 142 },
        { 6, 0x02, 142 },
        { 10, 0x02, 142 },
        { 15, 0x02, 142 },
        { 24, 0x02, 142 },
        { 31, 0x02, 142 },
        { 41, 0x02, 142 },
        { 56, 0x03, 142 },
    },
    /* 177 */
    {
        { 2, 0x02, 144 },
        { 9, 0x02, 144 },
        { 23, 0x02, 144 },
        { 40, 0x03, 144 },
        { 2, 0x02, 145 },
        { 9, 0x02, 145 },
        { 23, 0x02, 145 },
        { 40, 0x03, 145 },
        { 2, 0x02, 148 },
        { 9, 0x02, 148 },
        { 23, 0x02, 148 },
        { 40, 0x03, 148 },
        { 2, 0x02, 159 },
        { 9, 0x02, 159 },
        { 23, 0x02, 159 },
        { 40, 0x03, 159 },
    },
    /* 178 */
    {
        { 3, 0x02, 144 },
        { 6, 0x02, 144 },
        { 10, 0x02, 144 },
        { 15, 0x02, 144 },
        { 24, 0x02, 144 },
        { 31, 0x02, 144 },
        { 41, 0x02, 144 },
        { 56, 0x03, 144 },
        { 3, 0x02, 145 },
        { 6, 0x02, 145 },
        { 10, 0x02, 145 },
        { 15, 0x02, 145 },
        { 24, 0x02, 145 },
        { 31, 0x02, 145 },
        { 41, 0x02, 145 },
        { 56, 0x03, 145 },
    },
    /* 179 */
    {
        { 3, 0x02, 148 },
        { 6, 0x02, 148 },
        { 10, 0x02, 148 },
        { 15, 0x02, 148 },
        { 24, 0x02, 148 },
        { 31, 0x02, 148 },
        { 41, 0x02, 148 },
        { 56, 0x03, 148 },
        { 3, 0x02, 159 },
        { 6, 0x02, 159 },
        { 10, 0x02, 159 },
        { 15, 0x02, 159 },
        { 24, 0x02, 159 },
        { 31, 0x02, 159 },
        { 41, 0x02, 159 },
        { 56, 0x03, 159 },
    },
    /* 180 */
    {
        { 0, 0x03, 171 },
        { 0, 0x03, 206 },
        { 0, 0x03, 215 },
        { 0, 0x03, 225 },
        { 0, 0x03, 236 },
        { 0, 0x03, 237 },
        { 188, 0x00, 0 },
        { 189, 0x00, 0 },
        { 193, 0x00, 0 },
        { 196, 0x00, 0 },
        { 200, 0x00, 0 },
        { 203, 0x00, 0 },
        { 209, 0x00, 0 },
        { 216, 0x00, 0 },
        { 224, 0x00, 0 },
        { 238, 0x00, 0 },
    },
    /* 181 */
    {
        { 1, 0x02, 171 },
        { 22, 0x03, 171 },
        { 1, 0x02, 206 },
        { 22, 0x03, 206 },
        { 1, 0x02, 215 },
        { 22, 0x03, 215 },
        { 1, 0x02, 225 },
        { 22, 0x03, 225 },
        { 1, 0x02, 236 },
        { 22, 0x03, 236 },
        { 1, 0x02, 237 },
        { 22, 0x03, 237 },
        { 0, 0x03, 199 },
        { 0, 0x03, 207 },
        { 0, 0x03, 234 },
        { 0, 0x03, 235 },
    },
    /* 182 */
    {
        { 2, 0x02, 171 },
        { 9, 0x02, 171 },
        { 23, 0x02, 171 },
        { 40, 0x03, 171 },
        { 2, 0x02, 206 },
        { 9, 0x02, 206 },
        { 23, 0x02, 206 },
        { 40, 0x03, 206 },
        { 2, 0x02, 215 },
        { 9, 0x02, 215 },
        { 23, 0x02, 215 },
        { 40, 0x03, 215 },
        { 2, 0x02, 225 },
        { 9, 0x02, 225 },
        { 23, 0x02, 225 },
        { 40, 0x03, 225 },
    },
    /* 183 */
    {
        { 3, 0x02, 171 },
        { 6, 0x02, 171 },
        { 10, 0x02, 171 },
        { 15, 0x02, 171 },
        { 24, 0x02, 171 },
        { 31, 0x02, 171 },
        { 41, 0x02, 171 },
        { 56, 0x03, 171 },
        { 3, 0x02, 206 },
        { 6, 0x02, 206 },
        { 10, 0x02, 206 },
        { 15, 0x02, 206 },
        { 24, 0x02, 206 },
        { 31, 0x02, 206 },
        { 41, 0x02, 206 },
        { 56, 0x03, 206 },
    },
    /* 184 */
    {
        { 3, 0x02, 215 },
        { 6, 0x02, 215 },
        { 10, 0x02, 215 },
        { 15, 0x02, 215 },
        { 24, 0x02, 215 },
        { 31, 0x02, 215 },
        { 41, 0x02, 215 },
        { 56, 0x03, 215 },
        { 3, 0x02, 225 },
        { 6, 0x02, 225 },
        { 10, 0x02, 225 },
        { 15, 0x02, 225 },
        { 24, 0x02, 225 },
        { 31, 0x02, 225 },
        { 41, 0x02, 225 },
        { 56, 0x03, 225 },
    },
    /* 185 */
    {
        { 2, 0x02, 236 },
        { 9, 0x02, 236 },
        { 23, 0x02, 236 },
        { 40, 0x03, 236 },
        { 2, 0x02, 237 },
        { 9, 0x02, 237 },
        { 23, 0x02, 237 },
        { 40, 0x03, 237 },
        { 1, 0x02, 199 },
        { 22, 0x03, 199 },
        { 1, 0x02, 207 },
        { 22, 0x03, 207 },
        { 1, 0x02, 234 },
        { 22, 0x03, 234 },
        { 1, 0x02, 235 },
        { 22, 0x03, 235 },
    },
    /* 186 */
    {
        { 3, 0x02, 236 },
        { 6, 0x02, 236 },
        { 10, 0x02, 236 },
        { 15, 0x02, 236 },
        { 24, 0x02, 236 },
        { 31, 0x02, 236 },
        { 41, 0x02, 236 },
        { 56, 0x03, 236 },
        { 3, 0x02, 237 },
        { 6, 0x02, 237 },
        { 10, 0x02, 237 },
        { 15, 0x02, 237 },
        { 24, 0x02, 237 },
        { 31, 0x02, 237 },
        { 41, 0x02, 237 },
        { 56, 0x03, 237 },
    },
    /* 187 */
    {
        { 2, 0x02, 199 },
        { 9, 0x02, 199 },
        { 23, 0x02, 199 },
        { 40, 0x03, 199 },
        { 2, 0x02, 207 },
        { 9, 0x02, 207 },
        { 23, 0x02, 207 },
        { 40, 0x03, 207 },
        { 2, 0x02, 234 },
        { 9, 0x02, 234 },
        { 23, 0x02, 234 },
        { 40, 0x03, 234 },
        { 2, 0x02, 235 },
        { 9, 0x02, 235 },
        { 23, 0x02, 235 },
        { 40, 0x03, 235 },
    },
    /* 188 */
    {
        { 3, 0x02, 199 },
        { 6, 0x02, 199 },
        { 10, 0x02, 199 },
        { 15, 0x02, 199 },
        { 24, 0x02, 199 },
        { 31, 0x02, 199 },
        { 41, 0x02, 199 },
        { 56, 0x03, 199 },
        { 3, 0x02, 207 },
        { 6, 0x02, 207 },
        { 10, 0x02, 207 },
        { 15, 0x02, 207 },
        { 24, 0x02, 207 },
        { 31, 0x02, 207 },
        { 41, 0x02, 207 },
        { 56, 0x03, 207 },
    },
    /* 189 */
    {
        { 3, 0x02, 234 },
        { 6, 0x02, 234 },
        { 10, 0x02, 234 },
        { 15, 0x02, 234 },
        { 24, 0x02, 234 },
        { 31, 0x02, 234 },
        { 41, 0x02, 234 },
        { 56, 0x03, 234 },
        { 3, 0x02, 235 },
        { 6, 0x02, 235 },
        { 10, 0x02, 235 },
        { 15, 0x02, 235 },
        { 24, 0x02, 235 },
        { 31, 0x02, 235 },
        { 41, 0x02, 235 },
        { 56, 0x03, 235 },
    },
    /* 190 */
    {
        { 194, 0x00, 0 },
        { 195, 0x00, 0 },
        { 197, 0x00, 0 },
        { 198, 0x00, 0 },
        { 201, 0x00, 0 },
        { 202, 0x00, 0 },
        { 204, 0x00, 0 },
        { 205, 0x00, 0 },
        { 210, 0x00, 0 },
        { 213, 0x00, 0 },
        { 217, 0x00, 0 },
        { 220, 0x00, 0 },
        { 225, 0x00, 0 },
        { 231, 0x00, 0 },
        { 239, 0x00, 0 },
        { 246, 0x00, 0 },
    },
    /* 191 */
    {
        { 0, 0x03, 192 },
        { 0, 0x03, 193 },
        { 0, 0x03, 200 },
        { 0, 0x03, 201 },
        { 0, 0x03, 202 },
        { 0, 0x03, 205 },
        { 0, 0x03, 210 },
        { 0, 0x03, 213 },
        { 0, 0x03, 218 },
        { 0, 0x03, 219 },
        { 0, 0x03, 238 },
        { 0, 0x03, 240 },
        { 0, 0x03, 242 },
        { 0, 0x03, 243 },
        { 0, 0x03, 255 },
        { 206, 0x00, 0 },
    },
    /* 192 */
    {
        { 1, 0x02, 192 },
        { 22, 0x03, 192 },
        { 1, 0x02, 193 },
        { 22, 0x03, 193 },
        { 1, 0x02, 200 },
        { 22, 0x03, 200 },
        { 1, 0x02, 201 },
        { 22, 0x03, 201 },
        { 1, 0x02, 202 },
        { 22, 0x03, 202 },
        { 1, 0x02, 205 },
        { 22, 0x03, 205 },
        { 1, 0x02, 210 },
        { 22, 0x03, 210 },
        { 1, 0x02, 213 },
        { 22, 0x03, 213 },
    },
    /* 193 */
    {
        { 2, 0x02, 192 },
        { 9, 0x02, 192 },
        { 23, 0x02, 192 },
        { 40, 0x03, 192 },
        { 2, 0x02, 193 },
        { 9, 0x02, 193 },
        { 23, 0x02, 193 },
        { 40, 0x03, 193 },
        { 2, 0x02, 200 },
        { 9, 0x02, 200 },
        { 23, 0x02, 200 },
        { 40, 0x03, 200 },
        { 2, 0x02, 201 },
        { 9, 0x02, 201 },
        { 23, 0x02, 201 },
        { 40, 0x03, 201 },
    },
    /* 194 */
    {
        { 3, 0x02, 192 },
        { 6, 0x02, 192 },
        { 10, 0x02, 192 },
        { 15, 0x02, 192 },
        { 24, 0x02, 192 },
        { 31, 0x02, 192 },
        { 41, 0x02, 192 },
        { 56, 0x03, 192 },
        { 3, 0x02, 193 },
        { 6, 0x02, 193 },
        { 10, 0x02, 193 },
        { 15, 0x02, 193 },
        { 24, 0x02, 193 },
        { 31, 0x02, 193 },
        { 41, 0x02, 193 },
        { 56, 0x03, 193 },
    },
    /* 195 */
    {
        { 3, 0x02, 200 },
        { 6, 0x02, 200 },
        { 10, 0x02, 200 },
        { 15, 0x02, 200 },
        { 24, 0x02, 200 },
        { 31, 0x02, 200 },
        { 41, 0x02, 200 },
        { 56, 0x03, 200 },
        { 3, 0x02, 201 },
        { 6, 0x02, 201 },
        { 10, 0x02, 201 },
        { 15, 0x02, 201 },
        { 24, 0x02, 201 },
        { 31, 0x02, 201 },
        { 41, 0x02, 201 },
        { 56, 0x03, 201 },
    },
    /* 196 */
    {
        { 2, 0x02, 202 },
        { 9, 0x02, 202 },
        { 23, 0x02, 202 },
        { 40, 0x03, 202 },
        { 2, 0x02, 205 },
        { 9, 0x02, 205 },
        { 23, 0x02, 205 },
        { 40, 0x03, 205 },
        { 2, 0x02, 210 },
        { 9, 0x02, 210 },
        { 23, 0x02, 210 },
        { 40, 0x03, 210 },
        { 2, 0x02, 213 },
        { 9, 0x02, 213 },
        { 23, 0x02, 213 },
        { 40, 0x03, 213 },
    },
    /* 197 */
    {
        { 3, 0x02, 202 },
        { 6, 0x02, 202 },
        { 10, 0x02, 202 },
        { 15, 0x02, 202 },
        { 24, 0x02, 202 },
        { 31, 0x02, 202 },
        { 41, 0x02, 202 },
        { 56, 0x03, 202 },
        { 3, 0x02, 205 },
        { 6, 0x02, 205 },
        { 10, 0x02, 205 },
        { 15, 0x02, 205 },
        { 24, 0x02, 205 },
        { 31, 0x02, 205 },
        { 41, 0x02, 205 },
        { 56, 0x03, 205 },
    },
    /* 198 */
    {
        { 3, 0x02, 210 },
        { 6, 0x02, 210 },
        { 10, 0x02, 210 },
        { 15, 0x02, 210 },
        { 24, 0x02, 210 },
        { 31, 0x02, 210 },
        { 41, 0x02, 210 },
        { 56, 0x03, 210 },
        { 3, 0x02, 213 },
        { 6, 0x02, 213 },
        { 10, 0x02, 213 },
        { 15, 0x02, 213 },
        { 24, 0x02, 213 },
        { 31, 0x02, 213 },
        { 41, 0x02, 213 },
        { 56, 0x03, 213 },
    },
    /* 199 */
    {
        { 1, 0x02, 218 },
        { 22, 0x03, 218 },
        { 1, 0x02, 219 },
        { 22, 0x03, 219 },
        { 1, 0x02, 238 },
        { 22, 0x03, 238 },
        { 1, 0x02, 240 },
        { 22, 0x03, 240 },
        { 1, 0x02, 242 },
        { 22, 0x03, 242 },
        { 1, 0x02, 243 },
        { 22, 0x03, 243 },
        { 1, 0x02, 255 },
        { 22, 0x03, 255 },
        { 0, 0x03, 203 },
        { 0, 0x03, 204 },
    },
    /* 200 */
    {
        { 2, 0x02, 218 },
        { 9, 0x02, 218 },
        { 23, 0x02, 218 },
        { 40, 0x03, 218 },
        { 2, 0x02, 219 },
        { 9, 0x02, 219 },
        { 23, 0x02, 219 },
        { 40, 0x03, 219 },
        { 2, 0x02, 238 },
        { 9, 0x02, 238 },
        { 23, 0x02, 238 },
        { 40, 0x03, 238 },
        { 2, 0x02, 240 },
        { 9, 0x02, 240 },
        { 23, 0x02, 240 },
        { 40, 0x03, 240 },
    },
    /* 201 */
    {
        { 3, 0x02, 218 },
        { 6, 0x02, 218 },
        { 10, 0x02, 218 },
        { 15, 0x02, 218 },
        { 24, 0x02, 218 },
        { 31, 0x02, 218 },
        { 41, 0x02, 218 },
        { 56, 0x03, 218 },
        { 3, 0x02, 219 },
        { 6, 0x02, 219 },
        { 10, 0x02, 219 },
        { 15, 0x02, 219 },
        { 24, 0x02, 219 },
        { 31, 0x02, 219 },
        { 41, 0x02, 219 },
        { 56, 0x03, 219 },
    },
    /* 202 */
    {
        { 3, 0x02, 238 },
        { 6, 0x02, 238 },
        { 10, 0x02, 238 },
        { 15, 0x02, 238 },
        { 24, 0x02, 238 },
        { 31, 0x02, 238 },
        { 41, 0x02, 238 },
        { 56, 0x03, 238 },
        { 3, 0x02, 240 },
        { 6, 0x02, 240 },
        { 10, 0x02, 240 },
        { 15, 0x02, 240 },
        { 24, 0x02, 240 },
        { 31, 0x02, 240 },
        { 41, 0x02, 240 },
        { 56, 0x03, 240 },
    },
    /* 203 */
    {
        { 2, 0x02, 242 },
        { 9, 0x02, 242 },
        { 23, 0x02, 242 },
        { 40, 0x03, 242 },
        { 2, 0x02, 243 },
        { 9, 0x02, 243 },
        { 23, 0x02, 243 },
        { 40, 0x03, 243 },
        { 2, 0x02, 255 },
        { 9, 0x02, 255 },
        { 23, 0x02, 255 },
        { 40, 0x03, 255 },
        { 1, 0x02, 203 },
        { 22, 0x03, 203 },
        { 1, 0x02, 204 },
        { 22, 0x03, 204 },
    },
    /* 204 */
    {
        { 3, 0x02, 242 },
        { 6, 0x02, 242 },
        { 10, 0x02, 242 },
        { 15, 0x02, 242 },
        { 24, 0x02, 242 },
        { 31, 0x02, 242 },
        { 41, 0x02, 242 },
        { 56, 0x03, 242 },
        { 3, 0x02, 243 },
        { 6, 0x02, 243 },
        { 10, 0x02, 243 },
        { 15, 0x02, 243 },
        { 24, 0x02, 243 },
        { 31, 0x02, 243 },
        { 41, 0x02, 243 },
        { 56, 0x03, 243 },
    },
    /* 205 */
    {
        { 3, 0x02, 255 },
        { 6, 0x02, 255 },
        { 10, 0x02, 255 },
        { 15, 0x02, 255 },
        { 24, 0x02, 255 },
        { 31, 0x02, 255 },
        { 41, 0x02, 255 },
        { 56, 0x03, 255 },
        { 2, 0x02, 203 },
        { 9, 0x02, 203 },
        { 23, 0x02, 203 },
        { 40, 0x03, 203 },
        { 2, 0x02, 204 },
        { 9, 0x02, 204 },
        { 23, 0x02, 204 },
        { 40, 0x03, 204 },
    },
    /* 206 */
    {
        { 3, 0x02, 203 },
        { 6, 0x02, 203 },
        { 10, 0x02, 203 },
        { 15, 0x02, 203 },
        { 24, 0x02, 203 },
        { 31, 0x02, 203 },
        { 41, 0x02, 203 },
        { 56, 0x03, 203 },
        { 3, 0x02, 204 },
        { 6, 0x02, 204 },
        { 10, 0x02, 204 },
        { 15, 0x02, 204 },
        { 24, 0x02, 204 },
        { 31, 0x02, 204 },
        { 41, 0x02, 204 },
        { 56, 0x03, 204 },
    },
    /* 207 */
    {
        { 211, 0x00, 0 },
        { 212, 0x00, 0 },
        { 214, 0x00, 0 },
        { 215, 0x00, 0 },
        { 218, 0x00, 0 },
        { 219, 0x00, 0 },
        { 221, 0x00, 0 },
        { 222, 0x00, 0 },
        { 226, 0x00, 0 },
        { 228, 0x00, 0 },
        { 232, 0x00, 0 },
        { 235, 0x00, 0 },
        { 240, 0x00, 0 },
        { 243, 0x00, 0 },
        { 247, 0x00, 0 },
        { 250, 0x00, 0 },
    },
    /* 208 */
    {
        { 0, 0x03, 211 },
        { 0, 0x03, 212 },
        { 0, 0x03, 214 },
        { 0, 0x03, 221 },
        { 0, 0x03, 222 },
        { 0, 0x03, 223 },
        { 0, 0x03, 241 },
        { 0, 0x03, 244 },
        { 0, 0x03, 245 },
        { 0, 0x03, 246 },
        { 0, 0x03, 247 },
        { 0, 0x03, 248 },
        { 0, 0x03, 250 },
        { 0, 0x03, 251 },
        { 0, 0x03, 252 },
        { 0, 0x03, 253 },
    },
    /* 209 */
    {
        { 1, 0x02, 211 },
        { 22, 0x03, 211 },
        { 1, 0x02, 212 },
        { 22, 0x03, 212 },
        { 1, 0x02, 214 },
        { 22, 0x03, 214 },
        { 1, 0x02, 221 },
        { 22, 0x03, 221 },
        { 1, 0x02, 222 },
        { 22, 0x03, 222 },
        { 1, 0x02, 223 },
        { 22, 0x03, 223 },
        { 1, 0x02, 241 },
        { 22, 0x03, 241 },
        { 1, 0x02, 244 },
        { 22, 0x03, 244 },
    },
    /* 210 */
    {
        { 2, 0x02, 211 },
        { 9, 0x02, 211 },
        { 23, 0x02, 211 },
        { 40, 0x03, 211 },
        { 2, 0x02, 212 },
        { 9, 0x02, 212 },
        { 23, 0x02, 212 },
        { 40, 0x03, 212 },
        { 2, 0x02, 214 },
        { 9, 0x02, 214 },
        { 23, 0x02, 214 },
        { 40, 0x03, 214 },
        { 2, 0x02, 221 },
        { 9, 0x02, 221 },
        { 23, 0x02, 221 },
        { 40, 0x03, 221 },
    },
    /* 211 */
    {
        { 3, 0x02, 211 },
        { 6, 0x02, 211 },
        { 10, 0x02, 211 },
        { 15, 0x02, 211 },
        { 24, 0x02, 211 },
        { 31, 0x02, 211 },
        { 41, 0x02, 211 },
        { 56, 0x03, 211 },
        { 3, 0x02, 212 },
        { 6, 0x02, 212 },
        { 10, 0x02, 212 },
        { 15, 0x02, 212 },
        { 24, 0x02, 212 },
        { 31, 0x02, 212 },
        { 41, 0x02, 212 },
        { 56, 0x03, 212 },
    },
    /* 212 */
    {
        { 3, 0x02, 214 },
        { 6, 0x02, 214 },
        { 10, 0x02, 214 },
        { 15, 0x02, 214 },
        { 24, 0x02, 214 },
        { 31, 0x02, 214 },
        { 41, 0x02, 214 },
        { 56, 0x03, 214 },
        { 3, 0x02, 221 },
        { 6, 0x02, 221 },
        { 10, 0x02, 221 },
        { 15, 0x02, 221 },
        { 24, 0x02, 221 },
        { 31, 0x02, 221 },
        { 41, 0x02, 221 },
        { 56, 0x03, 221 },
    },
    /* 213 */
    {
        { 2, 0x02, 222 },
        { 9, 0x02, 222 },
        { 23, 0x02, 222 },
        { 40, 0x03, 222 },
        { 2, 0x02, 223 },
        { 9, 0x02, 223 },
        { 23, 0x02, 223 },
        { 40, 0x03, 223 },
        { 2, 0x02, 241 },
        { 9, 0x02, 241 },
        { 23, 0x02, 241 },
        { 40, 0x03, 241 },
        { 2, 0x02, 244 },
        { 9, 0x02, 244 },
        { 23, 0x02, 244 },
        { 40, 0x03, 244 },
    },
    /* 214 */
    {
        { 3, 0x02, 222 },
        { 6, 0x02, 222 },
        { 10, 0x02, 222 },
        { 15, 0x02, 222 },
        { 24, 0x02, 222 },
        { 31, 0x02, 222 },
        { 41, 0x02, 222 },
        { 56, 0x03, 222 },
        { 3, 0x02, 223 },
        { 6, 0x02, 223 },
        { 10, 0x02, 223 },
        { 15, 0x02, 223 },
        { 24, 0x02, 223 },
        { 31, 0x02, 223 },
        { 41, 0x02, 223 },
        { 56, 0x03, 223 },
    },
    /* 215 */
    {
        { 3, 0x02, 241 },
        { 6, 0x02, 241 },
        { 10, 0x02, 241 },
        { 15, 0x02, 241 },
        { 24, 0x02, 241 },
        { 31, 0x02, 241 },
        { 41, 0x02, 241 },
        { 56, 0x03, 241 },
        { 3, 0x02, 244 },
        { 6, 0x02, 244 },
        { 10, 0x02, 244 },
        { 15, 0x02, 244 },
        { 24, 0x02, 244 },
        { 31, 0x02, 244 },
        { 41, 0x02, 244 },
        { 56, 0x03, 244 },
    },
    /* 216 */
    {
        { 1, 0x02, 245 },
        { 22, 0x03, 245 },
        { 1, 0x02, 246 },
        { 22, 0x03, 246 },
        { 1, 0x02, 247 },
        { 22, 0x03, 247 },
        { 1, 0x02, 248 },
        { 22, 0x03, 248 },
        { 1, 0x02, 250 },
        { 22, 0x03, 250 },
        { 1, 0x02, 251 },
        { 22, 0x03, 251 },
        { 1, 0x02, 252 },
        { 22, 0x03, 252 },
        { 1, 0x02, 253 },
        { 22, 0x03, 253 },
    },
    /* 217 */
    {
        { 2, 0x02, 245 },
        { 9, 0x02, 245 },
        { 23, 0x02, 245 },
        { 40, 0x03, 245 },
        { 2, 0x02, 246 },
        { 9, 0x02, 246 },
        { 23, 0x02, 246 },
        { 40, 0x03, 246 },
        { 2, 0x02, 247 },
        { 9, 0x02, 247 },
        { 23, 0x02, 247 },
        { 40, 0x03, 247 },
        { 2, 0x02, 248 },
        { 9, 0x02, 248 },
        { 23, 0x02, 248 },
        { 40, 0x03, 248 },
    },
    /* 218 */
    {
        { 3, 0x02, 245 },
        { 6, 0x02, 245 },
        { 10, 0x02, 245 },
        { 15, 0x02, 245 },
        { 24, 0x02, 245 },
        { 31, 0x02, 245 },
        { 41, 0x02, 245 },
        { 56, 0x03, 245 },
        { 3, 0x02, 246 },
        { 6, 0x02, 246 },
        { 10, 0x02, 246 },
        { 15, 0x02, 246 },
        { 24, 0x02, 246 },
        { 31, 0x02, 246 },
        { 41, 0x02, 246 },
        { 56, 0x03, 246 },
    },
    /* 219 */
    {
        { 3, 0x02, 247 },
        { 6, 0x02, 247 },
        { 10, 0x02, 247 },
        { 15, 0x02, 247 },
        { 24, 0x02, 247 },
        { 31, 0x02, 247 },
        { 41, 0x02, 247 },
        { 56, 0x03, 247 },
        { 3, 0x02, 248 },
        { 6, 0x02, 248 },
        { 10, 0x02, 248 },
        { 15, 0x02, 248 },
        { 24, 0x02, 248 },
        { 31, 0x02, 248 },
        { 41, 0x02, 248 },
        { 56, 0x03, 248 },
    },
    /* 220 */
    {
        { 2, 0x02, 250 },
        { 9, 0x02, 250 },
        { 23, 0x02, 250 },
        { 40, 0x03, 250 },
        { 2, 0x02, 251 },
        { 9, 0x02, 251 },
        { 23, 0x02, 251 },
        { 40, 0x03, 251 },
        { 2, 0x02, 252 },
        { 9, 0x02, 252 },
        { 23, 0x02, 252 },
        { 40, 0x03, 252 },
        { 2, 0x02, 253 },
        { 9, 0x02, 253 },
        { 23, 0x02, 253 },
        { 40, 0x03, 253 },
    },
    /* 221 */
    {
        { 3, 0x02, 250 },
        { 6, 0x02, 250 },
        { 10, 0x02, 250 },
        { 15, 0x02, 250 },
        { 24, 0x02, 250 },
        { 31, 0x02, 250 },
        { 41, 0x02, 250 },
        { 56, 0x03, 250 },
        { 3, 0x02, 251 },
        { 6, 0x02, 251 },
        { 10, 0x02, 251 },
        { 15, 0x02, 251 },
        { 24, 0x02, 251 },
        { 31, 0x02, 251 },
        { 41, 0x02, 251 },
        { 56, 0x03, 251 },
    },
    /* 222 */
    {
        { 3, 0x02, 252 },
        { 6, 0x02, 252 },
        { 10, 0x02, 252 },
        { 15, 0x02, 252 },
        { 24, 0x02, 252 },
        { 31, 0x02, 252 },
        { 41, 0x02, 252 },
        { 56, 0x03, 252 },
        { 3, 0x02, 253 },
        { 6, 0x02, 253 },
        { 10, 0x02, 253 },
        { 15, 0x02, 253 },
        { 24, 0x02, 253 },
        { 31, 0x02, 253 },
        { 41, 0x02, 253 },
        { 56, 0x03, 253 },
    },
    /* 223 */
    {
        { 0, 0x03, 254 },
        { 227, 0x00, 0 },
        { 229, 0x00, 0 },
        { 230, 0x00, 0 },
        { 233, 0x00, 0 },
        { 234, 0x00, 0 },
        { 236, 0x00, 0 },
        { 237, 0x00, 0 },
        { 241, 0x00, 0 },
        { 242, 0x00, 0 },
        { 244, 0x00, 0 },
        { 245, 0x00, 0 },
        { 248, 0x00, 0 },
        { 249, 0x00, 0 },
        { 251, 0x00, 0 },
        { 252, 0x00, 0 },
    },
    /* 224 */
    {
        { 1, 0x02, 254 },
        { 22, 0x03, 254 },
        { 0, 0x03, 2 },
        { 0, 0x03, 3 },
        { 0, 0x03, 4 },
        { 0, 0x03, 5 },
        { 0, 0x03, 6 },
        { 0, 0x03, 7 },
        { 0, 0x03, 8 },
        { 0, 0x03, 11 },
        { 0, 0x03, 12 },
        { 0, 0x03, 14 },
        { 0, 0x03, 15 },
        { 0, 0x03, 16 },
        { 0, 0x03, 17 },
        { 0, 0x03, 18 },
    },
    /* 225 */
    {
        { 2, 0x02, 254 },
        { 9, 0x02, 254 },
        { 23, 0x02, 254 },
        { 40, 0x03, 254 },
        { 1, 0x02, 2 },
        { 22, 0x03, 2 },
        { 1, 0x02, 3 },
        { 22, 0x03, 3 },
        { 1, 0x02, 4 },
        { 22, 0x03, 4 },
        { 1, 0x02, 5 },
        { 22, 0x03, 5 },
        { 1, 0x02, 6 },
        { 22, 0x03, 6 },
        { 1, 0x02, 7 },
        { 22, 0x03, 7 },
    },
    /* 226 */
    {
        { 3, 0x02, 254 },
        { 6, 0x02, 254 },
        { 10, 0x02, 254 },
        { 15, 0x02, 254 },
        { 24, 0x02, 254 },
        { 31, 0x02, 254 },
        { 41, 0x02, 254 },
        { 56, 0x03, 254 },
        { 2, 0x02, 2 },
        { 9, 0x02, 2 },
        { 23, 0x02, 2 },
        { 40, 0x03, 2 },
        { 2, 0x02, 3 },
        { 9, 0x02, 3 },
        { 23, 0x02, 3 },
        { 40, 0x03, 3 },
    },
    /* 227 */
    {
        { 3, 0x02, 2 },
        { 6, 0x02, 2 },
        { 10, 0x02, 2 },
        { 15, 0x02, 2 },
        { 24, 0x02, 2 },
        { 31, 0x02, 2 },
        { 41, 0x02, 2 },
        { 56, 0x03, 2 },
        { 3, 0x02, 3 },
        { 6, 0x02, 3 },
        { 10, 0x02, 3 },
        { 15, 0x02, 3 },
        { 24, 0x02, 3 },
        { 31, 0x02, 3 },
        { 41, 0x02, 3 },
        { 56, 0x03, 3 },
    },
    /* 228 */
    {
        { 2, 0x02, 4 },
        { 9, 0x02, 4 },
        { 23, 0x02, 4 },
        { 40, 0x03, 4 },
        { 2, 0x02, 5 },
        { 9, 0x02, 5 },
        { 23, 0x02, 5 },
        { 40, 0x03, 5 },
        { 2, 0x02, 6 },
        { 9, 0x02, 6 },
        { 23, 0x02, 6 },
        { 40, 0x03, 6 },
        { 2, 0x02, 7 },
        { 9, 0x02, 7 },
        { 23, 0x02, 7 },
        { 40, 0x03, 7 },
    },
    /* 229 */
    {
        { 3, 0x02, 4 },
        { 6, 0x02, 4 },
        { 10, 0x02, 4 },
        { 15, 0x02, 4 },
        { 24, 0x02, 4 },
        { 31, 0x02, 4 },
        { 41, 0x02, 4 },
        { 56, 0x03, 4 },
        { 3, 0x02, 5 },
        { 6, 0x02, 5 },
        { 10, 0x02, 5 },
        { 15, 0x02, 5 },
        { 24, 0x02, 5 },
        { 31, 0x02, 5 },
        { 41, 0x02, 5 },
        { 56, 0x03, 5 },
    },
    /* 230 */
    {
        { 3, 0x02, 6 },
        { 6, 0x02, 6 },
        { 10, 0x02, 6 },
        { 15, 0x02, 6 },
        { 24, 0x02, 6 },
        { 31, 0x02, 6 },
        { 41, 0x02, 6 },
        { 56, 0x03, 6 },
        { 3, 0x02, 7 },
        { 6, 0x02, 7 },
        { 10, 0x02, 7 },
        { 15, 0x02, 7 },
        { 24, 0x02, 7 },
        { 31, 0x02, 7 },
        { 41, 0x02, 7 },
        { 56, 0x03, 7 },
    },
    /* 231 */
    {
        { 1, 0x02, 8 },
        { 22, 0x03, 8 },
        { 1, 0x02, 11 },
        { 22, 0x03, 11 },
        { 1, 0x02, 12 },
        { 22, 0x03, 12 },
        { 1, 0x02, 14 },
        { 22, 0x03, 14 },
        { 1, 0x02, 15 },
        { 22, 0x03, 15 },
        { 1, 0x02, 16 },
        { 22, 0x03, 16 },
        { 1, 0x02, 17 },
        { 22, 0x03, 17 },
        { 1, 0x02, 18 },
        { 22, 0x03, 18 },
    },
    /* 232 */
    {
        { 2, 0x02, 8 },
        { 9, 0x02, 8 },
        { 23, 0x02, 8 },
        { 40, 0x03, 8 },
        { 2, 0x02, 11 },
        { 9, 0x02, 11 },
        { 23, 0x02, 11 },
        { 40, 0x03, 11 },
        { 2, 0x02, 12 },
        { 9, 0x02, 12 },
        { 23, 0x02, 12 },
        { 40, 0x03, 12 },
        { 2, 0x02, 14 },
        { 9, 0x02, 14 },
        { 23, 0x02, 14 },
        { 40, 0x03, 14 },
    },
    /* 233 */
    {
        { 3, 0x02, 8 },
        { 6, 0x02, 8 },
        { 10, 0x02, 8 },
        { 15, 0x02, 8 },
        { 24, 0x02, 8 },
        { 31, 0x02, 8 },
        { 41, 0x02, 8 },
        { 56, 0x03, 8 },
        { 3, 0x02, 11 },
        { 6, 0x02, 11 },
        { 10, 0x02, 11 },
        { 15, 0x02, 11 },
        { 24, 0x02, 11 },
        { 31, 0x02, 11 },
        { 41, 0x02, 11 },
        { 56, 0x03, 11 },
    },
    /* 234 */
    {
        { 3, 0x02, 12 },
        { 6, 0x02, 12 },
        { 10, 0x02, 12 },
        { 15, 0x02, 12 },
        { 24, 0x02, 12 },
        { 31, 0x02, 12 },
        { 41, 0x02, 12 },
        { 56, 0x03, 12 },
        { 3, 0x02, 14 },
        { 6, 0x02, 14 },
        { 10, 0x02, 14 },
        { 15, 0x02, 14 },
        { 24, 0x02, 14 },
        { 31, 0x02, 14 },
        { 41, 0x02, 14 },
        { 56, 0x03, 14 },
    },
    /* 235 */
    {
        { 2, 0x02, 15 },
        { 9, 0x02, 15 },
        { 23, 0x02, 15 },
        { 40, 0x03, 15 },
        { 2, 0x02, 16 },
        { 9, 0x02, 16 },
        { 23, 0x02, 16 },
        { 40, 0x03, 16 },
        { 2, 0x02, 17 },
        { 9, 0x02, 17 },
        { 23, 0x02, 17 },
        { 40, 0x03, 17 },
        { 2, 0x02, 18 },
        { 9, 0x02, 18 },
        { 23, 0x02, 18 },
        { 40, 0x03, 18 },
    },
    /* 236 */
    {
        { 3, 0x02, 15 },
        { 6, 0x02, 15 },
        { 10, 0x02, 15 },
        { 15, 0x02, 15 },
        { 24, 0x02, 15 },
        { 31, 0x02, 15 },
        { 41, 0x02, 15 },
        { 56, 0x03, 15 },
        { 3, 0x02, 16 },
        { 6, 0x02, 16 },
        { 10, 0x02, 16 },
        { 15, 0x02, 16 },
        { 24, 0x02, 16 },
        { 31, 0x02, 16 },
        { 41, 0x02, 16 },
        { 56, 0x03, 16 },
    },
    /* 237 */
    {
        { 3, 0x02, 17 },
        { 6, 0x02, 17 },
        { 10, 0x02, 17 },
        { 15, 0x02, 17 },
        { 24, 0x02, 17 },
        { 31, 0x02, 17 },
        { 41, 0x02, 17 },
        { 56, 0x03, 17 },
        { 3, 0x02, 18 },
        { 6, 0x02, 18 },
        { 10, 0x02, 18 },
        { 15, 0x02, 18 },
        { 24, 0x02, 18 },
        { 31, 0x02, 18 },
        { 41, 0x02, 18 },
        { 56, 0x03, 18 },
    },
    /* 238 */
    {
        { 0, 0x03, 19 },
        { 0, 0x03, 20 },
        { 0, 0x03, 21 },
        { 0, 0x03, 23 },
        { 0, 0x03, 24 },
        { 0, 0x03, 25 },
        { 0, 0x03, 26 },
        { 0, 0x03, 27 },
        { 0, 0x03, 28 },
        { 0, 0x03, 29 },
        { 0, 0x03, 30 },
        { 0, 0x03, 31 },
        { 0, 0x03, 127 },
        { 0, 0x03, 220 },
        { 0, 0x03, 249 },
        { 253, 0x00, 0 },
    },
    /* 239 */
    {
        { 1, 0x02, 19 },
        { 22, 0x03, 19 },
        { 1, 0x02, 20 },
        { 22, 0x03, 20 },
        { 1, 0x02, 21 },
        { 22, 0x03, 21 },
        { 1, 0x02, 23 },
        { 22, 0x03, 23 },
        { 1, 0x02, 24 },
        { 22, 0x03, 24 },
        { 1, 0x02, 25 },
        { 22, 0x03, 25 },
        { 1, 0x02, 26 },
        { 22, 0x03, 26 },
        { 1, 0x02, 27 },
        { 22, 0x03, 27 },
    },
    /* 240 */
    {
        { 2, 0x02, 19 },
        { 9, 0x02, 19 },
        { 23, 0x02, 19 },
        { 40, 0x03, 19 },
        { 2, 0x02, 20 },
        { 9, 0x02, 20 },
        { 23, 0x02, 20 },
        { 40, 0x03, 20 },
        { 2, 0x02, 21 },
        { 9, 0x02, 21 },
        { 23, 0x02, 21 },
        { 40, 0x03, 21 },
        { 2, 0x02, 23 },
        { 9, 0x02, 23 },
        { 23, 0x02, 23 },
        { 40, 0x03, 23 },
    },
    /* 241 */
    {
        { 3, 0x02, 19 },
        { 6, 0x02, 19 },
        { 10, 0x02, 19 },
        { 15, 0x02, 19 },
        { 24, 0x02, 19 },
        { 31, 0x02, 19 },
        { 41, 0x02, 19 },
        { 56, 0x03, 19 },
        { 3, 0x02, 20 },
        { 6, 0x02, 20 },
        { 10, 0x02, 20 },
        { 15, 0x02, 20 },
        { 24, 0x02, 20 },
        { 31, 0x02, 20 },
        { 41, 0x02, 20 },
        { 56, 0x03, 20 },
    },
    /* 242 */
    {
        { 3, 0x02, 21 },
        { 6, 0x02, 21 },
        { 10, 0x02, 21 },
        { 15, 0x02, 21 },
        { 24, 0x02, 21 },
        { 31, 0x02, 21 },
        { 41, 0x02, 21 },
        { 56, 0x03, 21 },
        { 3, 0x02, 23 },
        { 6, 0x02, 23 },
        { 10, 0x02, 23 },
        { 15, 0x02, 23 },
        { 24, 0x02, 23 },
        { 31, 0x02, 23 },
        { 41, 0x02, 23 },
        { 56, 0x03, 23 },
    },
    /* 243 */
    {
        { 2, 0x02, 24 },
        { 9, 0x02, 24 },
        { 23, 0x02, 24 },
        { 40, 0x03, 24 },
        { 2, 0x02, 25 },
        { 9, 0x02, 25 },
        { 23, 0x02, 25 },
        { 40, 0x03, 25 },
        { 2, 0x02, 26 },
        { 9, 0x02, 26 },
        { 23, 0x02, 26 },
        { 40, 0x03, 26 },
        { 2, 0x02, 27 },
        { 9, 0x02, 27 },
        { 23, 0x02, 27 },
        { 40, 0x03, 27 },
    },
    /* 244 */
    {
        { 3, 0x02, 24 },
        { 6, 0x02, 24 },
        { 10, 0x02, 24 },
        { 15, 0x02, 24 },
        { 24, 0x02, 24 },
        { 31, 0x02, 24 },
        { 41, 0x02, 24 },
        { 56, 0x03, 24 },
        { 3, 0x02, 25 },
        { 6, 0x02, 25 },
        { 10, 0x02, 25 },
        { 15, 0x02, 25 },
        { 24, 0x02, 25 },
        { 31, 0x02, 25 },
        { 41, 0x02, 25 },
        { 56, 0x03, 25 },
    },
    /* 245 */
    {
        { 3, 0x02, 26 },
        { 6, 0x02, 26 },
        { 10, 0x02, 26 },
        { 15, 0x02, 26 },
        { 24, 0x02, 26 },
        { 31, 0x02, 26 },
        { 41, 0x02, 26 },
        { 56, 0x03, 26 },
        { 3, 0x02, 27 },
        { 6, 0x02, 27 },
        { 10, 0x02, 27 },
        { 15, 0x02, 27 },
        { 24, 0x02, 27 },
        { 31, 0x02, 27 },
        { 41, 0x02, 27 },
        { 56, 0x03, 27 },
    },
    /* 246 */
    {
        { 1, 0x02, 28 },
        { 22, 0x03, 28 },
        { 1, 0x02, 29 },
        { 22, 0x03, 29 },
        { 1, 0x02, 30 },
        { 22, 0x03, 30 },
        { 1, 0x02, 31 },
        { 22, 0x03, 31 },
        { 1, 0x02, 127 },
        { 22, 0x03, 127 },
        { 1, 0x02, 220 },
        { 22, 0x03, 220 },
        { 1, 0x02, 249 },
        { 22, 0x03, 249 },
        { 254, 0x00, 0 },
        { 255, 0x00, 0 },
    },
    /* 247 */
    {
        { 2, 0x02, 28 },
        { 9, 0x02, 28 },
        { 23, 0x02, 28 },
        { 40, 0x03, 28 },
        { 2, 0x02, 29 },
        { 9, 0x02, 29 },
        { 23, 0x02, 29 },
        { 40, 0x03, 29 },
        { 2, 0x02, 30 },
        { 9, 0x02, 30 },
        { 23, 0x02, 30 },
        { 40, 0x03, 30 },
        { 2, 0x02, 31 },
        { 9, 0x02, 31 },
        { 23, 0x02, 31 },
        { 40, 0x03, 31 },
    },
    /* 248 */
    {
        { 3, 0x02, 28 },
        { 6, 0x02, 28 },
        { 10, 0x02, 28 },
        { 15, 0x02, 28 },
        { 24, 0x02, 28 },
        { 31, 0x02, 28 },
        { 41, 0x02, 28 },
        { 56, 0x03, 28 },
        { 3, 0x02, 29 },
        { 6, 0x02, 29 },
        { 10, 0x02, 29 },
        { 15, 0x02, 29 },
        { 24, 0x02, 29 },
        { 31, 0x02, 29 },
        { 41, 0x02, 29 },
        { 56, 0x03, 29 },
    },
    /* 249 */
    {
        { 3, 0x02, 30 },
        { 6, 0x02, 30 },
        { 10, 0x02, 30 },
        { 15, 0x02, 30 },
        { 24, 0x02, 30 },
        { 31, 0x02, 30 },
        { 41, 0x02, 30 },
        { 56, 0x03, 30 },
        { 3, 0x02, 31 },
        { 6, 0x02, 31 },
        { 10, 0x02, 31 },
        { 15, 0x02, 31 },
        { 24, 0x02, 31 },
        { 31, 0x02, 31 },
        { 41, 0x02, 31 },
        { 56, 0x03, 31 },
    },
    /* 250 */
    {
        { 2, 0x02, 127 },
        { 9, 0x02, 127 },
        { 23, 0x02, 127 },
        { 40, 0x03, 127 },
        { 2, 0x02, 220 },
        { 9, 0x02, 220 },
        { 23, 0x02, 220 },
        { 40, 0x03, 220 },
        { 2, 0x02, 249 },
        { 9, 0x02, 249 },
        { 23, 0x02, 249 },
        { 40, 0x03, 249 },
        { 0, 0x03, 10 },
        { 0, 0x03, 13 },
        { 0, 0x03, 22 },
        { 0, 0x04, 0 },
    },
    /* 251 */
    {
        { 3, 0x02, 127 },
        { 6, 0x02, 127 },
        { 10, 0x02, 127 },
        { 15, 0x02, 127 },
        { 24, 0x02, 127 },
        { 31, 0x02, 127 },
        { 41, 0x02, 127 },
        { 56, 0x03, 127 },
        { 3, 0x02, 220 },
        { 6, 0x02, 220 },
        { 10, 0x02, 220 },
        { 15, 0x02, 220 },
        { 24, 0x02, 220 },
        { 31, 0x02, 220 },
        { 41, 0x02, 220 },
        { 56, 0x03, 220 },
    },
    /* 252 */
    {
        { 3, 0x02, 249 },
        { 6, 0x02, 249 },
        { 10, 0x02, 249 },
        { 15, 0x02, 249 },
        { 24, 0x02, 249 },
        { 31, 0x02, 249 },
        { 41, 0x02, 249 },
        { 56, 0x03, 249 },
        { 1, 0x02, 10 },
        { 22, 0x03, 10 },
        { 1, 0x02, 13 },
        { 22, 0x03, 13 },
        { 1, 0x02, 22 },
        { 22, 0x03, 22 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
    },
    /* 253 */
    {
        { 2, 0x02, 10 },
        { 9, 0x02, 10 },
        { 23, 0x02, 10 },
        { 40, 0x03, 10 },
        { 2, 0x02, 13 },
        { 9, 0x02, 13 },
        { 23, 0x02, 13 },
        { 40, 0x03, 13 },
        { 2, 0x02, 22 },
        { 9, 0x02, 22 },
        { 23, 0x02, 22 },
        { 40, 0x03, 22 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
    },
    /* 254 */
    {
        { 3, 0x02, 10 },
        { 6, 0x02, 10 },
        { 10, 0x02, 10 },
        { 15, 0x02, 10 },
        { 24, 0x02, 10 },
        { 31, 0x02, 10 },
        { 41, 0x02, 10 },
        { 56, 0x03, 10 },
        { 3, 0x02, 13 },
        { 6, 0x02, 13 },
        { 10, 0x02, 13 },
        { 15, 0x02, 13 },
        { 24, 0x02, 13 },
        { 31, 0x02, 13 },
        { 41, 0x02, 13 },
        { 56, 0x03, 13 },
    },
    /* 255 */
    {
        { 3, 0x02, 22 },
        { 6, 0x02, 22 },
        { 10, 0x02, 22 },
        { 15, 0x02, 22 },
        { 24, 0x02, 22 },
        { 31, 0x02, 22 },
        { 41, 0x02, 22 },
        { 56, 0x03, 22 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
        { 0, 0x04, 0 },
    },
};

#define lshpack_arr_init(a) do {                                        \
    memset((a), 0, sizeof(*(a)));                                       \
} while (0)

#define lshpack_arr_cleanup(a) do {                                     \
    free((a)->els);                                                     \
    memset((a), 0, sizeof(*(a)));                                       \
} while (0)

#define lshpack_arr_get(a, i) (                                         \
    assert((i) < (a)->nelem),                                           \
    (a)->els[(a)->off + (i)]                                            \
)

#define lshpack_arr_shift(a) (                                          \
    assert((a)->nelem > 0),                                             \
    (a)->nelem -= 1,                                                    \
    (a)->els[(a)->off++]                                                \
)

#define lshpack_arr_pop(a) (                                            \
    assert((a)->nelem > 0),                                             \
    (a)->nelem -= 1,                                                    \
    (a)->els[(a)->off + (a)->nelem]                                     \
)

#define lshpack_arr_count(a) (+(a)->nelem)

static int
lshpack_arr_push (struct lshpack_arr *arr, uintptr_t val)
{
    uintptr_t *new_els;
    unsigned n;

    if (arr->off + arr->nelem < arr->nalloc)
    {
        arr->els[arr->off + arr->nelem] = val;
        ++arr->nelem;
        return 0;
    }

    if (arr->off > arr->nalloc / 2)
    {
        memmove(arr->els, arr->els + arr->off,
                                        sizeof(arr->els[0]) * arr->nelem);
        arr->off = 0;
        arr->els[arr->nelem] = val;
        ++arr->nelem;
        return 0;
    }

    if (arr->nalloc)
        n = arr->nalloc * 2;
    else
        n = 64;
    new_els = malloc(n * sizeof(arr->els[0]));
    if (!new_els)
        return -1;
    memcpy(new_els, arr->els + arr->off, sizeof(arr->els[0]) * arr->nelem);
    free(arr->els);
    arr->off = 0;
    arr->els = new_els;
    arr->nalloc = n;
    arr->els[arr->off + arr->nelem] = val;
    ++arr->nelem;
    return 0;
}

struct lshpack_double_enc_head
{
    struct lshpack_enc_head by_name;
    struct lshpack_enc_head by_nameval;
};

struct lshpack_enc_table_entry
{
    /* An entry always lives on all three lists */
    STAILQ_ENTRY(lshpack_enc_table_entry)
                                    ete_next_nameval,
                                    ete_next_name,
                                    ete_next_all;
    unsigned                        ete_id;
    unsigned                        ete_nameval_hash;
    unsigned                        ete_name_hash;
    lshpack_strlen_t                ete_name_len;
    lshpack_strlen_t                ete_val_len;
    char                            ete_buf[0];
};

#define ETE_NAME(ete) ((ete)->ete_buf)
#define ETE_VALUE(ete) (&(ete)->ete_buf[(ete)->ete_name_len])


#define N_BUCKETS(n_bits) (1U << (n_bits))
#define BUCKNO(n_bits, hash) ((hash) & (N_BUCKETS(n_bits) - 1))

int
lshpack_enc_init (struct lshpack_enc *enc)
{
    struct lshpack_double_enc_head *buckets;
    unsigned nbits = 2;
    unsigned i;

    buckets = malloc(sizeof(buckets[0]) * N_BUCKETS(nbits));
    if (!buckets)
        return -1;

    for (i = 0; i < N_BUCKETS(nbits); ++i)
    {
        STAILQ_INIT(&buckets[i].by_name);
        STAILQ_INIT(&buckets[i].by_nameval);
    }

    memset(enc, 0, sizeof(*enc));
    STAILQ_INIT(&enc->hpe_all_entries);
    enc->hpe_max_capacity = INITIAL_DYNAMIC_TABLE_SIZE;
    enc->hpe_buckets      = buckets;
    /* The initial value of the entry ID is completely arbitrary.  As long as
     * there are fewer than 2^32 dynamic table entries, the math to calculate
     * the entry ID works.  To prove to ourselves that the wraparound works
     * and to have the unit tests cover it, we initialize the next ID so that
     * it is just about to wrap around.
     */
    enc->hpe_next_id      = ~0 - 3;
    enc->hpe_nbits        = nbits;
    enc->hpe_nelem        = 0;
    return 0;
}


void
lshpack_enc_cleanup (struct lshpack_enc *enc)
{
    struct lshpack_enc_table_entry *entry, *next;
    for (entry = STAILQ_FIRST(&enc->hpe_all_entries); entry; entry = next)
    {
        next = STAILQ_NEXT(entry, ete_next_all);
        free(entry);
    }
    free(enc->hpe_buckets);
}


//not find return 0, otherwise return the index
#if !LS_HPACK_EMIT_TEST_CODE
static
#endif
       unsigned
lshpack_enc_get_stx_tab_id (const char *name, lshpack_strlen_t name_len,
                const char *val, lshpack_strlen_t val_len, int *val_matched)
{
    if (name_len < 3)
        return 0;

    *val_matched = 0;

    //check value first
    int i = -1;
    switch (*val)
    {
        case 'G':
            i = 1;
            break;
        case 'P':
            i = 2;
            break;
        case '/':
            if (val_len == 1)
                i = 3;
            else if (val_len == 11)
                i = 4;
            break;
        case 'h':
            if (val_len == 4)
                i = 5;
            else if (val_len == 5)
                i = 6;
            break;
        case '2':
            if (val_len == 3)
            {
                switch (*(val + 2))
                {
                    case '0':
                        i = 7;
                        break;
                    case '4':
                        i = 8;
                        break;
                    case '6':
                        i = 9;
                        break;
                    default:
                        break;
                }
            }
            break;
        case '3':
            i = 10;
            break;
        case '4':
            if (val_len == 3)
            {
                switch (*(val + 2))
                {
                    case '0':
                        i = 11;
                        break;
                    case '4':
                        i = 12;
                    default:
                        break;
                }
            }
            break;
        case '5':
            i = 13;
            break;
        case 'g':
            i = 15;
            break;
        default:
            break;
    }

    if (i > 0 && static_table[i].val_len == val_len
            && static_table[i].name_len == name_len
            && memcmp(val, static_table[i].val, val_len) == 0
            && memcmp(name, static_table[i].name, name_len) == 0)
    {
        *val_matched = 1;
        return i + 1;
    }

    //macth name only checking
    i = -1;
    switch (*name)
    {
        case ':':
            switch (*(name + 1))
            {
                case 'a':
                    i = 0;
                    break;
                case 'm':
                    i = 1;
                    break;
                case 'p':
                    i = 3;
                    break;
                case 's':
                    if (*(name + 2) == 'c') //:scheme
                        i = 5;
                    else
                        i = 7;
                    break;
                default:
                    break;
            }
            break;
        case 'a':
            switch (name_len)
            {
                case 3:
                    i = 20; //age
                    break;
                case 5:
                    i = 21; //allow
                    break;
                case 6:
                    i = 18; //accept
                    break;
                case 13:
                    if (*(name + 1) == 'u')
                        i = 22; //authorization
                    else
                        i = 17; //accept-ranges
                    break;
                case 14:
                    i  = 14; //accept-charset
                    break;
                case 15:
                    if (*(name + 7) == 'l')
                        i = 16; //accept-language,
                    else
                        i = 15;// accept-encoding
                    break;
                case 27:
                    i = 19;//access-control-allow-origin
                    break;
                default:
                    break;
            }
            break;
        case 'c':
            switch (name_len)
            {
                case 6:
                    i = 31; //cookie
                    break;
                case 12:
                    i = 30; //content-type
                    break;
                case 13:
                    if (*(name + 1) == 'a')
                        i = 23; //cache-control
                    else
                        i = 29; //content-range
                    break;
                case 14:
                    i = 27; //content-length
                    break;
                case 16:
                    switch (*(name + 9))
                    {
                        case 'n':
                            i = 25 ;//content-encoding
                            break;
                        case 'a':
                            i = 26; //content-language
                            break;
                        case 'o':
                            i = 28; //content-location
                        default:
                            break;
                    }
                    break;
                case 19:
                    i = 24; //content-disposition
                    break;
            }
            break;
        case 'd':
            i = 32 ;//date
            break;
        case 'e':
            switch (name_len)
            {
                case 4:
                    i = 33; //etag
                    break;
                case 6:
                    i = 34;
                    break;
                case 7:
                    i = 35;
                    break;
                default:
                    break;
            }
            break;
        case 'f':
            i = 36; //from
            break;
        case 'h':
            i = 37; //host
            break;
        case 'i':
            switch (name_len)
            {
                case 8:
                    if (*(name + 3) == 'm')
                        i = 38; //if-match
                    else
                        i = 41; //if-range
                    break;
                case 13:
                    i = 40; //if-none-match
                    break;
                case 17:
                    i = 39; //if-modified-since
                    break;
                case 19:
                    i = 42; //if-unmodified-since
                    break;
                default:
                    break;
            }
            break;
        case 'l':
            switch (name_len)
            {
                case 4:
                    i = 44; //link
                    break;
                case 8:
                    i = 45; //location
                    break;
                case 13:
                    i = 43; //last-modified
                    break;
                default:
                    break;
            }
            break;
        case 'm':
            i = 46; //max-forwards
            break;
        case 'p':
            if (name_len == 18)
                i = 47; //proxy-authenticate
            else
                i = 48; //proxy-authorization
            break;
        case 'r':
            if (name_len >= 5)
            {
                switch (*(name + 4))
                {
                    case 'e':
                        if (name_len == 5)
                            i = 49; //range
                        else
                            i = 51; //refresh
                        break;
                    case 'r':
                        i = 50; //referer
                        break;
                    case 'y':
                        i = 52; //retry-after
                        break;
                    default:
                        break;
                }
            }
            break;
        case 's':
            switch (name_len)
            {
                case 6:
                    i = 53; //server
                    break;
                case 10:
                    i = 54; //set-cookie
                    break;
                case 25:
                    i = 55; //strict-transport-security
                    break;
                default:
                    break;
            }
            break;
        case 't':
            i = 56;//transfer-encoding
            break;
        case 'u':
            i = 57; //user-agent
            break;
        case 'v':
            if (name_len == 4)
                i = 58;
            else
                i = 59;
            break;
        case 'w':
            i = 60;
            break;
        default:
            break;
    }

    if (i >= 0
            && static_table[i].name_len == name_len
            && memcmp(name, static_table[i].name, name_len) == 0)
        return i + 1;

    return 0;
}


/* Given a dynamic entry, return its table ID */
static unsigned
henc_calc_table_id (const struct lshpack_enc *enc,
                                    const struct lshpack_enc_table_entry *entry)
{
    return HPACK_STATIC_TABLE_SIZE
         + (enc->hpe_next_id - entry->ete_id)
    ;
}


static unsigned
henc_find_table_id (struct lshpack_enc *enc, const char *name,
        lshpack_strlen_t name_len, const char *value,
        lshpack_strlen_t value_len, int *val_matched)
{
    struct lshpack_enc_table_entry *entry;
    unsigned name_hash, nameval_hash, buckno, static_table_id;
    XXH32_state_t hash_state;

    /* First, look for a match in the static table: */
    static_table_id = lshpack_enc_get_stx_tab_id(name, name_len, value,
                                                    value_len, val_matched);
    if (static_table_id > 0 && *val_matched)
        return static_table_id;

    /* Search by name and value: */
    XXH32_reset(&hash_state, (uintptr_t) enc);
    XXH32_update(&hash_state, &name_len, sizeof(name_len));
    XXH32_update(&hash_state, name, name_len);
    name_hash = XXH32_digest(&hash_state);
    XXH32_update(&hash_state,  &value_len, sizeof(value_len));
    XXH32_update(&hash_state,  value, value_len);
    nameval_hash = XXH32_digest(&hash_state);
    buckno = BUCKNO(enc->hpe_nbits, nameval_hash);
    STAILQ_FOREACH(entry, &enc->hpe_buckets[buckno].by_nameval,
                                                        ete_next_nameval)
        if (nameval_hash == entry->ete_nameval_hash &&
            name_len == entry->ete_name_len &&
            value_len == entry->ete_val_len &&
            0 == memcmp(name, ETE_NAME(entry), name_len) &&
            0 == memcmp(value, ETE_VALUE(entry), value_len))
        {
            *val_matched = 1;
            return henc_calc_table_id(enc, entry);
        }

    /* Name/value match is not found, but if the caller found a matching
     * static table entry, no need to continue to search:
     */
    if (static_table_id > 0)
        return static_table_id;

    /* Search by name only: */
    buckno = BUCKNO(enc->hpe_nbits, name_hash);
    STAILQ_FOREACH(entry, &enc->hpe_buckets[buckno].by_name, ete_next_name)
        if (name_hash == entry->ete_name_hash &&
            name_len == entry->ete_name_len &&
            0 == memcmp(name, ETE_NAME(entry), name_len))
        {
            *val_matched = 0;
            return henc_calc_table_id(enc, entry);
        }

    return 0;
}


static unsigned char *
henc_enc_int (unsigned char *dst, unsigned char *const end, uint32_t value,
                                                        uint8_t prefix_bits)
{
    unsigned char *const dst_orig = dst;

    /* This function assumes that at least one byte is available */
    assert(dst < end);
    if (value < (uint32_t)(1 << prefix_bits) - 1)
        *dst++ |= value;
    else
    {
        *dst++ |= (1 << prefix_bits) - 1;
        value -= (1 << prefix_bits) - 1;
        while (value >= 128)
        {
            if (dst < end)
            {
                *dst++ = (0x80 | value);
                value >>= 7;
            }
            else
                return dst_orig;
        }
        if (dst < end)
            *dst++ = value;
        else
            return dst_orig;
    }
    return dst;
}


int
lshpack_enc_huff_encode (const unsigned char *src,
    const unsigned char *const src_end, unsigned char *const dst, int dst_len)
{
    const unsigned char *p_src = src;
    unsigned char *p_dst = dst;
    unsigned char *dst_end = p_dst + dst_len;
    uint64_t bits = 0;
    int bits_left = 40;
    struct encode_el cur_enc_code;

    assert(dst_len > 0);

    while (p_src != src_end)
    {
        cur_enc_code = encode_table[(int) *p_src++];
        assert(bits_left >= cur_enc_code.bits); //  (possible negative shift, undefined behavior)
        bits |= (uint64_t)cur_enc_code.code << (bits_left - cur_enc_code.bits);
        bits_left -= cur_enc_code.bits;
        while (bits_left <= 32)
        {
            *p_dst++ = bits >> 32;
            bits <<= 8;
            bits_left += 8;
            if (p_dst == dst_end)
                return -1;  //dst does not have enough space
        }
    }

    if (bits_left != 40)
    {
        assert(bits_left < 40 && bits_left > 0);
        bits |= ((uint64_t)1 << bits_left) - 1;
        *p_dst++ = bits >> 32;
    }

    return p_dst - dst;
}


#if !LS_HPACK_EMIT_TEST_CODE
static
#endif
       int
lshpack_enc_enc_str (unsigned char *const dst, size_t dst_len,
                        const unsigned char *str, lshpack_strlen_t str_len)
{
    unsigned char size_buf[4];
    unsigned char *p;
    unsigned size_len;
    int rc;

    if (dst_len > 1)
        /* We guess that the string size fits into a single byte -- meaning
         * compressed string of size 126 and smaller -- which is the normal
         * case.  Thus, we immediately write compressed string to the output
         * buffer.  If our guess is not correct, we fix it later.
         */
        rc = lshpack_enc_huff_encode(str, str + str_len, dst + 1, dst_len - 1);
    else if (dst_len == 1)
        /* Here, the call can only succeed if the string to encode is empty. */
        rc = 0;
    else
        return -1;

    /*
     * Check if need huffman encoding or not
     * Comment: (size_t)rc <= str_len   = means if same length, still use
     *                                                              Huffman
     *                     ^
     */
    if (rc > 0 && (size_t)rc <= str_len)
    {
        if (rc < 127)
        {
            *dst = 0x80 | rc;
            return 1 + rc;
        }
        size_buf[0] = 0x80;
        str_len = rc;
        str = dst + 1;
    }
    else if (str_len <= dst_len - 1)
    {
        if (str_len < 127)
        {
            *dst = str_len;
            memcpy(dst + 1, str, str_len);
            return 1 + str_len;
        }
        size_buf[0] = 0x00;
    }
    else
        return -1;

    /* The guess of one-byte size was incorrect.  Perform necessary
     * adjustments.
     */
    p = henc_enc_int(size_buf, size_buf + sizeof(size_buf), str_len, 7);
    if (p == size_buf)
        return -1;

    size_len = p - size_buf;
    assert(size_len > 1);

    /* Check if there is enough room in the output buffer for both
     * encoded size and the string.
     */
    if (size_len + str_len > dst_len)
        return -1;

    memmove(dst + size_len, str, str_len);
    memcpy(dst, size_buf, size_len);
    return size_len + str_len;
}


static void
henc_drop_oldest_entry (struct lshpack_enc *enc)
{
    struct lshpack_enc_table_entry *entry;
    unsigned buckno;

    entry = STAILQ_FIRST(&enc->hpe_all_entries);
    assert(entry);
    STAILQ_REMOVE_HEAD(&enc->hpe_all_entries, ete_next_all);
    buckno = BUCKNO(enc->hpe_nbits, entry->ete_nameval_hash);
    assert(entry == STAILQ_FIRST(&enc->hpe_buckets[buckno].by_nameval));
    STAILQ_REMOVE_HEAD(&enc->hpe_buckets[buckno].by_nameval, ete_next_nameval);
    buckno = BUCKNO(enc->hpe_nbits, entry->ete_name_hash);
    assert(entry == STAILQ_FIRST(&enc->hpe_buckets[buckno].by_name));
    STAILQ_REMOVE_HEAD(&enc->hpe_buckets[buckno].by_name, ete_next_name);

    enc->hpe_cur_capacity -= DYNAMIC_ENTRY_OVERHEAD + entry->ete_name_len
                                                        + entry->ete_val_len;
    --enc->hpe_nelem;
    free(entry);
}


static void
henc_remove_overflow_entries (struct lshpack_enc *enc)
{
    while (enc->hpe_cur_capacity > enc->hpe_max_capacity)
        henc_drop_oldest_entry(enc);
}


static int
henc_grow_tables (struct lshpack_enc *enc)
{
    struct lshpack_double_enc_head *new_buckets, *new[2];
    struct lshpack_enc_table_entry *entry;
    unsigned n, old_nbits;
    int idx;

    old_nbits = enc->hpe_nbits;
    new_buckets = malloc(sizeof(enc->hpe_buckets[0])
                                                * N_BUCKETS(old_nbits + 1));
    if (!new_buckets)
        return -1;

    for (n = 0; n < N_BUCKETS(old_nbits); ++n)
    {
        new[0] = &new_buckets[n];
        new[1] = &new_buckets[n + N_BUCKETS(old_nbits)];
        STAILQ_INIT(&new[0]->by_name);
        STAILQ_INIT(&new[1]->by_name);
        STAILQ_INIT(&new[0]->by_nameval);
        STAILQ_INIT(&new[1]->by_nameval);
        while ((entry = STAILQ_FIRST(&enc->hpe_buckets[n].by_name)))
        {
            STAILQ_REMOVE_HEAD(&enc->hpe_buckets[n].by_name, ete_next_name);
            idx = (BUCKNO(old_nbits + 1, entry->ete_name_hash)
                                                        >> old_nbits) & 1;
            STAILQ_INSERT_TAIL(&new[idx]->by_name, entry, ete_next_name);
        }
        while ((entry = STAILQ_FIRST(&enc->hpe_buckets[n].by_nameval)))
        {
            STAILQ_REMOVE_HEAD(&enc->hpe_buckets[n].by_nameval,
                                                        ete_next_nameval);
            idx = (BUCKNO(old_nbits + 1, entry->ete_nameval_hash)
                                                        >> old_nbits) & 1;
            STAILQ_INSERT_TAIL(&new[idx]->by_nameval, entry,
                                                        ete_next_nameval);
        }
    }

    free(enc->hpe_buckets);
    enc->hpe_nbits   = old_nbits + 1;
    enc->hpe_buckets = new_buckets;
    return 0;
}

#if !LS_HPACK_EMIT_TEST_CODE
static
#endif
       int
lshpack_enc_push_entry (struct lshpack_enc *enc, const char *name,
                        lshpack_strlen_t name_len, const char *value,
                        lshpack_strlen_t value_len)
{
    unsigned name_hash, nameval_hash, buckno;
    struct lshpack_enc_table_entry *entry;
    XXH32_state_t hash_state;
    size_t size;

    if (enc->hpe_nelem >= N_BUCKETS(enc->hpe_nbits) / 2 &&
                                                0 != henc_grow_tables(enc))
        return -1;

    size = sizeof(*entry) + name_len + value_len;
    entry = malloc(size);
    if (!entry)
        return -1;

    XXH32_reset(&hash_state, (uintptr_t) enc);
    XXH32_update(&hash_state, &name_len, sizeof(name_len));
    XXH32_update(&hash_state, name, name_len);
    name_hash = XXH32_digest(&hash_state);
    XXH32_update(&hash_state,  &value_len, sizeof(value_len));
    XXH32_update(&hash_state,  value, value_len);
    nameval_hash = XXH32_digest(&hash_state);

    entry->ete_name_hash = name_hash;
    entry->ete_nameval_hash = nameval_hash;
    entry->ete_name_len = name_len;
    entry->ete_val_len = value_len;
    entry->ete_id = enc->hpe_next_id++;
    memcpy(ETE_NAME(entry), name, name_len);
    memcpy(ETE_VALUE(entry), value, value_len);

    STAILQ_INSERT_TAIL(&enc->hpe_all_entries, entry, ete_next_all);
    buckno = BUCKNO(enc->hpe_nbits, nameval_hash);
    STAILQ_INSERT_TAIL(&enc->hpe_buckets[buckno].by_nameval, entry,
                                                        ete_next_nameval);
    buckno = BUCKNO(enc->hpe_nbits, name_hash);
    STAILQ_INSERT_TAIL(&enc->hpe_buckets[buckno].by_name, entry,
                                                        ete_next_name);

    enc->hpe_cur_capacity += DYNAMIC_ENTRY_OVERHEAD + name_len + value_len;
    ++enc->hpe_nelem;
    henc_remove_overflow_entries(enc);
    return 0;
}


unsigned char *
lshpack_enc_encode (struct lshpack_enc *enc, unsigned char *dst,
        unsigned char *dst_end, const char *name, lshpack_strlen_t name_len,
        const char *value, lshpack_strlen_t value_len, int indexed_type)
{
    //indexed_type: 0, Add, 1,: without, 2: never
    static const char indexed_prefix_number[] = {0x40, 0x00, 0x10};
    unsigned char *const dst_org = dst;
    int val_matched, rc;
    unsigned table_id;

    assert(indexed_type >= 0 && indexed_type <= 2);

    if (dst_end <= dst)
        return dst_org;

    table_id = henc_find_table_id(enc, name, name_len, value, value_len,
                                                                &val_matched);
    if (table_id > 0)
    {
        if (val_matched)
        {
            *dst = 0x80;
            dst = henc_enc_int(dst, dst_end, table_id, 7);
            /* No need to check return value: we pass it up as-is because
             * the behavior is the same.
             */
            return dst;
        }
        else
        {
            *dst = indexed_prefix_number[indexed_type];
            dst = henc_enc_int(dst, dst_end, table_id,
                                            ((indexed_type == 0) ? 6 : 4));
            if (dst == dst_org)
                return dst_org;
        }
    }
    else
    {
        *dst++ = indexed_prefix_number[indexed_type];
        rc = lshpack_enc_enc_str(dst, dst_end - dst,
                                    (const unsigned char *)name, name_len);
        if (rc < 0)
            return dst_org; //Failed to enc this header, return unchanged ptr.
        dst += rc;
    }

    rc = lshpack_enc_enc_str(dst, dst_end - dst,
                                    (const unsigned char *)value, value_len);
    if (rc < 0)
        return dst_org; //Failed to enc this header, return unchanged ptr.
    dst += rc;

    if (indexed_type == 0)
    {
        rc = lshpack_enc_push_entry(enc, name, name_len, value, value_len);
        if (rc != 0)
            return dst_org; //Failed to enc this header, return unchanged ptr.
    }

    return dst;
}


void
lshpack_enc_set_max_capacity (struct lshpack_enc *enc, unsigned max_capacity)
{
    enc->hpe_max_capacity = max_capacity;
    henc_remove_overflow_entries(enc);
}

#if LS_HPACK_EMIT_TEST_CODE
void
lshpack_enc_iter_init (struct lshpack_enc *enc, void **iter)
{
    *iter = STAILQ_FIRST(&enc->hpe_all_entries);
}


/* Returns 0 if entry is found */
int
lshpack_enc_iter_next (struct lshpack_enc *enc, void **iter,
                                        struct enc_dyn_table_entry *retval)
{
    const struct lshpack_enc_table_entry *entry;

    entry = *iter;
    if (!entry)
        return -1;

    *iter = STAILQ_NEXT(entry, ete_next_all);

    retval->name = ETE_NAME(entry);
    retval->value = ETE_VALUE(entry);
    retval->name_len = entry->ete_name_len;
    retval->value_len = entry->ete_val_len;
    retval->entry_id = henc_calc_table_id(enc, entry);
    return 0;
}
#endif


/* Dynamic table entry: */
struct dec_table_entry
{
    uint16_t    dte_name_len;
    uint16_t    dte_val_len;
    uint8_t     dte_name_idx;
    char        dte_buf[0];     /* Contains both name and value */
};

#define DTE_NAME(dte) ((dte)->dte_buf)
#define DTE_VALUE(dte) (&(dte)->dte_buf[(dte)->dte_name_len])

enum
{
    HPACK_HUFFMAN_FLAG_ACCEPTED = 0x01,
    HPACK_HUFFMAN_FLAG_SYM = 0x02,
    HPACK_HUFFMAN_FLAG_FAIL = 0x04,
};

struct decode_status
{
    uint8_t state;
    uint8_t eos;
};


void
lshpack_dec_init (struct lshpack_dec *dec)
{
    memset(dec, 0, sizeof(*dec));
    dec->hpd_max_capacity = INITIAL_DYNAMIC_TABLE_SIZE;
    dec->hpd_cur_max_capacity = INITIAL_DYNAMIC_TABLE_SIZE;
    lshpack_arr_init(&dec->hpd_dyn_table);
}


void
lshpack_dec_cleanup (struct lshpack_dec *dec)
{
    uintptr_t val;

    while (lshpack_arr_count(&dec->hpd_dyn_table) > 0)
    {
        val = lshpack_arr_pop(&dec->hpd_dyn_table);
        free((struct dec_table_entry *) val);
    }
    lshpack_arr_cleanup(&dec->hpd_dyn_table);
}


#if !LS_HPACK_EMIT_TEST_CODE
static
#endif
       int
lshpack_dec_dec_int (const unsigned char **src, const unsigned char *src_end,
                                        uint8_t prefix_bits, uint32_t *value)
{
    uint32_t B, M;
    uint8_t prefix_max = (1 << prefix_bits) - 1;

    *value = (*(*src)++ & prefix_max);

    if (*value < prefix_max)
        return 0;

    /* To optimize the loop for the normal case, the overflow is checked
     * outside the loop.  The decoder is limited to 28-bit integer values,
     * which is far above limitations imposed by the APIs (16-bit integers).
     */
    M = 0;
    do
    {
        if ((*src) >= src_end)
            return -1;
        B = *(*src)++;
        *value = *value + ((B & 0x7f) << M);
        M += 7;
    }
    while (B & 0x80);

    return -(M > sizeof(*value) * 8);
}


static void
hdec_drop_oldest_entry (struct lshpack_dec *dec)
{
    struct dec_table_entry *entry;
    entry = (void *) lshpack_arr_shift(&dec->hpd_dyn_table);
    dec->hpd_cur_capacity -= DYNAMIC_ENTRY_OVERHEAD + entry->dte_name_len
                                                        + entry->dte_val_len;
    free(entry);
}


static void
hdec_remove_overflow_entries (struct lshpack_dec *dec)
{
    while (dec->hpd_cur_capacity > dec->hpd_cur_max_capacity)
        hdec_drop_oldest_entry(dec);
}


static void
hdec_update_max_capacity (struct lshpack_dec *dec, uint32_t new_capacity)
{
    dec->hpd_cur_max_capacity = new_capacity;
    hdec_remove_overflow_entries(dec);
}


void
lshpack_dec_set_max_capacity (struct lshpack_dec *dec, unsigned max_capacity)
{
    dec->hpd_max_capacity = max_capacity;
    hdec_update_max_capacity(dec, max_capacity);
}


static unsigned char *
hdec_huff_dec4bits (uint8_t src_4bits, unsigned char *dst,
                                        struct decode_status *status)
{
    const struct decode_el cur_dec_code =
        decode_tables[status->state][src_4bits];
    if (cur_dec_code.flags & HPACK_HUFFMAN_FLAG_FAIL) {
        return NULL; //failed
    }
    if (cur_dec_code.flags & HPACK_HUFFMAN_FLAG_SYM)
    {
        *dst = cur_dec_code.sym;
        dst++;
    }

    status->state = cur_dec_code.state;
    status->eos = ((cur_dec_code.flags & HPACK_HUFFMAN_FLAG_ACCEPTED) != 0);
    return dst;
}


int
lshpack_dec_huff_decode_full (const unsigned char *src, int src_len,
                                            unsigned char *dst, int dst_len)
{
    const unsigned char *p_src = src;
    const unsigned char *const src_end = src + src_len;
    unsigned char *p_dst = dst;
    unsigned char *dst_end = dst + dst_len;
    struct decode_status status = { 0, 1 };

    while (p_src != src_end)
    {
        if (p_dst == dst_end)
            return -2;
        if ((p_dst = hdec_huff_dec4bits(*p_src >> 4, p_dst, &status))
                == NULL)
            return -1;
        if (p_dst == dst_end)
            return -2;
        if ((p_dst = hdec_huff_dec4bits(*p_src & 0xf, p_dst, &status))
                == NULL)
            return -1;
        ++p_src;
    }

    if (!status.eos)
        return -1;

    return p_dst - dst;
}


int
lshpack_dec_huff_decode (const unsigned char *src, int src_len,
                                            unsigned char *dst, int dst_len);


//reutrn the length in the dst, also update the src
#if !LS_HPACK_EMIT_TEST_CODE
static
#endif
       int
hdec_dec_str (unsigned char *dst, size_t dst_len, const unsigned char **src,
        const unsigned char *src_end)
{
    if ((*src) == src_end)
        return 0;

    int is_huffman = (*(*src) & 0x80);
    uint32_t len;
    if (0 != lshpack_dec_dec_int(src, src_end, 7, &len))
        return -2;  //wrong int

    int ret = 0;
    if ((uint32_t)(src_end - (*src)) < len) {
        return -2;  //wrong int
    }

    if (is_huffman)
    {
        ret = lshpack_dec_huff_decode(*src, len, dst, dst_len);
        if (ret < 0)
            return -3; //Wrong code

        (*src) += len;
    }
    else
    {
        if (dst_len < len)
            ret = -3;  //dst not enough space
        else
        {
            memcpy(dst, (*src), len);
            (*src) += len;
            ret = len;
        }
    }

    return ret;
}


/* hpd_dyn_table is a dynamic array.  New entries are pushed onto it,
 * while old entries are shifted from it.
 */
static struct dec_table_entry *
hdec_get_table_entry (struct lshpack_dec *dec, uint32_t index)
{
    uintptr_t val;

    index -= HPACK_STATIC_TABLE_SIZE;
    if (index == 0 || index > lshpack_arr_count(&dec->hpd_dyn_table))
        return NULL;

    index = lshpack_arr_count(&dec->hpd_dyn_table) - index;
    val = lshpack_arr_get(&dec->hpd_dyn_table, index);
    return (struct dec_table_entry *) val;
}


#if !LS_HPACK_EMIT_TEST_CODE
static
#endif
       int
lshpack_dec_push_entry (struct lshpack_dec *dec, uint8_t name_idx, const char *name,
                        uint16_t name_len, const char *val, uint16_t val_len)
{
    struct dec_table_entry *entry;
    size_t size;

    size = sizeof(*entry) + name_len + val_len;
    entry = malloc(size);
    if (!entry)
        return -1;

    if (0 != lshpack_arr_push(&dec->hpd_dyn_table, (uintptr_t) entry))
    {
        free(entry);
        return -1;
    }

    dec->hpd_cur_capacity += DYNAMIC_ENTRY_OVERHEAD + name_len + val_len;
    entry->dte_name_len = name_len;
    entry->dte_val_len = val_len;
    entry->dte_name_idx = name_idx;
    memcpy(DTE_NAME(entry), name, name_len);
    memcpy(DTE_VALUE(entry), val, val_len);
    return 0;
}


int
lshpack_dec_decode (struct lshpack_dec *dec,
    const unsigned char **src, const unsigned char *src_end,
    char *dst, char *const dst_end, uint16_t *name_len, uint16_t *val_len,
    uint32_t *name_idx)
{
    struct dec_table_entry *entry;
    uint32_t index, new_capacity;
    int indexed_type, len;

    if ((*src) == src_end)
        return -1;

    while ((*(*src) & 0xe0) == 0x20)    //001 xxxxx
    {
        if (0 != lshpack_dec_dec_int(src, src_end, 5, &new_capacity))
            return -1;
        if (new_capacity > dec->hpd_max_capacity)
            return -1;
        hdec_update_max_capacity(dec, new_capacity);
        if (*src == src_end)
            return -1;
    }

    /* lshpack_dec_dec_int() sets `index' and advances `src'.  If we do not
     * call it, we set `index' and advance `src' ourselves:
     */
    if (*(*src) & 0x80) //1 xxxxxxx
    {
        if (0 != lshpack_dec_dec_int(src, src_end, 7, &index))
            return -1;

        indexed_type = 3; //need to parse value
    }
    else if (*(*src) > 0x40) //01 xxxxxx
    {
        if (0 != lshpack_dec_dec_int(src, src_end, 6, &index))
            return -1;

        indexed_type = 0;
    }
    else if (*(*src) == 0x40) //custmized //0100 0000
    {
        indexed_type = 0;
        index = 0;
        ++(*src);
    }

    //Never indexed
    else if (*(*src) == 0x10)  //00010000
    {
        indexed_type = 2;
        index = 0;
        ++(*src);
    }
    else if ((*(*src) & 0xf0) == 0x10)  //0001 xxxx
    {
        if (0 != lshpack_dec_dec_int(src, src_end, 4, &index))
            return -1;

        indexed_type = 2;
    }

    //without indexed
    else if (*(*src) == 0x00)  //0000 0000
    {
        indexed_type = 1;
        index = 0;
        ++(*src);
    }
    else // 0000 xxxx
    {
        if (0 != lshpack_dec_dec_int(src, src_end, 4, &index))
            return -1;

        indexed_type = 1;
    }
    *name_idx = index;

    char *const name = dst;
    if (index > 0)
    {
        if (index <= HPACK_STATIC_TABLE_SIZE) //static table
        {
            if (static_table[index - 1].name_len > dst_end - dst)
                return -1;
            *name_len = static_table[index - 1].name_len;
            memcpy(name, static_table[index - 1].name, *name_len);
            if (indexed_type == 3)
            {
                if (static_table[index - 1].name_len +
                    static_table[index - 1].val_len > dst_end - dst)
                    return -1;
                *val_len = static_table[index - 1].val_len;
                memcpy(name + *name_len, static_table[index - 1].val, *val_len);
                return 0;
            }
        }
        else
        {
            entry = hdec_get_table_entry(dec, index);
            if (entry == NULL)
                return -1;
            if (entry->dte_name_len > dst_end - dst)
                return -1;

            *name_len = entry->dte_name_len;
            memcpy(name, DTE_NAME(entry), *name_len);
            if (entry->dte_name_idx)
                *name_idx = entry->dte_name_idx;
            if (indexed_type == 3)
            {
                if (entry->dte_name_len + entry->dte_val_len > dst_end - dst)
                    return -1;
                *val_len = entry->dte_val_len;
                memcpy(name + *name_len, DTE_VALUE(entry), *val_len);
                return 0;
            }
        }
    }
    else
    {
        len = hdec_dec_str((unsigned char *)name, dst_end - dst, src, src_end);
        if (len < 0)
            return len; //error
        if (len > UINT16_MAX)
            return -2;
        *name_len = len;
    }

    len = hdec_dec_str((unsigned char *)name + *name_len,
                                    dst_end - dst - *name_len, src, src_end);
    if (len < 0)
        return len; //error
    if (len > UINT16_MAX)
        return -2;
    *val_len = len;

    if (indexed_type == 0)
    {
        if (index > HPACK_STATIC_TABLE_SIZE)
            index = 0;
        if (0 != lshpack_dec_push_entry(dec, index, name, *name_len,
                                            name + *name_len, *val_len))
            return -1;  //error
    }

    return 0;
}


static const struct hdec { uint8_t lens; uint8_t out[3]; } hdecs[] =
{
/*
          ,------------- Number of bits consumed by the index
          |      ,------ Number of bytes in output; 0 means long or invalid code
          |      |  ,--- Output, aligned to the left
          V      V  V                                                         */
/* 0 */ {(15<<2)|3,{48,48,48}},
/* 1 */ {(15<<2)|3,{48,48,48}},
/* 2 */ {(15<<2)|3,{48,48,49}},
/* 3 */ {(15<<2)|3,{48,48,49}},
/* 4 */ {(15<<2)|3,{48,48,50}},
/* 5 */ {(15<<2)|3,{48,48,50}},
/* 6 */ {(15<<2)|3,{48,48,97}},
/* 7 */ {(15<<2)|3,{48,48,97}},
/* 8 */ {(15<<2)|3,{48,48,99}},
/* 9 */ {(15<<2)|3,{48,48,99}},
/* 10 */ {(15<<2)|3,{48,48,101}},
/* 11 */ {(15<<2)|3,{48,48,101}},
/* 12 */ {(15<<2)|3,{48,48,105}},
/* 13 */ {(15<<2)|3,{48,48,105}},
/* 14 */ {(15<<2)|3,{48,48,111}},
/* 15 */ {(15<<2)|3,{48,48,111}},
/* 16 */ {(15<<2)|3,{48,48,115}},
/* 17 */ {(15<<2)|3,{48,48,115}},
/* 18 */ {(15<<2)|3,{48,48,116}},
/* 19 */ {(15<<2)|3,{48,48,116}},
/* 20 */ {(16<<2)|3,{48,48,32}},
/* 21 */ {(16<<2)|3,{48,48,37}},
/* 22 */ {(16<<2)|3,{48,48,45}},
/* 23 */ {(16<<2)|3,{48,48,46}},
/* 24 */ {(16<<2)|3,{48,48,47}},
/* 25 */ {(16<<2)|3,{48,48,51}},
/* 26 */ {(16<<2)|3,{48,48,52}},
/* 27 */ {(16<<2)|3,{48,48,53}},
/* 28 */ {(16<<2)|3,{48,48,54}},
/* 29 */ {(16<<2)|3,{48,48,55}},
/* 30 */ {(16<<2)|3,{48,48,56}},
/* 31 */ {(16<<2)|3,{48,48,57}},
/* 32 */ {(16<<2)|3,{48,48,61}},
/* 33 */ {(16<<2)|3,{48,48,65}},
/* 34 */ {(16<<2)|3,{48,48,95}},
/* 35 */ {(16<<2)|3,{48,48,98}},
/* 36 */ {(16<<2)|3,{48,48,100}},
/* 37 */ {(16<<2)|3,{48,48,102}},
/* 38 */ {(16<<2)|3,{48,48,103}},
/* 39 */ {(16<<2)|3,{48,48,104}},
/* 40 */ {(16<<2)|3,{48,48,108}},
/* 41 */ {(16<<2)|3,{48,48,109}},
/* 42 */ {(16<<2)|3,{48,48,110}},
/* 43 */ {(16<<2)|3,{48,48,112}},
/* 44 */ {(16<<2)|3,{48,48,114}},
/* 45 */ {(16<<2)|3,{48,48,117}},
/* 46 */ {(10<<2)|2,{48,48,0}},
/* 47 */ {(10<<2)|2,{48,48,0}},
/* 48 */ {(10<<2)|2,{48,48,0}},
/* 49 */ {(10<<2)|2,{48,48,0}},
/* 50 */ {(10<<2)|2,{48,48,0}},
/* 51 */ {(10<<2)|2,{48,48,0}},
/* 52 */ {(10<<2)|2,{48,48,0}},
/* 53 */ {(10<<2)|2,{48,48,0}},
/* 54 */ {(10<<2)|2,{48,48,0}},
/* 55 */ {(10<<2)|2,{48,48,0}},
/* 56 */ {(10<<2)|2,{48,48,0}},
/* 57 */ {(10<<2)|2,{48,48,0}},
/* 58 */ {(10<<2)|2,{48,48,0}},
/* 59 */ {(10<<2)|2,{48,48,0}},
/* 60 */ {(10<<2)|2,{48,48,0}},
/* 61 */ {(10<<2)|2,{48,48,0}},
/* 62 */ {(10<<2)|2,{48,48,0}},
/* 63 */ {(10<<2)|2,{48,48,0}},
/* 64 */ {(15<<2)|3,{48,49,48}},
/* 65 */ {(15<<2)|3,{48,49,48}},
/* 66 */ {(15<<2)|3,{48,49,49}},
/* 67 */ {(15<<2)|3,{48,49,49}},
/* 68 */ {(15<<2)|3,{48,49,50}},
/* 69 */ {(15<<2)|3,{48,49,50}},
/* 70 */ {(15<<2)|3,{48,49,97}},
/* 71 */ {(15<<2)|3,{48,49,97}},
/* 72 */ {(15<<2)|3,{48,49,99}},
/* 73 */ {(15<<2)|3,{48,49,99}},
/* 74 */ {(15<<2)|3,{48,49,101}},
/* 75 */ {(15<<2)|3,{48,49,101}},
/* 76 */ {(15<<2)|3,{48,49,105}},
/* 77 */ {(15<<2)|3,{48,49,105}},
/* 78 */ {(15<<2)|3,{48,49,111}},
/* 79 */ {(15<<2)|3,{48,49,111}},
/* 80 */ {(15<<2)|3,{48,49,115}},
/* 81 */ {(15<<2)|3,{48,49,115}},
/* 82 */ {(15<<2)|3,{48,49,116}},
/* 83 */ {(15<<2)|3,{48,49,116}},
/* 84 */ {(16<<2)|3,{48,49,32}},
/* 85 */ {(16<<2)|3,{48,49,37}},
/* 86 */ {(16<<2)|3,{48,49,45}},
/* 87 */ {(16<<2)|3,{48,49,46}},
/* 88 */ {(16<<2)|3,{48,49,47}},
/* 89 */ {(16<<2)|3,{48,49,51}},
/* 90 */ {(16<<2)|3,{48,49,52}},
/* 91 */ {(16<<2)|3,{48,49,53}},
/* 92 */ {(16<<2)|3,{48,49,54}},
/* 93 */ {(16<<2)|3,{48,49,55}},
/* 94 */ {(16<<2)|3,{48,49,56}},
/* 95 */ {(16<<2)|3,{48,49,57}},
/* 96 */ {(16<<2)|3,{48,49,61}},
/* 97 */ {(16<<2)|3,{48,49,65}},
/* 98 */ {(16<<2)|3,{48,49,95}},
/* 99 */ {(16<<2)|3,{48,49,98}},
/* 100 */ {(16<<2)|3,{48,49,100}},
/* 101 */ {(16<<2)|3,{48,49,102}},
/* 102 */ {(16<<2)|3,{48,49,103}},
/* 103 */ {(16<<2)|3,{48,49,104}},
/* 104 */ {(16<<2)|3,{48,49,108}},
/* 105 */ {(16<<2)|3,{48,49,109}},
/* 106 */ {(16<<2)|3,{48,49,110}},
/* 107 */ {(16<<2)|3,{48,49,112}},
/* 108 */ {(16<<2)|3,{48,49,114}},
/* 109 */ {(16<<2)|3,{48,49,117}},
/* 110 */ {(10<<2)|2,{48,49,0}},
/* 111 */ {(10<<2)|2,{48,49,0}},
/* 112 */ {(10<<2)|2,{48,49,0}},
/* 113 */ {(10<<2)|2,{48,49,0}},
/* 114 */ {(10<<2)|2,{48,49,0}},
/* 115 */ {(10<<2)|2,{48,49,0}},
/* 116 */ {(10<<2)|2,{48,49,0}},
/* 117 */ {(10<<2)|2,{48,49,0}},
/* 118 */ {(10<<2)|2,{48,49,0}},
/* 119 */ {(10<<2)|2,{48,49,0}},
/* 120 */ {(10<<2)|2,{48,49,0}},
/* 121 */ {(10<<2)|2,{48,49,0}},
/* 122 */ {(10<<2)|2,{48,49,0}},
/* 123 */ {(10<<2)|2,{48,49,0}},
/* 124 */ {(10<<2)|2,{48,49,0}},
/* 125 */ {(10<<2)|2,{48,49,0}},
/* 126 */ {(10<<2)|2,{48,49,0}},
/* 127 */ {(10<<2)|2,{48,49,0}},
/* 128 */ {(15<<2)|3,{48,50,48}},
/* 129 */ {(15<<2)|3,{48,50,48}},
/* 130 */ {(15<<2)|3,{48,50,49}},
/* 131 */ {(15<<2)|3,{48,50,49}},
/* 132 */ {(15<<2)|3,{48,50,50}},
/* 133 */ {(15<<2)|3,{48,50,50}},
/* 134 */ {(15<<2)|3,{48,50,97}},
/* 135 */ {(15<<2)|3,{48,50,97}},
/* 136 */ {(15<<2)|3,{48,50,99}},
/* 137 */ {(15<<2)|3,{48,50,99}},
/* 138 */ {(15<<2)|3,{48,50,101}},
/* 139 */ {(15<<2)|3,{48,50,101}},
/* 140 */ {(15<<2)|3,{48,50,105}},
/* 141 */ {(15<<2)|3,{48,50,105}},
/* 142 */ {(15<<2)|3,{48,50,111}},
/* 143 */ {(15<<2)|3,{48,50,111}},
/* 144 */ {(15<<2)|3,{48,50,115}},
/* 145 */ {(15<<2)|3,{48,50,115}},
/* 146 */ {(15<<2)|3,{48,50,116}},
/* 147 */ {(15<<2)|3,{48,50,116}},
/* 148 */ {(16<<2)|3,{48,50,32}},
/* 149 */ {(16<<2)|3,{48,50,37}},
/* 150 */ {(16<<2)|3,{48,50,45}},
/* 151 */ {(16<<2)|3,{48,50,46}},
/* 152 */ {(16<<2)|3,{48,50,47}},
/* 153 */ {(16<<2)|3,{48,50,51}},
/* 154 */ {(16<<2)|3,{48,50,52}},
/* 155 */ {(16<<2)|3,{48,50,53}},
/* 156 */ {(16<<2)|3,{48,50,54}},
/* 157 */ {(16<<2)|3,{48,50,55}},
/* 158 */ {(16<<2)|3,{48,50,56}},
/* 159 */ {(16<<2)|3,{48,50,57}},
/* 160 */ {(16<<2)|3,{48,50,61}},
/* 161 */ {(16<<2)|3,{48,50,65}},
/* 162 */ {(16<<2)|3,{48,50,95}},
/* 163 */ {(16<<2)|3,{48,50,98}},
/* 164 */ {(16<<2)|3,{48,50,100}},
/* 165 */ {(16<<2)|3,{48,50,102}},
/* 166 */ {(16<<2)|3,{48,50,103}},
/* 167 */ {(16<<2)|3,{48,50,104}},
/* 168 */ {(16<<2)|3,{48,50,108}},
/* 169 */ {(16<<2)|3,{48,50,109}},
/* 170 */ {(16<<2)|3,{48,50,110}},
/* 171 */ {(16<<2)|3,{48,50,112}},
/* 172 */ {(16<<2)|3,{48,50,114}},
/* 173 */ {(16<<2)|3,{48,50,117}},
/* 174 */ {(10<<2)|2,{48,50,0}},
/* 175 */ {(10<<2)|2,{48,50,0}},
/* 176 */ {(10<<2)|2,{48,50,0}},
/* 177 */ {(10<<2)|2,{48,50,0}},
/* 178 */ {(10<<2)|2,{48,50,0}},
/* 179 */ {(10<<2)|2,{48,50,0}},
/* 180 */ {(10<<2)|2,{48,50,0}},
/* 181 */ {(10<<2)|2,{48,50,0}},
/* 182 */ {(10<<2)|2,{48,50,0}},
/* 183 */ {(10<<2)|2,{48,50,0}},
/* 184 */ {(10<<2)|2,{48,50,0}},
/* 185 */ {(10<<2)|2,{48,50,0}},
/* 186 */ {(10<<2)|2,{48,50,0}},
/* 187 */ {(10<<2)|2,{48,50,0}},
/* 188 */ {(10<<2)|2,{48,50,0}},
/* 189 */ {(10<<2)|2,{48,50,0}},
/* 190 */ {(10<<2)|2,{48,50,0}},
/* 191 */ {(10<<2)|2,{48,50,0}},
/* 192 */ {(15<<2)|3,{48,97,48}},
/* 193 */ {(15<<2)|3,{48,97,48}},
/* 194 */ {(15<<2)|3,{48,97,49}},
/* 195 */ {(15<<2)|3,{48,97,49}},
/* 196 */ {(15<<2)|3,{48,97,50}},
/* 197 */ {(15<<2)|3,{48,97,50}},
/* 198 */ {(15<<2)|3,{48,97,97}},
/* 199 */ {(15<<2)|3,{48,97,97}},
/* 200 */ {(15<<2)|3,{48,97,99}},
/* 201 */ {(15<<2)|3,{48,97,99}},
/* 202 */ {(15<<2)|3,{48,97,101}},
/* 203 */ {(15<<2)|3,{48,97,101}},
/* 204 */ {(15<<2)|3,{48,97,105}},
/* 205 */ {(15<<2)|3,{48,97,105}},
/* 206 */ {(15<<2)|3,{48,97,111}},
/* 207 */ {(15<<2)|3,{48,97,111}},
/* 208 */ {(15<<2)|3,{48,97,115}},
/* 209 */ {(15<<2)|3,{48,97,115}},
/* 210 */ {(15<<2)|3,{48,97,116}},
/* 211 */ {(15<<2)|3,{48,97,116}},
/* 212 */ {(16<<2)|3,{48,97,32}},
/* 213 */ {(16<<2)|3,{48,97,37}},
/* 214 */ {(16<<2)|3,{48,97,45}},
/* 215 */ {(16<<2)|3,{48,97,46}},
/* 216 */ {(16<<2)|3,{48,97,47}},
/* 217 */ {(16<<2)|3,{48,97,51}},
/* 218 */ {(16<<2)|3,{48,97,52}},
/* 219 */ {(16<<2)|3,{48,97,53}},
/* 220 */ {(16<<2)|3,{48,97,54}},
/* 221 */ {(16<<2)|3,{48,97,55}},
/* 222 */ {(16<<2)|3,{48,97,56}},
/* 223 */ {(16<<2)|3,{48,97,57}},
/* 224 */ {(16<<2)|3,{48,97,61}},
/* 225 */ {(16<<2)|3,{48,97,65}},
/* 226 */ {(16<<2)|3,{48,97,95}},
/* 227 */ {(16<<2)|3,{48,97,98}},
/* 228 */ {(16<<2)|3,{48,97,100}},
/* 229 */ {(16<<2)|3,{48,97,102}},
/* 230 */ {(16<<2)|3,{48,97,103}},
/* 231 */ {(16<<2)|3,{48,97,104}},
/* 232 */ {(16<<2)|3,{48,97,108}},
/* 233 */ {(16<<2)|3,{48,97,109}},
/* 234 */ {(16<<2)|3,{48,97,110}},
/* 235 */ {(16<<2)|3,{48,97,112}},
/* 236 */ {(16<<2)|3,{48,97,114}},
/* 237 */ {(16<<2)|3,{48,97,117}},
/* 238 */ {(10<<2)|2,{48,97,0}},
/* 239 */ {(10<<2)|2,{48,97,0}},
/* 240 */ {(10<<2)|2,{48,97,0}},
/* 241 */ {(10<<2)|2,{48,97,0}},
/* 242 */ {(10<<2)|2,{48,97,0}},
/* 243 */ {(10<<2)|2,{48,97,0}},
/* 244 */ {(10<<2)|2,{48,97,0}},
/* 245 */ {(10<<2)|2,{48,97,0}},
/* 246 */ {(10<<2)|2,{48,97,0}},
/* 247 */ {(10<<2)|2,{48,97,0}},
/* 248 */ {(10<<2)|2,{48,97,0}},
/* 249 */ {(10<<2)|2,{48,97,0}},
/* 250 */ {(10<<2)|2,{48,97,0}},
/* 251 */ {(10<<2)|2,{48,97,0}},
/* 252 */ {(10<<2)|2,{48,97,0}},
/* 253 */ {(10<<2)|2,{48,97,0}},
/* 254 */ {(10<<2)|2,{48,97,0}},
/* 255 */ {(10<<2)|2,{48,97,0}},
/* 256 */ {(15<<2)|3,{48,99,48}},
/* 257 */ {(15<<2)|3,{48,99,48}},
/* 258 */ {(15<<2)|3,{48,99,49}},
/* 259 */ {(15<<2)|3,{48,99,49}},
/* 260 */ {(15<<2)|3,{48,99,50}},
/* 261 */ {(15<<2)|3,{48,99,50}},
/* 262 */ {(15<<2)|3,{48,99,97}},
/* 263 */ {(15<<2)|3,{48,99,97}},
/* 264 */ {(15<<2)|3,{48,99,99}},
/* 265 */ {(15<<2)|3,{48,99,99}},
/* 266 */ {(15<<2)|3,{48,99,101}},
/* 267 */ {(15<<2)|3,{48,99,101}},
/* 268 */ {(15<<2)|3,{48,99,105}},
/* 269 */ {(15<<2)|3,{48,99,105}},
/* 270 */ {(15<<2)|3,{48,99,111}},
/* 271 */ {(15<<2)|3,{48,99,111}},
/* 272 */ {(15<<2)|3,{48,99,115}},
/* 273 */ {(15<<2)|3,{48,99,115}},
/* 274 */ {(15<<2)|3,{48,99,116}},
/* 275 */ {(15<<2)|3,{48,99,116}},
/* 276 */ {(16<<2)|3,{48,99,32}},
/* 277 */ {(16<<2)|3,{48,99,37}},
/* 278 */ {(16<<2)|3,{48,99,45}},
/* 279 */ {(16<<2)|3,{48,99,46}},
/* 280 */ {(16<<2)|3,{48,99,47}},
/* 281 */ {(16<<2)|3,{48,99,51}},
/* 282 */ {(16<<2)|3,{48,99,52}},
/* 283 */ {(16<<2)|3,{48,99,53}},
/* 284 */ {(16<<2)|3,{48,99,54}},
/* 285 */ {(16<<2)|3,{48,99,55}},
/* 286 */ {(16<<2)|3,{48,99,56}},
/* 287 */ {(16<<2)|3,{48,99,57}},
/* 288 */ {(16<<2)|3,{48,99,61}},
/* 289 */ {(16<<2)|3,{48,99,65}},
/* 290 */ {(16<<2)|3,{48,99,95}},
/* 291 */ {(16<<2)|3,{48,99,98}},
/* 292 */ {(16<<2)|3,{48,99,100}},
/* 293 */ {(16<<2)|3,{48,99,102}},
/* 294 */ {(16<<2)|3,{48,99,103}},
/* 295 */ {(16<<2)|3,{48,99,104}},
/* 296 */ {(16<<2)|3,{48,99,108}},
/* 297 */ {(16<<2)|3,{48,99,109}},
/* 298 */ {(16<<2)|3,{48,99,110}},
/* 299 */ {(16<<2)|3,{48,99,112}},
/* 300 */ {(16<<2)|3,{48,99,114}},
/* 301 */ {(16<<2)|3,{48,99,117}},
/* 302 */ {(10<<2)|2,{48,99,0}},
/* 303 */ {(10<<2)|2,{48,99,0}},
/* 304 */ {(10<<2)|2,{48,99,0}},
/* 305 */ {(10<<2)|2,{48,99,0}},
/* 306 */ {(10<<2)|2,{48,99,0}},
/* 307 */ {(10<<2)|2,{48,99,0}},
/* 308 */ {(10<<2)|2,{48,99,0}},
/* 309 */ {(10<<2)|2,{48,99,0}},
/* 310 */ {(10<<2)|2,{48,99,0}},
/* 311 */ {(10<<2)|2,{48,99,0}},
/* 312 */ {(10<<2)|2,{48,99,0}},
/* 313 */ {(10<<2)|2,{48,99,0}},
/* 314 */ {(10<<2)|2,{48,99,0}},
/* 315 */ {(10<<2)|2,{48,99,0}},
/* 316 */ {(10<<2)|2,{48,99,0}},
/* 317 */ {(10<<2)|2,{48,99,0}},
/* 318 */ {(10<<2)|2,{48,99,0}},
/* 319 */ {(10<<2)|2,{48,99,0}},
/* 320 */ {(15<<2)|3,{48,101,48}},
/* 321 */ {(15<<2)|3,{48,101,48}},
/* 322 */ {(15<<2)|3,{48,101,49}},
/* 323 */ {(15<<2)|3,{48,101,49}},
/* 324 */ {(15<<2)|3,{48,101,50}},
/* 325 */ {(15<<2)|3,{48,101,50}},
/* 326 */ {(15<<2)|3,{48,101,97}},
/* 327 */ {(15<<2)|3,{48,101,97}},
/* 328 */ {(15<<2)|3,{48,101,99}},
/* 329 */ {(15<<2)|3,{48,101,99}},
/* 330 */ {(15<<2)|3,{48,101,101}},
/* 331 */ {(15<<2)|3,{48,101,101}},
/* 332 */ {(15<<2)|3,{48,101,105}},
/* 333 */ {(15<<2)|3,{48,101,105}},
/* 334 */ {(15<<2)|3,{48,101,111}},
/* 335 */ {(15<<2)|3,{48,101,111}},
/* 336 */ {(15<<2)|3,{48,101,115}},
/* 337 */ {(15<<2)|3,{48,101,115}},
/* 338 */ {(15<<2)|3,{48,101,116}},
/* 339 */ {(15<<2)|3,{48,101,116}},
/* 340 */ {(16<<2)|3,{48,101,32}},
/* 341 */ {(16<<2)|3,{48,101,37}},
/* 342 */ {(16<<2)|3,{48,101,45}},
/* 343 */ {(16<<2)|3,{48,101,46}},
/* 344 */ {(16<<2)|3,{48,101,47}},
/* 345 */ {(16<<2)|3,{48,101,51}},
/* 346 */ {(16<<2)|3,{48,101,52}},
/* 347 */ {(16<<2)|3,{48,101,53}},
/* 348 */ {(16<<2)|3,{48,101,54}},
/* 349 */ {(16<<2)|3,{48,101,55}},
/* 350 */ {(16<<2)|3,{48,101,56}},
/* 351 */ {(16<<2)|3,{48,101,57}},
/* 352 */ {(16<<2)|3,{48,101,61}},
/* 353 */ {(16<<2)|3,{48,101,65}},
/* 354 */ {(16<<2)|3,{48,101,95}},
/* 355 */ {(16<<2)|3,{48,101,98}},
/* 356 */ {(16<<2)|3,{48,101,100}},
/* 357 */ {(16<<2)|3,{48,101,102}},
/* 358 */ {(16<<2)|3,{48,101,103}},
/* 359 */ {(16<<2)|3,{48,101,104}},
/* 360 */ {(16<<2)|3,{48,101,108}},
/* 361 */ {(16<<2)|3,{48,101,109}},
/* 362 */ {(16<<2)|3,{48,101,110}},
/* 363 */ {(16<<2)|3,{48,101,112}},
/* 364 */ {(16<<2)|3,{48,101,114}},
/* 365 */ {(16<<2)|3,{48,101,117}},
/* 366 */ {(10<<2)|2,{48,101,0}},
/* 367 */ {(10<<2)|2,{48,101,0}},
/* 368 */ {(10<<2)|2,{48,101,0}},
/* 369 */ {(10<<2)|2,{48,101,0}},
/* 370 */ {(10<<2)|2,{48,101,0}},
/* 371 */ {(10<<2)|2,{48,101,0}},
/* 372 */ {(10<<2)|2,{48,101,0}},
/* 373 */ {(10<<2)|2,{48,101,0}},
/* 374 */ {(10<<2)|2,{48,101,0}},
/* 375 */ {(10<<2)|2,{48,101,0}},
/* 376 */ {(10<<2)|2,{48,101,0}},
/* 377 */ {(10<<2)|2,{48,101,0}},
/* 378 */ {(10<<2)|2,{48,101,0}},
/* 379 */ {(10<<2)|2,{48,101,0}},
/* 380 */ {(10<<2)|2,{48,101,0}},
/* 381 */ {(10<<2)|2,{48,101,0}},
/* 382 */ {(10<<2)|2,{48,101,0}},
/* 383 */ {(10<<2)|2,{48,101,0}},
/* 384 */ {(15<<2)|3,{48,105,48}},
/* 385 */ {(15<<2)|3,{48,105,48}},
/* 386 */ {(15<<2)|3,{48,105,49}},
/* 387 */ {(15<<2)|3,{48,105,49}},
/* 388 */ {(15<<2)|3,{48,105,50}},
/* 389 */ {(15<<2)|3,{48,105,50}},
/* 390 */ {(15<<2)|3,{48,105,97}},
/* 391 */ {(15<<2)|3,{48,105,97}},
/* 392 */ {(15<<2)|3,{48,105,99}},
/* 393 */ {(15<<2)|3,{48,105,99}},
/* 394 */ {(15<<2)|3,{48,105,101}},
/* 395 */ {(15<<2)|3,{48,105,101}},
/* 396 */ {(15<<2)|3,{48,105,105}},
/* 397 */ {(15<<2)|3,{48,105,105}},
/* 398 */ {(15<<2)|3,{48,105,111}},
/* 399 */ {(15<<2)|3,{48,105,111}},
/* 400 */ {(15<<2)|3,{48,105,115}},
/* 401 */ {(15<<2)|3,{48,105,115}},
/* 402 */ {(15<<2)|3,{48,105,116}},
/* 403 */ {(15<<2)|3,{48,105,116}},
/* 404 */ {(16<<2)|3,{48,105,32}},
/* 405 */ {(16<<2)|3,{48,105,37}},
/* 406 */ {(16<<2)|3,{48,105,45}},
/* 407 */ {(16<<2)|3,{48,105,46}},
/* 408 */ {(16<<2)|3,{48,105,47}},
/* 409 */ {(16<<2)|3,{48,105,51}},
/* 410 */ {(16<<2)|3,{48,105,52}},
/* 411 */ {(16<<2)|3,{48,105,53}},
/* 412 */ {(16<<2)|3,{48,105,54}},
/* 413 */ {(16<<2)|3,{48,105,55}},
/* 414 */ {(16<<2)|3,{48,105,56}},
/* 415 */ {(16<<2)|3,{48,105,57}},
/* 416 */ {(16<<2)|3,{48,105,61}},
/* 417 */ {(16<<2)|3,{48,105,65}},
/* 418 */ {(16<<2)|3,{48,105,95}},
/* 419 */ {(16<<2)|3,{48,105,98}},
/* 420 */ {(16<<2)|3,{48,105,100}},
/* 421 */ {(16<<2)|3,{48,105,102}},
/* 422 */ {(16<<2)|3,{48,105,103}},
/* 423 */ {(16<<2)|3,{48,105,104}},
/* 424 */ {(16<<2)|3,{48,105,108}},
/* 425 */ {(16<<2)|3,{48,105,109}},
/* 426 */ {(16<<2)|3,{48,105,110}},
/* 427 */ {(16<<2)|3,{48,105,112}},
/* 428 */ {(16<<2)|3,{48,105,114}},
/* 429 */ {(16<<2)|3,{48,105,117}},
/* 430 */ {(10<<2)|2,{48,105,0}},
/* 431 */ {(10<<2)|2,{48,105,0}},
/* 432 */ {(10<<2)|2,{48,105,0}},
/* 433 */ {(10<<2)|2,{48,105,0}},
/* 434 */ {(10<<2)|2,{48,105,0}},
/* 435 */ {(10<<2)|2,{48,105,0}},
/* 436 */ {(10<<2)|2,{48,105,0}},
/* 437 */ {(10<<2)|2,{48,105,0}},
/* 438 */ {(10<<2)|2,{48,105,0}},
/* 439 */ {(10<<2)|2,{48,105,0}},
/* 440 */ {(10<<2)|2,{48,105,0}},
/* 441 */ {(10<<2)|2,{48,105,0}},
/* 442 */ {(10<<2)|2,{48,105,0}},
/* 443 */ {(10<<2)|2,{48,105,0}},
/* 444 */ {(10<<2)|2,{48,105,0}},
/* 445 */ {(10<<2)|2,{48,105,0}},
/* 446 */ {(10<<2)|2,{48,105,0}},
/* 447 */ {(10<<2)|2,{48,105,0}},
/* 448 */ {(15<<2)|3,{48,111,48}},
/* 449 */ {(15<<2)|3,{48,111,48}},
/* 450 */ {(15<<2)|3,{48,111,49}},
/* 451 */ {(15<<2)|3,{48,111,49}},
/* 452 */ {(15<<2)|3,{48,111,50}},
/* 453 */ {(15<<2)|3,{48,111,50}},
/* 454 */ {(15<<2)|3,{48,111,97}},
/* 455 */ {(15<<2)|3,{48,111,97}},
/* 456 */ {(15<<2)|3,{48,111,99}},
/* 457 */ {(15<<2)|3,{48,111,99}},
/* 458 */ {(15<<2)|3,{48,111,101}},
/* 459 */ {(15<<2)|3,{48,111,101}},
/* 460 */ {(15<<2)|3,{48,111,105}},
/* 461 */ {(15<<2)|3,{48,111,105}},
/* 462 */ {(15<<2)|3,{48,111,111}},
/* 463 */ {(15<<2)|3,{48,111,111}},
/* 464 */ {(15<<2)|3,{48,111,115}},
/* 465 */ {(15<<2)|3,{48,111,115}},
/* 466 */ {(15<<2)|3,{48,111,116}},
/* 467 */ {(15<<2)|3,{48,111,116}},
/* 468 */ {(16<<2)|3,{48,111,32}},
/* 469 */ {(16<<2)|3,{48,111,37}},
/* 470 */ {(16<<2)|3,{48,111,45}},
/* 471 */ {(16<<2)|3,{48,111,46}},
/* 472 */ {(16<<2)|3,{48,111,47}},
/* 473 */ {(16<<2)|3,{48,111,51}},
/* 474 */ {(16<<2)|3,{48,111,52}},
/* 475 */ {(16<<2)|3,{48,111,53}},
/* 476 */ {(16<<2)|3,{48,111,54}},
/* 477 */ {(16<<2)|3,{48,111,55}},
/* 478 */ {(16<<2)|3,{48,111,56}},
/* 479 */ {(16<<2)|3,{48,111,57}},
/* 480 */ {(16<<2)|3,{48,111,61}},
/* 481 */ {(16<<2)|3,{48,111,65}},
/* 482 */ {(16<<2)|3,{48,111,95}},
/* 483 */ {(16<<2)|3,{48,111,98}},
/* 484 */ {(16<<2)|3,{48,111,100}},
/* 485 */ {(16<<2)|3,{48,111,102}},
/* 486 */ {(16<<2)|3,{48,111,103}},
/* 487 */ {(16<<2)|3,{48,111,104}},
/* 488 */ {(16<<2)|3,{48,111,108}},
/* 489 */ {(16<<2)|3,{48,111,109}},
/* 490 */ {(16<<2)|3,{48,111,110}},
/* 491 */ {(16<<2)|3,{48,111,112}},
/* 492 */ {(16<<2)|3,{48,111,114}},
/* 493 */ {(16<<2)|3,{48,111,117}},
/* 494 */ {(10<<2)|2,{48,111,0}},
/* 495 */ {(10<<2)|2,{48,111,0}},
/* 496 */ {(10<<2)|2,{48,111,0}},
/* 497 */ {(10<<2)|2,{48,111,0}},
/* 498 */ {(10<<2)|2,{48,111,0}},
/* 499 */ {(10<<2)|2,{48,111,0}},
/* 500 */ {(10<<2)|2,{48,111,0}},
/* 501 */ {(10<<2)|2,{48,111,0}},
/* 502 */ {(10<<2)|2,{48,111,0}},
/* 503 */ {(10<<2)|2,{48,111,0}},
/* 504 */ {(10<<2)|2,{48,111,0}},
/* 505 */ {(10<<2)|2,{48,111,0}},
/* 506 */ {(10<<2)|2,{48,111,0}},
/* 507 */ {(10<<2)|2,{48,111,0}},
/* 508 */ {(10<<2)|2,{48,111,0}},
/* 509 */ {(10<<2)|2,{48,111,0}},
/* 510 */ {(10<<2)|2,{48,111,0}},
/* 511 */ {(10<<2)|2,{48,111,0}},
/* 512 */ {(15<<2)|3,{48,115,48}},
/* 513 */ {(15<<2)|3,{48,115,48}},
/* 514 */ {(15<<2)|3,{48,115,49}},
/* 515 */ {(15<<2)|3,{48,115,49}},
/* 516 */ {(15<<2)|3,{48,115,50}},
/* 517 */ {(15<<2)|3,{48,115,50}},
/* 518 */ {(15<<2)|3,{48,115,97}},
/* 519 */ {(15<<2)|3,{48,115,97}},
/* 520 */ {(15<<2)|3,{48,115,99}},
/* 521 */ {(15<<2)|3,{48,115,99}},
/* 522 */ {(15<<2)|3,{48,115,101}},
/* 523 */ {(15<<2)|3,{48,115,101}},
/* 524 */ {(15<<2)|3,{48,115,105}},
/* 525 */ {(15<<2)|3,{48,115,105}},
/* 526 */ {(15<<2)|3,{48,115,111}},
/* 527 */ {(15<<2)|3,{48,115,111}},
/* 528 */ {(15<<2)|3,{48,115,115}},
/* 529 */ {(15<<2)|3,{48,115,115}},
/* 530 */ {(15<<2)|3,{48,115,116}},
/* 531 */ {(15<<2)|3,{48,115,116}},
/* 532 */ {(16<<2)|3,{48,115,32}},
/* 533 */ {(16<<2)|3,{48,115,37}},
/* 534 */ {(16<<2)|3,{48,115,45}},
/* 535 */ {(16<<2)|3,{48,115,46}},
/* 536 */ {(16<<2)|3,{48,115,47}},
/* 537 */ {(16<<2)|3,{48,115,51}},
/* 538 */ {(16<<2)|3,{48,115,52}},
/* 539 */ {(16<<2)|3,{48,115,53}},
/* 540 */ {(16<<2)|3,{48,115,54}},
/* 541 */ {(16<<2)|3,{48,115,55}},
/* 542 */ {(16<<2)|3,{48,115,56}},
/* 543 */ {(16<<2)|3,{48,115,57}},
/* 544 */ {(16<<2)|3,{48,115,61}},
/* 545 */ {(16<<2)|3,{48,115,65}},
/* 546 */ {(16<<2)|3,{48,115,95}},
/* 547 */ {(16<<2)|3,{48,115,98}},
/* 548 */ {(16<<2)|3,{48,115,100}},
/* 549 */ {(16<<2)|3,{48,115,102}},
/* 550 */ {(16<<2)|3,{48,115,103}},
/* 551 */ {(16<<2)|3,{48,115,104}},
/* 552 */ {(16<<2)|3,{48,115,108}},
/* 553 */ {(16<<2)|3,{48,115,109}},
/* 554 */ {(16<<2)|3,{48,115,110}},
/* 555 */ {(16<<2)|3,{48,115,112}},
/* 556 */ {(16<<2)|3,{48,115,114}},
/* 557 */ {(16<<2)|3,{48,115,117}},
/* 558 */ {(10<<2)|2,{48,115,0}},
/* 559 */ {(10<<2)|2,{48,115,0}},
/* 560 */ {(10<<2)|2,{48,115,0}},
/* 561 */ {(10<<2)|2,{48,115,0}},
/* 562 */ {(10<<2)|2,{48,115,0}},
/* 563 */ {(10<<2)|2,{48,115,0}},
/* 564 */ {(10<<2)|2,{48,115,0}},
/* 565 */ {(10<<2)|2,{48,115,0}},
/* 566 */ {(10<<2)|2,{48,115,0}},
/* 567 */ {(10<<2)|2,{48,115,0}},
/* 568 */ {(10<<2)|2,{48,115,0}},
/* 569 */ {(10<<2)|2,{48,115,0}},
/* 570 */ {(10<<2)|2,{48,115,0}},
/* 571 */ {(10<<2)|2,{48,115,0}},
/* 572 */ {(10<<2)|2,{48,115,0}},
/* 573 */ {(10<<2)|2,{48,115,0}},
/* 574 */ {(10<<2)|2,{48,115,0}},
/* 575 */ {(10<<2)|2,{48,115,0}},
/* 576 */ {(15<<2)|3,{48,116,48}},
/* 577 */ {(15<<2)|3,{48,116,48}},
/* 578 */ {(15<<2)|3,{48,116,49}},
/* 579 */ {(15<<2)|3,{48,116,49}},
/* 580 */ {(15<<2)|3,{48,116,50}},
/* 581 */ {(15<<2)|3,{48,116,50}},
/* 582 */ {(15<<2)|3,{48,116,97}},
/* 583 */ {(15<<2)|3,{48,116,97}},
/* 584 */ {(15<<2)|3,{48,116,99}},
/* 585 */ {(15<<2)|3,{48,116,99}},
/* 586 */ {(15<<2)|3,{48,116,101}},
/* 587 */ {(15<<2)|3,{48,116,101}},
/* 588 */ {(15<<2)|3,{48,116,105}},
/* 589 */ {(15<<2)|3,{48,116,105}},
/* 590 */ {(15<<2)|3,{48,116,111}},
/* 591 */ {(15<<2)|3,{48,116,111}},
/* 592 */ {(15<<2)|3,{48,116,115}},
/* 593 */ {(15<<2)|3,{48,116,115}},
/* 594 */ {(15<<2)|3,{48,116,116}},
/* 595 */ {(15<<2)|3,{48,116,116}},
/* 596 */ {(16<<2)|3,{48,116,32}},
/* 597 */ {(16<<2)|3,{48,116,37}},
/* 598 */ {(16<<2)|3,{48,116,45}},
/* 599 */ {(16<<2)|3,{48,116,46}},
/* 600 */ {(16<<2)|3,{48,116,47}},
/* 601 */ {(16<<2)|3,{48,116,51}},
/* 602 */ {(16<<2)|3,{48,116,52}},
/* 603 */ {(16<<2)|3,{48,116,53}},
/* 604 */ {(16<<2)|3,{48,116,54}},
/* 605 */ {(16<<2)|3,{48,116,55}},
/* 606 */ {(16<<2)|3,{48,116,56}},
/* 607 */ {(16<<2)|3,{48,116,57}},
/* 608 */ {(16<<2)|3,{48,116,61}},
/* 609 */ {(16<<2)|3,{48,116,65}},
/* 610 */ {(16<<2)|3,{48,116,95}},
/* 611 */ {(16<<2)|3,{48,116,98}},
/* 612 */ {(16<<2)|3,{48,116,100}},
/* 613 */ {(16<<2)|3,{48,116,102}},
/* 614 */ {(16<<2)|3,{48,116,103}},
/* 615 */ {(16<<2)|3,{48,116,104}},
/* 616 */ {(16<<2)|3,{48,116,108}},
/* 617 */ {(16<<2)|3,{48,116,109}},
/* 618 */ {(16<<2)|3,{48,116,110}},
/* 619 */ {(16<<2)|3,{48,116,112}},
/* 620 */ {(16<<2)|3,{48,116,114}},
/* 621 */ {(16<<2)|3,{48,116,117}},
/* 622 */ {(10<<2)|2,{48,116,0}},
/* 623 */ {(10<<2)|2,{48,116,0}},
/* 624 */ {(10<<2)|2,{48,116,0}},
/* 625 */ {(10<<2)|2,{48,116,0}},
/* 626 */ {(10<<2)|2,{48,116,0}},
/* 627 */ {(10<<2)|2,{48,116,0}},
/* 628 */ {(10<<2)|2,{48,116,0}},
/* 629 */ {(10<<2)|2,{48,116,0}},
/* 630 */ {(10<<2)|2,{48,116,0}},
/* 631 */ {(10<<2)|2,{48,116,0}},
/* 632 */ {(10<<2)|2,{48,116,0}},
/* 633 */ {(10<<2)|2,{48,116,0}},
/* 634 */ {(10<<2)|2,{48,116,0}},
/* 635 */ {(10<<2)|2,{48,116,0}},
/* 636 */ {(10<<2)|2,{48,116,0}},
/* 637 */ {(10<<2)|2,{48,116,0}},
/* 638 */ {(10<<2)|2,{48,116,0}},
/* 639 */ {(10<<2)|2,{48,116,0}},
/* 640 */ {(16<<2)|3,{48,32,48}},
/* 641 */ {(16<<2)|3,{48,32,49}},
/* 642 */ {(16<<2)|3,{48,32,50}},
/* 643 */ {(16<<2)|3,{48,32,97}},
/* 644 */ {(16<<2)|3,{48,32,99}},
/* 645 */ {(16<<2)|3,{48,32,101}},
/* 646 */ {(16<<2)|3,{48,32,105}},
/* 647 */ {(16<<2)|3,{48,32,111}},
/* 648 */ {(16<<2)|3,{48,32,115}},
/* 649 */ {(16<<2)|3,{48,32,116}},
/* 650 */ {(11<<2)|2,{48,32,0}},
/* 651 */ {(11<<2)|2,{48,32,0}},
/* 652 */ {(11<<2)|2,{48,32,0}},
/* 653 */ {(11<<2)|2,{48,32,0}},
/* 654 */ {(11<<2)|2,{48,32,0}},
/* 655 */ {(11<<2)|2,{48,32,0}},
/* 656 */ {(11<<2)|2,{48,32,0}},
/* 657 */ {(11<<2)|2,{48,32,0}},
/* 658 */ {(11<<2)|2,{48,32,0}},
/* 659 */ {(11<<2)|2,{48,32,0}},
/* 660 */ {(11<<2)|2,{48,32,0}},
/* 661 */ {(11<<2)|2,{48,32,0}},
/* 662 */ {(11<<2)|2,{48,32,0}},
/* 663 */ {(11<<2)|2,{48,32,0}},
/* 664 */ {(11<<2)|2,{48,32,0}},
/* 665 */ {(11<<2)|2,{48,32,0}},
/* 666 */ {(11<<2)|2,{48,32,0}},
/* 667 */ {(11<<2)|2,{48,32,0}},
/* 668 */ {(11<<2)|2,{48,32,0}},
/* 669 */ {(11<<2)|2,{48,32,0}},
/* 670 */ {(11<<2)|2,{48,32,0}},
/* 671 */ {(11<<2)|2,{48,32,0}},
/* 672 */ {(16<<2)|3,{48,37,48}},
/* 673 */ {(16<<2)|3,{48,37,49}},
/* 674 */ {(16<<2)|3,{48,37,50}},
/* 675 */ {(16<<2)|3,{48,37,97}},
/* 676 */ {(16<<2)|3,{48,37,99}},
/* 677 */ {(16<<2)|3,{48,37,101}},
/* 678 */ {(16<<2)|3,{48,37,105}},
/* 679 */ {(16<<2)|3,{48,37,111}},
/* 680 */ {(16<<2)|3,{48,37,115}},
/* 681 */ {(16<<2)|3,{48,37,116}},
/* 682 */ {(11<<2)|2,{48,37,0}},
/* 683 */ {(11<<2)|2,{48,37,0}},
/* 684 */ {(11<<2)|2,{48,37,0}},
/* 685 */ {(11<<2)|2,{48,37,0}},
/* 686 */ {(11<<2)|2,{48,37,0}},
/* 687 */ {(11<<2)|2,{48,37,0}},
/* 688 */ {(11<<2)|2,{48,37,0}},
/* 689 */ {(11<<2)|2,{48,37,0}},
/* 690 */ {(11<<2)|2,{48,37,0}},
/* 691 */ {(11<<2)|2,{48,37,0}},
/* 692 */ {(11<<2)|2,{48,37,0}},
/* 693 */ {(11<<2)|2,{48,37,0}},
/* 694 */ {(11<<2)|2,{48,37,0}},
/* 695 */ {(11<<2)|2,{48,37,0}},
/* 696 */ {(11<<2)|2,{48,37,0}},
/* 697 */ {(11<<2)|2,{48,37,0}},
/* 698 */ {(11<<2)|2,{48,37,0}},
/* 699 */ {(11<<2)|2,{48,37,0}},
/* 700 */ {(11<<2)|2,{48,37,0}},
/* 701 */ {(11<<2)|2,{48,37,0}},
/* 702 */ {(11<<2)|2,{48,37,0}},
/* 703 */ {(11<<2)|2,{48,37,0}},
/* 704 */ {(16<<2)|3,{48,45,48}},
/* 705 */ {(16<<2)|3,{48,45,49}},
/* 706 */ {(16<<2)|3,{48,45,50}},
/* 707 */ {(16<<2)|3,{48,45,97}},
/* 708 */ {(16<<2)|3,{48,45,99}},
/* 709 */ {(16<<2)|3,{48,45,101}},
/* 710 */ {(16<<2)|3,{48,45,105}},
/* 711 */ {(16<<2)|3,{48,45,111}},
/* 712 */ {(16<<2)|3,{48,45,115}},
/* 713 */ {(16<<2)|3,{48,45,116}},
/* 714 */ {(11<<2)|2,{48,45,0}},
/* 715 */ {(11<<2)|2,{48,45,0}},
/* 716 */ {(11<<2)|2,{48,45,0}},
/* 717 */ {(11<<2)|2,{48,45,0}},
/* 718 */ {(11<<2)|2,{48,45,0}},
/* 719 */ {(11<<2)|2,{48,45,0}},
/* 720 */ {(11<<2)|2,{48,45,0}},
/* 721 */ {(11<<2)|2,{48,45,0}},
/* 722 */ {(11<<2)|2,{48,45,0}},
/* 723 */ {(11<<2)|2,{48,45,0}},
/* 724 */ {(11<<2)|2,{48,45,0}},
/* 725 */ {(11<<2)|2,{48,45,0}},
/* 726 */ {(11<<2)|2,{48,45,0}},
/* 727 */ {(11<<2)|2,{48,45,0}},
/* 728 */ {(11<<2)|2,{48,45,0}},
/* 729 */ {(11<<2)|2,{48,45,0}},
/* 730 */ {(11<<2)|2,{48,45,0}},
/* 731 */ {(11<<2)|2,{48,45,0}},
/* 732 */ {(11<<2)|2,{48,45,0}},
/* 733 */ {(11<<2)|2,{48,45,0}},
/* 734 */ {(11<<2)|2,{48,45,0}},
/* 735 */ {(11<<2)|2,{48,45,0}},
/* 736 */ {(16<<2)|3,{48,46,48}},
/* 737 */ {(16<<2)|3,{48,46,49}},
/* 738 */ {(16<<2)|3,{48,46,50}},
/* 739 */ {(16<<2)|3,{48,46,97}},
/* 740 */ {(16<<2)|3,{48,46,99}},
/* 741 */ {(16<<2)|3,{48,46,101}},
/* 742 */ {(16<<2)|3,{48,46,105}},
/* 743 */ {(16<<2)|3,{48,46,111}},
/* 744 */ {(16<<2)|3,{48,46,115}},
/* 745 */ {(16<<2)|3,{48,46,116}},
/* 746 */ {(11<<2)|2,{48,46,0}},
/* 747 */ {(11<<2)|2,{48,46,0}},
/* 748 */ {(11<<2)|2,{48,46,0}},
/* 749 */ {(11<<2)|2,{48,46,0}},
/* 750 */ {(11<<2)|2,{48,46,0}},
/* 751 */ {(11<<2)|2,{48,46,0}},
/* 752 */ {(11<<2)|2,{48,46,0}},
/* 753 */ {(11<<2)|2,{48,46,0}},
/* 754 */ {(11<<2)|2,{48,46,0}},
/* 755 */ {(11<<2)|2,{48,46,0}},
/* 756 */ {(11<<2)|2,{48,46,0}},
/* 757 */ {(11<<2)|2,{48,46,0}},
/* 758 */ {(11<<2)|2,{48,46,0}},
/* 759 */ {(11<<2)|2,{48,46,0}},
/* 760 */ {(11<<2)|2,{48,46,0}},
/* 761 */ {(11<<2)|2,{48,46,0}},
/* 762 */ {(11<<2)|2,{48,46,0}},
/* 763 */ {(11<<2)|2,{48,46,0}},
/* 764 */ {(11<<2)|2,{48,46,0}},
/* 765 */ {(11<<2)|2,{48,46,0}},
/* 766 */ {(11<<2)|2,{48,46,0}},
/* 767 */ {(11<<2)|2,{48,46,0}},
/* 768 */ {(16<<2)|3,{48,47,48}},
/* 769 */ {(16<<2)|3,{48,47,49}},
/* 770 */ {(16<<2)|3,{48,47,50}},
/* 771 */ {(16<<2)|3,{48,47,97}},
/* 772 */ {(16<<2)|3,{48,47,99}},
/* 773 */ {(16<<2)|3,{48,47,101}},
/* 774 */ {(16<<2)|3,{48,47,105}},
/* 775 */ {(16<<2)|3,{48,47,111}},
/* 776 */ {(16<<2)|3,{48,47,115}},
/* 777 */ {(16<<2)|3,{48,47,116}},
/* 778 */ {(11<<2)|2,{48,47,0}},
/* 779 */ {(11<<2)|2,{48,47,0}},
/* 780 */ {(11<<2)|2,{48,47,0}},
/* 781 */ {(11<<2)|2,{48,47,0}},
/* 782 */ {(11<<2)|2,{48,47,0}},
/* 783 */ {(11<<2)|2,{48,47,0}},
/* 784 */ {(11<<2)|2,{48,47,0}},
/* 785 */ {(11<<2)|2,{48,47,0}},
/* 786 */ {(11<<2)|2,{48,47,0}},
/* 787 */ {(11<<2)|2,{48,47,0}},
/* 788 */ {(11<<2)|2,{48,47,0}},
/* 789 */ {(11<<2)|2,{48,47,0}},
/* 790 */ {(11<<2)|2,{48,47,0}},
/* 791 */ {(11<<2)|2,{48,47,0}},
/* 792 */ {(11<<2)|2,{48,47,0}},
/* 793 */ {(11<<2)|2,{48,47,0}},
/* 794 */ {(11<<2)|2,{48,47,0}},
/* 795 */ {(11<<2)|2,{48,47,0}},
/* 796 */ {(11<<2)|2,{48,47,0}},
/* 797 */ {(11<<2)|2,{48,47,0}},
/* 798 */ {(11<<2)|2,{48,47,0}},
/* 799 */ {(11<<2)|2,{48,47,0}},
/* 800 */ {(16<<2)|3,{48,51,48}},
/* 801 */ {(16<<2)|3,{48,51,49}},
/* 802 */ {(16<<2)|3,{48,51,50}},
/* 803 */ {(16<<2)|3,{48,51,97}},
/* 804 */ {(16<<2)|3,{48,51,99}},
/* 805 */ {(16<<2)|3,{48,51,101}},
/* 806 */ {(16<<2)|3,{48,51,105}},
/* 807 */ {(16<<2)|3,{48,51,111}},
/* 808 */ {(16<<2)|3,{48,51,115}},
/* 809 */ {(16<<2)|3,{48,51,116}},
/* 810 */ {(11<<2)|2,{48,51,0}},
/* 811 */ {(11<<2)|2,{48,51,0}},
/* 812 */ {(11<<2)|2,{48,51,0}},
/* 813 */ {(11<<2)|2,{48,51,0}},
/* 814 */ {(11<<2)|2,{48,51,0}},
/* 815 */ {(11<<2)|2,{48,51,0}},
/* 816 */ {(11<<2)|2,{48,51,0}},
/* 817 */ {(11<<2)|2,{48,51,0}},
/* 818 */ {(11<<2)|2,{48,51,0}},
/* 819 */ {(11<<2)|2,{48,51,0}},
/* 820 */ {(11<<2)|2,{48,51,0}},
/* 821 */ {(11<<2)|2,{48,51,0}},
/* 822 */ {(11<<2)|2,{48,51,0}},
/* 823 */ {(11<<2)|2,{48,51,0}},
/* 824 */ {(11<<2)|2,{48,51,0}},
/* 825 */ {(11<<2)|2,{48,51,0}},
/* 826 */ {(11<<2)|2,{48,51,0}},
/* 827 */ {(11<<2)|2,{48,51,0}},
/* 828 */ {(11<<2)|2,{48,51,0}},
/* 829 */ {(11<<2)|2,{48,51,0}},
/* 830 */ {(11<<2)|2,{48,51,0}},
/* 831 */ {(11<<2)|2,{48,51,0}},
/* 832 */ {(16<<2)|3,{48,52,48}},
/* 833 */ {(16<<2)|3,{48,52,49}},
/* 834 */ {(16<<2)|3,{48,52,50}},
/* 835 */ {(16<<2)|3,{48,52,97}},
/* 836 */ {(16<<2)|3,{48,52,99}},
/* 837 */ {(16<<2)|3,{48,52,101}},
/* 838 */ {(16<<2)|3,{48,52,105}},
/* 839 */ {(16<<2)|3,{48,52,111}},
/* 840 */ {(16<<2)|3,{48,52,115}},
/* 841 */ {(16<<2)|3,{48,52,116}},
/* 842 */ {(11<<2)|2,{48,52,0}},
/* 843 */ {(11<<2)|2,{48,52,0}},
/* 844 */ {(11<<2)|2,{48,52,0}},
/* 845 */ {(11<<2)|2,{48,52,0}},
/* 846 */ {(11<<2)|2,{48,52,0}},
/* 847 */ {(11<<2)|2,{48,52,0}},
/* 848 */ {(11<<2)|2,{48,52,0}},
/* 849 */ {(11<<2)|2,{48,52,0}},
/* 850 */ {(11<<2)|2,{48,52,0}},
/* 851 */ {(11<<2)|2,{48,52,0}},
/* 852 */ {(11<<2)|2,{48,52,0}},
/* 853 */ {(11<<2)|2,{48,52,0}},
/* 854 */ {(11<<2)|2,{48,52,0}},
/* 855 */ {(11<<2)|2,{48,52,0}},
/* 856 */ {(11<<2)|2,{48,52,0}},
/* 857 */ {(11<<2)|2,{48,52,0}},
/* 858 */ {(11<<2)|2,{48,52,0}},
/* 859 */ {(11<<2)|2,{48,52,0}},
/* 860 */ {(11<<2)|2,{48,52,0}},
/* 861 */ {(11<<2)|2,{48,52,0}},
/* 862 */ {(11<<2)|2,{48,52,0}},
/* 863 */ {(11<<2)|2,{48,52,0}},
/* 864 */ {(16<<2)|3,{48,53,48}},
/* 865 */ {(16<<2)|3,{48,53,49}},
/* 866 */ {(16<<2)|3,{48,53,50}},
/* 867 */ {(16<<2)|3,{48,53,97}},
/* 868 */ {(16<<2)|3,{48,53,99}},
/* 869 */ {(16<<2)|3,{48,53,101}},
/* 870 */ {(16<<2)|3,{48,53,105}},
/* 871 */ {(16<<2)|3,{48,53,111}},
/* 872 */ {(16<<2)|3,{48,53,115}},
/* 873 */ {(16<<2)|3,{48,53,116}},
/* 874 */ {(11<<2)|2,{48,53,0}},
/* 875 */ {(11<<2)|2,{48,53,0}},
/* 876 */ {(11<<2)|2,{48,53,0}},
/* 877 */ {(11<<2)|2,{48,53,0}},
/* 878 */ {(11<<2)|2,{48,53,0}},
/* 879 */ {(11<<2)|2,{48,53,0}},
/* 880 */ {(11<<2)|2,{48,53,0}},
/* 881 */ {(11<<2)|2,{48,53,0}},
/* 882 */ {(11<<2)|2,{48,53,0}},
/* 883 */ {(11<<2)|2,{48,53,0}},
/* 884 */ {(11<<2)|2,{48,53,0}},
/* 885 */ {(11<<2)|2,{48,53,0}},
/* 886 */ {(11<<2)|2,{48,53,0}},
/* 887 */ {(11<<2)|2,{48,53,0}},
/* 888 */ {(11<<2)|2,{48,53,0}},
/* 889 */ {(11<<2)|2,{48,53,0}},
/* 890 */ {(11<<2)|2,{48,53,0}},
/* 891 */ {(11<<2)|2,{48,53,0}},
/* 892 */ {(11<<2)|2,{48,53,0}},
/* 893 */ {(11<<2)|2,{48,53,0}},
/* 894 */ {(11<<2)|2,{48,53,0}},
/* 895 */ {(11<<2)|2,{48,53,0}},
/* 896 */ {(16<<2)|3,{48,54,48}},
/* 897 */ {(16<<2)|3,{48,54,49}},
/* 898 */ {(16<<2)|3,{48,54,50}},
/* 899 */ {(16<<2)|3,{48,54,97}},
/* 900 */ {(16<<2)|3,{48,54,99}},
/* 901 */ {(16<<2)|3,{48,54,101}},
/* 902 */ {(16<<2)|3,{48,54,105}},
/* 903 */ {(16<<2)|3,{48,54,111}},
/* 904 */ {(16<<2)|3,{48,54,115}},
/* 905 */ {(16<<2)|3,{48,54,116}},
/* 906 */ {(11<<2)|2,{48,54,0}},
/* 907 */ {(11<<2)|2,{48,54,0}},
/* 908 */ {(11<<2)|2,{48,54,0}},
/* 909 */ {(11<<2)|2,{48,54,0}},
/* 910 */ {(11<<2)|2,{48,54,0}},
/* 911 */ {(11<<2)|2,{48,54,0}},
/* 912 */ {(11<<2)|2,{48,54,0}},
/* 913 */ {(11<<2)|2,{48,54,0}},
/* 914 */ {(11<<2)|2,{48,54,0}},
/* 915 */ {(11<<2)|2,{48,54,0}},
/* 916 */ {(11<<2)|2,{48,54,0}},
/* 917 */ {(11<<2)|2,{48,54,0}},
/* 918 */ {(11<<2)|2,{48,54,0}},
/* 919 */ {(11<<2)|2,{48,54,0}},
/* 920 */ {(11<<2)|2,{48,54,0}},
/* 921 */ {(11<<2)|2,{48,54,0}},
/* 922 */ {(11<<2)|2,{48,54,0}},
/* 923 */ {(11<<2)|2,{48,54,0}},
/* 924 */ {(11<<2)|2,{48,54,0}},
/* 925 */ {(11<<2)|2,{48,54,0}},
/* 926 */ {(11<<2)|2,{48,54,0}},
/* 927 */ {(11<<2)|2,{48,54,0}},
/* 928 */ {(16<<2)|3,{48,55,48}},
/* 929 */ {(16<<2)|3,{48,55,49}},
/* 930 */ {(16<<2)|3,{48,55,50}},
/* 931 */ {(16<<2)|3,{48,55,97}},
/* 932 */ {(16<<2)|3,{48,55,99}},
/* 933 */ {(16<<2)|3,{48,55,101}},
/* 934 */ {(16<<2)|3,{48,55,105}},
/* 935 */ {(16<<2)|3,{48,55,111}},
/* 936 */ {(16<<2)|3,{48,55,115}},
/* 937 */ {(16<<2)|3,{48,55,116}},
/* 938 */ {(11<<2)|2,{48,55,0}},
/* 939 */ {(11<<2)|2,{48,55,0}},
/* 940 */ {(11<<2)|2,{48,55,0}},
/* 941 */ {(11<<2)|2,{48,55,0}},
/* 942 */ {(11<<2)|2,{48,55,0}},
/* 943 */ {(11<<2)|2,{48,55,0}},
/* 944 */ {(11<<2)|2,{48,55,0}},
/* 945 */ {(11<<2)|2,{48,55,0}},
/* 946 */ {(11<<2)|2,{48,55,0}},
/* 947 */ {(11<<2)|2,{48,55,0}},
/* 948 */ {(11<<2)|2,{48,55,0}},
/* 949 */ {(11<<2)|2,{48,55,0}},
/* 950 */ {(11<<2)|2,{48,55,0}},
/* 951 */ {(11<<2)|2,{48,55,0}},
/* 952 */ {(11<<2)|2,{48,55,0}},
/* 953 */ {(11<<2)|2,{48,55,0}},
/* 954 */ {(11<<2)|2,{48,55,0}},
/* 955 */ {(11<<2)|2,{48,55,0}},
/* 956 */ {(11<<2)|2,{48,55,0}},
/* 957 */ {(11<<2)|2,{48,55,0}},
/* 958 */ {(11<<2)|2,{48,55,0}},
/* 959 */ {(11<<2)|2,{48,55,0}},
/* 960 */ {(16<<2)|3,{48,56,48}},
/* 961 */ {(16<<2)|3,{48,56,49}},
/* 962 */ {(16<<2)|3,{48,56,50}},
/* 963 */ {(16<<2)|3,{48,56,97}},
/* 964 */ {(16<<2)|3,{48,56,99}},
/* 965 */ {(16<<2)|3,{48,56,101}},
/* 966 */ {(16<<2)|3,{48,56,105}},
/* 967 */ {(16<<2)|3,{48,56,111}},
/* 968 */ {(16<<2)|3,{48,56,115}},
/* 969 */ {(16<<2)|3,{48,56,116}},
/* 970 */ {(11<<2)|2,{48,56,0}},
/* 971 */ {(11<<2)|2,{48,56,0}},
/* 972 */ {(11<<2)|2,{48,56,0}},
/* 973 */ {(11<<2)|2,{48,56,0}},
/* 974 */ {(11<<2)|2,{48,56,0}},
/* 975 */ {(11<<2)|2,{48,56,0}},
/* 976 */ {(11<<2)|2,{48,56,0}},
/* 977 */ {(11<<2)|2,{48,56,0}},
/* 978 */ {(11<<2)|2,{48,56,0}},
/* 979 */ {(11<<2)|2,{48,56,0}},
/* 980 */ {(11<<2)|2,{48,56,0}},
/* 981 */ {(11<<2)|2,{48,56,0}},
/* 982 */ {(11<<2)|2,{48,56,0}},
/* 983 */ {(11<<2)|2,{48,56,0}},
/* 984 */ {(11<<2)|2,{48,56,0}},
/* 985 */ {(11<<2)|2,{48,56,0}},
/* 986 */ {(11<<2)|2,{48,56,0}},
/* 987 */ {(11<<2)|2,{48,56,0}},
/* 988 */ {(11<<2)|2,{48,56,0}},
/* 989 */ {(11<<2)|2,{48,56,0}},
/* 990 */ {(11<<2)|2,{48,56,0}},
/* 991 */ {(11<<2)|2,{48,56,0}},
/* 992 */ {(16<<2)|3,{48,57,48}},
/* 993 */ {(16<<2)|3,{48,57,49}},
/* 994 */ {(16<<2)|3,{48,57,50}},
/* 995 */ {(16<<2)|3,{48,57,97}},
/* 996 */ {(16<<2)|3,{48,57,99}},
/* 997 */ {(16<<2)|3,{48,57,101}},
/* 998 */ {(16<<2)|3,{48,57,105}},
/* 999 */ {(16<<2)|3,{48,57,111}},
/* 1000 */ {(16<<2)|3,{48,57,115}},
/* 1001 */ {(16<<2)|3,{48,57,116}},
/* 1002 */ {(11<<2)|2,{48,57,0}},
/* 1003 */ {(11<<2)|2,{48,57,0}},
/* 1004 */ {(11<<2)|2,{48,57,0}},
/* 1005 */ {(11<<2)|2,{48,57,0}},
/* 1006 */ {(11<<2)|2,{48,57,0}},
/* 1007 */ {(11<<2)|2,{48,57,0}},
/* 1008 */ {(11<<2)|2,{48,57,0}},
/* 1009 */ {(11<<2)|2,{48,57,0}},
/* 1010 */ {(11<<2)|2,{48,57,0}},
/* 1011 */ {(11<<2)|2,{48,57,0}},
/* 1012 */ {(11<<2)|2,{48,57,0}},
/* 1013 */ {(11<<2)|2,{48,57,0}},
/* 1014 */ {(11<<2)|2,{48,57,0}},
/* 1015 */ {(11<<2)|2,{48,57,0}},
/* 1016 */ {(11<<2)|2,{48,57,0}},
/* 1017 */ {(11<<2)|2,{48,57,0}},
/* 1018 */ {(11<<2)|2,{48,57,0}},
/* 1019 */ {(11<<2)|2,{48,57,0}},
/* 1020 */ {(11<<2)|2,{48,57,0}},
/* 1021 */ {(11<<2)|2,{48,57,0}},
/* 1022 */ {(11<<2)|2,{48,57,0}},
/* 1023 */ {(11<<2)|2,{48,57,0}},
/* 1024 */ {(16<<2)|3,{48,61,48}},
/* 1025 */ {(16<<2)|3,{48,61,49}},
/* 1026 */ {(16<<2)|3,{48,61,50}},
/* 1027 */ {(16<<2)|3,{48,61,97}},
/* 1028 */ {(16<<2)|3,{48,61,99}},
/* 1029 */ {(16<<2)|3,{48,61,101}},
/* 1030 */ {(16<<2)|3,{48,61,105}},
/* 1031 */ {(16<<2)|3,{48,61,111}},
/* 1032 */ {(16<<2)|3,{48,61,115}},
/* 1033 */ {(16<<2)|3,{48,61,116}},
/* 1034 */ {(11<<2)|2,{48,61,0}},
/* 1035 */ {(11<<2)|2,{48,61,0}},
/* 1036 */ {(11<<2)|2,{48,61,0}},
/* 1037 */ {(11<<2)|2,{48,61,0}},
/* 1038 */ {(11<<2)|2,{48,61,0}},
/* 1039 */ {(11<<2)|2,{48,61,0}},
/* 1040 */ {(11<<2)|2,{48,61,0}},
/* 1041 */ {(11<<2)|2,{48,61,0}},
/* 1042 */ {(11<<2)|2,{48,61,0}},
/* 1043 */ {(11<<2)|2,{48,61,0}},
/* 1044 */ {(11<<2)|2,{48,61,0}},
/* 1045 */ {(11<<2)|2,{48,61,0}},
/* 1046 */ {(11<<2)|2,{48,61,0}},
/* 1047 */ {(11<<2)|2,{48,61,0}},
/* 1048 */ {(11<<2)|2,{48,61,0}},
/* 1049 */ {(11<<2)|2,{48,61,0}},
/* 1050 */ {(11<<2)|2,{48,61,0}},
/* 1051 */ {(11<<2)|2,{48,61,0}},
/* 1052 */ {(11<<2)|2,{48,61,0}},
/* 1053 */ {(11<<2)|2,{48,61,0}},
/* 1054 */ {(11<<2)|2,{48,61,0}},
/* 1055 */ {(11<<2)|2,{48,61,0}},
/* 1056 */ {(16<<2)|3,{48,65,48}},
/* 1057 */ {(16<<2)|3,{48,65,49}},
/* 1058 */ {(16<<2)|3,{48,65,50}},
/* 1059 */ {(16<<2)|3,{48,65,97}},
/* 1060 */ {(16<<2)|3,{48,65,99}},
/* 1061 */ {(16<<2)|3,{48,65,101}},
/* 1062 */ {(16<<2)|3,{48,65,105}},
/* 1063 */ {(16<<2)|3,{48,65,111}},
/* 1064 */ {(16<<2)|3,{48,65,115}},
/* 1065 */ {(16<<2)|3,{48,65,116}},
/* 1066 */ {(11<<2)|2,{48,65,0}},
/* 1067 */ {(11<<2)|2,{48,65,0}},
/* 1068 */ {(11<<2)|2,{48,65,0}},
/* 1069 */ {(11<<2)|2,{48,65,0}},
/* 1070 */ {(11<<2)|2,{48,65,0}},
/* 1071 */ {(11<<2)|2,{48,65,0}},
/* 1072 */ {(11<<2)|2,{48,65,0}},
/* 1073 */ {(11<<2)|2,{48,65,0}},
/* 1074 */ {(11<<2)|2,{48,65,0}},
/* 1075 */ {(11<<2)|2,{48,65,0}},
/* 1076 */ {(11<<2)|2,{48,65,0}},
/* 1077 */ {(11<<2)|2,{48,65,0}},
/* 1078 */ {(11<<2)|2,{48,65,0}},
/* 1079 */ {(11<<2)|2,{48,65,0}},
/* 1080 */ {(11<<2)|2,{48,65,0}},
/* 1081 */ {(11<<2)|2,{48,65,0}},
/* 1082 */ {(11<<2)|2,{48,65,0}},
/* 1083 */ {(11<<2)|2,{48,65,0}},
/* 1084 */ {(11<<2)|2,{48,65,0}},
/* 1085 */ {(11<<2)|2,{48,65,0}},
/* 1086 */ {(11<<2)|2,{48,65,0}},
/* 1087 */ {(11<<2)|2,{48,65,0}},
/* 1088 */ {(16<<2)|3,{48,95,48}},
/* 1089 */ {(16<<2)|3,{48,95,49}},
/* 1090 */ {(16<<2)|3,{48,95,50}},
/* 1091 */ {(16<<2)|3,{48,95,97}},
/* 1092 */ {(16<<2)|3,{48,95,99}},
/* 1093 */ {(16<<2)|3,{48,95,101}},
/* 1094 */ {(16<<2)|3,{48,95,105}},
/* 1095 */ {(16<<2)|3,{48,95,111}},
/* 1096 */ {(16<<2)|3,{48,95,115}},
/* 1097 */ {(16<<2)|3,{48,95,116}},
/* 1098 */ {(11<<2)|2,{48,95,0}},
/* 1099 */ {(11<<2)|2,{48,95,0}},
/* 1100 */ {(11<<2)|2,{48,95,0}},
/* 1101 */ {(11<<2)|2,{48,95,0}},
/* 1102 */ {(11<<2)|2,{48,95,0}},
/* 1103 */ {(11<<2)|2,{48,95,0}},
/* 1104 */ {(11<<2)|2,{48,95,0}},
/* 1105 */ {(11<<2)|2,{48,95,0}},
/* 1106 */ {(11<<2)|2,{48,95,0}},
/* 1107 */ {(11<<2)|2,{48,95,0}},
/* 1108 */ {(11<<2)|2,{48,95,0}},
/* 1109 */ {(11<<2)|2,{48,95,0}},
/* 1110 */ {(11<<2)|2,{48,95,0}},
/* 1111 */ {(11<<2)|2,{48,95,0}},
/* 1112 */ {(11<<2)|2,{48,95,0}},
/* 1113 */ {(11<<2)|2,{48,95,0}},
/* 1114 */ {(11<<2)|2,{48,95,0}},
/* 1115 */ {(11<<2)|2,{48,95,0}},
/* 1116 */ {(11<<2)|2,{48,95,0}},
/* 1117 */ {(11<<2)|2,{48,95,0}},
/* 1118 */ {(11<<2)|2,{48,95,0}},
/* 1119 */ {(11<<2)|2,{48,95,0}},
/* 1120 */ {(16<<2)|3,{48,98,48}},
/* 1121 */ {(16<<2)|3,{48,98,49}},
/* 1122 */ {(16<<2)|3,{48,98,50}},
/* 1123 */ {(16<<2)|3,{48,98,97}},
/* 1124 */ {(16<<2)|3,{48,98,99}},
/* 1125 */ {(16<<2)|3,{48,98,101}},
/* 1126 */ {(16<<2)|3,{48,98,105}},
/* 1127 */ {(16<<2)|3,{48,98,111}},
/* 1128 */ {(16<<2)|3,{48,98,115}},
/* 1129 */ {(16<<2)|3,{48,98,116}},
/* 1130 */ {(11<<2)|2,{48,98,0}},
/* 1131 */ {(11<<2)|2,{48,98,0}},
/* 1132 */ {(11<<2)|2,{48,98,0}},
/* 1133 */ {(11<<2)|2,{48,98,0}},
/* 1134 */ {(11<<2)|2,{48,98,0}},
/* 1135 */ {(11<<2)|2,{48,98,0}},
/* 1136 */ {(11<<2)|2,{48,98,0}},
/* 1137 */ {(11<<2)|2,{48,98,0}},
/* 1138 */ {(11<<2)|2,{48,98,0}},
/* 1139 */ {(11<<2)|2,{48,98,0}},
/* 1140 */ {(11<<2)|2,{48,98,0}},
/* 1141 */ {(11<<2)|2,{48,98,0}},
/* 1142 */ {(11<<2)|2,{48,98,0}},
/* 1143 */ {(11<<2)|2,{48,98,0}},
/* 1144 */ {(11<<2)|2,{48,98,0}},
/* 1145 */ {(11<<2)|2,{48,98,0}},
/* 1146 */ {(11<<2)|2,{48,98,0}},
/* 1147 */ {(11<<2)|2,{48,98,0}},
/* 1148 */ {(11<<2)|2,{48,98,0}},
/* 1149 */ {(11<<2)|2,{48,98,0}},
/* 1150 */ {(11<<2)|2,{48,98,0}},
/* 1151 */ {(11<<2)|2,{48,98,0}},
/* 1152 */ {(16<<2)|3,{48,100,48}},
/* 1153 */ {(16<<2)|3,{48,100,49}},
/* 1154 */ {(16<<2)|3,{48,100,50}},
/* 1155 */ {(16<<2)|3,{48,100,97}},
/* 1156 */ {(16<<2)|3,{48,100,99}},
/* 1157 */ {(16<<2)|3,{48,100,101}},
/* 1158 */ {(16<<2)|3,{48,100,105}},
/* 1159 */ {(16<<2)|3,{48,100,111}},
/* 1160 */ {(16<<2)|3,{48,100,115}},
/* 1161 */ {(16<<2)|3,{48,100,116}},
/* 1162 */ {(11<<2)|2,{48,100,0}},
/* 1163 */ {(11<<2)|2,{48,100,0}},
/* 1164 */ {(11<<2)|2,{48,100,0}},
/* 1165 */ {(11<<2)|2,{48,100,0}},
/* 1166 */ {(11<<2)|2,{48,100,0}},
/* 1167 */ {(11<<2)|2,{48,100,0}},
/* 1168 */ {(11<<2)|2,{48,100,0}},
/* 1169 */ {(11<<2)|2,{48,100,0}},
/* 1170 */ {(11<<2)|2,{48,100,0}},
/* 1171 */ {(11<<2)|2,{48,100,0}},
/* 1172 */ {(11<<2)|2,{48,100,0}},
/* 1173 */ {(11<<2)|2,{48,100,0}},
/* 1174 */ {(11<<2)|2,{48,100,0}},
/* 1175 */ {(11<<2)|2,{48,100,0}},
/* 1176 */ {(11<<2)|2,{48,100,0}},
/* 1177 */ {(11<<2)|2,{48,100,0}},
/* 1178 */ {(11<<2)|2,{48,100,0}},
/* 1179 */ {(11<<2)|2,{48,100,0}},
/* 1180 */ {(11<<2)|2,{48,100,0}},
/* 1181 */ {(11<<2)|2,{48,100,0}},
/* 1182 */ {(11<<2)|2,{48,100,0}},
/* 1183 */ {(11<<2)|2,{48,100,0}},
/* 1184 */ {(16<<2)|3,{48,102,48}},
/* 1185 */ {(16<<2)|3,{48,102,49}},
/* 1186 */ {(16<<2)|3,{48,102,50}},
/* 1187 */ {(16<<2)|3,{48,102,97}},
/* 1188 */ {(16<<2)|3,{48,102,99}},
/* 1189 */ {(16<<2)|3,{48,102,101}},
/* 1190 */ {(16<<2)|3,{48,102,105}},
/* 1191 */ {(16<<2)|3,{48,102,111}},
/* 1192 */ {(16<<2)|3,{48,102,115}},
/* 1193 */ {(16<<2)|3,{48,102,116}},
/* 1194 */ {(11<<2)|2,{48,102,0}},
/* 1195 */ {(11<<2)|2,{48,102,0}},
/* 1196 */ {(11<<2)|2,{48,102,0}},
/* 1197 */ {(11<<2)|2,{48,102,0}},
/* 1198 */ {(11<<2)|2,{48,102,0}},
/* 1199 */ {(11<<2)|2,{48,102,0}},
/* 1200 */ {(11<<2)|2,{48,102,0}},
/* 1201 */ {(11<<2)|2,{48,102,0}},
/* 1202 */ {(11<<2)|2,{48,102,0}},
/* 1203 */ {(11<<2)|2,{48,102,0}},
/* 1204 */ {(11<<2)|2,{48,102,0}},
/* 1205 */ {(11<<2)|2,{48,102,0}},
/* 1206 */ {(11<<2)|2,{48,102,0}},
/* 1207 */ {(11<<2)|2,{48,102,0}},
/* 1208 */ {(11<<2)|2,{48,102,0}},
/* 1209 */ {(11<<2)|2,{48,102,0}},
/* 1210 */ {(11<<2)|2,{48,102,0}},
/* 1211 */ {(11<<2)|2,{48,102,0}},
/* 1212 */ {(11<<2)|2,{48,102,0}},
/* 1213 */ {(11<<2)|2,{48,102,0}},
/* 1214 */ {(11<<2)|2,{48,102,0}},
/* 1215 */ {(11<<2)|2,{48,102,0}},
/* 1216 */ {(16<<2)|3,{48,103,48}},
/* 1217 */ {(16<<2)|3,{48,103,49}},
/* 1218 */ {(16<<2)|3,{48,103,50}},
/* 1219 */ {(16<<2)|3,{48,103,97}},
/* 1220 */ {(16<<2)|3,{48,103,99}},
/* 1221 */ {(16<<2)|3,{48,103,101}},
/* 1222 */ {(16<<2)|3,{48,103,105}},
/* 1223 */ {(16<<2)|3,{48,103,111}},
/* 1224 */ {(16<<2)|3,{48,103,115}},
/* 1225 */ {(16<<2)|3,{48,103,116}},
/* 1226 */ {(11<<2)|2,{48,103,0}},
/* 1227 */ {(11<<2)|2,{48,103,0}},
/* 1228 */ {(11<<2)|2,{48,103,0}},
/* 1229 */ {(11<<2)|2,{48,103,0}},
/* 1230 */ {(11<<2)|2,{48,103,0}},
/* 1231 */ {(11<<2)|2,{48,103,0}},
/* 1232 */ {(11<<2)|2,{48,103,0}},
/* 1233 */ {(11<<2)|2,{48,103,0}},
/* 1234 */ {(11<<2)|2,{48,103,0}},
/* 1235 */ {(11<<2)|2,{48,103,0}},
/* 1236 */ {(11<<2)|2,{48,103,0}},
/* 1237 */ {(11<<2)|2,{48,103,0}},
/* 1238 */ {(11<<2)|2,{48,103,0}},
/* 1239 */ {(11<<2)|2,{48,103,0}},
/* 1240 */ {(11<<2)|2,{48,103,0}},
/* 1241 */ {(11<<2)|2,{48,103,0}},
/* 1242 */ {(11<<2)|2,{48,103,0}},
/* 1243 */ {(11<<2)|2,{48,103,0}},
/* 1244 */ {(11<<2)|2,{48,103,0}},
/* 1245 */ {(11<<2)|2,{48,103,0}},
/* 1246 */ {(11<<2)|2,{48,103,0}},
/* 1247 */ {(11<<2)|2,{48,103,0}},
/* 1248 */ {(16<<2)|3,{48,104,48}},
/* 1249 */ {(16<<2)|3,{48,104,49}},
/* 1250 */ {(16<<2)|3,{48,104,50}},
/* 1251 */ {(16<<2)|3,{48,104,97}},
/* 1252 */ {(16<<2)|3,{48,104,99}},
/* 1253 */ {(16<<2)|3,{48,104,101}},
/* 1254 */ {(16<<2)|3,{48,104,105}},
/* 1255 */ {(16<<2)|3,{48,104,111}},
/* 1256 */ {(16<<2)|3,{48,104,115}},
/* 1257 */ {(16<<2)|3,{48,104,116}},
/* 1258 */ {(11<<2)|2,{48,104,0}},
/* 1259 */ {(11<<2)|2,{48,104,0}},
/* 1260 */ {(11<<2)|2,{48,104,0}},
/* 1261 */ {(11<<2)|2,{48,104,0}},
/* 1262 */ {(11<<2)|2,{48,104,0}},
/* 1263 */ {(11<<2)|2,{48,104,0}},
/* 1264 */ {(11<<2)|2,{48,104,0}},
/* 1265 */ {(11<<2)|2,{48,104,0}},
/* 1266 */ {(11<<2)|2,{48,104,0}},
/* 1267 */ {(11<<2)|2,{48,104,0}},
/* 1268 */ {(11<<2)|2,{48,104,0}},
/* 1269 */ {(11<<2)|2,{48,104,0}},
/* 1270 */ {(11<<2)|2,{48,104,0}},
/* 1271 */ {(11<<2)|2,{48,104,0}},
/* 1272 */ {(11<<2)|2,{48,104,0}},
/* 1273 */ {(11<<2)|2,{48,104,0}},
/* 1274 */ {(11<<2)|2,{48,104,0}},
/* 1275 */ {(11<<2)|2,{48,104,0}},
/* 1276 */ {(11<<2)|2,{48,104,0}},
/* 1277 */ {(11<<2)|2,{48,104,0}},
/* 1278 */ {(11<<2)|2,{48,104,0}},
/* 1279 */ {(11<<2)|2,{48,104,0}},
/* 1280 */ {(16<<2)|3,{48,108,48}},
/* 1281 */ {(16<<2)|3,{48,108,49}},
/* 1282 */ {(16<<2)|3,{48,108,50}},
/* 1283 */ {(16<<2)|3,{48,108,97}},
/* 1284 */ {(16<<2)|3,{48,108,99}},
/* 1285 */ {(16<<2)|3,{48,108,101}},
/* 1286 */ {(16<<2)|3,{48,108,105}},
/* 1287 */ {(16<<2)|3,{48,108,111}},
/* 1288 */ {(16<<2)|3,{48,108,115}},
/* 1289 */ {(16<<2)|3,{48,108,116}},
/* 1290 */ {(11<<2)|2,{48,108,0}},
/* 1291 */ {(11<<2)|2,{48,108,0}},
/* 1292 */ {(11<<2)|2,{48,108,0}},
/* 1293 */ {(11<<2)|2,{48,108,0}},
/* 1294 */ {(11<<2)|2,{48,108,0}},
/* 1295 */ {(11<<2)|2,{48,108,0}},
/* 1296 */ {(11<<2)|2,{48,108,0}},
/* 1297 */ {(11<<2)|2,{48,108,0}},
/* 1298 */ {(11<<2)|2,{48,108,0}},
/* 1299 */ {(11<<2)|2,{48,108,0}},
/* 1300 */ {(11<<2)|2,{48,108,0}},
/* 1301 */ {(11<<2)|2,{48,108,0}},
/* 1302 */ {(11<<2)|2,{48,108,0}},
/* 1303 */ {(11<<2)|2,{48,108,0}},
/* 1304 */ {(11<<2)|2,{48,108,0}},
/* 1305 */ {(11<<2)|2,{48,108,0}},
/* 1306 */ {(11<<2)|2,{48,108,0}},
/* 1307 */ {(11<<2)|2,{48,108,0}},
/* 1308 */ {(11<<2)|2,{48,108,0}},
/* 1309 */ {(11<<2)|2,{48,108,0}},
/* 1310 */ {(11<<2)|2,{48,108,0}},
/* 1311 */ {(11<<2)|2,{48,108,0}},
/* 1312 */ {(16<<2)|3,{48,109,48}},
/* 1313 */ {(16<<2)|3,{48,109,49}},
/* 1314 */ {(16<<2)|3,{48,109,50}},
/* 1315 */ {(16<<2)|3,{48,109,97}},
/* 1316 */ {(16<<2)|3,{48,109,99}},
/* 1317 */ {(16<<2)|3,{48,109,101}},
/* 1318 */ {(16<<2)|3,{48,109,105}},
/* 1319 */ {(16<<2)|3,{48,109,111}},
/* 1320 */ {(16<<2)|3,{48,109,115}},
/* 1321 */ {(16<<2)|3,{48,109,116}},
/* 1322 */ {(11<<2)|2,{48,109,0}},
/* 1323 */ {(11<<2)|2,{48,109,0}},
/* 1324 */ {(11<<2)|2,{48,109,0}},
/* 1325 */ {(11<<2)|2,{48,109,0}},
/* 1326 */ {(11<<2)|2,{48,109,0}},
/* 1327 */ {(11<<2)|2,{48,109,0}},
/* 1328 */ {(11<<2)|2,{48,109,0}},
/* 1329 */ {(11<<2)|2,{48,109,0}},
/* 1330 */ {(11<<2)|2,{48,109,0}},
/* 1331 */ {(11<<2)|2,{48,109,0}},
/* 1332 */ {(11<<2)|2,{48,109,0}},
/* 1333 */ {(11<<2)|2,{48,109,0}},
/* 1334 */ {(11<<2)|2,{48,109,0}},
/* 1335 */ {(11<<2)|2,{48,109,0}},
/* 1336 */ {(11<<2)|2,{48,109,0}},
/* 1337 */ {(11<<2)|2,{48,109,0}},
/* 1338 */ {(11<<2)|2,{48,109,0}},
/* 1339 */ {(11<<2)|2,{48,109,0}},
/* 1340 */ {(11<<2)|2,{48,109,0}},
/* 1341 */ {(11<<2)|2,{48,109,0}},
/* 1342 */ {(11<<2)|2,{48,109,0}},
/* 1343 */ {(11<<2)|2,{48,109,0}},
/* 1344 */ {(16<<2)|3,{48,110,48}},
/* 1345 */ {(16<<2)|3,{48,110,49}},
/* 1346 */ {(16<<2)|3,{48,110,50}},
/* 1347 */ {(16<<2)|3,{48,110,97}},
/* 1348 */ {(16<<2)|3,{48,110,99}},
/* 1349 */ {(16<<2)|3,{48,110,101}},
/* 1350 */ {(16<<2)|3,{48,110,105}},
/* 1351 */ {(16<<2)|3,{48,110,111}},
/* 1352 */ {(16<<2)|3,{48,110,115}},
/* 1353 */ {(16<<2)|3,{48,110,116}},
/* 1354 */ {(11<<2)|2,{48,110,0}},
/* 1355 */ {(11<<2)|2,{48,110,0}},
/* 1356 */ {(11<<2)|2,{48,110,0}},
/* 1357 */ {(11<<2)|2,{48,110,0}},
/* 1358 */ {(11<<2)|2,{48,110,0}},
/* 1359 */ {(11<<2)|2,{48,110,0}},
/* 1360 */ {(11<<2)|2,{48,110,0}},
/* 1361 */ {(11<<2)|2,{48,110,0}},
/* 1362 */ {(11<<2)|2,{48,110,0}},
/* 1363 */ {(11<<2)|2,{48,110,0}},
/* 1364 */ {(11<<2)|2,{48,110,0}},
/* 1365 */ {(11<<2)|2,{48,110,0}},
/* 1366 */ {(11<<2)|2,{48,110,0}},
/* 1367 */ {(11<<2)|2,{48,110,0}},
/* 1368 */ {(11<<2)|2,{48,110,0}},
/* 1369 */ {(11<<2)|2,{48,110,0}},
/* 1370 */ {(11<<2)|2,{48,110,0}},
/* 1371 */ {(11<<2)|2,{48,110,0}},
/* 1372 */ {(11<<2)|2,{48,110,0}},
/* 1373 */ {(11<<2)|2,{48,110,0}},
/* 1374 */ {(11<<2)|2,{48,110,0}},
/* 1375 */ {(11<<2)|2,{48,110,0}},
/* 1376 */ {(16<<2)|3,{48,112,48}},
/* 1377 */ {(16<<2)|3,{48,112,49}},
/* 1378 */ {(16<<2)|3,{48,112,50}},
/* 1379 */ {(16<<2)|3,{48,112,97}},
/* 1380 */ {(16<<2)|3,{48,112,99}},
/* 1381 */ {(16<<2)|3,{48,112,101}},
/* 1382 */ {(16<<2)|3,{48,112,105}},
/* 1383 */ {(16<<2)|3,{48,112,111}},
/* 1384 */ {(16<<2)|3,{48,112,115}},
/* 1385 */ {(16<<2)|3,{48,112,116}},
/* 1386 */ {(11<<2)|2,{48,112,0}},
/* 1387 */ {(11<<2)|2,{48,112,0}},
/* 1388 */ {(11<<2)|2,{48,112,0}},
/* 1389 */ {(11<<2)|2,{48,112,0}},
/* 1390 */ {(11<<2)|2,{48,112,0}},
/* 1391 */ {(11<<2)|2,{48,112,0}},
/* 1392 */ {(11<<2)|2,{48,112,0}},
/* 1393 */ {(11<<2)|2,{48,112,0}},
/* 1394 */ {(11<<2)|2,{48,112,0}},
/* 1395 */ {(11<<2)|2,{48,112,0}},
/* 1396 */ {(11<<2)|2,{48,112,0}},
/* 1397 */ {(11<<2)|2,{48,112,0}},
/* 1398 */ {(11<<2)|2,{48,112,0}},
/* 1399 */ {(11<<2)|2,{48,112,0}},
/* 1400 */ {(11<<2)|2,{48,112,0}},
/* 1401 */ {(11<<2)|2,{48,112,0}},
/* 1402 */ {(11<<2)|2,{48,112,0}},
/* 1403 */ {(11<<2)|2,{48,112,0}},
/* 1404 */ {(11<<2)|2,{48,112,0}},
/* 1405 */ {(11<<2)|2,{48,112,0}},
/* 1406 */ {(11<<2)|2,{48,112,0}},
/* 1407 */ {(11<<2)|2,{48,112,0}},
/* 1408 */ {(16<<2)|3,{48,114,48}},
/* 1409 */ {(16<<2)|3,{48,114,49}},
/* 1410 */ {(16<<2)|3,{48,114,50}},
/* 1411 */ {(16<<2)|3,{48,114,97}},
/* 1412 */ {(16<<2)|3,{48,114,99}},
/* 1413 */ {(16<<2)|3,{48,114,101}},
/* 1414 */ {(16<<2)|3,{48,114,105}},
/* 1415 */ {(16<<2)|3,{48,114,111}},
/* 1416 */ {(16<<2)|3,{48,114,115}},
/* 1417 */ {(16<<2)|3,{48,114,116}},
/* 1418 */ {(11<<2)|2,{48,114,0}},
/* 1419 */ {(11<<2)|2,{48,114,0}},
/* 1420 */ {(11<<2)|2,{48,114,0}},
/* 1421 */ {(11<<2)|2,{48,114,0}},
/* 1422 */ {(11<<2)|2,{48,114,0}},
/* 1423 */ {(11<<2)|2,{48,114,0}},
/* 1424 */ {(11<<2)|2,{48,114,0}},
/* 1425 */ {(11<<2)|2,{48,114,0}},
/* 1426 */ {(11<<2)|2,{48,114,0}},
/* 1427 */ {(11<<2)|2,{48,114,0}},
/* 1428 */ {(11<<2)|2,{48,114,0}},
/* 1429 */ {(11<<2)|2,{48,114,0}},
/* 1430 */ {(11<<2)|2,{48,114,0}},
/* 1431 */ {(11<<2)|2,{48,114,0}},
/* 1432 */ {(11<<2)|2,{48,114,0}},
/* 1433 */ {(11<<2)|2,{48,114,0}},
/* 1434 */ {(11<<2)|2,{48,114,0}},
/* 1435 */ {(11<<2)|2,{48,114,0}},
/* 1436 */ {(11<<2)|2,{48,114,0}},
/* 1437 */ {(11<<2)|2,{48,114,0}},
/* 1438 */ {(11<<2)|2,{48,114,0}},
/* 1439 */ {(11<<2)|2,{48,114,0}},
/* 1440 */ {(16<<2)|3,{48,117,48}},
/* 1441 */ {(16<<2)|3,{48,117,49}},
/* 1442 */ {(16<<2)|3,{48,117,50}},
/* 1443 */ {(16<<2)|3,{48,117,97}},
/* 1444 */ {(16<<2)|3,{48,117,99}},
/* 1445 */ {(16<<2)|3,{48,117,101}},
/* 1446 */ {(16<<2)|3,{48,117,105}},
/* 1447 */ {(16<<2)|3,{48,117,111}},
/* 1448 */ {(16<<2)|3,{48,117,115}},
/* 1449 */ {(16<<2)|3,{48,117,116}},
/* 1450 */ {(11<<2)|2,{48,117,0}},
/* 1451 */ {(11<<2)|2,{48,117,0}},
/* 1452 */ {(11<<2)|2,{48,117,0}},
/* 1453 */ {(11<<2)|2,{48,117,0}},
/* 1454 */ {(11<<2)|2,{48,117,0}},
/* 1455 */ {(11<<2)|2,{48,117,0}},
/* 1456 */ {(11<<2)|2,{48,117,0}},
/* 1457 */ {(11<<2)|2,{48,117,0}},
/* 1458 */ {(11<<2)|2,{48,117,0}},
/* 1459 */ {(11<<2)|2,{48,117,0}},
/* 1460 */ {(11<<2)|2,{48,117,0}},
/* 1461 */ {(11<<2)|2,{48,117,0}},
/* 1462 */ {(11<<2)|2,{48,117,0}},
/* 1463 */ {(11<<2)|2,{48,117,0}},
/* 1464 */ {(11<<2)|2,{48,117,0}},
/* 1465 */ {(11<<2)|2,{48,117,0}},
/* 1466 */ {(11<<2)|2,{48,117,0}},
/* 1467 */ {(11<<2)|2,{48,117,0}},
/* 1468 */ {(11<<2)|2,{48,117,0}},
/* 1469 */ {(11<<2)|2,{48,117,0}},
/* 1470 */ {(11<<2)|2,{48,117,0}},
/* 1471 */ {(11<<2)|2,{48,117,0}},
/* 1472 */ {(12<<2)|2,{48,58,0}},
/* 1473 */ {(12<<2)|2,{48,58,0}},
/* 1474 */ {(12<<2)|2,{48,58,0}},
/* 1475 */ {(12<<2)|2,{48,58,0}},
/* 1476 */ {(12<<2)|2,{48,58,0}},
/* 1477 */ {(12<<2)|2,{48,58,0}},
/* 1478 */ {(12<<2)|2,{48,58,0}},
/* 1479 */ {(12<<2)|2,{48,58,0}},
/* 1480 */ {(12<<2)|2,{48,58,0}},
/* 1481 */ {(12<<2)|2,{48,58,0}},
/* 1482 */ {(12<<2)|2,{48,58,0}},
/* 1483 */ {(12<<2)|2,{48,58,0}},
/* 1484 */ {(12<<2)|2,{48,58,0}},
/* 1485 */ {(12<<2)|2,{48,58,0}},
/* 1486 */ {(12<<2)|2,{48,58,0}},
/* 1487 */ {(12<<2)|2,{48,58,0}},
/* 1488 */ {(12<<2)|2,{48,66,0}},
/* 1489 */ {(12<<2)|2,{48,66,0}},
/* 1490 */ {(12<<2)|2,{48,66,0}},
/* 1491 */ {(12<<2)|2,{48,66,0}},
/* 1492 */ {(12<<2)|2,{48,66,0}},
/* 1493 */ {(12<<2)|2,{48,66,0}},
/* 1494 */ {(12<<2)|2,{48,66,0}},
/* 1495 */ {(12<<2)|2,{48,66,0}},
/* 1496 */ {(12<<2)|2,{48,66,0}},
/* 1497 */ {(12<<2)|2,{48,66,0}},
/* 1498 */ {(12<<2)|2,{48,66,0}},
/* 1499 */ {(12<<2)|2,{48,66,0}},
/* 1500 */ {(12<<2)|2,{48,66,0}},
/* 1501 */ {(12<<2)|2,{48,66,0}},
/* 1502 */ {(12<<2)|2,{48,66,0}},
/* 1503 */ {(12<<2)|2,{48,66,0}},
/* 1504 */ {(12<<2)|2,{48,67,0}},
/* 1505 */ {(12<<2)|2,{48,67,0}},
/* 1506 */ {(12<<2)|2,{48,67,0}},
/* 1507 */ {(12<<2)|2,{48,67,0}},
/* 1508 */ {(12<<2)|2,{48,67,0}},
/* 1509 */ {(12<<2)|2,{48,67,0}},
/* 1510 */ {(12<<2)|2,{48,67,0}},
/* 1511 */ {(12<<2)|2,{48,67,0}},
/* 1512 */ {(12<<2)|2,{48,67,0}},
/* 1513 */ {(12<<2)|2,{48,67,0}},
/* 1514 */ {(12<<2)|2,{48,67,0}},
/* 1515 */ {(12<<2)|2,{48,67,0}},
/* 1516 */ {(12<<2)|2,{48,67,0}},
/* 1517 */ {(12<<2)|2,{48,67,0}},
/* 1518 */ {(12<<2)|2,{48,67,0}},
/* 1519 */ {(12<<2)|2,{48,67,0}},
/* 1520 */ {(12<<2)|2,{48,68,0}},
/* 1521 */ {(12<<2)|2,{48,68,0}},
/* 1522 */ {(12<<2)|2,{48,68,0}},
/* 1523 */ {(12<<2)|2,{48,68,0}},
/* 1524 */ {(12<<2)|2,{48,68,0}},
/* 1525 */ {(12<<2)|2,{48,68,0}},
/* 1526 */ {(12<<2)|2,{48,68,0}},
/* 1527 */ {(12<<2)|2,{48,68,0}},
/* 1528 */ {(12<<2)|2,{48,68,0}},
/* 1529 */ {(12<<2)|2,{48,68,0}},
/* 1530 */ {(12<<2)|2,{48,68,0}},
/* 1531 */ {(12<<2)|2,{48,68,0}},
/* 1532 */ {(12<<2)|2,{48,68,0}},
/* 1533 */ {(12<<2)|2,{48,68,0}},
/* 1534 */ {(12<<2)|2,{48,68,0}},
/* 1535 */ {(12<<2)|2,{48,68,0}},
/* 1536 */ {(12<<2)|2,{48,69,0}},
/* 1537 */ {(12<<2)|2,{48,69,0}},
/* 1538 */ {(12<<2)|2,{48,69,0}},
/* 1539 */ {(12<<2)|2,{48,69,0}},
/* 1540 */ {(12<<2)|2,{48,69,0}},
/* 1541 */ {(12<<2)|2,{48,69,0}},
/* 1542 */ {(12<<2)|2,{48,69,0}},
/* 1543 */ {(12<<2)|2,{48,69,0}},
/* 1544 */ {(12<<2)|2,{48,69,0}},
/* 1545 */ {(12<<2)|2,{48,69,0}},
/* 1546 */ {(12<<2)|2,{48,69,0}},
/* 1547 */ {(12<<2)|2,{48,69,0}},
/* 1548 */ {(12<<2)|2,{48,69,0}},
/* 1549 */ {(12<<2)|2,{48,69,0}},
/* 1550 */ {(12<<2)|2,{48,69,0}},
/* 1551 */ {(12<<2)|2,{48,69,0}},
/* 1552 */ {(12<<2)|2,{48,70,0}},
/* 1553 */ {(12<<2)|2,{48,70,0}},
/* 1554 */ {(12<<2)|2,{48,70,0}},
/* 1555 */ {(12<<2)|2,{48,70,0}},
/* 1556 */ {(12<<2)|2,{48,70,0}},
/* 1557 */ {(12<<2)|2,{48,70,0}},
/* 1558 */ {(12<<2)|2,{48,70,0}},
/* 1559 */ {(12<<2)|2,{48,70,0}},
/* 1560 */ {(12<<2)|2,{48,70,0}},
/* 1561 */ {(12<<2)|2,{48,70,0}},
/* 1562 */ {(12<<2)|2,{48,70,0}},
/* 1563 */ {(12<<2)|2,{48,70,0}},
/* 1564 */ {(12<<2)|2,{48,70,0}},
/* 1565 */ {(12<<2)|2,{48,70,0}},
/* 1566 */ {(12<<2)|2,{48,70,0}},
/* 1567 */ {(12<<2)|2,{48,70,0}},
/* 1568 */ {(12<<2)|2,{48,71,0}},
/* 1569 */ {(12<<2)|2,{48,71,0}},
/* 1570 */ {(12<<2)|2,{48,71,0}},
/* 1571 */ {(12<<2)|2,{48,71,0}},
/* 1572 */ {(12<<2)|2,{48,71,0}},
/* 1573 */ {(12<<2)|2,{48,71,0}},
/* 1574 */ {(12<<2)|2,{48,71,0}},
/* 1575 */ {(12<<2)|2,{48,71,0}},
/* 1576 */ {(12<<2)|2,{48,71,0}},
/* 1577 */ {(12<<2)|2,{48,71,0}},
/* 1578 */ {(12<<2)|2,{48,71,0}},
/* 1579 */ {(12<<2)|2,{48,71,0}},
/* 1580 */ {(12<<2)|2,{48,71,0}},
/* 1581 */ {(12<<2)|2,{48,71,0}},
/* 1582 */ {(12<<2)|2,{48,71,0}},
/* 1583 */ {(12<<2)|2,{48,71,0}},
/* 1584 */ {(12<<2)|2,{48,72,0}},
/* 1585 */ {(12<<2)|2,{48,72,0}},
/* 1586 */ {(12<<2)|2,{48,72,0}},
/* 1587 */ {(12<<2)|2,{48,72,0}},
/* 1588 */ {(12<<2)|2,{48,72,0}},
/* 1589 */ {(12<<2)|2,{48,72,0}},
/* 1590 */ {(12<<2)|2,{48,72,0}},
/* 1591 */ {(12<<2)|2,{48,72,0}},
/* 1592 */ {(12<<2)|2,{48,72,0}},
/* 1593 */ {(12<<2)|2,{48,72,0}},
/* 1594 */ {(12<<2)|2,{48,72,0}},
/* 1595 */ {(12<<2)|2,{48,72,0}},
/* 1596 */ {(12<<2)|2,{48,72,0}},
/* 1597 */ {(12<<2)|2,{48,72,0}},
/* 1598 */ {(12<<2)|2,{48,72,0}},
/* 1599 */ {(12<<2)|2,{48,72,0}},
/* 1600 */ {(12<<2)|2,{48,73,0}},
/* 1601 */ {(12<<2)|2,{48,73,0}},
/* 1602 */ {(12<<2)|2,{48,73,0}},
/* 1603 */ {(12<<2)|2,{48,73,0}},
/* 1604 */ {(12<<2)|2,{48,73,0}},
/* 1605 */ {(12<<2)|2,{48,73,0}},
/* 1606 */ {(12<<2)|2,{48,73,0}},
/* 1607 */ {(12<<2)|2,{48,73,0}},
/* 1608 */ {(12<<2)|2,{48,73,0}},
/* 1609 */ {(12<<2)|2,{48,73,0}},
/* 1610 */ {(12<<2)|2,{48,73,0}},
/* 1611 */ {(12<<2)|2,{48,73,0}},
/* 1612 */ {(12<<2)|2,{48,73,0}},
/* 1613 */ {(12<<2)|2,{48,73,0}},
/* 1614 */ {(12<<2)|2,{48,73,0}},
/* 1615 */ {(12<<2)|2,{48,73,0}},
/* 1616 */ {(12<<2)|2,{48,74,0}},
/* 1617 */ {(12<<2)|2,{48,74,0}},
/* 1618 */ {(12<<2)|2,{48,74,0}},
/* 1619 */ {(12<<2)|2,{48,74,0}},
/* 1620 */ {(12<<2)|2,{48,74,0}},
/* 1621 */ {(12<<2)|2,{48,74,0}},
/* 1622 */ {(12<<2)|2,{48,74,0}},
/* 1623 */ {(12<<2)|2,{48,74,0}},
/* 1624 */ {(12<<2)|2,{48,74,0}},
/* 1625 */ {(12<<2)|2,{48,74,0}},
/* 1626 */ {(12<<2)|2,{48,74,0}},
/* 1627 */ {(12<<2)|2,{48,74,0}},
/* 1628 */ {(12<<2)|2,{48,74,0}},
/* 1629 */ {(12<<2)|2,{48,74,0}},
/* 1630 */ {(12<<2)|2,{48,74,0}},
/* 1631 */ {(12<<2)|2,{48,74,0}},
/* 1632 */ {(12<<2)|2,{48,75,0}},
/* 1633 */ {(12<<2)|2,{48,75,0}},
/* 1634 */ {(12<<2)|2,{48,75,0}},
/* 1635 */ {(12<<2)|2,{48,75,0}},
/* 1636 */ {(12<<2)|2,{48,75,0}},
/* 1637 */ {(12<<2)|2,{48,75,0}},
/* 1638 */ {(12<<2)|2,{48,75,0}},
/* 1639 */ {(12<<2)|2,{48,75,0}},
/* 1640 */ {(12<<2)|2,{48,75,0}},
/* 1641 */ {(12<<2)|2,{48,75,0}},
/* 1642 */ {(12<<2)|2,{48,75,0}},
/* 1643 */ {(12<<2)|2,{48,75,0}},
/* 1644 */ {(12<<2)|2,{48,75,0}},
/* 1645 */ {(12<<2)|2,{48,75,0}},
/* 1646 */ {(12<<2)|2,{48,75,0}},
/* 1647 */ {(12<<2)|2,{48,75,0}},
/* 1648 */ {(12<<2)|2,{48,76,0}},
/* 1649 */ {(12<<2)|2,{48,76,0}},
/* 1650 */ {(12<<2)|2,{48,76,0}},
/* 1651 */ {(12<<2)|2,{48,76,0}},
/* 1652 */ {(12<<2)|2,{48,76,0}},
/* 1653 */ {(12<<2)|2,{48,76,0}},
/* 1654 */ {(12<<2)|2,{48,76,0}},
/* 1655 */ {(12<<2)|2,{48,76,0}},
/* 1656 */ {(12<<2)|2,{48,76,0}},
/* 1657 */ {(12<<2)|2,{48,76,0}},
/* 1658 */ {(12<<2)|2,{48,76,0}},
/* 1659 */ {(12<<2)|2,{48,76,0}},
/* 1660 */ {(12<<2)|2,{48,76,0}},
/* 1661 */ {(12<<2)|2,{48,76,0}},
/* 1662 */ {(12<<2)|2,{48,76,0}},
/* 1663 */ {(12<<2)|2,{48,76,0}},
/* 1664 */ {(12<<2)|2,{48,77,0}},
/* 1665 */ {(12<<2)|2,{48,77,0}},
/* 1666 */ {(12<<2)|2,{48,77,0}},
/* 1667 */ {(12<<2)|2,{48,77,0}},
/* 1668 */ {(12<<2)|2,{48,77,0}},
/* 1669 */ {(12<<2)|2,{48,77,0}},
/* 1670 */ {(12<<2)|2,{48,77,0}},
/* 1671 */ {(12<<2)|2,{48,77,0}},
/* 1672 */ {(12<<2)|2,{48,77,0}},
/* 1673 */ {(12<<2)|2,{48,77,0}},
/* 1674 */ {(12<<2)|2,{48,77,0}},
/* 1675 */ {(12<<2)|2,{48,77,0}},
/* 1676 */ {(12<<2)|2,{48,77,0}},
/* 1677 */ {(12<<2)|2,{48,77,0}},
/* 1678 */ {(12<<2)|2,{48,77,0}},
/* 1679 */ {(12<<2)|2,{48,77,0}},
/* 1680 */ {(12<<2)|2,{48,78,0}},
/* 1681 */ {(12<<2)|2,{48,78,0}},
/* 1682 */ {(12<<2)|2,{48,78,0}},
/* 1683 */ {(12<<2)|2,{48,78,0}},
/* 1684 */ {(12<<2)|2,{48,78,0}},
/* 1685 */ {(12<<2)|2,{48,78,0}},
/* 1686 */ {(12<<2)|2,{48,78,0}},
/* 1687 */ {(12<<2)|2,{48,78,0}},
/* 1688 */ {(12<<2)|2,{48,78,0}},
/* 1689 */ {(12<<2)|2,{48,78,0}},
/* 1690 */ {(12<<2)|2,{48,78,0}},
/* 1691 */ {(12<<2)|2,{48,78,0}},
/* 1692 */ {(12<<2)|2,{48,78,0}},
/* 1693 */ {(12<<2)|2,{48,78,0}},
/* 1694 */ {(12<<2)|2,{48,78,0}},
/* 1695 */ {(12<<2)|2,{48,78,0}},
/* 1696 */ {(12<<2)|2,{48,79,0}},
/* 1697 */ {(12<<2)|2,{48,79,0}},
/* 1698 */ {(12<<2)|2,{48,79,0}},
/* 1699 */ {(12<<2)|2,{48,79,0}},
/* 1700 */ {(12<<2)|2,{48,79,0}},
/* 1701 */ {(12<<2)|2,{48,79,0}},
/* 1702 */ {(12<<2)|2,{48,79,0}},
/* 1703 */ {(12<<2)|2,{48,79,0}},
/* 1704 */ {(12<<2)|2,{48,79,0}},
/* 1705 */ {(12<<2)|2,{48,79,0}},
/* 1706 */ {(12<<2)|2,{48,79,0}},
/* 1707 */ {(12<<2)|2,{48,79,0}},
/* 1708 */ {(12<<2)|2,{48,79,0}},
/* 1709 */ {(12<<2)|2,{48,79,0}},
/* 1710 */ {(12<<2)|2,{48,79,0}},
/* 1711 */ {(12<<2)|2,{48,79,0}},
/* 1712 */ {(12<<2)|2,{48,80,0}},
/* 1713 */ {(12<<2)|2,{48,80,0}},
/* 1714 */ {(12<<2)|2,{48,80,0}},
/* 1715 */ {(12<<2)|2,{48,80,0}},
/* 1716 */ {(12<<2)|2,{48,80,0}},
/* 1717 */ {(12<<2)|2,{48,80,0}},
/* 1718 */ {(12<<2)|2,{48,80,0}},
/* 1719 */ {(12<<2)|2,{48,80,0}},
/* 1720 */ {(12<<2)|2,{48,80,0}},
/* 1721 */ {(12<<2)|2,{48,80,0}},
/* 1722 */ {(12<<2)|2,{48,80,0}},
/* 1723 */ {(12<<2)|2,{48,80,0}},
/* 1724 */ {(12<<2)|2,{48,80,0}},
/* 1725 */ {(12<<2)|2,{48,80,0}},
/* 1726 */ {(12<<2)|2,{48,80,0}},
/* 1727 */ {(12<<2)|2,{48,80,0}},
/* 1728 */ {(12<<2)|2,{48,81,0}},
/* 1729 */ {(12<<2)|2,{48,81,0}},
/* 1730 */ {(12<<2)|2,{48,81,0}},
/* 1731 */ {(12<<2)|2,{48,81,0}},
/* 1732 */ {(12<<2)|2,{48,81,0}},
/* 1733 */ {(12<<2)|2,{48,81,0}},
/* 1734 */ {(12<<2)|2,{48,81,0}},
/* 1735 */ {(12<<2)|2,{48,81,0}},
/* 1736 */ {(12<<2)|2,{48,81,0}},
/* 1737 */ {(12<<2)|2,{48,81,0}},
/* 1738 */ {(12<<2)|2,{48,81,0}},
/* 1739 */ {(12<<2)|2,{48,81,0}},
/* 1740 */ {(12<<2)|2,{48,81,0}},
/* 1741 */ {(12<<2)|2,{48,81,0}},
/* 1742 */ {(12<<2)|2,{48,81,0}},
/* 1743 */ {(12<<2)|2,{48,81,0}},
/* 1744 */ {(12<<2)|2,{48,82,0}},
/* 1745 */ {(12<<2)|2,{48,82,0}},
/* 1746 */ {(12<<2)|2,{48,82,0}},
/* 1747 */ {(12<<2)|2,{48,82,0}},
/* 1748 */ {(12<<2)|2,{48,82,0}},
/* 1749 */ {(12<<2)|2,{48,82,0}},
/* 1750 */ {(12<<2)|2,{48,82,0}},
/* 1751 */ {(12<<2)|2,{48,82,0}},
/* 1752 */ {(12<<2)|2,{48,82,0}},
/* 1753 */ {(12<<2)|2,{48,82,0}},
/* 1754 */ {(12<<2)|2,{48,82,0}},
/* 1755 */ {(12<<2)|2,{48,82,0}},
/* 1756 */ {(12<<2)|2,{48,82,0}},
/* 1757 */ {(12<<2)|2,{48,82,0}},
/* 1758 */ {(12<<2)|2,{48,82,0}},
/* 1759 */ {(12<<2)|2,{48,82,0}},
/* 1760 */ {(12<<2)|2,{48,83,0}},
/* 1761 */ {(12<<2)|2,{48,83,0}},
/* 1762 */ {(12<<2)|2,{48,83,0}},
/* 1763 */ {(12<<2)|2,{48,83,0}},
/* 1764 */ {(12<<2)|2,{48,83,0}},
/* 1765 */ {(12<<2)|2,{48,83,0}},
/* 1766 */ {(12<<2)|2,{48,83,0}},
/* 1767 */ {(12<<2)|2,{48,83,0}},
/* 1768 */ {(12<<2)|2,{48,83,0}},
/* 1769 */ {(12<<2)|2,{48,83,0}},
/* 1770 */ {(12<<2)|2,{48,83,0}},
/* 1771 */ {(12<<2)|2,{48,83,0}},
/* 1772 */ {(12<<2)|2,{48,83,0}},
/* 1773 */ {(12<<2)|2,{48,83,0}},
/* 1774 */ {(12<<2)|2,{48,83,0}},
/* 1775 */ {(12<<2)|2,{48,83,0}},
/* 1776 */ {(12<<2)|2,{48,84,0}},
/* 1777 */ {(12<<2)|2,{48,84,0}},
/* 1778 */ {(12<<2)|2,{48,84,0}},
/* 1779 */ {(12<<2)|2,{48,84,0}},
/* 1780 */ {(12<<2)|2,{48,84,0}},
/* 1781 */ {(12<<2)|2,{48,84,0}},
/* 1782 */ {(12<<2)|2,{48,84,0}},
/* 1783 */ {(12<<2)|2,{48,84,0}},
/* 1784 */ {(12<<2)|2,{48,84,0}},
/* 1785 */ {(12<<2)|2,{48,84,0}},
/* 1786 */ {(12<<2)|2,{48,84,0}},
/* 1787 */ {(12<<2)|2,{48,84,0}},
/* 1788 */ {(12<<2)|2,{48,84,0}},
/* 1789 */ {(12<<2)|2,{48,84,0}},
/* 1790 */ {(12<<2)|2,{48,84,0}},
/* 1791 */ {(12<<2)|2,{48,84,0}},
/* 1792 */ {(12<<2)|2,{48,85,0}},
/* 1793 */ {(12<<2)|2,{48,85,0}},
/* 1794 */ {(12<<2)|2,{48,85,0}},
/* 1795 */ {(12<<2)|2,{48,85,0}},
/* 1796 */ {(12<<2)|2,{48,85,0}},
/* 1797 */ {(12<<2)|2,{48,85,0}},
/* 1798 */ {(12<<2)|2,{48,85,0}},
/* 1799 */ {(12<<2)|2,{48,85,0}},
/* 1800 */ {(12<<2)|2,{48,85,0}},
/* 1801 */ {(12<<2)|2,{48,85,0}},
/* 1802 */ {(12<<2)|2,{48,85,0}},
/* 1803 */ {(12<<2)|2,{48,85,0}},
/* 1804 */ {(12<<2)|2,{48,85,0}},
/* 1805 */ {(12<<2)|2,{48,85,0}},
/* 1806 */ {(12<<2)|2,{48,85,0}},
/* 1807 */ {(12<<2)|2,{48,85,0}},
/* 1808 */ {(12<<2)|2,{48,86,0}},
/* 1809 */ {(12<<2)|2,{48,86,0}},
/* 1810 */ {(12<<2)|2,{48,86,0}},
/* 1811 */ {(12<<2)|2,{48,86,0}},
/* 1812 */ {(12<<2)|2,{48,86,0}},
/* 1813 */ {(12<<2)|2,{48,86,0}},
/* 1814 */ {(12<<2)|2,{48,86,0}},
/* 1815 */ {(12<<2)|2,{48,86,0}},
/* 1816 */ {(12<<2)|2,{48,86,0}},
/* 1817 */ {(12<<2)|2,{48,86,0}},
/* 1818 */ {(12<<2)|2,{48,86,0}},
/* 1819 */ {(12<<2)|2,{48,86,0}},
/* 1820 */ {(12<<2)|2,{48,86,0}},
/* 1821 */ {(12<<2)|2,{48,86,0}},
/* 1822 */ {(12<<2)|2,{48,86,0}},
/* 1823 */ {(12<<2)|2,{48,86,0}},
/* 1824 */ {(12<<2)|2,{48,87,0}},
/* 1825 */ {(12<<2)|2,{48,87,0}},
/* 1826 */ {(12<<2)|2,{48,87,0}},
/* 1827 */ {(12<<2)|2,{48,87,0}},
/* 1828 */ {(12<<2)|2,{48,87,0}},
/* 1829 */ {(12<<2)|2,{48,87,0}},
/* 1830 */ {(12<<2)|2,{48,87,0}},
/* 1831 */ {(12<<2)|2,{48,87,0}},
/* 1832 */ {(12<<2)|2,{48,87,0}},
/* 1833 */ {(12<<2)|2,{48,87,0}},
/* 1834 */ {(12<<2)|2,{48,87,0}},
/* 1835 */ {(12<<2)|2,{48,87,0}},
/* 1836 */ {(12<<2)|2,{48,87,0}},
/* 1837 */ {(12<<2)|2,{48,87,0}},
/* 1838 */ {(12<<2)|2,{48,87,0}},
/* 1839 */ {(12<<2)|2,{48,87,0}},
/* 1840 */ {(12<<2)|2,{48,89,0}},
/* 1841 */ {(12<<2)|2,{48,89,0}},
/* 1842 */ {(12<<2)|2,{48,89,0}},
/* 1843 */ {(12<<2)|2,{48,89,0}},
/* 1844 */ {(12<<2)|2,{48,89,0}},
/* 1845 */ {(12<<2)|2,{48,89,0}},
/* 1846 */ {(12<<2)|2,{48,89,0}},
/* 1847 */ {(12<<2)|2,{48,89,0}},
/* 1848 */ {(12<<2)|2,{48,89,0}},
/* 1849 */ {(12<<2)|2,{48,89,0}},
/* 1850 */ {(12<<2)|2,{48,89,0}},
/* 1851 */ {(12<<2)|2,{48,89,0}},
/* 1852 */ {(12<<2)|2,{48,89,0}},
/* 1853 */ {(12<<2)|2,{48,89,0}},
/* 1854 */ {(12<<2)|2,{48,89,0}},
/* 1855 */ {(12<<2)|2,{48,89,0}},
/* 1856 */ {(12<<2)|2,{48,106,0}},
/* 1857 */ {(12<<2)|2,{48,106,0}},
/* 1858 */ {(12<<2)|2,{48,106,0}},
/* 1859 */ {(12<<2)|2,{48,106,0}},
/* 1860 */ {(12<<2)|2,{48,106,0}},
/* 1861 */ {(12<<2)|2,{48,106,0}},
/* 1862 */ {(12<<2)|2,{48,106,0}},
/* 1863 */ {(12<<2)|2,{48,106,0}},
/* 1864 */ {(12<<2)|2,{48,106,0}},
/* 1865 */ {(12<<2)|2,{48,106,0}},
/* 1866 */ {(12<<2)|2,{48,106,0}},
/* 1867 */ {(12<<2)|2,{48,106,0}},
/* 1868 */ {(12<<2)|2,{48,106,0}},
/* 1869 */ {(12<<2)|2,{48,106,0}},
/* 1870 */ {(12<<2)|2,{48,106,0}},
/* 1871 */ {(12<<2)|2,{48,106,0}},
/* 1872 */ {(12<<2)|2,{48,107,0}},
/* 1873 */ {(12<<2)|2,{48,107,0}},
/* 1874 */ {(12<<2)|2,{48,107,0}},
/* 1875 */ {(12<<2)|2,{48,107,0}},
/* 1876 */ {(12<<2)|2,{48,107,0}},
/* 1877 */ {(12<<2)|2,{48,107,0}},
/* 1878 */ {(12<<2)|2,{48,107,0}},
/* 1879 */ {(12<<2)|2,{48,107,0}},
/* 1880 */ {(12<<2)|2,{48,107,0}},
/* 1881 */ {(12<<2)|2,{48,107,0}},
/* 1882 */ {(12<<2)|2,{48,107,0}},
/* 1883 */ {(12<<2)|2,{48,107,0}},
/* 1884 */ {(12<<2)|2,{48,107,0}},
/* 1885 */ {(12<<2)|2,{48,107,0}},
/* 1886 */ {(12<<2)|2,{48,107,0}},
/* 1887 */ {(12<<2)|2,{48,107,0}},
/* 1888 */ {(12<<2)|2,{48,113,0}},
/* 1889 */ {(12<<2)|2,{48,113,0}},
/* 1890 */ {(12<<2)|2,{48,113,0}},
/* 1891 */ {(12<<2)|2,{48,113,0}},
/* 1892 */ {(12<<2)|2,{48,113,0}},
/* 1893 */ {(12<<2)|2,{48,113,0}},
/* 1894 */ {(12<<2)|2,{48,113,0}},
/* 1895 */ {(12<<2)|2,{48,113,0}},
/* 1896 */ {(12<<2)|2,{48,113,0}},
/* 1897 */ {(12<<2)|2,{48,113,0}},
/* 1898 */ {(12<<2)|2,{48,113,0}},
/* 1899 */ {(12<<2)|2,{48,113,0}},
/* 1900 */ {(12<<2)|2,{48,113,0}},
/* 1901 */ {(12<<2)|2,{48,113,0}},
/* 1902 */ {(12<<2)|2,{48,113,0}},
/* 1903 */ {(12<<2)|2,{48,113,0}},
/* 1904 */ {(12<<2)|2,{48,118,0}},
/* 1905 */ {(12<<2)|2,{48,118,0}},
/* 1906 */ {(12<<2)|2,{48,118,0}},
/* 1907 */ {(12<<2)|2,{48,118,0}},
/* 1908 */ {(12<<2)|2,{48,118,0}},
/* 1909 */ {(12<<2)|2,{48,118,0}},
/* 1910 */ {(12<<2)|2,{48,118,0}},
/* 1911 */ {(12<<2)|2,{48,118,0}},
/* 1912 */ {(12<<2)|2,{48,118,0}},
/* 1913 */ {(12<<2)|2,{48,118,0}},
/* 1914 */ {(12<<2)|2,{48,118,0}},
/* 1915 */ {(12<<2)|2,{48,118,0}},
/* 1916 */ {(12<<2)|2,{48,118,0}},
/* 1917 */ {(12<<2)|2,{48,118,0}},
/* 1918 */ {(12<<2)|2,{48,118,0}},
/* 1919 */ {(12<<2)|2,{48,118,0}},
/* 1920 */ {(12<<2)|2,{48,119,0}},
/* 1921 */ {(12<<2)|2,{48,119,0}},
/* 1922 */ {(12<<2)|2,{48,119,0}},
/* 1923 */ {(12<<2)|2,{48,119,0}},
/* 1924 */ {(12<<2)|2,{48,119,0}},
/* 1925 */ {(12<<2)|2,{48,119,0}},
/* 1926 */ {(12<<2)|2,{48,119,0}},
/* 1927 */ {(12<<2)|2,{48,119,0}},
/* 1928 */ {(12<<2)|2,{48,119,0}},
/* 1929 */ {(12<<2)|2,{48,119,0}},
/* 1930 */ {(12<<2)|2,{48,119,0}},
/* 1931 */ {(12<<2)|2,{48,119,0}},
/* 1932 */ {(12<<2)|2,{48,119,0}},
/* 1933 */ {(12<<2)|2,{48,119,0}},
/* 1934 */ {(12<<2)|2,{48,119,0}},
/* 1935 */ {(12<<2)|2,{48,119,0}},
/* 1936 */ {(12<<2)|2,{48,120,0}},
/* 1937 */ {(12<<2)|2,{48,120,0}},
/* 1938 */ {(12<<2)|2,{48,120,0}},
/* 1939 */ {(12<<2)|2,{48,120,0}},
/* 1940 */ {(12<<2)|2,{48,120,0}},
/* 1941 */ {(12<<2)|2,{48,120,0}},
/* 1942 */ {(12<<2)|2,{48,120,0}},
/* 1943 */ {(12<<2)|2,{48,120,0}},
/* 1944 */ {(12<<2)|2,{48,120,0}},
/* 1945 */ {(12<<2)|2,{48,120,0}},
/* 1946 */ {(12<<2)|2,{48,120,0}},
/* 1947 */ {(12<<2)|2,{48,120,0}},
/* 1948 */ {(12<<2)|2,{48,120,0}},
/* 1949 */ {(12<<2)|2,{48,120,0}},
/* 1950 */ {(12<<2)|2,{48,120,0}},
/* 1951 */ {(12<<2)|2,{48,120,0}},
/* 1952 */ {(12<<2)|2,{48,121,0}},
/* 1953 */ {(12<<2)|2,{48,121,0}},
/* 1954 */ {(12<<2)|2,{48,121,0}},
/* 1955 */ {(12<<2)|2,{48,121,0}},
/* 1956 */ {(12<<2)|2,{48,121,0}},
/* 1957 */ {(12<<2)|2,{48,121,0}},
/* 1958 */ {(12<<2)|2,{48,121,0}},
/* 1959 */ {(12<<2)|2,{48,121,0}},
/* 1960 */ {(12<<2)|2,{48,121,0}},
/* 1961 */ {(12<<2)|2,{48,121,0}},
/* 1962 */ {(12<<2)|2,{48,121,0}},
/* 1963 */ {(12<<2)|2,{48,121,0}},
/* 1964 */ {(12<<2)|2,{48,121,0}},
/* 1965 */ {(12<<2)|2,{48,121,0}},
/* 1966 */ {(12<<2)|2,{48,121,0}},
/* 1967 */ {(12<<2)|2,{48,121,0}},
/* 1968 */ {(12<<2)|2,{48,122,0}},
/* 1969 */ {(12<<2)|2,{48,122,0}},
/* 1970 */ {(12<<2)|2,{48,122,0}},
/* 1971 */ {(12<<2)|2,{48,122,0}},
/* 1972 */ {(12<<2)|2,{48,122,0}},
/* 1973 */ {(12<<2)|2,{48,122,0}},
/* 1974 */ {(12<<2)|2,{48,122,0}},
/* 1975 */ {(12<<2)|2,{48,122,0}},
/* 1976 */ {(12<<2)|2,{48,122,0}},
/* 1977 */ {(12<<2)|2,{48,122,0}},
/* 1978 */ {(12<<2)|2,{48,122,0}},
/* 1979 */ {(12<<2)|2,{48,122,0}},
/* 1980 */ {(12<<2)|2,{48,122,0}},
/* 1981 */ {(12<<2)|2,{48,122,0}},
/* 1982 */ {(12<<2)|2,{48,122,0}},
/* 1983 */ {(12<<2)|2,{48,122,0}},
/* 1984 */ {(13<<2)|2,{48,38,0}},
/* 1985 */ {(13<<2)|2,{48,38,0}},
/* 1986 */ {(13<<2)|2,{48,38,0}},
/* 1987 */ {(13<<2)|2,{48,38,0}},
/* 1988 */ {(13<<2)|2,{48,38,0}},
/* 1989 */ {(13<<2)|2,{48,38,0}},
/* 1990 */ {(13<<2)|2,{48,38,0}},
/* 1991 */ {(13<<2)|2,{48,38,0}},
/* 1992 */ {(13<<2)|2,{48,42,0}},
/* 1993 */ {(13<<2)|2,{48,42,0}},
/* 1994 */ {(13<<2)|2,{48,42,0}},
/* 1995 */ {(13<<2)|2,{48,42,0}},
/* 1996 */ {(13<<2)|2,{48,42,0}},
/* 1997 */ {(13<<2)|2,{48,42,0}},
/* 1998 */ {(13<<2)|2,{48,42,0}},
/* 1999 */ {(13<<2)|2,{48,42,0}},
/* 2000 */ {(13<<2)|2,{48,44,0}},
/* 2001 */ {(13<<2)|2,{48,44,0}},
/* 2002 */ {(13<<2)|2,{48,44,0}},
/* 2003 */ {(13<<2)|2,{48,44,0}},
/* 2004 */ {(13<<2)|2,{48,44,0}},
/* 2005 */ {(13<<2)|2,{48,44,0}},
/* 2006 */ {(13<<2)|2,{48,44,0}},
/* 2007 */ {(13<<2)|2,{48,44,0}},
/* 2008 */ {(13<<2)|2,{48,59,0}},
/* 2009 */ {(13<<2)|2,{48,59,0}},
/* 2010 */ {(13<<2)|2,{48,59,0}},
/* 2011 */ {(13<<2)|2,{48,59,0}},
/* 2012 */ {(13<<2)|2,{48,59,0}},
/* 2013 */ {(13<<2)|2,{48,59,0}},
/* 2014 */ {(13<<2)|2,{48,59,0}},
/* 2015 */ {(13<<2)|2,{48,59,0}},
/* 2016 */ {(13<<2)|2,{48,88,0}},
/* 2017 */ {(13<<2)|2,{48,88,0}},
/* 2018 */ {(13<<2)|2,{48,88,0}},
/* 2019 */ {(13<<2)|2,{48,88,0}},
/* 2020 */ {(13<<2)|2,{48,88,0}},
/* 2021 */ {(13<<2)|2,{48,88,0}},
/* 2022 */ {(13<<2)|2,{48,88,0}},
/* 2023 */ {(13<<2)|2,{48,88,0}},
/* 2024 */ {(13<<2)|2,{48,90,0}},
/* 2025 */ {(13<<2)|2,{48,90,0}},
/* 2026 */ {(13<<2)|2,{48,90,0}},
/* 2027 */ {(13<<2)|2,{48,90,0}},
/* 2028 */ {(13<<2)|2,{48,90,0}},
/* 2029 */ {(13<<2)|2,{48,90,0}},
/* 2030 */ {(13<<2)|2,{48,90,0}},
/* 2031 */ {(13<<2)|2,{48,90,0}},
/* 2032 */ {(15<<2)|2,{48,33,0}},
/* 2033 */ {(15<<2)|2,{48,33,0}},
/* 2034 */ {(15<<2)|2,{48,34,0}},
/* 2035 */ {(15<<2)|2,{48,34,0}},
/* 2036 */ {(15<<2)|2,{48,40,0}},
/* 2037 */ {(15<<2)|2,{48,40,0}},
/* 2038 */ {(15<<2)|2,{48,41,0}},
/* 2039 */ {(15<<2)|2,{48,41,0}},
/* 2040 */ {(15<<2)|2,{48,63,0}},
/* 2041 */ {(15<<2)|2,{48,63,0}},
/* 2042 */ {(16<<2)|2,{48,39,0}},
/* 2043 */ {(16<<2)|2,{48,43,0}},
/* 2044 */ {(16<<2)|2,{48,124,0}},
/* 2045 */ {(5<<2)|1,{48,0,0}},
/* 2046 */ {(5<<2)|1,{48,0,0}},
/* 2047 */ {(5<<2)|1,{48,0,0}},
/* 2048 */ {(15<<2)|3,{49,48,48}},
/* 2049 */ {(15<<2)|3,{49,48,48}},
/* 2050 */ {(15<<2)|3,{49,48,49}},
/* 2051 */ {(15<<2)|3,{49,48,49}},
/* 2052 */ {(15<<2)|3,{49,48,50}},
/* 2053 */ {(15<<2)|3,{49,48,50}},
/* 2054 */ {(15<<2)|3,{49,48,97}},
/* 2055 */ {(15<<2)|3,{49,48,97}},
/* 2056 */ {(15<<2)|3,{49,48,99}},
/* 2057 */ {(15<<2)|3,{49,48,99}},
/* 2058 */ {(15<<2)|3,{49,48,101}},
/* 2059 */ {(15<<2)|3,{49,48,101}},
/* 2060 */ {(15<<2)|3,{49,48,105}},
/* 2061 */ {(15<<2)|3,{49,48,105}},
/* 2062 */ {(15<<2)|3,{49,48,111}},
/* 2063 */ {(15<<2)|3,{49,48,111}},
/* 2064 */ {(15<<2)|3,{49,48,115}},
/* 2065 */ {(15<<2)|3,{49,48,115}},
/* 2066 */ {(15<<2)|3,{49,48,116}},
/* 2067 */ {(15<<2)|3,{49,48,116}},
/* 2068 */ {(16<<2)|3,{49,48,32}},
/* 2069 */ {(16<<2)|3,{49,48,37}},
/* 2070 */ {(16<<2)|3,{49,48,45}},
/* 2071 */ {(16<<2)|3,{49,48,46}},
/* 2072 */ {(16<<2)|3,{49,48,47}},
/* 2073 */ {(16<<2)|3,{49,48,51}},
/* 2074 */ {(16<<2)|3,{49,48,52}},
/* 2075 */ {(16<<2)|3,{49,48,53}},
/* 2076 */ {(16<<2)|3,{49,48,54}},
/* 2077 */ {(16<<2)|3,{49,48,55}},
/* 2078 */ {(16<<2)|3,{49,48,56}},
/* 2079 */ {(16<<2)|3,{49,48,57}},
/* 2080 */ {(16<<2)|3,{49,48,61}},
/* 2081 */ {(16<<2)|3,{49,48,65}},
/* 2082 */ {(16<<2)|3,{49,48,95}},
/* 2083 */ {(16<<2)|3,{49,48,98}},
/* 2084 */ {(16<<2)|3,{49,48,100}},
/* 2085 */ {(16<<2)|3,{49,48,102}},
/* 2086 */ {(16<<2)|3,{49,48,103}},
/* 2087 */ {(16<<2)|3,{49,48,104}},
/* 2088 */ {(16<<2)|3,{49,48,108}},
/* 2089 */ {(16<<2)|3,{49,48,109}},
/* 2090 */ {(16<<2)|3,{49,48,110}},
/* 2091 */ {(16<<2)|3,{49,48,112}},
/* 2092 */ {(16<<2)|3,{49,48,114}},
/* 2093 */ {(16<<2)|3,{49,48,117}},
/* 2094 */ {(10<<2)|2,{49,48,0}},
/* 2095 */ {(10<<2)|2,{49,48,0}},
/* 2096 */ {(10<<2)|2,{49,48,0}},
/* 2097 */ {(10<<2)|2,{49,48,0}},
/* 2098 */ {(10<<2)|2,{49,48,0}},
/* 2099 */ {(10<<2)|2,{49,48,0}},
/* 2100 */ {(10<<2)|2,{49,48,0}},
/* 2101 */ {(10<<2)|2,{49,48,0}},
/* 2102 */ {(10<<2)|2,{49,48,0}},
/* 2103 */ {(10<<2)|2,{49,48,0}},
/* 2104 */ {(10<<2)|2,{49,48,0}},
/* 2105 */ {(10<<2)|2,{49,48,0}},
/* 2106 */ {(10<<2)|2,{49,48,0}},
/* 2107 */ {(10<<2)|2,{49,48,0}},
/* 2108 */ {(10<<2)|2,{49,48,0}},
/* 2109 */ {(10<<2)|2,{49,48,0}},
/* 2110 */ {(10<<2)|2,{49,48,0}},
/* 2111 */ {(10<<2)|2,{49,48,0}},
/* 2112 */ {(15<<2)|3,{49,49,48}},
/* 2113 */ {(15<<2)|3,{49,49,48}},
/* 2114 */ {(15<<2)|3,{49,49,49}},
/* 2115 */ {(15<<2)|3,{49,49,49}},
/* 2116 */ {(15<<2)|3,{49,49,50}},
/* 2117 */ {(15<<2)|3,{49,49,50}},
/* 2118 */ {(15<<2)|3,{49,49,97}},
/* 2119 */ {(15<<2)|3,{49,49,97}},
/* 2120 */ {(15<<2)|3,{49,49,99}},
/* 2121 */ {(15<<2)|3,{49,49,99}},
/* 2122 */ {(15<<2)|3,{49,49,101}},
/* 2123 */ {(15<<2)|3,{49,49,101}},
/* 2124 */ {(15<<2)|3,{49,49,105}},
/* 2125 */ {(15<<2)|3,{49,49,105}},
/* 2126 */ {(15<<2)|3,{49,49,111}},
/* 2127 */ {(15<<2)|3,{49,49,111}},
/* 2128 */ {(15<<2)|3,{49,49,115}},
/* 2129 */ {(15<<2)|3,{49,49,115}},
/* 2130 */ {(15<<2)|3,{49,49,116}},
/* 2131 */ {(15<<2)|3,{49,49,116}},
/* 2132 */ {(16<<2)|3,{49,49,32}},
/* 2133 */ {(16<<2)|3,{49,49,37}},
/* 2134 */ {(16<<2)|3,{49,49,45}},
/* 2135 */ {(16<<2)|3,{49,49,46}},
/* 2136 */ {(16<<2)|3,{49,49,47}},
/* 2137 */ {(16<<2)|3,{49,49,51}},
/* 2138 */ {(16<<2)|3,{49,49,52}},
/* 2139 */ {(16<<2)|3,{49,49,53}},
/* 2140 */ {(16<<2)|3,{49,49,54}},
/* 2141 */ {(16<<2)|3,{49,49,55}},
/* 2142 */ {(16<<2)|3,{49,49,56}},
/* 2143 */ {(16<<2)|3,{49,49,57}},
/* 2144 */ {(16<<2)|3,{49,49,61}},
/* 2145 */ {(16<<2)|3,{49,49,65}},
/* 2146 */ {(16<<2)|3,{49,49,95}},
/* 2147 */ {(16<<2)|3,{49,49,98}},
/* 2148 */ {(16<<2)|3,{49,49,100}},
/* 2149 */ {(16<<2)|3,{49,49,102}},
/* 2150 */ {(16<<2)|3,{49,49,103}},
/* 2151 */ {(16<<2)|3,{49,49,104}},
/* 2152 */ {(16<<2)|3,{49,49,108}},
/* 2153 */ {(16<<2)|3,{49,49,109}},
/* 2154 */ {(16<<2)|3,{49,49,110}},
/* 2155 */ {(16<<2)|3,{49,49,112}},
/* 2156 */ {(16<<2)|3,{49,49,114}},
/* 2157 */ {(16<<2)|3,{49,49,117}},
/* 2158 */ {(10<<2)|2,{49,49,0}},
/* 2159 */ {(10<<2)|2,{49,49,0}},
/* 2160 */ {(10<<2)|2,{49,49,0}},
/* 2161 */ {(10<<2)|2,{49,49,0}},
/* 2162 */ {(10<<2)|2,{49,49,0}},
/* 2163 */ {(10<<2)|2,{49,49,0}},
/* 2164 */ {(10<<2)|2,{49,49,0}},
/* 2165 */ {(10<<2)|2,{49,49,0}},
/* 2166 */ {(10<<2)|2,{49,49,0}},
/* 2167 */ {(10<<2)|2,{49,49,0}},
/* 2168 */ {(10<<2)|2,{49,49,0}},
/* 2169 */ {(10<<2)|2,{49,49,0}},
/* 2170 */ {(10<<2)|2,{49,49,0}},
/* 2171 */ {(10<<2)|2,{49,49,0}},
/* 2172 */ {(10<<2)|2,{49,49,0}},
/* 2173 */ {(10<<2)|2,{49,49,0}},
/* 2174 */ {(10<<2)|2,{49,49,0}},
/* 2175 */ {(10<<2)|2,{49,49,0}},
/* 2176 */ {(15<<2)|3,{49,50,48}},
/* 2177 */ {(15<<2)|3,{49,50,48}},
/* 2178 */ {(15<<2)|3,{49,50,49}},
/* 2179 */ {(15<<2)|3,{49,50,49}},
/* 2180 */ {(15<<2)|3,{49,50,50}},
/* 2181 */ {(15<<2)|3,{49,50,50}},
/* 2182 */ {(15<<2)|3,{49,50,97}},
/* 2183 */ {(15<<2)|3,{49,50,97}},
/* 2184 */ {(15<<2)|3,{49,50,99}},
/* 2185 */ {(15<<2)|3,{49,50,99}},
/* 2186 */ {(15<<2)|3,{49,50,101}},
/* 2187 */ {(15<<2)|3,{49,50,101}},
/* 2188 */ {(15<<2)|3,{49,50,105}},
/* 2189 */ {(15<<2)|3,{49,50,105}},
/* 2190 */ {(15<<2)|3,{49,50,111}},
/* 2191 */ {(15<<2)|3,{49,50,111}},
/* 2192 */ {(15<<2)|3,{49,50,115}},
/* 2193 */ {(15<<2)|3,{49,50,115}},
/* 2194 */ {(15<<2)|3,{49,50,116}},
/* 2195 */ {(15<<2)|3,{49,50,116}},
/* 2196 */ {(16<<2)|3,{49,50,32}},
/* 2197 */ {(16<<2)|3,{49,50,37}},
/* 2198 */ {(16<<2)|3,{49,50,45}},
/* 2199 */ {(16<<2)|3,{49,50,46}},
/* 2200 */ {(16<<2)|3,{49,50,47}},
/* 2201 */ {(16<<2)|3,{49,50,51}},
/* 2202 */ {(16<<2)|3,{49,50,52}},
/* 2203 */ {(16<<2)|3,{49,50,53}},
/* 2204 */ {(16<<2)|3,{49,50,54}},
/* 2205 */ {(16<<2)|3,{49,50,55}},
/* 2206 */ {(16<<2)|3,{49,50,56}},
/* 2207 */ {(16<<2)|3,{49,50,57}},
/* 2208 */ {(16<<2)|3,{49,50,61}},
/* 2209 */ {(16<<2)|3,{49,50,65}},
/* 2210 */ {(16<<2)|3,{49,50,95}},
/* 2211 */ {(16<<2)|3,{49,50,98}},
/* 2212 */ {(16<<2)|3,{49,50,100}},
/* 2213 */ {(16<<2)|3,{49,50,102}},
/* 2214 */ {(16<<2)|3,{49,50,103}},
/* 2215 */ {(16<<2)|3,{49,50,104}},
/* 2216 */ {(16<<2)|3,{49,50,108}},
/* 2217 */ {(16<<2)|3,{49,50,109}},
/* 2218 */ {(16<<2)|3,{49,50,110}},
/* 2219 */ {(16<<2)|3,{49,50,112}},
/* 2220 */ {(16<<2)|3,{49,50,114}},
/* 2221 */ {(16<<2)|3,{49,50,117}},
/* 2222 */ {(10<<2)|2,{49,50,0}},
/* 2223 */ {(10<<2)|2,{49,50,0}},
/* 2224 */ {(10<<2)|2,{49,50,0}},
/* 2225 */ {(10<<2)|2,{49,50,0}},
/* 2226 */ {(10<<2)|2,{49,50,0}},
/* 2227 */ {(10<<2)|2,{49,50,0}},
/* 2228 */ {(10<<2)|2,{49,50,0}},
/* 2229 */ {(10<<2)|2,{49,50,0}},
/* 2230 */ {(10<<2)|2,{49,50,0}},
/* 2231 */ {(10<<2)|2,{49,50,0}},
/* 2232 */ {(10<<2)|2,{49,50,0}},
/* 2233 */ {(10<<2)|2,{49,50,0}},
/* 2234 */ {(10<<2)|2,{49,50,0}},
/* 2235 */ {(10<<2)|2,{49,50,0}},
/* 2236 */ {(10<<2)|2,{49,50,0}},
/* 2237 */ {(10<<2)|2,{49,50,0}},
/* 2238 */ {(10<<2)|2,{49,50,0}},
/* 2239 */ {(10<<2)|2,{49,50,0}},
/* 2240 */ {(15<<2)|3,{49,97,48}},
/* 2241 */ {(15<<2)|3,{49,97,48}},
/* 2242 */ {(15<<2)|3,{49,97,49}},
/* 2243 */ {(15<<2)|3,{49,97,49}},
/* 2244 */ {(15<<2)|3,{49,97,50}},
/* 2245 */ {(15<<2)|3,{49,97,50}},
/* 2246 */ {(15<<2)|3,{49,97,97}},
/* 2247 */ {(15<<2)|3,{49,97,97}},
/* 2248 */ {(15<<2)|3,{49,97,99}},
/* 2249 */ {(15<<2)|3,{49,97,99}},
/* 2250 */ {(15<<2)|3,{49,97,101}},
/* 2251 */ {(15<<2)|3,{49,97,101}},
/* 2252 */ {(15<<2)|3,{49,97,105}},
/* 2253 */ {(15<<2)|3,{49,97,105}},
/* 2254 */ {(15<<2)|3,{49,97,111}},
/* 2255 */ {(15<<2)|3,{49,97,111}},
/* 2256 */ {(15<<2)|3,{49,97,115}},
/* 2257 */ {(15<<2)|3,{49,97,115}},
/* 2258 */ {(15<<2)|3,{49,97,116}},
/* 2259 */ {(15<<2)|3,{49,97,116}},
/* 2260 */ {(16<<2)|3,{49,97,32}},
/* 2261 */ {(16<<2)|3,{49,97,37}},
/* 2262 */ {(16<<2)|3,{49,97,45}},
/* 2263 */ {(16<<2)|3,{49,97,46}},
/* 2264 */ {(16<<2)|3,{49,97,47}},
/* 2265 */ {(16<<2)|3,{49,97,51}},
/* 2266 */ {(16<<2)|3,{49,97,52}},
/* 2267 */ {(16<<2)|3,{49,97,53}},
/* 2268 */ {(16<<2)|3,{49,97,54}},
/* 2269 */ {(16<<2)|3,{49,97,55}},
/* 2270 */ {(16<<2)|3,{49,97,56}},
/* 2271 */ {(16<<2)|3,{49,97,57}},
/* 2272 */ {(16<<2)|3,{49,97,61}},
/* 2273 */ {(16<<2)|3,{49,97,65}},
/* 2274 */ {(16<<2)|3,{49,97,95}},
/* 2275 */ {(16<<2)|3,{49,97,98}},
/* 2276 */ {(16<<2)|3,{49,97,100}},
/* 2277 */ {(16<<2)|3,{49,97,102}},
/* 2278 */ {(16<<2)|3,{49,97,103}},
/* 2279 */ {(16<<2)|3,{49,97,104}},
/* 2280 */ {(16<<2)|3,{49,97,108}},
/* 2281 */ {(16<<2)|3,{49,97,109}},
/* 2282 */ {(16<<2)|3,{49,97,110}},
/* 2283 */ {(16<<2)|3,{49,97,112}},
/* 2284 */ {(16<<2)|3,{49,97,114}},
/* 2285 */ {(16<<2)|3,{49,97,117}},
/* 2286 */ {(10<<2)|2,{49,97,0}},
/* 2287 */ {(10<<2)|2,{49,97,0}},
/* 2288 */ {(10<<2)|2,{49,97,0}},
/* 2289 */ {(10<<2)|2,{49,97,0}},
/* 2290 */ {(10<<2)|2,{49,97,0}},
/* 2291 */ {(10<<2)|2,{49,97,0}},
/* 2292 */ {(10<<2)|2,{49,97,0}},
/* 2293 */ {(10<<2)|2,{49,97,0}},
/* 2294 */ {(10<<2)|2,{49,97,0}},
/* 2295 */ {(10<<2)|2,{49,97,0}},
/* 2296 */ {(10<<2)|2,{49,97,0}},
/* 2297 */ {(10<<2)|2,{49,97,0}},
/* 2298 */ {(10<<2)|2,{49,97,0}},
/* 2299 */ {(10<<2)|2,{49,97,0}},
/* 2300 */ {(10<<2)|2,{49,97,0}},
/* 2301 */ {(10<<2)|2,{49,97,0}},
/* 2302 */ {(10<<2)|2,{49,97,0}},
/* 2303 */ {(10<<2)|2,{49,97,0}},
/* 2304 */ {(15<<2)|3,{49,99,48}},
/* 2305 */ {(15<<2)|3,{49,99,48}},
/* 2306 */ {(15<<2)|3,{49,99,49}},
/* 2307 */ {(15<<2)|3,{49,99,49}},
/* 2308 */ {(15<<2)|3,{49,99,50}},
/* 2309 */ {(15<<2)|3,{49,99,50}},
/* 2310 */ {(15<<2)|3,{49,99,97}},
/* 2311 */ {(15<<2)|3,{49,99,97}},
/* 2312 */ {(15<<2)|3,{49,99,99}},
/* 2313 */ {(15<<2)|3,{49,99,99}},
/* 2314 */ {(15<<2)|3,{49,99,101}},
/* 2315 */ {(15<<2)|3,{49,99,101}},
/* 2316 */ {(15<<2)|3,{49,99,105}},
/* 2317 */ {(15<<2)|3,{49,99,105}},
/* 2318 */ {(15<<2)|3,{49,99,111}},
/* 2319 */ {(15<<2)|3,{49,99,111}},
/* 2320 */ {(15<<2)|3,{49,99,115}},
/* 2321 */ {(15<<2)|3,{49,99,115}},
/* 2322 */ {(15<<2)|3,{49,99,116}},
/* 2323 */ {(15<<2)|3,{49,99,116}},
/* 2324 */ {(16<<2)|3,{49,99,32}},
/* 2325 */ {(16<<2)|3,{49,99,37}},
/* 2326 */ {(16<<2)|3,{49,99,45}},
/* 2327 */ {(16<<2)|3,{49,99,46}},
/* 2328 */ {(16<<2)|3,{49,99,47}},
/* 2329 */ {(16<<2)|3,{49,99,51}},
/* 2330 */ {(16<<2)|3,{49,99,52}},
/* 2331 */ {(16<<2)|3,{49,99,53}},
/* 2332 */ {(16<<2)|3,{49,99,54}},
/* 2333 */ {(16<<2)|3,{49,99,55}},
/* 2334 */ {(16<<2)|3,{49,99,56}},
/* 2335 */ {(16<<2)|3,{49,99,57}},
/* 2336 */ {(16<<2)|3,{49,99,61}},
/* 2337 */ {(16<<2)|3,{49,99,65}},
/* 2338 */ {(16<<2)|3,{49,99,95}},
/* 2339 */ {(16<<2)|3,{49,99,98}},
/* 2340 */ {(16<<2)|3,{49,99,100}},
/* 2341 */ {(16<<2)|3,{49,99,102}},
/* 2342 */ {(16<<2)|3,{49,99,103}},
/* 2343 */ {(16<<2)|3,{49,99,104}},
/* 2344 */ {(16<<2)|3,{49,99,108}},
/* 2345 */ {(16<<2)|3,{49,99,109}},
/* 2346 */ {(16<<2)|3,{49,99,110}},
/* 2347 */ {(16<<2)|3,{49,99,112}},
/* 2348 */ {(16<<2)|3,{49,99,114}},
/* 2349 */ {(16<<2)|3,{49,99,117}},
/* 2350 */ {(10<<2)|2,{49,99,0}},
/* 2351 */ {(10<<2)|2,{49,99,0}},
/* 2352 */ {(10<<2)|2,{49,99,0}},
/* 2353 */ {(10<<2)|2,{49,99,0}},
/* 2354 */ {(10<<2)|2,{49,99,0}},
/* 2355 */ {(10<<2)|2,{49,99,0}},
/* 2356 */ {(10<<2)|2,{49,99,0}},
/* 2357 */ {(10<<2)|2,{49,99,0}},
/* 2358 */ {(10<<2)|2,{49,99,0}},
/* 2359 */ {(10<<2)|2,{49,99,0}},
/* 2360 */ {(10<<2)|2,{49,99,0}},
/* 2361 */ {(10<<2)|2,{49,99,0}},
/* 2362 */ {(10<<2)|2,{49,99,0}},
/* 2363 */ {(10<<2)|2,{49,99,0}},
/* 2364 */ {(10<<2)|2,{49,99,0}},
/* 2365 */ {(10<<2)|2,{49,99,0}},
/* 2366 */ {(10<<2)|2,{49,99,0}},
/* 2367 */ {(10<<2)|2,{49,99,0}},
/* 2368 */ {(15<<2)|3,{49,101,48}},
/* 2369 */ {(15<<2)|3,{49,101,48}},
/* 2370 */ {(15<<2)|3,{49,101,49}},
/* 2371 */ {(15<<2)|3,{49,101,49}},
/* 2372 */ {(15<<2)|3,{49,101,50}},
/* 2373 */ {(15<<2)|3,{49,101,50}},
/* 2374 */ {(15<<2)|3,{49,101,97}},
/* 2375 */ {(15<<2)|3,{49,101,97}},
/* 2376 */ {(15<<2)|3,{49,101,99}},
/* 2377 */ {(15<<2)|3,{49,101,99}},
/* 2378 */ {(15<<2)|3,{49,101,101}},
/* 2379 */ {(15<<2)|3,{49,101,101}},
/* 2380 */ {(15<<2)|3,{49,101,105}},
/* 2381 */ {(15<<2)|3,{49,101,105}},
/* 2382 */ {(15<<2)|3,{49,101,111}},
/* 2383 */ {(15<<2)|3,{49,101,111}},
/* 2384 */ {(15<<2)|3,{49,101,115}},
/* 2385 */ {(15<<2)|3,{49,101,115}},
/* 2386 */ {(15<<2)|3,{49,101,116}},
/* 2387 */ {(15<<2)|3,{49,101,116}},
/* 2388 */ {(16<<2)|3,{49,101,32}},
/* 2389 */ {(16<<2)|3,{49,101,37}},
/* 2390 */ {(16<<2)|3,{49,101,45}},
/* 2391 */ {(16<<2)|3,{49,101,46}},
/* 2392 */ {(16<<2)|3,{49,101,47}},
/* 2393 */ {(16<<2)|3,{49,101,51}},
/* 2394 */ {(16<<2)|3,{49,101,52}},
/* 2395 */ {(16<<2)|3,{49,101,53}},
/* 2396 */ {(16<<2)|3,{49,101,54}},
/* 2397 */ {(16<<2)|3,{49,101,55}},
/* 2398 */ {(16<<2)|3,{49,101,56}},
/* 2399 */ {(16<<2)|3,{49,101,57}},
/* 2400 */ {(16<<2)|3,{49,101,61}},
/* 2401 */ {(16<<2)|3,{49,101,65}},
/* 2402 */ {(16<<2)|3,{49,101,95}},
/* 2403 */ {(16<<2)|3,{49,101,98}},
/* 2404 */ {(16<<2)|3,{49,101,100}},
/* 2405 */ {(16<<2)|3,{49,101,102}},
/* 2406 */ {(16<<2)|3,{49,101,103}},
/* 2407 */ {(16<<2)|3,{49,101,104}},
/* 2408 */ {(16<<2)|3,{49,101,108}},
/* 2409 */ {(16<<2)|3,{49,101,109}},
/* 2410 */ {(16<<2)|3,{49,101,110}},
/* 2411 */ {(16<<2)|3,{49,101,112}},
/* 2412 */ {(16<<2)|3,{49,101,114}},
/* 2413 */ {(16<<2)|3,{49,101,117}},
/* 2414 */ {(10<<2)|2,{49,101,0}},
/* 2415 */ {(10<<2)|2,{49,101,0}},
/* 2416 */ {(10<<2)|2,{49,101,0}},
/* 2417 */ {(10<<2)|2,{49,101,0}},
/* 2418 */ {(10<<2)|2,{49,101,0}},
/* 2419 */ {(10<<2)|2,{49,101,0}},
/* 2420 */ {(10<<2)|2,{49,101,0}},
/* 2421 */ {(10<<2)|2,{49,101,0}},
/* 2422 */ {(10<<2)|2,{49,101,0}},
/* 2423 */ {(10<<2)|2,{49,101,0}},
/* 2424 */ {(10<<2)|2,{49,101,0}},
/* 2425 */ {(10<<2)|2,{49,101,0}},
/* 2426 */ {(10<<2)|2,{49,101,0}},
/* 2427 */ {(10<<2)|2,{49,101,0}},
/* 2428 */ {(10<<2)|2,{49,101,0}},
/* 2429 */ {(10<<2)|2,{49,101,0}},
/* 2430 */ {(10<<2)|2,{49,101,0}},
/* 2431 */ {(10<<2)|2,{49,101,0}},
/* 2432 */ {(15<<2)|3,{49,105,48}},
/* 2433 */ {(15<<2)|3,{49,105,48}},
/* 2434 */ {(15<<2)|3,{49,105,49}},
/* 2435 */ {(15<<2)|3,{49,105,49}},
/* 2436 */ {(15<<2)|3,{49,105,50}},
/* 2437 */ {(15<<2)|3,{49,105,50}},
/* 2438 */ {(15<<2)|3,{49,105,97}},
/* 2439 */ {(15<<2)|3,{49,105,97}},
/* 2440 */ {(15<<2)|3,{49,105,99}},
/* 2441 */ {(15<<2)|3,{49,105,99}},
/* 2442 */ {(15<<2)|3,{49,105,101}},
/* 2443 */ {(15<<2)|3,{49,105,101}},
/* 2444 */ {(15<<2)|3,{49,105,105}},
/* 2445 */ {(15<<2)|3,{49,105,105}},
/* 2446 */ {(15<<2)|3,{49,105,111}},
/* 2447 */ {(15<<2)|3,{49,105,111}},
/* 2448 */ {(15<<2)|3,{49,105,115}},
/* 2449 */ {(15<<2)|3,{49,105,115}},
/* 2450 */ {(15<<2)|3,{49,105,116}},
/* 2451 */ {(15<<2)|3,{49,105,116}},
/* 2452 */ {(16<<2)|3,{49,105,32}},
/* 2453 */ {(16<<2)|3,{49,105,37}},
/* 2454 */ {(16<<2)|3,{49,105,45}},
/* 2455 */ {(16<<2)|3,{49,105,46}},
/* 2456 */ {(16<<2)|3,{49,105,47}},
/* 2457 */ {(16<<2)|3,{49,105,51}},
/* 2458 */ {(16<<2)|3,{49,105,52}},
/* 2459 */ {(16<<2)|3,{49,105,53}},
/* 2460 */ {(16<<2)|3,{49,105,54}},
/* 2461 */ {(16<<2)|3,{49,105,55}},
/* 2462 */ {(16<<2)|3,{49,105,56}},
/* 2463 */ {(16<<2)|3,{49,105,57}},
/* 2464 */ {(16<<2)|3,{49,105,61}},
/* 2465 */ {(16<<2)|3,{49,105,65}},
/* 2466 */ {(16<<2)|3,{49,105,95}},
/* 2467 */ {(16<<2)|3,{49,105,98}},
/* 2468 */ {(16<<2)|3,{49,105,100}},
/* 2469 */ {(16<<2)|3,{49,105,102}},
/* 2470 */ {(16<<2)|3,{49,105,103}},
/* 2471 */ {(16<<2)|3,{49,105,104}},
/* 2472 */ {(16<<2)|3,{49,105,108}},
/* 2473 */ {(16<<2)|3,{49,105,109}},
/* 2474 */ {(16<<2)|3,{49,105,110}},
/* 2475 */ {(16<<2)|3,{49,105,112}},
/* 2476 */ {(16<<2)|3,{49,105,114}},
/* 2477 */ {(16<<2)|3,{49,105,117}},
/* 2478 */ {(10<<2)|2,{49,105,0}},
/* 2479 */ {(10<<2)|2,{49,105,0}},
/* 2480 */ {(10<<2)|2,{49,105,0}},
/* 2481 */ {(10<<2)|2,{49,105,0}},
/* 2482 */ {(10<<2)|2,{49,105,0}},
/* 2483 */ {(10<<2)|2,{49,105,0}},
/* 2484 */ {(10<<2)|2,{49,105,0}},
/* 2485 */ {(10<<2)|2,{49,105,0}},
/* 2486 */ {(10<<2)|2,{49,105,0}},
/* 2487 */ {(10<<2)|2,{49,105,0}},
/* 2488 */ {(10<<2)|2,{49,105,0}},
/* 2489 */ {(10<<2)|2,{49,105,0}},
/* 2490 */ {(10<<2)|2,{49,105,0}},
/* 2491 */ {(10<<2)|2,{49,105,0}},
/* 2492 */ {(10<<2)|2,{49,105,0}},
/* 2493 */ {(10<<2)|2,{49,105,0}},
/* 2494 */ {(10<<2)|2,{49,105,0}},
/* 2495 */ {(10<<2)|2,{49,105,0}},
/* 2496 */ {(15<<2)|3,{49,111,48}},
/* 2497 */ {(15<<2)|3,{49,111,48}},
/* 2498 */ {(15<<2)|3,{49,111,49}},
/* 2499 */ {(15<<2)|3,{49,111,49}},
/* 2500 */ {(15<<2)|3,{49,111,50}},
/* 2501 */ {(15<<2)|3,{49,111,50}},
/* 2502 */ {(15<<2)|3,{49,111,97}},
/* 2503 */ {(15<<2)|3,{49,111,97}},
/* 2504 */ {(15<<2)|3,{49,111,99}},
/* 2505 */ {(15<<2)|3,{49,111,99}},
/* 2506 */ {(15<<2)|3,{49,111,101}},
/* 2507 */ {(15<<2)|3,{49,111,101}},
/* 2508 */ {(15<<2)|3,{49,111,105}},
/* 2509 */ {(15<<2)|3,{49,111,105}},
/* 2510 */ {(15<<2)|3,{49,111,111}},
/* 2511 */ {(15<<2)|3,{49,111,111}},
/* 2512 */ {(15<<2)|3,{49,111,115}},
/* 2513 */ {(15<<2)|3,{49,111,115}},
/* 2514 */ {(15<<2)|3,{49,111,116}},
/* 2515 */ {(15<<2)|3,{49,111,116}},
/* 2516 */ {(16<<2)|3,{49,111,32}},
/* 2517 */ {(16<<2)|3,{49,111,37}},
/* 2518 */ {(16<<2)|3,{49,111,45}},
/* 2519 */ {(16<<2)|3,{49,111,46}},
/* 2520 */ {(16<<2)|3,{49,111,47}},
/* 2521 */ {(16<<2)|3,{49,111,51}},
/* 2522 */ {(16<<2)|3,{49,111,52}},
/* 2523 */ {(16<<2)|3,{49,111,53}},
/* 2524 */ {(16<<2)|3,{49,111,54}},
/* 2525 */ {(16<<2)|3,{49,111,55}},
/* 2526 */ {(16<<2)|3,{49,111,56}},
/* 2527 */ {(16<<2)|3,{49,111,57}},
/* 2528 */ {(16<<2)|3,{49,111,61}},
/* 2529 */ {(16<<2)|3,{49,111,65}},
/* 2530 */ {(16<<2)|3,{49,111,95}},
/* 2531 */ {(16<<2)|3,{49,111,98}},
/* 2532 */ {(16<<2)|3,{49,111,100}},
/* 2533 */ {(16<<2)|3,{49,111,102}},
/* 2534 */ {(16<<2)|3,{49,111,103}},
/* 2535 */ {(16<<2)|3,{49,111,104}},
/* 2536 */ {(16<<2)|3,{49,111,108}},
/* 2537 */ {(16<<2)|3,{49,111,109}},
/* 2538 */ {(16<<2)|3,{49,111,110}},
/* 2539 */ {(16<<2)|3,{49,111,112}},
/* 2540 */ {(16<<2)|3,{49,111,114}},
/* 2541 */ {(16<<2)|3,{49,111,117}},
/* 2542 */ {(10<<2)|2,{49,111,0}},
/* 2543 */ {(10<<2)|2,{49,111,0}},
/* 2544 */ {(10<<2)|2,{49,111,0}},
/* 2545 */ {(10<<2)|2,{49,111,0}},
/* 2546 */ {(10<<2)|2,{49,111,0}},
/* 2547 */ {(10<<2)|2,{49,111,0}},
/* 2548 */ {(10<<2)|2,{49,111,0}},
/* 2549 */ {(10<<2)|2,{49,111,0}},
/* 2550 */ {(10<<2)|2,{49,111,0}},
/* 2551 */ {(10<<2)|2,{49,111,0}},
/* 2552 */ {(10<<2)|2,{49,111,0}},
/* 2553 */ {(10<<2)|2,{49,111,0}},
/* 2554 */ {(10<<2)|2,{49,111,0}},
/* 2555 */ {(10<<2)|2,{49,111,0}},
/* 2556 */ {(10<<2)|2,{49,111,0}},
/* 2557 */ {(10<<2)|2,{49,111,0}},
/* 2558 */ {(10<<2)|2,{49,111,0}},
/* 2559 */ {(10<<2)|2,{49,111,0}},
/* 2560 */ {(15<<2)|3,{49,115,48}},
/* 2561 */ {(15<<2)|3,{49,115,48}},
/* 2562 */ {(15<<2)|3,{49,115,49}},
/* 2563 */ {(15<<2)|3,{49,115,49}},
/* 2564 */ {(15<<2)|3,{49,115,50}},
/* 2565 */ {(15<<2)|3,{49,115,50}},
/* 2566 */ {(15<<2)|3,{49,115,97}},
/* 2567 */ {(15<<2)|3,{49,115,97}},
/* 2568 */ {(15<<2)|3,{49,115,99}},
/* 2569 */ {(15<<2)|3,{49,115,99}},
/* 2570 */ {(15<<2)|3,{49,115,101}},
/* 2571 */ {(15<<2)|3,{49,115,101}},
/* 2572 */ {(15<<2)|3,{49,115,105}},
/* 2573 */ {(15<<2)|3,{49,115,105}},
/* 2574 */ {(15<<2)|3,{49,115,111}},
/* 2575 */ {(15<<2)|3,{49,115,111}},
/* 2576 */ {(15<<2)|3,{49,115,115}},
/* 2577 */ {(15<<2)|3,{49,115,115}},
/* 2578 */ {(15<<2)|3,{49,115,116}},
/* 2579 */ {(15<<2)|3,{49,115,116}},
/* 2580 */ {(16<<2)|3,{49,115,32}},
/* 2581 */ {(16<<2)|3,{49,115,37}},
/* 2582 */ {(16<<2)|3,{49,115,45}},
/* 2583 */ {(16<<2)|3,{49,115,46}},
/* 2584 */ {(16<<2)|3,{49,115,47}},
/* 2585 */ {(16<<2)|3,{49,115,51}},
/* 2586 */ {(16<<2)|3,{49,115,52}},
/* 2587 */ {(16<<2)|3,{49,115,53}},
/* 2588 */ {(16<<2)|3,{49,115,54}},
/* 2589 */ {(16<<2)|3,{49,115,55}},
/* 2590 */ {(16<<2)|3,{49,115,56}},
/* 2591 */ {(16<<2)|3,{49,115,57}},
/* 2592 */ {(16<<2)|3,{49,115,61}},
/* 2593 */ {(16<<2)|3,{49,115,65}},
/* 2594 */ {(16<<2)|3,{49,115,95}},
/* 2595 */ {(16<<2)|3,{49,115,98}},
/* 2596 */ {(16<<2)|3,{49,115,100}},
/* 2597 */ {(16<<2)|3,{49,115,102}},
/* 2598 */ {(16<<2)|3,{49,115,103}},
/* 2599 */ {(16<<2)|3,{49,115,104}},
/* 2600 */ {(16<<2)|3,{49,115,108}},
/* 2601 */ {(16<<2)|3,{49,115,109}},
/* 2602 */ {(16<<2)|3,{49,115,110}},
/* 2603 */ {(16<<2)|3,{49,115,112}},
/* 2604 */ {(16<<2)|3,{49,115,114}},
/* 2605 */ {(16<<2)|3,{49,115,117}},
/* 2606 */ {(10<<2)|2,{49,115,0}},
/* 2607 */ {(10<<2)|2,{49,115,0}},
/* 2608 */ {(10<<2)|2,{49,115,0}},
/* 2609 */ {(10<<2)|2,{49,115,0}},
/* 2610 */ {(10<<2)|2,{49,115,0}},
/* 2611 */ {(10<<2)|2,{49,115,0}},
/* 2612 */ {(10<<2)|2,{49,115,0}},
/* 2613 */ {(10<<2)|2,{49,115,0}},
/* 2614 */ {(10<<2)|2,{49,115,0}},
/* 2615 */ {(10<<2)|2,{49,115,0}},
/* 2616 */ {(10<<2)|2,{49,115,0}},
/* 2617 */ {(10<<2)|2,{49,115,0}},
/* 2618 */ {(10<<2)|2,{49,115,0}},
/* 2619 */ {(10<<2)|2,{49,115,0}},
/* 2620 */ {(10<<2)|2,{49,115,0}},
/* 2621 */ {(10<<2)|2,{49,115,0}},
/* 2622 */ {(10<<2)|2,{49,115,0}},
/* 2623 */ {(10<<2)|2,{49,115,0}},
/* 2624 */ {(15<<2)|3,{49,116,48}},
/* 2625 */ {(15<<2)|3,{49,116,48}},
/* 2626 */ {(15<<2)|3,{49,116,49}},
/* 2627 */ {(15<<2)|3,{49,116,49}},
/* 2628 */ {(15<<2)|3,{49,116,50}},
/* 2629 */ {(15<<2)|3,{49,116,50}},
/* 2630 */ {(15<<2)|3,{49,116,97}},
/* 2631 */ {(15<<2)|3,{49,116,97}},
/* 2632 */ {(15<<2)|3,{49,116,99}},
/* 2633 */ {(15<<2)|3,{49,116,99}},
/* 2634 */ {(15<<2)|3,{49,116,101}},
/* 2635 */ {(15<<2)|3,{49,116,101}},
/* 2636 */ {(15<<2)|3,{49,116,105}},
/* 2637 */ {(15<<2)|3,{49,116,105}},
/* 2638 */ {(15<<2)|3,{49,116,111}},
/* 2639 */ {(15<<2)|3,{49,116,111}},
/* 2640 */ {(15<<2)|3,{49,116,115}},
/* 2641 */ {(15<<2)|3,{49,116,115}},
/* 2642 */ {(15<<2)|3,{49,116,116}},
/* 2643 */ {(15<<2)|3,{49,116,116}},
/* 2644 */ {(16<<2)|3,{49,116,32}},
/* 2645 */ {(16<<2)|3,{49,116,37}},
/* 2646 */ {(16<<2)|3,{49,116,45}},
/* 2647 */ {(16<<2)|3,{49,116,46}},
/* 2648 */ {(16<<2)|3,{49,116,47}},
/* 2649 */ {(16<<2)|3,{49,116,51}},
/* 2650 */ {(16<<2)|3,{49,116,52}},
/* 2651 */ {(16<<2)|3,{49,116,53}},
/* 2652 */ {(16<<2)|3,{49,116,54}},
/* 2653 */ {(16<<2)|3,{49,116,55}},
/* 2654 */ {(16<<2)|3,{49,116,56}},
/* 2655 */ {(16<<2)|3,{49,116,57}},
/* 2656 */ {(16<<2)|3,{49,116,61}},
/* 2657 */ {(16<<2)|3,{49,116,65}},
/* 2658 */ {(16<<2)|3,{49,116,95}},
/* 2659 */ {(16<<2)|3,{49,116,98}},
/* 2660 */ {(16<<2)|3,{49,116,100}},
/* 2661 */ {(16<<2)|3,{49,116,102}},
/* 2662 */ {(16<<2)|3,{49,116,103}},
/* 2663 */ {(16<<2)|3,{49,116,104}},
/* 2664 */ {(16<<2)|3,{49,116,108}},
/* 2665 */ {(16<<2)|3,{49,116,109}},
/* 2666 */ {(16<<2)|3,{49,116,110}},
/* 2667 */ {(16<<2)|3,{49,116,112}},
/* 2668 */ {(16<<2)|3,{49,116,114}},
/* 2669 */ {(16<<2)|3,{49,116,117}},
/* 2670 */ {(10<<2)|2,{49,116,0}},
/* 2671 */ {(10<<2)|2,{49,116,0}},
/* 2672 */ {(10<<2)|2,{49,116,0}},
/* 2673 */ {(10<<2)|2,{49,116,0}},
/* 2674 */ {(10<<2)|2,{49,116,0}},
/* 2675 */ {(10<<2)|2,{49,116,0}},
/* 2676 */ {(10<<2)|2,{49,116,0}},
/* 2677 */ {(10<<2)|2,{49,116,0}},
/* 2678 */ {(10<<2)|2,{49,116,0}},
/* 2679 */ {(10<<2)|2,{49,116,0}},
/* 2680 */ {(10<<2)|2,{49,116,0}},
/* 2681 */ {(10<<2)|2,{49,116,0}},
/* 2682 */ {(10<<2)|2,{49,116,0}},
/* 2683 */ {(10<<2)|2,{49,116,0}},
/* 2684 */ {(10<<2)|2,{49,116,0}},
/* 2685 */ {(10<<2)|2,{49,116,0}},
/* 2686 */ {(10<<2)|2,{49,116,0}},
/* 2687 */ {(10<<2)|2,{49,116,0}},
/* 2688 */ {(16<<2)|3,{49,32,48}},
/* 2689 */ {(16<<2)|3,{49,32,49}},
/* 2690 */ {(16<<2)|3,{49,32,50}},
/* 2691 */ {(16<<2)|3,{49,32,97}},
/* 2692 */ {(16<<2)|3,{49,32,99}},
/* 2693 */ {(16<<2)|3,{49,32,101}},
/* 2694 */ {(16<<2)|3,{49,32,105}},
/* 2695 */ {(16<<2)|3,{49,32,111}},
/* 2696 */ {(16<<2)|3,{49,32,115}},
/* 2697 */ {(16<<2)|3,{49,32,116}},
/* 2698 */ {(11<<2)|2,{49,32,0}},
/* 2699 */ {(11<<2)|2,{49,32,0}},
/* 2700 */ {(11<<2)|2,{49,32,0}},
/* 2701 */ {(11<<2)|2,{49,32,0}},
/* 2702 */ {(11<<2)|2,{49,32,0}},
/* 2703 */ {(11<<2)|2,{49,32,0}},
/* 2704 */ {(11<<2)|2,{49,32,0}},
/* 2705 */ {(11<<2)|2,{49,32,0}},
/* 2706 */ {(11<<2)|2,{49,32,0}},
/* 2707 */ {(11<<2)|2,{49,32,0}},
/* 2708 */ {(11<<2)|2,{49,32,0}},
/* 2709 */ {(11<<2)|2,{49,32,0}},
/* 2710 */ {(11<<2)|2,{49,32,0}},
/* 2711 */ {(11<<2)|2,{49,32,0}},
/* 2712 */ {(11<<2)|2,{49,32,0}},
/* 2713 */ {(11<<2)|2,{49,32,0}},
/* 2714 */ {(11<<2)|2,{49,32,0}},
/* 2715 */ {(11<<2)|2,{49,32,0}},
/* 2716 */ {(11<<2)|2,{49,32,0}},
/* 2717 */ {(11<<2)|2,{49,32,0}},
/* 2718 */ {(11<<2)|2,{49,32,0}},
/* 2719 */ {(11<<2)|2,{49,32,0}},
/* 2720 */ {(16<<2)|3,{49,37,48}},
/* 2721 */ {(16<<2)|3,{49,37,49}},
/* 2722 */ {(16<<2)|3,{49,37,50}},
/* 2723 */ {(16<<2)|3,{49,37,97}},
/* 2724 */ {(16<<2)|3,{49,37,99}},
/* 2725 */ {(16<<2)|3,{49,37,101}},
/* 2726 */ {(16<<2)|3,{49,37,105}},
/* 2727 */ {(16<<2)|3,{49,37,111}},
/* 2728 */ {(16<<2)|3,{49,37,115}},
/* 2729 */ {(16<<2)|3,{49,37,116}},
/* 2730 */ {(11<<2)|2,{49,37,0}},
/* 2731 */ {(11<<2)|2,{49,37,0}},
/* 2732 */ {(11<<2)|2,{49,37,0}},
/* 2733 */ {(11<<2)|2,{49,37,0}},
/* 2734 */ {(11<<2)|2,{49,37,0}},
/* 2735 */ {(11<<2)|2,{49,37,0}},
/* 2736 */ {(11<<2)|2,{49,37,0}},
/* 2737 */ {(11<<2)|2,{49,37,0}},
/* 2738 */ {(11<<2)|2,{49,37,0}},
/* 2739 */ {(11<<2)|2,{49,37,0}},
/* 2740 */ {(11<<2)|2,{49,37,0}},
/* 2741 */ {(11<<2)|2,{49,37,0}},
/* 2742 */ {(11<<2)|2,{49,37,0}},
/* 2743 */ {(11<<2)|2,{49,37,0}},
/* 2744 */ {(11<<2)|2,{49,37,0}},
/* 2745 */ {(11<<2)|2,{49,37,0}},
/* 2746 */ {(11<<2)|2,{49,37,0}},
/* 2747 */ {(11<<2)|2,{49,37,0}},
/* 2748 */ {(11<<2)|2,{49,37,0}},
/* 2749 */ {(11<<2)|2,{49,37,0}},
/* 2750 */ {(11<<2)|2,{49,37,0}},
/* 2751 */ {(11<<2)|2,{49,37,0}},
/* 2752 */ {(16<<2)|3,{49,45,48}},
/* 2753 */ {(16<<2)|3,{49,45,49}},
/* 2754 */ {(16<<2)|3,{49,45,50}},
/* 2755 */ {(16<<2)|3,{49,45,97}},
/* 2756 */ {(16<<2)|3,{49,45,99}},
/* 2757 */ {(16<<2)|3,{49,45,101}},
/* 2758 */ {(16<<2)|3,{49,45,105}},
/* 2759 */ {(16<<2)|3,{49,45,111}},
/* 2760 */ {(16<<2)|3,{49,45,115}},
/* 2761 */ {(16<<2)|3,{49,45,116}},
/* 2762 */ {(11<<2)|2,{49,45,0}},
/* 2763 */ {(11<<2)|2,{49,45,0}},
/* 2764 */ {(11<<2)|2,{49,45,0}},
/* 2765 */ {(11<<2)|2,{49,45,0}},
/* 2766 */ {(11<<2)|2,{49,45,0}},
/* 2767 */ {(11<<2)|2,{49,45,0}},
/* 2768 */ {(11<<2)|2,{49,45,0}},
/* 2769 */ {(11<<2)|2,{49,45,0}},
/* 2770 */ {(11<<2)|2,{49,45,0}},
/* 2771 */ {(11<<2)|2,{49,45,0}},
/* 2772 */ {(11<<2)|2,{49,45,0}},
/* 2773 */ {(11<<2)|2,{49,45,0}},
/* 2774 */ {(11<<2)|2,{49,45,0}},
/* 2775 */ {(11<<2)|2,{49,45,0}},
/* 2776 */ {(11<<2)|2,{49,45,0}},
/* 2777 */ {(11<<2)|2,{49,45,0}},
/* 2778 */ {(11<<2)|2,{49,45,0}},
/* 2779 */ {(11<<2)|2,{49,45,0}},
/* 2780 */ {(11<<2)|2,{49,45,0}},
/* 2781 */ {(11<<2)|2,{49,45,0}},
/* 2782 */ {(11<<2)|2,{49,45,0}},
/* 2783 */ {(11<<2)|2,{49,45,0}},
/* 2784 */ {(16<<2)|3,{49,46,48}},
/* 2785 */ {(16<<2)|3,{49,46,49}},
/* 2786 */ {(16<<2)|3,{49,46,50}},
/* 2787 */ {(16<<2)|3,{49,46,97}},
/* 2788 */ {(16<<2)|3,{49,46,99}},
/* 2789 */ {(16<<2)|3,{49,46,101}},
/* 2790 */ {(16<<2)|3,{49,46,105}},
/* 2791 */ {(16<<2)|3,{49,46,111}},
/* 2792 */ {(16<<2)|3,{49,46,115}},
/* 2793 */ {(16<<2)|3,{49,46,116}},
/* 2794 */ {(11<<2)|2,{49,46,0}},
/* 2795 */ {(11<<2)|2,{49,46,0}},
/* 2796 */ {(11<<2)|2,{49,46,0}},
/* 2797 */ {(11<<2)|2,{49,46,0}},
/* 2798 */ {(11<<2)|2,{49,46,0}},
/* 2799 */ {(11<<2)|2,{49,46,0}},
/* 2800 */ {(11<<2)|2,{49,46,0}},
/* 2801 */ {(11<<2)|2,{49,46,0}},
/* 2802 */ {(11<<2)|2,{49,46,0}},
/* 2803 */ {(11<<2)|2,{49,46,0}},
/* 2804 */ {(11<<2)|2,{49,46,0}},
/* 2805 */ {(11<<2)|2,{49,46,0}},
/* 2806 */ {(11<<2)|2,{49,46,0}},
/* 2807 */ {(11<<2)|2,{49,46,0}},
/* 2808 */ {(11<<2)|2,{49,46,0}},
/* 2809 */ {(11<<2)|2,{49,46,0}},
/* 2810 */ {(11<<2)|2,{49,46,0}},
/* 2811 */ {(11<<2)|2,{49,46,0}},
/* 2812 */ {(11<<2)|2,{49,46,0}},
/* 2813 */ {(11<<2)|2,{49,46,0}},
/* 2814 */ {(11<<2)|2,{49,46,0}},
/* 2815 */ {(11<<2)|2,{49,46,0}},
/* 2816 */ {(16<<2)|3,{49,47,48}},
/* 2817 */ {(16<<2)|3,{49,47,49}},
/* 2818 */ {(16<<2)|3,{49,47,50}},
/* 2819 */ {(16<<2)|3,{49,47,97}},
/* 2820 */ {(16<<2)|3,{49,47,99}},
/* 2821 */ {(16<<2)|3,{49,47,101}},
/* 2822 */ {(16<<2)|3,{49,47,105}},
/* 2823 */ {(16<<2)|3,{49,47,111}},
/* 2824 */ {(16<<2)|3,{49,47,115}},
/* 2825 */ {(16<<2)|3,{49,47,116}},
/* 2826 */ {(11<<2)|2,{49,47,0}},
/* 2827 */ {(11<<2)|2,{49,47,0}},
/* 2828 */ {(11<<2)|2,{49,47,0}},
/* 2829 */ {(11<<2)|2,{49,47,0}},
/* 2830 */ {(11<<2)|2,{49,47,0}},
/* 2831 */ {(11<<2)|2,{49,47,0}},
/* 2832 */ {(11<<2)|2,{49,47,0}},
/* 2833 */ {(11<<2)|2,{49,47,0}},
/* 2834 */ {(11<<2)|2,{49,47,0}},
/* 2835 */ {(11<<2)|2,{49,47,0}},
/* 2836 */ {(11<<2)|2,{49,47,0}},
/* 2837 */ {(11<<2)|2,{49,47,0}},
/* 2838 */ {(11<<2)|2,{49,47,0}},
/* 2839 */ {(11<<2)|2,{49,47,0}},
/* 2840 */ {(11<<2)|2,{49,47,0}},
/* 2841 */ {(11<<2)|2,{49,47,0}},
/* 2842 */ {(11<<2)|2,{49,47,0}},
/* 2843 */ {(11<<2)|2,{49,47,0}},
/* 2844 */ {(11<<2)|2,{49,47,0}},
/* 2845 */ {(11<<2)|2,{49,47,0}},
/* 2846 */ {(11<<2)|2,{49,47,0}},
/* 2847 */ {(11<<2)|2,{49,47,0}},
/* 2848 */ {(16<<2)|3,{49,51,48}},
/* 2849 */ {(16<<2)|3,{49,51,49}},
/* 2850 */ {(16<<2)|3,{49,51,50}},
/* 2851 */ {(16<<2)|3,{49,51,97}},
/* 2852 */ {(16<<2)|3,{49,51,99}},
/* 2853 */ {(16<<2)|3,{49,51,101}},
/* 2854 */ {(16<<2)|3,{49,51,105}},
/* 2855 */ {(16<<2)|3,{49,51,111}},
/* 2856 */ {(16<<2)|3,{49,51,115}},
/* 2857 */ {(16<<2)|3,{49,51,116}},
/* 2858 */ {(11<<2)|2,{49,51,0}},
/* 2859 */ {(11<<2)|2,{49,51,0}},
/* 2860 */ {(11<<2)|2,{49,51,0}},
/* 2861 */ {(11<<2)|2,{49,51,0}},
/* 2862 */ {(11<<2)|2,{49,51,0}},
/* 2863 */ {(11<<2)|2,{49,51,0}},
/* 2864 */ {(11<<2)|2,{49,51,0}},
/* 2865 */ {(11<<2)|2,{49,51,0}},
/* 2866 */ {(11<<2)|2,{49,51,0}},
/* 2867 */ {(11<<2)|2,{49,51,0}},
/* 2868 */ {(11<<2)|2,{49,51,0}},
/* 2869 */ {(11<<2)|2,{49,51,0}},
/* 2870 */ {(11<<2)|2,{49,51,0}},
/* 2871 */ {(11<<2)|2,{49,51,0}},
/* 2872 */ {(11<<2)|2,{49,51,0}},
/* 2873 */ {(11<<2)|2,{49,51,0}},
/* 2874 */ {(11<<2)|2,{49,51,0}},
/* 2875 */ {(11<<2)|2,{49,51,0}},
/* 2876 */ {(11<<2)|2,{49,51,0}},
/* 2877 */ {(11<<2)|2,{49,51,0}},
/* 2878 */ {(11<<2)|2,{49,51,0}},
/* 2879 */ {(11<<2)|2,{49,51,0}},
/* 2880 */ {(16<<2)|3,{49,52,48}},
/* 2881 */ {(16<<2)|3,{49,52,49}},
/* 2882 */ {(16<<2)|3,{49,52,50}},
/* 2883 */ {(16<<2)|3,{49,52,97}},
/* 2884 */ {(16<<2)|3,{49,52,99}},
/* 2885 */ {(16<<2)|3,{49,52,101}},
/* 2886 */ {(16<<2)|3,{49,52,105}},
/* 2887 */ {(16<<2)|3,{49,52,111}},
/* 2888 */ {(16<<2)|3,{49,52,115}},
/* 2889 */ {(16<<2)|3,{49,52,116}},
/* 2890 */ {(11<<2)|2,{49,52,0}},
/* 2891 */ {(11<<2)|2,{49,52,0}},
/* 2892 */ {(11<<2)|2,{49,52,0}},
/* 2893 */ {(11<<2)|2,{49,52,0}},
/* 2894 */ {(11<<2)|2,{49,52,0}},
/* 2895 */ {(11<<2)|2,{49,52,0}},
/* 2896 */ {(11<<2)|2,{49,52,0}},
/* 2897 */ {(11<<2)|2,{49,52,0}},
/* 2898 */ {(11<<2)|2,{49,52,0}},
/* 2899 */ {(11<<2)|2,{49,52,0}},
/* 2900 */ {(11<<2)|2,{49,52,0}},
/* 2901 */ {(11<<2)|2,{49,52,0}},
/* 2902 */ {(11<<2)|2,{49,52,0}},
/* 2903 */ {(11<<2)|2,{49,52,0}},
/* 2904 */ {(11<<2)|2,{49,52,0}},
/* 2905 */ {(11<<2)|2,{49,52,0}},
/* 2906 */ {(11<<2)|2,{49,52,0}},
/* 2907 */ {(11<<2)|2,{49,52,0}},
/* 2908 */ {(11<<2)|2,{49,52,0}},
/* 2909 */ {(11<<2)|2,{49,52,0}},
/* 2910 */ {(11<<2)|2,{49,52,0}},
/* 2911 */ {(11<<2)|2,{49,52,0}},
/* 2912 */ {(16<<2)|3,{49,53,48}},
/* 2913 */ {(16<<2)|3,{49,53,49}},
/* 2914 */ {(16<<2)|3,{49,53,50}},
/* 2915 */ {(16<<2)|3,{49,53,97}},
/* 2916 */ {(16<<2)|3,{49,53,99}},
/* 2917 */ {(16<<2)|3,{49,53,101}},
/* 2918 */ {(16<<2)|3,{49,53,105}},
/* 2919 */ {(16<<2)|3,{49,53,111}},
/* 2920 */ {(16<<2)|3,{49,53,115}},
/* 2921 */ {(16<<2)|3,{49,53,116}},
/* 2922 */ {(11<<2)|2,{49,53,0}},
/* 2923 */ {(11<<2)|2,{49,53,0}},
/* 2924 */ {(11<<2)|2,{49,53,0}},
/* 2925 */ {(11<<2)|2,{49,53,0}},
/* 2926 */ {(11<<2)|2,{49,53,0}},
/* 2927 */ {(11<<2)|2,{49,53,0}},
/* 2928 */ {(11<<2)|2,{49,53,0}},
/* 2929 */ {(11<<2)|2,{49,53,0}},
/* 2930 */ {(11<<2)|2,{49,53,0}},
/* 2931 */ {(11<<2)|2,{49,53,0}},
/* 2932 */ {(11<<2)|2,{49,53,0}},
/* 2933 */ {(11<<2)|2,{49,53,0}},
/* 2934 */ {(11<<2)|2,{49,53,0}},
/* 2935 */ {(11<<2)|2,{49,53,0}},
/* 2936 */ {(11<<2)|2,{49,53,0}},
/* 2937 */ {(11<<2)|2,{49,53,0}},
/* 2938 */ {(11<<2)|2,{49,53,0}},
/* 2939 */ {(11<<2)|2,{49,53,0}},
/* 2940 */ {(11<<2)|2,{49,53,0}},
/* 2941 */ {(11<<2)|2,{49,53,0}},
/* 2942 */ {(11<<2)|2,{49,53,0}},
/* 2943 */ {(11<<2)|2,{49,53,0}},
/* 2944 */ {(16<<2)|3,{49,54,48}},
/* 2945 */ {(16<<2)|3,{49,54,49}},
/* 2946 */ {(16<<2)|3,{49,54,50}},
/* 2947 */ {(16<<2)|3,{49,54,97}},
/* 2948 */ {(16<<2)|3,{49,54,99}},
/* 2949 */ {(16<<2)|3,{49,54,101}},
/* 2950 */ {(16<<2)|3,{49,54,105}},
/* 2951 */ {(16<<2)|3,{49,54,111}},
/* 2952 */ {(16<<2)|3,{49,54,115}},
/* 2953 */ {(16<<2)|3,{49,54,116}},
/* 2954 */ {(11<<2)|2,{49,54,0}},
/* 2955 */ {(11<<2)|2,{49,54,0}},
/* 2956 */ {(11<<2)|2,{49,54,0}},
/* 2957 */ {(11<<2)|2,{49,54,0}},
/* 2958 */ {(11<<2)|2,{49,54,0}},
/* 2959 */ {(11<<2)|2,{49,54,0}},
/* 2960 */ {(11<<2)|2,{49,54,0}},
/* 2961 */ {(11<<2)|2,{49,54,0}},
/* 2962 */ {(11<<2)|2,{49,54,0}},
/* 2963 */ {(11<<2)|2,{49,54,0}},
/* 2964 */ {(11<<2)|2,{49,54,0}},
/* 2965 */ {(11<<2)|2,{49,54,0}},
/* 2966 */ {(11<<2)|2,{49,54,0}},
/* 2967 */ {(11<<2)|2,{49,54,0}},
/* 2968 */ {(11<<2)|2,{49,54,0}},
/* 2969 */ {(11<<2)|2,{49,54,0}},
/* 2970 */ {(11<<2)|2,{49,54,0}},
/* 2971 */ {(11<<2)|2,{49,54,0}},
/* 2972 */ {(11<<2)|2,{49,54,0}},
/* 2973 */ {(11<<2)|2,{49,54,0}},
/* 2974 */ {(11<<2)|2,{49,54,0}},
/* 2975 */ {(11<<2)|2,{49,54,0}},
/* 2976 */ {(16<<2)|3,{49,55,48}},
/* 2977 */ {(16<<2)|3,{49,55,49}},
/* 2978 */ {(16<<2)|3,{49,55,50}},
/* 2979 */ {(16<<2)|3,{49,55,97}},
/* 2980 */ {(16<<2)|3,{49,55,99}},
/* 2981 */ {(16<<2)|3,{49,55,101}},
/* 2982 */ {(16<<2)|3,{49,55,105}},
/* 2983 */ {(16<<2)|3,{49,55,111}},
/* 2984 */ {(16<<2)|3,{49,55,115}},
/* 2985 */ {(16<<2)|3,{49,55,116}},
/* 2986 */ {(11<<2)|2,{49,55,0}},
/* 2987 */ {(11<<2)|2,{49,55,0}},
/* 2988 */ {(11<<2)|2,{49,55,0}},
/* 2989 */ {(11<<2)|2,{49,55,0}},
/* 2990 */ {(11<<2)|2,{49,55,0}},
/* 2991 */ {(11<<2)|2,{49,55,0}},
/* 2992 */ {(11<<2)|2,{49,55,0}},
/* 2993 */ {(11<<2)|2,{49,55,0}},
/* 2994 */ {(11<<2)|2,{49,55,0}},
/* 2995 */ {(11<<2)|2,{49,55,0}},
/* 2996 */ {(11<<2)|2,{49,55,0}},
/* 2997 */ {(11<<2)|2,{49,55,0}},
/* 2998 */ {(11<<2)|2,{49,55,0}},
/* 2999 */ {(11<<2)|2,{49,55,0}},
/* 3000 */ {(11<<2)|2,{49,55,0}},
/* 3001 */ {(11<<2)|2,{49,55,0}},
/* 3002 */ {(11<<2)|2,{49,55,0}},
/* 3003 */ {(11<<2)|2,{49,55,0}},
/* 3004 */ {(11<<2)|2,{49,55,0}},
/* 3005 */ {(11<<2)|2,{49,55,0}},
/* 3006 */ {(11<<2)|2,{49,55,0}},
/* 3007 */ {(11<<2)|2,{49,55,0}},
/* 3008 */ {(16<<2)|3,{49,56,48}},
/* 3009 */ {(16<<2)|3,{49,56,49}},
/* 3010 */ {(16<<2)|3,{49,56,50}},
/* 3011 */ {(16<<2)|3,{49,56,97}},
/* 3012 */ {(16<<2)|3,{49,56,99}},
/* 3013 */ {(16<<2)|3,{49,56,101}},
/* 3014 */ {(16<<2)|3,{49,56,105}},
/* 3015 */ {(16<<2)|3,{49,56,111}},
/* 3016 */ {(16<<2)|3,{49,56,115}},
/* 3017 */ {(16<<2)|3,{49,56,116}},
/* 3018 */ {(11<<2)|2,{49,56,0}},
/* 3019 */ {(11<<2)|2,{49,56,0}},
/* 3020 */ {(11<<2)|2,{49,56,0}},
/* 3021 */ {(11<<2)|2,{49,56,0}},
/* 3022 */ {(11<<2)|2,{49,56,0}},
/* 3023 */ {(11<<2)|2,{49,56,0}},
/* 3024 */ {(11<<2)|2,{49,56,0}},
/* 3025 */ {(11<<2)|2,{49,56,0}},
/* 3026 */ {(11<<2)|2,{49,56,0}},
/* 3027 */ {(11<<2)|2,{49,56,0}},
/* 3028 */ {(11<<2)|2,{49,56,0}},
/* 3029 */ {(11<<2)|2,{49,56,0}},
/* 3030 */ {(11<<2)|2,{49,56,0}},
/* 3031 */ {(11<<2)|2,{49,56,0}},
/* 3032 */ {(11<<2)|2,{49,56,0}},
/* 3033 */ {(11<<2)|2,{49,56,0}},
/* 3034 */ {(11<<2)|2,{49,56,0}},
/* 3035 */ {(11<<2)|2,{49,56,0}},
/* 3036 */ {(11<<2)|2,{49,56,0}},
/* 3037 */ {(11<<2)|2,{49,56,0}},
/* 3038 */ {(11<<2)|2,{49,56,0}},
/* 3039 */ {(11<<2)|2,{49,56,0}},
/* 3040 */ {(16<<2)|3,{49,57,48}},
/* 3041 */ {(16<<2)|3,{49,57,49}},
/* 3042 */ {(16<<2)|3,{49,57,50}},
/* 3043 */ {(16<<2)|3,{49,57,97}},
/* 3044 */ {(16<<2)|3,{49,57,99}},
/* 3045 */ {(16<<2)|3,{49,57,101}},
/* 3046 */ {(16<<2)|3,{49,57,105}},
/* 3047 */ {(16<<2)|3,{49,57,111}},
/* 3048 */ {(16<<2)|3,{49,57,115}},
/* 3049 */ {(16<<2)|3,{49,57,116}},
/* 3050 */ {(11<<2)|2,{49,57,0}},
/* 3051 */ {(11<<2)|2,{49,57,0}},
/* 3052 */ {(11<<2)|2,{49,57,0}},
/* 3053 */ {(11<<2)|2,{49,57,0}},
/* 3054 */ {(11<<2)|2,{49,57,0}},
/* 3055 */ {(11<<2)|2,{49,57,0}},
/* 3056 */ {(11<<2)|2,{49,57,0}},
/* 3057 */ {(11<<2)|2,{49,57,0}},
/* 3058 */ {(11<<2)|2,{49,57,0}},
/* 3059 */ {(11<<2)|2,{49,57,0}},
/* 3060 */ {(11<<2)|2,{49,57,0}},
/* 3061 */ {(11<<2)|2,{49,57,0}},
/* 3062 */ {(11<<2)|2,{49,57,0}},
/* 3063 */ {(11<<2)|2,{49,57,0}},
/* 3064 */ {(11<<2)|2,{49,57,0}},
/* 3065 */ {(11<<2)|2,{49,57,0}},
/* 3066 */ {(11<<2)|2,{49,57,0}},
/* 3067 */ {(11<<2)|2,{49,57,0}},
/* 3068 */ {(11<<2)|2,{49,57,0}},
/* 3069 */ {(11<<2)|2,{49,57,0}},
/* 3070 */ {(11<<2)|2,{49,57,0}},
/* 3071 */ {(11<<2)|2,{49,57,0}},
/* 3072 */ {(16<<2)|3,{49,61,48}},
/* 3073 */ {(16<<2)|3,{49,61,49}},
/* 3074 */ {(16<<2)|3,{49,61,50}},
/* 3075 */ {(16<<2)|3,{49,61,97}},
/* 3076 */ {(16<<2)|3,{49,61,99}},
/* 3077 */ {(16<<2)|3,{49,61,101}},
/* 3078 */ {(16<<2)|3,{49,61,105}},
/* 3079 */ {(16<<2)|3,{49,61,111}},
/* 3080 */ {(16<<2)|3,{49,61,115}},
/* 3081 */ {(16<<2)|3,{49,61,116}},
/* 3082 */ {(11<<2)|2,{49,61,0}},
/* 3083 */ {(11<<2)|2,{49,61,0}},
/* 3084 */ {(11<<2)|2,{49,61,0}},
/* 3085 */ {(11<<2)|2,{49,61,0}},
/* 3086 */ {(11<<2)|2,{49,61,0}},
/* 3087 */ {(11<<2)|2,{49,61,0}},
/* 3088 */ {(11<<2)|2,{49,61,0}},
/* 3089 */ {(11<<2)|2,{49,61,0}},
/* 3090 */ {(11<<2)|2,{49,61,0}},
/* 3091 */ {(11<<2)|2,{49,61,0}},
/* 3092 */ {(11<<2)|2,{49,61,0}},
/* 3093 */ {(11<<2)|2,{49,61,0}},
/* 3094 */ {(11<<2)|2,{49,61,0}},
/* 3095 */ {(11<<2)|2,{49,61,0}},
/* 3096 */ {(11<<2)|2,{49,61,0}},
/* 3097 */ {(11<<2)|2,{49,61,0}},
/* 3098 */ {(11<<2)|2,{49,61,0}},
/* 3099 */ {(11<<2)|2,{49,61,0}},
/* 3100 */ {(11<<2)|2,{49,61,0}},
/* 3101 */ {(11<<2)|2,{49,61,0}},
/* 3102 */ {(11<<2)|2,{49,61,0}},
/* 3103 */ {(11<<2)|2,{49,61,0}},
/* 3104 */ {(16<<2)|3,{49,65,48}},
/* 3105 */ {(16<<2)|3,{49,65,49}},
/* 3106 */ {(16<<2)|3,{49,65,50}},
/* 3107 */ {(16<<2)|3,{49,65,97}},
/* 3108 */ {(16<<2)|3,{49,65,99}},
/* 3109 */ {(16<<2)|3,{49,65,101}},
/* 3110 */ {(16<<2)|3,{49,65,105}},
/* 3111 */ {(16<<2)|3,{49,65,111}},
/* 3112 */ {(16<<2)|3,{49,65,115}},
/* 3113 */ {(16<<2)|3,{49,65,116}},
/* 3114 */ {(11<<2)|2,{49,65,0}},
/* 3115 */ {(11<<2)|2,{49,65,0}},
/* 3116 */ {(11<<2)|2,{49,65,0}},
/* 3117 */ {(11<<2)|2,{49,65,0}},
/* 3118 */ {(11<<2)|2,{49,65,0}},
/* 3119 */ {(11<<2)|2,{49,65,0}},
/* 3120 */ {(11<<2)|2,{49,65,0}},
/* 3121 */ {(11<<2)|2,{49,65,0}},
/* 3122 */ {(11<<2)|2,{49,65,0}},
/* 3123 */ {(11<<2)|2,{49,65,0}},
/* 3124 */ {(11<<2)|2,{49,65,0}},
/* 3125 */ {(11<<2)|2,{49,65,0}},
/* 3126 */ {(11<<2)|2,{49,65,0}},
/* 3127 */ {(11<<2)|2,{49,65,0}},
/* 3128 */ {(11<<2)|2,{49,65,0}},
/* 3129 */ {(11<<2)|2,{49,65,0}},
/* 3130 */ {(11<<2)|2,{49,65,0}},
/* 3131 */ {(11<<2)|2,{49,65,0}},
/* 3132 */ {(11<<2)|2,{49,65,0}},
/* 3133 */ {(11<<2)|2,{49,65,0}},
/* 3134 */ {(11<<2)|2,{49,65,0}},
/* 3135 */ {(11<<2)|2,{49,65,0}},
/* 3136 */ {(16<<2)|3,{49,95,48}},
/* 3137 */ {(16<<2)|3,{49,95,49}},
/* 3138 */ {(16<<2)|3,{49,95,50}},
/* 3139 */ {(16<<2)|3,{49,95,97}},
/* 3140 */ {(16<<2)|3,{49,95,99}},
/* 3141 */ {(16<<2)|3,{49,95,101}},
/* 3142 */ {(16<<2)|3,{49,95,105}},
/* 3143 */ {(16<<2)|3,{49,95,111}},
/* 3144 */ {(16<<2)|3,{49,95,115}},
/* 3145 */ {(16<<2)|3,{49,95,116}},
/* 3146 */ {(11<<2)|2,{49,95,0}},
/* 3147 */ {(11<<2)|2,{49,95,0}},
/* 3148 */ {(11<<2)|2,{49,95,0}},
/* 3149 */ {(11<<2)|2,{49,95,0}},
/* 3150 */ {(11<<2)|2,{49,95,0}},
/* 3151 */ {(11<<2)|2,{49,95,0}},
/* 3152 */ {(11<<2)|2,{49,95,0}},
/* 3153 */ {(11<<2)|2,{49,95,0}},
/* 3154 */ {(11<<2)|2,{49,95,0}},
/* 3155 */ {(11<<2)|2,{49,95,0}},
/* 3156 */ {(11<<2)|2,{49,95,0}},
/* 3157 */ {(11<<2)|2,{49,95,0}},
/* 3158 */ {(11<<2)|2,{49,95,0}},
/* 3159 */ {(11<<2)|2,{49,95,0}},
/* 3160 */ {(11<<2)|2,{49,95,0}},
/* 3161 */ {(11<<2)|2,{49,95,0}},
/* 3162 */ {(11<<2)|2,{49,95,0}},
/* 3163 */ {(11<<2)|2,{49,95,0}},
/* 3164 */ {(11<<2)|2,{49,95,0}},
/* 3165 */ {(11<<2)|2,{49,95,0}},
/* 3166 */ {(11<<2)|2,{49,95,0}},
/* 3167 */ {(11<<2)|2,{49,95,0}},
/* 3168 */ {(16<<2)|3,{49,98,48}},
/* 3169 */ {(16<<2)|3,{49,98,49}},
/* 3170 */ {(16<<2)|3,{49,98,50}},
/* 3171 */ {(16<<2)|3,{49,98,97}},
/* 3172 */ {(16<<2)|3,{49,98,99}},
/* 3173 */ {(16<<2)|3,{49,98,101}},
/* 3174 */ {(16<<2)|3,{49,98,105}},
/* 3175 */ {(16<<2)|3,{49,98,111}},
/* 3176 */ {(16<<2)|3,{49,98,115}},
/* 3177 */ {(16<<2)|3,{49,98,116}},
/* 3178 */ {(11<<2)|2,{49,98,0}},
/* 3179 */ {(11<<2)|2,{49,98,0}},
/* 3180 */ {(11<<2)|2,{49,98,0}},
/* 3181 */ {(11<<2)|2,{49,98,0}},
/* 3182 */ {(11<<2)|2,{49,98,0}},
/* 3183 */ {(11<<2)|2,{49,98,0}},
/* 3184 */ {(11<<2)|2,{49,98,0}},
/* 3185 */ {(11<<2)|2,{49,98,0}},
/* 3186 */ {(11<<2)|2,{49,98,0}},
/* 3187 */ {(11<<2)|2,{49,98,0}},
/* 3188 */ {(11<<2)|2,{49,98,0}},
/* 3189 */ {(11<<2)|2,{49,98,0}},
/* 3190 */ {(11<<2)|2,{49,98,0}},
/* 3191 */ {(11<<2)|2,{49,98,0}},
/* 3192 */ {(11<<2)|2,{49,98,0}},
/* 3193 */ {(11<<2)|2,{49,98,0}},
/* 3194 */ {(11<<2)|2,{49,98,0}},
/* 3195 */ {(11<<2)|2,{49,98,0}},
/* 3196 */ {(11<<2)|2,{49,98,0}},
/* 3197 */ {(11<<2)|2,{49,98,0}},
/* 3198 */ {(11<<2)|2,{49,98,0}},
/* 3199 */ {(11<<2)|2,{49,98,0}},
/* 3200 */ {(16<<2)|3,{49,100,48}},
/* 3201 */ {(16<<2)|3,{49,100,49}},
/* 3202 */ {(16<<2)|3,{49,100,50}},
/* 3203 */ {(16<<2)|3,{49,100,97}},
/* 3204 */ {(16<<2)|3,{49,100,99}},
/* 3205 */ {(16<<2)|3,{49,100,101}},
/* 3206 */ {(16<<2)|3,{49,100,105}},
/* 3207 */ {(16<<2)|3,{49,100,111}},
/* 3208 */ {(16<<2)|3,{49,100,115}},
/* 3209 */ {(16<<2)|3,{49,100,116}},
/* 3210 */ {(11<<2)|2,{49,100,0}},
/* 3211 */ {(11<<2)|2,{49,100,0}},
/* 3212 */ {(11<<2)|2,{49,100,0}},
/* 3213 */ {(11<<2)|2,{49,100,0}},
/* 3214 */ {(11<<2)|2,{49,100,0}},
/* 3215 */ {(11<<2)|2,{49,100,0}},
/* 3216 */ {(11<<2)|2,{49,100,0}},
/* 3217 */ {(11<<2)|2,{49,100,0}},
/* 3218 */ {(11<<2)|2,{49,100,0}},
/* 3219 */ {(11<<2)|2,{49,100,0}},
/* 3220 */ {(11<<2)|2,{49,100,0}},
/* 3221 */ {(11<<2)|2,{49,100,0}},
/* 3222 */ {(11<<2)|2,{49,100,0}},
/* 3223 */ {(11<<2)|2,{49,100,0}},
/* 3224 */ {(11<<2)|2,{49,100,0}},
/* 3225 */ {(11<<2)|2,{49,100,0}},
/* 3226 */ {(11<<2)|2,{49,100,0}},
/* 3227 */ {(11<<2)|2,{49,100,0}},
/* 3228 */ {(11<<2)|2,{49,100,0}},
/* 3229 */ {(11<<2)|2,{49,100,0}},
/* 3230 */ {(11<<2)|2,{49,100,0}},
/* 3231 */ {(11<<2)|2,{49,100,0}},
/* 3232 */ {(16<<2)|3,{49,102,48}},
/* 3233 */ {(16<<2)|3,{49,102,49}},
/* 3234 */ {(16<<2)|3,{49,102,50}},
/* 3235 */ {(16<<2)|3,{49,102,97}},
/* 3236 */ {(16<<2)|3,{49,102,99}},
/* 3237 */ {(16<<2)|3,{49,102,101}},
/* 3238 */ {(16<<2)|3,{49,102,105}},
/* 3239 */ {(16<<2)|3,{49,102,111}},
/* 3240 */ {(16<<2)|3,{49,102,115}},
/* 3241 */ {(16<<2)|3,{49,102,116}},
/* 3242 */ {(11<<2)|2,{49,102,0}},
/* 3243 */ {(11<<2)|2,{49,102,0}},
/* 3244 */ {(11<<2)|2,{49,102,0}},
/* 3245 */ {(11<<2)|2,{49,102,0}},
/* 3246 */ {(11<<2)|2,{49,102,0}},
/* 3247 */ {(11<<2)|2,{49,102,0}},
/* 3248 */ {(11<<2)|2,{49,102,0}},
/* 3249 */ {(11<<2)|2,{49,102,0}},
/* 3250 */ {(11<<2)|2,{49,102,0}},
/* 3251 */ {(11<<2)|2,{49,102,0}},
/* 3252 */ {(11<<2)|2,{49,102,0}},
/* 3253 */ {(11<<2)|2,{49,102,0}},
/* 3254 */ {(11<<2)|2,{49,102,0}},
/* 3255 */ {(11<<2)|2,{49,102,0}},
/* 3256 */ {(11<<2)|2,{49,102,0}},
/* 3257 */ {(11<<2)|2,{49,102,0}},
/* 3258 */ {(11<<2)|2,{49,102,0}},
/* 3259 */ {(11<<2)|2,{49,102,0}},
/* 3260 */ {(11<<2)|2,{49,102,0}},
/* 3261 */ {(11<<2)|2,{49,102,0}},
/* 3262 */ {(11<<2)|2,{49,102,0}},
/* 3263 */ {(11<<2)|2,{49,102,0}},
/* 3264 */ {(16<<2)|3,{49,103,48}},
/* 3265 */ {(16<<2)|3,{49,103,49}},
/* 3266 */ {(16<<2)|3,{49,103,50}},
/* 3267 */ {(16<<2)|3,{49,103,97}},
/* 3268 */ {(16<<2)|3,{49,103,99}},
/* 3269 */ {(16<<2)|3,{49,103,101}},
/* 3270 */ {(16<<2)|3,{49,103,105}},
/* 3271 */ {(16<<2)|3,{49,103,111}},
/* 3272 */ {(16<<2)|3,{49,103,115}},
/* 3273 */ {(16<<2)|3,{49,103,116}},
/* 3274 */ {(11<<2)|2,{49,103,0}},
/* 3275 */ {(11<<2)|2,{49,103,0}},
/* 3276 */ {(11<<2)|2,{49,103,0}},
/* 3277 */ {(11<<2)|2,{49,103,0}},
/* 3278 */ {(11<<2)|2,{49,103,0}},
/* 3279 */ {(11<<2)|2,{49,103,0}},
/* 3280 */ {(11<<2)|2,{49,103,0}},
/* 3281 */ {(11<<2)|2,{49,103,0}},
/* 3282 */ {(11<<2)|2,{49,103,0}},
/* 3283 */ {(11<<2)|2,{49,103,0}},
/* 3284 */ {(11<<2)|2,{49,103,0}},
/* 3285 */ {(11<<2)|2,{49,103,0}},
/* 3286 */ {(11<<2)|2,{49,103,0}},
/* 3287 */ {(11<<2)|2,{49,103,0}},
/* 3288 */ {(11<<2)|2,{49,103,0}},
/* 3289 */ {(11<<2)|2,{49,103,0}},
/* 3290 */ {(11<<2)|2,{49,103,0}},
/* 3291 */ {(11<<2)|2,{49,103,0}},
/* 3292 */ {(11<<2)|2,{49,103,0}},
/* 3293 */ {(11<<2)|2,{49,103,0}},
/* 3294 */ {(11<<2)|2,{49,103,0}},
/* 3295 */ {(11<<2)|2,{49,103,0}},
/* 3296 */ {(16<<2)|3,{49,104,48}},
/* 3297 */ {(16<<2)|3,{49,104,49}},
/* 3298 */ {(16<<2)|3,{49,104,50}},
/* 3299 */ {(16<<2)|3,{49,104,97}},
/* 3300 */ {(16<<2)|3,{49,104,99}},
/* 3301 */ {(16<<2)|3,{49,104,101}},
/* 3302 */ {(16<<2)|3,{49,104,105}},
/* 3303 */ {(16<<2)|3,{49,104,111}},
/* 3304 */ {(16<<2)|3,{49,104,115}},
/* 3305 */ {(16<<2)|3,{49,104,116}},
/* 3306 */ {(11<<2)|2,{49,104,0}},
/* 3307 */ {(11<<2)|2,{49,104,0}},
/* 3308 */ {(11<<2)|2,{49,104,0}},
/* 3309 */ {(11<<2)|2,{49,104,0}},
/* 3310 */ {(11<<2)|2,{49,104,0}},
/* 3311 */ {(11<<2)|2,{49,104,0}},
/* 3312 */ {(11<<2)|2,{49,104,0}},
/* 3313 */ {(11<<2)|2,{49,104,0}},
/* 3314 */ {(11<<2)|2,{49,104,0}},
/* 3315 */ {(11<<2)|2,{49,104,0}},
/* 3316 */ {(11<<2)|2,{49,104,0}},
/* 3317 */ {(11<<2)|2,{49,104,0}},
/* 3318 */ {(11<<2)|2,{49,104,0}},
/* 3319 */ {(11<<2)|2,{49,104,0}},
/* 3320 */ {(11<<2)|2,{49,104,0}},
/* 3321 */ {(11<<2)|2,{49,104,0}},
/* 3322 */ {(11<<2)|2,{49,104,0}},
/* 3323 */ {(11<<2)|2,{49,104,0}},
/* 3324 */ {(11<<2)|2,{49,104,0}},
/* 3325 */ {(11<<2)|2,{49,104,0}},
/* 3326 */ {(11<<2)|2,{49,104,0}},
/* 3327 */ {(11<<2)|2,{49,104,0}},
/* 3328 */ {(16<<2)|3,{49,108,48}},
/* 3329 */ {(16<<2)|3,{49,108,49}},
/* 3330 */ {(16<<2)|3,{49,108,50}},
/* 3331 */ {(16<<2)|3,{49,108,97}},
/* 3332 */ {(16<<2)|3,{49,108,99}},
/* 3333 */ {(16<<2)|3,{49,108,101}},
/* 3334 */ {(16<<2)|3,{49,108,105}},
/* 3335 */ {(16<<2)|3,{49,108,111}},
/* 3336 */ {(16<<2)|3,{49,108,115}},
/* 3337 */ {(16<<2)|3,{49,108,116}},
/* 3338 */ {(11<<2)|2,{49,108,0}},
/* 3339 */ {(11<<2)|2,{49,108,0}},
/* 3340 */ {(11<<2)|2,{49,108,0}},
/* 3341 */ {(11<<2)|2,{49,108,0}},
/* 3342 */ {(11<<2)|2,{49,108,0}},
/* 3343 */ {(11<<2)|2,{49,108,0}},
/* 3344 */ {(11<<2)|2,{49,108,0}},
/* 3345 */ {(11<<2)|2,{49,108,0}},
/* 3346 */ {(11<<2)|2,{49,108,0}},
/* 3347 */ {(11<<2)|2,{49,108,0}},
/* 3348 */ {(11<<2)|2,{49,108,0}},
/* 3349 */ {(11<<2)|2,{49,108,0}},
/* 3350 */ {(11<<2)|2,{49,108,0}},
/* 3351 */ {(11<<2)|2,{49,108,0}},
/* 3352 */ {(11<<2)|2,{49,108,0}},
/* 3353 */ {(11<<2)|2,{49,108,0}},
/* 3354 */ {(11<<2)|2,{49,108,0}},
/* 3355 */ {(11<<2)|2,{49,108,0}},
/* 3356 */ {(11<<2)|2,{49,108,0}},
/* 3357 */ {(11<<2)|2,{49,108,0}},
/* 3358 */ {(11<<2)|2,{49,108,0}},
/* 3359 */ {(11<<2)|2,{49,108,0}},
/* 3360 */ {(16<<2)|3,{49,109,48}},
/* 3361 */ {(16<<2)|3,{49,109,49}},
/* 3362 */ {(16<<2)|3,{49,109,50}},
/* 3363 */ {(16<<2)|3,{49,109,97}},
/* 3364 */ {(16<<2)|3,{49,109,99}},
/* 3365 */ {(16<<2)|3,{49,109,101}},
/* 3366 */ {(16<<2)|3,{49,109,105}},
/* 3367 */ {(16<<2)|3,{49,109,111}},
/* 3368 */ {(16<<2)|3,{49,109,115}},
/* 3369 */ {(16<<2)|3,{49,109,116}},
/* 3370 */ {(11<<2)|2,{49,109,0}},
/* 3371 */ {(11<<2)|2,{49,109,0}},
/* 3372 */ {(11<<2)|2,{49,109,0}},
/* 3373 */ {(11<<2)|2,{49,109,0}},
/* 3374 */ {(11<<2)|2,{49,109,0}},
/* 3375 */ {(11<<2)|2,{49,109,0}},
/* 3376 */ {(11<<2)|2,{49,109,0}},
/* 3377 */ {(11<<2)|2,{49,109,0}},
/* 3378 */ {(11<<2)|2,{49,109,0}},
/* 3379 */ {(11<<2)|2,{49,109,0}},
/* 3380 */ {(11<<2)|2,{49,109,0}},
/* 3381 */ {(11<<2)|2,{49,109,0}},
/* 3382 */ {(11<<2)|2,{49,109,0}},
/* 3383 */ {(11<<2)|2,{49,109,0}},
/* 3384 */ {(11<<2)|2,{49,109,0}},
/* 3385 */ {(11<<2)|2,{49,109,0}},
/* 3386 */ {(11<<2)|2,{49,109,0}},
/* 3387 */ {(11<<2)|2,{49,109,0}},
/* 3388 */ {(11<<2)|2,{49,109,0}},
/* 3389 */ {(11<<2)|2,{49,109,0}},
/* 3390 */ {(11<<2)|2,{49,109,0}},
/* 3391 */ {(11<<2)|2,{49,109,0}},
/* 3392 */ {(16<<2)|3,{49,110,48}},
/* 3393 */ {(16<<2)|3,{49,110,49}},
/* 3394 */ {(16<<2)|3,{49,110,50}},
/* 3395 */ {(16<<2)|3,{49,110,97}},
/* 3396 */ {(16<<2)|3,{49,110,99}},
/* 3397 */ {(16<<2)|3,{49,110,101}},
/* 3398 */ {(16<<2)|3,{49,110,105}},
/* 3399 */ {(16<<2)|3,{49,110,111}},
/* 3400 */ {(16<<2)|3,{49,110,115}},
/* 3401 */ {(16<<2)|3,{49,110,116}},
/* 3402 */ {(11<<2)|2,{49,110,0}},
/* 3403 */ {(11<<2)|2,{49,110,0}},
/* 3404 */ {(11<<2)|2,{49,110,0}},
/* 3405 */ {(11<<2)|2,{49,110,0}},
/* 3406 */ {(11<<2)|2,{49,110,0}},
/* 3407 */ {(11<<2)|2,{49,110,0}},
/* 3408 */ {(11<<2)|2,{49,110,0}},
/* 3409 */ {(11<<2)|2,{49,110,0}},
/* 3410 */ {(11<<2)|2,{49,110,0}},
/* 3411 */ {(11<<2)|2,{49,110,0}},
/* 3412 */ {(11<<2)|2,{49,110,0}},
/* 3413 */ {(11<<2)|2,{49,110,0}},
/* 3414 */ {(11<<2)|2,{49,110,0}},
/* 3415 */ {(11<<2)|2,{49,110,0}},
/* 3416 */ {(11<<2)|2,{49,110,0}},
/* 3417 */ {(11<<2)|2,{49,110,0}},
/* 3418 */ {(11<<2)|2,{49,110,0}},
/* 3419 */ {(11<<2)|2,{49,110,0}},
/* 3420 */ {(11<<2)|2,{49,110,0}},
/* 3421 */ {(11<<2)|2,{49,110,0}},
/* 3422 */ {(11<<2)|2,{49,110,0}},
/* 3423 */ {(11<<2)|2,{49,110,0}},
/* 3424 */ {(16<<2)|3,{49,112,48}},
/* 3425 */ {(16<<2)|3,{49,112,49}},
/* 3426 */ {(16<<2)|3,{49,112,50}},
/* 3427 */ {(16<<2)|3,{49,112,97}},
/* 3428 */ {(16<<2)|3,{49,112,99}},
/* 3429 */ {(16<<2)|3,{49,112,101}},
/* 3430 */ {(16<<2)|3,{49,112,105}},
/* 3431 */ {(16<<2)|3,{49,112,111}},
/* 3432 */ {(16<<2)|3,{49,112,115}},
/* 3433 */ {(16<<2)|3,{49,112,116}},
/* 3434 */ {(11<<2)|2,{49,112,0}},
/* 3435 */ {(11<<2)|2,{49,112,0}},
/* 3436 */ {(11<<2)|2,{49,112,0}},
/* 3437 */ {(11<<2)|2,{49,112,0}},
/* 3438 */ {(11<<2)|2,{49,112,0}},
/* 3439 */ {(11<<2)|2,{49,112,0}},
/* 3440 */ {(11<<2)|2,{49,112,0}},
/* 3441 */ {(11<<2)|2,{49,112,0}},
/* 3442 */ {(11<<2)|2,{49,112,0}},
/* 3443 */ {(11<<2)|2,{49,112,0}},
/* 3444 */ {(11<<2)|2,{49,112,0}},
/* 3445 */ {(11<<2)|2,{49,112,0}},
/* 3446 */ {(11<<2)|2,{49,112,0}},
/* 3447 */ {(11<<2)|2,{49,112,0}},
/* 3448 */ {(11<<2)|2,{49,112,0}},
/* 3449 */ {(11<<2)|2,{49,112,0}},
/* 3450 */ {(11<<2)|2,{49,112,0}},
/* 3451 */ {(11<<2)|2,{49,112,0}},
/* 3452 */ {(11<<2)|2,{49,112,0}},
/* 3453 */ {(11<<2)|2,{49,112,0}},
/* 3454 */ {(11<<2)|2,{49,112,0}},
/* 3455 */ {(11<<2)|2,{49,112,0}},
/* 3456 */ {(16<<2)|3,{49,114,48}},
/* 3457 */ {(16<<2)|3,{49,114,49}},
/* 3458 */ {(16<<2)|3,{49,114,50}},
/* 3459 */ {(16<<2)|3,{49,114,97}},
/* 3460 */ {(16<<2)|3,{49,114,99}},
/* 3461 */ {(16<<2)|3,{49,114,101}},
/* 3462 */ {(16<<2)|3,{49,114,105}},
/* 3463 */ {(16<<2)|3,{49,114,111}},
/* 3464 */ {(16<<2)|3,{49,114,115}},
/* 3465 */ {(16<<2)|3,{49,114,116}},
/* 3466 */ {(11<<2)|2,{49,114,0}},
/* 3467 */ {(11<<2)|2,{49,114,0}},
/* 3468 */ {(11<<2)|2,{49,114,0}},
/* 3469 */ {(11<<2)|2,{49,114,0}},
/* 3470 */ {(11<<2)|2,{49,114,0}},
/* 3471 */ {(11<<2)|2,{49,114,0}},
/* 3472 */ {(11<<2)|2,{49,114,0}},
/* 3473 */ {(11<<2)|2,{49,114,0}},
/* 3474 */ {(11<<2)|2,{49,114,0}},
/* 3475 */ {(11<<2)|2,{49,114,0}},
/* 3476 */ {(11<<2)|2,{49,114,0}},
/* 3477 */ {(11<<2)|2,{49,114,0}},
/* 3478 */ {(11<<2)|2,{49,114,0}},
/* 3479 */ {(11<<2)|2,{49,114,0}},
/* 3480 */ {(11<<2)|2,{49,114,0}},
/* 3481 */ {(11<<2)|2,{49,114,0}},
/* 3482 */ {(11<<2)|2,{49,114,0}},
/* 3483 */ {(11<<2)|2,{49,114,0}},
/* 3484 */ {(11<<2)|2,{49,114,0}},
/* 3485 */ {(11<<2)|2,{49,114,0}},
/* 3486 */ {(11<<2)|2,{49,114,0}},
/* 3487 */ {(11<<2)|2,{49,114,0}},
/* 3488 */ {(16<<2)|3,{49,117,48}},
/* 3489 */ {(16<<2)|3,{49,117,49}},
/* 3490 */ {(16<<2)|3,{49,117,50}},
/* 3491 */ {(16<<2)|3,{49,117,97}},
/* 3492 */ {(16<<2)|3,{49,117,99}},
/* 3493 */ {(16<<2)|3,{49,117,101}},
/* 3494 */ {(16<<2)|3,{49,117,105}},
/* 3495 */ {(16<<2)|3,{49,117,111}},
/* 3496 */ {(16<<2)|3,{49,117,115}},
/* 3497 */ {(16<<2)|3,{49,117,116}},
/* 3498 */ {(11<<2)|2,{49,117,0}},
/* 3499 */ {(11<<2)|2,{49,117,0}},
/* 3500 */ {(11<<2)|2,{49,117,0}},
/* 3501 */ {(11<<2)|2,{49,117,0}},
/* 3502 */ {(11<<2)|2,{49,117,0}},
/* 3503 */ {(11<<2)|2,{49,117,0}},
/* 3504 */ {(11<<2)|2,{49,117,0}},
/* 3505 */ {(11<<2)|2,{49,117,0}},
/* 3506 */ {(11<<2)|2,{49,117,0}},
/* 3507 */ {(11<<2)|2,{49,117,0}},
/* 3508 */ {(11<<2)|2,{49,117,0}},
/* 3509 */ {(11<<2)|2,{49,117,0}},
/* 3510 */ {(11<<2)|2,{49,117,0}},
/* 3511 */ {(11<<2)|2,{49,117,0}},
/* 3512 */ {(11<<2)|2,{49,117,0}},
/* 3513 */ {(11<<2)|2,{49,117,0}},
/* 3514 */ {(11<<2)|2,{49,117,0}},
/* 3515 */ {(11<<2)|2,{49,117,0}},
/* 3516 */ {(11<<2)|2,{49,117,0}},
/* 3517 */ {(11<<2)|2,{49,117,0}},
/* 3518 */ {(11<<2)|2,{49,117,0}},
/* 3519 */ {(11<<2)|2,{49,117,0}},
/* 3520 */ {(12<<2)|2,{49,58,0}},
/* 3521 */ {(12<<2)|2,{49,58,0}},
/* 3522 */ {(12<<2)|2,{49,58,0}},
/* 3523 */ {(12<<2)|2,{49,58,0}},
/* 3524 */ {(12<<2)|2,{49,58,0}},
/* 3525 */ {(12<<2)|2,{49,58,0}},
/* 3526 */ {(12<<2)|2,{49,58,0}},
/* 3527 */ {(12<<2)|2,{49,58,0}},
/* 3528 */ {(12<<2)|2,{49,58,0}},
/* 3529 */ {(12<<2)|2,{49,58,0}},
/* 3530 */ {(12<<2)|2,{49,58,0}},
/* 3531 */ {(12<<2)|2,{49,58,0}},
/* 3532 */ {(12<<2)|2,{49,58,0}},
/* 3533 */ {(12<<2)|2,{49,58,0}},
/* 3534 */ {(12<<2)|2,{49,58,0}},
/* 3535 */ {(12<<2)|2,{49,58,0}},
/* 3536 */ {(12<<2)|2,{49,66,0}},
/* 3537 */ {(12<<2)|2,{49,66,0}},
/* 3538 */ {(12<<2)|2,{49,66,0}},
/* 3539 */ {(12<<2)|2,{49,66,0}},
/* 3540 */ {(12<<2)|2,{49,66,0}},
/* 3541 */ {(12<<2)|2,{49,66,0}},
/* 3542 */ {(12<<2)|2,{49,66,0}},
/* 3543 */ {(12<<2)|2,{49,66,0}},
/* 3544 */ {(12<<2)|2,{49,66,0}},
/* 3545 */ {(12<<2)|2,{49,66,0}},
/* 3546 */ {(12<<2)|2,{49,66,0}},
/* 3547 */ {(12<<2)|2,{49,66,0}},
/* 3548 */ {(12<<2)|2,{49,66,0}},
/* 3549 */ {(12<<2)|2,{49,66,0}},
/* 3550 */ {(12<<2)|2,{49,66,0}},
/* 3551 */ {(12<<2)|2,{49,66,0}},
/* 3552 */ {(12<<2)|2,{49,67,0}},
/* 3553 */ {(12<<2)|2,{49,67,0}},
/* 3554 */ {(12<<2)|2,{49,67,0}},
/* 3555 */ {(12<<2)|2,{49,67,0}},
/* 3556 */ {(12<<2)|2,{49,67,0}},
/* 3557 */ {(12<<2)|2,{49,67,0}},
/* 3558 */ {(12<<2)|2,{49,67,0}},
/* 3559 */ {(12<<2)|2,{49,67,0}},
/* 3560 */ {(12<<2)|2,{49,67,0}},
/* 3561 */ {(12<<2)|2,{49,67,0}},
/* 3562 */ {(12<<2)|2,{49,67,0}},
/* 3563 */ {(12<<2)|2,{49,67,0}},
/* 3564 */ {(12<<2)|2,{49,67,0}},
/* 3565 */ {(12<<2)|2,{49,67,0}},
/* 3566 */ {(12<<2)|2,{49,67,0}},
/* 3567 */ {(12<<2)|2,{49,67,0}},
/* 3568 */ {(12<<2)|2,{49,68,0}},
/* 3569 */ {(12<<2)|2,{49,68,0}},
/* 3570 */ {(12<<2)|2,{49,68,0}},
/* 3571 */ {(12<<2)|2,{49,68,0}},
/* 3572 */ {(12<<2)|2,{49,68,0}},
/* 3573 */ {(12<<2)|2,{49,68,0}},
/* 3574 */ {(12<<2)|2,{49,68,0}},
/* 3575 */ {(12<<2)|2,{49,68,0}},
/* 3576 */ {(12<<2)|2,{49,68,0}},
/* 3577 */ {(12<<2)|2,{49,68,0}},
/* 3578 */ {(12<<2)|2,{49,68,0}},
/* 3579 */ {(12<<2)|2,{49,68,0}},
/* 3580 */ {(12<<2)|2,{49,68,0}},
/* 3581 */ {(12<<2)|2,{49,68,0}},
/* 3582 */ {(12<<2)|2,{49,68,0}},
/* 3583 */ {(12<<2)|2,{49,68,0}},
/* 3584 */ {(12<<2)|2,{49,69,0}},
/* 3585 */ {(12<<2)|2,{49,69,0}},
/* 3586 */ {(12<<2)|2,{49,69,0}},
/* 3587 */ {(12<<2)|2,{49,69,0}},
/* 3588 */ {(12<<2)|2,{49,69,0}},
/* 3589 */ {(12<<2)|2,{49,69,0}},
/* 3590 */ {(12<<2)|2,{49,69,0}},
/* 3591 */ {(12<<2)|2,{49,69,0}},
/* 3592 */ {(12<<2)|2,{49,69,0}},
/* 3593 */ {(12<<2)|2,{49,69,0}},
/* 3594 */ {(12<<2)|2,{49,69,0}},
/* 3595 */ {(12<<2)|2,{49,69,0}},
/* 3596 */ {(12<<2)|2,{49,69,0}},
/* 3597 */ {(12<<2)|2,{49,69,0}},
/* 3598 */ {(12<<2)|2,{49,69,0}},
/* 3599 */ {(12<<2)|2,{49,69,0}},
/* 3600 */ {(12<<2)|2,{49,70,0}},
/* 3601 */ {(12<<2)|2,{49,70,0}},
/* 3602 */ {(12<<2)|2,{49,70,0}},
/* 3603 */ {(12<<2)|2,{49,70,0}},
/* 3604 */ {(12<<2)|2,{49,70,0}},
/* 3605 */ {(12<<2)|2,{49,70,0}},
/* 3606 */ {(12<<2)|2,{49,70,0}},
/* 3607 */ {(12<<2)|2,{49,70,0}},
/* 3608 */ {(12<<2)|2,{49,70,0}},
/* 3609 */ {(12<<2)|2,{49,70,0}},
/* 3610 */ {(12<<2)|2,{49,70,0}},
/* 3611 */ {(12<<2)|2,{49,70,0}},
/* 3612 */ {(12<<2)|2,{49,70,0}},
/* 3613 */ {(12<<2)|2,{49,70,0}},
/* 3614 */ {(12<<2)|2,{49,70,0}},
/* 3615 */ {(12<<2)|2,{49,70,0}},
/* 3616 */ {(12<<2)|2,{49,71,0}},
/* 3617 */ {(12<<2)|2,{49,71,0}},
/* 3618 */ {(12<<2)|2,{49,71,0}},
/* 3619 */ {(12<<2)|2,{49,71,0}},
/* 3620 */ {(12<<2)|2,{49,71,0}},
/* 3621 */ {(12<<2)|2,{49,71,0}},
/* 3622 */ {(12<<2)|2,{49,71,0}},
/* 3623 */ {(12<<2)|2,{49,71,0}},
/* 3624 */ {(12<<2)|2,{49,71,0}},
/* 3625 */ {(12<<2)|2,{49,71,0}},
/* 3626 */ {(12<<2)|2,{49,71,0}},
/* 3627 */ {(12<<2)|2,{49,71,0}},
/* 3628 */ {(12<<2)|2,{49,71,0}},
/* 3629 */ {(12<<2)|2,{49,71,0}},
/* 3630 */ {(12<<2)|2,{49,71,0}},
/* 3631 */ {(12<<2)|2,{49,71,0}},
/* 3632 */ {(12<<2)|2,{49,72,0}},
/* 3633 */ {(12<<2)|2,{49,72,0}},
/* 3634 */ {(12<<2)|2,{49,72,0}},
/* 3635 */ {(12<<2)|2,{49,72,0}},
/* 3636 */ {(12<<2)|2,{49,72,0}},
/* 3637 */ {(12<<2)|2,{49,72,0}},
/* 3638 */ {(12<<2)|2,{49,72,0}},
/* 3639 */ {(12<<2)|2,{49,72,0}},
/* 3640 */ {(12<<2)|2,{49,72,0}},
/* 3641 */ {(12<<2)|2,{49,72,0}},
/* 3642 */ {(12<<2)|2,{49,72,0}},
/* 3643 */ {(12<<2)|2,{49,72,0}},
/* 3644 */ {(12<<2)|2,{49,72,0}},
/* 3645 */ {(12<<2)|2,{49,72,0}},
/* 3646 */ {(12<<2)|2,{49,72,0}},
/* 3647 */ {(12<<2)|2,{49,72,0}},
/* 3648 */ {(12<<2)|2,{49,73,0}},
/* 3649 */ {(12<<2)|2,{49,73,0}},
/* 3650 */ {(12<<2)|2,{49,73,0}},
/* 3651 */ {(12<<2)|2,{49,73,0}},
/* 3652 */ {(12<<2)|2,{49,73,0}},
/* 3653 */ {(12<<2)|2,{49,73,0}},
/* 3654 */ {(12<<2)|2,{49,73,0}},
/* 3655 */ {(12<<2)|2,{49,73,0}},
/* 3656 */ {(12<<2)|2,{49,73,0}},
/* 3657 */ {(12<<2)|2,{49,73,0}},
/* 3658 */ {(12<<2)|2,{49,73,0}},
/* 3659 */ {(12<<2)|2,{49,73,0}},
/* 3660 */ {(12<<2)|2,{49,73,0}},
/* 3661 */ {(12<<2)|2,{49,73,0}},
/* 3662 */ {(12<<2)|2,{49,73,0}},
/* 3663 */ {(12<<2)|2,{49,73,0}},
/* 3664 */ {(12<<2)|2,{49,74,0}},
/* 3665 */ {(12<<2)|2,{49,74,0}},
/* 3666 */ {(12<<2)|2,{49,74,0}},
/* 3667 */ {(12<<2)|2,{49,74,0}},
/* 3668 */ {(12<<2)|2,{49,74,0}},
/* 3669 */ {(12<<2)|2,{49,74,0}},
/* 3670 */ {(12<<2)|2,{49,74,0}},
/* 3671 */ {(12<<2)|2,{49,74,0}},
/* 3672 */ {(12<<2)|2,{49,74,0}},
/* 3673 */ {(12<<2)|2,{49,74,0}},
/* 3674 */ {(12<<2)|2,{49,74,0}},
/* 3675 */ {(12<<2)|2,{49,74,0}},
/* 3676 */ {(12<<2)|2,{49,74,0}},
/* 3677 */ {(12<<2)|2,{49,74,0}},
/* 3678 */ {(12<<2)|2,{49,74,0}},
/* 3679 */ {(12<<2)|2,{49,74,0}},
/* 3680 */ {(12<<2)|2,{49,75,0}},
/* 3681 */ {(12<<2)|2,{49,75,0}},
/* 3682 */ {(12<<2)|2,{49,75,0}},
/* 3683 */ {(12<<2)|2,{49,75,0}},
/* 3684 */ {(12<<2)|2,{49,75,0}},
/* 3685 */ {(12<<2)|2,{49,75,0}},
/* 3686 */ {(12<<2)|2,{49,75,0}},
/* 3687 */ {(12<<2)|2,{49,75,0}},
/* 3688 */ {(12<<2)|2,{49,75,0}},
/* 3689 */ {(12<<2)|2,{49,75,0}},
/* 3690 */ {(12<<2)|2,{49,75,0}},
/* 3691 */ {(12<<2)|2,{49,75,0}},
/* 3692 */ {(12<<2)|2,{49,75,0}},
/* 3693 */ {(12<<2)|2,{49,75,0}},
/* 3694 */ {(12<<2)|2,{49,75,0}},
/* 3695 */ {(12<<2)|2,{49,75,0}},
/* 3696 */ {(12<<2)|2,{49,76,0}},
/* 3697 */ {(12<<2)|2,{49,76,0}},
/* 3698 */ {(12<<2)|2,{49,76,0}},
/* 3699 */ {(12<<2)|2,{49,76,0}},
/* 3700 */ {(12<<2)|2,{49,76,0}},
/* 3701 */ {(12<<2)|2,{49,76,0}},
/* 3702 */ {(12<<2)|2,{49,76,0}},
/* 3703 */ {(12<<2)|2,{49,76,0}},
/* 3704 */ {(12<<2)|2,{49,76,0}},
/* 3705 */ {(12<<2)|2,{49,76,0}},
/* 3706 */ {(12<<2)|2,{49,76,0}},
/* 3707 */ {(12<<2)|2,{49,76,0}},
/* 3708 */ {(12<<2)|2,{49,76,0}},
/* 3709 */ {(12<<2)|2,{49,76,0}},
/* 3710 */ {(12<<2)|2,{49,76,0}},
/* 3711 */ {(12<<2)|2,{49,76,0}},
/* 3712 */ {(12<<2)|2,{49,77,0}},
/* 3713 */ {(12<<2)|2,{49,77,0}},
/* 3714 */ {(12<<2)|2,{49,77,0}},
/* 3715 */ {(12<<2)|2,{49,77,0}},
/* 3716 */ {(12<<2)|2,{49,77,0}},
/* 3717 */ {(12<<2)|2,{49,77,0}},
/* 3718 */ {(12<<2)|2,{49,77,0}},
/* 3719 */ {(12<<2)|2,{49,77,0}},
/* 3720 */ {(12<<2)|2,{49,77,0}},
/* 3721 */ {(12<<2)|2,{49,77,0}},
/* 3722 */ {(12<<2)|2,{49,77,0}},
/* 3723 */ {(12<<2)|2,{49,77,0}},
/* 3724 */ {(12<<2)|2,{49,77,0}},
/* 3725 */ {(12<<2)|2,{49,77,0}},
/* 3726 */ {(12<<2)|2,{49,77,0}},
/* 3727 */ {(12<<2)|2,{49,77,0}},
/* 3728 */ {(12<<2)|2,{49,78,0}},
/* 3729 */ {(12<<2)|2,{49,78,0}},
/* 3730 */ {(12<<2)|2,{49,78,0}},
/* 3731 */ {(12<<2)|2,{49,78,0}},
/* 3732 */ {(12<<2)|2,{49,78,0}},
/* 3733 */ {(12<<2)|2,{49,78,0}},
/* 3734 */ {(12<<2)|2,{49,78,0}},
/* 3735 */ {(12<<2)|2,{49,78,0}},
/* 3736 */ {(12<<2)|2,{49,78,0}},
/* 3737 */ {(12<<2)|2,{49,78,0}},
/* 3738 */ {(12<<2)|2,{49,78,0}},
/* 3739 */ {(12<<2)|2,{49,78,0}},
/* 3740 */ {(12<<2)|2,{49,78,0}},
/* 3741 */ {(12<<2)|2,{49,78,0}},
/* 3742 */ {(12<<2)|2,{49,78,0}},
/* 3743 */ {(12<<2)|2,{49,78,0}},
/* 3744 */ {(12<<2)|2,{49,79,0}},
/* 3745 */ {(12<<2)|2,{49,79,0}},
/* 3746 */ {(12<<2)|2,{49,79,0}},
/* 3747 */ {(12<<2)|2,{49,79,0}},
/* 3748 */ {(12<<2)|2,{49,79,0}},
/* 3749 */ {(12<<2)|2,{49,79,0}},
/* 3750 */ {(12<<2)|2,{49,79,0}},
/* 3751 */ {(12<<2)|2,{49,79,0}},
/* 3752 */ {(12<<2)|2,{49,79,0}},
/* 3753 */ {(12<<2)|2,{49,79,0}},
/* 3754 */ {(12<<2)|2,{49,79,0}},
/* 3755 */ {(12<<2)|2,{49,79,0}},
/* 3756 */ {(12<<2)|2,{49,79,0}},
/* 3757 */ {(12<<2)|2,{49,79,0}},
/* 3758 */ {(12<<2)|2,{49,79,0}},
/* 3759 */ {(12<<2)|2,{49,79,0}},
/* 3760 */ {(12<<2)|2,{49,80,0}},
/* 3761 */ {(12<<2)|2,{49,80,0}},
/* 3762 */ {(12<<2)|2,{49,80,0}},
/* 3763 */ {(12<<2)|2,{49,80,0}},
/* 3764 */ {(12<<2)|2,{49,80,0}},
/* 3765 */ {(12<<2)|2,{49,80,0}},
/* 3766 */ {(12<<2)|2,{49,80,0}},
/* 3767 */ {(12<<2)|2,{49,80,0}},
/* 3768 */ {(12<<2)|2,{49,80,0}},
/* 3769 */ {(12<<2)|2,{49,80,0}},
/* 3770 */ {(12<<2)|2,{49,80,0}},
/* 3771 */ {(12<<2)|2,{49,80,0}},
/* 3772 */ {(12<<2)|2,{49,80,0}},
/* 3773 */ {(12<<2)|2,{49,80,0}},
/* 3774 */ {(12<<2)|2,{49,80,0}},
/* 3775 */ {(12<<2)|2,{49,80,0}},
/* 3776 */ {(12<<2)|2,{49,81,0}},
/* 3777 */ {(12<<2)|2,{49,81,0}},
/* 3778 */ {(12<<2)|2,{49,81,0}},
/* 3779 */ {(12<<2)|2,{49,81,0}},
/* 3780 */ {(12<<2)|2,{49,81,0}},
/* 3781 */ {(12<<2)|2,{49,81,0}},
/* 3782 */ {(12<<2)|2,{49,81,0}},
/* 3783 */ {(12<<2)|2,{49,81,0}},
/* 3784 */ {(12<<2)|2,{49,81,0}},
/* 3785 */ {(12<<2)|2,{49,81,0}},
/* 3786 */ {(12<<2)|2,{49,81,0}},
/* 3787 */ {(12<<2)|2,{49,81,0}},
/* 3788 */ {(12<<2)|2,{49,81,0}},
/* 3789 */ {(12<<2)|2,{49,81,0}},
/* 3790 */ {(12<<2)|2,{49,81,0}},
/* 3791 */ {(12<<2)|2,{49,81,0}},
/* 3792 */ {(12<<2)|2,{49,82,0}},
/* 3793 */ {(12<<2)|2,{49,82,0}},
/* 3794 */ {(12<<2)|2,{49,82,0}},
/* 3795 */ {(12<<2)|2,{49,82,0}},
/* 3796 */ {(12<<2)|2,{49,82,0}},
/* 3797 */ {(12<<2)|2,{49,82,0}},
/* 3798 */ {(12<<2)|2,{49,82,0}},
/* 3799 */ {(12<<2)|2,{49,82,0}},
/* 3800 */ {(12<<2)|2,{49,82,0}},
/* 3801 */ {(12<<2)|2,{49,82,0}},
/* 3802 */ {(12<<2)|2,{49,82,0}},
/* 3803 */ {(12<<2)|2,{49,82,0}},
/* 3804 */ {(12<<2)|2,{49,82,0}},
/* 3805 */ {(12<<2)|2,{49,82,0}},
/* 3806 */ {(12<<2)|2,{49,82,0}},
/* 3807 */ {(12<<2)|2,{49,82,0}},
/* 3808 */ {(12<<2)|2,{49,83,0}},
/* 3809 */ {(12<<2)|2,{49,83,0}},
/* 3810 */ {(12<<2)|2,{49,83,0}},
/* 3811 */ {(12<<2)|2,{49,83,0}},
/* 3812 */ {(12<<2)|2,{49,83,0}},
/* 3813 */ {(12<<2)|2,{49,83,0}},
/* 3814 */ {(12<<2)|2,{49,83,0}},
/* 3815 */ {(12<<2)|2,{49,83,0}},
/* 3816 */ {(12<<2)|2,{49,83,0}},
/* 3817 */ {(12<<2)|2,{49,83,0}},
/* 3818 */ {(12<<2)|2,{49,83,0}},
/* 3819 */ {(12<<2)|2,{49,83,0}},
/* 3820 */ {(12<<2)|2,{49,83,0}},
/* 3821 */ {(12<<2)|2,{49,83,0}},
/* 3822 */ {(12<<2)|2,{49,83,0}},
/* 3823 */ {(12<<2)|2,{49,83,0}},
/* 3824 */ {(12<<2)|2,{49,84,0}},
/* 3825 */ {(12<<2)|2,{49,84,0}},
/* 3826 */ {(12<<2)|2,{49,84,0}},
/* 3827 */ {(12<<2)|2,{49,84,0}},
/* 3828 */ {(12<<2)|2,{49,84,0}},
/* 3829 */ {(12<<2)|2,{49,84,0}},
/* 3830 */ {(12<<2)|2,{49,84,0}},
/* 3831 */ {(12<<2)|2,{49,84,0}},
/* 3832 */ {(12<<2)|2,{49,84,0}},
/* 3833 */ {(12<<2)|2,{49,84,0}},
/* 3834 */ {(12<<2)|2,{49,84,0}},
/* 3835 */ {(12<<2)|2,{49,84,0}},
/* 3836 */ {(12<<2)|2,{49,84,0}},
/* 3837 */ {(12<<2)|2,{49,84,0}},
/* 3838 */ {(12<<2)|2,{49,84,0}},
/* 3839 */ {(12<<2)|2,{49,84,0}},
/* 3840 */ {(12<<2)|2,{49,85,0}},
/* 3841 */ {(12<<2)|2,{49,85,0}},
/* 3842 */ {(12<<2)|2,{49,85,0}},
/* 3843 */ {(12<<2)|2,{49,85,0}},
/* 3844 */ {(12<<2)|2,{49,85,0}},
/* 3845 */ {(12<<2)|2,{49,85,0}},
/* 3846 */ {(12<<2)|2,{49,85,0}},
/* 3847 */ {(12<<2)|2,{49,85,0}},
/* 3848 */ {(12<<2)|2,{49,85,0}},
/* 3849 */ {(12<<2)|2,{49,85,0}},
/* 3850 */ {(12<<2)|2,{49,85,0}},
/* 3851 */ {(12<<2)|2,{49,85,0}},
/* 3852 */ {(12<<2)|2,{49,85,0}},
/* 3853 */ {(12<<2)|2,{49,85,0}},
/* 3854 */ {(12<<2)|2,{49,85,0}},
/* 3855 */ {(12<<2)|2,{49,85,0}},
/* 3856 */ {(12<<2)|2,{49,86,0}},
/* 3857 */ {(12<<2)|2,{49,86,0}},
/* 3858 */ {(12<<2)|2,{49,86,0}},
/* 3859 */ {(12<<2)|2,{49,86,0}},
/* 3860 */ {(12<<2)|2,{49,86,0}},
/* 3861 */ {(12<<2)|2,{49,86,0}},
/* 3862 */ {(12<<2)|2,{49,86,0}},
/* 3863 */ {(12<<2)|2,{49,86,0}},
/* 3864 */ {(12<<2)|2,{49,86,0}},
/* 3865 */ {(12<<2)|2,{49,86,0}},
/* 3866 */ {(12<<2)|2,{49,86,0}},
/* 3867 */ {(12<<2)|2,{49,86,0}},
/* 3868 */ {(12<<2)|2,{49,86,0}},
/* 3869 */ {(12<<2)|2,{49,86,0}},
/* 3870 */ {(12<<2)|2,{49,86,0}},
/* 3871 */ {(12<<2)|2,{49,86,0}},
/* 3872 */ {(12<<2)|2,{49,87,0}},
/* 3873 */ {(12<<2)|2,{49,87,0}},
/* 3874 */ {(12<<2)|2,{49,87,0}},
/* 3875 */ {(12<<2)|2,{49,87,0}},
/* 3876 */ {(12<<2)|2,{49,87,0}},
/* 3877 */ {(12<<2)|2,{49,87,0}},
/* 3878 */ {(12<<2)|2,{49,87,0}},
/* 3879 */ {(12<<2)|2,{49,87,0}},
/* 3880 */ {(12<<2)|2,{49,87,0}},
/* 3881 */ {(12<<2)|2,{49,87,0}},
/* 3882 */ {(12<<2)|2,{49,87,0}},
/* 3883 */ {(12<<2)|2,{49,87,0}},
/* 3884 */ {(12<<2)|2,{49,87,0}},
/* 3885 */ {(12<<2)|2,{49,87,0}},
/* 3886 */ {(12<<2)|2,{49,87,0}},
/* 3887 */ {(12<<2)|2,{49,87,0}},
/* 3888 */ {(12<<2)|2,{49,89,0}},
/* 3889 */ {(12<<2)|2,{49,89,0}},
/* 3890 */ {(12<<2)|2,{49,89,0}},
/* 3891 */ {(12<<2)|2,{49,89,0}},
/* 3892 */ {(12<<2)|2,{49,89,0}},
/* 3893 */ {(12<<2)|2,{49,89,0}},
/* 3894 */ {(12<<2)|2,{49,89,0}},
/* 3895 */ {(12<<2)|2,{49,89,0}},
/* 3896 */ {(12<<2)|2,{49,89,0}},
/* 3897 */ {(12<<2)|2,{49,89,0}},
/* 3898 */ {(12<<2)|2,{49,89,0}},
/* 3899 */ {(12<<2)|2,{49,89,0}},
/* 3900 */ {(12<<2)|2,{49,89,0}},
/* 3901 */ {(12<<2)|2,{49,89,0}},
/* 3902 */ {(12<<2)|2,{49,89,0}},
/* 3903 */ {(12<<2)|2,{49,89,0}},
/* 3904 */ {(12<<2)|2,{49,106,0}},
/* 3905 */ {(12<<2)|2,{49,106,0}},
/* 3906 */ {(12<<2)|2,{49,106,0}},
/* 3907 */ {(12<<2)|2,{49,106,0}},
/* 3908 */ {(12<<2)|2,{49,106,0}},
/* 3909 */ {(12<<2)|2,{49,106,0}},
/* 3910 */ {(12<<2)|2,{49,106,0}},
/* 3911 */ {(12<<2)|2,{49,106,0}},
/* 3912 */ {(12<<2)|2,{49,106,0}},
/* 3913 */ {(12<<2)|2,{49,106,0}},
/* 3914 */ {(12<<2)|2,{49,106,0}},
/* 3915 */ {(12<<2)|2,{49,106,0}},
/* 3916 */ {(12<<2)|2,{49,106,0}},
/* 3917 */ {(12<<2)|2,{49,106,0}},
/* 3918 */ {(12<<2)|2,{49,106,0}},
/* 3919 */ {(12<<2)|2,{49,106,0}},
/* 3920 */ {(12<<2)|2,{49,107,0}},
/* 3921 */ {(12<<2)|2,{49,107,0}},
/* 3922 */ {(12<<2)|2,{49,107,0}},
/* 3923 */ {(12<<2)|2,{49,107,0}},
/* 3924 */ {(12<<2)|2,{49,107,0}},
/* 3925 */ {(12<<2)|2,{49,107,0}},
/* 3926 */ {(12<<2)|2,{49,107,0}},
/* 3927 */ {(12<<2)|2,{49,107,0}},
/* 3928 */ {(12<<2)|2,{49,107,0}},
/* 3929 */ {(12<<2)|2,{49,107,0}},
/* 3930 */ {(12<<2)|2,{49,107,0}},
/* 3931 */ {(12<<2)|2,{49,107,0}},
/* 3932 */ {(12<<2)|2,{49,107,0}},
/* 3933 */ {(12<<2)|2,{49,107,0}},
/* 3934 */ {(12<<2)|2,{49,107,0}},
/* 3935 */ {(12<<2)|2,{49,107,0}},
/* 3936 */ {(12<<2)|2,{49,113,0}},
/* 3937 */ {(12<<2)|2,{49,113,0}},
/* 3938 */ {(12<<2)|2,{49,113,0}},
/* 3939 */ {(12<<2)|2,{49,113,0}},
/* 3940 */ {(12<<2)|2,{49,113,0}},
/* 3941 */ {(12<<2)|2,{49,113,0}},
/* 3942 */ {(12<<2)|2,{49,113,0}},
/* 3943 */ {(12<<2)|2,{49,113,0}},
/* 3944 */ {(12<<2)|2,{49,113,0}},
/* 3945 */ {(12<<2)|2,{49,113,0}},
/* 3946 */ {(12<<2)|2,{49,113,0}},
/* 3947 */ {(12<<2)|2,{49,113,0}},
/* 3948 */ {(12<<2)|2,{49,113,0}},
/* 3949 */ {(12<<2)|2,{49,113,0}},
/* 3950 */ {(12<<2)|2,{49,113,0}},
/* 3951 */ {(12<<2)|2,{49,113,0}},
/* 3952 */ {(12<<2)|2,{49,118,0}},
/* 3953 */ {(12<<2)|2,{49,118,0}},
/* 3954 */ {(12<<2)|2,{49,118,0}},
/* 3955 */ {(12<<2)|2,{49,118,0}},
/* 3956 */ {(12<<2)|2,{49,118,0}},
/* 3957 */ {(12<<2)|2,{49,118,0}},
/* 3958 */ {(12<<2)|2,{49,118,0}},
/* 3959 */ {(12<<2)|2,{49,118,0}},
/* 3960 */ {(12<<2)|2,{49,118,0}},
/* 3961 */ {(12<<2)|2,{49,118,0}},
/* 3962 */ {(12<<2)|2,{49,118,0}},
/* 3963 */ {(12<<2)|2,{49,118,0}},
/* 3964 */ {(12<<2)|2,{49,118,0}},
/* 3965 */ {(12<<2)|2,{49,118,0}},
/* 3966 */ {(12<<2)|2,{49,118,0}},
/* 3967 */ {(12<<2)|2,{49,118,0}},
/* 3968 */ {(12<<2)|2,{49,119,0}},
/* 3969 */ {(12<<2)|2,{49,119,0}},
/* 3970 */ {(12<<2)|2,{49,119,0}},
/* 3971 */ {(12<<2)|2,{49,119,0}},
/* 3972 */ {(12<<2)|2,{49,119,0}},
/* 3973 */ {(12<<2)|2,{49,119,0}},
/* 3974 */ {(12<<2)|2,{49,119,0}},
/* 3975 */ {(12<<2)|2,{49,119,0}},
/* 3976 */ {(12<<2)|2,{49,119,0}},
/* 3977 */ {(12<<2)|2,{49,119,0}},
/* 3978 */ {(12<<2)|2,{49,119,0}},
/* 3979 */ {(12<<2)|2,{49,119,0}},
/* 3980 */ {(12<<2)|2,{49,119,0}},
/* 3981 */ {(12<<2)|2,{49,119,0}},
/* 3982 */ {(12<<2)|2,{49,119,0}},
/* 3983 */ {(12<<2)|2,{49,119,0}},
/* 3984 */ {(12<<2)|2,{49,120,0}},
/* 3985 */ {(12<<2)|2,{49,120,0}},
/* 3986 */ {(12<<2)|2,{49,120,0}},
/* 3987 */ {(12<<2)|2,{49,120,0}},
/* 3988 */ {(12<<2)|2,{49,120,0}},
/* 3989 */ {(12<<2)|2,{49,120,0}},
/* 3990 */ {(12<<2)|2,{49,120,0}},
/* 3991 */ {(12<<2)|2,{49,120,0}},
/* 3992 */ {(12<<2)|2,{49,120,0}},
/* 3993 */ {(12<<2)|2,{49,120,0}},
/* 3994 */ {(12<<2)|2,{49,120,0}},
/* 3995 */ {(12<<2)|2,{49,120,0}},
/* 3996 */ {(12<<2)|2,{49,120,0}},
/* 3997 */ {(12<<2)|2,{49,120,0}},
/* 3998 */ {(12<<2)|2,{49,120,0}},
/* 3999 */ {(12<<2)|2,{49,120,0}},
/* 4000 */ {(12<<2)|2,{49,121,0}},
/* 4001 */ {(12<<2)|2,{49,121,0}},
/* 4002 */ {(12<<2)|2,{49,121,0}},
/* 4003 */ {(12<<2)|2,{49,121,0}},
/* 4004 */ {(12<<2)|2,{49,121,0}},
/* 4005 */ {(12<<2)|2,{49,121,0}},
/* 4006 */ {(12<<2)|2,{49,121,0}},
/* 4007 */ {(12<<2)|2,{49,121,0}},
/* 4008 */ {(12<<2)|2,{49,121,0}},
/* 4009 */ {(12<<2)|2,{49,121,0}},
/* 4010 */ {(12<<2)|2,{49,121,0}},
/* 4011 */ {(12<<2)|2,{49,121,0}},
/* 4012 */ {(12<<2)|2,{49,121,0}},
/* 4013 */ {(12<<2)|2,{49,121,0}},
/* 4014 */ {(12<<2)|2,{49,121,0}},
/* 4015 */ {(12<<2)|2,{49,121,0}},
/* 4016 */ {(12<<2)|2,{49,122,0}},
/* 4017 */ {(12<<2)|2,{49,122,0}},
/* 4018 */ {(12<<2)|2,{49,122,0}},
/* 4019 */ {(12<<2)|2,{49,122,0}},
/* 4020 */ {(12<<2)|2,{49,122,0}},
/* 4021 */ {(12<<2)|2,{49,122,0}},
/* 4022 */ {(12<<2)|2,{49,122,0}},
/* 4023 */ {(12<<2)|2,{49,122,0}},
/* 4024 */ {(12<<2)|2,{49,122,0}},
/* 4025 */ {(12<<2)|2,{49,122,0}},
/* 4026 */ {(12<<2)|2,{49,122,0}},
/* 4027 */ {(12<<2)|2,{49,122,0}},
/* 4028 */ {(12<<2)|2,{49,122,0}},
/* 4029 */ {(12<<2)|2,{49,122,0}},
/* 4030 */ {(12<<2)|2,{49,122,0}},
/* 4031 */ {(12<<2)|2,{49,122,0}},
/* 4032 */ {(13<<2)|2,{49,38,0}},
/* 4033 */ {(13<<2)|2,{49,38,0}},
/* 4034 */ {(13<<2)|2,{49,38,0}},
/* 4035 */ {(13<<2)|2,{49,38,0}},
/* 4036 */ {(13<<2)|2,{49,38,0}},
/* 4037 */ {(13<<2)|2,{49,38,0}},
/* 4038 */ {(13<<2)|2,{49,38,0}},
/* 4039 */ {(13<<2)|2,{49,38,0}},
/* 4040 */ {(13<<2)|2,{49,42,0}},
/* 4041 */ {(13<<2)|2,{49,42,0}},
/* 4042 */ {(13<<2)|2,{49,42,0}},
/* 4043 */ {(13<<2)|2,{49,42,0}},
/* 4044 */ {(13<<2)|2,{49,42,0}},
/* 4045 */ {(13<<2)|2,{49,42,0}},
/* 4046 */ {(13<<2)|2,{49,42,0}},
/* 4047 */ {(13<<2)|2,{49,42,0}},
/* 4048 */ {(13<<2)|2,{49,44,0}},
/* 4049 */ {(13<<2)|2,{49,44,0}},
/* 4050 */ {(13<<2)|2,{49,44,0}},
/* 4051 */ {(13<<2)|2,{49,44,0}},
/* 4052 */ {(13<<2)|2,{49,44,0}},
/* 4053 */ {(13<<2)|2,{49,44,0}},
/* 4054 */ {(13<<2)|2,{49,44,0}},
/* 4055 */ {(13<<2)|2,{49,44,0}},
/* 4056 */ {(13<<2)|2,{49,59,0}},
/* 4057 */ {(13<<2)|2,{49,59,0}},
/* 4058 */ {(13<<2)|2,{49,59,0}},
/* 4059 */ {(13<<2)|2,{49,59,0}},
/* 4060 */ {(13<<2)|2,{49,59,0}},
/* 4061 */ {(13<<2)|2,{49,59,0}},
/* 4062 */ {(13<<2)|2,{49,59,0}},
/* 4063 */ {(13<<2)|2,{49,59,0}},
/* 4064 */ {(13<<2)|2,{49,88,0}},
/* 4065 */ {(13<<2)|2,{49,88,0}},
/* 4066 */ {(13<<2)|2,{49,88,0}},
/* 4067 */ {(13<<2)|2,{49,88,0}},
/* 4068 */ {(13<<2)|2,{49,88,0}},
/* 4069 */ {(13<<2)|2,{49,88,0}},
/* 4070 */ {(13<<2)|2,{49,88,0}},
/* 4071 */ {(13<<2)|2,{49,88,0}},
/* 4072 */ {(13<<2)|2,{49,90,0}},
/* 4073 */ {(13<<2)|2,{49,90,0}},
/* 4074 */ {(13<<2)|2,{49,90,0}},
/* 4075 */ {(13<<2)|2,{49,90,0}},
/* 4076 */ {(13<<2)|2,{49,90,0}},
/* 4077 */ {(13<<2)|2,{49,90,0}},
/* 4078 */ {(13<<2)|2,{49,90,0}},
/* 4079 */ {(13<<2)|2,{49,90,0}},
/* 4080 */ {(15<<2)|2,{49,33,0}},
/* 4081 */ {(15<<2)|2,{49,33,0}},
/* 4082 */ {(15<<2)|2,{49,34,0}},
/* 4083 */ {(15<<2)|2,{49,34,0}},
/* 4084 */ {(15<<2)|2,{49,40,0}},
/* 4085 */ {(15<<2)|2,{49,40,0}},
/* 4086 */ {(15<<2)|2,{49,41,0}},
/* 4087 */ {(15<<2)|2,{49,41,0}},
/* 4088 */ {(15<<2)|2,{49,63,0}},
/* 4089 */ {(15<<2)|2,{49,63,0}},
/* 4090 */ {(16<<2)|2,{49,39,0}},
/* 4091 */ {(16<<2)|2,{49,43,0}},
/* 4092 */ {(16<<2)|2,{49,124,0}},
/* 4093 */ {(5<<2)|1,{49,0,0}},
/* 4094 */ {(5<<2)|1,{49,0,0}},
/* 4095 */ {(5<<2)|1,{49,0,0}},
/* 4096 */ {(15<<2)|3,{50,48,48}},
/* 4097 */ {(15<<2)|3,{50,48,48}},
/* 4098 */ {(15<<2)|3,{50,48,49}},
/* 4099 */ {(15<<2)|3,{50,48,49}},
/* 4100 */ {(15<<2)|3,{50,48,50}},
/* 4101 */ {(15<<2)|3,{50,48,50}},
/* 4102 */ {(15<<2)|3,{50,48,97}},
/* 4103 */ {(15<<2)|3,{50,48,97}},
/* 4104 */ {(15<<2)|3,{50,48,99}},
/* 4105 */ {(15<<2)|3,{50,48,99}},
/* 4106 */ {(15<<2)|3,{50,48,101}},
/* 4107 */ {(15<<2)|3,{50,48,101}},
/* 4108 */ {(15<<2)|3,{50,48,105}},
/* 4109 */ {(15<<2)|3,{50,48,105}},
/* 4110 */ {(15<<2)|3,{50,48,111}},
/* 4111 */ {(15<<2)|3,{50,48,111}},
/* 4112 */ {(15<<2)|3,{50,48,115}},
/* 4113 */ {(15<<2)|3,{50,48,115}},
/* 4114 */ {(15<<2)|3,{50,48,116}},
/* 4115 */ {(15<<2)|3,{50,48,116}},
/* 4116 */ {(16<<2)|3,{50,48,32}},
/* 4117 */ {(16<<2)|3,{50,48,37}},
/* 4118 */ {(16<<2)|3,{50,48,45}},
/* 4119 */ {(16<<2)|3,{50,48,46}},
/* 4120 */ {(16<<2)|3,{50,48,47}},
/* 4121 */ {(16<<2)|3,{50,48,51}},
/* 4122 */ {(16<<2)|3,{50,48,52}},
/* 4123 */ {(16<<2)|3,{50,48,53}},
/* 4124 */ {(16<<2)|3,{50,48,54}},
/* 4125 */ {(16<<2)|3,{50,48,55}},
/* 4126 */ {(16<<2)|3,{50,48,56}},
/* 4127 */ {(16<<2)|3,{50,48,57}},
/* 4128 */ {(16<<2)|3,{50,48,61}},
/* 4129 */ {(16<<2)|3,{50,48,65}},
/* 4130 */ {(16<<2)|3,{50,48,95}},
/* 4131 */ {(16<<2)|3,{50,48,98}},
/* 4132 */ {(16<<2)|3,{50,48,100}},
/* 4133 */ {(16<<2)|3,{50,48,102}},
/* 4134 */ {(16<<2)|3,{50,48,103}},
/* 4135 */ {(16<<2)|3,{50,48,104}},
/* 4136 */ {(16<<2)|3,{50,48,108}},
/* 4137 */ {(16<<2)|3,{50,48,109}},
/* 4138 */ {(16<<2)|3,{50,48,110}},
/* 4139 */ {(16<<2)|3,{50,48,112}},
/* 4140 */ {(16<<2)|3,{50,48,114}},
/* 4141 */ {(16<<2)|3,{50,48,117}},
/* 4142 */ {(10<<2)|2,{50,48,0}},
/* 4143 */ {(10<<2)|2,{50,48,0}},
/* 4144 */ {(10<<2)|2,{50,48,0}},
/* 4145 */ {(10<<2)|2,{50,48,0}},
/* 4146 */ {(10<<2)|2,{50,48,0}},
/* 4147 */ {(10<<2)|2,{50,48,0}},
/* 4148 */ {(10<<2)|2,{50,48,0}},
/* 4149 */ {(10<<2)|2,{50,48,0}},
/* 4150 */ {(10<<2)|2,{50,48,0}},
/* 4151 */ {(10<<2)|2,{50,48,0}},
/* 4152 */ {(10<<2)|2,{50,48,0}},
/* 4153 */ {(10<<2)|2,{50,48,0}},
/* 4154 */ {(10<<2)|2,{50,48,0}},
/* 4155 */ {(10<<2)|2,{50,48,0}},
/* 4156 */ {(10<<2)|2,{50,48,0}},
/* 4157 */ {(10<<2)|2,{50,48,0}},
/* 4158 */ {(10<<2)|2,{50,48,0}},
/* 4159 */ {(10<<2)|2,{50,48,0}},
/* 4160 */ {(15<<2)|3,{50,49,48}},
/* 4161 */ {(15<<2)|3,{50,49,48}},
/* 4162 */ {(15<<2)|3,{50,49,49}},
/* 4163 */ {(15<<2)|3,{50,49,49}},
/* 4164 */ {(15<<2)|3,{50,49,50}},
/* 4165 */ {(15<<2)|3,{50,49,50}},
/* 4166 */ {(15<<2)|3,{50,49,97}},
/* 4167 */ {(15<<2)|3,{50,49,97}},
/* 4168 */ {(15<<2)|3,{50,49,99}},
/* 4169 */ {(15<<2)|3,{50,49,99}},
/* 4170 */ {(15<<2)|3,{50,49,101}},
/* 4171 */ {(15<<2)|3,{50,49,101}},
/* 4172 */ {(15<<2)|3,{50,49,105}},
/* 4173 */ {(15<<2)|3,{50,49,105}},
/* 4174 */ {(15<<2)|3,{50,49,111}},
/* 4175 */ {(15<<2)|3,{50,49,111}},
/* 4176 */ {(15<<2)|3,{50,49,115}},
/* 4177 */ {(15<<2)|3,{50,49,115}},
/* 4178 */ {(15<<2)|3,{50,49,116}},
/* 4179 */ {(15<<2)|3,{50,49,116}},
/* 4180 */ {(16<<2)|3,{50,49,32}},
/* 4181 */ {(16<<2)|3,{50,49,37}},
/* 4182 */ {(16<<2)|3,{50,49,45}},
/* 4183 */ {(16<<2)|3,{50,49,46}},
/* 4184 */ {(16<<2)|3,{50,49,47}},
/* 4185 */ {(16<<2)|3,{50,49,51}},
/* 4186 */ {(16<<2)|3,{50,49,52}},
/* 4187 */ {(16<<2)|3,{50,49,53}},
/* 4188 */ {(16<<2)|3,{50,49,54}},
/* 4189 */ {(16<<2)|3,{50,49,55}},
/* 4190 */ {(16<<2)|3,{50,49,56}},
/* 4191 */ {(16<<2)|3,{50,49,57}},
/* 4192 */ {(16<<2)|3,{50,49,61}},
/* 4193 */ {(16<<2)|3,{50,49,65}},
/* 4194 */ {(16<<2)|3,{50,49,95}},
/* 4195 */ {(16<<2)|3,{50,49,98}},
/* 4196 */ {(16<<2)|3,{50,49,100}},
/* 4197 */ {(16<<2)|3,{50,49,102}},
/* 4198 */ {(16<<2)|3,{50,49,103}},
/* 4199 */ {(16<<2)|3,{50,49,104}},
/* 4200 */ {(16<<2)|3,{50,49,108}},
/* 4201 */ {(16<<2)|3,{50,49,109}},
/* 4202 */ {(16<<2)|3,{50,49,110}},
/* 4203 */ {(16<<2)|3,{50,49,112}},
/* 4204 */ {(16<<2)|3,{50,49,114}},
/* 4205 */ {(16<<2)|3,{50,49,117}},
/* 4206 */ {(10<<2)|2,{50,49,0}},
/* 4207 */ {(10<<2)|2,{50,49,0}},
/* 4208 */ {(10<<2)|2,{50,49,0}},
/* 4209 */ {(10<<2)|2,{50,49,0}},
/* 4210 */ {(10<<2)|2,{50,49,0}},
/* 4211 */ {(10<<2)|2,{50,49,0}},
/* 4212 */ {(10<<2)|2,{50,49,0}},
/* 4213 */ {(10<<2)|2,{50,49,0}},
/* 4214 */ {(10<<2)|2,{50,49,0}},
/* 4215 */ {(10<<2)|2,{50,49,0}},
/* 4216 */ {(10<<2)|2,{50,49,0}},
/* 4217 */ {(10<<2)|2,{50,49,0}},
/* 4218 */ {(10<<2)|2,{50,49,0}},
/* 4219 */ {(10<<2)|2,{50,49,0}},
/* 4220 */ {(10<<2)|2,{50,49,0}},
/* 4221 */ {(10<<2)|2,{50,49,0}},
/* 4222 */ {(10<<2)|2,{50,49,0}},
/* 4223 */ {(10<<2)|2,{50,49,0}},
/* 4224 */ {(15<<2)|3,{50,50,48}},
/* 4225 */ {(15<<2)|3,{50,50,48}},
/* 4226 */ {(15<<2)|3,{50,50,49}},
/* 4227 */ {(15<<2)|3,{50,50,49}},
/* 4228 */ {(15<<2)|3,{50,50,50}},
/* 4229 */ {(15<<2)|3,{50,50,50}},
/* 4230 */ {(15<<2)|3,{50,50,97}},
/* 4231 */ {(15<<2)|3,{50,50,97}},
/* 4232 */ {(15<<2)|3,{50,50,99}},
/* 4233 */ {(15<<2)|3,{50,50,99}},
/* 4234 */ {(15<<2)|3,{50,50,101}},
/* 4235 */ {(15<<2)|3,{50,50,101}},
/* 4236 */ {(15<<2)|3,{50,50,105}},
/* 4237 */ {(15<<2)|3,{50,50,105}},
/* 4238 */ {(15<<2)|3,{50,50,111}},
/* 4239 */ {(15<<2)|3,{50,50,111}},
/* 4240 */ {(15<<2)|3,{50,50,115}},
/* 4241 */ {(15<<2)|3,{50,50,115}},
/* 4242 */ {(15<<2)|3,{50,50,116}},
/* 4243 */ {(15<<2)|3,{50,50,116}},
/* 4244 */ {(16<<2)|3,{50,50,32}},
/* 4245 */ {(16<<2)|3,{50,50,37}},
/* 4246 */ {(16<<2)|3,{50,50,45}},
/* 4247 */ {(16<<2)|3,{50,50,46}},
/* 4248 */ {(16<<2)|3,{50,50,47}},
/* 4249 */ {(16<<2)|3,{50,50,51}},
/* 4250 */ {(16<<2)|3,{50,50,52}},
/* 4251 */ {(16<<2)|3,{50,50,53}},
/* 4252 */ {(16<<2)|3,{50,50,54}},
/* 4253 */ {(16<<2)|3,{50,50,55}},
/* 4254 */ {(16<<2)|3,{50,50,56}},
/* 4255 */ {(16<<2)|3,{50,50,57}},
/* 4256 */ {(16<<2)|3,{50,50,61}},
/* 4257 */ {(16<<2)|3,{50,50,65}},
/* 4258 */ {(16<<2)|3,{50,50,95}},
/* 4259 */ {(16<<2)|3,{50,50,98}},
/* 4260 */ {(16<<2)|3,{50,50,100}},
/* 4261 */ {(16<<2)|3,{50,50,102}},
/* 4262 */ {(16<<2)|3,{50,50,103}},
/* 4263 */ {(16<<2)|3,{50,50,104}},
/* 4264 */ {(16<<2)|3,{50,50,108}},
/* 4265 */ {(16<<2)|3,{50,50,109}},
/* 4266 */ {(16<<2)|3,{50,50,110}},
/* 4267 */ {(16<<2)|3,{50,50,112}},
/* 4268 */ {(16<<2)|3,{50,50,114}},
/* 4269 */ {(16<<2)|3,{50,50,117}},
/* 4270 */ {(10<<2)|2,{50,50,0}},
/* 4271 */ {(10<<2)|2,{50,50,0}},
/* 4272 */ {(10<<2)|2,{50,50,0}},
/* 4273 */ {(10<<2)|2,{50,50,0}},
/* 4274 */ {(10<<2)|2,{50,50,0}},
/* 4275 */ {(10<<2)|2,{50,50,0}},
/* 4276 */ {(10<<2)|2,{50,50,0}},
/* 4277 */ {(10<<2)|2,{50,50,0}},
/* 4278 */ {(10<<2)|2,{50,50,0}},
/* 4279 */ {(10<<2)|2,{50,50,0}},
/* 4280 */ {(10<<2)|2,{50,50,0}},
/* 4281 */ {(10<<2)|2,{50,50,0}},
/* 4282 */ {(10<<2)|2,{50,50,0}},
/* 4283 */ {(10<<2)|2,{50,50,0}},
/* 4284 */ {(10<<2)|2,{50,50,0}},
/* 4285 */ {(10<<2)|2,{50,50,0}},
/* 4286 */ {(10<<2)|2,{50,50,0}},
/* 4287 */ {(10<<2)|2,{50,50,0}},
/* 4288 */ {(15<<2)|3,{50,97,48}},
/* 4289 */ {(15<<2)|3,{50,97,48}},
/* 4290 */ {(15<<2)|3,{50,97,49}},
/* 4291 */ {(15<<2)|3,{50,97,49}},
/* 4292 */ {(15<<2)|3,{50,97,50}},
/* 4293 */ {(15<<2)|3,{50,97,50}},
/* 4294 */ {(15<<2)|3,{50,97,97}},
/* 4295 */ {(15<<2)|3,{50,97,97}},
/* 4296 */ {(15<<2)|3,{50,97,99}},
/* 4297 */ {(15<<2)|3,{50,97,99}},
/* 4298 */ {(15<<2)|3,{50,97,101}},
/* 4299 */ {(15<<2)|3,{50,97,101}},
/* 4300 */ {(15<<2)|3,{50,97,105}},
/* 4301 */ {(15<<2)|3,{50,97,105}},
/* 4302 */ {(15<<2)|3,{50,97,111}},
/* 4303 */ {(15<<2)|3,{50,97,111}},
/* 4304 */ {(15<<2)|3,{50,97,115}},
/* 4305 */ {(15<<2)|3,{50,97,115}},
/* 4306 */ {(15<<2)|3,{50,97,116}},
/* 4307 */ {(15<<2)|3,{50,97,116}},
/* 4308 */ {(16<<2)|3,{50,97,32}},
/* 4309 */ {(16<<2)|3,{50,97,37}},
/* 4310 */ {(16<<2)|3,{50,97,45}},
/* 4311 */ {(16<<2)|3,{50,97,46}},
/* 4312 */ {(16<<2)|3,{50,97,47}},
/* 4313 */ {(16<<2)|3,{50,97,51}},
/* 4314 */ {(16<<2)|3,{50,97,52}},
/* 4315 */ {(16<<2)|3,{50,97,53}},
/* 4316 */ {(16<<2)|3,{50,97,54}},
/* 4317 */ {(16<<2)|3,{50,97,55}},
/* 4318 */ {(16<<2)|3,{50,97,56}},
/* 4319 */ {(16<<2)|3,{50,97,57}},
/* 4320 */ {(16<<2)|3,{50,97,61}},
/* 4321 */ {(16<<2)|3,{50,97,65}},
/* 4322 */ {(16<<2)|3,{50,97,95}},
/* 4323 */ {(16<<2)|3,{50,97,98}},
/* 4324 */ {(16<<2)|3,{50,97,100}},
/* 4325 */ {(16<<2)|3,{50,97,102}},
/* 4326 */ {(16<<2)|3,{50,97,103}},
/* 4327 */ {(16<<2)|3,{50,97,104}},
/* 4328 */ {(16<<2)|3,{50,97,108}},
/* 4329 */ {(16<<2)|3,{50,97,109}},
/* 4330 */ {(16<<2)|3,{50,97,110}},
/* 4331 */ {(16<<2)|3,{50,97,112}},
/* 4332 */ {(16<<2)|3,{50,97,114}},
/* 4333 */ {(16<<2)|3,{50,97,117}},
/* 4334 */ {(10<<2)|2,{50,97,0}},
/* 4335 */ {(10<<2)|2,{50,97,0}},
/* 4336 */ {(10<<2)|2,{50,97,0}},
/* 4337 */ {(10<<2)|2,{50,97,0}},
/* 4338 */ {(10<<2)|2,{50,97,0}},
/* 4339 */ {(10<<2)|2,{50,97,0}},
/* 4340 */ {(10<<2)|2,{50,97,0}},
/* 4341 */ {(10<<2)|2,{50,97,0}},
/* 4342 */ {(10<<2)|2,{50,97,0}},
/* 4343 */ {(10<<2)|2,{50,97,0}},
/* 4344 */ {(10<<2)|2,{50,97,0}},
/* 4345 */ {(10<<2)|2,{50,97,0}},
/* 4346 */ {(10<<2)|2,{50,97,0}},
/* 4347 */ {(10<<2)|2,{50,97,0}},
/* 4348 */ {(10<<2)|2,{50,97,0}},
/* 4349 */ {(10<<2)|2,{50,97,0}},
/* 4350 */ {(10<<2)|2,{50,97,0}},
/* 4351 */ {(10<<2)|2,{50,97,0}},
/* 4352 */ {(15<<2)|3,{50,99,48}},
/* 4353 */ {(15<<2)|3,{50,99,48}},
/* 4354 */ {(15<<2)|3,{50,99,49}},
/* 4355 */ {(15<<2)|3,{50,99,49}},
/* 4356 */ {(15<<2)|3,{50,99,50}},
/* 4357 */ {(15<<2)|3,{50,99,50}},
/* 4358 */ {(15<<2)|3,{50,99,97}},
/* 4359 */ {(15<<2)|3,{50,99,97}},
/* 4360 */ {(15<<2)|3,{50,99,99}},
/* 4361 */ {(15<<2)|3,{50,99,99}},
/* 4362 */ {(15<<2)|3,{50,99,101}},
/* 4363 */ {(15<<2)|3,{50,99,101}},
/* 4364 */ {(15<<2)|3,{50,99,105}},
/* 4365 */ {(15<<2)|3,{50,99,105}},
/* 4366 */ {(15<<2)|3,{50,99,111}},
/* 4367 */ {(15<<2)|3,{50,99,111}},
/* 4368 */ {(15<<2)|3,{50,99,115}},
/* 4369 */ {(15<<2)|3,{50,99,115}},
/* 4370 */ {(15<<2)|3,{50,99,116}},
/* 4371 */ {(15<<2)|3,{50,99,116}},
/* 4372 */ {(16<<2)|3,{50,99,32}},
/* 4373 */ {(16<<2)|3,{50,99,37}},
/* 4374 */ {(16<<2)|3,{50,99,45}},
/* 4375 */ {(16<<2)|3,{50,99,46}},
/* 4376 */ {(16<<2)|3,{50,99,47}},
/* 4377 */ {(16<<2)|3,{50,99,51}},
/* 4378 */ {(16<<2)|3,{50,99,52}},
/* 4379 */ {(16<<2)|3,{50,99,53}},
/* 4380 */ {(16<<2)|3,{50,99,54}},
/* 4381 */ {(16<<2)|3,{50,99,55}},
/* 4382 */ {(16<<2)|3,{50,99,56}},
/* 4383 */ {(16<<2)|3,{50,99,57}},
/* 4384 */ {(16<<2)|3,{50,99,61}},
/* 4385 */ {(16<<2)|3,{50,99,65}},
/* 4386 */ {(16<<2)|3,{50,99,95}},
/* 4387 */ {(16<<2)|3,{50,99,98}},
/* 4388 */ {(16<<2)|3,{50,99,100}},
/* 4389 */ {(16<<2)|3,{50,99,102}},
/* 4390 */ {(16<<2)|3,{50,99,103}},
/* 4391 */ {(16<<2)|3,{50,99,104}},
/* 4392 */ {(16<<2)|3,{50,99,108}},
/* 4393 */ {(16<<2)|3,{50,99,109}},
/* 4394 */ {(16<<2)|3,{50,99,110}},
/* 4395 */ {(16<<2)|3,{50,99,112}},
/* 4396 */ {(16<<2)|3,{50,99,114}},
/* 4397 */ {(16<<2)|3,{50,99,117}},
/* 4398 */ {(10<<2)|2,{50,99,0}},
/* 4399 */ {(10<<2)|2,{50,99,0}},
/* 4400 */ {(10<<2)|2,{50,99,0}},
/* 4401 */ {(10<<2)|2,{50,99,0}},
/* 4402 */ {(10<<2)|2,{50,99,0}},
/* 4403 */ {(10<<2)|2,{50,99,0}},
/* 4404 */ {(10<<2)|2,{50,99,0}},
/* 4405 */ {(10<<2)|2,{50,99,0}},
/* 4406 */ {(10<<2)|2,{50,99,0}},
/* 4407 */ {(10<<2)|2,{50,99,0}},
/* 4408 */ {(10<<2)|2,{50,99,0}},
/* 4409 */ {(10<<2)|2,{50,99,0}},
/* 4410 */ {(10<<2)|2,{50,99,0}},
/* 4411 */ {(10<<2)|2,{50,99,0}},
/* 4412 */ {(10<<2)|2,{50,99,0}},
/* 4413 */ {(10<<2)|2,{50,99,0}},
/* 4414 */ {(10<<2)|2,{50,99,0}},
/* 4415 */ {(10<<2)|2,{50,99,0}},
/* 4416 */ {(15<<2)|3,{50,101,48}},
/* 4417 */ {(15<<2)|3,{50,101,48}},
/* 4418 */ {(15<<2)|3,{50,101,49}},
/* 4419 */ {(15<<2)|3,{50,101,49}},
/* 4420 */ {(15<<2)|3,{50,101,50}},
/* 4421 */ {(15<<2)|3,{50,101,50}},
/* 4422 */ {(15<<2)|3,{50,101,97}},
/* 4423 */ {(15<<2)|3,{50,101,97}},
/* 4424 */ {(15<<2)|3,{50,101,99}},
/* 4425 */ {(15<<2)|3,{50,101,99}},
/* 4426 */ {(15<<2)|3,{50,101,101}},
/* 4427 */ {(15<<2)|3,{50,101,101}},
/* 4428 */ {(15<<2)|3,{50,101,105}},
/* 4429 */ {(15<<2)|3,{50,101,105}},
/* 4430 */ {(15<<2)|3,{50,101,111}},
/* 4431 */ {(15<<2)|3,{50,101,111}},
/* 4432 */ {(15<<2)|3,{50,101,115}},
/* 4433 */ {(15<<2)|3,{50,101,115}},
/* 4434 */ {(15<<2)|3,{50,101,116}},
/* 4435 */ {(15<<2)|3,{50,101,116}},
/* 4436 */ {(16<<2)|3,{50,101,32}},
/* 4437 */ {(16<<2)|3,{50,101,37}},
/* 4438 */ {(16<<2)|3,{50,101,45}},
/* 4439 */ {(16<<2)|3,{50,101,46}},
/* 4440 */ {(16<<2)|3,{50,101,47}},
/* 4441 */ {(16<<2)|3,{50,101,51}},
/* 4442 */ {(16<<2)|3,{50,101,52}},
/* 4443 */ {(16<<2)|3,{50,101,53}},
/* 4444 */ {(16<<2)|3,{50,101,54}},
/* 4445 */ {(16<<2)|3,{50,101,55}},
/* 4446 */ {(16<<2)|3,{50,101,56}},
/* 4447 */ {(16<<2)|3,{50,101,57}},
/* 4448 */ {(16<<2)|3,{50,101,61}},
/* 4449 */ {(16<<2)|3,{50,101,65}},
/* 4450 */ {(16<<2)|3,{50,101,95}},
/* 4451 */ {(16<<2)|3,{50,101,98}},
/* 4452 */ {(16<<2)|3,{50,101,100}},
/* 4453 */ {(16<<2)|3,{50,101,102}},
/* 4454 */ {(16<<2)|3,{50,101,103}},
/* 4455 */ {(16<<2)|3,{50,101,104}},
/* 4456 */ {(16<<2)|3,{50,101,108}},
/* 4457 */ {(16<<2)|3,{50,101,109}},
/* 4458 */ {(16<<2)|3,{50,101,110}},
/* 4459 */ {(16<<2)|3,{50,101,112}},
/* 4460 */ {(16<<2)|3,{50,101,114}},
/* 4461 */ {(16<<2)|3,{50,101,117}},
/* 4462 */ {(10<<2)|2,{50,101,0}},
/* 4463 */ {(10<<2)|2,{50,101,0}},
/* 4464 */ {(10<<2)|2,{50,101,0}},
/* 4465 */ {(10<<2)|2,{50,101,0}},
/* 4466 */ {(10<<2)|2,{50,101,0}},
/* 4467 */ {(10<<2)|2,{50,101,0}},
/* 4468 */ {(10<<2)|2,{50,101,0}},
/* 4469 */ {(10<<2)|2,{50,101,0}},
/* 4470 */ {(10<<2)|2,{50,101,0}},
/* 4471 */ {(10<<2)|2,{50,101,0}},
/* 4472 */ {(10<<2)|2,{50,101,0}},
/* 4473 */ {(10<<2)|2,{50,101,0}},
/* 4474 */ {(10<<2)|2,{50,101,0}},
/* 4475 */ {(10<<2)|2,{50,101,0}},
/* 4476 */ {(10<<2)|2,{50,101,0}},
/* 4477 */ {(10<<2)|2,{50,101,0}},
/* 4478 */ {(10<<2)|2,{50,101,0}},
/* 4479 */ {(10<<2)|2,{50,101,0}},
/* 4480 */ {(15<<2)|3,{50,105,48}},
/* 4481 */ {(15<<2)|3,{50,105,48}},
/* 4482 */ {(15<<2)|3,{50,105,49}},
/* 4483 */ {(15<<2)|3,{50,105,49}},
/* 4484 */ {(15<<2)|3,{50,105,50}},
/* 4485 */ {(15<<2)|3,{50,105,50}},
/* 4486 */ {(15<<2)|3,{50,105,97}},
/* 4487 */ {(15<<2)|3,{50,105,97}},
/* 4488 */ {(15<<2)|3,{50,105,99}},
/* 4489 */ {(15<<2)|3,{50,105,99}},
/* 4490 */ {(15<<2)|3,{50,105,101}},
/* 4491 */ {(15<<2)|3,{50,105,101}},
/* 4492 */ {(15<<2)|3,{50,105,105}},
/* 4493 */ {(15<<2)|3,{50,105,105}},
/* 4494 */ {(15<<2)|3,{50,105,111}},
/* 4495 */ {(15<<2)|3,{50,105,111}},
/* 4496 */ {(15<<2)|3,{50,105,115}},
/* 4497 */ {(15<<2)|3,{50,105,115}},
/* 4498 */ {(15<<2)|3,{50,105,116}},
/* 4499 */ {(15<<2)|3,{50,105,116}},
/* 4500 */ {(16<<2)|3,{50,105,32}},
/* 4501 */ {(16<<2)|3,{50,105,37}},
/* 4502 */ {(16<<2)|3,{50,105,45}},
/* 4503 */ {(16<<2)|3,{50,105,46}},
/* 4504 */ {(16<<2)|3,{50,105,47}},
/* 4505 */ {(16<<2)|3,{50,105,51}},
/* 4506 */ {(16<<2)|3,{50,105,52}},
/* 4507 */ {(16<<2)|3,{50,105,53}},
/* 4508 */ {(16<<2)|3,{50,105,54}},
/* 4509 */ {(16<<2)|3,{50,105,55}},
/* 4510 */ {(16<<2)|3,{50,105,56}},
/* 4511 */ {(16<<2)|3,{50,105,57}},
/* 4512 */ {(16<<2)|3,{50,105,61}},
/* 4513 */ {(16<<2)|3,{50,105,65}},
/* 4514 */ {(16<<2)|3,{50,105,95}},
/* 4515 */ {(16<<2)|3,{50,105,98}},
/* 4516 */ {(16<<2)|3,{50,105,100}},
/* 4517 */ {(16<<2)|3,{50,105,102}},
/* 4518 */ {(16<<2)|3,{50,105,103}},
/* 4519 */ {(16<<2)|3,{50,105,104}},
/* 4520 */ {(16<<2)|3,{50,105,108}},
/* 4521 */ {(16<<2)|3,{50,105,109}},
/* 4522 */ {(16<<2)|3,{50,105,110}},
/* 4523 */ {(16<<2)|3,{50,105,112}},
/* 4524 */ {(16<<2)|3,{50,105,114}},
/* 4525 */ {(16<<2)|3,{50,105,117}},
/* 4526 */ {(10<<2)|2,{50,105,0}},
/* 4527 */ {(10<<2)|2,{50,105,0}},
/* 4528 */ {(10<<2)|2,{50,105,0}},
/* 4529 */ {(10<<2)|2,{50,105,0}},
/* 4530 */ {(10<<2)|2,{50,105,0}},
/* 4531 */ {(10<<2)|2,{50,105,0}},
/* 4532 */ {(10<<2)|2,{50,105,0}},
/* 4533 */ {(10<<2)|2,{50,105,0}},
/* 4534 */ {(10<<2)|2,{50,105,0}},
/* 4535 */ {(10<<2)|2,{50,105,0}},
/* 4536 */ {(10<<2)|2,{50,105,0}},
/* 4537 */ {(10<<2)|2,{50,105,0}},
/* 4538 */ {(10<<2)|2,{50,105,0}},
/* 4539 */ {(10<<2)|2,{50,105,0}},
/* 4540 */ {(10<<2)|2,{50,105,0}},
/* 4541 */ {(10<<2)|2,{50,105,0}},
/* 4542 */ {(10<<2)|2,{50,105,0}},
/* 4543 */ {(10<<2)|2,{50,105,0}},
/* 4544 */ {(15<<2)|3,{50,111,48}},
/* 4545 */ {(15<<2)|3,{50,111,48}},
/* 4546 */ {(15<<2)|3,{50,111,49}},
/* 4547 */ {(15<<2)|3,{50,111,49}},
/* 4548 */ {(15<<2)|3,{50,111,50}},
/* 4549 */ {(15<<2)|3,{50,111,50}},
/* 4550 */ {(15<<2)|3,{50,111,97}},
/* 4551 */ {(15<<2)|3,{50,111,97}},
/* 4552 */ {(15<<2)|3,{50,111,99}},
/* 4553 */ {(15<<2)|3,{50,111,99}},
/* 4554 */ {(15<<2)|3,{50,111,101}},
/* 4555 */ {(15<<2)|3,{50,111,101}},
/* 4556 */ {(15<<2)|3,{50,111,105}},
/* 4557 */ {(15<<2)|3,{50,111,105}},
/* 4558 */ {(15<<2)|3,{50,111,111}},
/* 4559 */ {(15<<2)|3,{50,111,111}},
/* 4560 */ {(15<<2)|3,{50,111,115}},
/* 4561 */ {(15<<2)|3,{50,111,115}},
/* 4562 */ {(15<<2)|3,{50,111,116}},
/* 4563 */ {(15<<2)|3,{50,111,116}},
/* 4564 */ {(16<<2)|3,{50,111,32}},
/* 4565 */ {(16<<2)|3,{50,111,37}},
/* 4566 */ {(16<<2)|3,{50,111,45}},
/* 4567 */ {(16<<2)|3,{50,111,46}},
/* 4568 */ {(16<<2)|3,{50,111,47}},
/* 4569 */ {(16<<2)|3,{50,111,51}},
/* 4570 */ {(16<<2)|3,{50,111,52}},
/* 4571 */ {(16<<2)|3,{50,111,53}},
/* 4572 */ {(16<<2)|3,{50,111,54}},
/* 4573 */ {(16<<2)|3,{50,111,55}},
/* 4574 */ {(16<<2)|3,{50,111,56}},
/* 4575 */ {(16<<2)|3,{50,111,57}},
/* 4576 */ {(16<<2)|3,{50,111,61}},
/* 4577 */ {(16<<2)|3,{50,111,65}},
/* 4578 */ {(16<<2)|3,{50,111,95}},
/* 4579 */ {(16<<2)|3,{50,111,98}},
/* 4580 */ {(16<<2)|3,{50,111,100}},
/* 4581 */ {(16<<2)|3,{50,111,102}},
/* 4582 */ {(16<<2)|3,{50,111,103}},
/* 4583 */ {(16<<2)|3,{50,111,104}},
/* 4584 */ {(16<<2)|3,{50,111,108}},
/* 4585 */ {(16<<2)|3,{50,111,109}},
/* 4586 */ {(16<<2)|3,{50,111,110}},
/* 4587 */ {(16<<2)|3,{50,111,112}},
/* 4588 */ {(16<<2)|3,{50,111,114}},
/* 4589 */ {(16<<2)|3,{50,111,117}},
/* 4590 */ {(10<<2)|2,{50,111,0}},
/* 4591 */ {(10<<2)|2,{50,111,0}},
/* 4592 */ {(10<<2)|2,{50,111,0}},
/* 4593 */ {(10<<2)|2,{50,111,0}},
/* 4594 */ {(10<<2)|2,{50,111,0}},
/* 4595 */ {(10<<2)|2,{50,111,0}},
/* 4596 */ {(10<<2)|2,{50,111,0}},
/* 4597 */ {(10<<2)|2,{50,111,0}},
/* 4598 */ {(10<<2)|2,{50,111,0}},
/* 4599 */ {(10<<2)|2,{50,111,0}},
/* 4600 */ {(10<<2)|2,{50,111,0}},
/* 4601 */ {(10<<2)|2,{50,111,0}},
/* 4602 */ {(10<<2)|2,{50,111,0}},
/* 4603 */ {(10<<2)|2,{50,111,0}},
/* 4604 */ {(10<<2)|2,{50,111,0}},
/* 4605 */ {(10<<2)|2,{50,111,0}},
/* 4606 */ {(10<<2)|2,{50,111,0}},
/* 4607 */ {(10<<2)|2,{50,111,0}},
/* 4608 */ {(15<<2)|3,{50,115,48}},
/* 4609 */ {(15<<2)|3,{50,115,48}},
/* 4610 */ {(15<<2)|3,{50,115,49}},
/* 4611 */ {(15<<2)|3,{50,115,49}},
/* 4612 */ {(15<<2)|3,{50,115,50}},
/* 4613 */ {(15<<2)|3,{50,115,50}},
/* 4614 */ {(15<<2)|3,{50,115,97}},
/* 4615 */ {(15<<2)|3,{50,115,97}},
/* 4616 */ {(15<<2)|3,{50,115,99}},
/* 4617 */ {(15<<2)|3,{50,115,99}},
/* 4618 */ {(15<<2)|3,{50,115,101}},
/* 4619 */ {(15<<2)|3,{50,115,101}},
/* 4620 */ {(15<<2)|3,{50,115,105}},
/* 4621 */ {(15<<2)|3,{50,115,105}},
/* 4622 */ {(15<<2)|3,{50,115,111}},
/* 4623 */ {(15<<2)|3,{50,115,111}},
/* 4624 */ {(15<<2)|3,{50,115,115}},
/* 4625 */ {(15<<2)|3,{50,115,115}},
/* 4626 */ {(15<<2)|3,{50,115,116}},
/* 4627 */ {(15<<2)|3,{50,115,116}},
/* 4628 */ {(16<<2)|3,{50,115,32}},
/* 4629 */ {(16<<2)|3,{50,115,37}},
/* 4630 */ {(16<<2)|3,{50,115,45}},
/* 4631 */ {(16<<2)|3,{50,115,46}},
/* 4632 */ {(16<<2)|3,{50,115,47}},
/* 4633 */ {(16<<2)|3,{50,115,51}},
/* 4634 */ {(16<<2)|3,{50,115,52}},
/* 4635 */ {(16<<2)|3,{50,115,53}},
/* 4636 */ {(16<<2)|3,{50,115,54}},
/* 4637 */ {(16<<2)|3,{50,115,55}},
/* 4638 */ {(16<<2)|3,{50,115,56}},
/* 4639 */ {(16<<2)|3,{50,115,57}},
/* 4640 */ {(16<<2)|3,{50,115,61}},
/* 4641 */ {(16<<2)|3,{50,115,65}},
/* 4642 */ {(16<<2)|3,{50,115,95}},
/* 4643 */ {(16<<2)|3,{50,115,98}},
/* 4644 */ {(16<<2)|3,{50,115,100}},
/* 4645 */ {(16<<2)|3,{50,115,102}},
/* 4646 */ {(16<<2)|3,{50,115,103}},
/* 4647 */ {(16<<2)|3,{50,115,104}},
/* 4648 */ {(16<<2)|3,{50,115,108}},
/* 4649 */ {(16<<2)|3,{50,115,109}},
/* 4650 */ {(16<<2)|3,{50,115,110}},
/* 4651 */ {(16<<2)|3,{50,115,112}},
/* 4652 */ {(16<<2)|3,{50,115,114}},
/* 4653 */ {(16<<2)|3,{50,115,117}},
/* 4654 */ {(10<<2)|2,{50,115,0}},
/* 4655 */ {(10<<2)|2,{50,115,0}},
/* 4656 */ {(10<<2)|2,{50,115,0}},
/* 4657 */ {(10<<2)|2,{50,115,0}},
/* 4658 */ {(10<<2)|2,{50,115,0}},
/* 4659 */ {(10<<2)|2,{50,115,0}},
/* 4660 */ {(10<<2)|2,{50,115,0}},
/* 4661 */ {(10<<2)|2,{50,115,0}},
/* 4662 */ {(10<<2)|2,{50,115,0}},
/* 4663 */ {(10<<2)|2,{50,115,0}},
/* 4664 */ {(10<<2)|2,{50,115,0}},
/* 4665 */ {(10<<2)|2,{50,115,0}},
/* 4666 */ {(10<<2)|2,{50,115,0}},
/* 4667 */ {(10<<2)|2,{50,115,0}},
/* 4668 */ {(10<<2)|2,{50,115,0}},
/* 4669 */ {(10<<2)|2,{50,115,0}},
/* 4670 */ {(10<<2)|2,{50,115,0}},
/* 4671 */ {(10<<2)|2,{50,115,0}},
/* 4672 */ {(15<<2)|3,{50,116,48}},
/* 4673 */ {(15<<2)|3,{50,116,48}},
/* 4674 */ {(15<<2)|3,{50,116,49}},
/* 4675 */ {(15<<2)|3,{50,116,49}},
/* 4676 */ {(15<<2)|3,{50,116,50}},
/* 4677 */ {(15<<2)|3,{50,116,50}},
/* 4678 */ {(15<<2)|3,{50,116,97}},
/* 4679 */ {(15<<2)|3,{50,116,97}},
/* 4680 */ {(15<<2)|3,{50,116,99}},
/* 4681 */ {(15<<2)|3,{50,116,99}},
/* 4682 */ {(15<<2)|3,{50,116,101}},
/* 4683 */ {(15<<2)|3,{50,116,101}},
/* 4684 */ {(15<<2)|3,{50,116,105}},
/* 4685 */ {(15<<2)|3,{50,116,105}},
/* 4686 */ {(15<<2)|3,{50,116,111}},
/* 4687 */ {(15<<2)|3,{50,116,111}},
/* 4688 */ {(15<<2)|3,{50,116,115}},
/* 4689 */ {(15<<2)|3,{50,116,115}},
/* 4690 */ {(15<<2)|3,{50,116,116}},
/* 4691 */ {(15<<2)|3,{50,116,116}},
/* 4692 */ {(16<<2)|3,{50,116,32}},
/* 4693 */ {(16<<2)|3,{50,116,37}},
/* 4694 */ {(16<<2)|3,{50,116,45}},
/* 4695 */ {(16<<2)|3,{50,116,46}},
/* 4696 */ {(16<<2)|3,{50,116,47}},
/* 4697 */ {(16<<2)|3,{50,116,51}},
/* 4698 */ {(16<<2)|3,{50,116,52}},
/* 4699 */ {(16<<2)|3,{50,116,53}},
/* 4700 */ {(16<<2)|3,{50,116,54}},
/* 4701 */ {(16<<2)|3,{50,116,55}},
/* 4702 */ {(16<<2)|3,{50,116,56}},
/* 4703 */ {(16<<2)|3,{50,116,57}},
/* 4704 */ {(16<<2)|3,{50,116,61}},
/* 4705 */ {(16<<2)|3,{50,116,65}},
/* 4706 */ {(16<<2)|3,{50,116,95}},
/* 4707 */ {(16<<2)|3,{50,116,98}},
/* 4708 */ {(16<<2)|3,{50,116,100}},
/* 4709 */ {(16<<2)|3,{50,116,102}},
/* 4710 */ {(16<<2)|3,{50,116,103}},
/* 4711 */ {(16<<2)|3,{50,116,104}},
/* 4712 */ {(16<<2)|3,{50,116,108}},
/* 4713 */ {(16<<2)|3,{50,116,109}},
/* 4714 */ {(16<<2)|3,{50,116,110}},
/* 4715 */ {(16<<2)|3,{50,116,112}},
/* 4716 */ {(16<<2)|3,{50,116,114}},
/* 4717 */ {(16<<2)|3,{50,116,117}},
/* 4718 */ {(10<<2)|2,{50,116,0}},
/* 4719 */ {(10<<2)|2,{50,116,0}},
/* 4720 */ {(10<<2)|2,{50,116,0}},
/* 4721 */ {(10<<2)|2,{50,116,0}},
/* 4722 */ {(10<<2)|2,{50,116,0}},
/* 4723 */ {(10<<2)|2,{50,116,0}},
/* 4724 */ {(10<<2)|2,{50,116,0}},
/* 4725 */ {(10<<2)|2,{50,116,0}},
/* 4726 */ {(10<<2)|2,{50,116,0}},
/* 4727 */ {(10<<2)|2,{50,116,0}},
/* 4728 */ {(10<<2)|2,{50,116,0}},
/* 4729 */ {(10<<2)|2,{50,116,0}},
/* 4730 */ {(10<<2)|2,{50,116,0}},
/* 4731 */ {(10<<2)|2,{50,116,0}},
/* 4732 */ {(10<<2)|2,{50,116,0}},
/* 4733 */ {(10<<2)|2,{50,116,0}},
/* 4734 */ {(10<<2)|2,{50,116,0}},
/* 4735 */ {(10<<2)|2,{50,116,0}},
/* 4736 */ {(16<<2)|3,{50,32,48}},
/* 4737 */ {(16<<2)|3,{50,32,49}},
/* 4738 */ {(16<<2)|3,{50,32,50}},
/* 4739 */ {(16<<2)|3,{50,32,97}},
/* 4740 */ {(16<<2)|3,{50,32,99}},
/* 4741 */ {(16<<2)|3,{50,32,101}},
/* 4742 */ {(16<<2)|3,{50,32,105}},
/* 4743 */ {(16<<2)|3,{50,32,111}},
/* 4744 */ {(16<<2)|3,{50,32,115}},
/* 4745 */ {(16<<2)|3,{50,32,116}},
/* 4746 */ {(11<<2)|2,{50,32,0}},
/* 4747 */ {(11<<2)|2,{50,32,0}},
/* 4748 */ {(11<<2)|2,{50,32,0}},
/* 4749 */ {(11<<2)|2,{50,32,0}},
/* 4750 */ {(11<<2)|2,{50,32,0}},
/* 4751 */ {(11<<2)|2,{50,32,0}},
/* 4752 */ {(11<<2)|2,{50,32,0}},
/* 4753 */ {(11<<2)|2,{50,32,0}},
/* 4754 */ {(11<<2)|2,{50,32,0}},
/* 4755 */ {(11<<2)|2,{50,32,0}},
/* 4756 */ {(11<<2)|2,{50,32,0}},
/* 4757 */ {(11<<2)|2,{50,32,0}},
/* 4758 */ {(11<<2)|2,{50,32,0}},
/* 4759 */ {(11<<2)|2,{50,32,0}},
/* 4760 */ {(11<<2)|2,{50,32,0}},
/* 4761 */ {(11<<2)|2,{50,32,0}},
/* 4762 */ {(11<<2)|2,{50,32,0}},
/* 4763 */ {(11<<2)|2,{50,32,0}},
/* 4764 */ {(11<<2)|2,{50,32,0}},
/* 4765 */ {(11<<2)|2,{50,32,0}},
/* 4766 */ {(11<<2)|2,{50,32,0}},
/* 4767 */ {(11<<2)|2,{50,32,0}},
/* 4768 */ {(16<<2)|3,{50,37,48}},
/* 4769 */ {(16<<2)|3,{50,37,49}},
/* 4770 */ {(16<<2)|3,{50,37,50}},
/* 4771 */ {(16<<2)|3,{50,37,97}},
/* 4772 */ {(16<<2)|3,{50,37,99}},
/* 4773 */ {(16<<2)|3,{50,37,101}},
/* 4774 */ {(16<<2)|3,{50,37,105}},
/* 4775 */ {(16<<2)|3,{50,37,111}},
/* 4776 */ {(16<<2)|3,{50,37,115}},
/* 4777 */ {(16<<2)|3,{50,37,116}},
/* 4778 */ {(11<<2)|2,{50,37,0}},
/* 4779 */ {(11<<2)|2,{50,37,0}},
/* 4780 */ {(11<<2)|2,{50,37,0}},
/* 4781 */ {(11<<2)|2,{50,37,0}},
/* 4782 */ {(11<<2)|2,{50,37,0}},
/* 4783 */ {(11<<2)|2,{50,37,0}},
/* 4784 */ {(11<<2)|2,{50,37,0}},
/* 4785 */ {(11<<2)|2,{50,37,0}},
/* 4786 */ {(11<<2)|2,{50,37,0}},
/* 4787 */ {(11<<2)|2,{50,37,0}},
/* 4788 */ {(11<<2)|2,{50,37,0}},
/* 4789 */ {(11<<2)|2,{50,37,0}},
/* 4790 */ {(11<<2)|2,{50,37,0}},
/* 4791 */ {(11<<2)|2,{50,37,0}},
/* 4792 */ {(11<<2)|2,{50,37,0}},
/* 4793 */ {(11<<2)|2,{50,37,0}},
/* 4794 */ {(11<<2)|2,{50,37,0}},
/* 4795 */ {(11<<2)|2,{50,37,0}},
/* 4796 */ {(11<<2)|2,{50,37,0}},
/* 4797 */ {(11<<2)|2,{50,37,0}},
/* 4798 */ {(11<<2)|2,{50,37,0}},
/* 4799 */ {(11<<2)|2,{50,37,0}},
/* 4800 */ {(16<<2)|3,{50,45,48}},
/* 4801 */ {(16<<2)|3,{50,45,49}},
/* 4802 */ {(16<<2)|3,{50,45,50}},
/* 4803 */ {(16<<2)|3,{50,45,97}},
/* 4804 */ {(16<<2)|3,{50,45,99}},
/* 4805 */ {(16<<2)|3,{50,45,101}},
/* 4806 */ {(16<<2)|3,{50,45,105}},
/* 4807 */ {(16<<2)|3,{50,45,111}},
/* 4808 */ {(16<<2)|3,{50,45,115}},
/* 4809 */ {(16<<2)|3,{50,45,116}},
/* 4810 */ {(11<<2)|2,{50,45,0}},
/* 4811 */ {(11<<2)|2,{50,45,0}},
/* 4812 */ {(11<<2)|2,{50,45,0}},
/* 4813 */ {(11<<2)|2,{50,45,0}},
/* 4814 */ {(11<<2)|2,{50,45,0}},
/* 4815 */ {(11<<2)|2,{50,45,0}},
/* 4816 */ {(11<<2)|2,{50,45,0}},
/* 4817 */ {(11<<2)|2,{50,45,0}},
/* 4818 */ {(11<<2)|2,{50,45,0}},
/* 4819 */ {(11<<2)|2,{50,45,0}},
/* 4820 */ {(11<<2)|2,{50,45,0}},
/* 4821 */ {(11<<2)|2,{50,45,0}},
/* 4822 */ {(11<<2)|2,{50,45,0}},
/* 4823 */ {(11<<2)|2,{50,45,0}},
/* 4824 */ {(11<<2)|2,{50,45,0}},
/* 4825 */ {(11<<2)|2,{50,45,0}},
/* 4826 */ {(11<<2)|2,{50,45,0}},
/* 4827 */ {(11<<2)|2,{50,45,0}},
/* 4828 */ {(11<<2)|2,{50,45,0}},
/* 4829 */ {(11<<2)|2,{50,45,0}},
/* 4830 */ {(11<<2)|2,{50,45,0}},
/* 4831 */ {(11<<2)|2,{50,45,0}},
/* 4832 */ {(16<<2)|3,{50,46,48}},
/* 4833 */ {(16<<2)|3,{50,46,49}},
/* 4834 */ {(16<<2)|3,{50,46,50}},
/* 4835 */ {(16<<2)|3,{50,46,97}},
/* 4836 */ {(16<<2)|3,{50,46,99}},
/* 4837 */ {(16<<2)|3,{50,46,101}},
/* 4838 */ {(16<<2)|3,{50,46,105}},
/* 4839 */ {(16<<2)|3,{50,46,111}},
/* 4840 */ {(16<<2)|3,{50,46,115}},
/* 4841 */ {(16<<2)|3,{50,46,116}},
/* 4842 */ {(11<<2)|2,{50,46,0}},
/* 4843 */ {(11<<2)|2,{50,46,0}},
/* 4844 */ {(11<<2)|2,{50,46,0}},
/* 4845 */ {(11<<2)|2,{50,46,0}},
/* 4846 */ {(11<<2)|2,{50,46,0}},
/* 4847 */ {(11<<2)|2,{50,46,0}},
/* 4848 */ {(11<<2)|2,{50,46,0}},
/* 4849 */ {(11<<2)|2,{50,46,0}},
/* 4850 */ {(11<<2)|2,{50,46,0}},
/* 4851 */ {(11<<2)|2,{50,46,0}},
/* 4852 */ {(11<<2)|2,{50,46,0}},
/* 4853 */ {(11<<2)|2,{50,46,0}},
/* 4854 */ {(11<<2)|2,{50,46,0}},
/* 4855 */ {(11<<2)|2,{50,46,0}},
/* 4856 */ {(11<<2)|2,{50,46,0}},
/* 4857 */ {(11<<2)|2,{50,46,0}},
/* 4858 */ {(11<<2)|2,{50,46,0}},
/* 4859 */ {(11<<2)|2,{50,46,0}},
/* 4860 */ {(11<<2)|2,{50,46,0}},
/* 4861 */ {(11<<2)|2,{50,46,0}},
/* 4862 */ {(11<<2)|2,{50,46,0}},
/* 4863 */ {(11<<2)|2,{50,46,0}},
/* 4864 */ {(16<<2)|3,{50,47,48}},
/* 4865 */ {(16<<2)|3,{50,47,49}},
/* 4866 */ {(16<<2)|3,{50,47,50}},
/* 4867 */ {(16<<2)|3,{50,47,97}},
/* 4868 */ {(16<<2)|3,{50,47,99}},
/* 4869 */ {(16<<2)|3,{50,47,101}},
/* 4870 */ {(16<<2)|3,{50,47,105}},
/* 4871 */ {(16<<2)|3,{50,47,111}},
/* 4872 */ {(16<<2)|3,{50,47,115}},
/* 4873 */ {(16<<2)|3,{50,47,116}},
/* 4874 */ {(11<<2)|2,{50,47,0}},
/* 4875 */ {(11<<2)|2,{50,47,0}},
/* 4876 */ {(11<<2)|2,{50,47,0}},
/* 4877 */ {(11<<2)|2,{50,47,0}},
/* 4878 */ {(11<<2)|2,{50,47,0}},
/* 4879 */ {(11<<2)|2,{50,47,0}},
/* 4880 */ {(11<<2)|2,{50,47,0}},
/* 4881 */ {(11<<2)|2,{50,47,0}},
/* 4882 */ {(11<<2)|2,{50,47,0}},
/* 4883 */ {(11<<2)|2,{50,47,0}},
/* 4884 */ {(11<<2)|2,{50,47,0}},
/* 4885 */ {(11<<2)|2,{50,47,0}},
/* 4886 */ {(11<<2)|2,{50,47,0}},
/* 4887 */ {(11<<2)|2,{50,47,0}},
/* 4888 */ {(11<<2)|2,{50,47,0}},
/* 4889 */ {(11<<2)|2,{50,47,0}},
/* 4890 */ {(11<<2)|2,{50,47,0}},
/* 4891 */ {(11<<2)|2,{50,47,0}},
/* 4892 */ {(11<<2)|2,{50,47,0}},
/* 4893 */ {(11<<2)|2,{50,47,0}},
/* 4894 */ {(11<<2)|2,{50,47,0}},
/* 4895 */ {(11<<2)|2,{50,47,0}},
/* 4896 */ {(16<<2)|3,{50,51,48}},
/* 4897 */ {(16<<2)|3,{50,51,49}},
/* 4898 */ {(16<<2)|3,{50,51,50}},
/* 4899 */ {(16<<2)|3,{50,51,97}},
/* 4900 */ {(16<<2)|3,{50,51,99}},
/* 4901 */ {(16<<2)|3,{50,51,101}},
/* 4902 */ {(16<<2)|3,{50,51,105}},
/* 4903 */ {(16<<2)|3,{50,51,111}},
/* 4904 */ {(16<<2)|3,{50,51,115}},
/* 4905 */ {(16<<2)|3,{50,51,116}},
/* 4906 */ {(11<<2)|2,{50,51,0}},
/* 4907 */ {(11<<2)|2,{50,51,0}},
/* 4908 */ {(11<<2)|2,{50,51,0}},
/* 4909 */ {(11<<2)|2,{50,51,0}},
/* 4910 */ {(11<<2)|2,{50,51,0}},
/* 4911 */ {(11<<2)|2,{50,51,0}},
/* 4912 */ {(11<<2)|2,{50,51,0}},
/* 4913 */ {(11<<2)|2,{50,51,0}},
/* 4914 */ {(11<<2)|2,{50,51,0}},
/* 4915 */ {(11<<2)|2,{50,51,0}},
/* 4916 */ {(11<<2)|2,{50,51,0}},
/* 4917 */ {(11<<2)|2,{50,51,0}},
/* 4918 */ {(11<<2)|2,{50,51,0}},
/* 4919 */ {(11<<2)|2,{50,51,0}},
/* 4920 */ {(11<<2)|2,{50,51,0}},
/* 4921 */ {(11<<2)|2,{50,51,0}},
/* 4922 */ {(11<<2)|2,{50,51,0}},
/* 4923 */ {(11<<2)|2,{50,51,0}},
/* 4924 */ {(11<<2)|2,{50,51,0}},
/* 4925 */ {(11<<2)|2,{50,51,0}},
/* 4926 */ {(11<<2)|2,{50,51,0}},
/* 4927 */ {(11<<2)|2,{50,51,0}},
/* 4928 */ {(16<<2)|3,{50,52,48}},
/* 4929 */ {(16<<2)|3,{50,52,49}},
/* 4930 */ {(16<<2)|3,{50,52,50}},
/* 4931 */ {(16<<2)|3,{50,52,97}},
/* 4932 */ {(16<<2)|3,{50,52,99}},
/* 4933 */ {(16<<2)|3,{50,52,101}},
/* 4934 */ {(16<<2)|3,{50,52,105}},
/* 4935 */ {(16<<2)|3,{50,52,111}},
/* 4936 */ {(16<<2)|3,{50,52,115}},
/* 4937 */ {(16<<2)|3,{50,52,116}},
/* 4938 */ {(11<<2)|2,{50,52,0}},
/* 4939 */ {(11<<2)|2,{50,52,0}},
/* 4940 */ {(11<<2)|2,{50,52,0}},
/* 4941 */ {(11<<2)|2,{50,52,0}},
/* 4942 */ {(11<<2)|2,{50,52,0}},
/* 4943 */ {(11<<2)|2,{50,52,0}},
/* 4944 */ {(11<<2)|2,{50,52,0}},
/* 4945 */ {(11<<2)|2,{50,52,0}},
/* 4946 */ {(11<<2)|2,{50,52,0}},
/* 4947 */ {(11<<2)|2,{50,52,0}},
/* 4948 */ {(11<<2)|2,{50,52,0}},
/* 4949 */ {(11<<2)|2,{50,52,0}},
/* 4950 */ {(11<<2)|2,{50,52,0}},
/* 4951 */ {(11<<2)|2,{50,52,0}},
/* 4952 */ {(11<<2)|2,{50,52,0}},
/* 4953 */ {(11<<2)|2,{50,52,0}},
/* 4954 */ {(11<<2)|2,{50,52,0}},
/* 4955 */ {(11<<2)|2,{50,52,0}},
/* 4956 */ {(11<<2)|2,{50,52,0}},
/* 4957 */ {(11<<2)|2,{50,52,0}},
/* 4958 */ {(11<<2)|2,{50,52,0}},
/* 4959 */ {(11<<2)|2,{50,52,0}},
/* 4960 */ {(16<<2)|3,{50,53,48}},
/* 4961 */ {(16<<2)|3,{50,53,49}},
/* 4962 */ {(16<<2)|3,{50,53,50}},
/* 4963 */ {(16<<2)|3,{50,53,97}},
/* 4964 */ {(16<<2)|3,{50,53,99}},
/* 4965 */ {(16<<2)|3,{50,53,101}},
/* 4966 */ {(16<<2)|3,{50,53,105}},
/* 4967 */ {(16<<2)|3,{50,53,111}},
/* 4968 */ {(16<<2)|3,{50,53,115}},
/* 4969 */ {(16<<2)|3,{50,53,116}},
/* 4970 */ {(11<<2)|2,{50,53,0}},
/* 4971 */ {(11<<2)|2,{50,53,0}},
/* 4972 */ {(11<<2)|2,{50,53,0}},
/* 4973 */ {(11<<2)|2,{50,53,0}},
/* 4974 */ {(11<<2)|2,{50,53,0}},
/* 4975 */ {(11<<2)|2,{50,53,0}},
/* 4976 */ {(11<<2)|2,{50,53,0}},
/* 4977 */ {(11<<2)|2,{50,53,0}},
/* 4978 */ {(11<<2)|2,{50,53,0}},
/* 4979 */ {(11<<2)|2,{50,53,0}},
/* 4980 */ {(11<<2)|2,{50,53,0}},
/* 4981 */ {(11<<2)|2,{50,53,0}},
/* 4982 */ {(11<<2)|2,{50,53,0}},
/* 4983 */ {(11<<2)|2,{50,53,0}},
/* 4984 */ {(11<<2)|2,{50,53,0}},
/* 4985 */ {(11<<2)|2,{50,53,0}},
/* 4986 */ {(11<<2)|2,{50,53,0}},
/* 4987 */ {(11<<2)|2,{50,53,0}},
/* 4988 */ {(11<<2)|2,{50,53,0}},
/* 4989 */ {(11<<2)|2,{50,53,0}},
/* 4990 */ {(11<<2)|2,{50,53,0}},
/* 4991 */ {(11<<2)|2,{50,53,0}},
/* 4992 */ {(16<<2)|3,{50,54,48}},
/* 4993 */ {(16<<2)|3,{50,54,49}},
/* 4994 */ {(16<<2)|3,{50,54,50}},
/* 4995 */ {(16<<2)|3,{50,54,97}},
/* 4996 */ {(16<<2)|3,{50,54,99}},
/* 4997 */ {(16<<2)|3,{50,54,101}},
/* 4998 */ {(16<<2)|3,{50,54,105}},
/* 4999 */ {(16<<2)|3,{50,54,111}},
/* 5000 */ {(16<<2)|3,{50,54,115}},
/* 5001 */ {(16<<2)|3,{50,54,116}},
/* 5002 */ {(11<<2)|2,{50,54,0}},
/* 5003 */ {(11<<2)|2,{50,54,0}},
/* 5004 */ {(11<<2)|2,{50,54,0}},
/* 5005 */ {(11<<2)|2,{50,54,0}},
/* 5006 */ {(11<<2)|2,{50,54,0}},
/* 5007 */ {(11<<2)|2,{50,54,0}},
/* 5008 */ {(11<<2)|2,{50,54,0}},
/* 5009 */ {(11<<2)|2,{50,54,0}},
/* 5010 */ {(11<<2)|2,{50,54,0}},
/* 5011 */ {(11<<2)|2,{50,54,0}},
/* 5012 */ {(11<<2)|2,{50,54,0}},
/* 5013 */ {(11<<2)|2,{50,54,0}},
/* 5014 */ {(11<<2)|2,{50,54,0}},
/* 5015 */ {(11<<2)|2,{50,54,0}},
/* 5016 */ {(11<<2)|2,{50,54,0}},
/* 5017 */ {(11<<2)|2,{50,54,0}},
/* 5018 */ {(11<<2)|2,{50,54,0}},
/* 5019 */ {(11<<2)|2,{50,54,0}},
/* 5020 */ {(11<<2)|2,{50,54,0}},
/* 5021 */ {(11<<2)|2,{50,54,0}},
/* 5022 */ {(11<<2)|2,{50,54,0}},
/* 5023 */ {(11<<2)|2,{50,54,0}},
/* 5024 */ {(16<<2)|3,{50,55,48}},
/* 5025 */ {(16<<2)|3,{50,55,49}},
/* 5026 */ {(16<<2)|3,{50,55,50}},
/* 5027 */ {(16<<2)|3,{50,55,97}},
/* 5028 */ {(16<<2)|3,{50,55,99}},
/* 5029 */ {(16<<2)|3,{50,55,101}},
/* 5030 */ {(16<<2)|3,{50,55,105}},
/* 5031 */ {(16<<2)|3,{50,55,111}},
/* 5032 */ {(16<<2)|3,{50,55,115}},
/* 5033 */ {(16<<2)|3,{50,55,116}},
/* 5034 */ {(11<<2)|2,{50,55,0}},
/* 5035 */ {(11<<2)|2,{50,55,0}},
/* 5036 */ {(11<<2)|2,{50,55,0}},
/* 5037 */ {(11<<2)|2,{50,55,0}},
/* 5038 */ {(11<<2)|2,{50,55,0}},
/* 5039 */ {(11<<2)|2,{50,55,0}},
/* 5040 */ {(11<<2)|2,{50,55,0}},
/* 5041 */ {(11<<2)|2,{50,55,0}},
/* 5042 */ {(11<<2)|2,{50,55,0}},
/* 5043 */ {(11<<2)|2,{50,55,0}},
/* 5044 */ {(11<<2)|2,{50,55,0}},
/* 5045 */ {(11<<2)|2,{50,55,0}},
/* 5046 */ {(11<<2)|2,{50,55,0}},
/* 5047 */ {(11<<2)|2,{50,55,0}},
/* 5048 */ {(11<<2)|2,{50,55,0}},
/* 5049 */ {(11<<2)|2,{50,55,0}},
/* 5050 */ {(11<<2)|2,{50,55,0}},
/* 5051 */ {(11<<2)|2,{50,55,0}},
/* 5052 */ {(11<<2)|2,{50,55,0}},
/* 5053 */ {(11<<2)|2,{50,55,0}},
/* 5054 */ {(11<<2)|2,{50,55,0}},
/* 5055 */ {(11<<2)|2,{50,55,0}},
/* 5056 */ {(16<<2)|3,{50,56,48}},
/* 5057 */ {(16<<2)|3,{50,56,49}},
/* 5058 */ {(16<<2)|3,{50,56,50}},
/* 5059 */ {(16<<2)|3,{50,56,97}},
/* 5060 */ {(16<<2)|3,{50,56,99}},
/* 5061 */ {(16<<2)|3,{50,56,101}},
/* 5062 */ {(16<<2)|3,{50,56,105}},
/* 5063 */ {(16<<2)|3,{50,56,111}},
/* 5064 */ {(16<<2)|3,{50,56,115}},
/* 5065 */ {(16<<2)|3,{50,56,116}},
/* 5066 */ {(11<<2)|2,{50,56,0}},
/* 5067 */ {(11<<2)|2,{50,56,0}},
/* 5068 */ {(11<<2)|2,{50,56,0}},
/* 5069 */ {(11<<2)|2,{50,56,0}},
/* 5070 */ {(11<<2)|2,{50,56,0}},
/* 5071 */ {(11<<2)|2,{50,56,0}},
/* 5072 */ {(11<<2)|2,{50,56,0}},
/* 5073 */ {(11<<2)|2,{50,56,0}},
/* 5074 */ {(11<<2)|2,{50,56,0}},
/* 5075 */ {(11<<2)|2,{50,56,0}},
/* 5076 */ {(11<<2)|2,{50,56,0}},
/* 5077 */ {(11<<2)|2,{50,56,0}},
/* 5078 */ {(11<<2)|2,{50,56,0}},
/* 5079 */ {(11<<2)|2,{50,56,0}},
/* 5080 */ {(11<<2)|2,{50,56,0}},
/* 5081 */ {(11<<2)|2,{50,56,0}},
/* 5082 */ {(11<<2)|2,{50,56,0}},
/* 5083 */ {(11<<2)|2,{50,56,0}},
/* 5084 */ {(11<<2)|2,{50,56,0}},
/* 5085 */ {(11<<2)|2,{50,56,0}},
/* 5086 */ {(11<<2)|2,{50,56,0}},
/* 5087 */ {(11<<2)|2,{50,56,0}},
/* 5088 */ {(16<<2)|3,{50,57,48}},
/* 5089 */ {(16<<2)|3,{50,57,49}},
/* 5090 */ {(16<<2)|3,{50,57,50}},
/* 5091 */ {(16<<2)|3,{50,57,97}},
/* 5092 */ {(16<<2)|3,{50,57,99}},
/* 5093 */ {(16<<2)|3,{50,57,101}},
/* 5094 */ {(16<<2)|3,{50,57,105}},
/* 5095 */ {(16<<2)|3,{50,57,111}},
/* 5096 */ {(16<<2)|3,{50,57,115}},
/* 5097 */ {(16<<2)|3,{50,57,116}},
/* 5098 */ {(11<<2)|2,{50,57,0}},
/* 5099 */ {(11<<2)|2,{50,57,0}},
/* 5100 */ {(11<<2)|2,{50,57,0}},
/* 5101 */ {(11<<2)|2,{50,57,0}},
/* 5102 */ {(11<<2)|2,{50,57,0}},
/* 5103 */ {(11<<2)|2,{50,57,0}},
/* 5104 */ {(11<<2)|2,{50,57,0}},
/* 5105 */ {(11<<2)|2,{50,57,0}},
/* 5106 */ {(11<<2)|2,{50,57,0}},
/* 5107 */ {(11<<2)|2,{50,57,0}},
/* 5108 */ {(11<<2)|2,{50,57,0}},
/* 5109 */ {(11<<2)|2,{50,57,0}},
/* 5110 */ {(11<<2)|2,{50,57,0}},
/* 5111 */ {(11<<2)|2,{50,57,0}},
/* 5112 */ {(11<<2)|2,{50,57,0}},
/* 5113 */ {(11<<2)|2,{50,57,0}},
/* 5114 */ {(11<<2)|2,{50,57,0}},
/* 5115 */ {(11<<2)|2,{50,57,0}},
/* 5116 */ {(11<<2)|2,{50,57,0}},
/* 5117 */ {(11<<2)|2,{50,57,0}},
/* 5118 */ {(11<<2)|2,{50,57,0}},
/* 5119 */ {(11<<2)|2,{50,57,0}},
/* 5120 */ {(16<<2)|3,{50,61,48}},
/* 5121 */ {(16<<2)|3,{50,61,49}},
/* 5122 */ {(16<<2)|3,{50,61,50}},
/* 5123 */ {(16<<2)|3,{50,61,97}},
/* 5124 */ {(16<<2)|3,{50,61,99}},
/* 5125 */ {(16<<2)|3,{50,61,101}},
/* 5126 */ {(16<<2)|3,{50,61,105}},
/* 5127 */ {(16<<2)|3,{50,61,111}},
/* 5128 */ {(16<<2)|3,{50,61,115}},
/* 5129 */ {(16<<2)|3,{50,61,116}},
/* 5130 */ {(11<<2)|2,{50,61,0}},
/* 5131 */ {(11<<2)|2,{50,61,0}},
/* 5132 */ {(11<<2)|2,{50,61,0}},
/* 5133 */ {(11<<2)|2,{50,61,0}},
/* 5134 */ {(11<<2)|2,{50,61,0}},
/* 5135 */ {(11<<2)|2,{50,61,0}},
/* 5136 */ {(11<<2)|2,{50,61,0}},
/* 5137 */ {(11<<2)|2,{50,61,0}},
/* 5138 */ {(11<<2)|2,{50,61,0}},
/* 5139 */ {(11<<2)|2,{50,61,0}},
/* 5140 */ {(11<<2)|2,{50,61,0}},
/* 5141 */ {(11<<2)|2,{50,61,0}},
/* 5142 */ {(11<<2)|2,{50,61,0}},
/* 5143 */ {(11<<2)|2,{50,61,0}},
/* 5144 */ {(11<<2)|2,{50,61,0}},
/* 5145 */ {(11<<2)|2,{50,61,0}},
/* 5146 */ {(11<<2)|2,{50,61,0}},
/* 5147 */ {(11<<2)|2,{50,61,0}},
/* 5148 */ {(11<<2)|2,{50,61,0}},
/* 5149 */ {(11<<2)|2,{50,61,0}},
/* 5150 */ {(11<<2)|2,{50,61,0}},
/* 5151 */ {(11<<2)|2,{50,61,0}},
/* 5152 */ {(16<<2)|3,{50,65,48}},
/* 5153 */ {(16<<2)|3,{50,65,49}},
/* 5154 */ {(16<<2)|3,{50,65,50}},
/* 5155 */ {(16<<2)|3,{50,65,97}},
/* 5156 */ {(16<<2)|3,{50,65,99}},
/* 5157 */ {(16<<2)|3,{50,65,101}},
/* 5158 */ {(16<<2)|3,{50,65,105}},
/* 5159 */ {(16<<2)|3,{50,65,111}},
/* 5160 */ {(16<<2)|3,{50,65,115}},
/* 5161 */ {(16<<2)|3,{50,65,116}},
/* 5162 */ {(11<<2)|2,{50,65,0}},
/* 5163 */ {(11<<2)|2,{50,65,0}},
/* 5164 */ {(11<<2)|2,{50,65,0}},
/* 5165 */ {(11<<2)|2,{50,65,0}},
/* 5166 */ {(11<<2)|2,{50,65,0}},
/* 5167 */ {(11<<2)|2,{50,65,0}},
/* 5168 */ {(11<<2)|2,{50,65,0}},
/* 5169 */ {(11<<2)|2,{50,65,0}},
/* 5170 */ {(11<<2)|2,{50,65,0}},
/* 5171 */ {(11<<2)|2,{50,65,0}},
/* 5172 */ {(11<<2)|2,{50,65,0}},
/* 5173 */ {(11<<2)|2,{50,65,0}},
/* 5174 */ {(11<<2)|2,{50,65,0}},
/* 5175 */ {(11<<2)|2,{50,65,0}},
/* 5176 */ {(11<<2)|2,{50,65,0}},
/* 5177 */ {(11<<2)|2,{50,65,0}},
/* 5178 */ {(11<<2)|2,{50,65,0}},
/* 5179 */ {(11<<2)|2,{50,65,0}},
/* 5180 */ {(11<<2)|2,{50,65,0}},
/* 5181 */ {(11<<2)|2,{50,65,0}},
/* 5182 */ {(11<<2)|2,{50,65,0}},
/* 5183 */ {(11<<2)|2,{50,65,0}},
/* 5184 */ {(16<<2)|3,{50,95,48}},
/* 5185 */ {(16<<2)|3,{50,95,49}},
/* 5186 */ {(16<<2)|3,{50,95,50}},
/* 5187 */ {(16<<2)|3,{50,95,97}},
/* 5188 */ {(16<<2)|3,{50,95,99}},
/* 5189 */ {(16<<2)|3,{50,95,101}},
/* 5190 */ {(16<<2)|3,{50,95,105}},
/* 5191 */ {(16<<2)|3,{50,95,111}},
/* 5192 */ {(16<<2)|3,{50,95,115}},
/* 5193 */ {(16<<2)|3,{50,95,116}},
/* 5194 */ {(11<<2)|2,{50,95,0}},
/* 5195 */ {(11<<2)|2,{50,95,0}},
/* 5196 */ {(11<<2)|2,{50,95,0}},
/* 5197 */ {(11<<2)|2,{50,95,0}},
/* 5198 */ {(11<<2)|2,{50,95,0}},
/* 5199 */ {(11<<2)|2,{50,95,0}},
/* 5200 */ {(11<<2)|2,{50,95,0}},
/* 5201 */ {(11<<2)|2,{50,95,0}},
/* 5202 */ {(11<<2)|2,{50,95,0}},
/* 5203 */ {(11<<2)|2,{50,95,0}},
/* 5204 */ {(11<<2)|2,{50,95,0}},
/* 5205 */ {(11<<2)|2,{50,95,0}},
/* 5206 */ {(11<<2)|2,{50,95,0}},
/* 5207 */ {(11<<2)|2,{50,95,0}},
/* 5208 */ {(11<<2)|2,{50,95,0}},
/* 5209 */ {(11<<2)|2,{50,95,0}},
/* 5210 */ {(11<<2)|2,{50,95,0}},
/* 5211 */ {(11<<2)|2,{50,95,0}},
/* 5212 */ {(11<<2)|2,{50,95,0}},
/* 5213 */ {(11<<2)|2,{50,95,0}},
/* 5214 */ {(11<<2)|2,{50,95,0}},
/* 5215 */ {(11<<2)|2,{50,95,0}},
/* 5216 */ {(16<<2)|3,{50,98,48}},
/* 5217 */ {(16<<2)|3,{50,98,49}},
/* 5218 */ {(16<<2)|3,{50,98,50}},
/* 5219 */ {(16<<2)|3,{50,98,97}},
/* 5220 */ {(16<<2)|3,{50,98,99}},
/* 5221 */ {(16<<2)|3,{50,98,101}},
/* 5222 */ {(16<<2)|3,{50,98,105}},
/* 5223 */ {(16<<2)|3,{50,98,111}},
/* 5224 */ {(16<<2)|3,{50,98,115}},
/* 5225 */ {(16<<2)|3,{50,98,116}},
/* 5226 */ {(11<<2)|2,{50,98,0}},
/* 5227 */ {(11<<2)|2,{50,98,0}},
/* 5228 */ {(11<<2)|2,{50,98,0}},
/* 5229 */ {(11<<2)|2,{50,98,0}},
/* 5230 */ {(11<<2)|2,{50,98,0}},
/* 5231 */ {(11<<2)|2,{50,98,0}},
/* 5232 */ {(11<<2)|2,{50,98,0}},
/* 5233 */ {(11<<2)|2,{50,98,0}},
/* 5234 */ {(11<<2)|2,{50,98,0}},
/* 5235 */ {(11<<2)|2,{50,98,0}},
/* 5236 */ {(11<<2)|2,{50,98,0}},
/* 5237 */ {(11<<2)|2,{50,98,0}},
/* 5238 */ {(11<<2)|2,{50,98,0}},
/* 5239 */ {(11<<2)|2,{50,98,0}},
/* 5240 */ {(11<<2)|2,{50,98,0}},
/* 5241 */ {(11<<2)|2,{50,98,0}},
/* 5242 */ {(11<<2)|2,{50,98,0}},
/* 5243 */ {(11<<2)|2,{50,98,0}},
/* 5244 */ {(11<<2)|2,{50,98,0}},
/* 5245 */ {(11<<2)|2,{50,98,0}},
/* 5246 */ {(11<<2)|2,{50,98,0}},
/* 5247 */ {(11<<2)|2,{50,98,0}},
/* 5248 */ {(16<<2)|3,{50,100,48}},
/* 5249 */ {(16<<2)|3,{50,100,49}},
/* 5250 */ {(16<<2)|3,{50,100,50}},
/* 5251 */ {(16<<2)|3,{50,100,97}},
/* 5252 */ {(16<<2)|3,{50,100,99}},
/* 5253 */ {(16<<2)|3,{50,100,101}},
/* 5254 */ {(16<<2)|3,{50,100,105}},
/* 5255 */ {(16<<2)|3,{50,100,111}},
/* 5256 */ {(16<<2)|3,{50,100,115}},
/* 5257 */ {(16<<2)|3,{50,100,116}},
/* 5258 */ {(11<<2)|2,{50,100,0}},
/* 5259 */ {(11<<2)|2,{50,100,0}},
/* 5260 */ {(11<<2)|2,{50,100,0}},
/* 5261 */ {(11<<2)|2,{50,100,0}},
/* 5262 */ {(11<<2)|2,{50,100,0}},
/* 5263 */ {(11<<2)|2,{50,100,0}},
/* 5264 */ {(11<<2)|2,{50,100,0}},
/* 5265 */ {(11<<2)|2,{50,100,0}},
/* 5266 */ {(11<<2)|2,{50,100,0}},
/* 5267 */ {(11<<2)|2,{50,100,0}},
/* 5268 */ {(11<<2)|2,{50,100,0}},
/* 5269 */ {(11<<2)|2,{50,100,0}},
/* 5270 */ {(11<<2)|2,{50,100,0}},
/* 5271 */ {(11<<2)|2,{50,100,0}},
/* 5272 */ {(11<<2)|2,{50,100,0}},
/* 5273 */ {(11<<2)|2,{50,100,0}},
/* 5274 */ {(11<<2)|2,{50,100,0}},
/* 5275 */ {(11<<2)|2,{50,100,0}},
/* 5276 */ {(11<<2)|2,{50,100,0}},
/* 5277 */ {(11<<2)|2,{50,100,0}},
/* 5278 */ {(11<<2)|2,{50,100,0}},
/* 5279 */ {(11<<2)|2,{50,100,0}},
/* 5280 */ {(16<<2)|3,{50,102,48}},
/* 5281 */ {(16<<2)|3,{50,102,49}},
/* 5282 */ {(16<<2)|3,{50,102,50}},
/* 5283 */ {(16<<2)|3,{50,102,97}},
/* 5284 */ {(16<<2)|3,{50,102,99}},
/* 5285 */ {(16<<2)|3,{50,102,101}},
/* 5286 */ {(16<<2)|3,{50,102,105}},
/* 5287 */ {(16<<2)|3,{50,102,111}},
/* 5288 */ {(16<<2)|3,{50,102,115}},
/* 5289 */ {(16<<2)|3,{50,102,116}},
/* 5290 */ {(11<<2)|2,{50,102,0}},
/* 5291 */ {(11<<2)|2,{50,102,0}},
/* 5292 */ {(11<<2)|2,{50,102,0}},
/* 5293 */ {(11<<2)|2,{50,102,0}},
/* 5294 */ {(11<<2)|2,{50,102,0}},
/* 5295 */ {(11<<2)|2,{50,102,0}},
/* 5296 */ {(11<<2)|2,{50,102,0}},
/* 5297 */ {(11<<2)|2,{50,102,0}},
/* 5298 */ {(11<<2)|2,{50,102,0}},
/* 5299 */ {(11<<2)|2,{50,102,0}},
/* 5300 */ {(11<<2)|2,{50,102,0}},
/* 5301 */ {(11<<2)|2,{50,102,0}},
/* 5302 */ {(11<<2)|2,{50,102,0}},
/* 5303 */ {(11<<2)|2,{50,102,0}},
/* 5304 */ {(11<<2)|2,{50,102,0}},
/* 5305 */ {(11<<2)|2,{50,102,0}},
/* 5306 */ {(11<<2)|2,{50,102,0}},
/* 5307 */ {(11<<2)|2,{50,102,0}},
/* 5308 */ {(11<<2)|2,{50,102,0}},
/* 5309 */ {(11<<2)|2,{50,102,0}},
/* 5310 */ {(11<<2)|2,{50,102,0}},
/* 5311 */ {(11<<2)|2,{50,102,0}},
/* 5312 */ {(16<<2)|3,{50,103,48}},
/* 5313 */ {(16<<2)|3,{50,103,49}},
/* 5314 */ {(16<<2)|3,{50,103,50}},
/* 5315 */ {(16<<2)|3,{50,103,97}},
/* 5316 */ {(16<<2)|3,{50,103,99}},
/* 5317 */ {(16<<2)|3,{50,103,101}},
/* 5318 */ {(16<<2)|3,{50,103,105}},
/* 5319 */ {(16<<2)|3,{50,103,111}},
/* 5320 */ {(16<<2)|3,{50,103,115}},
/* 5321 */ {(16<<2)|3,{50,103,116}},
/* 5322 */ {(11<<2)|2,{50,103,0}},
/* 5323 */ {(11<<2)|2,{50,103,0}},
/* 5324 */ {(11<<2)|2,{50,103,0}},
/* 5325 */ {(11<<2)|2,{50,103,0}},
/* 5326 */ {(11<<2)|2,{50,103,0}},
/* 5327 */ {(11<<2)|2,{50,103,0}},
/* 5328 */ {(11<<2)|2,{50,103,0}},
/* 5329 */ {(11<<2)|2,{50,103,0}},
/* 5330 */ {(11<<2)|2,{50,103,0}},
/* 5331 */ {(11<<2)|2,{50,103,0}},
/* 5332 */ {(11<<2)|2,{50,103,0}},
/* 5333 */ {(11<<2)|2,{50,103,0}},
/* 5334 */ {(11<<2)|2,{50,103,0}},
/* 5335 */ {(11<<2)|2,{50,103,0}},
/* 5336 */ {(11<<2)|2,{50,103,0}},
/* 5337 */ {(11<<2)|2,{50,103,0}},
/* 5338 */ {(11<<2)|2,{50,103,0}},
/* 5339 */ {(11<<2)|2,{50,103,0}},
/* 5340 */ {(11<<2)|2,{50,103,0}},
/* 5341 */ {(11<<2)|2,{50,103,0}},
/* 5342 */ {(11<<2)|2,{50,103,0}},
/* 5343 */ {(11<<2)|2,{50,103,0}},
/* 5344 */ {(16<<2)|3,{50,104,48}},
/* 5345 */ {(16<<2)|3,{50,104,49}},
/* 5346 */ {(16<<2)|3,{50,104,50}},
/* 5347 */ {(16<<2)|3,{50,104,97}},
/* 5348 */ {(16<<2)|3,{50,104,99}},
/* 5349 */ {(16<<2)|3,{50,104,101}},
/* 5350 */ {(16<<2)|3,{50,104,105}},
/* 5351 */ {(16<<2)|3,{50,104,111}},
/* 5352 */ {(16<<2)|3,{50,104,115}},
/* 5353 */ {(16<<2)|3,{50,104,116}},
/* 5354 */ {(11<<2)|2,{50,104,0}},
/* 5355 */ {(11<<2)|2,{50,104,0}},
/* 5356 */ {(11<<2)|2,{50,104,0}},
/* 5357 */ {(11<<2)|2,{50,104,0}},
/* 5358 */ {(11<<2)|2,{50,104,0}},
/* 5359 */ {(11<<2)|2,{50,104,0}},
/* 5360 */ {(11<<2)|2,{50,104,0}},
/* 5361 */ {(11<<2)|2,{50,104,0}},
/* 5362 */ {(11<<2)|2,{50,104,0}},
/* 5363 */ {(11<<2)|2,{50,104,0}},
/* 5364 */ {(11<<2)|2,{50,104,0}},
/* 5365 */ {(11<<2)|2,{50,104,0}},
/* 5366 */ {(11<<2)|2,{50,104,0}},
/* 5367 */ {(11<<2)|2,{50,104,0}},
/* 5368 */ {(11<<2)|2,{50,104,0}},
/* 5369 */ {(11<<2)|2,{50,104,0}},
/* 5370 */ {(11<<2)|2,{50,104,0}},
/* 5371 */ {(11<<2)|2,{50,104,0}},
/* 5372 */ {(11<<2)|2,{50,104,0}},
/* 5373 */ {(11<<2)|2,{50,104,0}},
/* 5374 */ {(11<<2)|2,{50,104,0}},
/* 5375 */ {(11<<2)|2,{50,104,0}},
/* 5376 */ {(16<<2)|3,{50,108,48}},
/* 5377 */ {(16<<2)|3,{50,108,49}},
/* 5378 */ {(16<<2)|3,{50,108,50}},
/* 5379 */ {(16<<2)|3,{50,108,97}},
/* 5380 */ {(16<<2)|3,{50,108,99}},
/* 5381 */ {(16<<2)|3,{50,108,101}},
/* 5382 */ {(16<<2)|3,{50,108,105}},
/* 5383 */ {(16<<2)|3,{50,108,111}},
/* 5384 */ {(16<<2)|3,{50,108,115}},
/* 5385 */ {(16<<2)|3,{50,108,116}},
/* 5386 */ {(11<<2)|2,{50,108,0}},
/* 5387 */ {(11<<2)|2,{50,108,0}},
/* 5388 */ {(11<<2)|2,{50,108,0}},
/* 5389 */ {(11<<2)|2,{50,108,0}},
/* 5390 */ {(11<<2)|2,{50,108,0}},
/* 5391 */ {(11<<2)|2,{50,108,0}},
/* 5392 */ {(11<<2)|2,{50,108,0}},
/* 5393 */ {(11<<2)|2,{50,108,0}},
/* 5394 */ {(11<<2)|2,{50,108,0}},
/* 5395 */ {(11<<2)|2,{50,108,0}},
/* 5396 */ {(11<<2)|2,{50,108,0}},
/* 5397 */ {(11<<2)|2,{50,108,0}},
/* 5398 */ {(11<<2)|2,{50,108,0}},
/* 5399 */ {(11<<2)|2,{50,108,0}},
/* 5400 */ {(11<<2)|2,{50,108,0}},
/* 5401 */ {(11<<2)|2,{50,108,0}},
/* 5402 */ {(11<<2)|2,{50,108,0}},
/* 5403 */ {(11<<2)|2,{50,108,0}},
/* 5404 */ {(11<<2)|2,{50,108,0}},
/* 5405 */ {(11<<2)|2,{50,108,0}},
/* 5406 */ {(11<<2)|2,{50,108,0}},
/* 5407 */ {(11<<2)|2,{50,108,0}},
/* 5408 */ {(16<<2)|3,{50,109,48}},
/* 5409 */ {(16<<2)|3,{50,109,49}},
/* 5410 */ {(16<<2)|3,{50,109,50}},
/* 5411 */ {(16<<2)|3,{50,109,97}},
/* 5412 */ {(16<<2)|3,{50,109,99}},
/* 5413 */ {(16<<2)|3,{50,109,101}},
/* 5414 */ {(16<<2)|3,{50,109,105}},
/* 5415 */ {(16<<2)|3,{50,109,111}},
/* 5416 */ {(16<<2)|3,{50,109,115}},
/* 5417 */ {(16<<2)|3,{50,109,116}},
/* 5418 */ {(11<<2)|2,{50,109,0}},
/* 5419 */ {(11<<2)|2,{50,109,0}},
/* 5420 */ {(11<<2)|2,{50,109,0}},
/* 5421 */ {(11<<2)|2,{50,109,0}},
/* 5422 */ {(11<<2)|2,{50,109,0}},
/* 5423 */ {(11<<2)|2,{50,109,0}},
/* 5424 */ {(11<<2)|2,{50,109,0}},
/* 5425 */ {(11<<2)|2,{50,109,0}},
/* 5426 */ {(11<<2)|2,{50,109,0}},
/* 5427 */ {(11<<2)|2,{50,109,0}},
/* 5428 */ {(11<<2)|2,{50,109,0}},
/* 5429 */ {(11<<2)|2,{50,109,0}},
/* 5430 */ {(11<<2)|2,{50,109,0}},
/* 5431 */ {(11<<2)|2,{50,109,0}},
/* 5432 */ {(11<<2)|2,{50,109,0}},
/* 5433 */ {(11<<2)|2,{50,109,0}},
/* 5434 */ {(11<<2)|2,{50,109,0}},
/* 5435 */ {(11<<2)|2,{50,109,0}},
/* 5436 */ {(11<<2)|2,{50,109,0}},
/* 5437 */ {(11<<2)|2,{50,109,0}},
/* 5438 */ {(11<<2)|2,{50,109,0}},
/* 5439 */ {(11<<2)|2,{50,109,0}},
/* 5440 */ {(16<<2)|3,{50,110,48}},
/* 5441 */ {(16<<2)|3,{50,110,49}},
/* 5442 */ {(16<<2)|3,{50,110,50}},
/* 5443 */ {(16<<2)|3,{50,110,97}},
/* 5444 */ {(16<<2)|3,{50,110,99}},
/* 5445 */ {(16<<2)|3,{50,110,101}},
/* 5446 */ {(16<<2)|3,{50,110,105}},
/* 5447 */ {(16<<2)|3,{50,110,111}},
/* 5448 */ {(16<<2)|3,{50,110,115}},
/* 5449 */ {(16<<2)|3,{50,110,116}},
/* 5450 */ {(11<<2)|2,{50,110,0}},
/* 5451 */ {(11<<2)|2,{50,110,0}},
/* 5452 */ {(11<<2)|2,{50,110,0}},
/* 5453 */ {(11<<2)|2,{50,110,0}},
/* 5454 */ {(11<<2)|2,{50,110,0}},
/* 5455 */ {(11<<2)|2,{50,110,0}},
/* 5456 */ {(11<<2)|2,{50,110,0}},
/* 5457 */ {(11<<2)|2,{50,110,0}},
/* 5458 */ {(11<<2)|2,{50,110,0}},
/* 5459 */ {(11<<2)|2,{50,110,0}},
/* 5460 */ {(11<<2)|2,{50,110,0}},
/* 5461 */ {(11<<2)|2,{50,110,0}},
/* 5462 */ {(11<<2)|2,{50,110,0}},
/* 5463 */ {(11<<2)|2,{50,110,0}},
/* 5464 */ {(11<<2)|2,{50,110,0}},
/* 5465 */ {(11<<2)|2,{50,110,0}},
/* 5466 */ {(11<<2)|2,{50,110,0}},
/* 5467 */ {(11<<2)|2,{50,110,0}},
/* 5468 */ {(11<<2)|2,{50,110,0}},
/* 5469 */ {(11<<2)|2,{50,110,0}},
/* 5470 */ {(11<<2)|2,{50,110,0}},
/* 5471 */ {(11<<2)|2,{50,110,0}},
/* 5472 */ {(16<<2)|3,{50,112,48}},
/* 5473 */ {(16<<2)|3,{50,112,49}},
/* 5474 */ {(16<<2)|3,{50,112,50}},
/* 5475 */ {(16<<2)|3,{50,112,97}},
/* 5476 */ {(16<<2)|3,{50,112,99}},
/* 5477 */ {(16<<2)|3,{50,112,101}},
/* 5478 */ {(16<<2)|3,{50,112,105}},
/* 5479 */ {(16<<2)|3,{50,112,111}},
/* 5480 */ {(16<<2)|3,{50,112,115}},
/* 5481 */ {(16<<2)|3,{50,112,116}},
/* 5482 */ {(11<<2)|2,{50,112,0}},
/* 5483 */ {(11<<2)|2,{50,112,0}},
/* 5484 */ {(11<<2)|2,{50,112,0}},
/* 5485 */ {(11<<2)|2,{50,112,0}},
/* 5486 */ {(11<<2)|2,{50,112,0}},
/* 5487 */ {(11<<2)|2,{50,112,0}},
/* 5488 */ {(11<<2)|2,{50,112,0}},
/* 5489 */ {(11<<2)|2,{50,112,0}},
/* 5490 */ {(11<<2)|2,{50,112,0}},
/* 5491 */ {(11<<2)|2,{50,112,0}},
/* 5492 */ {(11<<2)|2,{50,112,0}},
/* 5493 */ {(11<<2)|2,{50,112,0}},
/* 5494 */ {(11<<2)|2,{50,112,0}},
/* 5495 */ {(11<<2)|2,{50,112,0}},
/* 5496 */ {(11<<2)|2,{50,112,0}},
/* 5497 */ {(11<<2)|2,{50,112,0}},
/* 5498 */ {(11<<2)|2,{50,112,0}},
/* 5499 */ {(11<<2)|2,{50,112,0}},
/* 5500 */ {(11<<2)|2,{50,112,0}},
/* 5501 */ {(11<<2)|2,{50,112,0}},
/* 5502 */ {(11<<2)|2,{50,112,0}},
/* 5503 */ {(11<<2)|2,{50,112,0}},
/* 5504 */ {(16<<2)|3,{50,114,48}},
/* 5505 */ {(16<<2)|3,{50,114,49}},
/* 5506 */ {(16<<2)|3,{50,114,50}},
/* 5507 */ {(16<<2)|3,{50,114,97}},
/* 5508 */ {(16<<2)|3,{50,114,99}},
/* 5509 */ {(16<<2)|3,{50,114,101}},
/* 5510 */ {(16<<2)|3,{50,114,105}},
/* 5511 */ {(16<<2)|3,{50,114,111}},
/* 5512 */ {(16<<2)|3,{50,114,115}},
/* 5513 */ {(16<<2)|3,{50,114,116}},
/* 5514 */ {(11<<2)|2,{50,114,0}},
/* 5515 */ {(11<<2)|2,{50,114,0}},
/* 5516 */ {(11<<2)|2,{50,114,0}},
/* 5517 */ {(11<<2)|2,{50,114,0}},
/* 5518 */ {(11<<2)|2,{50,114,0}},
/* 5519 */ {(11<<2)|2,{50,114,0}},
/* 5520 */ {(11<<2)|2,{50,114,0}},
/* 5521 */ {(11<<2)|2,{50,114,0}},
/* 5522 */ {(11<<2)|2,{50,114,0}},
/* 5523 */ {(11<<2)|2,{50,114,0}},
/* 5524 */ {(11<<2)|2,{50,114,0}},
/* 5525 */ {(11<<2)|2,{50,114,0}},
/* 5526 */ {(11<<2)|2,{50,114,0}},
/* 5527 */ {(11<<2)|2,{50,114,0}},
/* 5528 */ {(11<<2)|2,{50,114,0}},
/* 5529 */ {(11<<2)|2,{50,114,0}},
/* 5530 */ {(11<<2)|2,{50,114,0}},
/* 5531 */ {(11<<2)|2,{50,114,0}},
/* 5532 */ {(11<<2)|2,{50,114,0}},
/* 5533 */ {(11<<2)|2,{50,114,0}},
/* 5534 */ {(11<<2)|2,{50,114,0}},
/* 5535 */ {(11<<2)|2,{50,114,0}},
/* 5536 */ {(16<<2)|3,{50,117,48}},
/* 5537 */ {(16<<2)|3,{50,117,49}},
/* 5538 */ {(16<<2)|3,{50,117,50}},
/* 5539 */ {(16<<2)|3,{50,117,97}},
/* 5540 */ {(16<<2)|3,{50,117,99}},
/* 5541 */ {(16<<2)|3,{50,117,101}},
/* 5542 */ {(16<<2)|3,{50,117,105}},
/* 5543 */ {(16<<2)|3,{50,117,111}},
/* 5544 */ {(16<<2)|3,{50,117,115}},
/* 5545 */ {(16<<2)|3,{50,117,116}},
/* 5546 */ {(11<<2)|2,{50,117,0}},
/* 5547 */ {(11<<2)|2,{50,117,0}},
/* 5548 */ {(11<<2)|2,{50,117,0}},
/* 5549 */ {(11<<2)|2,{50,117,0}},
/* 5550 */ {(11<<2)|2,{50,117,0}},
/* 5551 */ {(11<<2)|2,{50,117,0}},
/* 5552 */ {(11<<2)|2,{50,117,0}},
/* 5553 */ {(11<<2)|2,{50,117,0}},
/* 5554 */ {(11<<2)|2,{50,117,0}},
/* 5555 */ {(11<<2)|2,{50,117,0}},
/* 5556 */ {(11<<2)|2,{50,117,0}},
/* 5557 */ {(11<<2)|2,{50,117,0}},
/* 5558 */ {(11<<2)|2,{50,117,0}},
/* 5559 */ {(11<<2)|2,{50,117,0}},
/* 5560 */ {(11<<2)|2,{50,117,0}},
/* 5561 */ {(11<<2)|2,{50,117,0}},
/* 5562 */ {(11<<2)|2,{50,117,0}},
/* 5563 */ {(11<<2)|2,{50,117,0}},
/* 5564 */ {(11<<2)|2,{50,117,0}},
/* 5565 */ {(11<<2)|2,{50,117,0}},
/* 5566 */ {(11<<2)|2,{50,117,0}},
/* 5567 */ {(11<<2)|2,{50,117,0}},
/* 5568 */ {(12<<2)|2,{50,58,0}},
/* 5569 */ {(12<<2)|2,{50,58,0}},
/* 5570 */ {(12<<2)|2,{50,58,0}},
/* 5571 */ {(12<<2)|2,{50,58,0}},
/* 5572 */ {(12<<2)|2,{50,58,0}},
/* 5573 */ {(12<<2)|2,{50,58,0}},
/* 5574 */ {(12<<2)|2,{50,58,0}},
/* 5575 */ {(12<<2)|2,{50,58,0}},
/* 5576 */ {(12<<2)|2,{50,58,0}},
/* 5577 */ {(12<<2)|2,{50,58,0}},
/* 5578 */ {(12<<2)|2,{50,58,0}},
/* 5579 */ {(12<<2)|2,{50,58,0}},
/* 5580 */ {(12<<2)|2,{50,58,0}},
/* 5581 */ {(12<<2)|2,{50,58,0}},
/* 5582 */ {(12<<2)|2,{50,58,0}},
/* 5583 */ {(12<<2)|2,{50,58,0}},
/* 5584 */ {(12<<2)|2,{50,66,0}},
/* 5585 */ {(12<<2)|2,{50,66,0}},
/* 5586 */ {(12<<2)|2,{50,66,0}},
/* 5587 */ {(12<<2)|2,{50,66,0}},
/* 5588 */ {(12<<2)|2,{50,66,0}},
/* 5589 */ {(12<<2)|2,{50,66,0}},
/* 5590 */ {(12<<2)|2,{50,66,0}},
/* 5591 */ {(12<<2)|2,{50,66,0}},
/* 5592 */ {(12<<2)|2,{50,66,0}},
/* 5593 */ {(12<<2)|2,{50,66,0}},
/* 5594 */ {(12<<2)|2,{50,66,0}},
/* 5595 */ {(12<<2)|2,{50,66,0}},
/* 5596 */ {(12<<2)|2,{50,66,0}},
/* 5597 */ {(12<<2)|2,{50,66,0}},
/* 5598 */ {(12<<2)|2,{50,66,0}},
/* 5599 */ {(12<<2)|2,{50,66,0}},
/* 5600 */ {(12<<2)|2,{50,67,0}},
/* 5601 */ {(12<<2)|2,{50,67,0}},
/* 5602 */ {(12<<2)|2,{50,67,0}},
/* 5603 */ {(12<<2)|2,{50,67,0}},
/* 5604 */ {(12<<2)|2,{50,67,0}},
/* 5605 */ {(12<<2)|2,{50,67,0}},
/* 5606 */ {(12<<2)|2,{50,67,0}},
/* 5607 */ {(12<<2)|2,{50,67,0}},
/* 5608 */ {(12<<2)|2,{50,67,0}},
/* 5609 */ {(12<<2)|2,{50,67,0}},
/* 5610 */ {(12<<2)|2,{50,67,0}},
/* 5611 */ {(12<<2)|2,{50,67,0}},
/* 5612 */ {(12<<2)|2,{50,67,0}},
/* 5613 */ {(12<<2)|2,{50,67,0}},
/* 5614 */ {(12<<2)|2,{50,67,0}},
/* 5615 */ {(12<<2)|2,{50,67,0}},
/* 5616 */ {(12<<2)|2,{50,68,0}},
/* 5617 */ {(12<<2)|2,{50,68,0}},
/* 5618 */ {(12<<2)|2,{50,68,0}},
/* 5619 */ {(12<<2)|2,{50,68,0}},
/* 5620 */ {(12<<2)|2,{50,68,0}},
/* 5621 */ {(12<<2)|2,{50,68,0}},
/* 5622 */ {(12<<2)|2,{50,68,0}},
/* 5623 */ {(12<<2)|2,{50,68,0}},
/* 5624 */ {(12<<2)|2,{50,68,0}},
/* 5625 */ {(12<<2)|2,{50,68,0}},
/* 5626 */ {(12<<2)|2,{50,68,0}},
/* 5627 */ {(12<<2)|2,{50,68,0}},
/* 5628 */ {(12<<2)|2,{50,68,0}},
/* 5629 */ {(12<<2)|2,{50,68,0}},
/* 5630 */ {(12<<2)|2,{50,68,0}},
/* 5631 */ {(12<<2)|2,{50,68,0}},
/* 5632 */ {(12<<2)|2,{50,69,0}},
/* 5633 */ {(12<<2)|2,{50,69,0}},
/* 5634 */ {(12<<2)|2,{50,69,0}},
/* 5635 */ {(12<<2)|2,{50,69,0}},
/* 5636 */ {(12<<2)|2,{50,69,0}},
/* 5637 */ {(12<<2)|2,{50,69,0}},
/* 5638 */ {(12<<2)|2,{50,69,0}},
/* 5639 */ {(12<<2)|2,{50,69,0}},
/* 5640 */ {(12<<2)|2,{50,69,0}},
/* 5641 */ {(12<<2)|2,{50,69,0}},
/* 5642 */ {(12<<2)|2,{50,69,0}},
/* 5643 */ {(12<<2)|2,{50,69,0}},
/* 5644 */ {(12<<2)|2,{50,69,0}},
/* 5645 */ {(12<<2)|2,{50,69,0}},
/* 5646 */ {(12<<2)|2,{50,69,0}},
/* 5647 */ {(12<<2)|2,{50,69,0}},
/* 5648 */ {(12<<2)|2,{50,70,0}},
/* 5649 */ {(12<<2)|2,{50,70,0}},
/* 5650 */ {(12<<2)|2,{50,70,0}},
/* 5651 */ {(12<<2)|2,{50,70,0}},
/* 5652 */ {(12<<2)|2,{50,70,0}},
/* 5653 */ {(12<<2)|2,{50,70,0}},
/* 5654 */ {(12<<2)|2,{50,70,0}},
/* 5655 */ {(12<<2)|2,{50,70,0}},
/* 5656 */ {(12<<2)|2,{50,70,0}},
/* 5657 */ {(12<<2)|2,{50,70,0}},
/* 5658 */ {(12<<2)|2,{50,70,0}},
/* 5659 */ {(12<<2)|2,{50,70,0}},
/* 5660 */ {(12<<2)|2,{50,70,0}},
/* 5661 */ {(12<<2)|2,{50,70,0}},
/* 5662 */ {(12<<2)|2,{50,70,0}},
/* 5663 */ {(12<<2)|2,{50,70,0}},
/* 5664 */ {(12<<2)|2,{50,71,0}},
/* 5665 */ {(12<<2)|2,{50,71,0}},
/* 5666 */ {(12<<2)|2,{50,71,0}},
/* 5667 */ {(12<<2)|2,{50,71,0}},
/* 5668 */ {(12<<2)|2,{50,71,0}},
/* 5669 */ {(12<<2)|2,{50,71,0}},
/* 5670 */ {(12<<2)|2,{50,71,0}},
/* 5671 */ {(12<<2)|2,{50,71,0}},
/* 5672 */ {(12<<2)|2,{50,71,0}},
/* 5673 */ {(12<<2)|2,{50,71,0}},
/* 5674 */ {(12<<2)|2,{50,71,0}},
/* 5675 */ {(12<<2)|2,{50,71,0}},
/* 5676 */ {(12<<2)|2,{50,71,0}},
/* 5677 */ {(12<<2)|2,{50,71,0}},
/* 5678 */ {(12<<2)|2,{50,71,0}},
/* 5679 */ {(12<<2)|2,{50,71,0}},
/* 5680 */ {(12<<2)|2,{50,72,0}},
/* 5681 */ {(12<<2)|2,{50,72,0}},
/* 5682 */ {(12<<2)|2,{50,72,0}},
/* 5683 */ {(12<<2)|2,{50,72,0}},
/* 5684 */ {(12<<2)|2,{50,72,0}},
/* 5685 */ {(12<<2)|2,{50,72,0}},
/* 5686 */ {(12<<2)|2,{50,72,0}},
/* 5687 */ {(12<<2)|2,{50,72,0}},
/* 5688 */ {(12<<2)|2,{50,72,0}},
/* 5689 */ {(12<<2)|2,{50,72,0}},
/* 5690 */ {(12<<2)|2,{50,72,0}},
/* 5691 */ {(12<<2)|2,{50,72,0}},
/* 5692 */ {(12<<2)|2,{50,72,0}},
/* 5693 */ {(12<<2)|2,{50,72,0}},
/* 5694 */ {(12<<2)|2,{50,72,0}},
/* 5695 */ {(12<<2)|2,{50,72,0}},
/* 5696 */ {(12<<2)|2,{50,73,0}},
/* 5697 */ {(12<<2)|2,{50,73,0}},
/* 5698 */ {(12<<2)|2,{50,73,0}},
/* 5699 */ {(12<<2)|2,{50,73,0}},
/* 5700 */ {(12<<2)|2,{50,73,0}},
/* 5701 */ {(12<<2)|2,{50,73,0}},
/* 5702 */ {(12<<2)|2,{50,73,0}},
/* 5703 */ {(12<<2)|2,{50,73,0}},
/* 5704 */ {(12<<2)|2,{50,73,0}},
/* 5705 */ {(12<<2)|2,{50,73,0}},
/* 5706 */ {(12<<2)|2,{50,73,0}},
/* 5707 */ {(12<<2)|2,{50,73,0}},
/* 5708 */ {(12<<2)|2,{50,73,0}},
/* 5709 */ {(12<<2)|2,{50,73,0}},
/* 5710 */ {(12<<2)|2,{50,73,0}},
/* 5711 */ {(12<<2)|2,{50,73,0}},
/* 5712 */ {(12<<2)|2,{50,74,0}},
/* 5713 */ {(12<<2)|2,{50,74,0}},
/* 5714 */ {(12<<2)|2,{50,74,0}},
/* 5715 */ {(12<<2)|2,{50,74,0}},
/* 5716 */ {(12<<2)|2,{50,74,0}},
/* 5717 */ {(12<<2)|2,{50,74,0}},
/* 5718 */ {(12<<2)|2,{50,74,0}},
/* 5719 */ {(12<<2)|2,{50,74,0}},
/* 5720 */ {(12<<2)|2,{50,74,0}},
/* 5721 */ {(12<<2)|2,{50,74,0}},
/* 5722 */ {(12<<2)|2,{50,74,0}},
/* 5723 */ {(12<<2)|2,{50,74,0}},
/* 5724 */ {(12<<2)|2,{50,74,0}},
/* 5725 */ {(12<<2)|2,{50,74,0}},
/* 5726 */ {(12<<2)|2,{50,74,0}},
/* 5727 */ {(12<<2)|2,{50,74,0}},
/* 5728 */ {(12<<2)|2,{50,75,0}},
/* 5729 */ {(12<<2)|2,{50,75,0}},
/* 5730 */ {(12<<2)|2,{50,75,0}},
/* 5731 */ {(12<<2)|2,{50,75,0}},
/* 5732 */ {(12<<2)|2,{50,75,0}},
/* 5733 */ {(12<<2)|2,{50,75,0}},
/* 5734 */ {(12<<2)|2,{50,75,0}},
/* 5735 */ {(12<<2)|2,{50,75,0}},
/* 5736 */ {(12<<2)|2,{50,75,0}},
/* 5737 */ {(12<<2)|2,{50,75,0}},
/* 5738 */ {(12<<2)|2,{50,75,0}},
/* 5739 */ {(12<<2)|2,{50,75,0}},
/* 5740 */ {(12<<2)|2,{50,75,0}},
/* 5741 */ {(12<<2)|2,{50,75,0}},
/* 5742 */ {(12<<2)|2,{50,75,0}},
/* 5743 */ {(12<<2)|2,{50,75,0}},
/* 5744 */ {(12<<2)|2,{50,76,0}},
/* 5745 */ {(12<<2)|2,{50,76,0}},
/* 5746 */ {(12<<2)|2,{50,76,0}},
/* 5747 */ {(12<<2)|2,{50,76,0}},
/* 5748 */ {(12<<2)|2,{50,76,0}},
/* 5749 */ {(12<<2)|2,{50,76,0}},
/* 5750 */ {(12<<2)|2,{50,76,0}},
/* 5751 */ {(12<<2)|2,{50,76,0}},
/* 5752 */ {(12<<2)|2,{50,76,0}},
/* 5753 */ {(12<<2)|2,{50,76,0}},
/* 5754 */ {(12<<2)|2,{50,76,0}},
/* 5755 */ {(12<<2)|2,{50,76,0}},
/* 5756 */ {(12<<2)|2,{50,76,0}},
/* 5757 */ {(12<<2)|2,{50,76,0}},
/* 5758 */ {(12<<2)|2,{50,76,0}},
/* 5759 */ {(12<<2)|2,{50,76,0}},
/* 5760 */ {(12<<2)|2,{50,77,0}},
/* 5761 */ {(12<<2)|2,{50,77,0}},
/* 5762 */ {(12<<2)|2,{50,77,0}},
/* 5763 */ {(12<<2)|2,{50,77,0}},
/* 5764 */ {(12<<2)|2,{50,77,0}},
/* 5765 */ {(12<<2)|2,{50,77,0}},
/* 5766 */ {(12<<2)|2,{50,77,0}},
/* 5767 */ {(12<<2)|2,{50,77,0}},
/* 5768 */ {(12<<2)|2,{50,77,0}},
/* 5769 */ {(12<<2)|2,{50,77,0}},
/* 5770 */ {(12<<2)|2,{50,77,0}},
/* 5771 */ {(12<<2)|2,{50,77,0}},
/* 5772 */ {(12<<2)|2,{50,77,0}},
/* 5773 */ {(12<<2)|2,{50,77,0}},
/* 5774 */ {(12<<2)|2,{50,77,0}},
/* 5775 */ {(12<<2)|2,{50,77,0}},
/* 5776 */ {(12<<2)|2,{50,78,0}},
/* 5777 */ {(12<<2)|2,{50,78,0}},
/* 5778 */ {(12<<2)|2,{50,78,0}},
/* 5779 */ {(12<<2)|2,{50,78,0}},
/* 5780 */ {(12<<2)|2,{50,78,0}},
/* 5781 */ {(12<<2)|2,{50,78,0}},
/* 5782 */ {(12<<2)|2,{50,78,0}},
/* 5783 */ {(12<<2)|2,{50,78,0}},
/* 5784 */ {(12<<2)|2,{50,78,0}},
/* 5785 */ {(12<<2)|2,{50,78,0}},
/* 5786 */ {(12<<2)|2,{50,78,0}},
/* 5787 */ {(12<<2)|2,{50,78,0}},
/* 5788 */ {(12<<2)|2,{50,78,0}},
/* 5789 */ {(12<<2)|2,{50,78,0}},
/* 5790 */ {(12<<2)|2,{50,78,0}},
/* 5791 */ {(12<<2)|2,{50,78,0}},
/* 5792 */ {(12<<2)|2,{50,79,0}},
/* 5793 */ {(12<<2)|2,{50,79,0}},
/* 5794 */ {(12<<2)|2,{50,79,0}},
/* 5795 */ {(12<<2)|2,{50,79,0}},
/* 5796 */ {(12<<2)|2,{50,79,0}},
/* 5797 */ {(12<<2)|2,{50,79,0}},
/* 5798 */ {(12<<2)|2,{50,79,0}},
/* 5799 */ {(12<<2)|2,{50,79,0}},
/* 5800 */ {(12<<2)|2,{50,79,0}},
/* 5801 */ {(12<<2)|2,{50,79,0}},
/* 5802 */ {(12<<2)|2,{50,79,0}},
/* 5803 */ {(12<<2)|2,{50,79,0}},
/* 5804 */ {(12<<2)|2,{50,79,0}},
/* 5805 */ {(12<<2)|2,{50,79,0}},
/* 5806 */ {(12<<2)|2,{50,79,0}},
/* 5807 */ {(12<<2)|2,{50,79,0}},
/* 5808 */ {(12<<2)|2,{50,80,0}},
/* 5809 */ {(12<<2)|2,{50,80,0}},
/* 5810 */ {(12<<2)|2,{50,80,0}},
/* 5811 */ {(12<<2)|2,{50,80,0}},
/* 5812 */ {(12<<2)|2,{50,80,0}},
/* 5813 */ {(12<<2)|2,{50,80,0}},
/* 5814 */ {(12<<2)|2,{50,80,0}},
/* 5815 */ {(12<<2)|2,{50,80,0}},
/* 5816 */ {(12<<2)|2,{50,80,0}},
/* 5817 */ {(12<<2)|2,{50,80,0}},
/* 5818 */ {(12<<2)|2,{50,80,0}},
/* 5819 */ {(12<<2)|2,{50,80,0}},
/* 5820 */ {(12<<2)|2,{50,80,0}},
/* 5821 */ {(12<<2)|2,{50,80,0}},
/* 5822 */ {(12<<2)|2,{50,80,0}},
/* 5823 */ {(12<<2)|2,{50,80,0}},
/* 5824 */ {(12<<2)|2,{50,81,0}},
/* 5825 */ {(12<<2)|2,{50,81,0}},
/* 5826 */ {(12<<2)|2,{50,81,0}},
/* 5827 */ {(12<<2)|2,{50,81,0}},
/* 5828 */ {(12<<2)|2,{50,81,0}},
/* 5829 */ {(12<<2)|2,{50,81,0}},
/* 5830 */ {(12<<2)|2,{50,81,0}},
/* 5831 */ {(12<<2)|2,{50,81,0}},
/* 5832 */ {(12<<2)|2,{50,81,0}},
/* 5833 */ {(12<<2)|2,{50,81,0}},
/* 5834 */ {(12<<2)|2,{50,81,0}},
/* 5835 */ {(12<<2)|2,{50,81,0}},
/* 5836 */ {(12<<2)|2,{50,81,0}},
/* 5837 */ {(12<<2)|2,{50,81,0}},
/* 5838 */ {(12<<2)|2,{50,81,0}},
/* 5839 */ {(12<<2)|2,{50,81,0}},
/* 5840 */ {(12<<2)|2,{50,82,0}},
/* 5841 */ {(12<<2)|2,{50,82,0}},
/* 5842 */ {(12<<2)|2,{50,82,0}},
/* 5843 */ {(12<<2)|2,{50,82,0}},
/* 5844 */ {(12<<2)|2,{50,82,0}},
/* 5845 */ {(12<<2)|2,{50,82,0}},
/* 5846 */ {(12<<2)|2,{50,82,0}},
/* 5847 */ {(12<<2)|2,{50,82,0}},
/* 5848 */ {(12<<2)|2,{50,82,0}},
/* 5849 */ {(12<<2)|2,{50,82,0}},
/* 5850 */ {(12<<2)|2,{50,82,0}},
/* 5851 */ {(12<<2)|2,{50,82,0}},
/* 5852 */ {(12<<2)|2,{50,82,0}},
/* 5853 */ {(12<<2)|2,{50,82,0}},
/* 5854 */ {(12<<2)|2,{50,82,0}},
/* 5855 */ {(12<<2)|2,{50,82,0}},
/* 5856 */ {(12<<2)|2,{50,83,0}},
/* 5857 */ {(12<<2)|2,{50,83,0}},
/* 5858 */ {(12<<2)|2,{50,83,0}},
/* 5859 */ {(12<<2)|2,{50,83,0}},
/* 5860 */ {(12<<2)|2,{50,83,0}},
/* 5861 */ {(12<<2)|2,{50,83,0}},
/* 5862 */ {(12<<2)|2,{50,83,0}},
/* 5863 */ {(12<<2)|2,{50,83,0}},
/* 5864 */ {(12<<2)|2,{50,83,0}},
/* 5865 */ {(12<<2)|2,{50,83,0}},
/* 5866 */ {(12<<2)|2,{50,83,0}},
/* 5867 */ {(12<<2)|2,{50,83,0}},
/* 5868 */ {(12<<2)|2,{50,83,0}},
/* 5869 */ {(12<<2)|2,{50,83,0}},
/* 5870 */ {(12<<2)|2,{50,83,0}},
/* 5871 */ {(12<<2)|2,{50,83,0}},
/* 5872 */ {(12<<2)|2,{50,84,0}},
/* 5873 */ {(12<<2)|2,{50,84,0}},
/* 5874 */ {(12<<2)|2,{50,84,0}},
/* 5875 */ {(12<<2)|2,{50,84,0}},
/* 5876 */ {(12<<2)|2,{50,84,0}},
/* 5877 */ {(12<<2)|2,{50,84,0}},
/* 5878 */ {(12<<2)|2,{50,84,0}},
/* 5879 */ {(12<<2)|2,{50,84,0}},
/* 5880 */ {(12<<2)|2,{50,84,0}},
/* 5881 */ {(12<<2)|2,{50,84,0}},
/* 5882 */ {(12<<2)|2,{50,84,0}},
/* 5883 */ {(12<<2)|2,{50,84,0}},
/* 5884 */ {(12<<2)|2,{50,84,0}},
/* 5885 */ {(12<<2)|2,{50,84,0}},
/* 5886 */ {(12<<2)|2,{50,84,0}},
/* 5887 */ {(12<<2)|2,{50,84,0}},
/* 5888 */ {(12<<2)|2,{50,85,0}},
/* 5889 */ {(12<<2)|2,{50,85,0}},
/* 5890 */ {(12<<2)|2,{50,85,0}},
/* 5891 */ {(12<<2)|2,{50,85,0}},
/* 5892 */ {(12<<2)|2,{50,85,0}},
/* 5893 */ {(12<<2)|2,{50,85,0}},
/* 5894 */ {(12<<2)|2,{50,85,0}},
/* 5895 */ {(12<<2)|2,{50,85,0}},
/* 5896 */ {(12<<2)|2,{50,85,0}},
/* 5897 */ {(12<<2)|2,{50,85,0}},
/* 5898 */ {(12<<2)|2,{50,85,0}},
/* 5899 */ {(12<<2)|2,{50,85,0}},
/* 5900 */ {(12<<2)|2,{50,85,0}},
/* 5901 */ {(12<<2)|2,{50,85,0}},
/* 5902 */ {(12<<2)|2,{50,85,0}},
/* 5903 */ {(12<<2)|2,{50,85,0}},
/* 5904 */ {(12<<2)|2,{50,86,0}},
/* 5905 */ {(12<<2)|2,{50,86,0}},
/* 5906 */ {(12<<2)|2,{50,86,0}},
/* 5907 */ {(12<<2)|2,{50,86,0}},
/* 5908 */ {(12<<2)|2,{50,86,0}},
/* 5909 */ {(12<<2)|2,{50,86,0}},
/* 5910 */ {(12<<2)|2,{50,86,0}},
/* 5911 */ {(12<<2)|2,{50,86,0}},
/* 5912 */ {(12<<2)|2,{50,86,0}},
/* 5913 */ {(12<<2)|2,{50,86,0}},
/* 5914 */ {(12<<2)|2,{50,86,0}},
/* 5915 */ {(12<<2)|2,{50,86,0}},
/* 5916 */ {(12<<2)|2,{50,86,0}},
/* 5917 */ {(12<<2)|2,{50,86,0}},
/* 5918 */ {(12<<2)|2,{50,86,0}},
/* 5919 */ {(12<<2)|2,{50,86,0}},
/* 5920 */ {(12<<2)|2,{50,87,0}},
/* 5921 */ {(12<<2)|2,{50,87,0}},
/* 5922 */ {(12<<2)|2,{50,87,0}},
/* 5923 */ {(12<<2)|2,{50,87,0}},
/* 5924 */ {(12<<2)|2,{50,87,0}},
/* 5925 */ {(12<<2)|2,{50,87,0}},
/* 5926 */ {(12<<2)|2,{50,87,0}},
/* 5927 */ {(12<<2)|2,{50,87,0}},
/* 5928 */ {(12<<2)|2,{50,87,0}},
/* 5929 */ {(12<<2)|2,{50,87,0}},
/* 5930 */ {(12<<2)|2,{50,87,0}},
/* 5931 */ {(12<<2)|2,{50,87,0}},
/* 5932 */ {(12<<2)|2,{50,87,0}},
/* 5933 */ {(12<<2)|2,{50,87,0}},
/* 5934 */ {(12<<2)|2,{50,87,0}},
/* 5935 */ {(12<<2)|2,{50,87,0}},
/* 5936 */ {(12<<2)|2,{50,89,0}},
/* 5937 */ {(12<<2)|2,{50,89,0}},
/* 5938 */ {(12<<2)|2,{50,89,0}},
/* 5939 */ {(12<<2)|2,{50,89,0}},
/* 5940 */ {(12<<2)|2,{50,89,0}},
/* 5941 */ {(12<<2)|2,{50,89,0}},
/* 5942 */ {(12<<2)|2,{50,89,0}},
/* 5943 */ {(12<<2)|2,{50,89,0}},
/* 5944 */ {(12<<2)|2,{50,89,0}},
/* 5945 */ {(12<<2)|2,{50,89,0}},
/* 5946 */ {(12<<2)|2,{50,89,0}},
/* 5947 */ {(12<<2)|2,{50,89,0}},
/* 5948 */ {(12<<2)|2,{50,89,0}},
/* 5949 */ {(12<<2)|2,{50,89,0}},
/* 5950 */ {(12<<2)|2,{50,89,0}},
/* 5951 */ {(12<<2)|2,{50,89,0}},
/* 5952 */ {(12<<2)|2,{50,106,0}},
/* 5953 */ {(12<<2)|2,{50,106,0}},
/* 5954 */ {(12<<2)|2,{50,106,0}},
/* 5955 */ {(12<<2)|2,{50,106,0}},
/* 5956 */ {(12<<2)|2,{50,106,0}},
/* 5957 */ {(12<<2)|2,{50,106,0}},
/* 5958 */ {(12<<2)|2,{50,106,0}},
/* 5959 */ {(12<<2)|2,{50,106,0}},
/* 5960 */ {(12<<2)|2,{50,106,0}},
/* 5961 */ {(12<<2)|2,{50,106,0}},
/* 5962 */ {(12<<2)|2,{50,106,0}},
/* 5963 */ {(12<<2)|2,{50,106,0}},
/* 5964 */ {(12<<2)|2,{50,106,0}},
/* 5965 */ {(12<<2)|2,{50,106,0}},
/* 5966 */ {(12<<2)|2,{50,106,0}},
/* 5967 */ {(12<<2)|2,{50,106,0}},
/* 5968 */ {(12<<2)|2,{50,107,0}},
/* 5969 */ {(12<<2)|2,{50,107,0}},
/* 5970 */ {(12<<2)|2,{50,107,0}},
/* 5971 */ {(12<<2)|2,{50,107,0}},
/* 5972 */ {(12<<2)|2,{50,107,0}},
/* 5973 */ {(12<<2)|2,{50,107,0}},
/* 5974 */ {(12<<2)|2,{50,107,0}},
/* 5975 */ {(12<<2)|2,{50,107,0}},
/* 5976 */ {(12<<2)|2,{50,107,0}},
/* 5977 */ {(12<<2)|2,{50,107,0}},
/* 5978 */ {(12<<2)|2,{50,107,0}},
/* 5979 */ {(12<<2)|2,{50,107,0}},
/* 5980 */ {(12<<2)|2,{50,107,0}},
/* 5981 */ {(12<<2)|2,{50,107,0}},
/* 5982 */ {(12<<2)|2,{50,107,0}},
/* 5983 */ {(12<<2)|2,{50,107,0}},
/* 5984 */ {(12<<2)|2,{50,113,0}},
/* 5985 */ {(12<<2)|2,{50,113,0}},
/* 5986 */ {(12<<2)|2,{50,113,0}},
/* 5987 */ {(12<<2)|2,{50,113,0}},
/* 5988 */ {(12<<2)|2,{50,113,0}},
/* 5989 */ {(12<<2)|2,{50,113,0}},
/* 5990 */ {(12<<2)|2,{50,113,0}},
/* 5991 */ {(12<<2)|2,{50,113,0}},
/* 5992 */ {(12<<2)|2,{50,113,0}},
/* 5993 */ {(12<<2)|2,{50,113,0}},
/* 5994 */ {(12<<2)|2,{50,113,0}},
/* 5995 */ {(12<<2)|2,{50,113,0}},
/* 5996 */ {(12<<2)|2,{50,113,0}},
/* 5997 */ {(12<<2)|2,{50,113,0}},
/* 5998 */ {(12<<2)|2,{50,113,0}},
/* 5999 */ {(12<<2)|2,{50,113,0}},
/* 6000 */ {(12<<2)|2,{50,118,0}},
/* 6001 */ {(12<<2)|2,{50,118,0}},
/* 6002 */ {(12<<2)|2,{50,118,0}},
/* 6003 */ {(12<<2)|2,{50,118,0}},
/* 6004 */ {(12<<2)|2,{50,118,0}},
/* 6005 */ {(12<<2)|2,{50,118,0}},
/* 6006 */ {(12<<2)|2,{50,118,0}},
/* 6007 */ {(12<<2)|2,{50,118,0}},
/* 6008 */ {(12<<2)|2,{50,118,0}},
/* 6009 */ {(12<<2)|2,{50,118,0}},
/* 6010 */ {(12<<2)|2,{50,118,0}},
/* 6011 */ {(12<<2)|2,{50,118,0}},
/* 6012 */ {(12<<2)|2,{50,118,0}},
/* 6013 */ {(12<<2)|2,{50,118,0}},
/* 6014 */ {(12<<2)|2,{50,118,0}},
/* 6015 */ {(12<<2)|2,{50,118,0}},
/* 6016 */ {(12<<2)|2,{50,119,0}},
/* 6017 */ {(12<<2)|2,{50,119,0}},
/* 6018 */ {(12<<2)|2,{50,119,0}},
/* 6019 */ {(12<<2)|2,{50,119,0}},
/* 6020 */ {(12<<2)|2,{50,119,0}},
/* 6021 */ {(12<<2)|2,{50,119,0}},
/* 6022 */ {(12<<2)|2,{50,119,0}},
/* 6023 */ {(12<<2)|2,{50,119,0}},
/* 6024 */ {(12<<2)|2,{50,119,0}},
/* 6025 */ {(12<<2)|2,{50,119,0}},
/* 6026 */ {(12<<2)|2,{50,119,0}},
/* 6027 */ {(12<<2)|2,{50,119,0}},
/* 6028 */ {(12<<2)|2,{50,119,0}},
/* 6029 */ {(12<<2)|2,{50,119,0}},
/* 6030 */ {(12<<2)|2,{50,119,0}},
/* 6031 */ {(12<<2)|2,{50,119,0}},
/* 6032 */ {(12<<2)|2,{50,120,0}},
/* 6033 */ {(12<<2)|2,{50,120,0}},
/* 6034 */ {(12<<2)|2,{50,120,0}},
/* 6035 */ {(12<<2)|2,{50,120,0}},
/* 6036 */ {(12<<2)|2,{50,120,0}},
/* 6037 */ {(12<<2)|2,{50,120,0}},
/* 6038 */ {(12<<2)|2,{50,120,0}},
/* 6039 */ {(12<<2)|2,{50,120,0}},
/* 6040 */ {(12<<2)|2,{50,120,0}},
/* 6041 */ {(12<<2)|2,{50,120,0}},
/* 6042 */ {(12<<2)|2,{50,120,0}},
/* 6043 */ {(12<<2)|2,{50,120,0}},
/* 6044 */ {(12<<2)|2,{50,120,0}},
/* 6045 */ {(12<<2)|2,{50,120,0}},
/* 6046 */ {(12<<2)|2,{50,120,0}},
/* 6047 */ {(12<<2)|2,{50,120,0}},
/* 6048 */ {(12<<2)|2,{50,121,0}},
/* 6049 */ {(12<<2)|2,{50,121,0}},
/* 6050 */ {(12<<2)|2,{50,121,0}},
/* 6051 */ {(12<<2)|2,{50,121,0}},
/* 6052 */ {(12<<2)|2,{50,121,0}},
/* 6053 */ {(12<<2)|2,{50,121,0}},
/* 6054 */ {(12<<2)|2,{50,121,0}},
/* 6055 */ {(12<<2)|2,{50,121,0}},
/* 6056 */ {(12<<2)|2,{50,121,0}},
/* 6057 */ {(12<<2)|2,{50,121,0}},
/* 6058 */ {(12<<2)|2,{50,121,0}},
/* 6059 */ {(12<<2)|2,{50,121,0}},
/* 6060 */ {(12<<2)|2,{50,121,0}},
/* 6061 */ {(12<<2)|2,{50,121,0}},
/* 6062 */ {(12<<2)|2,{50,121,0}},
/* 6063 */ {(12<<2)|2,{50,121,0}},
/* 6064 */ {(12<<2)|2,{50,122,0}},
/* 6065 */ {(12<<2)|2,{50,122,0}},
/* 6066 */ {(12<<2)|2,{50,122,0}},
/* 6067 */ {(12<<2)|2,{50,122,0}},
/* 6068 */ {(12<<2)|2,{50,122,0}},
/* 6069 */ {(12<<2)|2,{50,122,0}},
/* 6070 */ {(12<<2)|2,{50,122,0}},
/* 6071 */ {(12<<2)|2,{50,122,0}},
/* 6072 */ {(12<<2)|2,{50,122,0}},
/* 6073 */ {(12<<2)|2,{50,122,0}},
/* 6074 */ {(12<<2)|2,{50,122,0}},
/* 6075 */ {(12<<2)|2,{50,122,0}},
/* 6076 */ {(12<<2)|2,{50,122,0}},
/* 6077 */ {(12<<2)|2,{50,122,0}},
/* 6078 */ {(12<<2)|2,{50,122,0}},
/* 6079 */ {(12<<2)|2,{50,122,0}},
/* 6080 */ {(13<<2)|2,{50,38,0}},
/* 6081 */ {(13<<2)|2,{50,38,0}},
/* 6082 */ {(13<<2)|2,{50,38,0}},
/* 6083 */ {(13<<2)|2,{50,38,0}},
/* 6084 */ {(13<<2)|2,{50,38,0}},
/* 6085 */ {(13<<2)|2,{50,38,0}},
/* 6086 */ {(13<<2)|2,{50,38,0}},
/* 6087 */ {(13<<2)|2,{50,38,0}},
/* 6088 */ {(13<<2)|2,{50,42,0}},
/* 6089 */ {(13<<2)|2,{50,42,0}},
/* 6090 */ {(13<<2)|2,{50,42,0}},
/* 6091 */ {(13<<2)|2,{50,42,0}},
/* 6092 */ {(13<<2)|2,{50,42,0}},
/* 6093 */ {(13<<2)|2,{50,42,0}},
/* 6094 */ {(13<<2)|2,{50,42,0}},
/* 6095 */ {(13<<2)|2,{50,42,0}},
/* 6096 */ {(13<<2)|2,{50,44,0}},
/* 6097 */ {(13<<2)|2,{50,44,0}},
/* 6098 */ {(13<<2)|2,{50,44,0}},
/* 6099 */ {(13<<2)|2,{50,44,0}},
/* 6100 */ {(13<<2)|2,{50,44,0}},
/* 6101 */ {(13<<2)|2,{50,44,0}},
/* 6102 */ {(13<<2)|2,{50,44,0}},
/* 6103 */ {(13<<2)|2,{50,44,0}},
/* 6104 */ {(13<<2)|2,{50,59,0}},
/* 6105 */ {(13<<2)|2,{50,59,0}},
/* 6106 */ {(13<<2)|2,{50,59,0}},
/* 6107 */ {(13<<2)|2,{50,59,0}},
/* 6108 */ {(13<<2)|2,{50,59,0}},
/* 6109 */ {(13<<2)|2,{50,59,0}},
/* 6110 */ {(13<<2)|2,{50,59,0}},
/* 6111 */ {(13<<2)|2,{50,59,0}},
/* 6112 */ {(13<<2)|2,{50,88,0}},
/* 6113 */ {(13<<2)|2,{50,88,0}},
/* 6114 */ {(13<<2)|2,{50,88,0}},
/* 6115 */ {(13<<2)|2,{50,88,0}},
/* 6116 */ {(13<<2)|2,{50,88,0}},
/* 6117 */ {(13<<2)|2,{50,88,0}},
/* 6118 */ {(13<<2)|2,{50,88,0}},
/* 6119 */ {(13<<2)|2,{50,88,0}},
/* 6120 */ {(13<<2)|2,{50,90,0}},
/* 6121 */ {(13<<2)|2,{50,90,0}},
/* 6122 */ {(13<<2)|2,{50,90,0}},
/* 6123 */ {(13<<2)|2,{50,90,0}},
/* 6124 */ {(13<<2)|2,{50,90,0}},
/* 6125 */ {(13<<2)|2,{50,90,0}},
/* 6126 */ {(13<<2)|2,{50,90,0}},
/* 6127 */ {(13<<2)|2,{50,90,0}},
/* 6128 */ {(15<<2)|2,{50,33,0}},
/* 6129 */ {(15<<2)|2,{50,33,0}},
/* 6130 */ {(15<<2)|2,{50,34,0}},
/* 6131 */ {(15<<2)|2,{50,34,0}},
/* 6132 */ {(15<<2)|2,{50,40,0}},
/* 6133 */ {(15<<2)|2,{50,40,0}},
/* 6134 */ {(15<<2)|2,{50,41,0}},
/* 6135 */ {(15<<2)|2,{50,41,0}},
/* 6136 */ {(15<<2)|2,{50,63,0}},
/* 6137 */ {(15<<2)|2,{50,63,0}},
/* 6138 */ {(16<<2)|2,{50,39,0}},
/* 6139 */ {(16<<2)|2,{50,43,0}},
/* 6140 */ {(16<<2)|2,{50,124,0}},
/* 6141 */ {(5<<2)|1,{50,0,0}},
/* 6142 */ {(5<<2)|1,{50,0,0}},
/* 6143 */ {(5<<2)|1,{50,0,0}},
/* 6144 */ {(15<<2)|3,{97,48,48}},
/* 6145 */ {(15<<2)|3,{97,48,48}},
/* 6146 */ {(15<<2)|3,{97,48,49}},
/* 6147 */ {(15<<2)|3,{97,48,49}},
/* 6148 */ {(15<<2)|3,{97,48,50}},
/* 6149 */ {(15<<2)|3,{97,48,50}},
/* 6150 */ {(15<<2)|3,{97,48,97}},
/* 6151 */ {(15<<2)|3,{97,48,97}},
/* 6152 */ {(15<<2)|3,{97,48,99}},
/* 6153 */ {(15<<2)|3,{97,48,99}},
/* 6154 */ {(15<<2)|3,{97,48,101}},
/* 6155 */ {(15<<2)|3,{97,48,101}},
/* 6156 */ {(15<<2)|3,{97,48,105}},
/* 6157 */ {(15<<2)|3,{97,48,105}},
/* 6158 */ {(15<<2)|3,{97,48,111}},
/* 6159 */ {(15<<2)|3,{97,48,111}},
/* 6160 */ {(15<<2)|3,{97,48,115}},
/* 6161 */ {(15<<2)|3,{97,48,115}},
/* 6162 */ {(15<<2)|3,{97,48,116}},
/* 6163 */ {(15<<2)|3,{97,48,116}},
/* 6164 */ {(16<<2)|3,{97,48,32}},
/* 6165 */ {(16<<2)|3,{97,48,37}},
/* 6166 */ {(16<<2)|3,{97,48,45}},
/* 6167 */ {(16<<2)|3,{97,48,46}},
/* 6168 */ {(16<<2)|3,{97,48,47}},
/* 6169 */ {(16<<2)|3,{97,48,51}},
/* 6170 */ {(16<<2)|3,{97,48,52}},
/* 6171 */ {(16<<2)|3,{97,48,53}},
/* 6172 */ {(16<<2)|3,{97,48,54}},
/* 6173 */ {(16<<2)|3,{97,48,55}},
/* 6174 */ {(16<<2)|3,{97,48,56}},
/* 6175 */ {(16<<2)|3,{97,48,57}},
/* 6176 */ {(16<<2)|3,{97,48,61}},
/* 6177 */ {(16<<2)|3,{97,48,65}},
/* 6178 */ {(16<<2)|3,{97,48,95}},
/* 6179 */ {(16<<2)|3,{97,48,98}},
/* 6180 */ {(16<<2)|3,{97,48,100}},
/* 6181 */ {(16<<2)|3,{97,48,102}},
/* 6182 */ {(16<<2)|3,{97,48,103}},
/* 6183 */ {(16<<2)|3,{97,48,104}},
/* 6184 */ {(16<<2)|3,{97,48,108}},
/* 6185 */ {(16<<2)|3,{97,48,109}},
/* 6186 */ {(16<<2)|3,{97,48,110}},
/* 6187 */ {(16<<2)|3,{97,48,112}},
/* 6188 */ {(16<<2)|3,{97,48,114}},
/* 6189 */ {(16<<2)|3,{97,48,117}},
/* 6190 */ {(10<<2)|2,{97,48,0}},
/* 6191 */ {(10<<2)|2,{97,48,0}},
/* 6192 */ {(10<<2)|2,{97,48,0}},
/* 6193 */ {(10<<2)|2,{97,48,0}},
/* 6194 */ {(10<<2)|2,{97,48,0}},
/* 6195 */ {(10<<2)|2,{97,48,0}},
/* 6196 */ {(10<<2)|2,{97,48,0}},
/* 6197 */ {(10<<2)|2,{97,48,0}},
/* 6198 */ {(10<<2)|2,{97,48,0}},
/* 6199 */ {(10<<2)|2,{97,48,0}},
/* 6200 */ {(10<<2)|2,{97,48,0}},
/* 6201 */ {(10<<2)|2,{97,48,0}},
/* 6202 */ {(10<<2)|2,{97,48,0}},
/* 6203 */ {(10<<2)|2,{97,48,0}},
/* 6204 */ {(10<<2)|2,{97,48,0}},
/* 6205 */ {(10<<2)|2,{97,48,0}},
/* 6206 */ {(10<<2)|2,{97,48,0}},
/* 6207 */ {(10<<2)|2,{97,48,0}},
/* 6208 */ {(15<<2)|3,{97,49,48}},
/* 6209 */ {(15<<2)|3,{97,49,48}},
/* 6210 */ {(15<<2)|3,{97,49,49}},
/* 6211 */ {(15<<2)|3,{97,49,49}},
/* 6212 */ {(15<<2)|3,{97,49,50}},
/* 6213 */ {(15<<2)|3,{97,49,50}},
/* 6214 */ {(15<<2)|3,{97,49,97}},
/* 6215 */ {(15<<2)|3,{97,49,97}},
/* 6216 */ {(15<<2)|3,{97,49,99}},
/* 6217 */ {(15<<2)|3,{97,49,99}},
/* 6218 */ {(15<<2)|3,{97,49,101}},
/* 6219 */ {(15<<2)|3,{97,49,101}},
/* 6220 */ {(15<<2)|3,{97,49,105}},
/* 6221 */ {(15<<2)|3,{97,49,105}},
/* 6222 */ {(15<<2)|3,{97,49,111}},
/* 6223 */ {(15<<2)|3,{97,49,111}},
/* 6224 */ {(15<<2)|3,{97,49,115}},
/* 6225 */ {(15<<2)|3,{97,49,115}},
/* 6226 */ {(15<<2)|3,{97,49,116}},
/* 6227 */ {(15<<2)|3,{97,49,116}},
/* 6228 */ {(16<<2)|3,{97,49,32}},
/* 6229 */ {(16<<2)|3,{97,49,37}},
/* 6230 */ {(16<<2)|3,{97,49,45}},
/* 6231 */ {(16<<2)|3,{97,49,46}},
/* 6232 */ {(16<<2)|3,{97,49,47}},
/* 6233 */ {(16<<2)|3,{97,49,51}},
/* 6234 */ {(16<<2)|3,{97,49,52}},
/* 6235 */ {(16<<2)|3,{97,49,53}},
/* 6236 */ {(16<<2)|3,{97,49,54}},
/* 6237 */ {(16<<2)|3,{97,49,55}},
/* 6238 */ {(16<<2)|3,{97,49,56}},
/* 6239 */ {(16<<2)|3,{97,49,57}},
/* 6240 */ {(16<<2)|3,{97,49,61}},
/* 6241 */ {(16<<2)|3,{97,49,65}},
/* 6242 */ {(16<<2)|3,{97,49,95}},
/* 6243 */ {(16<<2)|3,{97,49,98}},
/* 6244 */ {(16<<2)|3,{97,49,100}},
/* 6245 */ {(16<<2)|3,{97,49,102}},
/* 6246 */ {(16<<2)|3,{97,49,103}},
/* 6247 */ {(16<<2)|3,{97,49,104}},
/* 6248 */ {(16<<2)|3,{97,49,108}},
/* 6249 */ {(16<<2)|3,{97,49,109}},
/* 6250 */ {(16<<2)|3,{97,49,110}},
/* 6251 */ {(16<<2)|3,{97,49,112}},
/* 6252 */ {(16<<2)|3,{97,49,114}},
/* 6253 */ {(16<<2)|3,{97,49,117}},
/* 6254 */ {(10<<2)|2,{97,49,0}},
/* 6255 */ {(10<<2)|2,{97,49,0}},
/* 6256 */ {(10<<2)|2,{97,49,0}},
/* 6257 */ {(10<<2)|2,{97,49,0}},
/* 6258 */ {(10<<2)|2,{97,49,0}},
/* 6259 */ {(10<<2)|2,{97,49,0}},
/* 6260 */ {(10<<2)|2,{97,49,0}},
/* 6261 */ {(10<<2)|2,{97,49,0}},
/* 6262 */ {(10<<2)|2,{97,49,0}},
/* 6263 */ {(10<<2)|2,{97,49,0}},
/* 6264 */ {(10<<2)|2,{97,49,0}},
/* 6265 */ {(10<<2)|2,{97,49,0}},
/* 6266 */ {(10<<2)|2,{97,49,0}},
/* 6267 */ {(10<<2)|2,{97,49,0}},
/* 6268 */ {(10<<2)|2,{97,49,0}},
/* 6269 */ {(10<<2)|2,{97,49,0}},
/* 6270 */ {(10<<2)|2,{97,49,0}},
/* 6271 */ {(10<<2)|2,{97,49,0}},
/* 6272 */ {(15<<2)|3,{97,50,48}},
/* 6273 */ {(15<<2)|3,{97,50,48}},
/* 6274 */ {(15<<2)|3,{97,50,49}},
/* 6275 */ {(15<<2)|3,{97,50,49}},
/* 6276 */ {(15<<2)|3,{97,50,50}},
/* 6277 */ {(15<<2)|3,{97,50,50}},
/* 6278 */ {(15<<2)|3,{97,50,97}},
/* 6279 */ {(15<<2)|3,{97,50,97}},
/* 6280 */ {(15<<2)|3,{97,50,99}},
/* 6281 */ {(15<<2)|3,{97,50,99}},
/* 6282 */ {(15<<2)|3,{97,50,101}},
/* 6283 */ {(15<<2)|3,{97,50,101}},
/* 6284 */ {(15<<2)|3,{97,50,105}},
/* 6285 */ {(15<<2)|3,{97,50,105}},
/* 6286 */ {(15<<2)|3,{97,50,111}},
/* 6287 */ {(15<<2)|3,{97,50,111}},
/* 6288 */ {(15<<2)|3,{97,50,115}},
/* 6289 */ {(15<<2)|3,{97,50,115}},
/* 6290 */ {(15<<2)|3,{97,50,116}},
/* 6291 */ {(15<<2)|3,{97,50,116}},
/* 6292 */ {(16<<2)|3,{97,50,32}},
/* 6293 */ {(16<<2)|3,{97,50,37}},
/* 6294 */ {(16<<2)|3,{97,50,45}},
/* 6295 */ {(16<<2)|3,{97,50,46}},
/* 6296 */ {(16<<2)|3,{97,50,47}},
/* 6297 */ {(16<<2)|3,{97,50,51}},
/* 6298 */ {(16<<2)|3,{97,50,52}},
/* 6299 */ {(16<<2)|3,{97,50,53}},
/* 6300 */ {(16<<2)|3,{97,50,54}},
/* 6301 */ {(16<<2)|3,{97,50,55}},
/* 6302 */ {(16<<2)|3,{97,50,56}},
/* 6303 */ {(16<<2)|3,{97,50,57}},
/* 6304 */ {(16<<2)|3,{97,50,61}},
/* 6305 */ {(16<<2)|3,{97,50,65}},
/* 6306 */ {(16<<2)|3,{97,50,95}},
/* 6307 */ {(16<<2)|3,{97,50,98}},
/* 6308 */ {(16<<2)|3,{97,50,100}},
/* 6309 */ {(16<<2)|3,{97,50,102}},
/* 6310 */ {(16<<2)|3,{97,50,103}},
/* 6311 */ {(16<<2)|3,{97,50,104}},
/* 6312 */ {(16<<2)|3,{97,50,108}},
/* 6313 */ {(16<<2)|3,{97,50,109}},
/* 6314 */ {(16<<2)|3,{97,50,110}},
/* 6315 */ {(16<<2)|3,{97,50,112}},
/* 6316 */ {(16<<2)|3,{97,50,114}},
/* 6317 */ {(16<<2)|3,{97,50,117}},
/* 6318 */ {(10<<2)|2,{97,50,0}},
/* 6319 */ {(10<<2)|2,{97,50,0}},
/* 6320 */ {(10<<2)|2,{97,50,0}},
/* 6321 */ {(10<<2)|2,{97,50,0}},
/* 6322 */ {(10<<2)|2,{97,50,0}},
/* 6323 */ {(10<<2)|2,{97,50,0}},
/* 6324 */ {(10<<2)|2,{97,50,0}},
/* 6325 */ {(10<<2)|2,{97,50,0}},
/* 6326 */ {(10<<2)|2,{97,50,0}},
/* 6327 */ {(10<<2)|2,{97,50,0}},
/* 6328 */ {(10<<2)|2,{97,50,0}},
/* 6329 */ {(10<<2)|2,{97,50,0}},
/* 6330 */ {(10<<2)|2,{97,50,0}},
/* 6331 */ {(10<<2)|2,{97,50,0}},
/* 6332 */ {(10<<2)|2,{97,50,0}},
/* 6333 */ {(10<<2)|2,{97,50,0}},
/* 6334 */ {(10<<2)|2,{97,50,0}},
/* 6335 */ {(10<<2)|2,{97,50,0}},
/* 6336 */ {(15<<2)|3,{97,97,48}},
/* 6337 */ {(15<<2)|3,{97,97,48}},
/* 6338 */ {(15<<2)|3,{97,97,49}},
/* 6339 */ {(15<<2)|3,{97,97,49}},
/* 6340 */ {(15<<2)|3,{97,97,50}},
/* 6341 */ {(15<<2)|3,{97,97,50}},
/* 6342 */ {(15<<2)|3,{97,97,97}},
/* 6343 */ {(15<<2)|3,{97,97,97}},
/* 6344 */ {(15<<2)|3,{97,97,99}},
/* 6345 */ {(15<<2)|3,{97,97,99}},
/* 6346 */ {(15<<2)|3,{97,97,101}},
/* 6347 */ {(15<<2)|3,{97,97,101}},
/* 6348 */ {(15<<2)|3,{97,97,105}},
/* 6349 */ {(15<<2)|3,{97,97,105}},
/* 6350 */ {(15<<2)|3,{97,97,111}},
/* 6351 */ {(15<<2)|3,{97,97,111}},
/* 6352 */ {(15<<2)|3,{97,97,115}},
/* 6353 */ {(15<<2)|3,{97,97,115}},
/* 6354 */ {(15<<2)|3,{97,97,116}},
/* 6355 */ {(15<<2)|3,{97,97,116}},
/* 6356 */ {(16<<2)|3,{97,97,32}},
/* 6357 */ {(16<<2)|3,{97,97,37}},
/* 6358 */ {(16<<2)|3,{97,97,45}},
/* 6359 */ {(16<<2)|3,{97,97,46}},
/* 6360 */ {(16<<2)|3,{97,97,47}},
/* 6361 */ {(16<<2)|3,{97,97,51}},
/* 6362 */ {(16<<2)|3,{97,97,52}},
/* 6363 */ {(16<<2)|3,{97,97,53}},
/* 6364 */ {(16<<2)|3,{97,97,54}},
/* 6365 */ {(16<<2)|3,{97,97,55}},
/* 6366 */ {(16<<2)|3,{97,97,56}},
/* 6367 */ {(16<<2)|3,{97,97,57}},
/* 6368 */ {(16<<2)|3,{97,97,61}},
/* 6369 */ {(16<<2)|3,{97,97,65}},
/* 6370 */ {(16<<2)|3,{97,97,95}},
/* 6371 */ {(16<<2)|3,{97,97,98}},
/* 6372 */ {(16<<2)|3,{97,97,100}},
/* 6373 */ {(16<<2)|3,{97,97,102}},
/* 6374 */ {(16<<2)|3,{97,97,103}},
/* 6375 */ {(16<<2)|3,{97,97,104}},
/* 6376 */ {(16<<2)|3,{97,97,108}},
/* 6377 */ {(16<<2)|3,{97,97,109}},
/* 6378 */ {(16<<2)|3,{97,97,110}},
/* 6379 */ {(16<<2)|3,{97,97,112}},
/* 6380 */ {(16<<2)|3,{97,97,114}},
/* 6381 */ {(16<<2)|3,{97,97,117}},
/* 6382 */ {(10<<2)|2,{97,97,0}},
/* 6383 */ {(10<<2)|2,{97,97,0}},
/* 6384 */ {(10<<2)|2,{97,97,0}},
/* 6385 */ {(10<<2)|2,{97,97,0}},
/* 6386 */ {(10<<2)|2,{97,97,0}},
/* 6387 */ {(10<<2)|2,{97,97,0}},
/* 6388 */ {(10<<2)|2,{97,97,0}},
/* 6389 */ {(10<<2)|2,{97,97,0}},
/* 6390 */ {(10<<2)|2,{97,97,0}},
/* 6391 */ {(10<<2)|2,{97,97,0}},
/* 6392 */ {(10<<2)|2,{97,97,0}},
/* 6393 */ {(10<<2)|2,{97,97,0}},
/* 6394 */ {(10<<2)|2,{97,97,0}},
/* 6395 */ {(10<<2)|2,{97,97,0}},
/* 6396 */ {(10<<2)|2,{97,97,0}},
/* 6397 */ {(10<<2)|2,{97,97,0}},
/* 6398 */ {(10<<2)|2,{97,97,0}},
/* 6399 */ {(10<<2)|2,{97,97,0}},
/* 6400 */ {(15<<2)|3,{97,99,48}},
/* 6401 */ {(15<<2)|3,{97,99,48}},
/* 6402 */ {(15<<2)|3,{97,99,49}},
/* 6403 */ {(15<<2)|3,{97,99,49}},
/* 6404 */ {(15<<2)|3,{97,99,50}},
/* 6405 */ {(15<<2)|3,{97,99,50}},
/* 6406 */ {(15<<2)|3,{97,99,97}},
/* 6407 */ {(15<<2)|3,{97,99,97}},
/* 6408 */ {(15<<2)|3,{97,99,99}},
/* 6409 */ {(15<<2)|3,{97,99,99}},
/* 6410 */ {(15<<2)|3,{97,99,101}},
/* 6411 */ {(15<<2)|3,{97,99,101}},
/* 6412 */ {(15<<2)|3,{97,99,105}},
/* 6413 */ {(15<<2)|3,{97,99,105}},
/* 6414 */ {(15<<2)|3,{97,99,111}},
/* 6415 */ {(15<<2)|3,{97,99,111}},
/* 6416 */ {(15<<2)|3,{97,99,115}},
/* 6417 */ {(15<<2)|3,{97,99,115}},
/* 6418 */ {(15<<2)|3,{97,99,116}},
/* 6419 */ {(15<<2)|3,{97,99,116}},
/* 6420 */ {(16<<2)|3,{97,99,32}},
/* 6421 */ {(16<<2)|3,{97,99,37}},
/* 6422 */ {(16<<2)|3,{97,99,45}},
/* 6423 */ {(16<<2)|3,{97,99,46}},
/* 6424 */ {(16<<2)|3,{97,99,47}},
/* 6425 */ {(16<<2)|3,{97,99,51}},
/* 6426 */ {(16<<2)|3,{97,99,52}},
/* 6427 */ {(16<<2)|3,{97,99,53}},
/* 6428 */ {(16<<2)|3,{97,99,54}},
/* 6429 */ {(16<<2)|3,{97,99,55}},
/* 6430 */ {(16<<2)|3,{97,99,56}},
/* 6431 */ {(16<<2)|3,{97,99,57}},
/* 6432 */ {(16<<2)|3,{97,99,61}},
/* 6433 */ {(16<<2)|3,{97,99,65}},
/* 6434 */ {(16<<2)|3,{97,99,95}},
/* 6435 */ {(16<<2)|3,{97,99,98}},
/* 6436 */ {(16<<2)|3,{97,99,100}},
/* 6437 */ {(16<<2)|3,{97,99,102}},
/* 6438 */ {(16<<2)|3,{97,99,103}},
/* 6439 */ {(16<<2)|3,{97,99,104}},
/* 6440 */ {(16<<2)|3,{97,99,108}},
/* 6441 */ {(16<<2)|3,{97,99,109}},
/* 6442 */ {(16<<2)|3,{97,99,110}},
/* 6443 */ {(16<<2)|3,{97,99,112}},
/* 6444 */ {(16<<2)|3,{97,99,114}},
/* 6445 */ {(16<<2)|3,{97,99,117}},
/* 6446 */ {(10<<2)|2,{97,99,0}},
/* 6447 */ {(10<<2)|2,{97,99,0}},
/* 6448 */ {(10<<2)|2,{97,99,0}},
/* 6449 */ {(10<<2)|2,{97,99,0}},
/* 6450 */ {(10<<2)|2,{97,99,0}},
/* 6451 */ {(10<<2)|2,{97,99,0}},
/* 6452 */ {(10<<2)|2,{97,99,0}},
/* 6453 */ {(10<<2)|2,{97,99,0}},
/* 6454 */ {(10<<2)|2,{97,99,0}},
/* 6455 */ {(10<<2)|2,{97,99,0}},
/* 6456 */ {(10<<2)|2,{97,99,0}},
/* 6457 */ {(10<<2)|2,{97,99,0}},
/* 6458 */ {(10<<2)|2,{97,99,0}},
/* 6459 */ {(10<<2)|2,{97,99,0}},
/* 6460 */ {(10<<2)|2,{97,99,0}},
/* 6461 */ {(10<<2)|2,{97,99,0}},
/* 6462 */ {(10<<2)|2,{97,99,0}},
/* 6463 */ {(10<<2)|2,{97,99,0}},
/* 6464 */ {(15<<2)|3,{97,101,48}},
/* 6465 */ {(15<<2)|3,{97,101,48}},
/* 6466 */ {(15<<2)|3,{97,101,49}},
/* 6467 */ {(15<<2)|3,{97,101,49}},
/* 6468 */ {(15<<2)|3,{97,101,50}},
/* 6469 */ {(15<<2)|3,{97,101,50}},
/* 6470 */ {(15<<2)|3,{97,101,97}},
/* 6471 */ {(15<<2)|3,{97,101,97}},
/* 6472 */ {(15<<2)|3,{97,101,99}},
/* 6473 */ {(15<<2)|3,{97,101,99}},
/* 6474 */ {(15<<2)|3,{97,101,101}},
/* 6475 */ {(15<<2)|3,{97,101,101}},
/* 6476 */ {(15<<2)|3,{97,101,105}},
/* 6477 */ {(15<<2)|3,{97,101,105}},
/* 6478 */ {(15<<2)|3,{97,101,111}},
/* 6479 */ {(15<<2)|3,{97,101,111}},
/* 6480 */ {(15<<2)|3,{97,101,115}},
/* 6481 */ {(15<<2)|3,{97,101,115}},
/* 6482 */ {(15<<2)|3,{97,101,116}},
/* 6483 */ {(15<<2)|3,{97,101,116}},
/* 6484 */ {(16<<2)|3,{97,101,32}},
/* 6485 */ {(16<<2)|3,{97,101,37}},
/* 6486 */ {(16<<2)|3,{97,101,45}},
/* 6487 */ {(16<<2)|3,{97,101,46}},
/* 6488 */ {(16<<2)|3,{97,101,47}},
/* 6489 */ {(16<<2)|3,{97,101,51}},
/* 6490 */ {(16<<2)|3,{97,101,52}},
/* 6491 */ {(16<<2)|3,{97,101,53}},
/* 6492 */ {(16<<2)|3,{97,101,54}},
/* 6493 */ {(16<<2)|3,{97,101,55}},
/* 6494 */ {(16<<2)|3,{97,101,56}},
/* 6495 */ {(16<<2)|3,{97,101,57}},
/* 6496 */ {(16<<2)|3,{97,101,61}},
/* 6497 */ {(16<<2)|3,{97,101,65}},
/* 6498 */ {(16<<2)|3,{97,101,95}},
/* 6499 */ {(16<<2)|3,{97,101,98}},
/* 6500 */ {(16<<2)|3,{97,101,100}},
/* 6501 */ {(16<<2)|3,{97,101,102}},
/* 6502 */ {(16<<2)|3,{97,101,103}},
/* 6503 */ {(16<<2)|3,{97,101,104}},
/* 6504 */ {(16<<2)|3,{97,101,108}},
/* 6505 */ {(16<<2)|3,{97,101,109}},
/* 6506 */ {(16<<2)|3,{97,101,110}},
/* 6507 */ {(16<<2)|3,{97,101,112}},
/* 6508 */ {(16<<2)|3,{97,101,114}},
/* 6509 */ {(16<<2)|3,{97,101,117}},
/* 6510 */ {(10<<2)|2,{97,101,0}},
/* 6511 */ {(10<<2)|2,{97,101,0}},
/* 6512 */ {(10<<2)|2,{97,101,0}},
/* 6513 */ {(10<<2)|2,{97,101,0}},
/* 6514 */ {(10<<2)|2,{97,101,0}},
/* 6515 */ {(10<<2)|2,{97,101,0}},
/* 6516 */ {(10<<2)|2,{97,101,0}},
/* 6517 */ {(10<<2)|2,{97,101,0}},
/* 6518 */ {(10<<2)|2,{97,101,0}},
/* 6519 */ {(10<<2)|2,{97,101,0}},
/* 6520 */ {(10<<2)|2,{97,101,0}},
/* 6521 */ {(10<<2)|2,{97,101,0}},
/* 6522 */ {(10<<2)|2,{97,101,0}},
/* 6523 */ {(10<<2)|2,{97,101,0}},
/* 6524 */ {(10<<2)|2,{97,101,0}},
/* 6525 */ {(10<<2)|2,{97,101,0}},
/* 6526 */ {(10<<2)|2,{97,101,0}},
/* 6527 */ {(10<<2)|2,{97,101,0}},
/* 6528 */ {(15<<2)|3,{97,105,48}},
/* 6529 */ {(15<<2)|3,{97,105,48}},
/* 6530 */ {(15<<2)|3,{97,105,49}},
/* 6531 */ {(15<<2)|3,{97,105,49}},
/* 6532 */ {(15<<2)|3,{97,105,50}},
/* 6533 */ {(15<<2)|3,{97,105,50}},
/* 6534 */ {(15<<2)|3,{97,105,97}},
/* 6535 */ {(15<<2)|3,{97,105,97}},
/* 6536 */ {(15<<2)|3,{97,105,99}},
/* 6537 */ {(15<<2)|3,{97,105,99}},
/* 6538 */ {(15<<2)|3,{97,105,101}},
/* 6539 */ {(15<<2)|3,{97,105,101}},
/* 6540 */ {(15<<2)|3,{97,105,105}},
/* 6541 */ {(15<<2)|3,{97,105,105}},
/* 6542 */ {(15<<2)|3,{97,105,111}},
/* 6543 */ {(15<<2)|3,{97,105,111}},
/* 6544 */ {(15<<2)|3,{97,105,115}},
/* 6545 */ {(15<<2)|3,{97,105,115}},
/* 6546 */ {(15<<2)|3,{97,105,116}},
/* 6547 */ {(15<<2)|3,{97,105,116}},
/* 6548 */ {(16<<2)|3,{97,105,32}},
/* 6549 */ {(16<<2)|3,{97,105,37}},
/* 6550 */ {(16<<2)|3,{97,105,45}},
/* 6551 */ {(16<<2)|3,{97,105,46}},
/* 6552 */ {(16<<2)|3,{97,105,47}},
/* 6553 */ {(16<<2)|3,{97,105,51}},
/* 6554 */ {(16<<2)|3,{97,105,52}},
/* 6555 */ {(16<<2)|3,{97,105,53}},
/* 6556 */ {(16<<2)|3,{97,105,54}},
/* 6557 */ {(16<<2)|3,{97,105,55}},
/* 6558 */ {(16<<2)|3,{97,105,56}},
/* 6559 */ {(16<<2)|3,{97,105,57}},
/* 6560 */ {(16<<2)|3,{97,105,61}},
/* 6561 */ {(16<<2)|3,{97,105,65}},
/* 6562 */ {(16<<2)|3,{97,105,95}},
/* 6563 */ {(16<<2)|3,{97,105,98}},
/* 6564 */ {(16<<2)|3,{97,105,100}},
/* 6565 */ {(16<<2)|3,{97,105,102}},
/* 6566 */ {(16<<2)|3,{97,105,103}},
/* 6567 */ {(16<<2)|3,{97,105,104}},
/* 6568 */ {(16<<2)|3,{97,105,108}},
/* 6569 */ {(16<<2)|3,{97,105,109}},
/* 6570 */ {(16<<2)|3,{97,105,110}},
/* 6571 */ {(16<<2)|3,{97,105,112}},
/* 6572 */ {(16<<2)|3,{97,105,114}},
/* 6573 */ {(16<<2)|3,{97,105,117}},
/* 6574 */ {(10<<2)|2,{97,105,0}},
/* 6575 */ {(10<<2)|2,{97,105,0}},
/* 6576 */ {(10<<2)|2,{97,105,0}},
/* 6577 */ {(10<<2)|2,{97,105,0}},
/* 6578 */ {(10<<2)|2,{97,105,0}},
/* 6579 */ {(10<<2)|2,{97,105,0}},
/* 6580 */ {(10<<2)|2,{97,105,0}},
/* 6581 */ {(10<<2)|2,{97,105,0}},
/* 6582 */ {(10<<2)|2,{97,105,0}},
/* 6583 */ {(10<<2)|2,{97,105,0}},
/* 6584 */ {(10<<2)|2,{97,105,0}},
/* 6585 */ {(10<<2)|2,{97,105,0}},
/* 6586 */ {(10<<2)|2,{97,105,0}},
/* 6587 */ {(10<<2)|2,{97,105,0}},
/* 6588 */ {(10<<2)|2,{97,105,0}},
/* 6589 */ {(10<<2)|2,{97,105,0}},
/* 6590 */ {(10<<2)|2,{97,105,0}},
/* 6591 */ {(10<<2)|2,{97,105,0}},
/* 6592 */ {(15<<2)|3,{97,111,48}},
/* 6593 */ {(15<<2)|3,{97,111,48}},
/* 6594 */ {(15<<2)|3,{97,111,49}},
/* 6595 */ {(15<<2)|3,{97,111,49}},
/* 6596 */ {(15<<2)|3,{97,111,50}},
/* 6597 */ {(15<<2)|3,{97,111,50}},
/* 6598 */ {(15<<2)|3,{97,111,97}},
/* 6599 */ {(15<<2)|3,{97,111,97}},
/* 6600 */ {(15<<2)|3,{97,111,99}},
/* 6601 */ {(15<<2)|3,{97,111,99}},
/* 6602 */ {(15<<2)|3,{97,111,101}},
/* 6603 */ {(15<<2)|3,{97,111,101}},
/* 6604 */ {(15<<2)|3,{97,111,105}},
/* 6605 */ {(15<<2)|3,{97,111,105}},
/* 6606 */ {(15<<2)|3,{97,111,111}},
/* 6607 */ {(15<<2)|3,{97,111,111}},
/* 6608 */ {(15<<2)|3,{97,111,115}},
/* 6609 */ {(15<<2)|3,{97,111,115}},
/* 6610 */ {(15<<2)|3,{97,111,116}},
/* 6611 */ {(15<<2)|3,{97,111,116}},
/* 6612 */ {(16<<2)|3,{97,111,32}},
/* 6613 */ {(16<<2)|3,{97,111,37}},
/* 6614 */ {(16<<2)|3,{97,111,45}},
/* 6615 */ {(16<<2)|3,{97,111,46}},
/* 6616 */ {(16<<2)|3,{97,111,47}},
/* 6617 */ {(16<<2)|3,{97,111,51}},
/* 6618 */ {(16<<2)|3,{97,111,52}},
/* 6619 */ {(16<<2)|3,{97,111,53}},
/* 6620 */ {(16<<2)|3,{97,111,54}},
/* 6621 */ {(16<<2)|3,{97,111,55}},
/* 6622 */ {(16<<2)|3,{97,111,56}},
/* 6623 */ {(16<<2)|3,{97,111,57}},
/* 6624 */ {(16<<2)|3,{97,111,61}},
/* 6625 */ {(16<<2)|3,{97,111,65}},
/* 6626 */ {(16<<2)|3,{97,111,95}},
/* 6627 */ {(16<<2)|3,{97,111,98}},
/* 6628 */ {(16<<2)|3,{97,111,100}},
/* 6629 */ {(16<<2)|3,{97,111,102}},
/* 6630 */ {(16<<2)|3,{97,111,103}},
/* 6631 */ {(16<<2)|3,{97,111,104}},
/* 6632 */ {(16<<2)|3,{97,111,108}},
/* 6633 */ {(16<<2)|3,{97,111,109}},
/* 6634 */ {(16<<2)|3,{97,111,110}},
/* 6635 */ {(16<<2)|3,{97,111,112}},
/* 6636 */ {(16<<2)|3,{97,111,114}},
/* 6637 */ {(16<<2)|3,{97,111,117}},
/* 6638 */ {(10<<2)|2,{97,111,0}},
/* 6639 */ {(10<<2)|2,{97,111,0}},
/* 6640 */ {(10<<2)|2,{97,111,0}},
/* 6641 */ {(10<<2)|2,{97,111,0}},
/* 6642 */ {(10<<2)|2,{97,111,0}},
/* 6643 */ {(10<<2)|2,{97,111,0}},
/* 6644 */ {(10<<2)|2,{97,111,0}},
/* 6645 */ {(10<<2)|2,{97,111,0}},
/* 6646 */ {(10<<2)|2,{97,111,0}},
/* 6647 */ {(10<<2)|2,{97,111,0}},
/* 6648 */ {(10<<2)|2,{97,111,0}},
/* 6649 */ {(10<<2)|2,{97,111,0}},
/* 6650 */ {(10<<2)|2,{97,111,0}},
/* 6651 */ {(10<<2)|2,{97,111,0}},
/* 6652 */ {(10<<2)|2,{97,111,0}},
/* 6653 */ {(10<<2)|2,{97,111,0}},
/* 6654 */ {(10<<2)|2,{97,111,0}},
/* 6655 */ {(10<<2)|2,{97,111,0}},
/* 6656 */ {(15<<2)|3,{97,115,48}},
/* 6657 */ {(15<<2)|3,{97,115,48}},
/* 6658 */ {(15<<2)|3,{97,115,49}},
/* 6659 */ {(15<<2)|3,{97,115,49}},
/* 6660 */ {(15<<2)|3,{97,115,50}},
/* 6661 */ {(15<<2)|3,{97,115,50}},
/* 6662 */ {(15<<2)|3,{97,115,97}},
/* 6663 */ {(15<<2)|3,{97,115,97}},
/* 6664 */ {(15<<2)|3,{97,115,99}},
/* 6665 */ {(15<<2)|3,{97,115,99}},
/* 6666 */ {(15<<2)|3,{97,115,101}},
/* 6667 */ {(15<<2)|3,{97,115,101}},
/* 6668 */ {(15<<2)|3,{97,115,105}},
/* 6669 */ {(15<<2)|3,{97,115,105}},
/* 6670 */ {(15<<2)|3,{97,115,111}},
/* 6671 */ {(15<<2)|3,{97,115,111}},
/* 6672 */ {(15<<2)|3,{97,115,115}},
/* 6673 */ {(15<<2)|3,{97,115,115}},
/* 6674 */ {(15<<2)|3,{97,115,116}},
/* 6675 */ {(15<<2)|3,{97,115,116}},
/* 6676 */ {(16<<2)|3,{97,115,32}},
/* 6677 */ {(16<<2)|3,{97,115,37}},
/* 6678 */ {(16<<2)|3,{97,115,45}},
/* 6679 */ {(16<<2)|3,{97,115,46}},
/* 6680 */ {(16<<2)|3,{97,115,47}},
/* 6681 */ {(16<<2)|3,{97,115,51}},
/* 6682 */ {(16<<2)|3,{97,115,52}},
/* 6683 */ {(16<<2)|3,{97,115,53}},
/* 6684 */ {(16<<2)|3,{97,115,54}},
/* 6685 */ {(16<<2)|3,{97,115,55}},
/* 6686 */ {(16<<2)|3,{97,115,56}},
/* 6687 */ {(16<<2)|3,{97,115,57}},
/* 6688 */ {(16<<2)|3,{97,115,61}},
/* 6689 */ {(16<<2)|3,{97,115,65}},
/* 6690 */ {(16<<2)|3,{97,115,95}},
/* 6691 */ {(16<<2)|3,{97,115,98}},
/* 6692 */ {(16<<2)|3,{97,115,100}},
/* 6693 */ {(16<<2)|3,{97,115,102}},
/* 6694 */ {(16<<2)|3,{97,115,103}},
/* 6695 */ {(16<<2)|3,{97,115,104}},
/* 6696 */ {(16<<2)|3,{97,115,108}},
/* 6697 */ {(16<<2)|3,{97,115,109}},
/* 6698 */ {(16<<2)|3,{97,115,110}},
/* 6699 */ {(16<<2)|3,{97,115,112}},
/* 6700 */ {(16<<2)|3,{97,115,114}},
/* 6701 */ {(16<<2)|3,{97,115,117}},
/* 6702 */ {(10<<2)|2,{97,115,0}},
/* 6703 */ {(10<<2)|2,{97,115,0}},
/* 6704 */ {(10<<2)|2,{97,115,0}},
/* 6705 */ {(10<<2)|2,{97,115,0}},
/* 6706 */ {(10<<2)|2,{97,115,0}},
/* 6707 */ {(10<<2)|2,{97,115,0}},
/* 6708 */ {(10<<2)|2,{97,115,0}},
/* 6709 */ {(10<<2)|2,{97,115,0}},
/* 6710 */ {(10<<2)|2,{97,115,0}},
/* 6711 */ {(10<<2)|2,{97,115,0}},
/* 6712 */ {(10<<2)|2,{97,115,0}},
/* 6713 */ {(10<<2)|2,{97,115,0}},
/* 6714 */ {(10<<2)|2,{97,115,0}},
/* 6715 */ {(10<<2)|2,{97,115,0}},
/* 6716 */ {(10<<2)|2,{97,115,0}},
/* 6717 */ {(10<<2)|2,{97,115,0}},
/* 6718 */ {(10<<2)|2,{97,115,0}},
/* 6719 */ {(10<<2)|2,{97,115,0}},
/* 6720 */ {(15<<2)|3,{97,116,48}},
/* 6721 */ {(15<<2)|3,{97,116,48}},
/* 6722 */ {(15<<2)|3,{97,116,49}},
/* 6723 */ {(15<<2)|3,{97,116,49}},
/* 6724 */ {(15<<2)|3,{97,116,50}},
/* 6725 */ {(15<<2)|3,{97,116,50}},
/* 6726 */ {(15<<2)|3,{97,116,97}},
/* 6727 */ {(15<<2)|3,{97,116,97}},
/* 6728 */ {(15<<2)|3,{97,116,99}},
/* 6729 */ {(15<<2)|3,{97,116,99}},
/* 6730 */ {(15<<2)|3,{97,116,101}},
/* 6731 */ {(15<<2)|3,{97,116,101}},
/* 6732 */ {(15<<2)|3,{97,116,105}},
/* 6733 */ {(15<<2)|3,{97,116,105}},
/* 6734 */ {(15<<2)|3,{97,116,111}},
/* 6735 */ {(15<<2)|3,{97,116,111}},
/* 6736 */ {(15<<2)|3,{97,116,115}},
/* 6737 */ {(15<<2)|3,{97,116,115}},
/* 6738 */ {(15<<2)|3,{97,116,116}},
/* 6739 */ {(15<<2)|3,{97,116,116}},
/* 6740 */ {(16<<2)|3,{97,116,32}},
/* 6741 */ {(16<<2)|3,{97,116,37}},
/* 6742 */ {(16<<2)|3,{97,116,45}},
/* 6743 */ {(16<<2)|3,{97,116,46}},
/* 6744 */ {(16<<2)|3,{97,116,47}},
/* 6745 */ {(16<<2)|3,{97,116,51}},
/* 6746 */ {(16<<2)|3,{97,116,52}},
/* 6747 */ {(16<<2)|3,{97,116,53}},
/* 6748 */ {(16<<2)|3,{97,116,54}},
/* 6749 */ {(16<<2)|3,{97,116,55}},
/* 6750 */ {(16<<2)|3,{97,116,56}},
/* 6751 */ {(16<<2)|3,{97,116,57}},
/* 6752 */ {(16<<2)|3,{97,116,61}},
/* 6753 */ {(16<<2)|3,{97,116,65}},
/* 6754 */ {(16<<2)|3,{97,116,95}},
/* 6755 */ {(16<<2)|3,{97,116,98}},
/* 6756 */ {(16<<2)|3,{97,116,100}},
/* 6757 */ {(16<<2)|3,{97,116,102}},
/* 6758 */ {(16<<2)|3,{97,116,103}},
/* 6759 */ {(16<<2)|3,{97,116,104}},
/* 6760 */ {(16<<2)|3,{97,116,108}},
/* 6761 */ {(16<<2)|3,{97,116,109}},
/* 6762 */ {(16<<2)|3,{97,116,110}},
/* 6763 */ {(16<<2)|3,{97,116,112}},
/* 6764 */ {(16<<2)|3,{97,116,114}},
/* 6765 */ {(16<<2)|3,{97,116,117}},
/* 6766 */ {(10<<2)|2,{97,116,0}},
/* 6767 */ {(10<<2)|2,{97,116,0}},
/* 6768 */ {(10<<2)|2,{97,116,0}},
/* 6769 */ {(10<<2)|2,{97,116,0}},
/* 6770 */ {(10<<2)|2,{97,116,0}},
/* 6771 */ {(10<<2)|2,{97,116,0}},
/* 6772 */ {(10<<2)|2,{97,116,0}},
/* 6773 */ {(10<<2)|2,{97,116,0}},
/* 6774 */ {(10<<2)|2,{97,116,0}},
/* 6775 */ {(10<<2)|2,{97,116,0}},
/* 6776 */ {(10<<2)|2,{97,116,0}},
/* 6777 */ {(10<<2)|2,{97,116,0}},
/* 6778 */ {(10<<2)|2,{97,116,0}},
/* 6779 */ {(10<<2)|2,{97,116,0}},
/* 6780 */ {(10<<2)|2,{97,116,0}},
/* 6781 */ {(10<<2)|2,{97,116,0}},
/* 6782 */ {(10<<2)|2,{97,116,0}},
/* 6783 */ {(10<<2)|2,{97,116,0}},
/* 6784 */ {(16<<2)|3,{97,32,48}},
/* 6785 */ {(16<<2)|3,{97,32,49}},
/* 6786 */ {(16<<2)|3,{97,32,50}},
/* 6787 */ {(16<<2)|3,{97,32,97}},
/* 6788 */ {(16<<2)|3,{97,32,99}},
/* 6789 */ {(16<<2)|3,{97,32,101}},
/* 6790 */ {(16<<2)|3,{97,32,105}},
/* 6791 */ {(16<<2)|3,{97,32,111}},
/* 6792 */ {(16<<2)|3,{97,32,115}},
/* 6793 */ {(16<<2)|3,{97,32,116}},
/* 6794 */ {(11<<2)|2,{97,32,0}},
/* 6795 */ {(11<<2)|2,{97,32,0}},
/* 6796 */ {(11<<2)|2,{97,32,0}},
/* 6797 */ {(11<<2)|2,{97,32,0}},
/* 6798 */ {(11<<2)|2,{97,32,0}},
/* 6799 */ {(11<<2)|2,{97,32,0}},
/* 6800 */ {(11<<2)|2,{97,32,0}},
/* 6801 */ {(11<<2)|2,{97,32,0}},
/* 6802 */ {(11<<2)|2,{97,32,0}},
/* 6803 */ {(11<<2)|2,{97,32,0}},
/* 6804 */ {(11<<2)|2,{97,32,0}},
/* 6805 */ {(11<<2)|2,{97,32,0}},
/* 6806 */ {(11<<2)|2,{97,32,0}},
/* 6807 */ {(11<<2)|2,{97,32,0}},
/* 6808 */ {(11<<2)|2,{97,32,0}},
/* 6809 */ {(11<<2)|2,{97,32,0}},
/* 6810 */ {(11<<2)|2,{97,32,0}},
/* 6811 */ {(11<<2)|2,{97,32,0}},
/* 6812 */ {(11<<2)|2,{97,32,0}},
/* 6813 */ {(11<<2)|2,{97,32,0}},
/* 6814 */ {(11<<2)|2,{97,32,0}},
/* 6815 */ {(11<<2)|2,{97,32,0}},
/* 6816 */ {(16<<2)|3,{97,37,48}},
/* 6817 */ {(16<<2)|3,{97,37,49}},
/* 6818 */ {(16<<2)|3,{97,37,50}},
/* 6819 */ {(16<<2)|3,{97,37,97}},
/* 6820 */ {(16<<2)|3,{97,37,99}},
/* 6821 */ {(16<<2)|3,{97,37,101}},
/* 6822 */ {(16<<2)|3,{97,37,105}},
/* 6823 */ {(16<<2)|3,{97,37,111}},
/* 6824 */ {(16<<2)|3,{97,37,115}},
/* 6825 */ {(16<<2)|3,{97,37,116}},
/* 6826 */ {(11<<2)|2,{97,37,0}},
/* 6827 */ {(11<<2)|2,{97,37,0}},
/* 6828 */ {(11<<2)|2,{97,37,0}},
/* 6829 */ {(11<<2)|2,{97,37,0}},
/* 6830 */ {(11<<2)|2,{97,37,0}},
/* 6831 */ {(11<<2)|2,{97,37,0}},
/* 6832 */ {(11<<2)|2,{97,37,0}},
/* 6833 */ {(11<<2)|2,{97,37,0}},
/* 6834 */ {(11<<2)|2,{97,37,0}},
/* 6835 */ {(11<<2)|2,{97,37,0}},
/* 6836 */ {(11<<2)|2,{97,37,0}},
/* 6837 */ {(11<<2)|2,{97,37,0}},
/* 6838 */ {(11<<2)|2,{97,37,0}},
/* 6839 */ {(11<<2)|2,{97,37,0}},
/* 6840 */ {(11<<2)|2,{97,37,0}},
/* 6841 */ {(11<<2)|2,{97,37,0}},
/* 6842 */ {(11<<2)|2,{97,37,0}},
/* 6843 */ {(11<<2)|2,{97,37,0}},
/* 6844 */ {(11<<2)|2,{97,37,0}},
/* 6845 */ {(11<<2)|2,{97,37,0}},
/* 6846 */ {(11<<2)|2,{97,37,0}},
/* 6847 */ {(11<<2)|2,{97,37,0}},
/* 6848 */ {(16<<2)|3,{97,45,48}},
/* 6849 */ {(16<<2)|3,{97,45,49}},
/* 6850 */ {(16<<2)|3,{97,45,50}},
/* 6851 */ {(16<<2)|3,{97,45,97}},
/* 6852 */ {(16<<2)|3,{97,45,99}},
/* 6853 */ {(16<<2)|3,{97,45,101}},
/* 6854 */ {(16<<2)|3,{97,45,105}},
/* 6855 */ {(16<<2)|3,{97,45,111}},
/* 6856 */ {(16<<2)|3,{97,45,115}},
/* 6857 */ {(16<<2)|3,{97,45,116}},
/* 6858 */ {(11<<2)|2,{97,45,0}},
/* 6859 */ {(11<<2)|2,{97,45,0}},
/* 6860 */ {(11<<2)|2,{97,45,0}},
/* 6861 */ {(11<<2)|2,{97,45,0}},
/* 6862 */ {(11<<2)|2,{97,45,0}},
/* 6863 */ {(11<<2)|2,{97,45,0}},
/* 6864 */ {(11<<2)|2,{97,45,0}},
/* 6865 */ {(11<<2)|2,{97,45,0}},
/* 6866 */ {(11<<2)|2,{97,45,0}},
/* 6867 */ {(11<<2)|2,{97,45,0}},
/* 6868 */ {(11<<2)|2,{97,45,0}},
/* 6869 */ {(11<<2)|2,{97,45,0}},
/* 6870 */ {(11<<2)|2,{97,45,0}},
/* 6871 */ {(11<<2)|2,{97,45,0}},
/* 6872 */ {(11<<2)|2,{97,45,0}},
/* 6873 */ {(11<<2)|2,{97,45,0}},
/* 6874 */ {(11<<2)|2,{97,45,0}},
/* 6875 */ {(11<<2)|2,{97,45,0}},
/* 6876 */ {(11<<2)|2,{97,45,0}},
/* 6877 */ {(11<<2)|2,{97,45,0}},
/* 6878 */ {(11<<2)|2,{97,45,0}},
/* 6879 */ {(11<<2)|2,{97,45,0}},
/* 6880 */ {(16<<2)|3,{97,46,48}},
/* 6881 */ {(16<<2)|3,{97,46,49}},
/* 6882 */ {(16<<2)|3,{97,46,50}},
/* 6883 */ {(16<<2)|3,{97,46,97}},
/* 6884 */ {(16<<2)|3,{97,46,99}},
/* 6885 */ {(16<<2)|3,{97,46,101}},
/* 6886 */ {(16<<2)|3,{97,46,105}},
/* 6887 */ {(16<<2)|3,{97,46,111}},
/* 6888 */ {(16<<2)|3,{97,46,115}},
/* 6889 */ {(16<<2)|3,{97,46,116}},
/* 6890 */ {(11<<2)|2,{97,46,0}},
/* 6891 */ {(11<<2)|2,{97,46,0}},
/* 6892 */ {(11<<2)|2,{97,46,0}},
/* 6893 */ {(11<<2)|2,{97,46,0}},
/* 6894 */ {(11<<2)|2,{97,46,0}},
/* 6895 */ {(11<<2)|2,{97,46,0}},
/* 6896 */ {(11<<2)|2,{97,46,0}},
/* 6897 */ {(11<<2)|2,{97,46,0}},
/* 6898 */ {(11<<2)|2,{97,46,0}},
/* 6899 */ {(11<<2)|2,{97,46,0}},
/* 6900 */ {(11<<2)|2,{97,46,0}},
/* 6901 */ {(11<<2)|2,{97,46,0}},
/* 6902 */ {(11<<2)|2,{97,46,0}},
/* 6903 */ {(11<<2)|2,{97,46,0}},
/* 6904 */ {(11<<2)|2,{97,46,0}},
/* 6905 */ {(11<<2)|2,{97,46,0}},
/* 6906 */ {(11<<2)|2,{97,46,0}},
/* 6907 */ {(11<<2)|2,{97,46,0}},
/* 6908 */ {(11<<2)|2,{97,46,0}},
/* 6909 */ {(11<<2)|2,{97,46,0}},
/* 6910 */ {(11<<2)|2,{97,46,0}},
/* 6911 */ {(11<<2)|2,{97,46,0}},
/* 6912 */ {(16<<2)|3,{97,47,48}},
/* 6913 */ {(16<<2)|3,{97,47,49}},
/* 6914 */ {(16<<2)|3,{97,47,50}},
/* 6915 */ {(16<<2)|3,{97,47,97}},
/* 6916 */ {(16<<2)|3,{97,47,99}},
/* 6917 */ {(16<<2)|3,{97,47,101}},
/* 6918 */ {(16<<2)|3,{97,47,105}},
/* 6919 */ {(16<<2)|3,{97,47,111}},
/* 6920 */ {(16<<2)|3,{97,47,115}},
/* 6921 */ {(16<<2)|3,{97,47,116}},
/* 6922 */ {(11<<2)|2,{97,47,0}},
/* 6923 */ {(11<<2)|2,{97,47,0}},
/* 6924 */ {(11<<2)|2,{97,47,0}},
/* 6925 */ {(11<<2)|2,{97,47,0}},
/* 6926 */ {(11<<2)|2,{97,47,0}},
/* 6927 */ {(11<<2)|2,{97,47,0}},
/* 6928 */ {(11<<2)|2,{97,47,0}},
/* 6929 */ {(11<<2)|2,{97,47,0}},
/* 6930 */ {(11<<2)|2,{97,47,0}},
/* 6931 */ {(11<<2)|2,{97,47,0}},
/* 6932 */ {(11<<2)|2,{97,47,0}},
/* 6933 */ {(11<<2)|2,{97,47,0}},
/* 6934 */ {(11<<2)|2,{97,47,0}},
/* 6935 */ {(11<<2)|2,{97,47,0}},
/* 6936 */ {(11<<2)|2,{97,47,0}},
/* 6937 */ {(11<<2)|2,{97,47,0}},
/* 6938 */ {(11<<2)|2,{97,47,0}},
/* 6939 */ {(11<<2)|2,{97,47,0}},
/* 6940 */ {(11<<2)|2,{97,47,0}},
/* 6941 */ {(11<<2)|2,{97,47,0}},
/* 6942 */ {(11<<2)|2,{97,47,0}},
/* 6943 */ {(11<<2)|2,{97,47,0}},
/* 6944 */ {(16<<2)|3,{97,51,48}},
/* 6945 */ {(16<<2)|3,{97,51,49}},
/* 6946 */ {(16<<2)|3,{97,51,50}},
/* 6947 */ {(16<<2)|3,{97,51,97}},
/* 6948 */ {(16<<2)|3,{97,51,99}},
/* 6949 */ {(16<<2)|3,{97,51,101}},
/* 6950 */ {(16<<2)|3,{97,51,105}},
/* 6951 */ {(16<<2)|3,{97,51,111}},
/* 6952 */ {(16<<2)|3,{97,51,115}},
/* 6953 */ {(16<<2)|3,{97,51,116}},
/* 6954 */ {(11<<2)|2,{97,51,0}},
/* 6955 */ {(11<<2)|2,{97,51,0}},
/* 6956 */ {(11<<2)|2,{97,51,0}},
/* 6957 */ {(11<<2)|2,{97,51,0}},
/* 6958 */ {(11<<2)|2,{97,51,0}},
/* 6959 */ {(11<<2)|2,{97,51,0}},
/* 6960 */ {(11<<2)|2,{97,51,0}},
/* 6961 */ {(11<<2)|2,{97,51,0}},
/* 6962 */ {(11<<2)|2,{97,51,0}},
/* 6963 */ {(11<<2)|2,{97,51,0}},
/* 6964 */ {(11<<2)|2,{97,51,0}},
/* 6965 */ {(11<<2)|2,{97,51,0}},
/* 6966 */ {(11<<2)|2,{97,51,0}},
/* 6967 */ {(11<<2)|2,{97,51,0}},
/* 6968 */ {(11<<2)|2,{97,51,0}},
/* 6969 */ {(11<<2)|2,{97,51,0}},
/* 6970 */ {(11<<2)|2,{97,51,0}},
/* 6971 */ {(11<<2)|2,{97,51,0}},
/* 6972 */ {(11<<2)|2,{97,51,0}},
/* 6973 */ {(11<<2)|2,{97,51,0}},
/* 6974 */ {(11<<2)|2,{97,51,0}},
/* 6975 */ {(11<<2)|2,{97,51,0}},
/* 6976 */ {(16<<2)|3,{97,52,48}},
/* 6977 */ {(16<<2)|3,{97,52,49}},
/* 6978 */ {(16<<2)|3,{97,52,50}},
/* 6979 */ {(16<<2)|3,{97,52,97}},
/* 6980 */ {(16<<2)|3,{97,52,99}},
/* 6981 */ {(16<<2)|3,{97,52,101}},
/* 6982 */ {(16<<2)|3,{97,52,105}},
/* 6983 */ {(16<<2)|3,{97,52,111}},
/* 6984 */ {(16<<2)|3,{97,52,115}},
/* 6985 */ {(16<<2)|3,{97,52,116}},
/* 6986 */ {(11<<2)|2,{97,52,0}},
/* 6987 */ {(11<<2)|2,{97,52,0}},
/* 6988 */ {(11<<2)|2,{97,52,0}},
/* 6989 */ {(11<<2)|2,{97,52,0}},
/* 6990 */ {(11<<2)|2,{97,52,0}},
/* 6991 */ {(11<<2)|2,{97,52,0}},
/* 6992 */ {(11<<2)|2,{97,52,0}},
/* 6993 */ {(11<<2)|2,{97,52,0}},
/* 6994 */ {(11<<2)|2,{97,52,0}},
/* 6995 */ {(11<<2)|2,{97,52,0}},
/* 6996 */ {(11<<2)|2,{97,52,0}},
/* 6997 */ {(11<<2)|2,{97,52,0}},
/* 6998 */ {(11<<2)|2,{97,52,0}},
/* 6999 */ {(11<<2)|2,{97,52,0}},
/* 7000 */ {(11<<2)|2,{97,52,0}},
/* 7001 */ {(11<<2)|2,{97,52,0}},
/* 7002 */ {(11<<2)|2,{97,52,0}},
/* 7003 */ {(11<<2)|2,{97,52,0}},
/* 7004 */ {(11<<2)|2,{97,52,0}},
/* 7005 */ {(11<<2)|2,{97,52,0}},
/* 7006 */ {(11<<2)|2,{97,52,0}},
/* 7007 */ {(11<<2)|2,{97,52,0}},
/* 7008 */ {(16<<2)|3,{97,53,48}},
/* 7009 */ {(16<<2)|3,{97,53,49}},
/* 7010 */ {(16<<2)|3,{97,53,50}},
/* 7011 */ {(16<<2)|3,{97,53,97}},
/* 7012 */ {(16<<2)|3,{97,53,99}},
/* 7013 */ {(16<<2)|3,{97,53,101}},
/* 7014 */ {(16<<2)|3,{97,53,105}},
/* 7015 */ {(16<<2)|3,{97,53,111}},
/* 7016 */ {(16<<2)|3,{97,53,115}},
/* 7017 */ {(16<<2)|3,{97,53,116}},
/* 7018 */ {(11<<2)|2,{97,53,0}},
/* 7019 */ {(11<<2)|2,{97,53,0}},
/* 7020 */ {(11<<2)|2,{97,53,0}},
/* 7021 */ {(11<<2)|2,{97,53,0}},
/* 7022 */ {(11<<2)|2,{97,53,0}},
/* 7023 */ {(11<<2)|2,{97,53,0}},
/* 7024 */ {(11<<2)|2,{97,53,0}},
/* 7025 */ {(11<<2)|2,{97,53,0}},
/* 7026 */ {(11<<2)|2,{97,53,0}},
/* 7027 */ {(11<<2)|2,{97,53,0}},
/* 7028 */ {(11<<2)|2,{97,53,0}},
/* 7029 */ {(11<<2)|2,{97,53,0}},
/* 7030 */ {(11<<2)|2,{97,53,0}},
/* 7031 */ {(11<<2)|2,{97,53,0}},
/* 7032 */ {(11<<2)|2,{97,53,0}},
/* 7033 */ {(11<<2)|2,{97,53,0}},
/* 7034 */ {(11<<2)|2,{97,53,0}},
/* 7035 */ {(11<<2)|2,{97,53,0}},
/* 7036 */ {(11<<2)|2,{97,53,0}},
/* 7037 */ {(11<<2)|2,{97,53,0}},
/* 7038 */ {(11<<2)|2,{97,53,0}},
/* 7039 */ {(11<<2)|2,{97,53,0}},
/* 7040 */ {(16<<2)|3,{97,54,48}},
/* 7041 */ {(16<<2)|3,{97,54,49}},
/* 7042 */ {(16<<2)|3,{97,54,50}},
/* 7043 */ {(16<<2)|3,{97,54,97}},
/* 7044 */ {(16<<2)|3,{97,54,99}},
/* 7045 */ {(16<<2)|3,{97,54,101}},
/* 7046 */ {(16<<2)|3,{97,54,105}},
/* 7047 */ {(16<<2)|3,{97,54,111}},
/* 7048 */ {(16<<2)|3,{97,54,115}},
/* 7049 */ {(16<<2)|3,{97,54,116}},
/* 7050 */ {(11<<2)|2,{97,54,0}},
/* 7051 */ {(11<<2)|2,{97,54,0}},
/* 7052 */ {(11<<2)|2,{97,54,0}},
/* 7053 */ {(11<<2)|2,{97,54,0}},
/* 7054 */ {(11<<2)|2,{97,54,0}},
/* 7055 */ {(11<<2)|2,{97,54,0}},
/* 7056 */ {(11<<2)|2,{97,54,0}},
/* 7057 */ {(11<<2)|2,{97,54,0}},
/* 7058 */ {(11<<2)|2,{97,54,0}},
/* 7059 */ {(11<<2)|2,{97,54,0}},
/* 7060 */ {(11<<2)|2,{97,54,0}},
/* 7061 */ {(11<<2)|2,{97,54,0}},
/* 7062 */ {(11<<2)|2,{97,54,0}},
/* 7063 */ {(11<<2)|2,{97,54,0}},
/* 7064 */ {(11<<2)|2,{97,54,0}},
/* 7065 */ {(11<<2)|2,{97,54,0}},
/* 7066 */ {(11<<2)|2,{97,54,0}},
/* 7067 */ {(11<<2)|2,{97,54,0}},
/* 7068 */ {(11<<2)|2,{97,54,0}},
/* 7069 */ {(11<<2)|2,{97,54,0}},
/* 7070 */ {(11<<2)|2,{97,54,0}},
/* 7071 */ {(11<<2)|2,{97,54,0}},
/* 7072 */ {(16<<2)|3,{97,55,48}},
/* 7073 */ {(16<<2)|3,{97,55,49}},
/* 7074 */ {(16<<2)|3,{97,55,50}},
/* 7075 */ {(16<<2)|3,{97,55,97}},
/* 7076 */ {(16<<2)|3,{97,55,99}},
/* 7077 */ {(16<<2)|3,{97,55,101}},
/* 7078 */ {(16<<2)|3,{97,55,105}},
/* 7079 */ {(16<<2)|3,{97,55,111}},
/* 7080 */ {(16<<2)|3,{97,55,115}},
/* 7081 */ {(16<<2)|3,{97,55,116}},
/* 7082 */ {(11<<2)|2,{97,55,0}},
/* 7083 */ {(11<<2)|2,{97,55,0}},
/* 7084 */ {(11<<2)|2,{97,55,0}},
/* 7085 */ {(11<<2)|2,{97,55,0}},
/* 7086 */ {(11<<2)|2,{97,55,0}},
/* 7087 */ {(11<<2)|2,{97,55,0}},
/* 7088 */ {(11<<2)|2,{97,55,0}},
/* 7089 */ {(11<<2)|2,{97,55,0}},
/* 7090 */ {(11<<2)|2,{97,55,0}},
/* 7091 */ {(11<<2)|2,{97,55,0}},
/* 7092 */ {(11<<2)|2,{97,55,0}},
/* 7093 */ {(11<<2)|2,{97,55,0}},
/* 7094 */ {(11<<2)|2,{97,55,0}},
/* 7095 */ {(11<<2)|2,{97,55,0}},
/* 7096 */ {(11<<2)|2,{97,55,0}},
/* 7097 */ {(11<<2)|2,{97,55,0}},
/* 7098 */ {(11<<2)|2,{97,55,0}},
/* 7099 */ {(11<<2)|2,{97,55,0}},
/* 7100 */ {(11<<2)|2,{97,55,0}},
/* 7101 */ {(11<<2)|2,{97,55,0}},
/* 7102 */ {(11<<2)|2,{97,55,0}},
/* 7103 */ {(11<<2)|2,{97,55,0}},
/* 7104 */ {(16<<2)|3,{97,56,48}},
/* 7105 */ {(16<<2)|3,{97,56,49}},
/* 7106 */ {(16<<2)|3,{97,56,50}},
/* 7107 */ {(16<<2)|3,{97,56,97}},
/* 7108 */ {(16<<2)|3,{97,56,99}},
/* 7109 */ {(16<<2)|3,{97,56,101}},
/* 7110 */ {(16<<2)|3,{97,56,105}},
/* 7111 */ {(16<<2)|3,{97,56,111}},
/* 7112 */ {(16<<2)|3,{97,56,115}},
/* 7113 */ {(16<<2)|3,{97,56,116}},
/* 7114 */ {(11<<2)|2,{97,56,0}},
/* 7115 */ {(11<<2)|2,{97,56,0}},
/* 7116 */ {(11<<2)|2,{97,56,0}},
/* 7117 */ {(11<<2)|2,{97,56,0}},
/* 7118 */ {(11<<2)|2,{97,56,0}},
/* 7119 */ {(11<<2)|2,{97,56,0}},
/* 7120 */ {(11<<2)|2,{97,56,0}},
/* 7121 */ {(11<<2)|2,{97,56,0}},
/* 7122 */ {(11<<2)|2,{97,56,0}},
/* 7123 */ {(11<<2)|2,{97,56,0}},
/* 7124 */ {(11<<2)|2,{97,56,0}},
/* 7125 */ {(11<<2)|2,{97,56,0}},
/* 7126 */ {(11<<2)|2,{97,56,0}},
/* 7127 */ {(11<<2)|2,{97,56,0}},
/* 7128 */ {(11<<2)|2,{97,56,0}},
/* 7129 */ {(11<<2)|2,{97,56,0}},
/* 7130 */ {(11<<2)|2,{97,56,0}},
/* 7131 */ {(11<<2)|2,{97,56,0}},
/* 7132 */ {(11<<2)|2,{97,56,0}},
/* 7133 */ {(11<<2)|2,{97,56,0}},
/* 7134 */ {(11<<2)|2,{97,56,0}},
/* 7135 */ {(11<<2)|2,{97,56,0}},
/* 7136 */ {(16<<2)|3,{97,57,48}},
/* 7137 */ {(16<<2)|3,{97,57,49}},
/* 7138 */ {(16<<2)|3,{97,57,50}},
/* 7139 */ {(16<<2)|3,{97,57,97}},
/* 7140 */ {(16<<2)|3,{97,57,99}},
/* 7141 */ {(16<<2)|3,{97,57,101}},
/* 7142 */ {(16<<2)|3,{97,57,105}},
/* 7143 */ {(16<<2)|3,{97,57,111}},
/* 7144 */ {(16<<2)|3,{97,57,115}},
/* 7145 */ {(16<<2)|3,{97,57,116}},
/* 7146 */ {(11<<2)|2,{97,57,0}},
/* 7147 */ {(11<<2)|2,{97,57,0}},
/* 7148 */ {(11<<2)|2,{97,57,0}},
/* 7149 */ {(11<<2)|2,{97,57,0}},
/* 7150 */ {(11<<2)|2,{97,57,0}},
/* 7151 */ {(11<<2)|2,{97,57,0}},
/* 7152 */ {(11<<2)|2,{97,57,0}},
/* 7153 */ {(11<<2)|2,{97,57,0}},
/* 7154 */ {(11<<2)|2,{97,57,0}},
/* 7155 */ {(11<<2)|2,{97,57,0}},
/* 7156 */ {(11<<2)|2,{97,57,0}},
/* 7157 */ {(11<<2)|2,{97,57,0}},
/* 7158 */ {(11<<2)|2,{97,57,0}},
/* 7159 */ {(11<<2)|2,{97,57,0}},
/* 7160 */ {(11<<2)|2,{97,57,0}},
/* 7161 */ {(11<<2)|2,{97,57,0}},
/* 7162 */ {(11<<2)|2,{97,57,0}},
/* 7163 */ {(11<<2)|2,{97,57,0}},
/* 7164 */ {(11<<2)|2,{97,57,0}},
/* 7165 */ {(11<<2)|2,{97,57,0}},
/* 7166 */ {(11<<2)|2,{97,57,0}},
/* 7167 */ {(11<<2)|2,{97,57,0}},
/* 7168 */ {(16<<2)|3,{97,61,48}},
/* 7169 */ {(16<<2)|3,{97,61,49}},
/* 7170 */ {(16<<2)|3,{97,61,50}},
/* 7171 */ {(16<<2)|3,{97,61,97}},
/* 7172 */ {(16<<2)|3,{97,61,99}},
/* 7173 */ {(16<<2)|3,{97,61,101}},
/* 7174 */ {(16<<2)|3,{97,61,105}},
/* 7175 */ {(16<<2)|3,{97,61,111}},
/* 7176 */ {(16<<2)|3,{97,61,115}},
/* 7177 */ {(16<<2)|3,{97,61,116}},
/* 7178 */ {(11<<2)|2,{97,61,0}},
/* 7179 */ {(11<<2)|2,{97,61,0}},
/* 7180 */ {(11<<2)|2,{97,61,0}},
/* 7181 */ {(11<<2)|2,{97,61,0}},
/* 7182 */ {(11<<2)|2,{97,61,0}},
/* 7183 */ {(11<<2)|2,{97,61,0}},
/* 7184 */ {(11<<2)|2,{97,61,0}},
/* 7185 */ {(11<<2)|2,{97,61,0}},
/* 7186 */ {(11<<2)|2,{97,61,0}},
/* 7187 */ {(11<<2)|2,{97,61,0}},
/* 7188 */ {(11<<2)|2,{97,61,0}},
/* 7189 */ {(11<<2)|2,{97,61,0}},
/* 7190 */ {(11<<2)|2,{97,61,0}},
/* 7191 */ {(11<<2)|2,{97,61,0}},
/* 7192 */ {(11<<2)|2,{97,61,0}},
/* 7193 */ {(11<<2)|2,{97,61,0}},
/* 7194 */ {(11<<2)|2,{97,61,0}},
/* 7195 */ {(11<<2)|2,{97,61,0}},
/* 7196 */ {(11<<2)|2,{97,61,0}},
/* 7197 */ {(11<<2)|2,{97,61,0}},
/* 7198 */ {(11<<2)|2,{97,61,0}},
/* 7199 */ {(11<<2)|2,{97,61,0}},
/* 7200 */ {(16<<2)|3,{97,65,48}},
/* 7201 */ {(16<<2)|3,{97,65,49}},
/* 7202 */ {(16<<2)|3,{97,65,50}},
/* 7203 */ {(16<<2)|3,{97,65,97}},
/* 7204 */ {(16<<2)|3,{97,65,99}},
/* 7205 */ {(16<<2)|3,{97,65,101}},
/* 7206 */ {(16<<2)|3,{97,65,105}},
/* 7207 */ {(16<<2)|3,{97,65,111}},
/* 7208 */ {(16<<2)|3,{97,65,115}},
/* 7209 */ {(16<<2)|3,{97,65,116}},
/* 7210 */ {(11<<2)|2,{97,65,0}},
/* 7211 */ {(11<<2)|2,{97,65,0}},
/* 7212 */ {(11<<2)|2,{97,65,0}},
/* 7213 */ {(11<<2)|2,{97,65,0}},
/* 7214 */ {(11<<2)|2,{97,65,0}},
/* 7215 */ {(11<<2)|2,{97,65,0}},
/* 7216 */ {(11<<2)|2,{97,65,0}},
/* 7217 */ {(11<<2)|2,{97,65,0}},
/* 7218 */ {(11<<2)|2,{97,65,0}},
/* 7219 */ {(11<<2)|2,{97,65,0}},
/* 7220 */ {(11<<2)|2,{97,65,0}},
/* 7221 */ {(11<<2)|2,{97,65,0}},
/* 7222 */ {(11<<2)|2,{97,65,0}},
/* 7223 */ {(11<<2)|2,{97,65,0}},
/* 7224 */ {(11<<2)|2,{97,65,0}},
/* 7225 */ {(11<<2)|2,{97,65,0}},
/* 7226 */ {(11<<2)|2,{97,65,0}},
/* 7227 */ {(11<<2)|2,{97,65,0}},
/* 7228 */ {(11<<2)|2,{97,65,0}},
/* 7229 */ {(11<<2)|2,{97,65,0}},
/* 7230 */ {(11<<2)|2,{97,65,0}},
/* 7231 */ {(11<<2)|2,{97,65,0}},
/* 7232 */ {(16<<2)|3,{97,95,48}},
/* 7233 */ {(16<<2)|3,{97,95,49}},
/* 7234 */ {(16<<2)|3,{97,95,50}},
/* 7235 */ {(16<<2)|3,{97,95,97}},
/* 7236 */ {(16<<2)|3,{97,95,99}},
/* 7237 */ {(16<<2)|3,{97,95,101}},
/* 7238 */ {(16<<2)|3,{97,95,105}},
/* 7239 */ {(16<<2)|3,{97,95,111}},
/* 7240 */ {(16<<2)|3,{97,95,115}},
/* 7241 */ {(16<<2)|3,{97,95,116}},
/* 7242 */ {(11<<2)|2,{97,95,0}},
/* 7243 */ {(11<<2)|2,{97,95,0}},
/* 7244 */ {(11<<2)|2,{97,95,0}},
/* 7245 */ {(11<<2)|2,{97,95,0}},
/* 7246 */ {(11<<2)|2,{97,95,0}},
/* 7247 */ {(11<<2)|2,{97,95,0}},
/* 7248 */ {(11<<2)|2,{97,95,0}},
/* 7249 */ {(11<<2)|2,{97,95,0}},
/* 7250 */ {(11<<2)|2,{97,95,0}},
/* 7251 */ {(11<<2)|2,{97,95,0}},
/* 7252 */ {(11<<2)|2,{97,95,0}},
/* 7253 */ {(11<<2)|2,{97,95,0}},
/* 7254 */ {(11<<2)|2,{97,95,0}},
/* 7255 */ {(11<<2)|2,{97,95,0}},
/* 7256 */ {(11<<2)|2,{97,95,0}},
/* 7257 */ {(11<<2)|2,{97,95,0}},
/* 7258 */ {(11<<2)|2,{97,95,0}},
/* 7259 */ {(11<<2)|2,{97,95,0}},
/* 7260 */ {(11<<2)|2,{97,95,0}},
/* 7261 */ {(11<<2)|2,{97,95,0}},
/* 7262 */ {(11<<2)|2,{97,95,0}},
/* 7263 */ {(11<<2)|2,{97,95,0}},
/* 7264 */ {(16<<2)|3,{97,98,48}},
/* 7265 */ {(16<<2)|3,{97,98,49}},
/* 7266 */ {(16<<2)|3,{97,98,50}},
/* 7267 */ {(16<<2)|3,{97,98,97}},
/* 7268 */ {(16<<2)|3,{97,98,99}},
/* 7269 */ {(16<<2)|3,{97,98,101}},
/* 7270 */ {(16<<2)|3,{97,98,105}},
/* 7271 */ {(16<<2)|3,{97,98,111}},
/* 7272 */ {(16<<2)|3,{97,98,115}},
/* 7273 */ {(16<<2)|3,{97,98,116}},
/* 7274 */ {(11<<2)|2,{97,98,0}},
/* 7275 */ {(11<<2)|2,{97,98,0}},
/* 7276 */ {(11<<2)|2,{97,98,0}},
/* 7277 */ {(11<<2)|2,{97,98,0}},
/* 7278 */ {(11<<2)|2,{97,98,0}},
/* 7279 */ {(11<<2)|2,{97,98,0}},
/* 7280 */ {(11<<2)|2,{97,98,0}},
/* 7281 */ {(11<<2)|2,{97,98,0}},
/* 7282 */ {(11<<2)|2,{97,98,0}},
/* 7283 */ {(11<<2)|2,{97,98,0}},
/* 7284 */ {(11<<2)|2,{97,98,0}},
/* 7285 */ {(11<<2)|2,{97,98,0}},
/* 7286 */ {(11<<2)|2,{97,98,0}},
/* 7287 */ {(11<<2)|2,{97,98,0}},
/* 7288 */ {(11<<2)|2,{97,98,0}},
/* 7289 */ {(11<<2)|2,{97,98,0}},
/* 7290 */ {(11<<2)|2,{97,98,0}},
/* 7291 */ {(11<<2)|2,{97,98,0}},
/* 7292 */ {(11<<2)|2,{97,98,0}},
/* 7293 */ {(11<<2)|2,{97,98,0}},
/* 7294 */ {(11<<2)|2,{97,98,0}},
/* 7295 */ {(11<<2)|2,{97,98,0}},
/* 7296 */ {(16<<2)|3,{97,100,48}},
/* 7297 */ {(16<<2)|3,{97,100,49}},
/* 7298 */ {(16<<2)|3,{97,100,50}},
/* 7299 */ {(16<<2)|3,{97,100,97}},
/* 7300 */ {(16<<2)|3,{97,100,99}},
/* 7301 */ {(16<<2)|3,{97,100,101}},
/* 7302 */ {(16<<2)|3,{97,100,105}},
/* 7303 */ {(16<<2)|3,{97,100,111}},
/* 7304 */ {(16<<2)|3,{97,100,115}},
/* 7305 */ {(16<<2)|3,{97,100,116}},
/* 7306 */ {(11<<2)|2,{97,100,0}},
/* 7307 */ {(11<<2)|2,{97,100,0}},
/* 7308 */ {(11<<2)|2,{97,100,0}},
/* 7309 */ {(11<<2)|2,{97,100,0}},
/* 7310 */ {(11<<2)|2,{97,100,0}},
/* 7311 */ {(11<<2)|2,{97,100,0}},
/* 7312 */ {(11<<2)|2,{97,100,0}},
/* 7313 */ {(11<<2)|2,{97,100,0}},
/* 7314 */ {(11<<2)|2,{97,100,0}},
/* 7315 */ {(11<<2)|2,{97,100,0}},
/* 7316 */ {(11<<2)|2,{97,100,0}},
/* 7317 */ {(11<<2)|2,{97,100,0}},
/* 7318 */ {(11<<2)|2,{97,100,0}},
/* 7319 */ {(11<<2)|2,{97,100,0}},
/* 7320 */ {(11<<2)|2,{97,100,0}},
/* 7321 */ {(11<<2)|2,{97,100,0}},
/* 7322 */ {(11<<2)|2,{97,100,0}},
/* 7323 */ {(11<<2)|2,{97,100,0}},
/* 7324 */ {(11<<2)|2,{97,100,0}},
/* 7325 */ {(11<<2)|2,{97,100,0}},
/* 7326 */ {(11<<2)|2,{97,100,0}},
/* 7327 */ {(11<<2)|2,{97,100,0}},
/* 7328 */ {(16<<2)|3,{97,102,48}},
/* 7329 */ {(16<<2)|3,{97,102,49}},
/* 7330 */ {(16<<2)|3,{97,102,50}},
/* 7331 */ {(16<<2)|3,{97,102,97}},
/* 7332 */ {(16<<2)|3,{97,102,99}},
/* 7333 */ {(16<<2)|3,{97,102,101}},
/* 7334 */ {(16<<2)|3,{97,102,105}},
/* 7335 */ {(16<<2)|3,{97,102,111}},
/* 7336 */ {(16<<2)|3,{97,102,115}},
/* 7337 */ {(16<<2)|3,{97,102,116}},
/* 7338 */ {(11<<2)|2,{97,102,0}},
/* 7339 */ {(11<<2)|2,{97,102,0}},
/* 7340 */ {(11<<2)|2,{97,102,0}},
/* 7341 */ {(11<<2)|2,{97,102,0}},
/* 7342 */ {(11<<2)|2,{97,102,0}},
/* 7343 */ {(11<<2)|2,{97,102,0}},
/* 7344 */ {(11<<2)|2,{97,102,0}},
/* 7345 */ {(11<<2)|2,{97,102,0}},
/* 7346 */ {(11<<2)|2,{97,102,0}},
/* 7347 */ {(11<<2)|2,{97,102,0}},
/* 7348 */ {(11<<2)|2,{97,102,0}},
/* 7349 */ {(11<<2)|2,{97,102,0}},
/* 7350 */ {(11<<2)|2,{97,102,0}},
/* 7351 */ {(11<<2)|2,{97,102,0}},
/* 7352 */ {(11<<2)|2,{97,102,0}},
/* 7353 */ {(11<<2)|2,{97,102,0}},
/* 7354 */ {(11<<2)|2,{97,102,0}},
/* 7355 */ {(11<<2)|2,{97,102,0}},
/* 7356 */ {(11<<2)|2,{97,102,0}},
/* 7357 */ {(11<<2)|2,{97,102,0}},
/* 7358 */ {(11<<2)|2,{97,102,0}},
/* 7359 */ {(11<<2)|2,{97,102,0}},
/* 7360 */ {(16<<2)|3,{97,103,48}},
/* 7361 */ {(16<<2)|3,{97,103,49}},
/* 7362 */ {(16<<2)|3,{97,103,50}},
/* 7363 */ {(16<<2)|3,{97,103,97}},
/* 7364 */ {(16<<2)|3,{97,103,99}},
/* 7365 */ {(16<<2)|3,{97,103,101}},
/* 7366 */ {(16<<2)|3,{97,103,105}},
/* 7367 */ {(16<<2)|3,{97,103,111}},
/* 7368 */ {(16<<2)|3,{97,103,115}},
/* 7369 */ {(16<<2)|3,{97,103,116}},
/* 7370 */ {(11<<2)|2,{97,103,0}},
/* 7371 */ {(11<<2)|2,{97,103,0}},
/* 7372 */ {(11<<2)|2,{97,103,0}},
/* 7373 */ {(11<<2)|2,{97,103,0}},
/* 7374 */ {(11<<2)|2,{97,103,0}},
/* 7375 */ {(11<<2)|2,{97,103,0}},
/* 7376 */ {(11<<2)|2,{97,103,0}},
/* 7377 */ {(11<<2)|2,{97,103,0}},
/* 7378 */ {(11<<2)|2,{97,103,0}},
/* 7379 */ {(11<<2)|2,{97,103,0}},
/* 7380 */ {(11<<2)|2,{97,103,0}},
/* 7381 */ {(11<<2)|2,{97,103,0}},
/* 7382 */ {(11<<2)|2,{97,103,0}},
/* 7383 */ {(11<<2)|2,{97,103,0}},
/* 7384 */ {(11<<2)|2,{97,103,0}},
/* 7385 */ {(11<<2)|2,{97,103,0}},
/* 7386 */ {(11<<2)|2,{97,103,0}},
/* 7387 */ {(11<<2)|2,{97,103,0}},
/* 7388 */ {(11<<2)|2,{97,103,0}},
/* 7389 */ {(11<<2)|2,{97,103,0}},
/* 7390 */ {(11<<2)|2,{97,103,0}},
/* 7391 */ {(11<<2)|2,{97,103,0}},
/* 7392 */ {(16<<2)|3,{97,104,48}},
/* 7393 */ {(16<<2)|3,{97,104,49}},
/* 7394 */ {(16<<2)|3,{97,104,50}},
/* 7395 */ {(16<<2)|3,{97,104,97}},
/* 7396 */ {(16<<2)|3,{97,104,99}},
/* 7397 */ {(16<<2)|3,{97,104,101}},
/* 7398 */ {(16<<2)|3,{97,104,105}},
/* 7399 */ {(16<<2)|3,{97,104,111}},
/* 7400 */ {(16<<2)|3,{97,104,115}},
/* 7401 */ {(16<<2)|3,{97,104,116}},
/* 7402 */ {(11<<2)|2,{97,104,0}},
/* 7403 */ {(11<<2)|2,{97,104,0}},
/* 7404 */ {(11<<2)|2,{97,104,0}},
/* 7405 */ {(11<<2)|2,{97,104,0}},
/* 7406 */ {(11<<2)|2,{97,104,0}},
/* 7407 */ {(11<<2)|2,{97,104,0}},
/* 7408 */ {(11<<2)|2,{97,104,0}},
/* 7409 */ {(11<<2)|2,{97,104,0}},
/* 7410 */ {(11<<2)|2,{97,104,0}},
/* 7411 */ {(11<<2)|2,{97,104,0}},
/* 7412 */ {(11<<2)|2,{97,104,0}},
/* 7413 */ {(11<<2)|2,{97,104,0}},
/* 7414 */ {(11<<2)|2,{97,104,0}},
/* 7415 */ {(11<<2)|2,{97,104,0}},
/* 7416 */ {(11<<2)|2,{97,104,0}},
/* 7417 */ {(11<<2)|2,{97,104,0}},
/* 7418 */ {(11<<2)|2,{97,104,0}},
/* 7419 */ {(11<<2)|2,{97,104,0}},
/* 7420 */ {(11<<2)|2,{97,104,0}},
/* 7421 */ {(11<<2)|2,{97,104,0}},
/* 7422 */ {(11<<2)|2,{97,104,0}},
/* 7423 */ {(11<<2)|2,{97,104,0}},
/* 7424 */ {(16<<2)|3,{97,108,48}},
/* 7425 */ {(16<<2)|3,{97,108,49}},
/* 7426 */ {(16<<2)|3,{97,108,50}},
/* 7427 */ {(16<<2)|3,{97,108,97}},
/* 7428 */ {(16<<2)|3,{97,108,99}},
/* 7429 */ {(16<<2)|3,{97,108,101}},
/* 7430 */ {(16<<2)|3,{97,108,105}},
/* 7431 */ {(16<<2)|3,{97,108,111}},
/* 7432 */ {(16<<2)|3,{97,108,115}},
/* 7433 */ {(16<<2)|3,{97,108,116}},
/* 7434 */ {(11<<2)|2,{97,108,0}},
/* 7435 */ {(11<<2)|2,{97,108,0}},
/* 7436 */ {(11<<2)|2,{97,108,0}},
/* 7437 */ {(11<<2)|2,{97,108,0}},
/* 7438 */ {(11<<2)|2,{97,108,0}},
/* 7439 */ {(11<<2)|2,{97,108,0}},
/* 7440 */ {(11<<2)|2,{97,108,0}},
/* 7441 */ {(11<<2)|2,{97,108,0}},
/* 7442 */ {(11<<2)|2,{97,108,0}},
/* 7443 */ {(11<<2)|2,{97,108,0}},
/* 7444 */ {(11<<2)|2,{97,108,0}},
/* 7445 */ {(11<<2)|2,{97,108,0}},
/* 7446 */ {(11<<2)|2,{97,108,0}},
/* 7447 */ {(11<<2)|2,{97,108,0}},
/* 7448 */ {(11<<2)|2,{97,108,0}},
/* 7449 */ {(11<<2)|2,{97,108,0}},
/* 7450 */ {(11<<2)|2,{97,108,0}},
/* 7451 */ {(11<<2)|2,{97,108,0}},
/* 7452 */ {(11<<2)|2,{97,108,0}},
/* 7453 */ {(11<<2)|2,{97,108,0}},
/* 7454 */ {(11<<2)|2,{97,108,0}},
/* 7455 */ {(11<<2)|2,{97,108,0}},
/* 7456 */ {(16<<2)|3,{97,109,48}},
/* 7457 */ {(16<<2)|3,{97,109,49}},
/* 7458 */ {(16<<2)|3,{97,109,50}},
/* 7459 */ {(16<<2)|3,{97,109,97}},
/* 7460 */ {(16<<2)|3,{97,109,99}},
/* 7461 */ {(16<<2)|3,{97,109,101}},
/* 7462 */ {(16<<2)|3,{97,109,105}},
/* 7463 */ {(16<<2)|3,{97,109,111}},
/* 7464 */ {(16<<2)|3,{97,109,115}},
/* 7465 */ {(16<<2)|3,{97,109,116}},
/* 7466 */ {(11<<2)|2,{97,109,0}},
/* 7467 */ {(11<<2)|2,{97,109,0}},
/* 7468 */ {(11<<2)|2,{97,109,0}},
/* 7469 */ {(11<<2)|2,{97,109,0}},
/* 7470 */ {(11<<2)|2,{97,109,0}},
/* 7471 */ {(11<<2)|2,{97,109,0}},
/* 7472 */ {(11<<2)|2,{97,109,0}},
/* 7473 */ {(11<<2)|2,{97,109,0}},
/* 7474 */ {(11<<2)|2,{97,109,0}},
/* 7475 */ {(11<<2)|2,{97,109,0}},
/* 7476 */ {(11<<2)|2,{97,109,0}},
/* 7477 */ {(11<<2)|2,{97,109,0}},
/* 7478 */ {(11<<2)|2,{97,109,0}},
/* 7479 */ {(11<<2)|2,{97,109,0}},
/* 7480 */ {(11<<2)|2,{97,109,0}},
/* 7481 */ {(11<<2)|2,{97,109,0}},
/* 7482 */ {(11<<2)|2,{97,109,0}},
/* 7483 */ {(11<<2)|2,{97,109,0}},
/* 7484 */ {(11<<2)|2,{97,109,0}},
/* 7485 */ {(11<<2)|2,{97,109,0}},
/* 7486 */ {(11<<2)|2,{97,109,0}},
/* 7487 */ {(11<<2)|2,{97,109,0}},
/* 7488 */ {(16<<2)|3,{97,110,48}},
/* 7489 */ {(16<<2)|3,{97,110,49}},
/* 7490 */ {(16<<2)|3,{97,110,50}},
/* 7491 */ {(16<<2)|3,{97,110,97}},
/* 7492 */ {(16<<2)|3,{97,110,99}},
/* 7493 */ {(16<<2)|3,{97,110,101}},
/* 7494 */ {(16<<2)|3,{97,110,105}},
/* 7495 */ {(16<<2)|3,{97,110,111}},
/* 7496 */ {(16<<2)|3,{97,110,115}},
/* 7497 */ {(16<<2)|3,{97,110,116}},
/* 7498 */ {(11<<2)|2,{97,110,0}},
/* 7499 */ {(11<<2)|2,{97,110,0}},
/* 7500 */ {(11<<2)|2,{97,110,0}},
/* 7501 */ {(11<<2)|2,{97,110,0}},
/* 7502 */ {(11<<2)|2,{97,110,0}},
/* 7503 */ {(11<<2)|2,{97,110,0}},
/* 7504 */ {(11<<2)|2,{97,110,0}},
/* 7505 */ {(11<<2)|2,{97,110,0}},
/* 7506 */ {(11<<2)|2,{97,110,0}},
/* 7507 */ {(11<<2)|2,{97,110,0}},
/* 7508 */ {(11<<2)|2,{97,110,0}},
/* 7509 */ {(11<<2)|2,{97,110,0}},
/* 7510 */ {(11<<2)|2,{97,110,0}},
/* 7511 */ {(11<<2)|2,{97,110,0}},
/* 7512 */ {(11<<2)|2,{97,110,0}},
/* 7513 */ {(11<<2)|2,{97,110,0}},
/* 7514 */ {(11<<2)|2,{97,110,0}},
/* 7515 */ {(11<<2)|2,{97,110,0}},
/* 7516 */ {(11<<2)|2,{97,110,0}},
/* 7517 */ {(11<<2)|2,{97,110,0}},
/* 7518 */ {(11<<2)|2,{97,110,0}},
/* 7519 */ {(11<<2)|2,{97,110,0}},
/* 7520 */ {(16<<2)|3,{97,112,48}},
/* 7521 */ {(16<<2)|3,{97,112,49}},
/* 7522 */ {(16<<2)|3,{97,112,50}},
/* 7523 */ {(16<<2)|3,{97,112,97}},
/* 7524 */ {(16<<2)|3,{97,112,99}},
/* 7525 */ {(16<<2)|3,{97,112,101}},
/* 7526 */ {(16<<2)|3,{97,112,105}},
/* 7527 */ {(16<<2)|3,{97,112,111}},
/* 7528 */ {(16<<2)|3,{97,112,115}},
/* 7529 */ {(16<<2)|3,{97,112,116}},
/* 7530 */ {(11<<2)|2,{97,112,0}},
/* 7531 */ {(11<<2)|2,{97,112,0}},
/* 7532 */ {(11<<2)|2,{97,112,0}},
/* 7533 */ {(11<<2)|2,{97,112,0}},
/* 7534 */ {(11<<2)|2,{97,112,0}},
/* 7535 */ {(11<<2)|2,{97,112,0}},
/* 7536 */ {(11<<2)|2,{97,112,0}},
/* 7537 */ {(11<<2)|2,{97,112,0}},
/* 7538 */ {(11<<2)|2,{97,112,0}},
/* 7539 */ {(11<<2)|2,{97,112,0}},
/* 7540 */ {(11<<2)|2,{97,112,0}},
/* 7541 */ {(11<<2)|2,{97,112,0}},
/* 7542 */ {(11<<2)|2,{97,112,0}},
/* 7543 */ {(11<<2)|2,{97,112,0}},
/* 7544 */ {(11<<2)|2,{97,112,0}},
/* 7545 */ {(11<<2)|2,{97,112,0}},
/* 7546 */ {(11<<2)|2,{97,112,0}},
/* 7547 */ {(11<<2)|2,{97,112,0}},
/* 7548 */ {(11<<2)|2,{97,112,0}},
/* 7549 */ {(11<<2)|2,{97,112,0}},
/* 7550 */ {(11<<2)|2,{97,112,0}},
/* 7551 */ {(11<<2)|2,{97,112,0}},
/* 7552 */ {(16<<2)|3,{97,114,48}},
/* 7553 */ {(16<<2)|3,{97,114,49}},
/* 7554 */ {(16<<2)|3,{97,114,50}},
/* 7555 */ {(16<<2)|3,{97,114,97}},
/* 7556 */ {(16<<2)|3,{97,114,99}},
/* 7557 */ {(16<<2)|3,{97,114,101}},
/* 7558 */ {(16<<2)|3,{97,114,105}},
/* 7559 */ {(16<<2)|3,{97,114,111}},
/* 7560 */ {(16<<2)|3,{97,114,115}},
/* 7561 */ {(16<<2)|3,{97,114,116}},
/* 7562 */ {(11<<2)|2,{97,114,0}},
/* 7563 */ {(11<<2)|2,{97,114,0}},
/* 7564 */ {(11<<2)|2,{97,114,0}},
/* 7565 */ {(11<<2)|2,{97,114,0}},
/* 7566 */ {(11<<2)|2,{97,114,0}},
/* 7567 */ {(11<<2)|2,{97,114,0}},
/* 7568 */ {(11<<2)|2,{97,114,0}},
/* 7569 */ {(11<<2)|2,{97,114,0}},
/* 7570 */ {(11<<2)|2,{97,114,0}},
/* 7571 */ {(11<<2)|2,{97,114,0}},
/* 7572 */ {(11<<2)|2,{97,114,0}},
/* 7573 */ {(11<<2)|2,{97,114,0}},
/* 7574 */ {(11<<2)|2,{97,114,0}},
/* 7575 */ {(11<<2)|2,{97,114,0}},
/* 7576 */ {(11<<2)|2,{97,114,0}},
/* 7577 */ {(11<<2)|2,{97,114,0}},
/* 7578 */ {(11<<2)|2,{97,114,0}},
/* 7579 */ {(11<<2)|2,{97,114,0}},
/* 7580 */ {(11<<2)|2,{97,114,0}},
/* 7581 */ {(11<<2)|2,{97,114,0}},
/* 7582 */ {(11<<2)|2,{97,114,0}},
/* 7583 */ {(11<<2)|2,{97,114,0}},
/* 7584 */ {(16<<2)|3,{97,117,48}},
/* 7585 */ {(16<<2)|3,{97,117,49}},
/* 7586 */ {(16<<2)|3,{97,117,50}},
/* 7587 */ {(16<<2)|3,{97,117,97}},
/* 7588 */ {(16<<2)|3,{97,117,99}},
/* 7589 */ {(16<<2)|3,{97,117,101}},
/* 7590 */ {(16<<2)|3,{97,117,105}},
/* 7591 */ {(16<<2)|3,{97,117,111}},
/* 7592 */ {(16<<2)|3,{97,117,115}},
/* 7593 */ {(16<<2)|3,{97,117,116}},
/* 7594 */ {(11<<2)|2,{97,117,0}},
/* 7595 */ {(11<<2)|2,{97,117,0}},
/* 7596 */ {(11<<2)|2,{97,117,0}},
/* 7597 */ {(11<<2)|2,{97,117,0}},
/* 7598 */ {(11<<2)|2,{97,117,0}},
/* 7599 */ {(11<<2)|2,{97,117,0}},
/* 7600 */ {(11<<2)|2,{97,117,0}},
/* 7601 */ {(11<<2)|2,{97,117,0}},
/* 7602 */ {(11<<2)|2,{97,117,0}},
/* 7603 */ {(11<<2)|2,{97,117,0}},
/* 7604 */ {(11<<2)|2,{97,117,0}},
/* 7605 */ {(11<<2)|2,{97,117,0}},
/* 7606 */ {(11<<2)|2,{97,117,0}},
/* 7607 */ {(11<<2)|2,{97,117,0}},
/* 7608 */ {(11<<2)|2,{97,117,0}},
/* 7609 */ {(11<<2)|2,{97,117,0}},
/* 7610 */ {(11<<2)|2,{97,117,0}},
/* 7611 */ {(11<<2)|2,{97,117,0}},
/* 7612 */ {(11<<2)|2,{97,117,0}},
/* 7613 */ {(11<<2)|2,{97,117,0}},
/* 7614 */ {(11<<2)|2,{97,117,0}},
/* 7615 */ {(11<<2)|2,{97,117,0}},
/* 7616 */ {(12<<2)|2,{97,58,0}},
/* 7617 */ {(12<<2)|2,{97,58,0}},
/* 7618 */ {(12<<2)|2,{97,58,0}},
/* 7619 */ {(12<<2)|2,{97,58,0}},
/* 7620 */ {(12<<2)|2,{97,58,0}},
/* 7621 */ {(12<<2)|2,{97,58,0}},
/* 7622 */ {(12<<2)|2,{97,58,0}},
/* 7623 */ {(12<<2)|2,{97,58,0}},
/* 7624 */ {(12<<2)|2,{97,58,0}},
/* 7625 */ {(12<<2)|2,{97,58,0}},
/* 7626 */ {(12<<2)|2,{97,58,0}},
/* 7627 */ {(12<<2)|2,{97,58,0}},
/* 7628 */ {(12<<2)|2,{97,58,0}},
/* 7629 */ {(12<<2)|2,{97,58,0}},
/* 7630 */ {(12<<2)|2,{97,58,0}},
/* 7631 */ {(12<<2)|2,{97,58,0}},
/* 7632 */ {(12<<2)|2,{97,66,0}},
/* 7633 */ {(12<<2)|2,{97,66,0}},
/* 7634 */ {(12<<2)|2,{97,66,0}},
/* 7635 */ {(12<<2)|2,{97,66,0}},
/* 7636 */ {(12<<2)|2,{97,66,0}},
/* 7637 */ {(12<<2)|2,{97,66,0}},
/* 7638 */ {(12<<2)|2,{97,66,0}},
/* 7639 */ {(12<<2)|2,{97,66,0}},
/* 7640 */ {(12<<2)|2,{97,66,0}},
/* 7641 */ {(12<<2)|2,{97,66,0}},
/* 7642 */ {(12<<2)|2,{97,66,0}},
/* 7643 */ {(12<<2)|2,{97,66,0}},
/* 7644 */ {(12<<2)|2,{97,66,0}},
/* 7645 */ {(12<<2)|2,{97,66,0}},
/* 7646 */ {(12<<2)|2,{97,66,0}},
/* 7647 */ {(12<<2)|2,{97,66,0}},
/* 7648 */ {(12<<2)|2,{97,67,0}},
/* 7649 */ {(12<<2)|2,{97,67,0}},
/* 7650 */ {(12<<2)|2,{97,67,0}},
/* 7651 */ {(12<<2)|2,{97,67,0}},
/* 7652 */ {(12<<2)|2,{97,67,0}},
/* 7653 */ {(12<<2)|2,{97,67,0}},
/* 7654 */ {(12<<2)|2,{97,67,0}},
/* 7655 */ {(12<<2)|2,{97,67,0}},
/* 7656 */ {(12<<2)|2,{97,67,0}},
/* 7657 */ {(12<<2)|2,{97,67,0}},
/* 7658 */ {(12<<2)|2,{97,67,0}},
/* 7659 */ {(12<<2)|2,{97,67,0}},
/* 7660 */ {(12<<2)|2,{97,67,0}},
/* 7661 */ {(12<<2)|2,{97,67,0}},
/* 7662 */ {(12<<2)|2,{97,67,0}},
/* 7663 */ {(12<<2)|2,{97,67,0}},
/* 7664 */ {(12<<2)|2,{97,68,0}},
/* 7665 */ {(12<<2)|2,{97,68,0}},
/* 7666 */ {(12<<2)|2,{97,68,0}},
/* 7667 */ {(12<<2)|2,{97,68,0}},
/* 7668 */ {(12<<2)|2,{97,68,0}},
/* 7669 */ {(12<<2)|2,{97,68,0}},
/* 7670 */ {(12<<2)|2,{97,68,0}},
/* 7671 */ {(12<<2)|2,{97,68,0}},
/* 7672 */ {(12<<2)|2,{97,68,0}},
/* 7673 */ {(12<<2)|2,{97,68,0}},
/* 7674 */ {(12<<2)|2,{97,68,0}},
/* 7675 */ {(12<<2)|2,{97,68,0}},
/* 7676 */ {(12<<2)|2,{97,68,0}},
/* 7677 */ {(12<<2)|2,{97,68,0}},
/* 7678 */ {(12<<2)|2,{97,68,0}},
/* 7679 */ {(12<<2)|2,{97,68,0}},
/* 7680 */ {(12<<2)|2,{97,69,0}},
/* 7681 */ {(12<<2)|2,{97,69,0}},
/* 7682 */ {(12<<2)|2,{97,69,0}},
/* 7683 */ {(12<<2)|2,{97,69,0}},
/* 7684 */ {(12<<2)|2,{97,69,0}},
/* 7685 */ {(12<<2)|2,{97,69,0}},
/* 7686 */ {(12<<2)|2,{97,69,0}},
/* 7687 */ {(12<<2)|2,{97,69,0}},
/* 7688 */ {(12<<2)|2,{97,69,0}},
/* 7689 */ {(12<<2)|2,{97,69,0}},
/* 7690 */ {(12<<2)|2,{97,69,0}},
/* 7691 */ {(12<<2)|2,{97,69,0}},
/* 7692 */ {(12<<2)|2,{97,69,0}},
/* 7693 */ {(12<<2)|2,{97,69,0}},
/* 7694 */ {(12<<2)|2,{97,69,0}},
/* 7695 */ {(12<<2)|2,{97,69,0}},
/* 7696 */ {(12<<2)|2,{97,70,0}},
/* 7697 */ {(12<<2)|2,{97,70,0}},
/* 7698 */ {(12<<2)|2,{97,70,0}},
/* 7699 */ {(12<<2)|2,{97,70,0}},
/* 7700 */ {(12<<2)|2,{97,70,0}},
/* 7701 */ {(12<<2)|2,{97,70,0}},
/* 7702 */ {(12<<2)|2,{97,70,0}},
/* 7703 */ {(12<<2)|2,{97,70,0}},
/* 7704 */ {(12<<2)|2,{97,70,0}},
/* 7705 */ {(12<<2)|2,{97,70,0}},
/* 7706 */ {(12<<2)|2,{97,70,0}},
/* 7707 */ {(12<<2)|2,{97,70,0}},
/* 7708 */ {(12<<2)|2,{97,70,0}},
/* 7709 */ {(12<<2)|2,{97,70,0}},
/* 7710 */ {(12<<2)|2,{97,70,0}},
/* 7711 */ {(12<<2)|2,{97,70,0}},
/* 7712 */ {(12<<2)|2,{97,71,0}},
/* 7713 */ {(12<<2)|2,{97,71,0}},
/* 7714 */ {(12<<2)|2,{97,71,0}},
/* 7715 */ {(12<<2)|2,{97,71,0}},
/* 7716 */ {(12<<2)|2,{97,71,0}},
/* 7717 */ {(12<<2)|2,{97,71,0}},
/* 7718 */ {(12<<2)|2,{97,71,0}},
/* 7719 */ {(12<<2)|2,{97,71,0}},
/* 7720 */ {(12<<2)|2,{97,71,0}},
/* 7721 */ {(12<<2)|2,{97,71,0}},
/* 7722 */ {(12<<2)|2,{97,71,0}},
/* 7723 */ {(12<<2)|2,{97,71,0}},
/* 7724 */ {(12<<2)|2,{97,71,0}},
/* 7725 */ {(12<<2)|2,{97,71,0}},
/* 7726 */ {(12<<2)|2,{97,71,0}},
/* 7727 */ {(12<<2)|2,{97,71,0}},
/* 7728 */ {(12<<2)|2,{97,72,0}},
/* 7729 */ {(12<<2)|2,{97,72,0}},
/* 7730 */ {(12<<2)|2,{97,72,0}},
/* 7731 */ {(12<<2)|2,{97,72,0}},
/* 7732 */ {(12<<2)|2,{97,72,0}},
/* 7733 */ {(12<<2)|2,{97,72,0}},
/* 7734 */ {(12<<2)|2,{97,72,0}},
/* 7735 */ {(12<<2)|2,{97,72,0}},
/* 7736 */ {(12<<2)|2,{97,72,0}},
/* 7737 */ {(12<<2)|2,{97,72,0}},
/* 7738 */ {(12<<2)|2,{97,72,0}},
/* 7739 */ {(12<<2)|2,{97,72,0}},
/* 7740 */ {(12<<2)|2,{97,72,0}},
/* 7741 */ {(12<<2)|2,{97,72,0}},
/* 7742 */ {(12<<2)|2,{97,72,0}},
/* 7743 */ {(12<<2)|2,{97,72,0}},
/* 7744 */ {(12<<2)|2,{97,73,0}},
/* 7745 */ {(12<<2)|2,{97,73,0}},
/* 7746 */ {(12<<2)|2,{97,73,0}},
/* 7747 */ {(12<<2)|2,{97,73,0}},
/* 7748 */ {(12<<2)|2,{97,73,0}},
/* 7749 */ {(12<<2)|2,{97,73,0}},
/* 7750 */ {(12<<2)|2,{97,73,0}},
/* 7751 */ {(12<<2)|2,{97,73,0}},
/* 7752 */ {(12<<2)|2,{97,73,0}},
/* 7753 */ {(12<<2)|2,{97,73,0}},
/* 7754 */ {(12<<2)|2,{97,73,0}},
/* 7755 */ {(12<<2)|2,{97,73,0}},
/* 7756 */ {(12<<2)|2,{97,73,0}},
/* 7757 */ {(12<<2)|2,{97,73,0}},
/* 7758 */ {(12<<2)|2,{97,73,0}},
/* 7759 */ {(12<<2)|2,{97,73,0}},
/* 7760 */ {(12<<2)|2,{97,74,0}},
/* 7761 */ {(12<<2)|2,{97,74,0}},
/* 7762 */ {(12<<2)|2,{97,74,0}},
/* 7763 */ {(12<<2)|2,{97,74,0}},
/* 7764 */ {(12<<2)|2,{97,74,0}},
/* 7765 */ {(12<<2)|2,{97,74,0}},
/* 7766 */ {(12<<2)|2,{97,74,0}},
/* 7767 */ {(12<<2)|2,{97,74,0}},
/* 7768 */ {(12<<2)|2,{97,74,0}},
/* 7769 */ {(12<<2)|2,{97,74,0}},
/* 7770 */ {(12<<2)|2,{97,74,0}},
/* 7771 */ {(12<<2)|2,{97,74,0}},
/* 7772 */ {(12<<2)|2,{97,74,0}},
/* 7773 */ {(12<<2)|2,{97,74,0}},
/* 7774 */ {(12<<2)|2,{97,74,0}},
/* 7775 */ {(12<<2)|2,{97,74,0}},
/* 7776 */ {(12<<2)|2,{97,75,0}},
/* 7777 */ {(12<<2)|2,{97,75,0}},
/* 7778 */ {(12<<2)|2,{97,75,0}},
/* 7779 */ {(12<<2)|2,{97,75,0}},
/* 7780 */ {(12<<2)|2,{97,75,0}},
/* 7781 */ {(12<<2)|2,{97,75,0}},
/* 7782 */ {(12<<2)|2,{97,75,0}},
/* 7783 */ {(12<<2)|2,{97,75,0}},
/* 7784 */ {(12<<2)|2,{97,75,0}},
/* 7785 */ {(12<<2)|2,{97,75,0}},
/* 7786 */ {(12<<2)|2,{97,75,0}},
/* 7787 */ {(12<<2)|2,{97,75,0}},
/* 7788 */ {(12<<2)|2,{97,75,0}},
/* 7789 */ {(12<<2)|2,{97,75,0}},
/* 7790 */ {(12<<2)|2,{97,75,0}},
/* 7791 */ {(12<<2)|2,{97,75,0}},
/* 7792 */ {(12<<2)|2,{97,76,0}},
/* 7793 */ {(12<<2)|2,{97,76,0}},
/* 7794 */ {(12<<2)|2,{97,76,0}},
/* 7795 */ {(12<<2)|2,{97,76,0}},
/* 7796 */ {(12<<2)|2,{97,76,0}},
/* 7797 */ {(12<<2)|2,{97,76,0}},
/* 7798 */ {(12<<2)|2,{97,76,0}},
/* 7799 */ {(12<<2)|2,{97,76,0}},
/* 7800 */ {(12<<2)|2,{97,76,0}},
/* 7801 */ {(12<<2)|2,{97,76,0}},
/* 7802 */ {(12<<2)|2,{97,76,0}},
/* 7803 */ {(12<<2)|2,{97,76,0}},
/* 7804 */ {(12<<2)|2,{97,76,0}},
/* 7805 */ {(12<<2)|2,{97,76,0}},
/* 7806 */ {(12<<2)|2,{97,76,0}},
/* 7807 */ {(12<<2)|2,{97,76,0}},
/* 7808 */ {(12<<2)|2,{97,77,0}},
/* 7809 */ {(12<<2)|2,{97,77,0}},
/* 7810 */ {(12<<2)|2,{97,77,0}},
/* 7811 */ {(12<<2)|2,{97,77,0}},
/* 7812 */ {(12<<2)|2,{97,77,0}},
/* 7813 */ {(12<<2)|2,{97,77,0}},
/* 7814 */ {(12<<2)|2,{97,77,0}},
/* 7815 */ {(12<<2)|2,{97,77,0}},
/* 7816 */ {(12<<2)|2,{97,77,0}},
/* 7817 */ {(12<<2)|2,{97,77,0}},
/* 7818 */ {(12<<2)|2,{97,77,0}},
/* 7819 */ {(12<<2)|2,{97,77,0}},
/* 7820 */ {(12<<2)|2,{97,77,0}},
/* 7821 */ {(12<<2)|2,{97,77,0}},
/* 7822 */ {(12<<2)|2,{97,77,0}},
/* 7823 */ {(12<<2)|2,{97,77,0}},
/* 7824 */ {(12<<2)|2,{97,78,0}},
/* 7825 */ {(12<<2)|2,{97,78,0}},
/* 7826 */ {(12<<2)|2,{97,78,0}},
/* 7827 */ {(12<<2)|2,{97,78,0}},
/* 7828 */ {(12<<2)|2,{97,78,0}},
/* 7829 */ {(12<<2)|2,{97,78,0}},
/* 7830 */ {(12<<2)|2,{97,78,0}},
/* 7831 */ {(12<<2)|2,{97,78,0}},
/* 7832 */ {(12<<2)|2,{97,78,0}},
/* 7833 */ {(12<<2)|2,{97,78,0}},
/* 7834 */ {(12<<2)|2,{97,78,0}},
/* 7835 */ {(12<<2)|2,{97,78,0}},
/* 7836 */ {(12<<2)|2,{97,78,0}},
/* 7837 */ {(12<<2)|2,{97,78,0}},
/* 7838 */ {(12<<2)|2,{97,78,0}},
/* 7839 */ {(12<<2)|2,{97,78,0}},
/* 7840 */ {(12<<2)|2,{97,79,0}},
/* 7841 */ {(12<<2)|2,{97,79,0}},
/* 7842 */ {(12<<2)|2,{97,79,0}},
/* 7843 */ {(12<<2)|2,{97,79,0}},
/* 7844 */ {(12<<2)|2,{97,79,0}},
/* 7845 */ {(12<<2)|2,{97,79,0}},
/* 7846 */ {(12<<2)|2,{97,79,0}},
/* 7847 */ {(12<<2)|2,{97,79,0}},
/* 7848 */ {(12<<2)|2,{97,79,0}},
/* 7849 */ {(12<<2)|2,{97,79,0}},
/* 7850 */ {(12<<2)|2,{97,79,0}},
/* 7851 */ {(12<<2)|2,{97,79,0}},
/* 7852 */ {(12<<2)|2,{97,79,0}},
/* 7853 */ {(12<<2)|2,{97,79,0}},
/* 7854 */ {(12<<2)|2,{97,79,0}},
/* 7855 */ {(12<<2)|2,{97,79,0}},
/* 7856 */ {(12<<2)|2,{97,80,0}},
/* 7857 */ {(12<<2)|2,{97,80,0}},
/* 7858 */ {(12<<2)|2,{97,80,0}},
/* 7859 */ {(12<<2)|2,{97,80,0}},
/* 7860 */ {(12<<2)|2,{97,80,0}},
/* 7861 */ {(12<<2)|2,{97,80,0}},
/* 7862 */ {(12<<2)|2,{97,80,0}},
/* 7863 */ {(12<<2)|2,{97,80,0}},
/* 7864 */ {(12<<2)|2,{97,80,0}},
/* 7865 */ {(12<<2)|2,{97,80,0}},
/* 7866 */ {(12<<2)|2,{97,80,0}},
/* 7867 */ {(12<<2)|2,{97,80,0}},
/* 7868 */ {(12<<2)|2,{97,80,0}},
/* 7869 */ {(12<<2)|2,{97,80,0}},
/* 7870 */ {(12<<2)|2,{97,80,0}},
/* 7871 */ {(12<<2)|2,{97,80,0}},
/* 7872 */ {(12<<2)|2,{97,81,0}},
/* 7873 */ {(12<<2)|2,{97,81,0}},
/* 7874 */ {(12<<2)|2,{97,81,0}},
/* 7875 */ {(12<<2)|2,{97,81,0}},
/* 7876 */ {(12<<2)|2,{97,81,0}},
/* 7877 */ {(12<<2)|2,{97,81,0}},
/* 7878 */ {(12<<2)|2,{97,81,0}},
/* 7879 */ {(12<<2)|2,{97,81,0}},
/* 7880 */ {(12<<2)|2,{97,81,0}},
/* 7881 */ {(12<<2)|2,{97,81,0}},
/* 7882 */ {(12<<2)|2,{97,81,0}},
/* 7883 */ {(12<<2)|2,{97,81,0}},
/* 7884 */ {(12<<2)|2,{97,81,0}},
/* 7885 */ {(12<<2)|2,{97,81,0}},
/* 7886 */ {(12<<2)|2,{97,81,0}},
/* 7887 */ {(12<<2)|2,{97,81,0}},
/* 7888 */ {(12<<2)|2,{97,82,0}},
/* 7889 */ {(12<<2)|2,{97,82,0}},
/* 7890 */ {(12<<2)|2,{97,82,0}},
/* 7891 */ {(12<<2)|2,{97,82,0}},
/* 7892 */ {(12<<2)|2,{97,82,0}},
/* 7893 */ {(12<<2)|2,{97,82,0}},
/* 7894 */ {(12<<2)|2,{97,82,0}},
/* 7895 */ {(12<<2)|2,{97,82,0}},
/* 7896 */ {(12<<2)|2,{97,82,0}},
/* 7897 */ {(12<<2)|2,{97,82,0}},
/* 7898 */ {(12<<2)|2,{97,82,0}},
/* 7899 */ {(12<<2)|2,{97,82,0}},
/* 7900 */ {(12<<2)|2,{97,82,0}},
/* 7901 */ {(12<<2)|2,{97,82,0}},
/* 7902 */ {(12<<2)|2,{97,82,0}},
/* 7903 */ {(12<<2)|2,{97,82,0}},
/* 7904 */ {(12<<2)|2,{97,83,0}},
/* 7905 */ {(12<<2)|2,{97,83,0}},
/* 7906 */ {(12<<2)|2,{97,83,0}},
/* 7907 */ {(12<<2)|2,{97,83,0}},
/* 7908 */ {(12<<2)|2,{97,83,0}},
/* 7909 */ {(12<<2)|2,{97,83,0}},
/* 7910 */ {(12<<2)|2,{97,83,0}},
/* 7911 */ {(12<<2)|2,{97,83,0}},
/* 7912 */ {(12<<2)|2,{97,83,0}},
/* 7913 */ {(12<<2)|2,{97,83,0}},
/* 7914 */ {(12<<2)|2,{97,83,0}},
/* 7915 */ {(12<<2)|2,{97,83,0}},
/* 7916 */ {(12<<2)|2,{97,83,0}},
/* 7917 */ {(12<<2)|2,{97,83,0}},
/* 7918 */ {(12<<2)|2,{97,83,0}},
/* 7919 */ {(12<<2)|2,{97,83,0}},
/* 7920 */ {(12<<2)|2,{97,84,0}},
/* 7921 */ {(12<<2)|2,{97,84,0}},
/* 7922 */ {(12<<2)|2,{97,84,0}},
/* 7923 */ {(12<<2)|2,{97,84,0}},
/* 7924 */ {(12<<2)|2,{97,84,0}},
/* 7925 */ {(12<<2)|2,{97,84,0}},
/* 7926 */ {(12<<2)|2,{97,84,0}},
/* 7927 */ {(12<<2)|2,{97,84,0}},
/* 7928 */ {(12<<2)|2,{97,84,0}},
/* 7929 */ {(12<<2)|2,{97,84,0}},
/* 7930 */ {(12<<2)|2,{97,84,0}},
/* 7931 */ {(12<<2)|2,{97,84,0}},
/* 7932 */ {(12<<2)|2,{97,84,0}},
/* 7933 */ {(12<<2)|2,{97,84,0}},
/* 7934 */ {(12<<2)|2,{97,84,0}},
/* 7935 */ {(12<<2)|2,{97,84,0}},
/* 7936 */ {(12<<2)|2,{97,85,0}},
/* 7937 */ {(12<<2)|2,{97,85,0}},
/* 7938 */ {(12<<2)|2,{97,85,0}},
/* 7939 */ {(12<<2)|2,{97,85,0}},
/* 7940 */ {(12<<2)|2,{97,85,0}},
/* 7941 */ {(12<<2)|2,{97,85,0}},
/* 7942 */ {(12<<2)|2,{97,85,0}},
/* 7943 */ {(12<<2)|2,{97,85,0}},
/* 7944 */ {(12<<2)|2,{97,85,0}},
/* 7945 */ {(12<<2)|2,{97,85,0}},
/* 7946 */ {(12<<2)|2,{97,85,0}},
/* 7947 */ {(12<<2)|2,{97,85,0}},
/* 7948 */ {(12<<2)|2,{97,85,0}},
/* 7949 */ {(12<<2)|2,{97,85,0}},
/* 7950 */ {(12<<2)|2,{97,85,0}},
/* 7951 */ {(12<<2)|2,{97,85,0}},
/* 7952 */ {(12<<2)|2,{97,86,0}},
/* 7953 */ {(12<<2)|2,{97,86,0}},
/* 7954 */ {(12<<2)|2,{97,86,0}},
/* 7955 */ {(12<<2)|2,{97,86,0}},
/* 7956 */ {(12<<2)|2,{97,86,0}},
/* 7957 */ {(12<<2)|2,{97,86,0}},
/* 7958 */ {(12<<2)|2,{97,86,0}},
/* 7959 */ {(12<<2)|2,{97,86,0}},
/* 7960 */ {(12<<2)|2,{97,86,0}},
/* 7961 */ {(12<<2)|2,{97,86,0}},
/* 7962 */ {(12<<2)|2,{97,86,0}},
/* 7963 */ {(12<<2)|2,{97,86,0}},
/* 7964 */ {(12<<2)|2,{97,86,0}},
/* 7965 */ {(12<<2)|2,{97,86,0}},
/* 7966 */ {(12<<2)|2,{97,86,0}},
/* 7967 */ {(12<<2)|2,{97,86,0}},
/* 7968 */ {(12<<2)|2,{97,87,0}},
/* 7969 */ {(12<<2)|2,{97,87,0}},
/* 7970 */ {(12<<2)|2,{97,87,0}},
/* 7971 */ {(12<<2)|2,{97,87,0}},
/* 7972 */ {(12<<2)|2,{97,87,0}},
/* 7973 */ {(12<<2)|2,{97,87,0}},
/* 7974 */ {(12<<2)|2,{97,87,0}},
/* 7975 */ {(12<<2)|2,{97,87,0}},
/* 7976 */ {(12<<2)|2,{97,87,0}},
/* 7977 */ {(12<<2)|2,{97,87,0}},
/* 7978 */ {(12<<2)|2,{97,87,0}},
/* 7979 */ {(12<<2)|2,{97,87,0}},
/* 7980 */ {(12<<2)|2,{97,87,0}},
/* 7981 */ {(12<<2)|2,{97,87,0}},
/* 7982 */ {(12<<2)|2,{97,87,0}},
/* 7983 */ {(12<<2)|2,{97,87,0}},
/* 7984 */ {(12<<2)|2,{97,89,0}},
/* 7985 */ {(12<<2)|2,{97,89,0}},
/* 7986 */ {(12<<2)|2,{97,89,0}},
/* 7987 */ {(12<<2)|2,{97,89,0}},
/* 7988 */ {(12<<2)|2,{97,89,0}},
/* 7989 */ {(12<<2)|2,{97,89,0}},
/* 7990 */ {(12<<2)|2,{97,89,0}},
/* 7991 */ {(12<<2)|2,{97,89,0}},
/* 7992 */ {(12<<2)|2,{97,89,0}},
/* 7993 */ {(12<<2)|2,{97,89,0}},
/* 7994 */ {(12<<2)|2,{97,89,0}},
/* 7995 */ {(12<<2)|2,{97,89,0}},
/* 7996 */ {(12<<2)|2,{97,89,0}},
/* 7997 */ {(12<<2)|2,{97,89,0}},
/* 7998 */ {(12<<2)|2,{97,89,0}},
/* 7999 */ {(12<<2)|2,{97,89,0}},
/* 8000 */ {(12<<2)|2,{97,106,0}},
/* 8001 */ {(12<<2)|2,{97,106,0}},
/* 8002 */ {(12<<2)|2,{97,106,0}},
/* 8003 */ {(12<<2)|2,{97,106,0}},
/* 8004 */ {(12<<2)|2,{97,106,0}},
/* 8005 */ {(12<<2)|2,{97,106,0}},
/* 8006 */ {(12<<2)|2,{97,106,0}},
/* 8007 */ {(12<<2)|2,{97,106,0}},
/* 8008 */ {(12<<2)|2,{97,106,0}},
/* 8009 */ {(12<<2)|2,{97,106,0}},
/* 8010 */ {(12<<2)|2,{97,106,0}},
/* 8011 */ {(12<<2)|2,{97,106,0}},
/* 8012 */ {(12<<2)|2,{97,106,0}},
/* 8013 */ {(12<<2)|2,{97,106,0}},
/* 8014 */ {(12<<2)|2,{97,106,0}},
/* 8015 */ {(12<<2)|2,{97,106,0}},
/* 8016 */ {(12<<2)|2,{97,107,0}},
/* 8017 */ {(12<<2)|2,{97,107,0}},
/* 8018 */ {(12<<2)|2,{97,107,0}},
/* 8019 */ {(12<<2)|2,{97,107,0}},
/* 8020 */ {(12<<2)|2,{97,107,0}},
/* 8021 */ {(12<<2)|2,{97,107,0}},
/* 8022 */ {(12<<2)|2,{97,107,0}},
/* 8023 */ {(12<<2)|2,{97,107,0}},
/* 8024 */ {(12<<2)|2,{97,107,0}},
/* 8025 */ {(12<<2)|2,{97,107,0}},
/* 8026 */ {(12<<2)|2,{97,107,0}},
/* 8027 */ {(12<<2)|2,{97,107,0}},
/* 8028 */ {(12<<2)|2,{97,107,0}},
/* 8029 */ {(12<<2)|2,{97,107,0}},
/* 8030 */ {(12<<2)|2,{97,107,0}},
/* 8031 */ {(12<<2)|2,{97,107,0}},
/* 8032 */ {(12<<2)|2,{97,113,0}},
/* 8033 */ {(12<<2)|2,{97,113,0}},
/* 8034 */ {(12<<2)|2,{97,113,0}},
/* 8035 */ {(12<<2)|2,{97,113,0}},
/* 8036 */ {(12<<2)|2,{97,113,0}},
/* 8037 */ {(12<<2)|2,{97,113,0}},
/* 8038 */ {(12<<2)|2,{97,113,0}},
/* 8039 */ {(12<<2)|2,{97,113,0}},
/* 8040 */ {(12<<2)|2,{97,113,0}},
/* 8041 */ {(12<<2)|2,{97,113,0}},
/* 8042 */ {(12<<2)|2,{97,113,0}},
/* 8043 */ {(12<<2)|2,{97,113,0}},
/* 8044 */ {(12<<2)|2,{97,113,0}},
/* 8045 */ {(12<<2)|2,{97,113,0}},
/* 8046 */ {(12<<2)|2,{97,113,0}},
/* 8047 */ {(12<<2)|2,{97,113,0}},
/* 8048 */ {(12<<2)|2,{97,118,0}},
/* 8049 */ {(12<<2)|2,{97,118,0}},
/* 8050 */ {(12<<2)|2,{97,118,0}},
/* 8051 */ {(12<<2)|2,{97,118,0}},
/* 8052 */ {(12<<2)|2,{97,118,0}},
/* 8053 */ {(12<<2)|2,{97,118,0}},
/* 8054 */ {(12<<2)|2,{97,118,0}},
/* 8055 */ {(12<<2)|2,{97,118,0}},
/* 8056 */ {(12<<2)|2,{97,118,0}},
/* 8057 */ {(12<<2)|2,{97,118,0}},
/* 8058 */ {(12<<2)|2,{97,118,0}},
/* 8059 */ {(12<<2)|2,{97,118,0}},
/* 8060 */ {(12<<2)|2,{97,118,0}},
/* 8061 */ {(12<<2)|2,{97,118,0}},
/* 8062 */ {(12<<2)|2,{97,118,0}},
/* 8063 */ {(12<<2)|2,{97,118,0}},
/* 8064 */ {(12<<2)|2,{97,119,0}},
/* 8065 */ {(12<<2)|2,{97,119,0}},
/* 8066 */ {(12<<2)|2,{97,119,0}},
/* 8067 */ {(12<<2)|2,{97,119,0}},
/* 8068 */ {(12<<2)|2,{97,119,0}},
/* 8069 */ {(12<<2)|2,{97,119,0}},
/* 8070 */ {(12<<2)|2,{97,119,0}},
/* 8071 */ {(12<<2)|2,{97,119,0}},
/* 8072 */ {(12<<2)|2,{97,119,0}},
/* 8073 */ {(12<<2)|2,{97,119,0}},
/* 8074 */ {(12<<2)|2,{97,119,0}},
/* 8075 */ {(12<<2)|2,{97,119,0}},
/* 8076 */ {(12<<2)|2,{97,119,0}},
/* 8077 */ {(12<<2)|2,{97,119,0}},
/* 8078 */ {(12<<2)|2,{97,119,0}},
/* 8079 */ {(12<<2)|2,{97,119,0}},
/* 8080 */ {(12<<2)|2,{97,120,0}},
/* 8081 */ {(12<<2)|2,{97,120,0}},
/* 8082 */ {(12<<2)|2,{97,120,0}},
/* 8083 */ {(12<<2)|2,{97,120,0}},
/* 8084 */ {(12<<2)|2,{97,120,0}},
/* 8085 */ {(12<<2)|2,{97,120,0}},
/* 8086 */ {(12<<2)|2,{97,120,0}},
/* 8087 */ {(12<<2)|2,{97,120,0}},
/* 8088 */ {(12<<2)|2,{97,120,0}},
/* 8089 */ {(12<<2)|2,{97,120,0}},
/* 8090 */ {(12<<2)|2,{97,120,0}},
/* 8091 */ {(12<<2)|2,{97,120,0}},
/* 8092 */ {(12<<2)|2,{97,120,0}},
/* 8093 */ {(12<<2)|2,{97,120,0}},
/* 8094 */ {(12<<2)|2,{97,120,0}},
/* 8095 */ {(12<<2)|2,{97,120,0}},
/* 8096 */ {(12<<2)|2,{97,121,0}},
/* 8097 */ {(12<<2)|2,{97,121,0}},
/* 8098 */ {(12<<2)|2,{97,121,0}},
/* 8099 */ {(12<<2)|2,{97,121,0}},
/* 8100 */ {(12<<2)|2,{97,121,0}},
/* 8101 */ {(12<<2)|2,{97,121,0}},
/* 8102 */ {(12<<2)|2,{97,121,0}},
/* 8103 */ {(12<<2)|2,{97,121,0}},
/* 8104 */ {(12<<2)|2,{97,121,0}},
/* 8105 */ {(12<<2)|2,{97,121,0}},
/* 8106 */ {(12<<2)|2,{97,121,0}},
/* 8107 */ {(12<<2)|2,{97,121,0}},
/* 8108 */ {(12<<2)|2,{97,121,0}},
/* 8109 */ {(12<<2)|2,{97,121,0}},
/* 8110 */ {(12<<2)|2,{97,121,0}},
/* 8111 */ {(12<<2)|2,{97,121,0}},
/* 8112 */ {(12<<2)|2,{97,122,0}},
/* 8113 */ {(12<<2)|2,{97,122,0}},
/* 8114 */ {(12<<2)|2,{97,122,0}},
/* 8115 */ {(12<<2)|2,{97,122,0}},
/* 8116 */ {(12<<2)|2,{97,122,0}},
/* 8117 */ {(12<<2)|2,{97,122,0}},
/* 8118 */ {(12<<2)|2,{97,122,0}},
/* 8119 */ {(12<<2)|2,{97,122,0}},
/* 8120 */ {(12<<2)|2,{97,122,0}},
/* 8121 */ {(12<<2)|2,{97,122,0}},
/* 8122 */ {(12<<2)|2,{97,122,0}},
/* 8123 */ {(12<<2)|2,{97,122,0}},
/* 8124 */ {(12<<2)|2,{97,122,0}},
/* 8125 */ {(12<<2)|2,{97,122,0}},
/* 8126 */ {(12<<2)|2,{97,122,0}},
/* 8127 */ {(12<<2)|2,{97,122,0}},
/* 8128 */ {(13<<2)|2,{97,38,0}},
/* 8129 */ {(13<<2)|2,{97,38,0}},
/* 8130 */ {(13<<2)|2,{97,38,0}},
/* 8131 */ {(13<<2)|2,{97,38,0}},
/* 8132 */ {(13<<2)|2,{97,38,0}},
/* 8133 */ {(13<<2)|2,{97,38,0}},
/* 8134 */ {(13<<2)|2,{97,38,0}},
/* 8135 */ {(13<<2)|2,{97,38,0}},
/* 8136 */ {(13<<2)|2,{97,42,0}},
/* 8137 */ {(13<<2)|2,{97,42,0}},
/* 8138 */ {(13<<2)|2,{97,42,0}},
/* 8139 */ {(13<<2)|2,{97,42,0}},
/* 8140 */ {(13<<2)|2,{97,42,0}},
/* 8141 */ {(13<<2)|2,{97,42,0}},
/* 8142 */ {(13<<2)|2,{97,42,0}},
/* 8143 */ {(13<<2)|2,{97,42,0}},
/* 8144 */ {(13<<2)|2,{97,44,0}},
/* 8145 */ {(13<<2)|2,{97,44,0}},
/* 8146 */ {(13<<2)|2,{97,44,0}},
/* 8147 */ {(13<<2)|2,{97,44,0}},
/* 8148 */ {(13<<2)|2,{97,44,0}},
/* 8149 */ {(13<<2)|2,{97,44,0}},
/* 8150 */ {(13<<2)|2,{97,44,0}},
/* 8151 */ {(13<<2)|2,{97,44,0}},
/* 8152 */ {(13<<2)|2,{97,59,0}},
/* 8153 */ {(13<<2)|2,{97,59,0}},
/* 8154 */ {(13<<2)|2,{97,59,0}},
/* 8155 */ {(13<<2)|2,{97,59,0}},
/* 8156 */ {(13<<2)|2,{97,59,0}},
/* 8157 */ {(13<<2)|2,{97,59,0}},
/* 8158 */ {(13<<2)|2,{97,59,0}},
/* 8159 */ {(13<<2)|2,{97,59,0}},
/* 8160 */ {(13<<2)|2,{97,88,0}},
/* 8161 */ {(13<<2)|2,{97,88,0}},
/* 8162 */ {(13<<2)|2,{97,88,0}},
/* 8163 */ {(13<<2)|2,{97,88,0}},
/* 8164 */ {(13<<2)|2,{97,88,0}},
/* 8165 */ {(13<<2)|2,{97,88,0}},
/* 8166 */ {(13<<2)|2,{97,88,0}},
/* 8167 */ {(13<<2)|2,{97,88,0}},
/* 8168 */ {(13<<2)|2,{97,90,0}},
/* 8169 */ {(13<<2)|2,{97,90,0}},
/* 8170 */ {(13<<2)|2,{97,90,0}},
/* 8171 */ {(13<<2)|2,{97,90,0}},
/* 8172 */ {(13<<2)|2,{97,90,0}},
/* 8173 */ {(13<<2)|2,{97,90,0}},
/* 8174 */ {(13<<2)|2,{97,90,0}},
/* 8175 */ {(13<<2)|2,{97,90,0}},
/* 8176 */ {(15<<2)|2,{97,33,0}},
/* 8177 */ {(15<<2)|2,{97,33,0}},
/* 8178 */ {(15<<2)|2,{97,34,0}},
/* 8179 */ {(15<<2)|2,{97,34,0}},
/* 8180 */ {(15<<2)|2,{97,40,0}},
/* 8181 */ {(15<<2)|2,{97,40,0}},
/* 8182 */ {(15<<2)|2,{97,41,0}},
/* 8183 */ {(15<<2)|2,{97,41,0}},
/* 8184 */ {(15<<2)|2,{97,63,0}},
/* 8185 */ {(15<<2)|2,{97,63,0}},
/* 8186 */ {(16<<2)|2,{97,39,0}},
/* 8187 */ {(16<<2)|2,{97,43,0}},
/* 8188 */ {(16<<2)|2,{97,124,0}},
/* 8189 */ {(5<<2)|1,{97,0,0}},
/* 8190 */ {(5<<2)|1,{97,0,0}},
/* 8191 */ {(5<<2)|1,{97,0,0}},
/* 8192 */ {(15<<2)|3,{99,48,48}},
/* 8193 */ {(15<<2)|3,{99,48,48}},
/* 8194 */ {(15<<2)|3,{99,48,49}},
/* 8195 */ {(15<<2)|3,{99,48,49}},
/* 8196 */ {(15<<2)|3,{99,48,50}},
/* 8197 */ {(15<<2)|3,{99,48,50}},
/* 8198 */ {(15<<2)|3,{99,48,97}},
/* 8199 */ {(15<<2)|3,{99,48,97}},
/* 8200 */ {(15<<2)|3,{99,48,99}},
/* 8201 */ {(15<<2)|3,{99,48,99}},
/* 8202 */ {(15<<2)|3,{99,48,101}},
/* 8203 */ {(15<<2)|3,{99,48,101}},
/* 8204 */ {(15<<2)|3,{99,48,105}},
/* 8205 */ {(15<<2)|3,{99,48,105}},
/* 8206 */ {(15<<2)|3,{99,48,111}},
/* 8207 */ {(15<<2)|3,{99,48,111}},
/* 8208 */ {(15<<2)|3,{99,48,115}},
/* 8209 */ {(15<<2)|3,{99,48,115}},
/* 8210 */ {(15<<2)|3,{99,48,116}},
/* 8211 */ {(15<<2)|3,{99,48,116}},
/* 8212 */ {(16<<2)|3,{99,48,32}},
/* 8213 */ {(16<<2)|3,{99,48,37}},
/* 8214 */ {(16<<2)|3,{99,48,45}},
/* 8215 */ {(16<<2)|3,{99,48,46}},
/* 8216 */ {(16<<2)|3,{99,48,47}},
/* 8217 */ {(16<<2)|3,{99,48,51}},
/* 8218 */ {(16<<2)|3,{99,48,52}},
/* 8219 */ {(16<<2)|3,{99,48,53}},
/* 8220 */ {(16<<2)|3,{99,48,54}},
/* 8221 */ {(16<<2)|3,{99,48,55}},
/* 8222 */ {(16<<2)|3,{99,48,56}},
/* 8223 */ {(16<<2)|3,{99,48,57}},
/* 8224 */ {(16<<2)|3,{99,48,61}},
/* 8225 */ {(16<<2)|3,{99,48,65}},
/* 8226 */ {(16<<2)|3,{99,48,95}},
/* 8227 */ {(16<<2)|3,{99,48,98}},
/* 8228 */ {(16<<2)|3,{99,48,100}},
/* 8229 */ {(16<<2)|3,{99,48,102}},
/* 8230 */ {(16<<2)|3,{99,48,103}},
/* 8231 */ {(16<<2)|3,{99,48,104}},
/* 8232 */ {(16<<2)|3,{99,48,108}},
/* 8233 */ {(16<<2)|3,{99,48,109}},
/* 8234 */ {(16<<2)|3,{99,48,110}},
/* 8235 */ {(16<<2)|3,{99,48,112}},
/* 8236 */ {(16<<2)|3,{99,48,114}},
/* 8237 */ {(16<<2)|3,{99,48,117}},
/* 8238 */ {(10<<2)|2,{99,48,0}},
/* 8239 */ {(10<<2)|2,{99,48,0}},
/* 8240 */ {(10<<2)|2,{99,48,0}},
/* 8241 */ {(10<<2)|2,{99,48,0}},
/* 8242 */ {(10<<2)|2,{99,48,0}},
/* 8243 */ {(10<<2)|2,{99,48,0}},
/* 8244 */ {(10<<2)|2,{99,48,0}},
/* 8245 */ {(10<<2)|2,{99,48,0}},
/* 8246 */ {(10<<2)|2,{99,48,0}},
/* 8247 */ {(10<<2)|2,{99,48,0}},
/* 8248 */ {(10<<2)|2,{99,48,0}},
/* 8249 */ {(10<<2)|2,{99,48,0}},
/* 8250 */ {(10<<2)|2,{99,48,0}},
/* 8251 */ {(10<<2)|2,{99,48,0}},
/* 8252 */ {(10<<2)|2,{99,48,0}},
/* 8253 */ {(10<<2)|2,{99,48,0}},
/* 8254 */ {(10<<2)|2,{99,48,0}},
/* 8255 */ {(10<<2)|2,{99,48,0}},
/* 8256 */ {(15<<2)|3,{99,49,48}},
/* 8257 */ {(15<<2)|3,{99,49,48}},
/* 8258 */ {(15<<2)|3,{99,49,49}},
/* 8259 */ {(15<<2)|3,{99,49,49}},
/* 8260 */ {(15<<2)|3,{99,49,50}},
/* 8261 */ {(15<<2)|3,{99,49,50}},
/* 8262 */ {(15<<2)|3,{99,49,97}},
/* 8263 */ {(15<<2)|3,{99,49,97}},
/* 8264 */ {(15<<2)|3,{99,49,99}},
/* 8265 */ {(15<<2)|3,{99,49,99}},
/* 8266 */ {(15<<2)|3,{99,49,101}},
/* 8267 */ {(15<<2)|3,{99,49,101}},
/* 8268 */ {(15<<2)|3,{99,49,105}},
/* 8269 */ {(15<<2)|3,{99,49,105}},
/* 8270 */ {(15<<2)|3,{99,49,111}},
/* 8271 */ {(15<<2)|3,{99,49,111}},
/* 8272 */ {(15<<2)|3,{99,49,115}},
/* 8273 */ {(15<<2)|3,{99,49,115}},
/* 8274 */ {(15<<2)|3,{99,49,116}},
/* 8275 */ {(15<<2)|3,{99,49,116}},
/* 8276 */ {(16<<2)|3,{99,49,32}},
/* 8277 */ {(16<<2)|3,{99,49,37}},
/* 8278 */ {(16<<2)|3,{99,49,45}},
/* 8279 */ {(16<<2)|3,{99,49,46}},
/* 8280 */ {(16<<2)|3,{99,49,47}},
/* 8281 */ {(16<<2)|3,{99,49,51}},
/* 8282 */ {(16<<2)|3,{99,49,52}},
/* 8283 */ {(16<<2)|3,{99,49,53}},
/* 8284 */ {(16<<2)|3,{99,49,54}},
/* 8285 */ {(16<<2)|3,{99,49,55}},
/* 8286 */ {(16<<2)|3,{99,49,56}},
/* 8287 */ {(16<<2)|3,{99,49,57}},
/* 8288 */ {(16<<2)|3,{99,49,61}},
/* 8289 */ {(16<<2)|3,{99,49,65}},
/* 8290 */ {(16<<2)|3,{99,49,95}},
/* 8291 */ {(16<<2)|3,{99,49,98}},
/* 8292 */ {(16<<2)|3,{99,49,100}},
/* 8293 */ {(16<<2)|3,{99,49,102}},
/* 8294 */ {(16<<2)|3,{99,49,103}},
/* 8295 */ {(16<<2)|3,{99,49,104}},
/* 8296 */ {(16<<2)|3,{99,49,108}},
/* 8297 */ {(16<<2)|3,{99,49,109}},
/* 8298 */ {(16<<2)|3,{99,49,110}},
/* 8299 */ {(16<<2)|3,{99,49,112}},
/* 8300 */ {(16<<2)|3,{99,49,114}},
/* 8301 */ {(16<<2)|3,{99,49,117}},
/* 8302 */ {(10<<2)|2,{99,49,0}},
/* 8303 */ {(10<<2)|2,{99,49,0}},
/* 8304 */ {(10<<2)|2,{99,49,0}},
/* 8305 */ {(10<<2)|2,{99,49,0}},
/* 8306 */ {(10<<2)|2,{99,49,0}},
/* 8307 */ {(10<<2)|2,{99,49,0}},
/* 8308 */ {(10<<2)|2,{99,49,0}},
/* 8309 */ {(10<<2)|2,{99,49,0}},
/* 8310 */ {(10<<2)|2,{99,49,0}},
/* 8311 */ {(10<<2)|2,{99,49,0}},
/* 8312 */ {(10<<2)|2,{99,49,0}},
/* 8313 */ {(10<<2)|2,{99,49,0}},
/* 8314 */ {(10<<2)|2,{99,49,0}},
/* 8315 */ {(10<<2)|2,{99,49,0}},
/* 8316 */ {(10<<2)|2,{99,49,0}},
/* 8317 */ {(10<<2)|2,{99,49,0}},
/* 8318 */ {(10<<2)|2,{99,49,0}},
/* 8319 */ {(10<<2)|2,{99,49,0}},
/* 8320 */ {(15<<2)|3,{99,50,48}},
/* 8321 */ {(15<<2)|3,{99,50,48}},
/* 8322 */ {(15<<2)|3,{99,50,49}},
/* 8323 */ {(15<<2)|3,{99,50,49}},
/* 8324 */ {(15<<2)|3,{99,50,50}},
/* 8325 */ {(15<<2)|3,{99,50,50}},
/* 8326 */ {(15<<2)|3,{99,50,97}},
/* 8327 */ {(15<<2)|3,{99,50,97}},
/* 8328 */ {(15<<2)|3,{99,50,99}},
/* 8329 */ {(15<<2)|3,{99,50,99}},
/* 8330 */ {(15<<2)|3,{99,50,101}},
/* 8331 */ {(15<<2)|3,{99,50,101}},
/* 8332 */ {(15<<2)|3,{99,50,105}},
/* 8333 */ {(15<<2)|3,{99,50,105}},
/* 8334 */ {(15<<2)|3,{99,50,111}},
/* 8335 */ {(15<<2)|3,{99,50,111}},
/* 8336 */ {(15<<2)|3,{99,50,115}},
/* 8337 */ {(15<<2)|3,{99,50,115}},
/* 8338 */ {(15<<2)|3,{99,50,116}},
/* 8339 */ {(15<<2)|3,{99,50,116}},
/* 8340 */ {(16<<2)|3,{99,50,32}},
/* 8341 */ {(16<<2)|3,{99,50,37}},
/* 8342 */ {(16<<2)|3,{99,50,45}},
/* 8343 */ {(16<<2)|3,{99,50,46}},
/* 8344 */ {(16<<2)|3,{99,50,47}},
/* 8345 */ {(16<<2)|3,{99,50,51}},
/* 8346 */ {(16<<2)|3,{99,50,52}},
/* 8347 */ {(16<<2)|3,{99,50,53}},
/* 8348 */ {(16<<2)|3,{99,50,54}},
/* 8349 */ {(16<<2)|3,{99,50,55}},
/* 8350 */ {(16<<2)|3,{99,50,56}},
/* 8351 */ {(16<<2)|3,{99,50,57}},
/* 8352 */ {(16<<2)|3,{99,50,61}},
/* 8353 */ {(16<<2)|3,{99,50,65}},
/* 8354 */ {(16<<2)|3,{99,50,95}},
/* 8355 */ {(16<<2)|3,{99,50,98}},
/* 8356 */ {(16<<2)|3,{99,50,100}},
/* 8357 */ {(16<<2)|3,{99,50,102}},
/* 8358 */ {(16<<2)|3,{99,50,103}},
/* 8359 */ {(16<<2)|3,{99,50,104}},
/* 8360 */ {(16<<2)|3,{99,50,108}},
/* 8361 */ {(16<<2)|3,{99,50,109}},
/* 8362 */ {(16<<2)|3,{99,50,110}},
/* 8363 */ {(16<<2)|3,{99,50,112}},
/* 8364 */ {(16<<2)|3,{99,50,114}},
/* 8365 */ {(16<<2)|3,{99,50,117}},
/* 8366 */ {(10<<2)|2,{99,50,0}},
/* 8367 */ {(10<<2)|2,{99,50,0}},
/* 8368 */ {(10<<2)|2,{99,50,0}},
/* 8369 */ {(10<<2)|2,{99,50,0}},
/* 8370 */ {(10<<2)|2,{99,50,0}},
/* 8371 */ {(10<<2)|2,{99,50,0}},
/* 8372 */ {(10<<2)|2,{99,50,0}},
/* 8373 */ {(10<<2)|2,{99,50,0}},
/* 8374 */ {(10<<2)|2,{99,50,0}},
/* 8375 */ {(10<<2)|2,{99,50,0}},
/* 8376 */ {(10<<2)|2,{99,50,0}},
/* 8377 */ {(10<<2)|2,{99,50,0}},
/* 8378 */ {(10<<2)|2,{99,50,0}},
/* 8379 */ {(10<<2)|2,{99,50,0}},
/* 8380 */ {(10<<2)|2,{99,50,0}},
/* 8381 */ {(10<<2)|2,{99,50,0}},
/* 8382 */ {(10<<2)|2,{99,50,0}},
/* 8383 */ {(10<<2)|2,{99,50,0}},
/* 8384 */ {(15<<2)|3,{99,97,48}},
/* 8385 */ {(15<<2)|3,{99,97,48}},
/* 8386 */ {(15<<2)|3,{99,97,49}},
/* 8387 */ {(15<<2)|3,{99,97,49}},
/* 8388 */ {(15<<2)|3,{99,97,50}},
/* 8389 */ {(15<<2)|3,{99,97,50}},
/* 8390 */ {(15<<2)|3,{99,97,97}},
/* 8391 */ {(15<<2)|3,{99,97,97}},
/* 8392 */ {(15<<2)|3,{99,97,99}},
/* 8393 */ {(15<<2)|3,{99,97,99}},
/* 8394 */ {(15<<2)|3,{99,97,101}},
/* 8395 */ {(15<<2)|3,{99,97,101}},
/* 8396 */ {(15<<2)|3,{99,97,105}},
/* 8397 */ {(15<<2)|3,{99,97,105}},
/* 8398 */ {(15<<2)|3,{99,97,111}},
/* 8399 */ {(15<<2)|3,{99,97,111}},
/* 8400 */ {(15<<2)|3,{99,97,115}},
/* 8401 */ {(15<<2)|3,{99,97,115}},
/* 8402 */ {(15<<2)|3,{99,97,116}},
/* 8403 */ {(15<<2)|3,{99,97,116}},
/* 8404 */ {(16<<2)|3,{99,97,32}},
/* 8405 */ {(16<<2)|3,{99,97,37}},
/* 8406 */ {(16<<2)|3,{99,97,45}},
/* 8407 */ {(16<<2)|3,{99,97,46}},
/* 8408 */ {(16<<2)|3,{99,97,47}},
/* 8409 */ {(16<<2)|3,{99,97,51}},
/* 8410 */ {(16<<2)|3,{99,97,52}},
/* 8411 */ {(16<<2)|3,{99,97,53}},
/* 8412 */ {(16<<2)|3,{99,97,54}},
/* 8413 */ {(16<<2)|3,{99,97,55}},
/* 8414 */ {(16<<2)|3,{99,97,56}},
/* 8415 */ {(16<<2)|3,{99,97,57}},
/* 8416 */ {(16<<2)|3,{99,97,61}},
/* 8417 */ {(16<<2)|3,{99,97,65}},
/* 8418 */ {(16<<2)|3,{99,97,95}},
/* 8419 */ {(16<<2)|3,{99,97,98}},
/* 8420 */ {(16<<2)|3,{99,97,100}},
/* 8421 */ {(16<<2)|3,{99,97,102}},
/* 8422 */ {(16<<2)|3,{99,97,103}},
/* 8423 */ {(16<<2)|3,{99,97,104}},
/* 8424 */ {(16<<2)|3,{99,97,108}},
/* 8425 */ {(16<<2)|3,{99,97,109}},
/* 8426 */ {(16<<2)|3,{99,97,110}},
/* 8427 */ {(16<<2)|3,{99,97,112}},
/* 8428 */ {(16<<2)|3,{99,97,114}},
/* 8429 */ {(16<<2)|3,{99,97,117}},
/* 8430 */ {(10<<2)|2,{99,97,0}},
/* 8431 */ {(10<<2)|2,{99,97,0}},
/* 8432 */ {(10<<2)|2,{99,97,0}},
/* 8433 */ {(10<<2)|2,{99,97,0}},
/* 8434 */ {(10<<2)|2,{99,97,0}},
/* 8435 */ {(10<<2)|2,{99,97,0}},
/* 8436 */ {(10<<2)|2,{99,97,0}},
/* 8437 */ {(10<<2)|2,{99,97,0}},
/* 8438 */ {(10<<2)|2,{99,97,0}},
/* 8439 */ {(10<<2)|2,{99,97,0}},
/* 8440 */ {(10<<2)|2,{99,97,0}},
/* 8441 */ {(10<<2)|2,{99,97,0}},
/* 8442 */ {(10<<2)|2,{99,97,0}},
/* 8443 */ {(10<<2)|2,{99,97,0}},
/* 8444 */ {(10<<2)|2,{99,97,0}},
/* 8445 */ {(10<<2)|2,{99,97,0}},
/* 8446 */ {(10<<2)|2,{99,97,0}},
/* 8447 */ {(10<<2)|2,{99,97,0}},
/* 8448 */ {(15<<2)|3,{99,99,48}},
/* 8449 */ {(15<<2)|3,{99,99,48}},
/* 8450 */ {(15<<2)|3,{99,99,49}},
/* 8451 */ {(15<<2)|3,{99,99,49}},
/* 8452 */ {(15<<2)|3,{99,99,50}},
/* 8453 */ {(15<<2)|3,{99,99,50}},
/* 8454 */ {(15<<2)|3,{99,99,97}},
/* 8455 */ {(15<<2)|3,{99,99,97}},
/* 8456 */ {(15<<2)|3,{99,99,99}},
/* 8457 */ {(15<<2)|3,{99,99,99}},
/* 8458 */ {(15<<2)|3,{99,99,101}},
/* 8459 */ {(15<<2)|3,{99,99,101}},
/* 8460 */ {(15<<2)|3,{99,99,105}},
/* 8461 */ {(15<<2)|3,{99,99,105}},
/* 8462 */ {(15<<2)|3,{99,99,111}},
/* 8463 */ {(15<<2)|3,{99,99,111}},
/* 8464 */ {(15<<2)|3,{99,99,115}},
/* 8465 */ {(15<<2)|3,{99,99,115}},
/* 8466 */ {(15<<2)|3,{99,99,116}},
/* 8467 */ {(15<<2)|3,{99,99,116}},
/* 8468 */ {(16<<2)|3,{99,99,32}},
/* 8469 */ {(16<<2)|3,{99,99,37}},
/* 8470 */ {(16<<2)|3,{99,99,45}},
/* 8471 */ {(16<<2)|3,{99,99,46}},
/* 8472 */ {(16<<2)|3,{99,99,47}},
/* 8473 */ {(16<<2)|3,{99,99,51}},
/* 8474 */ {(16<<2)|3,{99,99,52}},
/* 8475 */ {(16<<2)|3,{99,99,53}},
/* 8476 */ {(16<<2)|3,{99,99,54}},
/* 8477 */ {(16<<2)|3,{99,99,55}},
/* 8478 */ {(16<<2)|3,{99,99,56}},
/* 8479 */ {(16<<2)|3,{99,99,57}},
/* 8480 */ {(16<<2)|3,{99,99,61}},
/* 8481 */ {(16<<2)|3,{99,99,65}},
/* 8482 */ {(16<<2)|3,{99,99,95}},
/* 8483 */ {(16<<2)|3,{99,99,98}},
/* 8484 */ {(16<<2)|3,{99,99,100}},
/* 8485 */ {(16<<2)|3,{99,99,102}},
/* 8486 */ {(16<<2)|3,{99,99,103}},
/* 8487 */ {(16<<2)|3,{99,99,104}},
/* 8488 */ {(16<<2)|3,{99,99,108}},
/* 8489 */ {(16<<2)|3,{99,99,109}},
/* 8490 */ {(16<<2)|3,{99,99,110}},
/* 8491 */ {(16<<2)|3,{99,99,112}},
/* 8492 */ {(16<<2)|3,{99,99,114}},
/* 8493 */ {(16<<2)|3,{99,99,117}},
/* 8494 */ {(10<<2)|2,{99,99,0}},
/* 8495 */ {(10<<2)|2,{99,99,0}},
/* 8496 */ {(10<<2)|2,{99,99,0}},
/* 8497 */ {(10<<2)|2,{99,99,0}},
/* 8498 */ {(10<<2)|2,{99,99,0}},
/* 8499 */ {(10<<2)|2,{99,99,0}},
/* 8500 */ {(10<<2)|2,{99,99,0}},
/* 8501 */ {(10<<2)|2,{99,99,0}},
/* 8502 */ {(10<<2)|2,{99,99,0}},
/* 8503 */ {(10<<2)|2,{99,99,0}},
/* 8504 */ {(10<<2)|2,{99,99,0}},
/* 8505 */ {(10<<2)|2,{99,99,0}},
/* 8506 */ {(10<<2)|2,{99,99,0}},
/* 8507 */ {(10<<2)|2,{99,99,0}},
/* 8508 */ {(10<<2)|2,{99,99,0}},
/* 8509 */ {(10<<2)|2,{99,99,0}},
/* 8510 */ {(10<<2)|2,{99,99,0}},
/* 8511 */ {(10<<2)|2,{99,99,0}},
/* 8512 */ {(15<<2)|3,{99,101,48}},
/* 8513 */ {(15<<2)|3,{99,101,48}},
/* 8514 */ {(15<<2)|3,{99,101,49}},
/* 8515 */ {(15<<2)|3,{99,101,49}},
/* 8516 */ {(15<<2)|3,{99,101,50}},
/* 8517 */ {(15<<2)|3,{99,101,50}},
/* 8518 */ {(15<<2)|3,{99,101,97}},
/* 8519 */ {(15<<2)|3,{99,101,97}},
/* 8520 */ {(15<<2)|3,{99,101,99}},
/* 8521 */ {(15<<2)|3,{99,101,99}},
/* 8522 */ {(15<<2)|3,{99,101,101}},
/* 8523 */ {(15<<2)|3,{99,101,101}},
/* 8524 */ {(15<<2)|3,{99,101,105}},
/* 8525 */ {(15<<2)|3,{99,101,105}},
/* 8526 */ {(15<<2)|3,{99,101,111}},
/* 8527 */ {(15<<2)|3,{99,101,111}},
/* 8528 */ {(15<<2)|3,{99,101,115}},
/* 8529 */ {(15<<2)|3,{99,101,115}},
/* 8530 */ {(15<<2)|3,{99,101,116}},
/* 8531 */ {(15<<2)|3,{99,101,116}},
/* 8532 */ {(16<<2)|3,{99,101,32}},
/* 8533 */ {(16<<2)|3,{99,101,37}},
/* 8534 */ {(16<<2)|3,{99,101,45}},
/* 8535 */ {(16<<2)|3,{99,101,46}},
/* 8536 */ {(16<<2)|3,{99,101,47}},
/* 8537 */ {(16<<2)|3,{99,101,51}},
/* 8538 */ {(16<<2)|3,{99,101,52}},
/* 8539 */ {(16<<2)|3,{99,101,53}},
/* 8540 */ {(16<<2)|3,{99,101,54}},
/* 8541 */ {(16<<2)|3,{99,101,55}},
/* 8542 */ {(16<<2)|3,{99,101,56}},
/* 8543 */ {(16<<2)|3,{99,101,57}},
/* 8544 */ {(16<<2)|3,{99,101,61}},
/* 8545 */ {(16<<2)|3,{99,101,65}},
/* 8546 */ {(16<<2)|3,{99,101,95}},
/* 8547 */ {(16<<2)|3,{99,101,98}},
/* 8548 */ {(16<<2)|3,{99,101,100}},
/* 8549 */ {(16<<2)|3,{99,101,102}},
/* 8550 */ {(16<<2)|3,{99,101,103}},
/* 8551 */ {(16<<2)|3,{99,101,104}},
/* 8552 */ {(16<<2)|3,{99,101,108}},
/* 8553 */ {(16<<2)|3,{99,101,109}},
/* 8554 */ {(16<<2)|3,{99,101,110}},
/* 8555 */ {(16<<2)|3,{99,101,112}},
/* 8556 */ {(16<<2)|3,{99,101,114}},
/* 8557 */ {(16<<2)|3,{99,101,117}},
/* 8558 */ {(10<<2)|2,{99,101,0}},
/* 8559 */ {(10<<2)|2,{99,101,0}},
/* 8560 */ {(10<<2)|2,{99,101,0}},
/* 8561 */ {(10<<2)|2,{99,101,0}},
/* 8562 */ {(10<<2)|2,{99,101,0}},
/* 8563 */ {(10<<2)|2,{99,101,0}},
/* 8564 */ {(10<<2)|2,{99,101,0}},
/* 8565 */ {(10<<2)|2,{99,101,0}},
/* 8566 */ {(10<<2)|2,{99,101,0}},
/* 8567 */ {(10<<2)|2,{99,101,0}},
/* 8568 */ {(10<<2)|2,{99,101,0}},
/* 8569 */ {(10<<2)|2,{99,101,0}},
/* 8570 */ {(10<<2)|2,{99,101,0}},
/* 8571 */ {(10<<2)|2,{99,101,0}},
/* 8572 */ {(10<<2)|2,{99,101,0}},
/* 8573 */ {(10<<2)|2,{99,101,0}},
/* 8574 */ {(10<<2)|2,{99,101,0}},
/* 8575 */ {(10<<2)|2,{99,101,0}},
/* 8576 */ {(15<<2)|3,{99,105,48}},
/* 8577 */ {(15<<2)|3,{99,105,48}},
/* 8578 */ {(15<<2)|3,{99,105,49}},
/* 8579 */ {(15<<2)|3,{99,105,49}},
/* 8580 */ {(15<<2)|3,{99,105,50}},
/* 8581 */ {(15<<2)|3,{99,105,50}},
/* 8582 */ {(15<<2)|3,{99,105,97}},
/* 8583 */ {(15<<2)|3,{99,105,97}},
/* 8584 */ {(15<<2)|3,{99,105,99}},
/* 8585 */ {(15<<2)|3,{99,105,99}},
/* 8586 */ {(15<<2)|3,{99,105,101}},
/* 8587 */ {(15<<2)|3,{99,105,101}},
/* 8588 */ {(15<<2)|3,{99,105,105}},
/* 8589 */ {(15<<2)|3,{99,105,105}},
/* 8590 */ {(15<<2)|3,{99,105,111}},
/* 8591 */ {(15<<2)|3,{99,105,111}},
/* 8592 */ {(15<<2)|3,{99,105,115}},
/* 8593 */ {(15<<2)|3,{99,105,115}},
/* 8594 */ {(15<<2)|3,{99,105,116}},
/* 8595 */ {(15<<2)|3,{99,105,116}},
/* 8596 */ {(16<<2)|3,{99,105,32}},
/* 8597 */ {(16<<2)|3,{99,105,37}},
/* 8598 */ {(16<<2)|3,{99,105,45}},
/* 8599 */ {(16<<2)|3,{99,105,46}},
/* 8600 */ {(16<<2)|3,{99,105,47}},
/* 8601 */ {(16<<2)|3,{99,105,51}},
/* 8602 */ {(16<<2)|3,{99,105,52}},
/* 8603 */ {(16<<2)|3,{99,105,53}},
/* 8604 */ {(16<<2)|3,{99,105,54}},
/* 8605 */ {(16<<2)|3,{99,105,55}},
/* 8606 */ {(16<<2)|3,{99,105,56}},
/* 8607 */ {(16<<2)|3,{99,105,57}},
/* 8608 */ {(16<<2)|3,{99,105,61}},
/* 8609 */ {(16<<2)|3,{99,105,65}},
/* 8610 */ {(16<<2)|3,{99,105,95}},
/* 8611 */ {(16<<2)|3,{99,105,98}},
/* 8612 */ {(16<<2)|3,{99,105,100}},
/* 8613 */ {(16<<2)|3,{99,105,102}},
/* 8614 */ {(16<<2)|3,{99,105,103}},
/* 8615 */ {(16<<2)|3,{99,105,104}},
/* 8616 */ {(16<<2)|3,{99,105,108}},
/* 8617 */ {(16<<2)|3,{99,105,109}},
/* 8618 */ {(16<<2)|3,{99,105,110}},
/* 8619 */ {(16<<2)|3,{99,105,112}},
/* 8620 */ {(16<<2)|3,{99,105,114}},
/* 8621 */ {(16<<2)|3,{99,105,117}},
/* 8622 */ {(10<<2)|2,{99,105,0}},
/* 8623 */ {(10<<2)|2,{99,105,0}},
/* 8624 */ {(10<<2)|2,{99,105,0}},
/* 8625 */ {(10<<2)|2,{99,105,0}},
/* 8626 */ {(10<<2)|2,{99,105,0}},
/* 8627 */ {(10<<2)|2,{99,105,0}},
/* 8628 */ {(10<<2)|2,{99,105,0}},
/* 8629 */ {(10<<2)|2,{99,105,0}},
/* 8630 */ {(10<<2)|2,{99,105,0}},
/* 8631 */ {(10<<2)|2,{99,105,0}},
/* 8632 */ {(10<<2)|2,{99,105,0}},
/* 8633 */ {(10<<2)|2,{99,105,0}},
/* 8634 */ {(10<<2)|2,{99,105,0}},
/* 8635 */ {(10<<2)|2,{99,105,0}},
/* 8636 */ {(10<<2)|2,{99,105,0}},
/* 8637 */ {(10<<2)|2,{99,105,0}},
/* 8638 */ {(10<<2)|2,{99,105,0}},
/* 8639 */ {(10<<2)|2,{99,105,0}},
/* 8640 */ {(15<<2)|3,{99,111,48}},
/* 8641 */ {(15<<2)|3,{99,111,48}},
/* 8642 */ {(15<<2)|3,{99,111,49}},
/* 8643 */ {(15<<2)|3,{99,111,49}},
/* 8644 */ {(15<<2)|3,{99,111,50}},
/* 8645 */ {(15<<2)|3,{99,111,50}},
/* 8646 */ {(15<<2)|3,{99,111,97}},
/* 8647 */ {(15<<2)|3,{99,111,97}},
/* 8648 */ {(15<<2)|3,{99,111,99}},
/* 8649 */ {(15<<2)|3,{99,111,99}},
/* 8650 */ {(15<<2)|3,{99,111,101}},
/* 8651 */ {(15<<2)|3,{99,111,101}},
/* 8652 */ {(15<<2)|3,{99,111,105}},
/* 8653 */ {(15<<2)|3,{99,111,105}},
/* 8654 */ {(15<<2)|3,{99,111,111}},
/* 8655 */ {(15<<2)|3,{99,111,111}},
/* 8656 */ {(15<<2)|3,{99,111,115}},
/* 8657 */ {(15<<2)|3,{99,111,115}},
/* 8658 */ {(15<<2)|3,{99,111,116}},
/* 8659 */ {(15<<2)|3,{99,111,116}},
/* 8660 */ {(16<<2)|3,{99,111,32}},
/* 8661 */ {(16<<2)|3,{99,111,37}},
/* 8662 */ {(16<<2)|3,{99,111,45}},
/* 8663 */ {(16<<2)|3,{99,111,46}},
/* 8664 */ {(16<<2)|3,{99,111,47}},
/* 8665 */ {(16<<2)|3,{99,111,51}},
/* 8666 */ {(16<<2)|3,{99,111,52}},
/* 8667 */ {(16<<2)|3,{99,111,53}},
/* 8668 */ {(16<<2)|3,{99,111,54}},
/* 8669 */ {(16<<2)|3,{99,111,55}},
/* 8670 */ {(16<<2)|3,{99,111,56}},
/* 8671 */ {(16<<2)|3,{99,111,57}},
/* 8672 */ {(16<<2)|3,{99,111,61}},
/* 8673 */ {(16<<2)|3,{99,111,65}},
/* 8674 */ {(16<<2)|3,{99,111,95}},
/* 8675 */ {(16<<2)|3,{99,111,98}},
/* 8676 */ {(16<<2)|3,{99,111,100}},
/* 8677 */ {(16<<2)|3,{99,111,102}},
/* 8678 */ {(16<<2)|3,{99,111,103}},
/* 8679 */ {(16<<2)|3,{99,111,104}},
/* 8680 */ {(16<<2)|3,{99,111,108}},
/* 8681 */ {(16<<2)|3,{99,111,109}},
/* 8682 */ {(16<<2)|3,{99,111,110}},
/* 8683 */ {(16<<2)|3,{99,111,112}},
/* 8684 */ {(16<<2)|3,{99,111,114}},
/* 8685 */ {(16<<2)|3,{99,111,117}},
/* 8686 */ {(10<<2)|2,{99,111,0}},
/* 8687 */ {(10<<2)|2,{99,111,0}},
/* 8688 */ {(10<<2)|2,{99,111,0}},
/* 8689 */ {(10<<2)|2,{99,111,0}},
/* 8690 */ {(10<<2)|2,{99,111,0}},
/* 8691 */ {(10<<2)|2,{99,111,0}},
/* 8692 */ {(10<<2)|2,{99,111,0}},
/* 8693 */ {(10<<2)|2,{99,111,0}},
/* 8694 */ {(10<<2)|2,{99,111,0}},
/* 8695 */ {(10<<2)|2,{99,111,0}},
/* 8696 */ {(10<<2)|2,{99,111,0}},
/* 8697 */ {(10<<2)|2,{99,111,0}},
/* 8698 */ {(10<<2)|2,{99,111,0}},
/* 8699 */ {(10<<2)|2,{99,111,0}},
/* 8700 */ {(10<<2)|2,{99,111,0}},
/* 8701 */ {(10<<2)|2,{99,111,0}},
/* 8702 */ {(10<<2)|2,{99,111,0}},
/* 8703 */ {(10<<2)|2,{99,111,0}},
/* 8704 */ {(15<<2)|3,{99,115,48}},
/* 8705 */ {(15<<2)|3,{99,115,48}},
/* 8706 */ {(15<<2)|3,{99,115,49}},
/* 8707 */ {(15<<2)|3,{99,115,49}},
/* 8708 */ {(15<<2)|3,{99,115,50}},
/* 8709 */ {(15<<2)|3,{99,115,50}},
/* 8710 */ {(15<<2)|3,{99,115,97}},
/* 8711 */ {(15<<2)|3,{99,115,97}},
/* 8712 */ {(15<<2)|3,{99,115,99}},
/* 8713 */ {(15<<2)|3,{99,115,99}},
/* 8714 */ {(15<<2)|3,{99,115,101}},
/* 8715 */ {(15<<2)|3,{99,115,101}},
/* 8716 */ {(15<<2)|3,{99,115,105}},
/* 8717 */ {(15<<2)|3,{99,115,105}},
/* 8718 */ {(15<<2)|3,{99,115,111}},
/* 8719 */ {(15<<2)|3,{99,115,111}},
/* 8720 */ {(15<<2)|3,{99,115,115}},
/* 8721 */ {(15<<2)|3,{99,115,115}},
/* 8722 */ {(15<<2)|3,{99,115,116}},
/* 8723 */ {(15<<2)|3,{99,115,116}},
/* 8724 */ {(16<<2)|3,{99,115,32}},
/* 8725 */ {(16<<2)|3,{99,115,37}},
/* 8726 */ {(16<<2)|3,{99,115,45}},
/* 8727 */ {(16<<2)|3,{99,115,46}},
/* 8728 */ {(16<<2)|3,{99,115,47}},
/* 8729 */ {(16<<2)|3,{99,115,51}},
/* 8730 */ {(16<<2)|3,{99,115,52}},
/* 8731 */ {(16<<2)|3,{99,115,53}},
/* 8732 */ {(16<<2)|3,{99,115,54}},
/* 8733 */ {(16<<2)|3,{99,115,55}},
/* 8734 */ {(16<<2)|3,{99,115,56}},
/* 8735 */ {(16<<2)|3,{99,115,57}},
/* 8736 */ {(16<<2)|3,{99,115,61}},
/* 8737 */ {(16<<2)|3,{99,115,65}},
/* 8738 */ {(16<<2)|3,{99,115,95}},
/* 8739 */ {(16<<2)|3,{99,115,98}},
/* 8740 */ {(16<<2)|3,{99,115,100}},
/* 8741 */ {(16<<2)|3,{99,115,102}},
/* 8742 */ {(16<<2)|3,{99,115,103}},
/* 8743 */ {(16<<2)|3,{99,115,104}},
/* 8744 */ {(16<<2)|3,{99,115,108}},
/* 8745 */ {(16<<2)|3,{99,115,109}},
/* 8746 */ {(16<<2)|3,{99,115,110}},
/* 8747 */ {(16<<2)|3,{99,115,112}},
/* 8748 */ {(16<<2)|3,{99,115,114}},
/* 8749 */ {(16<<2)|3,{99,115,117}},
/* 8750 */ {(10<<2)|2,{99,115,0}},
/* 8751 */ {(10<<2)|2,{99,115,0}},
/* 8752 */ {(10<<2)|2,{99,115,0}},
/* 8753 */ {(10<<2)|2,{99,115,0}},
/* 8754 */ {(10<<2)|2,{99,115,0}},
/* 8755 */ {(10<<2)|2,{99,115,0}},
/* 8756 */ {(10<<2)|2,{99,115,0}},
/* 8757 */ {(10<<2)|2,{99,115,0}},
/* 8758 */ {(10<<2)|2,{99,115,0}},
/* 8759 */ {(10<<2)|2,{99,115,0}},
/* 8760 */ {(10<<2)|2,{99,115,0}},
/* 8761 */ {(10<<2)|2,{99,115,0}},
/* 8762 */ {(10<<2)|2,{99,115,0}},
/* 8763 */ {(10<<2)|2,{99,115,0}},
/* 8764 */ {(10<<2)|2,{99,115,0}},
/* 8765 */ {(10<<2)|2,{99,115,0}},
/* 8766 */ {(10<<2)|2,{99,115,0}},
/* 8767 */ {(10<<2)|2,{99,115,0}},
/* 8768 */ {(15<<2)|3,{99,116,48}},
/* 8769 */ {(15<<2)|3,{99,116,48}},
/* 8770 */ {(15<<2)|3,{99,116,49}},
/* 8771 */ {(15<<2)|3,{99,116,49}},
/* 8772 */ {(15<<2)|3,{99,116,50}},
/* 8773 */ {(15<<2)|3,{99,116,50}},
/* 8774 */ {(15<<2)|3,{99,116,97}},
/* 8775 */ {(15<<2)|3,{99,116,97}},
/* 8776 */ {(15<<2)|3,{99,116,99}},
/* 8777 */ {(15<<2)|3,{99,116,99}},
/* 8778 */ {(15<<2)|3,{99,116,101}},
/* 8779 */ {(15<<2)|3,{99,116,101}},
/* 8780 */ {(15<<2)|3,{99,116,105}},
/* 8781 */ {(15<<2)|3,{99,116,105}},
/* 8782 */ {(15<<2)|3,{99,116,111}},
/* 8783 */ {(15<<2)|3,{99,116,111}},
/* 8784 */ {(15<<2)|3,{99,116,115}},
/* 8785 */ {(15<<2)|3,{99,116,115}},
/* 8786 */ {(15<<2)|3,{99,116,116}},
/* 8787 */ {(15<<2)|3,{99,116,116}},
/* 8788 */ {(16<<2)|3,{99,116,32}},
/* 8789 */ {(16<<2)|3,{99,116,37}},
/* 8790 */ {(16<<2)|3,{99,116,45}},
/* 8791 */ {(16<<2)|3,{99,116,46}},
/* 8792 */ {(16<<2)|3,{99,116,47}},
/* 8793 */ {(16<<2)|3,{99,116,51}},
/* 8794 */ {(16<<2)|3,{99,116,52}},
/* 8795 */ {(16<<2)|3,{99,116,53}},
/* 8796 */ {(16<<2)|3,{99,116,54}},
/* 8797 */ {(16<<2)|3,{99,116,55}},
/* 8798 */ {(16<<2)|3,{99,116,56}},
/* 8799 */ {(16<<2)|3,{99,116,57}},
/* 8800 */ {(16<<2)|3,{99,116,61}},
/* 8801 */ {(16<<2)|3,{99,116,65}},
/* 8802 */ {(16<<2)|3,{99,116,95}},
/* 8803 */ {(16<<2)|3,{99,116,98}},
/* 8804 */ {(16<<2)|3,{99,116,100}},
/* 8805 */ {(16<<2)|3,{99,116,102}},
/* 8806 */ {(16<<2)|3,{99,116,103}},
/* 8807 */ {(16<<2)|3,{99,116,104}},
/* 8808 */ {(16<<2)|3,{99,116,108}},
/* 8809 */ {(16<<2)|3,{99,116,109}},
/* 8810 */ {(16<<2)|3,{99,116,110}},
/* 8811 */ {(16<<2)|3,{99,116,112}},
/* 8812 */ {(16<<2)|3,{99,116,114}},
/* 8813 */ {(16<<2)|3,{99,116,117}},
/* 8814 */ {(10<<2)|2,{99,116,0}},
/* 8815 */ {(10<<2)|2,{99,116,0}},
/* 8816 */ {(10<<2)|2,{99,116,0}},
/* 8817 */ {(10<<2)|2,{99,116,0}},
/* 8818 */ {(10<<2)|2,{99,116,0}},
/* 8819 */ {(10<<2)|2,{99,116,0}},
/* 8820 */ {(10<<2)|2,{99,116,0}},
/* 8821 */ {(10<<2)|2,{99,116,0}},
/* 8822 */ {(10<<2)|2,{99,116,0}},
/* 8823 */ {(10<<2)|2,{99,116,0}},
/* 8824 */ {(10<<2)|2,{99,116,0}},
/* 8825 */ {(10<<2)|2,{99,116,0}},
/* 8826 */ {(10<<2)|2,{99,116,0}},
/* 8827 */ {(10<<2)|2,{99,116,0}},
/* 8828 */ {(10<<2)|2,{99,116,0}},
/* 8829 */ {(10<<2)|2,{99,116,0}},
/* 8830 */ {(10<<2)|2,{99,116,0}},
/* 8831 */ {(10<<2)|2,{99,116,0}},
/* 8832 */ {(16<<2)|3,{99,32,48}},
/* 8833 */ {(16<<2)|3,{99,32,49}},
/* 8834 */ {(16<<2)|3,{99,32,50}},
/* 8835 */ {(16<<2)|3,{99,32,97}},
/* 8836 */ {(16<<2)|3,{99,32,99}},
/* 8837 */ {(16<<2)|3,{99,32,101}},
/* 8838 */ {(16<<2)|3,{99,32,105}},
/* 8839 */ {(16<<2)|3,{99,32,111}},
/* 8840 */ {(16<<2)|3,{99,32,115}},
/* 8841 */ {(16<<2)|3,{99,32,116}},
/* 8842 */ {(11<<2)|2,{99,32,0}},
/* 8843 */ {(11<<2)|2,{99,32,0}},
/* 8844 */ {(11<<2)|2,{99,32,0}},
/* 8845 */ {(11<<2)|2,{99,32,0}},
/* 8846 */ {(11<<2)|2,{99,32,0}},
/* 8847 */ {(11<<2)|2,{99,32,0}},
/* 8848 */ {(11<<2)|2,{99,32,0}},
/* 8849 */ {(11<<2)|2,{99,32,0}},
/* 8850 */ {(11<<2)|2,{99,32,0}},
/* 8851 */ {(11<<2)|2,{99,32,0}},
/* 8852 */ {(11<<2)|2,{99,32,0}},
/* 8853 */ {(11<<2)|2,{99,32,0}},
/* 8854 */ {(11<<2)|2,{99,32,0}},
/* 8855 */ {(11<<2)|2,{99,32,0}},
/* 8856 */ {(11<<2)|2,{99,32,0}},
/* 8857 */ {(11<<2)|2,{99,32,0}},
/* 8858 */ {(11<<2)|2,{99,32,0}},
/* 8859 */ {(11<<2)|2,{99,32,0}},
/* 8860 */ {(11<<2)|2,{99,32,0}},
/* 8861 */ {(11<<2)|2,{99,32,0}},
/* 8862 */ {(11<<2)|2,{99,32,0}},
/* 8863 */ {(11<<2)|2,{99,32,0}},
/* 8864 */ {(16<<2)|3,{99,37,48}},
/* 8865 */ {(16<<2)|3,{99,37,49}},
/* 8866 */ {(16<<2)|3,{99,37,50}},
/* 8867 */ {(16<<2)|3,{99,37,97}},
/* 8868 */ {(16<<2)|3,{99,37,99}},
/* 8869 */ {(16<<2)|3,{99,37,101}},
/* 8870 */ {(16<<2)|3,{99,37,105}},
/* 8871 */ {(16<<2)|3,{99,37,111}},
/* 8872 */ {(16<<2)|3,{99,37,115}},
/* 8873 */ {(16<<2)|3,{99,37,116}},
/* 8874 */ {(11<<2)|2,{99,37,0}},
/* 8875 */ {(11<<2)|2,{99,37,0}},
/* 8876 */ {(11<<2)|2,{99,37,0}},
/* 8877 */ {(11<<2)|2,{99,37,0}},
/* 8878 */ {(11<<2)|2,{99,37,0}},
/* 8879 */ {(11<<2)|2,{99,37,0}},
/* 8880 */ {(11<<2)|2,{99,37,0}},
/* 8881 */ {(11<<2)|2,{99,37,0}},
/* 8882 */ {(11<<2)|2,{99,37,0}},
/* 8883 */ {(11<<2)|2,{99,37,0}},
/* 8884 */ {(11<<2)|2,{99,37,0}},
/* 8885 */ {(11<<2)|2,{99,37,0}},
/* 8886 */ {(11<<2)|2,{99,37,0}},
/* 8887 */ {(11<<2)|2,{99,37,0}},
/* 8888 */ {(11<<2)|2,{99,37,0}},
/* 8889 */ {(11<<2)|2,{99,37,0}},
/* 8890 */ {(11<<2)|2,{99,37,0}},
/* 8891 */ {(11<<2)|2,{99,37,0}},
/* 8892 */ {(11<<2)|2,{99,37,0}},
/* 8893 */ {(11<<2)|2,{99,37,0}},
/* 8894 */ {(11<<2)|2,{99,37,0}},
/* 8895 */ {(11<<2)|2,{99,37,0}},
/* 8896 */ {(16<<2)|3,{99,45,48}},
/* 8897 */ {(16<<2)|3,{99,45,49}},
/* 8898 */ {(16<<2)|3,{99,45,50}},
/* 8899 */ {(16<<2)|3,{99,45,97}},
/* 8900 */ {(16<<2)|3,{99,45,99}},
/* 8901 */ {(16<<2)|3,{99,45,101}},
/* 8902 */ {(16<<2)|3,{99,45,105}},
/* 8903 */ {(16<<2)|3,{99,45,111}},
/* 8904 */ {(16<<2)|3,{99,45,115}},
/* 8905 */ {(16<<2)|3,{99,45,116}},
/* 8906 */ {(11<<2)|2,{99,45,0}},
/* 8907 */ {(11<<2)|2,{99,45,0}},
/* 8908 */ {(11<<2)|2,{99,45,0}},
/* 8909 */ {(11<<2)|2,{99,45,0}},
/* 8910 */ {(11<<2)|2,{99,45,0}},
/* 8911 */ {(11<<2)|2,{99,45,0}},
/* 8912 */ {(11<<2)|2,{99,45,0}},
/* 8913 */ {(11<<2)|2,{99,45,0}},
/* 8914 */ {(11<<2)|2,{99,45,0}},
/* 8915 */ {(11<<2)|2,{99,45,0}},
/* 8916 */ {(11<<2)|2,{99,45,0}},
/* 8917 */ {(11<<2)|2,{99,45,0}},
/* 8918 */ {(11<<2)|2,{99,45,0}},
/* 8919 */ {(11<<2)|2,{99,45,0}},
/* 8920 */ {(11<<2)|2,{99,45,0}},
/* 8921 */ {(11<<2)|2,{99,45,0}},
/* 8922 */ {(11<<2)|2,{99,45,0}},
/* 8923 */ {(11<<2)|2,{99,45,0}},
/* 8924 */ {(11<<2)|2,{99,45,0}},
/* 8925 */ {(11<<2)|2,{99,45,0}},
/* 8926 */ {(11<<2)|2,{99,45,0}},
/* 8927 */ {(11<<2)|2,{99,45,0}},
/* 8928 */ {(16<<2)|3,{99,46,48}},
/* 8929 */ {(16<<2)|3,{99,46,49}},
/* 8930 */ {(16<<2)|3,{99,46,50}},
/* 8931 */ {(16<<2)|3,{99,46,97}},
/* 8932 */ {(16<<2)|3,{99,46,99}},
/* 8933 */ {(16<<2)|3,{99,46,101}},
/* 8934 */ {(16<<2)|3,{99,46,105}},
/* 8935 */ {(16<<2)|3,{99,46,111}},
/* 8936 */ {(16<<2)|3,{99,46,115}},
/* 8937 */ {(16<<2)|3,{99,46,116}},
/* 8938 */ {(11<<2)|2,{99,46,0}},
/* 8939 */ {(11<<2)|2,{99,46,0}},
/* 8940 */ {(11<<2)|2,{99,46,0}},
/* 8941 */ {(11<<2)|2,{99,46,0}},
/* 8942 */ {(11<<2)|2,{99,46,0}},
/* 8943 */ {(11<<2)|2,{99,46,0}},
/* 8944 */ {(11<<2)|2,{99,46,0}},
/* 8945 */ {(11<<2)|2,{99,46,0}},
/* 8946 */ {(11<<2)|2,{99,46,0}},
/* 8947 */ {(11<<2)|2,{99,46,0}},
/* 8948 */ {(11<<2)|2,{99,46,0}},
/* 8949 */ {(11<<2)|2,{99,46,0}},
/* 8950 */ {(11<<2)|2,{99,46,0}},
/* 8951 */ {(11<<2)|2,{99,46,0}},
/* 8952 */ {(11<<2)|2,{99,46,0}},
/* 8953 */ {(11<<2)|2,{99,46,0}},
/* 8954 */ {(11<<2)|2,{99,46,0}},
/* 8955 */ {(11<<2)|2,{99,46,0}},
/* 8956 */ {(11<<2)|2,{99,46,0}},
/* 8957 */ {(11<<2)|2,{99,46,0}},
/* 8958 */ {(11<<2)|2,{99,46,0}},
/* 8959 */ {(11<<2)|2,{99,46,0}},
/* 8960 */ {(16<<2)|3,{99,47,48}},
/* 8961 */ {(16<<2)|3,{99,47,49}},
/* 8962 */ {(16<<2)|3,{99,47,50}},
/* 8963 */ {(16<<2)|3,{99,47,97}},
/* 8964 */ {(16<<2)|3,{99,47,99}},
/* 8965 */ {(16<<2)|3,{99,47,101}},
/* 8966 */ {(16<<2)|3,{99,47,105}},
/* 8967 */ {(16<<2)|3,{99,47,111}},
/* 8968 */ {(16<<2)|3,{99,47,115}},
/* 8969 */ {(16<<2)|3,{99,47,116}},
/* 8970 */ {(11<<2)|2,{99,47,0}},
/* 8971 */ {(11<<2)|2,{99,47,0}},
/* 8972 */ {(11<<2)|2,{99,47,0}},
/* 8973 */ {(11<<2)|2,{99,47,0}},
/* 8974 */ {(11<<2)|2,{99,47,0}},
/* 8975 */ {(11<<2)|2,{99,47,0}},
/* 8976 */ {(11<<2)|2,{99,47,0}},
/* 8977 */ {(11<<2)|2,{99,47,0}},
/* 8978 */ {(11<<2)|2,{99,47,0}},
/* 8979 */ {(11<<2)|2,{99,47,0}},
/* 8980 */ {(11<<2)|2,{99,47,0}},
/* 8981 */ {(11<<2)|2,{99,47,0}},
/* 8982 */ {(11<<2)|2,{99,47,0}},
/* 8983 */ {(11<<2)|2,{99,47,0}},
/* 8984 */ {(11<<2)|2,{99,47,0}},
/* 8985 */ {(11<<2)|2,{99,47,0}},
/* 8986 */ {(11<<2)|2,{99,47,0}},
/* 8987 */ {(11<<2)|2,{99,47,0}},
/* 8988 */ {(11<<2)|2,{99,47,0}},
/* 8989 */ {(11<<2)|2,{99,47,0}},
/* 8990 */ {(11<<2)|2,{99,47,0}},
/* 8991 */ {(11<<2)|2,{99,47,0}},
/* 8992 */ {(16<<2)|3,{99,51,48}},
/* 8993 */ {(16<<2)|3,{99,51,49}},
/* 8994 */ {(16<<2)|3,{99,51,50}},
/* 8995 */ {(16<<2)|3,{99,51,97}},
/* 8996 */ {(16<<2)|3,{99,51,99}},
/* 8997 */ {(16<<2)|3,{99,51,101}},
/* 8998 */ {(16<<2)|3,{99,51,105}},
/* 8999 */ {(16<<2)|3,{99,51,111}},
/* 9000 */ {(16<<2)|3,{99,51,115}},
/* 9001 */ {(16<<2)|3,{99,51,116}},
/* 9002 */ {(11<<2)|2,{99,51,0}},
/* 9003 */ {(11<<2)|2,{99,51,0}},
/* 9004 */ {(11<<2)|2,{99,51,0}},
/* 9005 */ {(11<<2)|2,{99,51,0}},
/* 9006 */ {(11<<2)|2,{99,51,0}},
/* 9007 */ {(11<<2)|2,{99,51,0}},
/* 9008 */ {(11<<2)|2,{99,51,0}},
/* 9009 */ {(11<<2)|2,{99,51,0}},
/* 9010 */ {(11<<2)|2,{99,51,0}},
/* 9011 */ {(11<<2)|2,{99,51,0}},
/* 9012 */ {(11<<2)|2,{99,51,0}},
/* 9013 */ {(11<<2)|2,{99,51,0}},
/* 9014 */ {(11<<2)|2,{99,51,0}},
/* 9015 */ {(11<<2)|2,{99,51,0}},
/* 9016 */ {(11<<2)|2,{99,51,0}},
/* 9017 */ {(11<<2)|2,{99,51,0}},
/* 9018 */ {(11<<2)|2,{99,51,0}},
/* 9019 */ {(11<<2)|2,{99,51,0}},
/* 9020 */ {(11<<2)|2,{99,51,0}},
/* 9021 */ {(11<<2)|2,{99,51,0}},
/* 9022 */ {(11<<2)|2,{99,51,0}},
/* 9023 */ {(11<<2)|2,{99,51,0}},
/* 9024 */ {(16<<2)|3,{99,52,48}},
/* 9025 */ {(16<<2)|3,{99,52,49}},
/* 9026 */ {(16<<2)|3,{99,52,50}},
/* 9027 */ {(16<<2)|3,{99,52,97}},
/* 9028 */ {(16<<2)|3,{99,52,99}},
/* 9029 */ {(16<<2)|3,{99,52,101}},
/* 9030 */ {(16<<2)|3,{99,52,105}},
/* 9031 */ {(16<<2)|3,{99,52,111}},
/* 9032 */ {(16<<2)|3,{99,52,115}},
/* 9033 */ {(16<<2)|3,{99,52,116}},
/* 9034 */ {(11<<2)|2,{99,52,0}},
/* 9035 */ {(11<<2)|2,{99,52,0}},
/* 9036 */ {(11<<2)|2,{99,52,0}},
/* 9037 */ {(11<<2)|2,{99,52,0}},
/* 9038 */ {(11<<2)|2,{99,52,0}},
/* 9039 */ {(11<<2)|2,{99,52,0}},
/* 9040 */ {(11<<2)|2,{99,52,0}},
/* 9041 */ {(11<<2)|2,{99,52,0}},
/* 9042 */ {(11<<2)|2,{99,52,0}},
/* 9043 */ {(11<<2)|2,{99,52,0}},
/* 9044 */ {(11<<2)|2,{99,52,0}},
/* 9045 */ {(11<<2)|2,{99,52,0}},
/* 9046 */ {(11<<2)|2,{99,52,0}},
/* 9047 */ {(11<<2)|2,{99,52,0}},
/* 9048 */ {(11<<2)|2,{99,52,0}},
/* 9049 */ {(11<<2)|2,{99,52,0}},
/* 9050 */ {(11<<2)|2,{99,52,0}},
/* 9051 */ {(11<<2)|2,{99,52,0}},
/* 9052 */ {(11<<2)|2,{99,52,0}},
/* 9053 */ {(11<<2)|2,{99,52,0}},
/* 9054 */ {(11<<2)|2,{99,52,0}},
/* 9055 */ {(11<<2)|2,{99,52,0}},
/* 9056 */ {(16<<2)|3,{99,53,48}},
/* 9057 */ {(16<<2)|3,{99,53,49}},
/* 9058 */ {(16<<2)|3,{99,53,50}},
/* 9059 */ {(16<<2)|3,{99,53,97}},
/* 9060 */ {(16<<2)|3,{99,53,99}},
/* 9061 */ {(16<<2)|3,{99,53,101}},
/* 9062 */ {(16<<2)|3,{99,53,105}},
/* 9063 */ {(16<<2)|3,{99,53,111}},
/* 9064 */ {(16<<2)|3,{99,53,115}},
/* 9065 */ {(16<<2)|3,{99,53,116}},
/* 9066 */ {(11<<2)|2,{99,53,0}},
/* 9067 */ {(11<<2)|2,{99,53,0}},
/* 9068 */ {(11<<2)|2,{99,53,0}},
/* 9069 */ {(11<<2)|2,{99,53,0}},
/* 9070 */ {(11<<2)|2,{99,53,0}},
/* 9071 */ {(11<<2)|2,{99,53,0}},
/* 9072 */ {(11<<2)|2,{99,53,0}},
/* 9073 */ {(11<<2)|2,{99,53,0}},
/* 9074 */ {(11<<2)|2,{99,53,0}},
/* 9075 */ {(11<<2)|2,{99,53,0}},
/* 9076 */ {(11<<2)|2,{99,53,0}},
/* 9077 */ {(11<<2)|2,{99,53,0}},
/* 9078 */ {(11<<2)|2,{99,53,0}},
/* 9079 */ {(11<<2)|2,{99,53,0}},
/* 9080 */ {(11<<2)|2,{99,53,0}},
/* 9081 */ {(11<<2)|2,{99,53,0}},
/* 9082 */ {(11<<2)|2,{99,53,0}},
/* 9083 */ {(11<<2)|2,{99,53,0}},
/* 9084 */ {(11<<2)|2,{99,53,0}},
/* 9085 */ {(11<<2)|2,{99,53,0}},
/* 9086 */ {(11<<2)|2,{99,53,0}},
/* 9087 */ {(11<<2)|2,{99,53,0}},
/* 9088 */ {(16<<2)|3,{99,54,48}},
/* 9089 */ {(16<<2)|3,{99,54,49}},
/* 9090 */ {(16<<2)|3,{99,54,50}},
/* 9091 */ {(16<<2)|3,{99,54,97}},
/* 9092 */ {(16<<2)|3,{99,54,99}},
/* 9093 */ {(16<<2)|3,{99,54,101}},
/* 9094 */ {(16<<2)|3,{99,54,105}},
/* 9095 */ {(16<<2)|3,{99,54,111}},
/* 9096 */ {(16<<2)|3,{99,54,115}},
/* 9097 */ {(16<<2)|3,{99,54,116}},
/* 9098 */ {(11<<2)|2,{99,54,0}},
/* 9099 */ {(11<<2)|2,{99,54,0}},
/* 9100 */ {(11<<2)|2,{99,54,0}},
/* 9101 */ {(11<<2)|2,{99,54,0}},
/* 9102 */ {(11<<2)|2,{99,54,0}},
/* 9103 */ {(11<<2)|2,{99,54,0}},
/* 9104 */ {(11<<2)|2,{99,54,0}},
/* 9105 */ {(11<<2)|2,{99,54,0}},
/* 9106 */ {(11<<2)|2,{99,54,0}},
/* 9107 */ {(11<<2)|2,{99,54,0}},
/* 9108 */ {(11<<2)|2,{99,54,0}},
/* 9109 */ {(11<<2)|2,{99,54,0}},
/* 9110 */ {(11<<2)|2,{99,54,0}},
/* 9111 */ {(11<<2)|2,{99,54,0}},
/* 9112 */ {(11<<2)|2,{99,54,0}},
/* 9113 */ {(11<<2)|2,{99,54,0}},
/* 9114 */ {(11<<2)|2,{99,54,0}},
/* 9115 */ {(11<<2)|2,{99,54,0}},
/* 9116 */ {(11<<2)|2,{99,54,0}},
/* 9117 */ {(11<<2)|2,{99,54,0}},
/* 9118 */ {(11<<2)|2,{99,54,0}},
/* 9119 */ {(11<<2)|2,{99,54,0}},
/* 9120 */ {(16<<2)|3,{99,55,48}},
/* 9121 */ {(16<<2)|3,{99,55,49}},
/* 9122 */ {(16<<2)|3,{99,55,50}},
/* 9123 */ {(16<<2)|3,{99,55,97}},
/* 9124 */ {(16<<2)|3,{99,55,99}},
/* 9125 */ {(16<<2)|3,{99,55,101}},
/* 9126 */ {(16<<2)|3,{99,55,105}},
/* 9127 */ {(16<<2)|3,{99,55,111}},
/* 9128 */ {(16<<2)|3,{99,55,115}},
/* 9129 */ {(16<<2)|3,{99,55,116}},
/* 9130 */ {(11<<2)|2,{99,55,0}},
/* 9131 */ {(11<<2)|2,{99,55,0}},
/* 9132 */ {(11<<2)|2,{99,55,0}},
/* 9133 */ {(11<<2)|2,{99,55,0}},
/* 9134 */ {(11<<2)|2,{99,55,0}},
/* 9135 */ {(11<<2)|2,{99,55,0}},
/* 9136 */ {(11<<2)|2,{99,55,0}},
/* 9137 */ {(11<<2)|2,{99,55,0}},
/* 9138 */ {(11<<2)|2,{99,55,0}},
/* 9139 */ {(11<<2)|2,{99,55,0}},
/* 9140 */ {(11<<2)|2,{99,55,0}},
/* 9141 */ {(11<<2)|2,{99,55,0}},
/* 9142 */ {(11<<2)|2,{99,55,0}},
/* 9143 */ {(11<<2)|2,{99,55,0}},
/* 9144 */ {(11<<2)|2,{99,55,0}},
/* 9145 */ {(11<<2)|2,{99,55,0}},
/* 9146 */ {(11<<2)|2,{99,55,0}},
/* 9147 */ {(11<<2)|2,{99,55,0}},
/* 9148 */ {(11<<2)|2,{99,55,0}},
/* 9149 */ {(11<<2)|2,{99,55,0}},
/* 9150 */ {(11<<2)|2,{99,55,0}},
/* 9151 */ {(11<<2)|2,{99,55,0}},
/* 9152 */ {(16<<2)|3,{99,56,48}},
/* 9153 */ {(16<<2)|3,{99,56,49}},
/* 9154 */ {(16<<2)|3,{99,56,50}},
/* 9155 */ {(16<<2)|3,{99,56,97}},
/* 9156 */ {(16<<2)|3,{99,56,99}},
/* 9157 */ {(16<<2)|3,{99,56,101}},
/* 9158 */ {(16<<2)|3,{99,56,105}},
/* 9159 */ {(16<<2)|3,{99,56,111}},
/* 9160 */ {(16<<2)|3,{99,56,115}},
/* 9161 */ {(16<<2)|3,{99,56,116}},
/* 9162 */ {(11<<2)|2,{99,56,0}},
/* 9163 */ {(11<<2)|2,{99,56,0}},
/* 9164 */ {(11<<2)|2,{99,56,0}},
/* 9165 */ {(11<<2)|2,{99,56,0}},
/* 9166 */ {(11<<2)|2,{99,56,0}},
/* 9167 */ {(11<<2)|2,{99,56,0}},
/* 9168 */ {(11<<2)|2,{99,56,0}},
/* 9169 */ {(11<<2)|2,{99,56,0}},
/* 9170 */ {(11<<2)|2,{99,56,0}},
/* 9171 */ {(11<<2)|2,{99,56,0}},
/* 9172 */ {(11<<2)|2,{99,56,0}},
/* 9173 */ {(11<<2)|2,{99,56,0}},
/* 9174 */ {(11<<2)|2,{99,56,0}},
/* 9175 */ {(11<<2)|2,{99,56,0}},
/* 9176 */ {(11<<2)|2,{99,56,0}},
/* 9177 */ {(11<<2)|2,{99,56,0}},
/* 9178 */ {(11<<2)|2,{99,56,0}},
/* 9179 */ {(11<<2)|2,{99,56,0}},
/* 9180 */ {(11<<2)|2,{99,56,0}},
/* 9181 */ {(11<<2)|2,{99,56,0}},
/* 9182 */ {(11<<2)|2,{99,56,0}},
/* 9183 */ {(11<<2)|2,{99,56,0}},
/* 9184 */ {(16<<2)|3,{99,57,48}},
/* 9185 */ {(16<<2)|3,{99,57,49}},
/* 9186 */ {(16<<2)|3,{99,57,50}},
/* 9187 */ {(16<<2)|3,{99,57,97}},
/* 9188 */ {(16<<2)|3,{99,57,99}},
/* 9189 */ {(16<<2)|3,{99,57,101}},
/* 9190 */ {(16<<2)|3,{99,57,105}},
/* 9191 */ {(16<<2)|3,{99,57,111}},
/* 9192 */ {(16<<2)|3,{99,57,115}},
/* 9193 */ {(16<<2)|3,{99,57,116}},
/* 9194 */ {(11<<2)|2,{99,57,0}},
/* 9195 */ {(11<<2)|2,{99,57,0}},
/* 9196 */ {(11<<2)|2,{99,57,0}},
/* 9197 */ {(11<<2)|2,{99,57,0}},
/* 9198 */ {(11<<2)|2,{99,57,0}},
/* 9199 */ {(11<<2)|2,{99,57,0}},
/* 9200 */ {(11<<2)|2,{99,57,0}},
/* 9201 */ {(11<<2)|2,{99,57,0}},
/* 9202 */ {(11<<2)|2,{99,57,0}},
/* 9203 */ {(11<<2)|2,{99,57,0}},
/* 9204 */ {(11<<2)|2,{99,57,0}},
/* 9205 */ {(11<<2)|2,{99,57,0}},
/* 9206 */ {(11<<2)|2,{99,57,0}},
/* 9207 */ {(11<<2)|2,{99,57,0}},
/* 9208 */ {(11<<2)|2,{99,57,0}},
/* 9209 */ {(11<<2)|2,{99,57,0}},
/* 9210 */ {(11<<2)|2,{99,57,0}},
/* 9211 */ {(11<<2)|2,{99,57,0}},
/* 9212 */ {(11<<2)|2,{99,57,0}},
/* 9213 */ {(11<<2)|2,{99,57,0}},
/* 9214 */ {(11<<2)|2,{99,57,0}},
/* 9215 */ {(11<<2)|2,{99,57,0}},
/* 9216 */ {(16<<2)|3,{99,61,48}},
/* 9217 */ {(16<<2)|3,{99,61,49}},
/* 9218 */ {(16<<2)|3,{99,61,50}},
/* 9219 */ {(16<<2)|3,{99,61,97}},
/* 9220 */ {(16<<2)|3,{99,61,99}},
/* 9221 */ {(16<<2)|3,{99,61,101}},
/* 9222 */ {(16<<2)|3,{99,61,105}},
/* 9223 */ {(16<<2)|3,{99,61,111}},
/* 9224 */ {(16<<2)|3,{99,61,115}},
/* 9225 */ {(16<<2)|3,{99,61,116}},
/* 9226 */ {(11<<2)|2,{99,61,0}},
/* 9227 */ {(11<<2)|2,{99,61,0}},
/* 9228 */ {(11<<2)|2,{99,61,0}},
/* 9229 */ {(11<<2)|2,{99,61,0}},
/* 9230 */ {(11<<2)|2,{99,61,0}},
/* 9231 */ {(11<<2)|2,{99,61,0}},
/* 9232 */ {(11<<2)|2,{99,61,0}},
/* 9233 */ {(11<<2)|2,{99,61,0}},
/* 9234 */ {(11<<2)|2,{99,61,0}},
/* 9235 */ {(11<<2)|2,{99,61,0}},
/* 9236 */ {(11<<2)|2,{99,61,0}},
/* 9237 */ {(11<<2)|2,{99,61,0}},
/* 9238 */ {(11<<2)|2,{99,61,0}},
/* 9239 */ {(11<<2)|2,{99,61,0}},
/* 9240 */ {(11<<2)|2,{99,61,0}},
/* 9241 */ {(11<<2)|2,{99,61,0}},
/* 9242 */ {(11<<2)|2,{99,61,0}},
/* 9243 */ {(11<<2)|2,{99,61,0}},
/* 9244 */ {(11<<2)|2,{99,61,0}},
/* 9245 */ {(11<<2)|2,{99,61,0}},
/* 9246 */ {(11<<2)|2,{99,61,0}},
/* 9247 */ {(11<<2)|2,{99,61,0}},
/* 9248 */ {(16<<2)|3,{99,65,48}},
/* 9249 */ {(16<<2)|3,{99,65,49}},
/* 9250 */ {(16<<2)|3,{99,65,50}},
/* 9251 */ {(16<<2)|3,{99,65,97}},
/* 9252 */ {(16<<2)|3,{99,65,99}},
/* 9253 */ {(16<<2)|3,{99,65,101}},
/* 9254 */ {(16<<2)|3,{99,65,105}},
/* 9255 */ {(16<<2)|3,{99,65,111}},
/* 9256 */ {(16<<2)|3,{99,65,115}},
/* 9257 */ {(16<<2)|3,{99,65,116}},
/* 9258 */ {(11<<2)|2,{99,65,0}},
/* 9259 */ {(11<<2)|2,{99,65,0}},
/* 9260 */ {(11<<2)|2,{99,65,0}},
/* 9261 */ {(11<<2)|2,{99,65,0}},
/* 9262 */ {(11<<2)|2,{99,65,0}},
/* 9263 */ {(11<<2)|2,{99,65,0}},
/* 9264 */ {(11<<2)|2,{99,65,0}},
/* 9265 */ {(11<<2)|2,{99,65,0}},
/* 9266 */ {(11<<2)|2,{99,65,0}},
/* 9267 */ {(11<<2)|2,{99,65,0}},
/* 9268 */ {(11<<2)|2,{99,65,0}},
/* 9269 */ {(11<<2)|2,{99,65,0}},
/* 9270 */ {(11<<2)|2,{99,65,0}},
/* 9271 */ {(11<<2)|2,{99,65,0}},
/* 9272 */ {(11<<2)|2,{99,65,0}},
/* 9273 */ {(11<<2)|2,{99,65,0}},
/* 9274 */ {(11<<2)|2,{99,65,0}},
/* 9275 */ {(11<<2)|2,{99,65,0}},
/* 9276 */ {(11<<2)|2,{99,65,0}},
/* 9277 */ {(11<<2)|2,{99,65,0}},
/* 9278 */ {(11<<2)|2,{99,65,0}},
/* 9279 */ {(11<<2)|2,{99,65,0}},
/* 9280 */ {(16<<2)|3,{99,95,48}},
/* 9281 */ {(16<<2)|3,{99,95,49}},
/* 9282 */ {(16<<2)|3,{99,95,50}},
/* 9283 */ {(16<<2)|3,{99,95,97}},
/* 9284 */ {(16<<2)|3,{99,95,99}},
/* 9285 */ {(16<<2)|3,{99,95,101}},
/* 9286 */ {(16<<2)|3,{99,95,105}},
/* 9287 */ {(16<<2)|3,{99,95,111}},
/* 9288 */ {(16<<2)|3,{99,95,115}},
/* 9289 */ {(16<<2)|3,{99,95,116}},
/* 9290 */ {(11<<2)|2,{99,95,0}},
/* 9291 */ {(11<<2)|2,{99,95,0}},
/* 9292 */ {(11<<2)|2,{99,95,0}},
/* 9293 */ {(11<<2)|2,{99,95,0}},
/* 9294 */ {(11<<2)|2,{99,95,0}},
/* 9295 */ {(11<<2)|2,{99,95,0}},
/* 9296 */ {(11<<2)|2,{99,95,0}},
/* 9297 */ {(11<<2)|2,{99,95,0}},
/* 9298 */ {(11<<2)|2,{99,95,0}},
/* 9299 */ {(11<<2)|2,{99,95,0}},
/* 9300 */ {(11<<2)|2,{99,95,0}},
/* 9301 */ {(11<<2)|2,{99,95,0}},
/* 9302 */ {(11<<2)|2,{99,95,0}},
/* 9303 */ {(11<<2)|2,{99,95,0}},
/* 9304 */ {(11<<2)|2,{99,95,0}},
/* 9305 */ {(11<<2)|2,{99,95,0}},
/* 9306 */ {(11<<2)|2,{99,95,0}},
/* 9307 */ {(11<<2)|2,{99,95,0}},
/* 9308 */ {(11<<2)|2,{99,95,0}},
/* 9309 */ {(11<<2)|2,{99,95,0}},
/* 9310 */ {(11<<2)|2,{99,95,0}},
/* 9311 */ {(11<<2)|2,{99,95,0}},
/* 9312 */ {(16<<2)|3,{99,98,48}},
/* 9313 */ {(16<<2)|3,{99,98,49}},
/* 9314 */ {(16<<2)|3,{99,98,50}},
/* 9315 */ {(16<<2)|3,{99,98,97}},
/* 9316 */ {(16<<2)|3,{99,98,99}},
/* 9317 */ {(16<<2)|3,{99,98,101}},
/* 9318 */ {(16<<2)|3,{99,98,105}},
/* 9319 */ {(16<<2)|3,{99,98,111}},
/* 9320 */ {(16<<2)|3,{99,98,115}},
/* 9321 */ {(16<<2)|3,{99,98,116}},
/* 9322 */ {(11<<2)|2,{99,98,0}},
/* 9323 */ {(11<<2)|2,{99,98,0}},
/* 9324 */ {(11<<2)|2,{99,98,0}},
/* 9325 */ {(11<<2)|2,{99,98,0}},
/* 9326 */ {(11<<2)|2,{99,98,0}},
/* 9327 */ {(11<<2)|2,{99,98,0}},
/* 9328 */ {(11<<2)|2,{99,98,0}},
/* 9329 */ {(11<<2)|2,{99,98,0}},
/* 9330 */ {(11<<2)|2,{99,98,0}},
/* 9331 */ {(11<<2)|2,{99,98,0}},
/* 9332 */ {(11<<2)|2,{99,98,0}},
/* 9333 */ {(11<<2)|2,{99,98,0}},
/* 9334 */ {(11<<2)|2,{99,98,0}},
/* 9335 */ {(11<<2)|2,{99,98,0}},
/* 9336 */ {(11<<2)|2,{99,98,0}},
/* 9337 */ {(11<<2)|2,{99,98,0}},
/* 9338 */ {(11<<2)|2,{99,98,0}},
/* 9339 */ {(11<<2)|2,{99,98,0}},
/* 9340 */ {(11<<2)|2,{99,98,0}},
/* 9341 */ {(11<<2)|2,{99,98,0}},
/* 9342 */ {(11<<2)|2,{99,98,0}},
/* 9343 */ {(11<<2)|2,{99,98,0}},
/* 9344 */ {(16<<2)|3,{99,100,48}},
/* 9345 */ {(16<<2)|3,{99,100,49}},
/* 9346 */ {(16<<2)|3,{99,100,50}},
/* 9347 */ {(16<<2)|3,{99,100,97}},
/* 9348 */ {(16<<2)|3,{99,100,99}},
/* 9349 */ {(16<<2)|3,{99,100,101}},
/* 9350 */ {(16<<2)|3,{99,100,105}},
/* 9351 */ {(16<<2)|3,{99,100,111}},
/* 9352 */ {(16<<2)|3,{99,100,115}},
/* 9353 */ {(16<<2)|3,{99,100,116}},
/* 9354 */ {(11<<2)|2,{99,100,0}},
/* 9355 */ {(11<<2)|2,{99,100,0}},
/* 9356 */ {(11<<2)|2,{99,100,0}},
/* 9357 */ {(11<<2)|2,{99,100,0}},
/* 9358 */ {(11<<2)|2,{99,100,0}},
/* 9359 */ {(11<<2)|2,{99,100,0}},
/* 9360 */ {(11<<2)|2,{99,100,0}},
/* 9361 */ {(11<<2)|2,{99,100,0}},
/* 9362 */ {(11<<2)|2,{99,100,0}},
/* 9363 */ {(11<<2)|2,{99,100,0}},
/* 9364 */ {(11<<2)|2,{99,100,0}},
/* 9365 */ {(11<<2)|2,{99,100,0}},
/* 9366 */ {(11<<2)|2,{99,100,0}},
/* 9367 */ {(11<<2)|2,{99,100,0}},
/* 9368 */ {(11<<2)|2,{99,100,0}},
/* 9369 */ {(11<<2)|2,{99,100,0}},
/* 9370 */ {(11<<2)|2,{99,100,0}},
/* 9371 */ {(11<<2)|2,{99,100,0}},
/* 9372 */ {(11<<2)|2,{99,100,0}},
/* 9373 */ {(11<<2)|2,{99,100,0}},
/* 9374 */ {(11<<2)|2,{99,100,0}},
/* 9375 */ {(11<<2)|2,{99,100,0}},
/* 9376 */ {(16<<2)|3,{99,102,48}},
/* 9377 */ {(16<<2)|3,{99,102,49}},
/* 9378 */ {(16<<2)|3,{99,102,50}},
/* 9379 */ {(16<<2)|3,{99,102,97}},
/* 9380 */ {(16<<2)|3,{99,102,99}},
/* 9381 */ {(16<<2)|3,{99,102,101}},
/* 9382 */ {(16<<2)|3,{99,102,105}},
/* 9383 */ {(16<<2)|3,{99,102,111}},
/* 9384 */ {(16<<2)|3,{99,102,115}},
/* 9385 */ {(16<<2)|3,{99,102,116}},
/* 9386 */ {(11<<2)|2,{99,102,0}},
/* 9387 */ {(11<<2)|2,{99,102,0}},
/* 9388 */ {(11<<2)|2,{99,102,0}},
/* 9389 */ {(11<<2)|2,{99,102,0}},
/* 9390 */ {(11<<2)|2,{99,102,0}},
/* 9391 */ {(11<<2)|2,{99,102,0}},
/* 9392 */ {(11<<2)|2,{99,102,0}},
/* 9393 */ {(11<<2)|2,{99,102,0}},
/* 9394 */ {(11<<2)|2,{99,102,0}},
/* 9395 */ {(11<<2)|2,{99,102,0}},
/* 9396 */ {(11<<2)|2,{99,102,0}},
/* 9397 */ {(11<<2)|2,{99,102,0}},
/* 9398 */ {(11<<2)|2,{99,102,0}},
/* 9399 */ {(11<<2)|2,{99,102,0}},
/* 9400 */ {(11<<2)|2,{99,102,0}},
/* 9401 */ {(11<<2)|2,{99,102,0}},
/* 9402 */ {(11<<2)|2,{99,102,0}},
/* 9403 */ {(11<<2)|2,{99,102,0}},
/* 9404 */ {(11<<2)|2,{99,102,0}},
/* 9405 */ {(11<<2)|2,{99,102,0}},
/* 9406 */ {(11<<2)|2,{99,102,0}},
/* 9407 */ {(11<<2)|2,{99,102,0}},
/* 9408 */ {(16<<2)|3,{99,103,48}},
/* 9409 */ {(16<<2)|3,{99,103,49}},
/* 9410 */ {(16<<2)|3,{99,103,50}},
/* 9411 */ {(16<<2)|3,{99,103,97}},
/* 9412 */ {(16<<2)|3,{99,103,99}},
/* 9413 */ {(16<<2)|3,{99,103,101}},
/* 9414 */ {(16<<2)|3,{99,103,105}},
/* 9415 */ {(16<<2)|3,{99,103,111}},
/* 9416 */ {(16<<2)|3,{99,103,115}},
/* 9417 */ {(16<<2)|3,{99,103,116}},
/* 9418 */ {(11<<2)|2,{99,103,0}},
/* 9419 */ {(11<<2)|2,{99,103,0}},
/* 9420 */ {(11<<2)|2,{99,103,0}},
/* 9421 */ {(11<<2)|2,{99,103,0}},
/* 9422 */ {(11<<2)|2,{99,103,0}},
/* 9423 */ {(11<<2)|2,{99,103,0}},
/* 9424 */ {(11<<2)|2,{99,103,0}},
/* 9425 */ {(11<<2)|2,{99,103,0}},
/* 9426 */ {(11<<2)|2,{99,103,0}},
/* 9427 */ {(11<<2)|2,{99,103,0}},
/* 9428 */ {(11<<2)|2,{99,103,0}},
/* 9429 */ {(11<<2)|2,{99,103,0}},
/* 9430 */ {(11<<2)|2,{99,103,0}},
/* 9431 */ {(11<<2)|2,{99,103,0}},
/* 9432 */ {(11<<2)|2,{99,103,0}},
/* 9433 */ {(11<<2)|2,{99,103,0}},
/* 9434 */ {(11<<2)|2,{99,103,0}},
/* 9435 */ {(11<<2)|2,{99,103,0}},
/* 9436 */ {(11<<2)|2,{99,103,0}},
/* 9437 */ {(11<<2)|2,{99,103,0}},
/* 9438 */ {(11<<2)|2,{99,103,0}},
/* 9439 */ {(11<<2)|2,{99,103,0}},
/* 9440 */ {(16<<2)|3,{99,104,48}},
/* 9441 */ {(16<<2)|3,{99,104,49}},
/* 9442 */ {(16<<2)|3,{99,104,50}},
/* 9443 */ {(16<<2)|3,{99,104,97}},
/* 9444 */ {(16<<2)|3,{99,104,99}},
/* 9445 */ {(16<<2)|3,{99,104,101}},
/* 9446 */ {(16<<2)|3,{99,104,105}},
/* 9447 */ {(16<<2)|3,{99,104,111}},
/* 9448 */ {(16<<2)|3,{99,104,115}},
/* 9449 */ {(16<<2)|3,{99,104,116}},
/* 9450 */ {(11<<2)|2,{99,104,0}},
/* 9451 */ {(11<<2)|2,{99,104,0}},
/* 9452 */ {(11<<2)|2,{99,104,0}},
/* 9453 */ {(11<<2)|2,{99,104,0}},
/* 9454 */ {(11<<2)|2,{99,104,0}},
/* 9455 */ {(11<<2)|2,{99,104,0}},
/* 9456 */ {(11<<2)|2,{99,104,0}},
/* 9457 */ {(11<<2)|2,{99,104,0}},
/* 9458 */ {(11<<2)|2,{99,104,0}},
/* 9459 */ {(11<<2)|2,{99,104,0}},
/* 9460 */ {(11<<2)|2,{99,104,0}},
/* 9461 */ {(11<<2)|2,{99,104,0}},
/* 9462 */ {(11<<2)|2,{99,104,0}},
/* 9463 */ {(11<<2)|2,{99,104,0}},
/* 9464 */ {(11<<2)|2,{99,104,0}},
/* 9465 */ {(11<<2)|2,{99,104,0}},
/* 9466 */ {(11<<2)|2,{99,104,0}},
/* 9467 */ {(11<<2)|2,{99,104,0}},
/* 9468 */ {(11<<2)|2,{99,104,0}},
/* 9469 */ {(11<<2)|2,{99,104,0}},
/* 9470 */ {(11<<2)|2,{99,104,0}},
/* 9471 */ {(11<<2)|2,{99,104,0}},
/* 9472 */ {(16<<2)|3,{99,108,48}},
/* 9473 */ {(16<<2)|3,{99,108,49}},
/* 9474 */ {(16<<2)|3,{99,108,50}},
/* 9475 */ {(16<<2)|3,{99,108,97}},
/* 9476 */ {(16<<2)|3,{99,108,99}},
/* 9477 */ {(16<<2)|3,{99,108,101}},
/* 9478 */ {(16<<2)|3,{99,108,105}},
/* 9479 */ {(16<<2)|3,{99,108,111}},
/* 9480 */ {(16<<2)|3,{99,108,115}},
/* 9481 */ {(16<<2)|3,{99,108,116}},
/* 9482 */ {(11<<2)|2,{99,108,0}},
/* 9483 */ {(11<<2)|2,{99,108,0}},
/* 9484 */ {(11<<2)|2,{99,108,0}},
/* 9485 */ {(11<<2)|2,{99,108,0}},
/* 9486 */ {(11<<2)|2,{99,108,0}},
/* 9487 */ {(11<<2)|2,{99,108,0}},
/* 9488 */ {(11<<2)|2,{99,108,0}},
/* 9489 */ {(11<<2)|2,{99,108,0}},
/* 9490 */ {(11<<2)|2,{99,108,0}},
/* 9491 */ {(11<<2)|2,{99,108,0}},
/* 9492 */ {(11<<2)|2,{99,108,0}},
/* 9493 */ {(11<<2)|2,{99,108,0}},
/* 9494 */ {(11<<2)|2,{99,108,0}},
/* 9495 */ {(11<<2)|2,{99,108,0}},
/* 9496 */ {(11<<2)|2,{99,108,0}},
/* 9497 */ {(11<<2)|2,{99,108,0}},
/* 9498 */ {(11<<2)|2,{99,108,0}},
/* 9499 */ {(11<<2)|2,{99,108,0}},
/* 9500 */ {(11<<2)|2,{99,108,0}},
/* 9501 */ {(11<<2)|2,{99,108,0}},
/* 9502 */ {(11<<2)|2,{99,108,0}},
/* 9503 */ {(11<<2)|2,{99,108,0}},
/* 9504 */ {(16<<2)|3,{99,109,48}},
/* 9505 */ {(16<<2)|3,{99,109,49}},
/* 9506 */ {(16<<2)|3,{99,109,50}},
/* 9507 */ {(16<<2)|3,{99,109,97}},
/* 9508 */ {(16<<2)|3,{99,109,99}},
/* 9509 */ {(16<<2)|3,{99,109,101}},
/* 9510 */ {(16<<2)|3,{99,109,105}},
/* 9511 */ {(16<<2)|3,{99,109,111}},
/* 9512 */ {(16<<2)|3,{99,109,115}},
/* 9513 */ {(16<<2)|3,{99,109,116}},
/* 9514 */ {(11<<2)|2,{99,109,0}},
/* 9515 */ {(11<<2)|2,{99,109,0}},
/* 9516 */ {(11<<2)|2,{99,109,0}},
/* 9517 */ {(11<<2)|2,{99,109,0}},
/* 9518 */ {(11<<2)|2,{99,109,0}},
/* 9519 */ {(11<<2)|2,{99,109,0}},
/* 9520 */ {(11<<2)|2,{99,109,0}},
/* 9521 */ {(11<<2)|2,{99,109,0}},
/* 9522 */ {(11<<2)|2,{99,109,0}},
/* 9523 */ {(11<<2)|2,{99,109,0}},
/* 9524 */ {(11<<2)|2,{99,109,0}},
/* 9525 */ {(11<<2)|2,{99,109,0}},
/* 9526 */ {(11<<2)|2,{99,109,0}},
/* 9527 */ {(11<<2)|2,{99,109,0}},
/* 9528 */ {(11<<2)|2,{99,109,0}},
/* 9529 */ {(11<<2)|2,{99,109,0}},
/* 9530 */ {(11<<2)|2,{99,109,0}},
/* 9531 */ {(11<<2)|2,{99,109,0}},
/* 9532 */ {(11<<2)|2,{99,109,0}},
/* 9533 */ {(11<<2)|2,{99,109,0}},
/* 9534 */ {(11<<2)|2,{99,109,0}},
/* 9535 */ {(11<<2)|2,{99,109,0}},
/* 9536 */ {(16<<2)|3,{99,110,48}},
/* 9537 */ {(16<<2)|3,{99,110,49}},
/* 9538 */ {(16<<2)|3,{99,110,50}},
/* 9539 */ {(16<<2)|3,{99,110,97}},
/* 9540 */ {(16<<2)|3,{99,110,99}},
/* 9541 */ {(16<<2)|3,{99,110,101}},
/* 9542 */ {(16<<2)|3,{99,110,105}},
/* 9543 */ {(16<<2)|3,{99,110,111}},
/* 9544 */ {(16<<2)|3,{99,110,115}},
/* 9545 */ {(16<<2)|3,{99,110,116}},
/* 9546 */ {(11<<2)|2,{99,110,0}},
/* 9547 */ {(11<<2)|2,{99,110,0}},
/* 9548 */ {(11<<2)|2,{99,110,0}},
/* 9549 */ {(11<<2)|2,{99,110,0}},
/* 9550 */ {(11<<2)|2,{99,110,0}},
/* 9551 */ {(11<<2)|2,{99,110,0}},
/* 9552 */ {(11<<2)|2,{99,110,0}},
/* 9553 */ {(11<<2)|2,{99,110,0}},
/* 9554 */ {(11<<2)|2,{99,110,0}},
/* 9555 */ {(11<<2)|2,{99,110,0}},
/* 9556 */ {(11<<2)|2,{99,110,0}},
/* 9557 */ {(11<<2)|2,{99,110,0}},
/* 9558 */ {(11<<2)|2,{99,110,0}},
/* 9559 */ {(11<<2)|2,{99,110,0}},
/* 9560 */ {(11<<2)|2,{99,110,0}},
/* 9561 */ {(11<<2)|2,{99,110,0}},
/* 9562 */ {(11<<2)|2,{99,110,0}},
/* 9563 */ {(11<<2)|2,{99,110,0}},
/* 9564 */ {(11<<2)|2,{99,110,0}},
/* 9565 */ {(11<<2)|2,{99,110,0}},
/* 9566 */ {(11<<2)|2,{99,110,0}},
/* 9567 */ {(11<<2)|2,{99,110,0}},
/* 9568 */ {(16<<2)|3,{99,112,48}},
/* 9569 */ {(16<<2)|3,{99,112,49}},
/* 9570 */ {(16<<2)|3,{99,112,50}},
/* 9571 */ {(16<<2)|3,{99,112,97}},
/* 9572 */ {(16<<2)|3,{99,112,99}},
/* 9573 */ {(16<<2)|3,{99,112,101}},
/* 9574 */ {(16<<2)|3,{99,112,105}},
/* 9575 */ {(16<<2)|3,{99,112,111}},
/* 9576 */ {(16<<2)|3,{99,112,115}},
/* 9577 */ {(16<<2)|3,{99,112,116}},
/* 9578 */ {(11<<2)|2,{99,112,0}},
/* 9579 */ {(11<<2)|2,{99,112,0}},
/* 9580 */ {(11<<2)|2,{99,112,0}},
/* 9581 */ {(11<<2)|2,{99,112,0}},
/* 9582 */ {(11<<2)|2,{99,112,0}},
/* 9583 */ {(11<<2)|2,{99,112,0}},
/* 9584 */ {(11<<2)|2,{99,112,0}},
/* 9585 */ {(11<<2)|2,{99,112,0}},
/* 9586 */ {(11<<2)|2,{99,112,0}},
/* 9587 */ {(11<<2)|2,{99,112,0}},
/* 9588 */ {(11<<2)|2,{99,112,0}},
/* 9589 */ {(11<<2)|2,{99,112,0}},
/* 9590 */ {(11<<2)|2,{99,112,0}},
/* 9591 */ {(11<<2)|2,{99,112,0}},
/* 9592 */ {(11<<2)|2,{99,112,0}},
/* 9593 */ {(11<<2)|2,{99,112,0}},
/* 9594 */ {(11<<2)|2,{99,112,0}},
/* 9595 */ {(11<<2)|2,{99,112,0}},
/* 9596 */ {(11<<2)|2,{99,112,0}},
/* 9597 */ {(11<<2)|2,{99,112,0}},
/* 9598 */ {(11<<2)|2,{99,112,0}},
/* 9599 */ {(11<<2)|2,{99,112,0}},
/* 9600 */ {(16<<2)|3,{99,114,48}},
/* 9601 */ {(16<<2)|3,{99,114,49}},
/* 9602 */ {(16<<2)|3,{99,114,50}},
/* 9603 */ {(16<<2)|3,{99,114,97}},
/* 9604 */ {(16<<2)|3,{99,114,99}},
/* 9605 */ {(16<<2)|3,{99,114,101}},
/* 9606 */ {(16<<2)|3,{99,114,105}},
/* 9607 */ {(16<<2)|3,{99,114,111}},
/* 9608 */ {(16<<2)|3,{99,114,115}},
/* 9609 */ {(16<<2)|3,{99,114,116}},
/* 9610 */ {(11<<2)|2,{99,114,0}},
/* 9611 */ {(11<<2)|2,{99,114,0}},
/* 9612 */ {(11<<2)|2,{99,114,0}},
/* 9613 */ {(11<<2)|2,{99,114,0}},
/* 9614 */ {(11<<2)|2,{99,114,0}},
/* 9615 */ {(11<<2)|2,{99,114,0}},
/* 9616 */ {(11<<2)|2,{99,114,0}},
/* 9617 */ {(11<<2)|2,{99,114,0}},
/* 9618 */ {(11<<2)|2,{99,114,0}},
/* 9619 */ {(11<<2)|2,{99,114,0}},
/* 9620 */ {(11<<2)|2,{99,114,0}},
/* 9621 */ {(11<<2)|2,{99,114,0}},
/* 9622 */ {(11<<2)|2,{99,114,0}},
/* 9623 */ {(11<<2)|2,{99,114,0}},
/* 9624 */ {(11<<2)|2,{99,114,0}},
/* 9625 */ {(11<<2)|2,{99,114,0}},
/* 9626 */ {(11<<2)|2,{99,114,0}},
/* 9627 */ {(11<<2)|2,{99,114,0}},
/* 9628 */ {(11<<2)|2,{99,114,0}},
/* 9629 */ {(11<<2)|2,{99,114,0}},
/* 9630 */ {(11<<2)|2,{99,114,0}},
/* 9631 */ {(11<<2)|2,{99,114,0}},
/* 9632 */ {(16<<2)|3,{99,117,48}},
/* 9633 */ {(16<<2)|3,{99,117,49}},
/* 9634 */ {(16<<2)|3,{99,117,50}},
/* 9635 */ {(16<<2)|3,{99,117,97}},
/* 9636 */ {(16<<2)|3,{99,117,99}},
/* 9637 */ {(16<<2)|3,{99,117,101}},
/* 9638 */ {(16<<2)|3,{99,117,105}},
/* 9639 */ {(16<<2)|3,{99,117,111}},
/* 9640 */ {(16<<2)|3,{99,117,115}},
/* 9641 */ {(16<<2)|3,{99,117,116}},
/* 9642 */ {(11<<2)|2,{99,117,0}},
/* 9643 */ {(11<<2)|2,{99,117,0}},
/* 9644 */ {(11<<2)|2,{99,117,0}},
/* 9645 */ {(11<<2)|2,{99,117,0}},
/* 9646 */ {(11<<2)|2,{99,117,0}},
/* 9647 */ {(11<<2)|2,{99,117,0}},
/* 9648 */ {(11<<2)|2,{99,117,0}},
/* 9649 */ {(11<<2)|2,{99,117,0}},
/* 9650 */ {(11<<2)|2,{99,117,0}},
/* 9651 */ {(11<<2)|2,{99,117,0}},
/* 9652 */ {(11<<2)|2,{99,117,0}},
/* 9653 */ {(11<<2)|2,{99,117,0}},
/* 9654 */ {(11<<2)|2,{99,117,0}},
/* 9655 */ {(11<<2)|2,{99,117,0}},
/* 9656 */ {(11<<2)|2,{99,117,0}},
/* 9657 */ {(11<<2)|2,{99,117,0}},
/* 9658 */ {(11<<2)|2,{99,117,0}},
/* 9659 */ {(11<<2)|2,{99,117,0}},
/* 9660 */ {(11<<2)|2,{99,117,0}},
/* 9661 */ {(11<<2)|2,{99,117,0}},
/* 9662 */ {(11<<2)|2,{99,117,0}},
/* 9663 */ {(11<<2)|2,{99,117,0}},
/* 9664 */ {(12<<2)|2,{99,58,0}},
/* 9665 */ {(12<<2)|2,{99,58,0}},
/* 9666 */ {(12<<2)|2,{99,58,0}},
/* 9667 */ {(12<<2)|2,{99,58,0}},
/* 9668 */ {(12<<2)|2,{99,58,0}},
/* 9669 */ {(12<<2)|2,{99,58,0}},
/* 9670 */ {(12<<2)|2,{99,58,0}},
/* 9671 */ {(12<<2)|2,{99,58,0}},
/* 9672 */ {(12<<2)|2,{99,58,0}},
/* 9673 */ {(12<<2)|2,{99,58,0}},
/* 9674 */ {(12<<2)|2,{99,58,0}},
/* 9675 */ {(12<<2)|2,{99,58,0}},
/* 9676 */ {(12<<2)|2,{99,58,0}},
/* 9677 */ {(12<<2)|2,{99,58,0}},
/* 9678 */ {(12<<2)|2,{99,58,0}},
/* 9679 */ {(12<<2)|2,{99,58,0}},
/* 9680 */ {(12<<2)|2,{99,66,0}},
/* 9681 */ {(12<<2)|2,{99,66,0}},
/* 9682 */ {(12<<2)|2,{99,66,0}},
/* 9683 */ {(12<<2)|2,{99,66,0}},
/* 9684 */ {(12<<2)|2,{99,66,0}},
/* 9685 */ {(12<<2)|2,{99,66,0}},
/* 9686 */ {(12<<2)|2,{99,66,0}},
/* 9687 */ {(12<<2)|2,{99,66,0}},
/* 9688 */ {(12<<2)|2,{99,66,0}},
/* 9689 */ {(12<<2)|2,{99,66,0}},
/* 9690 */ {(12<<2)|2,{99,66,0}},
/* 9691 */ {(12<<2)|2,{99,66,0}},
/* 9692 */ {(12<<2)|2,{99,66,0}},
/* 9693 */ {(12<<2)|2,{99,66,0}},
/* 9694 */ {(12<<2)|2,{99,66,0}},
/* 9695 */ {(12<<2)|2,{99,66,0}},
/* 9696 */ {(12<<2)|2,{99,67,0}},
/* 9697 */ {(12<<2)|2,{99,67,0}},
/* 9698 */ {(12<<2)|2,{99,67,0}},
/* 9699 */ {(12<<2)|2,{99,67,0}},
/* 9700 */ {(12<<2)|2,{99,67,0}},
/* 9701 */ {(12<<2)|2,{99,67,0}},
/* 9702 */ {(12<<2)|2,{99,67,0}},
/* 9703 */ {(12<<2)|2,{99,67,0}},
/* 9704 */ {(12<<2)|2,{99,67,0}},
/* 9705 */ {(12<<2)|2,{99,67,0}},
/* 9706 */ {(12<<2)|2,{99,67,0}},
/* 9707 */ {(12<<2)|2,{99,67,0}},
/* 9708 */ {(12<<2)|2,{99,67,0}},
/* 9709 */ {(12<<2)|2,{99,67,0}},
/* 9710 */ {(12<<2)|2,{99,67,0}},
/* 9711 */ {(12<<2)|2,{99,67,0}},
/* 9712 */ {(12<<2)|2,{99,68,0}},
/* 9713 */ {(12<<2)|2,{99,68,0}},
/* 9714 */ {(12<<2)|2,{99,68,0}},
/* 9715 */ {(12<<2)|2,{99,68,0}},
/* 9716 */ {(12<<2)|2,{99,68,0}},
/* 9717 */ {(12<<2)|2,{99,68,0}},
/* 9718 */ {(12<<2)|2,{99,68,0}},
/* 9719 */ {(12<<2)|2,{99,68,0}},
/* 9720 */ {(12<<2)|2,{99,68,0}},
/* 9721 */ {(12<<2)|2,{99,68,0}},
/* 9722 */ {(12<<2)|2,{99,68,0}},
/* 9723 */ {(12<<2)|2,{99,68,0}},
/* 9724 */ {(12<<2)|2,{99,68,0}},
/* 9725 */ {(12<<2)|2,{99,68,0}},
/* 9726 */ {(12<<2)|2,{99,68,0}},
/* 9727 */ {(12<<2)|2,{99,68,0}},
/* 9728 */ {(12<<2)|2,{99,69,0}},
/* 9729 */ {(12<<2)|2,{99,69,0}},
/* 9730 */ {(12<<2)|2,{99,69,0}},
/* 9731 */ {(12<<2)|2,{99,69,0}},
/* 9732 */ {(12<<2)|2,{99,69,0}},
/* 9733 */ {(12<<2)|2,{99,69,0}},
/* 9734 */ {(12<<2)|2,{99,69,0}},
/* 9735 */ {(12<<2)|2,{99,69,0}},
/* 9736 */ {(12<<2)|2,{99,69,0}},
/* 9737 */ {(12<<2)|2,{99,69,0}},
/* 9738 */ {(12<<2)|2,{99,69,0}},
/* 9739 */ {(12<<2)|2,{99,69,0}},
/* 9740 */ {(12<<2)|2,{99,69,0}},
/* 9741 */ {(12<<2)|2,{99,69,0}},
/* 9742 */ {(12<<2)|2,{99,69,0}},
/* 9743 */ {(12<<2)|2,{99,69,0}},
/* 9744 */ {(12<<2)|2,{99,70,0}},
/* 9745 */ {(12<<2)|2,{99,70,0}},
/* 9746 */ {(12<<2)|2,{99,70,0}},
/* 9747 */ {(12<<2)|2,{99,70,0}},
/* 9748 */ {(12<<2)|2,{99,70,0}},
/* 9749 */ {(12<<2)|2,{99,70,0}},
/* 9750 */ {(12<<2)|2,{99,70,0}},
/* 9751 */ {(12<<2)|2,{99,70,0}},
/* 9752 */ {(12<<2)|2,{99,70,0}},
/* 9753 */ {(12<<2)|2,{99,70,0}},
/* 9754 */ {(12<<2)|2,{99,70,0}},
/* 9755 */ {(12<<2)|2,{99,70,0}},
/* 9756 */ {(12<<2)|2,{99,70,0}},
/* 9757 */ {(12<<2)|2,{99,70,0}},
/* 9758 */ {(12<<2)|2,{99,70,0}},
/* 9759 */ {(12<<2)|2,{99,70,0}},
/* 9760 */ {(12<<2)|2,{99,71,0}},
/* 9761 */ {(12<<2)|2,{99,71,0}},
/* 9762 */ {(12<<2)|2,{99,71,0}},
/* 9763 */ {(12<<2)|2,{99,71,0}},
/* 9764 */ {(12<<2)|2,{99,71,0}},
/* 9765 */ {(12<<2)|2,{99,71,0}},
/* 9766 */ {(12<<2)|2,{99,71,0}},
/* 9767 */ {(12<<2)|2,{99,71,0}},
/* 9768 */ {(12<<2)|2,{99,71,0}},
/* 9769 */ {(12<<2)|2,{99,71,0}},
/* 9770 */ {(12<<2)|2,{99,71,0}},
/* 9771 */ {(12<<2)|2,{99,71,0}},
/* 9772 */ {(12<<2)|2,{99,71,0}},
/* 9773 */ {(12<<2)|2,{99,71,0}},
/* 9774 */ {(12<<2)|2,{99,71,0}},
/* 9775 */ {(12<<2)|2,{99,71,0}},
/* 9776 */ {(12<<2)|2,{99,72,0}},
/* 9777 */ {(12<<2)|2,{99,72,0}},
/* 9778 */ {(12<<2)|2,{99,72,0}},
/* 9779 */ {(12<<2)|2,{99,72,0}},
/* 9780 */ {(12<<2)|2,{99,72,0}},
/* 9781 */ {(12<<2)|2,{99,72,0}},
/* 9782 */ {(12<<2)|2,{99,72,0}},
/* 9783 */ {(12<<2)|2,{99,72,0}},
/* 9784 */ {(12<<2)|2,{99,72,0}},
/* 9785 */ {(12<<2)|2,{99,72,0}},
/* 9786 */ {(12<<2)|2,{99,72,0}},
/* 9787 */ {(12<<2)|2,{99,72,0}},
/* 9788 */ {(12<<2)|2,{99,72,0}},
/* 9789 */ {(12<<2)|2,{99,72,0}},
/* 9790 */ {(12<<2)|2,{99,72,0}},
/* 9791 */ {(12<<2)|2,{99,72,0}},
/* 9792 */ {(12<<2)|2,{99,73,0}},
/* 9793 */ {(12<<2)|2,{99,73,0}},
/* 9794 */ {(12<<2)|2,{99,73,0}},
/* 9795 */ {(12<<2)|2,{99,73,0}},
/* 9796 */ {(12<<2)|2,{99,73,0}},
/* 9797 */ {(12<<2)|2,{99,73,0}},
/* 9798 */ {(12<<2)|2,{99,73,0}},
/* 9799 */ {(12<<2)|2,{99,73,0}},
/* 9800 */ {(12<<2)|2,{99,73,0}},
/* 9801 */ {(12<<2)|2,{99,73,0}},
/* 9802 */ {(12<<2)|2,{99,73,0}},
/* 9803 */ {(12<<2)|2,{99,73,0}},
/* 9804 */ {(12<<2)|2,{99,73,0}},
/* 9805 */ {(12<<2)|2,{99,73,0}},
/* 9806 */ {(12<<2)|2,{99,73,0}},
/* 9807 */ {(12<<2)|2,{99,73,0}},
/* 9808 */ {(12<<2)|2,{99,74,0}},
/* 9809 */ {(12<<2)|2,{99,74,0}},
/* 9810 */ {(12<<2)|2,{99,74,0}},
/* 9811 */ {(12<<2)|2,{99,74,0}},
/* 9812 */ {(12<<2)|2,{99,74,0}},
/* 9813 */ {(12<<2)|2,{99,74,0}},
/* 9814 */ {(12<<2)|2,{99,74,0}},
/* 9815 */ {(12<<2)|2,{99,74,0}},
/* 9816 */ {(12<<2)|2,{99,74,0}},
/* 9817 */ {(12<<2)|2,{99,74,0}},
/* 9818 */ {(12<<2)|2,{99,74,0}},
/* 9819 */ {(12<<2)|2,{99,74,0}},
/* 9820 */ {(12<<2)|2,{99,74,0}},
/* 9821 */ {(12<<2)|2,{99,74,0}},
/* 9822 */ {(12<<2)|2,{99,74,0}},
/* 9823 */ {(12<<2)|2,{99,74,0}},
/* 9824 */ {(12<<2)|2,{99,75,0}},
/* 9825 */ {(12<<2)|2,{99,75,0}},
/* 9826 */ {(12<<2)|2,{99,75,0}},
/* 9827 */ {(12<<2)|2,{99,75,0}},
/* 9828 */ {(12<<2)|2,{99,75,0}},
/* 9829 */ {(12<<2)|2,{99,75,0}},
/* 9830 */ {(12<<2)|2,{99,75,0}},
/* 9831 */ {(12<<2)|2,{99,75,0}},
/* 9832 */ {(12<<2)|2,{99,75,0}},
/* 9833 */ {(12<<2)|2,{99,75,0}},
/* 9834 */ {(12<<2)|2,{99,75,0}},
/* 9835 */ {(12<<2)|2,{99,75,0}},
/* 9836 */ {(12<<2)|2,{99,75,0}},
/* 9837 */ {(12<<2)|2,{99,75,0}},
/* 9838 */ {(12<<2)|2,{99,75,0}},
/* 9839 */ {(12<<2)|2,{99,75,0}},
/* 9840 */ {(12<<2)|2,{99,76,0}},
/* 9841 */ {(12<<2)|2,{99,76,0}},
/* 9842 */ {(12<<2)|2,{99,76,0}},
/* 9843 */ {(12<<2)|2,{99,76,0}},
/* 9844 */ {(12<<2)|2,{99,76,0}},
/* 9845 */ {(12<<2)|2,{99,76,0}},
/* 9846 */ {(12<<2)|2,{99,76,0}},
/* 9847 */ {(12<<2)|2,{99,76,0}},
/* 9848 */ {(12<<2)|2,{99,76,0}},
/* 9849 */ {(12<<2)|2,{99,76,0}},
/* 9850 */ {(12<<2)|2,{99,76,0}},
/* 9851 */ {(12<<2)|2,{99,76,0}},
/* 9852 */ {(12<<2)|2,{99,76,0}},
/* 9853 */ {(12<<2)|2,{99,76,0}},
/* 9854 */ {(12<<2)|2,{99,76,0}},
/* 9855 */ {(12<<2)|2,{99,76,0}},
/* 9856 */ {(12<<2)|2,{99,77,0}},
/* 9857 */ {(12<<2)|2,{99,77,0}},
/* 9858 */ {(12<<2)|2,{99,77,0}},
/* 9859 */ {(12<<2)|2,{99,77,0}},
/* 9860 */ {(12<<2)|2,{99,77,0}},
/* 9861 */ {(12<<2)|2,{99,77,0}},
/* 9862 */ {(12<<2)|2,{99,77,0}},
/* 9863 */ {(12<<2)|2,{99,77,0}},
/* 9864 */ {(12<<2)|2,{99,77,0}},
/* 9865 */ {(12<<2)|2,{99,77,0}},
/* 9866 */ {(12<<2)|2,{99,77,0}},
/* 9867 */ {(12<<2)|2,{99,77,0}},
/* 9868 */ {(12<<2)|2,{99,77,0}},
/* 9869 */ {(12<<2)|2,{99,77,0}},
/* 9870 */ {(12<<2)|2,{99,77,0}},
/* 9871 */ {(12<<2)|2,{99,77,0}},
/* 9872 */ {(12<<2)|2,{99,78,0}},
/* 9873 */ {(12<<2)|2,{99,78,0}},
/* 9874 */ {(12<<2)|2,{99,78,0}},
/* 9875 */ {(12<<2)|2,{99,78,0}},
/* 9876 */ {(12<<2)|2,{99,78,0}},
/* 9877 */ {(12<<2)|2,{99,78,0}},
/* 9878 */ {(12<<2)|2,{99,78,0}},
/* 9879 */ {(12<<2)|2,{99,78,0}},
/* 9880 */ {(12<<2)|2,{99,78,0}},
/* 9881 */ {(12<<2)|2,{99,78,0}},
/* 9882 */ {(12<<2)|2,{99,78,0}},
/* 9883 */ {(12<<2)|2,{99,78,0}},
/* 9884 */ {(12<<2)|2,{99,78,0}},
/* 9885 */ {(12<<2)|2,{99,78,0}},
/* 9886 */ {(12<<2)|2,{99,78,0}},
/* 9887 */ {(12<<2)|2,{99,78,0}},
/* 9888 */ {(12<<2)|2,{99,79,0}},
/* 9889 */ {(12<<2)|2,{99,79,0}},
/* 9890 */ {(12<<2)|2,{99,79,0}},
/* 9891 */ {(12<<2)|2,{99,79,0}},
/* 9892 */ {(12<<2)|2,{99,79,0}},
/* 9893 */ {(12<<2)|2,{99,79,0}},
/* 9894 */ {(12<<2)|2,{99,79,0}},
/* 9895 */ {(12<<2)|2,{99,79,0}},
/* 9896 */ {(12<<2)|2,{99,79,0}},
/* 9897 */ {(12<<2)|2,{99,79,0}},
/* 9898 */ {(12<<2)|2,{99,79,0}},
/* 9899 */ {(12<<2)|2,{99,79,0}},
/* 9900 */ {(12<<2)|2,{99,79,0}},
/* 9901 */ {(12<<2)|2,{99,79,0}},
/* 9902 */ {(12<<2)|2,{99,79,0}},
/* 9903 */ {(12<<2)|2,{99,79,0}},
/* 9904 */ {(12<<2)|2,{99,80,0}},
/* 9905 */ {(12<<2)|2,{99,80,0}},
/* 9906 */ {(12<<2)|2,{99,80,0}},
/* 9907 */ {(12<<2)|2,{99,80,0}},
/* 9908 */ {(12<<2)|2,{99,80,0}},
/* 9909 */ {(12<<2)|2,{99,80,0}},
/* 9910 */ {(12<<2)|2,{99,80,0}},
/* 9911 */ {(12<<2)|2,{99,80,0}},
/* 9912 */ {(12<<2)|2,{99,80,0}},
/* 9913 */ {(12<<2)|2,{99,80,0}},
/* 9914 */ {(12<<2)|2,{99,80,0}},
/* 9915 */ {(12<<2)|2,{99,80,0}},
/* 9916 */ {(12<<2)|2,{99,80,0}},
/* 9917 */ {(12<<2)|2,{99,80,0}},
/* 9918 */ {(12<<2)|2,{99,80,0}},
/* 9919 */ {(12<<2)|2,{99,80,0}},
/* 9920 */ {(12<<2)|2,{99,81,0}},
/* 9921 */ {(12<<2)|2,{99,81,0}},
/* 9922 */ {(12<<2)|2,{99,81,0}},
/* 9923 */ {(12<<2)|2,{99,81,0}},
/* 9924 */ {(12<<2)|2,{99,81,0}},
/* 9925 */ {(12<<2)|2,{99,81,0}},
/* 9926 */ {(12<<2)|2,{99,81,0}},
/* 9927 */ {(12<<2)|2,{99,81,0}},
/* 9928 */ {(12<<2)|2,{99,81,0}},
/* 9929 */ {(12<<2)|2,{99,81,0}},
/* 9930 */ {(12<<2)|2,{99,81,0}},
/* 9931 */ {(12<<2)|2,{99,81,0}},
/* 9932 */ {(12<<2)|2,{99,81,0}},
/* 9933 */ {(12<<2)|2,{99,81,0}},
/* 9934 */ {(12<<2)|2,{99,81,0}},
/* 9935 */ {(12<<2)|2,{99,81,0}},
/* 9936 */ {(12<<2)|2,{99,82,0}},
/* 9937 */ {(12<<2)|2,{99,82,0}},
/* 9938 */ {(12<<2)|2,{99,82,0}},
/* 9939 */ {(12<<2)|2,{99,82,0}},
/* 9940 */ {(12<<2)|2,{99,82,0}},
/* 9941 */ {(12<<2)|2,{99,82,0}},
/* 9942 */ {(12<<2)|2,{99,82,0}},
/* 9943 */ {(12<<2)|2,{99,82,0}},
/* 9944 */ {(12<<2)|2,{99,82,0}},
/* 9945 */ {(12<<2)|2,{99,82,0}},
/* 9946 */ {(12<<2)|2,{99,82,0}},
/* 9947 */ {(12<<2)|2,{99,82,0}},
/* 9948 */ {(12<<2)|2,{99,82,0}},
/* 9949 */ {(12<<2)|2,{99,82,0}},
/* 9950 */ {(12<<2)|2,{99,82,0}},
/* 9951 */ {(12<<2)|2,{99,82,0}},
/* 9952 */ {(12<<2)|2,{99,83,0}},
/* 9953 */ {(12<<2)|2,{99,83,0}},
/* 9954 */ {(12<<2)|2,{99,83,0}},
/* 9955 */ {(12<<2)|2,{99,83,0}},
/* 9956 */ {(12<<2)|2,{99,83,0}},
/* 9957 */ {(12<<2)|2,{99,83,0}},
/* 9958 */ {(12<<2)|2,{99,83,0}},
/* 9959 */ {(12<<2)|2,{99,83,0}},
/* 9960 */ {(12<<2)|2,{99,83,0}},
/* 9961 */ {(12<<2)|2,{99,83,0}},
/* 9962 */ {(12<<2)|2,{99,83,0}},
/* 9963 */ {(12<<2)|2,{99,83,0}},
/* 9964 */ {(12<<2)|2,{99,83,0}},
/* 9965 */ {(12<<2)|2,{99,83,0}},
/* 9966 */ {(12<<2)|2,{99,83,0}},
/* 9967 */ {(12<<2)|2,{99,83,0}},
/* 9968 */ {(12<<2)|2,{99,84,0}},
/* 9969 */ {(12<<2)|2,{99,84,0}},
/* 9970 */ {(12<<2)|2,{99,84,0}},
/* 9971 */ {(12<<2)|2,{99,84,0}},
/* 9972 */ {(12<<2)|2,{99,84,0}},
/* 9973 */ {(12<<2)|2,{99,84,0}},
/* 9974 */ {(12<<2)|2,{99,84,0}},
/* 9975 */ {(12<<2)|2,{99,84,0}},
/* 9976 */ {(12<<2)|2,{99,84,0}},
/* 9977 */ {(12<<2)|2,{99,84,0}},
/* 9978 */ {(12<<2)|2,{99,84,0}},
/* 9979 */ {(12<<2)|2,{99,84,0}},
/* 9980 */ {(12<<2)|2,{99,84,0}},
/* 9981 */ {(12<<2)|2,{99,84,0}},
/* 9982 */ {(12<<2)|2,{99,84,0}},
/* 9983 */ {(12<<2)|2,{99,84,0}},
/* 9984 */ {(12<<2)|2,{99,85,0}},
/* 9985 */ {(12<<2)|2,{99,85,0}},
/* 9986 */ {(12<<2)|2,{99,85,0}},
/* 9987 */ {(12<<2)|2,{99,85,0}},
/* 9988 */ {(12<<2)|2,{99,85,0}},
/* 9989 */ {(12<<2)|2,{99,85,0}},
/* 9990 */ {(12<<2)|2,{99,85,0}},
/* 9991 */ {(12<<2)|2,{99,85,0}},
/* 9992 */ {(12<<2)|2,{99,85,0}},
/* 9993 */ {(12<<2)|2,{99,85,0}},
/* 9994 */ {(12<<2)|2,{99,85,0}},
/* 9995 */ {(12<<2)|2,{99,85,0}},
/* 9996 */ {(12<<2)|2,{99,85,0}},
/* 9997 */ {(12<<2)|2,{99,85,0}},
/* 9998 */ {(12<<2)|2,{99,85,0}},
/* 9999 */ {(12<<2)|2,{99,85,0}},
/* 10000 */ {(12<<2)|2,{99,86,0}},
/* 10001 */ {(12<<2)|2,{99,86,0}},
/* 10002 */ {(12<<2)|2,{99,86,0}},
/* 10003 */ {(12<<2)|2,{99,86,0}},
/* 10004 */ {(12<<2)|2,{99,86,0}},
/* 10005 */ {(12<<2)|2,{99,86,0}},
/* 10006 */ {(12<<2)|2,{99,86,0}},
/* 10007 */ {(12<<2)|2,{99,86,0}},
/* 10008 */ {(12<<2)|2,{99,86,0}},
/* 10009 */ {(12<<2)|2,{99,86,0}},
/* 10010 */ {(12<<2)|2,{99,86,0}},
/* 10011 */ {(12<<2)|2,{99,86,0}},
/* 10012 */ {(12<<2)|2,{99,86,0}},
/* 10013 */ {(12<<2)|2,{99,86,0}},
/* 10014 */ {(12<<2)|2,{99,86,0}},
/* 10015 */ {(12<<2)|2,{99,86,0}},
/* 10016 */ {(12<<2)|2,{99,87,0}},
/* 10017 */ {(12<<2)|2,{99,87,0}},
/* 10018 */ {(12<<2)|2,{99,87,0}},
/* 10019 */ {(12<<2)|2,{99,87,0}},
/* 10020 */ {(12<<2)|2,{99,87,0}},
/* 10021 */ {(12<<2)|2,{99,87,0}},
/* 10022 */ {(12<<2)|2,{99,87,0}},
/* 10023 */ {(12<<2)|2,{99,87,0}},
/* 10024 */ {(12<<2)|2,{99,87,0}},
/* 10025 */ {(12<<2)|2,{99,87,0}},
/* 10026 */ {(12<<2)|2,{99,87,0}},
/* 10027 */ {(12<<2)|2,{99,87,0}},
/* 10028 */ {(12<<2)|2,{99,87,0}},
/* 10029 */ {(12<<2)|2,{99,87,0}},
/* 10030 */ {(12<<2)|2,{99,87,0}},
/* 10031 */ {(12<<2)|2,{99,87,0}},
/* 10032 */ {(12<<2)|2,{99,89,0}},
/* 10033 */ {(12<<2)|2,{99,89,0}},
/* 10034 */ {(12<<2)|2,{99,89,0}},
/* 10035 */ {(12<<2)|2,{99,89,0}},
/* 10036 */ {(12<<2)|2,{99,89,0}},
/* 10037 */ {(12<<2)|2,{99,89,0}},
/* 10038 */ {(12<<2)|2,{99,89,0}},
/* 10039 */ {(12<<2)|2,{99,89,0}},
/* 10040 */ {(12<<2)|2,{99,89,0}},
/* 10041 */ {(12<<2)|2,{99,89,0}},
/* 10042 */ {(12<<2)|2,{99,89,0}},
/* 10043 */ {(12<<2)|2,{99,89,0}},
/* 10044 */ {(12<<2)|2,{99,89,0}},
/* 10045 */ {(12<<2)|2,{99,89,0}},
/* 10046 */ {(12<<2)|2,{99,89,0}},
/* 10047 */ {(12<<2)|2,{99,89,0}},
/* 10048 */ {(12<<2)|2,{99,106,0}},
/* 10049 */ {(12<<2)|2,{99,106,0}},
/* 10050 */ {(12<<2)|2,{99,106,0}},
/* 10051 */ {(12<<2)|2,{99,106,0}},
/* 10052 */ {(12<<2)|2,{99,106,0}},
/* 10053 */ {(12<<2)|2,{99,106,0}},
/* 10054 */ {(12<<2)|2,{99,106,0}},
/* 10055 */ {(12<<2)|2,{99,106,0}},
/* 10056 */ {(12<<2)|2,{99,106,0}},
/* 10057 */ {(12<<2)|2,{99,106,0}},
/* 10058 */ {(12<<2)|2,{99,106,0}},
/* 10059 */ {(12<<2)|2,{99,106,0}},
/* 10060 */ {(12<<2)|2,{99,106,0}},
/* 10061 */ {(12<<2)|2,{99,106,0}},
/* 10062 */ {(12<<2)|2,{99,106,0}},
/* 10063 */ {(12<<2)|2,{99,106,0}},
/* 10064 */ {(12<<2)|2,{99,107,0}},
/* 10065 */ {(12<<2)|2,{99,107,0}},
/* 10066 */ {(12<<2)|2,{99,107,0}},
/* 10067 */ {(12<<2)|2,{99,107,0}},
/* 10068 */ {(12<<2)|2,{99,107,0}},
/* 10069 */ {(12<<2)|2,{99,107,0}},
/* 10070 */ {(12<<2)|2,{99,107,0}},
/* 10071 */ {(12<<2)|2,{99,107,0}},
/* 10072 */ {(12<<2)|2,{99,107,0}},
/* 10073 */ {(12<<2)|2,{99,107,0}},
/* 10074 */ {(12<<2)|2,{99,107,0}},
/* 10075 */ {(12<<2)|2,{99,107,0}},
/* 10076 */ {(12<<2)|2,{99,107,0}},
/* 10077 */ {(12<<2)|2,{99,107,0}},
/* 10078 */ {(12<<2)|2,{99,107,0}},
/* 10079 */ {(12<<2)|2,{99,107,0}},
/* 10080 */ {(12<<2)|2,{99,113,0}},
/* 10081 */ {(12<<2)|2,{99,113,0}},
/* 10082 */ {(12<<2)|2,{99,113,0}},
/* 10083 */ {(12<<2)|2,{99,113,0}},
/* 10084 */ {(12<<2)|2,{99,113,0}},
/* 10085 */ {(12<<2)|2,{99,113,0}},
/* 10086 */ {(12<<2)|2,{99,113,0}},
/* 10087 */ {(12<<2)|2,{99,113,0}},
/* 10088 */ {(12<<2)|2,{99,113,0}},
/* 10089 */ {(12<<2)|2,{99,113,0}},
/* 10090 */ {(12<<2)|2,{99,113,0}},
/* 10091 */ {(12<<2)|2,{99,113,0}},
/* 10092 */ {(12<<2)|2,{99,113,0}},
/* 10093 */ {(12<<2)|2,{99,113,0}},
/* 10094 */ {(12<<2)|2,{99,113,0}},
/* 10095 */ {(12<<2)|2,{99,113,0}},
/* 10096 */ {(12<<2)|2,{99,118,0}},
/* 10097 */ {(12<<2)|2,{99,118,0}},
/* 10098 */ {(12<<2)|2,{99,118,0}},
/* 10099 */ {(12<<2)|2,{99,118,0}},
/* 10100 */ {(12<<2)|2,{99,118,0}},
/* 10101 */ {(12<<2)|2,{99,118,0}},
/* 10102 */ {(12<<2)|2,{99,118,0}},
/* 10103 */ {(12<<2)|2,{99,118,0}},
/* 10104 */ {(12<<2)|2,{99,118,0}},
/* 10105 */ {(12<<2)|2,{99,118,0}},
/* 10106 */ {(12<<2)|2,{99,118,0}},
/* 10107 */ {(12<<2)|2,{99,118,0}},
/* 10108 */ {(12<<2)|2,{99,118,0}},
/* 10109 */ {(12<<2)|2,{99,118,0}},
/* 10110 */ {(12<<2)|2,{99,118,0}},
/* 10111 */ {(12<<2)|2,{99,118,0}},
/* 10112 */ {(12<<2)|2,{99,119,0}},
/* 10113 */ {(12<<2)|2,{99,119,0}},
/* 10114 */ {(12<<2)|2,{99,119,0}},
/* 10115 */ {(12<<2)|2,{99,119,0}},
/* 10116 */ {(12<<2)|2,{99,119,0}},
/* 10117 */ {(12<<2)|2,{99,119,0}},
/* 10118 */ {(12<<2)|2,{99,119,0}},
/* 10119 */ {(12<<2)|2,{99,119,0}},
/* 10120 */ {(12<<2)|2,{99,119,0}},
/* 10121 */ {(12<<2)|2,{99,119,0}},
/* 10122 */ {(12<<2)|2,{99,119,0}},
/* 10123 */ {(12<<2)|2,{99,119,0}},
/* 10124 */ {(12<<2)|2,{99,119,0}},
/* 10125 */ {(12<<2)|2,{99,119,0}},
/* 10126 */ {(12<<2)|2,{99,119,0}},
/* 10127 */ {(12<<2)|2,{99,119,0}},
/* 10128 */ {(12<<2)|2,{99,120,0}},
/* 10129 */ {(12<<2)|2,{99,120,0}},
/* 10130 */ {(12<<2)|2,{99,120,0}},
/* 10131 */ {(12<<2)|2,{99,120,0}},
/* 10132 */ {(12<<2)|2,{99,120,0}},
/* 10133 */ {(12<<2)|2,{99,120,0}},
/* 10134 */ {(12<<2)|2,{99,120,0}},
/* 10135 */ {(12<<2)|2,{99,120,0}},
/* 10136 */ {(12<<2)|2,{99,120,0}},
/* 10137 */ {(12<<2)|2,{99,120,0}},
/* 10138 */ {(12<<2)|2,{99,120,0}},
/* 10139 */ {(12<<2)|2,{99,120,0}},
/* 10140 */ {(12<<2)|2,{99,120,0}},
/* 10141 */ {(12<<2)|2,{99,120,0}},
/* 10142 */ {(12<<2)|2,{99,120,0}},
/* 10143 */ {(12<<2)|2,{99,120,0}},
/* 10144 */ {(12<<2)|2,{99,121,0}},
/* 10145 */ {(12<<2)|2,{99,121,0}},
/* 10146 */ {(12<<2)|2,{99,121,0}},
/* 10147 */ {(12<<2)|2,{99,121,0}},
/* 10148 */ {(12<<2)|2,{99,121,0}},
/* 10149 */ {(12<<2)|2,{99,121,0}},
/* 10150 */ {(12<<2)|2,{99,121,0}},
/* 10151 */ {(12<<2)|2,{99,121,0}},
/* 10152 */ {(12<<2)|2,{99,121,0}},
/* 10153 */ {(12<<2)|2,{99,121,0}},
/* 10154 */ {(12<<2)|2,{99,121,0}},
/* 10155 */ {(12<<2)|2,{99,121,0}},
/* 10156 */ {(12<<2)|2,{99,121,0}},
/* 10157 */ {(12<<2)|2,{99,121,0}},
/* 10158 */ {(12<<2)|2,{99,121,0}},
/* 10159 */ {(12<<2)|2,{99,121,0}},
/* 10160 */ {(12<<2)|2,{99,122,0}},
/* 10161 */ {(12<<2)|2,{99,122,0}},
/* 10162 */ {(12<<2)|2,{99,122,0}},
/* 10163 */ {(12<<2)|2,{99,122,0}},
/* 10164 */ {(12<<2)|2,{99,122,0}},
/* 10165 */ {(12<<2)|2,{99,122,0}},
/* 10166 */ {(12<<2)|2,{99,122,0}},
/* 10167 */ {(12<<2)|2,{99,122,0}},
/* 10168 */ {(12<<2)|2,{99,122,0}},
/* 10169 */ {(12<<2)|2,{99,122,0}},
/* 10170 */ {(12<<2)|2,{99,122,0}},
/* 10171 */ {(12<<2)|2,{99,122,0}},
/* 10172 */ {(12<<2)|2,{99,122,0}},
/* 10173 */ {(12<<2)|2,{99,122,0}},
/* 10174 */ {(12<<2)|2,{99,122,0}},
/* 10175 */ {(12<<2)|2,{99,122,0}},
/* 10176 */ {(13<<2)|2,{99,38,0}},
/* 10177 */ {(13<<2)|2,{99,38,0}},
/* 10178 */ {(13<<2)|2,{99,38,0}},
/* 10179 */ {(13<<2)|2,{99,38,0}},
/* 10180 */ {(13<<2)|2,{99,38,0}},
/* 10181 */ {(13<<2)|2,{99,38,0}},
/* 10182 */ {(13<<2)|2,{99,38,0}},
/* 10183 */ {(13<<2)|2,{99,38,0}},
/* 10184 */ {(13<<2)|2,{99,42,0}},
/* 10185 */ {(13<<2)|2,{99,42,0}},
/* 10186 */ {(13<<2)|2,{99,42,0}},
/* 10187 */ {(13<<2)|2,{99,42,0}},
/* 10188 */ {(13<<2)|2,{99,42,0}},
/* 10189 */ {(13<<2)|2,{99,42,0}},
/* 10190 */ {(13<<2)|2,{99,42,0}},
/* 10191 */ {(13<<2)|2,{99,42,0}},
/* 10192 */ {(13<<2)|2,{99,44,0}},
/* 10193 */ {(13<<2)|2,{99,44,0}},
/* 10194 */ {(13<<2)|2,{99,44,0}},
/* 10195 */ {(13<<2)|2,{99,44,0}},
/* 10196 */ {(13<<2)|2,{99,44,0}},
/* 10197 */ {(13<<2)|2,{99,44,0}},
/* 10198 */ {(13<<2)|2,{99,44,0}},
/* 10199 */ {(13<<2)|2,{99,44,0}},
/* 10200 */ {(13<<2)|2,{99,59,0}},
/* 10201 */ {(13<<2)|2,{99,59,0}},
/* 10202 */ {(13<<2)|2,{99,59,0}},
/* 10203 */ {(13<<2)|2,{99,59,0}},
/* 10204 */ {(13<<2)|2,{99,59,0}},
/* 10205 */ {(13<<2)|2,{99,59,0}},
/* 10206 */ {(13<<2)|2,{99,59,0}},
/* 10207 */ {(13<<2)|2,{99,59,0}},
/* 10208 */ {(13<<2)|2,{99,88,0}},
/* 10209 */ {(13<<2)|2,{99,88,0}},
/* 10210 */ {(13<<2)|2,{99,88,0}},
/* 10211 */ {(13<<2)|2,{99,88,0}},
/* 10212 */ {(13<<2)|2,{99,88,0}},
/* 10213 */ {(13<<2)|2,{99,88,0}},
/* 10214 */ {(13<<2)|2,{99,88,0}},
/* 10215 */ {(13<<2)|2,{99,88,0}},
/* 10216 */ {(13<<2)|2,{99,90,0}},
/* 10217 */ {(13<<2)|2,{99,90,0}},
/* 10218 */ {(13<<2)|2,{99,90,0}},
/* 10219 */ {(13<<2)|2,{99,90,0}},
/* 10220 */ {(13<<2)|2,{99,90,0}},
/* 10221 */ {(13<<2)|2,{99,90,0}},
/* 10222 */ {(13<<2)|2,{99,90,0}},
/* 10223 */ {(13<<2)|2,{99,90,0}},
/* 10224 */ {(15<<2)|2,{99,33,0}},
/* 10225 */ {(15<<2)|2,{99,33,0}},
/* 10226 */ {(15<<2)|2,{99,34,0}},
/* 10227 */ {(15<<2)|2,{99,34,0}},
/* 10228 */ {(15<<2)|2,{99,40,0}},
/* 10229 */ {(15<<2)|2,{99,40,0}},
/* 10230 */ {(15<<2)|2,{99,41,0}},
/* 10231 */ {(15<<2)|2,{99,41,0}},
/* 10232 */ {(15<<2)|2,{99,63,0}},
/* 10233 */ {(15<<2)|2,{99,63,0}},
/* 10234 */ {(16<<2)|2,{99,39,0}},
/* 10235 */ {(16<<2)|2,{99,43,0}},
/* 10236 */ {(16<<2)|2,{99,124,0}},
/* 10237 */ {(5<<2)|1,{99,0,0}},
/* 10238 */ {(5<<2)|1,{99,0,0}},
/* 10239 */ {(5<<2)|1,{99,0,0}},
/* 10240 */ {(15<<2)|3,{101,48,48}},
/* 10241 */ {(15<<2)|3,{101,48,48}},
/* 10242 */ {(15<<2)|3,{101,48,49}},
/* 10243 */ {(15<<2)|3,{101,48,49}},
/* 10244 */ {(15<<2)|3,{101,48,50}},
/* 10245 */ {(15<<2)|3,{101,48,50}},
/* 10246 */ {(15<<2)|3,{101,48,97}},
/* 10247 */ {(15<<2)|3,{101,48,97}},
/* 10248 */ {(15<<2)|3,{101,48,99}},
/* 10249 */ {(15<<2)|3,{101,48,99}},
/* 10250 */ {(15<<2)|3,{101,48,101}},
/* 10251 */ {(15<<2)|3,{101,48,101}},
/* 10252 */ {(15<<2)|3,{101,48,105}},
/* 10253 */ {(15<<2)|3,{101,48,105}},
/* 10254 */ {(15<<2)|3,{101,48,111}},
/* 10255 */ {(15<<2)|3,{101,48,111}},
/* 10256 */ {(15<<2)|3,{101,48,115}},
/* 10257 */ {(15<<2)|3,{101,48,115}},
/* 10258 */ {(15<<2)|3,{101,48,116}},
/* 10259 */ {(15<<2)|3,{101,48,116}},
/* 10260 */ {(16<<2)|3,{101,48,32}},
/* 10261 */ {(16<<2)|3,{101,48,37}},
/* 10262 */ {(16<<2)|3,{101,48,45}},
/* 10263 */ {(16<<2)|3,{101,48,46}},
/* 10264 */ {(16<<2)|3,{101,48,47}},
/* 10265 */ {(16<<2)|3,{101,48,51}},
/* 10266 */ {(16<<2)|3,{101,48,52}},
/* 10267 */ {(16<<2)|3,{101,48,53}},
/* 10268 */ {(16<<2)|3,{101,48,54}},
/* 10269 */ {(16<<2)|3,{101,48,55}},
/* 10270 */ {(16<<2)|3,{101,48,56}},
/* 10271 */ {(16<<2)|3,{101,48,57}},
/* 10272 */ {(16<<2)|3,{101,48,61}},
/* 10273 */ {(16<<2)|3,{101,48,65}},
/* 10274 */ {(16<<2)|3,{101,48,95}},
/* 10275 */ {(16<<2)|3,{101,48,98}},
/* 10276 */ {(16<<2)|3,{101,48,100}},
/* 10277 */ {(16<<2)|3,{101,48,102}},
/* 10278 */ {(16<<2)|3,{101,48,103}},
/* 10279 */ {(16<<2)|3,{101,48,104}},
/* 10280 */ {(16<<2)|3,{101,48,108}},
/* 10281 */ {(16<<2)|3,{101,48,109}},
/* 10282 */ {(16<<2)|3,{101,48,110}},
/* 10283 */ {(16<<2)|3,{101,48,112}},
/* 10284 */ {(16<<2)|3,{101,48,114}},
/* 10285 */ {(16<<2)|3,{101,48,117}},
/* 10286 */ {(10<<2)|2,{101,48,0}},
/* 10287 */ {(10<<2)|2,{101,48,0}},
/* 10288 */ {(10<<2)|2,{101,48,0}},
/* 10289 */ {(10<<2)|2,{101,48,0}},
/* 10290 */ {(10<<2)|2,{101,48,0}},
/* 10291 */ {(10<<2)|2,{101,48,0}},
/* 10292 */ {(10<<2)|2,{101,48,0}},
/* 10293 */ {(10<<2)|2,{101,48,0}},
/* 10294 */ {(10<<2)|2,{101,48,0}},
/* 10295 */ {(10<<2)|2,{101,48,0}},
/* 10296 */ {(10<<2)|2,{101,48,0}},
/* 10297 */ {(10<<2)|2,{101,48,0}},
/* 10298 */ {(10<<2)|2,{101,48,0}},
/* 10299 */ {(10<<2)|2,{101,48,0}},
/* 10300 */ {(10<<2)|2,{101,48,0}},
/* 10301 */ {(10<<2)|2,{101,48,0}},
/* 10302 */ {(10<<2)|2,{101,48,0}},
/* 10303 */ {(10<<2)|2,{101,48,0}},
/* 10304 */ {(15<<2)|3,{101,49,48}},
/* 10305 */ {(15<<2)|3,{101,49,48}},
/* 10306 */ {(15<<2)|3,{101,49,49}},
/* 10307 */ {(15<<2)|3,{101,49,49}},
/* 10308 */ {(15<<2)|3,{101,49,50}},
/* 10309 */ {(15<<2)|3,{101,49,50}},
/* 10310 */ {(15<<2)|3,{101,49,97}},
/* 10311 */ {(15<<2)|3,{101,49,97}},
/* 10312 */ {(15<<2)|3,{101,49,99}},
/* 10313 */ {(15<<2)|3,{101,49,99}},
/* 10314 */ {(15<<2)|3,{101,49,101}},
/* 10315 */ {(15<<2)|3,{101,49,101}},
/* 10316 */ {(15<<2)|3,{101,49,105}},
/* 10317 */ {(15<<2)|3,{101,49,105}},
/* 10318 */ {(15<<2)|3,{101,49,111}},
/* 10319 */ {(15<<2)|3,{101,49,111}},
/* 10320 */ {(15<<2)|3,{101,49,115}},
/* 10321 */ {(15<<2)|3,{101,49,115}},
/* 10322 */ {(15<<2)|3,{101,49,116}},
/* 10323 */ {(15<<2)|3,{101,49,116}},
/* 10324 */ {(16<<2)|3,{101,49,32}},
/* 10325 */ {(16<<2)|3,{101,49,37}},
/* 10326 */ {(16<<2)|3,{101,49,45}},
/* 10327 */ {(16<<2)|3,{101,49,46}},
/* 10328 */ {(16<<2)|3,{101,49,47}},
/* 10329 */ {(16<<2)|3,{101,49,51}},
/* 10330 */ {(16<<2)|3,{101,49,52}},
/* 10331 */ {(16<<2)|3,{101,49,53}},
/* 10332 */ {(16<<2)|3,{101,49,54}},
/* 10333 */ {(16<<2)|3,{101,49,55}},
/* 10334 */ {(16<<2)|3,{101,49,56}},
/* 10335 */ {(16<<2)|3,{101,49,57}},
/* 10336 */ {(16<<2)|3,{101,49,61}},
/* 10337 */ {(16<<2)|3,{101,49,65}},
/* 10338 */ {(16<<2)|3,{101,49,95}},
/* 10339 */ {(16<<2)|3,{101,49,98}},
/* 10340 */ {(16<<2)|3,{101,49,100}},
/* 10341 */ {(16<<2)|3,{101,49,102}},
/* 10342 */ {(16<<2)|3,{101,49,103}},
/* 10343 */ {(16<<2)|3,{101,49,104}},
/* 10344 */ {(16<<2)|3,{101,49,108}},
/* 10345 */ {(16<<2)|3,{101,49,109}},
/* 10346 */ {(16<<2)|3,{101,49,110}},
/* 10347 */ {(16<<2)|3,{101,49,112}},
/* 10348 */ {(16<<2)|3,{101,49,114}},
/* 10349 */ {(16<<2)|3,{101,49,117}},
/* 10350 */ {(10<<2)|2,{101,49,0}},
/* 10351 */ {(10<<2)|2,{101,49,0}},
/* 10352 */ {(10<<2)|2,{101,49,0}},
/* 10353 */ {(10<<2)|2,{101,49,0}},
/* 10354 */ {(10<<2)|2,{101,49,0}},
/* 10355 */ {(10<<2)|2,{101,49,0}},
/* 10356 */ {(10<<2)|2,{101,49,0}},
/* 10357 */ {(10<<2)|2,{101,49,0}},
/* 10358 */ {(10<<2)|2,{101,49,0}},
/* 10359 */ {(10<<2)|2,{101,49,0}},
/* 10360 */ {(10<<2)|2,{101,49,0}},
/* 10361 */ {(10<<2)|2,{101,49,0}},
/* 10362 */ {(10<<2)|2,{101,49,0}},
/* 10363 */ {(10<<2)|2,{101,49,0}},
/* 10364 */ {(10<<2)|2,{101,49,0}},
/* 10365 */ {(10<<2)|2,{101,49,0}},
/* 10366 */ {(10<<2)|2,{101,49,0}},
/* 10367 */ {(10<<2)|2,{101,49,0}},
/* 10368 */ {(15<<2)|3,{101,50,48}},
/* 10369 */ {(15<<2)|3,{101,50,48}},
/* 10370 */ {(15<<2)|3,{101,50,49}},
/* 10371 */ {(15<<2)|3,{101,50,49}},
/* 10372 */ {(15<<2)|3,{101,50,50}},
/* 10373 */ {(15<<2)|3,{101,50,50}},
/* 10374 */ {(15<<2)|3,{101,50,97}},
/* 10375 */ {(15<<2)|3,{101,50,97}},
/* 10376 */ {(15<<2)|3,{101,50,99}},
/* 10377 */ {(15<<2)|3,{101,50,99}},
/* 10378 */ {(15<<2)|3,{101,50,101}},
/* 10379 */ {(15<<2)|3,{101,50,101}},
/* 10380 */ {(15<<2)|3,{101,50,105}},
/* 10381 */ {(15<<2)|3,{101,50,105}},
/* 10382 */ {(15<<2)|3,{101,50,111}},
/* 10383 */ {(15<<2)|3,{101,50,111}},
/* 10384 */ {(15<<2)|3,{101,50,115}},
/* 10385 */ {(15<<2)|3,{101,50,115}},
/* 10386 */ {(15<<2)|3,{101,50,116}},
/* 10387 */ {(15<<2)|3,{101,50,116}},
/* 10388 */ {(16<<2)|3,{101,50,32}},
/* 10389 */ {(16<<2)|3,{101,50,37}},
/* 10390 */ {(16<<2)|3,{101,50,45}},
/* 10391 */ {(16<<2)|3,{101,50,46}},
/* 10392 */ {(16<<2)|3,{101,50,47}},
/* 10393 */ {(16<<2)|3,{101,50,51}},
/* 10394 */ {(16<<2)|3,{101,50,52}},
/* 10395 */ {(16<<2)|3,{101,50,53}},
/* 10396 */ {(16<<2)|3,{101,50,54}},
/* 10397 */ {(16<<2)|3,{101,50,55}},
/* 10398 */ {(16<<2)|3,{101,50,56}},
/* 10399 */ {(16<<2)|3,{101,50,57}},
/* 10400 */ {(16<<2)|3,{101,50,61}},
/* 10401 */ {(16<<2)|3,{101,50,65}},
/* 10402 */ {(16<<2)|3,{101,50,95}},
/* 10403 */ {(16<<2)|3,{101,50,98}},
/* 10404 */ {(16<<2)|3,{101,50,100}},
/* 10405 */ {(16<<2)|3,{101,50,102}},
/* 10406 */ {(16<<2)|3,{101,50,103}},
/* 10407 */ {(16<<2)|3,{101,50,104}},
/* 10408 */ {(16<<2)|3,{101,50,108}},
/* 10409 */ {(16<<2)|3,{101,50,109}},
/* 10410 */ {(16<<2)|3,{101,50,110}},
/* 10411 */ {(16<<2)|3,{101,50,112}},
/* 10412 */ {(16<<2)|3,{101,50,114}},
/* 10413 */ {(16<<2)|3,{101,50,117}},
/* 10414 */ {(10<<2)|2,{101,50,0}},
/* 10415 */ {(10<<2)|2,{101,50,0}},
/* 10416 */ {(10<<2)|2,{101,50,0}},
/* 10417 */ {(10<<2)|2,{101,50,0}},
/* 10418 */ {(10<<2)|2,{101,50,0}},
/* 10419 */ {(10<<2)|2,{101,50,0}},
/* 10420 */ {(10<<2)|2,{101,50,0}},
/* 10421 */ {(10<<2)|2,{101,50,0}},
/* 10422 */ {(10<<2)|2,{101,50,0}},
/* 10423 */ {(10<<2)|2,{101,50,0}},
/* 10424 */ {(10<<2)|2,{101,50,0}},
/* 10425 */ {(10<<2)|2,{101,50,0}},
/* 10426 */ {(10<<2)|2,{101,50,0}},
/* 10427 */ {(10<<2)|2,{101,50,0}},
/* 10428 */ {(10<<2)|2,{101,50,0}},
/* 10429 */ {(10<<2)|2,{101,50,0}},
/* 10430 */ {(10<<2)|2,{101,50,0}},
/* 10431 */ {(10<<2)|2,{101,50,0}},
/* 10432 */ {(15<<2)|3,{101,97,48}},
/* 10433 */ {(15<<2)|3,{101,97,48}},
/* 10434 */ {(15<<2)|3,{101,97,49}},
/* 10435 */ {(15<<2)|3,{101,97,49}},
/* 10436 */ {(15<<2)|3,{101,97,50}},
/* 10437 */ {(15<<2)|3,{101,97,50}},
/* 10438 */ {(15<<2)|3,{101,97,97}},
/* 10439 */ {(15<<2)|3,{101,97,97}},
/* 10440 */ {(15<<2)|3,{101,97,99}},
/* 10441 */ {(15<<2)|3,{101,97,99}},
/* 10442 */ {(15<<2)|3,{101,97,101}},
/* 10443 */ {(15<<2)|3,{101,97,101}},
/* 10444 */ {(15<<2)|3,{101,97,105}},
/* 10445 */ {(15<<2)|3,{101,97,105}},
/* 10446 */ {(15<<2)|3,{101,97,111}},
/* 10447 */ {(15<<2)|3,{101,97,111}},
/* 10448 */ {(15<<2)|3,{101,97,115}},
/* 10449 */ {(15<<2)|3,{101,97,115}},
/* 10450 */ {(15<<2)|3,{101,97,116}},
/* 10451 */ {(15<<2)|3,{101,97,116}},
/* 10452 */ {(16<<2)|3,{101,97,32}},
/* 10453 */ {(16<<2)|3,{101,97,37}},
/* 10454 */ {(16<<2)|3,{101,97,45}},
/* 10455 */ {(16<<2)|3,{101,97,46}},
/* 10456 */ {(16<<2)|3,{101,97,47}},
/* 10457 */ {(16<<2)|3,{101,97,51}},
/* 10458 */ {(16<<2)|3,{101,97,52}},
/* 10459 */ {(16<<2)|3,{101,97,53}},
/* 10460 */ {(16<<2)|3,{101,97,54}},
/* 10461 */ {(16<<2)|3,{101,97,55}},
/* 10462 */ {(16<<2)|3,{101,97,56}},
/* 10463 */ {(16<<2)|3,{101,97,57}},
/* 10464 */ {(16<<2)|3,{101,97,61}},
/* 10465 */ {(16<<2)|3,{101,97,65}},
/* 10466 */ {(16<<2)|3,{101,97,95}},
/* 10467 */ {(16<<2)|3,{101,97,98}},
/* 10468 */ {(16<<2)|3,{101,97,100}},
/* 10469 */ {(16<<2)|3,{101,97,102}},
/* 10470 */ {(16<<2)|3,{101,97,103}},
/* 10471 */ {(16<<2)|3,{101,97,104}},
/* 10472 */ {(16<<2)|3,{101,97,108}},
/* 10473 */ {(16<<2)|3,{101,97,109}},
/* 10474 */ {(16<<2)|3,{101,97,110}},
/* 10475 */ {(16<<2)|3,{101,97,112}},
/* 10476 */ {(16<<2)|3,{101,97,114}},
/* 10477 */ {(16<<2)|3,{101,97,117}},
/* 10478 */ {(10<<2)|2,{101,97,0}},
/* 10479 */ {(10<<2)|2,{101,97,0}},
/* 10480 */ {(10<<2)|2,{101,97,0}},
/* 10481 */ {(10<<2)|2,{101,97,0}},
/* 10482 */ {(10<<2)|2,{101,97,0}},
/* 10483 */ {(10<<2)|2,{101,97,0}},
/* 10484 */ {(10<<2)|2,{101,97,0}},
/* 10485 */ {(10<<2)|2,{101,97,0}},
/* 10486 */ {(10<<2)|2,{101,97,0}},
/* 10487 */ {(10<<2)|2,{101,97,0}},
/* 10488 */ {(10<<2)|2,{101,97,0}},
/* 10489 */ {(10<<2)|2,{101,97,0}},
/* 10490 */ {(10<<2)|2,{101,97,0}},
/* 10491 */ {(10<<2)|2,{101,97,0}},
/* 10492 */ {(10<<2)|2,{101,97,0}},
/* 10493 */ {(10<<2)|2,{101,97,0}},
/* 10494 */ {(10<<2)|2,{101,97,0}},
/* 10495 */ {(10<<2)|2,{101,97,0}},
/* 10496 */ {(15<<2)|3,{101,99,48}},
/* 10497 */ {(15<<2)|3,{101,99,48}},
/* 10498 */ {(15<<2)|3,{101,99,49}},
/* 10499 */ {(15<<2)|3,{101,99,49}},
/* 10500 */ {(15<<2)|3,{101,99,50}},
/* 10501 */ {(15<<2)|3,{101,99,50}},
/* 10502 */ {(15<<2)|3,{101,99,97}},
/* 10503 */ {(15<<2)|3,{101,99,97}},
/* 10504 */ {(15<<2)|3,{101,99,99}},
/* 10505 */ {(15<<2)|3,{101,99,99}},
/* 10506 */ {(15<<2)|3,{101,99,101}},
/* 10507 */ {(15<<2)|3,{101,99,101}},
/* 10508 */ {(15<<2)|3,{101,99,105}},
/* 10509 */ {(15<<2)|3,{101,99,105}},
/* 10510 */ {(15<<2)|3,{101,99,111}},
/* 10511 */ {(15<<2)|3,{101,99,111}},
/* 10512 */ {(15<<2)|3,{101,99,115}},
/* 10513 */ {(15<<2)|3,{101,99,115}},
/* 10514 */ {(15<<2)|3,{101,99,116}},
/* 10515 */ {(15<<2)|3,{101,99,116}},
/* 10516 */ {(16<<2)|3,{101,99,32}},
/* 10517 */ {(16<<2)|3,{101,99,37}},
/* 10518 */ {(16<<2)|3,{101,99,45}},
/* 10519 */ {(16<<2)|3,{101,99,46}},
/* 10520 */ {(16<<2)|3,{101,99,47}},
/* 10521 */ {(16<<2)|3,{101,99,51}},
/* 10522 */ {(16<<2)|3,{101,99,52}},
/* 10523 */ {(16<<2)|3,{101,99,53}},
/* 10524 */ {(16<<2)|3,{101,99,54}},
/* 10525 */ {(16<<2)|3,{101,99,55}},
/* 10526 */ {(16<<2)|3,{101,99,56}},
/* 10527 */ {(16<<2)|3,{101,99,57}},
/* 10528 */ {(16<<2)|3,{101,99,61}},
/* 10529 */ {(16<<2)|3,{101,99,65}},
/* 10530 */ {(16<<2)|3,{101,99,95}},
/* 10531 */ {(16<<2)|3,{101,99,98}},
/* 10532 */ {(16<<2)|3,{101,99,100}},
/* 10533 */ {(16<<2)|3,{101,99,102}},
/* 10534 */ {(16<<2)|3,{101,99,103}},
/* 10535 */ {(16<<2)|3,{101,99,104}},
/* 10536 */ {(16<<2)|3,{101,99,108}},
/* 10537 */ {(16<<2)|3,{101,99,109}},
/* 10538 */ {(16<<2)|3,{101,99,110}},
/* 10539 */ {(16<<2)|3,{101,99,112}},
/* 10540 */ {(16<<2)|3,{101,99,114}},
/* 10541 */ {(16<<2)|3,{101,99,117}},
/* 10542 */ {(10<<2)|2,{101,99,0}},
/* 10543 */ {(10<<2)|2,{101,99,0}},
/* 10544 */ {(10<<2)|2,{101,99,0}},
/* 10545 */ {(10<<2)|2,{101,99,0}},
/* 10546 */ {(10<<2)|2,{101,99,0}},
/* 10547 */ {(10<<2)|2,{101,99,0}},
/* 10548 */ {(10<<2)|2,{101,99,0}},
/* 10549 */ {(10<<2)|2,{101,99,0}},
/* 10550 */ {(10<<2)|2,{101,99,0}},
/* 10551 */ {(10<<2)|2,{101,99,0}},
/* 10552 */ {(10<<2)|2,{101,99,0}},
/* 10553 */ {(10<<2)|2,{101,99,0}},
/* 10554 */ {(10<<2)|2,{101,99,0}},
/* 10555 */ {(10<<2)|2,{101,99,0}},
/* 10556 */ {(10<<2)|2,{101,99,0}},
/* 10557 */ {(10<<2)|2,{101,99,0}},
/* 10558 */ {(10<<2)|2,{101,99,0}},
/* 10559 */ {(10<<2)|2,{101,99,0}},
/* 10560 */ {(15<<2)|3,{101,101,48}},
/* 10561 */ {(15<<2)|3,{101,101,48}},
/* 10562 */ {(15<<2)|3,{101,101,49}},
/* 10563 */ {(15<<2)|3,{101,101,49}},
/* 10564 */ {(15<<2)|3,{101,101,50}},
/* 10565 */ {(15<<2)|3,{101,101,50}},
/* 10566 */ {(15<<2)|3,{101,101,97}},
/* 10567 */ {(15<<2)|3,{101,101,97}},
/* 10568 */ {(15<<2)|3,{101,101,99}},
/* 10569 */ {(15<<2)|3,{101,101,99}},
/* 10570 */ {(15<<2)|3,{101,101,101}},
/* 10571 */ {(15<<2)|3,{101,101,101}},
/* 10572 */ {(15<<2)|3,{101,101,105}},
/* 10573 */ {(15<<2)|3,{101,101,105}},
/* 10574 */ {(15<<2)|3,{101,101,111}},
/* 10575 */ {(15<<2)|3,{101,101,111}},
/* 10576 */ {(15<<2)|3,{101,101,115}},
/* 10577 */ {(15<<2)|3,{101,101,115}},
/* 10578 */ {(15<<2)|3,{101,101,116}},
/* 10579 */ {(15<<2)|3,{101,101,116}},
/* 10580 */ {(16<<2)|3,{101,101,32}},
/* 10581 */ {(16<<2)|3,{101,101,37}},
/* 10582 */ {(16<<2)|3,{101,101,45}},
/* 10583 */ {(16<<2)|3,{101,101,46}},
/* 10584 */ {(16<<2)|3,{101,101,47}},
/* 10585 */ {(16<<2)|3,{101,101,51}},
/* 10586 */ {(16<<2)|3,{101,101,52}},
/* 10587 */ {(16<<2)|3,{101,101,53}},
/* 10588 */ {(16<<2)|3,{101,101,54}},
/* 10589 */ {(16<<2)|3,{101,101,55}},
/* 10590 */ {(16<<2)|3,{101,101,56}},
/* 10591 */ {(16<<2)|3,{101,101,57}},
/* 10592 */ {(16<<2)|3,{101,101,61}},
/* 10593 */ {(16<<2)|3,{101,101,65}},
/* 10594 */ {(16<<2)|3,{101,101,95}},
/* 10595 */ {(16<<2)|3,{101,101,98}},
/* 10596 */ {(16<<2)|3,{101,101,100}},
/* 10597 */ {(16<<2)|3,{101,101,102}},
/* 10598 */ {(16<<2)|3,{101,101,103}},
/* 10599 */ {(16<<2)|3,{101,101,104}},
/* 10600 */ {(16<<2)|3,{101,101,108}},
/* 10601 */ {(16<<2)|3,{101,101,109}},
/* 10602 */ {(16<<2)|3,{101,101,110}},
/* 10603 */ {(16<<2)|3,{101,101,112}},
/* 10604 */ {(16<<2)|3,{101,101,114}},
/* 10605 */ {(16<<2)|3,{101,101,117}},
/* 10606 */ {(10<<2)|2,{101,101,0}},
/* 10607 */ {(10<<2)|2,{101,101,0}},
/* 10608 */ {(10<<2)|2,{101,101,0}},
/* 10609 */ {(10<<2)|2,{101,101,0}},
/* 10610 */ {(10<<2)|2,{101,101,0}},
/* 10611 */ {(10<<2)|2,{101,101,0}},
/* 10612 */ {(10<<2)|2,{101,101,0}},
/* 10613 */ {(10<<2)|2,{101,101,0}},
/* 10614 */ {(10<<2)|2,{101,101,0}},
/* 10615 */ {(10<<2)|2,{101,101,0}},
/* 10616 */ {(10<<2)|2,{101,101,0}},
/* 10617 */ {(10<<2)|2,{101,101,0}},
/* 10618 */ {(10<<2)|2,{101,101,0}},
/* 10619 */ {(10<<2)|2,{101,101,0}},
/* 10620 */ {(10<<2)|2,{101,101,0}},
/* 10621 */ {(10<<2)|2,{101,101,0}},
/* 10622 */ {(10<<2)|2,{101,101,0}},
/* 10623 */ {(10<<2)|2,{101,101,0}},
/* 10624 */ {(15<<2)|3,{101,105,48}},
/* 10625 */ {(15<<2)|3,{101,105,48}},
/* 10626 */ {(15<<2)|3,{101,105,49}},
/* 10627 */ {(15<<2)|3,{101,105,49}},
/* 10628 */ {(15<<2)|3,{101,105,50}},
/* 10629 */ {(15<<2)|3,{101,105,50}},
/* 10630 */ {(15<<2)|3,{101,105,97}},
/* 10631 */ {(15<<2)|3,{101,105,97}},
/* 10632 */ {(15<<2)|3,{101,105,99}},
/* 10633 */ {(15<<2)|3,{101,105,99}},
/* 10634 */ {(15<<2)|3,{101,105,101}},
/* 10635 */ {(15<<2)|3,{101,105,101}},
/* 10636 */ {(15<<2)|3,{101,105,105}},
/* 10637 */ {(15<<2)|3,{101,105,105}},
/* 10638 */ {(15<<2)|3,{101,105,111}},
/* 10639 */ {(15<<2)|3,{101,105,111}},
/* 10640 */ {(15<<2)|3,{101,105,115}},
/* 10641 */ {(15<<2)|3,{101,105,115}},
/* 10642 */ {(15<<2)|3,{101,105,116}},
/* 10643 */ {(15<<2)|3,{101,105,116}},
/* 10644 */ {(16<<2)|3,{101,105,32}},
/* 10645 */ {(16<<2)|3,{101,105,37}},
/* 10646 */ {(16<<2)|3,{101,105,45}},
/* 10647 */ {(16<<2)|3,{101,105,46}},
/* 10648 */ {(16<<2)|3,{101,105,47}},
/* 10649 */ {(16<<2)|3,{101,105,51}},
/* 10650 */ {(16<<2)|3,{101,105,52}},
/* 10651 */ {(16<<2)|3,{101,105,53}},
/* 10652 */ {(16<<2)|3,{101,105,54}},
/* 10653 */ {(16<<2)|3,{101,105,55}},
/* 10654 */ {(16<<2)|3,{101,105,56}},
/* 10655 */ {(16<<2)|3,{101,105,57}},
/* 10656 */ {(16<<2)|3,{101,105,61}},
/* 10657 */ {(16<<2)|3,{101,105,65}},
/* 10658 */ {(16<<2)|3,{101,105,95}},
/* 10659 */ {(16<<2)|3,{101,105,98}},
/* 10660 */ {(16<<2)|3,{101,105,100}},
/* 10661 */ {(16<<2)|3,{101,105,102}},
/* 10662 */ {(16<<2)|3,{101,105,103}},
/* 10663 */ {(16<<2)|3,{101,105,104}},
/* 10664 */ {(16<<2)|3,{101,105,108}},
/* 10665 */ {(16<<2)|3,{101,105,109}},
/* 10666 */ {(16<<2)|3,{101,105,110}},
/* 10667 */ {(16<<2)|3,{101,105,112}},
/* 10668 */ {(16<<2)|3,{101,105,114}},
/* 10669 */ {(16<<2)|3,{101,105,117}},
/* 10670 */ {(10<<2)|2,{101,105,0}},
/* 10671 */ {(10<<2)|2,{101,105,0}},
/* 10672 */ {(10<<2)|2,{101,105,0}},
/* 10673 */ {(10<<2)|2,{101,105,0}},
/* 10674 */ {(10<<2)|2,{101,105,0}},
/* 10675 */ {(10<<2)|2,{101,105,0}},
/* 10676 */ {(10<<2)|2,{101,105,0}},
/* 10677 */ {(10<<2)|2,{101,105,0}},
/* 10678 */ {(10<<2)|2,{101,105,0}},
/* 10679 */ {(10<<2)|2,{101,105,0}},
/* 10680 */ {(10<<2)|2,{101,105,0}},
/* 10681 */ {(10<<2)|2,{101,105,0}},
/* 10682 */ {(10<<2)|2,{101,105,0}},
/* 10683 */ {(10<<2)|2,{101,105,0}},
/* 10684 */ {(10<<2)|2,{101,105,0}},
/* 10685 */ {(10<<2)|2,{101,105,0}},
/* 10686 */ {(10<<2)|2,{101,105,0}},
/* 10687 */ {(10<<2)|2,{101,105,0}},
/* 10688 */ {(15<<2)|3,{101,111,48}},
/* 10689 */ {(15<<2)|3,{101,111,48}},
/* 10690 */ {(15<<2)|3,{101,111,49}},
/* 10691 */ {(15<<2)|3,{101,111,49}},
/* 10692 */ {(15<<2)|3,{101,111,50}},
/* 10693 */ {(15<<2)|3,{101,111,50}},
/* 10694 */ {(15<<2)|3,{101,111,97}},
/* 10695 */ {(15<<2)|3,{101,111,97}},
/* 10696 */ {(15<<2)|3,{101,111,99}},
/* 10697 */ {(15<<2)|3,{101,111,99}},
/* 10698 */ {(15<<2)|3,{101,111,101}},
/* 10699 */ {(15<<2)|3,{101,111,101}},
/* 10700 */ {(15<<2)|3,{101,111,105}},
/* 10701 */ {(15<<2)|3,{101,111,105}},
/* 10702 */ {(15<<2)|3,{101,111,111}},
/* 10703 */ {(15<<2)|3,{101,111,111}},
/* 10704 */ {(15<<2)|3,{101,111,115}},
/* 10705 */ {(15<<2)|3,{101,111,115}},
/* 10706 */ {(15<<2)|3,{101,111,116}},
/* 10707 */ {(15<<2)|3,{101,111,116}},
/* 10708 */ {(16<<2)|3,{101,111,32}},
/* 10709 */ {(16<<2)|3,{101,111,37}},
/* 10710 */ {(16<<2)|3,{101,111,45}},
/* 10711 */ {(16<<2)|3,{101,111,46}},
/* 10712 */ {(16<<2)|3,{101,111,47}},
/* 10713 */ {(16<<2)|3,{101,111,51}},
/* 10714 */ {(16<<2)|3,{101,111,52}},
/* 10715 */ {(16<<2)|3,{101,111,53}},
/* 10716 */ {(16<<2)|3,{101,111,54}},
/* 10717 */ {(16<<2)|3,{101,111,55}},
/* 10718 */ {(16<<2)|3,{101,111,56}},
/* 10719 */ {(16<<2)|3,{101,111,57}},
/* 10720 */ {(16<<2)|3,{101,111,61}},
/* 10721 */ {(16<<2)|3,{101,111,65}},
/* 10722 */ {(16<<2)|3,{101,111,95}},
/* 10723 */ {(16<<2)|3,{101,111,98}},
/* 10724 */ {(16<<2)|3,{101,111,100}},
/* 10725 */ {(16<<2)|3,{101,111,102}},
/* 10726 */ {(16<<2)|3,{101,111,103}},
/* 10727 */ {(16<<2)|3,{101,111,104}},
/* 10728 */ {(16<<2)|3,{101,111,108}},
/* 10729 */ {(16<<2)|3,{101,111,109}},
/* 10730 */ {(16<<2)|3,{101,111,110}},
/* 10731 */ {(16<<2)|3,{101,111,112}},
/* 10732 */ {(16<<2)|3,{101,111,114}},
/* 10733 */ {(16<<2)|3,{101,111,117}},
/* 10734 */ {(10<<2)|2,{101,111,0}},
/* 10735 */ {(10<<2)|2,{101,111,0}},
/* 10736 */ {(10<<2)|2,{101,111,0}},
/* 10737 */ {(10<<2)|2,{101,111,0}},
/* 10738 */ {(10<<2)|2,{101,111,0}},
/* 10739 */ {(10<<2)|2,{101,111,0}},
/* 10740 */ {(10<<2)|2,{101,111,0}},
/* 10741 */ {(10<<2)|2,{101,111,0}},
/* 10742 */ {(10<<2)|2,{101,111,0}},
/* 10743 */ {(10<<2)|2,{101,111,0}},
/* 10744 */ {(10<<2)|2,{101,111,0}},
/* 10745 */ {(10<<2)|2,{101,111,0}},
/* 10746 */ {(10<<2)|2,{101,111,0}},
/* 10747 */ {(10<<2)|2,{101,111,0}},
/* 10748 */ {(10<<2)|2,{101,111,0}},
/* 10749 */ {(10<<2)|2,{101,111,0}},
/* 10750 */ {(10<<2)|2,{101,111,0}},
/* 10751 */ {(10<<2)|2,{101,111,0}},
/* 10752 */ {(15<<2)|3,{101,115,48}},
/* 10753 */ {(15<<2)|3,{101,115,48}},
/* 10754 */ {(15<<2)|3,{101,115,49}},
/* 10755 */ {(15<<2)|3,{101,115,49}},
/* 10756 */ {(15<<2)|3,{101,115,50}},
/* 10757 */ {(15<<2)|3,{101,115,50}},
/* 10758 */ {(15<<2)|3,{101,115,97}},
/* 10759 */ {(15<<2)|3,{101,115,97}},
/* 10760 */ {(15<<2)|3,{101,115,99}},
/* 10761 */ {(15<<2)|3,{101,115,99}},
/* 10762 */ {(15<<2)|3,{101,115,101}},
/* 10763 */ {(15<<2)|3,{101,115,101}},
/* 10764 */ {(15<<2)|3,{101,115,105}},
/* 10765 */ {(15<<2)|3,{101,115,105}},
/* 10766 */ {(15<<2)|3,{101,115,111}},
/* 10767 */ {(15<<2)|3,{101,115,111}},
/* 10768 */ {(15<<2)|3,{101,115,115}},
/* 10769 */ {(15<<2)|3,{101,115,115}},
/* 10770 */ {(15<<2)|3,{101,115,116}},
/* 10771 */ {(15<<2)|3,{101,115,116}},
/* 10772 */ {(16<<2)|3,{101,115,32}},
/* 10773 */ {(16<<2)|3,{101,115,37}},
/* 10774 */ {(16<<2)|3,{101,115,45}},
/* 10775 */ {(16<<2)|3,{101,115,46}},
/* 10776 */ {(16<<2)|3,{101,115,47}},
/* 10777 */ {(16<<2)|3,{101,115,51}},
/* 10778 */ {(16<<2)|3,{101,115,52}},
/* 10779 */ {(16<<2)|3,{101,115,53}},
/* 10780 */ {(16<<2)|3,{101,115,54}},
/* 10781 */ {(16<<2)|3,{101,115,55}},
/* 10782 */ {(16<<2)|3,{101,115,56}},
/* 10783 */ {(16<<2)|3,{101,115,57}},
/* 10784 */ {(16<<2)|3,{101,115,61}},
/* 10785 */ {(16<<2)|3,{101,115,65}},
/* 10786 */ {(16<<2)|3,{101,115,95}},
/* 10787 */ {(16<<2)|3,{101,115,98}},
/* 10788 */ {(16<<2)|3,{101,115,100}},
/* 10789 */ {(16<<2)|3,{101,115,102}},
/* 10790 */ {(16<<2)|3,{101,115,103}},
/* 10791 */ {(16<<2)|3,{101,115,104}},
/* 10792 */ {(16<<2)|3,{101,115,108}},
/* 10793 */ {(16<<2)|3,{101,115,109}},
/* 10794 */ {(16<<2)|3,{101,115,110}},
/* 10795 */ {(16<<2)|3,{101,115,112}},
/* 10796 */ {(16<<2)|3,{101,115,114}},
/* 10797 */ {(16<<2)|3,{101,115,117}},
/* 10798 */ {(10<<2)|2,{101,115,0}},
/* 10799 */ {(10<<2)|2,{101,115,0}},
/* 10800 */ {(10<<2)|2,{101,115,0}},
/* 10801 */ {(10<<2)|2,{101,115,0}},
/* 10802 */ {(10<<2)|2,{101,115,0}},
/* 10803 */ {(10<<2)|2,{101,115,0}},
/* 10804 */ {(10<<2)|2,{101,115,0}},
/* 10805 */ {(10<<2)|2,{101,115,0}},
/* 10806 */ {(10<<2)|2,{101,115,0}},
/* 10807 */ {(10<<2)|2,{101,115,0}},
/* 10808 */ {(10<<2)|2,{101,115,0}},
/* 10809 */ {(10<<2)|2,{101,115,0}},
/* 10810 */ {(10<<2)|2,{101,115,0}},
/* 10811 */ {(10<<2)|2,{101,115,0}},
/* 10812 */ {(10<<2)|2,{101,115,0}},
/* 10813 */ {(10<<2)|2,{101,115,0}},
/* 10814 */ {(10<<2)|2,{101,115,0}},
/* 10815 */ {(10<<2)|2,{101,115,0}},
/* 10816 */ {(15<<2)|3,{101,116,48}},
/* 10817 */ {(15<<2)|3,{101,116,48}},
/* 10818 */ {(15<<2)|3,{101,116,49}},
/* 10819 */ {(15<<2)|3,{101,116,49}},
/* 10820 */ {(15<<2)|3,{101,116,50}},
/* 10821 */ {(15<<2)|3,{101,116,50}},
/* 10822 */ {(15<<2)|3,{101,116,97}},
/* 10823 */ {(15<<2)|3,{101,116,97}},
/* 10824 */ {(15<<2)|3,{101,116,99}},
/* 10825 */ {(15<<2)|3,{101,116,99}},
/* 10826 */ {(15<<2)|3,{101,116,101}},
/* 10827 */ {(15<<2)|3,{101,116,101}},
/* 10828 */ {(15<<2)|3,{101,116,105}},
/* 10829 */ {(15<<2)|3,{101,116,105}},
/* 10830 */ {(15<<2)|3,{101,116,111}},
/* 10831 */ {(15<<2)|3,{101,116,111}},
/* 10832 */ {(15<<2)|3,{101,116,115}},
/* 10833 */ {(15<<2)|3,{101,116,115}},
/* 10834 */ {(15<<2)|3,{101,116,116}},
/* 10835 */ {(15<<2)|3,{101,116,116}},
/* 10836 */ {(16<<2)|3,{101,116,32}},
/* 10837 */ {(16<<2)|3,{101,116,37}},
/* 10838 */ {(16<<2)|3,{101,116,45}},
/* 10839 */ {(16<<2)|3,{101,116,46}},
/* 10840 */ {(16<<2)|3,{101,116,47}},
/* 10841 */ {(16<<2)|3,{101,116,51}},
/* 10842 */ {(16<<2)|3,{101,116,52}},
/* 10843 */ {(16<<2)|3,{101,116,53}},
/* 10844 */ {(16<<2)|3,{101,116,54}},
/* 10845 */ {(16<<2)|3,{101,116,55}},
/* 10846 */ {(16<<2)|3,{101,116,56}},
/* 10847 */ {(16<<2)|3,{101,116,57}},
/* 10848 */ {(16<<2)|3,{101,116,61}},
/* 10849 */ {(16<<2)|3,{101,116,65}},
/* 10850 */ {(16<<2)|3,{101,116,95}},
/* 10851 */ {(16<<2)|3,{101,116,98}},
/* 10852 */ {(16<<2)|3,{101,116,100}},
/* 10853 */ {(16<<2)|3,{101,116,102}},
/* 10854 */ {(16<<2)|3,{101,116,103}},
/* 10855 */ {(16<<2)|3,{101,116,104}},
/* 10856 */ {(16<<2)|3,{101,116,108}},
/* 10857 */ {(16<<2)|3,{101,116,109}},
/* 10858 */ {(16<<2)|3,{101,116,110}},
/* 10859 */ {(16<<2)|3,{101,116,112}},
/* 10860 */ {(16<<2)|3,{101,116,114}},
/* 10861 */ {(16<<2)|3,{101,116,117}},
/* 10862 */ {(10<<2)|2,{101,116,0}},
/* 10863 */ {(10<<2)|2,{101,116,0}},
/* 10864 */ {(10<<2)|2,{101,116,0}},
/* 10865 */ {(10<<2)|2,{101,116,0}},
/* 10866 */ {(10<<2)|2,{101,116,0}},
/* 10867 */ {(10<<2)|2,{101,116,0}},
/* 10868 */ {(10<<2)|2,{101,116,0}},
/* 10869 */ {(10<<2)|2,{101,116,0}},
/* 10870 */ {(10<<2)|2,{101,116,0}},
/* 10871 */ {(10<<2)|2,{101,116,0}},
/* 10872 */ {(10<<2)|2,{101,116,0}},
/* 10873 */ {(10<<2)|2,{101,116,0}},
/* 10874 */ {(10<<2)|2,{101,116,0}},
/* 10875 */ {(10<<2)|2,{101,116,0}},
/* 10876 */ {(10<<2)|2,{101,116,0}},
/* 10877 */ {(10<<2)|2,{101,116,0}},
/* 10878 */ {(10<<2)|2,{101,116,0}},
/* 10879 */ {(10<<2)|2,{101,116,0}},
/* 10880 */ {(16<<2)|3,{101,32,48}},
/* 10881 */ {(16<<2)|3,{101,32,49}},
/* 10882 */ {(16<<2)|3,{101,32,50}},
/* 10883 */ {(16<<2)|3,{101,32,97}},
/* 10884 */ {(16<<2)|3,{101,32,99}},
/* 10885 */ {(16<<2)|3,{101,32,101}},
/* 10886 */ {(16<<2)|3,{101,32,105}},
/* 10887 */ {(16<<2)|3,{101,32,111}},
/* 10888 */ {(16<<2)|3,{101,32,115}},
/* 10889 */ {(16<<2)|3,{101,32,116}},
/* 10890 */ {(11<<2)|2,{101,32,0}},
/* 10891 */ {(11<<2)|2,{101,32,0}},
/* 10892 */ {(11<<2)|2,{101,32,0}},
/* 10893 */ {(11<<2)|2,{101,32,0}},
/* 10894 */ {(11<<2)|2,{101,32,0}},
/* 10895 */ {(11<<2)|2,{101,32,0}},
/* 10896 */ {(11<<2)|2,{101,32,0}},
/* 10897 */ {(11<<2)|2,{101,32,0}},
/* 10898 */ {(11<<2)|2,{101,32,0}},
/* 10899 */ {(11<<2)|2,{101,32,0}},
/* 10900 */ {(11<<2)|2,{101,32,0}},
/* 10901 */ {(11<<2)|2,{101,32,0}},
/* 10902 */ {(11<<2)|2,{101,32,0}},
/* 10903 */ {(11<<2)|2,{101,32,0}},
/* 10904 */ {(11<<2)|2,{101,32,0}},
/* 10905 */ {(11<<2)|2,{101,32,0}},
/* 10906 */ {(11<<2)|2,{101,32,0}},
/* 10907 */ {(11<<2)|2,{101,32,0}},
/* 10908 */ {(11<<2)|2,{101,32,0}},
/* 10909 */ {(11<<2)|2,{101,32,0}},
/* 10910 */ {(11<<2)|2,{101,32,0}},
/* 10911 */ {(11<<2)|2,{101,32,0}},
/* 10912 */ {(16<<2)|3,{101,37,48}},
/* 10913 */ {(16<<2)|3,{101,37,49}},
/* 10914 */ {(16<<2)|3,{101,37,50}},
/* 10915 */ {(16<<2)|3,{101,37,97}},
/* 10916 */ {(16<<2)|3,{101,37,99}},
/* 10917 */ {(16<<2)|3,{101,37,101}},
/* 10918 */ {(16<<2)|3,{101,37,105}},
/* 10919 */ {(16<<2)|3,{101,37,111}},
/* 10920 */ {(16<<2)|3,{101,37,115}},
/* 10921 */ {(16<<2)|3,{101,37,116}},
/* 10922 */ {(11<<2)|2,{101,37,0}},
/* 10923 */ {(11<<2)|2,{101,37,0}},
/* 10924 */ {(11<<2)|2,{101,37,0}},
/* 10925 */ {(11<<2)|2,{101,37,0}},
/* 10926 */ {(11<<2)|2,{101,37,0}},
/* 10927 */ {(11<<2)|2,{101,37,0}},
/* 10928 */ {(11<<2)|2,{101,37,0}},
/* 10929 */ {(11<<2)|2,{101,37,0}},
/* 10930 */ {(11<<2)|2,{101,37,0}},
/* 10931 */ {(11<<2)|2,{101,37,0}},
/* 10932 */ {(11<<2)|2,{101,37,0}},
/* 10933 */ {(11<<2)|2,{101,37,0}},
/* 10934 */ {(11<<2)|2,{101,37,0}},
/* 10935 */ {(11<<2)|2,{101,37,0}},
/* 10936 */ {(11<<2)|2,{101,37,0}},
/* 10937 */ {(11<<2)|2,{101,37,0}},
/* 10938 */ {(11<<2)|2,{101,37,0}},
/* 10939 */ {(11<<2)|2,{101,37,0}},
/* 10940 */ {(11<<2)|2,{101,37,0}},
/* 10941 */ {(11<<2)|2,{101,37,0}},
/* 10942 */ {(11<<2)|2,{101,37,0}},
/* 10943 */ {(11<<2)|2,{101,37,0}},
/* 10944 */ {(16<<2)|3,{101,45,48}},
/* 10945 */ {(16<<2)|3,{101,45,49}},
/* 10946 */ {(16<<2)|3,{101,45,50}},
/* 10947 */ {(16<<2)|3,{101,45,97}},
/* 10948 */ {(16<<2)|3,{101,45,99}},
/* 10949 */ {(16<<2)|3,{101,45,101}},
/* 10950 */ {(16<<2)|3,{101,45,105}},
/* 10951 */ {(16<<2)|3,{101,45,111}},
/* 10952 */ {(16<<2)|3,{101,45,115}},
/* 10953 */ {(16<<2)|3,{101,45,116}},
/* 10954 */ {(11<<2)|2,{101,45,0}},
/* 10955 */ {(11<<2)|2,{101,45,0}},
/* 10956 */ {(11<<2)|2,{101,45,0}},
/* 10957 */ {(11<<2)|2,{101,45,0}},
/* 10958 */ {(11<<2)|2,{101,45,0}},
/* 10959 */ {(11<<2)|2,{101,45,0}},
/* 10960 */ {(11<<2)|2,{101,45,0}},
/* 10961 */ {(11<<2)|2,{101,45,0}},
/* 10962 */ {(11<<2)|2,{101,45,0}},
/* 10963 */ {(11<<2)|2,{101,45,0}},
/* 10964 */ {(11<<2)|2,{101,45,0}},
/* 10965 */ {(11<<2)|2,{101,45,0}},
/* 10966 */ {(11<<2)|2,{101,45,0}},
/* 10967 */ {(11<<2)|2,{101,45,0}},
/* 10968 */ {(11<<2)|2,{101,45,0}},
/* 10969 */ {(11<<2)|2,{101,45,0}},
/* 10970 */ {(11<<2)|2,{101,45,0}},
/* 10971 */ {(11<<2)|2,{101,45,0}},
/* 10972 */ {(11<<2)|2,{101,45,0}},
/* 10973 */ {(11<<2)|2,{101,45,0}},
/* 10974 */ {(11<<2)|2,{101,45,0}},
/* 10975 */ {(11<<2)|2,{101,45,0}},
/* 10976 */ {(16<<2)|3,{101,46,48}},
/* 10977 */ {(16<<2)|3,{101,46,49}},
/* 10978 */ {(16<<2)|3,{101,46,50}},
/* 10979 */ {(16<<2)|3,{101,46,97}},
/* 10980 */ {(16<<2)|3,{101,46,99}},
/* 10981 */ {(16<<2)|3,{101,46,101}},
/* 10982 */ {(16<<2)|3,{101,46,105}},
/* 10983 */ {(16<<2)|3,{101,46,111}},
/* 10984 */ {(16<<2)|3,{101,46,115}},
/* 10985 */ {(16<<2)|3,{101,46,116}},
/* 10986 */ {(11<<2)|2,{101,46,0}},
/* 10987 */ {(11<<2)|2,{101,46,0}},
/* 10988 */ {(11<<2)|2,{101,46,0}},
/* 10989 */ {(11<<2)|2,{101,46,0}},
/* 10990 */ {(11<<2)|2,{101,46,0}},
/* 10991 */ {(11<<2)|2,{101,46,0}},
/* 10992 */ {(11<<2)|2,{101,46,0}},
/* 10993 */ {(11<<2)|2,{101,46,0}},
/* 10994 */ {(11<<2)|2,{101,46,0}},
/* 10995 */ {(11<<2)|2,{101,46,0}},
/* 10996 */ {(11<<2)|2,{101,46,0}},
/* 10997 */ {(11<<2)|2,{101,46,0}},
/* 10998 */ {(11<<2)|2,{101,46,0}},
/* 10999 */ {(11<<2)|2,{101,46,0}},
/* 11000 */ {(11<<2)|2,{101,46,0}},
/* 11001 */ {(11<<2)|2,{101,46,0}},
/* 11002 */ {(11<<2)|2,{101,46,0}},
/* 11003 */ {(11<<2)|2,{101,46,0}},
/* 11004 */ {(11<<2)|2,{101,46,0}},
/* 11005 */ {(11<<2)|2,{101,46,0}},
/* 11006 */ {(11<<2)|2,{101,46,0}},
/* 11007 */ {(11<<2)|2,{101,46,0}},
/* 11008 */ {(16<<2)|3,{101,47,48}},
/* 11009 */ {(16<<2)|3,{101,47,49}},
/* 11010 */ {(16<<2)|3,{101,47,50}},
/* 11011 */ {(16<<2)|3,{101,47,97}},
/* 11012 */ {(16<<2)|3,{101,47,99}},
/* 11013 */ {(16<<2)|3,{101,47,101}},
/* 11014 */ {(16<<2)|3,{101,47,105}},
/* 11015 */ {(16<<2)|3,{101,47,111}},
/* 11016 */ {(16<<2)|3,{101,47,115}},
/* 11017 */ {(16<<2)|3,{101,47,116}},
/* 11018 */ {(11<<2)|2,{101,47,0}},
/* 11019 */ {(11<<2)|2,{101,47,0}},
/* 11020 */ {(11<<2)|2,{101,47,0}},
/* 11021 */ {(11<<2)|2,{101,47,0}},
/* 11022 */ {(11<<2)|2,{101,47,0}},
/* 11023 */ {(11<<2)|2,{101,47,0}},
/* 11024 */ {(11<<2)|2,{101,47,0}},
/* 11025 */ {(11<<2)|2,{101,47,0}},
/* 11026 */ {(11<<2)|2,{101,47,0}},
/* 11027 */ {(11<<2)|2,{101,47,0}},
/* 11028 */ {(11<<2)|2,{101,47,0}},
/* 11029 */ {(11<<2)|2,{101,47,0}},
/* 11030 */ {(11<<2)|2,{101,47,0}},
/* 11031 */ {(11<<2)|2,{101,47,0}},
/* 11032 */ {(11<<2)|2,{101,47,0}},
/* 11033 */ {(11<<2)|2,{101,47,0}},
/* 11034 */ {(11<<2)|2,{101,47,0}},
/* 11035 */ {(11<<2)|2,{101,47,0}},
/* 11036 */ {(11<<2)|2,{101,47,0}},
/* 11037 */ {(11<<2)|2,{101,47,0}},
/* 11038 */ {(11<<2)|2,{101,47,0}},
/* 11039 */ {(11<<2)|2,{101,47,0}},
/* 11040 */ {(16<<2)|3,{101,51,48}},
/* 11041 */ {(16<<2)|3,{101,51,49}},
/* 11042 */ {(16<<2)|3,{101,51,50}},
/* 11043 */ {(16<<2)|3,{101,51,97}},
/* 11044 */ {(16<<2)|3,{101,51,99}},
/* 11045 */ {(16<<2)|3,{101,51,101}},
/* 11046 */ {(16<<2)|3,{101,51,105}},
/* 11047 */ {(16<<2)|3,{101,51,111}},
/* 11048 */ {(16<<2)|3,{101,51,115}},
/* 11049 */ {(16<<2)|3,{101,51,116}},
/* 11050 */ {(11<<2)|2,{101,51,0}},
/* 11051 */ {(11<<2)|2,{101,51,0}},
/* 11052 */ {(11<<2)|2,{101,51,0}},
/* 11053 */ {(11<<2)|2,{101,51,0}},
/* 11054 */ {(11<<2)|2,{101,51,0}},
/* 11055 */ {(11<<2)|2,{101,51,0}},
/* 11056 */ {(11<<2)|2,{101,51,0}},
/* 11057 */ {(11<<2)|2,{101,51,0}},
/* 11058 */ {(11<<2)|2,{101,51,0}},
/* 11059 */ {(11<<2)|2,{101,51,0}},
/* 11060 */ {(11<<2)|2,{101,51,0}},
/* 11061 */ {(11<<2)|2,{101,51,0}},
/* 11062 */ {(11<<2)|2,{101,51,0}},
/* 11063 */ {(11<<2)|2,{101,51,0}},
/* 11064 */ {(11<<2)|2,{101,51,0}},
/* 11065 */ {(11<<2)|2,{101,51,0}},
/* 11066 */ {(11<<2)|2,{101,51,0}},
/* 11067 */ {(11<<2)|2,{101,51,0}},
/* 11068 */ {(11<<2)|2,{101,51,0}},
/* 11069 */ {(11<<2)|2,{101,51,0}},
/* 11070 */ {(11<<2)|2,{101,51,0}},
/* 11071 */ {(11<<2)|2,{101,51,0}},
/* 11072 */ {(16<<2)|3,{101,52,48}},
/* 11073 */ {(16<<2)|3,{101,52,49}},
/* 11074 */ {(16<<2)|3,{101,52,50}},
/* 11075 */ {(16<<2)|3,{101,52,97}},
/* 11076 */ {(16<<2)|3,{101,52,99}},
/* 11077 */ {(16<<2)|3,{101,52,101}},
/* 11078 */ {(16<<2)|3,{101,52,105}},
/* 11079 */ {(16<<2)|3,{101,52,111}},
/* 11080 */ {(16<<2)|3,{101,52,115}},
/* 11081 */ {(16<<2)|3,{101,52,116}},
/* 11082 */ {(11<<2)|2,{101,52,0}},
/* 11083 */ {(11<<2)|2,{101,52,0}},
/* 11084 */ {(11<<2)|2,{101,52,0}},
/* 11085 */ {(11<<2)|2,{101,52,0}},
/* 11086 */ {(11<<2)|2,{101,52,0}},
/* 11087 */ {(11<<2)|2,{101,52,0}},
/* 11088 */ {(11<<2)|2,{101,52,0}},
/* 11089 */ {(11<<2)|2,{101,52,0}},
/* 11090 */ {(11<<2)|2,{101,52,0}},
/* 11091 */ {(11<<2)|2,{101,52,0}},
/* 11092 */ {(11<<2)|2,{101,52,0}},
/* 11093 */ {(11<<2)|2,{101,52,0}},
/* 11094 */ {(11<<2)|2,{101,52,0}},
/* 11095 */ {(11<<2)|2,{101,52,0}},
/* 11096 */ {(11<<2)|2,{101,52,0}},
/* 11097 */ {(11<<2)|2,{101,52,0}},
/* 11098 */ {(11<<2)|2,{101,52,0}},
/* 11099 */ {(11<<2)|2,{101,52,0}},
/* 11100 */ {(11<<2)|2,{101,52,0}},
/* 11101 */ {(11<<2)|2,{101,52,0}},
/* 11102 */ {(11<<2)|2,{101,52,0}},
/* 11103 */ {(11<<2)|2,{101,52,0}},
/* 11104 */ {(16<<2)|3,{101,53,48}},
/* 11105 */ {(16<<2)|3,{101,53,49}},
/* 11106 */ {(16<<2)|3,{101,53,50}},
/* 11107 */ {(16<<2)|3,{101,53,97}},
/* 11108 */ {(16<<2)|3,{101,53,99}},
/* 11109 */ {(16<<2)|3,{101,53,101}},
/* 11110 */ {(16<<2)|3,{101,53,105}},
/* 11111 */ {(16<<2)|3,{101,53,111}},
/* 11112 */ {(16<<2)|3,{101,53,115}},
/* 11113 */ {(16<<2)|3,{101,53,116}},
/* 11114 */ {(11<<2)|2,{101,53,0}},
/* 11115 */ {(11<<2)|2,{101,53,0}},
/* 11116 */ {(11<<2)|2,{101,53,0}},
/* 11117 */ {(11<<2)|2,{101,53,0}},
/* 11118 */ {(11<<2)|2,{101,53,0}},
/* 11119 */ {(11<<2)|2,{101,53,0}},
/* 11120 */ {(11<<2)|2,{101,53,0}},
/* 11121 */ {(11<<2)|2,{101,53,0}},
/* 11122 */ {(11<<2)|2,{101,53,0}},
/* 11123 */ {(11<<2)|2,{101,53,0}},
/* 11124 */ {(11<<2)|2,{101,53,0}},
/* 11125 */ {(11<<2)|2,{101,53,0}},
/* 11126 */ {(11<<2)|2,{101,53,0}},
/* 11127 */ {(11<<2)|2,{101,53,0}},
/* 11128 */ {(11<<2)|2,{101,53,0}},
/* 11129 */ {(11<<2)|2,{101,53,0}},
/* 11130 */ {(11<<2)|2,{101,53,0}},
/* 11131 */ {(11<<2)|2,{101,53,0}},
/* 11132 */ {(11<<2)|2,{101,53,0}},
/* 11133 */ {(11<<2)|2,{101,53,0}},
/* 11134 */ {(11<<2)|2,{101,53,0}},
/* 11135 */ {(11<<2)|2,{101,53,0}},
/* 11136 */ {(16<<2)|3,{101,54,48}},
/* 11137 */ {(16<<2)|3,{101,54,49}},
/* 11138 */ {(16<<2)|3,{101,54,50}},
/* 11139 */ {(16<<2)|3,{101,54,97}},
/* 11140 */ {(16<<2)|3,{101,54,99}},
/* 11141 */ {(16<<2)|3,{101,54,101}},
/* 11142 */ {(16<<2)|3,{101,54,105}},
/* 11143 */ {(16<<2)|3,{101,54,111}},
/* 11144 */ {(16<<2)|3,{101,54,115}},
/* 11145 */ {(16<<2)|3,{101,54,116}},
/* 11146 */ {(11<<2)|2,{101,54,0}},
/* 11147 */ {(11<<2)|2,{101,54,0}},
/* 11148 */ {(11<<2)|2,{101,54,0}},
/* 11149 */ {(11<<2)|2,{101,54,0}},
/* 11150 */ {(11<<2)|2,{101,54,0}},
/* 11151 */ {(11<<2)|2,{101,54,0}},
/* 11152 */ {(11<<2)|2,{101,54,0}},
/* 11153 */ {(11<<2)|2,{101,54,0}},
/* 11154 */ {(11<<2)|2,{101,54,0}},
/* 11155 */ {(11<<2)|2,{101,54,0}},
/* 11156 */ {(11<<2)|2,{101,54,0}},
/* 11157 */ {(11<<2)|2,{101,54,0}},
/* 11158 */ {(11<<2)|2,{101,54,0}},
/* 11159 */ {(11<<2)|2,{101,54,0}},
/* 11160 */ {(11<<2)|2,{101,54,0}},
/* 11161 */ {(11<<2)|2,{101,54,0}},
/* 11162 */ {(11<<2)|2,{101,54,0}},
/* 11163 */ {(11<<2)|2,{101,54,0}},
/* 11164 */ {(11<<2)|2,{101,54,0}},
/* 11165 */ {(11<<2)|2,{101,54,0}},
/* 11166 */ {(11<<2)|2,{101,54,0}},
/* 11167 */ {(11<<2)|2,{101,54,0}},
/* 11168 */ {(16<<2)|3,{101,55,48}},
/* 11169 */ {(16<<2)|3,{101,55,49}},
/* 11170 */ {(16<<2)|3,{101,55,50}},
/* 11171 */ {(16<<2)|3,{101,55,97}},
/* 11172 */ {(16<<2)|3,{101,55,99}},
/* 11173 */ {(16<<2)|3,{101,55,101}},
/* 11174 */ {(16<<2)|3,{101,55,105}},
/* 11175 */ {(16<<2)|3,{101,55,111}},
/* 11176 */ {(16<<2)|3,{101,55,115}},
/* 11177 */ {(16<<2)|3,{101,55,116}},
/* 11178 */ {(11<<2)|2,{101,55,0}},
/* 11179 */ {(11<<2)|2,{101,55,0}},
/* 11180 */ {(11<<2)|2,{101,55,0}},
/* 11181 */ {(11<<2)|2,{101,55,0}},
/* 11182 */ {(11<<2)|2,{101,55,0}},
/* 11183 */ {(11<<2)|2,{101,55,0}},
/* 11184 */ {(11<<2)|2,{101,55,0}},
/* 11185 */ {(11<<2)|2,{101,55,0}},
/* 11186 */ {(11<<2)|2,{101,55,0}},
/* 11187 */ {(11<<2)|2,{101,55,0}},
/* 11188 */ {(11<<2)|2,{101,55,0}},
/* 11189 */ {(11<<2)|2,{101,55,0}},
/* 11190 */ {(11<<2)|2,{101,55,0}},
/* 11191 */ {(11<<2)|2,{101,55,0}},
/* 11192 */ {(11<<2)|2,{101,55,0}},
/* 11193 */ {(11<<2)|2,{101,55,0}},
/* 11194 */ {(11<<2)|2,{101,55,0}},
/* 11195 */ {(11<<2)|2,{101,55,0}},
/* 11196 */ {(11<<2)|2,{101,55,0}},
/* 11197 */ {(11<<2)|2,{101,55,0}},
/* 11198 */ {(11<<2)|2,{101,55,0}},
/* 11199 */ {(11<<2)|2,{101,55,0}},
/* 11200 */ {(16<<2)|3,{101,56,48}},
/* 11201 */ {(16<<2)|3,{101,56,49}},
/* 11202 */ {(16<<2)|3,{101,56,50}},
/* 11203 */ {(16<<2)|3,{101,56,97}},
/* 11204 */ {(16<<2)|3,{101,56,99}},
/* 11205 */ {(16<<2)|3,{101,56,101}},
/* 11206 */ {(16<<2)|3,{101,56,105}},
/* 11207 */ {(16<<2)|3,{101,56,111}},
/* 11208 */ {(16<<2)|3,{101,56,115}},
/* 11209 */ {(16<<2)|3,{101,56,116}},
/* 11210 */ {(11<<2)|2,{101,56,0}},
/* 11211 */ {(11<<2)|2,{101,56,0}},
/* 11212 */ {(11<<2)|2,{101,56,0}},
/* 11213 */ {(11<<2)|2,{101,56,0}},
/* 11214 */ {(11<<2)|2,{101,56,0}},
/* 11215 */ {(11<<2)|2,{101,56,0}},
/* 11216 */ {(11<<2)|2,{101,56,0}},
/* 11217 */ {(11<<2)|2,{101,56,0}},
/* 11218 */ {(11<<2)|2,{101,56,0}},
/* 11219 */ {(11<<2)|2,{101,56,0}},
/* 11220 */ {(11<<2)|2,{101,56,0}},
/* 11221 */ {(11<<2)|2,{101,56,0}},
/* 11222 */ {(11<<2)|2,{101,56,0}},
/* 11223 */ {(11<<2)|2,{101,56,0}},
/* 11224 */ {(11<<2)|2,{101,56,0}},
/* 11225 */ {(11<<2)|2,{101,56,0}},
/* 11226 */ {(11<<2)|2,{101,56,0}},
/* 11227 */ {(11<<2)|2,{101,56,0}},
/* 11228 */ {(11<<2)|2,{101,56,0}},
/* 11229 */ {(11<<2)|2,{101,56,0}},
/* 11230 */ {(11<<2)|2,{101,56,0}},
/* 11231 */ {(11<<2)|2,{101,56,0}},
/* 11232 */ {(16<<2)|3,{101,57,48}},
/* 11233 */ {(16<<2)|3,{101,57,49}},
/* 11234 */ {(16<<2)|3,{101,57,50}},
/* 11235 */ {(16<<2)|3,{101,57,97}},
/* 11236 */ {(16<<2)|3,{101,57,99}},
/* 11237 */ {(16<<2)|3,{101,57,101}},
/* 11238 */ {(16<<2)|3,{101,57,105}},
/* 11239 */ {(16<<2)|3,{101,57,111}},
/* 11240 */ {(16<<2)|3,{101,57,115}},
/* 11241 */ {(16<<2)|3,{101,57,116}},
/* 11242 */ {(11<<2)|2,{101,57,0}},
/* 11243 */ {(11<<2)|2,{101,57,0}},
/* 11244 */ {(11<<2)|2,{101,57,0}},
/* 11245 */ {(11<<2)|2,{101,57,0}},
/* 11246 */ {(11<<2)|2,{101,57,0}},
/* 11247 */ {(11<<2)|2,{101,57,0}},
/* 11248 */ {(11<<2)|2,{101,57,0}},
/* 11249 */ {(11<<2)|2,{101,57,0}},
/* 11250 */ {(11<<2)|2,{101,57,0}},
/* 11251 */ {(11<<2)|2,{101,57,0}},
/* 11252 */ {(11<<2)|2,{101,57,0}},
/* 11253 */ {(11<<2)|2,{101,57,0}},
/* 11254 */ {(11<<2)|2,{101,57,0}},
/* 11255 */ {(11<<2)|2,{101,57,0}},
/* 11256 */ {(11<<2)|2,{101,57,0}},
/* 11257 */ {(11<<2)|2,{101,57,0}},
/* 11258 */ {(11<<2)|2,{101,57,0}},
/* 11259 */ {(11<<2)|2,{101,57,0}},
/* 11260 */ {(11<<2)|2,{101,57,0}},
/* 11261 */ {(11<<2)|2,{101,57,0}},
/* 11262 */ {(11<<2)|2,{101,57,0}},
/* 11263 */ {(11<<2)|2,{101,57,0}},
/* 11264 */ {(16<<2)|3,{101,61,48}},
/* 11265 */ {(16<<2)|3,{101,61,49}},
/* 11266 */ {(16<<2)|3,{101,61,50}},
/* 11267 */ {(16<<2)|3,{101,61,97}},
/* 11268 */ {(16<<2)|3,{101,61,99}},
/* 11269 */ {(16<<2)|3,{101,61,101}},
/* 11270 */ {(16<<2)|3,{101,61,105}},
/* 11271 */ {(16<<2)|3,{101,61,111}},
/* 11272 */ {(16<<2)|3,{101,61,115}},
/* 11273 */ {(16<<2)|3,{101,61,116}},
/* 11274 */ {(11<<2)|2,{101,61,0}},
/* 11275 */ {(11<<2)|2,{101,61,0}},
/* 11276 */ {(11<<2)|2,{101,61,0}},
/* 11277 */ {(11<<2)|2,{101,61,0}},
/* 11278 */ {(11<<2)|2,{101,61,0}},
/* 11279 */ {(11<<2)|2,{101,61,0}},
/* 11280 */ {(11<<2)|2,{101,61,0}},
/* 11281 */ {(11<<2)|2,{101,61,0}},
/* 11282 */ {(11<<2)|2,{101,61,0}},
/* 11283 */ {(11<<2)|2,{101,61,0}},
/* 11284 */ {(11<<2)|2,{101,61,0}},
/* 11285 */ {(11<<2)|2,{101,61,0}},
/* 11286 */ {(11<<2)|2,{101,61,0}},
/* 11287 */ {(11<<2)|2,{101,61,0}},
/* 11288 */ {(11<<2)|2,{101,61,0}},
/* 11289 */ {(11<<2)|2,{101,61,0}},
/* 11290 */ {(11<<2)|2,{101,61,0}},
/* 11291 */ {(11<<2)|2,{101,61,0}},
/* 11292 */ {(11<<2)|2,{101,61,0}},
/* 11293 */ {(11<<2)|2,{101,61,0}},
/* 11294 */ {(11<<2)|2,{101,61,0}},
/* 11295 */ {(11<<2)|2,{101,61,0}},
/* 11296 */ {(16<<2)|3,{101,65,48}},
/* 11297 */ {(16<<2)|3,{101,65,49}},
/* 11298 */ {(16<<2)|3,{101,65,50}},
/* 11299 */ {(16<<2)|3,{101,65,97}},
/* 11300 */ {(16<<2)|3,{101,65,99}},
/* 11301 */ {(16<<2)|3,{101,65,101}},
/* 11302 */ {(16<<2)|3,{101,65,105}},
/* 11303 */ {(16<<2)|3,{101,65,111}},
/* 11304 */ {(16<<2)|3,{101,65,115}},
/* 11305 */ {(16<<2)|3,{101,65,116}},
/* 11306 */ {(11<<2)|2,{101,65,0}},
/* 11307 */ {(11<<2)|2,{101,65,0}},
/* 11308 */ {(11<<2)|2,{101,65,0}},
/* 11309 */ {(11<<2)|2,{101,65,0}},
/* 11310 */ {(11<<2)|2,{101,65,0}},
/* 11311 */ {(11<<2)|2,{101,65,0}},
/* 11312 */ {(11<<2)|2,{101,65,0}},
/* 11313 */ {(11<<2)|2,{101,65,0}},
/* 11314 */ {(11<<2)|2,{101,65,0}},
/* 11315 */ {(11<<2)|2,{101,65,0}},
/* 11316 */ {(11<<2)|2,{101,65,0}},
/* 11317 */ {(11<<2)|2,{101,65,0}},
/* 11318 */ {(11<<2)|2,{101,65,0}},
/* 11319 */ {(11<<2)|2,{101,65,0}},
/* 11320 */ {(11<<2)|2,{101,65,0}},
/* 11321 */ {(11<<2)|2,{101,65,0}},
/* 11322 */ {(11<<2)|2,{101,65,0}},
/* 11323 */ {(11<<2)|2,{101,65,0}},
/* 11324 */ {(11<<2)|2,{101,65,0}},
/* 11325 */ {(11<<2)|2,{101,65,0}},
/* 11326 */ {(11<<2)|2,{101,65,0}},
/* 11327 */ {(11<<2)|2,{101,65,0}},
/* 11328 */ {(16<<2)|3,{101,95,48}},
/* 11329 */ {(16<<2)|3,{101,95,49}},
/* 11330 */ {(16<<2)|3,{101,95,50}},
/* 11331 */ {(16<<2)|3,{101,95,97}},
/* 11332 */ {(16<<2)|3,{101,95,99}},
/* 11333 */ {(16<<2)|3,{101,95,101}},
/* 11334 */ {(16<<2)|3,{101,95,105}},
/* 11335 */ {(16<<2)|3,{101,95,111}},
/* 11336 */ {(16<<2)|3,{101,95,115}},
/* 11337 */ {(16<<2)|3,{101,95,116}},
/* 11338 */ {(11<<2)|2,{101,95,0}},
/* 11339 */ {(11<<2)|2,{101,95,0}},
/* 11340 */ {(11<<2)|2,{101,95,0}},
/* 11341 */ {(11<<2)|2,{101,95,0}},
/* 11342 */ {(11<<2)|2,{101,95,0}},
/* 11343 */ {(11<<2)|2,{101,95,0}},
/* 11344 */ {(11<<2)|2,{101,95,0}},
/* 11345 */ {(11<<2)|2,{101,95,0}},
/* 11346 */ {(11<<2)|2,{101,95,0}},
/* 11347 */ {(11<<2)|2,{101,95,0}},
/* 11348 */ {(11<<2)|2,{101,95,0}},
/* 11349 */ {(11<<2)|2,{101,95,0}},
/* 11350 */ {(11<<2)|2,{101,95,0}},
/* 11351 */ {(11<<2)|2,{101,95,0}},
/* 11352 */ {(11<<2)|2,{101,95,0}},
/* 11353 */ {(11<<2)|2,{101,95,0}},
/* 11354 */ {(11<<2)|2,{101,95,0}},
/* 11355 */ {(11<<2)|2,{101,95,0}},
/* 11356 */ {(11<<2)|2,{101,95,0}},
/* 11357 */ {(11<<2)|2,{101,95,0}},
/* 11358 */ {(11<<2)|2,{101,95,0}},
/* 11359 */ {(11<<2)|2,{101,95,0}},
/* 11360 */ {(16<<2)|3,{101,98,48}},
/* 11361 */ {(16<<2)|3,{101,98,49}},
/* 11362 */ {(16<<2)|3,{101,98,50}},
/* 11363 */ {(16<<2)|3,{101,98,97}},
/* 11364 */ {(16<<2)|3,{101,98,99}},
/* 11365 */ {(16<<2)|3,{101,98,101}},
/* 11366 */ {(16<<2)|3,{101,98,105}},
/* 11367 */ {(16<<2)|3,{101,98,111}},
/* 11368 */ {(16<<2)|3,{101,98,115}},
/* 11369 */ {(16<<2)|3,{101,98,116}},
/* 11370 */ {(11<<2)|2,{101,98,0}},
/* 11371 */ {(11<<2)|2,{101,98,0}},
/* 11372 */ {(11<<2)|2,{101,98,0}},
/* 11373 */ {(11<<2)|2,{101,98,0}},
/* 11374 */ {(11<<2)|2,{101,98,0}},
/* 11375 */ {(11<<2)|2,{101,98,0}},
/* 11376 */ {(11<<2)|2,{101,98,0}},
/* 11377 */ {(11<<2)|2,{101,98,0}},
/* 11378 */ {(11<<2)|2,{101,98,0}},
/* 11379 */ {(11<<2)|2,{101,98,0}},
/* 11380 */ {(11<<2)|2,{101,98,0}},
/* 11381 */ {(11<<2)|2,{101,98,0}},
/* 11382 */ {(11<<2)|2,{101,98,0}},
/* 11383 */ {(11<<2)|2,{101,98,0}},
/* 11384 */ {(11<<2)|2,{101,98,0}},
/* 11385 */ {(11<<2)|2,{101,98,0}},
/* 11386 */ {(11<<2)|2,{101,98,0}},
/* 11387 */ {(11<<2)|2,{101,98,0}},
/* 11388 */ {(11<<2)|2,{101,98,0}},
/* 11389 */ {(11<<2)|2,{101,98,0}},
/* 11390 */ {(11<<2)|2,{101,98,0}},
/* 11391 */ {(11<<2)|2,{101,98,0}},
/* 11392 */ {(16<<2)|3,{101,100,48}},
/* 11393 */ {(16<<2)|3,{101,100,49}},
/* 11394 */ {(16<<2)|3,{101,100,50}},
/* 11395 */ {(16<<2)|3,{101,100,97}},
/* 11396 */ {(16<<2)|3,{101,100,99}},
/* 11397 */ {(16<<2)|3,{101,100,101}},
/* 11398 */ {(16<<2)|3,{101,100,105}},
/* 11399 */ {(16<<2)|3,{101,100,111}},
/* 11400 */ {(16<<2)|3,{101,100,115}},
/* 11401 */ {(16<<2)|3,{101,100,116}},
/* 11402 */ {(11<<2)|2,{101,100,0}},
/* 11403 */ {(11<<2)|2,{101,100,0}},
/* 11404 */ {(11<<2)|2,{101,100,0}},
/* 11405 */ {(11<<2)|2,{101,100,0}},
/* 11406 */ {(11<<2)|2,{101,100,0}},
/* 11407 */ {(11<<2)|2,{101,100,0}},
/* 11408 */ {(11<<2)|2,{101,100,0}},
/* 11409 */ {(11<<2)|2,{101,100,0}},
/* 11410 */ {(11<<2)|2,{101,100,0}},
/* 11411 */ {(11<<2)|2,{101,100,0}},
/* 11412 */ {(11<<2)|2,{101,100,0}},
/* 11413 */ {(11<<2)|2,{101,100,0}},
/* 11414 */ {(11<<2)|2,{101,100,0}},
/* 11415 */ {(11<<2)|2,{101,100,0}},
/* 11416 */ {(11<<2)|2,{101,100,0}},
/* 11417 */ {(11<<2)|2,{101,100,0}},
/* 11418 */ {(11<<2)|2,{101,100,0}},
/* 11419 */ {(11<<2)|2,{101,100,0}},
/* 11420 */ {(11<<2)|2,{101,100,0}},
/* 11421 */ {(11<<2)|2,{101,100,0}},
/* 11422 */ {(11<<2)|2,{101,100,0}},
/* 11423 */ {(11<<2)|2,{101,100,0}},
/* 11424 */ {(16<<2)|3,{101,102,48}},
/* 11425 */ {(16<<2)|3,{101,102,49}},
/* 11426 */ {(16<<2)|3,{101,102,50}},
/* 11427 */ {(16<<2)|3,{101,102,97}},
/* 11428 */ {(16<<2)|3,{101,102,99}},
/* 11429 */ {(16<<2)|3,{101,102,101}},
/* 11430 */ {(16<<2)|3,{101,102,105}},
/* 11431 */ {(16<<2)|3,{101,102,111}},
/* 11432 */ {(16<<2)|3,{101,102,115}},
/* 11433 */ {(16<<2)|3,{101,102,116}},
/* 11434 */ {(11<<2)|2,{101,102,0}},
/* 11435 */ {(11<<2)|2,{101,102,0}},
/* 11436 */ {(11<<2)|2,{101,102,0}},
/* 11437 */ {(11<<2)|2,{101,102,0}},
/* 11438 */ {(11<<2)|2,{101,102,0}},
/* 11439 */ {(11<<2)|2,{101,102,0}},
/* 11440 */ {(11<<2)|2,{101,102,0}},
/* 11441 */ {(11<<2)|2,{101,102,0}},
/* 11442 */ {(11<<2)|2,{101,102,0}},
/* 11443 */ {(11<<2)|2,{101,102,0}},
/* 11444 */ {(11<<2)|2,{101,102,0}},
/* 11445 */ {(11<<2)|2,{101,102,0}},
/* 11446 */ {(11<<2)|2,{101,102,0}},
/* 11447 */ {(11<<2)|2,{101,102,0}},
/* 11448 */ {(11<<2)|2,{101,102,0}},
/* 11449 */ {(11<<2)|2,{101,102,0}},
/* 11450 */ {(11<<2)|2,{101,102,0}},
/* 11451 */ {(11<<2)|2,{101,102,0}},
/* 11452 */ {(11<<2)|2,{101,102,0}},
/* 11453 */ {(11<<2)|2,{101,102,0}},
/* 11454 */ {(11<<2)|2,{101,102,0}},
/* 11455 */ {(11<<2)|2,{101,102,0}},
/* 11456 */ {(16<<2)|3,{101,103,48}},
/* 11457 */ {(16<<2)|3,{101,103,49}},
/* 11458 */ {(16<<2)|3,{101,103,50}},
/* 11459 */ {(16<<2)|3,{101,103,97}},
/* 11460 */ {(16<<2)|3,{101,103,99}},
/* 11461 */ {(16<<2)|3,{101,103,101}},
/* 11462 */ {(16<<2)|3,{101,103,105}},
/* 11463 */ {(16<<2)|3,{101,103,111}},
/* 11464 */ {(16<<2)|3,{101,103,115}},
/* 11465 */ {(16<<2)|3,{101,103,116}},
/* 11466 */ {(11<<2)|2,{101,103,0}},
/* 11467 */ {(11<<2)|2,{101,103,0}},
/* 11468 */ {(11<<2)|2,{101,103,0}},
/* 11469 */ {(11<<2)|2,{101,103,0}},
/* 11470 */ {(11<<2)|2,{101,103,0}},
/* 11471 */ {(11<<2)|2,{101,103,0}},
/* 11472 */ {(11<<2)|2,{101,103,0}},
/* 11473 */ {(11<<2)|2,{101,103,0}},
/* 11474 */ {(11<<2)|2,{101,103,0}},
/* 11475 */ {(11<<2)|2,{101,103,0}},
/* 11476 */ {(11<<2)|2,{101,103,0}},
/* 11477 */ {(11<<2)|2,{101,103,0}},
/* 11478 */ {(11<<2)|2,{101,103,0}},
/* 11479 */ {(11<<2)|2,{101,103,0}},
/* 11480 */ {(11<<2)|2,{101,103,0}},
/* 11481 */ {(11<<2)|2,{101,103,0}},
/* 11482 */ {(11<<2)|2,{101,103,0}},
/* 11483 */ {(11<<2)|2,{101,103,0}},
/* 11484 */ {(11<<2)|2,{101,103,0}},
/* 11485 */ {(11<<2)|2,{101,103,0}},
/* 11486 */ {(11<<2)|2,{101,103,0}},
/* 11487 */ {(11<<2)|2,{101,103,0}},
/* 11488 */ {(16<<2)|3,{101,104,48}},
/* 11489 */ {(16<<2)|3,{101,104,49}},
/* 11490 */ {(16<<2)|3,{101,104,50}},
/* 11491 */ {(16<<2)|3,{101,104,97}},
/* 11492 */ {(16<<2)|3,{101,104,99}},
/* 11493 */ {(16<<2)|3,{101,104,101}},
/* 11494 */ {(16<<2)|3,{101,104,105}},
/* 11495 */ {(16<<2)|3,{101,104,111}},
/* 11496 */ {(16<<2)|3,{101,104,115}},
/* 11497 */ {(16<<2)|3,{101,104,116}},
/* 11498 */ {(11<<2)|2,{101,104,0}},
/* 11499 */ {(11<<2)|2,{101,104,0}},
/* 11500 */ {(11<<2)|2,{101,104,0}},
/* 11501 */ {(11<<2)|2,{101,104,0}},
/* 11502 */ {(11<<2)|2,{101,104,0}},
/* 11503 */ {(11<<2)|2,{101,104,0}},
/* 11504 */ {(11<<2)|2,{101,104,0}},
/* 11505 */ {(11<<2)|2,{101,104,0}},
/* 11506 */ {(11<<2)|2,{101,104,0}},
/* 11507 */ {(11<<2)|2,{101,104,0}},
/* 11508 */ {(11<<2)|2,{101,104,0}},
/* 11509 */ {(11<<2)|2,{101,104,0}},
/* 11510 */ {(11<<2)|2,{101,104,0}},
/* 11511 */ {(11<<2)|2,{101,104,0}},
/* 11512 */ {(11<<2)|2,{101,104,0}},
/* 11513 */ {(11<<2)|2,{101,104,0}},
/* 11514 */ {(11<<2)|2,{101,104,0}},
/* 11515 */ {(11<<2)|2,{101,104,0}},
/* 11516 */ {(11<<2)|2,{101,104,0}},
/* 11517 */ {(11<<2)|2,{101,104,0}},
/* 11518 */ {(11<<2)|2,{101,104,0}},
/* 11519 */ {(11<<2)|2,{101,104,0}},
/* 11520 */ {(16<<2)|3,{101,108,48}},
/* 11521 */ {(16<<2)|3,{101,108,49}},
/* 11522 */ {(16<<2)|3,{101,108,50}},
/* 11523 */ {(16<<2)|3,{101,108,97}},
/* 11524 */ {(16<<2)|3,{101,108,99}},
/* 11525 */ {(16<<2)|3,{101,108,101}},
/* 11526 */ {(16<<2)|3,{101,108,105}},
/* 11527 */ {(16<<2)|3,{101,108,111}},
/* 11528 */ {(16<<2)|3,{101,108,115}},
/* 11529 */ {(16<<2)|3,{101,108,116}},
/* 11530 */ {(11<<2)|2,{101,108,0}},
/* 11531 */ {(11<<2)|2,{101,108,0}},
/* 11532 */ {(11<<2)|2,{101,108,0}},
/* 11533 */ {(11<<2)|2,{101,108,0}},
/* 11534 */ {(11<<2)|2,{101,108,0}},
/* 11535 */ {(11<<2)|2,{101,108,0}},
/* 11536 */ {(11<<2)|2,{101,108,0}},
/* 11537 */ {(11<<2)|2,{101,108,0}},
/* 11538 */ {(11<<2)|2,{101,108,0}},
/* 11539 */ {(11<<2)|2,{101,108,0}},
/* 11540 */ {(11<<2)|2,{101,108,0}},
/* 11541 */ {(11<<2)|2,{101,108,0}},
/* 11542 */ {(11<<2)|2,{101,108,0}},
/* 11543 */ {(11<<2)|2,{101,108,0}},
/* 11544 */ {(11<<2)|2,{101,108,0}},
/* 11545 */ {(11<<2)|2,{101,108,0}},
/* 11546 */ {(11<<2)|2,{101,108,0}},
/* 11547 */ {(11<<2)|2,{101,108,0}},
/* 11548 */ {(11<<2)|2,{101,108,0}},
/* 11549 */ {(11<<2)|2,{101,108,0}},
/* 11550 */ {(11<<2)|2,{101,108,0}},
/* 11551 */ {(11<<2)|2,{101,108,0}},
/* 11552 */ {(16<<2)|3,{101,109,48}},
/* 11553 */ {(16<<2)|3,{101,109,49}},
/* 11554 */ {(16<<2)|3,{101,109,50}},
/* 11555 */ {(16<<2)|3,{101,109,97}},
/* 11556 */ {(16<<2)|3,{101,109,99}},
/* 11557 */ {(16<<2)|3,{101,109,101}},
/* 11558 */ {(16<<2)|3,{101,109,105}},
/* 11559 */ {(16<<2)|3,{101,109,111}},
/* 11560 */ {(16<<2)|3,{101,109,115}},
/* 11561 */ {(16<<2)|3,{101,109,116}},
/* 11562 */ {(11<<2)|2,{101,109,0}},
/* 11563 */ {(11<<2)|2,{101,109,0}},
/* 11564 */ {(11<<2)|2,{101,109,0}},
/* 11565 */ {(11<<2)|2,{101,109,0}},
/* 11566 */ {(11<<2)|2,{101,109,0}},
/* 11567 */ {(11<<2)|2,{101,109,0}},
/* 11568 */ {(11<<2)|2,{101,109,0}},
/* 11569 */ {(11<<2)|2,{101,109,0}},
/* 11570 */ {(11<<2)|2,{101,109,0}},
/* 11571 */ {(11<<2)|2,{101,109,0}},
/* 11572 */ {(11<<2)|2,{101,109,0}},
/* 11573 */ {(11<<2)|2,{101,109,0}},
/* 11574 */ {(11<<2)|2,{101,109,0}},
/* 11575 */ {(11<<2)|2,{101,109,0}},
/* 11576 */ {(11<<2)|2,{101,109,0}},
/* 11577 */ {(11<<2)|2,{101,109,0}},
/* 11578 */ {(11<<2)|2,{101,109,0}},
/* 11579 */ {(11<<2)|2,{101,109,0}},
/* 11580 */ {(11<<2)|2,{101,109,0}},
/* 11581 */ {(11<<2)|2,{101,109,0}},
/* 11582 */ {(11<<2)|2,{101,109,0}},
/* 11583 */ {(11<<2)|2,{101,109,0}},
/* 11584 */ {(16<<2)|3,{101,110,48}},
/* 11585 */ {(16<<2)|3,{101,110,49}},
/* 11586 */ {(16<<2)|3,{101,110,50}},
/* 11587 */ {(16<<2)|3,{101,110,97}},
/* 11588 */ {(16<<2)|3,{101,110,99}},
/* 11589 */ {(16<<2)|3,{101,110,101}},
/* 11590 */ {(16<<2)|3,{101,110,105}},
/* 11591 */ {(16<<2)|3,{101,110,111}},
/* 11592 */ {(16<<2)|3,{101,110,115}},
/* 11593 */ {(16<<2)|3,{101,110,116}},
/* 11594 */ {(11<<2)|2,{101,110,0}},
/* 11595 */ {(11<<2)|2,{101,110,0}},
/* 11596 */ {(11<<2)|2,{101,110,0}},
/* 11597 */ {(11<<2)|2,{101,110,0}},
/* 11598 */ {(11<<2)|2,{101,110,0}},
/* 11599 */ {(11<<2)|2,{101,110,0}},
/* 11600 */ {(11<<2)|2,{101,110,0}},
/* 11601 */ {(11<<2)|2,{101,110,0}},
/* 11602 */ {(11<<2)|2,{101,110,0}},
/* 11603 */ {(11<<2)|2,{101,110,0}},
/* 11604 */ {(11<<2)|2,{101,110,0}},
/* 11605 */ {(11<<2)|2,{101,110,0}},
/* 11606 */ {(11<<2)|2,{101,110,0}},
/* 11607 */ {(11<<2)|2,{101,110,0}},
/* 11608 */ {(11<<2)|2,{101,110,0}},
/* 11609 */ {(11<<2)|2,{101,110,0}},
/* 11610 */ {(11<<2)|2,{101,110,0}},
/* 11611 */ {(11<<2)|2,{101,110,0}},
/* 11612 */ {(11<<2)|2,{101,110,0}},
/* 11613 */ {(11<<2)|2,{101,110,0}},
/* 11614 */ {(11<<2)|2,{101,110,0}},
/* 11615 */ {(11<<2)|2,{101,110,0}},
/* 11616 */ {(16<<2)|3,{101,112,48}},
/* 11617 */ {(16<<2)|3,{101,112,49}},
/* 11618 */ {(16<<2)|3,{101,112,50}},
/* 11619 */ {(16<<2)|3,{101,112,97}},
/* 11620 */ {(16<<2)|3,{101,112,99}},
/* 11621 */ {(16<<2)|3,{101,112,101}},
/* 11622 */ {(16<<2)|3,{101,112,105}},
/* 11623 */ {(16<<2)|3,{101,112,111}},
/* 11624 */ {(16<<2)|3,{101,112,115}},
/* 11625 */ {(16<<2)|3,{101,112,116}},
/* 11626 */ {(11<<2)|2,{101,112,0}},
/* 11627 */ {(11<<2)|2,{101,112,0}},
/* 11628 */ {(11<<2)|2,{101,112,0}},
/* 11629 */ {(11<<2)|2,{101,112,0}},
/* 11630 */ {(11<<2)|2,{101,112,0}},
/* 11631 */ {(11<<2)|2,{101,112,0}},
/* 11632 */ {(11<<2)|2,{101,112,0}},
/* 11633 */ {(11<<2)|2,{101,112,0}},
/* 11634 */ {(11<<2)|2,{101,112,0}},
/* 11635 */ {(11<<2)|2,{101,112,0}},
/* 11636 */ {(11<<2)|2,{101,112,0}},
/* 11637 */ {(11<<2)|2,{101,112,0}},
/* 11638 */ {(11<<2)|2,{101,112,0}},
/* 11639 */ {(11<<2)|2,{101,112,0}},
/* 11640 */ {(11<<2)|2,{101,112,0}},
/* 11641 */ {(11<<2)|2,{101,112,0}},
/* 11642 */ {(11<<2)|2,{101,112,0}},
/* 11643 */ {(11<<2)|2,{101,112,0}},
/* 11644 */ {(11<<2)|2,{101,112,0}},
/* 11645 */ {(11<<2)|2,{101,112,0}},
/* 11646 */ {(11<<2)|2,{101,112,0}},
/* 11647 */ {(11<<2)|2,{101,112,0}},
/* 11648 */ {(16<<2)|3,{101,114,48}},
/* 11649 */ {(16<<2)|3,{101,114,49}},
/* 11650 */ {(16<<2)|3,{101,114,50}},
/* 11651 */ {(16<<2)|3,{101,114,97}},
/* 11652 */ {(16<<2)|3,{101,114,99}},
/* 11653 */ {(16<<2)|3,{101,114,101}},
/* 11654 */ {(16<<2)|3,{101,114,105}},
/* 11655 */ {(16<<2)|3,{101,114,111}},
/* 11656 */ {(16<<2)|3,{101,114,115}},
/* 11657 */ {(16<<2)|3,{101,114,116}},
/* 11658 */ {(11<<2)|2,{101,114,0}},
/* 11659 */ {(11<<2)|2,{101,114,0}},
/* 11660 */ {(11<<2)|2,{101,114,0}},
/* 11661 */ {(11<<2)|2,{101,114,0}},
/* 11662 */ {(11<<2)|2,{101,114,0}},
/* 11663 */ {(11<<2)|2,{101,114,0}},
/* 11664 */ {(11<<2)|2,{101,114,0}},
/* 11665 */ {(11<<2)|2,{101,114,0}},
/* 11666 */ {(11<<2)|2,{101,114,0}},
/* 11667 */ {(11<<2)|2,{101,114,0}},
/* 11668 */ {(11<<2)|2,{101,114,0}},
/* 11669 */ {(11<<2)|2,{101,114,0}},
/* 11670 */ {(11<<2)|2,{101,114,0}},
/* 11671 */ {(11<<2)|2,{101,114,0}},
/* 11672 */ {(11<<2)|2,{101,114,0}},
/* 11673 */ {(11<<2)|2,{101,114,0}},
/* 11674 */ {(11<<2)|2,{101,114,0}},
/* 11675 */ {(11<<2)|2,{101,114,0}},
/* 11676 */ {(11<<2)|2,{101,114,0}},
/* 11677 */ {(11<<2)|2,{101,114,0}},
/* 11678 */ {(11<<2)|2,{101,114,0}},
/* 11679 */ {(11<<2)|2,{101,114,0}},
/* 11680 */ {(16<<2)|3,{101,117,48}},
/* 11681 */ {(16<<2)|3,{101,117,49}},
/* 11682 */ {(16<<2)|3,{101,117,50}},
/* 11683 */ {(16<<2)|3,{101,117,97}},
/* 11684 */ {(16<<2)|3,{101,117,99}},
/* 11685 */ {(16<<2)|3,{101,117,101}},
/* 11686 */ {(16<<2)|3,{101,117,105}},
/* 11687 */ {(16<<2)|3,{101,117,111}},
/* 11688 */ {(16<<2)|3,{101,117,115}},
/* 11689 */ {(16<<2)|3,{101,117,116}},
/* 11690 */ {(11<<2)|2,{101,117,0}},
/* 11691 */ {(11<<2)|2,{101,117,0}},
/* 11692 */ {(11<<2)|2,{101,117,0}},
/* 11693 */ {(11<<2)|2,{101,117,0}},
/* 11694 */ {(11<<2)|2,{101,117,0}},
/* 11695 */ {(11<<2)|2,{101,117,0}},
/* 11696 */ {(11<<2)|2,{101,117,0}},
/* 11697 */ {(11<<2)|2,{101,117,0}},
/* 11698 */ {(11<<2)|2,{101,117,0}},
/* 11699 */ {(11<<2)|2,{101,117,0}},
/* 11700 */ {(11<<2)|2,{101,117,0}},
/* 11701 */ {(11<<2)|2,{101,117,0}},
/* 11702 */ {(11<<2)|2,{101,117,0}},
/* 11703 */ {(11<<2)|2,{101,117,0}},
/* 11704 */ {(11<<2)|2,{101,117,0}},
/* 11705 */ {(11<<2)|2,{101,117,0}},
/* 11706 */ {(11<<2)|2,{101,117,0}},
/* 11707 */ {(11<<2)|2,{101,117,0}},
/* 11708 */ {(11<<2)|2,{101,117,0}},
/* 11709 */ {(11<<2)|2,{101,117,0}},
/* 11710 */ {(11<<2)|2,{101,117,0}},
/* 11711 */ {(11<<2)|2,{101,117,0}},
/* 11712 */ {(12<<2)|2,{101,58,0}},
/* 11713 */ {(12<<2)|2,{101,58,0}},
/* 11714 */ {(12<<2)|2,{101,58,0}},
/* 11715 */ {(12<<2)|2,{101,58,0}},
/* 11716 */ {(12<<2)|2,{101,58,0}},
/* 11717 */ {(12<<2)|2,{101,58,0}},
/* 11718 */ {(12<<2)|2,{101,58,0}},
/* 11719 */ {(12<<2)|2,{101,58,0}},
/* 11720 */ {(12<<2)|2,{101,58,0}},
/* 11721 */ {(12<<2)|2,{101,58,0}},
/* 11722 */ {(12<<2)|2,{101,58,0}},
/* 11723 */ {(12<<2)|2,{101,58,0}},
/* 11724 */ {(12<<2)|2,{101,58,0}},
/* 11725 */ {(12<<2)|2,{101,58,0}},
/* 11726 */ {(12<<2)|2,{101,58,0}},
/* 11727 */ {(12<<2)|2,{101,58,0}},
/* 11728 */ {(12<<2)|2,{101,66,0}},
/* 11729 */ {(12<<2)|2,{101,66,0}},
/* 11730 */ {(12<<2)|2,{101,66,0}},
/* 11731 */ {(12<<2)|2,{101,66,0}},
/* 11732 */ {(12<<2)|2,{101,66,0}},
/* 11733 */ {(12<<2)|2,{101,66,0}},
/* 11734 */ {(12<<2)|2,{101,66,0}},
/* 11735 */ {(12<<2)|2,{101,66,0}},
/* 11736 */ {(12<<2)|2,{101,66,0}},
/* 11737 */ {(12<<2)|2,{101,66,0}},
/* 11738 */ {(12<<2)|2,{101,66,0}},
/* 11739 */ {(12<<2)|2,{101,66,0}},
/* 11740 */ {(12<<2)|2,{101,66,0}},
/* 11741 */ {(12<<2)|2,{101,66,0}},
/* 11742 */ {(12<<2)|2,{101,66,0}},
/* 11743 */ {(12<<2)|2,{101,66,0}},
/* 11744 */ {(12<<2)|2,{101,67,0}},
/* 11745 */ {(12<<2)|2,{101,67,0}},
/* 11746 */ {(12<<2)|2,{101,67,0}},
/* 11747 */ {(12<<2)|2,{101,67,0}},
/* 11748 */ {(12<<2)|2,{101,67,0}},
/* 11749 */ {(12<<2)|2,{101,67,0}},
/* 11750 */ {(12<<2)|2,{101,67,0}},
/* 11751 */ {(12<<2)|2,{101,67,0}},
/* 11752 */ {(12<<2)|2,{101,67,0}},
/* 11753 */ {(12<<2)|2,{101,67,0}},
/* 11754 */ {(12<<2)|2,{101,67,0}},
/* 11755 */ {(12<<2)|2,{101,67,0}},
/* 11756 */ {(12<<2)|2,{101,67,0}},
/* 11757 */ {(12<<2)|2,{101,67,0}},
/* 11758 */ {(12<<2)|2,{101,67,0}},
/* 11759 */ {(12<<2)|2,{101,67,0}},
/* 11760 */ {(12<<2)|2,{101,68,0}},
/* 11761 */ {(12<<2)|2,{101,68,0}},
/* 11762 */ {(12<<2)|2,{101,68,0}},
/* 11763 */ {(12<<2)|2,{101,68,0}},
/* 11764 */ {(12<<2)|2,{101,68,0}},
/* 11765 */ {(12<<2)|2,{101,68,0}},
/* 11766 */ {(12<<2)|2,{101,68,0}},
/* 11767 */ {(12<<2)|2,{101,68,0}},
/* 11768 */ {(12<<2)|2,{101,68,0}},
/* 11769 */ {(12<<2)|2,{101,68,0}},
/* 11770 */ {(12<<2)|2,{101,68,0}},
/* 11771 */ {(12<<2)|2,{101,68,0}},
/* 11772 */ {(12<<2)|2,{101,68,0}},
/* 11773 */ {(12<<2)|2,{101,68,0}},
/* 11774 */ {(12<<2)|2,{101,68,0}},
/* 11775 */ {(12<<2)|2,{101,68,0}},
/* 11776 */ {(12<<2)|2,{101,69,0}},
/* 11777 */ {(12<<2)|2,{101,69,0}},
/* 11778 */ {(12<<2)|2,{101,69,0}},
/* 11779 */ {(12<<2)|2,{101,69,0}},
/* 11780 */ {(12<<2)|2,{101,69,0}},
/* 11781 */ {(12<<2)|2,{101,69,0}},
/* 11782 */ {(12<<2)|2,{101,69,0}},
/* 11783 */ {(12<<2)|2,{101,69,0}},
/* 11784 */ {(12<<2)|2,{101,69,0}},
/* 11785 */ {(12<<2)|2,{101,69,0}},
/* 11786 */ {(12<<2)|2,{101,69,0}},
/* 11787 */ {(12<<2)|2,{101,69,0}},
/* 11788 */ {(12<<2)|2,{101,69,0}},
/* 11789 */ {(12<<2)|2,{101,69,0}},
/* 11790 */ {(12<<2)|2,{101,69,0}},
/* 11791 */ {(12<<2)|2,{101,69,0}},
/* 11792 */ {(12<<2)|2,{101,70,0}},
/* 11793 */ {(12<<2)|2,{101,70,0}},
/* 11794 */ {(12<<2)|2,{101,70,0}},
/* 11795 */ {(12<<2)|2,{101,70,0}},
/* 11796 */ {(12<<2)|2,{101,70,0}},
/* 11797 */ {(12<<2)|2,{101,70,0}},
/* 11798 */ {(12<<2)|2,{101,70,0}},
/* 11799 */ {(12<<2)|2,{101,70,0}},
/* 11800 */ {(12<<2)|2,{101,70,0}},
/* 11801 */ {(12<<2)|2,{101,70,0}},
/* 11802 */ {(12<<2)|2,{101,70,0}},
/* 11803 */ {(12<<2)|2,{101,70,0}},
/* 11804 */ {(12<<2)|2,{101,70,0}},
/* 11805 */ {(12<<2)|2,{101,70,0}},
/* 11806 */ {(12<<2)|2,{101,70,0}},
/* 11807 */ {(12<<2)|2,{101,70,0}},
/* 11808 */ {(12<<2)|2,{101,71,0}},
/* 11809 */ {(12<<2)|2,{101,71,0}},
/* 11810 */ {(12<<2)|2,{101,71,0}},
/* 11811 */ {(12<<2)|2,{101,71,0}},
/* 11812 */ {(12<<2)|2,{101,71,0}},
/* 11813 */ {(12<<2)|2,{101,71,0}},
/* 11814 */ {(12<<2)|2,{101,71,0}},
/* 11815 */ {(12<<2)|2,{101,71,0}},
/* 11816 */ {(12<<2)|2,{101,71,0}},
/* 11817 */ {(12<<2)|2,{101,71,0}},
/* 11818 */ {(12<<2)|2,{101,71,0}},
/* 11819 */ {(12<<2)|2,{101,71,0}},
/* 11820 */ {(12<<2)|2,{101,71,0}},
/* 11821 */ {(12<<2)|2,{101,71,0}},
/* 11822 */ {(12<<2)|2,{101,71,0}},
/* 11823 */ {(12<<2)|2,{101,71,0}},
/* 11824 */ {(12<<2)|2,{101,72,0}},
/* 11825 */ {(12<<2)|2,{101,72,0}},
/* 11826 */ {(12<<2)|2,{101,72,0}},
/* 11827 */ {(12<<2)|2,{101,72,0}},
/* 11828 */ {(12<<2)|2,{101,72,0}},
/* 11829 */ {(12<<2)|2,{101,72,0}},
/* 11830 */ {(12<<2)|2,{101,72,0}},
/* 11831 */ {(12<<2)|2,{101,72,0}},
/* 11832 */ {(12<<2)|2,{101,72,0}},
/* 11833 */ {(12<<2)|2,{101,72,0}},
/* 11834 */ {(12<<2)|2,{101,72,0}},
/* 11835 */ {(12<<2)|2,{101,72,0}},
/* 11836 */ {(12<<2)|2,{101,72,0}},
/* 11837 */ {(12<<2)|2,{101,72,0}},
/* 11838 */ {(12<<2)|2,{101,72,0}},
/* 11839 */ {(12<<2)|2,{101,72,0}},
/* 11840 */ {(12<<2)|2,{101,73,0}},
/* 11841 */ {(12<<2)|2,{101,73,0}},
/* 11842 */ {(12<<2)|2,{101,73,0}},
/* 11843 */ {(12<<2)|2,{101,73,0}},
/* 11844 */ {(12<<2)|2,{101,73,0}},
/* 11845 */ {(12<<2)|2,{101,73,0}},
/* 11846 */ {(12<<2)|2,{101,73,0}},
/* 11847 */ {(12<<2)|2,{101,73,0}},
/* 11848 */ {(12<<2)|2,{101,73,0}},
/* 11849 */ {(12<<2)|2,{101,73,0}},
/* 11850 */ {(12<<2)|2,{101,73,0}},
/* 11851 */ {(12<<2)|2,{101,73,0}},
/* 11852 */ {(12<<2)|2,{101,73,0}},
/* 11853 */ {(12<<2)|2,{101,73,0}},
/* 11854 */ {(12<<2)|2,{101,73,0}},
/* 11855 */ {(12<<2)|2,{101,73,0}},
/* 11856 */ {(12<<2)|2,{101,74,0}},
/* 11857 */ {(12<<2)|2,{101,74,0}},
/* 11858 */ {(12<<2)|2,{101,74,0}},
/* 11859 */ {(12<<2)|2,{101,74,0}},
/* 11860 */ {(12<<2)|2,{101,74,0}},
/* 11861 */ {(12<<2)|2,{101,74,0}},
/* 11862 */ {(12<<2)|2,{101,74,0}},
/* 11863 */ {(12<<2)|2,{101,74,0}},
/* 11864 */ {(12<<2)|2,{101,74,0}},
/* 11865 */ {(12<<2)|2,{101,74,0}},
/* 11866 */ {(12<<2)|2,{101,74,0}},
/* 11867 */ {(12<<2)|2,{101,74,0}},
/* 11868 */ {(12<<2)|2,{101,74,0}},
/* 11869 */ {(12<<2)|2,{101,74,0}},
/* 11870 */ {(12<<2)|2,{101,74,0}},
/* 11871 */ {(12<<2)|2,{101,74,0}},
/* 11872 */ {(12<<2)|2,{101,75,0}},
/* 11873 */ {(12<<2)|2,{101,75,0}},
/* 11874 */ {(12<<2)|2,{101,75,0}},
/* 11875 */ {(12<<2)|2,{101,75,0}},
/* 11876 */ {(12<<2)|2,{101,75,0}},
/* 11877 */ {(12<<2)|2,{101,75,0}},
/* 11878 */ {(12<<2)|2,{101,75,0}},
/* 11879 */ {(12<<2)|2,{101,75,0}},
/* 11880 */ {(12<<2)|2,{101,75,0}},
/* 11881 */ {(12<<2)|2,{101,75,0}},
/* 11882 */ {(12<<2)|2,{101,75,0}},
/* 11883 */ {(12<<2)|2,{101,75,0}},
/* 11884 */ {(12<<2)|2,{101,75,0}},
/* 11885 */ {(12<<2)|2,{101,75,0}},
/* 11886 */ {(12<<2)|2,{101,75,0}},
/* 11887 */ {(12<<2)|2,{101,75,0}},
/* 11888 */ {(12<<2)|2,{101,76,0}},
/* 11889 */ {(12<<2)|2,{101,76,0}},
/* 11890 */ {(12<<2)|2,{101,76,0}},
/* 11891 */ {(12<<2)|2,{101,76,0}},
/* 11892 */ {(12<<2)|2,{101,76,0}},
/* 11893 */ {(12<<2)|2,{101,76,0}},
/* 11894 */ {(12<<2)|2,{101,76,0}},
/* 11895 */ {(12<<2)|2,{101,76,0}},
/* 11896 */ {(12<<2)|2,{101,76,0}},
/* 11897 */ {(12<<2)|2,{101,76,0}},
/* 11898 */ {(12<<2)|2,{101,76,0}},
/* 11899 */ {(12<<2)|2,{101,76,0}},
/* 11900 */ {(12<<2)|2,{101,76,0}},
/* 11901 */ {(12<<2)|2,{101,76,0}},
/* 11902 */ {(12<<2)|2,{101,76,0}},
/* 11903 */ {(12<<2)|2,{101,76,0}},
/* 11904 */ {(12<<2)|2,{101,77,0}},
/* 11905 */ {(12<<2)|2,{101,77,0}},
/* 11906 */ {(12<<2)|2,{101,77,0}},
/* 11907 */ {(12<<2)|2,{101,77,0}},
/* 11908 */ {(12<<2)|2,{101,77,0}},
/* 11909 */ {(12<<2)|2,{101,77,0}},
/* 11910 */ {(12<<2)|2,{101,77,0}},
/* 11911 */ {(12<<2)|2,{101,77,0}},
/* 11912 */ {(12<<2)|2,{101,77,0}},
/* 11913 */ {(12<<2)|2,{101,77,0}},
/* 11914 */ {(12<<2)|2,{101,77,0}},
/* 11915 */ {(12<<2)|2,{101,77,0}},
/* 11916 */ {(12<<2)|2,{101,77,0}},
/* 11917 */ {(12<<2)|2,{101,77,0}},
/* 11918 */ {(12<<2)|2,{101,77,0}},
/* 11919 */ {(12<<2)|2,{101,77,0}},
/* 11920 */ {(12<<2)|2,{101,78,0}},
/* 11921 */ {(12<<2)|2,{101,78,0}},
/* 11922 */ {(12<<2)|2,{101,78,0}},
/* 11923 */ {(12<<2)|2,{101,78,0}},
/* 11924 */ {(12<<2)|2,{101,78,0}},
/* 11925 */ {(12<<2)|2,{101,78,0}},
/* 11926 */ {(12<<2)|2,{101,78,0}},
/* 11927 */ {(12<<2)|2,{101,78,0}},
/* 11928 */ {(12<<2)|2,{101,78,0}},
/* 11929 */ {(12<<2)|2,{101,78,0}},
/* 11930 */ {(12<<2)|2,{101,78,0}},
/* 11931 */ {(12<<2)|2,{101,78,0}},
/* 11932 */ {(12<<2)|2,{101,78,0}},
/* 11933 */ {(12<<2)|2,{101,78,0}},
/* 11934 */ {(12<<2)|2,{101,78,0}},
/* 11935 */ {(12<<2)|2,{101,78,0}},
/* 11936 */ {(12<<2)|2,{101,79,0}},
/* 11937 */ {(12<<2)|2,{101,79,0}},
/* 11938 */ {(12<<2)|2,{101,79,0}},
/* 11939 */ {(12<<2)|2,{101,79,0}},
/* 11940 */ {(12<<2)|2,{101,79,0}},
/* 11941 */ {(12<<2)|2,{101,79,0}},
/* 11942 */ {(12<<2)|2,{101,79,0}},
/* 11943 */ {(12<<2)|2,{101,79,0}},
/* 11944 */ {(12<<2)|2,{101,79,0}},
/* 11945 */ {(12<<2)|2,{101,79,0}},
/* 11946 */ {(12<<2)|2,{101,79,0}},
/* 11947 */ {(12<<2)|2,{101,79,0}},
/* 11948 */ {(12<<2)|2,{101,79,0}},
/* 11949 */ {(12<<2)|2,{101,79,0}},
/* 11950 */ {(12<<2)|2,{101,79,0}},
/* 11951 */ {(12<<2)|2,{101,79,0}},
/* 11952 */ {(12<<2)|2,{101,80,0}},
/* 11953 */ {(12<<2)|2,{101,80,0}},
/* 11954 */ {(12<<2)|2,{101,80,0}},
/* 11955 */ {(12<<2)|2,{101,80,0}},
/* 11956 */ {(12<<2)|2,{101,80,0}},
/* 11957 */ {(12<<2)|2,{101,80,0}},
/* 11958 */ {(12<<2)|2,{101,80,0}},
/* 11959 */ {(12<<2)|2,{101,80,0}},
/* 11960 */ {(12<<2)|2,{101,80,0}},
/* 11961 */ {(12<<2)|2,{101,80,0}},
/* 11962 */ {(12<<2)|2,{101,80,0}},
/* 11963 */ {(12<<2)|2,{101,80,0}},
/* 11964 */ {(12<<2)|2,{101,80,0}},
/* 11965 */ {(12<<2)|2,{101,80,0}},
/* 11966 */ {(12<<2)|2,{101,80,0}},
/* 11967 */ {(12<<2)|2,{101,80,0}},
/* 11968 */ {(12<<2)|2,{101,81,0}},
/* 11969 */ {(12<<2)|2,{101,81,0}},
/* 11970 */ {(12<<2)|2,{101,81,0}},
/* 11971 */ {(12<<2)|2,{101,81,0}},
/* 11972 */ {(12<<2)|2,{101,81,0}},
/* 11973 */ {(12<<2)|2,{101,81,0}},
/* 11974 */ {(12<<2)|2,{101,81,0}},
/* 11975 */ {(12<<2)|2,{101,81,0}},
/* 11976 */ {(12<<2)|2,{101,81,0}},
/* 11977 */ {(12<<2)|2,{101,81,0}},
/* 11978 */ {(12<<2)|2,{101,81,0}},
/* 11979 */ {(12<<2)|2,{101,81,0}},
/* 11980 */ {(12<<2)|2,{101,81,0}},
/* 11981 */ {(12<<2)|2,{101,81,0}},
/* 11982 */ {(12<<2)|2,{101,81,0}},
/* 11983 */ {(12<<2)|2,{101,81,0}},
/* 11984 */ {(12<<2)|2,{101,82,0}},
/* 11985 */ {(12<<2)|2,{101,82,0}},
/* 11986 */ {(12<<2)|2,{101,82,0}},
/* 11987 */ {(12<<2)|2,{101,82,0}},
/* 11988 */ {(12<<2)|2,{101,82,0}},
/* 11989 */ {(12<<2)|2,{101,82,0}},
/* 11990 */ {(12<<2)|2,{101,82,0}},
/* 11991 */ {(12<<2)|2,{101,82,0}},
/* 11992 */ {(12<<2)|2,{101,82,0}},
/* 11993 */ {(12<<2)|2,{101,82,0}},
/* 11994 */ {(12<<2)|2,{101,82,0}},
/* 11995 */ {(12<<2)|2,{101,82,0}},
/* 11996 */ {(12<<2)|2,{101,82,0}},
/* 11997 */ {(12<<2)|2,{101,82,0}},
/* 11998 */ {(12<<2)|2,{101,82,0}},
/* 11999 */ {(12<<2)|2,{101,82,0}},
/* 12000 */ {(12<<2)|2,{101,83,0}},
/* 12001 */ {(12<<2)|2,{101,83,0}},
/* 12002 */ {(12<<2)|2,{101,83,0}},
/* 12003 */ {(12<<2)|2,{101,83,0}},
/* 12004 */ {(12<<2)|2,{101,83,0}},
/* 12005 */ {(12<<2)|2,{101,83,0}},
/* 12006 */ {(12<<2)|2,{101,83,0}},
/* 12007 */ {(12<<2)|2,{101,83,0}},
/* 12008 */ {(12<<2)|2,{101,83,0}},
/* 12009 */ {(12<<2)|2,{101,83,0}},
/* 12010 */ {(12<<2)|2,{101,83,0}},
/* 12011 */ {(12<<2)|2,{101,83,0}},
/* 12012 */ {(12<<2)|2,{101,83,0}},
/* 12013 */ {(12<<2)|2,{101,83,0}},
/* 12014 */ {(12<<2)|2,{101,83,0}},
/* 12015 */ {(12<<2)|2,{101,83,0}},
/* 12016 */ {(12<<2)|2,{101,84,0}},
/* 12017 */ {(12<<2)|2,{101,84,0}},
/* 12018 */ {(12<<2)|2,{101,84,0}},
/* 12019 */ {(12<<2)|2,{101,84,0}},
/* 12020 */ {(12<<2)|2,{101,84,0}},
/* 12021 */ {(12<<2)|2,{101,84,0}},
/* 12022 */ {(12<<2)|2,{101,84,0}},
/* 12023 */ {(12<<2)|2,{101,84,0}},
/* 12024 */ {(12<<2)|2,{101,84,0}},
/* 12025 */ {(12<<2)|2,{101,84,0}},
/* 12026 */ {(12<<2)|2,{101,84,0}},
/* 12027 */ {(12<<2)|2,{101,84,0}},
/* 12028 */ {(12<<2)|2,{101,84,0}},
/* 12029 */ {(12<<2)|2,{101,84,0}},
/* 12030 */ {(12<<2)|2,{101,84,0}},
/* 12031 */ {(12<<2)|2,{101,84,0}},
/* 12032 */ {(12<<2)|2,{101,85,0}},
/* 12033 */ {(12<<2)|2,{101,85,0}},
/* 12034 */ {(12<<2)|2,{101,85,0}},
/* 12035 */ {(12<<2)|2,{101,85,0}},
/* 12036 */ {(12<<2)|2,{101,85,0}},
/* 12037 */ {(12<<2)|2,{101,85,0}},
/* 12038 */ {(12<<2)|2,{101,85,0}},
/* 12039 */ {(12<<2)|2,{101,85,0}},
/* 12040 */ {(12<<2)|2,{101,85,0}},
/* 12041 */ {(12<<2)|2,{101,85,0}},
/* 12042 */ {(12<<2)|2,{101,85,0}},
/* 12043 */ {(12<<2)|2,{101,85,0}},
/* 12044 */ {(12<<2)|2,{101,85,0}},
/* 12045 */ {(12<<2)|2,{101,85,0}},
/* 12046 */ {(12<<2)|2,{101,85,0}},
/* 12047 */ {(12<<2)|2,{101,85,0}},
/* 12048 */ {(12<<2)|2,{101,86,0}},
/* 12049 */ {(12<<2)|2,{101,86,0}},
/* 12050 */ {(12<<2)|2,{101,86,0}},
/* 12051 */ {(12<<2)|2,{101,86,0}},
/* 12052 */ {(12<<2)|2,{101,86,0}},
/* 12053 */ {(12<<2)|2,{101,86,0}},
/* 12054 */ {(12<<2)|2,{101,86,0}},
/* 12055 */ {(12<<2)|2,{101,86,0}},
/* 12056 */ {(12<<2)|2,{101,86,0}},
/* 12057 */ {(12<<2)|2,{101,86,0}},
/* 12058 */ {(12<<2)|2,{101,86,0}},
/* 12059 */ {(12<<2)|2,{101,86,0}},
/* 12060 */ {(12<<2)|2,{101,86,0}},
/* 12061 */ {(12<<2)|2,{101,86,0}},
/* 12062 */ {(12<<2)|2,{101,86,0}},
/* 12063 */ {(12<<2)|2,{101,86,0}},
/* 12064 */ {(12<<2)|2,{101,87,0}},
/* 12065 */ {(12<<2)|2,{101,87,0}},
/* 12066 */ {(12<<2)|2,{101,87,0}},
/* 12067 */ {(12<<2)|2,{101,87,0}},
/* 12068 */ {(12<<2)|2,{101,87,0}},
/* 12069 */ {(12<<2)|2,{101,87,0}},
/* 12070 */ {(12<<2)|2,{101,87,0}},
/* 12071 */ {(12<<2)|2,{101,87,0}},
/* 12072 */ {(12<<2)|2,{101,87,0}},
/* 12073 */ {(12<<2)|2,{101,87,0}},
/* 12074 */ {(12<<2)|2,{101,87,0}},
/* 12075 */ {(12<<2)|2,{101,87,0}},
/* 12076 */ {(12<<2)|2,{101,87,0}},
/* 12077 */ {(12<<2)|2,{101,87,0}},
/* 12078 */ {(12<<2)|2,{101,87,0}},
/* 12079 */ {(12<<2)|2,{101,87,0}},
/* 12080 */ {(12<<2)|2,{101,89,0}},
/* 12081 */ {(12<<2)|2,{101,89,0}},
/* 12082 */ {(12<<2)|2,{101,89,0}},
/* 12083 */ {(12<<2)|2,{101,89,0}},
/* 12084 */ {(12<<2)|2,{101,89,0}},
/* 12085 */ {(12<<2)|2,{101,89,0}},
/* 12086 */ {(12<<2)|2,{101,89,0}},
/* 12087 */ {(12<<2)|2,{101,89,0}},
/* 12088 */ {(12<<2)|2,{101,89,0}},
/* 12089 */ {(12<<2)|2,{101,89,0}},
/* 12090 */ {(12<<2)|2,{101,89,0}},
/* 12091 */ {(12<<2)|2,{101,89,0}},
/* 12092 */ {(12<<2)|2,{101,89,0}},
/* 12093 */ {(12<<2)|2,{101,89,0}},
/* 12094 */ {(12<<2)|2,{101,89,0}},
/* 12095 */ {(12<<2)|2,{101,89,0}},
/* 12096 */ {(12<<2)|2,{101,106,0}},
/* 12097 */ {(12<<2)|2,{101,106,0}},
/* 12098 */ {(12<<2)|2,{101,106,0}},
/* 12099 */ {(12<<2)|2,{101,106,0}},
/* 12100 */ {(12<<2)|2,{101,106,0}},
/* 12101 */ {(12<<2)|2,{101,106,0}},
/* 12102 */ {(12<<2)|2,{101,106,0}},
/* 12103 */ {(12<<2)|2,{101,106,0}},
/* 12104 */ {(12<<2)|2,{101,106,0}},
/* 12105 */ {(12<<2)|2,{101,106,0}},
/* 12106 */ {(12<<2)|2,{101,106,0}},
/* 12107 */ {(12<<2)|2,{101,106,0}},
/* 12108 */ {(12<<2)|2,{101,106,0}},
/* 12109 */ {(12<<2)|2,{101,106,0}},
/* 12110 */ {(12<<2)|2,{101,106,0}},
/* 12111 */ {(12<<2)|2,{101,106,0}},
/* 12112 */ {(12<<2)|2,{101,107,0}},
/* 12113 */ {(12<<2)|2,{101,107,0}},
/* 12114 */ {(12<<2)|2,{101,107,0}},
/* 12115 */ {(12<<2)|2,{101,107,0}},
/* 12116 */ {(12<<2)|2,{101,107,0}},
/* 12117 */ {(12<<2)|2,{101,107,0}},
/* 12118 */ {(12<<2)|2,{101,107,0}},
/* 12119 */ {(12<<2)|2,{101,107,0}},
/* 12120 */ {(12<<2)|2,{101,107,0}},
/* 12121 */ {(12<<2)|2,{101,107,0}},
/* 12122 */ {(12<<2)|2,{101,107,0}},
/* 12123 */ {(12<<2)|2,{101,107,0}},
/* 12124 */ {(12<<2)|2,{101,107,0}},
/* 12125 */ {(12<<2)|2,{101,107,0}},
/* 12126 */ {(12<<2)|2,{101,107,0}},
/* 12127 */ {(12<<2)|2,{101,107,0}},
/* 12128 */ {(12<<2)|2,{101,113,0}},
/* 12129 */ {(12<<2)|2,{101,113,0}},
/* 12130 */ {(12<<2)|2,{101,113,0}},
/* 12131 */ {(12<<2)|2,{101,113,0}},
/* 12132 */ {(12<<2)|2,{101,113,0}},
/* 12133 */ {(12<<2)|2,{101,113,0}},
/* 12134 */ {(12<<2)|2,{101,113,0}},
/* 12135 */ {(12<<2)|2,{101,113,0}},
/* 12136 */ {(12<<2)|2,{101,113,0}},
/* 12137 */ {(12<<2)|2,{101,113,0}},
/* 12138 */ {(12<<2)|2,{101,113,0}},
/* 12139 */ {(12<<2)|2,{101,113,0}},
/* 12140 */ {(12<<2)|2,{101,113,0}},
/* 12141 */ {(12<<2)|2,{101,113,0}},
/* 12142 */ {(12<<2)|2,{101,113,0}},
/* 12143 */ {(12<<2)|2,{101,113,0}},
/* 12144 */ {(12<<2)|2,{101,118,0}},
/* 12145 */ {(12<<2)|2,{101,118,0}},
/* 12146 */ {(12<<2)|2,{101,118,0}},
/* 12147 */ {(12<<2)|2,{101,118,0}},
/* 12148 */ {(12<<2)|2,{101,118,0}},
/* 12149 */ {(12<<2)|2,{101,118,0}},
/* 12150 */ {(12<<2)|2,{101,118,0}},
/* 12151 */ {(12<<2)|2,{101,118,0}},
/* 12152 */ {(12<<2)|2,{101,118,0}},
/* 12153 */ {(12<<2)|2,{101,118,0}},
/* 12154 */ {(12<<2)|2,{101,118,0}},
/* 12155 */ {(12<<2)|2,{101,118,0}},
/* 12156 */ {(12<<2)|2,{101,118,0}},
/* 12157 */ {(12<<2)|2,{101,118,0}},
/* 12158 */ {(12<<2)|2,{101,118,0}},
/* 12159 */ {(12<<2)|2,{101,118,0}},
/* 12160 */ {(12<<2)|2,{101,119,0}},
/* 12161 */ {(12<<2)|2,{101,119,0}},
/* 12162 */ {(12<<2)|2,{101,119,0}},
/* 12163 */ {(12<<2)|2,{101,119,0}},
/* 12164 */ {(12<<2)|2,{101,119,0}},
/* 12165 */ {(12<<2)|2,{101,119,0}},
/* 12166 */ {(12<<2)|2,{101,119,0}},
/* 12167 */ {(12<<2)|2,{101,119,0}},
/* 12168 */ {(12<<2)|2,{101,119,0}},
/* 12169 */ {(12<<2)|2,{101,119,0}},
/* 12170 */ {(12<<2)|2,{101,119,0}},
/* 12171 */ {(12<<2)|2,{101,119,0}},
/* 12172 */ {(12<<2)|2,{101,119,0}},
/* 12173 */ {(12<<2)|2,{101,119,0}},
/* 12174 */ {(12<<2)|2,{101,119,0}},
/* 12175 */ {(12<<2)|2,{101,119,0}},
/* 12176 */ {(12<<2)|2,{101,120,0}},
/* 12177 */ {(12<<2)|2,{101,120,0}},
/* 12178 */ {(12<<2)|2,{101,120,0}},
/* 12179 */ {(12<<2)|2,{101,120,0}},
/* 12180 */ {(12<<2)|2,{101,120,0}},
/* 12181 */ {(12<<2)|2,{101,120,0}},
/* 12182 */ {(12<<2)|2,{101,120,0}},
/* 12183 */ {(12<<2)|2,{101,120,0}},
/* 12184 */ {(12<<2)|2,{101,120,0}},
/* 12185 */ {(12<<2)|2,{101,120,0}},
/* 12186 */ {(12<<2)|2,{101,120,0}},
/* 12187 */ {(12<<2)|2,{101,120,0}},
/* 12188 */ {(12<<2)|2,{101,120,0}},
/* 12189 */ {(12<<2)|2,{101,120,0}},
/* 12190 */ {(12<<2)|2,{101,120,0}},
/* 12191 */ {(12<<2)|2,{101,120,0}},
/* 12192 */ {(12<<2)|2,{101,121,0}},
/* 12193 */ {(12<<2)|2,{101,121,0}},
/* 12194 */ {(12<<2)|2,{101,121,0}},
/* 12195 */ {(12<<2)|2,{101,121,0}},
/* 12196 */ {(12<<2)|2,{101,121,0}},
/* 12197 */ {(12<<2)|2,{101,121,0}},
/* 12198 */ {(12<<2)|2,{101,121,0}},
/* 12199 */ {(12<<2)|2,{101,121,0}},
/* 12200 */ {(12<<2)|2,{101,121,0}},
/* 12201 */ {(12<<2)|2,{101,121,0}},
/* 12202 */ {(12<<2)|2,{101,121,0}},
/* 12203 */ {(12<<2)|2,{101,121,0}},
/* 12204 */ {(12<<2)|2,{101,121,0}},
/* 12205 */ {(12<<2)|2,{101,121,0}},
/* 12206 */ {(12<<2)|2,{101,121,0}},
/* 12207 */ {(12<<2)|2,{101,121,0}},
/* 12208 */ {(12<<2)|2,{101,122,0}},
/* 12209 */ {(12<<2)|2,{101,122,0}},
/* 12210 */ {(12<<2)|2,{101,122,0}},
/* 12211 */ {(12<<2)|2,{101,122,0}},
/* 12212 */ {(12<<2)|2,{101,122,0}},
/* 12213 */ {(12<<2)|2,{101,122,0}},
/* 12214 */ {(12<<2)|2,{101,122,0}},
/* 12215 */ {(12<<2)|2,{101,122,0}},
/* 12216 */ {(12<<2)|2,{101,122,0}},
/* 12217 */ {(12<<2)|2,{101,122,0}},
/* 12218 */ {(12<<2)|2,{101,122,0}},
/* 12219 */ {(12<<2)|2,{101,122,0}},
/* 12220 */ {(12<<2)|2,{101,122,0}},
/* 12221 */ {(12<<2)|2,{101,122,0}},
/* 12222 */ {(12<<2)|2,{101,122,0}},
/* 12223 */ {(12<<2)|2,{101,122,0}},
/* 12224 */ {(13<<2)|2,{101,38,0}},
/* 12225 */ {(13<<2)|2,{101,38,0}},
/* 12226 */ {(13<<2)|2,{101,38,0}},
/* 12227 */ {(13<<2)|2,{101,38,0}},
/* 12228 */ {(13<<2)|2,{101,38,0}},
/* 12229 */ {(13<<2)|2,{101,38,0}},
/* 12230 */ {(13<<2)|2,{101,38,0}},
/* 12231 */ {(13<<2)|2,{101,38,0}},
/* 12232 */ {(13<<2)|2,{101,42,0}},
/* 12233 */ {(13<<2)|2,{101,42,0}},
/* 12234 */ {(13<<2)|2,{101,42,0}},
/* 12235 */ {(13<<2)|2,{101,42,0}},
/* 12236 */ {(13<<2)|2,{101,42,0}},
/* 12237 */ {(13<<2)|2,{101,42,0}},
/* 12238 */ {(13<<2)|2,{101,42,0}},
/* 12239 */ {(13<<2)|2,{101,42,0}},
/* 12240 */ {(13<<2)|2,{101,44,0}},
/* 12241 */ {(13<<2)|2,{101,44,0}},
/* 12242 */ {(13<<2)|2,{101,44,0}},
/* 12243 */ {(13<<2)|2,{101,44,0}},
/* 12244 */ {(13<<2)|2,{101,44,0}},
/* 12245 */ {(13<<2)|2,{101,44,0}},
/* 12246 */ {(13<<2)|2,{101,44,0}},
/* 12247 */ {(13<<2)|2,{101,44,0}},
/* 12248 */ {(13<<2)|2,{101,59,0}},
/* 12249 */ {(13<<2)|2,{101,59,0}},
/* 12250 */ {(13<<2)|2,{101,59,0}},
/* 12251 */ {(13<<2)|2,{101,59,0}},
/* 12252 */ {(13<<2)|2,{101,59,0}},
/* 12253 */ {(13<<2)|2,{101,59,0}},
/* 12254 */ {(13<<2)|2,{101,59,0}},
/* 12255 */ {(13<<2)|2,{101,59,0}},
/* 12256 */ {(13<<2)|2,{101,88,0}},
/* 12257 */ {(13<<2)|2,{101,88,0}},
/* 12258 */ {(13<<2)|2,{101,88,0}},
/* 12259 */ {(13<<2)|2,{101,88,0}},
/* 12260 */ {(13<<2)|2,{101,88,0}},
/* 12261 */ {(13<<2)|2,{101,88,0}},
/* 12262 */ {(13<<2)|2,{101,88,0}},
/* 12263 */ {(13<<2)|2,{101,88,0}},
/* 12264 */ {(13<<2)|2,{101,90,0}},
/* 12265 */ {(13<<2)|2,{101,90,0}},
/* 12266 */ {(13<<2)|2,{101,90,0}},
/* 12267 */ {(13<<2)|2,{101,90,0}},
/* 12268 */ {(13<<2)|2,{101,90,0}},
/* 12269 */ {(13<<2)|2,{101,90,0}},
/* 12270 */ {(13<<2)|2,{101,90,0}},
/* 12271 */ {(13<<2)|2,{101,90,0}},
/* 12272 */ {(15<<2)|2,{101,33,0}},
/* 12273 */ {(15<<2)|2,{101,33,0}},
/* 12274 */ {(15<<2)|2,{101,34,0}},
/* 12275 */ {(15<<2)|2,{101,34,0}},
/* 12276 */ {(15<<2)|2,{101,40,0}},
/* 12277 */ {(15<<2)|2,{101,40,0}},
/* 12278 */ {(15<<2)|2,{101,41,0}},
/* 12279 */ {(15<<2)|2,{101,41,0}},
/* 12280 */ {(15<<2)|2,{101,63,0}},
/* 12281 */ {(15<<2)|2,{101,63,0}},
/* 12282 */ {(16<<2)|2,{101,39,0}},
/* 12283 */ {(16<<2)|2,{101,43,0}},
/* 12284 */ {(16<<2)|2,{101,124,0}},
/* 12285 */ {(5<<2)|1,{101,0,0}},
/* 12286 */ {(5<<2)|1,{101,0,0}},
/* 12287 */ {(5<<2)|1,{101,0,0}},
/* 12288 */ {(15<<2)|3,{105,48,48}},
/* 12289 */ {(15<<2)|3,{105,48,48}},
/* 12290 */ {(15<<2)|3,{105,48,49}},
/* 12291 */ {(15<<2)|3,{105,48,49}},
/* 12292 */ {(15<<2)|3,{105,48,50}},
/* 12293 */ {(15<<2)|3,{105,48,50}},
/* 12294 */ {(15<<2)|3,{105,48,97}},
/* 12295 */ {(15<<2)|3,{105,48,97}},
/* 12296 */ {(15<<2)|3,{105,48,99}},
/* 12297 */ {(15<<2)|3,{105,48,99}},
/* 12298 */ {(15<<2)|3,{105,48,101}},
/* 12299 */ {(15<<2)|3,{105,48,101}},
/* 12300 */ {(15<<2)|3,{105,48,105}},
/* 12301 */ {(15<<2)|3,{105,48,105}},
/* 12302 */ {(15<<2)|3,{105,48,111}},
/* 12303 */ {(15<<2)|3,{105,48,111}},
/* 12304 */ {(15<<2)|3,{105,48,115}},
/* 12305 */ {(15<<2)|3,{105,48,115}},
/* 12306 */ {(15<<2)|3,{105,48,116}},
/* 12307 */ {(15<<2)|3,{105,48,116}},
/* 12308 */ {(16<<2)|3,{105,48,32}},
/* 12309 */ {(16<<2)|3,{105,48,37}},
/* 12310 */ {(16<<2)|3,{105,48,45}},
/* 12311 */ {(16<<2)|3,{105,48,46}},
/* 12312 */ {(16<<2)|3,{105,48,47}},
/* 12313 */ {(16<<2)|3,{105,48,51}},
/* 12314 */ {(16<<2)|3,{105,48,52}},
/* 12315 */ {(16<<2)|3,{105,48,53}},
/* 12316 */ {(16<<2)|3,{105,48,54}},
/* 12317 */ {(16<<2)|3,{105,48,55}},
/* 12318 */ {(16<<2)|3,{105,48,56}},
/* 12319 */ {(16<<2)|3,{105,48,57}},
/* 12320 */ {(16<<2)|3,{105,48,61}},
/* 12321 */ {(16<<2)|3,{105,48,65}},
/* 12322 */ {(16<<2)|3,{105,48,95}},
/* 12323 */ {(16<<2)|3,{105,48,98}},
/* 12324 */ {(16<<2)|3,{105,48,100}},
/* 12325 */ {(16<<2)|3,{105,48,102}},
/* 12326 */ {(16<<2)|3,{105,48,103}},
/* 12327 */ {(16<<2)|3,{105,48,104}},
/* 12328 */ {(16<<2)|3,{105,48,108}},
/* 12329 */ {(16<<2)|3,{105,48,109}},
/* 12330 */ {(16<<2)|3,{105,48,110}},
/* 12331 */ {(16<<2)|3,{105,48,112}},
/* 12332 */ {(16<<2)|3,{105,48,114}},
/* 12333 */ {(16<<2)|3,{105,48,117}},
/* 12334 */ {(10<<2)|2,{105,48,0}},
/* 12335 */ {(10<<2)|2,{105,48,0}},
/* 12336 */ {(10<<2)|2,{105,48,0}},
/* 12337 */ {(10<<2)|2,{105,48,0}},
/* 12338 */ {(10<<2)|2,{105,48,0}},
/* 12339 */ {(10<<2)|2,{105,48,0}},
/* 12340 */ {(10<<2)|2,{105,48,0}},
/* 12341 */ {(10<<2)|2,{105,48,0}},
/* 12342 */ {(10<<2)|2,{105,48,0}},
/* 12343 */ {(10<<2)|2,{105,48,0}},
/* 12344 */ {(10<<2)|2,{105,48,0}},
/* 12345 */ {(10<<2)|2,{105,48,0}},
/* 12346 */ {(10<<2)|2,{105,48,0}},
/* 12347 */ {(10<<2)|2,{105,48,0}},
/* 12348 */ {(10<<2)|2,{105,48,0}},
/* 12349 */ {(10<<2)|2,{105,48,0}},
/* 12350 */ {(10<<2)|2,{105,48,0}},
/* 12351 */ {(10<<2)|2,{105,48,0}},
/* 12352 */ {(15<<2)|3,{105,49,48}},
/* 12353 */ {(15<<2)|3,{105,49,48}},
/* 12354 */ {(15<<2)|3,{105,49,49}},
/* 12355 */ {(15<<2)|3,{105,49,49}},
/* 12356 */ {(15<<2)|3,{105,49,50}},
/* 12357 */ {(15<<2)|3,{105,49,50}},
/* 12358 */ {(15<<2)|3,{105,49,97}},
/* 12359 */ {(15<<2)|3,{105,49,97}},
/* 12360 */ {(15<<2)|3,{105,49,99}},
/* 12361 */ {(15<<2)|3,{105,49,99}},
/* 12362 */ {(15<<2)|3,{105,49,101}},
/* 12363 */ {(15<<2)|3,{105,49,101}},
/* 12364 */ {(15<<2)|3,{105,49,105}},
/* 12365 */ {(15<<2)|3,{105,49,105}},
/* 12366 */ {(15<<2)|3,{105,49,111}},
/* 12367 */ {(15<<2)|3,{105,49,111}},
/* 12368 */ {(15<<2)|3,{105,49,115}},
/* 12369 */ {(15<<2)|3,{105,49,115}},
/* 12370 */ {(15<<2)|3,{105,49,116}},
/* 12371 */ {(15<<2)|3,{105,49,116}},
/* 12372 */ {(16<<2)|3,{105,49,32}},
/* 12373 */ {(16<<2)|3,{105,49,37}},
/* 12374 */ {(16<<2)|3,{105,49,45}},
/* 12375 */ {(16<<2)|3,{105,49,46}},
/* 12376 */ {(16<<2)|3,{105,49,47}},
/* 12377 */ {(16<<2)|3,{105,49,51}},
/* 12378 */ {(16<<2)|3,{105,49,52}},
/* 12379 */ {(16<<2)|3,{105,49,53}},
/* 12380 */ {(16<<2)|3,{105,49,54}},
/* 12381 */ {(16<<2)|3,{105,49,55}},
/* 12382 */ {(16<<2)|3,{105,49,56}},
/* 12383 */ {(16<<2)|3,{105,49,57}},
/* 12384 */ {(16<<2)|3,{105,49,61}},
/* 12385 */ {(16<<2)|3,{105,49,65}},
/* 12386 */ {(16<<2)|3,{105,49,95}},
/* 12387 */ {(16<<2)|3,{105,49,98}},
/* 12388 */ {(16<<2)|3,{105,49,100}},
/* 12389 */ {(16<<2)|3,{105,49,102}},
/* 12390 */ {(16<<2)|3,{105,49,103}},
/* 12391 */ {(16<<2)|3,{105,49,104}},
/* 12392 */ {(16<<2)|3,{105,49,108}},
/* 12393 */ {(16<<2)|3,{105,49,109}},
/* 12394 */ {(16<<2)|3,{105,49,110}},
/* 12395 */ {(16<<2)|3,{105,49,112}},
/* 12396 */ {(16<<2)|3,{105,49,114}},
/* 12397 */ {(16<<2)|3,{105,49,117}},
/* 12398 */ {(10<<2)|2,{105,49,0}},
/* 12399 */ {(10<<2)|2,{105,49,0}},
/* 12400 */ {(10<<2)|2,{105,49,0}},
/* 12401 */ {(10<<2)|2,{105,49,0}},
/* 12402 */ {(10<<2)|2,{105,49,0}},
/* 12403 */ {(10<<2)|2,{105,49,0}},
/* 12404 */ {(10<<2)|2,{105,49,0}},
/* 12405 */ {(10<<2)|2,{105,49,0}},
/* 12406 */ {(10<<2)|2,{105,49,0}},
/* 12407 */ {(10<<2)|2,{105,49,0}},
/* 12408 */ {(10<<2)|2,{105,49,0}},
/* 12409 */ {(10<<2)|2,{105,49,0}},
/* 12410 */ {(10<<2)|2,{105,49,0}},
/* 12411 */ {(10<<2)|2,{105,49,0}},
/* 12412 */ {(10<<2)|2,{105,49,0}},
/* 12413 */ {(10<<2)|2,{105,49,0}},
/* 12414 */ {(10<<2)|2,{105,49,0}},
/* 12415 */ {(10<<2)|2,{105,49,0}},
/* 12416 */ {(15<<2)|3,{105,50,48}},
/* 12417 */ {(15<<2)|3,{105,50,48}},
/* 12418 */ {(15<<2)|3,{105,50,49}},
/* 12419 */ {(15<<2)|3,{105,50,49}},
/* 12420 */ {(15<<2)|3,{105,50,50}},
/* 12421 */ {(15<<2)|3,{105,50,50}},
/* 12422 */ {(15<<2)|3,{105,50,97}},
/* 12423 */ {(15<<2)|3,{105,50,97}},
/* 12424 */ {(15<<2)|3,{105,50,99}},
/* 12425 */ {(15<<2)|3,{105,50,99}},
/* 12426 */ {(15<<2)|3,{105,50,101}},
/* 12427 */ {(15<<2)|3,{105,50,101}},
/* 12428 */ {(15<<2)|3,{105,50,105}},
/* 12429 */ {(15<<2)|3,{105,50,105}},
/* 12430 */ {(15<<2)|3,{105,50,111}},
/* 12431 */ {(15<<2)|3,{105,50,111}},
/* 12432 */ {(15<<2)|3,{105,50,115}},
/* 12433 */ {(15<<2)|3,{105,50,115}},
/* 12434 */ {(15<<2)|3,{105,50,116}},
/* 12435 */ {(15<<2)|3,{105,50,116}},
/* 12436 */ {(16<<2)|3,{105,50,32}},
/* 12437 */ {(16<<2)|3,{105,50,37}},
/* 12438 */ {(16<<2)|3,{105,50,45}},
/* 12439 */ {(16<<2)|3,{105,50,46}},
/* 12440 */ {(16<<2)|3,{105,50,47}},
/* 12441 */ {(16<<2)|3,{105,50,51}},
/* 12442 */ {(16<<2)|3,{105,50,52}},
/* 12443 */ {(16<<2)|3,{105,50,53}},
/* 12444 */ {(16<<2)|3,{105,50,54}},
/* 12445 */ {(16<<2)|3,{105,50,55}},
/* 12446 */ {(16<<2)|3,{105,50,56}},
/* 12447 */ {(16<<2)|3,{105,50,57}},
/* 12448 */ {(16<<2)|3,{105,50,61}},
/* 12449 */ {(16<<2)|3,{105,50,65}},
/* 12450 */ {(16<<2)|3,{105,50,95}},
/* 12451 */ {(16<<2)|3,{105,50,98}},
/* 12452 */ {(16<<2)|3,{105,50,100}},
/* 12453 */ {(16<<2)|3,{105,50,102}},
/* 12454 */ {(16<<2)|3,{105,50,103}},
/* 12455 */ {(16<<2)|3,{105,50,104}},
/* 12456 */ {(16<<2)|3,{105,50,108}},
/* 12457 */ {(16<<2)|3,{105,50,109}},
/* 12458 */ {(16<<2)|3,{105,50,110}},
/* 12459 */ {(16<<2)|3,{105,50,112}},
/* 12460 */ {(16<<2)|3,{105,50,114}},
/* 12461 */ {(16<<2)|3,{105,50,117}},
/* 12462 */ {(10<<2)|2,{105,50,0}},
/* 12463 */ {(10<<2)|2,{105,50,0}},
/* 12464 */ {(10<<2)|2,{105,50,0}},
/* 12465 */ {(10<<2)|2,{105,50,0}},
/* 12466 */ {(10<<2)|2,{105,50,0}},
/* 12467 */ {(10<<2)|2,{105,50,0}},
/* 12468 */ {(10<<2)|2,{105,50,0}},
/* 12469 */ {(10<<2)|2,{105,50,0}},
/* 12470 */ {(10<<2)|2,{105,50,0}},
/* 12471 */ {(10<<2)|2,{105,50,0}},
/* 12472 */ {(10<<2)|2,{105,50,0}},
/* 12473 */ {(10<<2)|2,{105,50,0}},
/* 12474 */ {(10<<2)|2,{105,50,0}},
/* 12475 */ {(10<<2)|2,{105,50,0}},
/* 12476 */ {(10<<2)|2,{105,50,0}},
/* 12477 */ {(10<<2)|2,{105,50,0}},
/* 12478 */ {(10<<2)|2,{105,50,0}},
/* 12479 */ {(10<<2)|2,{105,50,0}},
/* 12480 */ {(15<<2)|3,{105,97,48}},
/* 12481 */ {(15<<2)|3,{105,97,48}},
/* 12482 */ {(15<<2)|3,{105,97,49}},
/* 12483 */ {(15<<2)|3,{105,97,49}},
/* 12484 */ {(15<<2)|3,{105,97,50}},
/* 12485 */ {(15<<2)|3,{105,97,50}},
/* 12486 */ {(15<<2)|3,{105,97,97}},
/* 12487 */ {(15<<2)|3,{105,97,97}},
/* 12488 */ {(15<<2)|3,{105,97,99}},
/* 12489 */ {(15<<2)|3,{105,97,99}},
/* 12490 */ {(15<<2)|3,{105,97,101}},
/* 12491 */ {(15<<2)|3,{105,97,101}},
/* 12492 */ {(15<<2)|3,{105,97,105}},
/* 12493 */ {(15<<2)|3,{105,97,105}},
/* 12494 */ {(15<<2)|3,{105,97,111}},
/* 12495 */ {(15<<2)|3,{105,97,111}},
/* 12496 */ {(15<<2)|3,{105,97,115}},
/* 12497 */ {(15<<2)|3,{105,97,115}},
/* 12498 */ {(15<<2)|3,{105,97,116}},
/* 12499 */ {(15<<2)|3,{105,97,116}},
/* 12500 */ {(16<<2)|3,{105,97,32}},
/* 12501 */ {(16<<2)|3,{105,97,37}},
/* 12502 */ {(16<<2)|3,{105,97,45}},
/* 12503 */ {(16<<2)|3,{105,97,46}},
/* 12504 */ {(16<<2)|3,{105,97,47}},
/* 12505 */ {(16<<2)|3,{105,97,51}},
/* 12506 */ {(16<<2)|3,{105,97,52}},
/* 12507 */ {(16<<2)|3,{105,97,53}},
/* 12508 */ {(16<<2)|3,{105,97,54}},
/* 12509 */ {(16<<2)|3,{105,97,55}},
/* 12510 */ {(16<<2)|3,{105,97,56}},
/* 12511 */ {(16<<2)|3,{105,97,57}},
/* 12512 */ {(16<<2)|3,{105,97,61}},
/* 12513 */ {(16<<2)|3,{105,97,65}},
/* 12514 */ {(16<<2)|3,{105,97,95}},
/* 12515 */ {(16<<2)|3,{105,97,98}},
/* 12516 */ {(16<<2)|3,{105,97,100}},
/* 12517 */ {(16<<2)|3,{105,97,102}},
/* 12518 */ {(16<<2)|3,{105,97,103}},
/* 12519 */ {(16<<2)|3,{105,97,104}},
/* 12520 */ {(16<<2)|3,{105,97,108}},
/* 12521 */ {(16<<2)|3,{105,97,109}},
/* 12522 */ {(16<<2)|3,{105,97,110}},
/* 12523 */ {(16<<2)|3,{105,97,112}},
/* 12524 */ {(16<<2)|3,{105,97,114}},
/* 12525 */ {(16<<2)|3,{105,97,117}},
/* 12526 */ {(10<<2)|2,{105,97,0}},
/* 12527 */ {(10<<2)|2,{105,97,0}},
/* 12528 */ {(10<<2)|2,{105,97,0}},
/* 12529 */ {(10<<2)|2,{105,97,0}},
/* 12530 */ {(10<<2)|2,{105,97,0}},
/* 12531 */ {(10<<2)|2,{105,97,0}},
/* 12532 */ {(10<<2)|2,{105,97,0}},
/* 12533 */ {(10<<2)|2,{105,97,0}},
/* 12534 */ {(10<<2)|2,{105,97,0}},
/* 12535 */ {(10<<2)|2,{105,97,0}},
/* 12536 */ {(10<<2)|2,{105,97,0}},
/* 12537 */ {(10<<2)|2,{105,97,0}},
/* 12538 */ {(10<<2)|2,{105,97,0}},
/* 12539 */ {(10<<2)|2,{105,97,0}},
/* 12540 */ {(10<<2)|2,{105,97,0}},
/* 12541 */ {(10<<2)|2,{105,97,0}},
/* 12542 */ {(10<<2)|2,{105,97,0}},
/* 12543 */ {(10<<2)|2,{105,97,0}},
/* 12544 */ {(15<<2)|3,{105,99,48}},
/* 12545 */ {(15<<2)|3,{105,99,48}},
/* 12546 */ {(15<<2)|3,{105,99,49}},
/* 12547 */ {(15<<2)|3,{105,99,49}},
/* 12548 */ {(15<<2)|3,{105,99,50}},
/* 12549 */ {(15<<2)|3,{105,99,50}},
/* 12550 */ {(15<<2)|3,{105,99,97}},
/* 12551 */ {(15<<2)|3,{105,99,97}},
/* 12552 */ {(15<<2)|3,{105,99,99}},
/* 12553 */ {(15<<2)|3,{105,99,99}},
/* 12554 */ {(15<<2)|3,{105,99,101}},
/* 12555 */ {(15<<2)|3,{105,99,101}},
/* 12556 */ {(15<<2)|3,{105,99,105}},
/* 12557 */ {(15<<2)|3,{105,99,105}},
/* 12558 */ {(15<<2)|3,{105,99,111}},
/* 12559 */ {(15<<2)|3,{105,99,111}},
/* 12560 */ {(15<<2)|3,{105,99,115}},
/* 12561 */ {(15<<2)|3,{105,99,115}},
/* 12562 */ {(15<<2)|3,{105,99,116}},
/* 12563 */ {(15<<2)|3,{105,99,116}},
/* 12564 */ {(16<<2)|3,{105,99,32}},
/* 12565 */ {(16<<2)|3,{105,99,37}},
/* 12566 */ {(16<<2)|3,{105,99,45}},
/* 12567 */ {(16<<2)|3,{105,99,46}},
/* 12568 */ {(16<<2)|3,{105,99,47}},
/* 12569 */ {(16<<2)|3,{105,99,51}},
/* 12570 */ {(16<<2)|3,{105,99,52}},
/* 12571 */ {(16<<2)|3,{105,99,53}},
/* 12572 */ {(16<<2)|3,{105,99,54}},
/* 12573 */ {(16<<2)|3,{105,99,55}},
/* 12574 */ {(16<<2)|3,{105,99,56}},
/* 12575 */ {(16<<2)|3,{105,99,57}},
/* 12576 */ {(16<<2)|3,{105,99,61}},
/* 12577 */ {(16<<2)|3,{105,99,65}},
/* 12578 */ {(16<<2)|3,{105,99,95}},
/* 12579 */ {(16<<2)|3,{105,99,98}},
/* 12580 */ {(16<<2)|3,{105,99,100}},
/* 12581 */ {(16<<2)|3,{105,99,102}},
/* 12582 */ {(16<<2)|3,{105,99,103}},
/* 12583 */ {(16<<2)|3,{105,99,104}},
/* 12584 */ {(16<<2)|3,{105,99,108}},
/* 12585 */ {(16<<2)|3,{105,99,109}},
/* 12586 */ {(16<<2)|3,{105,99,110}},
/* 12587 */ {(16<<2)|3,{105,99,112}},
/* 12588 */ {(16<<2)|3,{105,99,114}},
/* 12589 */ {(16<<2)|3,{105,99,117}},
/* 12590 */ {(10<<2)|2,{105,99,0}},
/* 12591 */ {(10<<2)|2,{105,99,0}},
/* 12592 */ {(10<<2)|2,{105,99,0}},
/* 12593 */ {(10<<2)|2,{105,99,0}},
/* 12594 */ {(10<<2)|2,{105,99,0}},
/* 12595 */ {(10<<2)|2,{105,99,0}},
/* 12596 */ {(10<<2)|2,{105,99,0}},
/* 12597 */ {(10<<2)|2,{105,99,0}},
/* 12598 */ {(10<<2)|2,{105,99,0}},
/* 12599 */ {(10<<2)|2,{105,99,0}},
/* 12600 */ {(10<<2)|2,{105,99,0}},
/* 12601 */ {(10<<2)|2,{105,99,0}},
/* 12602 */ {(10<<2)|2,{105,99,0}},
/* 12603 */ {(10<<2)|2,{105,99,0}},
/* 12604 */ {(10<<2)|2,{105,99,0}},
/* 12605 */ {(10<<2)|2,{105,99,0}},
/* 12606 */ {(10<<2)|2,{105,99,0}},
/* 12607 */ {(10<<2)|2,{105,99,0}},
/* 12608 */ {(15<<2)|3,{105,101,48}},
/* 12609 */ {(15<<2)|3,{105,101,48}},
/* 12610 */ {(15<<2)|3,{105,101,49}},
/* 12611 */ {(15<<2)|3,{105,101,49}},
/* 12612 */ {(15<<2)|3,{105,101,50}},
/* 12613 */ {(15<<2)|3,{105,101,50}},
/* 12614 */ {(15<<2)|3,{105,101,97}},
/* 12615 */ {(15<<2)|3,{105,101,97}},
/* 12616 */ {(15<<2)|3,{105,101,99}},
/* 12617 */ {(15<<2)|3,{105,101,99}},
/* 12618 */ {(15<<2)|3,{105,101,101}},
/* 12619 */ {(15<<2)|3,{105,101,101}},
/* 12620 */ {(15<<2)|3,{105,101,105}},
/* 12621 */ {(15<<2)|3,{105,101,105}},
/* 12622 */ {(15<<2)|3,{105,101,111}},
/* 12623 */ {(15<<2)|3,{105,101,111}},
/* 12624 */ {(15<<2)|3,{105,101,115}},
/* 12625 */ {(15<<2)|3,{105,101,115}},
/* 12626 */ {(15<<2)|3,{105,101,116}},
/* 12627 */ {(15<<2)|3,{105,101,116}},
/* 12628 */ {(16<<2)|3,{105,101,32}},
/* 12629 */ {(16<<2)|3,{105,101,37}},
/* 12630 */ {(16<<2)|3,{105,101,45}},
/* 12631 */ {(16<<2)|3,{105,101,46}},
/* 12632 */ {(16<<2)|3,{105,101,47}},
/* 12633 */ {(16<<2)|3,{105,101,51}},
/* 12634 */ {(16<<2)|3,{105,101,52}},
/* 12635 */ {(16<<2)|3,{105,101,53}},
/* 12636 */ {(16<<2)|3,{105,101,54}},
/* 12637 */ {(16<<2)|3,{105,101,55}},
/* 12638 */ {(16<<2)|3,{105,101,56}},
/* 12639 */ {(16<<2)|3,{105,101,57}},
/* 12640 */ {(16<<2)|3,{105,101,61}},
/* 12641 */ {(16<<2)|3,{105,101,65}},
/* 12642 */ {(16<<2)|3,{105,101,95}},
/* 12643 */ {(16<<2)|3,{105,101,98}},
/* 12644 */ {(16<<2)|3,{105,101,100}},
/* 12645 */ {(16<<2)|3,{105,101,102}},
/* 12646 */ {(16<<2)|3,{105,101,103}},
/* 12647 */ {(16<<2)|3,{105,101,104}},
/* 12648 */ {(16<<2)|3,{105,101,108}},
/* 12649 */ {(16<<2)|3,{105,101,109}},
/* 12650 */ {(16<<2)|3,{105,101,110}},
/* 12651 */ {(16<<2)|3,{105,101,112}},
/* 12652 */ {(16<<2)|3,{105,101,114}},
/* 12653 */ {(16<<2)|3,{105,101,117}},
/* 12654 */ {(10<<2)|2,{105,101,0}},
/* 12655 */ {(10<<2)|2,{105,101,0}},
/* 12656 */ {(10<<2)|2,{105,101,0}},
/* 12657 */ {(10<<2)|2,{105,101,0}},
/* 12658 */ {(10<<2)|2,{105,101,0}},
/* 12659 */ {(10<<2)|2,{105,101,0}},
/* 12660 */ {(10<<2)|2,{105,101,0}},
/* 12661 */ {(10<<2)|2,{105,101,0}},
/* 12662 */ {(10<<2)|2,{105,101,0}},
/* 12663 */ {(10<<2)|2,{105,101,0}},
/* 12664 */ {(10<<2)|2,{105,101,0}},
/* 12665 */ {(10<<2)|2,{105,101,0}},
/* 12666 */ {(10<<2)|2,{105,101,0}},
/* 12667 */ {(10<<2)|2,{105,101,0}},
/* 12668 */ {(10<<2)|2,{105,101,0}},
/* 12669 */ {(10<<2)|2,{105,101,0}},
/* 12670 */ {(10<<2)|2,{105,101,0}},
/* 12671 */ {(10<<2)|2,{105,101,0}},
/* 12672 */ {(15<<2)|3,{105,105,48}},
/* 12673 */ {(15<<2)|3,{105,105,48}},
/* 12674 */ {(15<<2)|3,{105,105,49}},
/* 12675 */ {(15<<2)|3,{105,105,49}},
/* 12676 */ {(15<<2)|3,{105,105,50}},
/* 12677 */ {(15<<2)|3,{105,105,50}},
/* 12678 */ {(15<<2)|3,{105,105,97}},
/* 12679 */ {(15<<2)|3,{105,105,97}},
/* 12680 */ {(15<<2)|3,{105,105,99}},
/* 12681 */ {(15<<2)|3,{105,105,99}},
/* 12682 */ {(15<<2)|3,{105,105,101}},
/* 12683 */ {(15<<2)|3,{105,105,101}},
/* 12684 */ {(15<<2)|3,{105,105,105}},
/* 12685 */ {(15<<2)|3,{105,105,105}},
/* 12686 */ {(15<<2)|3,{105,105,111}},
/* 12687 */ {(15<<2)|3,{105,105,111}},
/* 12688 */ {(15<<2)|3,{105,105,115}},
/* 12689 */ {(15<<2)|3,{105,105,115}},
/* 12690 */ {(15<<2)|3,{105,105,116}},
/* 12691 */ {(15<<2)|3,{105,105,116}},
/* 12692 */ {(16<<2)|3,{105,105,32}},
/* 12693 */ {(16<<2)|3,{105,105,37}},
/* 12694 */ {(16<<2)|3,{105,105,45}},
/* 12695 */ {(16<<2)|3,{105,105,46}},
/* 12696 */ {(16<<2)|3,{105,105,47}},
/* 12697 */ {(16<<2)|3,{105,105,51}},
/* 12698 */ {(16<<2)|3,{105,105,52}},
/* 12699 */ {(16<<2)|3,{105,105,53}},
/* 12700 */ {(16<<2)|3,{105,105,54}},
/* 12701 */ {(16<<2)|3,{105,105,55}},
/* 12702 */ {(16<<2)|3,{105,105,56}},
/* 12703 */ {(16<<2)|3,{105,105,57}},
/* 12704 */ {(16<<2)|3,{105,105,61}},
/* 12705 */ {(16<<2)|3,{105,105,65}},
/* 12706 */ {(16<<2)|3,{105,105,95}},
/* 12707 */ {(16<<2)|3,{105,105,98}},
/* 12708 */ {(16<<2)|3,{105,105,100}},
/* 12709 */ {(16<<2)|3,{105,105,102}},
/* 12710 */ {(16<<2)|3,{105,105,103}},
/* 12711 */ {(16<<2)|3,{105,105,104}},
/* 12712 */ {(16<<2)|3,{105,105,108}},
/* 12713 */ {(16<<2)|3,{105,105,109}},
/* 12714 */ {(16<<2)|3,{105,105,110}},
/* 12715 */ {(16<<2)|3,{105,105,112}},
/* 12716 */ {(16<<2)|3,{105,105,114}},
/* 12717 */ {(16<<2)|3,{105,105,117}},
/* 12718 */ {(10<<2)|2,{105,105,0}},
/* 12719 */ {(10<<2)|2,{105,105,0}},
/* 12720 */ {(10<<2)|2,{105,105,0}},
/* 12721 */ {(10<<2)|2,{105,105,0}},
/* 12722 */ {(10<<2)|2,{105,105,0}},
/* 12723 */ {(10<<2)|2,{105,105,0}},
/* 12724 */ {(10<<2)|2,{105,105,0}},
/* 12725 */ {(10<<2)|2,{105,105,0}},
/* 12726 */ {(10<<2)|2,{105,105,0}},
/* 12727 */ {(10<<2)|2,{105,105,0}},
/* 12728 */ {(10<<2)|2,{105,105,0}},
/* 12729 */ {(10<<2)|2,{105,105,0}},
/* 12730 */ {(10<<2)|2,{105,105,0}},
/* 12731 */ {(10<<2)|2,{105,105,0}},
/* 12732 */ {(10<<2)|2,{105,105,0}},
/* 12733 */ {(10<<2)|2,{105,105,0}},
/* 12734 */ {(10<<2)|2,{105,105,0}},
/* 12735 */ {(10<<2)|2,{105,105,0}},
/* 12736 */ {(15<<2)|3,{105,111,48}},
/* 12737 */ {(15<<2)|3,{105,111,48}},
/* 12738 */ {(15<<2)|3,{105,111,49}},
/* 12739 */ {(15<<2)|3,{105,111,49}},
/* 12740 */ {(15<<2)|3,{105,111,50}},
/* 12741 */ {(15<<2)|3,{105,111,50}},
/* 12742 */ {(15<<2)|3,{105,111,97}},
/* 12743 */ {(15<<2)|3,{105,111,97}},
/* 12744 */ {(15<<2)|3,{105,111,99}},
/* 12745 */ {(15<<2)|3,{105,111,99}},
/* 12746 */ {(15<<2)|3,{105,111,101}},
/* 12747 */ {(15<<2)|3,{105,111,101}},
/* 12748 */ {(15<<2)|3,{105,111,105}},
/* 12749 */ {(15<<2)|3,{105,111,105}},
/* 12750 */ {(15<<2)|3,{105,111,111}},
/* 12751 */ {(15<<2)|3,{105,111,111}},
/* 12752 */ {(15<<2)|3,{105,111,115}},
/* 12753 */ {(15<<2)|3,{105,111,115}},
/* 12754 */ {(15<<2)|3,{105,111,116}},
/* 12755 */ {(15<<2)|3,{105,111,116}},
/* 12756 */ {(16<<2)|3,{105,111,32}},
/* 12757 */ {(16<<2)|3,{105,111,37}},
/* 12758 */ {(16<<2)|3,{105,111,45}},
/* 12759 */ {(16<<2)|3,{105,111,46}},
/* 12760 */ {(16<<2)|3,{105,111,47}},
/* 12761 */ {(16<<2)|3,{105,111,51}},
/* 12762 */ {(16<<2)|3,{105,111,52}},
/* 12763 */ {(16<<2)|3,{105,111,53}},
/* 12764 */ {(16<<2)|3,{105,111,54}},
/* 12765 */ {(16<<2)|3,{105,111,55}},
/* 12766 */ {(16<<2)|3,{105,111,56}},
/* 12767 */ {(16<<2)|3,{105,111,57}},
/* 12768 */ {(16<<2)|3,{105,111,61}},
/* 12769 */ {(16<<2)|3,{105,111,65}},
/* 12770 */ {(16<<2)|3,{105,111,95}},
/* 12771 */ {(16<<2)|3,{105,111,98}},
/* 12772 */ {(16<<2)|3,{105,111,100}},
/* 12773 */ {(16<<2)|3,{105,111,102}},
/* 12774 */ {(16<<2)|3,{105,111,103}},
/* 12775 */ {(16<<2)|3,{105,111,104}},
/* 12776 */ {(16<<2)|3,{105,111,108}},
/* 12777 */ {(16<<2)|3,{105,111,109}},
/* 12778 */ {(16<<2)|3,{105,111,110}},
/* 12779 */ {(16<<2)|3,{105,111,112}},
/* 12780 */ {(16<<2)|3,{105,111,114}},
/* 12781 */ {(16<<2)|3,{105,111,117}},
/* 12782 */ {(10<<2)|2,{105,111,0}},
/* 12783 */ {(10<<2)|2,{105,111,0}},
/* 12784 */ {(10<<2)|2,{105,111,0}},
/* 12785 */ {(10<<2)|2,{105,111,0}},
/* 12786 */ {(10<<2)|2,{105,111,0}},
/* 12787 */ {(10<<2)|2,{105,111,0}},
/* 12788 */ {(10<<2)|2,{105,111,0}},
/* 12789 */ {(10<<2)|2,{105,111,0}},
/* 12790 */ {(10<<2)|2,{105,111,0}},
/* 12791 */ {(10<<2)|2,{105,111,0}},
/* 12792 */ {(10<<2)|2,{105,111,0}},
/* 12793 */ {(10<<2)|2,{105,111,0}},
/* 12794 */ {(10<<2)|2,{105,111,0}},
/* 12795 */ {(10<<2)|2,{105,111,0}},
/* 12796 */ {(10<<2)|2,{105,111,0}},
/* 12797 */ {(10<<2)|2,{105,111,0}},
/* 12798 */ {(10<<2)|2,{105,111,0}},
/* 12799 */ {(10<<2)|2,{105,111,0}},
/* 12800 */ {(15<<2)|3,{105,115,48}},
/* 12801 */ {(15<<2)|3,{105,115,48}},
/* 12802 */ {(15<<2)|3,{105,115,49}},
/* 12803 */ {(15<<2)|3,{105,115,49}},
/* 12804 */ {(15<<2)|3,{105,115,50}},
/* 12805 */ {(15<<2)|3,{105,115,50}},
/* 12806 */ {(15<<2)|3,{105,115,97}},
/* 12807 */ {(15<<2)|3,{105,115,97}},
/* 12808 */ {(15<<2)|3,{105,115,99}},
/* 12809 */ {(15<<2)|3,{105,115,99}},
/* 12810 */ {(15<<2)|3,{105,115,101}},
/* 12811 */ {(15<<2)|3,{105,115,101}},
/* 12812 */ {(15<<2)|3,{105,115,105}},
/* 12813 */ {(15<<2)|3,{105,115,105}},
/* 12814 */ {(15<<2)|3,{105,115,111}},
/* 12815 */ {(15<<2)|3,{105,115,111}},
/* 12816 */ {(15<<2)|3,{105,115,115}},
/* 12817 */ {(15<<2)|3,{105,115,115}},
/* 12818 */ {(15<<2)|3,{105,115,116}},
/* 12819 */ {(15<<2)|3,{105,115,116}},
/* 12820 */ {(16<<2)|3,{105,115,32}},
/* 12821 */ {(16<<2)|3,{105,115,37}},
/* 12822 */ {(16<<2)|3,{105,115,45}},
/* 12823 */ {(16<<2)|3,{105,115,46}},
/* 12824 */ {(16<<2)|3,{105,115,47}},
/* 12825 */ {(16<<2)|3,{105,115,51}},
/* 12826 */ {(16<<2)|3,{105,115,52}},
/* 12827 */ {(16<<2)|3,{105,115,53}},
/* 12828 */ {(16<<2)|3,{105,115,54}},
/* 12829 */ {(16<<2)|3,{105,115,55}},
/* 12830 */ {(16<<2)|3,{105,115,56}},
/* 12831 */ {(16<<2)|3,{105,115,57}},
/* 12832 */ {(16<<2)|3,{105,115,61}},
/* 12833 */ {(16<<2)|3,{105,115,65}},
/* 12834 */ {(16<<2)|3,{105,115,95}},
/* 12835 */ {(16<<2)|3,{105,115,98}},
/* 12836 */ {(16<<2)|3,{105,115,100}},
/* 12837 */ {(16<<2)|3,{105,115,102}},
/* 12838 */ {(16<<2)|3,{105,115,103}},
/* 12839 */ {(16<<2)|3,{105,115,104}},
/* 12840 */ {(16<<2)|3,{105,115,108}},
/* 12841 */ {(16<<2)|3,{105,115,109}},
/* 12842 */ {(16<<2)|3,{105,115,110}},
/* 12843 */ {(16<<2)|3,{105,115,112}},
/* 12844 */ {(16<<2)|3,{105,115,114}},
/* 12845 */ {(16<<2)|3,{105,115,117}},
/* 12846 */ {(10<<2)|2,{105,115,0}},
/* 12847 */ {(10<<2)|2,{105,115,0}},
/* 12848 */ {(10<<2)|2,{105,115,0}},
/* 12849 */ {(10<<2)|2,{105,115,0}},
/* 12850 */ {(10<<2)|2,{105,115,0}},
/* 12851 */ {(10<<2)|2,{105,115,0}},
/* 12852 */ {(10<<2)|2,{105,115,0}},
/* 12853 */ {(10<<2)|2,{105,115,0}},
/* 12854 */ {(10<<2)|2,{105,115,0}},
/* 12855 */ {(10<<2)|2,{105,115,0}},
/* 12856 */ {(10<<2)|2,{105,115,0}},
/* 12857 */ {(10<<2)|2,{105,115,0}},
/* 12858 */ {(10<<2)|2,{105,115,0}},
/* 12859 */ {(10<<2)|2,{105,115,0}},
/* 12860 */ {(10<<2)|2,{105,115,0}},
/* 12861 */ {(10<<2)|2,{105,115,0}},
/* 12862 */ {(10<<2)|2,{105,115,0}},
/* 12863 */ {(10<<2)|2,{105,115,0}},
/* 12864 */ {(15<<2)|3,{105,116,48}},
/* 12865 */ {(15<<2)|3,{105,116,48}},
/* 12866 */ {(15<<2)|3,{105,116,49}},
/* 12867 */ {(15<<2)|3,{105,116,49}},
/* 12868 */ {(15<<2)|3,{105,116,50}},
/* 12869 */ {(15<<2)|3,{105,116,50}},
/* 12870 */ {(15<<2)|3,{105,116,97}},
/* 12871 */ {(15<<2)|3,{105,116,97}},
/* 12872 */ {(15<<2)|3,{105,116,99}},
/* 12873 */ {(15<<2)|3,{105,116,99}},
/* 12874 */ {(15<<2)|3,{105,116,101}},
/* 12875 */ {(15<<2)|3,{105,116,101}},
/* 12876 */ {(15<<2)|3,{105,116,105}},
/* 12877 */ {(15<<2)|3,{105,116,105}},
/* 12878 */ {(15<<2)|3,{105,116,111}},
/* 12879 */ {(15<<2)|3,{105,116,111}},
/* 12880 */ {(15<<2)|3,{105,116,115}},
/* 12881 */ {(15<<2)|3,{105,116,115}},
/* 12882 */ {(15<<2)|3,{105,116,116}},
/* 12883 */ {(15<<2)|3,{105,116,116}},
/* 12884 */ {(16<<2)|3,{105,116,32}},
/* 12885 */ {(16<<2)|3,{105,116,37}},
/* 12886 */ {(16<<2)|3,{105,116,45}},
/* 12887 */ {(16<<2)|3,{105,116,46}},
/* 12888 */ {(16<<2)|3,{105,116,47}},
/* 12889 */ {(16<<2)|3,{105,116,51}},
/* 12890 */ {(16<<2)|3,{105,116,52}},
/* 12891 */ {(16<<2)|3,{105,116,53}},
/* 12892 */ {(16<<2)|3,{105,116,54}},
/* 12893 */ {(16<<2)|3,{105,116,55}},
/* 12894 */ {(16<<2)|3,{105,116,56}},
/* 12895 */ {(16<<2)|3,{105,116,57}},
/* 12896 */ {(16<<2)|3,{105,116,61}},
/* 12897 */ {(16<<2)|3,{105,116,65}},
/* 12898 */ {(16<<2)|3,{105,116,95}},
/* 12899 */ {(16<<2)|3,{105,116,98}},
/* 12900 */ {(16<<2)|3,{105,116,100}},
/* 12901 */ {(16<<2)|3,{105,116,102}},
/* 12902 */ {(16<<2)|3,{105,116,103}},
/* 12903 */ {(16<<2)|3,{105,116,104}},
/* 12904 */ {(16<<2)|3,{105,116,108}},
/* 12905 */ {(16<<2)|3,{105,116,109}},
/* 12906 */ {(16<<2)|3,{105,116,110}},
/* 12907 */ {(16<<2)|3,{105,116,112}},
/* 12908 */ {(16<<2)|3,{105,116,114}},
/* 12909 */ {(16<<2)|3,{105,116,117}},
/* 12910 */ {(10<<2)|2,{105,116,0}},
/* 12911 */ {(10<<2)|2,{105,116,0}},
/* 12912 */ {(10<<2)|2,{105,116,0}},
/* 12913 */ {(10<<2)|2,{105,116,0}},
/* 12914 */ {(10<<2)|2,{105,116,0}},
/* 12915 */ {(10<<2)|2,{105,116,0}},
/* 12916 */ {(10<<2)|2,{105,116,0}},
/* 12917 */ {(10<<2)|2,{105,116,0}},
/* 12918 */ {(10<<2)|2,{105,116,0}},
/* 12919 */ {(10<<2)|2,{105,116,0}},
/* 12920 */ {(10<<2)|2,{105,116,0}},
/* 12921 */ {(10<<2)|2,{105,116,0}},
/* 12922 */ {(10<<2)|2,{105,116,0}},
/* 12923 */ {(10<<2)|2,{105,116,0}},
/* 12924 */ {(10<<2)|2,{105,116,0}},
/* 12925 */ {(10<<2)|2,{105,116,0}},
/* 12926 */ {(10<<2)|2,{105,116,0}},
/* 12927 */ {(10<<2)|2,{105,116,0}},
/* 12928 */ {(16<<2)|3,{105,32,48}},
/* 12929 */ {(16<<2)|3,{105,32,49}},
/* 12930 */ {(16<<2)|3,{105,32,50}},
/* 12931 */ {(16<<2)|3,{105,32,97}},
/* 12932 */ {(16<<2)|3,{105,32,99}},
/* 12933 */ {(16<<2)|3,{105,32,101}},
/* 12934 */ {(16<<2)|3,{105,32,105}},
/* 12935 */ {(16<<2)|3,{105,32,111}},
/* 12936 */ {(16<<2)|3,{105,32,115}},
/* 12937 */ {(16<<2)|3,{105,32,116}},
/* 12938 */ {(11<<2)|2,{105,32,0}},
/* 12939 */ {(11<<2)|2,{105,32,0}},
/* 12940 */ {(11<<2)|2,{105,32,0}},
/* 12941 */ {(11<<2)|2,{105,32,0}},
/* 12942 */ {(11<<2)|2,{105,32,0}},
/* 12943 */ {(11<<2)|2,{105,32,0}},
/* 12944 */ {(11<<2)|2,{105,32,0}},
/* 12945 */ {(11<<2)|2,{105,32,0}},
/* 12946 */ {(11<<2)|2,{105,32,0}},
/* 12947 */ {(11<<2)|2,{105,32,0}},
/* 12948 */ {(11<<2)|2,{105,32,0}},
/* 12949 */ {(11<<2)|2,{105,32,0}},
/* 12950 */ {(11<<2)|2,{105,32,0}},
/* 12951 */ {(11<<2)|2,{105,32,0}},
/* 12952 */ {(11<<2)|2,{105,32,0}},
/* 12953 */ {(11<<2)|2,{105,32,0}},
/* 12954 */ {(11<<2)|2,{105,32,0}},
/* 12955 */ {(11<<2)|2,{105,32,0}},
/* 12956 */ {(11<<2)|2,{105,32,0}},
/* 12957 */ {(11<<2)|2,{105,32,0}},
/* 12958 */ {(11<<2)|2,{105,32,0}},
/* 12959 */ {(11<<2)|2,{105,32,0}},
/* 12960 */ {(16<<2)|3,{105,37,48}},
/* 12961 */ {(16<<2)|3,{105,37,49}},
/* 12962 */ {(16<<2)|3,{105,37,50}},
/* 12963 */ {(16<<2)|3,{105,37,97}},
/* 12964 */ {(16<<2)|3,{105,37,99}},
/* 12965 */ {(16<<2)|3,{105,37,101}},
/* 12966 */ {(16<<2)|3,{105,37,105}},
/* 12967 */ {(16<<2)|3,{105,37,111}},
/* 12968 */ {(16<<2)|3,{105,37,115}},
/* 12969 */ {(16<<2)|3,{105,37,116}},
/* 12970 */ {(11<<2)|2,{105,37,0}},
/* 12971 */ {(11<<2)|2,{105,37,0}},
/* 12972 */ {(11<<2)|2,{105,37,0}},
/* 12973 */ {(11<<2)|2,{105,37,0}},
/* 12974 */ {(11<<2)|2,{105,37,0}},
/* 12975 */ {(11<<2)|2,{105,37,0}},
/* 12976 */ {(11<<2)|2,{105,37,0}},
/* 12977 */ {(11<<2)|2,{105,37,0}},
/* 12978 */ {(11<<2)|2,{105,37,0}},
/* 12979 */ {(11<<2)|2,{105,37,0}},
/* 12980 */ {(11<<2)|2,{105,37,0}},
/* 12981 */ {(11<<2)|2,{105,37,0}},
/* 12982 */ {(11<<2)|2,{105,37,0}},
/* 12983 */ {(11<<2)|2,{105,37,0}},
/* 12984 */ {(11<<2)|2,{105,37,0}},
/* 12985 */ {(11<<2)|2,{105,37,0}},
/* 12986 */ {(11<<2)|2,{105,37,0}},
/* 12987 */ {(11<<2)|2,{105,37,0}},
/* 12988 */ {(11<<2)|2,{105,37,0}},
/* 12989 */ {(11<<2)|2,{105,37,0}},
/* 12990 */ {(11<<2)|2,{105,37,0}},
/* 12991 */ {(11<<2)|2,{105,37,0}},
/* 12992 */ {(16<<2)|3,{105,45,48}},
/* 12993 */ {(16<<2)|3,{105,45,49}},
/* 12994 */ {(16<<2)|3,{105,45,50}},
/* 12995 */ {(16<<2)|3,{105,45,97}},
/* 12996 */ {(16<<2)|3,{105,45,99}},
/* 12997 */ {(16<<2)|3,{105,45,101}},
/* 12998 */ {(16<<2)|3,{105,45,105}},
/* 12999 */ {(16<<2)|3,{105,45,111}},
/* 13000 */ {(16<<2)|3,{105,45,115}},
/* 13001 */ {(16<<2)|3,{105,45,116}},
/* 13002 */ {(11<<2)|2,{105,45,0}},
/* 13003 */ {(11<<2)|2,{105,45,0}},
/* 13004 */ {(11<<2)|2,{105,45,0}},
/* 13005 */ {(11<<2)|2,{105,45,0}},
/* 13006 */ {(11<<2)|2,{105,45,0}},
/* 13007 */ {(11<<2)|2,{105,45,0}},
/* 13008 */ {(11<<2)|2,{105,45,0}},
/* 13009 */ {(11<<2)|2,{105,45,0}},
/* 13010 */ {(11<<2)|2,{105,45,0}},
/* 13011 */ {(11<<2)|2,{105,45,0}},
/* 13012 */ {(11<<2)|2,{105,45,0}},
/* 13013 */ {(11<<2)|2,{105,45,0}},
/* 13014 */ {(11<<2)|2,{105,45,0}},
/* 13015 */ {(11<<2)|2,{105,45,0}},
/* 13016 */ {(11<<2)|2,{105,45,0}},
/* 13017 */ {(11<<2)|2,{105,45,0}},
/* 13018 */ {(11<<2)|2,{105,45,0}},
/* 13019 */ {(11<<2)|2,{105,45,0}},
/* 13020 */ {(11<<2)|2,{105,45,0}},
/* 13021 */ {(11<<2)|2,{105,45,0}},
/* 13022 */ {(11<<2)|2,{105,45,0}},
/* 13023 */ {(11<<2)|2,{105,45,0}},
/* 13024 */ {(16<<2)|3,{105,46,48}},
/* 13025 */ {(16<<2)|3,{105,46,49}},
/* 13026 */ {(16<<2)|3,{105,46,50}},
/* 13027 */ {(16<<2)|3,{105,46,97}},
/* 13028 */ {(16<<2)|3,{105,46,99}},
/* 13029 */ {(16<<2)|3,{105,46,101}},
/* 13030 */ {(16<<2)|3,{105,46,105}},
/* 13031 */ {(16<<2)|3,{105,46,111}},
/* 13032 */ {(16<<2)|3,{105,46,115}},
/* 13033 */ {(16<<2)|3,{105,46,116}},
/* 13034 */ {(11<<2)|2,{105,46,0}},
/* 13035 */ {(11<<2)|2,{105,46,0}},
/* 13036 */ {(11<<2)|2,{105,46,0}},
/* 13037 */ {(11<<2)|2,{105,46,0}},
/* 13038 */ {(11<<2)|2,{105,46,0}},
/* 13039 */ {(11<<2)|2,{105,46,0}},
/* 13040 */ {(11<<2)|2,{105,46,0}},
/* 13041 */ {(11<<2)|2,{105,46,0}},
/* 13042 */ {(11<<2)|2,{105,46,0}},
/* 13043 */ {(11<<2)|2,{105,46,0}},
/* 13044 */ {(11<<2)|2,{105,46,0}},
/* 13045 */ {(11<<2)|2,{105,46,0}},
/* 13046 */ {(11<<2)|2,{105,46,0}},
/* 13047 */ {(11<<2)|2,{105,46,0}},
/* 13048 */ {(11<<2)|2,{105,46,0}},
/* 13049 */ {(11<<2)|2,{105,46,0}},
/* 13050 */ {(11<<2)|2,{105,46,0}},
/* 13051 */ {(11<<2)|2,{105,46,0}},
/* 13052 */ {(11<<2)|2,{105,46,0}},
/* 13053 */ {(11<<2)|2,{105,46,0}},
/* 13054 */ {(11<<2)|2,{105,46,0}},
/* 13055 */ {(11<<2)|2,{105,46,0}},
/* 13056 */ {(16<<2)|3,{105,47,48}},
/* 13057 */ {(16<<2)|3,{105,47,49}},
/* 13058 */ {(16<<2)|3,{105,47,50}},
/* 13059 */ {(16<<2)|3,{105,47,97}},
/* 13060 */ {(16<<2)|3,{105,47,99}},
/* 13061 */ {(16<<2)|3,{105,47,101}},
/* 13062 */ {(16<<2)|3,{105,47,105}},
/* 13063 */ {(16<<2)|3,{105,47,111}},
/* 13064 */ {(16<<2)|3,{105,47,115}},
/* 13065 */ {(16<<2)|3,{105,47,116}},
/* 13066 */ {(11<<2)|2,{105,47,0}},
/* 13067 */ {(11<<2)|2,{105,47,0}},
/* 13068 */ {(11<<2)|2,{105,47,0}},
/* 13069 */ {(11<<2)|2,{105,47,0}},
/* 13070 */ {(11<<2)|2,{105,47,0}},
/* 13071 */ {(11<<2)|2,{105,47,0}},
/* 13072 */ {(11<<2)|2,{105,47,0}},
/* 13073 */ {(11<<2)|2,{105,47,0}},
/* 13074 */ {(11<<2)|2,{105,47,0}},
/* 13075 */ {(11<<2)|2,{105,47,0}},
/* 13076 */ {(11<<2)|2,{105,47,0}},
/* 13077 */ {(11<<2)|2,{105,47,0}},
/* 13078 */ {(11<<2)|2,{105,47,0}},
/* 13079 */ {(11<<2)|2,{105,47,0}},
/* 13080 */ {(11<<2)|2,{105,47,0}},
/* 13081 */ {(11<<2)|2,{105,47,0}},
/* 13082 */ {(11<<2)|2,{105,47,0}},
/* 13083 */ {(11<<2)|2,{105,47,0}},
/* 13084 */ {(11<<2)|2,{105,47,0}},
/* 13085 */ {(11<<2)|2,{105,47,0}},
/* 13086 */ {(11<<2)|2,{105,47,0}},
/* 13087 */ {(11<<2)|2,{105,47,0}},
/* 13088 */ {(16<<2)|3,{105,51,48}},
/* 13089 */ {(16<<2)|3,{105,51,49}},
/* 13090 */ {(16<<2)|3,{105,51,50}},
/* 13091 */ {(16<<2)|3,{105,51,97}},
/* 13092 */ {(16<<2)|3,{105,51,99}},
/* 13093 */ {(16<<2)|3,{105,51,101}},
/* 13094 */ {(16<<2)|3,{105,51,105}},
/* 13095 */ {(16<<2)|3,{105,51,111}},
/* 13096 */ {(16<<2)|3,{105,51,115}},
/* 13097 */ {(16<<2)|3,{105,51,116}},
/* 13098 */ {(11<<2)|2,{105,51,0}},
/* 13099 */ {(11<<2)|2,{105,51,0}},
/* 13100 */ {(11<<2)|2,{105,51,0}},
/* 13101 */ {(11<<2)|2,{105,51,0}},
/* 13102 */ {(11<<2)|2,{105,51,0}},
/* 13103 */ {(11<<2)|2,{105,51,0}},
/* 13104 */ {(11<<2)|2,{105,51,0}},
/* 13105 */ {(11<<2)|2,{105,51,0}},
/* 13106 */ {(11<<2)|2,{105,51,0}},
/* 13107 */ {(11<<2)|2,{105,51,0}},
/* 13108 */ {(11<<2)|2,{105,51,0}},
/* 13109 */ {(11<<2)|2,{105,51,0}},
/* 13110 */ {(11<<2)|2,{105,51,0}},
/* 13111 */ {(11<<2)|2,{105,51,0}},
/* 13112 */ {(11<<2)|2,{105,51,0}},
/* 13113 */ {(11<<2)|2,{105,51,0}},
/* 13114 */ {(11<<2)|2,{105,51,0}},
/* 13115 */ {(11<<2)|2,{105,51,0}},
/* 13116 */ {(11<<2)|2,{105,51,0}},
/* 13117 */ {(11<<2)|2,{105,51,0}},
/* 13118 */ {(11<<2)|2,{105,51,0}},
/* 13119 */ {(11<<2)|2,{105,51,0}},
/* 13120 */ {(16<<2)|3,{105,52,48}},
/* 13121 */ {(16<<2)|3,{105,52,49}},
/* 13122 */ {(16<<2)|3,{105,52,50}},
/* 13123 */ {(16<<2)|3,{105,52,97}},
/* 13124 */ {(16<<2)|3,{105,52,99}},
/* 13125 */ {(16<<2)|3,{105,52,101}},
/* 13126 */ {(16<<2)|3,{105,52,105}},
/* 13127 */ {(16<<2)|3,{105,52,111}},
/* 13128 */ {(16<<2)|3,{105,52,115}},
/* 13129 */ {(16<<2)|3,{105,52,116}},
/* 13130 */ {(11<<2)|2,{105,52,0}},
/* 13131 */ {(11<<2)|2,{105,52,0}},
/* 13132 */ {(11<<2)|2,{105,52,0}},
/* 13133 */ {(11<<2)|2,{105,52,0}},
/* 13134 */ {(11<<2)|2,{105,52,0}},
/* 13135 */ {(11<<2)|2,{105,52,0}},
/* 13136 */ {(11<<2)|2,{105,52,0}},
/* 13137 */ {(11<<2)|2,{105,52,0}},
/* 13138 */ {(11<<2)|2,{105,52,0}},
/* 13139 */ {(11<<2)|2,{105,52,0}},
/* 13140 */ {(11<<2)|2,{105,52,0}},
/* 13141 */ {(11<<2)|2,{105,52,0}},
/* 13142 */ {(11<<2)|2,{105,52,0}},
/* 13143 */ {(11<<2)|2,{105,52,0}},
/* 13144 */ {(11<<2)|2,{105,52,0}},
/* 13145 */ {(11<<2)|2,{105,52,0}},
/* 13146 */ {(11<<2)|2,{105,52,0}},
/* 13147 */ {(11<<2)|2,{105,52,0}},
/* 13148 */ {(11<<2)|2,{105,52,0}},
/* 13149 */ {(11<<2)|2,{105,52,0}},
/* 13150 */ {(11<<2)|2,{105,52,0}},
/* 13151 */ {(11<<2)|2,{105,52,0}},
/* 13152 */ {(16<<2)|3,{105,53,48}},
/* 13153 */ {(16<<2)|3,{105,53,49}},
/* 13154 */ {(16<<2)|3,{105,53,50}},
/* 13155 */ {(16<<2)|3,{105,53,97}},
/* 13156 */ {(16<<2)|3,{105,53,99}},
/* 13157 */ {(16<<2)|3,{105,53,101}},
/* 13158 */ {(16<<2)|3,{105,53,105}},
/* 13159 */ {(16<<2)|3,{105,53,111}},
/* 13160 */ {(16<<2)|3,{105,53,115}},
/* 13161 */ {(16<<2)|3,{105,53,116}},
/* 13162 */ {(11<<2)|2,{105,53,0}},
/* 13163 */ {(11<<2)|2,{105,53,0}},
/* 13164 */ {(11<<2)|2,{105,53,0}},
/* 13165 */ {(11<<2)|2,{105,53,0}},
/* 13166 */ {(11<<2)|2,{105,53,0}},
/* 13167 */ {(11<<2)|2,{105,53,0}},
/* 13168 */ {(11<<2)|2,{105,53,0}},
/* 13169 */ {(11<<2)|2,{105,53,0}},
/* 13170 */ {(11<<2)|2,{105,53,0}},
/* 13171 */ {(11<<2)|2,{105,53,0}},
/* 13172 */ {(11<<2)|2,{105,53,0}},
/* 13173 */ {(11<<2)|2,{105,53,0}},
/* 13174 */ {(11<<2)|2,{105,53,0}},
/* 13175 */ {(11<<2)|2,{105,53,0}},
/* 13176 */ {(11<<2)|2,{105,53,0}},
/* 13177 */ {(11<<2)|2,{105,53,0}},
/* 13178 */ {(11<<2)|2,{105,53,0}},
/* 13179 */ {(11<<2)|2,{105,53,0}},
/* 13180 */ {(11<<2)|2,{105,53,0}},
/* 13181 */ {(11<<2)|2,{105,53,0}},
/* 13182 */ {(11<<2)|2,{105,53,0}},
/* 13183 */ {(11<<2)|2,{105,53,0}},
/* 13184 */ {(16<<2)|3,{105,54,48}},
/* 13185 */ {(16<<2)|3,{105,54,49}},
/* 13186 */ {(16<<2)|3,{105,54,50}},
/* 13187 */ {(16<<2)|3,{105,54,97}},
/* 13188 */ {(16<<2)|3,{105,54,99}},
/* 13189 */ {(16<<2)|3,{105,54,101}},
/* 13190 */ {(16<<2)|3,{105,54,105}},
/* 13191 */ {(16<<2)|3,{105,54,111}},
/* 13192 */ {(16<<2)|3,{105,54,115}},
/* 13193 */ {(16<<2)|3,{105,54,116}},
/* 13194 */ {(11<<2)|2,{105,54,0}},
/* 13195 */ {(11<<2)|2,{105,54,0}},
/* 13196 */ {(11<<2)|2,{105,54,0}},
/* 13197 */ {(11<<2)|2,{105,54,0}},
/* 13198 */ {(11<<2)|2,{105,54,0}},
/* 13199 */ {(11<<2)|2,{105,54,0}},
/* 13200 */ {(11<<2)|2,{105,54,0}},
/* 13201 */ {(11<<2)|2,{105,54,0}},
/* 13202 */ {(11<<2)|2,{105,54,0}},
/* 13203 */ {(11<<2)|2,{105,54,0}},
/* 13204 */ {(11<<2)|2,{105,54,0}},
/* 13205 */ {(11<<2)|2,{105,54,0}},
/* 13206 */ {(11<<2)|2,{105,54,0}},
/* 13207 */ {(11<<2)|2,{105,54,0}},
/* 13208 */ {(11<<2)|2,{105,54,0}},
/* 13209 */ {(11<<2)|2,{105,54,0}},
/* 13210 */ {(11<<2)|2,{105,54,0}},
/* 13211 */ {(11<<2)|2,{105,54,0}},
/* 13212 */ {(11<<2)|2,{105,54,0}},
/* 13213 */ {(11<<2)|2,{105,54,0}},
/* 13214 */ {(11<<2)|2,{105,54,0}},
/* 13215 */ {(11<<2)|2,{105,54,0}},
/* 13216 */ {(16<<2)|3,{105,55,48}},
/* 13217 */ {(16<<2)|3,{105,55,49}},
/* 13218 */ {(16<<2)|3,{105,55,50}},
/* 13219 */ {(16<<2)|3,{105,55,97}},
/* 13220 */ {(16<<2)|3,{105,55,99}},
/* 13221 */ {(16<<2)|3,{105,55,101}},
/* 13222 */ {(16<<2)|3,{105,55,105}},
/* 13223 */ {(16<<2)|3,{105,55,111}},
/* 13224 */ {(16<<2)|3,{105,55,115}},
/* 13225 */ {(16<<2)|3,{105,55,116}},
/* 13226 */ {(11<<2)|2,{105,55,0}},
/* 13227 */ {(11<<2)|2,{105,55,0}},
/* 13228 */ {(11<<2)|2,{105,55,0}},
/* 13229 */ {(11<<2)|2,{105,55,0}},
/* 13230 */ {(11<<2)|2,{105,55,0}},
/* 13231 */ {(11<<2)|2,{105,55,0}},
/* 13232 */ {(11<<2)|2,{105,55,0}},
/* 13233 */ {(11<<2)|2,{105,55,0}},
/* 13234 */ {(11<<2)|2,{105,55,0}},
/* 13235 */ {(11<<2)|2,{105,55,0}},
/* 13236 */ {(11<<2)|2,{105,55,0}},
/* 13237 */ {(11<<2)|2,{105,55,0}},
/* 13238 */ {(11<<2)|2,{105,55,0}},
/* 13239 */ {(11<<2)|2,{105,55,0}},
/* 13240 */ {(11<<2)|2,{105,55,0}},
/* 13241 */ {(11<<2)|2,{105,55,0}},
/* 13242 */ {(11<<2)|2,{105,55,0}},
/* 13243 */ {(11<<2)|2,{105,55,0}},
/* 13244 */ {(11<<2)|2,{105,55,0}},
/* 13245 */ {(11<<2)|2,{105,55,0}},
/* 13246 */ {(11<<2)|2,{105,55,0}},
/* 13247 */ {(11<<2)|2,{105,55,0}},
/* 13248 */ {(16<<2)|3,{105,56,48}},
/* 13249 */ {(16<<2)|3,{105,56,49}},
/* 13250 */ {(16<<2)|3,{105,56,50}},
/* 13251 */ {(16<<2)|3,{105,56,97}},
/* 13252 */ {(16<<2)|3,{105,56,99}},
/* 13253 */ {(16<<2)|3,{105,56,101}},
/* 13254 */ {(16<<2)|3,{105,56,105}},
/* 13255 */ {(16<<2)|3,{105,56,111}},
/* 13256 */ {(16<<2)|3,{105,56,115}},
/* 13257 */ {(16<<2)|3,{105,56,116}},
/* 13258 */ {(11<<2)|2,{105,56,0}},
/* 13259 */ {(11<<2)|2,{105,56,0}},
/* 13260 */ {(11<<2)|2,{105,56,0}},
/* 13261 */ {(11<<2)|2,{105,56,0}},
/* 13262 */ {(11<<2)|2,{105,56,0}},
/* 13263 */ {(11<<2)|2,{105,56,0}},
/* 13264 */ {(11<<2)|2,{105,56,0}},
/* 13265 */ {(11<<2)|2,{105,56,0}},
/* 13266 */ {(11<<2)|2,{105,56,0}},
/* 13267 */ {(11<<2)|2,{105,56,0}},
/* 13268 */ {(11<<2)|2,{105,56,0}},
/* 13269 */ {(11<<2)|2,{105,56,0}},
/* 13270 */ {(11<<2)|2,{105,56,0}},
/* 13271 */ {(11<<2)|2,{105,56,0}},
/* 13272 */ {(11<<2)|2,{105,56,0}},
/* 13273 */ {(11<<2)|2,{105,56,0}},
/* 13274 */ {(11<<2)|2,{105,56,0}},
/* 13275 */ {(11<<2)|2,{105,56,0}},
/* 13276 */ {(11<<2)|2,{105,56,0}},
/* 13277 */ {(11<<2)|2,{105,56,0}},
/* 13278 */ {(11<<2)|2,{105,56,0}},
/* 13279 */ {(11<<2)|2,{105,56,0}},
/* 13280 */ {(16<<2)|3,{105,57,48}},
/* 13281 */ {(16<<2)|3,{105,57,49}},
/* 13282 */ {(16<<2)|3,{105,57,50}},
/* 13283 */ {(16<<2)|3,{105,57,97}},
/* 13284 */ {(16<<2)|3,{105,57,99}},
/* 13285 */ {(16<<2)|3,{105,57,101}},
/* 13286 */ {(16<<2)|3,{105,57,105}},
/* 13287 */ {(16<<2)|3,{105,57,111}},
/* 13288 */ {(16<<2)|3,{105,57,115}},
/* 13289 */ {(16<<2)|3,{105,57,116}},
/* 13290 */ {(11<<2)|2,{105,57,0}},
/* 13291 */ {(11<<2)|2,{105,57,0}},
/* 13292 */ {(11<<2)|2,{105,57,0}},
/* 13293 */ {(11<<2)|2,{105,57,0}},
/* 13294 */ {(11<<2)|2,{105,57,0}},
/* 13295 */ {(11<<2)|2,{105,57,0}},
/* 13296 */ {(11<<2)|2,{105,57,0}},
/* 13297 */ {(11<<2)|2,{105,57,0}},
/* 13298 */ {(11<<2)|2,{105,57,0}},
/* 13299 */ {(11<<2)|2,{105,57,0}},
/* 13300 */ {(11<<2)|2,{105,57,0}},
/* 13301 */ {(11<<2)|2,{105,57,0}},
/* 13302 */ {(11<<2)|2,{105,57,0}},
/* 13303 */ {(11<<2)|2,{105,57,0}},
/* 13304 */ {(11<<2)|2,{105,57,0}},
/* 13305 */ {(11<<2)|2,{105,57,0}},
/* 13306 */ {(11<<2)|2,{105,57,0}},
/* 13307 */ {(11<<2)|2,{105,57,0}},
/* 13308 */ {(11<<2)|2,{105,57,0}},
/* 13309 */ {(11<<2)|2,{105,57,0}},
/* 13310 */ {(11<<2)|2,{105,57,0}},
/* 13311 */ {(11<<2)|2,{105,57,0}},
/* 13312 */ {(16<<2)|3,{105,61,48}},
/* 13313 */ {(16<<2)|3,{105,61,49}},
/* 13314 */ {(16<<2)|3,{105,61,50}},
/* 13315 */ {(16<<2)|3,{105,61,97}},
/* 13316 */ {(16<<2)|3,{105,61,99}},
/* 13317 */ {(16<<2)|3,{105,61,101}},
/* 13318 */ {(16<<2)|3,{105,61,105}},
/* 13319 */ {(16<<2)|3,{105,61,111}},
/* 13320 */ {(16<<2)|3,{105,61,115}},
/* 13321 */ {(16<<2)|3,{105,61,116}},
/* 13322 */ {(11<<2)|2,{105,61,0}},
/* 13323 */ {(11<<2)|2,{105,61,0}},
/* 13324 */ {(11<<2)|2,{105,61,0}},
/* 13325 */ {(11<<2)|2,{105,61,0}},
/* 13326 */ {(11<<2)|2,{105,61,0}},
/* 13327 */ {(11<<2)|2,{105,61,0}},
/* 13328 */ {(11<<2)|2,{105,61,0}},
/* 13329 */ {(11<<2)|2,{105,61,0}},
/* 13330 */ {(11<<2)|2,{105,61,0}},
/* 13331 */ {(11<<2)|2,{105,61,0}},
/* 13332 */ {(11<<2)|2,{105,61,0}},
/* 13333 */ {(11<<2)|2,{105,61,0}},
/* 13334 */ {(11<<2)|2,{105,61,0}},
/* 13335 */ {(11<<2)|2,{105,61,0}},
/* 13336 */ {(11<<2)|2,{105,61,0}},
/* 13337 */ {(11<<2)|2,{105,61,0}},
/* 13338 */ {(11<<2)|2,{105,61,0}},
/* 13339 */ {(11<<2)|2,{105,61,0}},
/* 13340 */ {(11<<2)|2,{105,61,0}},
/* 13341 */ {(11<<2)|2,{105,61,0}},
/* 13342 */ {(11<<2)|2,{105,61,0}},
/* 13343 */ {(11<<2)|2,{105,61,0}},
/* 13344 */ {(16<<2)|3,{105,65,48}},
/* 13345 */ {(16<<2)|3,{105,65,49}},
/* 13346 */ {(16<<2)|3,{105,65,50}},
/* 13347 */ {(16<<2)|3,{105,65,97}},
/* 13348 */ {(16<<2)|3,{105,65,99}},
/* 13349 */ {(16<<2)|3,{105,65,101}},
/* 13350 */ {(16<<2)|3,{105,65,105}},
/* 13351 */ {(16<<2)|3,{105,65,111}},
/* 13352 */ {(16<<2)|3,{105,65,115}},
/* 13353 */ {(16<<2)|3,{105,65,116}},
/* 13354 */ {(11<<2)|2,{105,65,0}},
/* 13355 */ {(11<<2)|2,{105,65,0}},
/* 13356 */ {(11<<2)|2,{105,65,0}},
/* 13357 */ {(11<<2)|2,{105,65,0}},
/* 13358 */ {(11<<2)|2,{105,65,0}},
/* 13359 */ {(11<<2)|2,{105,65,0}},
/* 13360 */ {(11<<2)|2,{105,65,0}},
/* 13361 */ {(11<<2)|2,{105,65,0}},
/* 13362 */ {(11<<2)|2,{105,65,0}},
/* 13363 */ {(11<<2)|2,{105,65,0}},
/* 13364 */ {(11<<2)|2,{105,65,0}},
/* 13365 */ {(11<<2)|2,{105,65,0}},
/* 13366 */ {(11<<2)|2,{105,65,0}},
/* 13367 */ {(11<<2)|2,{105,65,0}},
/* 13368 */ {(11<<2)|2,{105,65,0}},
/* 13369 */ {(11<<2)|2,{105,65,0}},
/* 13370 */ {(11<<2)|2,{105,65,0}},
/* 13371 */ {(11<<2)|2,{105,65,0}},
/* 13372 */ {(11<<2)|2,{105,65,0}},
/* 13373 */ {(11<<2)|2,{105,65,0}},
/* 13374 */ {(11<<2)|2,{105,65,0}},
/* 13375 */ {(11<<2)|2,{105,65,0}},
/* 13376 */ {(16<<2)|3,{105,95,48}},
/* 13377 */ {(16<<2)|3,{105,95,49}},
/* 13378 */ {(16<<2)|3,{105,95,50}},
/* 13379 */ {(16<<2)|3,{105,95,97}},
/* 13380 */ {(16<<2)|3,{105,95,99}},
/* 13381 */ {(16<<2)|3,{105,95,101}},
/* 13382 */ {(16<<2)|3,{105,95,105}},
/* 13383 */ {(16<<2)|3,{105,95,111}},
/* 13384 */ {(16<<2)|3,{105,95,115}},
/* 13385 */ {(16<<2)|3,{105,95,116}},
/* 13386 */ {(11<<2)|2,{105,95,0}},
/* 13387 */ {(11<<2)|2,{105,95,0}},
/* 13388 */ {(11<<2)|2,{105,95,0}},
/* 13389 */ {(11<<2)|2,{105,95,0}},
/* 13390 */ {(11<<2)|2,{105,95,0}},
/* 13391 */ {(11<<2)|2,{105,95,0}},
/* 13392 */ {(11<<2)|2,{105,95,0}},
/* 13393 */ {(11<<2)|2,{105,95,0}},
/* 13394 */ {(11<<2)|2,{105,95,0}},
/* 13395 */ {(11<<2)|2,{105,95,0}},
/* 13396 */ {(11<<2)|2,{105,95,0}},
/* 13397 */ {(11<<2)|2,{105,95,0}},
/* 13398 */ {(11<<2)|2,{105,95,0}},
/* 13399 */ {(11<<2)|2,{105,95,0}},
/* 13400 */ {(11<<2)|2,{105,95,0}},
/* 13401 */ {(11<<2)|2,{105,95,0}},
/* 13402 */ {(11<<2)|2,{105,95,0}},
/* 13403 */ {(11<<2)|2,{105,95,0}},
/* 13404 */ {(11<<2)|2,{105,95,0}},
/* 13405 */ {(11<<2)|2,{105,95,0}},
/* 13406 */ {(11<<2)|2,{105,95,0}},
/* 13407 */ {(11<<2)|2,{105,95,0}},
/* 13408 */ {(16<<2)|3,{105,98,48}},
/* 13409 */ {(16<<2)|3,{105,98,49}},
/* 13410 */ {(16<<2)|3,{105,98,50}},
/* 13411 */ {(16<<2)|3,{105,98,97}},
/* 13412 */ {(16<<2)|3,{105,98,99}},
/* 13413 */ {(16<<2)|3,{105,98,101}},
/* 13414 */ {(16<<2)|3,{105,98,105}},
/* 13415 */ {(16<<2)|3,{105,98,111}},
/* 13416 */ {(16<<2)|3,{105,98,115}},
/* 13417 */ {(16<<2)|3,{105,98,116}},
/* 13418 */ {(11<<2)|2,{105,98,0}},
/* 13419 */ {(11<<2)|2,{105,98,0}},
/* 13420 */ {(11<<2)|2,{105,98,0}},
/* 13421 */ {(11<<2)|2,{105,98,0}},
/* 13422 */ {(11<<2)|2,{105,98,0}},
/* 13423 */ {(11<<2)|2,{105,98,0}},
/* 13424 */ {(11<<2)|2,{105,98,0}},
/* 13425 */ {(11<<2)|2,{105,98,0}},
/* 13426 */ {(11<<2)|2,{105,98,0}},
/* 13427 */ {(11<<2)|2,{105,98,0}},
/* 13428 */ {(11<<2)|2,{105,98,0}},
/* 13429 */ {(11<<2)|2,{105,98,0}},
/* 13430 */ {(11<<2)|2,{105,98,0}},
/* 13431 */ {(11<<2)|2,{105,98,0}},
/* 13432 */ {(11<<2)|2,{105,98,0}},
/* 13433 */ {(11<<2)|2,{105,98,0}},
/* 13434 */ {(11<<2)|2,{105,98,0}},
/* 13435 */ {(11<<2)|2,{105,98,0}},
/* 13436 */ {(11<<2)|2,{105,98,0}},
/* 13437 */ {(11<<2)|2,{105,98,0}},
/* 13438 */ {(11<<2)|2,{105,98,0}},
/* 13439 */ {(11<<2)|2,{105,98,0}},
/* 13440 */ {(16<<2)|3,{105,100,48}},
/* 13441 */ {(16<<2)|3,{105,100,49}},
/* 13442 */ {(16<<2)|3,{105,100,50}},
/* 13443 */ {(16<<2)|3,{105,100,97}},
/* 13444 */ {(16<<2)|3,{105,100,99}},
/* 13445 */ {(16<<2)|3,{105,100,101}},
/* 13446 */ {(16<<2)|3,{105,100,105}},
/* 13447 */ {(16<<2)|3,{105,100,111}},
/* 13448 */ {(16<<2)|3,{105,100,115}},
/* 13449 */ {(16<<2)|3,{105,100,116}},
/* 13450 */ {(11<<2)|2,{105,100,0}},
/* 13451 */ {(11<<2)|2,{105,100,0}},
/* 13452 */ {(11<<2)|2,{105,100,0}},
/* 13453 */ {(11<<2)|2,{105,100,0}},
/* 13454 */ {(11<<2)|2,{105,100,0}},
/* 13455 */ {(11<<2)|2,{105,100,0}},
/* 13456 */ {(11<<2)|2,{105,100,0}},
/* 13457 */ {(11<<2)|2,{105,100,0}},
/* 13458 */ {(11<<2)|2,{105,100,0}},
/* 13459 */ {(11<<2)|2,{105,100,0}},
/* 13460 */ {(11<<2)|2,{105,100,0}},
/* 13461 */ {(11<<2)|2,{105,100,0}},
/* 13462 */ {(11<<2)|2,{105,100,0}},
/* 13463 */ {(11<<2)|2,{105,100,0}},
/* 13464 */ {(11<<2)|2,{105,100,0}},
/* 13465 */ {(11<<2)|2,{105,100,0}},
/* 13466 */ {(11<<2)|2,{105,100,0}},
/* 13467 */ {(11<<2)|2,{105,100,0}},
/* 13468 */ {(11<<2)|2,{105,100,0}},
/* 13469 */ {(11<<2)|2,{105,100,0}},
/* 13470 */ {(11<<2)|2,{105,100,0}},
/* 13471 */ {(11<<2)|2,{105,100,0}},
/* 13472 */ {(16<<2)|3,{105,102,48}},
/* 13473 */ {(16<<2)|3,{105,102,49}},
/* 13474 */ {(16<<2)|3,{105,102,50}},
/* 13475 */ {(16<<2)|3,{105,102,97}},
/* 13476 */ {(16<<2)|3,{105,102,99}},
/* 13477 */ {(16<<2)|3,{105,102,101}},
/* 13478 */ {(16<<2)|3,{105,102,105}},
/* 13479 */ {(16<<2)|3,{105,102,111}},
/* 13480 */ {(16<<2)|3,{105,102,115}},
/* 13481 */ {(16<<2)|3,{105,102,116}},
/* 13482 */ {(11<<2)|2,{105,102,0}},
/* 13483 */ {(11<<2)|2,{105,102,0}},
/* 13484 */ {(11<<2)|2,{105,102,0}},
/* 13485 */ {(11<<2)|2,{105,102,0}},
/* 13486 */ {(11<<2)|2,{105,102,0}},
/* 13487 */ {(11<<2)|2,{105,102,0}},
/* 13488 */ {(11<<2)|2,{105,102,0}},
/* 13489 */ {(11<<2)|2,{105,102,0}},
/* 13490 */ {(11<<2)|2,{105,102,0}},
/* 13491 */ {(11<<2)|2,{105,102,0}},
/* 13492 */ {(11<<2)|2,{105,102,0}},
/* 13493 */ {(11<<2)|2,{105,102,0}},
/* 13494 */ {(11<<2)|2,{105,102,0}},
/* 13495 */ {(11<<2)|2,{105,102,0}},
/* 13496 */ {(11<<2)|2,{105,102,0}},
/* 13497 */ {(11<<2)|2,{105,102,0}},
/* 13498 */ {(11<<2)|2,{105,102,0}},
/* 13499 */ {(11<<2)|2,{105,102,0}},
/* 13500 */ {(11<<2)|2,{105,102,0}},
/* 13501 */ {(11<<2)|2,{105,102,0}},
/* 13502 */ {(11<<2)|2,{105,102,0}},
/* 13503 */ {(11<<2)|2,{105,102,0}},
/* 13504 */ {(16<<2)|3,{105,103,48}},
/* 13505 */ {(16<<2)|3,{105,103,49}},
/* 13506 */ {(16<<2)|3,{105,103,50}},
/* 13507 */ {(16<<2)|3,{105,103,97}},
/* 13508 */ {(16<<2)|3,{105,103,99}},
/* 13509 */ {(16<<2)|3,{105,103,101}},
/* 13510 */ {(16<<2)|3,{105,103,105}},
/* 13511 */ {(16<<2)|3,{105,103,111}},
/* 13512 */ {(16<<2)|3,{105,103,115}},
/* 13513 */ {(16<<2)|3,{105,103,116}},
/* 13514 */ {(11<<2)|2,{105,103,0}},
/* 13515 */ {(11<<2)|2,{105,103,0}},
/* 13516 */ {(11<<2)|2,{105,103,0}},
/* 13517 */ {(11<<2)|2,{105,103,0}},
/* 13518 */ {(11<<2)|2,{105,103,0}},
/* 13519 */ {(11<<2)|2,{105,103,0}},
/* 13520 */ {(11<<2)|2,{105,103,0}},
/* 13521 */ {(11<<2)|2,{105,103,0}},
/* 13522 */ {(11<<2)|2,{105,103,0}},
/* 13523 */ {(11<<2)|2,{105,103,0}},
/* 13524 */ {(11<<2)|2,{105,103,0}},
/* 13525 */ {(11<<2)|2,{105,103,0}},
/* 13526 */ {(11<<2)|2,{105,103,0}},
/* 13527 */ {(11<<2)|2,{105,103,0}},
/* 13528 */ {(11<<2)|2,{105,103,0}},
/* 13529 */ {(11<<2)|2,{105,103,0}},
/* 13530 */ {(11<<2)|2,{105,103,0}},
/* 13531 */ {(11<<2)|2,{105,103,0}},
/* 13532 */ {(11<<2)|2,{105,103,0}},
/* 13533 */ {(11<<2)|2,{105,103,0}},
/* 13534 */ {(11<<2)|2,{105,103,0}},
/* 13535 */ {(11<<2)|2,{105,103,0}},
/* 13536 */ {(16<<2)|3,{105,104,48}},
/* 13537 */ {(16<<2)|3,{105,104,49}},
/* 13538 */ {(16<<2)|3,{105,104,50}},
/* 13539 */ {(16<<2)|3,{105,104,97}},
/* 13540 */ {(16<<2)|3,{105,104,99}},
/* 13541 */ {(16<<2)|3,{105,104,101}},
/* 13542 */ {(16<<2)|3,{105,104,105}},
/* 13543 */ {(16<<2)|3,{105,104,111}},
/* 13544 */ {(16<<2)|3,{105,104,115}},
/* 13545 */ {(16<<2)|3,{105,104,116}},
/* 13546 */ {(11<<2)|2,{105,104,0}},
/* 13547 */ {(11<<2)|2,{105,104,0}},
/* 13548 */ {(11<<2)|2,{105,104,0}},
/* 13549 */ {(11<<2)|2,{105,104,0}},
/* 13550 */ {(11<<2)|2,{105,104,0}},
/* 13551 */ {(11<<2)|2,{105,104,0}},
/* 13552 */ {(11<<2)|2,{105,104,0}},
/* 13553 */ {(11<<2)|2,{105,104,0}},
/* 13554 */ {(11<<2)|2,{105,104,0}},
/* 13555 */ {(11<<2)|2,{105,104,0}},
/* 13556 */ {(11<<2)|2,{105,104,0}},
/* 13557 */ {(11<<2)|2,{105,104,0}},
/* 13558 */ {(11<<2)|2,{105,104,0}},
/* 13559 */ {(11<<2)|2,{105,104,0}},
/* 13560 */ {(11<<2)|2,{105,104,0}},
/* 13561 */ {(11<<2)|2,{105,104,0}},
/* 13562 */ {(11<<2)|2,{105,104,0}},
/* 13563 */ {(11<<2)|2,{105,104,0}},
/* 13564 */ {(11<<2)|2,{105,104,0}},
/* 13565 */ {(11<<2)|2,{105,104,0}},
/* 13566 */ {(11<<2)|2,{105,104,0}},
/* 13567 */ {(11<<2)|2,{105,104,0}},
/* 13568 */ {(16<<2)|3,{105,108,48}},
/* 13569 */ {(16<<2)|3,{105,108,49}},
/* 13570 */ {(16<<2)|3,{105,108,50}},
/* 13571 */ {(16<<2)|3,{105,108,97}},
/* 13572 */ {(16<<2)|3,{105,108,99}},
/* 13573 */ {(16<<2)|3,{105,108,101}},
/* 13574 */ {(16<<2)|3,{105,108,105}},
/* 13575 */ {(16<<2)|3,{105,108,111}},
/* 13576 */ {(16<<2)|3,{105,108,115}},
/* 13577 */ {(16<<2)|3,{105,108,116}},
/* 13578 */ {(11<<2)|2,{105,108,0}},
/* 13579 */ {(11<<2)|2,{105,108,0}},
/* 13580 */ {(11<<2)|2,{105,108,0}},
/* 13581 */ {(11<<2)|2,{105,108,0}},
/* 13582 */ {(11<<2)|2,{105,108,0}},
/* 13583 */ {(11<<2)|2,{105,108,0}},
/* 13584 */ {(11<<2)|2,{105,108,0}},
/* 13585 */ {(11<<2)|2,{105,108,0}},
/* 13586 */ {(11<<2)|2,{105,108,0}},
/* 13587 */ {(11<<2)|2,{105,108,0}},
/* 13588 */ {(11<<2)|2,{105,108,0}},
/* 13589 */ {(11<<2)|2,{105,108,0}},
/* 13590 */ {(11<<2)|2,{105,108,0}},
/* 13591 */ {(11<<2)|2,{105,108,0}},
/* 13592 */ {(11<<2)|2,{105,108,0}},
/* 13593 */ {(11<<2)|2,{105,108,0}},
/* 13594 */ {(11<<2)|2,{105,108,0}},
/* 13595 */ {(11<<2)|2,{105,108,0}},
/* 13596 */ {(11<<2)|2,{105,108,0}},
/* 13597 */ {(11<<2)|2,{105,108,0}},
/* 13598 */ {(11<<2)|2,{105,108,0}},
/* 13599 */ {(11<<2)|2,{105,108,0}},
/* 13600 */ {(16<<2)|3,{105,109,48}},
/* 13601 */ {(16<<2)|3,{105,109,49}},
/* 13602 */ {(16<<2)|3,{105,109,50}},
/* 13603 */ {(16<<2)|3,{105,109,97}},
/* 13604 */ {(16<<2)|3,{105,109,99}},
/* 13605 */ {(16<<2)|3,{105,109,101}},
/* 13606 */ {(16<<2)|3,{105,109,105}},
/* 13607 */ {(16<<2)|3,{105,109,111}},
/* 13608 */ {(16<<2)|3,{105,109,115}},
/* 13609 */ {(16<<2)|3,{105,109,116}},
/* 13610 */ {(11<<2)|2,{105,109,0}},
/* 13611 */ {(11<<2)|2,{105,109,0}},
/* 13612 */ {(11<<2)|2,{105,109,0}},
/* 13613 */ {(11<<2)|2,{105,109,0}},
/* 13614 */ {(11<<2)|2,{105,109,0}},
/* 13615 */ {(11<<2)|2,{105,109,0}},
/* 13616 */ {(11<<2)|2,{105,109,0}},
/* 13617 */ {(11<<2)|2,{105,109,0}},
/* 13618 */ {(11<<2)|2,{105,109,0}},
/* 13619 */ {(11<<2)|2,{105,109,0}},
/* 13620 */ {(11<<2)|2,{105,109,0}},
/* 13621 */ {(11<<2)|2,{105,109,0}},
/* 13622 */ {(11<<2)|2,{105,109,0}},
/* 13623 */ {(11<<2)|2,{105,109,0}},
/* 13624 */ {(11<<2)|2,{105,109,0}},
/* 13625 */ {(11<<2)|2,{105,109,0}},
/* 13626 */ {(11<<2)|2,{105,109,0}},
/* 13627 */ {(11<<2)|2,{105,109,0}},
/* 13628 */ {(11<<2)|2,{105,109,0}},
/* 13629 */ {(11<<2)|2,{105,109,0}},
/* 13630 */ {(11<<2)|2,{105,109,0}},
/* 13631 */ {(11<<2)|2,{105,109,0}},
/* 13632 */ {(16<<2)|3,{105,110,48}},
/* 13633 */ {(16<<2)|3,{105,110,49}},
/* 13634 */ {(16<<2)|3,{105,110,50}},
/* 13635 */ {(16<<2)|3,{105,110,97}},
/* 13636 */ {(16<<2)|3,{105,110,99}},
/* 13637 */ {(16<<2)|3,{105,110,101}},
/* 13638 */ {(16<<2)|3,{105,110,105}},
/* 13639 */ {(16<<2)|3,{105,110,111}},
/* 13640 */ {(16<<2)|3,{105,110,115}},
/* 13641 */ {(16<<2)|3,{105,110,116}},
/* 13642 */ {(11<<2)|2,{105,110,0}},
/* 13643 */ {(11<<2)|2,{105,110,0}},
/* 13644 */ {(11<<2)|2,{105,110,0}},
/* 13645 */ {(11<<2)|2,{105,110,0}},
/* 13646 */ {(11<<2)|2,{105,110,0}},
/* 13647 */ {(11<<2)|2,{105,110,0}},
/* 13648 */ {(11<<2)|2,{105,110,0}},
/* 13649 */ {(11<<2)|2,{105,110,0}},
/* 13650 */ {(11<<2)|2,{105,110,0}},
/* 13651 */ {(11<<2)|2,{105,110,0}},
/* 13652 */ {(11<<2)|2,{105,110,0}},
/* 13653 */ {(11<<2)|2,{105,110,0}},
/* 13654 */ {(11<<2)|2,{105,110,0}},
/* 13655 */ {(11<<2)|2,{105,110,0}},
/* 13656 */ {(11<<2)|2,{105,110,0}},
/* 13657 */ {(11<<2)|2,{105,110,0}},
/* 13658 */ {(11<<2)|2,{105,110,0}},
/* 13659 */ {(11<<2)|2,{105,110,0}},
/* 13660 */ {(11<<2)|2,{105,110,0}},
/* 13661 */ {(11<<2)|2,{105,110,0}},
/* 13662 */ {(11<<2)|2,{105,110,0}},
/* 13663 */ {(11<<2)|2,{105,110,0}},
/* 13664 */ {(16<<2)|3,{105,112,48}},
/* 13665 */ {(16<<2)|3,{105,112,49}},
/* 13666 */ {(16<<2)|3,{105,112,50}},
/* 13667 */ {(16<<2)|3,{105,112,97}},
/* 13668 */ {(16<<2)|3,{105,112,99}},
/* 13669 */ {(16<<2)|3,{105,112,101}},
/* 13670 */ {(16<<2)|3,{105,112,105}},
/* 13671 */ {(16<<2)|3,{105,112,111}},
/* 13672 */ {(16<<2)|3,{105,112,115}},
/* 13673 */ {(16<<2)|3,{105,112,116}},
/* 13674 */ {(11<<2)|2,{105,112,0}},
/* 13675 */ {(11<<2)|2,{105,112,0}},
/* 13676 */ {(11<<2)|2,{105,112,0}},
/* 13677 */ {(11<<2)|2,{105,112,0}},
/* 13678 */ {(11<<2)|2,{105,112,0}},
/* 13679 */ {(11<<2)|2,{105,112,0}},
/* 13680 */ {(11<<2)|2,{105,112,0}},
/* 13681 */ {(11<<2)|2,{105,112,0}},
/* 13682 */ {(11<<2)|2,{105,112,0}},
/* 13683 */ {(11<<2)|2,{105,112,0}},
/* 13684 */ {(11<<2)|2,{105,112,0}},
/* 13685 */ {(11<<2)|2,{105,112,0}},
/* 13686 */ {(11<<2)|2,{105,112,0}},
/* 13687 */ {(11<<2)|2,{105,112,0}},
/* 13688 */ {(11<<2)|2,{105,112,0}},
/* 13689 */ {(11<<2)|2,{105,112,0}},
/* 13690 */ {(11<<2)|2,{105,112,0}},
/* 13691 */ {(11<<2)|2,{105,112,0}},
/* 13692 */ {(11<<2)|2,{105,112,0}},
/* 13693 */ {(11<<2)|2,{105,112,0}},
/* 13694 */ {(11<<2)|2,{105,112,0}},
/* 13695 */ {(11<<2)|2,{105,112,0}},
/* 13696 */ {(16<<2)|3,{105,114,48}},
/* 13697 */ {(16<<2)|3,{105,114,49}},
/* 13698 */ {(16<<2)|3,{105,114,50}},
/* 13699 */ {(16<<2)|3,{105,114,97}},
/* 13700 */ {(16<<2)|3,{105,114,99}},
/* 13701 */ {(16<<2)|3,{105,114,101}},
/* 13702 */ {(16<<2)|3,{105,114,105}},
/* 13703 */ {(16<<2)|3,{105,114,111}},
/* 13704 */ {(16<<2)|3,{105,114,115}},
/* 13705 */ {(16<<2)|3,{105,114,116}},
/* 13706 */ {(11<<2)|2,{105,114,0}},
/* 13707 */ {(11<<2)|2,{105,114,0}},
/* 13708 */ {(11<<2)|2,{105,114,0}},
/* 13709 */ {(11<<2)|2,{105,114,0}},
/* 13710 */ {(11<<2)|2,{105,114,0}},
/* 13711 */ {(11<<2)|2,{105,114,0}},
/* 13712 */ {(11<<2)|2,{105,114,0}},
/* 13713 */ {(11<<2)|2,{105,114,0}},
/* 13714 */ {(11<<2)|2,{105,114,0}},
/* 13715 */ {(11<<2)|2,{105,114,0}},
/* 13716 */ {(11<<2)|2,{105,114,0}},
/* 13717 */ {(11<<2)|2,{105,114,0}},
/* 13718 */ {(11<<2)|2,{105,114,0}},
/* 13719 */ {(11<<2)|2,{105,114,0}},
/* 13720 */ {(11<<2)|2,{105,114,0}},
/* 13721 */ {(11<<2)|2,{105,114,0}},
/* 13722 */ {(11<<2)|2,{105,114,0}},
/* 13723 */ {(11<<2)|2,{105,114,0}},
/* 13724 */ {(11<<2)|2,{105,114,0}},
/* 13725 */ {(11<<2)|2,{105,114,0}},
/* 13726 */ {(11<<2)|2,{105,114,0}},
/* 13727 */ {(11<<2)|2,{105,114,0}},
/* 13728 */ {(16<<2)|3,{105,117,48}},
/* 13729 */ {(16<<2)|3,{105,117,49}},
/* 13730 */ {(16<<2)|3,{105,117,50}},
/* 13731 */ {(16<<2)|3,{105,117,97}},
/* 13732 */ {(16<<2)|3,{105,117,99}},
/* 13733 */ {(16<<2)|3,{105,117,101}},
/* 13734 */ {(16<<2)|3,{105,117,105}},
/* 13735 */ {(16<<2)|3,{105,117,111}},
/* 13736 */ {(16<<2)|3,{105,117,115}},
/* 13737 */ {(16<<2)|3,{105,117,116}},
/* 13738 */ {(11<<2)|2,{105,117,0}},
/* 13739 */ {(11<<2)|2,{105,117,0}},
/* 13740 */ {(11<<2)|2,{105,117,0}},
/* 13741 */ {(11<<2)|2,{105,117,0}},
/* 13742 */ {(11<<2)|2,{105,117,0}},
/* 13743 */ {(11<<2)|2,{105,117,0}},
/* 13744 */ {(11<<2)|2,{105,117,0}},
/* 13745 */ {(11<<2)|2,{105,117,0}},
/* 13746 */ {(11<<2)|2,{105,117,0}},
/* 13747 */ {(11<<2)|2,{105,117,0}},
/* 13748 */ {(11<<2)|2,{105,117,0}},
/* 13749 */ {(11<<2)|2,{105,117,0}},
/* 13750 */ {(11<<2)|2,{105,117,0}},
/* 13751 */ {(11<<2)|2,{105,117,0}},
/* 13752 */ {(11<<2)|2,{105,117,0}},
/* 13753 */ {(11<<2)|2,{105,117,0}},
/* 13754 */ {(11<<2)|2,{105,117,0}},
/* 13755 */ {(11<<2)|2,{105,117,0}},
/* 13756 */ {(11<<2)|2,{105,117,0}},
/* 13757 */ {(11<<2)|2,{105,117,0}},
/* 13758 */ {(11<<2)|2,{105,117,0}},
/* 13759 */ {(11<<2)|2,{105,117,0}},
/* 13760 */ {(12<<2)|2,{105,58,0}},
/* 13761 */ {(12<<2)|2,{105,58,0}},
/* 13762 */ {(12<<2)|2,{105,58,0}},
/* 13763 */ {(12<<2)|2,{105,58,0}},
/* 13764 */ {(12<<2)|2,{105,58,0}},
/* 13765 */ {(12<<2)|2,{105,58,0}},
/* 13766 */ {(12<<2)|2,{105,58,0}},
/* 13767 */ {(12<<2)|2,{105,58,0}},
/* 13768 */ {(12<<2)|2,{105,58,0}},
/* 13769 */ {(12<<2)|2,{105,58,0}},
/* 13770 */ {(12<<2)|2,{105,58,0}},
/* 13771 */ {(12<<2)|2,{105,58,0}},
/* 13772 */ {(12<<2)|2,{105,58,0}},
/* 13773 */ {(12<<2)|2,{105,58,0}},
/* 13774 */ {(12<<2)|2,{105,58,0}},
/* 13775 */ {(12<<2)|2,{105,58,0}},
/* 13776 */ {(12<<2)|2,{105,66,0}},
/* 13777 */ {(12<<2)|2,{105,66,0}},
/* 13778 */ {(12<<2)|2,{105,66,0}},
/* 13779 */ {(12<<2)|2,{105,66,0}},
/* 13780 */ {(12<<2)|2,{105,66,0}},
/* 13781 */ {(12<<2)|2,{105,66,0}},
/* 13782 */ {(12<<2)|2,{105,66,0}},
/* 13783 */ {(12<<2)|2,{105,66,0}},
/* 13784 */ {(12<<2)|2,{105,66,0}},
/* 13785 */ {(12<<2)|2,{105,66,0}},
/* 13786 */ {(12<<2)|2,{105,66,0}},
/* 13787 */ {(12<<2)|2,{105,66,0}},
/* 13788 */ {(12<<2)|2,{105,66,0}},
/* 13789 */ {(12<<2)|2,{105,66,0}},
/* 13790 */ {(12<<2)|2,{105,66,0}},
/* 13791 */ {(12<<2)|2,{105,66,0}},
/* 13792 */ {(12<<2)|2,{105,67,0}},
/* 13793 */ {(12<<2)|2,{105,67,0}},
/* 13794 */ {(12<<2)|2,{105,67,0}},
/* 13795 */ {(12<<2)|2,{105,67,0}},
/* 13796 */ {(12<<2)|2,{105,67,0}},
/* 13797 */ {(12<<2)|2,{105,67,0}},
/* 13798 */ {(12<<2)|2,{105,67,0}},
/* 13799 */ {(12<<2)|2,{105,67,0}},
/* 13800 */ {(12<<2)|2,{105,67,0}},
/* 13801 */ {(12<<2)|2,{105,67,0}},
/* 13802 */ {(12<<2)|2,{105,67,0}},
/* 13803 */ {(12<<2)|2,{105,67,0}},
/* 13804 */ {(12<<2)|2,{105,67,0}},
/* 13805 */ {(12<<2)|2,{105,67,0}},
/* 13806 */ {(12<<2)|2,{105,67,0}},
/* 13807 */ {(12<<2)|2,{105,67,0}},
/* 13808 */ {(12<<2)|2,{105,68,0}},
/* 13809 */ {(12<<2)|2,{105,68,0}},
/* 13810 */ {(12<<2)|2,{105,68,0}},
/* 13811 */ {(12<<2)|2,{105,68,0}},
/* 13812 */ {(12<<2)|2,{105,68,0}},
/* 13813 */ {(12<<2)|2,{105,68,0}},
/* 13814 */ {(12<<2)|2,{105,68,0}},
/* 13815 */ {(12<<2)|2,{105,68,0}},
/* 13816 */ {(12<<2)|2,{105,68,0}},
/* 13817 */ {(12<<2)|2,{105,68,0}},
/* 13818 */ {(12<<2)|2,{105,68,0}},
/* 13819 */ {(12<<2)|2,{105,68,0}},
/* 13820 */ {(12<<2)|2,{105,68,0}},
/* 13821 */ {(12<<2)|2,{105,68,0}},
/* 13822 */ {(12<<2)|2,{105,68,0}},
/* 13823 */ {(12<<2)|2,{105,68,0}},
/* 13824 */ {(12<<2)|2,{105,69,0}},
/* 13825 */ {(12<<2)|2,{105,69,0}},
/* 13826 */ {(12<<2)|2,{105,69,0}},
/* 13827 */ {(12<<2)|2,{105,69,0}},
/* 13828 */ {(12<<2)|2,{105,69,0}},
/* 13829 */ {(12<<2)|2,{105,69,0}},
/* 13830 */ {(12<<2)|2,{105,69,0}},
/* 13831 */ {(12<<2)|2,{105,69,0}},
/* 13832 */ {(12<<2)|2,{105,69,0}},
/* 13833 */ {(12<<2)|2,{105,69,0}},
/* 13834 */ {(12<<2)|2,{105,69,0}},
/* 13835 */ {(12<<2)|2,{105,69,0}},
/* 13836 */ {(12<<2)|2,{105,69,0}},
/* 13837 */ {(12<<2)|2,{105,69,0}},
/* 13838 */ {(12<<2)|2,{105,69,0}},
/* 13839 */ {(12<<2)|2,{105,69,0}},
/* 13840 */ {(12<<2)|2,{105,70,0}},
/* 13841 */ {(12<<2)|2,{105,70,0}},
/* 13842 */ {(12<<2)|2,{105,70,0}},
/* 13843 */ {(12<<2)|2,{105,70,0}},
/* 13844 */ {(12<<2)|2,{105,70,0}},
/* 13845 */ {(12<<2)|2,{105,70,0}},
/* 13846 */ {(12<<2)|2,{105,70,0}},
/* 13847 */ {(12<<2)|2,{105,70,0}},
/* 13848 */ {(12<<2)|2,{105,70,0}},
/* 13849 */ {(12<<2)|2,{105,70,0}},
/* 13850 */ {(12<<2)|2,{105,70,0}},
/* 13851 */ {(12<<2)|2,{105,70,0}},
/* 13852 */ {(12<<2)|2,{105,70,0}},
/* 13853 */ {(12<<2)|2,{105,70,0}},
/* 13854 */ {(12<<2)|2,{105,70,0}},
/* 13855 */ {(12<<2)|2,{105,70,0}},
/* 13856 */ {(12<<2)|2,{105,71,0}},
/* 13857 */ {(12<<2)|2,{105,71,0}},
/* 13858 */ {(12<<2)|2,{105,71,0}},
/* 13859 */ {(12<<2)|2,{105,71,0}},
/* 13860 */ {(12<<2)|2,{105,71,0}},
/* 13861 */ {(12<<2)|2,{105,71,0}},
/* 13862 */ {(12<<2)|2,{105,71,0}},
/* 13863 */ {(12<<2)|2,{105,71,0}},
/* 13864 */ {(12<<2)|2,{105,71,0}},
/* 13865 */ {(12<<2)|2,{105,71,0}},
/* 13866 */ {(12<<2)|2,{105,71,0}},
/* 13867 */ {(12<<2)|2,{105,71,0}},
/* 13868 */ {(12<<2)|2,{105,71,0}},
/* 13869 */ {(12<<2)|2,{105,71,0}},
/* 13870 */ {(12<<2)|2,{105,71,0}},
/* 13871 */ {(12<<2)|2,{105,71,0}},
/* 13872 */ {(12<<2)|2,{105,72,0}},
/* 13873 */ {(12<<2)|2,{105,72,0}},
/* 13874 */ {(12<<2)|2,{105,72,0}},
/* 13875 */ {(12<<2)|2,{105,72,0}},
/* 13876 */ {(12<<2)|2,{105,72,0}},
/* 13877 */ {(12<<2)|2,{105,72,0}},
/* 13878 */ {(12<<2)|2,{105,72,0}},
/* 13879 */ {(12<<2)|2,{105,72,0}},
/* 13880 */ {(12<<2)|2,{105,72,0}},
/* 13881 */ {(12<<2)|2,{105,72,0}},
/* 13882 */ {(12<<2)|2,{105,72,0}},
/* 13883 */ {(12<<2)|2,{105,72,0}},
/* 13884 */ {(12<<2)|2,{105,72,0}},
/* 13885 */ {(12<<2)|2,{105,72,0}},
/* 13886 */ {(12<<2)|2,{105,72,0}},
/* 13887 */ {(12<<2)|2,{105,72,0}},
/* 13888 */ {(12<<2)|2,{105,73,0}},
/* 13889 */ {(12<<2)|2,{105,73,0}},
/* 13890 */ {(12<<2)|2,{105,73,0}},
/* 13891 */ {(12<<2)|2,{105,73,0}},
/* 13892 */ {(12<<2)|2,{105,73,0}},
/* 13893 */ {(12<<2)|2,{105,73,0}},
/* 13894 */ {(12<<2)|2,{105,73,0}},
/* 13895 */ {(12<<2)|2,{105,73,0}},
/* 13896 */ {(12<<2)|2,{105,73,0}},
/* 13897 */ {(12<<2)|2,{105,73,0}},
/* 13898 */ {(12<<2)|2,{105,73,0}},
/* 13899 */ {(12<<2)|2,{105,73,0}},
/* 13900 */ {(12<<2)|2,{105,73,0}},
/* 13901 */ {(12<<2)|2,{105,73,0}},
/* 13902 */ {(12<<2)|2,{105,73,0}},
/* 13903 */ {(12<<2)|2,{105,73,0}},
/* 13904 */ {(12<<2)|2,{105,74,0}},
/* 13905 */ {(12<<2)|2,{105,74,0}},
/* 13906 */ {(12<<2)|2,{105,74,0}},
/* 13907 */ {(12<<2)|2,{105,74,0}},
/* 13908 */ {(12<<2)|2,{105,74,0}},
/* 13909 */ {(12<<2)|2,{105,74,0}},
/* 13910 */ {(12<<2)|2,{105,74,0}},
/* 13911 */ {(12<<2)|2,{105,74,0}},
/* 13912 */ {(12<<2)|2,{105,74,0}},
/* 13913 */ {(12<<2)|2,{105,74,0}},
/* 13914 */ {(12<<2)|2,{105,74,0}},
/* 13915 */ {(12<<2)|2,{105,74,0}},
/* 13916 */ {(12<<2)|2,{105,74,0}},
/* 13917 */ {(12<<2)|2,{105,74,0}},
/* 13918 */ {(12<<2)|2,{105,74,0}},
/* 13919 */ {(12<<2)|2,{105,74,0}},
/* 13920 */ {(12<<2)|2,{105,75,0}},
/* 13921 */ {(12<<2)|2,{105,75,0}},
/* 13922 */ {(12<<2)|2,{105,75,0}},
/* 13923 */ {(12<<2)|2,{105,75,0}},
/* 13924 */ {(12<<2)|2,{105,75,0}},
/* 13925 */ {(12<<2)|2,{105,75,0}},
/* 13926 */ {(12<<2)|2,{105,75,0}},
/* 13927 */ {(12<<2)|2,{105,75,0}},
/* 13928 */ {(12<<2)|2,{105,75,0}},
/* 13929 */ {(12<<2)|2,{105,75,0}},
/* 13930 */ {(12<<2)|2,{105,75,0}},
/* 13931 */ {(12<<2)|2,{105,75,0}},
/* 13932 */ {(12<<2)|2,{105,75,0}},
/* 13933 */ {(12<<2)|2,{105,75,0}},
/* 13934 */ {(12<<2)|2,{105,75,0}},
/* 13935 */ {(12<<2)|2,{105,75,0}},
/* 13936 */ {(12<<2)|2,{105,76,0}},
/* 13937 */ {(12<<2)|2,{105,76,0}},
/* 13938 */ {(12<<2)|2,{105,76,0}},
/* 13939 */ {(12<<2)|2,{105,76,0}},
/* 13940 */ {(12<<2)|2,{105,76,0}},
/* 13941 */ {(12<<2)|2,{105,76,0}},
/* 13942 */ {(12<<2)|2,{105,76,0}},
/* 13943 */ {(12<<2)|2,{105,76,0}},
/* 13944 */ {(12<<2)|2,{105,76,0}},
/* 13945 */ {(12<<2)|2,{105,76,0}},
/* 13946 */ {(12<<2)|2,{105,76,0}},
/* 13947 */ {(12<<2)|2,{105,76,0}},
/* 13948 */ {(12<<2)|2,{105,76,0}},
/* 13949 */ {(12<<2)|2,{105,76,0}},
/* 13950 */ {(12<<2)|2,{105,76,0}},
/* 13951 */ {(12<<2)|2,{105,76,0}},
/* 13952 */ {(12<<2)|2,{105,77,0}},
/* 13953 */ {(12<<2)|2,{105,77,0}},
/* 13954 */ {(12<<2)|2,{105,77,0}},
/* 13955 */ {(12<<2)|2,{105,77,0}},
/* 13956 */ {(12<<2)|2,{105,77,0}},
/* 13957 */ {(12<<2)|2,{105,77,0}},
/* 13958 */ {(12<<2)|2,{105,77,0}},
/* 13959 */ {(12<<2)|2,{105,77,0}},
/* 13960 */ {(12<<2)|2,{105,77,0}},
/* 13961 */ {(12<<2)|2,{105,77,0}},
/* 13962 */ {(12<<2)|2,{105,77,0}},
/* 13963 */ {(12<<2)|2,{105,77,0}},
/* 13964 */ {(12<<2)|2,{105,77,0}},
/* 13965 */ {(12<<2)|2,{105,77,0}},
/* 13966 */ {(12<<2)|2,{105,77,0}},
/* 13967 */ {(12<<2)|2,{105,77,0}},
/* 13968 */ {(12<<2)|2,{105,78,0}},
/* 13969 */ {(12<<2)|2,{105,78,0}},
/* 13970 */ {(12<<2)|2,{105,78,0}},
/* 13971 */ {(12<<2)|2,{105,78,0}},
/* 13972 */ {(12<<2)|2,{105,78,0}},
/* 13973 */ {(12<<2)|2,{105,78,0}},
/* 13974 */ {(12<<2)|2,{105,78,0}},
/* 13975 */ {(12<<2)|2,{105,78,0}},
/* 13976 */ {(12<<2)|2,{105,78,0}},
/* 13977 */ {(12<<2)|2,{105,78,0}},
/* 13978 */ {(12<<2)|2,{105,78,0}},
/* 13979 */ {(12<<2)|2,{105,78,0}},
/* 13980 */ {(12<<2)|2,{105,78,0}},
/* 13981 */ {(12<<2)|2,{105,78,0}},
/* 13982 */ {(12<<2)|2,{105,78,0}},
/* 13983 */ {(12<<2)|2,{105,78,0}},
/* 13984 */ {(12<<2)|2,{105,79,0}},
/* 13985 */ {(12<<2)|2,{105,79,0}},
/* 13986 */ {(12<<2)|2,{105,79,0}},
/* 13987 */ {(12<<2)|2,{105,79,0}},
/* 13988 */ {(12<<2)|2,{105,79,0}},
/* 13989 */ {(12<<2)|2,{105,79,0}},
/* 13990 */ {(12<<2)|2,{105,79,0}},
/* 13991 */ {(12<<2)|2,{105,79,0}},
/* 13992 */ {(12<<2)|2,{105,79,0}},
/* 13993 */ {(12<<2)|2,{105,79,0}},
/* 13994 */ {(12<<2)|2,{105,79,0}},
/* 13995 */ {(12<<2)|2,{105,79,0}},
/* 13996 */ {(12<<2)|2,{105,79,0}},
/* 13997 */ {(12<<2)|2,{105,79,0}},
/* 13998 */ {(12<<2)|2,{105,79,0}},
/* 13999 */ {(12<<2)|2,{105,79,0}},
/* 14000 */ {(12<<2)|2,{105,80,0}},
/* 14001 */ {(12<<2)|2,{105,80,0}},
/* 14002 */ {(12<<2)|2,{105,80,0}},
/* 14003 */ {(12<<2)|2,{105,80,0}},
/* 14004 */ {(12<<2)|2,{105,80,0}},
/* 14005 */ {(12<<2)|2,{105,80,0}},
/* 14006 */ {(12<<2)|2,{105,80,0}},
/* 14007 */ {(12<<2)|2,{105,80,0}},
/* 14008 */ {(12<<2)|2,{105,80,0}},
/* 14009 */ {(12<<2)|2,{105,80,0}},
/* 14010 */ {(12<<2)|2,{105,80,0}},
/* 14011 */ {(12<<2)|2,{105,80,0}},
/* 14012 */ {(12<<2)|2,{105,80,0}},
/* 14013 */ {(12<<2)|2,{105,80,0}},
/* 14014 */ {(12<<2)|2,{105,80,0}},
/* 14015 */ {(12<<2)|2,{105,80,0}},
/* 14016 */ {(12<<2)|2,{105,81,0}},
/* 14017 */ {(12<<2)|2,{105,81,0}},
/* 14018 */ {(12<<2)|2,{105,81,0}},
/* 14019 */ {(12<<2)|2,{105,81,0}},
/* 14020 */ {(12<<2)|2,{105,81,0}},
/* 14021 */ {(12<<2)|2,{105,81,0}},
/* 14022 */ {(12<<2)|2,{105,81,0}},
/* 14023 */ {(12<<2)|2,{105,81,0}},
/* 14024 */ {(12<<2)|2,{105,81,0}},
/* 14025 */ {(12<<2)|2,{105,81,0}},
/* 14026 */ {(12<<2)|2,{105,81,0}},
/* 14027 */ {(12<<2)|2,{105,81,0}},
/* 14028 */ {(12<<2)|2,{105,81,0}},
/* 14029 */ {(12<<2)|2,{105,81,0}},
/* 14030 */ {(12<<2)|2,{105,81,0}},
/* 14031 */ {(12<<2)|2,{105,81,0}},
/* 14032 */ {(12<<2)|2,{105,82,0}},
/* 14033 */ {(12<<2)|2,{105,82,0}},
/* 14034 */ {(12<<2)|2,{105,82,0}},
/* 14035 */ {(12<<2)|2,{105,82,0}},
/* 14036 */ {(12<<2)|2,{105,82,0}},
/* 14037 */ {(12<<2)|2,{105,82,0}},
/* 14038 */ {(12<<2)|2,{105,82,0}},
/* 14039 */ {(12<<2)|2,{105,82,0}},
/* 14040 */ {(12<<2)|2,{105,82,0}},
/* 14041 */ {(12<<2)|2,{105,82,0}},
/* 14042 */ {(12<<2)|2,{105,82,0}},
/* 14043 */ {(12<<2)|2,{105,82,0}},
/* 14044 */ {(12<<2)|2,{105,82,0}},
/* 14045 */ {(12<<2)|2,{105,82,0}},
/* 14046 */ {(12<<2)|2,{105,82,0}},
/* 14047 */ {(12<<2)|2,{105,82,0}},
/* 14048 */ {(12<<2)|2,{105,83,0}},
/* 14049 */ {(12<<2)|2,{105,83,0}},
/* 14050 */ {(12<<2)|2,{105,83,0}},
/* 14051 */ {(12<<2)|2,{105,83,0}},
/* 14052 */ {(12<<2)|2,{105,83,0}},
/* 14053 */ {(12<<2)|2,{105,83,0}},
/* 14054 */ {(12<<2)|2,{105,83,0}},
/* 14055 */ {(12<<2)|2,{105,83,0}},
/* 14056 */ {(12<<2)|2,{105,83,0}},
/* 14057 */ {(12<<2)|2,{105,83,0}},
/* 14058 */ {(12<<2)|2,{105,83,0}},
/* 14059 */ {(12<<2)|2,{105,83,0}},
/* 14060 */ {(12<<2)|2,{105,83,0}},
/* 14061 */ {(12<<2)|2,{105,83,0}},
/* 14062 */ {(12<<2)|2,{105,83,0}},
/* 14063 */ {(12<<2)|2,{105,83,0}},
/* 14064 */ {(12<<2)|2,{105,84,0}},
/* 14065 */ {(12<<2)|2,{105,84,0}},
/* 14066 */ {(12<<2)|2,{105,84,0}},
/* 14067 */ {(12<<2)|2,{105,84,0}},
/* 14068 */ {(12<<2)|2,{105,84,0}},
/* 14069 */ {(12<<2)|2,{105,84,0}},
/* 14070 */ {(12<<2)|2,{105,84,0}},
/* 14071 */ {(12<<2)|2,{105,84,0}},
/* 14072 */ {(12<<2)|2,{105,84,0}},
/* 14073 */ {(12<<2)|2,{105,84,0}},
/* 14074 */ {(12<<2)|2,{105,84,0}},
/* 14075 */ {(12<<2)|2,{105,84,0}},
/* 14076 */ {(12<<2)|2,{105,84,0}},
/* 14077 */ {(12<<2)|2,{105,84,0}},
/* 14078 */ {(12<<2)|2,{105,84,0}},
/* 14079 */ {(12<<2)|2,{105,84,0}},
/* 14080 */ {(12<<2)|2,{105,85,0}},
/* 14081 */ {(12<<2)|2,{105,85,0}},
/* 14082 */ {(12<<2)|2,{105,85,0}},
/* 14083 */ {(12<<2)|2,{105,85,0}},
/* 14084 */ {(12<<2)|2,{105,85,0}},
/* 14085 */ {(12<<2)|2,{105,85,0}},
/* 14086 */ {(12<<2)|2,{105,85,0}},
/* 14087 */ {(12<<2)|2,{105,85,0}},
/* 14088 */ {(12<<2)|2,{105,85,0}},
/* 14089 */ {(12<<2)|2,{105,85,0}},
/* 14090 */ {(12<<2)|2,{105,85,0}},
/* 14091 */ {(12<<2)|2,{105,85,0}},
/* 14092 */ {(12<<2)|2,{105,85,0}},
/* 14093 */ {(12<<2)|2,{105,85,0}},
/* 14094 */ {(12<<2)|2,{105,85,0}},
/* 14095 */ {(12<<2)|2,{105,85,0}},
/* 14096 */ {(12<<2)|2,{105,86,0}},
/* 14097 */ {(12<<2)|2,{105,86,0}},
/* 14098 */ {(12<<2)|2,{105,86,0}},
/* 14099 */ {(12<<2)|2,{105,86,0}},
/* 14100 */ {(12<<2)|2,{105,86,0}},
/* 14101 */ {(12<<2)|2,{105,86,0}},
/* 14102 */ {(12<<2)|2,{105,86,0}},
/* 14103 */ {(12<<2)|2,{105,86,0}},
/* 14104 */ {(12<<2)|2,{105,86,0}},
/* 14105 */ {(12<<2)|2,{105,86,0}},
/* 14106 */ {(12<<2)|2,{105,86,0}},
/* 14107 */ {(12<<2)|2,{105,86,0}},
/* 14108 */ {(12<<2)|2,{105,86,0}},
/* 14109 */ {(12<<2)|2,{105,86,0}},
/* 14110 */ {(12<<2)|2,{105,86,0}},
/* 14111 */ {(12<<2)|2,{105,86,0}},
/* 14112 */ {(12<<2)|2,{105,87,0}},
/* 14113 */ {(12<<2)|2,{105,87,0}},
/* 14114 */ {(12<<2)|2,{105,87,0}},
/* 14115 */ {(12<<2)|2,{105,87,0}},
/* 14116 */ {(12<<2)|2,{105,87,0}},
/* 14117 */ {(12<<2)|2,{105,87,0}},
/* 14118 */ {(12<<2)|2,{105,87,0}},
/* 14119 */ {(12<<2)|2,{105,87,0}},
/* 14120 */ {(12<<2)|2,{105,87,0}},
/* 14121 */ {(12<<2)|2,{105,87,0}},
/* 14122 */ {(12<<2)|2,{105,87,0}},
/* 14123 */ {(12<<2)|2,{105,87,0}},
/* 14124 */ {(12<<2)|2,{105,87,0}},
/* 14125 */ {(12<<2)|2,{105,87,0}},
/* 14126 */ {(12<<2)|2,{105,87,0}},
/* 14127 */ {(12<<2)|2,{105,87,0}},
/* 14128 */ {(12<<2)|2,{105,89,0}},
/* 14129 */ {(12<<2)|2,{105,89,0}},
/* 14130 */ {(12<<2)|2,{105,89,0}},
/* 14131 */ {(12<<2)|2,{105,89,0}},
/* 14132 */ {(12<<2)|2,{105,89,0}},
/* 14133 */ {(12<<2)|2,{105,89,0}},
/* 14134 */ {(12<<2)|2,{105,89,0}},
/* 14135 */ {(12<<2)|2,{105,89,0}},
/* 14136 */ {(12<<2)|2,{105,89,0}},
/* 14137 */ {(12<<2)|2,{105,89,0}},
/* 14138 */ {(12<<2)|2,{105,89,0}},
/* 14139 */ {(12<<2)|2,{105,89,0}},
/* 14140 */ {(12<<2)|2,{105,89,0}},
/* 14141 */ {(12<<2)|2,{105,89,0}},
/* 14142 */ {(12<<2)|2,{105,89,0}},
/* 14143 */ {(12<<2)|2,{105,89,0}},
/* 14144 */ {(12<<2)|2,{105,106,0}},
/* 14145 */ {(12<<2)|2,{105,106,0}},
/* 14146 */ {(12<<2)|2,{105,106,0}},
/* 14147 */ {(12<<2)|2,{105,106,0}},
/* 14148 */ {(12<<2)|2,{105,106,0}},
/* 14149 */ {(12<<2)|2,{105,106,0}},
/* 14150 */ {(12<<2)|2,{105,106,0}},
/* 14151 */ {(12<<2)|2,{105,106,0}},
/* 14152 */ {(12<<2)|2,{105,106,0}},
/* 14153 */ {(12<<2)|2,{105,106,0}},
/* 14154 */ {(12<<2)|2,{105,106,0}},
/* 14155 */ {(12<<2)|2,{105,106,0}},
/* 14156 */ {(12<<2)|2,{105,106,0}},
/* 14157 */ {(12<<2)|2,{105,106,0}},
/* 14158 */ {(12<<2)|2,{105,106,0}},
/* 14159 */ {(12<<2)|2,{105,106,0}},
/* 14160 */ {(12<<2)|2,{105,107,0}},
/* 14161 */ {(12<<2)|2,{105,107,0}},
/* 14162 */ {(12<<2)|2,{105,107,0}},
/* 14163 */ {(12<<2)|2,{105,107,0}},
/* 14164 */ {(12<<2)|2,{105,107,0}},
/* 14165 */ {(12<<2)|2,{105,107,0}},
/* 14166 */ {(12<<2)|2,{105,107,0}},
/* 14167 */ {(12<<2)|2,{105,107,0}},
/* 14168 */ {(12<<2)|2,{105,107,0}},
/* 14169 */ {(12<<2)|2,{105,107,0}},
/* 14170 */ {(12<<2)|2,{105,107,0}},
/* 14171 */ {(12<<2)|2,{105,107,0}},
/* 14172 */ {(12<<2)|2,{105,107,0}},
/* 14173 */ {(12<<2)|2,{105,107,0}},
/* 14174 */ {(12<<2)|2,{105,107,0}},
/* 14175 */ {(12<<2)|2,{105,107,0}},
/* 14176 */ {(12<<2)|2,{105,113,0}},
/* 14177 */ {(12<<2)|2,{105,113,0}},
/* 14178 */ {(12<<2)|2,{105,113,0}},
/* 14179 */ {(12<<2)|2,{105,113,0}},
/* 14180 */ {(12<<2)|2,{105,113,0}},
/* 14181 */ {(12<<2)|2,{105,113,0}},
/* 14182 */ {(12<<2)|2,{105,113,0}},
/* 14183 */ {(12<<2)|2,{105,113,0}},
/* 14184 */ {(12<<2)|2,{105,113,0}},
/* 14185 */ {(12<<2)|2,{105,113,0}},
/* 14186 */ {(12<<2)|2,{105,113,0}},
/* 14187 */ {(12<<2)|2,{105,113,0}},
/* 14188 */ {(12<<2)|2,{105,113,0}},
/* 14189 */ {(12<<2)|2,{105,113,0}},
/* 14190 */ {(12<<2)|2,{105,113,0}},
/* 14191 */ {(12<<2)|2,{105,113,0}},
/* 14192 */ {(12<<2)|2,{105,118,0}},
/* 14193 */ {(12<<2)|2,{105,118,0}},
/* 14194 */ {(12<<2)|2,{105,118,0}},
/* 14195 */ {(12<<2)|2,{105,118,0}},
/* 14196 */ {(12<<2)|2,{105,118,0}},
/* 14197 */ {(12<<2)|2,{105,118,0}},
/* 14198 */ {(12<<2)|2,{105,118,0}},
/* 14199 */ {(12<<2)|2,{105,118,0}},
/* 14200 */ {(12<<2)|2,{105,118,0}},
/* 14201 */ {(12<<2)|2,{105,118,0}},
/* 14202 */ {(12<<2)|2,{105,118,0}},
/* 14203 */ {(12<<2)|2,{105,118,0}},
/* 14204 */ {(12<<2)|2,{105,118,0}},
/* 14205 */ {(12<<2)|2,{105,118,0}},
/* 14206 */ {(12<<2)|2,{105,118,0}},
/* 14207 */ {(12<<2)|2,{105,118,0}},
/* 14208 */ {(12<<2)|2,{105,119,0}},
/* 14209 */ {(12<<2)|2,{105,119,0}},
/* 14210 */ {(12<<2)|2,{105,119,0}},
/* 14211 */ {(12<<2)|2,{105,119,0}},
/* 14212 */ {(12<<2)|2,{105,119,0}},
/* 14213 */ {(12<<2)|2,{105,119,0}},
/* 14214 */ {(12<<2)|2,{105,119,0}},
/* 14215 */ {(12<<2)|2,{105,119,0}},
/* 14216 */ {(12<<2)|2,{105,119,0}},
/* 14217 */ {(12<<2)|2,{105,119,0}},
/* 14218 */ {(12<<2)|2,{105,119,0}},
/* 14219 */ {(12<<2)|2,{105,119,0}},
/* 14220 */ {(12<<2)|2,{105,119,0}},
/* 14221 */ {(12<<2)|2,{105,119,0}},
/* 14222 */ {(12<<2)|2,{105,119,0}},
/* 14223 */ {(12<<2)|2,{105,119,0}},
/* 14224 */ {(12<<2)|2,{105,120,0}},
/* 14225 */ {(12<<2)|2,{105,120,0}},
/* 14226 */ {(12<<2)|2,{105,120,0}},
/* 14227 */ {(12<<2)|2,{105,120,0}},
/* 14228 */ {(12<<2)|2,{105,120,0}},
/* 14229 */ {(12<<2)|2,{105,120,0}},
/* 14230 */ {(12<<2)|2,{105,120,0}},
/* 14231 */ {(12<<2)|2,{105,120,0}},
/* 14232 */ {(12<<2)|2,{105,120,0}},
/* 14233 */ {(12<<2)|2,{105,120,0}},
/* 14234 */ {(12<<2)|2,{105,120,0}},
/* 14235 */ {(12<<2)|2,{105,120,0}},
/* 14236 */ {(12<<2)|2,{105,120,0}},
/* 14237 */ {(12<<2)|2,{105,120,0}},
/* 14238 */ {(12<<2)|2,{105,120,0}},
/* 14239 */ {(12<<2)|2,{105,120,0}},
/* 14240 */ {(12<<2)|2,{105,121,0}},
/* 14241 */ {(12<<2)|2,{105,121,0}},
/* 14242 */ {(12<<2)|2,{105,121,0}},
/* 14243 */ {(12<<2)|2,{105,121,0}},
/* 14244 */ {(12<<2)|2,{105,121,0}},
/* 14245 */ {(12<<2)|2,{105,121,0}},
/* 14246 */ {(12<<2)|2,{105,121,0}},
/* 14247 */ {(12<<2)|2,{105,121,0}},
/* 14248 */ {(12<<2)|2,{105,121,0}},
/* 14249 */ {(12<<2)|2,{105,121,0}},
/* 14250 */ {(12<<2)|2,{105,121,0}},
/* 14251 */ {(12<<2)|2,{105,121,0}},
/* 14252 */ {(12<<2)|2,{105,121,0}},
/* 14253 */ {(12<<2)|2,{105,121,0}},
/* 14254 */ {(12<<2)|2,{105,121,0}},
/* 14255 */ {(12<<2)|2,{105,121,0}},
/* 14256 */ {(12<<2)|2,{105,122,0}},
/* 14257 */ {(12<<2)|2,{105,122,0}},
/* 14258 */ {(12<<2)|2,{105,122,0}},
/* 14259 */ {(12<<2)|2,{105,122,0}},
/* 14260 */ {(12<<2)|2,{105,122,0}},
/* 14261 */ {(12<<2)|2,{105,122,0}},
/* 14262 */ {(12<<2)|2,{105,122,0}},
/* 14263 */ {(12<<2)|2,{105,122,0}},
/* 14264 */ {(12<<2)|2,{105,122,0}},
/* 14265 */ {(12<<2)|2,{105,122,0}},
/* 14266 */ {(12<<2)|2,{105,122,0}},
/* 14267 */ {(12<<2)|2,{105,122,0}},
/* 14268 */ {(12<<2)|2,{105,122,0}},
/* 14269 */ {(12<<2)|2,{105,122,0}},
/* 14270 */ {(12<<2)|2,{105,122,0}},
/* 14271 */ {(12<<2)|2,{105,122,0}},
/* 14272 */ {(13<<2)|2,{105,38,0}},
/* 14273 */ {(13<<2)|2,{105,38,0}},
/* 14274 */ {(13<<2)|2,{105,38,0}},
/* 14275 */ {(13<<2)|2,{105,38,0}},
/* 14276 */ {(13<<2)|2,{105,38,0}},
/* 14277 */ {(13<<2)|2,{105,38,0}},
/* 14278 */ {(13<<2)|2,{105,38,0}},
/* 14279 */ {(13<<2)|2,{105,38,0}},
/* 14280 */ {(13<<2)|2,{105,42,0}},
/* 14281 */ {(13<<2)|2,{105,42,0}},
/* 14282 */ {(13<<2)|2,{105,42,0}},
/* 14283 */ {(13<<2)|2,{105,42,0}},
/* 14284 */ {(13<<2)|2,{105,42,0}},
/* 14285 */ {(13<<2)|2,{105,42,0}},
/* 14286 */ {(13<<2)|2,{105,42,0}},
/* 14287 */ {(13<<2)|2,{105,42,0}},
/* 14288 */ {(13<<2)|2,{105,44,0}},
/* 14289 */ {(13<<2)|2,{105,44,0}},
/* 14290 */ {(13<<2)|2,{105,44,0}},
/* 14291 */ {(13<<2)|2,{105,44,0}},
/* 14292 */ {(13<<2)|2,{105,44,0}},
/* 14293 */ {(13<<2)|2,{105,44,0}},
/* 14294 */ {(13<<2)|2,{105,44,0}},
/* 14295 */ {(13<<2)|2,{105,44,0}},
/* 14296 */ {(13<<2)|2,{105,59,0}},
/* 14297 */ {(13<<2)|2,{105,59,0}},
/* 14298 */ {(13<<2)|2,{105,59,0}},
/* 14299 */ {(13<<2)|2,{105,59,0}},
/* 14300 */ {(13<<2)|2,{105,59,0}},
/* 14301 */ {(13<<2)|2,{105,59,0}},
/* 14302 */ {(13<<2)|2,{105,59,0}},
/* 14303 */ {(13<<2)|2,{105,59,0}},
/* 14304 */ {(13<<2)|2,{105,88,0}},
/* 14305 */ {(13<<2)|2,{105,88,0}},
/* 14306 */ {(13<<2)|2,{105,88,0}},
/* 14307 */ {(13<<2)|2,{105,88,0}},
/* 14308 */ {(13<<2)|2,{105,88,0}},
/* 14309 */ {(13<<2)|2,{105,88,0}},
/* 14310 */ {(13<<2)|2,{105,88,0}},
/* 14311 */ {(13<<2)|2,{105,88,0}},
/* 14312 */ {(13<<2)|2,{105,90,0}},
/* 14313 */ {(13<<2)|2,{105,90,0}},
/* 14314 */ {(13<<2)|2,{105,90,0}},
/* 14315 */ {(13<<2)|2,{105,90,0}},
/* 14316 */ {(13<<2)|2,{105,90,0}},
/* 14317 */ {(13<<2)|2,{105,90,0}},
/* 14318 */ {(13<<2)|2,{105,90,0}},
/* 14319 */ {(13<<2)|2,{105,90,0}},
/* 14320 */ {(15<<2)|2,{105,33,0}},
/* 14321 */ {(15<<2)|2,{105,33,0}},
/* 14322 */ {(15<<2)|2,{105,34,0}},
/* 14323 */ {(15<<2)|2,{105,34,0}},
/* 14324 */ {(15<<2)|2,{105,40,0}},
/* 14325 */ {(15<<2)|2,{105,40,0}},
/* 14326 */ {(15<<2)|2,{105,41,0}},
/* 14327 */ {(15<<2)|2,{105,41,0}},
/* 14328 */ {(15<<2)|2,{105,63,0}},
/* 14329 */ {(15<<2)|2,{105,63,0}},
/* 14330 */ {(16<<2)|2,{105,39,0}},
/* 14331 */ {(16<<2)|2,{105,43,0}},
/* 14332 */ {(16<<2)|2,{105,124,0}},
/* 14333 */ {(5<<2)|1,{105,0,0}},
/* 14334 */ {(5<<2)|1,{105,0,0}},
/* 14335 */ {(5<<2)|1,{105,0,0}},
/* 14336 */ {(15<<2)|3,{111,48,48}},
/* 14337 */ {(15<<2)|3,{111,48,48}},
/* 14338 */ {(15<<2)|3,{111,48,49}},
/* 14339 */ {(15<<2)|3,{111,48,49}},
/* 14340 */ {(15<<2)|3,{111,48,50}},
/* 14341 */ {(15<<2)|3,{111,48,50}},
/* 14342 */ {(15<<2)|3,{111,48,97}},
/* 14343 */ {(15<<2)|3,{111,48,97}},
/* 14344 */ {(15<<2)|3,{111,48,99}},
/* 14345 */ {(15<<2)|3,{111,48,99}},
/* 14346 */ {(15<<2)|3,{111,48,101}},
/* 14347 */ {(15<<2)|3,{111,48,101}},
/* 14348 */ {(15<<2)|3,{111,48,105}},
/* 14349 */ {(15<<2)|3,{111,48,105}},
/* 14350 */ {(15<<2)|3,{111,48,111}},
/* 14351 */ {(15<<2)|3,{111,48,111}},
/* 14352 */ {(15<<2)|3,{111,48,115}},
/* 14353 */ {(15<<2)|3,{111,48,115}},
/* 14354 */ {(15<<2)|3,{111,48,116}},
/* 14355 */ {(15<<2)|3,{111,48,116}},
/* 14356 */ {(16<<2)|3,{111,48,32}},
/* 14357 */ {(16<<2)|3,{111,48,37}},
/* 14358 */ {(16<<2)|3,{111,48,45}},
/* 14359 */ {(16<<2)|3,{111,48,46}},
/* 14360 */ {(16<<2)|3,{111,48,47}},
/* 14361 */ {(16<<2)|3,{111,48,51}},
/* 14362 */ {(16<<2)|3,{111,48,52}},
/* 14363 */ {(16<<2)|3,{111,48,53}},
/* 14364 */ {(16<<2)|3,{111,48,54}},
/* 14365 */ {(16<<2)|3,{111,48,55}},
/* 14366 */ {(16<<2)|3,{111,48,56}},
/* 14367 */ {(16<<2)|3,{111,48,57}},
/* 14368 */ {(16<<2)|3,{111,48,61}},
/* 14369 */ {(16<<2)|3,{111,48,65}},
/* 14370 */ {(16<<2)|3,{111,48,95}},
/* 14371 */ {(16<<2)|3,{111,48,98}},
/* 14372 */ {(16<<2)|3,{111,48,100}},
/* 14373 */ {(16<<2)|3,{111,48,102}},
/* 14374 */ {(16<<2)|3,{111,48,103}},
/* 14375 */ {(16<<2)|3,{111,48,104}},
/* 14376 */ {(16<<2)|3,{111,48,108}},
/* 14377 */ {(16<<2)|3,{111,48,109}},
/* 14378 */ {(16<<2)|3,{111,48,110}},
/* 14379 */ {(16<<2)|3,{111,48,112}},
/* 14380 */ {(16<<2)|3,{111,48,114}},
/* 14381 */ {(16<<2)|3,{111,48,117}},
/* 14382 */ {(10<<2)|2,{111,48,0}},
/* 14383 */ {(10<<2)|2,{111,48,0}},
/* 14384 */ {(10<<2)|2,{111,48,0}},
/* 14385 */ {(10<<2)|2,{111,48,0}},
/* 14386 */ {(10<<2)|2,{111,48,0}},
/* 14387 */ {(10<<2)|2,{111,48,0}},
/* 14388 */ {(10<<2)|2,{111,48,0}},
/* 14389 */ {(10<<2)|2,{111,48,0}},
/* 14390 */ {(10<<2)|2,{111,48,0}},
/* 14391 */ {(10<<2)|2,{111,48,0}},
/* 14392 */ {(10<<2)|2,{111,48,0}},
/* 14393 */ {(10<<2)|2,{111,48,0}},
/* 14394 */ {(10<<2)|2,{111,48,0}},
/* 14395 */ {(10<<2)|2,{111,48,0}},
/* 14396 */ {(10<<2)|2,{111,48,0}},
/* 14397 */ {(10<<2)|2,{111,48,0}},
/* 14398 */ {(10<<2)|2,{111,48,0}},
/* 14399 */ {(10<<2)|2,{111,48,0}},
/* 14400 */ {(15<<2)|3,{111,49,48}},
/* 14401 */ {(15<<2)|3,{111,49,48}},
/* 14402 */ {(15<<2)|3,{111,49,49}},
/* 14403 */ {(15<<2)|3,{111,49,49}},
/* 14404 */ {(15<<2)|3,{111,49,50}},
/* 14405 */ {(15<<2)|3,{111,49,50}},
/* 14406 */ {(15<<2)|3,{111,49,97}},
/* 14407 */ {(15<<2)|3,{111,49,97}},
/* 14408 */ {(15<<2)|3,{111,49,99}},
/* 14409 */ {(15<<2)|3,{111,49,99}},
/* 14410 */ {(15<<2)|3,{111,49,101}},
/* 14411 */ {(15<<2)|3,{111,49,101}},
/* 14412 */ {(15<<2)|3,{111,49,105}},
/* 14413 */ {(15<<2)|3,{111,49,105}},
/* 14414 */ {(15<<2)|3,{111,49,111}},
/* 14415 */ {(15<<2)|3,{111,49,111}},
/* 14416 */ {(15<<2)|3,{111,49,115}},
/* 14417 */ {(15<<2)|3,{111,49,115}},
/* 14418 */ {(15<<2)|3,{111,49,116}},
/* 14419 */ {(15<<2)|3,{111,49,116}},
/* 14420 */ {(16<<2)|3,{111,49,32}},
/* 14421 */ {(16<<2)|3,{111,49,37}},
/* 14422 */ {(16<<2)|3,{111,49,45}},
/* 14423 */ {(16<<2)|3,{111,49,46}},
/* 14424 */ {(16<<2)|3,{111,49,47}},
/* 14425 */ {(16<<2)|3,{111,49,51}},
/* 14426 */ {(16<<2)|3,{111,49,52}},
/* 14427 */ {(16<<2)|3,{111,49,53}},
/* 14428 */ {(16<<2)|3,{111,49,54}},
/* 14429 */ {(16<<2)|3,{111,49,55}},
/* 14430 */ {(16<<2)|3,{111,49,56}},
/* 14431 */ {(16<<2)|3,{111,49,57}},
/* 14432 */ {(16<<2)|3,{111,49,61}},
/* 14433 */ {(16<<2)|3,{111,49,65}},
/* 14434 */ {(16<<2)|3,{111,49,95}},
/* 14435 */ {(16<<2)|3,{111,49,98}},
/* 14436 */ {(16<<2)|3,{111,49,100}},
/* 14437 */ {(16<<2)|3,{111,49,102}},
/* 14438 */ {(16<<2)|3,{111,49,103}},
/* 14439 */ {(16<<2)|3,{111,49,104}},
/* 14440 */ {(16<<2)|3,{111,49,108}},
/* 14441 */ {(16<<2)|3,{111,49,109}},
/* 14442 */ {(16<<2)|3,{111,49,110}},
/* 14443 */ {(16<<2)|3,{111,49,112}},
/* 14444 */ {(16<<2)|3,{111,49,114}},
/* 14445 */ {(16<<2)|3,{111,49,117}},
/* 14446 */ {(10<<2)|2,{111,49,0}},
/* 14447 */ {(10<<2)|2,{111,49,0}},
/* 14448 */ {(10<<2)|2,{111,49,0}},
/* 14449 */ {(10<<2)|2,{111,49,0}},
/* 14450 */ {(10<<2)|2,{111,49,0}},
/* 14451 */ {(10<<2)|2,{111,49,0}},
/* 14452 */ {(10<<2)|2,{111,49,0}},
/* 14453 */ {(10<<2)|2,{111,49,0}},
/* 14454 */ {(10<<2)|2,{111,49,0}},
/* 14455 */ {(10<<2)|2,{111,49,0}},
/* 14456 */ {(10<<2)|2,{111,49,0}},
/* 14457 */ {(10<<2)|2,{111,49,0}},
/* 14458 */ {(10<<2)|2,{111,49,0}},
/* 14459 */ {(10<<2)|2,{111,49,0}},
/* 14460 */ {(10<<2)|2,{111,49,0}},
/* 14461 */ {(10<<2)|2,{111,49,0}},
/* 14462 */ {(10<<2)|2,{111,49,0}},
/* 14463 */ {(10<<2)|2,{111,49,0}},
/* 14464 */ {(15<<2)|3,{111,50,48}},
/* 14465 */ {(15<<2)|3,{111,50,48}},
/* 14466 */ {(15<<2)|3,{111,50,49}},
/* 14467 */ {(15<<2)|3,{111,50,49}},
/* 14468 */ {(15<<2)|3,{111,50,50}},
/* 14469 */ {(15<<2)|3,{111,50,50}},
/* 14470 */ {(15<<2)|3,{111,50,97}},
/* 14471 */ {(15<<2)|3,{111,50,97}},
/* 14472 */ {(15<<2)|3,{111,50,99}},
/* 14473 */ {(15<<2)|3,{111,50,99}},
/* 14474 */ {(15<<2)|3,{111,50,101}},
/* 14475 */ {(15<<2)|3,{111,50,101}},
/* 14476 */ {(15<<2)|3,{111,50,105}},
/* 14477 */ {(15<<2)|3,{111,50,105}},
/* 14478 */ {(15<<2)|3,{111,50,111}},
/* 14479 */ {(15<<2)|3,{111,50,111}},
/* 14480 */ {(15<<2)|3,{111,50,115}},
/* 14481 */ {(15<<2)|3,{111,50,115}},
/* 14482 */ {(15<<2)|3,{111,50,116}},
/* 14483 */ {(15<<2)|3,{111,50,116}},
/* 14484 */ {(16<<2)|3,{111,50,32}},
/* 14485 */ {(16<<2)|3,{111,50,37}},
/* 14486 */ {(16<<2)|3,{111,50,45}},
/* 14487 */ {(16<<2)|3,{111,50,46}},
/* 14488 */ {(16<<2)|3,{111,50,47}},
/* 14489 */ {(16<<2)|3,{111,50,51}},
/* 14490 */ {(16<<2)|3,{111,50,52}},
/* 14491 */ {(16<<2)|3,{111,50,53}},
/* 14492 */ {(16<<2)|3,{111,50,54}},
/* 14493 */ {(16<<2)|3,{111,50,55}},
/* 14494 */ {(16<<2)|3,{111,50,56}},
/* 14495 */ {(16<<2)|3,{111,50,57}},
/* 14496 */ {(16<<2)|3,{111,50,61}},
/* 14497 */ {(16<<2)|3,{111,50,65}},
/* 14498 */ {(16<<2)|3,{111,50,95}},
/* 14499 */ {(16<<2)|3,{111,50,98}},
/* 14500 */ {(16<<2)|3,{111,50,100}},
/* 14501 */ {(16<<2)|3,{111,50,102}},
/* 14502 */ {(16<<2)|3,{111,50,103}},
/* 14503 */ {(16<<2)|3,{111,50,104}},
/* 14504 */ {(16<<2)|3,{111,50,108}},
/* 14505 */ {(16<<2)|3,{111,50,109}},
/* 14506 */ {(16<<2)|3,{111,50,110}},
/* 14507 */ {(16<<2)|3,{111,50,112}},
/* 14508 */ {(16<<2)|3,{111,50,114}},
/* 14509 */ {(16<<2)|3,{111,50,117}},
/* 14510 */ {(10<<2)|2,{111,50,0}},
/* 14511 */ {(10<<2)|2,{111,50,0}},
/* 14512 */ {(10<<2)|2,{111,50,0}},
/* 14513 */ {(10<<2)|2,{111,50,0}},
/* 14514 */ {(10<<2)|2,{111,50,0}},
/* 14515 */ {(10<<2)|2,{111,50,0}},
/* 14516 */ {(10<<2)|2,{111,50,0}},
/* 14517 */ {(10<<2)|2,{111,50,0}},
/* 14518 */ {(10<<2)|2,{111,50,0}},
/* 14519 */ {(10<<2)|2,{111,50,0}},
/* 14520 */ {(10<<2)|2,{111,50,0}},
/* 14521 */ {(10<<2)|2,{111,50,0}},
/* 14522 */ {(10<<2)|2,{111,50,0}},
/* 14523 */ {(10<<2)|2,{111,50,0}},
/* 14524 */ {(10<<2)|2,{111,50,0}},
/* 14525 */ {(10<<2)|2,{111,50,0}},
/* 14526 */ {(10<<2)|2,{111,50,0}},
/* 14527 */ {(10<<2)|2,{111,50,0}},
/* 14528 */ {(15<<2)|3,{111,97,48}},
/* 14529 */ {(15<<2)|3,{111,97,48}},
/* 14530 */ {(15<<2)|3,{111,97,49}},
/* 14531 */ {(15<<2)|3,{111,97,49}},
/* 14532 */ {(15<<2)|3,{111,97,50}},
/* 14533 */ {(15<<2)|3,{111,97,50}},
/* 14534 */ {(15<<2)|3,{111,97,97}},
/* 14535 */ {(15<<2)|3,{111,97,97}},
/* 14536 */ {(15<<2)|3,{111,97,99}},
/* 14537 */ {(15<<2)|3,{111,97,99}},
/* 14538 */ {(15<<2)|3,{111,97,101}},
/* 14539 */ {(15<<2)|3,{111,97,101}},
/* 14540 */ {(15<<2)|3,{111,97,105}},
/* 14541 */ {(15<<2)|3,{111,97,105}},
/* 14542 */ {(15<<2)|3,{111,97,111}},
/* 14543 */ {(15<<2)|3,{111,97,111}},
/* 14544 */ {(15<<2)|3,{111,97,115}},
/* 14545 */ {(15<<2)|3,{111,97,115}},
/* 14546 */ {(15<<2)|3,{111,97,116}},
/* 14547 */ {(15<<2)|3,{111,97,116}},
/* 14548 */ {(16<<2)|3,{111,97,32}},
/* 14549 */ {(16<<2)|3,{111,97,37}},
/* 14550 */ {(16<<2)|3,{111,97,45}},
/* 14551 */ {(16<<2)|3,{111,97,46}},
/* 14552 */ {(16<<2)|3,{111,97,47}},
/* 14553 */ {(16<<2)|3,{111,97,51}},
/* 14554 */ {(16<<2)|3,{111,97,52}},
/* 14555 */ {(16<<2)|3,{111,97,53}},
/* 14556 */ {(16<<2)|3,{111,97,54}},
/* 14557 */ {(16<<2)|3,{111,97,55}},
/* 14558 */ {(16<<2)|3,{111,97,56}},
/* 14559 */ {(16<<2)|3,{111,97,57}},
/* 14560 */ {(16<<2)|3,{111,97,61}},
/* 14561 */ {(16<<2)|3,{111,97,65}},
/* 14562 */ {(16<<2)|3,{111,97,95}},
/* 14563 */ {(16<<2)|3,{111,97,98}},
/* 14564 */ {(16<<2)|3,{111,97,100}},
/* 14565 */ {(16<<2)|3,{111,97,102}},
/* 14566 */ {(16<<2)|3,{111,97,103}},
/* 14567 */ {(16<<2)|3,{111,97,104}},
/* 14568 */ {(16<<2)|3,{111,97,108}},
/* 14569 */ {(16<<2)|3,{111,97,109}},
/* 14570 */ {(16<<2)|3,{111,97,110}},
/* 14571 */ {(16<<2)|3,{111,97,112}},
/* 14572 */ {(16<<2)|3,{111,97,114}},
/* 14573 */ {(16<<2)|3,{111,97,117}},
/* 14574 */ {(10<<2)|2,{111,97,0}},
/* 14575 */ {(10<<2)|2,{111,97,0}},
/* 14576 */ {(10<<2)|2,{111,97,0}},
/* 14577 */ {(10<<2)|2,{111,97,0}},
/* 14578 */ {(10<<2)|2,{111,97,0}},
/* 14579 */ {(10<<2)|2,{111,97,0}},
/* 14580 */ {(10<<2)|2,{111,97,0}},
/* 14581 */ {(10<<2)|2,{111,97,0}},
/* 14582 */ {(10<<2)|2,{111,97,0}},
/* 14583 */ {(10<<2)|2,{111,97,0}},
/* 14584 */ {(10<<2)|2,{111,97,0}},
/* 14585 */ {(10<<2)|2,{111,97,0}},
/* 14586 */ {(10<<2)|2,{111,97,0}},
/* 14587 */ {(10<<2)|2,{111,97,0}},
/* 14588 */ {(10<<2)|2,{111,97,0}},
/* 14589 */ {(10<<2)|2,{111,97,0}},
/* 14590 */ {(10<<2)|2,{111,97,0}},
/* 14591 */ {(10<<2)|2,{111,97,0}},
/* 14592 */ {(15<<2)|3,{111,99,48}},
/* 14593 */ {(15<<2)|3,{111,99,48}},
/* 14594 */ {(15<<2)|3,{111,99,49}},
/* 14595 */ {(15<<2)|3,{111,99,49}},
/* 14596 */ {(15<<2)|3,{111,99,50}},
/* 14597 */ {(15<<2)|3,{111,99,50}},
/* 14598 */ {(15<<2)|3,{111,99,97}},
/* 14599 */ {(15<<2)|3,{111,99,97}},
/* 14600 */ {(15<<2)|3,{111,99,99}},
/* 14601 */ {(15<<2)|3,{111,99,99}},
/* 14602 */ {(15<<2)|3,{111,99,101}},
/* 14603 */ {(15<<2)|3,{111,99,101}},
/* 14604 */ {(15<<2)|3,{111,99,105}},
/* 14605 */ {(15<<2)|3,{111,99,105}},
/* 14606 */ {(15<<2)|3,{111,99,111}},
/* 14607 */ {(15<<2)|3,{111,99,111}},
/* 14608 */ {(15<<2)|3,{111,99,115}},
/* 14609 */ {(15<<2)|3,{111,99,115}},
/* 14610 */ {(15<<2)|3,{111,99,116}},
/* 14611 */ {(15<<2)|3,{111,99,116}},
/* 14612 */ {(16<<2)|3,{111,99,32}},
/* 14613 */ {(16<<2)|3,{111,99,37}},
/* 14614 */ {(16<<2)|3,{111,99,45}},
/* 14615 */ {(16<<2)|3,{111,99,46}},
/* 14616 */ {(16<<2)|3,{111,99,47}},
/* 14617 */ {(16<<2)|3,{111,99,51}},
/* 14618 */ {(16<<2)|3,{111,99,52}},
/* 14619 */ {(16<<2)|3,{111,99,53}},
/* 14620 */ {(16<<2)|3,{111,99,54}},
/* 14621 */ {(16<<2)|3,{111,99,55}},
/* 14622 */ {(16<<2)|3,{111,99,56}},
/* 14623 */ {(16<<2)|3,{111,99,57}},
/* 14624 */ {(16<<2)|3,{111,99,61}},
/* 14625 */ {(16<<2)|3,{111,99,65}},
/* 14626 */ {(16<<2)|3,{111,99,95}},
/* 14627 */ {(16<<2)|3,{111,99,98}},
/* 14628 */ {(16<<2)|3,{111,99,100}},
/* 14629 */ {(16<<2)|3,{111,99,102}},
/* 14630 */ {(16<<2)|3,{111,99,103}},
/* 14631 */ {(16<<2)|3,{111,99,104}},
/* 14632 */ {(16<<2)|3,{111,99,108}},
/* 14633 */ {(16<<2)|3,{111,99,109}},
/* 14634 */ {(16<<2)|3,{111,99,110}},
/* 14635 */ {(16<<2)|3,{111,99,112}},
/* 14636 */ {(16<<2)|3,{111,99,114}},
/* 14637 */ {(16<<2)|3,{111,99,117}},
/* 14638 */ {(10<<2)|2,{111,99,0}},
/* 14639 */ {(10<<2)|2,{111,99,0}},
/* 14640 */ {(10<<2)|2,{111,99,0}},
/* 14641 */ {(10<<2)|2,{111,99,0}},
/* 14642 */ {(10<<2)|2,{111,99,0}},
/* 14643 */ {(10<<2)|2,{111,99,0}},
/* 14644 */ {(10<<2)|2,{111,99,0}},
/* 14645 */ {(10<<2)|2,{111,99,0}},
/* 14646 */ {(10<<2)|2,{111,99,0}},
/* 14647 */ {(10<<2)|2,{111,99,0}},
/* 14648 */ {(10<<2)|2,{111,99,0}},
/* 14649 */ {(10<<2)|2,{111,99,0}},
/* 14650 */ {(10<<2)|2,{111,99,0}},
/* 14651 */ {(10<<2)|2,{111,99,0}},
/* 14652 */ {(10<<2)|2,{111,99,0}},
/* 14653 */ {(10<<2)|2,{111,99,0}},
/* 14654 */ {(10<<2)|2,{111,99,0}},
/* 14655 */ {(10<<2)|2,{111,99,0}},
/* 14656 */ {(15<<2)|3,{111,101,48}},
/* 14657 */ {(15<<2)|3,{111,101,48}},
/* 14658 */ {(15<<2)|3,{111,101,49}},
/* 14659 */ {(15<<2)|3,{111,101,49}},
/* 14660 */ {(15<<2)|3,{111,101,50}},
/* 14661 */ {(15<<2)|3,{111,101,50}},
/* 14662 */ {(15<<2)|3,{111,101,97}},
/* 14663 */ {(15<<2)|3,{111,101,97}},
/* 14664 */ {(15<<2)|3,{111,101,99}},
/* 14665 */ {(15<<2)|3,{111,101,99}},
/* 14666 */ {(15<<2)|3,{111,101,101}},
/* 14667 */ {(15<<2)|3,{111,101,101}},
/* 14668 */ {(15<<2)|3,{111,101,105}},
/* 14669 */ {(15<<2)|3,{111,101,105}},
/* 14670 */ {(15<<2)|3,{111,101,111}},
/* 14671 */ {(15<<2)|3,{111,101,111}},
/* 14672 */ {(15<<2)|3,{111,101,115}},
/* 14673 */ {(15<<2)|3,{111,101,115}},
/* 14674 */ {(15<<2)|3,{111,101,116}},
/* 14675 */ {(15<<2)|3,{111,101,116}},
/* 14676 */ {(16<<2)|3,{111,101,32}},
/* 14677 */ {(16<<2)|3,{111,101,37}},
/* 14678 */ {(16<<2)|3,{111,101,45}},
/* 14679 */ {(16<<2)|3,{111,101,46}},
/* 14680 */ {(16<<2)|3,{111,101,47}},
/* 14681 */ {(16<<2)|3,{111,101,51}},
/* 14682 */ {(16<<2)|3,{111,101,52}},
/* 14683 */ {(16<<2)|3,{111,101,53}},
/* 14684 */ {(16<<2)|3,{111,101,54}},
/* 14685 */ {(16<<2)|3,{111,101,55}},
/* 14686 */ {(16<<2)|3,{111,101,56}},
/* 14687 */ {(16<<2)|3,{111,101,57}},
/* 14688 */ {(16<<2)|3,{111,101,61}},
/* 14689 */ {(16<<2)|3,{111,101,65}},
/* 14690 */ {(16<<2)|3,{111,101,95}},
/* 14691 */ {(16<<2)|3,{111,101,98}},
/* 14692 */ {(16<<2)|3,{111,101,100}},
/* 14693 */ {(16<<2)|3,{111,101,102}},
/* 14694 */ {(16<<2)|3,{111,101,103}},
/* 14695 */ {(16<<2)|3,{111,101,104}},
/* 14696 */ {(16<<2)|3,{111,101,108}},
/* 14697 */ {(16<<2)|3,{111,101,109}},
/* 14698 */ {(16<<2)|3,{111,101,110}},
/* 14699 */ {(16<<2)|3,{111,101,112}},
/* 14700 */ {(16<<2)|3,{111,101,114}},
/* 14701 */ {(16<<2)|3,{111,101,117}},
/* 14702 */ {(10<<2)|2,{111,101,0}},
/* 14703 */ {(10<<2)|2,{111,101,0}},
/* 14704 */ {(10<<2)|2,{111,101,0}},
/* 14705 */ {(10<<2)|2,{111,101,0}},
/* 14706 */ {(10<<2)|2,{111,101,0}},
/* 14707 */ {(10<<2)|2,{111,101,0}},
/* 14708 */ {(10<<2)|2,{111,101,0}},
/* 14709 */ {(10<<2)|2,{111,101,0}},
/* 14710 */ {(10<<2)|2,{111,101,0}},
/* 14711 */ {(10<<2)|2,{111,101,0}},
/* 14712 */ {(10<<2)|2,{111,101,0}},
/* 14713 */ {(10<<2)|2,{111,101,0}},
/* 14714 */ {(10<<2)|2,{111,101,0}},
/* 14715 */ {(10<<2)|2,{111,101,0}},
/* 14716 */ {(10<<2)|2,{111,101,0}},
/* 14717 */ {(10<<2)|2,{111,101,0}},
/* 14718 */ {(10<<2)|2,{111,101,0}},
/* 14719 */ {(10<<2)|2,{111,101,0}},
/* 14720 */ {(15<<2)|3,{111,105,48}},
/* 14721 */ {(15<<2)|3,{111,105,48}},
/* 14722 */ {(15<<2)|3,{111,105,49}},
/* 14723 */ {(15<<2)|3,{111,105,49}},
/* 14724 */ {(15<<2)|3,{111,105,50}},
/* 14725 */ {(15<<2)|3,{111,105,50}},
/* 14726 */ {(15<<2)|3,{111,105,97}},
/* 14727 */ {(15<<2)|3,{111,105,97}},
/* 14728 */ {(15<<2)|3,{111,105,99}},
/* 14729 */ {(15<<2)|3,{111,105,99}},
/* 14730 */ {(15<<2)|3,{111,105,101}},
/* 14731 */ {(15<<2)|3,{111,105,101}},
/* 14732 */ {(15<<2)|3,{111,105,105}},
/* 14733 */ {(15<<2)|3,{111,105,105}},
/* 14734 */ {(15<<2)|3,{111,105,111}},
/* 14735 */ {(15<<2)|3,{111,105,111}},
/* 14736 */ {(15<<2)|3,{111,105,115}},
/* 14737 */ {(15<<2)|3,{111,105,115}},
/* 14738 */ {(15<<2)|3,{111,105,116}},
/* 14739 */ {(15<<2)|3,{111,105,116}},
/* 14740 */ {(16<<2)|3,{111,105,32}},
/* 14741 */ {(16<<2)|3,{111,105,37}},
/* 14742 */ {(16<<2)|3,{111,105,45}},
/* 14743 */ {(16<<2)|3,{111,105,46}},
/* 14744 */ {(16<<2)|3,{111,105,47}},
/* 14745 */ {(16<<2)|3,{111,105,51}},
/* 14746 */ {(16<<2)|3,{111,105,52}},
/* 14747 */ {(16<<2)|3,{111,105,53}},
/* 14748 */ {(16<<2)|3,{111,105,54}},
/* 14749 */ {(16<<2)|3,{111,105,55}},
/* 14750 */ {(16<<2)|3,{111,105,56}},
/* 14751 */ {(16<<2)|3,{111,105,57}},
/* 14752 */ {(16<<2)|3,{111,105,61}},
/* 14753 */ {(16<<2)|3,{111,105,65}},
/* 14754 */ {(16<<2)|3,{111,105,95}},
/* 14755 */ {(16<<2)|3,{111,105,98}},
/* 14756 */ {(16<<2)|3,{111,105,100}},
/* 14757 */ {(16<<2)|3,{111,105,102}},
/* 14758 */ {(16<<2)|3,{111,105,103}},
/* 14759 */ {(16<<2)|3,{111,105,104}},
/* 14760 */ {(16<<2)|3,{111,105,108}},
/* 14761 */ {(16<<2)|3,{111,105,109}},
/* 14762 */ {(16<<2)|3,{111,105,110}},
/* 14763 */ {(16<<2)|3,{111,105,112}},
/* 14764 */ {(16<<2)|3,{111,105,114}},
/* 14765 */ {(16<<2)|3,{111,105,117}},
/* 14766 */ {(10<<2)|2,{111,105,0}},
/* 14767 */ {(10<<2)|2,{111,105,0}},
/* 14768 */ {(10<<2)|2,{111,105,0}},
/* 14769 */ {(10<<2)|2,{111,105,0}},
/* 14770 */ {(10<<2)|2,{111,105,0}},
/* 14771 */ {(10<<2)|2,{111,105,0}},
/* 14772 */ {(10<<2)|2,{111,105,0}},
/* 14773 */ {(10<<2)|2,{111,105,0}},
/* 14774 */ {(10<<2)|2,{111,105,0}},
/* 14775 */ {(10<<2)|2,{111,105,0}},
/* 14776 */ {(10<<2)|2,{111,105,0}},
/* 14777 */ {(10<<2)|2,{111,105,0}},
/* 14778 */ {(10<<2)|2,{111,105,0}},
/* 14779 */ {(10<<2)|2,{111,105,0}},
/* 14780 */ {(10<<2)|2,{111,105,0}},
/* 14781 */ {(10<<2)|2,{111,105,0}},
/* 14782 */ {(10<<2)|2,{111,105,0}},
/* 14783 */ {(10<<2)|2,{111,105,0}},
/* 14784 */ {(15<<2)|3,{111,111,48}},
/* 14785 */ {(15<<2)|3,{111,111,48}},
/* 14786 */ {(15<<2)|3,{111,111,49}},
/* 14787 */ {(15<<2)|3,{111,111,49}},
/* 14788 */ {(15<<2)|3,{111,111,50}},
/* 14789 */ {(15<<2)|3,{111,111,50}},
/* 14790 */ {(15<<2)|3,{111,111,97}},
/* 14791 */ {(15<<2)|3,{111,111,97}},
/* 14792 */ {(15<<2)|3,{111,111,99}},
/* 14793 */ {(15<<2)|3,{111,111,99}},
/* 14794 */ {(15<<2)|3,{111,111,101}},
/* 14795 */ {(15<<2)|3,{111,111,101}},
/* 14796 */ {(15<<2)|3,{111,111,105}},
/* 14797 */ {(15<<2)|3,{111,111,105}},
/* 14798 */ {(15<<2)|3,{111,111,111}},
/* 14799 */ {(15<<2)|3,{111,111,111}},
/* 14800 */ {(15<<2)|3,{111,111,115}},
/* 14801 */ {(15<<2)|3,{111,111,115}},
/* 14802 */ {(15<<2)|3,{111,111,116}},
/* 14803 */ {(15<<2)|3,{111,111,116}},
/* 14804 */ {(16<<2)|3,{111,111,32}},
/* 14805 */ {(16<<2)|3,{111,111,37}},
/* 14806 */ {(16<<2)|3,{111,111,45}},
/* 14807 */ {(16<<2)|3,{111,111,46}},
/* 14808 */ {(16<<2)|3,{111,111,47}},
/* 14809 */ {(16<<2)|3,{111,111,51}},
/* 14810 */ {(16<<2)|3,{111,111,52}},
/* 14811 */ {(16<<2)|3,{111,111,53}},
/* 14812 */ {(16<<2)|3,{111,111,54}},
/* 14813 */ {(16<<2)|3,{111,111,55}},
/* 14814 */ {(16<<2)|3,{111,111,56}},
/* 14815 */ {(16<<2)|3,{111,111,57}},
/* 14816 */ {(16<<2)|3,{111,111,61}},
/* 14817 */ {(16<<2)|3,{111,111,65}},
/* 14818 */ {(16<<2)|3,{111,111,95}},
/* 14819 */ {(16<<2)|3,{111,111,98}},
/* 14820 */ {(16<<2)|3,{111,111,100}},
/* 14821 */ {(16<<2)|3,{111,111,102}},
/* 14822 */ {(16<<2)|3,{111,111,103}},
/* 14823 */ {(16<<2)|3,{111,111,104}},
/* 14824 */ {(16<<2)|3,{111,111,108}},
/* 14825 */ {(16<<2)|3,{111,111,109}},
/* 14826 */ {(16<<2)|3,{111,111,110}},
/* 14827 */ {(16<<2)|3,{111,111,112}},
/* 14828 */ {(16<<2)|3,{111,111,114}},
/* 14829 */ {(16<<2)|3,{111,111,117}},
/* 14830 */ {(10<<2)|2,{111,111,0}},
/* 14831 */ {(10<<2)|2,{111,111,0}},
/* 14832 */ {(10<<2)|2,{111,111,0}},
/* 14833 */ {(10<<2)|2,{111,111,0}},
/* 14834 */ {(10<<2)|2,{111,111,0}},
/* 14835 */ {(10<<2)|2,{111,111,0}},
/* 14836 */ {(10<<2)|2,{111,111,0}},
/* 14837 */ {(10<<2)|2,{111,111,0}},
/* 14838 */ {(10<<2)|2,{111,111,0}},
/* 14839 */ {(10<<2)|2,{111,111,0}},
/* 14840 */ {(10<<2)|2,{111,111,0}},
/* 14841 */ {(10<<2)|2,{111,111,0}},
/* 14842 */ {(10<<2)|2,{111,111,0}},
/* 14843 */ {(10<<2)|2,{111,111,0}},
/* 14844 */ {(10<<2)|2,{111,111,0}},
/* 14845 */ {(10<<2)|2,{111,111,0}},
/* 14846 */ {(10<<2)|2,{111,111,0}},
/* 14847 */ {(10<<2)|2,{111,111,0}},
/* 14848 */ {(15<<2)|3,{111,115,48}},
/* 14849 */ {(15<<2)|3,{111,115,48}},
/* 14850 */ {(15<<2)|3,{111,115,49}},
/* 14851 */ {(15<<2)|3,{111,115,49}},
/* 14852 */ {(15<<2)|3,{111,115,50}},
/* 14853 */ {(15<<2)|3,{111,115,50}},
/* 14854 */ {(15<<2)|3,{111,115,97}},
/* 14855 */ {(15<<2)|3,{111,115,97}},
/* 14856 */ {(15<<2)|3,{111,115,99}},
/* 14857 */ {(15<<2)|3,{111,115,99}},
/* 14858 */ {(15<<2)|3,{111,115,101}},
/* 14859 */ {(15<<2)|3,{111,115,101}},
/* 14860 */ {(15<<2)|3,{111,115,105}},
/* 14861 */ {(15<<2)|3,{111,115,105}},
/* 14862 */ {(15<<2)|3,{111,115,111}},
/* 14863 */ {(15<<2)|3,{111,115,111}},
/* 14864 */ {(15<<2)|3,{111,115,115}},
/* 14865 */ {(15<<2)|3,{111,115,115}},
/* 14866 */ {(15<<2)|3,{111,115,116}},
/* 14867 */ {(15<<2)|3,{111,115,116}},
/* 14868 */ {(16<<2)|3,{111,115,32}},
/* 14869 */ {(16<<2)|3,{111,115,37}},
/* 14870 */ {(16<<2)|3,{111,115,45}},
/* 14871 */ {(16<<2)|3,{111,115,46}},
/* 14872 */ {(16<<2)|3,{111,115,47}},
/* 14873 */ {(16<<2)|3,{111,115,51}},
/* 14874 */ {(16<<2)|3,{111,115,52}},
/* 14875 */ {(16<<2)|3,{111,115,53}},
/* 14876 */ {(16<<2)|3,{111,115,54}},
/* 14877 */ {(16<<2)|3,{111,115,55}},
/* 14878 */ {(16<<2)|3,{111,115,56}},
/* 14879 */ {(16<<2)|3,{111,115,57}},
/* 14880 */ {(16<<2)|3,{111,115,61}},
/* 14881 */ {(16<<2)|3,{111,115,65}},
/* 14882 */ {(16<<2)|3,{111,115,95}},
/* 14883 */ {(16<<2)|3,{111,115,98}},
/* 14884 */ {(16<<2)|3,{111,115,100}},
/* 14885 */ {(16<<2)|3,{111,115,102}},
/* 14886 */ {(16<<2)|3,{111,115,103}},
/* 14887 */ {(16<<2)|3,{111,115,104}},
/* 14888 */ {(16<<2)|3,{111,115,108}},
/* 14889 */ {(16<<2)|3,{111,115,109}},
/* 14890 */ {(16<<2)|3,{111,115,110}},
/* 14891 */ {(16<<2)|3,{111,115,112}},
/* 14892 */ {(16<<2)|3,{111,115,114}},
/* 14893 */ {(16<<2)|3,{111,115,117}},
/* 14894 */ {(10<<2)|2,{111,115,0}},
/* 14895 */ {(10<<2)|2,{111,115,0}},
/* 14896 */ {(10<<2)|2,{111,115,0}},
/* 14897 */ {(10<<2)|2,{111,115,0}},
/* 14898 */ {(10<<2)|2,{111,115,0}},
/* 14899 */ {(10<<2)|2,{111,115,0}},
/* 14900 */ {(10<<2)|2,{111,115,0}},
/* 14901 */ {(10<<2)|2,{111,115,0}},
/* 14902 */ {(10<<2)|2,{111,115,0}},
/* 14903 */ {(10<<2)|2,{111,115,0}},
/* 14904 */ {(10<<2)|2,{111,115,0}},
/* 14905 */ {(10<<2)|2,{111,115,0}},
/* 14906 */ {(10<<2)|2,{111,115,0}},
/* 14907 */ {(10<<2)|2,{111,115,0}},
/* 14908 */ {(10<<2)|2,{111,115,0}},
/* 14909 */ {(10<<2)|2,{111,115,0}},
/* 14910 */ {(10<<2)|2,{111,115,0}},
/* 14911 */ {(10<<2)|2,{111,115,0}},
/* 14912 */ {(15<<2)|3,{111,116,48}},
/* 14913 */ {(15<<2)|3,{111,116,48}},
/* 14914 */ {(15<<2)|3,{111,116,49}},
/* 14915 */ {(15<<2)|3,{111,116,49}},
/* 14916 */ {(15<<2)|3,{111,116,50}},
/* 14917 */ {(15<<2)|3,{111,116,50}},
/* 14918 */ {(15<<2)|3,{111,116,97}},
/* 14919 */ {(15<<2)|3,{111,116,97}},
/* 14920 */ {(15<<2)|3,{111,116,99}},
/* 14921 */ {(15<<2)|3,{111,116,99}},
/* 14922 */ {(15<<2)|3,{111,116,101}},
/* 14923 */ {(15<<2)|3,{111,116,101}},
/* 14924 */ {(15<<2)|3,{111,116,105}},
/* 14925 */ {(15<<2)|3,{111,116,105}},
/* 14926 */ {(15<<2)|3,{111,116,111}},
/* 14927 */ {(15<<2)|3,{111,116,111}},
/* 14928 */ {(15<<2)|3,{111,116,115}},
/* 14929 */ {(15<<2)|3,{111,116,115}},
/* 14930 */ {(15<<2)|3,{111,116,116}},
/* 14931 */ {(15<<2)|3,{111,116,116}},
/* 14932 */ {(16<<2)|3,{111,116,32}},
/* 14933 */ {(16<<2)|3,{111,116,37}},
/* 14934 */ {(16<<2)|3,{111,116,45}},
/* 14935 */ {(16<<2)|3,{111,116,46}},
/* 14936 */ {(16<<2)|3,{111,116,47}},
/* 14937 */ {(16<<2)|3,{111,116,51}},
/* 14938 */ {(16<<2)|3,{111,116,52}},
/* 14939 */ {(16<<2)|3,{111,116,53}},
/* 14940 */ {(16<<2)|3,{111,116,54}},
/* 14941 */ {(16<<2)|3,{111,116,55}},
/* 14942 */ {(16<<2)|3,{111,116,56}},
/* 14943 */ {(16<<2)|3,{111,116,57}},
/* 14944 */ {(16<<2)|3,{111,116,61}},
/* 14945 */ {(16<<2)|3,{111,116,65}},
/* 14946 */ {(16<<2)|3,{111,116,95}},
/* 14947 */ {(16<<2)|3,{111,116,98}},
/* 14948 */ {(16<<2)|3,{111,116,100}},
/* 14949 */ {(16<<2)|3,{111,116,102}},
/* 14950 */ {(16<<2)|3,{111,116,103}},
/* 14951 */ {(16<<2)|3,{111,116,104}},
/* 14952 */ {(16<<2)|3,{111,116,108}},
/* 14953 */ {(16<<2)|3,{111,116,109}},
/* 14954 */ {(16<<2)|3,{111,116,110}},
/* 14955 */ {(16<<2)|3,{111,116,112}},
/* 14956 */ {(16<<2)|3,{111,116,114}},
/* 14957 */ {(16<<2)|3,{111,116,117}},
/* 14958 */ {(10<<2)|2,{111,116,0}},
/* 14959 */ {(10<<2)|2,{111,116,0}},
/* 14960 */ {(10<<2)|2,{111,116,0}},
/* 14961 */ {(10<<2)|2,{111,116,0}},
/* 14962 */ {(10<<2)|2,{111,116,0}},
/* 14963 */ {(10<<2)|2,{111,116,0}},
/* 14964 */ {(10<<2)|2,{111,116,0}},
/* 14965 */ {(10<<2)|2,{111,116,0}},
/* 14966 */ {(10<<2)|2,{111,116,0}},
/* 14967 */ {(10<<2)|2,{111,116,0}},
/* 14968 */ {(10<<2)|2,{111,116,0}},
/* 14969 */ {(10<<2)|2,{111,116,0}},
/* 14970 */ {(10<<2)|2,{111,116,0}},
/* 14971 */ {(10<<2)|2,{111,116,0}},
/* 14972 */ {(10<<2)|2,{111,116,0}},
/* 14973 */ {(10<<2)|2,{111,116,0}},
/* 14974 */ {(10<<2)|2,{111,116,0}},
/* 14975 */ {(10<<2)|2,{111,116,0}},
/* 14976 */ {(16<<2)|3,{111,32,48}},
/* 14977 */ {(16<<2)|3,{111,32,49}},
/* 14978 */ {(16<<2)|3,{111,32,50}},
/* 14979 */ {(16<<2)|3,{111,32,97}},
/* 14980 */ {(16<<2)|3,{111,32,99}},
/* 14981 */ {(16<<2)|3,{111,32,101}},
/* 14982 */ {(16<<2)|3,{111,32,105}},
/* 14983 */ {(16<<2)|3,{111,32,111}},
/* 14984 */ {(16<<2)|3,{111,32,115}},
/* 14985 */ {(16<<2)|3,{111,32,116}},
/* 14986 */ {(11<<2)|2,{111,32,0}},
/* 14987 */ {(11<<2)|2,{111,32,0}},
/* 14988 */ {(11<<2)|2,{111,32,0}},
/* 14989 */ {(11<<2)|2,{111,32,0}},
/* 14990 */ {(11<<2)|2,{111,32,0}},
/* 14991 */ {(11<<2)|2,{111,32,0}},
/* 14992 */ {(11<<2)|2,{111,32,0}},
/* 14993 */ {(11<<2)|2,{111,32,0}},
/* 14994 */ {(11<<2)|2,{111,32,0}},
/* 14995 */ {(11<<2)|2,{111,32,0}},
/* 14996 */ {(11<<2)|2,{111,32,0}},
/* 14997 */ {(11<<2)|2,{111,32,0}},
/* 14998 */ {(11<<2)|2,{111,32,0}},
/* 14999 */ {(11<<2)|2,{111,32,0}},
/* 15000 */ {(11<<2)|2,{111,32,0}},
/* 15001 */ {(11<<2)|2,{111,32,0}},
/* 15002 */ {(11<<2)|2,{111,32,0}},
/* 15003 */ {(11<<2)|2,{111,32,0}},
/* 15004 */ {(11<<2)|2,{111,32,0}},
/* 15005 */ {(11<<2)|2,{111,32,0}},
/* 15006 */ {(11<<2)|2,{111,32,0}},
/* 15007 */ {(11<<2)|2,{111,32,0}},
/* 15008 */ {(16<<2)|3,{111,37,48}},
/* 15009 */ {(16<<2)|3,{111,37,49}},
/* 15010 */ {(16<<2)|3,{111,37,50}},
/* 15011 */ {(16<<2)|3,{111,37,97}},
/* 15012 */ {(16<<2)|3,{111,37,99}},
/* 15013 */ {(16<<2)|3,{111,37,101}},
/* 15014 */ {(16<<2)|3,{111,37,105}},
/* 15015 */ {(16<<2)|3,{111,37,111}},
/* 15016 */ {(16<<2)|3,{111,37,115}},
/* 15017 */ {(16<<2)|3,{111,37,116}},
/* 15018 */ {(11<<2)|2,{111,37,0}},
/* 15019 */ {(11<<2)|2,{111,37,0}},
/* 15020 */ {(11<<2)|2,{111,37,0}},
/* 15021 */ {(11<<2)|2,{111,37,0}},
/* 15022 */ {(11<<2)|2,{111,37,0}},
/* 15023 */ {(11<<2)|2,{111,37,0}},
/* 15024 */ {(11<<2)|2,{111,37,0}},
/* 15025 */ {(11<<2)|2,{111,37,0}},
/* 15026 */ {(11<<2)|2,{111,37,0}},
/* 15027 */ {(11<<2)|2,{111,37,0}},
/* 15028 */ {(11<<2)|2,{111,37,0}},
/* 15029 */ {(11<<2)|2,{111,37,0}},
/* 15030 */ {(11<<2)|2,{111,37,0}},
/* 15031 */ {(11<<2)|2,{111,37,0}},
/* 15032 */ {(11<<2)|2,{111,37,0}},
/* 15033 */ {(11<<2)|2,{111,37,0}},
/* 15034 */ {(11<<2)|2,{111,37,0}},
/* 15035 */ {(11<<2)|2,{111,37,0}},
/* 15036 */ {(11<<2)|2,{111,37,0}},
/* 15037 */ {(11<<2)|2,{111,37,0}},
/* 15038 */ {(11<<2)|2,{111,37,0}},
/* 15039 */ {(11<<2)|2,{111,37,0}},
/* 15040 */ {(16<<2)|3,{111,45,48}},
/* 15041 */ {(16<<2)|3,{111,45,49}},
/* 15042 */ {(16<<2)|3,{111,45,50}},
/* 15043 */ {(16<<2)|3,{111,45,97}},
/* 15044 */ {(16<<2)|3,{111,45,99}},
/* 15045 */ {(16<<2)|3,{111,45,101}},
/* 15046 */ {(16<<2)|3,{111,45,105}},
/* 15047 */ {(16<<2)|3,{111,45,111}},
/* 15048 */ {(16<<2)|3,{111,45,115}},
/* 15049 */ {(16<<2)|3,{111,45,116}},
/* 15050 */ {(11<<2)|2,{111,45,0}},
/* 15051 */ {(11<<2)|2,{111,45,0}},
/* 15052 */ {(11<<2)|2,{111,45,0}},
/* 15053 */ {(11<<2)|2,{111,45,0}},
/* 15054 */ {(11<<2)|2,{111,45,0}},
/* 15055 */ {(11<<2)|2,{111,45,0}},
/* 15056 */ {(11<<2)|2,{111,45,0}},
/* 15057 */ {(11<<2)|2,{111,45,0}},
/* 15058 */ {(11<<2)|2,{111,45,0}},
/* 15059 */ {(11<<2)|2,{111,45,0}},
/* 15060 */ {(11<<2)|2,{111,45,0}},
/* 15061 */ {(11<<2)|2,{111,45,0}},
/* 15062 */ {(11<<2)|2,{111,45,0}},
/* 15063 */ {(11<<2)|2,{111,45,0}},
/* 15064 */ {(11<<2)|2,{111,45,0}},
/* 15065 */ {(11<<2)|2,{111,45,0}},
/* 15066 */ {(11<<2)|2,{111,45,0}},
/* 15067 */ {(11<<2)|2,{111,45,0}},
/* 15068 */ {(11<<2)|2,{111,45,0}},
/* 15069 */ {(11<<2)|2,{111,45,0}},
/* 15070 */ {(11<<2)|2,{111,45,0}},
/* 15071 */ {(11<<2)|2,{111,45,0}},
/* 15072 */ {(16<<2)|3,{111,46,48}},
/* 15073 */ {(16<<2)|3,{111,46,49}},
/* 15074 */ {(16<<2)|3,{111,46,50}},
/* 15075 */ {(16<<2)|3,{111,46,97}},
/* 15076 */ {(16<<2)|3,{111,46,99}},
/* 15077 */ {(16<<2)|3,{111,46,101}},
/* 15078 */ {(16<<2)|3,{111,46,105}},
/* 15079 */ {(16<<2)|3,{111,46,111}},
/* 15080 */ {(16<<2)|3,{111,46,115}},
/* 15081 */ {(16<<2)|3,{111,46,116}},
/* 15082 */ {(11<<2)|2,{111,46,0}},
/* 15083 */ {(11<<2)|2,{111,46,0}},
/* 15084 */ {(11<<2)|2,{111,46,0}},
/* 15085 */ {(11<<2)|2,{111,46,0}},
/* 15086 */ {(11<<2)|2,{111,46,0}},
/* 15087 */ {(11<<2)|2,{111,46,0}},
/* 15088 */ {(11<<2)|2,{111,46,0}},
/* 15089 */ {(11<<2)|2,{111,46,0}},
/* 15090 */ {(11<<2)|2,{111,46,0}},
/* 15091 */ {(11<<2)|2,{111,46,0}},
/* 15092 */ {(11<<2)|2,{111,46,0}},
/* 15093 */ {(11<<2)|2,{111,46,0}},
/* 15094 */ {(11<<2)|2,{111,46,0}},
/* 15095 */ {(11<<2)|2,{111,46,0}},
/* 15096 */ {(11<<2)|2,{111,46,0}},
/* 15097 */ {(11<<2)|2,{111,46,0}},
/* 15098 */ {(11<<2)|2,{111,46,0}},
/* 15099 */ {(11<<2)|2,{111,46,0}},
/* 15100 */ {(11<<2)|2,{111,46,0}},
/* 15101 */ {(11<<2)|2,{111,46,0}},
/* 15102 */ {(11<<2)|2,{111,46,0}},
/* 15103 */ {(11<<2)|2,{111,46,0}},
/* 15104 */ {(16<<2)|3,{111,47,48}},
/* 15105 */ {(16<<2)|3,{111,47,49}},
/* 15106 */ {(16<<2)|3,{111,47,50}},
/* 15107 */ {(16<<2)|3,{111,47,97}},
/* 15108 */ {(16<<2)|3,{111,47,99}},
/* 15109 */ {(16<<2)|3,{111,47,101}},
/* 15110 */ {(16<<2)|3,{111,47,105}},
/* 15111 */ {(16<<2)|3,{111,47,111}},
/* 15112 */ {(16<<2)|3,{111,47,115}},
/* 15113 */ {(16<<2)|3,{111,47,116}},
/* 15114 */ {(11<<2)|2,{111,47,0}},
/* 15115 */ {(11<<2)|2,{111,47,0}},
/* 15116 */ {(11<<2)|2,{111,47,0}},
/* 15117 */ {(11<<2)|2,{111,47,0}},
/* 15118 */ {(11<<2)|2,{111,47,0}},
/* 15119 */ {(11<<2)|2,{111,47,0}},
/* 15120 */ {(11<<2)|2,{111,47,0}},
/* 15121 */ {(11<<2)|2,{111,47,0}},
/* 15122 */ {(11<<2)|2,{111,47,0}},
/* 15123 */ {(11<<2)|2,{111,47,0}},
/* 15124 */ {(11<<2)|2,{111,47,0}},
/* 15125 */ {(11<<2)|2,{111,47,0}},
/* 15126 */ {(11<<2)|2,{111,47,0}},
/* 15127 */ {(11<<2)|2,{111,47,0}},
/* 15128 */ {(11<<2)|2,{111,47,0}},
/* 15129 */ {(11<<2)|2,{111,47,0}},
/* 15130 */ {(11<<2)|2,{111,47,0}},
/* 15131 */ {(11<<2)|2,{111,47,0}},
/* 15132 */ {(11<<2)|2,{111,47,0}},
/* 15133 */ {(11<<2)|2,{111,47,0}},
/* 15134 */ {(11<<2)|2,{111,47,0}},
/* 15135 */ {(11<<2)|2,{111,47,0}},
/* 15136 */ {(16<<2)|3,{111,51,48}},
/* 15137 */ {(16<<2)|3,{111,51,49}},
/* 15138 */ {(16<<2)|3,{111,51,50}},
/* 15139 */ {(16<<2)|3,{111,51,97}},
/* 15140 */ {(16<<2)|3,{111,51,99}},
/* 15141 */ {(16<<2)|3,{111,51,101}},
/* 15142 */ {(16<<2)|3,{111,51,105}},
/* 15143 */ {(16<<2)|3,{111,51,111}},
/* 15144 */ {(16<<2)|3,{111,51,115}},
/* 15145 */ {(16<<2)|3,{111,51,116}},
/* 15146 */ {(11<<2)|2,{111,51,0}},
/* 15147 */ {(11<<2)|2,{111,51,0}},
/* 15148 */ {(11<<2)|2,{111,51,0}},
/* 15149 */ {(11<<2)|2,{111,51,0}},
/* 15150 */ {(11<<2)|2,{111,51,0}},
/* 15151 */ {(11<<2)|2,{111,51,0}},
/* 15152 */ {(11<<2)|2,{111,51,0}},
/* 15153 */ {(11<<2)|2,{111,51,0}},
/* 15154 */ {(11<<2)|2,{111,51,0}},
/* 15155 */ {(11<<2)|2,{111,51,0}},
/* 15156 */ {(11<<2)|2,{111,51,0}},
/* 15157 */ {(11<<2)|2,{111,51,0}},
/* 15158 */ {(11<<2)|2,{111,51,0}},
/* 15159 */ {(11<<2)|2,{111,51,0}},
/* 15160 */ {(11<<2)|2,{111,51,0}},
/* 15161 */ {(11<<2)|2,{111,51,0}},
/* 15162 */ {(11<<2)|2,{111,51,0}},
/* 15163 */ {(11<<2)|2,{111,51,0}},
/* 15164 */ {(11<<2)|2,{111,51,0}},
/* 15165 */ {(11<<2)|2,{111,51,0}},
/* 15166 */ {(11<<2)|2,{111,51,0}},
/* 15167 */ {(11<<2)|2,{111,51,0}},
/* 15168 */ {(16<<2)|3,{111,52,48}},
/* 15169 */ {(16<<2)|3,{111,52,49}},
/* 15170 */ {(16<<2)|3,{111,52,50}},
/* 15171 */ {(16<<2)|3,{111,52,97}},
/* 15172 */ {(16<<2)|3,{111,52,99}},
/* 15173 */ {(16<<2)|3,{111,52,101}},
/* 15174 */ {(16<<2)|3,{111,52,105}},
/* 15175 */ {(16<<2)|3,{111,52,111}},
/* 15176 */ {(16<<2)|3,{111,52,115}},
/* 15177 */ {(16<<2)|3,{111,52,116}},
/* 15178 */ {(11<<2)|2,{111,52,0}},
/* 15179 */ {(11<<2)|2,{111,52,0}},
/* 15180 */ {(11<<2)|2,{111,52,0}},
/* 15181 */ {(11<<2)|2,{111,52,0}},
/* 15182 */ {(11<<2)|2,{111,52,0}},
/* 15183 */ {(11<<2)|2,{111,52,0}},
/* 15184 */ {(11<<2)|2,{111,52,0}},
/* 15185 */ {(11<<2)|2,{111,52,0}},
/* 15186 */ {(11<<2)|2,{111,52,0}},
/* 15187 */ {(11<<2)|2,{111,52,0}},
/* 15188 */ {(11<<2)|2,{111,52,0}},
/* 15189 */ {(11<<2)|2,{111,52,0}},
/* 15190 */ {(11<<2)|2,{111,52,0}},
/* 15191 */ {(11<<2)|2,{111,52,0}},
/* 15192 */ {(11<<2)|2,{111,52,0}},
/* 15193 */ {(11<<2)|2,{111,52,0}},
/* 15194 */ {(11<<2)|2,{111,52,0}},
/* 15195 */ {(11<<2)|2,{111,52,0}},
/* 15196 */ {(11<<2)|2,{111,52,0}},
/* 15197 */ {(11<<2)|2,{111,52,0}},
/* 15198 */ {(11<<2)|2,{111,52,0}},
/* 15199 */ {(11<<2)|2,{111,52,0}},
/* 15200 */ {(16<<2)|3,{111,53,48}},
/* 15201 */ {(16<<2)|3,{111,53,49}},
/* 15202 */ {(16<<2)|3,{111,53,50}},
/* 15203 */ {(16<<2)|3,{111,53,97}},
/* 15204 */ {(16<<2)|3,{111,53,99}},
/* 15205 */ {(16<<2)|3,{111,53,101}},
/* 15206 */ {(16<<2)|3,{111,53,105}},
/* 15207 */ {(16<<2)|3,{111,53,111}},
/* 15208 */ {(16<<2)|3,{111,53,115}},
/* 15209 */ {(16<<2)|3,{111,53,116}},
/* 15210 */ {(11<<2)|2,{111,53,0}},
/* 15211 */ {(11<<2)|2,{111,53,0}},
/* 15212 */ {(11<<2)|2,{111,53,0}},
/* 15213 */ {(11<<2)|2,{111,53,0}},
/* 15214 */ {(11<<2)|2,{111,53,0}},
/* 15215 */ {(11<<2)|2,{111,53,0}},
/* 15216 */ {(11<<2)|2,{111,53,0}},
/* 15217 */ {(11<<2)|2,{111,53,0}},
/* 15218 */ {(11<<2)|2,{111,53,0}},
/* 15219 */ {(11<<2)|2,{111,53,0}},
/* 15220 */ {(11<<2)|2,{111,53,0}},
/* 15221 */ {(11<<2)|2,{111,53,0}},
/* 15222 */ {(11<<2)|2,{111,53,0}},
/* 15223 */ {(11<<2)|2,{111,53,0}},
/* 15224 */ {(11<<2)|2,{111,53,0}},
/* 15225 */ {(11<<2)|2,{111,53,0}},
/* 15226 */ {(11<<2)|2,{111,53,0}},
/* 15227 */ {(11<<2)|2,{111,53,0}},
/* 15228 */ {(11<<2)|2,{111,53,0}},
/* 15229 */ {(11<<2)|2,{111,53,0}},
/* 15230 */ {(11<<2)|2,{111,53,0}},
/* 15231 */ {(11<<2)|2,{111,53,0}},
/* 15232 */ {(16<<2)|3,{111,54,48}},
/* 15233 */ {(16<<2)|3,{111,54,49}},
/* 15234 */ {(16<<2)|3,{111,54,50}},
/* 15235 */ {(16<<2)|3,{111,54,97}},
/* 15236 */ {(16<<2)|3,{111,54,99}},
/* 15237 */ {(16<<2)|3,{111,54,101}},
/* 15238 */ {(16<<2)|3,{111,54,105}},
/* 15239 */ {(16<<2)|3,{111,54,111}},
/* 15240 */ {(16<<2)|3,{111,54,115}},
/* 15241 */ {(16<<2)|3,{111,54,116}},
/* 15242 */ {(11<<2)|2,{111,54,0}},
/* 15243 */ {(11<<2)|2,{111,54,0}},
/* 15244 */ {(11<<2)|2,{111,54,0}},
/* 15245 */ {(11<<2)|2,{111,54,0}},
/* 15246 */ {(11<<2)|2,{111,54,0}},
/* 15247 */ {(11<<2)|2,{111,54,0}},
/* 15248 */ {(11<<2)|2,{111,54,0}},
/* 15249 */ {(11<<2)|2,{111,54,0}},
/* 15250 */ {(11<<2)|2,{111,54,0}},
/* 15251 */ {(11<<2)|2,{111,54,0}},
/* 15252 */ {(11<<2)|2,{111,54,0}},
/* 15253 */ {(11<<2)|2,{111,54,0}},
/* 15254 */ {(11<<2)|2,{111,54,0}},
/* 15255 */ {(11<<2)|2,{111,54,0}},
/* 15256 */ {(11<<2)|2,{111,54,0}},
/* 15257 */ {(11<<2)|2,{111,54,0}},
/* 15258 */ {(11<<2)|2,{111,54,0}},
/* 15259 */ {(11<<2)|2,{111,54,0}},
/* 15260 */ {(11<<2)|2,{111,54,0}},
/* 15261 */ {(11<<2)|2,{111,54,0}},
/* 15262 */ {(11<<2)|2,{111,54,0}},
/* 15263 */ {(11<<2)|2,{111,54,0}},
/* 15264 */ {(16<<2)|3,{111,55,48}},
/* 15265 */ {(16<<2)|3,{111,55,49}},
/* 15266 */ {(16<<2)|3,{111,55,50}},
/* 15267 */ {(16<<2)|3,{111,55,97}},
/* 15268 */ {(16<<2)|3,{111,55,99}},
/* 15269 */ {(16<<2)|3,{111,55,101}},
/* 15270 */ {(16<<2)|3,{111,55,105}},
/* 15271 */ {(16<<2)|3,{111,55,111}},
/* 15272 */ {(16<<2)|3,{111,55,115}},
/* 15273 */ {(16<<2)|3,{111,55,116}},
/* 15274 */ {(11<<2)|2,{111,55,0}},
/* 15275 */ {(11<<2)|2,{111,55,0}},
/* 15276 */ {(11<<2)|2,{111,55,0}},
/* 15277 */ {(11<<2)|2,{111,55,0}},
/* 15278 */ {(11<<2)|2,{111,55,0}},
/* 15279 */ {(11<<2)|2,{111,55,0}},
/* 15280 */ {(11<<2)|2,{111,55,0}},
/* 15281 */ {(11<<2)|2,{111,55,0}},
/* 15282 */ {(11<<2)|2,{111,55,0}},
/* 15283 */ {(11<<2)|2,{111,55,0}},
/* 15284 */ {(11<<2)|2,{111,55,0}},
/* 15285 */ {(11<<2)|2,{111,55,0}},
/* 15286 */ {(11<<2)|2,{111,55,0}},
/* 15287 */ {(11<<2)|2,{111,55,0}},
/* 15288 */ {(11<<2)|2,{111,55,0}},
/* 15289 */ {(11<<2)|2,{111,55,0}},
/* 15290 */ {(11<<2)|2,{111,55,0}},
/* 15291 */ {(11<<2)|2,{111,55,0}},
/* 15292 */ {(11<<2)|2,{111,55,0}},
/* 15293 */ {(11<<2)|2,{111,55,0}},
/* 15294 */ {(11<<2)|2,{111,55,0}},
/* 15295 */ {(11<<2)|2,{111,55,0}},
/* 15296 */ {(16<<2)|3,{111,56,48}},
/* 15297 */ {(16<<2)|3,{111,56,49}},
/* 15298 */ {(16<<2)|3,{111,56,50}},
/* 15299 */ {(16<<2)|3,{111,56,97}},
/* 15300 */ {(16<<2)|3,{111,56,99}},
/* 15301 */ {(16<<2)|3,{111,56,101}},
/* 15302 */ {(16<<2)|3,{111,56,105}},
/* 15303 */ {(16<<2)|3,{111,56,111}},
/* 15304 */ {(16<<2)|3,{111,56,115}},
/* 15305 */ {(16<<2)|3,{111,56,116}},
/* 15306 */ {(11<<2)|2,{111,56,0}},
/* 15307 */ {(11<<2)|2,{111,56,0}},
/* 15308 */ {(11<<2)|2,{111,56,0}},
/* 15309 */ {(11<<2)|2,{111,56,0}},
/* 15310 */ {(11<<2)|2,{111,56,0}},
/* 15311 */ {(11<<2)|2,{111,56,0}},
/* 15312 */ {(11<<2)|2,{111,56,0}},
/* 15313 */ {(11<<2)|2,{111,56,0}},
/* 15314 */ {(11<<2)|2,{111,56,0}},
/* 15315 */ {(11<<2)|2,{111,56,0}},
/* 15316 */ {(11<<2)|2,{111,56,0}},
/* 15317 */ {(11<<2)|2,{111,56,0}},
/* 15318 */ {(11<<2)|2,{111,56,0}},
/* 15319 */ {(11<<2)|2,{111,56,0}},
/* 15320 */ {(11<<2)|2,{111,56,0}},
/* 15321 */ {(11<<2)|2,{111,56,0}},
/* 15322 */ {(11<<2)|2,{111,56,0}},
/* 15323 */ {(11<<2)|2,{111,56,0}},
/* 15324 */ {(11<<2)|2,{111,56,0}},
/* 15325 */ {(11<<2)|2,{111,56,0}},
/* 15326 */ {(11<<2)|2,{111,56,0}},
/* 15327 */ {(11<<2)|2,{111,56,0}},
/* 15328 */ {(16<<2)|3,{111,57,48}},
/* 15329 */ {(16<<2)|3,{111,57,49}},
/* 15330 */ {(16<<2)|3,{111,57,50}},
/* 15331 */ {(16<<2)|3,{111,57,97}},
/* 15332 */ {(16<<2)|3,{111,57,99}},
/* 15333 */ {(16<<2)|3,{111,57,101}},
/* 15334 */ {(16<<2)|3,{111,57,105}},
/* 15335 */ {(16<<2)|3,{111,57,111}},
/* 15336 */ {(16<<2)|3,{111,57,115}},
/* 15337 */ {(16<<2)|3,{111,57,116}},
/* 15338 */ {(11<<2)|2,{111,57,0}},
/* 15339 */ {(11<<2)|2,{111,57,0}},
/* 15340 */ {(11<<2)|2,{111,57,0}},
/* 15341 */ {(11<<2)|2,{111,57,0}},
/* 15342 */ {(11<<2)|2,{111,57,0}},
/* 15343 */ {(11<<2)|2,{111,57,0}},
/* 15344 */ {(11<<2)|2,{111,57,0}},
/* 15345 */ {(11<<2)|2,{111,57,0}},
/* 15346 */ {(11<<2)|2,{111,57,0}},
/* 15347 */ {(11<<2)|2,{111,57,0}},
/* 15348 */ {(11<<2)|2,{111,57,0}},
/* 15349 */ {(11<<2)|2,{111,57,0}},
/* 15350 */ {(11<<2)|2,{111,57,0}},
/* 15351 */ {(11<<2)|2,{111,57,0}},
/* 15352 */ {(11<<2)|2,{111,57,0}},
/* 15353 */ {(11<<2)|2,{111,57,0}},
/* 15354 */ {(11<<2)|2,{111,57,0}},
/* 15355 */ {(11<<2)|2,{111,57,0}},
/* 15356 */ {(11<<2)|2,{111,57,0}},
/* 15357 */ {(11<<2)|2,{111,57,0}},
/* 15358 */ {(11<<2)|2,{111,57,0}},
/* 15359 */ {(11<<2)|2,{111,57,0}},
/* 15360 */ {(16<<2)|3,{111,61,48}},
/* 15361 */ {(16<<2)|3,{111,61,49}},
/* 15362 */ {(16<<2)|3,{111,61,50}},
/* 15363 */ {(16<<2)|3,{111,61,97}},
/* 15364 */ {(16<<2)|3,{111,61,99}},
/* 15365 */ {(16<<2)|3,{111,61,101}},
/* 15366 */ {(16<<2)|3,{111,61,105}},
/* 15367 */ {(16<<2)|3,{111,61,111}},
/* 15368 */ {(16<<2)|3,{111,61,115}},
/* 15369 */ {(16<<2)|3,{111,61,116}},
/* 15370 */ {(11<<2)|2,{111,61,0}},
/* 15371 */ {(11<<2)|2,{111,61,0}},
/* 15372 */ {(11<<2)|2,{111,61,0}},
/* 15373 */ {(11<<2)|2,{111,61,0}},
/* 15374 */ {(11<<2)|2,{111,61,0}},
/* 15375 */ {(11<<2)|2,{111,61,0}},
/* 15376 */ {(11<<2)|2,{111,61,0}},
/* 15377 */ {(11<<2)|2,{111,61,0}},
/* 15378 */ {(11<<2)|2,{111,61,0}},
/* 15379 */ {(11<<2)|2,{111,61,0}},
/* 15380 */ {(11<<2)|2,{111,61,0}},
/* 15381 */ {(11<<2)|2,{111,61,0}},
/* 15382 */ {(11<<2)|2,{111,61,0}},
/* 15383 */ {(11<<2)|2,{111,61,0}},
/* 15384 */ {(11<<2)|2,{111,61,0}},
/* 15385 */ {(11<<2)|2,{111,61,0}},
/* 15386 */ {(11<<2)|2,{111,61,0}},
/* 15387 */ {(11<<2)|2,{111,61,0}},
/* 15388 */ {(11<<2)|2,{111,61,0}},
/* 15389 */ {(11<<2)|2,{111,61,0}},
/* 15390 */ {(11<<2)|2,{111,61,0}},
/* 15391 */ {(11<<2)|2,{111,61,0}},
/* 15392 */ {(16<<2)|3,{111,65,48}},
/* 15393 */ {(16<<2)|3,{111,65,49}},
/* 15394 */ {(16<<2)|3,{111,65,50}},
/* 15395 */ {(16<<2)|3,{111,65,97}},
/* 15396 */ {(16<<2)|3,{111,65,99}},
/* 15397 */ {(16<<2)|3,{111,65,101}},
/* 15398 */ {(16<<2)|3,{111,65,105}},
/* 15399 */ {(16<<2)|3,{111,65,111}},
/* 15400 */ {(16<<2)|3,{111,65,115}},
/* 15401 */ {(16<<2)|3,{111,65,116}},
/* 15402 */ {(11<<2)|2,{111,65,0}},
/* 15403 */ {(11<<2)|2,{111,65,0}},
/* 15404 */ {(11<<2)|2,{111,65,0}},
/* 15405 */ {(11<<2)|2,{111,65,0}},
/* 15406 */ {(11<<2)|2,{111,65,0}},
/* 15407 */ {(11<<2)|2,{111,65,0}},
/* 15408 */ {(11<<2)|2,{111,65,0}},
/* 15409 */ {(11<<2)|2,{111,65,0}},
/* 15410 */ {(11<<2)|2,{111,65,0}},
/* 15411 */ {(11<<2)|2,{111,65,0}},
/* 15412 */ {(11<<2)|2,{111,65,0}},
/* 15413 */ {(11<<2)|2,{111,65,0}},
/* 15414 */ {(11<<2)|2,{111,65,0}},
/* 15415 */ {(11<<2)|2,{111,65,0}},
/* 15416 */ {(11<<2)|2,{111,65,0}},
/* 15417 */ {(11<<2)|2,{111,65,0}},
/* 15418 */ {(11<<2)|2,{111,65,0}},
/* 15419 */ {(11<<2)|2,{111,65,0}},
/* 15420 */ {(11<<2)|2,{111,65,0}},
/* 15421 */ {(11<<2)|2,{111,65,0}},
/* 15422 */ {(11<<2)|2,{111,65,0}},
/* 15423 */ {(11<<2)|2,{111,65,0}},
/* 15424 */ {(16<<2)|3,{111,95,48}},
/* 15425 */ {(16<<2)|3,{111,95,49}},
/* 15426 */ {(16<<2)|3,{111,95,50}},
/* 15427 */ {(16<<2)|3,{111,95,97}},
/* 15428 */ {(16<<2)|3,{111,95,99}},
/* 15429 */ {(16<<2)|3,{111,95,101}},
/* 15430 */ {(16<<2)|3,{111,95,105}},
/* 15431 */ {(16<<2)|3,{111,95,111}},
/* 15432 */ {(16<<2)|3,{111,95,115}},
/* 15433 */ {(16<<2)|3,{111,95,116}},
/* 15434 */ {(11<<2)|2,{111,95,0}},
/* 15435 */ {(11<<2)|2,{111,95,0}},
/* 15436 */ {(11<<2)|2,{111,95,0}},
/* 15437 */ {(11<<2)|2,{111,95,0}},
/* 15438 */ {(11<<2)|2,{111,95,0}},
/* 15439 */ {(11<<2)|2,{111,95,0}},
/* 15440 */ {(11<<2)|2,{111,95,0}},
/* 15441 */ {(11<<2)|2,{111,95,0}},
/* 15442 */ {(11<<2)|2,{111,95,0}},
/* 15443 */ {(11<<2)|2,{111,95,0}},
/* 15444 */ {(11<<2)|2,{111,95,0}},
/* 15445 */ {(11<<2)|2,{111,95,0}},
/* 15446 */ {(11<<2)|2,{111,95,0}},
/* 15447 */ {(11<<2)|2,{111,95,0}},
/* 15448 */ {(11<<2)|2,{111,95,0}},
/* 15449 */ {(11<<2)|2,{111,95,0}},
/* 15450 */ {(11<<2)|2,{111,95,0}},
/* 15451 */ {(11<<2)|2,{111,95,0}},
/* 15452 */ {(11<<2)|2,{111,95,0}},
/* 15453 */ {(11<<2)|2,{111,95,0}},
/* 15454 */ {(11<<2)|2,{111,95,0}},
/* 15455 */ {(11<<2)|2,{111,95,0}},
/* 15456 */ {(16<<2)|3,{111,98,48}},
/* 15457 */ {(16<<2)|3,{111,98,49}},
/* 15458 */ {(16<<2)|3,{111,98,50}},
/* 15459 */ {(16<<2)|3,{111,98,97}},
/* 15460 */ {(16<<2)|3,{111,98,99}},
/* 15461 */ {(16<<2)|3,{111,98,101}},
/* 15462 */ {(16<<2)|3,{111,98,105}},
/* 15463 */ {(16<<2)|3,{111,98,111}},
/* 15464 */ {(16<<2)|3,{111,98,115}},
/* 15465 */ {(16<<2)|3,{111,98,116}},
/* 15466 */ {(11<<2)|2,{111,98,0}},
/* 15467 */ {(11<<2)|2,{111,98,0}},
/* 15468 */ {(11<<2)|2,{111,98,0}},
/* 15469 */ {(11<<2)|2,{111,98,0}},
/* 15470 */ {(11<<2)|2,{111,98,0}},
/* 15471 */ {(11<<2)|2,{111,98,0}},
/* 15472 */ {(11<<2)|2,{111,98,0}},
/* 15473 */ {(11<<2)|2,{111,98,0}},
/* 15474 */ {(11<<2)|2,{111,98,0}},
/* 15475 */ {(11<<2)|2,{111,98,0}},
/* 15476 */ {(11<<2)|2,{111,98,0}},
/* 15477 */ {(11<<2)|2,{111,98,0}},
/* 15478 */ {(11<<2)|2,{111,98,0}},
/* 15479 */ {(11<<2)|2,{111,98,0}},
/* 15480 */ {(11<<2)|2,{111,98,0}},
/* 15481 */ {(11<<2)|2,{111,98,0}},
/* 15482 */ {(11<<2)|2,{111,98,0}},
/* 15483 */ {(11<<2)|2,{111,98,0}},
/* 15484 */ {(11<<2)|2,{111,98,0}},
/* 15485 */ {(11<<2)|2,{111,98,0}},
/* 15486 */ {(11<<2)|2,{111,98,0}},
/* 15487 */ {(11<<2)|2,{111,98,0}},
/* 15488 */ {(16<<2)|3,{111,100,48}},
/* 15489 */ {(16<<2)|3,{111,100,49}},
/* 15490 */ {(16<<2)|3,{111,100,50}},
/* 15491 */ {(16<<2)|3,{111,100,97}},
/* 15492 */ {(16<<2)|3,{111,100,99}},
/* 15493 */ {(16<<2)|3,{111,100,101}},
/* 15494 */ {(16<<2)|3,{111,100,105}},
/* 15495 */ {(16<<2)|3,{111,100,111}},
/* 15496 */ {(16<<2)|3,{111,100,115}},
/* 15497 */ {(16<<2)|3,{111,100,116}},
/* 15498 */ {(11<<2)|2,{111,100,0}},
/* 15499 */ {(11<<2)|2,{111,100,0}},
/* 15500 */ {(11<<2)|2,{111,100,0}},
/* 15501 */ {(11<<2)|2,{111,100,0}},
/* 15502 */ {(11<<2)|2,{111,100,0}},
/* 15503 */ {(11<<2)|2,{111,100,0}},
/* 15504 */ {(11<<2)|2,{111,100,0}},
/* 15505 */ {(11<<2)|2,{111,100,0}},
/* 15506 */ {(11<<2)|2,{111,100,0}},
/* 15507 */ {(11<<2)|2,{111,100,0}},
/* 15508 */ {(11<<2)|2,{111,100,0}},
/* 15509 */ {(11<<2)|2,{111,100,0}},
/* 15510 */ {(11<<2)|2,{111,100,0}},
/* 15511 */ {(11<<2)|2,{111,100,0}},
/* 15512 */ {(11<<2)|2,{111,100,0}},
/* 15513 */ {(11<<2)|2,{111,100,0}},
/* 15514 */ {(11<<2)|2,{111,100,0}},
/* 15515 */ {(11<<2)|2,{111,100,0}},
/* 15516 */ {(11<<2)|2,{111,100,0}},
/* 15517 */ {(11<<2)|2,{111,100,0}},
/* 15518 */ {(11<<2)|2,{111,100,0}},
/* 15519 */ {(11<<2)|2,{111,100,0}},
/* 15520 */ {(16<<2)|3,{111,102,48}},
/* 15521 */ {(16<<2)|3,{111,102,49}},
/* 15522 */ {(16<<2)|3,{111,102,50}},
/* 15523 */ {(16<<2)|3,{111,102,97}},
/* 15524 */ {(16<<2)|3,{111,102,99}},
/* 15525 */ {(16<<2)|3,{111,102,101}},
/* 15526 */ {(16<<2)|3,{111,102,105}},
/* 15527 */ {(16<<2)|3,{111,102,111}},
/* 15528 */ {(16<<2)|3,{111,102,115}},
/* 15529 */ {(16<<2)|3,{111,102,116}},
/* 15530 */ {(11<<2)|2,{111,102,0}},
/* 15531 */ {(11<<2)|2,{111,102,0}},
/* 15532 */ {(11<<2)|2,{111,102,0}},
/* 15533 */ {(11<<2)|2,{111,102,0}},
/* 15534 */ {(11<<2)|2,{111,102,0}},
/* 15535 */ {(11<<2)|2,{111,102,0}},
/* 15536 */ {(11<<2)|2,{111,102,0}},
/* 15537 */ {(11<<2)|2,{111,102,0}},
/* 15538 */ {(11<<2)|2,{111,102,0}},
/* 15539 */ {(11<<2)|2,{111,102,0}},
/* 15540 */ {(11<<2)|2,{111,102,0}},
/* 15541 */ {(11<<2)|2,{111,102,0}},
/* 15542 */ {(11<<2)|2,{111,102,0}},
/* 15543 */ {(11<<2)|2,{111,102,0}},
/* 15544 */ {(11<<2)|2,{111,102,0}},
/* 15545 */ {(11<<2)|2,{111,102,0}},
/* 15546 */ {(11<<2)|2,{111,102,0}},
/* 15547 */ {(11<<2)|2,{111,102,0}},
/* 15548 */ {(11<<2)|2,{111,102,0}},
/* 15549 */ {(11<<2)|2,{111,102,0}},
/* 15550 */ {(11<<2)|2,{111,102,0}},
/* 15551 */ {(11<<2)|2,{111,102,0}},
/* 15552 */ {(16<<2)|3,{111,103,48}},
/* 15553 */ {(16<<2)|3,{111,103,49}},
/* 15554 */ {(16<<2)|3,{111,103,50}},
/* 15555 */ {(16<<2)|3,{111,103,97}},
/* 15556 */ {(16<<2)|3,{111,103,99}},
/* 15557 */ {(16<<2)|3,{111,103,101}},
/* 15558 */ {(16<<2)|3,{111,103,105}},
/* 15559 */ {(16<<2)|3,{111,103,111}},
/* 15560 */ {(16<<2)|3,{111,103,115}},
/* 15561 */ {(16<<2)|3,{111,103,116}},
/* 15562 */ {(11<<2)|2,{111,103,0}},
/* 15563 */ {(11<<2)|2,{111,103,0}},
/* 15564 */ {(11<<2)|2,{111,103,0}},
/* 15565 */ {(11<<2)|2,{111,103,0}},
/* 15566 */ {(11<<2)|2,{111,103,0}},
/* 15567 */ {(11<<2)|2,{111,103,0}},
/* 15568 */ {(11<<2)|2,{111,103,0}},
/* 15569 */ {(11<<2)|2,{111,103,0}},
/* 15570 */ {(11<<2)|2,{111,103,0}},
/* 15571 */ {(11<<2)|2,{111,103,0}},
/* 15572 */ {(11<<2)|2,{111,103,0}},
/* 15573 */ {(11<<2)|2,{111,103,0}},
/* 15574 */ {(11<<2)|2,{111,103,0}},
/* 15575 */ {(11<<2)|2,{111,103,0}},
/* 15576 */ {(11<<2)|2,{111,103,0}},
/* 15577 */ {(11<<2)|2,{111,103,0}},
/* 15578 */ {(11<<2)|2,{111,103,0}},
/* 15579 */ {(11<<2)|2,{111,103,0}},
/* 15580 */ {(11<<2)|2,{111,103,0}},
/* 15581 */ {(11<<2)|2,{111,103,0}},
/* 15582 */ {(11<<2)|2,{111,103,0}},
/* 15583 */ {(11<<2)|2,{111,103,0}},
/* 15584 */ {(16<<2)|3,{111,104,48}},
/* 15585 */ {(16<<2)|3,{111,104,49}},
/* 15586 */ {(16<<2)|3,{111,104,50}},
/* 15587 */ {(16<<2)|3,{111,104,97}},
/* 15588 */ {(16<<2)|3,{111,104,99}},
/* 15589 */ {(16<<2)|3,{111,104,101}},
/* 15590 */ {(16<<2)|3,{111,104,105}},
/* 15591 */ {(16<<2)|3,{111,104,111}},
/* 15592 */ {(16<<2)|3,{111,104,115}},
/* 15593 */ {(16<<2)|3,{111,104,116}},
/* 15594 */ {(11<<2)|2,{111,104,0}},
/* 15595 */ {(11<<2)|2,{111,104,0}},
/* 15596 */ {(11<<2)|2,{111,104,0}},
/* 15597 */ {(11<<2)|2,{111,104,0}},
/* 15598 */ {(11<<2)|2,{111,104,0}},
/* 15599 */ {(11<<2)|2,{111,104,0}},
/* 15600 */ {(11<<2)|2,{111,104,0}},
/* 15601 */ {(11<<2)|2,{111,104,0}},
/* 15602 */ {(11<<2)|2,{111,104,0}},
/* 15603 */ {(11<<2)|2,{111,104,0}},
/* 15604 */ {(11<<2)|2,{111,104,0}},
/* 15605 */ {(11<<2)|2,{111,104,0}},
/* 15606 */ {(11<<2)|2,{111,104,0}},
/* 15607 */ {(11<<2)|2,{111,104,0}},
/* 15608 */ {(11<<2)|2,{111,104,0}},
/* 15609 */ {(11<<2)|2,{111,104,0}},
/* 15610 */ {(11<<2)|2,{111,104,0}},
/* 15611 */ {(11<<2)|2,{111,104,0}},
/* 15612 */ {(11<<2)|2,{111,104,0}},
/* 15613 */ {(11<<2)|2,{111,104,0}},
/* 15614 */ {(11<<2)|2,{111,104,0}},
/* 15615 */ {(11<<2)|2,{111,104,0}},
/* 15616 */ {(16<<2)|3,{111,108,48}},
/* 15617 */ {(16<<2)|3,{111,108,49}},
/* 15618 */ {(16<<2)|3,{111,108,50}},
/* 15619 */ {(16<<2)|3,{111,108,97}},
/* 15620 */ {(16<<2)|3,{111,108,99}},
/* 15621 */ {(16<<2)|3,{111,108,101}},
/* 15622 */ {(16<<2)|3,{111,108,105}},
/* 15623 */ {(16<<2)|3,{111,108,111}},
/* 15624 */ {(16<<2)|3,{111,108,115}},
/* 15625 */ {(16<<2)|3,{111,108,116}},
/* 15626 */ {(11<<2)|2,{111,108,0}},
/* 15627 */ {(11<<2)|2,{111,108,0}},
/* 15628 */ {(11<<2)|2,{111,108,0}},
/* 15629 */ {(11<<2)|2,{111,108,0}},
/* 15630 */ {(11<<2)|2,{111,108,0}},
/* 15631 */ {(11<<2)|2,{111,108,0}},
/* 15632 */ {(11<<2)|2,{111,108,0}},
/* 15633 */ {(11<<2)|2,{111,108,0}},
/* 15634 */ {(11<<2)|2,{111,108,0}},
/* 15635 */ {(11<<2)|2,{111,108,0}},
/* 15636 */ {(11<<2)|2,{111,108,0}},
/* 15637 */ {(11<<2)|2,{111,108,0}},
/* 15638 */ {(11<<2)|2,{111,108,0}},
/* 15639 */ {(11<<2)|2,{111,108,0}},
/* 15640 */ {(11<<2)|2,{111,108,0}},
/* 15641 */ {(11<<2)|2,{111,108,0}},
/* 15642 */ {(11<<2)|2,{111,108,0}},
/* 15643 */ {(11<<2)|2,{111,108,0}},
/* 15644 */ {(11<<2)|2,{111,108,0}},
/* 15645 */ {(11<<2)|2,{111,108,0}},
/* 15646 */ {(11<<2)|2,{111,108,0}},
/* 15647 */ {(11<<2)|2,{111,108,0}},
/* 15648 */ {(16<<2)|3,{111,109,48}},
/* 15649 */ {(16<<2)|3,{111,109,49}},
/* 15650 */ {(16<<2)|3,{111,109,50}},
/* 15651 */ {(16<<2)|3,{111,109,97}},
/* 15652 */ {(16<<2)|3,{111,109,99}},
/* 15653 */ {(16<<2)|3,{111,109,101}},
/* 15654 */ {(16<<2)|3,{111,109,105}},
/* 15655 */ {(16<<2)|3,{111,109,111}},
/* 15656 */ {(16<<2)|3,{111,109,115}},
/* 15657 */ {(16<<2)|3,{111,109,116}},
/* 15658 */ {(11<<2)|2,{111,109,0}},
/* 15659 */ {(11<<2)|2,{111,109,0}},
/* 15660 */ {(11<<2)|2,{111,109,0}},
/* 15661 */ {(11<<2)|2,{111,109,0}},
/* 15662 */ {(11<<2)|2,{111,109,0}},
/* 15663 */ {(11<<2)|2,{111,109,0}},
/* 15664 */ {(11<<2)|2,{111,109,0}},
/* 15665 */ {(11<<2)|2,{111,109,0}},
/* 15666 */ {(11<<2)|2,{111,109,0}},
/* 15667 */ {(11<<2)|2,{111,109,0}},
/* 15668 */ {(11<<2)|2,{111,109,0}},
/* 15669 */ {(11<<2)|2,{111,109,0}},
/* 15670 */ {(11<<2)|2,{111,109,0}},
/* 15671 */ {(11<<2)|2,{111,109,0}},
/* 15672 */ {(11<<2)|2,{111,109,0}},
/* 15673 */ {(11<<2)|2,{111,109,0}},
/* 15674 */ {(11<<2)|2,{111,109,0}},
/* 15675 */ {(11<<2)|2,{111,109,0}},
/* 15676 */ {(11<<2)|2,{111,109,0}},
/* 15677 */ {(11<<2)|2,{111,109,0}},
/* 15678 */ {(11<<2)|2,{111,109,0}},
/* 15679 */ {(11<<2)|2,{111,109,0}},
/* 15680 */ {(16<<2)|3,{111,110,48}},
/* 15681 */ {(16<<2)|3,{111,110,49}},
/* 15682 */ {(16<<2)|3,{111,110,50}},
/* 15683 */ {(16<<2)|3,{111,110,97}},
/* 15684 */ {(16<<2)|3,{111,110,99}},
/* 15685 */ {(16<<2)|3,{111,110,101}},
/* 15686 */ {(16<<2)|3,{111,110,105}},
/* 15687 */ {(16<<2)|3,{111,110,111}},
/* 15688 */ {(16<<2)|3,{111,110,115}},
/* 15689 */ {(16<<2)|3,{111,110,116}},
/* 15690 */ {(11<<2)|2,{111,110,0}},
/* 15691 */ {(11<<2)|2,{111,110,0}},
/* 15692 */ {(11<<2)|2,{111,110,0}},
/* 15693 */ {(11<<2)|2,{111,110,0}},
/* 15694 */ {(11<<2)|2,{111,110,0}},
/* 15695 */ {(11<<2)|2,{111,110,0}},
/* 15696 */ {(11<<2)|2,{111,110,0}},
/* 15697 */ {(11<<2)|2,{111,110,0}},
/* 15698 */ {(11<<2)|2,{111,110,0}},
/* 15699 */ {(11<<2)|2,{111,110,0}},
/* 15700 */ {(11<<2)|2,{111,110,0}},
/* 15701 */ {(11<<2)|2,{111,110,0}},
/* 15702 */ {(11<<2)|2,{111,110,0}},
/* 15703 */ {(11<<2)|2,{111,110,0}},
/* 15704 */ {(11<<2)|2,{111,110,0}},
/* 15705 */ {(11<<2)|2,{111,110,0}},
/* 15706 */ {(11<<2)|2,{111,110,0}},
/* 15707 */ {(11<<2)|2,{111,110,0}},
/* 15708 */ {(11<<2)|2,{111,110,0}},
/* 15709 */ {(11<<2)|2,{111,110,0}},
/* 15710 */ {(11<<2)|2,{111,110,0}},
/* 15711 */ {(11<<2)|2,{111,110,0}},
/* 15712 */ {(16<<2)|3,{111,112,48}},
/* 15713 */ {(16<<2)|3,{111,112,49}},
/* 15714 */ {(16<<2)|3,{111,112,50}},
/* 15715 */ {(16<<2)|3,{111,112,97}},
/* 15716 */ {(16<<2)|3,{111,112,99}},
/* 15717 */ {(16<<2)|3,{111,112,101}},
/* 15718 */ {(16<<2)|3,{111,112,105}},
/* 15719 */ {(16<<2)|3,{111,112,111}},
/* 15720 */ {(16<<2)|3,{111,112,115}},
/* 15721 */ {(16<<2)|3,{111,112,116}},
/* 15722 */ {(11<<2)|2,{111,112,0}},
/* 15723 */ {(11<<2)|2,{111,112,0}},
/* 15724 */ {(11<<2)|2,{111,112,0}},
/* 15725 */ {(11<<2)|2,{111,112,0}},
/* 15726 */ {(11<<2)|2,{111,112,0}},
/* 15727 */ {(11<<2)|2,{111,112,0}},
/* 15728 */ {(11<<2)|2,{111,112,0}},
/* 15729 */ {(11<<2)|2,{111,112,0}},
/* 15730 */ {(11<<2)|2,{111,112,0}},
/* 15731 */ {(11<<2)|2,{111,112,0}},
/* 15732 */ {(11<<2)|2,{111,112,0}},
/* 15733 */ {(11<<2)|2,{111,112,0}},
/* 15734 */ {(11<<2)|2,{111,112,0}},
/* 15735 */ {(11<<2)|2,{111,112,0}},
/* 15736 */ {(11<<2)|2,{111,112,0}},
/* 15737 */ {(11<<2)|2,{111,112,0}},
/* 15738 */ {(11<<2)|2,{111,112,0}},
/* 15739 */ {(11<<2)|2,{111,112,0}},
/* 15740 */ {(11<<2)|2,{111,112,0}},
/* 15741 */ {(11<<2)|2,{111,112,0}},
/* 15742 */ {(11<<2)|2,{111,112,0}},
/* 15743 */ {(11<<2)|2,{111,112,0}},
/* 15744 */ {(16<<2)|3,{111,114,48}},
/* 15745 */ {(16<<2)|3,{111,114,49}},
/* 15746 */ {(16<<2)|3,{111,114,50}},
/* 15747 */ {(16<<2)|3,{111,114,97}},
/* 15748 */ {(16<<2)|3,{111,114,99}},
/* 15749 */ {(16<<2)|3,{111,114,101}},
/* 15750 */ {(16<<2)|3,{111,114,105}},
/* 15751 */ {(16<<2)|3,{111,114,111}},
/* 15752 */ {(16<<2)|3,{111,114,115}},
/* 15753 */ {(16<<2)|3,{111,114,116}},
/* 15754 */ {(11<<2)|2,{111,114,0}},
/* 15755 */ {(11<<2)|2,{111,114,0}},
/* 15756 */ {(11<<2)|2,{111,114,0}},
/* 15757 */ {(11<<2)|2,{111,114,0}},
/* 15758 */ {(11<<2)|2,{111,114,0}},
/* 15759 */ {(11<<2)|2,{111,114,0}},
/* 15760 */ {(11<<2)|2,{111,114,0}},
/* 15761 */ {(11<<2)|2,{111,114,0}},
/* 15762 */ {(11<<2)|2,{111,114,0}},
/* 15763 */ {(11<<2)|2,{111,114,0}},
/* 15764 */ {(11<<2)|2,{111,114,0}},
/* 15765 */ {(11<<2)|2,{111,114,0}},
/* 15766 */ {(11<<2)|2,{111,114,0}},
/* 15767 */ {(11<<2)|2,{111,114,0}},
/* 15768 */ {(11<<2)|2,{111,114,0}},
/* 15769 */ {(11<<2)|2,{111,114,0}},
/* 15770 */ {(11<<2)|2,{111,114,0}},
/* 15771 */ {(11<<2)|2,{111,114,0}},
/* 15772 */ {(11<<2)|2,{111,114,0}},
/* 15773 */ {(11<<2)|2,{111,114,0}},
/* 15774 */ {(11<<2)|2,{111,114,0}},
/* 15775 */ {(11<<2)|2,{111,114,0}},
/* 15776 */ {(16<<2)|3,{111,117,48}},
/* 15777 */ {(16<<2)|3,{111,117,49}},
/* 15778 */ {(16<<2)|3,{111,117,50}},
/* 15779 */ {(16<<2)|3,{111,117,97}},
/* 15780 */ {(16<<2)|3,{111,117,99}},
/* 15781 */ {(16<<2)|3,{111,117,101}},
/* 15782 */ {(16<<2)|3,{111,117,105}},
/* 15783 */ {(16<<2)|3,{111,117,111}},
/* 15784 */ {(16<<2)|3,{111,117,115}},
/* 15785 */ {(16<<2)|3,{111,117,116}},
/* 15786 */ {(11<<2)|2,{111,117,0}},
/* 15787 */ {(11<<2)|2,{111,117,0}},
/* 15788 */ {(11<<2)|2,{111,117,0}},
/* 15789 */ {(11<<2)|2,{111,117,0}},
/* 15790 */ {(11<<2)|2,{111,117,0}},
/* 15791 */ {(11<<2)|2,{111,117,0}},
/* 15792 */ {(11<<2)|2,{111,117,0}},
/* 15793 */ {(11<<2)|2,{111,117,0}},
/* 15794 */ {(11<<2)|2,{111,117,0}},
/* 15795 */ {(11<<2)|2,{111,117,0}},
/* 15796 */ {(11<<2)|2,{111,117,0}},
/* 15797 */ {(11<<2)|2,{111,117,0}},
/* 15798 */ {(11<<2)|2,{111,117,0}},
/* 15799 */ {(11<<2)|2,{111,117,0}},
/* 15800 */ {(11<<2)|2,{111,117,0}},
/* 15801 */ {(11<<2)|2,{111,117,0}},
/* 15802 */ {(11<<2)|2,{111,117,0}},
/* 15803 */ {(11<<2)|2,{111,117,0}},
/* 15804 */ {(11<<2)|2,{111,117,0}},
/* 15805 */ {(11<<2)|2,{111,117,0}},
/* 15806 */ {(11<<2)|2,{111,117,0}},
/* 15807 */ {(11<<2)|2,{111,117,0}},
/* 15808 */ {(12<<2)|2,{111,58,0}},
/* 15809 */ {(12<<2)|2,{111,58,0}},
/* 15810 */ {(12<<2)|2,{111,58,0}},
/* 15811 */ {(12<<2)|2,{111,58,0}},
/* 15812 */ {(12<<2)|2,{111,58,0}},
/* 15813 */ {(12<<2)|2,{111,58,0}},
/* 15814 */ {(12<<2)|2,{111,58,0}},
/* 15815 */ {(12<<2)|2,{111,58,0}},
/* 15816 */ {(12<<2)|2,{111,58,0}},
/* 15817 */ {(12<<2)|2,{111,58,0}},
/* 15818 */ {(12<<2)|2,{111,58,0}},
/* 15819 */ {(12<<2)|2,{111,58,0}},
/* 15820 */ {(12<<2)|2,{111,58,0}},
/* 15821 */ {(12<<2)|2,{111,58,0}},
/* 15822 */ {(12<<2)|2,{111,58,0}},
/* 15823 */ {(12<<2)|2,{111,58,0}},
/* 15824 */ {(12<<2)|2,{111,66,0}},
/* 15825 */ {(12<<2)|2,{111,66,0}},
/* 15826 */ {(12<<2)|2,{111,66,0}},
/* 15827 */ {(12<<2)|2,{111,66,0}},
/* 15828 */ {(12<<2)|2,{111,66,0}},
/* 15829 */ {(12<<2)|2,{111,66,0}},
/* 15830 */ {(12<<2)|2,{111,66,0}},
/* 15831 */ {(12<<2)|2,{111,66,0}},
/* 15832 */ {(12<<2)|2,{111,66,0}},
/* 15833 */ {(12<<2)|2,{111,66,0}},
/* 15834 */ {(12<<2)|2,{111,66,0}},
/* 15835 */ {(12<<2)|2,{111,66,0}},
/* 15836 */ {(12<<2)|2,{111,66,0}},
/* 15837 */ {(12<<2)|2,{111,66,0}},
/* 15838 */ {(12<<2)|2,{111,66,0}},
/* 15839 */ {(12<<2)|2,{111,66,0}},
/* 15840 */ {(12<<2)|2,{111,67,0}},
/* 15841 */ {(12<<2)|2,{111,67,0}},
/* 15842 */ {(12<<2)|2,{111,67,0}},
/* 15843 */ {(12<<2)|2,{111,67,0}},
/* 15844 */ {(12<<2)|2,{111,67,0}},
/* 15845 */ {(12<<2)|2,{111,67,0}},
/* 15846 */ {(12<<2)|2,{111,67,0}},
/* 15847 */ {(12<<2)|2,{111,67,0}},
/* 15848 */ {(12<<2)|2,{111,67,0}},
/* 15849 */ {(12<<2)|2,{111,67,0}},
/* 15850 */ {(12<<2)|2,{111,67,0}},
/* 15851 */ {(12<<2)|2,{111,67,0}},
/* 15852 */ {(12<<2)|2,{111,67,0}},
/* 15853 */ {(12<<2)|2,{111,67,0}},
/* 15854 */ {(12<<2)|2,{111,67,0}},
/* 15855 */ {(12<<2)|2,{111,67,0}},
/* 15856 */ {(12<<2)|2,{111,68,0}},
/* 15857 */ {(12<<2)|2,{111,68,0}},
/* 15858 */ {(12<<2)|2,{111,68,0}},
/* 15859 */ {(12<<2)|2,{111,68,0}},
/* 15860 */ {(12<<2)|2,{111,68,0}},
/* 15861 */ {(12<<2)|2,{111,68,0}},
/* 15862 */ {(12<<2)|2,{111,68,0}},
/* 15863 */ {(12<<2)|2,{111,68,0}},
/* 15864 */ {(12<<2)|2,{111,68,0}},
/* 15865 */ {(12<<2)|2,{111,68,0}},
/* 15866 */ {(12<<2)|2,{111,68,0}},
/* 15867 */ {(12<<2)|2,{111,68,0}},
/* 15868 */ {(12<<2)|2,{111,68,0}},
/* 15869 */ {(12<<2)|2,{111,68,0}},
/* 15870 */ {(12<<2)|2,{111,68,0}},
/* 15871 */ {(12<<2)|2,{111,68,0}},
/* 15872 */ {(12<<2)|2,{111,69,0}},
/* 15873 */ {(12<<2)|2,{111,69,0}},
/* 15874 */ {(12<<2)|2,{111,69,0}},
/* 15875 */ {(12<<2)|2,{111,69,0}},
/* 15876 */ {(12<<2)|2,{111,69,0}},
/* 15877 */ {(12<<2)|2,{111,69,0}},
/* 15878 */ {(12<<2)|2,{111,69,0}},
/* 15879 */ {(12<<2)|2,{111,69,0}},
/* 15880 */ {(12<<2)|2,{111,69,0}},
/* 15881 */ {(12<<2)|2,{111,69,0}},
/* 15882 */ {(12<<2)|2,{111,69,0}},
/* 15883 */ {(12<<2)|2,{111,69,0}},
/* 15884 */ {(12<<2)|2,{111,69,0}},
/* 15885 */ {(12<<2)|2,{111,69,0}},
/* 15886 */ {(12<<2)|2,{111,69,0}},
/* 15887 */ {(12<<2)|2,{111,69,0}},
/* 15888 */ {(12<<2)|2,{111,70,0}},
/* 15889 */ {(12<<2)|2,{111,70,0}},
/* 15890 */ {(12<<2)|2,{111,70,0}},
/* 15891 */ {(12<<2)|2,{111,70,0}},
/* 15892 */ {(12<<2)|2,{111,70,0}},
/* 15893 */ {(12<<2)|2,{111,70,0}},
/* 15894 */ {(12<<2)|2,{111,70,0}},
/* 15895 */ {(12<<2)|2,{111,70,0}},
/* 15896 */ {(12<<2)|2,{111,70,0}},
/* 15897 */ {(12<<2)|2,{111,70,0}},
/* 15898 */ {(12<<2)|2,{111,70,0}},
/* 15899 */ {(12<<2)|2,{111,70,0}},
/* 15900 */ {(12<<2)|2,{111,70,0}},
/* 15901 */ {(12<<2)|2,{111,70,0}},
/* 15902 */ {(12<<2)|2,{111,70,0}},
/* 15903 */ {(12<<2)|2,{111,70,0}},
/* 15904 */ {(12<<2)|2,{111,71,0}},
/* 15905 */ {(12<<2)|2,{111,71,0}},
/* 15906 */ {(12<<2)|2,{111,71,0}},
/* 15907 */ {(12<<2)|2,{111,71,0}},
/* 15908 */ {(12<<2)|2,{111,71,0}},
/* 15909 */ {(12<<2)|2,{111,71,0}},
/* 15910 */ {(12<<2)|2,{111,71,0}},
/* 15911 */ {(12<<2)|2,{111,71,0}},
/* 15912 */ {(12<<2)|2,{111,71,0}},
/* 15913 */ {(12<<2)|2,{111,71,0}},
/* 15914 */ {(12<<2)|2,{111,71,0}},
/* 15915 */ {(12<<2)|2,{111,71,0}},
/* 15916 */ {(12<<2)|2,{111,71,0}},
/* 15917 */ {(12<<2)|2,{111,71,0}},
/* 15918 */ {(12<<2)|2,{111,71,0}},
/* 15919 */ {(12<<2)|2,{111,71,0}},
/* 15920 */ {(12<<2)|2,{111,72,0}},
/* 15921 */ {(12<<2)|2,{111,72,0}},
/* 15922 */ {(12<<2)|2,{111,72,0}},
/* 15923 */ {(12<<2)|2,{111,72,0}},
/* 15924 */ {(12<<2)|2,{111,72,0}},
/* 15925 */ {(12<<2)|2,{111,72,0}},
/* 15926 */ {(12<<2)|2,{111,72,0}},
/* 15927 */ {(12<<2)|2,{111,72,0}},
/* 15928 */ {(12<<2)|2,{111,72,0}},
/* 15929 */ {(12<<2)|2,{111,72,0}},
/* 15930 */ {(12<<2)|2,{111,72,0}},
/* 15931 */ {(12<<2)|2,{111,72,0}},
/* 15932 */ {(12<<2)|2,{111,72,0}},
/* 15933 */ {(12<<2)|2,{111,72,0}},
/* 15934 */ {(12<<2)|2,{111,72,0}},
/* 15935 */ {(12<<2)|2,{111,72,0}},
/* 15936 */ {(12<<2)|2,{111,73,0}},
/* 15937 */ {(12<<2)|2,{111,73,0}},
/* 15938 */ {(12<<2)|2,{111,73,0}},
/* 15939 */ {(12<<2)|2,{111,73,0}},
/* 15940 */ {(12<<2)|2,{111,73,0}},
/* 15941 */ {(12<<2)|2,{111,73,0}},
/* 15942 */ {(12<<2)|2,{111,73,0}},
/* 15943 */ {(12<<2)|2,{111,73,0}},
/* 15944 */ {(12<<2)|2,{111,73,0}},
/* 15945 */ {(12<<2)|2,{111,73,0}},
/* 15946 */ {(12<<2)|2,{111,73,0}},
/* 15947 */ {(12<<2)|2,{111,73,0}},
/* 15948 */ {(12<<2)|2,{111,73,0}},
/* 15949 */ {(12<<2)|2,{111,73,0}},
/* 15950 */ {(12<<2)|2,{111,73,0}},
/* 15951 */ {(12<<2)|2,{111,73,0}},
/* 15952 */ {(12<<2)|2,{111,74,0}},
/* 15953 */ {(12<<2)|2,{111,74,0}},
/* 15954 */ {(12<<2)|2,{111,74,0}},
/* 15955 */ {(12<<2)|2,{111,74,0}},
/* 15956 */ {(12<<2)|2,{111,74,0}},
/* 15957 */ {(12<<2)|2,{111,74,0}},
/* 15958 */ {(12<<2)|2,{111,74,0}},
/* 15959 */ {(12<<2)|2,{111,74,0}},
/* 15960 */ {(12<<2)|2,{111,74,0}},
/* 15961 */ {(12<<2)|2,{111,74,0}},
/* 15962 */ {(12<<2)|2,{111,74,0}},
/* 15963 */ {(12<<2)|2,{111,74,0}},
/* 15964 */ {(12<<2)|2,{111,74,0}},
/* 15965 */ {(12<<2)|2,{111,74,0}},
/* 15966 */ {(12<<2)|2,{111,74,0}},
/* 15967 */ {(12<<2)|2,{111,74,0}},
/* 15968 */ {(12<<2)|2,{111,75,0}},
/* 15969 */ {(12<<2)|2,{111,75,0}},
/* 15970 */ {(12<<2)|2,{111,75,0}},
/* 15971 */ {(12<<2)|2,{111,75,0}},
/* 15972 */ {(12<<2)|2,{111,75,0}},
/* 15973 */ {(12<<2)|2,{111,75,0}},
/* 15974 */ {(12<<2)|2,{111,75,0}},
/* 15975 */ {(12<<2)|2,{111,75,0}},
/* 15976 */ {(12<<2)|2,{111,75,0}},
/* 15977 */ {(12<<2)|2,{111,75,0}},
/* 15978 */ {(12<<2)|2,{111,75,0}},
/* 15979 */ {(12<<2)|2,{111,75,0}},
/* 15980 */ {(12<<2)|2,{111,75,0}},
/* 15981 */ {(12<<2)|2,{111,75,0}},
/* 15982 */ {(12<<2)|2,{111,75,0}},
/* 15983 */ {(12<<2)|2,{111,75,0}},
/* 15984 */ {(12<<2)|2,{111,76,0}},
/* 15985 */ {(12<<2)|2,{111,76,0}},
/* 15986 */ {(12<<2)|2,{111,76,0}},
/* 15987 */ {(12<<2)|2,{111,76,0}},
/* 15988 */ {(12<<2)|2,{111,76,0}},
/* 15989 */ {(12<<2)|2,{111,76,0}},
/* 15990 */ {(12<<2)|2,{111,76,0}},
/* 15991 */ {(12<<2)|2,{111,76,0}},
/* 15992 */ {(12<<2)|2,{111,76,0}},
/* 15993 */ {(12<<2)|2,{111,76,0}},
/* 15994 */ {(12<<2)|2,{111,76,0}},
/* 15995 */ {(12<<2)|2,{111,76,0}},
/* 15996 */ {(12<<2)|2,{111,76,0}},
/* 15997 */ {(12<<2)|2,{111,76,0}},
/* 15998 */ {(12<<2)|2,{111,76,0}},
/* 15999 */ {(12<<2)|2,{111,76,0}},
/* 16000 */ {(12<<2)|2,{111,77,0}},
/* 16001 */ {(12<<2)|2,{111,77,0}},
/* 16002 */ {(12<<2)|2,{111,77,0}},
/* 16003 */ {(12<<2)|2,{111,77,0}},
/* 16004 */ {(12<<2)|2,{111,77,0}},
/* 16005 */ {(12<<2)|2,{111,77,0}},
/* 16006 */ {(12<<2)|2,{111,77,0}},
/* 16007 */ {(12<<2)|2,{111,77,0}},
/* 16008 */ {(12<<2)|2,{111,77,0}},
/* 16009 */ {(12<<2)|2,{111,77,0}},
/* 16010 */ {(12<<2)|2,{111,77,0}},
/* 16011 */ {(12<<2)|2,{111,77,0}},
/* 16012 */ {(12<<2)|2,{111,77,0}},
/* 16013 */ {(12<<2)|2,{111,77,0}},
/* 16014 */ {(12<<2)|2,{111,77,0}},
/* 16015 */ {(12<<2)|2,{111,77,0}},
/* 16016 */ {(12<<2)|2,{111,78,0}},
/* 16017 */ {(12<<2)|2,{111,78,0}},
/* 16018 */ {(12<<2)|2,{111,78,0}},
/* 16019 */ {(12<<2)|2,{111,78,0}},
/* 16020 */ {(12<<2)|2,{111,78,0}},
/* 16021 */ {(12<<2)|2,{111,78,0}},
/* 16022 */ {(12<<2)|2,{111,78,0}},
/* 16023 */ {(12<<2)|2,{111,78,0}},
/* 16024 */ {(12<<2)|2,{111,78,0}},
/* 16025 */ {(12<<2)|2,{111,78,0}},
/* 16026 */ {(12<<2)|2,{111,78,0}},
/* 16027 */ {(12<<2)|2,{111,78,0}},
/* 16028 */ {(12<<2)|2,{111,78,0}},
/* 16029 */ {(12<<2)|2,{111,78,0}},
/* 16030 */ {(12<<2)|2,{111,78,0}},
/* 16031 */ {(12<<2)|2,{111,78,0}},
/* 16032 */ {(12<<2)|2,{111,79,0}},
/* 16033 */ {(12<<2)|2,{111,79,0}},
/* 16034 */ {(12<<2)|2,{111,79,0}},
/* 16035 */ {(12<<2)|2,{111,79,0}},
/* 16036 */ {(12<<2)|2,{111,79,0}},
/* 16037 */ {(12<<2)|2,{111,79,0}},
/* 16038 */ {(12<<2)|2,{111,79,0}},
/* 16039 */ {(12<<2)|2,{111,79,0}},
/* 16040 */ {(12<<2)|2,{111,79,0}},
/* 16041 */ {(12<<2)|2,{111,79,0}},
/* 16042 */ {(12<<2)|2,{111,79,0}},
/* 16043 */ {(12<<2)|2,{111,79,0}},
/* 16044 */ {(12<<2)|2,{111,79,0}},
/* 16045 */ {(12<<2)|2,{111,79,0}},
/* 16046 */ {(12<<2)|2,{111,79,0}},
/* 16047 */ {(12<<2)|2,{111,79,0}},
/* 16048 */ {(12<<2)|2,{111,80,0}},
/* 16049 */ {(12<<2)|2,{111,80,0}},
/* 16050 */ {(12<<2)|2,{111,80,0}},
/* 16051 */ {(12<<2)|2,{111,80,0}},
/* 16052 */ {(12<<2)|2,{111,80,0}},
/* 16053 */ {(12<<2)|2,{111,80,0}},
/* 16054 */ {(12<<2)|2,{111,80,0}},
/* 16055 */ {(12<<2)|2,{111,80,0}},
/* 16056 */ {(12<<2)|2,{111,80,0}},
/* 16057 */ {(12<<2)|2,{111,80,0}},
/* 16058 */ {(12<<2)|2,{111,80,0}},
/* 16059 */ {(12<<2)|2,{111,80,0}},
/* 16060 */ {(12<<2)|2,{111,80,0}},
/* 16061 */ {(12<<2)|2,{111,80,0}},
/* 16062 */ {(12<<2)|2,{111,80,0}},
/* 16063 */ {(12<<2)|2,{111,80,0}},
/* 16064 */ {(12<<2)|2,{111,81,0}},
/* 16065 */ {(12<<2)|2,{111,81,0}},
/* 16066 */ {(12<<2)|2,{111,81,0}},
/* 16067 */ {(12<<2)|2,{111,81,0}},
/* 16068 */ {(12<<2)|2,{111,81,0}},
/* 16069 */ {(12<<2)|2,{111,81,0}},
/* 16070 */ {(12<<2)|2,{111,81,0}},
/* 16071 */ {(12<<2)|2,{111,81,0}},
/* 16072 */ {(12<<2)|2,{111,81,0}},
/* 16073 */ {(12<<2)|2,{111,81,0}},
/* 16074 */ {(12<<2)|2,{111,81,0}},
/* 16075 */ {(12<<2)|2,{111,81,0}},
/* 16076 */ {(12<<2)|2,{111,81,0}},
/* 16077 */ {(12<<2)|2,{111,81,0}},
/* 16078 */ {(12<<2)|2,{111,81,0}},
/* 16079 */ {(12<<2)|2,{111,81,0}},
/* 16080 */ {(12<<2)|2,{111,82,0}},
/* 16081 */ {(12<<2)|2,{111,82,0}},
/* 16082 */ {(12<<2)|2,{111,82,0}},
/* 16083 */ {(12<<2)|2,{111,82,0}},
/* 16084 */ {(12<<2)|2,{111,82,0}},
/* 16085 */ {(12<<2)|2,{111,82,0}},
/* 16086 */ {(12<<2)|2,{111,82,0}},
/* 16087 */ {(12<<2)|2,{111,82,0}},
/* 16088 */ {(12<<2)|2,{111,82,0}},
/* 16089 */ {(12<<2)|2,{111,82,0}},
/* 16090 */ {(12<<2)|2,{111,82,0}},
/* 16091 */ {(12<<2)|2,{111,82,0}},
/* 16092 */ {(12<<2)|2,{111,82,0}},
/* 16093 */ {(12<<2)|2,{111,82,0}},
/* 16094 */ {(12<<2)|2,{111,82,0}},
/* 16095 */ {(12<<2)|2,{111,82,0}},
/* 16096 */ {(12<<2)|2,{111,83,0}},
/* 16097 */ {(12<<2)|2,{111,83,0}},
/* 16098 */ {(12<<2)|2,{111,83,0}},
/* 16099 */ {(12<<2)|2,{111,83,0}},
/* 16100 */ {(12<<2)|2,{111,83,0}},
/* 16101 */ {(12<<2)|2,{111,83,0}},
/* 16102 */ {(12<<2)|2,{111,83,0}},
/* 16103 */ {(12<<2)|2,{111,83,0}},
/* 16104 */ {(12<<2)|2,{111,83,0}},
/* 16105 */ {(12<<2)|2,{111,83,0}},
/* 16106 */ {(12<<2)|2,{111,83,0}},
/* 16107 */ {(12<<2)|2,{111,83,0}},
/* 16108 */ {(12<<2)|2,{111,83,0}},
/* 16109 */ {(12<<2)|2,{111,83,0}},
/* 16110 */ {(12<<2)|2,{111,83,0}},
/* 16111 */ {(12<<2)|2,{111,83,0}},
/* 16112 */ {(12<<2)|2,{111,84,0}},
/* 16113 */ {(12<<2)|2,{111,84,0}},
/* 16114 */ {(12<<2)|2,{111,84,0}},
/* 16115 */ {(12<<2)|2,{111,84,0}},
/* 16116 */ {(12<<2)|2,{111,84,0}},
/* 16117 */ {(12<<2)|2,{111,84,0}},
/* 16118 */ {(12<<2)|2,{111,84,0}},
/* 16119 */ {(12<<2)|2,{111,84,0}},
/* 16120 */ {(12<<2)|2,{111,84,0}},
/* 16121 */ {(12<<2)|2,{111,84,0}},
/* 16122 */ {(12<<2)|2,{111,84,0}},
/* 16123 */ {(12<<2)|2,{111,84,0}},
/* 16124 */ {(12<<2)|2,{111,84,0}},
/* 16125 */ {(12<<2)|2,{111,84,0}},
/* 16126 */ {(12<<2)|2,{111,84,0}},
/* 16127 */ {(12<<2)|2,{111,84,0}},
/* 16128 */ {(12<<2)|2,{111,85,0}},
/* 16129 */ {(12<<2)|2,{111,85,0}},
/* 16130 */ {(12<<2)|2,{111,85,0}},
/* 16131 */ {(12<<2)|2,{111,85,0}},
/* 16132 */ {(12<<2)|2,{111,85,0}},
/* 16133 */ {(12<<2)|2,{111,85,0}},
/* 16134 */ {(12<<2)|2,{111,85,0}},
/* 16135 */ {(12<<2)|2,{111,85,0}},
/* 16136 */ {(12<<2)|2,{111,85,0}},
/* 16137 */ {(12<<2)|2,{111,85,0}},
/* 16138 */ {(12<<2)|2,{111,85,0}},
/* 16139 */ {(12<<2)|2,{111,85,0}},
/* 16140 */ {(12<<2)|2,{111,85,0}},
/* 16141 */ {(12<<2)|2,{111,85,0}},
/* 16142 */ {(12<<2)|2,{111,85,0}},
/* 16143 */ {(12<<2)|2,{111,85,0}},
/* 16144 */ {(12<<2)|2,{111,86,0}},
/* 16145 */ {(12<<2)|2,{111,86,0}},
/* 16146 */ {(12<<2)|2,{111,86,0}},
/* 16147 */ {(12<<2)|2,{111,86,0}},
/* 16148 */ {(12<<2)|2,{111,86,0}},
/* 16149 */ {(12<<2)|2,{111,86,0}},
/* 16150 */ {(12<<2)|2,{111,86,0}},
/* 16151 */ {(12<<2)|2,{111,86,0}},
/* 16152 */ {(12<<2)|2,{111,86,0}},
/* 16153 */ {(12<<2)|2,{111,86,0}},
/* 16154 */ {(12<<2)|2,{111,86,0}},
/* 16155 */ {(12<<2)|2,{111,86,0}},
/* 16156 */ {(12<<2)|2,{111,86,0}},
/* 16157 */ {(12<<2)|2,{111,86,0}},
/* 16158 */ {(12<<2)|2,{111,86,0}},
/* 16159 */ {(12<<2)|2,{111,86,0}},
/* 16160 */ {(12<<2)|2,{111,87,0}},
/* 16161 */ {(12<<2)|2,{111,87,0}},
/* 16162 */ {(12<<2)|2,{111,87,0}},
/* 16163 */ {(12<<2)|2,{111,87,0}},
/* 16164 */ {(12<<2)|2,{111,87,0}},
/* 16165 */ {(12<<2)|2,{111,87,0}},
/* 16166 */ {(12<<2)|2,{111,87,0}},
/* 16167 */ {(12<<2)|2,{111,87,0}},
/* 16168 */ {(12<<2)|2,{111,87,0}},
/* 16169 */ {(12<<2)|2,{111,87,0}},
/* 16170 */ {(12<<2)|2,{111,87,0}},
/* 16171 */ {(12<<2)|2,{111,87,0}},
/* 16172 */ {(12<<2)|2,{111,87,0}},
/* 16173 */ {(12<<2)|2,{111,87,0}},
/* 16174 */ {(12<<2)|2,{111,87,0}},
/* 16175 */ {(12<<2)|2,{111,87,0}},
/* 16176 */ {(12<<2)|2,{111,89,0}},
/* 16177 */ {(12<<2)|2,{111,89,0}},
/* 16178 */ {(12<<2)|2,{111,89,0}},
/* 16179 */ {(12<<2)|2,{111,89,0}},
/* 16180 */ {(12<<2)|2,{111,89,0}},
/* 16181 */ {(12<<2)|2,{111,89,0}},
/* 16182 */ {(12<<2)|2,{111,89,0}},
/* 16183 */ {(12<<2)|2,{111,89,0}},
/* 16184 */ {(12<<2)|2,{111,89,0}},
/* 16185 */ {(12<<2)|2,{111,89,0}},
/* 16186 */ {(12<<2)|2,{111,89,0}},
/* 16187 */ {(12<<2)|2,{111,89,0}},
/* 16188 */ {(12<<2)|2,{111,89,0}},
/* 16189 */ {(12<<2)|2,{111,89,0}},
/* 16190 */ {(12<<2)|2,{111,89,0}},
/* 16191 */ {(12<<2)|2,{111,89,0}},
/* 16192 */ {(12<<2)|2,{111,106,0}},
/* 16193 */ {(12<<2)|2,{111,106,0}},
/* 16194 */ {(12<<2)|2,{111,106,0}},
/* 16195 */ {(12<<2)|2,{111,106,0}},
/* 16196 */ {(12<<2)|2,{111,106,0}},
/* 16197 */ {(12<<2)|2,{111,106,0}},
/* 16198 */ {(12<<2)|2,{111,106,0}},
/* 16199 */ {(12<<2)|2,{111,106,0}},
/* 16200 */ {(12<<2)|2,{111,106,0}},
/* 16201 */ {(12<<2)|2,{111,106,0}},
/* 16202 */ {(12<<2)|2,{111,106,0}},
/* 16203 */ {(12<<2)|2,{111,106,0}},
/* 16204 */ {(12<<2)|2,{111,106,0}},
/* 16205 */ {(12<<2)|2,{111,106,0}},
/* 16206 */ {(12<<2)|2,{111,106,0}},
/* 16207 */ {(12<<2)|2,{111,106,0}},
/* 16208 */ {(12<<2)|2,{111,107,0}},
/* 16209 */ {(12<<2)|2,{111,107,0}},
/* 16210 */ {(12<<2)|2,{111,107,0}},
/* 16211 */ {(12<<2)|2,{111,107,0}},
/* 16212 */ {(12<<2)|2,{111,107,0}},
/* 16213 */ {(12<<2)|2,{111,107,0}},
/* 16214 */ {(12<<2)|2,{111,107,0}},
/* 16215 */ {(12<<2)|2,{111,107,0}},
/* 16216 */ {(12<<2)|2,{111,107,0}},
/* 16217 */ {(12<<2)|2,{111,107,0}},
/* 16218 */ {(12<<2)|2,{111,107,0}},
/* 16219 */ {(12<<2)|2,{111,107,0}},
/* 16220 */ {(12<<2)|2,{111,107,0}},
/* 16221 */ {(12<<2)|2,{111,107,0}},
/* 16222 */ {(12<<2)|2,{111,107,0}},
/* 16223 */ {(12<<2)|2,{111,107,0}},
/* 16224 */ {(12<<2)|2,{111,113,0}},
/* 16225 */ {(12<<2)|2,{111,113,0}},
/* 16226 */ {(12<<2)|2,{111,113,0}},
/* 16227 */ {(12<<2)|2,{111,113,0}},
/* 16228 */ {(12<<2)|2,{111,113,0}},
/* 16229 */ {(12<<2)|2,{111,113,0}},
/* 16230 */ {(12<<2)|2,{111,113,0}},
/* 16231 */ {(12<<2)|2,{111,113,0}},
/* 16232 */ {(12<<2)|2,{111,113,0}},
/* 16233 */ {(12<<2)|2,{111,113,0}},
/* 16234 */ {(12<<2)|2,{111,113,0}},
/* 16235 */ {(12<<2)|2,{111,113,0}},
/* 16236 */ {(12<<2)|2,{111,113,0}},
/* 16237 */ {(12<<2)|2,{111,113,0}},
/* 16238 */ {(12<<2)|2,{111,113,0}},
/* 16239 */ {(12<<2)|2,{111,113,0}},
/* 16240 */ {(12<<2)|2,{111,118,0}},
/* 16241 */ {(12<<2)|2,{111,118,0}},
/* 16242 */ {(12<<2)|2,{111,118,0}},
/* 16243 */ {(12<<2)|2,{111,118,0}},
/* 16244 */ {(12<<2)|2,{111,118,0}},
/* 16245 */ {(12<<2)|2,{111,118,0}},
/* 16246 */ {(12<<2)|2,{111,118,0}},
/* 16247 */ {(12<<2)|2,{111,118,0}},
/* 16248 */ {(12<<2)|2,{111,118,0}},
/* 16249 */ {(12<<2)|2,{111,118,0}},
/* 16250 */ {(12<<2)|2,{111,118,0}},
/* 16251 */ {(12<<2)|2,{111,118,0}},
/* 16252 */ {(12<<2)|2,{111,118,0}},
/* 16253 */ {(12<<2)|2,{111,118,0}},
/* 16254 */ {(12<<2)|2,{111,118,0}},
/* 16255 */ {(12<<2)|2,{111,118,0}},
/* 16256 */ {(12<<2)|2,{111,119,0}},
/* 16257 */ {(12<<2)|2,{111,119,0}},
/* 16258 */ {(12<<2)|2,{111,119,0}},
/* 16259 */ {(12<<2)|2,{111,119,0}},
/* 16260 */ {(12<<2)|2,{111,119,0}},
/* 16261 */ {(12<<2)|2,{111,119,0}},
/* 16262 */ {(12<<2)|2,{111,119,0}},
/* 16263 */ {(12<<2)|2,{111,119,0}},
/* 16264 */ {(12<<2)|2,{111,119,0}},
/* 16265 */ {(12<<2)|2,{111,119,0}},
/* 16266 */ {(12<<2)|2,{111,119,0}},
/* 16267 */ {(12<<2)|2,{111,119,0}},
/* 16268 */ {(12<<2)|2,{111,119,0}},
/* 16269 */ {(12<<2)|2,{111,119,0}},
/* 16270 */ {(12<<2)|2,{111,119,0}},
/* 16271 */ {(12<<2)|2,{111,119,0}},
/* 16272 */ {(12<<2)|2,{111,120,0}},
/* 16273 */ {(12<<2)|2,{111,120,0}},
/* 16274 */ {(12<<2)|2,{111,120,0}},
/* 16275 */ {(12<<2)|2,{111,120,0}},
/* 16276 */ {(12<<2)|2,{111,120,0}},
/* 16277 */ {(12<<2)|2,{111,120,0}},
/* 16278 */ {(12<<2)|2,{111,120,0}},
/* 16279 */ {(12<<2)|2,{111,120,0}},
/* 16280 */ {(12<<2)|2,{111,120,0}},
/* 16281 */ {(12<<2)|2,{111,120,0}},
/* 16282 */ {(12<<2)|2,{111,120,0}},
/* 16283 */ {(12<<2)|2,{111,120,0}},
/* 16284 */ {(12<<2)|2,{111,120,0}},
/* 16285 */ {(12<<2)|2,{111,120,0}},
/* 16286 */ {(12<<2)|2,{111,120,0}},
/* 16287 */ {(12<<2)|2,{111,120,0}},
/* 16288 */ {(12<<2)|2,{111,121,0}},
/* 16289 */ {(12<<2)|2,{111,121,0}},
/* 16290 */ {(12<<2)|2,{111,121,0}},
/* 16291 */ {(12<<2)|2,{111,121,0}},
/* 16292 */ {(12<<2)|2,{111,121,0}},
/* 16293 */ {(12<<2)|2,{111,121,0}},
/* 16294 */ {(12<<2)|2,{111,121,0}},
/* 16295 */ {(12<<2)|2,{111,121,0}},
/* 16296 */ {(12<<2)|2,{111,121,0}},
/* 16297 */ {(12<<2)|2,{111,121,0}},
/* 16298 */ {(12<<2)|2,{111,121,0}},
/* 16299 */ {(12<<2)|2,{111,121,0}},
/* 16300 */ {(12<<2)|2,{111,121,0}},
/* 16301 */ {(12<<2)|2,{111,121,0}},
/* 16302 */ {(12<<2)|2,{111,121,0}},
/* 16303 */ {(12<<2)|2,{111,121,0}},
/* 16304 */ {(12<<2)|2,{111,122,0}},
/* 16305 */ {(12<<2)|2,{111,122,0}},
/* 16306 */ {(12<<2)|2,{111,122,0}},
/* 16307 */ {(12<<2)|2,{111,122,0}},
/* 16308 */ {(12<<2)|2,{111,122,0}},
/* 16309 */ {(12<<2)|2,{111,122,0}},
/* 16310 */ {(12<<2)|2,{111,122,0}},
/* 16311 */ {(12<<2)|2,{111,122,0}},
/* 16312 */ {(12<<2)|2,{111,122,0}},
/* 16313 */ {(12<<2)|2,{111,122,0}},
/* 16314 */ {(12<<2)|2,{111,122,0}},
/* 16315 */ {(12<<2)|2,{111,122,0}},
/* 16316 */ {(12<<2)|2,{111,122,0}},
/* 16317 */ {(12<<2)|2,{111,122,0}},
/* 16318 */ {(12<<2)|2,{111,122,0}},
/* 16319 */ {(12<<2)|2,{111,122,0}},
/* 16320 */ {(13<<2)|2,{111,38,0}},
/* 16321 */ {(13<<2)|2,{111,38,0}},
/* 16322 */ {(13<<2)|2,{111,38,0}},
/* 16323 */ {(13<<2)|2,{111,38,0}},
/* 16324 */ {(13<<2)|2,{111,38,0}},
/* 16325 */ {(13<<2)|2,{111,38,0}},
/* 16326 */ {(13<<2)|2,{111,38,0}},
/* 16327 */ {(13<<2)|2,{111,38,0}},
/* 16328 */ {(13<<2)|2,{111,42,0}},
/* 16329 */ {(13<<2)|2,{111,42,0}},
/* 16330 */ {(13<<2)|2,{111,42,0}},
/* 16331 */ {(13<<2)|2,{111,42,0}},
/* 16332 */ {(13<<2)|2,{111,42,0}},
/* 16333 */ {(13<<2)|2,{111,42,0}},
/* 16334 */ {(13<<2)|2,{111,42,0}},
/* 16335 */ {(13<<2)|2,{111,42,0}},
/* 16336 */ {(13<<2)|2,{111,44,0}},
/* 16337 */ {(13<<2)|2,{111,44,0}},
/* 16338 */ {(13<<2)|2,{111,44,0}},
/* 16339 */ {(13<<2)|2,{111,44,0}},
/* 16340 */ {(13<<2)|2,{111,44,0}},
/* 16341 */ {(13<<2)|2,{111,44,0}},
/* 16342 */ {(13<<2)|2,{111,44,0}},
/* 16343 */ {(13<<2)|2,{111,44,0}},
/* 16344 */ {(13<<2)|2,{111,59,0}},
/* 16345 */ {(13<<2)|2,{111,59,0}},
/* 16346 */ {(13<<2)|2,{111,59,0}},
/* 16347 */ {(13<<2)|2,{111,59,0}},
/* 16348 */ {(13<<2)|2,{111,59,0}},
/* 16349 */ {(13<<2)|2,{111,59,0}},
/* 16350 */ {(13<<2)|2,{111,59,0}},
/* 16351 */ {(13<<2)|2,{111,59,0}},
/* 16352 */ {(13<<2)|2,{111,88,0}},
/* 16353 */ {(13<<2)|2,{111,88,0}},
/* 16354 */ {(13<<2)|2,{111,88,0}},
/* 16355 */ {(13<<2)|2,{111,88,0}},
/* 16356 */ {(13<<2)|2,{111,88,0}},
/* 16357 */ {(13<<2)|2,{111,88,0}},
/* 16358 */ {(13<<2)|2,{111,88,0}},
/* 16359 */ {(13<<2)|2,{111,88,0}},
/* 16360 */ {(13<<2)|2,{111,90,0}},
/* 16361 */ {(13<<2)|2,{111,90,0}},
/* 16362 */ {(13<<2)|2,{111,90,0}},
/* 16363 */ {(13<<2)|2,{111,90,0}},
/* 16364 */ {(13<<2)|2,{111,90,0}},
/* 16365 */ {(13<<2)|2,{111,90,0}},
/* 16366 */ {(13<<2)|2,{111,90,0}},
/* 16367 */ {(13<<2)|2,{111,90,0}},
/* 16368 */ {(15<<2)|2,{111,33,0}},
/* 16369 */ {(15<<2)|2,{111,33,0}},
/* 16370 */ {(15<<2)|2,{111,34,0}},
/* 16371 */ {(15<<2)|2,{111,34,0}},
/* 16372 */ {(15<<2)|2,{111,40,0}},
/* 16373 */ {(15<<2)|2,{111,40,0}},
/* 16374 */ {(15<<2)|2,{111,41,0}},
/* 16375 */ {(15<<2)|2,{111,41,0}},
/* 16376 */ {(15<<2)|2,{111,63,0}},
/* 16377 */ {(15<<2)|2,{111,63,0}},
/* 16378 */ {(16<<2)|2,{111,39,0}},
/* 16379 */ {(16<<2)|2,{111,43,0}},
/* 16380 */ {(16<<2)|2,{111,124,0}},
/* 16381 */ {(5<<2)|1,{111,0,0}},
/* 16382 */ {(5<<2)|1,{111,0,0}},
/* 16383 */ {(5<<2)|1,{111,0,0}},
/* 16384 */ {(15<<2)|3,{115,48,48}},
/* 16385 */ {(15<<2)|3,{115,48,48}},
/* 16386 */ {(15<<2)|3,{115,48,49}},
/* 16387 */ {(15<<2)|3,{115,48,49}},
/* 16388 */ {(15<<2)|3,{115,48,50}},
/* 16389 */ {(15<<2)|3,{115,48,50}},
/* 16390 */ {(15<<2)|3,{115,48,97}},
/* 16391 */ {(15<<2)|3,{115,48,97}},
/* 16392 */ {(15<<2)|3,{115,48,99}},
/* 16393 */ {(15<<2)|3,{115,48,99}},
/* 16394 */ {(15<<2)|3,{115,48,101}},
/* 16395 */ {(15<<2)|3,{115,48,101}},
/* 16396 */ {(15<<2)|3,{115,48,105}},
/* 16397 */ {(15<<2)|3,{115,48,105}},
/* 16398 */ {(15<<2)|3,{115,48,111}},
/* 16399 */ {(15<<2)|3,{115,48,111}},
/* 16400 */ {(15<<2)|3,{115,48,115}},
/* 16401 */ {(15<<2)|3,{115,48,115}},
/* 16402 */ {(15<<2)|3,{115,48,116}},
/* 16403 */ {(15<<2)|3,{115,48,116}},
/* 16404 */ {(16<<2)|3,{115,48,32}},
/* 16405 */ {(16<<2)|3,{115,48,37}},
/* 16406 */ {(16<<2)|3,{115,48,45}},
/* 16407 */ {(16<<2)|3,{115,48,46}},
/* 16408 */ {(16<<2)|3,{115,48,47}},
/* 16409 */ {(16<<2)|3,{115,48,51}},
/* 16410 */ {(16<<2)|3,{115,48,52}},
/* 16411 */ {(16<<2)|3,{115,48,53}},
/* 16412 */ {(16<<2)|3,{115,48,54}},
/* 16413 */ {(16<<2)|3,{115,48,55}},
/* 16414 */ {(16<<2)|3,{115,48,56}},
/* 16415 */ {(16<<2)|3,{115,48,57}},
/* 16416 */ {(16<<2)|3,{115,48,61}},
/* 16417 */ {(16<<2)|3,{115,48,65}},
/* 16418 */ {(16<<2)|3,{115,48,95}},
/* 16419 */ {(16<<2)|3,{115,48,98}},
/* 16420 */ {(16<<2)|3,{115,48,100}},
/* 16421 */ {(16<<2)|3,{115,48,102}},
/* 16422 */ {(16<<2)|3,{115,48,103}},
/* 16423 */ {(16<<2)|3,{115,48,104}},
/* 16424 */ {(16<<2)|3,{115,48,108}},
/* 16425 */ {(16<<2)|3,{115,48,109}},
/* 16426 */ {(16<<2)|3,{115,48,110}},
/* 16427 */ {(16<<2)|3,{115,48,112}},
/* 16428 */ {(16<<2)|3,{115,48,114}},
/* 16429 */ {(16<<2)|3,{115,48,117}},
/* 16430 */ {(10<<2)|2,{115,48,0}},
/* 16431 */ {(10<<2)|2,{115,48,0}},
/* 16432 */ {(10<<2)|2,{115,48,0}},
/* 16433 */ {(10<<2)|2,{115,48,0}},
/* 16434 */ {(10<<2)|2,{115,48,0}},
/* 16435 */ {(10<<2)|2,{115,48,0}},
/* 16436 */ {(10<<2)|2,{115,48,0}},
/* 16437 */ {(10<<2)|2,{115,48,0}},
/* 16438 */ {(10<<2)|2,{115,48,0}},
/* 16439 */ {(10<<2)|2,{115,48,0}},
/* 16440 */ {(10<<2)|2,{115,48,0}},
/* 16441 */ {(10<<2)|2,{115,48,0}},
/* 16442 */ {(10<<2)|2,{115,48,0}},
/* 16443 */ {(10<<2)|2,{115,48,0}},
/* 16444 */ {(10<<2)|2,{115,48,0}},
/* 16445 */ {(10<<2)|2,{115,48,0}},
/* 16446 */ {(10<<2)|2,{115,48,0}},
/* 16447 */ {(10<<2)|2,{115,48,0}},
/* 16448 */ {(15<<2)|3,{115,49,48}},
/* 16449 */ {(15<<2)|3,{115,49,48}},
/* 16450 */ {(15<<2)|3,{115,49,49}},
/* 16451 */ {(15<<2)|3,{115,49,49}},
/* 16452 */ {(15<<2)|3,{115,49,50}},
/* 16453 */ {(15<<2)|3,{115,49,50}},
/* 16454 */ {(15<<2)|3,{115,49,97}},
/* 16455 */ {(15<<2)|3,{115,49,97}},
/* 16456 */ {(15<<2)|3,{115,49,99}},
/* 16457 */ {(15<<2)|3,{115,49,99}},
/* 16458 */ {(15<<2)|3,{115,49,101}},
/* 16459 */ {(15<<2)|3,{115,49,101}},
/* 16460 */ {(15<<2)|3,{115,49,105}},
/* 16461 */ {(15<<2)|3,{115,49,105}},
/* 16462 */ {(15<<2)|3,{115,49,111}},
/* 16463 */ {(15<<2)|3,{115,49,111}},
/* 16464 */ {(15<<2)|3,{115,49,115}},
/* 16465 */ {(15<<2)|3,{115,49,115}},
/* 16466 */ {(15<<2)|3,{115,49,116}},
/* 16467 */ {(15<<2)|3,{115,49,116}},
/* 16468 */ {(16<<2)|3,{115,49,32}},
/* 16469 */ {(16<<2)|3,{115,49,37}},
/* 16470 */ {(16<<2)|3,{115,49,45}},
/* 16471 */ {(16<<2)|3,{115,49,46}},
/* 16472 */ {(16<<2)|3,{115,49,47}},
/* 16473 */ {(16<<2)|3,{115,49,51}},
/* 16474 */ {(16<<2)|3,{115,49,52}},
/* 16475 */ {(16<<2)|3,{115,49,53}},
/* 16476 */ {(16<<2)|3,{115,49,54}},
/* 16477 */ {(16<<2)|3,{115,49,55}},
/* 16478 */ {(16<<2)|3,{115,49,56}},
/* 16479 */ {(16<<2)|3,{115,49,57}},
/* 16480 */ {(16<<2)|3,{115,49,61}},
/* 16481 */ {(16<<2)|3,{115,49,65}},
/* 16482 */ {(16<<2)|3,{115,49,95}},
/* 16483 */ {(16<<2)|3,{115,49,98}},
/* 16484 */ {(16<<2)|3,{115,49,100}},
/* 16485 */ {(16<<2)|3,{115,49,102}},
/* 16486 */ {(16<<2)|3,{115,49,103}},
/* 16487 */ {(16<<2)|3,{115,49,104}},
/* 16488 */ {(16<<2)|3,{115,49,108}},
/* 16489 */ {(16<<2)|3,{115,49,109}},
/* 16490 */ {(16<<2)|3,{115,49,110}},
/* 16491 */ {(16<<2)|3,{115,49,112}},
/* 16492 */ {(16<<2)|3,{115,49,114}},
/* 16493 */ {(16<<2)|3,{115,49,117}},
/* 16494 */ {(10<<2)|2,{115,49,0}},
/* 16495 */ {(10<<2)|2,{115,49,0}},
/* 16496 */ {(10<<2)|2,{115,49,0}},
/* 16497 */ {(10<<2)|2,{115,49,0}},
/* 16498 */ {(10<<2)|2,{115,49,0}},
/* 16499 */ {(10<<2)|2,{115,49,0}},
/* 16500 */ {(10<<2)|2,{115,49,0}},
/* 16501 */ {(10<<2)|2,{115,49,0}},
/* 16502 */ {(10<<2)|2,{115,49,0}},
/* 16503 */ {(10<<2)|2,{115,49,0}},
/* 16504 */ {(10<<2)|2,{115,49,0}},
/* 16505 */ {(10<<2)|2,{115,49,0}},
/* 16506 */ {(10<<2)|2,{115,49,0}},
/* 16507 */ {(10<<2)|2,{115,49,0}},
/* 16508 */ {(10<<2)|2,{115,49,0}},
/* 16509 */ {(10<<2)|2,{115,49,0}},
/* 16510 */ {(10<<2)|2,{115,49,0}},
/* 16511 */ {(10<<2)|2,{115,49,0}},
/* 16512 */ {(15<<2)|3,{115,50,48}},
/* 16513 */ {(15<<2)|3,{115,50,48}},
/* 16514 */ {(15<<2)|3,{115,50,49}},
/* 16515 */ {(15<<2)|3,{115,50,49}},
/* 16516 */ {(15<<2)|3,{115,50,50}},
/* 16517 */ {(15<<2)|3,{115,50,50}},
/* 16518 */ {(15<<2)|3,{115,50,97}},
/* 16519 */ {(15<<2)|3,{115,50,97}},
/* 16520 */ {(15<<2)|3,{115,50,99}},
/* 16521 */ {(15<<2)|3,{115,50,99}},
/* 16522 */ {(15<<2)|3,{115,50,101}},
/* 16523 */ {(15<<2)|3,{115,50,101}},
/* 16524 */ {(15<<2)|3,{115,50,105}},
/* 16525 */ {(15<<2)|3,{115,50,105}},
/* 16526 */ {(15<<2)|3,{115,50,111}},
/* 16527 */ {(15<<2)|3,{115,50,111}},
/* 16528 */ {(15<<2)|3,{115,50,115}},
/* 16529 */ {(15<<2)|3,{115,50,115}},
/* 16530 */ {(15<<2)|3,{115,50,116}},
/* 16531 */ {(15<<2)|3,{115,50,116}},
/* 16532 */ {(16<<2)|3,{115,50,32}},
/* 16533 */ {(16<<2)|3,{115,50,37}},
/* 16534 */ {(16<<2)|3,{115,50,45}},
/* 16535 */ {(16<<2)|3,{115,50,46}},
/* 16536 */ {(16<<2)|3,{115,50,47}},
/* 16537 */ {(16<<2)|3,{115,50,51}},
/* 16538 */ {(16<<2)|3,{115,50,52}},
/* 16539 */ {(16<<2)|3,{115,50,53}},
/* 16540 */ {(16<<2)|3,{115,50,54}},
/* 16541 */ {(16<<2)|3,{115,50,55}},
/* 16542 */ {(16<<2)|3,{115,50,56}},
/* 16543 */ {(16<<2)|3,{115,50,57}},
/* 16544 */ {(16<<2)|3,{115,50,61}},
/* 16545 */ {(16<<2)|3,{115,50,65}},
/* 16546 */ {(16<<2)|3,{115,50,95}},
/* 16547 */ {(16<<2)|3,{115,50,98}},
/* 16548 */ {(16<<2)|3,{115,50,100}},
/* 16549 */ {(16<<2)|3,{115,50,102}},
/* 16550 */ {(16<<2)|3,{115,50,103}},
/* 16551 */ {(16<<2)|3,{115,50,104}},
/* 16552 */ {(16<<2)|3,{115,50,108}},
/* 16553 */ {(16<<2)|3,{115,50,109}},
/* 16554 */ {(16<<2)|3,{115,50,110}},
/* 16555 */ {(16<<2)|3,{115,50,112}},
/* 16556 */ {(16<<2)|3,{115,50,114}},
/* 16557 */ {(16<<2)|3,{115,50,117}},
/* 16558 */ {(10<<2)|2,{115,50,0}},
/* 16559 */ {(10<<2)|2,{115,50,0}},
/* 16560 */ {(10<<2)|2,{115,50,0}},
/* 16561 */ {(10<<2)|2,{115,50,0}},
/* 16562 */ {(10<<2)|2,{115,50,0}},
/* 16563 */ {(10<<2)|2,{115,50,0}},
/* 16564 */ {(10<<2)|2,{115,50,0}},
/* 16565 */ {(10<<2)|2,{115,50,0}},
/* 16566 */ {(10<<2)|2,{115,50,0}},
/* 16567 */ {(10<<2)|2,{115,50,0}},
/* 16568 */ {(10<<2)|2,{115,50,0}},
/* 16569 */ {(10<<2)|2,{115,50,0}},
/* 16570 */ {(10<<2)|2,{115,50,0}},
/* 16571 */ {(10<<2)|2,{115,50,0}},
/* 16572 */ {(10<<2)|2,{115,50,0}},
/* 16573 */ {(10<<2)|2,{115,50,0}},
/* 16574 */ {(10<<2)|2,{115,50,0}},
/* 16575 */ {(10<<2)|2,{115,50,0}},
/* 16576 */ {(15<<2)|3,{115,97,48}},
/* 16577 */ {(15<<2)|3,{115,97,48}},
/* 16578 */ {(15<<2)|3,{115,97,49}},
/* 16579 */ {(15<<2)|3,{115,97,49}},
/* 16580 */ {(15<<2)|3,{115,97,50}},
/* 16581 */ {(15<<2)|3,{115,97,50}},
/* 16582 */ {(15<<2)|3,{115,97,97}},
/* 16583 */ {(15<<2)|3,{115,97,97}},
/* 16584 */ {(15<<2)|3,{115,97,99}},
/* 16585 */ {(15<<2)|3,{115,97,99}},
/* 16586 */ {(15<<2)|3,{115,97,101}},
/* 16587 */ {(15<<2)|3,{115,97,101}},
/* 16588 */ {(15<<2)|3,{115,97,105}},
/* 16589 */ {(15<<2)|3,{115,97,105}},
/* 16590 */ {(15<<2)|3,{115,97,111}},
/* 16591 */ {(15<<2)|3,{115,97,111}},
/* 16592 */ {(15<<2)|3,{115,97,115}},
/* 16593 */ {(15<<2)|3,{115,97,115}},
/* 16594 */ {(15<<2)|3,{115,97,116}},
/* 16595 */ {(15<<2)|3,{115,97,116}},
/* 16596 */ {(16<<2)|3,{115,97,32}},
/* 16597 */ {(16<<2)|3,{115,97,37}},
/* 16598 */ {(16<<2)|3,{115,97,45}},
/* 16599 */ {(16<<2)|3,{115,97,46}},
/* 16600 */ {(16<<2)|3,{115,97,47}},
/* 16601 */ {(16<<2)|3,{115,97,51}},
/* 16602 */ {(16<<2)|3,{115,97,52}},
/* 16603 */ {(16<<2)|3,{115,97,53}},
/* 16604 */ {(16<<2)|3,{115,97,54}},
/* 16605 */ {(16<<2)|3,{115,97,55}},
/* 16606 */ {(16<<2)|3,{115,97,56}},
/* 16607 */ {(16<<2)|3,{115,97,57}},
/* 16608 */ {(16<<2)|3,{115,97,61}},
/* 16609 */ {(16<<2)|3,{115,97,65}},
/* 16610 */ {(16<<2)|3,{115,97,95}},
/* 16611 */ {(16<<2)|3,{115,97,98}},
/* 16612 */ {(16<<2)|3,{115,97,100}},
/* 16613 */ {(16<<2)|3,{115,97,102}},
/* 16614 */ {(16<<2)|3,{115,97,103}},
/* 16615 */ {(16<<2)|3,{115,97,104}},
/* 16616 */ {(16<<2)|3,{115,97,108}},
/* 16617 */ {(16<<2)|3,{115,97,109}},
/* 16618 */ {(16<<2)|3,{115,97,110}},
/* 16619 */ {(16<<2)|3,{115,97,112}},
/* 16620 */ {(16<<2)|3,{115,97,114}},
/* 16621 */ {(16<<2)|3,{115,97,117}},
/* 16622 */ {(10<<2)|2,{115,97,0}},
/* 16623 */ {(10<<2)|2,{115,97,0}},
/* 16624 */ {(10<<2)|2,{115,97,0}},
/* 16625 */ {(10<<2)|2,{115,97,0}},
/* 16626 */ {(10<<2)|2,{115,97,0}},
/* 16627 */ {(10<<2)|2,{115,97,0}},
/* 16628 */ {(10<<2)|2,{115,97,0}},
/* 16629 */ {(10<<2)|2,{115,97,0}},
/* 16630 */ {(10<<2)|2,{115,97,0}},
/* 16631 */ {(10<<2)|2,{115,97,0}},
/* 16632 */ {(10<<2)|2,{115,97,0}},
/* 16633 */ {(10<<2)|2,{115,97,0}},
/* 16634 */ {(10<<2)|2,{115,97,0}},
/* 16635 */ {(10<<2)|2,{115,97,0}},
/* 16636 */ {(10<<2)|2,{115,97,0}},
/* 16637 */ {(10<<2)|2,{115,97,0}},
/* 16638 */ {(10<<2)|2,{115,97,0}},
/* 16639 */ {(10<<2)|2,{115,97,0}},
/* 16640 */ {(15<<2)|3,{115,99,48}},
/* 16641 */ {(15<<2)|3,{115,99,48}},
/* 16642 */ {(15<<2)|3,{115,99,49}},
/* 16643 */ {(15<<2)|3,{115,99,49}},
/* 16644 */ {(15<<2)|3,{115,99,50}},
/* 16645 */ {(15<<2)|3,{115,99,50}},
/* 16646 */ {(15<<2)|3,{115,99,97}},
/* 16647 */ {(15<<2)|3,{115,99,97}},
/* 16648 */ {(15<<2)|3,{115,99,99}},
/* 16649 */ {(15<<2)|3,{115,99,99}},
/* 16650 */ {(15<<2)|3,{115,99,101}},
/* 16651 */ {(15<<2)|3,{115,99,101}},
/* 16652 */ {(15<<2)|3,{115,99,105}},
/* 16653 */ {(15<<2)|3,{115,99,105}},
/* 16654 */ {(15<<2)|3,{115,99,111}},
/* 16655 */ {(15<<2)|3,{115,99,111}},
/* 16656 */ {(15<<2)|3,{115,99,115}},
/* 16657 */ {(15<<2)|3,{115,99,115}},
/* 16658 */ {(15<<2)|3,{115,99,116}},
/* 16659 */ {(15<<2)|3,{115,99,116}},
/* 16660 */ {(16<<2)|3,{115,99,32}},
/* 16661 */ {(16<<2)|3,{115,99,37}},
/* 16662 */ {(16<<2)|3,{115,99,45}},
/* 16663 */ {(16<<2)|3,{115,99,46}},
/* 16664 */ {(16<<2)|3,{115,99,47}},
/* 16665 */ {(16<<2)|3,{115,99,51}},
/* 16666 */ {(16<<2)|3,{115,99,52}},
/* 16667 */ {(16<<2)|3,{115,99,53}},
/* 16668 */ {(16<<2)|3,{115,99,54}},
/* 16669 */ {(16<<2)|3,{115,99,55}},
/* 16670 */ {(16<<2)|3,{115,99,56}},
/* 16671 */ {(16<<2)|3,{115,99,57}},
/* 16672 */ {(16<<2)|3,{115,99,61}},
/* 16673 */ {(16<<2)|3,{115,99,65}},
/* 16674 */ {(16<<2)|3,{115,99,95}},
/* 16675 */ {(16<<2)|3,{115,99,98}},
/* 16676 */ {(16<<2)|3,{115,99,100}},
/* 16677 */ {(16<<2)|3,{115,99,102}},
/* 16678 */ {(16<<2)|3,{115,99,103}},
/* 16679 */ {(16<<2)|3,{115,99,104}},
/* 16680 */ {(16<<2)|3,{115,99,108}},
/* 16681 */ {(16<<2)|3,{115,99,109}},
/* 16682 */ {(16<<2)|3,{115,99,110}},
/* 16683 */ {(16<<2)|3,{115,99,112}},
/* 16684 */ {(16<<2)|3,{115,99,114}},
/* 16685 */ {(16<<2)|3,{115,99,117}},
/* 16686 */ {(10<<2)|2,{115,99,0}},
/* 16687 */ {(10<<2)|2,{115,99,0}},
/* 16688 */ {(10<<2)|2,{115,99,0}},
/* 16689 */ {(10<<2)|2,{115,99,0}},
/* 16690 */ {(10<<2)|2,{115,99,0}},
/* 16691 */ {(10<<2)|2,{115,99,0}},
/* 16692 */ {(10<<2)|2,{115,99,0}},
/* 16693 */ {(10<<2)|2,{115,99,0}},
/* 16694 */ {(10<<2)|2,{115,99,0}},
/* 16695 */ {(10<<2)|2,{115,99,0}},
/* 16696 */ {(10<<2)|2,{115,99,0}},
/* 16697 */ {(10<<2)|2,{115,99,0}},
/* 16698 */ {(10<<2)|2,{115,99,0}},
/* 16699 */ {(10<<2)|2,{115,99,0}},
/* 16700 */ {(10<<2)|2,{115,99,0}},
/* 16701 */ {(10<<2)|2,{115,99,0}},
/* 16702 */ {(10<<2)|2,{115,99,0}},
/* 16703 */ {(10<<2)|2,{115,99,0}},
/* 16704 */ {(15<<2)|3,{115,101,48}},
/* 16705 */ {(15<<2)|3,{115,101,48}},
/* 16706 */ {(15<<2)|3,{115,101,49}},
/* 16707 */ {(15<<2)|3,{115,101,49}},
/* 16708 */ {(15<<2)|3,{115,101,50}},
/* 16709 */ {(15<<2)|3,{115,101,50}},
/* 16710 */ {(15<<2)|3,{115,101,97}},
/* 16711 */ {(15<<2)|3,{115,101,97}},
/* 16712 */ {(15<<2)|3,{115,101,99}},
/* 16713 */ {(15<<2)|3,{115,101,99}},
/* 16714 */ {(15<<2)|3,{115,101,101}},
/* 16715 */ {(15<<2)|3,{115,101,101}},
/* 16716 */ {(15<<2)|3,{115,101,105}},
/* 16717 */ {(15<<2)|3,{115,101,105}},
/* 16718 */ {(15<<2)|3,{115,101,111}},
/* 16719 */ {(15<<2)|3,{115,101,111}},
/* 16720 */ {(15<<2)|3,{115,101,115}},
/* 16721 */ {(15<<2)|3,{115,101,115}},
/* 16722 */ {(15<<2)|3,{115,101,116}},
/* 16723 */ {(15<<2)|3,{115,101,116}},
/* 16724 */ {(16<<2)|3,{115,101,32}},
/* 16725 */ {(16<<2)|3,{115,101,37}},
/* 16726 */ {(16<<2)|3,{115,101,45}},
/* 16727 */ {(16<<2)|3,{115,101,46}},
/* 16728 */ {(16<<2)|3,{115,101,47}},
/* 16729 */ {(16<<2)|3,{115,101,51}},
/* 16730 */ {(16<<2)|3,{115,101,52}},
/* 16731 */ {(16<<2)|3,{115,101,53}},
/* 16732 */ {(16<<2)|3,{115,101,54}},
/* 16733 */ {(16<<2)|3,{115,101,55}},
/* 16734 */ {(16<<2)|3,{115,101,56}},
/* 16735 */ {(16<<2)|3,{115,101,57}},
/* 16736 */ {(16<<2)|3,{115,101,61}},
/* 16737 */ {(16<<2)|3,{115,101,65}},
/* 16738 */ {(16<<2)|3,{115,101,95}},
/* 16739 */ {(16<<2)|3,{115,101,98}},
/* 16740 */ {(16<<2)|3,{115,101,100}},
/* 16741 */ {(16<<2)|3,{115,101,102}},
/* 16742 */ {(16<<2)|3,{115,101,103}},
/* 16743 */ {(16<<2)|3,{115,101,104}},
/* 16744 */ {(16<<2)|3,{115,101,108}},
/* 16745 */ {(16<<2)|3,{115,101,109}},
/* 16746 */ {(16<<2)|3,{115,101,110}},
/* 16747 */ {(16<<2)|3,{115,101,112}},
/* 16748 */ {(16<<2)|3,{115,101,114}},
/* 16749 */ {(16<<2)|3,{115,101,117}},
/* 16750 */ {(10<<2)|2,{115,101,0}},
/* 16751 */ {(10<<2)|2,{115,101,0}},
/* 16752 */ {(10<<2)|2,{115,101,0}},
/* 16753 */ {(10<<2)|2,{115,101,0}},
/* 16754 */ {(10<<2)|2,{115,101,0}},
/* 16755 */ {(10<<2)|2,{115,101,0}},
/* 16756 */ {(10<<2)|2,{115,101,0}},
/* 16757 */ {(10<<2)|2,{115,101,0}},
/* 16758 */ {(10<<2)|2,{115,101,0}},
/* 16759 */ {(10<<2)|2,{115,101,0}},
/* 16760 */ {(10<<2)|2,{115,101,0}},
/* 16761 */ {(10<<2)|2,{115,101,0}},
/* 16762 */ {(10<<2)|2,{115,101,0}},
/* 16763 */ {(10<<2)|2,{115,101,0}},
/* 16764 */ {(10<<2)|2,{115,101,0}},
/* 16765 */ {(10<<2)|2,{115,101,0}},
/* 16766 */ {(10<<2)|2,{115,101,0}},
/* 16767 */ {(10<<2)|2,{115,101,0}},
/* 16768 */ {(15<<2)|3,{115,105,48}},
/* 16769 */ {(15<<2)|3,{115,105,48}},
/* 16770 */ {(15<<2)|3,{115,105,49}},
/* 16771 */ {(15<<2)|3,{115,105,49}},
/* 16772 */ {(15<<2)|3,{115,105,50}},
/* 16773 */ {(15<<2)|3,{115,105,50}},
/* 16774 */ {(15<<2)|3,{115,105,97}},
/* 16775 */ {(15<<2)|3,{115,105,97}},
/* 16776 */ {(15<<2)|3,{115,105,99}},
/* 16777 */ {(15<<2)|3,{115,105,99}},
/* 16778 */ {(15<<2)|3,{115,105,101}},
/* 16779 */ {(15<<2)|3,{115,105,101}},
/* 16780 */ {(15<<2)|3,{115,105,105}},
/* 16781 */ {(15<<2)|3,{115,105,105}},
/* 16782 */ {(15<<2)|3,{115,105,111}},
/* 16783 */ {(15<<2)|3,{115,105,111}},
/* 16784 */ {(15<<2)|3,{115,105,115}},
/* 16785 */ {(15<<2)|3,{115,105,115}},
/* 16786 */ {(15<<2)|3,{115,105,116}},
/* 16787 */ {(15<<2)|3,{115,105,116}},
/* 16788 */ {(16<<2)|3,{115,105,32}},
/* 16789 */ {(16<<2)|3,{115,105,37}},
/* 16790 */ {(16<<2)|3,{115,105,45}},
/* 16791 */ {(16<<2)|3,{115,105,46}},
/* 16792 */ {(16<<2)|3,{115,105,47}},
/* 16793 */ {(16<<2)|3,{115,105,51}},
/* 16794 */ {(16<<2)|3,{115,105,52}},
/* 16795 */ {(16<<2)|3,{115,105,53}},
/* 16796 */ {(16<<2)|3,{115,105,54}},
/* 16797 */ {(16<<2)|3,{115,105,55}},
/* 16798 */ {(16<<2)|3,{115,105,56}},
/* 16799 */ {(16<<2)|3,{115,105,57}},
/* 16800 */ {(16<<2)|3,{115,105,61}},
/* 16801 */ {(16<<2)|3,{115,105,65}},
/* 16802 */ {(16<<2)|3,{115,105,95}},
/* 16803 */ {(16<<2)|3,{115,105,98}},
/* 16804 */ {(16<<2)|3,{115,105,100}},
/* 16805 */ {(16<<2)|3,{115,105,102}},
/* 16806 */ {(16<<2)|3,{115,105,103}},
/* 16807 */ {(16<<2)|3,{115,105,104}},
/* 16808 */ {(16<<2)|3,{115,105,108}},
/* 16809 */ {(16<<2)|3,{115,105,109}},
/* 16810 */ {(16<<2)|3,{115,105,110}},
/* 16811 */ {(16<<2)|3,{115,105,112}},
/* 16812 */ {(16<<2)|3,{115,105,114}},
/* 16813 */ {(16<<2)|3,{115,105,117}},
/* 16814 */ {(10<<2)|2,{115,105,0}},
/* 16815 */ {(10<<2)|2,{115,105,0}},
/* 16816 */ {(10<<2)|2,{115,105,0}},
/* 16817 */ {(10<<2)|2,{115,105,0}},
/* 16818 */ {(10<<2)|2,{115,105,0}},
/* 16819 */ {(10<<2)|2,{115,105,0}},
/* 16820 */ {(10<<2)|2,{115,105,0}},
/* 16821 */ {(10<<2)|2,{115,105,0}},
/* 16822 */ {(10<<2)|2,{115,105,0}},
/* 16823 */ {(10<<2)|2,{115,105,0}},
/* 16824 */ {(10<<2)|2,{115,105,0}},
/* 16825 */ {(10<<2)|2,{115,105,0}},
/* 16826 */ {(10<<2)|2,{115,105,0}},
/* 16827 */ {(10<<2)|2,{115,105,0}},
/* 16828 */ {(10<<2)|2,{115,105,0}},
/* 16829 */ {(10<<2)|2,{115,105,0}},
/* 16830 */ {(10<<2)|2,{115,105,0}},
/* 16831 */ {(10<<2)|2,{115,105,0}},
/* 16832 */ {(15<<2)|3,{115,111,48}},
/* 16833 */ {(15<<2)|3,{115,111,48}},
/* 16834 */ {(15<<2)|3,{115,111,49}},
/* 16835 */ {(15<<2)|3,{115,111,49}},
/* 16836 */ {(15<<2)|3,{115,111,50}},
/* 16837 */ {(15<<2)|3,{115,111,50}},
/* 16838 */ {(15<<2)|3,{115,111,97}},
/* 16839 */ {(15<<2)|3,{115,111,97}},
/* 16840 */ {(15<<2)|3,{115,111,99}},
/* 16841 */ {(15<<2)|3,{115,111,99}},
/* 16842 */ {(15<<2)|3,{115,111,101}},
/* 16843 */ {(15<<2)|3,{115,111,101}},
/* 16844 */ {(15<<2)|3,{115,111,105}},
/* 16845 */ {(15<<2)|3,{115,111,105}},
/* 16846 */ {(15<<2)|3,{115,111,111}},
/* 16847 */ {(15<<2)|3,{115,111,111}},
/* 16848 */ {(15<<2)|3,{115,111,115}},
/* 16849 */ {(15<<2)|3,{115,111,115}},
/* 16850 */ {(15<<2)|3,{115,111,116}},
/* 16851 */ {(15<<2)|3,{115,111,116}},
/* 16852 */ {(16<<2)|3,{115,111,32}},
/* 16853 */ {(16<<2)|3,{115,111,37}},
/* 16854 */ {(16<<2)|3,{115,111,45}},
/* 16855 */ {(16<<2)|3,{115,111,46}},
/* 16856 */ {(16<<2)|3,{115,111,47}},
/* 16857 */ {(16<<2)|3,{115,111,51}},
/* 16858 */ {(16<<2)|3,{115,111,52}},
/* 16859 */ {(16<<2)|3,{115,111,53}},
/* 16860 */ {(16<<2)|3,{115,111,54}},
/* 16861 */ {(16<<2)|3,{115,111,55}},
/* 16862 */ {(16<<2)|3,{115,111,56}},
/* 16863 */ {(16<<2)|3,{115,111,57}},
/* 16864 */ {(16<<2)|3,{115,111,61}},
/* 16865 */ {(16<<2)|3,{115,111,65}},
/* 16866 */ {(16<<2)|3,{115,111,95}},
/* 16867 */ {(16<<2)|3,{115,111,98}},
/* 16868 */ {(16<<2)|3,{115,111,100}},
/* 16869 */ {(16<<2)|3,{115,111,102}},
/* 16870 */ {(16<<2)|3,{115,111,103}},
/* 16871 */ {(16<<2)|3,{115,111,104}},
/* 16872 */ {(16<<2)|3,{115,111,108}},
/* 16873 */ {(16<<2)|3,{115,111,109}},
/* 16874 */ {(16<<2)|3,{115,111,110}},
/* 16875 */ {(16<<2)|3,{115,111,112}},
/* 16876 */ {(16<<2)|3,{115,111,114}},
/* 16877 */ {(16<<2)|3,{115,111,117}},
/* 16878 */ {(10<<2)|2,{115,111,0}},
/* 16879 */ {(10<<2)|2,{115,111,0}},
/* 16880 */ {(10<<2)|2,{115,111,0}},
/* 16881 */ {(10<<2)|2,{115,111,0}},
/* 16882 */ {(10<<2)|2,{115,111,0}},
/* 16883 */ {(10<<2)|2,{115,111,0}},
/* 16884 */ {(10<<2)|2,{115,111,0}},
/* 16885 */ {(10<<2)|2,{115,111,0}},
/* 16886 */ {(10<<2)|2,{115,111,0}},
/* 16887 */ {(10<<2)|2,{115,111,0}},
/* 16888 */ {(10<<2)|2,{115,111,0}},
/* 16889 */ {(10<<2)|2,{115,111,0}},
/* 16890 */ {(10<<2)|2,{115,111,0}},
/* 16891 */ {(10<<2)|2,{115,111,0}},
/* 16892 */ {(10<<2)|2,{115,111,0}},
/* 16893 */ {(10<<2)|2,{115,111,0}},
/* 16894 */ {(10<<2)|2,{115,111,0}},
/* 16895 */ {(10<<2)|2,{115,111,0}},
/* 16896 */ {(15<<2)|3,{115,115,48}},
/* 16897 */ {(15<<2)|3,{115,115,48}},
/* 16898 */ {(15<<2)|3,{115,115,49}},
/* 16899 */ {(15<<2)|3,{115,115,49}},
/* 16900 */ {(15<<2)|3,{115,115,50}},
/* 16901 */ {(15<<2)|3,{115,115,50}},
/* 16902 */ {(15<<2)|3,{115,115,97}},
/* 16903 */ {(15<<2)|3,{115,115,97}},
/* 16904 */ {(15<<2)|3,{115,115,99}},
/* 16905 */ {(15<<2)|3,{115,115,99}},
/* 16906 */ {(15<<2)|3,{115,115,101}},
/* 16907 */ {(15<<2)|3,{115,115,101}},
/* 16908 */ {(15<<2)|3,{115,115,105}},
/* 16909 */ {(15<<2)|3,{115,115,105}},
/* 16910 */ {(15<<2)|3,{115,115,111}},
/* 16911 */ {(15<<2)|3,{115,115,111}},
/* 16912 */ {(15<<2)|3,{115,115,115}},
/* 16913 */ {(15<<2)|3,{115,115,115}},
/* 16914 */ {(15<<2)|3,{115,115,116}},
/* 16915 */ {(15<<2)|3,{115,115,116}},
/* 16916 */ {(16<<2)|3,{115,115,32}},
/* 16917 */ {(16<<2)|3,{115,115,37}},
/* 16918 */ {(16<<2)|3,{115,115,45}},
/* 16919 */ {(16<<2)|3,{115,115,46}},
/* 16920 */ {(16<<2)|3,{115,115,47}},
/* 16921 */ {(16<<2)|3,{115,115,51}},
/* 16922 */ {(16<<2)|3,{115,115,52}},
/* 16923 */ {(16<<2)|3,{115,115,53}},
/* 16924 */ {(16<<2)|3,{115,115,54}},
/* 16925 */ {(16<<2)|3,{115,115,55}},
/* 16926 */ {(16<<2)|3,{115,115,56}},
/* 16927 */ {(16<<2)|3,{115,115,57}},
/* 16928 */ {(16<<2)|3,{115,115,61}},
/* 16929 */ {(16<<2)|3,{115,115,65}},
/* 16930 */ {(16<<2)|3,{115,115,95}},
/* 16931 */ {(16<<2)|3,{115,115,98}},
/* 16932 */ {(16<<2)|3,{115,115,100}},
/* 16933 */ {(16<<2)|3,{115,115,102}},
/* 16934 */ {(16<<2)|3,{115,115,103}},
/* 16935 */ {(16<<2)|3,{115,115,104}},
/* 16936 */ {(16<<2)|3,{115,115,108}},
/* 16937 */ {(16<<2)|3,{115,115,109}},
/* 16938 */ {(16<<2)|3,{115,115,110}},
/* 16939 */ {(16<<2)|3,{115,115,112}},
/* 16940 */ {(16<<2)|3,{115,115,114}},
/* 16941 */ {(16<<2)|3,{115,115,117}},
/* 16942 */ {(10<<2)|2,{115,115,0}},
/* 16943 */ {(10<<2)|2,{115,115,0}},
/* 16944 */ {(10<<2)|2,{115,115,0}},
/* 16945 */ {(10<<2)|2,{115,115,0}},
/* 16946 */ {(10<<2)|2,{115,115,0}},
/* 16947 */ {(10<<2)|2,{115,115,0}},
/* 16948 */ {(10<<2)|2,{115,115,0}},
/* 16949 */ {(10<<2)|2,{115,115,0}},
/* 16950 */ {(10<<2)|2,{115,115,0}},
/* 16951 */ {(10<<2)|2,{115,115,0}},
/* 16952 */ {(10<<2)|2,{115,115,0}},
/* 16953 */ {(10<<2)|2,{115,115,0}},
/* 16954 */ {(10<<2)|2,{115,115,0}},
/* 16955 */ {(10<<2)|2,{115,115,0}},
/* 16956 */ {(10<<2)|2,{115,115,0}},
/* 16957 */ {(10<<2)|2,{115,115,0}},
/* 16958 */ {(10<<2)|2,{115,115,0}},
/* 16959 */ {(10<<2)|2,{115,115,0}},
/* 16960 */ {(15<<2)|3,{115,116,48}},
/* 16961 */ {(15<<2)|3,{115,116,48}},
/* 16962 */ {(15<<2)|3,{115,116,49}},
/* 16963 */ {(15<<2)|3,{115,116,49}},
/* 16964 */ {(15<<2)|3,{115,116,50}},
/* 16965 */ {(15<<2)|3,{115,116,50}},
/* 16966 */ {(15<<2)|3,{115,116,97}},
/* 16967 */ {(15<<2)|3,{115,116,97}},
/* 16968 */ {(15<<2)|3,{115,116,99}},
/* 16969 */ {(15<<2)|3,{115,116,99}},
/* 16970 */ {(15<<2)|3,{115,116,101}},
/* 16971 */ {(15<<2)|3,{115,116,101}},
/* 16972 */ {(15<<2)|3,{115,116,105}},
/* 16973 */ {(15<<2)|3,{115,116,105}},
/* 16974 */ {(15<<2)|3,{115,116,111}},
/* 16975 */ {(15<<2)|3,{115,116,111}},
/* 16976 */ {(15<<2)|3,{115,116,115}},
/* 16977 */ {(15<<2)|3,{115,116,115}},
/* 16978 */ {(15<<2)|3,{115,116,116}},
/* 16979 */ {(15<<2)|3,{115,116,116}},
/* 16980 */ {(16<<2)|3,{115,116,32}},
/* 16981 */ {(16<<2)|3,{115,116,37}},
/* 16982 */ {(16<<2)|3,{115,116,45}},
/* 16983 */ {(16<<2)|3,{115,116,46}},
/* 16984 */ {(16<<2)|3,{115,116,47}},
/* 16985 */ {(16<<2)|3,{115,116,51}},
/* 16986 */ {(16<<2)|3,{115,116,52}},
/* 16987 */ {(16<<2)|3,{115,116,53}},
/* 16988 */ {(16<<2)|3,{115,116,54}},
/* 16989 */ {(16<<2)|3,{115,116,55}},
/* 16990 */ {(16<<2)|3,{115,116,56}},
/* 16991 */ {(16<<2)|3,{115,116,57}},
/* 16992 */ {(16<<2)|3,{115,116,61}},
/* 16993 */ {(16<<2)|3,{115,116,65}},
/* 16994 */ {(16<<2)|3,{115,116,95}},
/* 16995 */ {(16<<2)|3,{115,116,98}},
/* 16996 */ {(16<<2)|3,{115,116,100}},
/* 16997 */ {(16<<2)|3,{115,116,102}},
/* 16998 */ {(16<<2)|3,{115,116,103}},
/* 16999 */ {(16<<2)|3,{115,116,104}},
/* 17000 */ {(16<<2)|3,{115,116,108}},
/* 17001 */ {(16<<2)|3,{115,116,109}},
/* 17002 */ {(16<<2)|3,{115,116,110}},
/* 17003 */ {(16<<2)|3,{115,116,112}},
/* 17004 */ {(16<<2)|3,{115,116,114}},
/* 17005 */ {(16<<2)|3,{115,116,117}},
/* 17006 */ {(10<<2)|2,{115,116,0}},
/* 17007 */ {(10<<2)|2,{115,116,0}},
/* 17008 */ {(10<<2)|2,{115,116,0}},
/* 17009 */ {(10<<2)|2,{115,116,0}},
/* 17010 */ {(10<<2)|2,{115,116,0}},
/* 17011 */ {(10<<2)|2,{115,116,0}},
/* 17012 */ {(10<<2)|2,{115,116,0}},
/* 17013 */ {(10<<2)|2,{115,116,0}},
/* 17014 */ {(10<<2)|2,{115,116,0}},
/* 17015 */ {(10<<2)|2,{115,116,0}},
/* 17016 */ {(10<<2)|2,{115,116,0}},
/* 17017 */ {(10<<2)|2,{115,116,0}},
/* 17018 */ {(10<<2)|2,{115,116,0}},
/* 17019 */ {(10<<2)|2,{115,116,0}},
/* 17020 */ {(10<<2)|2,{115,116,0}},
/* 17021 */ {(10<<2)|2,{115,116,0}},
/* 17022 */ {(10<<2)|2,{115,116,0}},
/* 17023 */ {(10<<2)|2,{115,116,0}},
/* 17024 */ {(16<<2)|3,{115,32,48}},
/* 17025 */ {(16<<2)|3,{115,32,49}},
/* 17026 */ {(16<<2)|3,{115,32,50}},
/* 17027 */ {(16<<2)|3,{115,32,97}},
/* 17028 */ {(16<<2)|3,{115,32,99}},
/* 17029 */ {(16<<2)|3,{115,32,101}},
/* 17030 */ {(16<<2)|3,{115,32,105}},
/* 17031 */ {(16<<2)|3,{115,32,111}},
/* 17032 */ {(16<<2)|3,{115,32,115}},
/* 17033 */ {(16<<2)|3,{115,32,116}},
/* 17034 */ {(11<<2)|2,{115,32,0}},
/* 17035 */ {(11<<2)|2,{115,32,0}},
/* 17036 */ {(11<<2)|2,{115,32,0}},
/* 17037 */ {(11<<2)|2,{115,32,0}},
/* 17038 */ {(11<<2)|2,{115,32,0}},
/* 17039 */ {(11<<2)|2,{115,32,0}},
/* 17040 */ {(11<<2)|2,{115,32,0}},
/* 17041 */ {(11<<2)|2,{115,32,0}},
/* 17042 */ {(11<<2)|2,{115,32,0}},
/* 17043 */ {(11<<2)|2,{115,32,0}},
/* 17044 */ {(11<<2)|2,{115,32,0}},
/* 17045 */ {(11<<2)|2,{115,32,0}},
/* 17046 */ {(11<<2)|2,{115,32,0}},
/* 17047 */ {(11<<2)|2,{115,32,0}},
/* 17048 */ {(11<<2)|2,{115,32,0}},
/* 17049 */ {(11<<2)|2,{115,32,0}},
/* 17050 */ {(11<<2)|2,{115,32,0}},
/* 17051 */ {(11<<2)|2,{115,32,0}},
/* 17052 */ {(11<<2)|2,{115,32,0}},
/* 17053 */ {(11<<2)|2,{115,32,0}},
/* 17054 */ {(11<<2)|2,{115,32,0}},
/* 17055 */ {(11<<2)|2,{115,32,0}},
/* 17056 */ {(16<<2)|3,{115,37,48}},
/* 17057 */ {(16<<2)|3,{115,37,49}},
/* 17058 */ {(16<<2)|3,{115,37,50}},
/* 17059 */ {(16<<2)|3,{115,37,97}},
/* 17060 */ {(16<<2)|3,{115,37,99}},
/* 17061 */ {(16<<2)|3,{115,37,101}},
/* 17062 */ {(16<<2)|3,{115,37,105}},
/* 17063 */ {(16<<2)|3,{115,37,111}},
/* 17064 */ {(16<<2)|3,{115,37,115}},
/* 17065 */ {(16<<2)|3,{115,37,116}},
/* 17066 */ {(11<<2)|2,{115,37,0}},
/* 17067 */ {(11<<2)|2,{115,37,0}},
/* 17068 */ {(11<<2)|2,{115,37,0}},
/* 17069 */ {(11<<2)|2,{115,37,0}},
/* 17070 */ {(11<<2)|2,{115,37,0}},
/* 17071 */ {(11<<2)|2,{115,37,0}},
/* 17072 */ {(11<<2)|2,{115,37,0}},
/* 17073 */ {(11<<2)|2,{115,37,0}},
/* 17074 */ {(11<<2)|2,{115,37,0}},
/* 17075 */ {(11<<2)|2,{115,37,0}},
/* 17076 */ {(11<<2)|2,{115,37,0}},
/* 17077 */ {(11<<2)|2,{115,37,0}},
/* 17078 */ {(11<<2)|2,{115,37,0}},
/* 17079 */ {(11<<2)|2,{115,37,0}},
/* 17080 */ {(11<<2)|2,{115,37,0}},
/* 17081 */ {(11<<2)|2,{115,37,0}},
/* 17082 */ {(11<<2)|2,{115,37,0}},
/* 17083 */ {(11<<2)|2,{115,37,0}},
/* 17084 */ {(11<<2)|2,{115,37,0}},
/* 17085 */ {(11<<2)|2,{115,37,0}},
/* 17086 */ {(11<<2)|2,{115,37,0}},
/* 17087 */ {(11<<2)|2,{115,37,0}},
/* 17088 */ {(16<<2)|3,{115,45,48}},
/* 17089 */ {(16<<2)|3,{115,45,49}},
/* 17090 */ {(16<<2)|3,{115,45,50}},
/* 17091 */ {(16<<2)|3,{115,45,97}},
/* 17092 */ {(16<<2)|3,{115,45,99}},
/* 17093 */ {(16<<2)|3,{115,45,101}},
/* 17094 */ {(16<<2)|3,{115,45,105}},
/* 17095 */ {(16<<2)|3,{115,45,111}},
/* 17096 */ {(16<<2)|3,{115,45,115}},
/* 17097 */ {(16<<2)|3,{115,45,116}},
/* 17098 */ {(11<<2)|2,{115,45,0}},
/* 17099 */ {(11<<2)|2,{115,45,0}},
/* 17100 */ {(11<<2)|2,{115,45,0}},
/* 17101 */ {(11<<2)|2,{115,45,0}},
/* 17102 */ {(11<<2)|2,{115,45,0}},
/* 17103 */ {(11<<2)|2,{115,45,0}},
/* 17104 */ {(11<<2)|2,{115,45,0}},
/* 17105 */ {(11<<2)|2,{115,45,0}},
/* 17106 */ {(11<<2)|2,{115,45,0}},
/* 17107 */ {(11<<2)|2,{115,45,0}},
/* 17108 */ {(11<<2)|2,{115,45,0}},
/* 17109 */ {(11<<2)|2,{115,45,0}},
/* 17110 */ {(11<<2)|2,{115,45,0}},
/* 17111 */ {(11<<2)|2,{115,45,0}},
/* 17112 */ {(11<<2)|2,{115,45,0}},
/* 17113 */ {(11<<2)|2,{115,45,0}},
/* 17114 */ {(11<<2)|2,{115,45,0}},
/* 17115 */ {(11<<2)|2,{115,45,0}},
/* 17116 */ {(11<<2)|2,{115,45,0}},
/* 17117 */ {(11<<2)|2,{115,45,0}},
/* 17118 */ {(11<<2)|2,{115,45,0}},
/* 17119 */ {(11<<2)|2,{115,45,0}},
/* 17120 */ {(16<<2)|3,{115,46,48}},
/* 17121 */ {(16<<2)|3,{115,46,49}},
/* 17122 */ {(16<<2)|3,{115,46,50}},
/* 17123 */ {(16<<2)|3,{115,46,97}},
/* 17124 */ {(16<<2)|3,{115,46,99}},
/* 17125 */ {(16<<2)|3,{115,46,101}},
/* 17126 */ {(16<<2)|3,{115,46,105}},
/* 17127 */ {(16<<2)|3,{115,46,111}},
/* 17128 */ {(16<<2)|3,{115,46,115}},
/* 17129 */ {(16<<2)|3,{115,46,116}},
/* 17130 */ {(11<<2)|2,{115,46,0}},
/* 17131 */ {(11<<2)|2,{115,46,0}},
/* 17132 */ {(11<<2)|2,{115,46,0}},
/* 17133 */ {(11<<2)|2,{115,46,0}},
/* 17134 */ {(11<<2)|2,{115,46,0}},
/* 17135 */ {(11<<2)|2,{115,46,0}},
/* 17136 */ {(11<<2)|2,{115,46,0}},
/* 17137 */ {(11<<2)|2,{115,46,0}},
/* 17138 */ {(11<<2)|2,{115,46,0}},
/* 17139 */ {(11<<2)|2,{115,46,0}},
/* 17140 */ {(11<<2)|2,{115,46,0}},
/* 17141 */ {(11<<2)|2,{115,46,0}},
/* 17142 */ {(11<<2)|2,{115,46,0}},
/* 17143 */ {(11<<2)|2,{115,46,0}},
/* 17144 */ {(11<<2)|2,{115,46,0}},
/* 17145 */ {(11<<2)|2,{115,46,0}},
/* 17146 */ {(11<<2)|2,{115,46,0}},
/* 17147 */ {(11<<2)|2,{115,46,0}},
/* 17148 */ {(11<<2)|2,{115,46,0}},
/* 17149 */ {(11<<2)|2,{115,46,0}},
/* 17150 */ {(11<<2)|2,{115,46,0}},
/* 17151 */ {(11<<2)|2,{115,46,0}},
/* 17152 */ {(16<<2)|3,{115,47,48}},
/* 17153 */ {(16<<2)|3,{115,47,49}},
/* 17154 */ {(16<<2)|3,{115,47,50}},
/* 17155 */ {(16<<2)|3,{115,47,97}},
/* 17156 */ {(16<<2)|3,{115,47,99}},
/* 17157 */ {(16<<2)|3,{115,47,101}},
/* 17158 */ {(16<<2)|3,{115,47,105}},
/* 17159 */ {(16<<2)|3,{115,47,111}},
/* 17160 */ {(16<<2)|3,{115,47,115}},
/* 17161 */ {(16<<2)|3,{115,47,116}},
/* 17162 */ {(11<<2)|2,{115,47,0}},
/* 17163 */ {(11<<2)|2,{115,47,0}},
/* 17164 */ {(11<<2)|2,{115,47,0}},
/* 17165 */ {(11<<2)|2,{115,47,0}},
/* 17166 */ {(11<<2)|2,{115,47,0}},
/* 17167 */ {(11<<2)|2,{115,47,0}},
/* 17168 */ {(11<<2)|2,{115,47,0}},
/* 17169 */ {(11<<2)|2,{115,47,0}},
/* 17170 */ {(11<<2)|2,{115,47,0}},
/* 17171 */ {(11<<2)|2,{115,47,0}},
/* 17172 */ {(11<<2)|2,{115,47,0}},
/* 17173 */ {(11<<2)|2,{115,47,0}},
/* 17174 */ {(11<<2)|2,{115,47,0}},
/* 17175 */ {(11<<2)|2,{115,47,0}},
/* 17176 */ {(11<<2)|2,{115,47,0}},
/* 17177 */ {(11<<2)|2,{115,47,0}},
/* 17178 */ {(11<<2)|2,{115,47,0}},
/* 17179 */ {(11<<2)|2,{115,47,0}},
/* 17180 */ {(11<<2)|2,{115,47,0}},
/* 17181 */ {(11<<2)|2,{115,47,0}},
/* 17182 */ {(11<<2)|2,{115,47,0}},
/* 17183 */ {(11<<2)|2,{115,47,0}},
/* 17184 */ {(16<<2)|3,{115,51,48}},
/* 17185 */ {(16<<2)|3,{115,51,49}},
/* 17186 */ {(16<<2)|3,{115,51,50}},
/* 17187 */ {(16<<2)|3,{115,51,97}},
/* 17188 */ {(16<<2)|3,{115,51,99}},
/* 17189 */ {(16<<2)|3,{115,51,101}},
/* 17190 */ {(16<<2)|3,{115,51,105}},
/* 17191 */ {(16<<2)|3,{115,51,111}},
/* 17192 */ {(16<<2)|3,{115,51,115}},
/* 17193 */ {(16<<2)|3,{115,51,116}},
/* 17194 */ {(11<<2)|2,{115,51,0}},
/* 17195 */ {(11<<2)|2,{115,51,0}},
/* 17196 */ {(11<<2)|2,{115,51,0}},
/* 17197 */ {(11<<2)|2,{115,51,0}},
/* 17198 */ {(11<<2)|2,{115,51,0}},
/* 17199 */ {(11<<2)|2,{115,51,0}},
/* 17200 */ {(11<<2)|2,{115,51,0}},
/* 17201 */ {(11<<2)|2,{115,51,0}},
/* 17202 */ {(11<<2)|2,{115,51,0}},
/* 17203 */ {(11<<2)|2,{115,51,0}},
/* 17204 */ {(11<<2)|2,{115,51,0}},
/* 17205 */ {(11<<2)|2,{115,51,0}},
/* 17206 */ {(11<<2)|2,{115,51,0}},
/* 17207 */ {(11<<2)|2,{115,51,0}},
/* 17208 */ {(11<<2)|2,{115,51,0}},
/* 17209 */ {(11<<2)|2,{115,51,0}},
/* 17210 */ {(11<<2)|2,{115,51,0}},
/* 17211 */ {(11<<2)|2,{115,51,0}},
/* 17212 */ {(11<<2)|2,{115,51,0}},
/* 17213 */ {(11<<2)|2,{115,51,0}},
/* 17214 */ {(11<<2)|2,{115,51,0}},
/* 17215 */ {(11<<2)|2,{115,51,0}},
/* 17216 */ {(16<<2)|3,{115,52,48}},
/* 17217 */ {(16<<2)|3,{115,52,49}},
/* 17218 */ {(16<<2)|3,{115,52,50}},
/* 17219 */ {(16<<2)|3,{115,52,97}},
/* 17220 */ {(16<<2)|3,{115,52,99}},
/* 17221 */ {(16<<2)|3,{115,52,101}},
/* 17222 */ {(16<<2)|3,{115,52,105}},
/* 17223 */ {(16<<2)|3,{115,52,111}},
/* 17224 */ {(16<<2)|3,{115,52,115}},
/* 17225 */ {(16<<2)|3,{115,52,116}},
/* 17226 */ {(11<<2)|2,{115,52,0}},
/* 17227 */ {(11<<2)|2,{115,52,0}},
/* 17228 */ {(11<<2)|2,{115,52,0}},
/* 17229 */ {(11<<2)|2,{115,52,0}},
/* 17230 */ {(11<<2)|2,{115,52,0}},
/* 17231 */ {(11<<2)|2,{115,52,0}},
/* 17232 */ {(11<<2)|2,{115,52,0}},
/* 17233 */ {(11<<2)|2,{115,52,0}},
/* 17234 */ {(11<<2)|2,{115,52,0}},
/* 17235 */ {(11<<2)|2,{115,52,0}},
/* 17236 */ {(11<<2)|2,{115,52,0}},
/* 17237 */ {(11<<2)|2,{115,52,0}},
/* 17238 */ {(11<<2)|2,{115,52,0}},
/* 17239 */ {(11<<2)|2,{115,52,0}},
/* 17240 */ {(11<<2)|2,{115,52,0}},
/* 17241 */ {(11<<2)|2,{115,52,0}},
/* 17242 */ {(11<<2)|2,{115,52,0}},
/* 17243 */ {(11<<2)|2,{115,52,0}},
/* 17244 */ {(11<<2)|2,{115,52,0}},
/* 17245 */ {(11<<2)|2,{115,52,0}},
/* 17246 */ {(11<<2)|2,{115,52,0}},
/* 17247 */ {(11<<2)|2,{115,52,0}},
/* 17248 */ {(16<<2)|3,{115,53,48}},
/* 17249 */ {(16<<2)|3,{115,53,49}},
/* 17250 */ {(16<<2)|3,{115,53,50}},
/* 17251 */ {(16<<2)|3,{115,53,97}},
/* 17252 */ {(16<<2)|3,{115,53,99}},
/* 17253 */ {(16<<2)|3,{115,53,101}},
/* 17254 */ {(16<<2)|3,{115,53,105}},
/* 17255 */ {(16<<2)|3,{115,53,111}},
/* 17256 */ {(16<<2)|3,{115,53,115}},
/* 17257 */ {(16<<2)|3,{115,53,116}},
/* 17258 */ {(11<<2)|2,{115,53,0}},
/* 17259 */ {(11<<2)|2,{115,53,0}},
/* 17260 */ {(11<<2)|2,{115,53,0}},
/* 17261 */ {(11<<2)|2,{115,53,0}},
/* 17262 */ {(11<<2)|2,{115,53,0}},
/* 17263 */ {(11<<2)|2,{115,53,0}},
/* 17264 */ {(11<<2)|2,{115,53,0}},
/* 17265 */ {(11<<2)|2,{115,53,0}},
/* 17266 */ {(11<<2)|2,{115,53,0}},
/* 17267 */ {(11<<2)|2,{115,53,0}},
/* 17268 */ {(11<<2)|2,{115,53,0}},
/* 17269 */ {(11<<2)|2,{115,53,0}},
/* 17270 */ {(11<<2)|2,{115,53,0}},
/* 17271 */ {(11<<2)|2,{115,53,0}},
/* 17272 */ {(11<<2)|2,{115,53,0}},
/* 17273 */ {(11<<2)|2,{115,53,0}},
/* 17274 */ {(11<<2)|2,{115,53,0}},
/* 17275 */ {(11<<2)|2,{115,53,0}},
/* 17276 */ {(11<<2)|2,{115,53,0}},
/* 17277 */ {(11<<2)|2,{115,53,0}},
/* 17278 */ {(11<<2)|2,{115,53,0}},
/* 17279 */ {(11<<2)|2,{115,53,0}},
/* 17280 */ {(16<<2)|3,{115,54,48}},
/* 17281 */ {(16<<2)|3,{115,54,49}},
/* 17282 */ {(16<<2)|3,{115,54,50}},
/* 17283 */ {(16<<2)|3,{115,54,97}},
/* 17284 */ {(16<<2)|3,{115,54,99}},
/* 17285 */ {(16<<2)|3,{115,54,101}},
/* 17286 */ {(16<<2)|3,{115,54,105}},
/* 17287 */ {(16<<2)|3,{115,54,111}},
/* 17288 */ {(16<<2)|3,{115,54,115}},
/* 17289 */ {(16<<2)|3,{115,54,116}},
/* 17290 */ {(11<<2)|2,{115,54,0}},
/* 17291 */ {(11<<2)|2,{115,54,0}},
/* 17292 */ {(11<<2)|2,{115,54,0}},
/* 17293 */ {(11<<2)|2,{115,54,0}},
/* 17294 */ {(11<<2)|2,{115,54,0}},
/* 17295 */ {(11<<2)|2,{115,54,0}},
/* 17296 */ {(11<<2)|2,{115,54,0}},
/* 17297 */ {(11<<2)|2,{115,54,0}},
/* 17298 */ {(11<<2)|2,{115,54,0}},
/* 17299 */ {(11<<2)|2,{115,54,0}},
/* 17300 */ {(11<<2)|2,{115,54,0}},
/* 17301 */ {(11<<2)|2,{115,54,0}},
/* 17302 */ {(11<<2)|2,{115,54,0}},
/* 17303 */ {(11<<2)|2,{115,54,0}},
/* 17304 */ {(11<<2)|2,{115,54,0}},
/* 17305 */ {(11<<2)|2,{115,54,0}},
/* 17306 */ {(11<<2)|2,{115,54,0}},
/* 17307 */ {(11<<2)|2,{115,54,0}},
/* 17308 */ {(11<<2)|2,{115,54,0}},
/* 17309 */ {(11<<2)|2,{115,54,0}},
/* 17310 */ {(11<<2)|2,{115,54,0}},
/* 17311 */ {(11<<2)|2,{115,54,0}},
/* 17312 */ {(16<<2)|3,{115,55,48}},
/* 17313 */ {(16<<2)|3,{115,55,49}},
/* 17314 */ {(16<<2)|3,{115,55,50}},
/* 17315 */ {(16<<2)|3,{115,55,97}},
/* 17316 */ {(16<<2)|3,{115,55,99}},
/* 17317 */ {(16<<2)|3,{115,55,101}},
/* 17318 */ {(16<<2)|3,{115,55,105}},
/* 17319 */ {(16<<2)|3,{115,55,111}},
/* 17320 */ {(16<<2)|3,{115,55,115}},
/* 17321 */ {(16<<2)|3,{115,55,116}},
/* 17322 */ {(11<<2)|2,{115,55,0}},
/* 17323 */ {(11<<2)|2,{115,55,0}},
/* 17324 */ {(11<<2)|2,{115,55,0}},
/* 17325 */ {(11<<2)|2,{115,55,0}},
/* 17326 */ {(11<<2)|2,{115,55,0}},
/* 17327 */ {(11<<2)|2,{115,55,0}},
/* 17328 */ {(11<<2)|2,{115,55,0}},
/* 17329 */ {(11<<2)|2,{115,55,0}},
/* 17330 */ {(11<<2)|2,{115,55,0}},
/* 17331 */ {(11<<2)|2,{115,55,0}},
/* 17332 */ {(11<<2)|2,{115,55,0}},
/* 17333 */ {(11<<2)|2,{115,55,0}},
/* 17334 */ {(11<<2)|2,{115,55,0}},
/* 17335 */ {(11<<2)|2,{115,55,0}},
/* 17336 */ {(11<<2)|2,{115,55,0}},
/* 17337 */ {(11<<2)|2,{115,55,0}},
/* 17338 */ {(11<<2)|2,{115,55,0}},
/* 17339 */ {(11<<2)|2,{115,55,0}},
/* 17340 */ {(11<<2)|2,{115,55,0}},
/* 17341 */ {(11<<2)|2,{115,55,0}},
/* 17342 */ {(11<<2)|2,{115,55,0}},
/* 17343 */ {(11<<2)|2,{115,55,0}},
/* 17344 */ {(16<<2)|3,{115,56,48}},
/* 17345 */ {(16<<2)|3,{115,56,49}},
/* 17346 */ {(16<<2)|3,{115,56,50}},
/* 17347 */ {(16<<2)|3,{115,56,97}},
/* 17348 */ {(16<<2)|3,{115,56,99}},
/* 17349 */ {(16<<2)|3,{115,56,101}},
/* 17350 */ {(16<<2)|3,{115,56,105}},
/* 17351 */ {(16<<2)|3,{115,56,111}},
/* 17352 */ {(16<<2)|3,{115,56,115}},
/* 17353 */ {(16<<2)|3,{115,56,116}},
/* 17354 */ {(11<<2)|2,{115,56,0}},
/* 17355 */ {(11<<2)|2,{115,56,0}},
/* 17356 */ {(11<<2)|2,{115,56,0}},
/* 17357 */ {(11<<2)|2,{115,56,0}},
/* 17358 */ {(11<<2)|2,{115,56,0}},
/* 17359 */ {(11<<2)|2,{115,56,0}},
/* 17360 */ {(11<<2)|2,{115,56,0}},
/* 17361 */ {(11<<2)|2,{115,56,0}},
/* 17362 */ {(11<<2)|2,{115,56,0}},
/* 17363 */ {(11<<2)|2,{115,56,0}},
/* 17364 */ {(11<<2)|2,{115,56,0}},
/* 17365 */ {(11<<2)|2,{115,56,0}},
/* 17366 */ {(11<<2)|2,{115,56,0}},
/* 17367 */ {(11<<2)|2,{115,56,0}},
/* 17368 */ {(11<<2)|2,{115,56,0}},
/* 17369 */ {(11<<2)|2,{115,56,0}},
/* 17370 */ {(11<<2)|2,{115,56,0}},
/* 17371 */ {(11<<2)|2,{115,56,0}},
/* 17372 */ {(11<<2)|2,{115,56,0}},
/* 17373 */ {(11<<2)|2,{115,56,0}},
/* 17374 */ {(11<<2)|2,{115,56,0}},
/* 17375 */ {(11<<2)|2,{115,56,0}},
/* 17376 */ {(16<<2)|3,{115,57,48}},
/* 17377 */ {(16<<2)|3,{115,57,49}},
/* 17378 */ {(16<<2)|3,{115,57,50}},
/* 17379 */ {(16<<2)|3,{115,57,97}},
/* 17380 */ {(16<<2)|3,{115,57,99}},
/* 17381 */ {(16<<2)|3,{115,57,101}},
/* 17382 */ {(16<<2)|3,{115,57,105}},
/* 17383 */ {(16<<2)|3,{115,57,111}},
/* 17384 */ {(16<<2)|3,{115,57,115}},
/* 17385 */ {(16<<2)|3,{115,57,116}},
/* 17386 */ {(11<<2)|2,{115,57,0}},
/* 17387 */ {(11<<2)|2,{115,57,0}},
/* 17388 */ {(11<<2)|2,{115,57,0}},
/* 17389 */ {(11<<2)|2,{115,57,0}},
/* 17390 */ {(11<<2)|2,{115,57,0}},
/* 17391 */ {(11<<2)|2,{115,57,0}},
/* 17392 */ {(11<<2)|2,{115,57,0}},
/* 17393 */ {(11<<2)|2,{115,57,0}},
/* 17394 */ {(11<<2)|2,{115,57,0}},
/* 17395 */ {(11<<2)|2,{115,57,0}},
/* 17396 */ {(11<<2)|2,{115,57,0}},
/* 17397 */ {(11<<2)|2,{115,57,0}},
/* 17398 */ {(11<<2)|2,{115,57,0}},
/* 17399 */ {(11<<2)|2,{115,57,0}},
/* 17400 */ {(11<<2)|2,{115,57,0}},
/* 17401 */ {(11<<2)|2,{115,57,0}},
/* 17402 */ {(11<<2)|2,{115,57,0}},
/* 17403 */ {(11<<2)|2,{115,57,0}},
/* 17404 */ {(11<<2)|2,{115,57,0}},
/* 17405 */ {(11<<2)|2,{115,57,0}},
/* 17406 */ {(11<<2)|2,{115,57,0}},
/* 17407 */ {(11<<2)|2,{115,57,0}},
/* 17408 */ {(16<<2)|3,{115,61,48}},
/* 17409 */ {(16<<2)|3,{115,61,49}},
/* 17410 */ {(16<<2)|3,{115,61,50}},
/* 17411 */ {(16<<2)|3,{115,61,97}},
/* 17412 */ {(16<<2)|3,{115,61,99}},
/* 17413 */ {(16<<2)|3,{115,61,101}},
/* 17414 */ {(16<<2)|3,{115,61,105}},
/* 17415 */ {(16<<2)|3,{115,61,111}},
/* 17416 */ {(16<<2)|3,{115,61,115}},
/* 17417 */ {(16<<2)|3,{115,61,116}},
/* 17418 */ {(11<<2)|2,{115,61,0}},
/* 17419 */ {(11<<2)|2,{115,61,0}},
/* 17420 */ {(11<<2)|2,{115,61,0}},
/* 17421 */ {(11<<2)|2,{115,61,0}},
/* 17422 */ {(11<<2)|2,{115,61,0}},
/* 17423 */ {(11<<2)|2,{115,61,0}},
/* 17424 */ {(11<<2)|2,{115,61,0}},
/* 17425 */ {(11<<2)|2,{115,61,0}},
/* 17426 */ {(11<<2)|2,{115,61,0}},
/* 17427 */ {(11<<2)|2,{115,61,0}},
/* 17428 */ {(11<<2)|2,{115,61,0}},
/* 17429 */ {(11<<2)|2,{115,61,0}},
/* 17430 */ {(11<<2)|2,{115,61,0}},
/* 17431 */ {(11<<2)|2,{115,61,0}},
/* 17432 */ {(11<<2)|2,{115,61,0}},
/* 17433 */ {(11<<2)|2,{115,61,0}},
/* 17434 */ {(11<<2)|2,{115,61,0}},
/* 17435 */ {(11<<2)|2,{115,61,0}},
/* 17436 */ {(11<<2)|2,{115,61,0}},
/* 17437 */ {(11<<2)|2,{115,61,0}},
/* 17438 */ {(11<<2)|2,{115,61,0}},
/* 17439 */ {(11<<2)|2,{115,61,0}},
/* 17440 */ {(16<<2)|3,{115,65,48}},
/* 17441 */ {(16<<2)|3,{115,65,49}},
/* 17442 */ {(16<<2)|3,{115,65,50}},
/* 17443 */ {(16<<2)|3,{115,65,97}},
/* 17444 */ {(16<<2)|3,{115,65,99}},
/* 17445 */ {(16<<2)|3,{115,65,101}},
/* 17446 */ {(16<<2)|3,{115,65,105}},
/* 17447 */ {(16<<2)|3,{115,65,111}},
/* 17448 */ {(16<<2)|3,{115,65,115}},
/* 17449 */ {(16<<2)|3,{115,65,116}},
/* 17450 */ {(11<<2)|2,{115,65,0}},
/* 17451 */ {(11<<2)|2,{115,65,0}},
/* 17452 */ {(11<<2)|2,{115,65,0}},
/* 17453 */ {(11<<2)|2,{115,65,0}},
/* 17454 */ {(11<<2)|2,{115,65,0}},
/* 17455 */ {(11<<2)|2,{115,65,0}},
/* 17456 */ {(11<<2)|2,{115,65,0}},
/* 17457 */ {(11<<2)|2,{115,65,0}},
/* 17458 */ {(11<<2)|2,{115,65,0}},
/* 17459 */ {(11<<2)|2,{115,65,0}},
/* 17460 */ {(11<<2)|2,{115,65,0}},
/* 17461 */ {(11<<2)|2,{115,65,0}},
/* 17462 */ {(11<<2)|2,{115,65,0}},
/* 17463 */ {(11<<2)|2,{115,65,0}},
/* 17464 */ {(11<<2)|2,{115,65,0}},
/* 17465 */ {(11<<2)|2,{115,65,0}},
/* 17466 */ {(11<<2)|2,{115,65,0}},
/* 17467 */ {(11<<2)|2,{115,65,0}},
/* 17468 */ {(11<<2)|2,{115,65,0}},
/* 17469 */ {(11<<2)|2,{115,65,0}},
/* 17470 */ {(11<<2)|2,{115,65,0}},
/* 17471 */ {(11<<2)|2,{115,65,0}},
/* 17472 */ {(16<<2)|3,{115,95,48}},
/* 17473 */ {(16<<2)|3,{115,95,49}},
/* 17474 */ {(16<<2)|3,{115,95,50}},
/* 17475 */ {(16<<2)|3,{115,95,97}},
/* 17476 */ {(16<<2)|3,{115,95,99}},
/* 17477 */ {(16<<2)|3,{115,95,101}},
/* 17478 */ {(16<<2)|3,{115,95,105}},
/* 17479 */ {(16<<2)|3,{115,95,111}},
/* 17480 */ {(16<<2)|3,{115,95,115}},
/* 17481 */ {(16<<2)|3,{115,95,116}},
/* 17482 */ {(11<<2)|2,{115,95,0}},
/* 17483 */ {(11<<2)|2,{115,95,0}},
/* 17484 */ {(11<<2)|2,{115,95,0}},
/* 17485 */ {(11<<2)|2,{115,95,0}},
/* 17486 */ {(11<<2)|2,{115,95,0}},
/* 17487 */ {(11<<2)|2,{115,95,0}},
/* 17488 */ {(11<<2)|2,{115,95,0}},
/* 17489 */ {(11<<2)|2,{115,95,0}},
/* 17490 */ {(11<<2)|2,{115,95,0}},
/* 17491 */ {(11<<2)|2,{115,95,0}},
/* 17492 */ {(11<<2)|2,{115,95,0}},
/* 17493 */ {(11<<2)|2,{115,95,0}},
/* 17494 */ {(11<<2)|2,{115,95,0}},
/* 17495 */ {(11<<2)|2,{115,95,0}},
/* 17496 */ {(11<<2)|2,{115,95,0}},
/* 17497 */ {(11<<2)|2,{115,95,0}},
/* 17498 */ {(11<<2)|2,{115,95,0}},
/* 17499 */ {(11<<2)|2,{115,95,0}},
/* 17500 */ {(11<<2)|2,{115,95,0}},
/* 17501 */ {(11<<2)|2,{115,95,0}},
/* 17502 */ {(11<<2)|2,{115,95,0}},
/* 17503 */ {(11<<2)|2,{115,95,0}},
/* 17504 */ {(16<<2)|3,{115,98,48}},
/* 17505 */ {(16<<2)|3,{115,98,49}},
/* 17506 */ {(16<<2)|3,{115,98,50}},
/* 17507 */ {(16<<2)|3,{115,98,97}},
/* 17508 */ {(16<<2)|3,{115,98,99}},
/* 17509 */ {(16<<2)|3,{115,98,101}},
/* 17510 */ {(16<<2)|3,{115,98,105}},
/* 17511 */ {(16<<2)|3,{115,98,111}},
/* 17512 */ {(16<<2)|3,{115,98,115}},
/* 17513 */ {(16<<2)|3,{115,98,116}},
/* 17514 */ {(11<<2)|2,{115,98,0}},
/* 17515 */ {(11<<2)|2,{115,98,0}},
/* 17516 */ {(11<<2)|2,{115,98,0}},
/* 17517 */ {(11<<2)|2,{115,98,0}},
/* 17518 */ {(11<<2)|2,{115,98,0}},
/* 17519 */ {(11<<2)|2,{115,98,0}},
/* 17520 */ {(11<<2)|2,{115,98,0}},
/* 17521 */ {(11<<2)|2,{115,98,0}},
/* 17522 */ {(11<<2)|2,{115,98,0}},
/* 17523 */ {(11<<2)|2,{115,98,0}},
/* 17524 */ {(11<<2)|2,{115,98,0}},
/* 17525 */ {(11<<2)|2,{115,98,0}},
/* 17526 */ {(11<<2)|2,{115,98,0}},
/* 17527 */ {(11<<2)|2,{115,98,0}},
/* 17528 */ {(11<<2)|2,{115,98,0}},
/* 17529 */ {(11<<2)|2,{115,98,0}},
/* 17530 */ {(11<<2)|2,{115,98,0}},
/* 17531 */ {(11<<2)|2,{115,98,0}},
/* 17532 */ {(11<<2)|2,{115,98,0}},
/* 17533 */ {(11<<2)|2,{115,98,0}},
/* 17534 */ {(11<<2)|2,{115,98,0}},
/* 17535 */ {(11<<2)|2,{115,98,0}},
/* 17536 */ {(16<<2)|3,{115,100,48}},
/* 17537 */ {(16<<2)|3,{115,100,49}},
/* 17538 */ {(16<<2)|3,{115,100,50}},
/* 17539 */ {(16<<2)|3,{115,100,97}},
/* 17540 */ {(16<<2)|3,{115,100,99}},
/* 17541 */ {(16<<2)|3,{115,100,101}},
/* 17542 */ {(16<<2)|3,{115,100,105}},
/* 17543 */ {(16<<2)|3,{115,100,111}},
/* 17544 */ {(16<<2)|3,{115,100,115}},
/* 17545 */ {(16<<2)|3,{115,100,116}},
/* 17546 */ {(11<<2)|2,{115,100,0}},
/* 17547 */ {(11<<2)|2,{115,100,0}},
/* 17548 */ {(11<<2)|2,{115,100,0}},
/* 17549 */ {(11<<2)|2,{115,100,0}},
/* 17550 */ {(11<<2)|2,{115,100,0}},
/* 17551 */ {(11<<2)|2,{115,100,0}},
/* 17552 */ {(11<<2)|2,{115,100,0}},
/* 17553 */ {(11<<2)|2,{115,100,0}},
/* 17554 */ {(11<<2)|2,{115,100,0}},
/* 17555 */ {(11<<2)|2,{115,100,0}},
/* 17556 */ {(11<<2)|2,{115,100,0}},
/* 17557 */ {(11<<2)|2,{115,100,0}},
/* 17558 */ {(11<<2)|2,{115,100,0}},
/* 17559 */ {(11<<2)|2,{115,100,0}},
/* 17560 */ {(11<<2)|2,{115,100,0}},
/* 17561 */ {(11<<2)|2,{115,100,0}},
/* 17562 */ {(11<<2)|2,{115,100,0}},
/* 17563 */ {(11<<2)|2,{115,100,0}},
/* 17564 */ {(11<<2)|2,{115,100,0}},
/* 17565 */ {(11<<2)|2,{115,100,0}},
/* 17566 */ {(11<<2)|2,{115,100,0}},
/* 17567 */ {(11<<2)|2,{115,100,0}},
/* 17568 */ {(16<<2)|3,{115,102,48}},
/* 17569 */ {(16<<2)|3,{115,102,49}},
/* 17570 */ {(16<<2)|3,{115,102,50}},
/* 17571 */ {(16<<2)|3,{115,102,97}},
/* 17572 */ {(16<<2)|3,{115,102,99}},
/* 17573 */ {(16<<2)|3,{115,102,101}},
/* 17574 */ {(16<<2)|3,{115,102,105}},
/* 17575 */ {(16<<2)|3,{115,102,111}},
/* 17576 */ {(16<<2)|3,{115,102,115}},
/* 17577 */ {(16<<2)|3,{115,102,116}},
/* 17578 */ {(11<<2)|2,{115,102,0}},
/* 17579 */ {(11<<2)|2,{115,102,0}},
/* 17580 */ {(11<<2)|2,{115,102,0}},
/* 17581 */ {(11<<2)|2,{115,102,0}},
/* 17582 */ {(11<<2)|2,{115,102,0}},
/* 17583 */ {(11<<2)|2,{115,102,0}},
/* 17584 */ {(11<<2)|2,{115,102,0}},
/* 17585 */ {(11<<2)|2,{115,102,0}},
/* 17586 */ {(11<<2)|2,{115,102,0}},
/* 17587 */ {(11<<2)|2,{115,102,0}},
/* 17588 */ {(11<<2)|2,{115,102,0}},
/* 17589 */ {(11<<2)|2,{115,102,0}},
/* 17590 */ {(11<<2)|2,{115,102,0}},
/* 17591 */ {(11<<2)|2,{115,102,0}},
/* 17592 */ {(11<<2)|2,{115,102,0}},
/* 17593 */ {(11<<2)|2,{115,102,0}},
/* 17594 */ {(11<<2)|2,{115,102,0}},
/* 17595 */ {(11<<2)|2,{115,102,0}},
/* 17596 */ {(11<<2)|2,{115,102,0}},
/* 17597 */ {(11<<2)|2,{115,102,0}},
/* 17598 */ {(11<<2)|2,{115,102,0}},
/* 17599 */ {(11<<2)|2,{115,102,0}},
/* 17600 */ {(16<<2)|3,{115,103,48}},
/* 17601 */ {(16<<2)|3,{115,103,49}},
/* 17602 */ {(16<<2)|3,{115,103,50}},
/* 17603 */ {(16<<2)|3,{115,103,97}},
/* 17604 */ {(16<<2)|3,{115,103,99}},
/* 17605 */ {(16<<2)|3,{115,103,101}},
/* 17606 */ {(16<<2)|3,{115,103,105}},
/* 17607 */ {(16<<2)|3,{115,103,111}},
/* 17608 */ {(16<<2)|3,{115,103,115}},
/* 17609 */ {(16<<2)|3,{115,103,116}},
/* 17610 */ {(11<<2)|2,{115,103,0}},
/* 17611 */ {(11<<2)|2,{115,103,0}},
/* 17612 */ {(11<<2)|2,{115,103,0}},
/* 17613 */ {(11<<2)|2,{115,103,0}},
/* 17614 */ {(11<<2)|2,{115,103,0}},
/* 17615 */ {(11<<2)|2,{115,103,0}},
/* 17616 */ {(11<<2)|2,{115,103,0}},
/* 17617 */ {(11<<2)|2,{115,103,0}},
/* 17618 */ {(11<<2)|2,{115,103,0}},
/* 17619 */ {(11<<2)|2,{115,103,0}},
/* 17620 */ {(11<<2)|2,{115,103,0}},
/* 17621 */ {(11<<2)|2,{115,103,0}},
/* 17622 */ {(11<<2)|2,{115,103,0}},
/* 17623 */ {(11<<2)|2,{115,103,0}},
/* 17624 */ {(11<<2)|2,{115,103,0}},
/* 17625 */ {(11<<2)|2,{115,103,0}},
/* 17626 */ {(11<<2)|2,{115,103,0}},
/* 17627 */ {(11<<2)|2,{115,103,0}},
/* 17628 */ {(11<<2)|2,{115,103,0}},
/* 17629 */ {(11<<2)|2,{115,103,0}},
/* 17630 */ {(11<<2)|2,{115,103,0}},
/* 17631 */ {(11<<2)|2,{115,103,0}},
/* 17632 */ {(16<<2)|3,{115,104,48}},
/* 17633 */ {(16<<2)|3,{115,104,49}},
/* 17634 */ {(16<<2)|3,{115,104,50}},
/* 17635 */ {(16<<2)|3,{115,104,97}},
/* 17636 */ {(16<<2)|3,{115,104,99}},
/* 17637 */ {(16<<2)|3,{115,104,101}},
/* 17638 */ {(16<<2)|3,{115,104,105}},
/* 17639 */ {(16<<2)|3,{115,104,111}},
/* 17640 */ {(16<<2)|3,{115,104,115}},
/* 17641 */ {(16<<2)|3,{115,104,116}},
/* 17642 */ {(11<<2)|2,{115,104,0}},
/* 17643 */ {(11<<2)|2,{115,104,0}},
/* 17644 */ {(11<<2)|2,{115,104,0}},
/* 17645 */ {(11<<2)|2,{115,104,0}},
/* 17646 */ {(11<<2)|2,{115,104,0}},
/* 17647 */ {(11<<2)|2,{115,104,0}},
/* 17648 */ {(11<<2)|2,{115,104,0}},
/* 17649 */ {(11<<2)|2,{115,104,0}},
/* 17650 */ {(11<<2)|2,{115,104,0}},
/* 17651 */ {(11<<2)|2,{115,104,0}},
/* 17652 */ {(11<<2)|2,{115,104,0}},
/* 17653 */ {(11<<2)|2,{115,104,0}},
/* 17654 */ {(11<<2)|2,{115,104,0}},
/* 17655 */ {(11<<2)|2,{115,104,0}},
/* 17656 */ {(11<<2)|2,{115,104,0}},
/* 17657 */ {(11<<2)|2,{115,104,0}},
/* 17658 */ {(11<<2)|2,{115,104,0}},
/* 17659 */ {(11<<2)|2,{115,104,0}},
/* 17660 */ {(11<<2)|2,{115,104,0}},
/* 17661 */ {(11<<2)|2,{115,104,0}},
/* 17662 */ {(11<<2)|2,{115,104,0}},
/* 17663 */ {(11<<2)|2,{115,104,0}},
/* 17664 */ {(16<<2)|3,{115,108,48}},
/* 17665 */ {(16<<2)|3,{115,108,49}},
/* 17666 */ {(16<<2)|3,{115,108,50}},
/* 17667 */ {(16<<2)|3,{115,108,97}},
/* 17668 */ {(16<<2)|3,{115,108,99}},
/* 17669 */ {(16<<2)|3,{115,108,101}},
/* 17670 */ {(16<<2)|3,{115,108,105}},
/* 17671 */ {(16<<2)|3,{115,108,111}},
/* 17672 */ {(16<<2)|3,{115,108,115}},
/* 17673 */ {(16<<2)|3,{115,108,116}},
/* 17674 */ {(11<<2)|2,{115,108,0}},
/* 17675 */ {(11<<2)|2,{115,108,0}},
/* 17676 */ {(11<<2)|2,{115,108,0}},
/* 17677 */ {(11<<2)|2,{115,108,0}},
/* 17678 */ {(11<<2)|2,{115,108,0}},
/* 17679 */ {(11<<2)|2,{115,108,0}},
/* 17680 */ {(11<<2)|2,{115,108,0}},
/* 17681 */ {(11<<2)|2,{115,108,0}},
/* 17682 */ {(11<<2)|2,{115,108,0}},
/* 17683 */ {(11<<2)|2,{115,108,0}},
/* 17684 */ {(11<<2)|2,{115,108,0}},
/* 17685 */ {(11<<2)|2,{115,108,0}},
/* 17686 */ {(11<<2)|2,{115,108,0}},
/* 17687 */ {(11<<2)|2,{115,108,0}},
/* 17688 */ {(11<<2)|2,{115,108,0}},
/* 17689 */ {(11<<2)|2,{115,108,0}},
/* 17690 */ {(11<<2)|2,{115,108,0}},
/* 17691 */ {(11<<2)|2,{115,108,0}},
/* 17692 */ {(11<<2)|2,{115,108,0}},
/* 17693 */ {(11<<2)|2,{115,108,0}},
/* 17694 */ {(11<<2)|2,{115,108,0}},
/* 17695 */ {(11<<2)|2,{115,108,0}},
/* 17696 */ {(16<<2)|3,{115,109,48}},
/* 17697 */ {(16<<2)|3,{115,109,49}},
/* 17698 */ {(16<<2)|3,{115,109,50}},
/* 17699 */ {(16<<2)|3,{115,109,97}},
/* 17700 */ {(16<<2)|3,{115,109,99}},
/* 17701 */ {(16<<2)|3,{115,109,101}},
/* 17702 */ {(16<<2)|3,{115,109,105}},
/* 17703 */ {(16<<2)|3,{115,109,111}},
/* 17704 */ {(16<<2)|3,{115,109,115}},
/* 17705 */ {(16<<2)|3,{115,109,116}},
/* 17706 */ {(11<<2)|2,{115,109,0}},
/* 17707 */ {(11<<2)|2,{115,109,0}},
/* 17708 */ {(11<<2)|2,{115,109,0}},
/* 17709 */ {(11<<2)|2,{115,109,0}},
/* 17710 */ {(11<<2)|2,{115,109,0}},
/* 17711 */ {(11<<2)|2,{115,109,0}},
/* 17712 */ {(11<<2)|2,{115,109,0}},
/* 17713 */ {(11<<2)|2,{115,109,0}},
/* 17714 */ {(11<<2)|2,{115,109,0}},
/* 17715 */ {(11<<2)|2,{115,109,0}},
/* 17716 */ {(11<<2)|2,{115,109,0}},
/* 17717 */ {(11<<2)|2,{115,109,0}},
/* 17718 */ {(11<<2)|2,{115,109,0}},
/* 17719 */ {(11<<2)|2,{115,109,0}},
/* 17720 */ {(11<<2)|2,{115,109,0}},
/* 17721 */ {(11<<2)|2,{115,109,0}},
/* 17722 */ {(11<<2)|2,{115,109,0}},
/* 17723 */ {(11<<2)|2,{115,109,0}},
/* 17724 */ {(11<<2)|2,{115,109,0}},
/* 17725 */ {(11<<2)|2,{115,109,0}},
/* 17726 */ {(11<<2)|2,{115,109,0}},
/* 17727 */ {(11<<2)|2,{115,109,0}},
/* 17728 */ {(16<<2)|3,{115,110,48}},
/* 17729 */ {(16<<2)|3,{115,110,49}},
/* 17730 */ {(16<<2)|3,{115,110,50}},
/* 17731 */ {(16<<2)|3,{115,110,97}},
/* 17732 */ {(16<<2)|3,{115,110,99}},
/* 17733 */ {(16<<2)|3,{115,110,101}},
/* 17734 */ {(16<<2)|3,{115,110,105}},
/* 17735 */ {(16<<2)|3,{115,110,111}},
/* 17736 */ {(16<<2)|3,{115,110,115}},
/* 17737 */ {(16<<2)|3,{115,110,116}},
/* 17738 */ {(11<<2)|2,{115,110,0}},
/* 17739 */ {(11<<2)|2,{115,110,0}},
/* 17740 */ {(11<<2)|2,{115,110,0}},
/* 17741 */ {(11<<2)|2,{115,110,0}},
/* 17742 */ {(11<<2)|2,{115,110,0}},
/* 17743 */ {(11<<2)|2,{115,110,0}},
/* 17744 */ {(11<<2)|2,{115,110,0}},
/* 17745 */ {(11<<2)|2,{115,110,0}},
/* 17746 */ {(11<<2)|2,{115,110,0}},
/* 17747 */ {(11<<2)|2,{115,110,0}},
/* 17748 */ {(11<<2)|2,{115,110,0}},
/* 17749 */ {(11<<2)|2,{115,110,0}},
/* 17750 */ {(11<<2)|2,{115,110,0}},
/* 17751 */ {(11<<2)|2,{115,110,0}},
/* 17752 */ {(11<<2)|2,{115,110,0}},
/* 17753 */ {(11<<2)|2,{115,110,0}},
/* 17754 */ {(11<<2)|2,{115,110,0}},
/* 17755 */ {(11<<2)|2,{115,110,0}},
/* 17756 */ {(11<<2)|2,{115,110,0}},
/* 17757 */ {(11<<2)|2,{115,110,0}},
/* 17758 */ {(11<<2)|2,{115,110,0}},
/* 17759 */ {(11<<2)|2,{115,110,0}},
/* 17760 */ {(16<<2)|3,{115,112,48}},
/* 17761 */ {(16<<2)|3,{115,112,49}},
/* 17762 */ {(16<<2)|3,{115,112,50}},
/* 17763 */ {(16<<2)|3,{115,112,97}},
/* 17764 */ {(16<<2)|3,{115,112,99}},
/* 17765 */ {(16<<2)|3,{115,112,101}},
/* 17766 */ {(16<<2)|3,{115,112,105}},
/* 17767 */ {(16<<2)|3,{115,112,111}},
/* 17768 */ {(16<<2)|3,{115,112,115}},
/* 17769 */ {(16<<2)|3,{115,112,116}},
/* 17770 */ {(11<<2)|2,{115,112,0}},
/* 17771 */ {(11<<2)|2,{115,112,0}},
/* 17772 */ {(11<<2)|2,{115,112,0}},
/* 17773 */ {(11<<2)|2,{115,112,0}},
/* 17774 */ {(11<<2)|2,{115,112,0}},
/* 17775 */ {(11<<2)|2,{115,112,0}},
/* 17776 */ {(11<<2)|2,{115,112,0}},
/* 17777 */ {(11<<2)|2,{115,112,0}},
/* 17778 */ {(11<<2)|2,{115,112,0}},
/* 17779 */ {(11<<2)|2,{115,112,0}},
/* 17780 */ {(11<<2)|2,{115,112,0}},
/* 17781 */ {(11<<2)|2,{115,112,0}},
/* 17782 */ {(11<<2)|2,{115,112,0}},
/* 17783 */ {(11<<2)|2,{115,112,0}},
/* 17784 */ {(11<<2)|2,{115,112,0}},
/* 17785 */ {(11<<2)|2,{115,112,0}},
/* 17786 */ {(11<<2)|2,{115,112,0}},
/* 17787 */ {(11<<2)|2,{115,112,0}},
/* 17788 */ {(11<<2)|2,{115,112,0}},
/* 17789 */ {(11<<2)|2,{115,112,0}},
/* 17790 */ {(11<<2)|2,{115,112,0}},
/* 17791 */ {(11<<2)|2,{115,112,0}},
/* 17792 */ {(16<<2)|3,{115,114,48}},
/* 17793 */ {(16<<2)|3,{115,114,49}},
/* 17794 */ {(16<<2)|3,{115,114,50}},
/* 17795 */ {(16<<2)|3,{115,114,97}},
/* 17796 */ {(16<<2)|3,{115,114,99}},
/* 17797 */ {(16<<2)|3,{115,114,101}},
/* 17798 */ {(16<<2)|3,{115,114,105}},
/* 17799 */ {(16<<2)|3,{115,114,111}},
/* 17800 */ {(16<<2)|3,{115,114,115}},
/* 17801 */ {(16<<2)|3,{115,114,116}},
/* 17802 */ {(11<<2)|2,{115,114,0}},
/* 17803 */ {(11<<2)|2,{115,114,0}},
/* 17804 */ {(11<<2)|2,{115,114,0}},
/* 17805 */ {(11<<2)|2,{115,114,0}},
/* 17806 */ {(11<<2)|2,{115,114,0}},
/* 17807 */ {(11<<2)|2,{115,114,0}},
/* 17808 */ {(11<<2)|2,{115,114,0}},
/* 17809 */ {(11<<2)|2,{115,114,0}},
/* 17810 */ {(11<<2)|2,{115,114,0}},
/* 17811 */ {(11<<2)|2,{115,114,0}},
/* 17812 */ {(11<<2)|2,{115,114,0}},
/* 17813 */ {(11<<2)|2,{115,114,0}},
/* 17814 */ {(11<<2)|2,{115,114,0}},
/* 17815 */ {(11<<2)|2,{115,114,0}},
/* 17816 */ {(11<<2)|2,{115,114,0}},
/* 17817 */ {(11<<2)|2,{115,114,0}},
/* 17818 */ {(11<<2)|2,{115,114,0}},
/* 17819 */ {(11<<2)|2,{115,114,0}},
/* 17820 */ {(11<<2)|2,{115,114,0}},
/* 17821 */ {(11<<2)|2,{115,114,0}},
/* 17822 */ {(11<<2)|2,{115,114,0}},
/* 17823 */ {(11<<2)|2,{115,114,0}},
/* 17824 */ {(16<<2)|3,{115,117,48}},
/* 17825 */ {(16<<2)|3,{115,117,49}},
/* 17826 */ {(16<<2)|3,{115,117,50}},
/* 17827 */ {(16<<2)|3,{115,117,97}},
/* 17828 */ {(16<<2)|3,{115,117,99}},
/* 17829 */ {(16<<2)|3,{115,117,101}},
/* 17830 */ {(16<<2)|3,{115,117,105}},
/* 17831 */ {(16<<2)|3,{115,117,111}},
/* 17832 */ {(16<<2)|3,{115,117,115}},
/* 17833 */ {(16<<2)|3,{115,117,116}},
/* 17834 */ {(11<<2)|2,{115,117,0}},
/* 17835 */ {(11<<2)|2,{115,117,0}},
/* 17836 */ {(11<<2)|2,{115,117,0}},
/* 17837 */ {(11<<2)|2,{115,117,0}},
/* 17838 */ {(11<<2)|2,{115,117,0}},
/* 17839 */ {(11<<2)|2,{115,117,0}},
/* 17840 */ {(11<<2)|2,{115,117,0}},
/* 17841 */ {(11<<2)|2,{115,117,0}},
/* 17842 */ {(11<<2)|2,{115,117,0}},
/* 17843 */ {(11<<2)|2,{115,117,0}},
/* 17844 */ {(11<<2)|2,{115,117,0}},
/* 17845 */ {(11<<2)|2,{115,117,0}},
/* 17846 */ {(11<<2)|2,{115,117,0}},
/* 17847 */ {(11<<2)|2,{115,117,0}},
/* 17848 */ {(11<<2)|2,{115,117,0}},
/* 17849 */ {(11<<2)|2,{115,117,0}},
/* 17850 */ {(11<<2)|2,{115,117,0}},
/* 17851 */ {(11<<2)|2,{115,117,0}},
/* 17852 */ {(11<<2)|2,{115,117,0}},
/* 17853 */ {(11<<2)|2,{115,117,0}},
/* 17854 */ {(11<<2)|2,{115,117,0}},
/* 17855 */ {(11<<2)|2,{115,117,0}},
/* 17856 */ {(12<<2)|2,{115,58,0}},
/* 17857 */ {(12<<2)|2,{115,58,0}},
/* 17858 */ {(12<<2)|2,{115,58,0}},
/* 17859 */ {(12<<2)|2,{115,58,0}},
/* 17860 */ {(12<<2)|2,{115,58,0}},
/* 17861 */ {(12<<2)|2,{115,58,0}},
/* 17862 */ {(12<<2)|2,{115,58,0}},
/* 17863 */ {(12<<2)|2,{115,58,0}},
/* 17864 */ {(12<<2)|2,{115,58,0}},
/* 17865 */ {(12<<2)|2,{115,58,0}},
/* 17866 */ {(12<<2)|2,{115,58,0}},
/* 17867 */ {(12<<2)|2,{115,58,0}},
/* 17868 */ {(12<<2)|2,{115,58,0}},
/* 17869 */ {(12<<2)|2,{115,58,0}},
/* 17870 */ {(12<<2)|2,{115,58,0}},
/* 17871 */ {(12<<2)|2,{115,58,0}},
/* 17872 */ {(12<<2)|2,{115,66,0}},
/* 17873 */ {(12<<2)|2,{115,66,0}},
/* 17874 */ {(12<<2)|2,{115,66,0}},
/* 17875 */ {(12<<2)|2,{115,66,0}},
/* 17876 */ {(12<<2)|2,{115,66,0}},
/* 17877 */ {(12<<2)|2,{115,66,0}},
/* 17878 */ {(12<<2)|2,{115,66,0}},
/* 17879 */ {(12<<2)|2,{115,66,0}},
/* 17880 */ {(12<<2)|2,{115,66,0}},
/* 17881 */ {(12<<2)|2,{115,66,0}},
/* 17882 */ {(12<<2)|2,{115,66,0}},
/* 17883 */ {(12<<2)|2,{115,66,0}},
/* 17884 */ {(12<<2)|2,{115,66,0}},
/* 17885 */ {(12<<2)|2,{115,66,0}},
/* 17886 */ {(12<<2)|2,{115,66,0}},
/* 17887 */ {(12<<2)|2,{115,66,0}},
/* 17888 */ {(12<<2)|2,{115,67,0}},
/* 17889 */ {(12<<2)|2,{115,67,0}},
/* 17890 */ {(12<<2)|2,{115,67,0}},
/* 17891 */ {(12<<2)|2,{115,67,0}},
/* 17892 */ {(12<<2)|2,{115,67,0}},
/* 17893 */ {(12<<2)|2,{115,67,0}},
/* 17894 */ {(12<<2)|2,{115,67,0}},
/* 17895 */ {(12<<2)|2,{115,67,0}},
/* 17896 */ {(12<<2)|2,{115,67,0}},
/* 17897 */ {(12<<2)|2,{115,67,0}},
/* 17898 */ {(12<<2)|2,{115,67,0}},
/* 17899 */ {(12<<2)|2,{115,67,0}},
/* 17900 */ {(12<<2)|2,{115,67,0}},
/* 17901 */ {(12<<2)|2,{115,67,0}},
/* 17902 */ {(12<<2)|2,{115,67,0}},
/* 17903 */ {(12<<2)|2,{115,67,0}},
/* 17904 */ {(12<<2)|2,{115,68,0}},
/* 17905 */ {(12<<2)|2,{115,68,0}},
/* 17906 */ {(12<<2)|2,{115,68,0}},
/* 17907 */ {(12<<2)|2,{115,68,0}},
/* 17908 */ {(12<<2)|2,{115,68,0}},
/* 17909 */ {(12<<2)|2,{115,68,0}},
/* 17910 */ {(12<<2)|2,{115,68,0}},
/* 17911 */ {(12<<2)|2,{115,68,0}},
/* 17912 */ {(12<<2)|2,{115,68,0}},
/* 17913 */ {(12<<2)|2,{115,68,0}},
/* 17914 */ {(12<<2)|2,{115,68,0}},
/* 17915 */ {(12<<2)|2,{115,68,0}},
/* 17916 */ {(12<<2)|2,{115,68,0}},
/* 17917 */ {(12<<2)|2,{115,68,0}},
/* 17918 */ {(12<<2)|2,{115,68,0}},
/* 17919 */ {(12<<2)|2,{115,68,0}},
/* 17920 */ {(12<<2)|2,{115,69,0}},
/* 17921 */ {(12<<2)|2,{115,69,0}},
/* 17922 */ {(12<<2)|2,{115,69,0}},
/* 17923 */ {(12<<2)|2,{115,69,0}},
/* 17924 */ {(12<<2)|2,{115,69,0}},
/* 17925 */ {(12<<2)|2,{115,69,0}},
/* 17926 */ {(12<<2)|2,{115,69,0}},
/* 17927 */ {(12<<2)|2,{115,69,0}},
/* 17928 */ {(12<<2)|2,{115,69,0}},
/* 17929 */ {(12<<2)|2,{115,69,0}},
/* 17930 */ {(12<<2)|2,{115,69,0}},
/* 17931 */ {(12<<2)|2,{115,69,0}},
/* 17932 */ {(12<<2)|2,{115,69,0}},
/* 17933 */ {(12<<2)|2,{115,69,0}},
/* 17934 */ {(12<<2)|2,{115,69,0}},
/* 17935 */ {(12<<2)|2,{115,69,0}},
/* 17936 */ {(12<<2)|2,{115,70,0}},
/* 17937 */ {(12<<2)|2,{115,70,0}},
/* 17938 */ {(12<<2)|2,{115,70,0}},
/* 17939 */ {(12<<2)|2,{115,70,0}},
/* 17940 */ {(12<<2)|2,{115,70,0}},
/* 17941 */ {(12<<2)|2,{115,70,0}},
/* 17942 */ {(12<<2)|2,{115,70,0}},
/* 17943 */ {(12<<2)|2,{115,70,0}},
/* 17944 */ {(12<<2)|2,{115,70,0}},
/* 17945 */ {(12<<2)|2,{115,70,0}},
/* 17946 */ {(12<<2)|2,{115,70,0}},
/* 17947 */ {(12<<2)|2,{115,70,0}},
/* 17948 */ {(12<<2)|2,{115,70,0}},
/* 17949 */ {(12<<2)|2,{115,70,0}},
/* 17950 */ {(12<<2)|2,{115,70,0}},
/* 17951 */ {(12<<2)|2,{115,70,0}},
/* 17952 */ {(12<<2)|2,{115,71,0}},
/* 17953 */ {(12<<2)|2,{115,71,0}},
/* 17954 */ {(12<<2)|2,{115,71,0}},
/* 17955 */ {(12<<2)|2,{115,71,0}},
/* 17956 */ {(12<<2)|2,{115,71,0}},
/* 17957 */ {(12<<2)|2,{115,71,0}},
/* 17958 */ {(12<<2)|2,{115,71,0}},
/* 17959 */ {(12<<2)|2,{115,71,0}},
/* 17960 */ {(12<<2)|2,{115,71,0}},
/* 17961 */ {(12<<2)|2,{115,71,0}},
/* 17962 */ {(12<<2)|2,{115,71,0}},
/* 17963 */ {(12<<2)|2,{115,71,0}},
/* 17964 */ {(12<<2)|2,{115,71,0}},
/* 17965 */ {(12<<2)|2,{115,71,0}},
/* 17966 */ {(12<<2)|2,{115,71,0}},
/* 17967 */ {(12<<2)|2,{115,71,0}},
/* 17968 */ {(12<<2)|2,{115,72,0}},
/* 17969 */ {(12<<2)|2,{115,72,0}},
/* 17970 */ {(12<<2)|2,{115,72,0}},
/* 17971 */ {(12<<2)|2,{115,72,0}},
/* 17972 */ {(12<<2)|2,{115,72,0}},
/* 17973 */ {(12<<2)|2,{115,72,0}},
/* 17974 */ {(12<<2)|2,{115,72,0}},
/* 17975 */ {(12<<2)|2,{115,72,0}},
/* 17976 */ {(12<<2)|2,{115,72,0}},
/* 17977 */ {(12<<2)|2,{115,72,0}},
/* 17978 */ {(12<<2)|2,{115,72,0}},
/* 17979 */ {(12<<2)|2,{115,72,0}},
/* 17980 */ {(12<<2)|2,{115,72,0}},
/* 17981 */ {(12<<2)|2,{115,72,0}},
/* 17982 */ {(12<<2)|2,{115,72,0}},
/* 17983 */ {(12<<2)|2,{115,72,0}},
/* 17984 */ {(12<<2)|2,{115,73,0}},
/* 17985 */ {(12<<2)|2,{115,73,0}},
/* 17986 */ {(12<<2)|2,{115,73,0}},
/* 17987 */ {(12<<2)|2,{115,73,0}},
/* 17988 */ {(12<<2)|2,{115,73,0}},
/* 17989 */ {(12<<2)|2,{115,73,0}},
/* 17990 */ {(12<<2)|2,{115,73,0}},
/* 17991 */ {(12<<2)|2,{115,73,0}},
/* 17992 */ {(12<<2)|2,{115,73,0}},
/* 17993 */ {(12<<2)|2,{115,73,0}},
/* 17994 */ {(12<<2)|2,{115,73,0}},
/* 17995 */ {(12<<2)|2,{115,73,0}},
/* 17996 */ {(12<<2)|2,{115,73,0}},
/* 17997 */ {(12<<2)|2,{115,73,0}},
/* 17998 */ {(12<<2)|2,{115,73,0}},
/* 17999 */ {(12<<2)|2,{115,73,0}},
/* 18000 */ {(12<<2)|2,{115,74,0}},
/* 18001 */ {(12<<2)|2,{115,74,0}},
/* 18002 */ {(12<<2)|2,{115,74,0}},
/* 18003 */ {(12<<2)|2,{115,74,0}},
/* 18004 */ {(12<<2)|2,{115,74,0}},
/* 18005 */ {(12<<2)|2,{115,74,0}},
/* 18006 */ {(12<<2)|2,{115,74,0}},
/* 18007 */ {(12<<2)|2,{115,74,0}},
/* 18008 */ {(12<<2)|2,{115,74,0}},
/* 18009 */ {(12<<2)|2,{115,74,0}},
/* 18010 */ {(12<<2)|2,{115,74,0}},
/* 18011 */ {(12<<2)|2,{115,74,0}},
/* 18012 */ {(12<<2)|2,{115,74,0}},
/* 18013 */ {(12<<2)|2,{115,74,0}},
/* 18014 */ {(12<<2)|2,{115,74,0}},
/* 18015 */ {(12<<2)|2,{115,74,0}},
/* 18016 */ {(12<<2)|2,{115,75,0}},
/* 18017 */ {(12<<2)|2,{115,75,0}},
/* 18018 */ {(12<<2)|2,{115,75,0}},
/* 18019 */ {(12<<2)|2,{115,75,0}},
/* 18020 */ {(12<<2)|2,{115,75,0}},
/* 18021 */ {(12<<2)|2,{115,75,0}},
/* 18022 */ {(12<<2)|2,{115,75,0}},
/* 18023 */ {(12<<2)|2,{115,75,0}},
/* 18024 */ {(12<<2)|2,{115,75,0}},
/* 18025 */ {(12<<2)|2,{115,75,0}},
/* 18026 */ {(12<<2)|2,{115,75,0}},
/* 18027 */ {(12<<2)|2,{115,75,0}},
/* 18028 */ {(12<<2)|2,{115,75,0}},
/* 18029 */ {(12<<2)|2,{115,75,0}},
/* 18030 */ {(12<<2)|2,{115,75,0}},
/* 18031 */ {(12<<2)|2,{115,75,0}},
/* 18032 */ {(12<<2)|2,{115,76,0}},
/* 18033 */ {(12<<2)|2,{115,76,0}},
/* 18034 */ {(12<<2)|2,{115,76,0}},
/* 18035 */ {(12<<2)|2,{115,76,0}},
/* 18036 */ {(12<<2)|2,{115,76,0}},
/* 18037 */ {(12<<2)|2,{115,76,0}},
/* 18038 */ {(12<<2)|2,{115,76,0}},
/* 18039 */ {(12<<2)|2,{115,76,0}},
/* 18040 */ {(12<<2)|2,{115,76,0}},
/* 18041 */ {(12<<2)|2,{115,76,0}},
/* 18042 */ {(12<<2)|2,{115,76,0}},
/* 18043 */ {(12<<2)|2,{115,76,0}},
/* 18044 */ {(12<<2)|2,{115,76,0}},
/* 18045 */ {(12<<2)|2,{115,76,0}},
/* 18046 */ {(12<<2)|2,{115,76,0}},
/* 18047 */ {(12<<2)|2,{115,76,0}},
/* 18048 */ {(12<<2)|2,{115,77,0}},
/* 18049 */ {(12<<2)|2,{115,77,0}},
/* 18050 */ {(12<<2)|2,{115,77,0}},
/* 18051 */ {(12<<2)|2,{115,77,0}},
/* 18052 */ {(12<<2)|2,{115,77,0}},
/* 18053 */ {(12<<2)|2,{115,77,0}},
/* 18054 */ {(12<<2)|2,{115,77,0}},
/* 18055 */ {(12<<2)|2,{115,77,0}},
/* 18056 */ {(12<<2)|2,{115,77,0}},
/* 18057 */ {(12<<2)|2,{115,77,0}},
/* 18058 */ {(12<<2)|2,{115,77,0}},
/* 18059 */ {(12<<2)|2,{115,77,0}},
/* 18060 */ {(12<<2)|2,{115,77,0}},
/* 18061 */ {(12<<2)|2,{115,77,0}},
/* 18062 */ {(12<<2)|2,{115,77,0}},
/* 18063 */ {(12<<2)|2,{115,77,0}},
/* 18064 */ {(12<<2)|2,{115,78,0}},
/* 18065 */ {(12<<2)|2,{115,78,0}},
/* 18066 */ {(12<<2)|2,{115,78,0}},
/* 18067 */ {(12<<2)|2,{115,78,0}},
/* 18068 */ {(12<<2)|2,{115,78,0}},
/* 18069 */ {(12<<2)|2,{115,78,0}},
/* 18070 */ {(12<<2)|2,{115,78,0}},
/* 18071 */ {(12<<2)|2,{115,78,0}},
/* 18072 */ {(12<<2)|2,{115,78,0}},
/* 18073 */ {(12<<2)|2,{115,78,0}},
/* 18074 */ {(12<<2)|2,{115,78,0}},
/* 18075 */ {(12<<2)|2,{115,78,0}},
/* 18076 */ {(12<<2)|2,{115,78,0}},
/* 18077 */ {(12<<2)|2,{115,78,0}},
/* 18078 */ {(12<<2)|2,{115,78,0}},
/* 18079 */ {(12<<2)|2,{115,78,0}},
/* 18080 */ {(12<<2)|2,{115,79,0}},
/* 18081 */ {(12<<2)|2,{115,79,0}},
/* 18082 */ {(12<<2)|2,{115,79,0}},
/* 18083 */ {(12<<2)|2,{115,79,0}},
/* 18084 */ {(12<<2)|2,{115,79,0}},
/* 18085 */ {(12<<2)|2,{115,79,0}},
/* 18086 */ {(12<<2)|2,{115,79,0}},
/* 18087 */ {(12<<2)|2,{115,79,0}},
/* 18088 */ {(12<<2)|2,{115,79,0}},
/* 18089 */ {(12<<2)|2,{115,79,0}},
/* 18090 */ {(12<<2)|2,{115,79,0}},
/* 18091 */ {(12<<2)|2,{115,79,0}},
/* 18092 */ {(12<<2)|2,{115,79,0}},
/* 18093 */ {(12<<2)|2,{115,79,0}},
/* 18094 */ {(12<<2)|2,{115,79,0}},
/* 18095 */ {(12<<2)|2,{115,79,0}},
/* 18096 */ {(12<<2)|2,{115,80,0}},
/* 18097 */ {(12<<2)|2,{115,80,0}},
/* 18098 */ {(12<<2)|2,{115,80,0}},
/* 18099 */ {(12<<2)|2,{115,80,0}},
/* 18100 */ {(12<<2)|2,{115,80,0}},
/* 18101 */ {(12<<2)|2,{115,80,0}},
/* 18102 */ {(12<<2)|2,{115,80,0}},
/* 18103 */ {(12<<2)|2,{115,80,0}},
/* 18104 */ {(12<<2)|2,{115,80,0}},
/* 18105 */ {(12<<2)|2,{115,80,0}},
/* 18106 */ {(12<<2)|2,{115,80,0}},
/* 18107 */ {(12<<2)|2,{115,80,0}},
/* 18108 */ {(12<<2)|2,{115,80,0}},
/* 18109 */ {(12<<2)|2,{115,80,0}},
/* 18110 */ {(12<<2)|2,{115,80,0}},
/* 18111 */ {(12<<2)|2,{115,80,0}},
/* 18112 */ {(12<<2)|2,{115,81,0}},
/* 18113 */ {(12<<2)|2,{115,81,0}},
/* 18114 */ {(12<<2)|2,{115,81,0}},
/* 18115 */ {(12<<2)|2,{115,81,0}},
/* 18116 */ {(12<<2)|2,{115,81,0}},
/* 18117 */ {(12<<2)|2,{115,81,0}},
/* 18118 */ {(12<<2)|2,{115,81,0}},
/* 18119 */ {(12<<2)|2,{115,81,0}},
/* 18120 */ {(12<<2)|2,{115,81,0}},
/* 18121 */ {(12<<2)|2,{115,81,0}},
/* 18122 */ {(12<<2)|2,{115,81,0}},
/* 18123 */ {(12<<2)|2,{115,81,0}},
/* 18124 */ {(12<<2)|2,{115,81,0}},
/* 18125 */ {(12<<2)|2,{115,81,0}},
/* 18126 */ {(12<<2)|2,{115,81,0}},
/* 18127 */ {(12<<2)|2,{115,81,0}},
/* 18128 */ {(12<<2)|2,{115,82,0}},
/* 18129 */ {(12<<2)|2,{115,82,0}},
/* 18130 */ {(12<<2)|2,{115,82,0}},
/* 18131 */ {(12<<2)|2,{115,82,0}},
/* 18132 */ {(12<<2)|2,{115,82,0}},
/* 18133 */ {(12<<2)|2,{115,82,0}},
/* 18134 */ {(12<<2)|2,{115,82,0}},
/* 18135 */ {(12<<2)|2,{115,82,0}},
/* 18136 */ {(12<<2)|2,{115,82,0}},
/* 18137 */ {(12<<2)|2,{115,82,0}},
/* 18138 */ {(12<<2)|2,{115,82,0}},
/* 18139 */ {(12<<2)|2,{115,82,0}},
/* 18140 */ {(12<<2)|2,{115,82,0}},
/* 18141 */ {(12<<2)|2,{115,82,0}},
/* 18142 */ {(12<<2)|2,{115,82,0}},
/* 18143 */ {(12<<2)|2,{115,82,0}},
/* 18144 */ {(12<<2)|2,{115,83,0}},
/* 18145 */ {(12<<2)|2,{115,83,0}},
/* 18146 */ {(12<<2)|2,{115,83,0}},
/* 18147 */ {(12<<2)|2,{115,83,0}},
/* 18148 */ {(12<<2)|2,{115,83,0}},
/* 18149 */ {(12<<2)|2,{115,83,0}},
/* 18150 */ {(12<<2)|2,{115,83,0}},
/* 18151 */ {(12<<2)|2,{115,83,0}},
/* 18152 */ {(12<<2)|2,{115,83,0}},
/* 18153 */ {(12<<2)|2,{115,83,0}},
/* 18154 */ {(12<<2)|2,{115,83,0}},
/* 18155 */ {(12<<2)|2,{115,83,0}},
/* 18156 */ {(12<<2)|2,{115,83,0}},
/* 18157 */ {(12<<2)|2,{115,83,0}},
/* 18158 */ {(12<<2)|2,{115,83,0}},
/* 18159 */ {(12<<2)|2,{115,83,0}},
/* 18160 */ {(12<<2)|2,{115,84,0}},
/* 18161 */ {(12<<2)|2,{115,84,0}},
/* 18162 */ {(12<<2)|2,{115,84,0}},
/* 18163 */ {(12<<2)|2,{115,84,0}},
/* 18164 */ {(12<<2)|2,{115,84,0}},
/* 18165 */ {(12<<2)|2,{115,84,0}},
/* 18166 */ {(12<<2)|2,{115,84,0}},
/* 18167 */ {(12<<2)|2,{115,84,0}},
/* 18168 */ {(12<<2)|2,{115,84,0}},
/* 18169 */ {(12<<2)|2,{115,84,0}},
/* 18170 */ {(12<<2)|2,{115,84,0}},
/* 18171 */ {(12<<2)|2,{115,84,0}},
/* 18172 */ {(12<<2)|2,{115,84,0}},
/* 18173 */ {(12<<2)|2,{115,84,0}},
/* 18174 */ {(12<<2)|2,{115,84,0}},
/* 18175 */ {(12<<2)|2,{115,84,0}},
/* 18176 */ {(12<<2)|2,{115,85,0}},
/* 18177 */ {(12<<2)|2,{115,85,0}},
/* 18178 */ {(12<<2)|2,{115,85,0}},
/* 18179 */ {(12<<2)|2,{115,85,0}},
/* 18180 */ {(12<<2)|2,{115,85,0}},
/* 18181 */ {(12<<2)|2,{115,85,0}},
/* 18182 */ {(12<<2)|2,{115,85,0}},
/* 18183 */ {(12<<2)|2,{115,85,0}},
/* 18184 */ {(12<<2)|2,{115,85,0}},
/* 18185 */ {(12<<2)|2,{115,85,0}},
/* 18186 */ {(12<<2)|2,{115,85,0}},
/* 18187 */ {(12<<2)|2,{115,85,0}},
/* 18188 */ {(12<<2)|2,{115,85,0}},
/* 18189 */ {(12<<2)|2,{115,85,0}},
/* 18190 */ {(12<<2)|2,{115,85,0}},
/* 18191 */ {(12<<2)|2,{115,85,0}},
/* 18192 */ {(12<<2)|2,{115,86,0}},
/* 18193 */ {(12<<2)|2,{115,86,0}},
/* 18194 */ {(12<<2)|2,{115,86,0}},
/* 18195 */ {(12<<2)|2,{115,86,0}},
/* 18196 */ {(12<<2)|2,{115,86,0}},
/* 18197 */ {(12<<2)|2,{115,86,0}},
/* 18198 */ {(12<<2)|2,{115,86,0}},
/* 18199 */ {(12<<2)|2,{115,86,0}},
/* 18200 */ {(12<<2)|2,{115,86,0}},
/* 18201 */ {(12<<2)|2,{115,86,0}},
/* 18202 */ {(12<<2)|2,{115,86,0}},
/* 18203 */ {(12<<2)|2,{115,86,0}},
/* 18204 */ {(12<<2)|2,{115,86,0}},
/* 18205 */ {(12<<2)|2,{115,86,0}},
/* 18206 */ {(12<<2)|2,{115,86,0}},
/* 18207 */ {(12<<2)|2,{115,86,0}},
/* 18208 */ {(12<<2)|2,{115,87,0}},
/* 18209 */ {(12<<2)|2,{115,87,0}},
/* 18210 */ {(12<<2)|2,{115,87,0}},
/* 18211 */ {(12<<2)|2,{115,87,0}},
/* 18212 */ {(12<<2)|2,{115,87,0}},
/* 18213 */ {(12<<2)|2,{115,87,0}},
/* 18214 */ {(12<<2)|2,{115,87,0}},
/* 18215 */ {(12<<2)|2,{115,87,0}},
/* 18216 */ {(12<<2)|2,{115,87,0}},
/* 18217 */ {(12<<2)|2,{115,87,0}},
/* 18218 */ {(12<<2)|2,{115,87,0}},
/* 18219 */ {(12<<2)|2,{115,87,0}},
/* 18220 */ {(12<<2)|2,{115,87,0}},
/* 18221 */ {(12<<2)|2,{115,87,0}},
/* 18222 */ {(12<<2)|2,{115,87,0}},
/* 18223 */ {(12<<2)|2,{115,87,0}},
/* 18224 */ {(12<<2)|2,{115,89,0}},
/* 18225 */ {(12<<2)|2,{115,89,0}},
/* 18226 */ {(12<<2)|2,{115,89,0}},
/* 18227 */ {(12<<2)|2,{115,89,0}},
/* 18228 */ {(12<<2)|2,{115,89,0}},
/* 18229 */ {(12<<2)|2,{115,89,0}},
/* 18230 */ {(12<<2)|2,{115,89,0}},
/* 18231 */ {(12<<2)|2,{115,89,0}},
/* 18232 */ {(12<<2)|2,{115,89,0}},
/* 18233 */ {(12<<2)|2,{115,89,0}},
/* 18234 */ {(12<<2)|2,{115,89,0}},
/* 18235 */ {(12<<2)|2,{115,89,0}},
/* 18236 */ {(12<<2)|2,{115,89,0}},
/* 18237 */ {(12<<2)|2,{115,89,0}},
/* 18238 */ {(12<<2)|2,{115,89,0}},
/* 18239 */ {(12<<2)|2,{115,89,0}},
/* 18240 */ {(12<<2)|2,{115,106,0}},
/* 18241 */ {(12<<2)|2,{115,106,0}},
/* 18242 */ {(12<<2)|2,{115,106,0}},
/* 18243 */ {(12<<2)|2,{115,106,0}},
/* 18244 */ {(12<<2)|2,{115,106,0}},
/* 18245 */ {(12<<2)|2,{115,106,0}},
/* 18246 */ {(12<<2)|2,{115,106,0}},
/* 18247 */ {(12<<2)|2,{115,106,0}},
/* 18248 */ {(12<<2)|2,{115,106,0}},
/* 18249 */ {(12<<2)|2,{115,106,0}},
/* 18250 */ {(12<<2)|2,{115,106,0}},
/* 18251 */ {(12<<2)|2,{115,106,0}},
/* 18252 */ {(12<<2)|2,{115,106,0}},
/* 18253 */ {(12<<2)|2,{115,106,0}},
/* 18254 */ {(12<<2)|2,{115,106,0}},
/* 18255 */ {(12<<2)|2,{115,106,0}},
/* 18256 */ {(12<<2)|2,{115,107,0}},
/* 18257 */ {(12<<2)|2,{115,107,0}},
/* 18258 */ {(12<<2)|2,{115,107,0}},
/* 18259 */ {(12<<2)|2,{115,107,0}},
/* 18260 */ {(12<<2)|2,{115,107,0}},
/* 18261 */ {(12<<2)|2,{115,107,0}},
/* 18262 */ {(12<<2)|2,{115,107,0}},
/* 18263 */ {(12<<2)|2,{115,107,0}},
/* 18264 */ {(12<<2)|2,{115,107,0}},
/* 18265 */ {(12<<2)|2,{115,107,0}},
/* 18266 */ {(12<<2)|2,{115,107,0}},
/* 18267 */ {(12<<2)|2,{115,107,0}},
/* 18268 */ {(12<<2)|2,{115,107,0}},
/* 18269 */ {(12<<2)|2,{115,107,0}},
/* 18270 */ {(12<<2)|2,{115,107,0}},
/* 18271 */ {(12<<2)|2,{115,107,0}},
/* 18272 */ {(12<<2)|2,{115,113,0}},
/* 18273 */ {(12<<2)|2,{115,113,0}},
/* 18274 */ {(12<<2)|2,{115,113,0}},
/* 18275 */ {(12<<2)|2,{115,113,0}},
/* 18276 */ {(12<<2)|2,{115,113,0}},
/* 18277 */ {(12<<2)|2,{115,113,0}},
/* 18278 */ {(12<<2)|2,{115,113,0}},
/* 18279 */ {(12<<2)|2,{115,113,0}},
/* 18280 */ {(12<<2)|2,{115,113,0}},
/* 18281 */ {(12<<2)|2,{115,113,0}},
/* 18282 */ {(12<<2)|2,{115,113,0}},
/* 18283 */ {(12<<2)|2,{115,113,0}},
/* 18284 */ {(12<<2)|2,{115,113,0}},
/* 18285 */ {(12<<2)|2,{115,113,0}},
/* 18286 */ {(12<<2)|2,{115,113,0}},
/* 18287 */ {(12<<2)|2,{115,113,0}},
/* 18288 */ {(12<<2)|2,{115,118,0}},
/* 18289 */ {(12<<2)|2,{115,118,0}},
/* 18290 */ {(12<<2)|2,{115,118,0}},
/* 18291 */ {(12<<2)|2,{115,118,0}},
/* 18292 */ {(12<<2)|2,{115,118,0}},
/* 18293 */ {(12<<2)|2,{115,118,0}},
/* 18294 */ {(12<<2)|2,{115,118,0}},
/* 18295 */ {(12<<2)|2,{115,118,0}},
/* 18296 */ {(12<<2)|2,{115,118,0}},
/* 18297 */ {(12<<2)|2,{115,118,0}},
/* 18298 */ {(12<<2)|2,{115,118,0}},
/* 18299 */ {(12<<2)|2,{115,118,0}},
/* 18300 */ {(12<<2)|2,{115,118,0}},
/* 18301 */ {(12<<2)|2,{115,118,0}},
/* 18302 */ {(12<<2)|2,{115,118,0}},
/* 18303 */ {(12<<2)|2,{115,118,0}},
/* 18304 */ {(12<<2)|2,{115,119,0}},
/* 18305 */ {(12<<2)|2,{115,119,0}},
/* 18306 */ {(12<<2)|2,{115,119,0}},
/* 18307 */ {(12<<2)|2,{115,119,0}},
/* 18308 */ {(12<<2)|2,{115,119,0}},
/* 18309 */ {(12<<2)|2,{115,119,0}},
/* 18310 */ {(12<<2)|2,{115,119,0}},
/* 18311 */ {(12<<2)|2,{115,119,0}},
/* 18312 */ {(12<<2)|2,{115,119,0}},
/* 18313 */ {(12<<2)|2,{115,119,0}},
/* 18314 */ {(12<<2)|2,{115,119,0}},
/* 18315 */ {(12<<2)|2,{115,119,0}},
/* 18316 */ {(12<<2)|2,{115,119,0}},
/* 18317 */ {(12<<2)|2,{115,119,0}},
/* 18318 */ {(12<<2)|2,{115,119,0}},
/* 18319 */ {(12<<2)|2,{115,119,0}},
/* 18320 */ {(12<<2)|2,{115,120,0}},
/* 18321 */ {(12<<2)|2,{115,120,0}},
/* 18322 */ {(12<<2)|2,{115,120,0}},
/* 18323 */ {(12<<2)|2,{115,120,0}},
/* 18324 */ {(12<<2)|2,{115,120,0}},
/* 18325 */ {(12<<2)|2,{115,120,0}},
/* 18326 */ {(12<<2)|2,{115,120,0}},
/* 18327 */ {(12<<2)|2,{115,120,0}},
/* 18328 */ {(12<<2)|2,{115,120,0}},
/* 18329 */ {(12<<2)|2,{115,120,0}},
/* 18330 */ {(12<<2)|2,{115,120,0}},
/* 18331 */ {(12<<2)|2,{115,120,0}},
/* 18332 */ {(12<<2)|2,{115,120,0}},
/* 18333 */ {(12<<2)|2,{115,120,0}},
/* 18334 */ {(12<<2)|2,{115,120,0}},
/* 18335 */ {(12<<2)|2,{115,120,0}},
/* 18336 */ {(12<<2)|2,{115,121,0}},
/* 18337 */ {(12<<2)|2,{115,121,0}},
/* 18338 */ {(12<<2)|2,{115,121,0}},
/* 18339 */ {(12<<2)|2,{115,121,0}},
/* 18340 */ {(12<<2)|2,{115,121,0}},
/* 18341 */ {(12<<2)|2,{115,121,0}},
/* 18342 */ {(12<<2)|2,{115,121,0}},
/* 18343 */ {(12<<2)|2,{115,121,0}},
/* 18344 */ {(12<<2)|2,{115,121,0}},
/* 18345 */ {(12<<2)|2,{115,121,0}},
/* 18346 */ {(12<<2)|2,{115,121,0}},
/* 18347 */ {(12<<2)|2,{115,121,0}},
/* 18348 */ {(12<<2)|2,{115,121,0}},
/* 18349 */ {(12<<2)|2,{115,121,0}},
/* 18350 */ {(12<<2)|2,{115,121,0}},
/* 18351 */ {(12<<2)|2,{115,121,0}},
/* 18352 */ {(12<<2)|2,{115,122,0}},
/* 18353 */ {(12<<2)|2,{115,122,0}},
/* 18354 */ {(12<<2)|2,{115,122,0}},
/* 18355 */ {(12<<2)|2,{115,122,0}},
/* 18356 */ {(12<<2)|2,{115,122,0}},
/* 18357 */ {(12<<2)|2,{115,122,0}},
/* 18358 */ {(12<<2)|2,{115,122,0}},
/* 18359 */ {(12<<2)|2,{115,122,0}},
/* 18360 */ {(12<<2)|2,{115,122,0}},
/* 18361 */ {(12<<2)|2,{115,122,0}},
/* 18362 */ {(12<<2)|2,{115,122,0}},
/* 18363 */ {(12<<2)|2,{115,122,0}},
/* 18364 */ {(12<<2)|2,{115,122,0}},
/* 18365 */ {(12<<2)|2,{115,122,0}},
/* 18366 */ {(12<<2)|2,{115,122,0}},
/* 18367 */ {(12<<2)|2,{115,122,0}},
/* 18368 */ {(13<<2)|2,{115,38,0}},
/* 18369 */ {(13<<2)|2,{115,38,0}},
/* 18370 */ {(13<<2)|2,{115,38,0}},
/* 18371 */ {(13<<2)|2,{115,38,0}},
/* 18372 */ {(13<<2)|2,{115,38,0}},
/* 18373 */ {(13<<2)|2,{115,38,0}},
/* 18374 */ {(13<<2)|2,{115,38,0}},
/* 18375 */ {(13<<2)|2,{115,38,0}},
/* 18376 */ {(13<<2)|2,{115,42,0}},
/* 18377 */ {(13<<2)|2,{115,42,0}},
/* 18378 */ {(13<<2)|2,{115,42,0}},
/* 18379 */ {(13<<2)|2,{115,42,0}},
/* 18380 */ {(13<<2)|2,{115,42,0}},
/* 18381 */ {(13<<2)|2,{115,42,0}},
/* 18382 */ {(13<<2)|2,{115,42,0}},
/* 18383 */ {(13<<2)|2,{115,42,0}},
/* 18384 */ {(13<<2)|2,{115,44,0}},
/* 18385 */ {(13<<2)|2,{115,44,0}},
/* 18386 */ {(13<<2)|2,{115,44,0}},
/* 18387 */ {(13<<2)|2,{115,44,0}},
/* 18388 */ {(13<<2)|2,{115,44,0}},
/* 18389 */ {(13<<2)|2,{115,44,0}},
/* 18390 */ {(13<<2)|2,{115,44,0}},
/* 18391 */ {(13<<2)|2,{115,44,0}},
/* 18392 */ {(13<<2)|2,{115,59,0}},
/* 18393 */ {(13<<2)|2,{115,59,0}},
/* 18394 */ {(13<<2)|2,{115,59,0}},
/* 18395 */ {(13<<2)|2,{115,59,0}},
/* 18396 */ {(13<<2)|2,{115,59,0}},
/* 18397 */ {(13<<2)|2,{115,59,0}},
/* 18398 */ {(13<<2)|2,{115,59,0}},
/* 18399 */ {(13<<2)|2,{115,59,0}},
/* 18400 */ {(13<<2)|2,{115,88,0}},
/* 18401 */ {(13<<2)|2,{115,88,0}},
/* 18402 */ {(13<<2)|2,{115,88,0}},
/* 18403 */ {(13<<2)|2,{115,88,0}},
/* 18404 */ {(13<<2)|2,{115,88,0}},
/* 18405 */ {(13<<2)|2,{115,88,0}},
/* 18406 */ {(13<<2)|2,{115,88,0}},
/* 18407 */ {(13<<2)|2,{115,88,0}},
/* 18408 */ {(13<<2)|2,{115,90,0}},
/* 18409 */ {(13<<2)|2,{115,90,0}},
/* 18410 */ {(13<<2)|2,{115,90,0}},
/* 18411 */ {(13<<2)|2,{115,90,0}},
/* 18412 */ {(13<<2)|2,{115,90,0}},
/* 18413 */ {(13<<2)|2,{115,90,0}},
/* 18414 */ {(13<<2)|2,{115,90,0}},
/* 18415 */ {(13<<2)|2,{115,90,0}},
/* 18416 */ {(15<<2)|2,{115,33,0}},
/* 18417 */ {(15<<2)|2,{115,33,0}},
/* 18418 */ {(15<<2)|2,{115,34,0}},
/* 18419 */ {(15<<2)|2,{115,34,0}},
/* 18420 */ {(15<<2)|2,{115,40,0}},
/* 18421 */ {(15<<2)|2,{115,40,0}},
/* 18422 */ {(15<<2)|2,{115,41,0}},
/* 18423 */ {(15<<2)|2,{115,41,0}},
/* 18424 */ {(15<<2)|2,{115,63,0}},
/* 18425 */ {(15<<2)|2,{115,63,0}},
/* 18426 */ {(16<<2)|2,{115,39,0}},
/* 18427 */ {(16<<2)|2,{115,43,0}},
/* 18428 */ {(16<<2)|2,{115,124,0}},
/* 18429 */ {(5<<2)|1,{115,0,0}},
/* 18430 */ {(5<<2)|1,{115,0,0}},
/* 18431 */ {(5<<2)|1,{115,0,0}},
/* 18432 */ {(15<<2)|3,{116,48,48}},
/* 18433 */ {(15<<2)|3,{116,48,48}},
/* 18434 */ {(15<<2)|3,{116,48,49}},
/* 18435 */ {(15<<2)|3,{116,48,49}},
/* 18436 */ {(15<<2)|3,{116,48,50}},
/* 18437 */ {(15<<2)|3,{116,48,50}},
/* 18438 */ {(15<<2)|3,{116,48,97}},
/* 18439 */ {(15<<2)|3,{116,48,97}},
/* 18440 */ {(15<<2)|3,{116,48,99}},
/* 18441 */ {(15<<2)|3,{116,48,99}},
/* 18442 */ {(15<<2)|3,{116,48,101}},
/* 18443 */ {(15<<2)|3,{116,48,101}},
/* 18444 */ {(15<<2)|3,{116,48,105}},
/* 18445 */ {(15<<2)|3,{116,48,105}},
/* 18446 */ {(15<<2)|3,{116,48,111}},
/* 18447 */ {(15<<2)|3,{116,48,111}},
/* 18448 */ {(15<<2)|3,{116,48,115}},
/* 18449 */ {(15<<2)|3,{116,48,115}},
/* 18450 */ {(15<<2)|3,{116,48,116}},
/* 18451 */ {(15<<2)|3,{116,48,116}},
/* 18452 */ {(16<<2)|3,{116,48,32}},
/* 18453 */ {(16<<2)|3,{116,48,37}},
/* 18454 */ {(16<<2)|3,{116,48,45}},
/* 18455 */ {(16<<2)|3,{116,48,46}},
/* 18456 */ {(16<<2)|3,{116,48,47}},
/* 18457 */ {(16<<2)|3,{116,48,51}},
/* 18458 */ {(16<<2)|3,{116,48,52}},
/* 18459 */ {(16<<2)|3,{116,48,53}},
/* 18460 */ {(16<<2)|3,{116,48,54}},
/* 18461 */ {(16<<2)|3,{116,48,55}},
/* 18462 */ {(16<<2)|3,{116,48,56}},
/* 18463 */ {(16<<2)|3,{116,48,57}},
/* 18464 */ {(16<<2)|3,{116,48,61}},
/* 18465 */ {(16<<2)|3,{116,48,65}},
/* 18466 */ {(16<<2)|3,{116,48,95}},
/* 18467 */ {(16<<2)|3,{116,48,98}},
/* 18468 */ {(16<<2)|3,{116,48,100}},
/* 18469 */ {(16<<2)|3,{116,48,102}},
/* 18470 */ {(16<<2)|3,{116,48,103}},
/* 18471 */ {(16<<2)|3,{116,48,104}},
/* 18472 */ {(16<<2)|3,{116,48,108}},
/* 18473 */ {(16<<2)|3,{116,48,109}},
/* 18474 */ {(16<<2)|3,{116,48,110}},
/* 18475 */ {(16<<2)|3,{116,48,112}},
/* 18476 */ {(16<<2)|3,{116,48,114}},
/* 18477 */ {(16<<2)|3,{116,48,117}},
/* 18478 */ {(10<<2)|2,{116,48,0}},
/* 18479 */ {(10<<2)|2,{116,48,0}},
/* 18480 */ {(10<<2)|2,{116,48,0}},
/* 18481 */ {(10<<2)|2,{116,48,0}},
/* 18482 */ {(10<<2)|2,{116,48,0}},
/* 18483 */ {(10<<2)|2,{116,48,0}},
/* 18484 */ {(10<<2)|2,{116,48,0}},
/* 18485 */ {(10<<2)|2,{116,48,0}},
/* 18486 */ {(10<<2)|2,{116,48,0}},
/* 18487 */ {(10<<2)|2,{116,48,0}},
/* 18488 */ {(10<<2)|2,{116,48,0}},
/* 18489 */ {(10<<2)|2,{116,48,0}},
/* 18490 */ {(10<<2)|2,{116,48,0}},
/* 18491 */ {(10<<2)|2,{116,48,0}},
/* 18492 */ {(10<<2)|2,{116,48,0}},
/* 18493 */ {(10<<2)|2,{116,48,0}},
/* 18494 */ {(10<<2)|2,{116,48,0}},
/* 18495 */ {(10<<2)|2,{116,48,0}},
/* 18496 */ {(15<<2)|3,{116,49,48}},
/* 18497 */ {(15<<2)|3,{116,49,48}},
/* 18498 */ {(15<<2)|3,{116,49,49}},
/* 18499 */ {(15<<2)|3,{116,49,49}},
/* 18500 */ {(15<<2)|3,{116,49,50}},
/* 18501 */ {(15<<2)|3,{116,49,50}},
/* 18502 */ {(15<<2)|3,{116,49,97}},
/* 18503 */ {(15<<2)|3,{116,49,97}},
/* 18504 */ {(15<<2)|3,{116,49,99}},
/* 18505 */ {(15<<2)|3,{116,49,99}},
/* 18506 */ {(15<<2)|3,{116,49,101}},
/* 18507 */ {(15<<2)|3,{116,49,101}},
/* 18508 */ {(15<<2)|3,{116,49,105}},
/* 18509 */ {(15<<2)|3,{116,49,105}},
/* 18510 */ {(15<<2)|3,{116,49,111}},
/* 18511 */ {(15<<2)|3,{116,49,111}},
/* 18512 */ {(15<<2)|3,{116,49,115}},
/* 18513 */ {(15<<2)|3,{116,49,115}},
/* 18514 */ {(15<<2)|3,{116,49,116}},
/* 18515 */ {(15<<2)|3,{116,49,116}},
/* 18516 */ {(16<<2)|3,{116,49,32}},
/* 18517 */ {(16<<2)|3,{116,49,37}},
/* 18518 */ {(16<<2)|3,{116,49,45}},
/* 18519 */ {(16<<2)|3,{116,49,46}},
/* 18520 */ {(16<<2)|3,{116,49,47}},
/* 18521 */ {(16<<2)|3,{116,49,51}},
/* 18522 */ {(16<<2)|3,{116,49,52}},
/* 18523 */ {(16<<2)|3,{116,49,53}},
/* 18524 */ {(16<<2)|3,{116,49,54}},
/* 18525 */ {(16<<2)|3,{116,49,55}},
/* 18526 */ {(16<<2)|3,{116,49,56}},
/* 18527 */ {(16<<2)|3,{116,49,57}},
/* 18528 */ {(16<<2)|3,{116,49,61}},
/* 18529 */ {(16<<2)|3,{116,49,65}},
/* 18530 */ {(16<<2)|3,{116,49,95}},
/* 18531 */ {(16<<2)|3,{116,49,98}},
/* 18532 */ {(16<<2)|3,{116,49,100}},
/* 18533 */ {(16<<2)|3,{116,49,102}},
/* 18534 */ {(16<<2)|3,{116,49,103}},
/* 18535 */ {(16<<2)|3,{116,49,104}},
/* 18536 */ {(16<<2)|3,{116,49,108}},
/* 18537 */ {(16<<2)|3,{116,49,109}},
/* 18538 */ {(16<<2)|3,{116,49,110}},
/* 18539 */ {(16<<2)|3,{116,49,112}},
/* 18540 */ {(16<<2)|3,{116,49,114}},
/* 18541 */ {(16<<2)|3,{116,49,117}},
/* 18542 */ {(10<<2)|2,{116,49,0}},
/* 18543 */ {(10<<2)|2,{116,49,0}},
/* 18544 */ {(10<<2)|2,{116,49,0}},
/* 18545 */ {(10<<2)|2,{116,49,0}},
/* 18546 */ {(10<<2)|2,{116,49,0}},
/* 18547 */ {(10<<2)|2,{116,49,0}},
/* 18548 */ {(10<<2)|2,{116,49,0}},
/* 18549 */ {(10<<2)|2,{116,49,0}},
/* 18550 */ {(10<<2)|2,{116,49,0}},
/* 18551 */ {(10<<2)|2,{116,49,0}},
/* 18552 */ {(10<<2)|2,{116,49,0}},
/* 18553 */ {(10<<2)|2,{116,49,0}},
/* 18554 */ {(10<<2)|2,{116,49,0}},
/* 18555 */ {(10<<2)|2,{116,49,0}},
/* 18556 */ {(10<<2)|2,{116,49,0}},
/* 18557 */ {(10<<2)|2,{116,49,0}},
/* 18558 */ {(10<<2)|2,{116,49,0}},
/* 18559 */ {(10<<2)|2,{116,49,0}},
/* 18560 */ {(15<<2)|3,{116,50,48}},
/* 18561 */ {(15<<2)|3,{116,50,48}},
/* 18562 */ {(15<<2)|3,{116,50,49}},
/* 18563 */ {(15<<2)|3,{116,50,49}},
/* 18564 */ {(15<<2)|3,{116,50,50}},
/* 18565 */ {(15<<2)|3,{116,50,50}},
/* 18566 */ {(15<<2)|3,{116,50,97}},
/* 18567 */ {(15<<2)|3,{116,50,97}},
/* 18568 */ {(15<<2)|3,{116,50,99}},
/* 18569 */ {(15<<2)|3,{116,50,99}},
/* 18570 */ {(15<<2)|3,{116,50,101}},
/* 18571 */ {(15<<2)|3,{116,50,101}},
/* 18572 */ {(15<<2)|3,{116,50,105}},
/* 18573 */ {(15<<2)|3,{116,50,105}},
/* 18574 */ {(15<<2)|3,{116,50,111}},
/* 18575 */ {(15<<2)|3,{116,50,111}},
/* 18576 */ {(15<<2)|3,{116,50,115}},
/* 18577 */ {(15<<2)|3,{116,50,115}},
/* 18578 */ {(15<<2)|3,{116,50,116}},
/* 18579 */ {(15<<2)|3,{116,50,116}},
/* 18580 */ {(16<<2)|3,{116,50,32}},
/* 18581 */ {(16<<2)|3,{116,50,37}},
/* 18582 */ {(16<<2)|3,{116,50,45}},
/* 18583 */ {(16<<2)|3,{116,50,46}},
/* 18584 */ {(16<<2)|3,{116,50,47}},
/* 18585 */ {(16<<2)|3,{116,50,51}},
/* 18586 */ {(16<<2)|3,{116,50,52}},
/* 18587 */ {(16<<2)|3,{116,50,53}},
/* 18588 */ {(16<<2)|3,{116,50,54}},
/* 18589 */ {(16<<2)|3,{116,50,55}},
/* 18590 */ {(16<<2)|3,{116,50,56}},
/* 18591 */ {(16<<2)|3,{116,50,57}},
/* 18592 */ {(16<<2)|3,{116,50,61}},
/* 18593 */ {(16<<2)|3,{116,50,65}},
/* 18594 */ {(16<<2)|3,{116,50,95}},
/* 18595 */ {(16<<2)|3,{116,50,98}},
/* 18596 */ {(16<<2)|3,{116,50,100}},
/* 18597 */ {(16<<2)|3,{116,50,102}},
/* 18598 */ {(16<<2)|3,{116,50,103}},
/* 18599 */ {(16<<2)|3,{116,50,104}},
/* 18600 */ {(16<<2)|3,{116,50,108}},
/* 18601 */ {(16<<2)|3,{116,50,109}},
/* 18602 */ {(16<<2)|3,{116,50,110}},
/* 18603 */ {(16<<2)|3,{116,50,112}},
/* 18604 */ {(16<<2)|3,{116,50,114}},
/* 18605 */ {(16<<2)|3,{116,50,117}},
/* 18606 */ {(10<<2)|2,{116,50,0}},
/* 18607 */ {(10<<2)|2,{116,50,0}},
/* 18608 */ {(10<<2)|2,{116,50,0}},
/* 18609 */ {(10<<2)|2,{116,50,0}},
/* 18610 */ {(10<<2)|2,{116,50,0}},
/* 18611 */ {(10<<2)|2,{116,50,0}},
/* 18612 */ {(10<<2)|2,{116,50,0}},
/* 18613 */ {(10<<2)|2,{116,50,0}},
/* 18614 */ {(10<<2)|2,{116,50,0}},
/* 18615 */ {(10<<2)|2,{116,50,0}},
/* 18616 */ {(10<<2)|2,{116,50,0}},
/* 18617 */ {(10<<2)|2,{116,50,0}},
/* 18618 */ {(10<<2)|2,{116,50,0}},
/* 18619 */ {(10<<2)|2,{116,50,0}},
/* 18620 */ {(10<<2)|2,{116,50,0}},
/* 18621 */ {(10<<2)|2,{116,50,0}},
/* 18622 */ {(10<<2)|2,{116,50,0}},
/* 18623 */ {(10<<2)|2,{116,50,0}},
/* 18624 */ {(15<<2)|3,{116,97,48}},
/* 18625 */ {(15<<2)|3,{116,97,48}},
/* 18626 */ {(15<<2)|3,{116,97,49}},
/* 18627 */ {(15<<2)|3,{116,97,49}},
/* 18628 */ {(15<<2)|3,{116,97,50}},
/* 18629 */ {(15<<2)|3,{116,97,50}},
/* 18630 */ {(15<<2)|3,{116,97,97}},
/* 18631 */ {(15<<2)|3,{116,97,97}},
/* 18632 */ {(15<<2)|3,{116,97,99}},
/* 18633 */ {(15<<2)|3,{116,97,99}},
/* 18634 */ {(15<<2)|3,{116,97,101}},
/* 18635 */ {(15<<2)|3,{116,97,101}},
/* 18636 */ {(15<<2)|3,{116,97,105}},
/* 18637 */ {(15<<2)|3,{116,97,105}},
/* 18638 */ {(15<<2)|3,{116,97,111}},
/* 18639 */ {(15<<2)|3,{116,97,111}},
/* 18640 */ {(15<<2)|3,{116,97,115}},
/* 18641 */ {(15<<2)|3,{116,97,115}},
/* 18642 */ {(15<<2)|3,{116,97,116}},
/* 18643 */ {(15<<2)|3,{116,97,116}},
/* 18644 */ {(16<<2)|3,{116,97,32}},
/* 18645 */ {(16<<2)|3,{116,97,37}},
/* 18646 */ {(16<<2)|3,{116,97,45}},
/* 18647 */ {(16<<2)|3,{116,97,46}},
/* 18648 */ {(16<<2)|3,{116,97,47}},
/* 18649 */ {(16<<2)|3,{116,97,51}},
/* 18650 */ {(16<<2)|3,{116,97,52}},
/* 18651 */ {(16<<2)|3,{116,97,53}},
/* 18652 */ {(16<<2)|3,{116,97,54}},
/* 18653 */ {(16<<2)|3,{116,97,55}},
/* 18654 */ {(16<<2)|3,{116,97,56}},
/* 18655 */ {(16<<2)|3,{116,97,57}},
/* 18656 */ {(16<<2)|3,{116,97,61}},
/* 18657 */ {(16<<2)|3,{116,97,65}},
/* 18658 */ {(16<<2)|3,{116,97,95}},
/* 18659 */ {(16<<2)|3,{116,97,98}},
/* 18660 */ {(16<<2)|3,{116,97,100}},
/* 18661 */ {(16<<2)|3,{116,97,102}},
/* 18662 */ {(16<<2)|3,{116,97,103}},
/* 18663 */ {(16<<2)|3,{116,97,104}},
/* 18664 */ {(16<<2)|3,{116,97,108}},
/* 18665 */ {(16<<2)|3,{116,97,109}},
/* 18666 */ {(16<<2)|3,{116,97,110}},
/* 18667 */ {(16<<2)|3,{116,97,112}},
/* 18668 */ {(16<<2)|3,{116,97,114}},
/* 18669 */ {(16<<2)|3,{116,97,117}},
/* 18670 */ {(10<<2)|2,{116,97,0}},
/* 18671 */ {(10<<2)|2,{116,97,0}},
/* 18672 */ {(10<<2)|2,{116,97,0}},
/* 18673 */ {(10<<2)|2,{116,97,0}},
/* 18674 */ {(10<<2)|2,{116,97,0}},
/* 18675 */ {(10<<2)|2,{116,97,0}},
/* 18676 */ {(10<<2)|2,{116,97,0}},
/* 18677 */ {(10<<2)|2,{116,97,0}},
/* 18678 */ {(10<<2)|2,{116,97,0}},
/* 18679 */ {(10<<2)|2,{116,97,0}},
/* 18680 */ {(10<<2)|2,{116,97,0}},
/* 18681 */ {(10<<2)|2,{116,97,0}},
/* 18682 */ {(10<<2)|2,{116,97,0}},
/* 18683 */ {(10<<2)|2,{116,97,0}},
/* 18684 */ {(10<<2)|2,{116,97,0}},
/* 18685 */ {(10<<2)|2,{116,97,0}},
/* 18686 */ {(10<<2)|2,{116,97,0}},
/* 18687 */ {(10<<2)|2,{116,97,0}},
/* 18688 */ {(15<<2)|3,{116,99,48}},
/* 18689 */ {(15<<2)|3,{116,99,48}},
/* 18690 */ {(15<<2)|3,{116,99,49}},
/* 18691 */ {(15<<2)|3,{116,99,49}},
/* 18692 */ {(15<<2)|3,{116,99,50}},
/* 18693 */ {(15<<2)|3,{116,99,50}},
/* 18694 */ {(15<<2)|3,{116,99,97}},
/* 18695 */ {(15<<2)|3,{116,99,97}},
/* 18696 */ {(15<<2)|3,{116,99,99}},
/* 18697 */ {(15<<2)|3,{116,99,99}},
/* 18698 */ {(15<<2)|3,{116,99,101}},
/* 18699 */ {(15<<2)|3,{116,99,101}},
/* 18700 */ {(15<<2)|3,{116,99,105}},
/* 18701 */ {(15<<2)|3,{116,99,105}},
/* 18702 */ {(15<<2)|3,{116,99,111}},
/* 18703 */ {(15<<2)|3,{116,99,111}},
/* 18704 */ {(15<<2)|3,{116,99,115}},
/* 18705 */ {(15<<2)|3,{116,99,115}},
/* 18706 */ {(15<<2)|3,{116,99,116}},
/* 18707 */ {(15<<2)|3,{116,99,116}},
/* 18708 */ {(16<<2)|3,{116,99,32}},
/* 18709 */ {(16<<2)|3,{116,99,37}},
/* 18710 */ {(16<<2)|3,{116,99,45}},
/* 18711 */ {(16<<2)|3,{116,99,46}},
/* 18712 */ {(16<<2)|3,{116,99,47}},
/* 18713 */ {(16<<2)|3,{116,99,51}},
/* 18714 */ {(16<<2)|3,{116,99,52}},
/* 18715 */ {(16<<2)|3,{116,99,53}},
/* 18716 */ {(16<<2)|3,{116,99,54}},
/* 18717 */ {(16<<2)|3,{116,99,55}},
/* 18718 */ {(16<<2)|3,{116,99,56}},
/* 18719 */ {(16<<2)|3,{116,99,57}},
/* 18720 */ {(16<<2)|3,{116,99,61}},
/* 18721 */ {(16<<2)|3,{116,99,65}},
/* 18722 */ {(16<<2)|3,{116,99,95}},
/* 18723 */ {(16<<2)|3,{116,99,98}},
/* 18724 */ {(16<<2)|3,{116,99,100}},
/* 18725 */ {(16<<2)|3,{116,99,102}},
/* 18726 */ {(16<<2)|3,{116,99,103}},
/* 18727 */ {(16<<2)|3,{116,99,104}},
/* 18728 */ {(16<<2)|3,{116,99,108}},
/* 18729 */ {(16<<2)|3,{116,99,109}},
/* 18730 */ {(16<<2)|3,{116,99,110}},
/* 18731 */ {(16<<2)|3,{116,99,112}},
/* 18732 */ {(16<<2)|3,{116,99,114}},
/* 18733 */ {(16<<2)|3,{116,99,117}},
/* 18734 */ {(10<<2)|2,{116,99,0}},
/* 18735 */ {(10<<2)|2,{116,99,0}},
/* 18736 */ {(10<<2)|2,{116,99,0}},
/* 18737 */ {(10<<2)|2,{116,99,0}},
/* 18738 */ {(10<<2)|2,{116,99,0}},
/* 18739 */ {(10<<2)|2,{116,99,0}},
/* 18740 */ {(10<<2)|2,{116,99,0}},
/* 18741 */ {(10<<2)|2,{116,99,0}},
/* 18742 */ {(10<<2)|2,{116,99,0}},
/* 18743 */ {(10<<2)|2,{116,99,0}},
/* 18744 */ {(10<<2)|2,{116,99,0}},
/* 18745 */ {(10<<2)|2,{116,99,0}},
/* 18746 */ {(10<<2)|2,{116,99,0}},
/* 18747 */ {(10<<2)|2,{116,99,0}},
/* 18748 */ {(10<<2)|2,{116,99,0}},
/* 18749 */ {(10<<2)|2,{116,99,0}},
/* 18750 */ {(10<<2)|2,{116,99,0}},
/* 18751 */ {(10<<2)|2,{116,99,0}},
/* 18752 */ {(15<<2)|3,{116,101,48}},
/* 18753 */ {(15<<2)|3,{116,101,48}},
/* 18754 */ {(15<<2)|3,{116,101,49}},
/* 18755 */ {(15<<2)|3,{116,101,49}},
/* 18756 */ {(15<<2)|3,{116,101,50}},
/* 18757 */ {(15<<2)|3,{116,101,50}},
/* 18758 */ {(15<<2)|3,{116,101,97}},
/* 18759 */ {(15<<2)|3,{116,101,97}},
/* 18760 */ {(15<<2)|3,{116,101,99}},
/* 18761 */ {(15<<2)|3,{116,101,99}},
/* 18762 */ {(15<<2)|3,{116,101,101}},
/* 18763 */ {(15<<2)|3,{116,101,101}},
/* 18764 */ {(15<<2)|3,{116,101,105}},
/* 18765 */ {(15<<2)|3,{116,101,105}},
/* 18766 */ {(15<<2)|3,{116,101,111}},
/* 18767 */ {(15<<2)|3,{116,101,111}},
/* 18768 */ {(15<<2)|3,{116,101,115}},
/* 18769 */ {(15<<2)|3,{116,101,115}},
/* 18770 */ {(15<<2)|3,{116,101,116}},
/* 18771 */ {(15<<2)|3,{116,101,116}},
/* 18772 */ {(16<<2)|3,{116,101,32}},
/* 18773 */ {(16<<2)|3,{116,101,37}},
/* 18774 */ {(16<<2)|3,{116,101,45}},
/* 18775 */ {(16<<2)|3,{116,101,46}},
/* 18776 */ {(16<<2)|3,{116,101,47}},
/* 18777 */ {(16<<2)|3,{116,101,51}},
/* 18778 */ {(16<<2)|3,{116,101,52}},
/* 18779 */ {(16<<2)|3,{116,101,53}},
/* 18780 */ {(16<<2)|3,{116,101,54}},
/* 18781 */ {(16<<2)|3,{116,101,55}},
/* 18782 */ {(16<<2)|3,{116,101,56}},
/* 18783 */ {(16<<2)|3,{116,101,57}},
/* 18784 */ {(16<<2)|3,{116,101,61}},
/* 18785 */ {(16<<2)|3,{116,101,65}},
/* 18786 */ {(16<<2)|3,{116,101,95}},
/* 18787 */ {(16<<2)|3,{116,101,98}},
/* 18788 */ {(16<<2)|3,{116,101,100}},
/* 18789 */ {(16<<2)|3,{116,101,102}},
/* 18790 */ {(16<<2)|3,{116,101,103}},
/* 18791 */ {(16<<2)|3,{116,101,104}},
/* 18792 */ {(16<<2)|3,{116,101,108}},
/* 18793 */ {(16<<2)|3,{116,101,109}},
/* 18794 */ {(16<<2)|3,{116,101,110}},
/* 18795 */ {(16<<2)|3,{116,101,112}},
/* 18796 */ {(16<<2)|3,{116,101,114}},
/* 18797 */ {(16<<2)|3,{116,101,117}},
/* 18798 */ {(10<<2)|2,{116,101,0}},
/* 18799 */ {(10<<2)|2,{116,101,0}},
/* 18800 */ {(10<<2)|2,{116,101,0}},
/* 18801 */ {(10<<2)|2,{116,101,0}},
/* 18802 */ {(10<<2)|2,{116,101,0}},
/* 18803 */ {(10<<2)|2,{116,101,0}},
/* 18804 */ {(10<<2)|2,{116,101,0}},
/* 18805 */ {(10<<2)|2,{116,101,0}},
/* 18806 */ {(10<<2)|2,{116,101,0}},
/* 18807 */ {(10<<2)|2,{116,101,0}},
/* 18808 */ {(10<<2)|2,{116,101,0}},
/* 18809 */ {(10<<2)|2,{116,101,0}},
/* 18810 */ {(10<<2)|2,{116,101,0}},
/* 18811 */ {(10<<2)|2,{116,101,0}},
/* 18812 */ {(10<<2)|2,{116,101,0}},
/* 18813 */ {(10<<2)|2,{116,101,0}},
/* 18814 */ {(10<<2)|2,{116,101,0}},
/* 18815 */ {(10<<2)|2,{116,101,0}},
/* 18816 */ {(15<<2)|3,{116,105,48}},
/* 18817 */ {(15<<2)|3,{116,105,48}},
/* 18818 */ {(15<<2)|3,{116,105,49}},
/* 18819 */ {(15<<2)|3,{116,105,49}},
/* 18820 */ {(15<<2)|3,{116,105,50}},
/* 18821 */ {(15<<2)|3,{116,105,50}},
/* 18822 */ {(15<<2)|3,{116,105,97}},
/* 18823 */ {(15<<2)|3,{116,105,97}},
/* 18824 */ {(15<<2)|3,{116,105,99}},
/* 18825 */ {(15<<2)|3,{116,105,99}},
/* 18826 */ {(15<<2)|3,{116,105,101}},
/* 18827 */ {(15<<2)|3,{116,105,101}},
/* 18828 */ {(15<<2)|3,{116,105,105}},
/* 18829 */ {(15<<2)|3,{116,105,105}},
/* 18830 */ {(15<<2)|3,{116,105,111}},
/* 18831 */ {(15<<2)|3,{116,105,111}},
/* 18832 */ {(15<<2)|3,{116,105,115}},
/* 18833 */ {(15<<2)|3,{116,105,115}},
/* 18834 */ {(15<<2)|3,{116,105,116}},
/* 18835 */ {(15<<2)|3,{116,105,116}},
/* 18836 */ {(16<<2)|3,{116,105,32}},
/* 18837 */ {(16<<2)|3,{116,105,37}},
/* 18838 */ {(16<<2)|3,{116,105,45}},
/* 18839 */ {(16<<2)|3,{116,105,46}},
/* 18840 */ {(16<<2)|3,{116,105,47}},
/* 18841 */ {(16<<2)|3,{116,105,51}},
/* 18842 */ {(16<<2)|3,{116,105,52}},
/* 18843 */ {(16<<2)|3,{116,105,53}},
/* 18844 */ {(16<<2)|3,{116,105,54}},
/* 18845 */ {(16<<2)|3,{116,105,55}},
/* 18846 */ {(16<<2)|3,{116,105,56}},
/* 18847 */ {(16<<2)|3,{116,105,57}},
/* 18848 */ {(16<<2)|3,{116,105,61}},
/* 18849 */ {(16<<2)|3,{116,105,65}},
/* 18850 */ {(16<<2)|3,{116,105,95}},
/* 18851 */ {(16<<2)|3,{116,105,98}},
/* 18852 */ {(16<<2)|3,{116,105,100}},
/* 18853 */ {(16<<2)|3,{116,105,102}},
/* 18854 */ {(16<<2)|3,{116,105,103}},
/* 18855 */ {(16<<2)|3,{116,105,104}},
/* 18856 */ {(16<<2)|3,{116,105,108}},
/* 18857 */ {(16<<2)|3,{116,105,109}},
/* 18858 */ {(16<<2)|3,{116,105,110}},
/* 18859 */ {(16<<2)|3,{116,105,112}},
/* 18860 */ {(16<<2)|3,{116,105,114}},
/* 18861 */ {(16<<2)|3,{116,105,117}},
/* 18862 */ {(10<<2)|2,{116,105,0}},
/* 18863 */ {(10<<2)|2,{116,105,0}},
/* 18864 */ {(10<<2)|2,{116,105,0}},
/* 18865 */ {(10<<2)|2,{116,105,0}},
/* 18866 */ {(10<<2)|2,{116,105,0}},
/* 18867 */ {(10<<2)|2,{116,105,0}},
/* 18868 */ {(10<<2)|2,{116,105,0}},
/* 18869 */ {(10<<2)|2,{116,105,0}},
/* 18870 */ {(10<<2)|2,{116,105,0}},
/* 18871 */ {(10<<2)|2,{116,105,0}},
/* 18872 */ {(10<<2)|2,{116,105,0}},
/* 18873 */ {(10<<2)|2,{116,105,0}},
/* 18874 */ {(10<<2)|2,{116,105,0}},
/* 18875 */ {(10<<2)|2,{116,105,0}},
/* 18876 */ {(10<<2)|2,{116,105,0}},
/* 18877 */ {(10<<2)|2,{116,105,0}},
/* 18878 */ {(10<<2)|2,{116,105,0}},
/* 18879 */ {(10<<2)|2,{116,105,0}},
/* 18880 */ {(15<<2)|3,{116,111,48}},
/* 18881 */ {(15<<2)|3,{116,111,48}},
/* 18882 */ {(15<<2)|3,{116,111,49}},
/* 18883 */ {(15<<2)|3,{116,111,49}},
/* 18884 */ {(15<<2)|3,{116,111,50}},
/* 18885 */ {(15<<2)|3,{116,111,50}},
/* 18886 */ {(15<<2)|3,{116,111,97}},
/* 18887 */ {(15<<2)|3,{116,111,97}},
/* 18888 */ {(15<<2)|3,{116,111,99}},
/* 18889 */ {(15<<2)|3,{116,111,99}},
/* 18890 */ {(15<<2)|3,{116,111,101}},
/* 18891 */ {(15<<2)|3,{116,111,101}},
/* 18892 */ {(15<<2)|3,{116,111,105}},
/* 18893 */ {(15<<2)|3,{116,111,105}},
/* 18894 */ {(15<<2)|3,{116,111,111}},
/* 18895 */ {(15<<2)|3,{116,111,111}},
/* 18896 */ {(15<<2)|3,{116,111,115}},
/* 18897 */ {(15<<2)|3,{116,111,115}},
/* 18898 */ {(15<<2)|3,{116,111,116}},
/* 18899 */ {(15<<2)|3,{116,111,116}},
/* 18900 */ {(16<<2)|3,{116,111,32}},
/* 18901 */ {(16<<2)|3,{116,111,37}},
/* 18902 */ {(16<<2)|3,{116,111,45}},
/* 18903 */ {(16<<2)|3,{116,111,46}},
/* 18904 */ {(16<<2)|3,{116,111,47}},
/* 18905 */ {(16<<2)|3,{116,111,51}},
/* 18906 */ {(16<<2)|3,{116,111,52}},
/* 18907 */ {(16<<2)|3,{116,111,53}},
/* 18908 */ {(16<<2)|3,{116,111,54}},
/* 18909 */ {(16<<2)|3,{116,111,55}},
/* 18910 */ {(16<<2)|3,{116,111,56}},
/* 18911 */ {(16<<2)|3,{116,111,57}},
/* 18912 */ {(16<<2)|3,{116,111,61}},
/* 18913 */ {(16<<2)|3,{116,111,65}},
/* 18914 */ {(16<<2)|3,{116,111,95}},
/* 18915 */ {(16<<2)|3,{116,111,98}},
/* 18916 */ {(16<<2)|3,{116,111,100}},
/* 18917 */ {(16<<2)|3,{116,111,102}},
/* 18918 */ {(16<<2)|3,{116,111,103}},
/* 18919 */ {(16<<2)|3,{116,111,104}},
/* 18920 */ {(16<<2)|3,{116,111,108}},
/* 18921 */ {(16<<2)|3,{116,111,109}},
/* 18922 */ {(16<<2)|3,{116,111,110}},
/* 18923 */ {(16<<2)|3,{116,111,112}},
/* 18924 */ {(16<<2)|3,{116,111,114}},
/* 18925 */ {(16<<2)|3,{116,111,117}},
/* 18926 */ {(10<<2)|2,{116,111,0}},
/* 18927 */ {(10<<2)|2,{116,111,0}},
/* 18928 */ {(10<<2)|2,{116,111,0}},
/* 18929 */ {(10<<2)|2,{116,111,0}},
/* 18930 */ {(10<<2)|2,{116,111,0}},
/* 18931 */ {(10<<2)|2,{116,111,0}},
/* 18932 */ {(10<<2)|2,{116,111,0}},
/* 18933 */ {(10<<2)|2,{116,111,0}},
/* 18934 */ {(10<<2)|2,{116,111,0}},
/* 18935 */ {(10<<2)|2,{116,111,0}},
/* 18936 */ {(10<<2)|2,{116,111,0}},
/* 18937 */ {(10<<2)|2,{116,111,0}},
/* 18938 */ {(10<<2)|2,{116,111,0}},
/* 18939 */ {(10<<2)|2,{116,111,0}},
/* 18940 */ {(10<<2)|2,{116,111,0}},
/* 18941 */ {(10<<2)|2,{116,111,0}},
/* 18942 */ {(10<<2)|2,{116,111,0}},
/* 18943 */ {(10<<2)|2,{116,111,0}},
/* 18944 */ {(15<<2)|3,{116,115,48}},
/* 18945 */ {(15<<2)|3,{116,115,48}},
/* 18946 */ {(15<<2)|3,{116,115,49}},
/* 18947 */ {(15<<2)|3,{116,115,49}},
/* 18948 */ {(15<<2)|3,{116,115,50}},
/* 18949 */ {(15<<2)|3,{116,115,50}},
/* 18950 */ {(15<<2)|3,{116,115,97}},
/* 18951 */ {(15<<2)|3,{116,115,97}},
/* 18952 */ {(15<<2)|3,{116,115,99}},
/* 18953 */ {(15<<2)|3,{116,115,99}},
/* 18954 */ {(15<<2)|3,{116,115,101}},
/* 18955 */ {(15<<2)|3,{116,115,101}},
/* 18956 */ {(15<<2)|3,{116,115,105}},
/* 18957 */ {(15<<2)|3,{116,115,105}},
/* 18958 */ {(15<<2)|3,{116,115,111}},
/* 18959 */ {(15<<2)|3,{116,115,111}},
/* 18960 */ {(15<<2)|3,{116,115,115}},
/* 18961 */ {(15<<2)|3,{116,115,115}},
/* 18962 */ {(15<<2)|3,{116,115,116}},
/* 18963 */ {(15<<2)|3,{116,115,116}},
/* 18964 */ {(16<<2)|3,{116,115,32}},
/* 18965 */ {(16<<2)|3,{116,115,37}},
/* 18966 */ {(16<<2)|3,{116,115,45}},
/* 18967 */ {(16<<2)|3,{116,115,46}},
/* 18968 */ {(16<<2)|3,{116,115,47}},
/* 18969 */ {(16<<2)|3,{116,115,51}},
/* 18970 */ {(16<<2)|3,{116,115,52}},
/* 18971 */ {(16<<2)|3,{116,115,53}},
/* 18972 */ {(16<<2)|3,{116,115,54}},
/* 18973 */ {(16<<2)|3,{116,115,55}},
/* 18974 */ {(16<<2)|3,{116,115,56}},
/* 18975 */ {(16<<2)|3,{116,115,57}},
/* 18976 */ {(16<<2)|3,{116,115,61}},
/* 18977 */ {(16<<2)|3,{116,115,65}},
/* 18978 */ {(16<<2)|3,{116,115,95}},
/* 18979 */ {(16<<2)|3,{116,115,98}},
/* 18980 */ {(16<<2)|3,{116,115,100}},
/* 18981 */ {(16<<2)|3,{116,115,102}},
/* 18982 */ {(16<<2)|3,{116,115,103}},
/* 18983 */ {(16<<2)|3,{116,115,104}},
/* 18984 */ {(16<<2)|3,{116,115,108}},
/* 18985 */ {(16<<2)|3,{116,115,109}},
/* 18986 */ {(16<<2)|3,{116,115,110}},
/* 18987 */ {(16<<2)|3,{116,115,112}},
/* 18988 */ {(16<<2)|3,{116,115,114}},
/* 18989 */ {(16<<2)|3,{116,115,117}},
/* 18990 */ {(10<<2)|2,{116,115,0}},
/* 18991 */ {(10<<2)|2,{116,115,0}},
/* 18992 */ {(10<<2)|2,{116,115,0}},
/* 18993 */ {(10<<2)|2,{116,115,0}},
/* 18994 */ {(10<<2)|2,{116,115,0}},
/* 18995 */ {(10<<2)|2,{116,115,0}},
/* 18996 */ {(10<<2)|2,{116,115,0}},
/* 18997 */ {(10<<2)|2,{116,115,0}},
/* 18998 */ {(10<<2)|2,{116,115,0}},
/* 18999 */ {(10<<2)|2,{116,115,0}},
/* 19000 */ {(10<<2)|2,{116,115,0}},
/* 19001 */ {(10<<2)|2,{116,115,0}},
/* 19002 */ {(10<<2)|2,{116,115,0}},
/* 19003 */ {(10<<2)|2,{116,115,0}},
/* 19004 */ {(10<<2)|2,{116,115,0}},
/* 19005 */ {(10<<2)|2,{116,115,0}},
/* 19006 */ {(10<<2)|2,{116,115,0}},
/* 19007 */ {(10<<2)|2,{116,115,0}},
/* 19008 */ {(15<<2)|3,{116,116,48}},
/* 19009 */ {(15<<2)|3,{116,116,48}},
/* 19010 */ {(15<<2)|3,{116,116,49}},
/* 19011 */ {(15<<2)|3,{116,116,49}},
/* 19012 */ {(15<<2)|3,{116,116,50}},
/* 19013 */ {(15<<2)|3,{116,116,50}},
/* 19014 */ {(15<<2)|3,{116,116,97}},
/* 19015 */ {(15<<2)|3,{116,116,97}},
/* 19016 */ {(15<<2)|3,{116,116,99}},
/* 19017 */ {(15<<2)|3,{116,116,99}},
/* 19018 */ {(15<<2)|3,{116,116,101}},
/* 19019 */ {(15<<2)|3,{116,116,101}},
/* 19020 */ {(15<<2)|3,{116,116,105}},
/* 19021 */ {(15<<2)|3,{116,116,105}},
/* 19022 */ {(15<<2)|3,{116,116,111}},
/* 19023 */ {(15<<2)|3,{116,116,111}},
/* 19024 */ {(15<<2)|3,{116,116,115}},
/* 19025 */ {(15<<2)|3,{116,116,115}},
/* 19026 */ {(15<<2)|3,{116,116,116}},
/* 19027 */ {(15<<2)|3,{116,116,116}},
/* 19028 */ {(16<<2)|3,{116,116,32}},
/* 19029 */ {(16<<2)|3,{116,116,37}},
/* 19030 */ {(16<<2)|3,{116,116,45}},
/* 19031 */ {(16<<2)|3,{116,116,46}},
/* 19032 */ {(16<<2)|3,{116,116,47}},
/* 19033 */ {(16<<2)|3,{116,116,51}},
/* 19034 */ {(16<<2)|3,{116,116,52}},
/* 19035 */ {(16<<2)|3,{116,116,53}},
/* 19036 */ {(16<<2)|3,{116,116,54}},
/* 19037 */ {(16<<2)|3,{116,116,55}},
/* 19038 */ {(16<<2)|3,{116,116,56}},
/* 19039 */ {(16<<2)|3,{116,116,57}},
/* 19040 */ {(16<<2)|3,{116,116,61}},
/* 19041 */ {(16<<2)|3,{116,116,65}},
/* 19042 */ {(16<<2)|3,{116,116,95}},
/* 19043 */ {(16<<2)|3,{116,116,98}},
/* 19044 */ {(16<<2)|3,{116,116,100}},
/* 19045 */ {(16<<2)|3,{116,116,102}},
/* 19046 */ {(16<<2)|3,{116,116,103}},
/* 19047 */ {(16<<2)|3,{116,116,104}},
/* 19048 */ {(16<<2)|3,{116,116,108}},
/* 19049 */ {(16<<2)|3,{116,116,109}},
/* 19050 */ {(16<<2)|3,{116,116,110}},
/* 19051 */ {(16<<2)|3,{116,116,112}},
/* 19052 */ {(16<<2)|3,{116,116,114}},
/* 19053 */ {(16<<2)|3,{116,116,117}},
/* 19054 */ {(10<<2)|2,{116,116,0}},
/* 19055 */ {(10<<2)|2,{116,116,0}},
/* 19056 */ {(10<<2)|2,{116,116,0}},
/* 19057 */ {(10<<2)|2,{116,116,0}},
/* 19058 */ {(10<<2)|2,{116,116,0}},
/* 19059 */ {(10<<2)|2,{116,116,0}},
/* 19060 */ {(10<<2)|2,{116,116,0}},
/* 19061 */ {(10<<2)|2,{116,116,0}},
/* 19062 */ {(10<<2)|2,{116,116,0}},
/* 19063 */ {(10<<2)|2,{116,116,0}},
/* 19064 */ {(10<<2)|2,{116,116,0}},
/* 19065 */ {(10<<2)|2,{116,116,0}},
/* 19066 */ {(10<<2)|2,{116,116,0}},
/* 19067 */ {(10<<2)|2,{116,116,0}},
/* 19068 */ {(10<<2)|2,{116,116,0}},
/* 19069 */ {(10<<2)|2,{116,116,0}},
/* 19070 */ {(10<<2)|2,{116,116,0}},
/* 19071 */ {(10<<2)|2,{116,116,0}},
/* 19072 */ {(16<<2)|3,{116,32,48}},
/* 19073 */ {(16<<2)|3,{116,32,49}},
/* 19074 */ {(16<<2)|3,{116,32,50}},
/* 19075 */ {(16<<2)|3,{116,32,97}},
/* 19076 */ {(16<<2)|3,{116,32,99}},
/* 19077 */ {(16<<2)|3,{116,32,101}},
/* 19078 */ {(16<<2)|3,{116,32,105}},
/* 19079 */ {(16<<2)|3,{116,32,111}},
/* 19080 */ {(16<<2)|3,{116,32,115}},
/* 19081 */ {(16<<2)|3,{116,32,116}},
/* 19082 */ {(11<<2)|2,{116,32,0}},
/* 19083 */ {(11<<2)|2,{116,32,0}},
/* 19084 */ {(11<<2)|2,{116,32,0}},
/* 19085 */ {(11<<2)|2,{116,32,0}},
/* 19086 */ {(11<<2)|2,{116,32,0}},
/* 19087 */ {(11<<2)|2,{116,32,0}},
/* 19088 */ {(11<<2)|2,{116,32,0}},
/* 19089 */ {(11<<2)|2,{116,32,0}},
/* 19090 */ {(11<<2)|2,{116,32,0}},
/* 19091 */ {(11<<2)|2,{116,32,0}},
/* 19092 */ {(11<<2)|2,{116,32,0}},
/* 19093 */ {(11<<2)|2,{116,32,0}},
/* 19094 */ {(11<<2)|2,{116,32,0}},
/* 19095 */ {(11<<2)|2,{116,32,0}},
/* 19096 */ {(11<<2)|2,{116,32,0}},
/* 19097 */ {(11<<2)|2,{116,32,0}},
/* 19098 */ {(11<<2)|2,{116,32,0}},
/* 19099 */ {(11<<2)|2,{116,32,0}},
/* 19100 */ {(11<<2)|2,{116,32,0}},
/* 19101 */ {(11<<2)|2,{116,32,0}},
/* 19102 */ {(11<<2)|2,{116,32,0}},
/* 19103 */ {(11<<2)|2,{116,32,0}},
/* 19104 */ {(16<<2)|3,{116,37,48}},
/* 19105 */ {(16<<2)|3,{116,37,49}},
/* 19106 */ {(16<<2)|3,{116,37,50}},
/* 19107 */ {(16<<2)|3,{116,37,97}},
/* 19108 */ {(16<<2)|3,{116,37,99}},
/* 19109 */ {(16<<2)|3,{116,37,101}},
/* 19110 */ {(16<<2)|3,{116,37,105}},
/* 19111 */ {(16<<2)|3,{116,37,111}},
/* 19112 */ {(16<<2)|3,{116,37,115}},
/* 19113 */ {(16<<2)|3,{116,37,116}},
/* 19114 */ {(11<<2)|2,{116,37,0}},
/* 19115 */ {(11<<2)|2,{116,37,0}},
/* 19116 */ {(11<<2)|2,{116,37,0}},
/* 19117 */ {(11<<2)|2,{116,37,0}},
/* 19118 */ {(11<<2)|2,{116,37,0}},
/* 19119 */ {(11<<2)|2,{116,37,0}},
/* 19120 */ {(11<<2)|2,{116,37,0}},
/* 19121 */ {(11<<2)|2,{116,37,0}},
/* 19122 */ {(11<<2)|2,{116,37,0}},
/* 19123 */ {(11<<2)|2,{116,37,0}},
/* 19124 */ {(11<<2)|2,{116,37,0}},
/* 19125 */ {(11<<2)|2,{116,37,0}},
/* 19126 */ {(11<<2)|2,{116,37,0}},
/* 19127 */ {(11<<2)|2,{116,37,0}},
/* 19128 */ {(11<<2)|2,{116,37,0}},
/* 19129 */ {(11<<2)|2,{116,37,0}},
/* 19130 */ {(11<<2)|2,{116,37,0}},
/* 19131 */ {(11<<2)|2,{116,37,0}},
/* 19132 */ {(11<<2)|2,{116,37,0}},
/* 19133 */ {(11<<2)|2,{116,37,0}},
/* 19134 */ {(11<<2)|2,{116,37,0}},
/* 19135 */ {(11<<2)|2,{116,37,0}},
/* 19136 */ {(16<<2)|3,{116,45,48}},
/* 19137 */ {(16<<2)|3,{116,45,49}},
/* 19138 */ {(16<<2)|3,{116,45,50}},
/* 19139 */ {(16<<2)|3,{116,45,97}},
/* 19140 */ {(16<<2)|3,{116,45,99}},
/* 19141 */ {(16<<2)|3,{116,45,101}},
/* 19142 */ {(16<<2)|3,{116,45,105}},
/* 19143 */ {(16<<2)|3,{116,45,111}},
/* 19144 */ {(16<<2)|3,{116,45,115}},
/* 19145 */ {(16<<2)|3,{116,45,116}},
/* 19146 */ {(11<<2)|2,{116,45,0}},
/* 19147 */ {(11<<2)|2,{116,45,0}},
/* 19148 */ {(11<<2)|2,{116,45,0}},
/* 19149 */ {(11<<2)|2,{116,45,0}},
/* 19150 */ {(11<<2)|2,{116,45,0}},
/* 19151 */ {(11<<2)|2,{116,45,0}},
/* 19152 */ {(11<<2)|2,{116,45,0}},
/* 19153 */ {(11<<2)|2,{116,45,0}},
/* 19154 */ {(11<<2)|2,{116,45,0}},
/* 19155 */ {(11<<2)|2,{116,45,0}},
/* 19156 */ {(11<<2)|2,{116,45,0}},
/* 19157 */ {(11<<2)|2,{116,45,0}},
/* 19158 */ {(11<<2)|2,{116,45,0}},
/* 19159 */ {(11<<2)|2,{116,45,0}},
/* 19160 */ {(11<<2)|2,{116,45,0}},
/* 19161 */ {(11<<2)|2,{116,45,0}},
/* 19162 */ {(11<<2)|2,{116,45,0}},
/* 19163 */ {(11<<2)|2,{116,45,0}},
/* 19164 */ {(11<<2)|2,{116,45,0}},
/* 19165 */ {(11<<2)|2,{116,45,0}},
/* 19166 */ {(11<<2)|2,{116,45,0}},
/* 19167 */ {(11<<2)|2,{116,45,0}},
/* 19168 */ {(16<<2)|3,{116,46,48}},
/* 19169 */ {(16<<2)|3,{116,46,49}},
/* 19170 */ {(16<<2)|3,{116,46,50}},
/* 19171 */ {(16<<2)|3,{116,46,97}},
/* 19172 */ {(16<<2)|3,{116,46,99}},
/* 19173 */ {(16<<2)|3,{116,46,101}},
/* 19174 */ {(16<<2)|3,{116,46,105}},
/* 19175 */ {(16<<2)|3,{116,46,111}},
/* 19176 */ {(16<<2)|3,{116,46,115}},
/* 19177 */ {(16<<2)|3,{116,46,116}},
/* 19178 */ {(11<<2)|2,{116,46,0}},
/* 19179 */ {(11<<2)|2,{116,46,0}},
/* 19180 */ {(11<<2)|2,{116,46,0}},
/* 19181 */ {(11<<2)|2,{116,46,0}},
/* 19182 */ {(11<<2)|2,{116,46,0}},
/* 19183 */ {(11<<2)|2,{116,46,0}},
/* 19184 */ {(11<<2)|2,{116,46,0}},
/* 19185 */ {(11<<2)|2,{116,46,0}},
/* 19186 */ {(11<<2)|2,{116,46,0}},
/* 19187 */ {(11<<2)|2,{116,46,0}},
/* 19188 */ {(11<<2)|2,{116,46,0}},
/* 19189 */ {(11<<2)|2,{116,46,0}},
/* 19190 */ {(11<<2)|2,{116,46,0}},
/* 19191 */ {(11<<2)|2,{116,46,0}},
/* 19192 */ {(11<<2)|2,{116,46,0}},
/* 19193 */ {(11<<2)|2,{116,46,0}},
/* 19194 */ {(11<<2)|2,{116,46,0}},
/* 19195 */ {(11<<2)|2,{116,46,0}},
/* 19196 */ {(11<<2)|2,{116,46,0}},
/* 19197 */ {(11<<2)|2,{116,46,0}},
/* 19198 */ {(11<<2)|2,{116,46,0}},
/* 19199 */ {(11<<2)|2,{116,46,0}},
/* 19200 */ {(16<<2)|3,{116,47,48}},
/* 19201 */ {(16<<2)|3,{116,47,49}},
/* 19202 */ {(16<<2)|3,{116,47,50}},
/* 19203 */ {(16<<2)|3,{116,47,97}},
/* 19204 */ {(16<<2)|3,{116,47,99}},
/* 19205 */ {(16<<2)|3,{116,47,101}},
/* 19206 */ {(16<<2)|3,{116,47,105}},
/* 19207 */ {(16<<2)|3,{116,47,111}},
/* 19208 */ {(16<<2)|3,{116,47,115}},
/* 19209 */ {(16<<2)|3,{116,47,116}},
/* 19210 */ {(11<<2)|2,{116,47,0}},
/* 19211 */ {(11<<2)|2,{116,47,0}},
/* 19212 */ {(11<<2)|2,{116,47,0}},
/* 19213 */ {(11<<2)|2,{116,47,0}},
/* 19214 */ {(11<<2)|2,{116,47,0}},
/* 19215 */ {(11<<2)|2,{116,47,0}},
/* 19216 */ {(11<<2)|2,{116,47,0}},
/* 19217 */ {(11<<2)|2,{116,47,0}},
/* 19218 */ {(11<<2)|2,{116,47,0}},
/* 19219 */ {(11<<2)|2,{116,47,0}},
/* 19220 */ {(11<<2)|2,{116,47,0}},
/* 19221 */ {(11<<2)|2,{116,47,0}},
/* 19222 */ {(11<<2)|2,{116,47,0}},
/* 19223 */ {(11<<2)|2,{116,47,0}},
/* 19224 */ {(11<<2)|2,{116,47,0}},
/* 19225 */ {(11<<2)|2,{116,47,0}},
/* 19226 */ {(11<<2)|2,{116,47,0}},
/* 19227 */ {(11<<2)|2,{116,47,0}},
/* 19228 */ {(11<<2)|2,{116,47,0}},
/* 19229 */ {(11<<2)|2,{116,47,0}},
/* 19230 */ {(11<<2)|2,{116,47,0}},
/* 19231 */ {(11<<2)|2,{116,47,0}},
/* 19232 */ {(16<<2)|3,{116,51,48}},
/* 19233 */ {(16<<2)|3,{116,51,49}},
/* 19234 */ {(16<<2)|3,{116,51,50}},
/* 19235 */ {(16<<2)|3,{116,51,97}},
/* 19236 */ {(16<<2)|3,{116,51,99}},
/* 19237 */ {(16<<2)|3,{116,51,101}},
/* 19238 */ {(16<<2)|3,{116,51,105}},
/* 19239 */ {(16<<2)|3,{116,51,111}},
/* 19240 */ {(16<<2)|3,{116,51,115}},
/* 19241 */ {(16<<2)|3,{116,51,116}},
/* 19242 */ {(11<<2)|2,{116,51,0}},
/* 19243 */ {(11<<2)|2,{116,51,0}},
/* 19244 */ {(11<<2)|2,{116,51,0}},
/* 19245 */ {(11<<2)|2,{116,51,0}},
/* 19246 */ {(11<<2)|2,{116,51,0}},
/* 19247 */ {(11<<2)|2,{116,51,0}},
/* 19248 */ {(11<<2)|2,{116,51,0}},
/* 19249 */ {(11<<2)|2,{116,51,0}},
/* 19250 */ {(11<<2)|2,{116,51,0}},
/* 19251 */ {(11<<2)|2,{116,51,0}},
/* 19252 */ {(11<<2)|2,{116,51,0}},
/* 19253 */ {(11<<2)|2,{116,51,0}},
/* 19254 */ {(11<<2)|2,{116,51,0}},
/* 19255 */ {(11<<2)|2,{116,51,0}},
/* 19256 */ {(11<<2)|2,{116,51,0}},
/* 19257 */ {(11<<2)|2,{116,51,0}},
/* 19258 */ {(11<<2)|2,{116,51,0}},
/* 19259 */ {(11<<2)|2,{116,51,0}},
/* 19260 */ {(11<<2)|2,{116,51,0}},
/* 19261 */ {(11<<2)|2,{116,51,0}},
/* 19262 */ {(11<<2)|2,{116,51,0}},
/* 19263 */ {(11<<2)|2,{116,51,0}},
/* 19264 */ {(16<<2)|3,{116,52,48}},
/* 19265 */ {(16<<2)|3,{116,52,49}},
/* 19266 */ {(16<<2)|3,{116,52,50}},
/* 19267 */ {(16<<2)|3,{116,52,97}},
/* 19268 */ {(16<<2)|3,{116,52,99}},
/* 19269 */ {(16<<2)|3,{116,52,101}},
/* 19270 */ {(16<<2)|3,{116,52,105}},
/* 19271 */ {(16<<2)|3,{116,52,111}},
/* 19272 */ {(16<<2)|3,{116,52,115}},
/* 19273 */ {(16<<2)|3,{116,52,116}},
/* 19274 */ {(11<<2)|2,{116,52,0}},
/* 19275 */ {(11<<2)|2,{116,52,0}},
/* 19276 */ {(11<<2)|2,{116,52,0}},
/* 19277 */ {(11<<2)|2,{116,52,0}},
/* 19278 */ {(11<<2)|2,{116,52,0}},
/* 19279 */ {(11<<2)|2,{116,52,0}},
/* 19280 */ {(11<<2)|2,{116,52,0}},
/* 19281 */ {(11<<2)|2,{116,52,0}},
/* 19282 */ {(11<<2)|2,{116,52,0}},
/* 19283 */ {(11<<2)|2,{116,52,0}},
/* 19284 */ {(11<<2)|2,{116,52,0}},
/* 19285 */ {(11<<2)|2,{116,52,0}},
/* 19286 */ {(11<<2)|2,{116,52,0}},
/* 19287 */ {(11<<2)|2,{116,52,0}},
/* 19288 */ {(11<<2)|2,{116,52,0}},
/* 19289 */ {(11<<2)|2,{116,52,0}},
/* 19290 */ {(11<<2)|2,{116,52,0}},
/* 19291 */ {(11<<2)|2,{116,52,0}},
/* 19292 */ {(11<<2)|2,{116,52,0}},
/* 19293 */ {(11<<2)|2,{116,52,0}},
/* 19294 */ {(11<<2)|2,{116,52,0}},
/* 19295 */ {(11<<2)|2,{116,52,0}},
/* 19296 */ {(16<<2)|3,{116,53,48}},
/* 19297 */ {(16<<2)|3,{116,53,49}},
/* 19298 */ {(16<<2)|3,{116,53,50}},
/* 19299 */ {(16<<2)|3,{116,53,97}},
/* 19300 */ {(16<<2)|3,{116,53,99}},
/* 19301 */ {(16<<2)|3,{116,53,101}},
/* 19302 */ {(16<<2)|3,{116,53,105}},
/* 19303 */ {(16<<2)|3,{116,53,111}},
/* 19304 */ {(16<<2)|3,{116,53,115}},
/* 19305 */ {(16<<2)|3,{116,53,116}},
/* 19306 */ {(11<<2)|2,{116,53,0}},
/* 19307 */ {(11<<2)|2,{116,53,0}},
/* 19308 */ {(11<<2)|2,{116,53,0}},
/* 19309 */ {(11<<2)|2,{116,53,0}},
/* 19310 */ {(11<<2)|2,{116,53,0}},
/* 19311 */ {(11<<2)|2,{116,53,0}},
/* 19312 */ {(11<<2)|2,{116,53,0}},
/* 19313 */ {(11<<2)|2,{116,53,0}},
/* 19314 */ {(11<<2)|2,{116,53,0}},
/* 19315 */ {(11<<2)|2,{116,53,0}},
/* 19316 */ {(11<<2)|2,{116,53,0}},
/* 19317 */ {(11<<2)|2,{116,53,0}},
/* 19318 */ {(11<<2)|2,{116,53,0}},
/* 19319 */ {(11<<2)|2,{116,53,0}},
/* 19320 */ {(11<<2)|2,{116,53,0}},
/* 19321 */ {(11<<2)|2,{116,53,0}},
/* 19322 */ {(11<<2)|2,{116,53,0}},
/* 19323 */ {(11<<2)|2,{116,53,0}},
/* 19324 */ {(11<<2)|2,{116,53,0}},
/* 19325 */ {(11<<2)|2,{116,53,0}},
/* 19326 */ {(11<<2)|2,{116,53,0}},
/* 19327 */ {(11<<2)|2,{116,53,0}},
/* 19328 */ {(16<<2)|3,{116,54,48}},
/* 19329 */ {(16<<2)|3,{116,54,49}},
/* 19330 */ {(16<<2)|3,{116,54,50}},
/* 19331 */ {(16<<2)|3,{116,54,97}},
/* 19332 */ {(16<<2)|3,{116,54,99}},
/* 19333 */ {(16<<2)|3,{116,54,101}},
/* 19334 */ {(16<<2)|3,{116,54,105}},
/* 19335 */ {(16<<2)|3,{116,54,111}},
/* 19336 */ {(16<<2)|3,{116,54,115}},
/* 19337 */ {(16<<2)|3,{116,54,116}},
/* 19338 */ {(11<<2)|2,{116,54,0}},
/* 19339 */ {(11<<2)|2,{116,54,0}},
/* 19340 */ {(11<<2)|2,{116,54,0}},
/* 19341 */ {(11<<2)|2,{116,54,0}},
/* 19342 */ {(11<<2)|2,{116,54,0}},
/* 19343 */ {(11<<2)|2,{116,54,0}},
/* 19344 */ {(11<<2)|2,{116,54,0}},
/* 19345 */ {(11<<2)|2,{116,54,0}},
/* 19346 */ {(11<<2)|2,{116,54,0}},
/* 19347 */ {(11<<2)|2,{116,54,0}},
/* 19348 */ {(11<<2)|2,{116,54,0}},
/* 19349 */ {(11<<2)|2,{116,54,0}},
/* 19350 */ {(11<<2)|2,{116,54,0}},
/* 19351 */ {(11<<2)|2,{116,54,0}},
/* 19352 */ {(11<<2)|2,{116,54,0}},
/* 19353 */ {(11<<2)|2,{116,54,0}},
/* 19354 */ {(11<<2)|2,{116,54,0}},
/* 19355 */ {(11<<2)|2,{116,54,0}},
/* 19356 */ {(11<<2)|2,{116,54,0}},
/* 19357 */ {(11<<2)|2,{116,54,0}},
/* 19358 */ {(11<<2)|2,{116,54,0}},
/* 19359 */ {(11<<2)|2,{116,54,0}},
/* 19360 */ {(16<<2)|3,{116,55,48}},
/* 19361 */ {(16<<2)|3,{116,55,49}},
/* 19362 */ {(16<<2)|3,{116,55,50}},
/* 19363 */ {(16<<2)|3,{116,55,97}},
/* 19364 */ {(16<<2)|3,{116,55,99}},
/* 19365 */ {(16<<2)|3,{116,55,101}},
/* 19366 */ {(16<<2)|3,{116,55,105}},
/* 19367 */ {(16<<2)|3,{116,55,111}},
/* 19368 */ {(16<<2)|3,{116,55,115}},
/* 19369 */ {(16<<2)|3,{116,55,116}},
/* 19370 */ {(11<<2)|2,{116,55,0}},
/* 19371 */ {(11<<2)|2,{116,55,0}},
/* 19372 */ {(11<<2)|2,{116,55,0}},
/* 19373 */ {(11<<2)|2,{116,55,0}},
/* 19374 */ {(11<<2)|2,{116,55,0}},
/* 19375 */ {(11<<2)|2,{116,55,0}},
/* 19376 */ {(11<<2)|2,{116,55,0}},
/* 19377 */ {(11<<2)|2,{116,55,0}},
/* 19378 */ {(11<<2)|2,{116,55,0}},
/* 19379 */ {(11<<2)|2,{116,55,0}},
/* 19380 */ {(11<<2)|2,{116,55,0}},
/* 19381 */ {(11<<2)|2,{116,55,0}},
/* 19382 */ {(11<<2)|2,{116,55,0}},
/* 19383 */ {(11<<2)|2,{116,55,0}},
/* 19384 */ {(11<<2)|2,{116,55,0}},
/* 19385 */ {(11<<2)|2,{116,55,0}},
/* 19386 */ {(11<<2)|2,{116,55,0}},
/* 19387 */ {(11<<2)|2,{116,55,0}},
/* 19388 */ {(11<<2)|2,{116,55,0}},
/* 19389 */ {(11<<2)|2,{116,55,0}},
/* 19390 */ {(11<<2)|2,{116,55,0}},
/* 19391 */ {(11<<2)|2,{116,55,0}},
/* 19392 */ {(16<<2)|3,{116,56,48}},
/* 19393 */ {(16<<2)|3,{116,56,49}},
/* 19394 */ {(16<<2)|3,{116,56,50}},
/* 19395 */ {(16<<2)|3,{116,56,97}},
/* 19396 */ {(16<<2)|3,{116,56,99}},
/* 19397 */ {(16<<2)|3,{116,56,101}},
/* 19398 */ {(16<<2)|3,{116,56,105}},
/* 19399 */ {(16<<2)|3,{116,56,111}},
/* 19400 */ {(16<<2)|3,{116,56,115}},
/* 19401 */ {(16<<2)|3,{116,56,116}},
/* 19402 */ {(11<<2)|2,{116,56,0}},
/* 19403 */ {(11<<2)|2,{116,56,0}},
/* 19404 */ {(11<<2)|2,{116,56,0}},
/* 19405 */ {(11<<2)|2,{116,56,0}},
/* 19406 */ {(11<<2)|2,{116,56,0}},
/* 19407 */ {(11<<2)|2,{116,56,0}},
/* 19408 */ {(11<<2)|2,{116,56,0}},
/* 19409 */ {(11<<2)|2,{116,56,0}},
/* 19410 */ {(11<<2)|2,{116,56,0}},
/* 19411 */ {(11<<2)|2,{116,56,0}},
/* 19412 */ {(11<<2)|2,{116,56,0}},
/* 19413 */ {(11<<2)|2,{116,56,0}},
/* 19414 */ {(11<<2)|2,{116,56,0}},
/* 19415 */ {(11<<2)|2,{116,56,0}},
/* 19416 */ {(11<<2)|2,{116,56,0}},
/* 19417 */ {(11<<2)|2,{116,56,0}},
/* 19418 */ {(11<<2)|2,{116,56,0}},
/* 19419 */ {(11<<2)|2,{116,56,0}},
/* 19420 */ {(11<<2)|2,{116,56,0}},
/* 19421 */ {(11<<2)|2,{116,56,0}},
/* 19422 */ {(11<<2)|2,{116,56,0}},
/* 19423 */ {(11<<2)|2,{116,56,0}},
/* 19424 */ {(16<<2)|3,{116,57,48}},
/* 19425 */ {(16<<2)|3,{116,57,49}},
/* 19426 */ {(16<<2)|3,{116,57,50}},
/* 19427 */ {(16<<2)|3,{116,57,97}},
/* 19428 */ {(16<<2)|3,{116,57,99}},
/* 19429 */ {(16<<2)|3,{116,57,101}},
/* 19430 */ {(16<<2)|3,{116,57,105}},
/* 19431 */ {(16<<2)|3,{116,57,111}},
/* 19432 */ {(16<<2)|3,{116,57,115}},
/* 19433 */ {(16<<2)|3,{116,57,116}},
/* 19434 */ {(11<<2)|2,{116,57,0}},
/* 19435 */ {(11<<2)|2,{116,57,0}},
/* 19436 */ {(11<<2)|2,{116,57,0}},
/* 19437 */ {(11<<2)|2,{116,57,0}},
/* 19438 */ {(11<<2)|2,{116,57,0}},
/* 19439 */ {(11<<2)|2,{116,57,0}},
/* 19440 */ {(11<<2)|2,{116,57,0}},
/* 19441 */ {(11<<2)|2,{116,57,0}},
/* 19442 */ {(11<<2)|2,{116,57,0}},
/* 19443 */ {(11<<2)|2,{116,57,0}},
/* 19444 */ {(11<<2)|2,{116,57,0}},
/* 19445 */ {(11<<2)|2,{116,57,0}},
/* 19446 */ {(11<<2)|2,{116,57,0}},
/* 19447 */ {(11<<2)|2,{116,57,0}},
/* 19448 */ {(11<<2)|2,{116,57,0}},
/* 19449 */ {(11<<2)|2,{116,57,0}},
/* 19450 */ {(11<<2)|2,{116,57,0}},
/* 19451 */ {(11<<2)|2,{116,57,0}},
/* 19452 */ {(11<<2)|2,{116,57,0}},
/* 19453 */ {(11<<2)|2,{116,57,0}},
/* 19454 */ {(11<<2)|2,{116,57,0}},
/* 19455 */ {(11<<2)|2,{116,57,0}},
/* 19456 */ {(16<<2)|3,{116,61,48}},
/* 19457 */ {(16<<2)|3,{116,61,49}},
/* 19458 */ {(16<<2)|3,{116,61,50}},
/* 19459 */ {(16<<2)|3,{116,61,97}},
/* 19460 */ {(16<<2)|3,{116,61,99}},
/* 19461 */ {(16<<2)|3,{116,61,101}},
/* 19462 */ {(16<<2)|3,{116,61,105}},
/* 19463 */ {(16<<2)|3,{116,61,111}},
/* 19464 */ {(16<<2)|3,{116,61,115}},
/* 19465 */ {(16<<2)|3,{116,61,116}},
/* 19466 */ {(11<<2)|2,{116,61,0}},
/* 19467 */ {(11<<2)|2,{116,61,0}},
/* 19468 */ {(11<<2)|2,{116,61,0}},
/* 19469 */ {(11<<2)|2,{116,61,0}},
/* 19470 */ {(11<<2)|2,{116,61,0}},
/* 19471 */ {(11<<2)|2,{116,61,0}},
/* 19472 */ {(11<<2)|2,{116,61,0}},
/* 19473 */ {(11<<2)|2,{116,61,0}},
/* 19474 */ {(11<<2)|2,{116,61,0}},
/* 19475 */ {(11<<2)|2,{116,61,0}},
/* 19476 */ {(11<<2)|2,{116,61,0}},
/* 19477 */ {(11<<2)|2,{116,61,0}},
/* 19478 */ {(11<<2)|2,{116,61,0}},
/* 19479 */ {(11<<2)|2,{116,61,0}},
/* 19480 */ {(11<<2)|2,{116,61,0}},
/* 19481 */ {(11<<2)|2,{116,61,0}},
/* 19482 */ {(11<<2)|2,{116,61,0}},
/* 19483 */ {(11<<2)|2,{116,61,0}},
/* 19484 */ {(11<<2)|2,{116,61,0}},
/* 19485 */ {(11<<2)|2,{116,61,0}},
/* 19486 */ {(11<<2)|2,{116,61,0}},
/* 19487 */ {(11<<2)|2,{116,61,0}},
/* 19488 */ {(16<<2)|3,{116,65,48}},
/* 19489 */ {(16<<2)|3,{116,65,49}},
/* 19490 */ {(16<<2)|3,{116,65,50}},
/* 19491 */ {(16<<2)|3,{116,65,97}},
/* 19492 */ {(16<<2)|3,{116,65,99}},
/* 19493 */ {(16<<2)|3,{116,65,101}},
/* 19494 */ {(16<<2)|3,{116,65,105}},
/* 19495 */ {(16<<2)|3,{116,65,111}},
/* 19496 */ {(16<<2)|3,{116,65,115}},
/* 19497 */ {(16<<2)|3,{116,65,116}},
/* 19498 */ {(11<<2)|2,{116,65,0}},
/* 19499 */ {(11<<2)|2,{116,65,0}},
/* 19500 */ {(11<<2)|2,{116,65,0}},
/* 19501 */ {(11<<2)|2,{116,65,0}},
/* 19502 */ {(11<<2)|2,{116,65,0}},
/* 19503 */ {(11<<2)|2,{116,65,0}},
/* 19504 */ {(11<<2)|2,{116,65,0}},
/* 19505 */ {(11<<2)|2,{116,65,0}},
/* 19506 */ {(11<<2)|2,{116,65,0}},
/* 19507 */ {(11<<2)|2,{116,65,0}},
/* 19508 */ {(11<<2)|2,{116,65,0}},
/* 19509 */ {(11<<2)|2,{116,65,0}},
/* 19510 */ {(11<<2)|2,{116,65,0}},
/* 19511 */ {(11<<2)|2,{116,65,0}},
/* 19512 */ {(11<<2)|2,{116,65,0}},
/* 19513 */ {(11<<2)|2,{116,65,0}},
/* 19514 */ {(11<<2)|2,{116,65,0}},
/* 19515 */ {(11<<2)|2,{116,65,0}},
/* 19516 */ {(11<<2)|2,{116,65,0}},
/* 19517 */ {(11<<2)|2,{116,65,0}},
/* 19518 */ {(11<<2)|2,{116,65,0}},
/* 19519 */ {(11<<2)|2,{116,65,0}},
/* 19520 */ {(16<<2)|3,{116,95,48}},
/* 19521 */ {(16<<2)|3,{116,95,49}},
/* 19522 */ {(16<<2)|3,{116,95,50}},
/* 19523 */ {(16<<2)|3,{116,95,97}},
/* 19524 */ {(16<<2)|3,{116,95,99}},
/* 19525 */ {(16<<2)|3,{116,95,101}},
/* 19526 */ {(16<<2)|3,{116,95,105}},
/* 19527 */ {(16<<2)|3,{116,95,111}},
/* 19528 */ {(16<<2)|3,{116,95,115}},
/* 19529 */ {(16<<2)|3,{116,95,116}},
/* 19530 */ {(11<<2)|2,{116,95,0}},
/* 19531 */ {(11<<2)|2,{116,95,0}},
/* 19532 */ {(11<<2)|2,{116,95,0}},
/* 19533 */ {(11<<2)|2,{116,95,0}},
/* 19534 */ {(11<<2)|2,{116,95,0}},
/* 19535 */ {(11<<2)|2,{116,95,0}},
/* 19536 */ {(11<<2)|2,{116,95,0}},
/* 19537 */ {(11<<2)|2,{116,95,0}},
/* 19538 */ {(11<<2)|2,{116,95,0}},
/* 19539 */ {(11<<2)|2,{116,95,0}},
/* 19540 */ {(11<<2)|2,{116,95,0}},
/* 19541 */ {(11<<2)|2,{116,95,0}},
/* 19542 */ {(11<<2)|2,{116,95,0}},
/* 19543 */ {(11<<2)|2,{116,95,0}},
/* 19544 */ {(11<<2)|2,{116,95,0}},
/* 19545 */ {(11<<2)|2,{116,95,0}},
/* 19546 */ {(11<<2)|2,{116,95,0}},
/* 19547 */ {(11<<2)|2,{116,95,0}},
/* 19548 */ {(11<<2)|2,{116,95,0}},
/* 19549 */ {(11<<2)|2,{116,95,0}},
/* 19550 */ {(11<<2)|2,{116,95,0}},
/* 19551 */ {(11<<2)|2,{116,95,0}},
/* 19552 */ {(16<<2)|3,{116,98,48}},
/* 19553 */ {(16<<2)|3,{116,98,49}},
/* 19554 */ {(16<<2)|3,{116,98,50}},
/* 19555 */ {(16<<2)|3,{116,98,97}},
/* 19556 */ {(16<<2)|3,{116,98,99}},
/* 19557 */ {(16<<2)|3,{116,98,101}},
/* 19558 */ {(16<<2)|3,{116,98,105}},
/* 19559 */ {(16<<2)|3,{116,98,111}},
/* 19560 */ {(16<<2)|3,{116,98,115}},
/* 19561 */ {(16<<2)|3,{116,98,116}},
/* 19562 */ {(11<<2)|2,{116,98,0}},
/* 19563 */ {(11<<2)|2,{116,98,0}},
/* 19564 */ {(11<<2)|2,{116,98,0}},
/* 19565 */ {(11<<2)|2,{116,98,0}},
/* 19566 */ {(11<<2)|2,{116,98,0}},
/* 19567 */ {(11<<2)|2,{116,98,0}},
/* 19568 */ {(11<<2)|2,{116,98,0}},
/* 19569 */ {(11<<2)|2,{116,98,0}},
/* 19570 */ {(11<<2)|2,{116,98,0}},
/* 19571 */ {(11<<2)|2,{116,98,0}},
/* 19572 */ {(11<<2)|2,{116,98,0}},
/* 19573 */ {(11<<2)|2,{116,98,0}},
/* 19574 */ {(11<<2)|2,{116,98,0}},
/* 19575 */ {(11<<2)|2,{116,98,0}},
/* 19576 */ {(11<<2)|2,{116,98,0}},
/* 19577 */ {(11<<2)|2,{116,98,0}},
/* 19578 */ {(11<<2)|2,{116,98,0}},
/* 19579 */ {(11<<2)|2,{116,98,0}},
/* 19580 */ {(11<<2)|2,{116,98,0}},
/* 19581 */ {(11<<2)|2,{116,98,0}},
/* 19582 */ {(11<<2)|2,{116,98,0}},
/* 19583 */ {(11<<2)|2,{116,98,0}},
/* 19584 */ {(16<<2)|3,{116,100,48}},
/* 19585 */ {(16<<2)|3,{116,100,49}},
/* 19586 */ {(16<<2)|3,{116,100,50}},
/* 19587 */ {(16<<2)|3,{116,100,97}},
/* 19588 */ {(16<<2)|3,{116,100,99}},
/* 19589 */ {(16<<2)|3,{116,100,101}},
/* 19590 */ {(16<<2)|3,{116,100,105}},
/* 19591 */ {(16<<2)|3,{116,100,111}},
/* 19592 */ {(16<<2)|3,{116,100,115}},
/* 19593 */ {(16<<2)|3,{116,100,116}},
/* 19594 */ {(11<<2)|2,{116,100,0}},
/* 19595 */ {(11<<2)|2,{116,100,0}},
/* 19596 */ {(11<<2)|2,{116,100,0}},
/* 19597 */ {(11<<2)|2,{116,100,0}},
/* 19598 */ {(11<<2)|2,{116,100,0}},
/* 19599 */ {(11<<2)|2,{116,100,0}},
/* 19600 */ {(11<<2)|2,{116,100,0}},
/* 19601 */ {(11<<2)|2,{116,100,0}},
/* 19602 */ {(11<<2)|2,{116,100,0}},
/* 19603 */ {(11<<2)|2,{116,100,0}},
/* 19604 */ {(11<<2)|2,{116,100,0}},
/* 19605 */ {(11<<2)|2,{116,100,0}},
/* 19606 */ {(11<<2)|2,{116,100,0}},
/* 19607 */ {(11<<2)|2,{116,100,0}},
/* 19608 */ {(11<<2)|2,{116,100,0}},
/* 19609 */ {(11<<2)|2,{116,100,0}},
/* 19610 */ {(11<<2)|2,{116,100,0}},
/* 19611 */ {(11<<2)|2,{116,100,0}},
/* 19612 */ {(11<<2)|2,{116,100,0}},
/* 19613 */ {(11<<2)|2,{116,100,0}},
/* 19614 */ {(11<<2)|2,{116,100,0}},
/* 19615 */ {(11<<2)|2,{116,100,0}},
/* 19616 */ {(16<<2)|3,{116,102,48}},
/* 19617 */ {(16<<2)|3,{116,102,49}},
/* 19618 */ {(16<<2)|3,{116,102,50}},
/* 19619 */ {(16<<2)|3,{116,102,97}},
/* 19620 */ {(16<<2)|3,{116,102,99}},
/* 19621 */ {(16<<2)|3,{116,102,101}},
/* 19622 */ {(16<<2)|3,{116,102,105}},
/* 19623 */ {(16<<2)|3,{116,102,111}},
/* 19624 */ {(16<<2)|3,{116,102,115}},
/* 19625 */ {(16<<2)|3,{116,102,116}},
/* 19626 */ {(11<<2)|2,{116,102,0}},
/* 19627 */ {(11<<2)|2,{116,102,0}},
/* 19628 */ {(11<<2)|2,{116,102,0}},
/* 19629 */ {(11<<2)|2,{116,102,0}},
/* 19630 */ {(11<<2)|2,{116,102,0}},
/* 19631 */ {(11<<2)|2,{116,102,0}},
/* 19632 */ {(11<<2)|2,{116,102,0}},
/* 19633 */ {(11<<2)|2,{116,102,0}},
/* 19634 */ {(11<<2)|2,{116,102,0}},
/* 19635 */ {(11<<2)|2,{116,102,0}},
/* 19636 */ {(11<<2)|2,{116,102,0}},
/* 19637 */ {(11<<2)|2,{116,102,0}},
/* 19638 */ {(11<<2)|2,{116,102,0}},
/* 19639 */ {(11<<2)|2,{116,102,0}},
/* 19640 */ {(11<<2)|2,{116,102,0}},
/* 19641 */ {(11<<2)|2,{116,102,0}},
/* 19642 */ {(11<<2)|2,{116,102,0}},
/* 19643 */ {(11<<2)|2,{116,102,0}},
/* 19644 */ {(11<<2)|2,{116,102,0}},
/* 19645 */ {(11<<2)|2,{116,102,0}},
/* 19646 */ {(11<<2)|2,{116,102,0}},
/* 19647 */ {(11<<2)|2,{116,102,0}},
/* 19648 */ {(16<<2)|3,{116,103,48}},
/* 19649 */ {(16<<2)|3,{116,103,49}},
/* 19650 */ {(16<<2)|3,{116,103,50}},
/* 19651 */ {(16<<2)|3,{116,103,97}},
/* 19652 */ {(16<<2)|3,{116,103,99}},
/* 19653 */ {(16<<2)|3,{116,103,101}},
/* 19654 */ {(16<<2)|3,{116,103,105}},
/* 19655 */ {(16<<2)|3,{116,103,111}},
/* 19656 */ {(16<<2)|3,{116,103,115}},
/* 19657 */ {(16<<2)|3,{116,103,116}},
/* 19658 */ {(11<<2)|2,{116,103,0}},
/* 19659 */ {(11<<2)|2,{116,103,0}},
/* 19660 */ {(11<<2)|2,{116,103,0}},
/* 19661 */ {(11<<2)|2,{116,103,0}},
/* 19662 */ {(11<<2)|2,{116,103,0}},
/* 19663 */ {(11<<2)|2,{116,103,0}},
/* 19664 */ {(11<<2)|2,{116,103,0}},
/* 19665 */ {(11<<2)|2,{116,103,0}},
/* 19666 */ {(11<<2)|2,{116,103,0}},
/* 19667 */ {(11<<2)|2,{116,103,0}},
/* 19668 */ {(11<<2)|2,{116,103,0}},
/* 19669 */ {(11<<2)|2,{116,103,0}},
/* 19670 */ {(11<<2)|2,{116,103,0}},
/* 19671 */ {(11<<2)|2,{116,103,0}},
/* 19672 */ {(11<<2)|2,{116,103,0}},
/* 19673 */ {(11<<2)|2,{116,103,0}},
/* 19674 */ {(11<<2)|2,{116,103,0}},
/* 19675 */ {(11<<2)|2,{116,103,0}},
/* 19676 */ {(11<<2)|2,{116,103,0}},
/* 19677 */ {(11<<2)|2,{116,103,0}},
/* 19678 */ {(11<<2)|2,{116,103,0}},
/* 19679 */ {(11<<2)|2,{116,103,0}},
/* 19680 */ {(16<<2)|3,{116,104,48}},
/* 19681 */ {(16<<2)|3,{116,104,49}},
/* 19682 */ {(16<<2)|3,{116,104,50}},
/* 19683 */ {(16<<2)|3,{116,104,97}},
/* 19684 */ {(16<<2)|3,{116,104,99}},
/* 19685 */ {(16<<2)|3,{116,104,101}},
/* 19686 */ {(16<<2)|3,{116,104,105}},
/* 19687 */ {(16<<2)|3,{116,104,111}},
/* 19688 */ {(16<<2)|3,{116,104,115}},
/* 19689 */ {(16<<2)|3,{116,104,116}},
/* 19690 */ {(11<<2)|2,{116,104,0}},
/* 19691 */ {(11<<2)|2,{116,104,0}},
/* 19692 */ {(11<<2)|2,{116,104,0}},
/* 19693 */ {(11<<2)|2,{116,104,0}},
/* 19694 */ {(11<<2)|2,{116,104,0}},
/* 19695 */ {(11<<2)|2,{116,104,0}},
/* 19696 */ {(11<<2)|2,{116,104,0}},
/* 19697 */ {(11<<2)|2,{116,104,0}},
/* 19698 */ {(11<<2)|2,{116,104,0}},
/* 19699 */ {(11<<2)|2,{116,104,0}},
/* 19700 */ {(11<<2)|2,{116,104,0}},
/* 19701 */ {(11<<2)|2,{116,104,0}},
/* 19702 */ {(11<<2)|2,{116,104,0}},
/* 19703 */ {(11<<2)|2,{116,104,0}},
/* 19704 */ {(11<<2)|2,{116,104,0}},
/* 19705 */ {(11<<2)|2,{116,104,0}},
/* 19706 */ {(11<<2)|2,{116,104,0}},
/* 19707 */ {(11<<2)|2,{116,104,0}},
/* 19708 */ {(11<<2)|2,{116,104,0}},
/* 19709 */ {(11<<2)|2,{116,104,0}},
/* 19710 */ {(11<<2)|2,{116,104,0}},
/* 19711 */ {(11<<2)|2,{116,104,0}},
/* 19712 */ {(16<<2)|3,{116,108,48}},
/* 19713 */ {(16<<2)|3,{116,108,49}},
/* 19714 */ {(16<<2)|3,{116,108,50}},
/* 19715 */ {(16<<2)|3,{116,108,97}},
/* 19716 */ {(16<<2)|3,{116,108,99}},
/* 19717 */ {(16<<2)|3,{116,108,101}},
/* 19718 */ {(16<<2)|3,{116,108,105}},
/* 19719 */ {(16<<2)|3,{116,108,111}},
/* 19720 */ {(16<<2)|3,{116,108,115}},
/* 19721 */ {(16<<2)|3,{116,108,116}},
/* 19722 */ {(11<<2)|2,{116,108,0}},
/* 19723 */ {(11<<2)|2,{116,108,0}},
/* 19724 */ {(11<<2)|2,{116,108,0}},
/* 19725 */ {(11<<2)|2,{116,108,0}},
/* 19726 */ {(11<<2)|2,{116,108,0}},
/* 19727 */ {(11<<2)|2,{116,108,0}},
/* 19728 */ {(11<<2)|2,{116,108,0}},
/* 19729 */ {(11<<2)|2,{116,108,0}},
/* 19730 */ {(11<<2)|2,{116,108,0}},
/* 19731 */ {(11<<2)|2,{116,108,0}},
/* 19732 */ {(11<<2)|2,{116,108,0}},
/* 19733 */ {(11<<2)|2,{116,108,0}},
/* 19734 */ {(11<<2)|2,{116,108,0}},
/* 19735 */ {(11<<2)|2,{116,108,0}},
/* 19736 */ {(11<<2)|2,{116,108,0}},
/* 19737 */ {(11<<2)|2,{116,108,0}},
/* 19738 */ {(11<<2)|2,{116,108,0}},
/* 19739 */ {(11<<2)|2,{116,108,0}},
/* 19740 */ {(11<<2)|2,{116,108,0}},
/* 19741 */ {(11<<2)|2,{116,108,0}},
/* 19742 */ {(11<<2)|2,{116,108,0}},
/* 19743 */ {(11<<2)|2,{116,108,0}},
/* 19744 */ {(16<<2)|3,{116,109,48}},
/* 19745 */ {(16<<2)|3,{116,109,49}},
/* 19746 */ {(16<<2)|3,{116,109,50}},
/* 19747 */ {(16<<2)|3,{116,109,97}},
/* 19748 */ {(16<<2)|3,{116,109,99}},
/* 19749 */ {(16<<2)|3,{116,109,101}},
/* 19750 */ {(16<<2)|3,{116,109,105}},
/* 19751 */ {(16<<2)|3,{116,109,111}},
/* 19752 */ {(16<<2)|3,{116,109,115}},
/* 19753 */ {(16<<2)|3,{116,109,116}},
/* 19754 */ {(11<<2)|2,{116,109,0}},
/* 19755 */ {(11<<2)|2,{116,109,0}},
/* 19756 */ {(11<<2)|2,{116,109,0}},
/* 19757 */ {(11<<2)|2,{116,109,0}},
/* 19758 */ {(11<<2)|2,{116,109,0}},
/* 19759 */ {(11<<2)|2,{116,109,0}},
/* 19760 */ {(11<<2)|2,{116,109,0}},
/* 19761 */ {(11<<2)|2,{116,109,0}},
/* 19762 */ {(11<<2)|2,{116,109,0}},
/* 19763 */ {(11<<2)|2,{116,109,0}},
/* 19764 */ {(11<<2)|2,{116,109,0}},
/* 19765 */ {(11<<2)|2,{116,109,0}},
/* 19766 */ {(11<<2)|2,{116,109,0}},
/* 19767 */ {(11<<2)|2,{116,109,0}},
/* 19768 */ {(11<<2)|2,{116,109,0}},
/* 19769 */ {(11<<2)|2,{116,109,0}},
/* 19770 */ {(11<<2)|2,{116,109,0}},
/* 19771 */ {(11<<2)|2,{116,109,0}},
/* 19772 */ {(11<<2)|2,{116,109,0}},
/* 19773 */ {(11<<2)|2,{116,109,0}},
/* 19774 */ {(11<<2)|2,{116,109,0}},
/* 19775 */ {(11<<2)|2,{116,109,0}},
/* 19776 */ {(16<<2)|3,{116,110,48}},
/* 19777 */ {(16<<2)|3,{116,110,49}},
/* 19778 */ {(16<<2)|3,{116,110,50}},
/* 19779 */ {(16<<2)|3,{116,110,97}},
/* 19780 */ {(16<<2)|3,{116,110,99}},
/* 19781 */ {(16<<2)|3,{116,110,101}},
/* 19782 */ {(16<<2)|3,{116,110,105}},
/* 19783 */ {(16<<2)|3,{116,110,111}},
/* 19784 */ {(16<<2)|3,{116,110,115}},
/* 19785 */ {(16<<2)|3,{116,110,116}},
/* 19786 */ {(11<<2)|2,{116,110,0}},
/* 19787 */ {(11<<2)|2,{116,110,0}},
/* 19788 */ {(11<<2)|2,{116,110,0}},
/* 19789 */ {(11<<2)|2,{116,110,0}},
/* 19790 */ {(11<<2)|2,{116,110,0}},
/* 19791 */ {(11<<2)|2,{116,110,0}},
/* 19792 */ {(11<<2)|2,{116,110,0}},
/* 19793 */ {(11<<2)|2,{116,110,0}},
/* 19794 */ {(11<<2)|2,{116,110,0}},
/* 19795 */ {(11<<2)|2,{116,110,0}},
/* 19796 */ {(11<<2)|2,{116,110,0}},
/* 19797 */ {(11<<2)|2,{116,110,0}},
/* 19798 */ {(11<<2)|2,{116,110,0}},
/* 19799 */ {(11<<2)|2,{116,110,0}},
/* 19800 */ {(11<<2)|2,{116,110,0}},
/* 19801 */ {(11<<2)|2,{116,110,0}},
/* 19802 */ {(11<<2)|2,{116,110,0}},
/* 19803 */ {(11<<2)|2,{116,110,0}},
/* 19804 */ {(11<<2)|2,{116,110,0}},
/* 19805 */ {(11<<2)|2,{116,110,0}},
/* 19806 */ {(11<<2)|2,{116,110,0}},
/* 19807 */ {(11<<2)|2,{116,110,0}},
/* 19808 */ {(16<<2)|3,{116,112,48}},
/* 19809 */ {(16<<2)|3,{116,112,49}},
/* 19810 */ {(16<<2)|3,{116,112,50}},
/* 19811 */ {(16<<2)|3,{116,112,97}},
/* 19812 */ {(16<<2)|3,{116,112,99}},
/* 19813 */ {(16<<2)|3,{116,112,101}},
/* 19814 */ {(16<<2)|3,{116,112,105}},
/* 19815 */ {(16<<2)|3,{116,112,111}},
/* 19816 */ {(16<<2)|3,{116,112,115}},
/* 19817 */ {(16<<2)|3,{116,112,116}},
/* 19818 */ {(11<<2)|2,{116,112,0}},
/* 19819 */ {(11<<2)|2,{116,112,0}},
/* 19820 */ {(11<<2)|2,{116,112,0}},
/* 19821 */ {(11<<2)|2,{116,112,0}},
/* 19822 */ {(11<<2)|2,{116,112,0}},
/* 19823 */ {(11<<2)|2,{116,112,0}},
/* 19824 */ {(11<<2)|2,{116,112,0}},
/* 19825 */ {(11<<2)|2,{116,112,0}},
/* 19826 */ {(11<<2)|2,{116,112,0}},
/* 19827 */ {(11<<2)|2,{116,112,0}},
/* 19828 */ {(11<<2)|2,{116,112,0}},
/* 19829 */ {(11<<2)|2,{116,112,0}},
/* 19830 */ {(11<<2)|2,{116,112,0}},
/* 19831 */ {(11<<2)|2,{116,112,0}},
/* 19832 */ {(11<<2)|2,{116,112,0}},
/* 19833 */ {(11<<2)|2,{116,112,0}},
/* 19834 */ {(11<<2)|2,{116,112,0}},
/* 19835 */ {(11<<2)|2,{116,112,0}},
/* 19836 */ {(11<<2)|2,{116,112,0}},
/* 19837 */ {(11<<2)|2,{116,112,0}},
/* 19838 */ {(11<<2)|2,{116,112,0}},
/* 19839 */ {(11<<2)|2,{116,112,0}},
/* 19840 */ {(16<<2)|3,{116,114,48}},
/* 19841 */ {(16<<2)|3,{116,114,49}},
/* 19842 */ {(16<<2)|3,{116,114,50}},
/* 19843 */ {(16<<2)|3,{116,114,97}},
/* 19844 */ {(16<<2)|3,{116,114,99}},
/* 19845 */ {(16<<2)|3,{116,114,101}},
/* 19846 */ {(16<<2)|3,{116,114,105}},
/* 19847 */ {(16<<2)|3,{116,114,111}},
/* 19848 */ {(16<<2)|3,{116,114,115}},
/* 19849 */ {(16<<2)|3,{116,114,116}},
/* 19850 */ {(11<<2)|2,{116,114,0}},
/* 19851 */ {(11<<2)|2,{116,114,0}},
/* 19852 */ {(11<<2)|2,{116,114,0}},
/* 19853 */ {(11<<2)|2,{116,114,0}},
/* 19854 */ {(11<<2)|2,{116,114,0}},
/* 19855 */ {(11<<2)|2,{116,114,0}},
/* 19856 */ {(11<<2)|2,{116,114,0}},
/* 19857 */ {(11<<2)|2,{116,114,0}},
/* 19858 */ {(11<<2)|2,{116,114,0}},
/* 19859 */ {(11<<2)|2,{116,114,0}},
/* 19860 */ {(11<<2)|2,{116,114,0}},
/* 19861 */ {(11<<2)|2,{116,114,0}},
/* 19862 */ {(11<<2)|2,{116,114,0}},
/* 19863 */ {(11<<2)|2,{116,114,0}},
/* 19864 */ {(11<<2)|2,{116,114,0}},
/* 19865 */ {(11<<2)|2,{116,114,0}},
/* 19866 */ {(11<<2)|2,{116,114,0}},
/* 19867 */ {(11<<2)|2,{116,114,0}},
/* 19868 */ {(11<<2)|2,{116,114,0}},
/* 19869 */ {(11<<2)|2,{116,114,0}},
/* 19870 */ {(11<<2)|2,{116,114,0}},
/* 19871 */ {(11<<2)|2,{116,114,0}},
/* 19872 */ {(16<<2)|3,{116,117,48}},
/* 19873 */ {(16<<2)|3,{116,117,49}},
/* 19874 */ {(16<<2)|3,{116,117,50}},
/* 19875 */ {(16<<2)|3,{116,117,97}},
/* 19876 */ {(16<<2)|3,{116,117,99}},
/* 19877 */ {(16<<2)|3,{116,117,101}},
/* 19878 */ {(16<<2)|3,{116,117,105}},
/* 19879 */ {(16<<2)|3,{116,117,111}},
/* 19880 */ {(16<<2)|3,{116,117,115}},
/* 19881 */ {(16<<2)|3,{116,117,116}},
/* 19882 */ {(11<<2)|2,{116,117,0}},
/* 19883 */ {(11<<2)|2,{116,117,0}},
/* 19884 */ {(11<<2)|2,{116,117,0}},
/* 19885 */ {(11<<2)|2,{116,117,0}},
/* 19886 */ {(11<<2)|2,{116,117,0}},
/* 19887 */ {(11<<2)|2,{116,117,0}},
/* 19888 */ {(11<<2)|2,{116,117,0}},
/* 19889 */ {(11<<2)|2,{116,117,0}},
/* 19890 */ {(11<<2)|2,{116,117,0}},
/* 19891 */ {(11<<2)|2,{116,117,0}},
/* 19892 */ {(11<<2)|2,{116,117,0}},
/* 19893 */ {(11<<2)|2,{116,117,0}},
/* 19894 */ {(11<<2)|2,{116,117,0}},
/* 19895 */ {(11<<2)|2,{116,117,0}},
/* 19896 */ {(11<<2)|2,{116,117,0}},
/* 19897 */ {(11<<2)|2,{116,117,0}},
/* 19898 */ {(11<<2)|2,{116,117,0}},
/* 19899 */ {(11<<2)|2,{116,117,0}},
/* 19900 */ {(11<<2)|2,{116,117,0}},
/* 19901 */ {(11<<2)|2,{116,117,0}},
/* 19902 */ {(11<<2)|2,{116,117,0}},
/* 19903 */ {(11<<2)|2,{116,117,0}},
/* 19904 */ {(12<<2)|2,{116,58,0}},
/* 19905 */ {(12<<2)|2,{116,58,0}},
/* 19906 */ {(12<<2)|2,{116,58,0}},
/* 19907 */ {(12<<2)|2,{116,58,0}},
/* 19908 */ {(12<<2)|2,{116,58,0}},
/* 19909 */ {(12<<2)|2,{116,58,0}},
/* 19910 */ {(12<<2)|2,{116,58,0}},
/* 19911 */ {(12<<2)|2,{116,58,0}},
/* 19912 */ {(12<<2)|2,{116,58,0}},
/* 19913 */ {(12<<2)|2,{116,58,0}},
/* 19914 */ {(12<<2)|2,{116,58,0}},
/* 19915 */ {(12<<2)|2,{116,58,0}},
/* 19916 */ {(12<<2)|2,{116,58,0}},
/* 19917 */ {(12<<2)|2,{116,58,0}},
/* 19918 */ {(12<<2)|2,{116,58,0}},
/* 19919 */ {(12<<2)|2,{116,58,0}},
/* 19920 */ {(12<<2)|2,{116,66,0}},
/* 19921 */ {(12<<2)|2,{116,66,0}},
/* 19922 */ {(12<<2)|2,{116,66,0}},
/* 19923 */ {(12<<2)|2,{116,66,0}},
/* 19924 */ {(12<<2)|2,{116,66,0}},
/* 19925 */ {(12<<2)|2,{116,66,0}},
/* 19926 */ {(12<<2)|2,{116,66,0}},
/* 19927 */ {(12<<2)|2,{116,66,0}},
/* 19928 */ {(12<<2)|2,{116,66,0}},
/* 19929 */ {(12<<2)|2,{116,66,0}},
/* 19930 */ {(12<<2)|2,{116,66,0}},
/* 19931 */ {(12<<2)|2,{116,66,0}},
/* 19932 */ {(12<<2)|2,{116,66,0}},
/* 19933 */ {(12<<2)|2,{116,66,0}},
/* 19934 */ {(12<<2)|2,{116,66,0}},
/* 19935 */ {(12<<2)|2,{116,66,0}},
/* 19936 */ {(12<<2)|2,{116,67,0}},
/* 19937 */ {(12<<2)|2,{116,67,0}},
/* 19938 */ {(12<<2)|2,{116,67,0}},
/* 19939 */ {(12<<2)|2,{116,67,0}},
/* 19940 */ {(12<<2)|2,{116,67,0}},
/* 19941 */ {(12<<2)|2,{116,67,0}},
/* 19942 */ {(12<<2)|2,{116,67,0}},
/* 19943 */ {(12<<2)|2,{116,67,0}},
/* 19944 */ {(12<<2)|2,{116,67,0}},
/* 19945 */ {(12<<2)|2,{116,67,0}},
/* 19946 */ {(12<<2)|2,{116,67,0}},
/* 19947 */ {(12<<2)|2,{116,67,0}},
/* 19948 */ {(12<<2)|2,{116,67,0}},
/* 19949 */ {(12<<2)|2,{116,67,0}},
/* 19950 */ {(12<<2)|2,{116,67,0}},
/* 19951 */ {(12<<2)|2,{116,67,0}},
/* 19952 */ {(12<<2)|2,{116,68,0}},
/* 19953 */ {(12<<2)|2,{116,68,0}},
/* 19954 */ {(12<<2)|2,{116,68,0}},
/* 19955 */ {(12<<2)|2,{116,68,0}},
/* 19956 */ {(12<<2)|2,{116,68,0}},
/* 19957 */ {(12<<2)|2,{116,68,0}},
/* 19958 */ {(12<<2)|2,{116,68,0}},
/* 19959 */ {(12<<2)|2,{116,68,0}},
/* 19960 */ {(12<<2)|2,{116,68,0}},
/* 19961 */ {(12<<2)|2,{116,68,0}},
/* 19962 */ {(12<<2)|2,{116,68,0}},
/* 19963 */ {(12<<2)|2,{116,68,0}},
/* 19964 */ {(12<<2)|2,{116,68,0}},
/* 19965 */ {(12<<2)|2,{116,68,0}},
/* 19966 */ {(12<<2)|2,{116,68,0}},
/* 19967 */ {(12<<2)|2,{116,68,0}},
/* 19968 */ {(12<<2)|2,{116,69,0}},
/* 19969 */ {(12<<2)|2,{116,69,0}},
/* 19970 */ {(12<<2)|2,{116,69,0}},
/* 19971 */ {(12<<2)|2,{116,69,0}},
/* 19972 */ {(12<<2)|2,{116,69,0}},
/* 19973 */ {(12<<2)|2,{116,69,0}},
/* 19974 */ {(12<<2)|2,{116,69,0}},
/* 19975 */ {(12<<2)|2,{116,69,0}},
/* 19976 */ {(12<<2)|2,{116,69,0}},
/* 19977 */ {(12<<2)|2,{116,69,0}},
/* 19978 */ {(12<<2)|2,{116,69,0}},
/* 19979 */ {(12<<2)|2,{116,69,0}},
/* 19980 */ {(12<<2)|2,{116,69,0}},
/* 19981 */ {(12<<2)|2,{116,69,0}},
/* 19982 */ {(12<<2)|2,{116,69,0}},
/* 19983 */ {(12<<2)|2,{116,69,0}},
/* 19984 */ {(12<<2)|2,{116,70,0}},
/* 19985 */ {(12<<2)|2,{116,70,0}},
/* 19986 */ {(12<<2)|2,{116,70,0}},
/* 19987 */ {(12<<2)|2,{116,70,0}},
/* 19988 */ {(12<<2)|2,{116,70,0}},
/* 19989 */ {(12<<2)|2,{116,70,0}},
/* 19990 */ {(12<<2)|2,{116,70,0}},
/* 19991 */ {(12<<2)|2,{116,70,0}},
/* 19992 */ {(12<<2)|2,{116,70,0}},
/* 19993 */ {(12<<2)|2,{116,70,0}},
/* 19994 */ {(12<<2)|2,{116,70,0}},
/* 19995 */ {(12<<2)|2,{116,70,0}},
/* 19996 */ {(12<<2)|2,{116,70,0}},
/* 19997 */ {(12<<2)|2,{116,70,0}},
/* 19998 */ {(12<<2)|2,{116,70,0}},
/* 19999 */ {(12<<2)|2,{116,70,0}},
/* 20000 */ {(12<<2)|2,{116,71,0}},
/* 20001 */ {(12<<2)|2,{116,71,0}},
/* 20002 */ {(12<<2)|2,{116,71,0}},
/* 20003 */ {(12<<2)|2,{116,71,0}},
/* 20004 */ {(12<<2)|2,{116,71,0}},
/* 20005 */ {(12<<2)|2,{116,71,0}},
/* 20006 */ {(12<<2)|2,{116,71,0}},
/* 20007 */ {(12<<2)|2,{116,71,0}},
/* 20008 */ {(12<<2)|2,{116,71,0}},
/* 20009 */ {(12<<2)|2,{116,71,0}},
/* 20010 */ {(12<<2)|2,{116,71,0}},
/* 20011 */ {(12<<2)|2,{116,71,0}},
/* 20012 */ {(12<<2)|2,{116,71,0}},
/* 20013 */ {(12<<2)|2,{116,71,0}},
/* 20014 */ {(12<<2)|2,{116,71,0}},
/* 20015 */ {(12<<2)|2,{116,71,0}},
/* 20016 */ {(12<<2)|2,{116,72,0}},
/* 20017 */ {(12<<2)|2,{116,72,0}},
/* 20018 */ {(12<<2)|2,{116,72,0}},
/* 20019 */ {(12<<2)|2,{116,72,0}},
/* 20020 */ {(12<<2)|2,{116,72,0}},
/* 20021 */ {(12<<2)|2,{116,72,0}},
/* 20022 */ {(12<<2)|2,{116,72,0}},
/* 20023 */ {(12<<2)|2,{116,72,0}},
/* 20024 */ {(12<<2)|2,{116,72,0}},
/* 20025 */ {(12<<2)|2,{116,72,0}},
/* 20026 */ {(12<<2)|2,{116,72,0}},
/* 20027 */ {(12<<2)|2,{116,72,0}},
/* 20028 */ {(12<<2)|2,{116,72,0}},
/* 20029 */ {(12<<2)|2,{116,72,0}},
/* 20030 */ {(12<<2)|2,{116,72,0}},
/* 20031 */ {(12<<2)|2,{116,72,0}},
/* 20032 */ {(12<<2)|2,{116,73,0}},
/* 20033 */ {(12<<2)|2,{116,73,0}},
/* 20034 */ {(12<<2)|2,{116,73,0}},
/* 20035 */ {(12<<2)|2,{116,73,0}},
/* 20036 */ {(12<<2)|2,{116,73,0}},
/* 20037 */ {(12<<2)|2,{116,73,0}},
/* 20038 */ {(12<<2)|2,{116,73,0}},
/* 20039 */ {(12<<2)|2,{116,73,0}},
/* 20040 */ {(12<<2)|2,{116,73,0}},
/* 20041 */ {(12<<2)|2,{116,73,0}},
/* 20042 */ {(12<<2)|2,{116,73,0}},
/* 20043 */ {(12<<2)|2,{116,73,0}},
/* 20044 */ {(12<<2)|2,{116,73,0}},
/* 20045 */ {(12<<2)|2,{116,73,0}},
/* 20046 */ {(12<<2)|2,{116,73,0}},
/* 20047 */ {(12<<2)|2,{116,73,0}},
/* 20048 */ {(12<<2)|2,{116,74,0}},
/* 20049 */ {(12<<2)|2,{116,74,0}},
/* 20050 */ {(12<<2)|2,{116,74,0}},
/* 20051 */ {(12<<2)|2,{116,74,0}},
/* 20052 */ {(12<<2)|2,{116,74,0}},
/* 20053 */ {(12<<2)|2,{116,74,0}},
/* 20054 */ {(12<<2)|2,{116,74,0}},
/* 20055 */ {(12<<2)|2,{116,74,0}},
/* 20056 */ {(12<<2)|2,{116,74,0}},
/* 20057 */ {(12<<2)|2,{116,74,0}},
/* 20058 */ {(12<<2)|2,{116,74,0}},
/* 20059 */ {(12<<2)|2,{116,74,0}},
/* 20060 */ {(12<<2)|2,{116,74,0}},
/* 20061 */ {(12<<2)|2,{116,74,0}},
/* 20062 */ {(12<<2)|2,{116,74,0}},
/* 20063 */ {(12<<2)|2,{116,74,0}},
/* 20064 */ {(12<<2)|2,{116,75,0}},
/* 20065 */ {(12<<2)|2,{116,75,0}},
/* 20066 */ {(12<<2)|2,{116,75,0}},
/* 20067 */ {(12<<2)|2,{116,75,0}},
/* 20068 */ {(12<<2)|2,{116,75,0}},
/* 20069 */ {(12<<2)|2,{116,75,0}},
/* 20070 */ {(12<<2)|2,{116,75,0}},
/* 20071 */ {(12<<2)|2,{116,75,0}},
/* 20072 */ {(12<<2)|2,{116,75,0}},
/* 20073 */ {(12<<2)|2,{116,75,0}},
/* 20074 */ {(12<<2)|2,{116,75,0}},
/* 20075 */ {(12<<2)|2,{116,75,0}},
/* 20076 */ {(12<<2)|2,{116,75,0}},
/* 20077 */ {(12<<2)|2,{116,75,0}},
/* 20078 */ {(12<<2)|2,{116,75,0}},
/* 20079 */ {(12<<2)|2,{116,75,0}},
/* 20080 */ {(12<<2)|2,{116,76,0}},
/* 20081 */ {(12<<2)|2,{116,76,0}},
/* 20082 */ {(12<<2)|2,{116,76,0}},
/* 20083 */ {(12<<2)|2,{116,76,0}},
/* 20084 */ {(12<<2)|2,{116,76,0}},
/* 20085 */ {(12<<2)|2,{116,76,0}},
/* 20086 */ {(12<<2)|2,{116,76,0}},
/* 20087 */ {(12<<2)|2,{116,76,0}},
/* 20088 */ {(12<<2)|2,{116,76,0}},
/* 20089 */ {(12<<2)|2,{116,76,0}},
/* 20090 */ {(12<<2)|2,{116,76,0}},
/* 20091 */ {(12<<2)|2,{116,76,0}},
/* 20092 */ {(12<<2)|2,{116,76,0}},
/* 20093 */ {(12<<2)|2,{116,76,0}},
/* 20094 */ {(12<<2)|2,{116,76,0}},
/* 20095 */ {(12<<2)|2,{116,76,0}},
/* 20096 */ {(12<<2)|2,{116,77,0}},
/* 20097 */ {(12<<2)|2,{116,77,0}},
/* 20098 */ {(12<<2)|2,{116,77,0}},
/* 20099 */ {(12<<2)|2,{116,77,0}},
/* 20100 */ {(12<<2)|2,{116,77,0}},
/* 20101 */ {(12<<2)|2,{116,77,0}},
/* 20102 */ {(12<<2)|2,{116,77,0}},
/* 20103 */ {(12<<2)|2,{116,77,0}},
/* 20104 */ {(12<<2)|2,{116,77,0}},
/* 20105 */ {(12<<2)|2,{116,77,0}},
/* 20106 */ {(12<<2)|2,{116,77,0}},
/* 20107 */ {(12<<2)|2,{116,77,0}},
/* 20108 */ {(12<<2)|2,{116,77,0}},
/* 20109 */ {(12<<2)|2,{116,77,0}},
/* 20110 */ {(12<<2)|2,{116,77,0}},
/* 20111 */ {(12<<2)|2,{116,77,0}},
/* 20112 */ {(12<<2)|2,{116,78,0}},
/* 20113 */ {(12<<2)|2,{116,78,0}},
/* 20114 */ {(12<<2)|2,{116,78,0}},
/* 20115 */ {(12<<2)|2,{116,78,0}},
/* 20116 */ {(12<<2)|2,{116,78,0}},
/* 20117 */ {(12<<2)|2,{116,78,0}},
/* 20118 */ {(12<<2)|2,{116,78,0}},
/* 20119 */ {(12<<2)|2,{116,78,0}},
/* 20120 */ {(12<<2)|2,{116,78,0}},
/* 20121 */ {(12<<2)|2,{116,78,0}},
/* 20122 */ {(12<<2)|2,{116,78,0}},
/* 20123 */ {(12<<2)|2,{116,78,0}},
/* 20124 */ {(12<<2)|2,{116,78,0}},
/* 20125 */ {(12<<2)|2,{116,78,0}},
/* 20126 */ {(12<<2)|2,{116,78,0}},
/* 20127 */ {(12<<2)|2,{116,78,0}},
/* 20128 */ {(12<<2)|2,{116,79,0}},
/* 20129 */ {(12<<2)|2,{116,79,0}},
/* 20130 */ {(12<<2)|2,{116,79,0}},
/* 20131 */ {(12<<2)|2,{116,79,0}},
/* 20132 */ {(12<<2)|2,{116,79,0}},
/* 20133 */ {(12<<2)|2,{116,79,0}},
/* 20134 */ {(12<<2)|2,{116,79,0}},
/* 20135 */ {(12<<2)|2,{116,79,0}},
/* 20136 */ {(12<<2)|2,{116,79,0}},
/* 20137 */ {(12<<2)|2,{116,79,0}},
/* 20138 */ {(12<<2)|2,{116,79,0}},
/* 20139 */ {(12<<2)|2,{116,79,0}},
/* 20140 */ {(12<<2)|2,{116,79,0}},
/* 20141 */ {(12<<2)|2,{116,79,0}},
/* 20142 */ {(12<<2)|2,{116,79,0}},
/* 20143 */ {(12<<2)|2,{116,79,0}},
/* 20144 */ {(12<<2)|2,{116,80,0}},
/* 20145 */ {(12<<2)|2,{116,80,0}},
/* 20146 */ {(12<<2)|2,{116,80,0}},
/* 20147 */ {(12<<2)|2,{116,80,0}},
/* 20148 */ {(12<<2)|2,{116,80,0}},
/* 20149 */ {(12<<2)|2,{116,80,0}},
/* 20150 */ {(12<<2)|2,{116,80,0}},
/* 20151 */ {(12<<2)|2,{116,80,0}},
/* 20152 */ {(12<<2)|2,{116,80,0}},
/* 20153 */ {(12<<2)|2,{116,80,0}},
/* 20154 */ {(12<<2)|2,{116,80,0}},
/* 20155 */ {(12<<2)|2,{116,80,0}},
/* 20156 */ {(12<<2)|2,{116,80,0}},
/* 20157 */ {(12<<2)|2,{116,80,0}},
/* 20158 */ {(12<<2)|2,{116,80,0}},
/* 20159 */ {(12<<2)|2,{116,80,0}},
/* 20160 */ {(12<<2)|2,{116,81,0}},
/* 20161 */ {(12<<2)|2,{116,81,0}},
/* 20162 */ {(12<<2)|2,{116,81,0}},
/* 20163 */ {(12<<2)|2,{116,81,0}},
/* 20164 */ {(12<<2)|2,{116,81,0}},
/* 20165 */ {(12<<2)|2,{116,81,0}},
/* 20166 */ {(12<<2)|2,{116,81,0}},
/* 20167 */ {(12<<2)|2,{116,81,0}},
/* 20168 */ {(12<<2)|2,{116,81,0}},
/* 20169 */ {(12<<2)|2,{116,81,0}},
/* 20170 */ {(12<<2)|2,{116,81,0}},
/* 20171 */ {(12<<2)|2,{116,81,0}},
/* 20172 */ {(12<<2)|2,{116,81,0}},
/* 20173 */ {(12<<2)|2,{116,81,0}},
/* 20174 */ {(12<<2)|2,{116,81,0}},
/* 20175 */ {(12<<2)|2,{116,81,0}},
/* 20176 */ {(12<<2)|2,{116,82,0}},
/* 20177 */ {(12<<2)|2,{116,82,0}},
/* 20178 */ {(12<<2)|2,{116,82,0}},
/* 20179 */ {(12<<2)|2,{116,82,0}},
/* 20180 */ {(12<<2)|2,{116,82,0}},
/* 20181 */ {(12<<2)|2,{116,82,0}},
/* 20182 */ {(12<<2)|2,{116,82,0}},
/* 20183 */ {(12<<2)|2,{116,82,0}},
/* 20184 */ {(12<<2)|2,{116,82,0}},
/* 20185 */ {(12<<2)|2,{116,82,0}},
/* 20186 */ {(12<<2)|2,{116,82,0}},
/* 20187 */ {(12<<2)|2,{116,82,0}},
/* 20188 */ {(12<<2)|2,{116,82,0}},
/* 20189 */ {(12<<2)|2,{116,82,0}},
/* 20190 */ {(12<<2)|2,{116,82,0}},
/* 20191 */ {(12<<2)|2,{116,82,0}},
/* 20192 */ {(12<<2)|2,{116,83,0}},
/* 20193 */ {(12<<2)|2,{116,83,0}},
/* 20194 */ {(12<<2)|2,{116,83,0}},
/* 20195 */ {(12<<2)|2,{116,83,0}},
/* 20196 */ {(12<<2)|2,{116,83,0}},
/* 20197 */ {(12<<2)|2,{116,83,0}},
/* 20198 */ {(12<<2)|2,{116,83,0}},
/* 20199 */ {(12<<2)|2,{116,83,0}},
/* 20200 */ {(12<<2)|2,{116,83,0}},
/* 20201 */ {(12<<2)|2,{116,83,0}},
/* 20202 */ {(12<<2)|2,{116,83,0}},
/* 20203 */ {(12<<2)|2,{116,83,0}},
/* 20204 */ {(12<<2)|2,{116,83,0}},
/* 20205 */ {(12<<2)|2,{116,83,0}},
/* 20206 */ {(12<<2)|2,{116,83,0}},
/* 20207 */ {(12<<2)|2,{116,83,0}},
/* 20208 */ {(12<<2)|2,{116,84,0}},
/* 20209 */ {(12<<2)|2,{116,84,0}},
/* 20210 */ {(12<<2)|2,{116,84,0}},
/* 20211 */ {(12<<2)|2,{116,84,0}},
/* 20212 */ {(12<<2)|2,{116,84,0}},
/* 20213 */ {(12<<2)|2,{116,84,0}},
/* 20214 */ {(12<<2)|2,{116,84,0}},
/* 20215 */ {(12<<2)|2,{116,84,0}},
/* 20216 */ {(12<<2)|2,{116,84,0}},
/* 20217 */ {(12<<2)|2,{116,84,0}},
/* 20218 */ {(12<<2)|2,{116,84,0}},
/* 20219 */ {(12<<2)|2,{116,84,0}},
/* 20220 */ {(12<<2)|2,{116,84,0}},
/* 20221 */ {(12<<2)|2,{116,84,0}},
/* 20222 */ {(12<<2)|2,{116,84,0}},
/* 20223 */ {(12<<2)|2,{116,84,0}},
/* 20224 */ {(12<<2)|2,{116,85,0}},
/* 20225 */ {(12<<2)|2,{116,85,0}},
/* 20226 */ {(12<<2)|2,{116,85,0}},
/* 20227 */ {(12<<2)|2,{116,85,0}},
/* 20228 */ {(12<<2)|2,{116,85,0}},
/* 20229 */ {(12<<2)|2,{116,85,0}},
/* 20230 */ {(12<<2)|2,{116,85,0}},
/* 20231 */ {(12<<2)|2,{116,85,0}},
/* 20232 */ {(12<<2)|2,{116,85,0}},
/* 20233 */ {(12<<2)|2,{116,85,0}},
/* 20234 */ {(12<<2)|2,{116,85,0}},
/* 20235 */ {(12<<2)|2,{116,85,0}},
/* 20236 */ {(12<<2)|2,{116,85,0}},
/* 20237 */ {(12<<2)|2,{116,85,0}},
/* 20238 */ {(12<<2)|2,{116,85,0}},
/* 20239 */ {(12<<2)|2,{116,85,0}},
/* 20240 */ {(12<<2)|2,{116,86,0}},
/* 20241 */ {(12<<2)|2,{116,86,0}},
/* 20242 */ {(12<<2)|2,{116,86,0}},
/* 20243 */ {(12<<2)|2,{116,86,0}},
/* 20244 */ {(12<<2)|2,{116,86,0}},
/* 20245 */ {(12<<2)|2,{116,86,0}},
/* 20246 */ {(12<<2)|2,{116,86,0}},
/* 20247 */ {(12<<2)|2,{116,86,0}},
/* 20248 */ {(12<<2)|2,{116,86,0}},
/* 20249 */ {(12<<2)|2,{116,86,0}},
/* 20250 */ {(12<<2)|2,{116,86,0}},
/* 20251 */ {(12<<2)|2,{116,86,0}},
/* 20252 */ {(12<<2)|2,{116,86,0}},
/* 20253 */ {(12<<2)|2,{116,86,0}},
/* 20254 */ {(12<<2)|2,{116,86,0}},
/* 20255 */ {(12<<2)|2,{116,86,0}},
/* 20256 */ {(12<<2)|2,{116,87,0}},
/* 20257 */ {(12<<2)|2,{116,87,0}},
/* 20258 */ {(12<<2)|2,{116,87,0}},
/* 20259 */ {(12<<2)|2,{116,87,0}},
/* 20260 */ {(12<<2)|2,{116,87,0}},
/* 20261 */ {(12<<2)|2,{116,87,0}},
/* 20262 */ {(12<<2)|2,{116,87,0}},
/* 20263 */ {(12<<2)|2,{116,87,0}},
/* 20264 */ {(12<<2)|2,{116,87,0}},
/* 20265 */ {(12<<2)|2,{116,87,0}},
/* 20266 */ {(12<<2)|2,{116,87,0}},
/* 20267 */ {(12<<2)|2,{116,87,0}},
/* 20268 */ {(12<<2)|2,{116,87,0}},
/* 20269 */ {(12<<2)|2,{116,87,0}},
/* 20270 */ {(12<<2)|2,{116,87,0}},
/* 20271 */ {(12<<2)|2,{116,87,0}},
/* 20272 */ {(12<<2)|2,{116,89,0}},
/* 20273 */ {(12<<2)|2,{116,89,0}},
/* 20274 */ {(12<<2)|2,{116,89,0}},
/* 20275 */ {(12<<2)|2,{116,89,0}},
/* 20276 */ {(12<<2)|2,{116,89,0}},
/* 20277 */ {(12<<2)|2,{116,89,0}},
/* 20278 */ {(12<<2)|2,{116,89,0}},
/* 20279 */ {(12<<2)|2,{116,89,0}},
/* 20280 */ {(12<<2)|2,{116,89,0}},
/* 20281 */ {(12<<2)|2,{116,89,0}},
/* 20282 */ {(12<<2)|2,{116,89,0}},
/* 20283 */ {(12<<2)|2,{116,89,0}},
/* 20284 */ {(12<<2)|2,{116,89,0}},
/* 20285 */ {(12<<2)|2,{116,89,0}},
/* 20286 */ {(12<<2)|2,{116,89,0}},
/* 20287 */ {(12<<2)|2,{116,89,0}},
/* 20288 */ {(12<<2)|2,{116,106,0}},
/* 20289 */ {(12<<2)|2,{116,106,0}},
/* 20290 */ {(12<<2)|2,{116,106,0}},
/* 20291 */ {(12<<2)|2,{116,106,0}},
/* 20292 */ {(12<<2)|2,{116,106,0}},
/* 20293 */ {(12<<2)|2,{116,106,0}},
/* 20294 */ {(12<<2)|2,{116,106,0}},
/* 20295 */ {(12<<2)|2,{116,106,0}},
/* 20296 */ {(12<<2)|2,{116,106,0}},
/* 20297 */ {(12<<2)|2,{116,106,0}},
/* 20298 */ {(12<<2)|2,{116,106,0}},
/* 20299 */ {(12<<2)|2,{116,106,0}},
/* 20300 */ {(12<<2)|2,{116,106,0}},
/* 20301 */ {(12<<2)|2,{116,106,0}},
/* 20302 */ {(12<<2)|2,{116,106,0}},
/* 20303 */ {(12<<2)|2,{116,106,0}},
/* 20304 */ {(12<<2)|2,{116,107,0}},
/* 20305 */ {(12<<2)|2,{116,107,0}},
/* 20306 */ {(12<<2)|2,{116,107,0}},
/* 20307 */ {(12<<2)|2,{116,107,0}},
/* 20308 */ {(12<<2)|2,{116,107,0}},
/* 20309 */ {(12<<2)|2,{116,107,0}},
/* 20310 */ {(12<<2)|2,{116,107,0}},
/* 20311 */ {(12<<2)|2,{116,107,0}},
/* 20312 */ {(12<<2)|2,{116,107,0}},
/* 20313 */ {(12<<2)|2,{116,107,0}},
/* 20314 */ {(12<<2)|2,{116,107,0}},
/* 20315 */ {(12<<2)|2,{116,107,0}},
/* 20316 */ {(12<<2)|2,{116,107,0}},
/* 20317 */ {(12<<2)|2,{116,107,0}},
/* 20318 */ {(12<<2)|2,{116,107,0}},
/* 20319 */ {(12<<2)|2,{116,107,0}},
/* 20320 */ {(12<<2)|2,{116,113,0}},
/* 20321 */ {(12<<2)|2,{116,113,0}},
/* 20322 */ {(12<<2)|2,{116,113,0}},
/* 20323 */ {(12<<2)|2,{116,113,0}},
/* 20324 */ {(12<<2)|2,{116,113,0}},
/* 20325 */ {(12<<2)|2,{116,113,0}},
/* 20326 */ {(12<<2)|2,{116,113,0}},
/* 20327 */ {(12<<2)|2,{116,113,0}},
/* 20328 */ {(12<<2)|2,{116,113,0}},
/* 20329 */ {(12<<2)|2,{116,113,0}},
/* 20330 */ {(12<<2)|2,{116,113,0}},
/* 20331 */ {(12<<2)|2,{116,113,0}},
/* 20332 */ {(12<<2)|2,{116,113,0}},
/* 20333 */ {(12<<2)|2,{116,113,0}},
/* 20334 */ {(12<<2)|2,{116,113,0}},
/* 20335 */ {(12<<2)|2,{116,113,0}},
/* 20336 */ {(12<<2)|2,{116,118,0}},
/* 20337 */ {(12<<2)|2,{116,118,0}},
/* 20338 */ {(12<<2)|2,{116,118,0}},
/* 20339 */ {(12<<2)|2,{116,118,0}},
/* 20340 */ {(12<<2)|2,{116,118,0}},
/* 20341 */ {(12<<2)|2,{116,118,0}},
/* 20342 */ {(12<<2)|2,{116,118,0}},
/* 20343 */ {(12<<2)|2,{116,118,0}},
/* 20344 */ {(12<<2)|2,{116,118,0}},
/* 20345 */ {(12<<2)|2,{116,118,0}},
/* 20346 */ {(12<<2)|2,{116,118,0}},
/* 20347 */ {(12<<2)|2,{116,118,0}},
/* 20348 */ {(12<<2)|2,{116,118,0}},
/* 20349 */ {(12<<2)|2,{116,118,0}},
/* 20350 */ {(12<<2)|2,{116,118,0}},
/* 20351 */ {(12<<2)|2,{116,118,0}},
/* 20352 */ {(12<<2)|2,{116,119,0}},
/* 20353 */ {(12<<2)|2,{116,119,0}},
/* 20354 */ {(12<<2)|2,{116,119,0}},
/* 20355 */ {(12<<2)|2,{116,119,0}},
/* 20356 */ {(12<<2)|2,{116,119,0}},
/* 20357 */ {(12<<2)|2,{116,119,0}},
/* 20358 */ {(12<<2)|2,{116,119,0}},
/* 20359 */ {(12<<2)|2,{116,119,0}},
/* 20360 */ {(12<<2)|2,{116,119,0}},
/* 20361 */ {(12<<2)|2,{116,119,0}},
/* 20362 */ {(12<<2)|2,{116,119,0}},
/* 20363 */ {(12<<2)|2,{116,119,0}},
/* 20364 */ {(12<<2)|2,{116,119,0}},
/* 20365 */ {(12<<2)|2,{116,119,0}},
/* 20366 */ {(12<<2)|2,{116,119,0}},
/* 20367 */ {(12<<2)|2,{116,119,0}},
/* 20368 */ {(12<<2)|2,{116,120,0}},
/* 20369 */ {(12<<2)|2,{116,120,0}},
/* 20370 */ {(12<<2)|2,{116,120,0}},
/* 20371 */ {(12<<2)|2,{116,120,0}},
/* 20372 */ {(12<<2)|2,{116,120,0}},
/* 20373 */ {(12<<2)|2,{116,120,0}},
/* 20374 */ {(12<<2)|2,{116,120,0}},
/* 20375 */ {(12<<2)|2,{116,120,0}},
/* 20376 */ {(12<<2)|2,{116,120,0}},
/* 20377 */ {(12<<2)|2,{116,120,0}},
/* 20378 */ {(12<<2)|2,{116,120,0}},
/* 20379 */ {(12<<2)|2,{116,120,0}},
/* 20380 */ {(12<<2)|2,{116,120,0}},
/* 20381 */ {(12<<2)|2,{116,120,0}},
/* 20382 */ {(12<<2)|2,{116,120,0}},
/* 20383 */ {(12<<2)|2,{116,120,0}},
/* 20384 */ {(12<<2)|2,{116,121,0}},
/* 20385 */ {(12<<2)|2,{116,121,0}},
/* 20386 */ {(12<<2)|2,{116,121,0}},
/* 20387 */ {(12<<2)|2,{116,121,0}},
/* 20388 */ {(12<<2)|2,{116,121,0}},
/* 20389 */ {(12<<2)|2,{116,121,0}},
/* 20390 */ {(12<<2)|2,{116,121,0}},
/* 20391 */ {(12<<2)|2,{116,121,0}},
/* 20392 */ {(12<<2)|2,{116,121,0}},
/* 20393 */ {(12<<2)|2,{116,121,0}},
/* 20394 */ {(12<<2)|2,{116,121,0}},
/* 20395 */ {(12<<2)|2,{116,121,0}},
/* 20396 */ {(12<<2)|2,{116,121,0}},
/* 20397 */ {(12<<2)|2,{116,121,0}},
/* 20398 */ {(12<<2)|2,{116,121,0}},
/* 20399 */ {(12<<2)|2,{116,121,0}},
/* 20400 */ {(12<<2)|2,{116,122,0}},
/* 20401 */ {(12<<2)|2,{116,122,0}},
/* 20402 */ {(12<<2)|2,{116,122,0}},
/* 20403 */ {(12<<2)|2,{116,122,0}},
/* 20404 */ {(12<<2)|2,{116,122,0}},
/* 20405 */ {(12<<2)|2,{116,122,0}},
/* 20406 */ {(12<<2)|2,{116,122,0}},
/* 20407 */ {(12<<2)|2,{116,122,0}},
/* 20408 */ {(12<<2)|2,{116,122,0}},
/* 20409 */ {(12<<2)|2,{116,122,0}},
/* 20410 */ {(12<<2)|2,{116,122,0}},
/* 20411 */ {(12<<2)|2,{116,122,0}},
/* 20412 */ {(12<<2)|2,{116,122,0}},
/* 20413 */ {(12<<2)|2,{116,122,0}},
/* 20414 */ {(12<<2)|2,{116,122,0}},
/* 20415 */ {(12<<2)|2,{116,122,0}},
/* 20416 */ {(13<<2)|2,{116,38,0}},
/* 20417 */ {(13<<2)|2,{116,38,0}},
/* 20418 */ {(13<<2)|2,{116,38,0}},
/* 20419 */ {(13<<2)|2,{116,38,0}},
/* 20420 */ {(13<<2)|2,{116,38,0}},
/* 20421 */ {(13<<2)|2,{116,38,0}},
/* 20422 */ {(13<<2)|2,{116,38,0}},
/* 20423 */ {(13<<2)|2,{116,38,0}},
/* 20424 */ {(13<<2)|2,{116,42,0}},
/* 20425 */ {(13<<2)|2,{116,42,0}},
/* 20426 */ {(13<<2)|2,{116,42,0}},
/* 20427 */ {(13<<2)|2,{116,42,0}},
/* 20428 */ {(13<<2)|2,{116,42,0}},
/* 20429 */ {(13<<2)|2,{116,42,0}},
/* 20430 */ {(13<<2)|2,{116,42,0}},
/* 20431 */ {(13<<2)|2,{116,42,0}},
/* 20432 */ {(13<<2)|2,{116,44,0}},
/* 20433 */ {(13<<2)|2,{116,44,0}},
/* 20434 */ {(13<<2)|2,{116,44,0}},
/* 20435 */ {(13<<2)|2,{116,44,0}},
/* 20436 */ {(13<<2)|2,{116,44,0}},
/* 20437 */ {(13<<2)|2,{116,44,0}},
/* 20438 */ {(13<<2)|2,{116,44,0}},
/* 20439 */ {(13<<2)|2,{116,44,0}},
/* 20440 */ {(13<<2)|2,{116,59,0}},
/* 20441 */ {(13<<2)|2,{116,59,0}},
/* 20442 */ {(13<<2)|2,{116,59,0}},
/* 20443 */ {(13<<2)|2,{116,59,0}},
/* 20444 */ {(13<<2)|2,{116,59,0}},
/* 20445 */ {(13<<2)|2,{116,59,0}},
/* 20446 */ {(13<<2)|2,{116,59,0}},
/* 20447 */ {(13<<2)|2,{116,59,0}},
/* 20448 */ {(13<<2)|2,{116,88,0}},
/* 20449 */ {(13<<2)|2,{116,88,0}},
/* 20450 */ {(13<<2)|2,{116,88,0}},
/* 20451 */ {(13<<2)|2,{116,88,0}},
/* 20452 */ {(13<<2)|2,{116,88,0}},
/* 20453 */ {(13<<2)|2,{116,88,0}},
/* 20454 */ {(13<<2)|2,{116,88,0}},
/* 20455 */ {(13<<2)|2,{116,88,0}},
/* 20456 */ {(13<<2)|2,{116,90,0}},
/* 20457 */ {(13<<2)|2,{116,90,0}},
/* 20458 */ {(13<<2)|2,{116,90,0}},
/* 20459 */ {(13<<2)|2,{116,90,0}},
/* 20460 */ {(13<<2)|2,{116,90,0}},
/* 20461 */ {(13<<2)|2,{116,90,0}},
/* 20462 */ {(13<<2)|2,{116,90,0}},
/* 20463 */ {(13<<2)|2,{116,90,0}},
/* 20464 */ {(15<<2)|2,{116,33,0}},
/* 20465 */ {(15<<2)|2,{116,33,0}},
/* 20466 */ {(15<<2)|2,{116,34,0}},
/* 20467 */ {(15<<2)|2,{116,34,0}},
/* 20468 */ {(15<<2)|2,{116,40,0}},
/* 20469 */ {(15<<2)|2,{116,40,0}},
/* 20470 */ {(15<<2)|2,{116,41,0}},
/* 20471 */ {(15<<2)|2,{116,41,0}},
/* 20472 */ {(15<<2)|2,{116,63,0}},
/* 20473 */ {(15<<2)|2,{116,63,0}},
/* 20474 */ {(16<<2)|2,{116,39,0}},
/* 20475 */ {(16<<2)|2,{116,43,0}},
/* 20476 */ {(16<<2)|2,{116,124,0}},
/* 20477 */ {(5<<2)|1,{116,0,0}},
/* 20478 */ {(5<<2)|1,{116,0,0}},
/* 20479 */ {(5<<2)|1,{116,0,0}},
/* 20480 */ {(16<<2)|3,{32,48,48}},
/* 20481 */ {(16<<2)|3,{32,48,49}},
/* 20482 */ {(16<<2)|3,{32,48,50}},
/* 20483 */ {(16<<2)|3,{32,48,97}},
/* 20484 */ {(16<<2)|3,{32,48,99}},
/* 20485 */ {(16<<2)|3,{32,48,101}},
/* 20486 */ {(16<<2)|3,{32,48,105}},
/* 20487 */ {(16<<2)|3,{32,48,111}},
/* 20488 */ {(16<<2)|3,{32,48,115}},
/* 20489 */ {(16<<2)|3,{32,48,116}},
/* 20490 */ {(11<<2)|2,{32,48,0}},
/* 20491 */ {(11<<2)|2,{32,48,0}},
/* 20492 */ {(11<<2)|2,{32,48,0}},
/* 20493 */ {(11<<2)|2,{32,48,0}},
/* 20494 */ {(11<<2)|2,{32,48,0}},
/* 20495 */ {(11<<2)|2,{32,48,0}},
/* 20496 */ {(11<<2)|2,{32,48,0}},
/* 20497 */ {(11<<2)|2,{32,48,0}},
/* 20498 */ {(11<<2)|2,{32,48,0}},
/* 20499 */ {(11<<2)|2,{32,48,0}},
/* 20500 */ {(11<<2)|2,{32,48,0}},
/* 20501 */ {(11<<2)|2,{32,48,0}},
/* 20502 */ {(11<<2)|2,{32,48,0}},
/* 20503 */ {(11<<2)|2,{32,48,0}},
/* 20504 */ {(11<<2)|2,{32,48,0}},
/* 20505 */ {(11<<2)|2,{32,48,0}},
/* 20506 */ {(11<<2)|2,{32,48,0}},
/* 20507 */ {(11<<2)|2,{32,48,0}},
/* 20508 */ {(11<<2)|2,{32,48,0}},
/* 20509 */ {(11<<2)|2,{32,48,0}},
/* 20510 */ {(11<<2)|2,{32,48,0}},
/* 20511 */ {(11<<2)|2,{32,48,0}},
/* 20512 */ {(16<<2)|3,{32,49,48}},
/* 20513 */ {(16<<2)|3,{32,49,49}},
/* 20514 */ {(16<<2)|3,{32,49,50}},
/* 20515 */ {(16<<2)|3,{32,49,97}},
/* 20516 */ {(16<<2)|3,{32,49,99}},
/* 20517 */ {(16<<2)|3,{32,49,101}},
/* 20518 */ {(16<<2)|3,{32,49,105}},
/* 20519 */ {(16<<2)|3,{32,49,111}},
/* 20520 */ {(16<<2)|3,{32,49,115}},
/* 20521 */ {(16<<2)|3,{32,49,116}},
/* 20522 */ {(11<<2)|2,{32,49,0}},
/* 20523 */ {(11<<2)|2,{32,49,0}},
/* 20524 */ {(11<<2)|2,{32,49,0}},
/* 20525 */ {(11<<2)|2,{32,49,0}},
/* 20526 */ {(11<<2)|2,{32,49,0}},
/* 20527 */ {(11<<2)|2,{32,49,0}},
/* 20528 */ {(11<<2)|2,{32,49,0}},
/* 20529 */ {(11<<2)|2,{32,49,0}},
/* 20530 */ {(11<<2)|2,{32,49,0}},
/* 20531 */ {(11<<2)|2,{32,49,0}},
/* 20532 */ {(11<<2)|2,{32,49,0}},
/* 20533 */ {(11<<2)|2,{32,49,0}},
/* 20534 */ {(11<<2)|2,{32,49,0}},
/* 20535 */ {(11<<2)|2,{32,49,0}},
/* 20536 */ {(11<<2)|2,{32,49,0}},
/* 20537 */ {(11<<2)|2,{32,49,0}},
/* 20538 */ {(11<<2)|2,{32,49,0}},
/* 20539 */ {(11<<2)|2,{32,49,0}},
/* 20540 */ {(11<<2)|2,{32,49,0}},
/* 20541 */ {(11<<2)|2,{32,49,0}},
/* 20542 */ {(11<<2)|2,{32,49,0}},
/* 20543 */ {(11<<2)|2,{32,49,0}},
/* 20544 */ {(16<<2)|3,{32,50,48}},
/* 20545 */ {(16<<2)|3,{32,50,49}},
/* 20546 */ {(16<<2)|3,{32,50,50}},
/* 20547 */ {(16<<2)|3,{32,50,97}},
/* 20548 */ {(16<<2)|3,{32,50,99}},
/* 20549 */ {(16<<2)|3,{32,50,101}},
/* 20550 */ {(16<<2)|3,{32,50,105}},
/* 20551 */ {(16<<2)|3,{32,50,111}},
/* 20552 */ {(16<<2)|3,{32,50,115}},
/* 20553 */ {(16<<2)|3,{32,50,116}},
/* 20554 */ {(11<<2)|2,{32,50,0}},
/* 20555 */ {(11<<2)|2,{32,50,0}},
/* 20556 */ {(11<<2)|2,{32,50,0}},
/* 20557 */ {(11<<2)|2,{32,50,0}},
/* 20558 */ {(11<<2)|2,{32,50,0}},
/* 20559 */ {(11<<2)|2,{32,50,0}},
/* 20560 */ {(11<<2)|2,{32,50,0}},
/* 20561 */ {(11<<2)|2,{32,50,0}},
/* 20562 */ {(11<<2)|2,{32,50,0}},
/* 20563 */ {(11<<2)|2,{32,50,0}},
/* 20564 */ {(11<<2)|2,{32,50,0}},
/* 20565 */ {(11<<2)|2,{32,50,0}},
/* 20566 */ {(11<<2)|2,{32,50,0}},
/* 20567 */ {(11<<2)|2,{32,50,0}},
/* 20568 */ {(11<<2)|2,{32,50,0}},
/* 20569 */ {(11<<2)|2,{32,50,0}},
/* 20570 */ {(11<<2)|2,{32,50,0}},
/* 20571 */ {(11<<2)|2,{32,50,0}},
/* 20572 */ {(11<<2)|2,{32,50,0}},
/* 20573 */ {(11<<2)|2,{32,50,0}},
/* 20574 */ {(11<<2)|2,{32,50,0}},
/* 20575 */ {(11<<2)|2,{32,50,0}},
/* 20576 */ {(16<<2)|3,{32,97,48}},
/* 20577 */ {(16<<2)|3,{32,97,49}},
/* 20578 */ {(16<<2)|3,{32,97,50}},
/* 20579 */ {(16<<2)|3,{32,97,97}},
/* 20580 */ {(16<<2)|3,{32,97,99}},
/* 20581 */ {(16<<2)|3,{32,97,101}},
/* 20582 */ {(16<<2)|3,{32,97,105}},
/* 20583 */ {(16<<2)|3,{32,97,111}},
/* 20584 */ {(16<<2)|3,{32,97,115}},
/* 20585 */ {(16<<2)|3,{32,97,116}},
/* 20586 */ {(11<<2)|2,{32,97,0}},
/* 20587 */ {(11<<2)|2,{32,97,0}},
/* 20588 */ {(11<<2)|2,{32,97,0}},
/* 20589 */ {(11<<2)|2,{32,97,0}},
/* 20590 */ {(11<<2)|2,{32,97,0}},
/* 20591 */ {(11<<2)|2,{32,97,0}},
/* 20592 */ {(11<<2)|2,{32,97,0}},
/* 20593 */ {(11<<2)|2,{32,97,0}},
/* 20594 */ {(11<<2)|2,{32,97,0}},
/* 20595 */ {(11<<2)|2,{32,97,0}},
/* 20596 */ {(11<<2)|2,{32,97,0}},
/* 20597 */ {(11<<2)|2,{32,97,0}},
/* 20598 */ {(11<<2)|2,{32,97,0}},
/* 20599 */ {(11<<2)|2,{32,97,0}},
/* 20600 */ {(11<<2)|2,{32,97,0}},
/* 20601 */ {(11<<2)|2,{32,97,0}},
/* 20602 */ {(11<<2)|2,{32,97,0}},
/* 20603 */ {(11<<2)|2,{32,97,0}},
/* 20604 */ {(11<<2)|2,{32,97,0}},
/* 20605 */ {(11<<2)|2,{32,97,0}},
/* 20606 */ {(11<<2)|2,{32,97,0}},
/* 20607 */ {(11<<2)|2,{32,97,0}},
/* 20608 */ {(16<<2)|3,{32,99,48}},
/* 20609 */ {(16<<2)|3,{32,99,49}},
/* 20610 */ {(16<<2)|3,{32,99,50}},
/* 20611 */ {(16<<2)|3,{32,99,97}},
/* 20612 */ {(16<<2)|3,{32,99,99}},
/* 20613 */ {(16<<2)|3,{32,99,101}},
/* 20614 */ {(16<<2)|3,{32,99,105}},
/* 20615 */ {(16<<2)|3,{32,99,111}},
/* 20616 */ {(16<<2)|3,{32,99,115}},
/* 20617 */ {(16<<2)|3,{32,99,116}},
/* 20618 */ {(11<<2)|2,{32,99,0}},
/* 20619 */ {(11<<2)|2,{32,99,0}},
/* 20620 */ {(11<<2)|2,{32,99,0}},
/* 20621 */ {(11<<2)|2,{32,99,0}},
/* 20622 */ {(11<<2)|2,{32,99,0}},
/* 20623 */ {(11<<2)|2,{32,99,0}},
/* 20624 */ {(11<<2)|2,{32,99,0}},
/* 20625 */ {(11<<2)|2,{32,99,0}},
/* 20626 */ {(11<<2)|2,{32,99,0}},
/* 20627 */ {(11<<2)|2,{32,99,0}},
/* 20628 */ {(11<<2)|2,{32,99,0}},
/* 20629 */ {(11<<2)|2,{32,99,0}},
/* 20630 */ {(11<<2)|2,{32,99,0}},
/* 20631 */ {(11<<2)|2,{32,99,0}},
/* 20632 */ {(11<<2)|2,{32,99,0}},
/* 20633 */ {(11<<2)|2,{32,99,0}},
/* 20634 */ {(11<<2)|2,{32,99,0}},
/* 20635 */ {(11<<2)|2,{32,99,0}},
/* 20636 */ {(11<<2)|2,{32,99,0}},
/* 20637 */ {(11<<2)|2,{32,99,0}},
/* 20638 */ {(11<<2)|2,{32,99,0}},
/* 20639 */ {(11<<2)|2,{32,99,0}},
/* 20640 */ {(16<<2)|3,{32,101,48}},
/* 20641 */ {(16<<2)|3,{32,101,49}},
/* 20642 */ {(16<<2)|3,{32,101,50}},
/* 20643 */ {(16<<2)|3,{32,101,97}},
/* 20644 */ {(16<<2)|3,{32,101,99}},
/* 20645 */ {(16<<2)|3,{32,101,101}},
/* 20646 */ {(16<<2)|3,{32,101,105}},
/* 20647 */ {(16<<2)|3,{32,101,111}},
/* 20648 */ {(16<<2)|3,{32,101,115}},
/* 20649 */ {(16<<2)|3,{32,101,116}},
/* 20650 */ {(11<<2)|2,{32,101,0}},
/* 20651 */ {(11<<2)|2,{32,101,0}},
/* 20652 */ {(11<<2)|2,{32,101,0}},
/* 20653 */ {(11<<2)|2,{32,101,0}},
/* 20654 */ {(11<<2)|2,{32,101,0}},
/* 20655 */ {(11<<2)|2,{32,101,0}},
/* 20656 */ {(11<<2)|2,{32,101,0}},
/* 20657 */ {(11<<2)|2,{32,101,0}},
/* 20658 */ {(11<<2)|2,{32,101,0}},
/* 20659 */ {(11<<2)|2,{32,101,0}},
/* 20660 */ {(11<<2)|2,{32,101,0}},
/* 20661 */ {(11<<2)|2,{32,101,0}},
/* 20662 */ {(11<<2)|2,{32,101,0}},
/* 20663 */ {(11<<2)|2,{32,101,0}},
/* 20664 */ {(11<<2)|2,{32,101,0}},
/* 20665 */ {(11<<2)|2,{32,101,0}},
/* 20666 */ {(11<<2)|2,{32,101,0}},
/* 20667 */ {(11<<2)|2,{32,101,0}},
/* 20668 */ {(11<<2)|2,{32,101,0}},
/* 20669 */ {(11<<2)|2,{32,101,0}},
/* 20670 */ {(11<<2)|2,{32,101,0}},
/* 20671 */ {(11<<2)|2,{32,101,0}},
/* 20672 */ {(16<<2)|3,{32,105,48}},
/* 20673 */ {(16<<2)|3,{32,105,49}},
/* 20674 */ {(16<<2)|3,{32,105,50}},
/* 20675 */ {(16<<2)|3,{32,105,97}},
/* 20676 */ {(16<<2)|3,{32,105,99}},
/* 20677 */ {(16<<2)|3,{32,105,101}},
/* 20678 */ {(16<<2)|3,{32,105,105}},
/* 20679 */ {(16<<2)|3,{32,105,111}},
/* 20680 */ {(16<<2)|3,{32,105,115}},
/* 20681 */ {(16<<2)|3,{32,105,116}},
/* 20682 */ {(11<<2)|2,{32,105,0}},
/* 20683 */ {(11<<2)|2,{32,105,0}},
/* 20684 */ {(11<<2)|2,{32,105,0}},
/* 20685 */ {(11<<2)|2,{32,105,0}},
/* 20686 */ {(11<<2)|2,{32,105,0}},
/* 20687 */ {(11<<2)|2,{32,105,0}},
/* 20688 */ {(11<<2)|2,{32,105,0}},
/* 20689 */ {(11<<2)|2,{32,105,0}},
/* 20690 */ {(11<<2)|2,{32,105,0}},
/* 20691 */ {(11<<2)|2,{32,105,0}},
/* 20692 */ {(11<<2)|2,{32,105,0}},
/* 20693 */ {(11<<2)|2,{32,105,0}},
/* 20694 */ {(11<<2)|2,{32,105,0}},
/* 20695 */ {(11<<2)|2,{32,105,0}},
/* 20696 */ {(11<<2)|2,{32,105,0}},
/* 20697 */ {(11<<2)|2,{32,105,0}},
/* 20698 */ {(11<<2)|2,{32,105,0}},
/* 20699 */ {(11<<2)|2,{32,105,0}},
/* 20700 */ {(11<<2)|2,{32,105,0}},
/* 20701 */ {(11<<2)|2,{32,105,0}},
/* 20702 */ {(11<<2)|2,{32,105,0}},
/* 20703 */ {(11<<2)|2,{32,105,0}},
/* 20704 */ {(16<<2)|3,{32,111,48}},
/* 20705 */ {(16<<2)|3,{32,111,49}},
/* 20706 */ {(16<<2)|3,{32,111,50}},
/* 20707 */ {(16<<2)|3,{32,111,97}},
/* 20708 */ {(16<<2)|3,{32,111,99}},
/* 20709 */ {(16<<2)|3,{32,111,101}},
/* 20710 */ {(16<<2)|3,{32,111,105}},
/* 20711 */ {(16<<2)|3,{32,111,111}},
/* 20712 */ {(16<<2)|3,{32,111,115}},
/* 20713 */ {(16<<2)|3,{32,111,116}},
/* 20714 */ {(11<<2)|2,{32,111,0}},
/* 20715 */ {(11<<2)|2,{32,111,0}},
/* 20716 */ {(11<<2)|2,{32,111,0}},
/* 20717 */ {(11<<2)|2,{32,111,0}},
/* 20718 */ {(11<<2)|2,{32,111,0}},
/* 20719 */ {(11<<2)|2,{32,111,0}},
/* 20720 */ {(11<<2)|2,{32,111,0}},
/* 20721 */ {(11<<2)|2,{32,111,0}},
/* 20722 */ {(11<<2)|2,{32,111,0}},
/* 20723 */ {(11<<2)|2,{32,111,0}},
/* 20724 */ {(11<<2)|2,{32,111,0}},
/* 20725 */ {(11<<2)|2,{32,111,0}},
/* 20726 */ {(11<<2)|2,{32,111,0}},
/* 20727 */ {(11<<2)|2,{32,111,0}},
/* 20728 */ {(11<<2)|2,{32,111,0}},
/* 20729 */ {(11<<2)|2,{32,111,0}},
/* 20730 */ {(11<<2)|2,{32,111,0}},
/* 20731 */ {(11<<2)|2,{32,111,0}},
/* 20732 */ {(11<<2)|2,{32,111,0}},
/* 20733 */ {(11<<2)|2,{32,111,0}},
/* 20734 */ {(11<<2)|2,{32,111,0}},
/* 20735 */ {(11<<2)|2,{32,111,0}},
/* 20736 */ {(16<<2)|3,{32,115,48}},
/* 20737 */ {(16<<2)|3,{32,115,49}},
/* 20738 */ {(16<<2)|3,{32,115,50}},
/* 20739 */ {(16<<2)|3,{32,115,97}},
/* 20740 */ {(16<<2)|3,{32,115,99}},
/* 20741 */ {(16<<2)|3,{32,115,101}},
/* 20742 */ {(16<<2)|3,{32,115,105}},
/* 20743 */ {(16<<2)|3,{32,115,111}},
/* 20744 */ {(16<<2)|3,{32,115,115}},
/* 20745 */ {(16<<2)|3,{32,115,116}},
/* 20746 */ {(11<<2)|2,{32,115,0}},
/* 20747 */ {(11<<2)|2,{32,115,0}},
/* 20748 */ {(11<<2)|2,{32,115,0}},
/* 20749 */ {(11<<2)|2,{32,115,0}},
/* 20750 */ {(11<<2)|2,{32,115,0}},
/* 20751 */ {(11<<2)|2,{32,115,0}},
/* 20752 */ {(11<<2)|2,{32,115,0}},
/* 20753 */ {(11<<2)|2,{32,115,0}},
/* 20754 */ {(11<<2)|2,{32,115,0}},
/* 20755 */ {(11<<2)|2,{32,115,0}},
/* 20756 */ {(11<<2)|2,{32,115,0}},
/* 20757 */ {(11<<2)|2,{32,115,0}},
/* 20758 */ {(11<<2)|2,{32,115,0}},
/* 20759 */ {(11<<2)|2,{32,115,0}},
/* 20760 */ {(11<<2)|2,{32,115,0}},
/* 20761 */ {(11<<2)|2,{32,115,0}},
/* 20762 */ {(11<<2)|2,{32,115,0}},
/* 20763 */ {(11<<2)|2,{32,115,0}},
/* 20764 */ {(11<<2)|2,{32,115,0}},
/* 20765 */ {(11<<2)|2,{32,115,0}},
/* 20766 */ {(11<<2)|2,{32,115,0}},
/* 20767 */ {(11<<2)|2,{32,115,0}},
/* 20768 */ {(16<<2)|3,{32,116,48}},
/* 20769 */ {(16<<2)|3,{32,116,49}},
/* 20770 */ {(16<<2)|3,{32,116,50}},
/* 20771 */ {(16<<2)|3,{32,116,97}},
/* 20772 */ {(16<<2)|3,{32,116,99}},
/* 20773 */ {(16<<2)|3,{32,116,101}},
/* 20774 */ {(16<<2)|3,{32,116,105}},
/* 20775 */ {(16<<2)|3,{32,116,111}},
/* 20776 */ {(16<<2)|3,{32,116,115}},
/* 20777 */ {(16<<2)|3,{32,116,116}},
/* 20778 */ {(11<<2)|2,{32,116,0}},
/* 20779 */ {(11<<2)|2,{32,116,0}},
/* 20780 */ {(11<<2)|2,{32,116,0}},
/* 20781 */ {(11<<2)|2,{32,116,0}},
/* 20782 */ {(11<<2)|2,{32,116,0}},
/* 20783 */ {(11<<2)|2,{32,116,0}},
/* 20784 */ {(11<<2)|2,{32,116,0}},
/* 20785 */ {(11<<2)|2,{32,116,0}},
/* 20786 */ {(11<<2)|2,{32,116,0}},
/* 20787 */ {(11<<2)|2,{32,116,0}},
/* 20788 */ {(11<<2)|2,{32,116,0}},
/* 20789 */ {(11<<2)|2,{32,116,0}},
/* 20790 */ {(11<<2)|2,{32,116,0}},
/* 20791 */ {(11<<2)|2,{32,116,0}},
/* 20792 */ {(11<<2)|2,{32,116,0}},
/* 20793 */ {(11<<2)|2,{32,116,0}},
/* 20794 */ {(11<<2)|2,{32,116,0}},
/* 20795 */ {(11<<2)|2,{32,116,0}},
/* 20796 */ {(11<<2)|2,{32,116,0}},
/* 20797 */ {(11<<2)|2,{32,116,0}},
/* 20798 */ {(11<<2)|2,{32,116,0}},
/* 20799 */ {(11<<2)|2,{32,116,0}},
/* 20800 */ {(12<<2)|2,{32,32,0}},
/* 20801 */ {(12<<2)|2,{32,32,0}},
/* 20802 */ {(12<<2)|2,{32,32,0}},
/* 20803 */ {(12<<2)|2,{32,32,0}},
/* 20804 */ {(12<<2)|2,{32,32,0}},
/* 20805 */ {(12<<2)|2,{32,32,0}},
/* 20806 */ {(12<<2)|2,{32,32,0}},
/* 20807 */ {(12<<2)|2,{32,32,0}},
/* 20808 */ {(12<<2)|2,{32,32,0}},
/* 20809 */ {(12<<2)|2,{32,32,0}},
/* 20810 */ {(12<<2)|2,{32,32,0}},
/* 20811 */ {(12<<2)|2,{32,32,0}},
/* 20812 */ {(12<<2)|2,{32,32,0}},
/* 20813 */ {(12<<2)|2,{32,32,0}},
/* 20814 */ {(12<<2)|2,{32,32,0}},
/* 20815 */ {(12<<2)|2,{32,32,0}},
/* 20816 */ {(12<<2)|2,{32,37,0}},
/* 20817 */ {(12<<2)|2,{32,37,0}},
/* 20818 */ {(12<<2)|2,{32,37,0}},
/* 20819 */ {(12<<2)|2,{32,37,0}},
/* 20820 */ {(12<<2)|2,{32,37,0}},
/* 20821 */ {(12<<2)|2,{32,37,0}},
/* 20822 */ {(12<<2)|2,{32,37,0}},
/* 20823 */ {(12<<2)|2,{32,37,0}},
/* 20824 */ {(12<<2)|2,{32,37,0}},
/* 20825 */ {(12<<2)|2,{32,37,0}},
/* 20826 */ {(12<<2)|2,{32,37,0}},
/* 20827 */ {(12<<2)|2,{32,37,0}},
/* 20828 */ {(12<<2)|2,{32,37,0}},
/* 20829 */ {(12<<2)|2,{32,37,0}},
/* 20830 */ {(12<<2)|2,{32,37,0}},
/* 20831 */ {(12<<2)|2,{32,37,0}},
/* 20832 */ {(12<<2)|2,{32,45,0}},
/* 20833 */ {(12<<2)|2,{32,45,0}},
/* 20834 */ {(12<<2)|2,{32,45,0}},
/* 20835 */ {(12<<2)|2,{32,45,0}},
/* 20836 */ {(12<<2)|2,{32,45,0}},
/* 20837 */ {(12<<2)|2,{32,45,0}},
/* 20838 */ {(12<<2)|2,{32,45,0}},
/* 20839 */ {(12<<2)|2,{32,45,0}},
/* 20840 */ {(12<<2)|2,{32,45,0}},
/* 20841 */ {(12<<2)|2,{32,45,0}},
/* 20842 */ {(12<<2)|2,{32,45,0}},
/* 20843 */ {(12<<2)|2,{32,45,0}},
/* 20844 */ {(12<<2)|2,{32,45,0}},
/* 20845 */ {(12<<2)|2,{32,45,0}},
/* 20846 */ {(12<<2)|2,{32,45,0}},
/* 20847 */ {(12<<2)|2,{32,45,0}},
/* 20848 */ {(12<<2)|2,{32,46,0}},
/* 20849 */ {(12<<2)|2,{32,46,0}},
/* 20850 */ {(12<<2)|2,{32,46,0}},
/* 20851 */ {(12<<2)|2,{32,46,0}},
/* 20852 */ {(12<<2)|2,{32,46,0}},
/* 20853 */ {(12<<2)|2,{32,46,0}},
/* 20854 */ {(12<<2)|2,{32,46,0}},
/* 20855 */ {(12<<2)|2,{32,46,0}},
/* 20856 */ {(12<<2)|2,{32,46,0}},
/* 20857 */ {(12<<2)|2,{32,46,0}},
/* 20858 */ {(12<<2)|2,{32,46,0}},
/* 20859 */ {(12<<2)|2,{32,46,0}},
/* 20860 */ {(12<<2)|2,{32,46,0}},
/* 20861 */ {(12<<2)|2,{32,46,0}},
/* 20862 */ {(12<<2)|2,{32,46,0}},
/* 20863 */ {(12<<2)|2,{32,46,0}},
/* 20864 */ {(12<<2)|2,{32,47,0}},
/* 20865 */ {(12<<2)|2,{32,47,0}},
/* 20866 */ {(12<<2)|2,{32,47,0}},
/* 20867 */ {(12<<2)|2,{32,47,0}},
/* 20868 */ {(12<<2)|2,{32,47,0}},
/* 20869 */ {(12<<2)|2,{32,47,0}},
/* 20870 */ {(12<<2)|2,{32,47,0}},
/* 20871 */ {(12<<2)|2,{32,47,0}},
/* 20872 */ {(12<<2)|2,{32,47,0}},
/* 20873 */ {(12<<2)|2,{32,47,0}},
/* 20874 */ {(12<<2)|2,{32,47,0}},
/* 20875 */ {(12<<2)|2,{32,47,0}},
/* 20876 */ {(12<<2)|2,{32,47,0}},
/* 20877 */ {(12<<2)|2,{32,47,0}},
/* 20878 */ {(12<<2)|2,{32,47,0}},
/* 20879 */ {(12<<2)|2,{32,47,0}},
/* 20880 */ {(12<<2)|2,{32,51,0}},
/* 20881 */ {(12<<2)|2,{32,51,0}},
/* 20882 */ {(12<<2)|2,{32,51,0}},
/* 20883 */ {(12<<2)|2,{32,51,0}},
/* 20884 */ {(12<<2)|2,{32,51,0}},
/* 20885 */ {(12<<2)|2,{32,51,0}},
/* 20886 */ {(12<<2)|2,{32,51,0}},
/* 20887 */ {(12<<2)|2,{32,51,0}},
/* 20888 */ {(12<<2)|2,{32,51,0}},
/* 20889 */ {(12<<2)|2,{32,51,0}},
/* 20890 */ {(12<<2)|2,{32,51,0}},
/* 20891 */ {(12<<2)|2,{32,51,0}},
/* 20892 */ {(12<<2)|2,{32,51,0}},
/* 20893 */ {(12<<2)|2,{32,51,0}},
/* 20894 */ {(12<<2)|2,{32,51,0}},
/* 20895 */ {(12<<2)|2,{32,51,0}},
/* 20896 */ {(12<<2)|2,{32,52,0}},
/* 20897 */ {(12<<2)|2,{32,52,0}},
/* 20898 */ {(12<<2)|2,{32,52,0}},
/* 20899 */ {(12<<2)|2,{32,52,0}},
/* 20900 */ {(12<<2)|2,{32,52,0}},
/* 20901 */ {(12<<2)|2,{32,52,0}},
/* 20902 */ {(12<<2)|2,{32,52,0}},
/* 20903 */ {(12<<2)|2,{32,52,0}},
/* 20904 */ {(12<<2)|2,{32,52,0}},
/* 20905 */ {(12<<2)|2,{32,52,0}},
/* 20906 */ {(12<<2)|2,{32,52,0}},
/* 20907 */ {(12<<2)|2,{32,52,0}},
/* 20908 */ {(12<<2)|2,{32,52,0}},
/* 20909 */ {(12<<2)|2,{32,52,0}},
/* 20910 */ {(12<<2)|2,{32,52,0}},
/* 20911 */ {(12<<2)|2,{32,52,0}},
/* 20912 */ {(12<<2)|2,{32,53,0}},
/* 20913 */ {(12<<2)|2,{32,53,0}},
/* 20914 */ {(12<<2)|2,{32,53,0}},
/* 20915 */ {(12<<2)|2,{32,53,0}},
/* 20916 */ {(12<<2)|2,{32,53,0}},
/* 20917 */ {(12<<2)|2,{32,53,0}},
/* 20918 */ {(12<<2)|2,{32,53,0}},
/* 20919 */ {(12<<2)|2,{32,53,0}},
/* 20920 */ {(12<<2)|2,{32,53,0}},
/* 20921 */ {(12<<2)|2,{32,53,0}},
/* 20922 */ {(12<<2)|2,{32,53,0}},
/* 20923 */ {(12<<2)|2,{32,53,0}},
/* 20924 */ {(12<<2)|2,{32,53,0}},
/* 20925 */ {(12<<2)|2,{32,53,0}},
/* 20926 */ {(12<<2)|2,{32,53,0}},
/* 20927 */ {(12<<2)|2,{32,53,0}},
/* 20928 */ {(12<<2)|2,{32,54,0}},
/* 20929 */ {(12<<2)|2,{32,54,0}},
/* 20930 */ {(12<<2)|2,{32,54,0}},
/* 20931 */ {(12<<2)|2,{32,54,0}},
/* 20932 */ {(12<<2)|2,{32,54,0}},
/* 20933 */ {(12<<2)|2,{32,54,0}},
/* 20934 */ {(12<<2)|2,{32,54,0}},
/* 20935 */ {(12<<2)|2,{32,54,0}},
/* 20936 */ {(12<<2)|2,{32,54,0}},
/* 20937 */ {(12<<2)|2,{32,54,0}},
/* 20938 */ {(12<<2)|2,{32,54,0}},
/* 20939 */ {(12<<2)|2,{32,54,0}},
/* 20940 */ {(12<<2)|2,{32,54,0}},
/* 20941 */ {(12<<2)|2,{32,54,0}},
/* 20942 */ {(12<<2)|2,{32,54,0}},
/* 20943 */ {(12<<2)|2,{32,54,0}},
/* 20944 */ {(12<<2)|2,{32,55,0}},
/* 20945 */ {(12<<2)|2,{32,55,0}},
/* 20946 */ {(12<<2)|2,{32,55,0}},
/* 20947 */ {(12<<2)|2,{32,55,0}},
/* 20948 */ {(12<<2)|2,{32,55,0}},
/* 20949 */ {(12<<2)|2,{32,55,0}},
/* 20950 */ {(12<<2)|2,{32,55,0}},
/* 20951 */ {(12<<2)|2,{32,55,0}},
/* 20952 */ {(12<<2)|2,{32,55,0}},
/* 20953 */ {(12<<2)|2,{32,55,0}},
/* 20954 */ {(12<<2)|2,{32,55,0}},
/* 20955 */ {(12<<2)|2,{32,55,0}},
/* 20956 */ {(12<<2)|2,{32,55,0}},
/* 20957 */ {(12<<2)|2,{32,55,0}},
/* 20958 */ {(12<<2)|2,{32,55,0}},
/* 20959 */ {(12<<2)|2,{32,55,0}},
/* 20960 */ {(12<<2)|2,{32,56,0}},
/* 20961 */ {(12<<2)|2,{32,56,0}},
/* 20962 */ {(12<<2)|2,{32,56,0}},
/* 20963 */ {(12<<2)|2,{32,56,0}},
/* 20964 */ {(12<<2)|2,{32,56,0}},
/* 20965 */ {(12<<2)|2,{32,56,0}},
/* 20966 */ {(12<<2)|2,{32,56,0}},
/* 20967 */ {(12<<2)|2,{32,56,0}},
/* 20968 */ {(12<<2)|2,{32,56,0}},
/* 20969 */ {(12<<2)|2,{32,56,0}},
/* 20970 */ {(12<<2)|2,{32,56,0}},
/* 20971 */ {(12<<2)|2,{32,56,0}},
/* 20972 */ {(12<<2)|2,{32,56,0}},
/* 20973 */ {(12<<2)|2,{32,56,0}},
/* 20974 */ {(12<<2)|2,{32,56,0}},
/* 20975 */ {(12<<2)|2,{32,56,0}},
/* 20976 */ {(12<<2)|2,{32,57,0}},
/* 20977 */ {(12<<2)|2,{32,57,0}},
/* 20978 */ {(12<<2)|2,{32,57,0}},
/* 20979 */ {(12<<2)|2,{32,57,0}},
/* 20980 */ {(12<<2)|2,{32,57,0}},
/* 20981 */ {(12<<2)|2,{32,57,0}},
/* 20982 */ {(12<<2)|2,{32,57,0}},
/* 20983 */ {(12<<2)|2,{32,57,0}},
/* 20984 */ {(12<<2)|2,{32,57,0}},
/* 20985 */ {(12<<2)|2,{32,57,0}},
/* 20986 */ {(12<<2)|2,{32,57,0}},
/* 20987 */ {(12<<2)|2,{32,57,0}},
/* 20988 */ {(12<<2)|2,{32,57,0}},
/* 20989 */ {(12<<2)|2,{32,57,0}},
/* 20990 */ {(12<<2)|2,{32,57,0}},
/* 20991 */ {(12<<2)|2,{32,57,0}},
/* 20992 */ {(12<<2)|2,{32,61,0}},
/* 20993 */ {(12<<2)|2,{32,61,0}},
/* 20994 */ {(12<<2)|2,{32,61,0}},
/* 20995 */ {(12<<2)|2,{32,61,0}},
/* 20996 */ {(12<<2)|2,{32,61,0}},
/* 20997 */ {(12<<2)|2,{32,61,0}},
/* 20998 */ {(12<<2)|2,{32,61,0}},
/* 20999 */ {(12<<2)|2,{32,61,0}},
/* 21000 */ {(12<<2)|2,{32,61,0}},
/* 21001 */ {(12<<2)|2,{32,61,0}},
/* 21002 */ {(12<<2)|2,{32,61,0}},
/* 21003 */ {(12<<2)|2,{32,61,0}},
/* 21004 */ {(12<<2)|2,{32,61,0}},
/* 21005 */ {(12<<2)|2,{32,61,0}},
/* 21006 */ {(12<<2)|2,{32,61,0}},
/* 21007 */ {(12<<2)|2,{32,61,0}},
/* 21008 */ {(12<<2)|2,{32,65,0}},
/* 21009 */ {(12<<2)|2,{32,65,0}},
/* 21010 */ {(12<<2)|2,{32,65,0}},
/* 21011 */ {(12<<2)|2,{32,65,0}},
/* 21012 */ {(12<<2)|2,{32,65,0}},
/* 21013 */ {(12<<2)|2,{32,65,0}},
/* 21014 */ {(12<<2)|2,{32,65,0}},
/* 21015 */ {(12<<2)|2,{32,65,0}},
/* 21016 */ {(12<<2)|2,{32,65,0}},
/* 21017 */ {(12<<2)|2,{32,65,0}},
/* 21018 */ {(12<<2)|2,{32,65,0}},
/* 21019 */ {(12<<2)|2,{32,65,0}},
/* 21020 */ {(12<<2)|2,{32,65,0}},
/* 21021 */ {(12<<2)|2,{32,65,0}},
/* 21022 */ {(12<<2)|2,{32,65,0}},
/* 21023 */ {(12<<2)|2,{32,65,0}},
/* 21024 */ {(12<<2)|2,{32,95,0}},
/* 21025 */ {(12<<2)|2,{32,95,0}},
/* 21026 */ {(12<<2)|2,{32,95,0}},
/* 21027 */ {(12<<2)|2,{32,95,0}},
/* 21028 */ {(12<<2)|2,{32,95,0}},
/* 21029 */ {(12<<2)|2,{32,95,0}},
/* 21030 */ {(12<<2)|2,{32,95,0}},
/* 21031 */ {(12<<2)|2,{32,95,0}},
/* 21032 */ {(12<<2)|2,{32,95,0}},
/* 21033 */ {(12<<2)|2,{32,95,0}},
/* 21034 */ {(12<<2)|2,{32,95,0}},
/* 21035 */ {(12<<2)|2,{32,95,0}},
/* 21036 */ {(12<<2)|2,{32,95,0}},
/* 21037 */ {(12<<2)|2,{32,95,0}},
/* 21038 */ {(12<<2)|2,{32,95,0}},
/* 21039 */ {(12<<2)|2,{32,95,0}},
/* 21040 */ {(12<<2)|2,{32,98,0}},
/* 21041 */ {(12<<2)|2,{32,98,0}},
/* 21042 */ {(12<<2)|2,{32,98,0}},
/* 21043 */ {(12<<2)|2,{32,98,0}},
/* 21044 */ {(12<<2)|2,{32,98,0}},
/* 21045 */ {(12<<2)|2,{32,98,0}},
/* 21046 */ {(12<<2)|2,{32,98,0}},
/* 21047 */ {(12<<2)|2,{32,98,0}},
/* 21048 */ {(12<<2)|2,{32,98,0}},
/* 21049 */ {(12<<2)|2,{32,98,0}},
/* 21050 */ {(12<<2)|2,{32,98,0}},
/* 21051 */ {(12<<2)|2,{32,98,0}},
/* 21052 */ {(12<<2)|2,{32,98,0}},
/* 21053 */ {(12<<2)|2,{32,98,0}},
/* 21054 */ {(12<<2)|2,{32,98,0}},
/* 21055 */ {(12<<2)|2,{32,98,0}},
/* 21056 */ {(12<<2)|2,{32,100,0}},
/* 21057 */ {(12<<2)|2,{32,100,0}},
/* 21058 */ {(12<<2)|2,{32,100,0}},
/* 21059 */ {(12<<2)|2,{32,100,0}},
/* 21060 */ {(12<<2)|2,{32,100,0}},
/* 21061 */ {(12<<2)|2,{32,100,0}},
/* 21062 */ {(12<<2)|2,{32,100,0}},
/* 21063 */ {(12<<2)|2,{32,100,0}},
/* 21064 */ {(12<<2)|2,{32,100,0}},
/* 21065 */ {(12<<2)|2,{32,100,0}},
/* 21066 */ {(12<<2)|2,{32,100,0}},
/* 21067 */ {(12<<2)|2,{32,100,0}},
/* 21068 */ {(12<<2)|2,{32,100,0}},
/* 21069 */ {(12<<2)|2,{32,100,0}},
/* 21070 */ {(12<<2)|2,{32,100,0}},
/* 21071 */ {(12<<2)|2,{32,100,0}},
/* 21072 */ {(12<<2)|2,{32,102,0}},
/* 21073 */ {(12<<2)|2,{32,102,0}},
/* 21074 */ {(12<<2)|2,{32,102,0}},
/* 21075 */ {(12<<2)|2,{32,102,0}},
/* 21076 */ {(12<<2)|2,{32,102,0}},
/* 21077 */ {(12<<2)|2,{32,102,0}},
/* 21078 */ {(12<<2)|2,{32,102,0}},
/* 21079 */ {(12<<2)|2,{32,102,0}},
/* 21080 */ {(12<<2)|2,{32,102,0}},
/* 21081 */ {(12<<2)|2,{32,102,0}},
/* 21082 */ {(12<<2)|2,{32,102,0}},
/* 21083 */ {(12<<2)|2,{32,102,0}},
/* 21084 */ {(12<<2)|2,{32,102,0}},
/* 21085 */ {(12<<2)|2,{32,102,0}},
/* 21086 */ {(12<<2)|2,{32,102,0}},
/* 21087 */ {(12<<2)|2,{32,102,0}},
/* 21088 */ {(12<<2)|2,{32,103,0}},
/* 21089 */ {(12<<2)|2,{32,103,0}},
/* 21090 */ {(12<<2)|2,{32,103,0}},
/* 21091 */ {(12<<2)|2,{32,103,0}},
/* 21092 */ {(12<<2)|2,{32,103,0}},
/* 21093 */ {(12<<2)|2,{32,103,0}},
/* 21094 */ {(12<<2)|2,{32,103,0}},
/* 21095 */ {(12<<2)|2,{32,103,0}},
/* 21096 */ {(12<<2)|2,{32,103,0}},
/* 21097 */ {(12<<2)|2,{32,103,0}},
/* 21098 */ {(12<<2)|2,{32,103,0}},
/* 21099 */ {(12<<2)|2,{32,103,0}},
/* 21100 */ {(12<<2)|2,{32,103,0}},
/* 21101 */ {(12<<2)|2,{32,103,0}},
/* 21102 */ {(12<<2)|2,{32,103,0}},
/* 21103 */ {(12<<2)|2,{32,103,0}},
/* 21104 */ {(12<<2)|2,{32,104,0}},
/* 21105 */ {(12<<2)|2,{32,104,0}},
/* 21106 */ {(12<<2)|2,{32,104,0}},
/* 21107 */ {(12<<2)|2,{32,104,0}},
/* 21108 */ {(12<<2)|2,{32,104,0}},
/* 21109 */ {(12<<2)|2,{32,104,0}},
/* 21110 */ {(12<<2)|2,{32,104,0}},
/* 21111 */ {(12<<2)|2,{32,104,0}},
/* 21112 */ {(12<<2)|2,{32,104,0}},
/* 21113 */ {(12<<2)|2,{32,104,0}},
/* 21114 */ {(12<<2)|2,{32,104,0}},
/* 21115 */ {(12<<2)|2,{32,104,0}},
/* 21116 */ {(12<<2)|2,{32,104,0}},
/* 21117 */ {(12<<2)|2,{32,104,0}},
/* 21118 */ {(12<<2)|2,{32,104,0}},
/* 21119 */ {(12<<2)|2,{32,104,0}},
/* 21120 */ {(12<<2)|2,{32,108,0}},
/* 21121 */ {(12<<2)|2,{32,108,0}},
/* 21122 */ {(12<<2)|2,{32,108,0}},
/* 21123 */ {(12<<2)|2,{32,108,0}},
/* 21124 */ {(12<<2)|2,{32,108,0}},
/* 21125 */ {(12<<2)|2,{32,108,0}},
/* 21126 */ {(12<<2)|2,{32,108,0}},
/* 21127 */ {(12<<2)|2,{32,108,0}},
/* 21128 */ {(12<<2)|2,{32,108,0}},
/* 21129 */ {(12<<2)|2,{32,108,0}},
/* 21130 */ {(12<<2)|2,{32,108,0}},
/* 21131 */ {(12<<2)|2,{32,108,0}},
/* 21132 */ {(12<<2)|2,{32,108,0}},
/* 21133 */ {(12<<2)|2,{32,108,0}},
/* 21134 */ {(12<<2)|2,{32,108,0}},
/* 21135 */ {(12<<2)|2,{32,108,0}},
/* 21136 */ {(12<<2)|2,{32,109,0}},
/* 21137 */ {(12<<2)|2,{32,109,0}},
/* 21138 */ {(12<<2)|2,{32,109,0}},
/* 21139 */ {(12<<2)|2,{32,109,0}},
/* 21140 */ {(12<<2)|2,{32,109,0}},
/* 21141 */ {(12<<2)|2,{32,109,0}},
/* 21142 */ {(12<<2)|2,{32,109,0}},
/* 21143 */ {(12<<2)|2,{32,109,0}},
/* 21144 */ {(12<<2)|2,{32,109,0}},
/* 21145 */ {(12<<2)|2,{32,109,0}},
/* 21146 */ {(12<<2)|2,{32,109,0}},
/* 21147 */ {(12<<2)|2,{32,109,0}},
/* 21148 */ {(12<<2)|2,{32,109,0}},
/* 21149 */ {(12<<2)|2,{32,109,0}},
/* 21150 */ {(12<<2)|2,{32,109,0}},
/* 21151 */ {(12<<2)|2,{32,109,0}},
/* 21152 */ {(12<<2)|2,{32,110,0}},
/* 21153 */ {(12<<2)|2,{32,110,0}},
/* 21154 */ {(12<<2)|2,{32,110,0}},
/* 21155 */ {(12<<2)|2,{32,110,0}},
/* 21156 */ {(12<<2)|2,{32,110,0}},
/* 21157 */ {(12<<2)|2,{32,110,0}},
/* 21158 */ {(12<<2)|2,{32,110,0}},
/* 21159 */ {(12<<2)|2,{32,110,0}},
/* 21160 */ {(12<<2)|2,{32,110,0}},
/* 21161 */ {(12<<2)|2,{32,110,0}},
/* 21162 */ {(12<<2)|2,{32,110,0}},
/* 21163 */ {(12<<2)|2,{32,110,0}},
/* 21164 */ {(12<<2)|2,{32,110,0}},
/* 21165 */ {(12<<2)|2,{32,110,0}},
/* 21166 */ {(12<<2)|2,{32,110,0}},
/* 21167 */ {(12<<2)|2,{32,110,0}},
/* 21168 */ {(12<<2)|2,{32,112,0}},
/* 21169 */ {(12<<2)|2,{32,112,0}},
/* 21170 */ {(12<<2)|2,{32,112,0}},
/* 21171 */ {(12<<2)|2,{32,112,0}},
/* 21172 */ {(12<<2)|2,{32,112,0}},
/* 21173 */ {(12<<2)|2,{32,112,0}},
/* 21174 */ {(12<<2)|2,{32,112,0}},
/* 21175 */ {(12<<2)|2,{32,112,0}},
/* 21176 */ {(12<<2)|2,{32,112,0}},
/* 21177 */ {(12<<2)|2,{32,112,0}},
/* 21178 */ {(12<<2)|2,{32,112,0}},
/* 21179 */ {(12<<2)|2,{32,112,0}},
/* 21180 */ {(12<<2)|2,{32,112,0}},
/* 21181 */ {(12<<2)|2,{32,112,0}},
/* 21182 */ {(12<<2)|2,{32,112,0}},
/* 21183 */ {(12<<2)|2,{32,112,0}},
/* 21184 */ {(12<<2)|2,{32,114,0}},
/* 21185 */ {(12<<2)|2,{32,114,0}},
/* 21186 */ {(12<<2)|2,{32,114,0}},
/* 21187 */ {(12<<2)|2,{32,114,0}},
/* 21188 */ {(12<<2)|2,{32,114,0}},
/* 21189 */ {(12<<2)|2,{32,114,0}},
/* 21190 */ {(12<<2)|2,{32,114,0}},
/* 21191 */ {(12<<2)|2,{32,114,0}},
/* 21192 */ {(12<<2)|2,{32,114,0}},
/* 21193 */ {(12<<2)|2,{32,114,0}},
/* 21194 */ {(12<<2)|2,{32,114,0}},
/* 21195 */ {(12<<2)|2,{32,114,0}},
/* 21196 */ {(12<<2)|2,{32,114,0}},
/* 21197 */ {(12<<2)|2,{32,114,0}},
/* 21198 */ {(12<<2)|2,{32,114,0}},
/* 21199 */ {(12<<2)|2,{32,114,0}},
/* 21200 */ {(12<<2)|2,{32,117,0}},
/* 21201 */ {(12<<2)|2,{32,117,0}},
/* 21202 */ {(12<<2)|2,{32,117,0}},
/* 21203 */ {(12<<2)|2,{32,117,0}},
/* 21204 */ {(12<<2)|2,{32,117,0}},
/* 21205 */ {(12<<2)|2,{32,117,0}},
/* 21206 */ {(12<<2)|2,{32,117,0}},
/* 21207 */ {(12<<2)|2,{32,117,0}},
/* 21208 */ {(12<<2)|2,{32,117,0}},
/* 21209 */ {(12<<2)|2,{32,117,0}},
/* 21210 */ {(12<<2)|2,{32,117,0}},
/* 21211 */ {(12<<2)|2,{32,117,0}},
/* 21212 */ {(12<<2)|2,{32,117,0}},
/* 21213 */ {(12<<2)|2,{32,117,0}},
/* 21214 */ {(12<<2)|2,{32,117,0}},
/* 21215 */ {(12<<2)|2,{32,117,0}},
/* 21216 */ {(13<<2)|2,{32,58,0}},
/* 21217 */ {(13<<2)|2,{32,58,0}},
/* 21218 */ {(13<<2)|2,{32,58,0}},
/* 21219 */ {(13<<2)|2,{32,58,0}},
/* 21220 */ {(13<<2)|2,{32,58,0}},
/* 21221 */ {(13<<2)|2,{32,58,0}},
/* 21222 */ {(13<<2)|2,{32,58,0}},
/* 21223 */ {(13<<2)|2,{32,58,0}},
/* 21224 */ {(13<<2)|2,{32,66,0}},
/* 21225 */ {(13<<2)|2,{32,66,0}},
/* 21226 */ {(13<<2)|2,{32,66,0}},
/* 21227 */ {(13<<2)|2,{32,66,0}},
/* 21228 */ {(13<<2)|2,{32,66,0}},
/* 21229 */ {(13<<2)|2,{32,66,0}},
/* 21230 */ {(13<<2)|2,{32,66,0}},
/* 21231 */ {(13<<2)|2,{32,66,0}},
/* 21232 */ {(13<<2)|2,{32,67,0}},
/* 21233 */ {(13<<2)|2,{32,67,0}},
/* 21234 */ {(13<<2)|2,{32,67,0}},
/* 21235 */ {(13<<2)|2,{32,67,0}},
/* 21236 */ {(13<<2)|2,{32,67,0}},
/* 21237 */ {(13<<2)|2,{32,67,0}},
/* 21238 */ {(13<<2)|2,{32,67,0}},
/* 21239 */ {(13<<2)|2,{32,67,0}},
/* 21240 */ {(13<<2)|2,{32,68,0}},
/* 21241 */ {(13<<2)|2,{32,68,0}},
/* 21242 */ {(13<<2)|2,{32,68,0}},
/* 21243 */ {(13<<2)|2,{32,68,0}},
/* 21244 */ {(13<<2)|2,{32,68,0}},
/* 21245 */ {(13<<2)|2,{32,68,0}},
/* 21246 */ {(13<<2)|2,{32,68,0}},
/* 21247 */ {(13<<2)|2,{32,68,0}},
/* 21248 */ {(13<<2)|2,{32,69,0}},
/* 21249 */ {(13<<2)|2,{32,69,0}},
/* 21250 */ {(13<<2)|2,{32,69,0}},
/* 21251 */ {(13<<2)|2,{32,69,0}},
/* 21252 */ {(13<<2)|2,{32,69,0}},
/* 21253 */ {(13<<2)|2,{32,69,0}},
/* 21254 */ {(13<<2)|2,{32,69,0}},
/* 21255 */ {(13<<2)|2,{32,69,0}},
/* 21256 */ {(13<<2)|2,{32,70,0}},
/* 21257 */ {(13<<2)|2,{32,70,0}},
/* 21258 */ {(13<<2)|2,{32,70,0}},
/* 21259 */ {(13<<2)|2,{32,70,0}},
/* 21260 */ {(13<<2)|2,{32,70,0}},
/* 21261 */ {(13<<2)|2,{32,70,0}},
/* 21262 */ {(13<<2)|2,{32,70,0}},
/* 21263 */ {(13<<2)|2,{32,70,0}},
/* 21264 */ {(13<<2)|2,{32,71,0}},
/* 21265 */ {(13<<2)|2,{32,71,0}},
/* 21266 */ {(13<<2)|2,{32,71,0}},
/* 21267 */ {(13<<2)|2,{32,71,0}},
/* 21268 */ {(13<<2)|2,{32,71,0}},
/* 21269 */ {(13<<2)|2,{32,71,0}},
/* 21270 */ {(13<<2)|2,{32,71,0}},
/* 21271 */ {(13<<2)|2,{32,71,0}},
/* 21272 */ {(13<<2)|2,{32,72,0}},
/* 21273 */ {(13<<2)|2,{32,72,0}},
/* 21274 */ {(13<<2)|2,{32,72,0}},
/* 21275 */ {(13<<2)|2,{32,72,0}},
/* 21276 */ {(13<<2)|2,{32,72,0}},
/* 21277 */ {(13<<2)|2,{32,72,0}},
/* 21278 */ {(13<<2)|2,{32,72,0}},
/* 21279 */ {(13<<2)|2,{32,72,0}},
/* 21280 */ {(13<<2)|2,{32,73,0}},
/* 21281 */ {(13<<2)|2,{32,73,0}},
/* 21282 */ {(13<<2)|2,{32,73,0}},
/* 21283 */ {(13<<2)|2,{32,73,0}},
/* 21284 */ {(13<<2)|2,{32,73,0}},
/* 21285 */ {(13<<2)|2,{32,73,0}},
/* 21286 */ {(13<<2)|2,{32,73,0}},
/* 21287 */ {(13<<2)|2,{32,73,0}},
/* 21288 */ {(13<<2)|2,{32,74,0}},
/* 21289 */ {(13<<2)|2,{32,74,0}},
/* 21290 */ {(13<<2)|2,{32,74,0}},
/* 21291 */ {(13<<2)|2,{32,74,0}},
/* 21292 */ {(13<<2)|2,{32,74,0}},
/* 21293 */ {(13<<2)|2,{32,74,0}},
/* 21294 */ {(13<<2)|2,{32,74,0}},
/* 21295 */ {(13<<2)|2,{32,74,0}},
/* 21296 */ {(13<<2)|2,{32,75,0}},
/* 21297 */ {(13<<2)|2,{32,75,0}},
/* 21298 */ {(13<<2)|2,{32,75,0}},
/* 21299 */ {(13<<2)|2,{32,75,0}},
/* 21300 */ {(13<<2)|2,{32,75,0}},
/* 21301 */ {(13<<2)|2,{32,75,0}},
/* 21302 */ {(13<<2)|2,{32,75,0}},
/* 21303 */ {(13<<2)|2,{32,75,0}},
/* 21304 */ {(13<<2)|2,{32,76,0}},
/* 21305 */ {(13<<2)|2,{32,76,0}},
/* 21306 */ {(13<<2)|2,{32,76,0}},
/* 21307 */ {(13<<2)|2,{32,76,0}},
/* 21308 */ {(13<<2)|2,{32,76,0}},
/* 21309 */ {(13<<2)|2,{32,76,0}},
/* 21310 */ {(13<<2)|2,{32,76,0}},
/* 21311 */ {(13<<2)|2,{32,76,0}},
/* 21312 */ {(13<<2)|2,{32,77,0}},
/* 21313 */ {(13<<2)|2,{32,77,0}},
/* 21314 */ {(13<<2)|2,{32,77,0}},
/* 21315 */ {(13<<2)|2,{32,77,0}},
/* 21316 */ {(13<<2)|2,{32,77,0}},
/* 21317 */ {(13<<2)|2,{32,77,0}},
/* 21318 */ {(13<<2)|2,{32,77,0}},
/* 21319 */ {(13<<2)|2,{32,77,0}},
/* 21320 */ {(13<<2)|2,{32,78,0}},
/* 21321 */ {(13<<2)|2,{32,78,0}},
/* 21322 */ {(13<<2)|2,{32,78,0}},
/* 21323 */ {(13<<2)|2,{32,78,0}},
/* 21324 */ {(13<<2)|2,{32,78,0}},
/* 21325 */ {(13<<2)|2,{32,78,0}},
/* 21326 */ {(13<<2)|2,{32,78,0}},
/* 21327 */ {(13<<2)|2,{32,78,0}},
/* 21328 */ {(13<<2)|2,{32,79,0}},
/* 21329 */ {(13<<2)|2,{32,79,0}},
/* 21330 */ {(13<<2)|2,{32,79,0}},
/* 21331 */ {(13<<2)|2,{32,79,0}},
/* 21332 */ {(13<<2)|2,{32,79,0}},
/* 21333 */ {(13<<2)|2,{32,79,0}},
/* 21334 */ {(13<<2)|2,{32,79,0}},
/* 21335 */ {(13<<2)|2,{32,79,0}},
/* 21336 */ {(13<<2)|2,{32,80,0}},
/* 21337 */ {(13<<2)|2,{32,80,0}},
/* 21338 */ {(13<<2)|2,{32,80,0}},
/* 21339 */ {(13<<2)|2,{32,80,0}},
/* 21340 */ {(13<<2)|2,{32,80,0}},
/* 21341 */ {(13<<2)|2,{32,80,0}},
/* 21342 */ {(13<<2)|2,{32,80,0}},
/* 21343 */ {(13<<2)|2,{32,80,0}},
/* 21344 */ {(13<<2)|2,{32,81,0}},
/* 21345 */ {(13<<2)|2,{32,81,0}},
/* 21346 */ {(13<<2)|2,{32,81,0}},
/* 21347 */ {(13<<2)|2,{32,81,0}},
/* 21348 */ {(13<<2)|2,{32,81,0}},
/* 21349 */ {(13<<2)|2,{32,81,0}},
/* 21350 */ {(13<<2)|2,{32,81,0}},
/* 21351 */ {(13<<2)|2,{32,81,0}},
/* 21352 */ {(13<<2)|2,{32,82,0}},
/* 21353 */ {(13<<2)|2,{32,82,0}},
/* 21354 */ {(13<<2)|2,{32,82,0}},
/* 21355 */ {(13<<2)|2,{32,82,0}},
/* 21356 */ {(13<<2)|2,{32,82,0}},
/* 21357 */ {(13<<2)|2,{32,82,0}},
/* 21358 */ {(13<<2)|2,{32,82,0}},
/* 21359 */ {(13<<2)|2,{32,82,0}},
/* 21360 */ {(13<<2)|2,{32,83,0}},
/* 21361 */ {(13<<2)|2,{32,83,0}},
/* 21362 */ {(13<<2)|2,{32,83,0}},
/* 21363 */ {(13<<2)|2,{32,83,0}},
/* 21364 */ {(13<<2)|2,{32,83,0}},
/* 21365 */ {(13<<2)|2,{32,83,0}},
/* 21366 */ {(13<<2)|2,{32,83,0}},
/* 21367 */ {(13<<2)|2,{32,83,0}},
/* 21368 */ {(13<<2)|2,{32,84,0}},
/* 21369 */ {(13<<2)|2,{32,84,0}},
/* 21370 */ {(13<<2)|2,{32,84,0}},
/* 21371 */ {(13<<2)|2,{32,84,0}},
/* 21372 */ {(13<<2)|2,{32,84,0}},
/* 21373 */ {(13<<2)|2,{32,84,0}},
/* 21374 */ {(13<<2)|2,{32,84,0}},
/* 21375 */ {(13<<2)|2,{32,84,0}},
/* 21376 */ {(13<<2)|2,{32,85,0}},
/* 21377 */ {(13<<2)|2,{32,85,0}},
/* 21378 */ {(13<<2)|2,{32,85,0}},
/* 21379 */ {(13<<2)|2,{32,85,0}},
/* 21380 */ {(13<<2)|2,{32,85,0}},
/* 21381 */ {(13<<2)|2,{32,85,0}},
/* 21382 */ {(13<<2)|2,{32,85,0}},
/* 21383 */ {(13<<2)|2,{32,85,0}},
/* 21384 */ {(13<<2)|2,{32,86,0}},
/* 21385 */ {(13<<2)|2,{32,86,0}},
/* 21386 */ {(13<<2)|2,{32,86,0}},
/* 21387 */ {(13<<2)|2,{32,86,0}},
/* 21388 */ {(13<<2)|2,{32,86,0}},
/* 21389 */ {(13<<2)|2,{32,86,0}},
/* 21390 */ {(13<<2)|2,{32,86,0}},
/* 21391 */ {(13<<2)|2,{32,86,0}},
/* 21392 */ {(13<<2)|2,{32,87,0}},
/* 21393 */ {(13<<2)|2,{32,87,0}},
/* 21394 */ {(13<<2)|2,{32,87,0}},
/* 21395 */ {(13<<2)|2,{32,87,0}},
/* 21396 */ {(13<<2)|2,{32,87,0}},
/* 21397 */ {(13<<2)|2,{32,87,0}},
/* 21398 */ {(13<<2)|2,{32,87,0}},
/* 21399 */ {(13<<2)|2,{32,87,0}},
/* 21400 */ {(13<<2)|2,{32,89,0}},
/* 21401 */ {(13<<2)|2,{32,89,0}},
/* 21402 */ {(13<<2)|2,{32,89,0}},
/* 21403 */ {(13<<2)|2,{32,89,0}},
/* 21404 */ {(13<<2)|2,{32,89,0}},
/* 21405 */ {(13<<2)|2,{32,89,0}},
/* 21406 */ {(13<<2)|2,{32,89,0}},
/* 21407 */ {(13<<2)|2,{32,89,0}},
/* 21408 */ {(13<<2)|2,{32,106,0}},
/* 21409 */ {(13<<2)|2,{32,106,0}},
/* 21410 */ {(13<<2)|2,{32,106,0}},
/* 21411 */ {(13<<2)|2,{32,106,0}},
/* 21412 */ {(13<<2)|2,{32,106,0}},
/* 21413 */ {(13<<2)|2,{32,106,0}},
/* 21414 */ {(13<<2)|2,{32,106,0}},
/* 21415 */ {(13<<2)|2,{32,106,0}},
/* 21416 */ {(13<<2)|2,{32,107,0}},
/* 21417 */ {(13<<2)|2,{32,107,0}},
/* 21418 */ {(13<<2)|2,{32,107,0}},
/* 21419 */ {(13<<2)|2,{32,107,0}},
/* 21420 */ {(13<<2)|2,{32,107,0}},
/* 21421 */ {(13<<2)|2,{32,107,0}},
/* 21422 */ {(13<<2)|2,{32,107,0}},
/* 21423 */ {(13<<2)|2,{32,107,0}},
/* 21424 */ {(13<<2)|2,{32,113,0}},
/* 21425 */ {(13<<2)|2,{32,113,0}},
/* 21426 */ {(13<<2)|2,{32,113,0}},
/* 21427 */ {(13<<2)|2,{32,113,0}},
/* 21428 */ {(13<<2)|2,{32,113,0}},
/* 21429 */ {(13<<2)|2,{32,113,0}},
/* 21430 */ {(13<<2)|2,{32,113,0}},
/* 21431 */ {(13<<2)|2,{32,113,0}},
/* 21432 */ {(13<<2)|2,{32,118,0}},
/* 21433 */ {(13<<2)|2,{32,118,0}},
/* 21434 */ {(13<<2)|2,{32,118,0}},
/* 21435 */ {(13<<2)|2,{32,118,0}},
/* 21436 */ {(13<<2)|2,{32,118,0}},
/* 21437 */ {(13<<2)|2,{32,118,0}},
/* 21438 */ {(13<<2)|2,{32,118,0}},
/* 21439 */ {(13<<2)|2,{32,118,0}},
/* 21440 */ {(13<<2)|2,{32,119,0}},
/* 21441 */ {(13<<2)|2,{32,119,0}},
/* 21442 */ {(13<<2)|2,{32,119,0}},
/* 21443 */ {(13<<2)|2,{32,119,0}},
/* 21444 */ {(13<<2)|2,{32,119,0}},
/* 21445 */ {(13<<2)|2,{32,119,0}},
/* 21446 */ {(13<<2)|2,{32,119,0}},
/* 21447 */ {(13<<2)|2,{32,119,0}},
/* 21448 */ {(13<<2)|2,{32,120,0}},
/* 21449 */ {(13<<2)|2,{32,120,0}},
/* 21450 */ {(13<<2)|2,{32,120,0}},
/* 21451 */ {(13<<2)|2,{32,120,0}},
/* 21452 */ {(13<<2)|2,{32,120,0}},
/* 21453 */ {(13<<2)|2,{32,120,0}},
/* 21454 */ {(13<<2)|2,{32,120,0}},
/* 21455 */ {(13<<2)|2,{32,120,0}},
/* 21456 */ {(13<<2)|2,{32,121,0}},
/* 21457 */ {(13<<2)|2,{32,121,0}},
/* 21458 */ {(13<<2)|2,{32,121,0}},
/* 21459 */ {(13<<2)|2,{32,121,0}},
/* 21460 */ {(13<<2)|2,{32,121,0}},
/* 21461 */ {(13<<2)|2,{32,121,0}},
/* 21462 */ {(13<<2)|2,{32,121,0}},
/* 21463 */ {(13<<2)|2,{32,121,0}},
/* 21464 */ {(13<<2)|2,{32,122,0}},
/* 21465 */ {(13<<2)|2,{32,122,0}},
/* 21466 */ {(13<<2)|2,{32,122,0}},
/* 21467 */ {(13<<2)|2,{32,122,0}},
/* 21468 */ {(13<<2)|2,{32,122,0}},
/* 21469 */ {(13<<2)|2,{32,122,0}},
/* 21470 */ {(13<<2)|2,{32,122,0}},
/* 21471 */ {(13<<2)|2,{32,122,0}},
/* 21472 */ {(14<<2)|2,{32,38,0}},
/* 21473 */ {(14<<2)|2,{32,38,0}},
/* 21474 */ {(14<<2)|2,{32,38,0}},
/* 21475 */ {(14<<2)|2,{32,38,0}},
/* 21476 */ {(14<<2)|2,{32,42,0}},
/* 21477 */ {(14<<2)|2,{32,42,0}},
/* 21478 */ {(14<<2)|2,{32,42,0}},
/* 21479 */ {(14<<2)|2,{32,42,0}},
/* 21480 */ {(14<<2)|2,{32,44,0}},
/* 21481 */ {(14<<2)|2,{32,44,0}},
/* 21482 */ {(14<<2)|2,{32,44,0}},
/* 21483 */ {(14<<2)|2,{32,44,0}},
/* 21484 */ {(14<<2)|2,{32,59,0}},
/* 21485 */ {(14<<2)|2,{32,59,0}},
/* 21486 */ {(14<<2)|2,{32,59,0}},
/* 21487 */ {(14<<2)|2,{32,59,0}},
/* 21488 */ {(14<<2)|2,{32,88,0}},
/* 21489 */ {(14<<2)|2,{32,88,0}},
/* 21490 */ {(14<<2)|2,{32,88,0}},
/* 21491 */ {(14<<2)|2,{32,88,0}},
/* 21492 */ {(14<<2)|2,{32,90,0}},
/* 21493 */ {(14<<2)|2,{32,90,0}},
/* 21494 */ {(14<<2)|2,{32,90,0}},
/* 21495 */ {(14<<2)|2,{32,90,0}},
/* 21496 */ {(16<<2)|2,{32,33,0}},
/* 21497 */ {(16<<2)|2,{32,34,0}},
/* 21498 */ {(16<<2)|2,{32,40,0}},
/* 21499 */ {(16<<2)|2,{32,41,0}},
/* 21500 */ {(16<<2)|2,{32,63,0}},
/* 21501 */ {(6<<2)|1,{32,0,0}},
/* 21502 */ {(6<<2)|1,{32,0,0}},
/* 21503 */ {(6<<2)|1,{32,0,0}},
/* 21504 */ {(16<<2)|3,{37,48,48}},
/* 21505 */ {(16<<2)|3,{37,48,49}},
/* 21506 */ {(16<<2)|3,{37,48,50}},
/* 21507 */ {(16<<2)|3,{37,48,97}},
/* 21508 */ {(16<<2)|3,{37,48,99}},
/* 21509 */ {(16<<2)|3,{37,48,101}},
/* 21510 */ {(16<<2)|3,{37,48,105}},
/* 21511 */ {(16<<2)|3,{37,48,111}},
/* 21512 */ {(16<<2)|3,{37,48,115}},
/* 21513 */ {(16<<2)|3,{37,48,116}},
/* 21514 */ {(11<<2)|2,{37,48,0}},
/* 21515 */ {(11<<2)|2,{37,48,0}},
/* 21516 */ {(11<<2)|2,{37,48,0}},
/* 21517 */ {(11<<2)|2,{37,48,0}},
/* 21518 */ {(11<<2)|2,{37,48,0}},
/* 21519 */ {(11<<2)|2,{37,48,0}},
/* 21520 */ {(11<<2)|2,{37,48,0}},
/* 21521 */ {(11<<2)|2,{37,48,0}},
/* 21522 */ {(11<<2)|2,{37,48,0}},
/* 21523 */ {(11<<2)|2,{37,48,0}},
/* 21524 */ {(11<<2)|2,{37,48,0}},
/* 21525 */ {(11<<2)|2,{37,48,0}},
/* 21526 */ {(11<<2)|2,{37,48,0}},
/* 21527 */ {(11<<2)|2,{37,48,0}},
/* 21528 */ {(11<<2)|2,{37,48,0}},
/* 21529 */ {(11<<2)|2,{37,48,0}},
/* 21530 */ {(11<<2)|2,{37,48,0}},
/* 21531 */ {(11<<2)|2,{37,48,0}},
/* 21532 */ {(11<<2)|2,{37,48,0}},
/* 21533 */ {(11<<2)|2,{37,48,0}},
/* 21534 */ {(11<<2)|2,{37,48,0}},
/* 21535 */ {(11<<2)|2,{37,48,0}},
/* 21536 */ {(16<<2)|3,{37,49,48}},
/* 21537 */ {(16<<2)|3,{37,49,49}},
/* 21538 */ {(16<<2)|3,{37,49,50}},
/* 21539 */ {(16<<2)|3,{37,49,97}},
/* 21540 */ {(16<<2)|3,{37,49,99}},
/* 21541 */ {(16<<2)|3,{37,49,101}},
/* 21542 */ {(16<<2)|3,{37,49,105}},
/* 21543 */ {(16<<2)|3,{37,49,111}},
/* 21544 */ {(16<<2)|3,{37,49,115}},
/* 21545 */ {(16<<2)|3,{37,49,116}},
/* 21546 */ {(11<<2)|2,{37,49,0}},
/* 21547 */ {(11<<2)|2,{37,49,0}},
/* 21548 */ {(11<<2)|2,{37,49,0}},
/* 21549 */ {(11<<2)|2,{37,49,0}},
/* 21550 */ {(11<<2)|2,{37,49,0}},
/* 21551 */ {(11<<2)|2,{37,49,0}},
/* 21552 */ {(11<<2)|2,{37,49,0}},
/* 21553 */ {(11<<2)|2,{37,49,0}},
/* 21554 */ {(11<<2)|2,{37,49,0}},
/* 21555 */ {(11<<2)|2,{37,49,0}},
/* 21556 */ {(11<<2)|2,{37,49,0}},
/* 21557 */ {(11<<2)|2,{37,49,0}},
/* 21558 */ {(11<<2)|2,{37,49,0}},
/* 21559 */ {(11<<2)|2,{37,49,0}},
/* 21560 */ {(11<<2)|2,{37,49,0}},
/* 21561 */ {(11<<2)|2,{37,49,0}},
/* 21562 */ {(11<<2)|2,{37,49,0}},
/* 21563 */ {(11<<2)|2,{37,49,0}},
/* 21564 */ {(11<<2)|2,{37,49,0}},
/* 21565 */ {(11<<2)|2,{37,49,0}},
/* 21566 */ {(11<<2)|2,{37,49,0}},
/* 21567 */ {(11<<2)|2,{37,49,0}},
/* 21568 */ {(16<<2)|3,{37,50,48}},
/* 21569 */ {(16<<2)|3,{37,50,49}},
/* 21570 */ {(16<<2)|3,{37,50,50}},
/* 21571 */ {(16<<2)|3,{37,50,97}},
/* 21572 */ {(16<<2)|3,{37,50,99}},
/* 21573 */ {(16<<2)|3,{37,50,101}},
/* 21574 */ {(16<<2)|3,{37,50,105}},
/* 21575 */ {(16<<2)|3,{37,50,111}},
/* 21576 */ {(16<<2)|3,{37,50,115}},
/* 21577 */ {(16<<2)|3,{37,50,116}},
/* 21578 */ {(11<<2)|2,{37,50,0}},
/* 21579 */ {(11<<2)|2,{37,50,0}},
/* 21580 */ {(11<<2)|2,{37,50,0}},
/* 21581 */ {(11<<2)|2,{37,50,0}},
/* 21582 */ {(11<<2)|2,{37,50,0}},
/* 21583 */ {(11<<2)|2,{37,50,0}},
/* 21584 */ {(11<<2)|2,{37,50,0}},
/* 21585 */ {(11<<2)|2,{37,50,0}},
/* 21586 */ {(11<<2)|2,{37,50,0}},
/* 21587 */ {(11<<2)|2,{37,50,0}},
/* 21588 */ {(11<<2)|2,{37,50,0}},
/* 21589 */ {(11<<2)|2,{37,50,0}},
/* 21590 */ {(11<<2)|2,{37,50,0}},
/* 21591 */ {(11<<2)|2,{37,50,0}},
/* 21592 */ {(11<<2)|2,{37,50,0}},
/* 21593 */ {(11<<2)|2,{37,50,0}},
/* 21594 */ {(11<<2)|2,{37,50,0}},
/* 21595 */ {(11<<2)|2,{37,50,0}},
/* 21596 */ {(11<<2)|2,{37,50,0}},
/* 21597 */ {(11<<2)|2,{37,50,0}},
/* 21598 */ {(11<<2)|2,{37,50,0}},
/* 21599 */ {(11<<2)|2,{37,50,0}},
/* 21600 */ {(16<<2)|3,{37,97,48}},
/* 21601 */ {(16<<2)|3,{37,97,49}},
/* 21602 */ {(16<<2)|3,{37,97,50}},
/* 21603 */ {(16<<2)|3,{37,97,97}},
/* 21604 */ {(16<<2)|3,{37,97,99}},
/* 21605 */ {(16<<2)|3,{37,97,101}},
/* 21606 */ {(16<<2)|3,{37,97,105}},
/* 21607 */ {(16<<2)|3,{37,97,111}},
/* 21608 */ {(16<<2)|3,{37,97,115}},
/* 21609 */ {(16<<2)|3,{37,97,116}},
/* 21610 */ {(11<<2)|2,{37,97,0}},
/* 21611 */ {(11<<2)|2,{37,97,0}},
/* 21612 */ {(11<<2)|2,{37,97,0}},
/* 21613 */ {(11<<2)|2,{37,97,0}},
/* 21614 */ {(11<<2)|2,{37,97,0}},
/* 21615 */ {(11<<2)|2,{37,97,0}},
/* 21616 */ {(11<<2)|2,{37,97,0}},
/* 21617 */ {(11<<2)|2,{37,97,0}},
/* 21618 */ {(11<<2)|2,{37,97,0}},
/* 21619 */ {(11<<2)|2,{37,97,0}},
/* 21620 */ {(11<<2)|2,{37,97,0}},
/* 21621 */ {(11<<2)|2,{37,97,0}},
/* 21622 */ {(11<<2)|2,{37,97,0}},
/* 21623 */ {(11<<2)|2,{37,97,0}},
/* 21624 */ {(11<<2)|2,{37,97,0}},
/* 21625 */ {(11<<2)|2,{37,97,0}},
/* 21626 */ {(11<<2)|2,{37,97,0}},
/* 21627 */ {(11<<2)|2,{37,97,0}},
/* 21628 */ {(11<<2)|2,{37,97,0}},
/* 21629 */ {(11<<2)|2,{37,97,0}},
/* 21630 */ {(11<<2)|2,{37,97,0}},
/* 21631 */ {(11<<2)|2,{37,97,0}},
/* 21632 */ {(16<<2)|3,{37,99,48}},
/* 21633 */ {(16<<2)|3,{37,99,49}},
/* 21634 */ {(16<<2)|3,{37,99,50}},
/* 21635 */ {(16<<2)|3,{37,99,97}},
/* 21636 */ {(16<<2)|3,{37,99,99}},
/* 21637 */ {(16<<2)|3,{37,99,101}},
/* 21638 */ {(16<<2)|3,{37,99,105}},
/* 21639 */ {(16<<2)|3,{37,99,111}},
/* 21640 */ {(16<<2)|3,{37,99,115}},
/* 21641 */ {(16<<2)|3,{37,99,116}},
/* 21642 */ {(11<<2)|2,{37,99,0}},
/* 21643 */ {(11<<2)|2,{37,99,0}},
/* 21644 */ {(11<<2)|2,{37,99,0}},
/* 21645 */ {(11<<2)|2,{37,99,0}},
/* 21646 */ {(11<<2)|2,{37,99,0}},
/* 21647 */ {(11<<2)|2,{37,99,0}},
/* 21648 */ {(11<<2)|2,{37,99,0}},
/* 21649 */ {(11<<2)|2,{37,99,0}},
/* 21650 */ {(11<<2)|2,{37,99,0}},
/* 21651 */ {(11<<2)|2,{37,99,0}},
/* 21652 */ {(11<<2)|2,{37,99,0}},
/* 21653 */ {(11<<2)|2,{37,99,0}},
/* 21654 */ {(11<<2)|2,{37,99,0}},
/* 21655 */ {(11<<2)|2,{37,99,0}},
/* 21656 */ {(11<<2)|2,{37,99,0}},
/* 21657 */ {(11<<2)|2,{37,99,0}},
/* 21658 */ {(11<<2)|2,{37,99,0}},
/* 21659 */ {(11<<2)|2,{37,99,0}},
/* 21660 */ {(11<<2)|2,{37,99,0}},
/* 21661 */ {(11<<2)|2,{37,99,0}},
/* 21662 */ {(11<<2)|2,{37,99,0}},
/* 21663 */ {(11<<2)|2,{37,99,0}},
/* 21664 */ {(16<<2)|3,{37,101,48}},
/* 21665 */ {(16<<2)|3,{37,101,49}},
/* 21666 */ {(16<<2)|3,{37,101,50}},
/* 21667 */ {(16<<2)|3,{37,101,97}},
/* 21668 */ {(16<<2)|3,{37,101,99}},
/* 21669 */ {(16<<2)|3,{37,101,101}},
/* 21670 */ {(16<<2)|3,{37,101,105}},
/* 21671 */ {(16<<2)|3,{37,101,111}},
/* 21672 */ {(16<<2)|3,{37,101,115}},
/* 21673 */ {(16<<2)|3,{37,101,116}},
/* 21674 */ {(11<<2)|2,{37,101,0}},
/* 21675 */ {(11<<2)|2,{37,101,0}},
/* 21676 */ {(11<<2)|2,{37,101,0}},
/* 21677 */ {(11<<2)|2,{37,101,0}},
/* 21678 */ {(11<<2)|2,{37,101,0}},
/* 21679 */ {(11<<2)|2,{37,101,0}},
/* 21680 */ {(11<<2)|2,{37,101,0}},
/* 21681 */ {(11<<2)|2,{37,101,0}},
/* 21682 */ {(11<<2)|2,{37,101,0}},
/* 21683 */ {(11<<2)|2,{37,101,0}},
/* 21684 */ {(11<<2)|2,{37,101,0}},
/* 21685 */ {(11<<2)|2,{37,101,0}},
/* 21686 */ {(11<<2)|2,{37,101,0}},
/* 21687 */ {(11<<2)|2,{37,101,0}},
/* 21688 */ {(11<<2)|2,{37,101,0}},
/* 21689 */ {(11<<2)|2,{37,101,0}},
/* 21690 */ {(11<<2)|2,{37,101,0}},
/* 21691 */ {(11<<2)|2,{37,101,0}},
/* 21692 */ {(11<<2)|2,{37,101,0}},
/* 21693 */ {(11<<2)|2,{37,101,0}},
/* 21694 */ {(11<<2)|2,{37,101,0}},
/* 21695 */ {(11<<2)|2,{37,101,0}},
/* 21696 */ {(16<<2)|3,{37,105,48}},
/* 21697 */ {(16<<2)|3,{37,105,49}},
/* 21698 */ {(16<<2)|3,{37,105,50}},
/* 21699 */ {(16<<2)|3,{37,105,97}},
/* 21700 */ {(16<<2)|3,{37,105,99}},
/* 21701 */ {(16<<2)|3,{37,105,101}},
/* 21702 */ {(16<<2)|3,{37,105,105}},
/* 21703 */ {(16<<2)|3,{37,105,111}},
/* 21704 */ {(16<<2)|3,{37,105,115}},
/* 21705 */ {(16<<2)|3,{37,105,116}},
/* 21706 */ {(11<<2)|2,{37,105,0}},
/* 21707 */ {(11<<2)|2,{37,105,0}},
/* 21708 */ {(11<<2)|2,{37,105,0}},
/* 21709 */ {(11<<2)|2,{37,105,0}},
/* 21710 */ {(11<<2)|2,{37,105,0}},
/* 21711 */ {(11<<2)|2,{37,105,0}},
/* 21712 */ {(11<<2)|2,{37,105,0}},
/* 21713 */ {(11<<2)|2,{37,105,0}},
/* 21714 */ {(11<<2)|2,{37,105,0}},
/* 21715 */ {(11<<2)|2,{37,105,0}},
/* 21716 */ {(11<<2)|2,{37,105,0}},
/* 21717 */ {(11<<2)|2,{37,105,0}},
/* 21718 */ {(11<<2)|2,{37,105,0}},
/* 21719 */ {(11<<2)|2,{37,105,0}},
/* 21720 */ {(11<<2)|2,{37,105,0}},
/* 21721 */ {(11<<2)|2,{37,105,0}},
/* 21722 */ {(11<<2)|2,{37,105,0}},
/* 21723 */ {(11<<2)|2,{37,105,0}},
/* 21724 */ {(11<<2)|2,{37,105,0}},
/* 21725 */ {(11<<2)|2,{37,105,0}},
/* 21726 */ {(11<<2)|2,{37,105,0}},
/* 21727 */ {(11<<2)|2,{37,105,0}},
/* 21728 */ {(16<<2)|3,{37,111,48}},
/* 21729 */ {(16<<2)|3,{37,111,49}},
/* 21730 */ {(16<<2)|3,{37,111,50}},
/* 21731 */ {(16<<2)|3,{37,111,97}},
/* 21732 */ {(16<<2)|3,{37,111,99}},
/* 21733 */ {(16<<2)|3,{37,111,101}},
/* 21734 */ {(16<<2)|3,{37,111,105}},
/* 21735 */ {(16<<2)|3,{37,111,111}},
/* 21736 */ {(16<<2)|3,{37,111,115}},
/* 21737 */ {(16<<2)|3,{37,111,116}},
/* 21738 */ {(11<<2)|2,{37,111,0}},
/* 21739 */ {(11<<2)|2,{37,111,0}},
/* 21740 */ {(11<<2)|2,{37,111,0}},
/* 21741 */ {(11<<2)|2,{37,111,0}},
/* 21742 */ {(11<<2)|2,{37,111,0}},
/* 21743 */ {(11<<2)|2,{37,111,0}},
/* 21744 */ {(11<<2)|2,{37,111,0}},
/* 21745 */ {(11<<2)|2,{37,111,0}},
/* 21746 */ {(11<<2)|2,{37,111,0}},
/* 21747 */ {(11<<2)|2,{37,111,0}},
/* 21748 */ {(11<<2)|2,{37,111,0}},
/* 21749 */ {(11<<2)|2,{37,111,0}},
/* 21750 */ {(11<<2)|2,{37,111,0}},
/* 21751 */ {(11<<2)|2,{37,111,0}},
/* 21752 */ {(11<<2)|2,{37,111,0}},
/* 21753 */ {(11<<2)|2,{37,111,0}},
/* 21754 */ {(11<<2)|2,{37,111,0}},
/* 21755 */ {(11<<2)|2,{37,111,0}},
/* 21756 */ {(11<<2)|2,{37,111,0}},
/* 21757 */ {(11<<2)|2,{37,111,0}},
/* 21758 */ {(11<<2)|2,{37,111,0}},
/* 21759 */ {(11<<2)|2,{37,111,0}},
/* 21760 */ {(16<<2)|3,{37,115,48}},
/* 21761 */ {(16<<2)|3,{37,115,49}},
/* 21762 */ {(16<<2)|3,{37,115,50}},
/* 21763 */ {(16<<2)|3,{37,115,97}},
/* 21764 */ {(16<<2)|3,{37,115,99}},
/* 21765 */ {(16<<2)|3,{37,115,101}},
/* 21766 */ {(16<<2)|3,{37,115,105}},
/* 21767 */ {(16<<2)|3,{37,115,111}},
/* 21768 */ {(16<<2)|3,{37,115,115}},
/* 21769 */ {(16<<2)|3,{37,115,116}},
/* 21770 */ {(11<<2)|2,{37,115,0}},
/* 21771 */ {(11<<2)|2,{37,115,0}},
/* 21772 */ {(11<<2)|2,{37,115,0}},
/* 21773 */ {(11<<2)|2,{37,115,0}},
/* 21774 */ {(11<<2)|2,{37,115,0}},
/* 21775 */ {(11<<2)|2,{37,115,0}},
/* 21776 */ {(11<<2)|2,{37,115,0}},
/* 21777 */ {(11<<2)|2,{37,115,0}},
/* 21778 */ {(11<<2)|2,{37,115,0}},
/* 21779 */ {(11<<2)|2,{37,115,0}},
/* 21780 */ {(11<<2)|2,{37,115,0}},
/* 21781 */ {(11<<2)|2,{37,115,0}},
/* 21782 */ {(11<<2)|2,{37,115,0}},
/* 21783 */ {(11<<2)|2,{37,115,0}},
/* 21784 */ {(11<<2)|2,{37,115,0}},
/* 21785 */ {(11<<2)|2,{37,115,0}},
/* 21786 */ {(11<<2)|2,{37,115,0}},
/* 21787 */ {(11<<2)|2,{37,115,0}},
/* 21788 */ {(11<<2)|2,{37,115,0}},
/* 21789 */ {(11<<2)|2,{37,115,0}},
/* 21790 */ {(11<<2)|2,{37,115,0}},
/* 21791 */ {(11<<2)|2,{37,115,0}},
/* 21792 */ {(16<<2)|3,{37,116,48}},
/* 21793 */ {(16<<2)|3,{37,116,49}},
/* 21794 */ {(16<<2)|3,{37,116,50}},
/* 21795 */ {(16<<2)|3,{37,116,97}},
/* 21796 */ {(16<<2)|3,{37,116,99}},
/* 21797 */ {(16<<2)|3,{37,116,101}},
/* 21798 */ {(16<<2)|3,{37,116,105}},
/* 21799 */ {(16<<2)|3,{37,116,111}},
/* 21800 */ {(16<<2)|3,{37,116,115}},
/* 21801 */ {(16<<2)|3,{37,116,116}},
/* 21802 */ {(11<<2)|2,{37,116,0}},
/* 21803 */ {(11<<2)|2,{37,116,0}},
/* 21804 */ {(11<<2)|2,{37,116,0}},
/* 21805 */ {(11<<2)|2,{37,116,0}},
/* 21806 */ {(11<<2)|2,{37,116,0}},
/* 21807 */ {(11<<2)|2,{37,116,0}},
/* 21808 */ {(11<<2)|2,{37,116,0}},
/* 21809 */ {(11<<2)|2,{37,116,0}},
/* 21810 */ {(11<<2)|2,{37,116,0}},
/* 21811 */ {(11<<2)|2,{37,116,0}},
/* 21812 */ {(11<<2)|2,{37,116,0}},
/* 21813 */ {(11<<2)|2,{37,116,0}},
/* 21814 */ {(11<<2)|2,{37,116,0}},
/* 21815 */ {(11<<2)|2,{37,116,0}},
/* 21816 */ {(11<<2)|2,{37,116,0}},
/* 21817 */ {(11<<2)|2,{37,116,0}},
/* 21818 */ {(11<<2)|2,{37,116,0}},
/* 21819 */ {(11<<2)|2,{37,116,0}},
/* 21820 */ {(11<<2)|2,{37,116,0}},
/* 21821 */ {(11<<2)|2,{37,116,0}},
/* 21822 */ {(11<<2)|2,{37,116,0}},
/* 21823 */ {(11<<2)|2,{37,116,0}},
/* 21824 */ {(12<<2)|2,{37,32,0}},
/* 21825 */ {(12<<2)|2,{37,32,0}},
/* 21826 */ {(12<<2)|2,{37,32,0}},
/* 21827 */ {(12<<2)|2,{37,32,0}},
/* 21828 */ {(12<<2)|2,{37,32,0}},
/* 21829 */ {(12<<2)|2,{37,32,0}},
/* 21830 */ {(12<<2)|2,{37,32,0}},
/* 21831 */ {(12<<2)|2,{37,32,0}},
/* 21832 */ {(12<<2)|2,{37,32,0}},
/* 21833 */ {(12<<2)|2,{37,32,0}},
/* 21834 */ {(12<<2)|2,{37,32,0}},
/* 21835 */ {(12<<2)|2,{37,32,0}},
/* 21836 */ {(12<<2)|2,{37,32,0}},
/* 21837 */ {(12<<2)|2,{37,32,0}},
/* 21838 */ {(12<<2)|2,{37,32,0}},
/* 21839 */ {(12<<2)|2,{37,32,0}},
/* 21840 */ {(12<<2)|2,{37,37,0}},
/* 21841 */ {(12<<2)|2,{37,37,0}},
/* 21842 */ {(12<<2)|2,{37,37,0}},
/* 21843 */ {(12<<2)|2,{37,37,0}},
/* 21844 */ {(12<<2)|2,{37,37,0}},
/* 21845 */ {(12<<2)|2,{37,37,0}},
/* 21846 */ {(12<<2)|2,{37,37,0}},
/* 21847 */ {(12<<2)|2,{37,37,0}},
/* 21848 */ {(12<<2)|2,{37,37,0}},
/* 21849 */ {(12<<2)|2,{37,37,0}},
/* 21850 */ {(12<<2)|2,{37,37,0}},
/* 21851 */ {(12<<2)|2,{37,37,0}},
/* 21852 */ {(12<<2)|2,{37,37,0}},
/* 21853 */ {(12<<2)|2,{37,37,0}},
/* 21854 */ {(12<<2)|2,{37,37,0}},
/* 21855 */ {(12<<2)|2,{37,37,0}},
/* 21856 */ {(12<<2)|2,{37,45,0}},
/* 21857 */ {(12<<2)|2,{37,45,0}},
/* 21858 */ {(12<<2)|2,{37,45,0}},
/* 21859 */ {(12<<2)|2,{37,45,0}},
/* 21860 */ {(12<<2)|2,{37,45,0}},
/* 21861 */ {(12<<2)|2,{37,45,0}},
/* 21862 */ {(12<<2)|2,{37,45,0}},
/* 21863 */ {(12<<2)|2,{37,45,0}},
/* 21864 */ {(12<<2)|2,{37,45,0}},
/* 21865 */ {(12<<2)|2,{37,45,0}},
/* 21866 */ {(12<<2)|2,{37,45,0}},
/* 21867 */ {(12<<2)|2,{37,45,0}},
/* 21868 */ {(12<<2)|2,{37,45,0}},
/* 21869 */ {(12<<2)|2,{37,45,0}},
/* 21870 */ {(12<<2)|2,{37,45,0}},
/* 21871 */ {(12<<2)|2,{37,45,0}},
/* 21872 */ {(12<<2)|2,{37,46,0}},
/* 21873 */ {(12<<2)|2,{37,46,0}},
/* 21874 */ {(12<<2)|2,{37,46,0}},
/* 21875 */ {(12<<2)|2,{37,46,0}},
/* 21876 */ {(12<<2)|2,{37,46,0}},
/* 21877 */ {(12<<2)|2,{37,46,0}},
/* 21878 */ {(12<<2)|2,{37,46,0}},
/* 21879 */ {(12<<2)|2,{37,46,0}},
/* 21880 */ {(12<<2)|2,{37,46,0}},
/* 21881 */ {(12<<2)|2,{37,46,0}},
/* 21882 */ {(12<<2)|2,{37,46,0}},
/* 21883 */ {(12<<2)|2,{37,46,0}},
/* 21884 */ {(12<<2)|2,{37,46,0}},
/* 21885 */ {(12<<2)|2,{37,46,0}},
/* 21886 */ {(12<<2)|2,{37,46,0}},
/* 21887 */ {(12<<2)|2,{37,46,0}},
/* 21888 */ {(12<<2)|2,{37,47,0}},
/* 21889 */ {(12<<2)|2,{37,47,0}},
/* 21890 */ {(12<<2)|2,{37,47,0}},
/* 21891 */ {(12<<2)|2,{37,47,0}},
/* 21892 */ {(12<<2)|2,{37,47,0}},
/* 21893 */ {(12<<2)|2,{37,47,0}},
/* 21894 */ {(12<<2)|2,{37,47,0}},
/* 21895 */ {(12<<2)|2,{37,47,0}},
/* 21896 */ {(12<<2)|2,{37,47,0}},
/* 21897 */ {(12<<2)|2,{37,47,0}},
/* 21898 */ {(12<<2)|2,{37,47,0}},
/* 21899 */ {(12<<2)|2,{37,47,0}},
/* 21900 */ {(12<<2)|2,{37,47,0}},
/* 21901 */ {(12<<2)|2,{37,47,0}},
/* 21902 */ {(12<<2)|2,{37,47,0}},
/* 21903 */ {(12<<2)|2,{37,47,0}},
/* 21904 */ {(12<<2)|2,{37,51,0}},
/* 21905 */ {(12<<2)|2,{37,51,0}},
/* 21906 */ {(12<<2)|2,{37,51,0}},
/* 21907 */ {(12<<2)|2,{37,51,0}},
/* 21908 */ {(12<<2)|2,{37,51,0}},
/* 21909 */ {(12<<2)|2,{37,51,0}},
/* 21910 */ {(12<<2)|2,{37,51,0}},
/* 21911 */ {(12<<2)|2,{37,51,0}},
/* 21912 */ {(12<<2)|2,{37,51,0}},
/* 21913 */ {(12<<2)|2,{37,51,0}},
/* 21914 */ {(12<<2)|2,{37,51,0}},
/* 21915 */ {(12<<2)|2,{37,51,0}},
/* 21916 */ {(12<<2)|2,{37,51,0}},
/* 21917 */ {(12<<2)|2,{37,51,0}},
/* 21918 */ {(12<<2)|2,{37,51,0}},
/* 21919 */ {(12<<2)|2,{37,51,0}},
/* 21920 */ {(12<<2)|2,{37,52,0}},
/* 21921 */ {(12<<2)|2,{37,52,0}},
/* 21922 */ {(12<<2)|2,{37,52,0}},
/* 21923 */ {(12<<2)|2,{37,52,0}},
/* 21924 */ {(12<<2)|2,{37,52,0}},
/* 21925 */ {(12<<2)|2,{37,52,0}},
/* 21926 */ {(12<<2)|2,{37,52,0}},
/* 21927 */ {(12<<2)|2,{37,52,0}},
/* 21928 */ {(12<<2)|2,{37,52,0}},
/* 21929 */ {(12<<2)|2,{37,52,0}},
/* 21930 */ {(12<<2)|2,{37,52,0}},
/* 21931 */ {(12<<2)|2,{37,52,0}},
/* 21932 */ {(12<<2)|2,{37,52,0}},
/* 21933 */ {(12<<2)|2,{37,52,0}},
/* 21934 */ {(12<<2)|2,{37,52,0}},
/* 21935 */ {(12<<2)|2,{37,52,0}},
/* 21936 */ {(12<<2)|2,{37,53,0}},
/* 21937 */ {(12<<2)|2,{37,53,0}},
/* 21938 */ {(12<<2)|2,{37,53,0}},
/* 21939 */ {(12<<2)|2,{37,53,0}},
/* 21940 */ {(12<<2)|2,{37,53,0}},
/* 21941 */ {(12<<2)|2,{37,53,0}},
/* 21942 */ {(12<<2)|2,{37,53,0}},
/* 21943 */ {(12<<2)|2,{37,53,0}},
/* 21944 */ {(12<<2)|2,{37,53,0}},
/* 21945 */ {(12<<2)|2,{37,53,0}},
/* 21946 */ {(12<<2)|2,{37,53,0}},
/* 21947 */ {(12<<2)|2,{37,53,0}},
/* 21948 */ {(12<<2)|2,{37,53,0}},
/* 21949 */ {(12<<2)|2,{37,53,0}},
/* 21950 */ {(12<<2)|2,{37,53,0}},
/* 21951 */ {(12<<2)|2,{37,53,0}},
/* 21952 */ {(12<<2)|2,{37,54,0}},
/* 21953 */ {(12<<2)|2,{37,54,0}},
/* 21954 */ {(12<<2)|2,{37,54,0}},
/* 21955 */ {(12<<2)|2,{37,54,0}},
/* 21956 */ {(12<<2)|2,{37,54,0}},
/* 21957 */ {(12<<2)|2,{37,54,0}},
/* 21958 */ {(12<<2)|2,{37,54,0}},
/* 21959 */ {(12<<2)|2,{37,54,0}},
/* 21960 */ {(12<<2)|2,{37,54,0}},
/* 21961 */ {(12<<2)|2,{37,54,0}},
/* 21962 */ {(12<<2)|2,{37,54,0}},
/* 21963 */ {(12<<2)|2,{37,54,0}},
/* 21964 */ {(12<<2)|2,{37,54,0}},
/* 21965 */ {(12<<2)|2,{37,54,0}},
/* 21966 */ {(12<<2)|2,{37,54,0}},
/* 21967 */ {(12<<2)|2,{37,54,0}},
/* 21968 */ {(12<<2)|2,{37,55,0}},
/* 21969 */ {(12<<2)|2,{37,55,0}},
/* 21970 */ {(12<<2)|2,{37,55,0}},
/* 21971 */ {(12<<2)|2,{37,55,0}},
/* 21972 */ {(12<<2)|2,{37,55,0}},
/* 21973 */ {(12<<2)|2,{37,55,0}},
/* 21974 */ {(12<<2)|2,{37,55,0}},
/* 21975 */ {(12<<2)|2,{37,55,0}},
/* 21976 */ {(12<<2)|2,{37,55,0}},
/* 21977 */ {(12<<2)|2,{37,55,0}},
/* 21978 */ {(12<<2)|2,{37,55,0}},
/* 21979 */ {(12<<2)|2,{37,55,0}},
/* 21980 */ {(12<<2)|2,{37,55,0}},
/* 21981 */ {(12<<2)|2,{37,55,0}},
/* 21982 */ {(12<<2)|2,{37,55,0}},
/* 21983 */ {(12<<2)|2,{37,55,0}},
/* 21984 */ {(12<<2)|2,{37,56,0}},
/* 21985 */ {(12<<2)|2,{37,56,0}},
/* 21986 */ {(12<<2)|2,{37,56,0}},
/* 21987 */ {(12<<2)|2,{37,56,0}},
/* 21988 */ {(12<<2)|2,{37,56,0}},
/* 21989 */ {(12<<2)|2,{37,56,0}},
/* 21990 */ {(12<<2)|2,{37,56,0}},
/* 21991 */ {(12<<2)|2,{37,56,0}},
/* 21992 */ {(12<<2)|2,{37,56,0}},
/* 21993 */ {(12<<2)|2,{37,56,0}},
/* 21994 */ {(12<<2)|2,{37,56,0}},
/* 21995 */ {(12<<2)|2,{37,56,0}},
/* 21996 */ {(12<<2)|2,{37,56,0}},
/* 21997 */ {(12<<2)|2,{37,56,0}},
/* 21998 */ {(12<<2)|2,{37,56,0}},
/* 21999 */ {(12<<2)|2,{37,56,0}},
/* 22000 */ {(12<<2)|2,{37,57,0}},
/* 22001 */ {(12<<2)|2,{37,57,0}},
/* 22002 */ {(12<<2)|2,{37,57,0}},
/* 22003 */ {(12<<2)|2,{37,57,0}},
/* 22004 */ {(12<<2)|2,{37,57,0}},
/* 22005 */ {(12<<2)|2,{37,57,0}},
/* 22006 */ {(12<<2)|2,{37,57,0}},
/* 22007 */ {(12<<2)|2,{37,57,0}},
/* 22008 */ {(12<<2)|2,{37,57,0}},
/* 22009 */ {(12<<2)|2,{37,57,0}},
/* 22010 */ {(12<<2)|2,{37,57,0}},
/* 22011 */ {(12<<2)|2,{37,57,0}},
/* 22012 */ {(12<<2)|2,{37,57,0}},
/* 22013 */ {(12<<2)|2,{37,57,0}},
/* 22014 */ {(12<<2)|2,{37,57,0}},
/* 22015 */ {(12<<2)|2,{37,57,0}},
/* 22016 */ {(12<<2)|2,{37,61,0}},
/* 22017 */ {(12<<2)|2,{37,61,0}},
/* 22018 */ {(12<<2)|2,{37,61,0}},
/* 22019 */ {(12<<2)|2,{37,61,0}},
/* 22020 */ {(12<<2)|2,{37,61,0}},
/* 22021 */ {(12<<2)|2,{37,61,0}},
/* 22022 */ {(12<<2)|2,{37,61,0}},
/* 22023 */ {(12<<2)|2,{37,61,0}},
/* 22024 */ {(12<<2)|2,{37,61,0}},
/* 22025 */ {(12<<2)|2,{37,61,0}},
/* 22026 */ {(12<<2)|2,{37,61,0}},
/* 22027 */ {(12<<2)|2,{37,61,0}},
/* 22028 */ {(12<<2)|2,{37,61,0}},
/* 22029 */ {(12<<2)|2,{37,61,0}},
/* 22030 */ {(12<<2)|2,{37,61,0}},
/* 22031 */ {(12<<2)|2,{37,61,0}},
/* 22032 */ {(12<<2)|2,{37,65,0}},
/* 22033 */ {(12<<2)|2,{37,65,0}},
/* 22034 */ {(12<<2)|2,{37,65,0}},
/* 22035 */ {(12<<2)|2,{37,65,0}},
/* 22036 */ {(12<<2)|2,{37,65,0}},
/* 22037 */ {(12<<2)|2,{37,65,0}},
/* 22038 */ {(12<<2)|2,{37,65,0}},
/* 22039 */ {(12<<2)|2,{37,65,0}},
/* 22040 */ {(12<<2)|2,{37,65,0}},
/* 22041 */ {(12<<2)|2,{37,65,0}},
/* 22042 */ {(12<<2)|2,{37,65,0}},
/* 22043 */ {(12<<2)|2,{37,65,0}},
/* 22044 */ {(12<<2)|2,{37,65,0}},
/* 22045 */ {(12<<2)|2,{37,65,0}},
/* 22046 */ {(12<<2)|2,{37,65,0}},
/* 22047 */ {(12<<2)|2,{37,65,0}},
/* 22048 */ {(12<<2)|2,{37,95,0}},
/* 22049 */ {(12<<2)|2,{37,95,0}},
/* 22050 */ {(12<<2)|2,{37,95,0}},
/* 22051 */ {(12<<2)|2,{37,95,0}},
/* 22052 */ {(12<<2)|2,{37,95,0}},
/* 22053 */ {(12<<2)|2,{37,95,0}},
/* 22054 */ {(12<<2)|2,{37,95,0}},
/* 22055 */ {(12<<2)|2,{37,95,0}},
/* 22056 */ {(12<<2)|2,{37,95,0}},
/* 22057 */ {(12<<2)|2,{37,95,0}},
/* 22058 */ {(12<<2)|2,{37,95,0}},
/* 22059 */ {(12<<2)|2,{37,95,0}},
/* 22060 */ {(12<<2)|2,{37,95,0}},
/* 22061 */ {(12<<2)|2,{37,95,0}},
/* 22062 */ {(12<<2)|2,{37,95,0}},
/* 22063 */ {(12<<2)|2,{37,95,0}},
/* 22064 */ {(12<<2)|2,{37,98,0}},
/* 22065 */ {(12<<2)|2,{37,98,0}},
/* 22066 */ {(12<<2)|2,{37,98,0}},
/* 22067 */ {(12<<2)|2,{37,98,0}},
/* 22068 */ {(12<<2)|2,{37,98,0}},
/* 22069 */ {(12<<2)|2,{37,98,0}},
/* 22070 */ {(12<<2)|2,{37,98,0}},
/* 22071 */ {(12<<2)|2,{37,98,0}},
/* 22072 */ {(12<<2)|2,{37,98,0}},
/* 22073 */ {(12<<2)|2,{37,98,0}},
/* 22074 */ {(12<<2)|2,{37,98,0}},
/* 22075 */ {(12<<2)|2,{37,98,0}},
/* 22076 */ {(12<<2)|2,{37,98,0}},
/* 22077 */ {(12<<2)|2,{37,98,0}},
/* 22078 */ {(12<<2)|2,{37,98,0}},
/* 22079 */ {(12<<2)|2,{37,98,0}},
/* 22080 */ {(12<<2)|2,{37,100,0}},
/* 22081 */ {(12<<2)|2,{37,100,0}},
/* 22082 */ {(12<<2)|2,{37,100,0}},
/* 22083 */ {(12<<2)|2,{37,100,0}},
/* 22084 */ {(12<<2)|2,{37,100,0}},
/* 22085 */ {(12<<2)|2,{37,100,0}},
/* 22086 */ {(12<<2)|2,{37,100,0}},
/* 22087 */ {(12<<2)|2,{37,100,0}},
/* 22088 */ {(12<<2)|2,{37,100,0}},
/* 22089 */ {(12<<2)|2,{37,100,0}},
/* 22090 */ {(12<<2)|2,{37,100,0}},
/* 22091 */ {(12<<2)|2,{37,100,0}},
/* 22092 */ {(12<<2)|2,{37,100,0}},
/* 22093 */ {(12<<2)|2,{37,100,0}},
/* 22094 */ {(12<<2)|2,{37,100,0}},
/* 22095 */ {(12<<2)|2,{37,100,0}},
/* 22096 */ {(12<<2)|2,{37,102,0}},
/* 22097 */ {(12<<2)|2,{37,102,0}},
/* 22098 */ {(12<<2)|2,{37,102,0}},
/* 22099 */ {(12<<2)|2,{37,102,0}},
/* 22100 */ {(12<<2)|2,{37,102,0}},
/* 22101 */ {(12<<2)|2,{37,102,0}},
/* 22102 */ {(12<<2)|2,{37,102,0}},
/* 22103 */ {(12<<2)|2,{37,102,0}},
/* 22104 */ {(12<<2)|2,{37,102,0}},
/* 22105 */ {(12<<2)|2,{37,102,0}},
/* 22106 */ {(12<<2)|2,{37,102,0}},
/* 22107 */ {(12<<2)|2,{37,102,0}},
/* 22108 */ {(12<<2)|2,{37,102,0}},
/* 22109 */ {(12<<2)|2,{37,102,0}},
/* 22110 */ {(12<<2)|2,{37,102,0}},
/* 22111 */ {(12<<2)|2,{37,102,0}},
/* 22112 */ {(12<<2)|2,{37,103,0}},
/* 22113 */ {(12<<2)|2,{37,103,0}},
/* 22114 */ {(12<<2)|2,{37,103,0}},
/* 22115 */ {(12<<2)|2,{37,103,0}},
/* 22116 */ {(12<<2)|2,{37,103,0}},
/* 22117 */ {(12<<2)|2,{37,103,0}},
/* 22118 */ {(12<<2)|2,{37,103,0}},
/* 22119 */ {(12<<2)|2,{37,103,0}},
/* 22120 */ {(12<<2)|2,{37,103,0}},
/* 22121 */ {(12<<2)|2,{37,103,0}},
/* 22122 */ {(12<<2)|2,{37,103,0}},
/* 22123 */ {(12<<2)|2,{37,103,0}},
/* 22124 */ {(12<<2)|2,{37,103,0}},
/* 22125 */ {(12<<2)|2,{37,103,0}},
/* 22126 */ {(12<<2)|2,{37,103,0}},
/* 22127 */ {(12<<2)|2,{37,103,0}},
/* 22128 */ {(12<<2)|2,{37,104,0}},
/* 22129 */ {(12<<2)|2,{37,104,0}},
/* 22130 */ {(12<<2)|2,{37,104,0}},
/* 22131 */ {(12<<2)|2,{37,104,0}},
/* 22132 */ {(12<<2)|2,{37,104,0}},
/* 22133 */ {(12<<2)|2,{37,104,0}},
/* 22134 */ {(12<<2)|2,{37,104,0}},
/* 22135 */ {(12<<2)|2,{37,104,0}},
/* 22136 */ {(12<<2)|2,{37,104,0}},
/* 22137 */ {(12<<2)|2,{37,104,0}},
/* 22138 */ {(12<<2)|2,{37,104,0}},
/* 22139 */ {(12<<2)|2,{37,104,0}},
/* 22140 */ {(12<<2)|2,{37,104,0}},
/* 22141 */ {(12<<2)|2,{37,104,0}},
/* 22142 */ {(12<<2)|2,{37,104,0}},
/* 22143 */ {(12<<2)|2,{37,104,0}},
/* 22144 */ {(12<<2)|2,{37,108,0}},
/* 22145 */ {(12<<2)|2,{37,108,0}},
/* 22146 */ {(12<<2)|2,{37,108,0}},
/* 22147 */ {(12<<2)|2,{37,108,0}},
/* 22148 */ {(12<<2)|2,{37,108,0}},
/* 22149 */ {(12<<2)|2,{37,108,0}},
/* 22150 */ {(12<<2)|2,{37,108,0}},
/* 22151 */ {(12<<2)|2,{37,108,0}},
/* 22152 */ {(12<<2)|2,{37,108,0}},
/* 22153 */ {(12<<2)|2,{37,108,0}},
/* 22154 */ {(12<<2)|2,{37,108,0}},
/* 22155 */ {(12<<2)|2,{37,108,0}},
/* 22156 */ {(12<<2)|2,{37,108,0}},
/* 22157 */ {(12<<2)|2,{37,108,0}},
/* 22158 */ {(12<<2)|2,{37,108,0}},
/* 22159 */ {(12<<2)|2,{37,108,0}},
/* 22160 */ {(12<<2)|2,{37,109,0}},
/* 22161 */ {(12<<2)|2,{37,109,0}},
/* 22162 */ {(12<<2)|2,{37,109,0}},
/* 22163 */ {(12<<2)|2,{37,109,0}},
/* 22164 */ {(12<<2)|2,{37,109,0}},
/* 22165 */ {(12<<2)|2,{37,109,0}},
/* 22166 */ {(12<<2)|2,{37,109,0}},
/* 22167 */ {(12<<2)|2,{37,109,0}},
/* 22168 */ {(12<<2)|2,{37,109,0}},
/* 22169 */ {(12<<2)|2,{37,109,0}},
/* 22170 */ {(12<<2)|2,{37,109,0}},
/* 22171 */ {(12<<2)|2,{37,109,0}},
/* 22172 */ {(12<<2)|2,{37,109,0}},
/* 22173 */ {(12<<2)|2,{37,109,0}},
/* 22174 */ {(12<<2)|2,{37,109,0}},
/* 22175 */ {(12<<2)|2,{37,109,0}},
/* 22176 */ {(12<<2)|2,{37,110,0}},
/* 22177 */ {(12<<2)|2,{37,110,0}},
/* 22178 */ {(12<<2)|2,{37,110,0}},
/* 22179 */ {(12<<2)|2,{37,110,0}},
/* 22180 */ {(12<<2)|2,{37,110,0}},
/* 22181 */ {(12<<2)|2,{37,110,0}},
/* 22182 */ {(12<<2)|2,{37,110,0}},
/* 22183 */ {(12<<2)|2,{37,110,0}},
/* 22184 */ {(12<<2)|2,{37,110,0}},
/* 22185 */ {(12<<2)|2,{37,110,0}},
/* 22186 */ {(12<<2)|2,{37,110,0}},
/* 22187 */ {(12<<2)|2,{37,110,0}},
/* 22188 */ {(12<<2)|2,{37,110,0}},
/* 22189 */ {(12<<2)|2,{37,110,0}},
/* 22190 */ {(12<<2)|2,{37,110,0}},
/* 22191 */ {(12<<2)|2,{37,110,0}},
/* 22192 */ {(12<<2)|2,{37,112,0}},
/* 22193 */ {(12<<2)|2,{37,112,0}},
/* 22194 */ {(12<<2)|2,{37,112,0}},
/* 22195 */ {(12<<2)|2,{37,112,0}},
/* 22196 */ {(12<<2)|2,{37,112,0}},
/* 22197 */ {(12<<2)|2,{37,112,0}},
/* 22198 */ {(12<<2)|2,{37,112,0}},
/* 22199 */ {(12<<2)|2,{37,112,0}},
/* 22200 */ {(12<<2)|2,{37,112,0}},
/* 22201 */ {(12<<2)|2,{37,112,0}},
/* 22202 */ {(12<<2)|2,{37,112,0}},
/* 22203 */ {(12<<2)|2,{37,112,0}},
/* 22204 */ {(12<<2)|2,{37,112,0}},
/* 22205 */ {(12<<2)|2,{37,112,0}},
/* 22206 */ {(12<<2)|2,{37,112,0}},
/* 22207 */ {(12<<2)|2,{37,112,0}},
/* 22208 */ {(12<<2)|2,{37,114,0}},
/* 22209 */ {(12<<2)|2,{37,114,0}},
/* 22210 */ {(12<<2)|2,{37,114,0}},
/* 22211 */ {(12<<2)|2,{37,114,0}},
/* 22212 */ {(12<<2)|2,{37,114,0}},
/* 22213 */ {(12<<2)|2,{37,114,0}},
/* 22214 */ {(12<<2)|2,{37,114,0}},
/* 22215 */ {(12<<2)|2,{37,114,0}},
/* 22216 */ {(12<<2)|2,{37,114,0}},
/* 22217 */ {(12<<2)|2,{37,114,0}},
/* 22218 */ {(12<<2)|2,{37,114,0}},
/* 22219 */ {(12<<2)|2,{37,114,0}},
/* 22220 */ {(12<<2)|2,{37,114,0}},
/* 22221 */ {(12<<2)|2,{37,114,0}},
/* 22222 */ {(12<<2)|2,{37,114,0}},
/* 22223 */ {(12<<2)|2,{37,114,0}},
/* 22224 */ {(12<<2)|2,{37,117,0}},
/* 22225 */ {(12<<2)|2,{37,117,0}},
/* 22226 */ {(12<<2)|2,{37,117,0}},
/* 22227 */ {(12<<2)|2,{37,117,0}},
/* 22228 */ {(12<<2)|2,{37,117,0}},
/* 22229 */ {(12<<2)|2,{37,117,0}},
/* 22230 */ {(12<<2)|2,{37,117,0}},
/* 22231 */ {(12<<2)|2,{37,117,0}},
/* 22232 */ {(12<<2)|2,{37,117,0}},
/* 22233 */ {(12<<2)|2,{37,117,0}},
/* 22234 */ {(12<<2)|2,{37,117,0}},
/* 22235 */ {(12<<2)|2,{37,117,0}},
/* 22236 */ {(12<<2)|2,{37,117,0}},
/* 22237 */ {(12<<2)|2,{37,117,0}},
/* 22238 */ {(12<<2)|2,{37,117,0}},
/* 22239 */ {(12<<2)|2,{37,117,0}},
/* 22240 */ {(13<<2)|2,{37,58,0}},
/* 22241 */ {(13<<2)|2,{37,58,0}},
/* 22242 */ {(13<<2)|2,{37,58,0}},
/* 22243 */ {(13<<2)|2,{37,58,0}},
/* 22244 */ {(13<<2)|2,{37,58,0}},
/* 22245 */ {(13<<2)|2,{37,58,0}},
/* 22246 */ {(13<<2)|2,{37,58,0}},
/* 22247 */ {(13<<2)|2,{37,58,0}},
/* 22248 */ {(13<<2)|2,{37,66,0}},
/* 22249 */ {(13<<2)|2,{37,66,0}},
/* 22250 */ {(13<<2)|2,{37,66,0}},
/* 22251 */ {(13<<2)|2,{37,66,0}},
/* 22252 */ {(13<<2)|2,{37,66,0}},
/* 22253 */ {(13<<2)|2,{37,66,0}},
/* 22254 */ {(13<<2)|2,{37,66,0}},
/* 22255 */ {(13<<2)|2,{37,66,0}},
/* 22256 */ {(13<<2)|2,{37,67,0}},
/* 22257 */ {(13<<2)|2,{37,67,0}},
/* 22258 */ {(13<<2)|2,{37,67,0}},
/* 22259 */ {(13<<2)|2,{37,67,0}},
/* 22260 */ {(13<<2)|2,{37,67,0}},
/* 22261 */ {(13<<2)|2,{37,67,0}},
/* 22262 */ {(13<<2)|2,{37,67,0}},
/* 22263 */ {(13<<2)|2,{37,67,0}},
/* 22264 */ {(13<<2)|2,{37,68,0}},
/* 22265 */ {(13<<2)|2,{37,68,0}},
/* 22266 */ {(13<<2)|2,{37,68,0}},
/* 22267 */ {(13<<2)|2,{37,68,0}},
/* 22268 */ {(13<<2)|2,{37,68,0}},
/* 22269 */ {(13<<2)|2,{37,68,0}},
/* 22270 */ {(13<<2)|2,{37,68,0}},
/* 22271 */ {(13<<2)|2,{37,68,0}},
/* 22272 */ {(13<<2)|2,{37,69,0}},
/* 22273 */ {(13<<2)|2,{37,69,0}},
/* 22274 */ {(13<<2)|2,{37,69,0}},
/* 22275 */ {(13<<2)|2,{37,69,0}},
/* 22276 */ {(13<<2)|2,{37,69,0}},
/* 22277 */ {(13<<2)|2,{37,69,0}},
/* 22278 */ {(13<<2)|2,{37,69,0}},
/* 22279 */ {(13<<2)|2,{37,69,0}},
/* 22280 */ {(13<<2)|2,{37,70,0}},
/* 22281 */ {(13<<2)|2,{37,70,0}},
/* 22282 */ {(13<<2)|2,{37,70,0}},
/* 22283 */ {(13<<2)|2,{37,70,0}},
/* 22284 */ {(13<<2)|2,{37,70,0}},
/* 22285 */ {(13<<2)|2,{37,70,0}},
/* 22286 */ {(13<<2)|2,{37,70,0}},
/* 22287 */ {(13<<2)|2,{37,70,0}},
/* 22288 */ {(13<<2)|2,{37,71,0}},
/* 22289 */ {(13<<2)|2,{37,71,0}},
/* 22290 */ {(13<<2)|2,{37,71,0}},
/* 22291 */ {(13<<2)|2,{37,71,0}},
/* 22292 */ {(13<<2)|2,{37,71,0}},
/* 22293 */ {(13<<2)|2,{37,71,0}},
/* 22294 */ {(13<<2)|2,{37,71,0}},
/* 22295 */ {(13<<2)|2,{37,71,0}},
/* 22296 */ {(13<<2)|2,{37,72,0}},
/* 22297 */ {(13<<2)|2,{37,72,0}},
/* 22298 */ {(13<<2)|2,{37,72,0}},
/* 22299 */ {(13<<2)|2,{37,72,0}},
/* 22300 */ {(13<<2)|2,{37,72,0}},
/* 22301 */ {(13<<2)|2,{37,72,0}},
/* 22302 */ {(13<<2)|2,{37,72,0}},
/* 22303 */ {(13<<2)|2,{37,72,0}},
/* 22304 */ {(13<<2)|2,{37,73,0}},
/* 22305 */ {(13<<2)|2,{37,73,0}},
/* 22306 */ {(13<<2)|2,{37,73,0}},
/* 22307 */ {(13<<2)|2,{37,73,0}},
/* 22308 */ {(13<<2)|2,{37,73,0}},
/* 22309 */ {(13<<2)|2,{37,73,0}},
/* 22310 */ {(13<<2)|2,{37,73,0}},
/* 22311 */ {(13<<2)|2,{37,73,0}},
/* 22312 */ {(13<<2)|2,{37,74,0}},
/* 22313 */ {(13<<2)|2,{37,74,0}},
/* 22314 */ {(13<<2)|2,{37,74,0}},
/* 22315 */ {(13<<2)|2,{37,74,0}},
/* 22316 */ {(13<<2)|2,{37,74,0}},
/* 22317 */ {(13<<2)|2,{37,74,0}},
/* 22318 */ {(13<<2)|2,{37,74,0}},
/* 22319 */ {(13<<2)|2,{37,74,0}},
/* 22320 */ {(13<<2)|2,{37,75,0}},
/* 22321 */ {(13<<2)|2,{37,75,0}},
/* 22322 */ {(13<<2)|2,{37,75,0}},
/* 22323 */ {(13<<2)|2,{37,75,0}},
/* 22324 */ {(13<<2)|2,{37,75,0}},
/* 22325 */ {(13<<2)|2,{37,75,0}},
/* 22326 */ {(13<<2)|2,{37,75,0}},
/* 22327 */ {(13<<2)|2,{37,75,0}},
/* 22328 */ {(13<<2)|2,{37,76,0}},
/* 22329 */ {(13<<2)|2,{37,76,0}},
/* 22330 */ {(13<<2)|2,{37,76,0}},
/* 22331 */ {(13<<2)|2,{37,76,0}},
/* 22332 */ {(13<<2)|2,{37,76,0}},
/* 22333 */ {(13<<2)|2,{37,76,0}},
/* 22334 */ {(13<<2)|2,{37,76,0}},
/* 22335 */ {(13<<2)|2,{37,76,0}},
/* 22336 */ {(13<<2)|2,{37,77,0}},
/* 22337 */ {(13<<2)|2,{37,77,0}},
/* 22338 */ {(13<<2)|2,{37,77,0}},
/* 22339 */ {(13<<2)|2,{37,77,0}},
/* 22340 */ {(13<<2)|2,{37,77,0}},
/* 22341 */ {(13<<2)|2,{37,77,0}},
/* 22342 */ {(13<<2)|2,{37,77,0}},
/* 22343 */ {(13<<2)|2,{37,77,0}},
/* 22344 */ {(13<<2)|2,{37,78,0}},
/* 22345 */ {(13<<2)|2,{37,78,0}},
/* 22346 */ {(13<<2)|2,{37,78,0}},
/* 22347 */ {(13<<2)|2,{37,78,0}},
/* 22348 */ {(13<<2)|2,{37,78,0}},
/* 22349 */ {(13<<2)|2,{37,78,0}},
/* 22350 */ {(13<<2)|2,{37,78,0}},
/* 22351 */ {(13<<2)|2,{37,78,0}},
/* 22352 */ {(13<<2)|2,{37,79,0}},
/* 22353 */ {(13<<2)|2,{37,79,0}},
/* 22354 */ {(13<<2)|2,{37,79,0}},
/* 22355 */ {(13<<2)|2,{37,79,0}},
/* 22356 */ {(13<<2)|2,{37,79,0}},
/* 22357 */ {(13<<2)|2,{37,79,0}},
/* 22358 */ {(13<<2)|2,{37,79,0}},
/* 22359 */ {(13<<2)|2,{37,79,0}},
/* 22360 */ {(13<<2)|2,{37,80,0}},
/* 22361 */ {(13<<2)|2,{37,80,0}},
/* 22362 */ {(13<<2)|2,{37,80,0}},
/* 22363 */ {(13<<2)|2,{37,80,0}},
/* 22364 */ {(13<<2)|2,{37,80,0}},
/* 22365 */ {(13<<2)|2,{37,80,0}},
/* 22366 */ {(13<<2)|2,{37,80,0}},
/* 22367 */ {(13<<2)|2,{37,80,0}},
/* 22368 */ {(13<<2)|2,{37,81,0}},
/* 22369 */ {(13<<2)|2,{37,81,0}},
/* 22370 */ {(13<<2)|2,{37,81,0}},
/* 22371 */ {(13<<2)|2,{37,81,0}},
/* 22372 */ {(13<<2)|2,{37,81,0}},
/* 22373 */ {(13<<2)|2,{37,81,0}},
/* 22374 */ {(13<<2)|2,{37,81,0}},
/* 22375 */ {(13<<2)|2,{37,81,0}},
/* 22376 */ {(13<<2)|2,{37,82,0}},
/* 22377 */ {(13<<2)|2,{37,82,0}},
/* 22378 */ {(13<<2)|2,{37,82,0}},
/* 22379 */ {(13<<2)|2,{37,82,0}},
/* 22380 */ {(13<<2)|2,{37,82,0}},
/* 22381 */ {(13<<2)|2,{37,82,0}},
/* 22382 */ {(13<<2)|2,{37,82,0}},
/* 22383 */ {(13<<2)|2,{37,82,0}},
/* 22384 */ {(13<<2)|2,{37,83,0}},
/* 22385 */ {(13<<2)|2,{37,83,0}},
/* 22386 */ {(13<<2)|2,{37,83,0}},
/* 22387 */ {(13<<2)|2,{37,83,0}},
/* 22388 */ {(13<<2)|2,{37,83,0}},
/* 22389 */ {(13<<2)|2,{37,83,0}},
/* 22390 */ {(13<<2)|2,{37,83,0}},
/* 22391 */ {(13<<2)|2,{37,83,0}},
/* 22392 */ {(13<<2)|2,{37,84,0}},
/* 22393 */ {(13<<2)|2,{37,84,0}},
/* 22394 */ {(13<<2)|2,{37,84,0}},
/* 22395 */ {(13<<2)|2,{37,84,0}},
/* 22396 */ {(13<<2)|2,{37,84,0}},
/* 22397 */ {(13<<2)|2,{37,84,0}},
/* 22398 */ {(13<<2)|2,{37,84,0}},
/* 22399 */ {(13<<2)|2,{37,84,0}},
/* 22400 */ {(13<<2)|2,{37,85,0}},
/* 22401 */ {(13<<2)|2,{37,85,0}},
/* 22402 */ {(13<<2)|2,{37,85,0}},
/* 22403 */ {(13<<2)|2,{37,85,0}},
/* 22404 */ {(13<<2)|2,{37,85,0}},
/* 22405 */ {(13<<2)|2,{37,85,0}},
/* 22406 */ {(13<<2)|2,{37,85,0}},
/* 22407 */ {(13<<2)|2,{37,85,0}},
/* 22408 */ {(13<<2)|2,{37,86,0}},
/* 22409 */ {(13<<2)|2,{37,86,0}},
/* 22410 */ {(13<<2)|2,{37,86,0}},
/* 22411 */ {(13<<2)|2,{37,86,0}},
/* 22412 */ {(13<<2)|2,{37,86,0}},
/* 22413 */ {(13<<2)|2,{37,86,0}},
/* 22414 */ {(13<<2)|2,{37,86,0}},
/* 22415 */ {(13<<2)|2,{37,86,0}},
/* 22416 */ {(13<<2)|2,{37,87,0}},
/* 22417 */ {(13<<2)|2,{37,87,0}},
/* 22418 */ {(13<<2)|2,{37,87,0}},
/* 22419 */ {(13<<2)|2,{37,87,0}},
/* 22420 */ {(13<<2)|2,{37,87,0}},
/* 22421 */ {(13<<2)|2,{37,87,0}},
/* 22422 */ {(13<<2)|2,{37,87,0}},
/* 22423 */ {(13<<2)|2,{37,87,0}},
/* 22424 */ {(13<<2)|2,{37,89,0}},
/* 22425 */ {(13<<2)|2,{37,89,0}},
/* 22426 */ {(13<<2)|2,{37,89,0}},
/* 22427 */ {(13<<2)|2,{37,89,0}},
/* 22428 */ {(13<<2)|2,{37,89,0}},
/* 22429 */ {(13<<2)|2,{37,89,0}},
/* 22430 */ {(13<<2)|2,{37,89,0}},
/* 22431 */ {(13<<2)|2,{37,89,0}},
/* 22432 */ {(13<<2)|2,{37,106,0}},
/* 22433 */ {(13<<2)|2,{37,106,0}},
/* 22434 */ {(13<<2)|2,{37,106,0}},
/* 22435 */ {(13<<2)|2,{37,106,0}},
/* 22436 */ {(13<<2)|2,{37,106,0}},
/* 22437 */ {(13<<2)|2,{37,106,0}},
/* 22438 */ {(13<<2)|2,{37,106,0}},
/* 22439 */ {(13<<2)|2,{37,106,0}},
/* 22440 */ {(13<<2)|2,{37,107,0}},
/* 22441 */ {(13<<2)|2,{37,107,0}},
/* 22442 */ {(13<<2)|2,{37,107,0}},
/* 22443 */ {(13<<2)|2,{37,107,0}},
/* 22444 */ {(13<<2)|2,{37,107,0}},
/* 22445 */ {(13<<2)|2,{37,107,0}},
/* 22446 */ {(13<<2)|2,{37,107,0}},
/* 22447 */ {(13<<2)|2,{37,107,0}},
/* 22448 */ {(13<<2)|2,{37,113,0}},
/* 22449 */ {(13<<2)|2,{37,113,0}},
/* 22450 */ {(13<<2)|2,{37,113,0}},
/* 22451 */ {(13<<2)|2,{37,113,0}},
/* 22452 */ {(13<<2)|2,{37,113,0}},
/* 22453 */ {(13<<2)|2,{37,113,0}},
/* 22454 */ {(13<<2)|2,{37,113,0}},
/* 22455 */ {(13<<2)|2,{37,113,0}},
/* 22456 */ {(13<<2)|2,{37,118,0}},
/* 22457 */ {(13<<2)|2,{37,118,0}},
/* 22458 */ {(13<<2)|2,{37,118,0}},
/* 22459 */ {(13<<2)|2,{37,118,0}},
/* 22460 */ {(13<<2)|2,{37,118,0}},
/* 22461 */ {(13<<2)|2,{37,118,0}},
/* 22462 */ {(13<<2)|2,{37,118,0}},
/* 22463 */ {(13<<2)|2,{37,118,0}},
/* 22464 */ {(13<<2)|2,{37,119,0}},
/* 22465 */ {(13<<2)|2,{37,119,0}},
/* 22466 */ {(13<<2)|2,{37,119,0}},
/* 22467 */ {(13<<2)|2,{37,119,0}},
/* 22468 */ {(13<<2)|2,{37,119,0}},
/* 22469 */ {(13<<2)|2,{37,119,0}},
/* 22470 */ {(13<<2)|2,{37,119,0}},
/* 22471 */ {(13<<2)|2,{37,119,0}},
/* 22472 */ {(13<<2)|2,{37,120,0}},
/* 22473 */ {(13<<2)|2,{37,120,0}},
/* 22474 */ {(13<<2)|2,{37,120,0}},
/* 22475 */ {(13<<2)|2,{37,120,0}},
/* 22476 */ {(13<<2)|2,{37,120,0}},
/* 22477 */ {(13<<2)|2,{37,120,0}},
/* 22478 */ {(13<<2)|2,{37,120,0}},
/* 22479 */ {(13<<2)|2,{37,120,0}},
/* 22480 */ {(13<<2)|2,{37,121,0}},
/* 22481 */ {(13<<2)|2,{37,121,0}},
/* 22482 */ {(13<<2)|2,{37,121,0}},
/* 22483 */ {(13<<2)|2,{37,121,0}},
/* 22484 */ {(13<<2)|2,{37,121,0}},
/* 22485 */ {(13<<2)|2,{37,121,0}},
/* 22486 */ {(13<<2)|2,{37,121,0}},
/* 22487 */ {(13<<2)|2,{37,121,0}},
/* 22488 */ {(13<<2)|2,{37,122,0}},
/* 22489 */ {(13<<2)|2,{37,122,0}},
/* 22490 */ {(13<<2)|2,{37,122,0}},
/* 22491 */ {(13<<2)|2,{37,122,0}},
/* 22492 */ {(13<<2)|2,{37,122,0}},
/* 22493 */ {(13<<2)|2,{37,122,0}},
/* 22494 */ {(13<<2)|2,{37,122,0}},
/* 22495 */ {(13<<2)|2,{37,122,0}},
/* 22496 */ {(14<<2)|2,{37,38,0}},
/* 22497 */ {(14<<2)|2,{37,38,0}},
/* 22498 */ {(14<<2)|2,{37,38,0}},
/* 22499 */ {(14<<2)|2,{37,38,0}},
/* 22500 */ {(14<<2)|2,{37,42,0}},
/* 22501 */ {(14<<2)|2,{37,42,0}},
/* 22502 */ {(14<<2)|2,{37,42,0}},
/* 22503 */ {(14<<2)|2,{37,42,0}},
/* 22504 */ {(14<<2)|2,{37,44,0}},
/* 22505 */ {(14<<2)|2,{37,44,0}},
/* 22506 */ {(14<<2)|2,{37,44,0}},
/* 22507 */ {(14<<2)|2,{37,44,0}},
/* 22508 */ {(14<<2)|2,{37,59,0}},
/* 22509 */ {(14<<2)|2,{37,59,0}},
/* 22510 */ {(14<<2)|2,{37,59,0}},
/* 22511 */ {(14<<2)|2,{37,59,0}},
/* 22512 */ {(14<<2)|2,{37,88,0}},
/* 22513 */ {(14<<2)|2,{37,88,0}},
/* 22514 */ {(14<<2)|2,{37,88,0}},
/* 22515 */ {(14<<2)|2,{37,88,0}},
/* 22516 */ {(14<<2)|2,{37,90,0}},
/* 22517 */ {(14<<2)|2,{37,90,0}},
/* 22518 */ {(14<<2)|2,{37,90,0}},
/* 22519 */ {(14<<2)|2,{37,90,0}},
/* 22520 */ {(16<<2)|2,{37,33,0}},
/* 22521 */ {(16<<2)|2,{37,34,0}},
/* 22522 */ {(16<<2)|2,{37,40,0}},
/* 22523 */ {(16<<2)|2,{37,41,0}},
/* 22524 */ {(16<<2)|2,{37,63,0}},
/* 22525 */ {(6<<2)|1,{37,0,0}},
/* 22526 */ {(6<<2)|1,{37,0,0}},
/* 22527 */ {(6<<2)|1,{37,0,0}},
/* 22528 */ {(16<<2)|3,{45,48,48}},
/* 22529 */ {(16<<2)|3,{45,48,49}},
/* 22530 */ {(16<<2)|3,{45,48,50}},
/* 22531 */ {(16<<2)|3,{45,48,97}},
/* 22532 */ {(16<<2)|3,{45,48,99}},
/* 22533 */ {(16<<2)|3,{45,48,101}},
/* 22534 */ {(16<<2)|3,{45,48,105}},
/* 22535 */ {(16<<2)|3,{45,48,111}},
/* 22536 */ {(16<<2)|3,{45,48,115}},
/* 22537 */ {(16<<2)|3,{45,48,116}},
/* 22538 */ {(11<<2)|2,{45,48,0}},
/* 22539 */ {(11<<2)|2,{45,48,0}},
/* 22540 */ {(11<<2)|2,{45,48,0}},
/* 22541 */ {(11<<2)|2,{45,48,0}},
/* 22542 */ {(11<<2)|2,{45,48,0}},
/* 22543 */ {(11<<2)|2,{45,48,0}},
/* 22544 */ {(11<<2)|2,{45,48,0}},
/* 22545 */ {(11<<2)|2,{45,48,0}},
/* 22546 */ {(11<<2)|2,{45,48,0}},
/* 22547 */ {(11<<2)|2,{45,48,0}},
/* 22548 */ {(11<<2)|2,{45,48,0}},
/* 22549 */ {(11<<2)|2,{45,48,0}},
/* 22550 */ {(11<<2)|2,{45,48,0}},
/* 22551 */ {(11<<2)|2,{45,48,0}},
/* 22552 */ {(11<<2)|2,{45,48,0}},
/* 22553 */ {(11<<2)|2,{45,48,0}},
/* 22554 */ {(11<<2)|2,{45,48,0}},
/* 22555 */ {(11<<2)|2,{45,48,0}},
/* 22556 */ {(11<<2)|2,{45,48,0}},
/* 22557 */ {(11<<2)|2,{45,48,0}},
/* 22558 */ {(11<<2)|2,{45,48,0}},
/* 22559 */ {(11<<2)|2,{45,48,0}},
/* 22560 */ {(16<<2)|3,{45,49,48}},
/* 22561 */ {(16<<2)|3,{45,49,49}},
/* 22562 */ {(16<<2)|3,{45,49,50}},
/* 22563 */ {(16<<2)|3,{45,49,97}},
/* 22564 */ {(16<<2)|3,{45,49,99}},
/* 22565 */ {(16<<2)|3,{45,49,101}},
/* 22566 */ {(16<<2)|3,{45,49,105}},
/* 22567 */ {(16<<2)|3,{45,49,111}},
/* 22568 */ {(16<<2)|3,{45,49,115}},
/* 22569 */ {(16<<2)|3,{45,49,116}},
/* 22570 */ {(11<<2)|2,{45,49,0}},
/* 22571 */ {(11<<2)|2,{45,49,0}},
/* 22572 */ {(11<<2)|2,{45,49,0}},
/* 22573 */ {(11<<2)|2,{45,49,0}},
/* 22574 */ {(11<<2)|2,{45,49,0}},
/* 22575 */ {(11<<2)|2,{45,49,0}},
/* 22576 */ {(11<<2)|2,{45,49,0}},
/* 22577 */ {(11<<2)|2,{45,49,0}},
/* 22578 */ {(11<<2)|2,{45,49,0}},
/* 22579 */ {(11<<2)|2,{45,49,0}},
/* 22580 */ {(11<<2)|2,{45,49,0}},
/* 22581 */ {(11<<2)|2,{45,49,0}},
/* 22582 */ {(11<<2)|2,{45,49,0}},
/* 22583 */ {(11<<2)|2,{45,49,0}},
/* 22584 */ {(11<<2)|2,{45,49,0}},
/* 22585 */ {(11<<2)|2,{45,49,0}},
/* 22586 */ {(11<<2)|2,{45,49,0}},
/* 22587 */ {(11<<2)|2,{45,49,0}},
/* 22588 */ {(11<<2)|2,{45,49,0}},
/* 22589 */ {(11<<2)|2,{45,49,0}},
/* 22590 */ {(11<<2)|2,{45,49,0}},
/* 22591 */ {(11<<2)|2,{45,49,0}},
/* 22592 */ {(16<<2)|3,{45,50,48}},
/* 22593 */ {(16<<2)|3,{45,50,49}},
/* 22594 */ {(16<<2)|3,{45,50,50}},
/* 22595 */ {(16<<2)|3,{45,50,97}},
/* 22596 */ {(16<<2)|3,{45,50,99}},
/* 22597 */ {(16<<2)|3,{45,50,101}},
/* 22598 */ {(16<<2)|3,{45,50,105}},
/* 22599 */ {(16<<2)|3,{45,50,111}},
/* 22600 */ {(16<<2)|3,{45,50,115}},
/* 22601 */ {(16<<2)|3,{45,50,116}},
/* 22602 */ {(11<<2)|2,{45,50,0}},
/* 22603 */ {(11<<2)|2,{45,50,0}},
/* 22604 */ {(11<<2)|2,{45,50,0}},
/* 22605 */ {(11<<2)|2,{45,50,0}},
/* 22606 */ {(11<<2)|2,{45,50,0}},
/* 22607 */ {(11<<2)|2,{45,50,0}},
/* 22608 */ {(11<<2)|2,{45,50,0}},
/* 22609 */ {(11<<2)|2,{45,50,0}},
/* 22610 */ {(11<<2)|2,{45,50,0}},
/* 22611 */ {(11<<2)|2,{45,50,0}},
/* 22612 */ {(11<<2)|2,{45,50,0}},
/* 22613 */ {(11<<2)|2,{45,50,0}},
/* 22614 */ {(11<<2)|2,{45,50,0}},
/* 22615 */ {(11<<2)|2,{45,50,0}},
/* 22616 */ {(11<<2)|2,{45,50,0}},
/* 22617 */ {(11<<2)|2,{45,50,0}},
/* 22618 */ {(11<<2)|2,{45,50,0}},
/* 22619 */ {(11<<2)|2,{45,50,0}},
/* 22620 */ {(11<<2)|2,{45,50,0}},
/* 22621 */ {(11<<2)|2,{45,50,0}},
/* 22622 */ {(11<<2)|2,{45,50,0}},
/* 22623 */ {(11<<2)|2,{45,50,0}},
/* 22624 */ {(16<<2)|3,{45,97,48}},
/* 22625 */ {(16<<2)|3,{45,97,49}},
/* 22626 */ {(16<<2)|3,{45,97,50}},
/* 22627 */ {(16<<2)|3,{45,97,97}},
/* 22628 */ {(16<<2)|3,{45,97,99}},
/* 22629 */ {(16<<2)|3,{45,97,101}},
/* 22630 */ {(16<<2)|3,{45,97,105}},
/* 22631 */ {(16<<2)|3,{45,97,111}},
/* 22632 */ {(16<<2)|3,{45,97,115}},
/* 22633 */ {(16<<2)|3,{45,97,116}},
/* 22634 */ {(11<<2)|2,{45,97,0}},
/* 22635 */ {(11<<2)|2,{45,97,0}},
/* 22636 */ {(11<<2)|2,{45,97,0}},
/* 22637 */ {(11<<2)|2,{45,97,0}},
/* 22638 */ {(11<<2)|2,{45,97,0}},
/* 22639 */ {(11<<2)|2,{45,97,0}},
/* 22640 */ {(11<<2)|2,{45,97,0}},
/* 22641 */ {(11<<2)|2,{45,97,0}},
/* 22642 */ {(11<<2)|2,{45,97,0}},
/* 22643 */ {(11<<2)|2,{45,97,0}},
/* 22644 */ {(11<<2)|2,{45,97,0}},
/* 22645 */ {(11<<2)|2,{45,97,0}},
/* 22646 */ {(11<<2)|2,{45,97,0}},
/* 22647 */ {(11<<2)|2,{45,97,0}},
/* 22648 */ {(11<<2)|2,{45,97,0}},
/* 22649 */ {(11<<2)|2,{45,97,0}},
/* 22650 */ {(11<<2)|2,{45,97,0}},
/* 22651 */ {(11<<2)|2,{45,97,0}},
/* 22652 */ {(11<<2)|2,{45,97,0}},
/* 22653 */ {(11<<2)|2,{45,97,0}},
/* 22654 */ {(11<<2)|2,{45,97,0}},
/* 22655 */ {(11<<2)|2,{45,97,0}},
/* 22656 */ {(16<<2)|3,{45,99,48}},
/* 22657 */ {(16<<2)|3,{45,99,49}},
/* 22658 */ {(16<<2)|3,{45,99,50}},
/* 22659 */ {(16<<2)|3,{45,99,97}},
/* 22660 */ {(16<<2)|3,{45,99,99}},
/* 22661 */ {(16<<2)|3,{45,99,101}},
/* 22662 */ {(16<<2)|3,{45,99,105}},
/* 22663 */ {(16<<2)|3,{45,99,111}},
/* 22664 */ {(16<<2)|3,{45,99,115}},
/* 22665 */ {(16<<2)|3,{45,99,116}},
/* 22666 */ {(11<<2)|2,{45,99,0}},
/* 22667 */ {(11<<2)|2,{45,99,0}},
/* 22668 */ {(11<<2)|2,{45,99,0}},
/* 22669 */ {(11<<2)|2,{45,99,0}},
/* 22670 */ {(11<<2)|2,{45,99,0}},
/* 22671 */ {(11<<2)|2,{45,99,0}},
/* 22672 */ {(11<<2)|2,{45,99,0}},
/* 22673 */ {(11<<2)|2,{45,99,0}},
/* 22674 */ {(11<<2)|2,{45,99,0}},
/* 22675 */ {(11<<2)|2,{45,99,0}},
/* 22676 */ {(11<<2)|2,{45,99,0}},
/* 22677 */ {(11<<2)|2,{45,99,0}},
/* 22678 */ {(11<<2)|2,{45,99,0}},
/* 22679 */ {(11<<2)|2,{45,99,0}},
/* 22680 */ {(11<<2)|2,{45,99,0}},
/* 22681 */ {(11<<2)|2,{45,99,0}},
/* 22682 */ {(11<<2)|2,{45,99,0}},
/* 22683 */ {(11<<2)|2,{45,99,0}},
/* 22684 */ {(11<<2)|2,{45,99,0}},
/* 22685 */ {(11<<2)|2,{45,99,0}},
/* 22686 */ {(11<<2)|2,{45,99,0}},
/* 22687 */ {(11<<2)|2,{45,99,0}},
/* 22688 */ {(16<<2)|3,{45,101,48}},
/* 22689 */ {(16<<2)|3,{45,101,49}},
/* 22690 */ {(16<<2)|3,{45,101,50}},
/* 22691 */ {(16<<2)|3,{45,101,97}},
/* 22692 */ {(16<<2)|3,{45,101,99}},
/* 22693 */ {(16<<2)|3,{45,101,101}},
/* 22694 */ {(16<<2)|3,{45,101,105}},
/* 22695 */ {(16<<2)|3,{45,101,111}},
/* 22696 */ {(16<<2)|3,{45,101,115}},
/* 22697 */ {(16<<2)|3,{45,101,116}},
/* 22698 */ {(11<<2)|2,{45,101,0}},
/* 22699 */ {(11<<2)|2,{45,101,0}},
/* 22700 */ {(11<<2)|2,{45,101,0}},
/* 22701 */ {(11<<2)|2,{45,101,0}},
/* 22702 */ {(11<<2)|2,{45,101,0}},
/* 22703 */ {(11<<2)|2,{45,101,0}},
/* 22704 */ {(11<<2)|2,{45,101,0}},
/* 22705 */ {(11<<2)|2,{45,101,0}},
/* 22706 */ {(11<<2)|2,{45,101,0}},
/* 22707 */ {(11<<2)|2,{45,101,0}},
/* 22708 */ {(11<<2)|2,{45,101,0}},
/* 22709 */ {(11<<2)|2,{45,101,0}},
/* 22710 */ {(11<<2)|2,{45,101,0}},
/* 22711 */ {(11<<2)|2,{45,101,0}},
/* 22712 */ {(11<<2)|2,{45,101,0}},
/* 22713 */ {(11<<2)|2,{45,101,0}},
/* 22714 */ {(11<<2)|2,{45,101,0}},
/* 22715 */ {(11<<2)|2,{45,101,0}},
/* 22716 */ {(11<<2)|2,{45,101,0}},
/* 22717 */ {(11<<2)|2,{45,101,0}},
/* 22718 */ {(11<<2)|2,{45,101,0}},
/* 22719 */ {(11<<2)|2,{45,101,0}},
/* 22720 */ {(16<<2)|3,{45,105,48}},
/* 22721 */ {(16<<2)|3,{45,105,49}},
/* 22722 */ {(16<<2)|3,{45,105,50}},
/* 22723 */ {(16<<2)|3,{45,105,97}},
/* 22724 */ {(16<<2)|3,{45,105,99}},
/* 22725 */ {(16<<2)|3,{45,105,101}},
/* 22726 */ {(16<<2)|3,{45,105,105}},
/* 22727 */ {(16<<2)|3,{45,105,111}},
/* 22728 */ {(16<<2)|3,{45,105,115}},
/* 22729 */ {(16<<2)|3,{45,105,116}},
/* 22730 */ {(11<<2)|2,{45,105,0}},
/* 22731 */ {(11<<2)|2,{45,105,0}},
/* 22732 */ {(11<<2)|2,{45,105,0}},
/* 22733 */ {(11<<2)|2,{45,105,0}},
/* 22734 */ {(11<<2)|2,{45,105,0}},
/* 22735 */ {(11<<2)|2,{45,105,0}},
/* 22736 */ {(11<<2)|2,{45,105,0}},
/* 22737 */ {(11<<2)|2,{45,105,0}},
/* 22738 */ {(11<<2)|2,{45,105,0}},
/* 22739 */ {(11<<2)|2,{45,105,0}},
/* 22740 */ {(11<<2)|2,{45,105,0}},
/* 22741 */ {(11<<2)|2,{45,105,0}},
/* 22742 */ {(11<<2)|2,{45,105,0}},
/* 22743 */ {(11<<2)|2,{45,105,0}},
/* 22744 */ {(11<<2)|2,{45,105,0}},
/* 22745 */ {(11<<2)|2,{45,105,0}},
/* 22746 */ {(11<<2)|2,{45,105,0}},
/* 22747 */ {(11<<2)|2,{45,105,0}},
/* 22748 */ {(11<<2)|2,{45,105,0}},
/* 22749 */ {(11<<2)|2,{45,105,0}},
/* 22750 */ {(11<<2)|2,{45,105,0}},
/* 22751 */ {(11<<2)|2,{45,105,0}},
/* 22752 */ {(16<<2)|3,{45,111,48}},
/* 22753 */ {(16<<2)|3,{45,111,49}},
/* 22754 */ {(16<<2)|3,{45,111,50}},
/* 22755 */ {(16<<2)|3,{45,111,97}},
/* 22756 */ {(16<<2)|3,{45,111,99}},
/* 22757 */ {(16<<2)|3,{45,111,101}},
/* 22758 */ {(16<<2)|3,{45,111,105}},
/* 22759 */ {(16<<2)|3,{45,111,111}},
/* 22760 */ {(16<<2)|3,{45,111,115}},
/* 22761 */ {(16<<2)|3,{45,111,116}},
/* 22762 */ {(11<<2)|2,{45,111,0}},
/* 22763 */ {(11<<2)|2,{45,111,0}},
/* 22764 */ {(11<<2)|2,{45,111,0}},
/* 22765 */ {(11<<2)|2,{45,111,0}},
/* 22766 */ {(11<<2)|2,{45,111,0}},
/* 22767 */ {(11<<2)|2,{45,111,0}},
/* 22768 */ {(11<<2)|2,{45,111,0}},
/* 22769 */ {(11<<2)|2,{45,111,0}},
/* 22770 */ {(11<<2)|2,{45,111,0}},
/* 22771 */ {(11<<2)|2,{45,111,0}},
/* 22772 */ {(11<<2)|2,{45,111,0}},
/* 22773 */ {(11<<2)|2,{45,111,0}},
/* 22774 */ {(11<<2)|2,{45,111,0}},
/* 22775 */ {(11<<2)|2,{45,111,0}},
/* 22776 */ {(11<<2)|2,{45,111,0}},
/* 22777 */ {(11<<2)|2,{45,111,0}},
/* 22778 */ {(11<<2)|2,{45,111,0}},
/* 22779 */ {(11<<2)|2,{45,111,0}},
/* 22780 */ {(11<<2)|2,{45,111,0}},
/* 22781 */ {(11<<2)|2,{45,111,0}},
/* 22782 */ {(11<<2)|2,{45,111,0}},
/* 22783 */ {(11<<2)|2,{45,111,0}},
/* 22784 */ {(16<<2)|3,{45,115,48}},
/* 22785 */ {(16<<2)|3,{45,115,49}},
/* 22786 */ {(16<<2)|3,{45,115,50}},
/* 22787 */ {(16<<2)|3,{45,115,97}},
/* 22788 */ {(16<<2)|3,{45,115,99}},
/* 22789 */ {(16<<2)|3,{45,115,101}},
/* 22790 */ {(16<<2)|3,{45,115,105}},
/* 22791 */ {(16<<2)|3,{45,115,111}},
/* 22792 */ {(16<<2)|3,{45,115,115}},
/* 22793 */ {(16<<2)|3,{45,115,116}},
/* 22794 */ {(11<<2)|2,{45,115,0}},
/* 22795 */ {(11<<2)|2,{45,115,0}},
/* 22796 */ {(11<<2)|2,{45,115,0}},
/* 22797 */ {(11<<2)|2,{45,115,0}},
/* 22798 */ {(11<<2)|2,{45,115,0}},
/* 22799 */ {(11<<2)|2,{45,115,0}},
/* 22800 */ {(11<<2)|2,{45,115,0}},
/* 22801 */ {(11<<2)|2,{45,115,0}},
/* 22802 */ {(11<<2)|2,{45,115,0}},
/* 22803 */ {(11<<2)|2,{45,115,0}},
/* 22804 */ {(11<<2)|2,{45,115,0}},
/* 22805 */ {(11<<2)|2,{45,115,0}},
/* 22806 */ {(11<<2)|2,{45,115,0}},
/* 22807 */ {(11<<2)|2,{45,115,0}},
/* 22808 */ {(11<<2)|2,{45,115,0}},
/* 22809 */ {(11<<2)|2,{45,115,0}},
/* 22810 */ {(11<<2)|2,{45,115,0}},
/* 22811 */ {(11<<2)|2,{45,115,0}},
/* 22812 */ {(11<<2)|2,{45,115,0}},
/* 22813 */ {(11<<2)|2,{45,115,0}},
/* 22814 */ {(11<<2)|2,{45,115,0}},
/* 22815 */ {(11<<2)|2,{45,115,0}},
/* 22816 */ {(16<<2)|3,{45,116,48}},
/* 22817 */ {(16<<2)|3,{45,116,49}},
/* 22818 */ {(16<<2)|3,{45,116,50}},
/* 22819 */ {(16<<2)|3,{45,116,97}},
/* 22820 */ {(16<<2)|3,{45,116,99}},
/* 22821 */ {(16<<2)|3,{45,116,101}},
/* 22822 */ {(16<<2)|3,{45,116,105}},
/* 22823 */ {(16<<2)|3,{45,116,111}},
/* 22824 */ {(16<<2)|3,{45,116,115}},
/* 22825 */ {(16<<2)|3,{45,116,116}},
/* 22826 */ {(11<<2)|2,{45,116,0}},
/* 22827 */ {(11<<2)|2,{45,116,0}},
/* 22828 */ {(11<<2)|2,{45,116,0}},
/* 22829 */ {(11<<2)|2,{45,116,0}},
/* 22830 */ {(11<<2)|2,{45,116,0}},
/* 22831 */ {(11<<2)|2,{45,116,0}},
/* 22832 */ {(11<<2)|2,{45,116,0}},
/* 22833 */ {(11<<2)|2,{45,116,0}},
/* 22834 */ {(11<<2)|2,{45,116,0}},
/* 22835 */ {(11<<2)|2,{45,116,0}},
/* 22836 */ {(11<<2)|2,{45,116,0}},
/* 22837 */ {(11<<2)|2,{45,116,0}},
/* 22838 */ {(11<<2)|2,{45,116,0}},
/* 22839 */ {(11<<2)|2,{45,116,0}},
/* 22840 */ {(11<<2)|2,{45,116,0}},
/* 22841 */ {(11<<2)|2,{45,116,0}},
/* 22842 */ {(11<<2)|2,{45,116,0}},
/* 22843 */ {(11<<2)|2,{45,116,0}},
/* 22844 */ {(11<<2)|2,{45,116,0}},
/* 22845 */ {(11<<2)|2,{45,116,0}},
/* 22846 */ {(11<<2)|2,{45,116,0}},
/* 22847 */ {(11<<2)|2,{45,116,0}},
/* 22848 */ {(12<<2)|2,{45,32,0}},
/* 22849 */ {(12<<2)|2,{45,32,0}},
/* 22850 */ {(12<<2)|2,{45,32,0}},
/* 22851 */ {(12<<2)|2,{45,32,0}},
/* 22852 */ {(12<<2)|2,{45,32,0}},
/* 22853 */ {(12<<2)|2,{45,32,0}},
/* 22854 */ {(12<<2)|2,{45,32,0}},
/* 22855 */ {(12<<2)|2,{45,32,0}},
/* 22856 */ {(12<<2)|2,{45,32,0}},
/* 22857 */ {(12<<2)|2,{45,32,0}},
/* 22858 */ {(12<<2)|2,{45,32,0}},
/* 22859 */ {(12<<2)|2,{45,32,0}},
/* 22860 */ {(12<<2)|2,{45,32,0}},
/* 22861 */ {(12<<2)|2,{45,32,0}},
/* 22862 */ {(12<<2)|2,{45,32,0}},
/* 22863 */ {(12<<2)|2,{45,32,0}},
/* 22864 */ {(12<<2)|2,{45,37,0}},
/* 22865 */ {(12<<2)|2,{45,37,0}},
/* 22866 */ {(12<<2)|2,{45,37,0}},
/* 22867 */ {(12<<2)|2,{45,37,0}},
/* 22868 */ {(12<<2)|2,{45,37,0}},
/* 22869 */ {(12<<2)|2,{45,37,0}},
/* 22870 */ {(12<<2)|2,{45,37,0}},
/* 22871 */ {(12<<2)|2,{45,37,0}},
/* 22872 */ {(12<<2)|2,{45,37,0}},
/* 22873 */ {(12<<2)|2,{45,37,0}},
/* 22874 */ {(12<<2)|2,{45,37,0}},
/* 22875 */ {(12<<2)|2,{45,37,0}},
/* 22876 */ {(12<<2)|2,{45,37,0}},
/* 22877 */ {(12<<2)|2,{45,37,0}},
/* 22878 */ {(12<<2)|2,{45,37,0}},
/* 22879 */ {(12<<2)|2,{45,37,0}},
/* 22880 */ {(12<<2)|2,{45,45,0}},
/* 22881 */ {(12<<2)|2,{45,45,0}},
/* 22882 */ {(12<<2)|2,{45,45,0}},
/* 22883 */ {(12<<2)|2,{45,45,0}},
/* 22884 */ {(12<<2)|2,{45,45,0}},
/* 22885 */ {(12<<2)|2,{45,45,0}},
/* 22886 */ {(12<<2)|2,{45,45,0}},
/* 22887 */ {(12<<2)|2,{45,45,0}},
/* 22888 */ {(12<<2)|2,{45,45,0}},
/* 22889 */ {(12<<2)|2,{45,45,0}},
/* 22890 */ {(12<<2)|2,{45,45,0}},
/* 22891 */ {(12<<2)|2,{45,45,0}},
/* 22892 */ {(12<<2)|2,{45,45,0}},
/* 22893 */ {(12<<2)|2,{45,45,0}},
/* 22894 */ {(12<<2)|2,{45,45,0}},
/* 22895 */ {(12<<2)|2,{45,45,0}},
/* 22896 */ {(12<<2)|2,{45,46,0}},
/* 22897 */ {(12<<2)|2,{45,46,0}},
/* 22898 */ {(12<<2)|2,{45,46,0}},
/* 22899 */ {(12<<2)|2,{45,46,0}},
/* 22900 */ {(12<<2)|2,{45,46,0}},
/* 22901 */ {(12<<2)|2,{45,46,0}},
/* 22902 */ {(12<<2)|2,{45,46,0}},
/* 22903 */ {(12<<2)|2,{45,46,0}},
/* 22904 */ {(12<<2)|2,{45,46,0}},
/* 22905 */ {(12<<2)|2,{45,46,0}},
/* 22906 */ {(12<<2)|2,{45,46,0}},
/* 22907 */ {(12<<2)|2,{45,46,0}},
/* 22908 */ {(12<<2)|2,{45,46,0}},
/* 22909 */ {(12<<2)|2,{45,46,0}},
/* 22910 */ {(12<<2)|2,{45,46,0}},
/* 22911 */ {(12<<2)|2,{45,46,0}},
/* 22912 */ {(12<<2)|2,{45,47,0}},
/* 22913 */ {(12<<2)|2,{45,47,0}},
/* 22914 */ {(12<<2)|2,{45,47,0}},
/* 22915 */ {(12<<2)|2,{45,47,0}},
/* 22916 */ {(12<<2)|2,{45,47,0}},
/* 22917 */ {(12<<2)|2,{45,47,0}},
/* 22918 */ {(12<<2)|2,{45,47,0}},
/* 22919 */ {(12<<2)|2,{45,47,0}},
/* 22920 */ {(12<<2)|2,{45,47,0}},
/* 22921 */ {(12<<2)|2,{45,47,0}},
/* 22922 */ {(12<<2)|2,{45,47,0}},
/* 22923 */ {(12<<2)|2,{45,47,0}},
/* 22924 */ {(12<<2)|2,{45,47,0}},
/* 22925 */ {(12<<2)|2,{45,47,0}},
/* 22926 */ {(12<<2)|2,{45,47,0}},
/* 22927 */ {(12<<2)|2,{45,47,0}},
/* 22928 */ {(12<<2)|2,{45,51,0}},
/* 22929 */ {(12<<2)|2,{45,51,0}},
/* 22930 */ {(12<<2)|2,{45,51,0}},
/* 22931 */ {(12<<2)|2,{45,51,0}},
/* 22932 */ {(12<<2)|2,{45,51,0}},
/* 22933 */ {(12<<2)|2,{45,51,0}},
/* 22934 */ {(12<<2)|2,{45,51,0}},
/* 22935 */ {(12<<2)|2,{45,51,0}},
/* 22936 */ {(12<<2)|2,{45,51,0}},
/* 22937 */ {(12<<2)|2,{45,51,0}},
/* 22938 */ {(12<<2)|2,{45,51,0}},
/* 22939 */ {(12<<2)|2,{45,51,0}},
/* 22940 */ {(12<<2)|2,{45,51,0}},
/* 22941 */ {(12<<2)|2,{45,51,0}},
/* 22942 */ {(12<<2)|2,{45,51,0}},
/* 22943 */ {(12<<2)|2,{45,51,0}},
/* 22944 */ {(12<<2)|2,{45,52,0}},
/* 22945 */ {(12<<2)|2,{45,52,0}},
/* 22946 */ {(12<<2)|2,{45,52,0}},
/* 22947 */ {(12<<2)|2,{45,52,0}},
/* 22948 */ {(12<<2)|2,{45,52,0}},
/* 22949 */ {(12<<2)|2,{45,52,0}},
/* 22950 */ {(12<<2)|2,{45,52,0}},
/* 22951 */ {(12<<2)|2,{45,52,0}},
/* 22952 */ {(12<<2)|2,{45,52,0}},
/* 22953 */ {(12<<2)|2,{45,52,0}},
/* 22954 */ {(12<<2)|2,{45,52,0}},
/* 22955 */ {(12<<2)|2,{45,52,0}},
/* 22956 */ {(12<<2)|2,{45,52,0}},
/* 22957 */ {(12<<2)|2,{45,52,0}},
/* 22958 */ {(12<<2)|2,{45,52,0}},
/* 22959 */ {(12<<2)|2,{45,52,0}},
/* 22960 */ {(12<<2)|2,{45,53,0}},
/* 22961 */ {(12<<2)|2,{45,53,0}},
/* 22962 */ {(12<<2)|2,{45,53,0}},
/* 22963 */ {(12<<2)|2,{45,53,0}},
/* 22964 */ {(12<<2)|2,{45,53,0}},
/* 22965 */ {(12<<2)|2,{45,53,0}},
/* 22966 */ {(12<<2)|2,{45,53,0}},
/* 22967 */ {(12<<2)|2,{45,53,0}},
/* 22968 */ {(12<<2)|2,{45,53,0}},
/* 22969 */ {(12<<2)|2,{45,53,0}},
/* 22970 */ {(12<<2)|2,{45,53,0}},
/* 22971 */ {(12<<2)|2,{45,53,0}},
/* 22972 */ {(12<<2)|2,{45,53,0}},
/* 22973 */ {(12<<2)|2,{45,53,0}},
/* 22974 */ {(12<<2)|2,{45,53,0}},
/* 22975 */ {(12<<2)|2,{45,53,0}},
/* 22976 */ {(12<<2)|2,{45,54,0}},
/* 22977 */ {(12<<2)|2,{45,54,0}},
/* 22978 */ {(12<<2)|2,{45,54,0}},
/* 22979 */ {(12<<2)|2,{45,54,0}},
/* 22980 */ {(12<<2)|2,{45,54,0}},
/* 22981 */ {(12<<2)|2,{45,54,0}},
/* 22982 */ {(12<<2)|2,{45,54,0}},
/* 22983 */ {(12<<2)|2,{45,54,0}},
/* 22984 */ {(12<<2)|2,{45,54,0}},
/* 22985 */ {(12<<2)|2,{45,54,0}},
/* 22986 */ {(12<<2)|2,{45,54,0}},
/* 22987 */ {(12<<2)|2,{45,54,0}},
/* 22988 */ {(12<<2)|2,{45,54,0}},
/* 22989 */ {(12<<2)|2,{45,54,0}},
/* 22990 */ {(12<<2)|2,{45,54,0}},
/* 22991 */ {(12<<2)|2,{45,54,0}},
/* 22992 */ {(12<<2)|2,{45,55,0}},
/* 22993 */ {(12<<2)|2,{45,55,0}},
/* 22994 */ {(12<<2)|2,{45,55,0}},
/* 22995 */ {(12<<2)|2,{45,55,0}},
/* 22996 */ {(12<<2)|2,{45,55,0}},
/* 22997 */ {(12<<2)|2,{45,55,0}},
/* 22998 */ {(12<<2)|2,{45,55,0}},
/* 22999 */ {(12<<2)|2,{45,55,0}},
/* 23000 */ {(12<<2)|2,{45,55,0}},
/* 23001 */ {(12<<2)|2,{45,55,0}},
/* 23002 */ {(12<<2)|2,{45,55,0}},
/* 23003 */ {(12<<2)|2,{45,55,0}},
/* 23004 */ {(12<<2)|2,{45,55,0}},
/* 23005 */ {(12<<2)|2,{45,55,0}},
/* 23006 */ {(12<<2)|2,{45,55,0}},
/* 23007 */ {(12<<2)|2,{45,55,0}},
/* 23008 */ {(12<<2)|2,{45,56,0}},
/* 23009 */ {(12<<2)|2,{45,56,0}},
/* 23010 */ {(12<<2)|2,{45,56,0}},
/* 23011 */ {(12<<2)|2,{45,56,0}},
/* 23012 */ {(12<<2)|2,{45,56,0}},
/* 23013 */ {(12<<2)|2,{45,56,0}},
/* 23014 */ {(12<<2)|2,{45,56,0}},
/* 23015 */ {(12<<2)|2,{45,56,0}},
/* 23016 */ {(12<<2)|2,{45,56,0}},
/* 23017 */ {(12<<2)|2,{45,56,0}},
/* 23018 */ {(12<<2)|2,{45,56,0}},
/* 23019 */ {(12<<2)|2,{45,56,0}},
/* 23020 */ {(12<<2)|2,{45,56,0}},
/* 23021 */ {(12<<2)|2,{45,56,0}},
/* 23022 */ {(12<<2)|2,{45,56,0}},
/* 23023 */ {(12<<2)|2,{45,56,0}},
/* 23024 */ {(12<<2)|2,{45,57,0}},
/* 23025 */ {(12<<2)|2,{45,57,0}},
/* 23026 */ {(12<<2)|2,{45,57,0}},
/* 23027 */ {(12<<2)|2,{45,57,0}},
/* 23028 */ {(12<<2)|2,{45,57,0}},
/* 23029 */ {(12<<2)|2,{45,57,0}},
/* 23030 */ {(12<<2)|2,{45,57,0}},
/* 23031 */ {(12<<2)|2,{45,57,0}},
/* 23032 */ {(12<<2)|2,{45,57,0}},
/* 23033 */ {(12<<2)|2,{45,57,0}},
/* 23034 */ {(12<<2)|2,{45,57,0}},
/* 23035 */ {(12<<2)|2,{45,57,0}},
/* 23036 */ {(12<<2)|2,{45,57,0}},
/* 23037 */ {(12<<2)|2,{45,57,0}},
/* 23038 */ {(12<<2)|2,{45,57,0}},
/* 23039 */ {(12<<2)|2,{45,57,0}},
/* 23040 */ {(12<<2)|2,{45,61,0}},
/* 23041 */ {(12<<2)|2,{45,61,0}},
/* 23042 */ {(12<<2)|2,{45,61,0}},
/* 23043 */ {(12<<2)|2,{45,61,0}},
/* 23044 */ {(12<<2)|2,{45,61,0}},
/* 23045 */ {(12<<2)|2,{45,61,0}},
/* 23046 */ {(12<<2)|2,{45,61,0}},
/* 23047 */ {(12<<2)|2,{45,61,0}},
/* 23048 */ {(12<<2)|2,{45,61,0}},
/* 23049 */ {(12<<2)|2,{45,61,0}},
/* 23050 */ {(12<<2)|2,{45,61,0}},
/* 23051 */ {(12<<2)|2,{45,61,0}},
/* 23052 */ {(12<<2)|2,{45,61,0}},
/* 23053 */ {(12<<2)|2,{45,61,0}},
/* 23054 */ {(12<<2)|2,{45,61,0}},
/* 23055 */ {(12<<2)|2,{45,61,0}},
/* 23056 */ {(12<<2)|2,{45,65,0}},
/* 23057 */ {(12<<2)|2,{45,65,0}},
/* 23058 */ {(12<<2)|2,{45,65,0}},
/* 23059 */ {(12<<2)|2,{45,65,0}},
/* 23060 */ {(12<<2)|2,{45,65,0}},
/* 23061 */ {(12<<2)|2,{45,65,0}},
/* 23062 */ {(12<<2)|2,{45,65,0}},
/* 23063 */ {(12<<2)|2,{45,65,0}},
/* 23064 */ {(12<<2)|2,{45,65,0}},
/* 23065 */ {(12<<2)|2,{45,65,0}},
/* 23066 */ {(12<<2)|2,{45,65,0}},
/* 23067 */ {(12<<2)|2,{45,65,0}},
/* 23068 */ {(12<<2)|2,{45,65,0}},
/* 23069 */ {(12<<2)|2,{45,65,0}},
/* 23070 */ {(12<<2)|2,{45,65,0}},
/* 23071 */ {(12<<2)|2,{45,65,0}},
/* 23072 */ {(12<<2)|2,{45,95,0}},
/* 23073 */ {(12<<2)|2,{45,95,0}},
/* 23074 */ {(12<<2)|2,{45,95,0}},
/* 23075 */ {(12<<2)|2,{45,95,0}},
/* 23076 */ {(12<<2)|2,{45,95,0}},
/* 23077 */ {(12<<2)|2,{45,95,0}},
/* 23078 */ {(12<<2)|2,{45,95,0}},
/* 23079 */ {(12<<2)|2,{45,95,0}},
/* 23080 */ {(12<<2)|2,{45,95,0}},
/* 23081 */ {(12<<2)|2,{45,95,0}},
/* 23082 */ {(12<<2)|2,{45,95,0}},
/* 23083 */ {(12<<2)|2,{45,95,0}},
/* 23084 */ {(12<<2)|2,{45,95,0}},
/* 23085 */ {(12<<2)|2,{45,95,0}},
/* 23086 */ {(12<<2)|2,{45,95,0}},
/* 23087 */ {(12<<2)|2,{45,95,0}},
/* 23088 */ {(12<<2)|2,{45,98,0}},
/* 23089 */ {(12<<2)|2,{45,98,0}},
/* 23090 */ {(12<<2)|2,{45,98,0}},
/* 23091 */ {(12<<2)|2,{45,98,0}},
/* 23092 */ {(12<<2)|2,{45,98,0}},
/* 23093 */ {(12<<2)|2,{45,98,0}},
/* 23094 */ {(12<<2)|2,{45,98,0}},
/* 23095 */ {(12<<2)|2,{45,98,0}},
/* 23096 */ {(12<<2)|2,{45,98,0}},
/* 23097 */ {(12<<2)|2,{45,98,0}},
/* 23098 */ {(12<<2)|2,{45,98,0}},
/* 23099 */ {(12<<2)|2,{45,98,0}},
/* 23100 */ {(12<<2)|2,{45,98,0}},
/* 23101 */ {(12<<2)|2,{45,98,0}},
/* 23102 */ {(12<<2)|2,{45,98,0}},
/* 23103 */ {(12<<2)|2,{45,98,0}},
/* 23104 */ {(12<<2)|2,{45,100,0}},
/* 23105 */ {(12<<2)|2,{45,100,0}},
/* 23106 */ {(12<<2)|2,{45,100,0}},
/* 23107 */ {(12<<2)|2,{45,100,0}},
/* 23108 */ {(12<<2)|2,{45,100,0}},
/* 23109 */ {(12<<2)|2,{45,100,0}},
/* 23110 */ {(12<<2)|2,{45,100,0}},
/* 23111 */ {(12<<2)|2,{45,100,0}},
/* 23112 */ {(12<<2)|2,{45,100,0}},
/* 23113 */ {(12<<2)|2,{45,100,0}},
/* 23114 */ {(12<<2)|2,{45,100,0}},
/* 23115 */ {(12<<2)|2,{45,100,0}},
/* 23116 */ {(12<<2)|2,{45,100,0}},
/* 23117 */ {(12<<2)|2,{45,100,0}},
/* 23118 */ {(12<<2)|2,{45,100,0}},
/* 23119 */ {(12<<2)|2,{45,100,0}},
/* 23120 */ {(12<<2)|2,{45,102,0}},
/* 23121 */ {(12<<2)|2,{45,102,0}},
/* 23122 */ {(12<<2)|2,{45,102,0}},
/* 23123 */ {(12<<2)|2,{45,102,0}},
/* 23124 */ {(12<<2)|2,{45,102,0}},
/* 23125 */ {(12<<2)|2,{45,102,0}},
/* 23126 */ {(12<<2)|2,{45,102,0}},
/* 23127 */ {(12<<2)|2,{45,102,0}},
/* 23128 */ {(12<<2)|2,{45,102,0}},
/* 23129 */ {(12<<2)|2,{45,102,0}},
/* 23130 */ {(12<<2)|2,{45,102,0}},
/* 23131 */ {(12<<2)|2,{45,102,0}},
/* 23132 */ {(12<<2)|2,{45,102,0}},
/* 23133 */ {(12<<2)|2,{45,102,0}},
/* 23134 */ {(12<<2)|2,{45,102,0}},
/* 23135 */ {(12<<2)|2,{45,102,0}},
/* 23136 */ {(12<<2)|2,{45,103,0}},
/* 23137 */ {(12<<2)|2,{45,103,0}},
/* 23138 */ {(12<<2)|2,{45,103,0}},
/* 23139 */ {(12<<2)|2,{45,103,0}},
/* 23140 */ {(12<<2)|2,{45,103,0}},
/* 23141 */ {(12<<2)|2,{45,103,0}},
/* 23142 */ {(12<<2)|2,{45,103,0}},
/* 23143 */ {(12<<2)|2,{45,103,0}},
/* 23144 */ {(12<<2)|2,{45,103,0}},
/* 23145 */ {(12<<2)|2,{45,103,0}},
/* 23146 */ {(12<<2)|2,{45,103,0}},
/* 23147 */ {(12<<2)|2,{45,103,0}},
/* 23148 */ {(12<<2)|2,{45,103,0}},
/* 23149 */ {(12<<2)|2,{45,103,0}},
/* 23150 */ {(12<<2)|2,{45,103,0}},
/* 23151 */ {(12<<2)|2,{45,103,0}},
/* 23152 */ {(12<<2)|2,{45,104,0}},
/* 23153 */ {(12<<2)|2,{45,104,0}},
/* 23154 */ {(12<<2)|2,{45,104,0}},
/* 23155 */ {(12<<2)|2,{45,104,0}},
/* 23156 */ {(12<<2)|2,{45,104,0}},
/* 23157 */ {(12<<2)|2,{45,104,0}},
/* 23158 */ {(12<<2)|2,{45,104,0}},
/* 23159 */ {(12<<2)|2,{45,104,0}},
/* 23160 */ {(12<<2)|2,{45,104,0}},
/* 23161 */ {(12<<2)|2,{45,104,0}},
/* 23162 */ {(12<<2)|2,{45,104,0}},
/* 23163 */ {(12<<2)|2,{45,104,0}},
/* 23164 */ {(12<<2)|2,{45,104,0}},
/* 23165 */ {(12<<2)|2,{45,104,0}},
/* 23166 */ {(12<<2)|2,{45,104,0}},
/* 23167 */ {(12<<2)|2,{45,104,0}},
/* 23168 */ {(12<<2)|2,{45,108,0}},
/* 23169 */ {(12<<2)|2,{45,108,0}},
/* 23170 */ {(12<<2)|2,{45,108,0}},
/* 23171 */ {(12<<2)|2,{45,108,0}},
/* 23172 */ {(12<<2)|2,{45,108,0}},
/* 23173 */ {(12<<2)|2,{45,108,0}},
/* 23174 */ {(12<<2)|2,{45,108,0}},
/* 23175 */ {(12<<2)|2,{45,108,0}},
/* 23176 */ {(12<<2)|2,{45,108,0}},
/* 23177 */ {(12<<2)|2,{45,108,0}},
/* 23178 */ {(12<<2)|2,{45,108,0}},
/* 23179 */ {(12<<2)|2,{45,108,0}},
/* 23180 */ {(12<<2)|2,{45,108,0}},
/* 23181 */ {(12<<2)|2,{45,108,0}},
/* 23182 */ {(12<<2)|2,{45,108,0}},
/* 23183 */ {(12<<2)|2,{45,108,0}},
/* 23184 */ {(12<<2)|2,{45,109,0}},
/* 23185 */ {(12<<2)|2,{45,109,0}},
/* 23186 */ {(12<<2)|2,{45,109,0}},
/* 23187 */ {(12<<2)|2,{45,109,0}},
/* 23188 */ {(12<<2)|2,{45,109,0}},
/* 23189 */ {(12<<2)|2,{45,109,0}},
/* 23190 */ {(12<<2)|2,{45,109,0}},
/* 23191 */ {(12<<2)|2,{45,109,0}},
/* 23192 */ {(12<<2)|2,{45,109,0}},
/* 23193 */ {(12<<2)|2,{45,109,0}},
/* 23194 */ {(12<<2)|2,{45,109,0}},
/* 23195 */ {(12<<2)|2,{45,109,0}},
/* 23196 */ {(12<<2)|2,{45,109,0}},
/* 23197 */ {(12<<2)|2,{45,109,0}},
/* 23198 */ {(12<<2)|2,{45,109,0}},
/* 23199 */ {(12<<2)|2,{45,109,0}},
/* 23200